// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//                             
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit Global; // Global definitions

// ----------------------------------------------------------------------------
// Contains global definitions and some frequent used functions.
// ----------------------------------------------------------------------------

interface

uses windows, messages, classes, filectrl, stdctrls, Graphics,
     cServerBase, cServerNNTP, cServerPOP3, cServerIMAP, cServerSMTP, cServerReCo;

Var
  hWndMainWindow: THandle = 0;

Const
  WM_HAMSTER    = WM_USER + 1;
  WM_TRAYNOTIFY = WM_USER + 2;
  WM_NOTIFYGUI  = WM_USER + 3;

  HAM_MSG_HAMSTER_EXIT    = 1; // -
  HAM_MSG_RESET_COUNTERS  = 2; // -
  HAM_MSG_LOCALNNTP_ONOFF = 3; // 0/1
  HAM_MSG_LOCALPOP3_ONOFF = 4; // 0/1
  HAM_MSG_LOCALSMTP_ONOFF = 5; // 0/1
  HAM_MSG_LOCALIMAP_ONOFF = 14;// 0/1 //IMAP
  HAM_MSG_LOCALRECO_ONOFF = 11;
  HAM_MSG_SHOWWINDOW      = 6; // 0/1 + [2..10]
  HAM_MSG_SHOWICON        = 7; // 0/1
  HAM_MSG_STOPALLTASKS    = 8; // -
  HAM_MSG_CONFIG_RELOAD   = 9; // 0/1
  HAM_MSG_CONFIG_IPAccess = 10; //     //JW {Reload IPaccess}
  HAM_MSG_DIRS            = 12;
  HAM_MSG_SCRIPTS         = 13;
  HAM_MSG_NEWSKILLFILELOG = 15;
  HAM_MSG_PULLNEWGROUPS   = 16;

  GUI_MSG_SERVERSTATE_CHANGED = 1;
  GUI_MSG_RELOAD_SCRIPTS = 2;

  CR   = #13;                           // JAWO!! Params 8.10.2000 (new definitions)
  LF   = #10;
  FF   = #12;
  CRLF = CR+LF;

  INTERNALGROUP_DEFAULT        = 0;
  INTERNALGROUP_POSTOK         = 1;
  INTERNALGROUP_POSTERRORS     = 2;
  INTERNALGROUP_STATISTICS     = 3;
  INTERNALGROUP_CANCELNOTIFY   = 4;
  INTERNALGROUP_NEWGROUPS      = 5;
  INTERNALGROUP_UNKNOWNGROUP   = 6;
  INTERNALGROUP_PULLERRORS     = 7;
  INTERNALGROUP_LASTGROUP      = INTERNALGROUP_PULLERRORS;
  INTERNALGROUP_NOTSET_DEFAULT = 'internal.misc';

  EXT_DAT = '.dat';
  EXT_IDX = '.idx';
  EXT_CFG = '.ini';

var
  INTERNALGROUP: array[INTERNALGROUP_DEFAULT..INTERNALGROUP_LASTGROUP] of String;
  MUTEXSTRING  : String; {JW}
     {JW} // handle for public Event
   {JW} {purge event}
   EventPurge        : THandle;
   {JW}
   EventMain         : THandle;
   EventMailIn       : THandle;
   EventMailOut      : THandle;
   EventMailInternal : Thandle;
   EventNewsOut      : THandle;
   EventRASConnected : Thandle;
   EventRASHangup    : Thandle;
   {JW}


Const
  OUR_X_HEADER  = 'X-Hamster-Info:';
  GENERATEMID_NEVER     = 0;
  GENERATEMID_IFNOTSET  = 1;
   
Var
  OUR_VERINFO   : String = 'Hamster';  // rest siehe init
  XREF_HOSTNAME : String = 'localhost';

  PATH_BASE     : String = ''; // Pfade werden im Initialization-Abschnitt initialisiert
  PATH_HSC      : String = '';
  PATH_HSC_RC   : String = '';
  PATH_HSM      : String = '';
  PATH_LOGS     : String = '';
  PATH_SERVER   : String = '';
  PATH_GROUPS   : String = '';
  PATH_MAILS    : String = '';
  PATH_NEWS_OUT : String = '';
  PATH_MAIL_OUT : String = '';
  PATH_NEWS_ERR : String = '';
  PATH_TRASH    : String = ''; 

Const
  CFGFILE_SERVER_NNTP = 'Server.hst';   // PATH_BASE
  CFGFILE_SERVER_POP3 = 'SrvPOP3.hst';  // PATH_BASE
  CFGFILE_SERVER_SMTP = 'SrvSMTP.hst';  // PATH_BASE

  CFGFILE_ACTIVE  = 'Groups.hst';        // PATH_BASE
  CFGFILE_PULLS   = 'Pulls.hst';
  CFGFILE_SCORES  = 'Scores.hst';
  CFGFILE_SCORELOG = 'Kills.log';
  CFGFILE_SCORELOG_INFO = 'KillsStat.log';
  CFGFILE_MFILTER = 'MailFilt.hst';
  CFGFILE_IPACCESS= 'IPAccess.hst';
  CFGFILE_DEFSCRIPT = 'DefScript.txt';
  CFGFILE_DEFModule = 'DefModule.txt';
  CFGFILE_SHELLMOD = 'hs2Shell.hsm';
  CFGFILE_PASS    = 'Password.!!!';
  CFGFILE_ALIAS   = 'MAlias.hst'; {JW} {MailAlias}

  SRVFILE_ALLDESCS     = 'alldescs.txt';

  SRVFILE_INI          = 'Server.ini';
  SRVFILE_HELPTEXT     = 'help.txt';
  SRVFILE_GROUPS       = 'groups.txt';
  SRVFILE_GRPDESCS     = 'grpdescs.txt';
  SRVFILE_OVERVIEWFMT  = 'overview.txt';
  SRVFILE_GREETING     = 'greeting.txt';
  SRVFILE_GETMIDLIST   = 'getmids.txt';

  CFGFILE_RANDSEED     = 'RandSeed.!!!';    // File to save random seed {JW+MG}{PRNG}

Var
  {JW} {End}
//  CanCloseNow   : Boolean = True;
  CanCloseNow   : Integer=0;
  CloseByScript : Boolean = False;
  {JW}
  CriticalState : Integer = 0;
  RunAsService  : Boolean = False;
  AllShutDownReq: Boolean = false;
  ArchivMode: Boolean = false;

Const
  EnvelopeMAILFROM = '!MAIL FROM: ';
  EnvelopeRCPTTO = '!RCPT TO: ';

  CFG_LOCAL_AUTOSTART_NNTP = 'local.autostart.nntp';
  CFG_LOCAL_AUTOSTART_POP3 = 'local.autostart.pop3';
  CFG_LOCAL_AUTOSTART_SMTP = 'local.autostart.smtp';
  CFG_LOCAL_AUTOSTART_RECO = 'local.autostart.reco';

Var
{JW}{Socket Access}
  Def_LocalNNTPServerBind         : String  = '127.0.0.1';
  Def_LocalPOP3ServerBind         : String  = '127.0.0.1';
  Def_LocalSMTPServerBind         : String  = '127.0.0.1';
  Def_LocalReCoServerBind         : String  = '127.0.0.1';

  {MG}{SSL}
  SSLReady : Boolean = False;
  Def_SSLCipherString : String = 'HIGH';
  Def_UseSSLv3 : Boolean = True;
  Def_UseTLSv1 : Boolean = True;

  Def_LocalNntpTlsMode : Integer = 0;
  Def_LocalPop3TlsMode : Integer = 0;
  Def_LocalSmtpTlsMode : Integer = 0;
  Def_LocalRecoUseTls  : Boolean = False;
  {/SSL}
  
  Def_Pull_Limit              : Integer = -100;
  Def_Pull_Limit_First        : Integer =  100;

  {HSR} {CAPA1.0}
  Def_MailExpire              : String  = 'NEVER';
  Def_Login_Delay             : Integer = 30;
  {/HSR}
  Def_IgnoreConnectionLost    : Boolean = False;  // JAWO 26.01.02

{JW} {N2M}
  Def_Flup                        : Boolean = True;
  Def_Gate_UseHamsterEnvelope     : Boolean = True;
{JW}

  {JAWO|HSR}
  Def_parts_make              : Boolean = False;
  Def_parts_size_Min          : Integer = 250;  // default block size of parted pulls
  Def_parts_size_Max          : Integer = 300;  // max. block size of parted pulls (single/last one)
  {/JAWO|HSR}

  Def_ShowShell               : Integer = 0;
  Def_AdvancedConfiguration   : Boolean = false;

  Def_OLE_Server_Name: String = 'Hamster'; {JW}
  Def_OLE_Server: Boolean = true; {JW} {En/Disable OLE}

  Def_Purge_Daily             : Boolean = false;
  Def_Purge_Articles_KeepDays : Integer =    7;
  Def_Purge_History_KeepDays  : Integer =    7;
  Def_Purge_Kills_KeepDays    : Integer =    7;
  Def_Purge_MHistory_KeepDays : Integer =   14;
  Def_LocalNntpServer_Port    : Integer =  119;
  Def_LocalPop3Server_Port    : Integer =  110;
  Def_LocalSmtpServer_Port    : Integer =   25;
  Def_LocalReCoServer_Port    : Integer =   23;
  Def_LocalTimeoutInactivity  : Integer =   15; // min.
  Def_LocalNNTPLoginTimeout   : Integer = 10000; // msec. {JW} {Login}
  Def_LocalPOP3LoginTimeout   : Integer = 10000; // msec. {JW} {Login}
  Def_LocalSMTPLoginTimeout   : Integer = 10000; // msec. {JW} {Login}
  Def_LocalReCoLoginTimeout   : Integer = 10000; // msec. {JW} {Login}
  Def_Max_Local_NNTP_Servers  : Integer = 12; {JW} {LocalLimit}
  Def_Max_Local_POP3_Servers  : Integer = 10; {JW} {LocalLimit}
  Def_Max_Local_SMTP_Servers  : Integer = 3; {JW} {LocalLimit}
  Def_Max_Local_ReCo_Servers  : Integer = 1;
  Def_Max_Local_NNTP_Servers_Per_IP: Integer = 6;
  Def_Max_Local_POP3_Servers_Per_IP: Integer = 5;
  Def_Max_Local_SMTP_Servers_Per_IP: Integer = 3;
  Def_Max_Local_ReCo_Servers_Per_IP: Integer = 1;
  Def_LocalTimeoutQuitDelay   : Integer =  100; // msec.
  Def_SmtpAfterPop3Period     : Integer =  300; // sec.
  Def_RemoteTimeoutConnect    : Integer =  120; // sec.
  Def_RemoteTimeoutCommand    : Integer =  120; // sec.
  Def_PasswordCodeBase        : Integer =    0;
  Def_MaxScriptPWNumber       : Integer =   30;
  Def_LocalAuthReqSmtp        : Boolean = False;
  Def_LocalAuthReqESmtp       : Boolean = False;
  Def_LocalMailReqNotAuth     : Boolean = False;  
  Def_LocalFalseAuthCloseConnection
                              : Boolean = true;
  
Const
  SMTPSupportedSASL           = 'CRAM-SHA1 CRAM-MD5 LOGIN PLAIN'; {JW} {Local SASL}
  POP3SupportedSASL           = 'CRAM-SHA1 CRAM-MD5 LOGIN PLAIN'; {JW} {Local SASL}
  ReCoSupportedSASL           = 'CRAM-SHA1 CRAM-MD5 LOGIN PLAIN'; {JW} {Local SASL}
  POP3LocalSASL               = 'CRAM-SHA1 CRAM-MD5 LOGIN PLAIN';
  //POP3LocalSASL               = 'DIGEST-MD5 CRAM-SHA1 CRAM-MD5 LOGIN PLAIN';
  SMTPLocalSASL               = 'CRAM-SHA1 CRAM-MD5 LOGIN PLAIN';

Var
  Def_LocalSmtpSASL           : String = SMTPSupportedSASL; {JW} {Local SASL}
  Def_LocalPOP3SASL           : String = POP3SupportedSASL; {JW} {Local SASL}
  Def_LocalReCoSASL           : String = ReCoSupportedSASL; {JW} {Local SASL}

  Def_LocalNntpPostToUnknown  : Boolean = False;
  Def_LocalNntpSearchUnknownMIDs: Boolean = true;
  Def_NNTPAutoGetServerInfos  : Boolean = true;
  Def_NNTPDropResidualJobs    : Boolean = true;
  Def_LocalLimitLineLenNntp   : Integer = 1000;
  Def_LocalLimitLineLenPop3   : Integer = 1000;
  Def_LocalLimitLineLenSmtp   : Integer = 1000;
  Def_LocalLimitLineLenReCo   : Integer = 1000;
  Def_LocalLimitTextSizeNntp  : Integer = 2097152;
  Def_LocalLimitTextSizePop3  : Integer = 2097152;
  Def_LocalLimitTextSizeSmtp  : Integer = 2097152;
  Def_LocalLimitTextSizeReCo  : Integer = 2097152;
  Def_GenerateNewsMID         : Integer = GENERATEMID_NEVER;
  Def_GenerateMailMID         : Integer = GENERATEMID_NEVER;
  Def_BounceMailToSender      : Boolean = false; {JW} {bounce local}
  Def_MID_FQDN_local          : Boolean = true; {JW} {FQDN_MID_Local}
  Def_FQDNforMIDs             : String = '';
  Def_FQDN                    : String = '';
  Def_Mail_RemoveMids         : String = '';
  Def_Crypt_MID               : Boolean = false;  //JW {MID-Crypt}
  Def_NNTPClient_ModeReader   : Boolean = True; {Feeder} {JW}
  Def_Junkfeeder              : boolean =false; {Feeder} {JW}
  Def_Mail_AddXHamster        : Boolean = True;
  Def_Mail_AddReceived        : Boolean = True;
  Def_POP3_AddReceived        : Boolean = True; {JW} {Def_POP3_AddReceived}
  Def_POP3Delay               : integer = 0; {JW} {POP3 delay}
  Def_SMTPDelay               : integer = 0; {JW} {SMTP delay}
  Def_News_AddXHamster        : Boolean = False;
  Def_News_AddUserAgent       : Boolean = true;
  Def_News_AddXHTrace         : String = '';
  Def_News_LocalInjection     : Boolean = false; {JW} {li}
  Def_News_AddPath            : Boolean = false; {JW} {li}
  Def_News_XOverWithoutCRLF   : Boolean = true;

  Def_News_FeededCancel       : Boolean = false;
  Def_News_FeededSupersedes   : Boolean = false;
  Def_News_FeededCancelVerify : Boolean = True;
  Def_News_FeededCancelControlMsg: Boolean = false;
  Def_News_FeededCancelDelete : Boolean = false;

  Def_MaxUnknownGroupsInRe    : Integer = 2;
  Def_ListRequireAuth         : Boolean = True; {JW} {List Auth}
  Def_HistoryChunkBits        : Integer =    4;
  Def_IsLocalDomain           : String = '.*hamster|\[?127\.0\.0\.1\]?|.*invalid';
  Def_PostServer              : String = '';
  Def_SmtpServer              : String = '';
  Def_LeaveMailsOnServer      : Boolean = True;
  Def_FilterByUIDL            : Boolean = False;
  Def_MailSizeIgnore          : Integer = 0;
  Def_MailSizeKill            : Integer = 0;
  Def_MailSizeNotify          : Integer = 0;
  Def_NoOfTopLinesToLoad      : Integer = 20;
  Def_ScoreLogLimit           : Integer = -9999;
  Def_FilterNewGroupsInfo     : String = '';
  Def_GatewayMIDExtension     : String = 'n2m-g.';

{PW} {Killfilelog, DeleteThread}
   Def_KillLogDeleteThreadVal  : String = '=-9999';
   Def_KillLogDeleteThreadSec  : String = '[*]';
{/PW}
{PW} {Killfilelog, RetrieveThread}
   Def_KillLogRetrieveThreadVal: String = '=+9999';
   Def_KillLogRetrieveThreadSec: String = '[*]';
{/PW}
  Def_Postmaster              : String = 'admin';  //JW //Postmaster
{JW} {NNTPINFOMAIL}
   Def_NNTPINFOMAIL: Boolean = True;
   Def_Usenet      : String  = 'admin';
{/JW}

  Def_SendMailAttemptsMax     : Integer = 3;
  Def_SendMailAttemptsDel     : Boolean = False;
  Def_SendInfoMailLocalOnly   : Boolean = TRUE;   //JW //LOCAL_ONLY_ADMIN
  Def_MinimizeOnClose         : Boolean = False;
  Def_HideMenuItems           : String = '';
  Def_AutoActivateErrorLog    : Boolean = False;
  Def_EnvelopeFrom            : String = 'admin';
  Def_EditorApp               : String = 'notepad.exe';
  Def_EditorParams            : String = '%1';

{PW} {write Expire}{13.06.2001}
  Def_KillLogDeleteThreadExp  : Integer = 14;
  Def_KillLogRetrieveThreadExp: Integer = 14;
{/PW}

{JW} {Peer}
  Def_IHAVE_Auth              : Boolean = True;
  Def_Stream_Mode             : Boolean = False;
{JW}
{JW} {Tasks}
  Def_MaxTasks                : Integer = 0;
{JW}

  {JW} {MID}
  MID_Counter    : LongInt = 0;
  MID_OldDateTime: LongInt = 0;
  {/JW}

  Def_Cfg_Tree_Fullexpand     : Boolean = True;                            //HSR //Groups

  LOGFILEREFRESH: Integer = 5;

  CntByteIn     : Int64 = 0;
  CntByteOut    : Int64 = 0;
  CntMailNew    : LongInt = 0;
  CntArtNew     : LongInt = 0;
  CntArtLoad    : LongInt = 0;
  CntArtHist    : LongInt = 0;
  CntArtKill    : LongInt = 0;
  CntArtByMid   : LongInt = 0; //HSR //GetByMid-Counter
  CntActiveTasks: LongInt = 0;
  CntCustomValues: Array[0..9] of Longint = (0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
  {JW} {Tasks}
  CntWaitTasks    : LongInt = 0;
  {JW}
  CntOutboxChk  : LongInt = 0;
  CntOutboxM    : LongInt = 0;
  CntOutboxN    : LongInt = 0;
  CntWarnings   : Longint = 0;
  CntErrors     : Longint = 0;
  CntNewsErr    : Longint = 0;
  CntDoLogList  : LongInt = 0;

  ColoredTabs          : boolean = true;
  TabColorErrorText    : LongInt = clWhite;
  TabColorErrorBrush   : LongInt = clRed;
  TabColorWarningText  : LongInt = clBlack;
  TabColorWarningBrush : LongInt = clYellow;

CONST //IMAP
  CFG_LOCAL_AUTOSTART_IMAP = 'local.autostart.IMAP';
VAR //IMAP
  Def_IMAP_ID                 : Boolean = True; //JW //IMAP ID
  Def_Max_Local_IMAP_Servers  : Integer = 10;
  Def_LocalIMAPLoginTimeout   : Integer = 10000; // msec.
  Def_LocalLimitTextSizeIMAP  : Integer = 2097152;
  DEF_LOCALIMAPServer_PORT    : Integer = 143;
  Def_IMAPDelay               : integer = 0;
  Def_LocalIMAPServerBind     : String  = '127.0.0.1';
  CntIMAPCli                  : LongInt = 0;
  Def_LocalImapTlsMode        : Integer = 0;
  Def_Max_Local_IMAP_Servers_Per_IP : Integer = 10;
  Def_IMAP_DisableSASLLogin   : Boolean = true; //JW //IMAP-AUTH
  Def_IMAP_DisableLogin       : Boolean = true;
  Def_Mail_ExpiredFilterentries_Delete: Boolean = false;
  Def_News_ExpiredFilterentries_Delete: Boolean = false;
  Def_IMAPNCBrain             : Boolean = false; //NCBrain
  Def_IMAP8bitFilter          : boolean = false; //JW //IMAP 7Bit

var
  CS_COUNTER         : TRTLCriticalSection;
  CS_MAINTENANCE     : TRTLCriticalSection;
  CS_SAVE_ARTICLE    : TRTLCriticalSection;
  CS_LOCK_NEWSOUT    : TRTLCriticalSection;
  CS_LOCK_MAILBOX_ALL: TRTLCriticalSection;
  CS_LOCK_MAILOUT_USE: TRTLCriticalSection;
  CS_LOCK_MAILOUT_ADD: TRTLCriticalSection;
  CS_LOCK_InCancel   : TRTLCriticalSection; //HSR //ModeCancel
  CS_Limiter         : trtlCriticalSection;
  CS_LOCK_InFeed     : TRTLCriticalSection;
  CS_TEST_ARTICLE    : TRTLCriticalSection; //HRR:
  CS_MID             : TRTLCriticalSection; //JW //MID
  CS_Event           : TRTLCriticalSection; {JW} {critical event}
  CS_SHUTDOWNLIST    : TRTLCriticalSection; //HSR //One_SD
  CS_IMAPMBCreate    : TRTLCriticalSection; //HSR //MBCreate

  Def_Check_Path     : Boolean=true; {JW} {Check Path}

  EVT_STOPSCRIPT: THandle;
  EVT_ISIDLE    : THandle;

type
  TLocalServerTypes     = ( stNONE, stNNTP, stPOP3, stSMTP, stRECO, stIMAP );
  TLocalServerActivates = ( saSTOP, saSTART, saRESTART );

var
  LocalNntpServer: TSrvNNTP = nil;
  LocalPop3Server: TSrvPOP3 = nil;
  LocalSmtpServer: TSrvSMTP = nil;
  LocalReCoServer: TSrvRECO = NIL;
  LocalImapServer: TSrvIMAP = NIL;

  CntNntpCli   : LongInt = 0;
  CntPop3Cli   : LongInt = 0;
  CntSmtpCli   : LongInt = 0;

procedure ScriptAddLog( const Info: WideString; Typ: Integer );

procedure IncCounter( var CntVar: LongInt; Const IncVal: LongInt ); overload;
procedure SetCounter( var CntVar: LongInt; Const NewVal: LongInt ); overload;
function  GetCounter( var CntVar: LongInt ): LongInt; overload;

procedure IncCounter( var CntVar: Int64; Const IncVal: Int64 ); overload;
procedure SetCounter( var CntVar: Int64; Const NewVal: Int64 ); overload;
function  GetCounter( var CntVar: Int64 ): Int64; overload;

procedure CountOutboxes;

function  CreateEventGrandAccess(ManualReset,
   InitalState: Boolean; MutexString:PChar): THandle; {JW XP Event}

Function LockApplication: boolean;
procedure UnlockApplication;

function LocalServerTypeToStr( ServerType: TLocalServerTypes ): String;
function LocalServerStrToType( const ServerStr: String ): TLocalServerTypes;
function LocalServerIsActive( ServerType: TLocalServerTypes ): Boolean;
function LocalServerActivate( ServerType: TLocalServerTypes;
                              ServerActivate: TLocalServerActivates ): Boolean;
function LocalServerGetState( const Typ, Server: Integer ): Integer;

// JAWO!! Params 8.10.2000 (new function)
function  ScriptAvail (var ScriptFile, ParameterList:  string;
                       var WaitFlag : boolean): integer;

procedure HamFileAppendList( const Filename: String; const TS: TStringList );
/// ersetzt: uTools.AppendToFile( TS: TStringList; Filename: String );
procedure HamFileAppendLine( const Filename, LineText: String );
/// ersetzt: uTools.AppendLineToFile( Filename, LineText: String );
procedure HamFileRewriteLine( const Filename, LineText: String );
/// ersetzt: uTools.WriteLineToFile( Filename, LineText: String );

procedure Init_InternalGroups;

implementation

uses SysUtils, IniFiles, uTools, Forms, cStdForm, cIPAccess, Config, cLogFile;

procedure ScriptAddLog( const Info: WideString; Typ: Integer );
Var x: Word;
begin
   Case Typ of
      1: x := LOGID_DEBUG;
      2: x := LOGID_DETAIL;
      3: x := LOGID_INFO;
      4: x := LOGID_SYSTEM;
      5: x := LOGID_WARN;
      6: x := LOGID_ERROR;
      7: x := LOGID_STATUS;
      else x := LOGID_INFO
   end;
   Log( x, Info )
end;

{JW XP Event} {mit Untersttzung von Andreas Spangenberg erstellt}
function CreateEventGrandAccess (ManualReset,
   InitalState: Boolean; MutexString: PChar): THandle;
var
  saMutexSecurity : Security_Attributes;
  pMutexSD        : pSecurityDescriptor;
begin
  new(pMutexSD);
  try
    InitializeSecurityDescriptor(pMutexSD,SECURITY_DESCRIPTOR_REVISION);
    SetSecurityDescriptorDacl( pMutexSD, TRUE, nil, false);
    saMutexSecurity.nLength              := sizeof( SECURITY_ATTRIBUTES );
    saMutexSecurity.bInheritHandle       := False;
    saMutexSecurity.lpSecurityDescriptor := pMutexSD;
    {JW} {critical event}
    EnterCriticalSection(CS_Event);
    result := CreateEvent(@saMutexSecurity,
                          ManualReset,
                          InitalState,
                          MutexString);
    LeaveCriticalSection(CS_Event);
    {JW}
  finally
    Dispose(pMutexSD);
  end;
end;

{JW XP Mutex} {mit Untersttzung von Andreas Spangenberg erstellt}
function CreateMutexGrandAccess(bInitialOwner:Boolean;MutexString:PChar):
THandle;
var
   saMutexSecurity : Security_Attributes;
   pMutexSD        : pSecurityDescriptor;
begin
   new(pMutexSD);
//  GetMem(pGrandAcl,1000);
   try
     InitializeSecurityDescriptor(pMutexSD,SECURITY_DESCRIPTOR_REVISION);
     SetSecurityDescriptorDacl( pMutexSD, TRUE, nil, false);
     saMutexSecurity.nLength              := sizeof( SECURITY_ATTRIBUTES) ;
     saMutexSecurity.bInheritHandle       := False;
     saMutexSecurity.lpSecurityDescriptor := pMutexSD;
     result := CreateMutex(@saMutexSecurity, bInitialOwner,MutexString);
   finally
     Dispose(pMutexSD);
   end;
end;
{JW}

procedure IncCounter( var CntVar: LongInt; Const IncVal: LongInt );
begin
     EnterCriticalSection( CS_COUNTER );
     inc( CntVar, IncVal );
     LeaveCriticalSection( CS_COUNTER );
end;

procedure SetCounter( var CntVar: LongInt; Const NewVal: LongInt );
begin
     EnterCriticalSection( CS_COUNTER );
     CntVar := NewVal;
     LeaveCriticalSection( CS_COUNTER );
end;

function GetCounter( var CntVar: LongInt ): LongInt;
begin
     EnterCriticalSection( CS_COUNTER );
     Result := CntVar;
     LeaveCriticalSection( CS_COUNTER );
end;

procedure IncCounter( var CntVar: Int64; Const IncVal: Int64 );
begin
     EnterCriticalSection( CS_COUNTER );
     inc( CntVar, IncVal );
     LeaveCriticalSection( CS_COUNTER );
end;

procedure SetCounter( var CntVar: Int64; Const NewVal: Int64 );
begin
     EnterCriticalSection( CS_COUNTER );
     CntVar := NewVal;
     LeaveCriticalSection( CS_COUNTER );
end;

function GetCounter( var CntVar: Int64 ): Int64;
begin
     EnterCriticalSection( CS_COUNTER );
     Result := CntVar;
     LeaveCriticalSection( CS_COUNTER );
end;

procedure CountOutboxes;

   Function Count (Const Path, Ext: String): Longint;
   var  SR : TSearchRec;
   begin
      Result := 0;
      If FindFirst( Path + '*.' + Ext, faAnyFile, SR ) = 0 then begin
         Repeat Inc(Result) until FindNext(SR) <> 0;
         FindClose(SR)
      end
   end;

begin
     EnterCriticalSection( CS_COUNTER );
     try
        CntOutboxChk := 0;
        CntOutboxM := Count (PATH_MAIL_OUT, 'msg');
        CntOutboxN := Count (PATH_NEWS_OUT, CfgIni.ReadString( 'Setup', 'news.ext.out', 'msg' ));
        CntNewsErr := Count (PATH_NEWS_ERR, 'msg');
     finally
        LeaveCriticalSection( CS_COUNTER );
     end;
end;


Var HamsterMutex: THandle = 0;
// JAWO!! Params 12.12.2000 (Parameter-Reihenfolge)

function  ScriptAvail (var ScriptFile, ParameterList :  string;
                        var WaitFlag : boolean): integer;
var
    i, n: Integer;
    b, e: boolean; // JAWO ScriptAvail 18.12.2000 (new return value)
begin
    Result := 0;
    WaitFlag := false;
    ScriptFile := '';
    ParameterList := '';
    e := false;
    n := 1;                               // number of investigated parameter
    b := (n <= ParamCount);               // are there parameters at all?
    if b then begin
       while b do begin
          ScriptFile := ParamStr (n);     // parameter to investigate
          b := true;
          if (copy (ScriptFile, 1, 10)='-Embedding') then begin // OLE: -Embedding
          end else if (LowerCase( copy(ScriptFile,1,4))='/svc') then begin ///ff.
             RunAsService := True
          end else if (LowerCase (copy (ScriptFile, 1, 2))='/w') then begin // wait flag found,
             WaitFlag := true
          end else if (LowerCase (copy (ScriptFile, 1, 3))='/rw') then begin // wait flag found,
             ArchivMode := false
          end else if (LowerCase (copy (ScriptFile, 1, 3))='/ro') then begin // wait flag found,
             ArchivMode := true
          end else if (LowerCase (copy (ScriptFile, 1, 1))='/') then begin // junk found, complain
             e := true; b := false
          end else begin
             b := false
          end;
          if b then begin                 // program parameter found,
             Inc (n);                     // scriptfile must be the next parameter
             b := (n <= ParamCount);      // are there more parameters?
          end;
       end;  // while
       if (n > ParamCount) then begin     // no parameter left for scriptfile name
          ScriptFile := '';               // clear variable (contained last program parameter)
       end else begin                     // there's at least one parameter left
          if (n < ParamCount) then begin  // but are there script parameters?
             Inc (n);
             ParameterList := ParamStr (n); // first script parameter
             for i := n+1 to ParamCount do // add the other script parameters
                ParameterList := ParameterList + CR + ParamStr (i);
          end;
          if (e) then begin
            Result := 2  // 'ScriptFile' contains invalid parameter
          end else begin
            Result := 1; // 'ScriptFile' contains script file name
          end;
       end;
    end;  // if b
end;


Function LockApplication: boolean;
var  Classstring: PChar;
     s, ScriptFile, ParameterList: String;
     X: TCopyDataStruct;
     hdl: THandle;
     WaitFlag: boolean;
     i:integer;
     Ergebnis, ScriptTimeout:Longword;
begin
     Result := true;
{JW XP Mutex}
     If not IsWindowsXP then begin
       HamsterMutex := CreateMutex( nil, False, PChar(MutexString) );
       if WaitForSingleObject( HamsterMutex, 0 )=WAIT_OBJECT_0 then exit;
     end else begin
       HamsterMutex:=CreateMutexGrandAccess(False,PChar('Global\'+MutexString));
       if HamsterMutex=0 then begin
         result:=false;
         LogFile.AddMain ( LOGID_ERROR, TrGlF (kLog, 'MainMutex.init.failed',
           'Initialization of primary mutex failed, Error %s', IntToStr(GetLastError)) );
         exit
       end;
       if WaitForSingleObject( HamsterMutex, 0 )=WAIT_OBJECT_0 then exit;
     end;
{JW}
     CloseHandle( HamsterMutex );

     HamsterMutex := 0;

     i := ScriptAvail (ScriptFile, ParameterList, WaitFlag);
     If ( i <> 0 ) then begin // JAWO ScriptAvail 18.12.2000 (new return value)
        s := ScriptFile;
        if (ParameterList <> '') then
           s := s + LF + ParameterList;
        s := s + FF + IntToStr (Ord (WaitFlag)); // add wait flag
        s := s + FF + MutexString; // add mutex string
        // Format: <script> (LF <Param> [(CR <Param>)]*)? FF (0|1) FF <mutex>
        X.dwData := i; // JAWO ScriptAvail 18.12.2000 (new return value)
        X.cbData := length(s);
        X.lpData := @s[1];
        hdl := FindWindow( 'THamsterMainWindow', nil );
        hdl:=GetWindow(hdl,GW_HWNDFIRST);
        Classstring:=StrAlloc(255);
        while hdl<>0 do begin
           hdl:=GetWindow(hdl,GW_HWNDNEXT);
           GetClassName(hdl,Classstring,255);
           If PChar(Classstring)='THamsterMainWindow' then begin
              ScriptTimeout := CfgIni.ReadInteger( 'Setup', 'StartUp.ScriptTimeout', 30000 );
              if ScriptTimeout = 0 then begin
                 SendMessage( hdl, WM_COPYDATA, 41, LongInt(@X) )
              end else begin
                 If SendMessageTimeout( hdl, WM_COPYDATA, 41, LongInt(@X),
                    SMTO_ABORTIFHUNG, ScriptTimeout, Ergebnis)=0
                 then begin
                    LogFile.AddMain ( LOGID_Error, 'Error '+IntToStr(GetLasterror)+
                             ' when send script to first instance')
                 end
              end
           end
        end;
        {JW}
     end;
     Result := false
end;

procedure UnlockApplication;
begin
     if HamsterMutex<>0 then begin
        ReleaseMutex( HamsterMutex );
        CloseHandle( HamsterMutex );
     end;
end;

// Local Server-Functions

function LocalServerTypeToStr( ServerType: TLocalServerTypes ): String;
begin
   case ServerType of
      stNNTP: Result := 'NNTP';
      stPOP3: Result := 'POP3';
      stSMTP: Result := 'SMTP';
      stRECO: Result := 'ReCo';
      stIMAP: Result := 'IMAP'; //IMAP
      else    Result := 'NONE';
   end;
end;

function LocalServerStrToType( const ServerStr: String ): TLocalServerTypes;
var  s: String;
begin
   s := UpperCase( ServerStr );
   if      s='NNTP' then Result := stNNTP
   else if s='POP3' then Result := stPOP3
   else if s='SMTP' then Result := stSMTP
   else if s='RECO' then Result := stRECO
   else if s='IMAP' then Result := stIMAP //IMAP
   else                  Result := stNONE;
end;

function LocalServerIsActive( ServerType: TLocalServerTypes ): Boolean;
begin
   Result := False;

   case ServerType of
      stNNTP: if Assigned( LocalNntpServer ) then
                 if LocalNntpServer.Active then Result := True;
      stPOP3: if Assigned( LocalPop3Server ) then
                 if LocalPop3Server.Active then Result := True;
      stSMTP: if Assigned( LocalSmtpServer ) then
                 if LocalSmtpServer.Active then Result := True;
      stRECO: if Assigned( LocalReCoServer ) then
                 if LocalReCoServer.Active then Result := True;
      stIMAP: if Assigned( LocalIMAPServer ) then //IMAP
                 if LocalIMAPServer.Active then Result := True;
   end;
end;

function LocalServerActivate( ServerType: TLocalServerTypes;
                              ServerActivate: TLocalServerActivates ): Boolean;
var  N: String;
begin
   Result := True;

   N := LocalServerTypeToStr( ServerType );

   if ( ServerActivate in [ saSTOP, saRESTART ] ) and
      ( LocalServerIsActive( ServerType ) ) then begin

      Result := False;
      
      try
         Log( LOGID_INFO, TrGlF(kLog, 'LocalXXXServer.ShutDown',
            'Shutdown local %s-server ...', N ) );
         case ServerType of
            stNNTP: begin LocalNntpServer.Close; FreeAndNil( LocalNntpServer ); end;
            stPOP3: begin LocalPop3Server.Close; FreeAndNil( LocalPop3Server ); end;
            stSMTP: begin LocalSmtpServer.Close; FreeAndNil( LocalSmtpServer ); end;
            stRECO: begin LocalReCoServer.Close; FreeAndNil( LocalReCoServer ); end;
            stIMAP: Begin LocalImapServer.Close; FreeAndNil( LocalImapServer ); end; //IMAP
         end;

         Result := True;
         Log( LOGID_SYSTEM, TrGlF(kLog, 'LocalXXXServer.Stopped',
            'Local %s-server stopped.', N ) );

      except
         on E:Exception do
            Log( LOGID_ERROR, TrGlf(kLog, 'LocalXXXServer.StopFailed',
              'Stop of local %s-server failed: %s', [N, E.Message] ) );
      end;

   end;

   if ( ServerActivate in [ saSTART, saRESTART ] ) and
      ( not LocalServerIsActive( ServerType ) ) then begin

      Result := False;
      
      if CriticalState > 0 then begin
         Log( LOGID_ERROR, TrGlF(kLog, 'LocalXXXServer.StartRefused',
            'Start of local %s-server refused: Maintenance is active.', N ) );
         exit;
      end;

      try
         Log( LOGID_INFO, TrGlF(kLog, 'LocalXXXServer.Starting',
            'Starting local %s-server ...', N ) );
         case ServerType of
            stNNTP: begin
                       LocalNntpServer := TSrvNNTP.Create( nil );
                       LocalNntpServer.Active := True;
                       if SSLReady and (Def_LocalNntpTlsMode>0) then
                          LocalNntpServer.InitSSLContext; {MG}{SSL}
                       If (Def_LocalNntpTlsMode = 2) and // Required!
                          not Assigned( LocalNntpServer.SSLCtx )
                       then begin
                          Log( LOGID_ERROR, TrGlF(kLog, 'LocalXXXServer.StartFailed.TLSRequired',
                             'Start of local %s-server failed: TLS required, but not available.', N ) );
                          LocalNntpServer.Close;
                          FreeAndNil( LocalNntpServer );
                          exit
                       end
                    end;
            stPOP3: begin
                       LocalPop3Server := TSrvPOP3.Create( nil );
                       LocalPop3Server.Active := True;
                       if SSLReady and (Def_LocalPop3TlsMode>0) then
                          LocalPop3Server.InitSSLContext; {MG}{SSL}
                       if (Def_LocalPop3TlsMode=2) and
                          not Assigned( LocalPop3Server.SSLCtx )
                       then begin
                          Log( LOGID_ERROR, TrGlF(kLog, 'LocalXXXServer.StartFailed.TLSRequired',
                             'Start of local %s-server failed: TLS required, but not available.', N ) );
                          LocalPop3Server.Close;
                          FreeAndNil( LocalPop3Server );
                          exit
                       end;
                    end;
            stSMTP: begin
                       LocalSmtpServer := TSrvSMTP.Create( nil );
                       LocalSmtpServer.Active := True;
                       if SSLReady and (Def_LocalSmtpTlsMode>0) then
                          LocalSmtpServer.InitSSLContext; {MG}{SSL}
                       if (Def_LocalSmtpTlsMode=2) and
                          not Assigned( LocalSmtpServer.SSLCtx )
                       then begin
                          Log( LOGID_ERROR, TrGlF(kLog, 'LocalXXXServer.StartFailed.TLSRequired',
                             'Start of local %s-server failed: TLS required, but not available.', N ) );
                          LocalSmtpServer.Close;
                          FreeAndNil( LocalSmtpServer );
                          exit
                       end
                    end;
            stRECO: begin
                       LocalReCoServer := TSrvRECO.Create( nil );
                       LocalReCoServer.Active := True;
                       {MG}{SSL}
                       if Def_LocalRecoUseTls then begin
                          if SSLReady then LocalReCoServer.InitSSLContext;
                          if LocalReCoServer.SSLCtx = nil then begin
                          Log( LOGID_ERROR, TrGlF(kLog, 'LocalXXXServer.StartFailed.TLSRequired',
                             'Start of local %s-server failed: TLS required, but not available.', N ) );
                             LocalRecoServer.Close;
                             FreeAndNil( LocalRecoServer );
                             exit
                          end
                       end;
                       {/SSL}
                    end;
            stIMAP: begin //IMAP
                       LocalImapServer := TSrvImap.Create( nil );
                       LocalImapServer.Active := True;
                       if SSLReady and (Def_LocalImapTlsMode>0)
                          then LocalImapServer.InitSSLContext {MG}{SSL}
                          else Log(LOGID_WARN,'IMAP server startup without SSL'+
                                 ' capability and doesn''t conform RFC 3501');
                       if (Def_LocalImapTlsMode=2) and
                          not Assigned( LocalImapServer.SSLCtx )
                       then begin
                          Log( LOGID_ERROR, TrGlF(kLog, 'LocalXXXServer.StartFailed.TLSRequired',
                             'Start of local %s-server failed: TLS required, but not available.', N ) );
                          LocalImapServer.Close;
                          FreeAndNil( LocalImapServer );
                          exit
                       end
                    end;
         end;

         Log( LOGID_SYSTEM, TrGlF(kLog, 'LocalXXXServer.Started',
            'Local %s-server started.', N ) );
         Result := True;

      except
         on E:Exception do
            Log( LOGID_ERROR, TrGlF(kLog, 'LocalXXXServer.StartFailed.XXX',
               'Start of local %s-server failed: %s', [N, E.Message] ) );
      end;

      IPAccessCheck.WantReload := True;

   end;

   PostMessage( hWndMainWindow, WM_NOTIFYGUI, GUI_MSG_SERVERSTATE_CHANGED, 0 );
end;

function LocalServerGetState( const Typ, Server: Integer ): Integer;
begin
   Result := 0;
   Case Typ of
     1: { Aktiv oder nicht aktiv? }
        Case Server of
          1: if LocalServerIsActive(stNNTP) then Result:=1;
          2: if LocalServerIsActive(stPOP3) then Result:=1;
          3: if LocalServerIsActive(stSMTP) then Result:=1;
          4: if LocalServerIsActive(stIMAP) then Result:=1; //IMAP
          5: if LocalServerIsActive(stReCO) then Result:=1; //IMAP
          else Result := -1
        end;
     2: Case Server of
          1: if LocalServerIsActive(stNNTP) then
                Result := LocalNntpServer.ActiveConnections;
          2: if LocalServerIsActive(stPOP3) then
                Result := LocalPop3Server.ActiveConnections;
          3: if LocalServerIsActive(stSMTP) then
                Result := LocalSmtpServer.ActiveConnections;
          4: if LocalServerIsActive(stIMAP) then //IMAP
                Result := LocalIMAPServer.ActiveConnections;
          5: if LocalServerIsActive(stReCo) then //IMAP
                Result := LocalReCoServer.ActiveConnections;
          else Result:=-1;
        end;
     3: Case Server of
          1: result:=CntArtNew;
          2: result:=CntArtLoad;
          3: result:=CntArtHist;
          4: result:=CntArtKill;
          5: result:=CntOutboxN;
          6: result:=CntMailNew;
          7: result:=CntOutboxM;
          8: result:=CntArtByMid; //HSR //GetByMid-Counter
          9: Result:=CntErrors;
          10:Result:=CntWarnings;
          else Result:=-1;
        end;
     {HSR} {GetLOGMask}
     4: Case Server of
          1: result:=Logfile.ViewMask;
          2: result:=Logfile.FileMask;
        end;
     {/HSR}
     else Result := -1;
   end
end;

// -------------------

procedure Init_InternalGroups;
var  Ini: TIniFile;

   procedure GroupFromIni( IniKey: String; Index: Integer );
   var  s: String;
   begin
      INTERNALGROUP[Index] := INTERNALGROUP[INTERNALGROUP_DEFAULT];
      s := Ini.ReadString( 'Setup', 'internalgroup.' + IniKey, '' );
      if s='' then exit; // use default
      if IsNewsgroup( s ) then INTERNALGROUP[ Index ] := s;
   end;

begin
   INTERNALGROUP[INTERNALGROUP_DEFAULT] := INTERNALGROUP_NOTSET_DEFAULT;

   Ini := TIniFile.Create( PATH_BASE + 'Hamster.ini' );
   GroupFromIni( 'default',      INTERNALGROUP_DEFAULT      );
   GroupFromIni( 'postok',       INTERNALGROUP_POSTOK       );
   GroupFromIni( 'posterrors',   INTERNALGROUP_POSTERRORS   );
   GroupFromIni( 'statistics',   INTERNALGROUP_STATISTICS   );
   GroupFromIni( 'cancelnotify', INTERNALGROUP_CANCELNOTIFY );
   GroupFromIni( 'newgroups',    INTERNALGROUP_NEWGROUPS    );
   GroupFromIni( 'unknowngroup', INTERNALGROUP_UNKNOWNGROUP );
   GroupFromIni( 'pullerrors',   INTERNALGROUP_PULLERRORS   );
   Ini.Free;
end;

Function ExpandDir(Ini: TInifile; Const Key, Default, Main: String; Const AutoCreate: boolean): String;
Var i: Integer;
begin
   Result := Ini.ReadString('Directories', Key, Default);
   If (Result = '') and (Default > '') then Result := Default;
   For i := 1 to Length(Result) do If Result[i]='/' then Result[i] := '\';
   if Copy(Result, Length(Result), 1)<>'\' then Result := Result + '\';
   If (Copy(Result, 2, 1) <> ':') and (Copy(Result, 2, 2) <> '\\') then begin
      If Copy(Result, 1, 1) = '\' then Result := Copy(Main, 1, 2) + Result
                                  else Result := Main + Result
   end;
   If Not DirectoryExists ( Result ) then begin
      If AutoCreate then begin
         If ForceDirectories ( Result ) then begin
            LogFile.AddMain ( LOGID_WARN, TrGlF (kLog, 'Directory.AutoCreatead',
              'Path "%s" didn''t exist, but is now created automaticly!', [Result, default] ) );
         end else begin
            LogFile.AddMain ( LOGID_ERROR, TrGlF (kLog, 'Directory.Not.Createable',
              'Path "%s" couldn''t be created, Default-Path "%s" is used instead!', [Result, default] ) );
            Result := Default
         end
      end else begin
         LogFile.AddMain ( LOGID_ERROR, TrGlF (kLog, 'Directory.Doesnt.exists',
           'Path "%s" doesn''t exists, Default-Path "%s" is used instead!', [Result, default] ) );
         Result := Default
      end;
      If Copy(Result, Length(Result), 1)<>'\' then Result := Result + '\'
   end
end;

Procedure Init_Mutex;
Var i: Integer; c: Char;
begin
  // Mutex initialisieren
  MutexString := LowerCase( ParamStr(0) );
  for i:=1 to length( MutexString ) do begin
     c := MutexString[i];
     if not( c in ['a'..'z', '0'..'9'] ) then MutexString[i]:='_';
  end;
  MutexString := 'mutexHamster.App.' + MutexString;
  // Mutex den Event-Namen zugefgt.
  {Event XP}
  If not IsWindowsXP then  begin
    {JW} {critical event}
    EnterCriticalSection(CS_Event);
    EVT_ISIDLE:=CreateEvent( nil,True,False,
                PChar(MutexString+'evtHamster.IsIdle') );
    EVT_STOPSCRIPT := CreateEvent( nil, True, False,
                      PChar(MutexString+'evtHamster.StopScript') );
    LeaveCriticalSection(CS_Event);
    {JW}
  end else begin
    // Events unter XP nur lokal anlegen, um Kollisionen mit anderen
    // Usern in jedem Fall zu vermeiden
    {JW} {critical event}
    EnterCriticalSection(CS_Event);
    EVT_ISIDLE:=CreateEvent( nil,True,False,
                PChar('Local\'+MutexString+'evtHamster.IsIdle') );
    EVT_STOPSCRIPT:=CreateEvent( nil, True, False,
                    PChar('Local\'+MutexString+'evtHamster.StopScript') );
    LeaveCriticalSection(CS_Event);
    {JW}
  end;
{JW}
   If (EVT_ISIDLE=0) or (EVT_STOPSCRIPT=0) then begin
      LogFile.AddMain ( LOGID_ERROR, TrGlF (kLog, 'Mutex.init.failed',
        'Initialization of mutex failed, Error %s', IntToStr(GetLastError)) )
   end else begin
      LogFile.AddMain ( LOGID_SYSTEM, TrGl (kLog, 'Mutex.init.done',
        'Mutex "IsIdle" and "StopScript" successfully initialized') )
   end;
end;

Procedure Init_Pathes;
Var Ini: TInifile;
begin
  // Pfade
  PATH_BASE     := ExtractFilePath( Application.ExeName );
  Ini := TIniFile.Create(PATH_BASE + 'hamster.ini');
  With Ini do try
     PATH_BASE := ExpandDir(Ini, 'Main', PATH_BASE, PATH_BASE, false);
     SetCurrentDirectory( PChar(PATH_BASE) );
  finally free end;
  Ini := TIniFile.Create(PATH_BASE + 'hamster.ini');
  With Ini do try
     PATH_HSC      := ExpandDir(Ini, 'Scripts', PATH_BASE, PATH_BASE, True);
     PATH_HSC_RC   := ExpandDir(Ini, 'ScriptsRC', PATH_HSC, PATH_BASE, True);
     PATH_HSM      := ExpandDir(Ini, 'Modules', PATH_HSC, PATH_BASE, True);
     PATH_LOGS     := ExpandDir(Ini, 'Logs', PATH_BASE + 'Logs', PATH_BASE, True);
     PATH_SERVER   := ExpandDir(Ini, 'Server', PATH_BASE + 'Server', PATH_BASE, True);
     PATH_GROUPS   := ExpandDir(Ini, 'Groups', PATH_BASE + 'Groups', PATH_BASE, True);
     PATH_MAILS    := ExpandDir(Ini, 'Mails', PATH_BASE + 'Mails', PATH_BASE, True);
     PATH_NEWS_OUT := ExpandDir(Ini, 'News.Out', PATH_GROUPS + 'News.Out', PATH_BASE, True);
     PATH_MAIL_OUT := ExpandDir(Ini, 'Mail.Out', PATH_MAILS + 'Mail.Out', PATH_BASE, True);
     PATH_NEWS_ERR := ExpandDir(Ini, 'News.Err', PATH_GROUPS + 'News.Err', PATH_BASE, True);
     PATH_TRASH    := ExpandDir(Ini, 'Trash', PATH_BASE + 'Trash', PATH_BASE, True);
  finally free end
end;

Procedure Init_CriticalSections;
begin
  InitializeCriticalSection( CS_COUNTER    );
  InitializeCriticalSection( CS_MAINTENANCE );
  InitializeCriticalSection( CS_SAVE_ARTICLE );
  InitializeCriticalSection( CS_TEST_ARTICLE );   //HRR
  InitializeCriticalSection( CS_LOCK_MAILBOX_ALL );
  InitializeCriticalSection( CS_LOCK_MAILOUT_USE );
  InitializeCriticalSection( CS_LOCK_MAILOUT_ADD );
  InitializeCriticalSection( CS_LOCK_NEWSOUT );
  InitializeCriticalSection( CS_Limiter );
  InitializeCriticalSection( CS_LOCK_InFeed );
  InitializeCriticalSection( CS_MID ); //JW //MID
  InitializeCriticalSection( CS_LOCK_InCancel ); //HSR //ModeCancel
  InitializeCriticalSection( CS_EVENT ); {JW} {critical event}
  InitializeCriticalSection( CS_SHUTDOWNLIST ); //HSR //One_SD
  InitializeCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
end;

Procedure Exit_CriticalSections;
begin
  DeleteCriticalSection( CS_COUNTER );
  DeleteCriticalSection( CS_MAINTENANCE );
  DeleteCriticalSection( CS_SAVE_ARTICLE );
  DeleteCriticalSection( CS_TEST_ARTICLE );             //HRR
  DeleteCriticalSection( CS_LOCK_MAILBOX_ALL );
  DeleteCriticalSection( CS_LOCK_MAILOUT_USE );
  DeleteCriticalSection( CS_LOCK_MAILOUT_ADD );
  DeleteCriticalSection( CS_LOCK_NEWSOUT );
  DeleteCriticalSection( CS_Limiter );
  DeleteCriticalSection( CS_LOCK_InFeed );
  DeleteCriticalSection( CS_MID ); //JW //MID
  DeleteCriticalSection( CS_LOCK_InCancel ); //HSR //ModeCancel
  DeleteCriticalSection( CS_EVENT ); {JW} {critical event}
  DeleteCriticalSection( CS_SHUTDOWNLIST ); //HSR //One_SD
  DeleteCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
end;

procedure HamFileAppendList( const Filename: String; const TS: TStringList );
/// ersetzt: uTools.AppendToFile( TS: TStringList; Filename: String );
var  T : TextFile;
begin
   LogFile.Enter; /// EnterCriticalSection( CS_LOG );
   try
      try
         AssignFile( T, Filename );
         Filemode := 2;
         if FileExists2( Filename ) then Append( T ) else Rewrite( T );
         write( T, TS.Text );
         CloseFile( T );
      except
         on E: Exception do
            Log( LOGID_ERROR, 'FileAppendList '+ Filename + ': ' + E.Message );
      end;
   finally LogFile.Leave end;  /// LeaveCriticalSection( CS_LOG );
end;

procedure HamFileAppendLine( const Filename, LineText: String );
/// ersetzt: uTools.AppendLineToFile( Filename, LineText: String );
var  T : TextFile;
begin
   LogFile.Enter;
   try
      try
         AssignFile( T, Filename );
         FileMode := 2;
         if FileExists2( Filename ) then Append(T) else Rewrite(T);
         if IOResult<>0 then exit;
         writeln( T, LineText );
         CloseFile( T );
      except
         on E: Exception do
            Log( LOGID_ERROR, 'FileAppendLine '+ Filename + ': ' + E.Message );
      end;
   finally LogFile.Leave end;
end;

procedure HamFileRewriteLine( const Filename, LineText: String );
/// ersetzt: uTools.WriteLineToFile( Filename, LineText: String );
var  T : TextFile;
begin
   LogFile.Enter;
   try
      try
         AssignFile( T, Filename );
         FileMode := 1;
         Rewrite( T );
         writeln( T, LineText );
         CloseFile( T );
      except
         on E: Exception do
            Log( LOGID_ERROR, 'FileWriteLine '+ Filename + ': ' + E.Message );
      end;
   finally LogFile.Leave end;
end;

Procedure TestArchiveMode;
Var r: TSearchRec; s1, s2: String; b: Boolean;
begin
   If FindFirst (PATH_BASE+'*.*', faAnyfile, R) = 0 then begin
      ArchivMode := true;
      Repeat
         If (R.Attr and faDirectory) = 0 then
            If (R.Attr and faReadOnly) = 0 then ArchivMode := false
      until (Not ArchivMode) or (FindNext(r)<>0);
      FindClose(R)
   end;
   ScriptAvail (s1, s2, b)
end;


initialization
  Init_CriticalSections;
  Init_Pathes;
  TestArchiveMode;
  Logfile.Cached := true;
  TestLanguage;
  Init_Mutex;
  Init_InternalGroups;
  OUR_VERINFO   := OUR_VERINFO + '/' + GetExeVersion;
  if EVT_ISIDLE=0 then begin
     {JW} {critical event}
     EnterCriticalSection(CS_Event);
     SetEvent( EVT_ISIDLE ); {JW} {XP_EVT}
     LeaveCriticalSection(CS_Event);
     {JW}
  end;
finalization
  Exit_CriticalSections;
  CloseHandle( EVT_ISIDLE );
  CloseHandle( EVT_STOPSCRIPT );
end.
