// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR      m
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit HConfigIPAccess;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, ExtCtrls, dGroupSelect, uTools, cStdForm, Menus;

Function DialogIPAccess: Boolean;

type
  TViewMode = (vmList, vmEdit, vmNew);
  TfrmConfigIPAccess = class(THForm)
    OKBtn: TButton;
    CancelBtn: TButton;
    butHelp: TButton;
    Panel1: TPanel;
    lb: TListBox;
    butNew: TButton;
    butEdit: TButton;
    butSave: TButton;
    butCancel: TButton;
    butDuplicate: TButton;
    butDelete: TButton;
    butUp: TButton;
    butDown: TButton;
    GBEntry: TGroupBox;
    labServer: TLabel;
    labAccess: TLabel;
    labTo: TLabel;
    cbServer: TComboBox;
    cbAccess: TComboBox;
    txtIP1: TEdit;
    optRange: TRadioButton;
    optLocalAs: TRadioButton;
    txtIP2: TEdit;
    optComment: TRadioButton;
    txtComment: TEdit;
    cbIP3: TComboBox;
    procedure FormCreate(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure butNewClick(Sender: TObject);
    procedure butEditClick(Sender: TObject);
    procedure butSaveClick(Sender: TObject);
    procedure butCancelClick(Sender: TObject);
    procedure lbClick(Sender: TObject);
    procedure optRangeClick(Sender: TObject);
    procedure butUpClick(Sender: TObject);
    procedure butDownClick(Sender: TObject);
    procedure butDeleteClick(Sender: TObject);
    procedure lbDblClick(Sender: TObject);
    procedure butDuplicateClick(Sender: TObject);
    procedure butAdvancedSSLSettingsClick(Sender: TObject);
  private
    fViewmode: TViewmode;
    ValidEntry, DefLocal: Boolean;
    IP1, IP2, Scope, Access: Longint;

    procedure SetViewmode(const m: TViewMode);
    Property Viewmode: TViewMode read fViewmode Write SetViewmode;

    procedure LoadSettings;
    procedure SaveSettings;
  end;

implementation

uses Global, fAccount, HConfigSSLAdvanced, Config, dInput, fGroup, cIPAccess;

{$R *.DFM}

Function DialogIPAccess: Boolean;
begin
   With TfrmConfigIPAccess.Create(NIL) do try
      Result := ShowModal = mrOK
   finally Free end
end;

procedure TfrmConfigIPAccess.LoadSettings;
begin
   If FileExists2(PATH_BASE + CFGFILE_IPACCESS)
      then lb.Items.LoadFromFile(PATH_BASE + CFGFILE_IPACCESS)
      else lb.Items.Text := DefaultIPAccess;
   If lb.Items.Count > 0 then begin
      lb.ItemIndex := 0;
      lb.OnClick (NIL)
   end
end;

procedure TfrmConfigIPAccess.SaveSettings;
begin
   lb.Items.SaveToFile(PATH_BASE + CFGFILE_IPACCESS);
end;

procedure TfrmConfigIPAccess.FormCreate(Sender: TObject);
begin
   Viewmode := vmList;
   cbIP3.Items.Text := '127.0.0.1' + #13#10 + GetLocalIPs + '0.0.0.0';
   With cbServer.Items do begin
      Clear;
      Add ( Tr ( 'Entry-NNTP', 'NNTP: local newsserver') );
      Add ( Tr ( 'Entry-POP3', 'POP3: local POP3-mailserver') );
      Add ( Tr ( 'Entry-IMAP', 'IMAP: local IMAP-mailserver') );
      Add ( Tr ( 'Entry-SMTP', 'SMTP: local SMTP-mailserver') );
      Add ( Tr ( 'Entry-MAIL', 'MAIL: POP3+IMAP+SMTP') );
      Add ( Tr ( 'Entry-RECO', 'RECO: local telnet-server') );
      Add ( Tr ( 'Entry-ALL', 'ALL: all local servers') );
   end;
   LoadSettings;
   butSave.Left := butNew.Left;
   butCancel.Left := butNew.Left;
end;

procedure TfrmConfigIPAccess.SetViewmode(const m: TViewMode);
begin
   lb.Enabled := m = vmList;
   butNew.Visible := m = vmList;
   butSave.Visible := m > vmList;
   butCancel.Visible := m = vmEdit;
   With butEdit do begin
      Visible := m = vmList;
      Enabled := Visible and (lb.ItemIndex >= 0)
   end;
   With butDuplicate do begin
      Visible := m = vmList;
      Enabled := Visible and (lb.ItemIndex >= 0)
   end;
   With butUp do begin
      Visible := m = vmList;
      Enabled := visible and (lb.ItemIndex > 0)
   end;
   With butDown do begin
      Visible := m = vmList;
      Enabled := Visible and (lb.ItemIndex < lb.Items.Count-1)
   end;
   DoEnable ( GBEntry, m <> vmList );
   OKBtn.Enabled := m = vmList;
   If (fViewMode = vmList) and (m > vmList) then begin
      With txtIP1 do If Visible then SetFocus;
      With cbIP3 do If Visible then SetFocus;
      With txtComment do If Visible then SetFocus;
   end else If (fViewMode > vmList) and (m = vmList) then begin
      lb.SetFocus
   end;
   fViewMode := m;
end;

Function ScopeIndex(Const Scope: LongInt): Integer;
begin
   Result := -1;
   Case Scope of
      IPACC_SCOPE_NNTP: Result := 0;
      IPACC_SCOPE_POP3: Result := 1;
      IPACC_SCOPE_IMAP: Result := 2;
      IPACC_SCOPE_SMTP: Result := 3;
      IPACC_SCOPE_MAIL: Result := 4;
      IPACC_SCOPE_RECO: Result := 5;
      IPACC_SCOPE_ALL : Result := 6;
   end
end;
Function IndexToScopename (Const Nr: Integer): String;
begin
   Case Nr of
      0: Result := 'NNTP';
      1: Result := 'POP3';
      2: Result := 'IMAP';
      3: Result := 'SMTP';
      4: Result := 'MAIL';
      5: Result := 'RECO';
      6: Result := 'ALL';
      else Result := ''
   end
end;

Function AccessIndex(Const Access: LongInt): Integer;
begin
   Result := -1;
   Case Access of
      IPACC_ACCESS_NA: Result := 0;
      IPACC_ACCESS_RO: Result := 1;
      IPACC_ACCESS_WO: Result := 2;
      IPACC_ACCESS_RW: Result := 3;
      IPACC_ACCESS_ALL: Result := 4;
   end;
end;
Function IndexToAccessname (Const Nr: Integer): String;
begin
   Case Nr of
      0: Result := 'NA';
      1: Result := 'RO';
      2: Result := 'WO';
      3: Result := 'RW';
      4: Result := 'ALL';
      else Result := ''
   end
end;

procedure TfrmConfigIPAccess.OKBtnClick(Sender: TObject);
begin
   SaveSettings;
   ModalResult := mrOK
end;

procedure TfrmConfigIPAccess.butNewClick(Sender: TObject);
Var p: Integer; s: String;
begin
   // Update Dialog State
   ViewMode := vmNew;
   // Insert dummy-entry into listbox
   s := Tr('NewEntry', '<New Entry>');
   If lb.ItemIndex >= 0
      then begin p := lb.ItemIndex; lb.Items.Insert (p, s) end
      else p := lb.Items.Add(s);
   lb.ItemIndex := p;
   // Clear Entry
   optRange.Checked := true;
   txtIP1.text := '';
   txtIP2.text := '';
   cbIP3.text := '';
   cbServer.ItemIndex := -1;
   cbAccess.ItemIndex := -1;
   txtComment.Text := ''
end;

procedure TfrmConfigIPAccess.butEditClick(Sender: TObject);
begin
   ViewMode := vmEdit
end;

procedure TfrmConfigIPAccess.butSaveClick(Sender: TObject);
Var Msg, Res: String;
begin
   Msg := '';
   Res := lb.Items[lb.ItemIndex];
   If optComment.Checked then begin
      Res := txtComment.Text;
      If (Res > '') and (Not (Res[1] IN['#', ';'])) and (ViewMode = vmNew)
         then Res := '# '+Trim(Res)
   end else
   If optLocalAs.Checked then begin
      With cbIP3 do If IsValidIP(Text)
         then Res := 'ALL, NA, LOCAL, '+text
         else Msg := TrF('SaveError.NoValidIP', '"%s" is no valid IP', Text)
   end else
   If optRange.Checked then begin
      With txtIP1 do If Not IsValidIP(Text) then
         Msg := TrF('SaveError.NoValidIP', '"%s" is no valid IP', Text)
      else With txtIP2 do If (Text > '') and (Not IsValidIP(Text)) then
         Msg := TrF('SaveError.NoValidIP', '"%s" is no valid IP', Text)
      else If cbServer.ItemIndex < 0 then
         Msg := Tr('SaveError.NoServer', 'You have to choose a server')
      else If cbAccess.ItemIndex < 0 then
         Msg := Tr('SaveError.NoAccess', 'You have to choose an access-level');
      If Msg = '' then begin
         Res := IndexToScopename(cbServer.ItemIndex) + ', '
                + IndexToAccessname(cbAccess.ItemIndex) + ', '
                + txtIP1.Text;
         With txtIP2 do If Text > '' then Res := Res + ', '+Text
      end
   end ELSE begin
      Msg := Tr('SaveError.NoTypeChosen', 'No type for entry');
   end;
   If Msg > '' then begin
      Application.MessageBox(
         PChar(Tr('SaveError.MainMsg', 'Entry can not be saved: ') + Msg),
         PChar(caption),
         MB_ICONINFORMATION + MB_OK )
   end else begin
      lb.Items[lb.ItemIndex] := Res;
      lb.OnClick(NIL);
      ViewMode := vmList
   end
end;

procedure TfrmConfigIPAccess.butCancelClick(Sender: TObject);
begin
   ViewMode := vmList;
   lb.OnClick(NIL)
end;

procedure TfrmConfigIPAccess.lbClick(Sender: TObject);
Var empty: Boolean;
begin
   If Viewmode = vmList then begin
      If lb.ItemIndex >= 0
         then ValidEntry := ConvertLineToEntry ( lb.Items[lb.ItemIndex],
                              IP1, IP2, Scope, Access, Empty, DefLocal)
         else exit;
      txtComment.text := lb.Items[lb.ItemIndex];
      If ValidEntry then ValidEntry := Not Empty;
      If ValidEntry then begin
         If DefLocal
            then optLocalAs.Checked := true
            else optRange.Checked := true;
         txtIP1.Text := IP2Text(IP1);
         If IP2 <> IP1 then txtIP2.Text := IP2Text(IP2) else txtIP2.Text := '';
         cbIP3.Text := IP2Text(IP2);
         cbServer.ItemIndex := ScopeIndex(Scope);
         cbAccess.ItemIndex := AccessIndex(Access);
      end else begin
         optComment.Checked := true
      end;
      Viewmode := vmList; // Refresh state of buttons
   end;
end;

procedure TfrmConfigIPAccess.optRangeClick(Sender: TObject);
begin
   txtIP1.Visible := optRange.Checked;
   labTo.Visible := optRange.Checked;
   txtIP2.Visible := optRange.Checked;
   labServer.Visible := optRange.Checked;
   cbServer.Visible := optRange.Checked;
   labAccess.Visible := optRange.Checked;
   cbAccess.Visible := optRange.Checked;
   cbIP3.Visible := optLocalAs.Checked;
   txtComment.Visible := optComment.Checked;
end;

procedure TfrmConfigIPAccess.butUpClick(Sender: TObject);
Var s: String; i: Integer;
begin
   With lb do begin
      i := ItemIndex;
      If i < 1 then Exit;
      s := Items[i-1]; Items[i-1] := Items[i]; Items[i] := s;
      ItemIndex := i-1;
      lb.OnClick(NIL)
   end
end;

procedure TfrmConfigIPAccess.butDownClick(Sender: TObject);
Var s: String; i: Integer;
begin
   With lb do begin
      i := ItemIndex;
      If i > Items.Count-2 then Exit;
      s := Items[i+1]; Items[i+1] := Items[i]; Items[i] := s;
      ItemIndex := i+1;
      lb.OnClick(NIL)
   end
end;

procedure TfrmConfigIPAccess.butDeleteClick(Sender: TObject);
Var p: Integer;
begin
   p := lb.ItemIndex;
   If p < 0 then Exit;
   If (Viewmode = vmNew) or (Application.MessageBox(
      PChar(Tr('DeleteEntry.Ask', 'Do you really want to delete the selected entry?')),
      PChar(caption),
      MB_ICONQUESTION + MB_YESNO ) = IDYES)
   then begin
      lb.Items.Delete (p);
      If p = lb.Items.Count then Dec(p);
      lb.ItemIndex := p;
      lb.OnClick(NIL);
      ViewMode := vmList
   end
end;

procedure TfrmConfigIPAccess.lbDblClick(Sender: TObject);
begin
   With butEdit do If Visible and Enabled then OnClick(NIL)
end;

procedure TfrmConfigIPAccess.butDuplicateClick(Sender: TObject);
begin
   With lb do With Items do If ItemIndex >= 0 then begin
      Insert (ItemIndex, Items[ItemIndex])
   end
end;

procedure TfrmConfigIPAccess.butAdvancedSSLSettingsClick(
  Sender: TObject);
begin
   Dialog_AdvancedSSLSettings
end;

end.
