// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit XApp; // "Hamster.App"-object

// ----------------------------------------------------------------------------
// Contains Hamster's automation object "App"
// ----------------------------------------------------------------------------

interface

uses
  ComObj, ActiveX, Hamster_TLB, StdVcl, cMailRouter, cNewsJobs,
  IniFiles; //HSR //OLE-Newsgroup-Informations

const
  HAM_WAITIDLE_INFINITE = 0;

type
  TApp = class(TAutoObject, IApp)
  public //protected
    function ControlGetInfo: WideString; safecall;
    function ControlGetPath: WideString; safecall;
    function ControlGetVersion: WideString; safecall;
    function ControlIsIdle: WordBool; safecall;
    function ControlThreadCount: Integer; safecall;
    function ControlRunPurge(PurgeOptions: Integer): Integer; safecall;
    function ControlWaitIdle(WaitTimeout: Integer): WordBool; safecall;
    function NewsGrpCount: Integer; safecall;
    function NewsGrpName(GrpIdx: Integer): WideString; safecall;
    function NewsGrpIndex(const GrpName: WideString): Integer; safecall;
    function NewsGrpOpen(const GrpName: WideString): Integer; safecall;
    procedure NewsGrpClose(GrpHdl: Integer); safecall;
    function NewsArtCount(GrpHdl: Integer): Integer; safecall;
    function NewsArtNoMax(GrpHdl: Integer): Integer; safecall;
    function NewsArtNoMin(GrpHdl: Integer): Integer; safecall;
    function NewsArtText(GrpHdl, ArtNo: Integer): WideString; safecall;
    function NewsArtTextExport(GrpHdl, ArtNo: Integer): WideString; safecall;
    function NewsScoreListFor(const GrpName: WideString): WideString; safecall;
    function RasDial(const ConnectionID, Username,
      Password: WideString): WordBool; safecall;
    function RasIsConnected: WordBool; safecall;
    function RasLastError: Integer; safecall;
    procedure RasHangup; safecall;
    procedure ControlSetLogin(const Identifier, Username,
      Password: WideString); safecall;
    function ControlRunMail(const ServerList: WideString): Integer; safecall;
    function ControlRunNewsPost(const ServerList: WideString): Integer;
      safecall;
    function ControlRunNewsPull(const ServerList: WideString): Integer;
      safecall;
    function ControlRunFetchMail(const Server: WideString; const Port: WideString;
                                  const User: WideString; const Pass: WideString;
                                  const DestUser: WideString): Integer; safecall;
    function ControlRunSendMail(const Server: WideString; const Port: WideString;
                                 const FromSelection: WideString): Integer; safecall;
    function ControlRunFetchMailTLS(const Server, Port, User, Pass,
      Destuser: WideString; SSLMode, SSLVerify: SYSINT;
      const SSLCaFile: WideString): Integer; safecall;
    function ControlRunSendMailTLS(const Server, Port,
      FromSelection: WideString; SSLMode, SSLVerify: SYSINT;
      const SSLCaFile: WideString): Integer; safecall;
    function ControlRunSendMailAuthTLS(const Server, Port, User, Pass,
      FromSelection: WideString; SSLMode, SSLVerify: SYSINT;
      const SSLCaFile: WideString): Integer; safecall;
    {/SSL}
  protected
    function NewsLocateMID(const MessageID: WideString;
      var Groupname: WideString; var ArtNo: Integer): WordBool; safecall;
    function ControlRunRebuildHistory: Integer; safecall;
    function NewsScoreTest(const GrpName, ArtText: WideString;
      var MatchLog: WideString): Integer; safecall;
    function NewsDeleteByMID(const MessageID: WideString): WordBool; safecall;
    function NewsImport(const ArtText, OverrideGroups: WideString;
      IgnoreHistory, MarkNoArchive: WordBool): WordBool; safecall;
    function ControlMessage(Msg, Param: Integer): Integer; safecall;
    function NewsLocateMID2(const MessageID: WideString; var Groupname,
      ArtNo: OleVariant): WordBool; safecall;
    function ControlRunRebuildGlobalLists(Options: Integer): Integer; safecall;
    function NewsJobsClear: Integer; safecall;
    function NewsJobsPullDef(const ServerList: WideString): Integer; safecall;
    function NewsJobsPostDef(const ServerList: WideString): Integer; safecall;
    function NewsJobsPull(const Servername, reGrpSelect: WideString): Integer;
      safecall;
    function NewsJobsPost(const Servername, reGrpSelect,
      reMsgSelect: WideString): Integer; safecall;
    function NewsJobsStart(const ServerList: WideString): Integer; safecall;
    function ControlRunSendMailAuth(const Server: WideString; const Port: WideString;
                                     const User: WideString; const Pass: WideString;
                                     const FromSelection: WideString): Integer; safecall;
    function ControlFlush: Integer; safecall;
    function ControlGetHscPath: WideString; safecall;
    function ControlGetLogsPath: WideString; safecall;
    function ControlGetServerPath: WideString; safecall;
    function ControlGetGroupsPath: WideString; safecall;
    function ControlGetMailsPath: WideString; safecall;
    function ControlGetNewsOutPath: WideString; safecall;
    function ControlGetMailsOutPath: WideString; safecall;
    function ControlChangePassword(const AccountName, OldPW,
      NewPW: WideString): SYSINT; safecall;
    function ControlGetHsmPath: WideString; safecall;
    function ControlGetHsmPath2: WideString; safecall;
    procedure ControlAddLog(const Info: WideString; Typ: SYSINT); safecall;
    function ControlRunScript(const Script, Parameters: WideString;
      Wait: WordBool): Integer; safecall;
    function ControlGetStatus(Typ, Server: Integer): Integer; safecall;
    function ControlRunPurgeGroup(const GrpName: WideString): Integer;
      safecall;
    {JW}
    function ControlNewsJobsGetServer(Index: Integer): WideString; safecall;
    function ControlNewsJobsGetParam(Index: Integer): WideString; safecall;
    function ControlNewsJobsGetType(Index: Integer): Integer; safecall;
    function ControlNewsJobsGetPriority(Index: Integer): Integer; safecall;
    function ControlNewsJobsSetPriority(Index: Integer; Priority: Integer): Integer; safecall;
    function ControlNewsJobsDelete(Index: Integer): Integer; safecall;
    function ControlNewsJobsGetCounter(): Integer; safecall;
    {JW}
    function ControlGetProcessIdentifier: WideString; safecall;
    function ControlNewsJobsAdd(const Server: WideString; Typ: Integer;
                                const Parameter: WideString; Priority: Integer): Integer; safecall;
    function ControlNewMail(const MailFrom, MailTo,MailText: WideString): Integer; safecall;
{JW} {Feed}
    function  NewsJobsFeed(const Servername: WideString; const reGrpSelect: WideString): Integer; safecall;
{JW}
{HSR} {OLE-Header}
    function NewsArtXOver(GrpHdl, ArtNo: Integer) : widestring; safecall;
    function NewsGrpXOver(GrpHdl, start, finish: Integer) : widestring; safecall;
{/HSR}
    function NewsGrpInformation(GrpHdl: Integer) : widestring; safecall;
    function ControlGetNewsErrPath: WideString; safecall;
    function NewsGrpNameByHandle(GrpHdl: Integer): WideString; safecall;
    function ClearXCounter(AbNr, BisNr: Integer): Integer; safecall;
    function DecXCounter(Nr, Value: Integer): Integer; safecall;
    function IncXCounter(Nr, Value: Integer): Integer; safecall;
    function SetXCounter(Nr, Value: Integer): Integer; safecall;
    function XCounter(Nr: Integer): Integer; safecall;
    function RasGetConnection: WideString; safecall;
    function RasGetIP: WideString; safecall;
    function RasListEntries: WideString; safecall;
    function ControlAddGroup(const Group: WideString): Integer; safecall;
    function ControlDelGroup(const Group: WideString): Integer; safecall;
    function ControlAddPull(const Server, Group: WideString): Integer;
      safecall;
    function ControlDelPull(const Server, Group: WideString): Integer;
      safecall;
    function ControlRotateLog: Integer; safecall;
    function hs2AsyncCommand(const Command: WideString): Integer; safecall; //HSR //OLE-Newsgroup-Informations
  end;

procedure AddLog(const Info: WideString; Typ: SYSINT);

implementation

uses classes, Windows, SysUtils, ComServ, Forms, uTools, Global, Config, cArtFiles,
     cArticle, cFiltersNews, tTransfer, tMaintenance, cPasswordFile,
     cAccount, Main, TScript, cLogFile, cStdForm, uRasDyn;

function TApp.ControlGetInfo: WideString;
begin
     Result := 'Hamster Vr. ' + GetExeVersion + ', ' + Application.ExeName;
end;

function TApp.ControlGetPath: WideString;
begin
     Result := PATH_BASE
     {Result := ExtractFilePath( Application.ExeName );
     if copy(Result,length(Result),1)<>'\' then Result:=Result+'\';}
end;

function TApp.ControlGetHscPath: WideString;
begin
     Result := PATH_HSC
end;

function TApp.ControlGetHsmPath: WideString;
begin
     Result := PATH_HSM
end;

function TApp.ControlGetLogsPath: WideString;
begin
     Result := PATH_LOGS
end;

function TApp.ControlGetServerPath: WideString;
begin
     Result := PATH_SERVER
end;

function TApp.ControlGetGroupsPath: WideString;
begin
     Result := PATH_GROUPS
end;

function TApp.ControlGetMailsPath: WideString;
begin
     Result := PATH_MAILS
end;

function TApp.ControlGetNewsOutPath: WideString;
begin
     Result := PATH_NEWS_OUT
end;

function TApp.ControlGetMailsOutPath: WideString;
begin
     Result := PATH_MAIL_OUT
end;


function TApp.ControlGetVersion: WideString;
begin
     Result := GetExeVersion;
end;

function TApp.NewsGrpCount: Integer;
begin
     Result := CfgHamster.ActiveCount;
end;

function TApp.NewsGrpName(GrpIdx: Integer): WideString;
begin
     Result := CfgHamster.ActiveName[ GrpIdx ];
end;

function TApp.NewsGrpIndex(const GrpName: WideString): Integer;
begin
     Result := CfgHamster.ActiveIndexOf[ GrpName ];
end;

function TApp.NewsGrpOpen(const GrpName: WideString): Integer;
begin
     if CfgHamster.ActiveIndexOf[GrpName]>=0 then begin
        Result := ArticleBase.Open( GrpName );
     end else begin
        Result := -1;
     end;
end;

procedure TApp.NewsGrpClose(GrpHdl: Integer);
begin
     if GrpHdl>=0 then ArticleBase.Close( GrpHdl );
end;

function TApp.NewsArtCount(GrpHdl: Integer): Integer;
begin
     Result := ArticleBase.Count[ GrpHdl ];
end;

function TApp.NewsArtNoMin(GrpHdl: Integer): Integer;
begin
     Result := ArticleBase.LocalMin[ GrpHdl ];
end;

function TApp.NewsArtNoMax(GrpHdl: Integer): Integer;
begin
     Result := ArticleBase.LocalMax[ GrpHdl ];
end;

function TApp.NewsArtText(GrpHdl, ArtNo: Integer): WideString;
begin
     Result := ArticleBase.ReadArticle( GrpHdl, ArtNo );
end;

function TApp.NewsArtTextExport(GrpHdl, ArtNo: Integer): WideString;
Var Buf: String;
begin
   Buf := ArticleBase.ReadArticle( GrpHdl, ArtNo );
   If Buf > '' then begin
      With TArticle.Create do try
         Text := Buf;
         Buf := AsExportFormat
      finally
         Free
      end
   end;
   Result := Buf
end;

function TApp.NewsScoreListFor(const GrpName: WideString): WideString;
var  ScoreFile: TFiltersNews;
begin
     ScoreFile := TFiltersNews.Create( PATH_BASE + CFGFILE_SCORES );
     ScoreFile.SelectSections( GrpName );
     Result := ScoreFile.SelectedLines;
     ScoreFile.Free;
end;

function TApp.NewsScoreTest(const GrpName, ArtText: WideString;
  var MatchLog: WideString): Integer;
var  XOverRec : TXOverRec;
     s        : String;
     Art      : TArticle;
begin
   Art := TArticle.Create;
   With Art do try
      Text := ArtText;
      ArticleToXOverRec( Art, XOverRec )
   finally
      Free
   end;
   With TFiltersNews.Create( PATH_BASE + CFGFILE_SCORES ) do try
      SelectSections( GrpName );
      Result := ScoreBeforeLoad( XOverRec, @s ); // PG
   finally
      free
   end;
   MatchLog := s;
end;

function TApp.ControlIsIdle: WordBool;
begin
  Result := (GetCounter(CntActiveTasks)<= 0 );
end;

function TApp.ControlThreadCount: Integer;
begin
   Result := GetCounter(CntActiveTasks);
end;

function TApp.ControlWaitIdle(WaitTimeout: Integer): WordBool;
const WaitSteps = 100;
var  Waited: Integer;
begin
     // Hinweis: Da Object im Thread des Hauptprogramms luft, erfolgt
     // das Warten hier in kleinen Sleep-Hppchen, damit das Hauptprg.
     // nicht vllig lahmgelegt wird (B.A.D.).
     Waited := 0;
     repeat
        Result := ControlIsIdle;
        if Result=True then break;

        Sleep( WaitSteps );
        inc( Waited, WaitSteps );
        Application.ProcessMessages;

        if WaitTimeout<>HAM_WAITIDLE_INFINITE then begin
           if Waited>=WaitTimeout then break;
        end;
     until AllShutDownReq
end;

function TApp.ControlRunPurge(PurgeOptions: Integer): Integer;
begin
   if ControlIsIdle and (Not ArchivMode) then begin
      TThreadPurge.Create( PurgeOptions , '', NIL).resume;
      Result := 0
   end else begin
      Result := -1
   end;
end;

function TApp.RasDial(const ConnectionID, Username,
  Password: WideString): WordBool;
var  s, u, p: String;
begin
     Result := False;

     u := Username;
     p := Password;

     if (copy(u,1,1)='$') and (p='') then begin
        s := u;
        if not PasswordFile.UsePassword( s, u, p ) then begin
           Log( LOGID_WARN, 'Missing username/password for "' + s + '"!' );
           exit;
        end;
     end;

     try
        if RasDialer.Dial( ConnectionID, u, p ) = 0 then Result := True;
     except
        on E:Exception do Log( LOGID_DEBUG, 'OLE-RasDial failed: ' + E.Message );
     end;
end;

procedure TApp.RasHangup;
begin
     try
        RasDialer.HangUp;
     except
        on E:Exception do Log( LOGID_DEBUG, 'OLE-RasHangup failed: ' + E.Message );
     end;
end;

function TApp.RasIsConnected: WordBool;
begin
   Result := RasDynIsConnected
end;

function TApp.RasLastError: Integer;
begin
   Result := RasDialer.LastError;
end;

procedure TApp.ControlSetLogin(const Identifier, Username,
  Password: WideString);
var  s, u, p: String;
begin
     u := Username;
     p := Password;

     if (copy(u,1,1)='$') and (p='') then begin
        s := u;
        if not PasswordFile.UsePassword( s, u, p ) then begin
           Log( LOGID_WARN, 'Missing username/password for "' + s + '"!' );
           exit;
        end;
     end;

     PasswordFile.SavePassword( False, Identifier, u, p );
end;

function TApp.ControlRunMail(const ServerList: WideString): Integer;
var  SrvList: String;
begin
     Result := -1;
     If ArchivMode then Exit;

     AllShutdownReq := False;

     SrvList := ServerList;
     if SrvList<>'' then SrvList := ';' + LowerCase(SrvList) + ';';

     if (CfgHamster.Pop3ServerCount>0) or (CfgHamster.SmtpServerCount>0) then begin
        TThreadPop3AndSmtp.Create(SrvList, 0).resume;
        Result := 0;
     end;
end;

function TApp.ControlRunNewsPost(const ServerList: WideString): Integer;
var  SrvList: String;
begin
     Result := -1;
     If ArchivMode then Exit;

     AllShutdownReq := False;

     SrvList := ServerList;
     if SrvList>'' then SrvList := ';' + LowerCase(SrvList) + ';';

     if CfgHamster.ServerCount>0 then begin
        if FileExists2( PATH_NEWS_OUT + '*.' +
           CfgIni.ReadString( 'Setup', 'news.ext.out', 'msg' ) )
        then begin
           TThreadNewsPostSimulate.Create( SrvList ).resume;
           Sleep( 1000 ); // give some time to startup and "lock" the server before pull
           Result := 0
        end;
     end;
end;

function TApp.ControlRunNewsPull(const ServerList: WideString): Integer;
var  SrvList, s : String;
     LfdServer  : Integer;
     LfdThread  : Integer;
     OK         : Boolean;
begin
     Result := -1;
     If ArchivMode then Exit;

     AllShutdownReq := False;

     SrvList := ServerList;
     if SrvList>'' then SrvList := ';' + LowerCase(SrvList) + ';';

     if (CfgHamster.ServerCount>0) then begin
        for LfdServer:=0 to CfgHamster.ServerCount-1 do begin
           if SrvList='' then begin
              OK := True;
           end else begin
              s := CfgHamster.ServerName[LfdServer] + ',' + CfgHamster.ServerPort[LfdServer];
              if Pos( ';' + LowerCase(s) + ';', SrvList ) > 0 then OK:=True
                                                              else OK:=False;
           end;
           if OK then begin
              // refresh Pull-List for server
              NewsJobs.AddPullSrv( CfgHamster.ServerName[LfdServer], '' );
              for LfdThread:=1 to CfgHamster.ServerPullThreads(LfdServer,True) do begin
                 if AllShutDownReq then break;
                 TThreadNewsJobs.Create( LfdServer).resume;
                 Result := 0
              end;
           end;
        end;
     end;
end;

function TApp.NewsLocateMID(const MessageID: WideString;
  var Groupname: WideString; var ArtNo: Integer): WordBool;
var  s: String;
     i: Integer;
begin
     if NewsHistory.LocateMID( MessageID, s, i ) then begin
        Groupname := s;
        ArtNo     := i;
        Result    := True;
     end else begin
        Result    := False;
     end;
end;

function TApp.NewsDeleteByMID(const MessageID: WideString): WordBool;
begin
   if ArchivMode then begin
      Result := false
   end else begin
      Result := ArticleBase.DeleteArticleByMID( MessageID, true )
   end
end;

function TApp.ControlRunRebuildHistory: Integer;
begin
    if ControlIsIdle and (Not ArchivMode) then begin
       TThreadHistoryRebuild.Create.resume;
       Result := 0;
    end else begin
       Result := -1;
    end;
end;

function TApp.NewsImport(const ArtText, OverrideGroups: WideString;
  IgnoreHistory, MarkNoArchive: WordBool): WordBool;
begin
   If ArchivMode
      then Result := false
      else Result := ImportArticle( ArtText, OverrideGroups,
                              IgnoreHistory, MarkNoArchive );
end;

function TApp.ControlMessage(Msg, Param: Integer): Integer;
begin
   try
      If Msg=HAM_MSG_HAMSTER_EXIT then begin
         if Param<100 then Param:=100;
         PostMessage( hWndMainWindow, WM_HAMSTER, Msg, Param );
         Result:=0
      end else begin
         Result := SendMessage( hWndMainWindow, WM_HAMSTER, Msg, Param )
      end
   except
      Result := 3;
   end;
end;

function TApp.NewsLocateMID2(const MessageID: WideString; var Groupname,
  ArtNo: OleVariant): WordBool;
var  s: String;
     i: Integer;
begin
     if NewsHistory.LocateMID( MessageID, s, i ) then begin
        Groupname := s;
        ArtNo     := i;
        Result    := True;
     end else begin
        Result    := False;
     end;
end;

function TApp.ControlRunRebuildGlobalLists(Options: Integer): Integer;
begin
     if ControlIsIdle and (Not ArchivMode) then begin
        TThreadRebuildGlobalLists.Create.resume;
        Result := 0
     end else begin
        Result := -1
     end;
end;

function TApp.NewsJobsClear: Integer;
begin
   If ArchivMode
      then Result := 0
      else Result := NewsJobs.Clear;
end;

function TApp.NewsJobsPullDef(const ServerList: WideString): Integer;
begin
   If ArchivMode
      then Result := 0
      else Result := NewsJobs.AddPullDef( ServerList ); 
end;

function TApp.NewsJobsPostDef(const ServerList: WideString): Integer;
begin
   If ArchivMode
      then Result := 0
      else Result := NewsJobs.AddPostDef( ServerList );
end;

function TApp.NewsJobsPull(const Servername,
  reGrpSelect: WideString): Integer;
begin
   If ArchivMode
      then Result := 0
      else Result := NewsJobs.AddPullSrv( Servername, reGrpSelect );
end;

function TApp.NewsJobsPost(const Servername, reGrpSelect,
  reMsgSelect: WideString): Integer;
begin
   If ArchivMode
      then Result := 0
      else Result := NewsJobs.AddPostSrv( Servername, reGrpSelect, reMsgSelect );
end;

function TApp.NewsJobsStart(const ServerList: WideString): Integer;
begin
   If ArchivMode
      then Result := 0
      else Result := NewsJobs.StartThreads( ServerList );
end;

function TApp.ControlFlush: Integer;
begin
   Result := 0;
   If Not ArchivMode then NewsHistory.SaveToFile;
end;

procedure TApp.ControlAddLog(const Info: WideString; Typ: SYSINT);
begin
   AddLog (Info, Typ)
end;

procedure AddLog(const Info: WideString; Typ: SYSINT);
Var x: Word;
begin
   Case Typ of
      1: x := LOGID_DEBUG;
      2: x := LOGID_DETAIL;
      3: x := LOGID_INFO;
      4: x := LOGID_SYSTEM;
      5: x := LOGID_WARN;
      6: x := LOGID_ERROR;
      7: x := LOGID_STATUS; {kms}
      else x := LOGID_INFO
   end;
   Log( x, Info )
end;

function TApp.ControlChangePassword(const AccountName, OldPW,
  NewPW: WideString): SYSINT;
var UID: Integer;
begin
   Result := 2;
   If ArchivMode then Exit;
   UID := CfgAccounts.UserIDOf(AccountName);
   Result := 0;
   If CfgAccounts.Value[UID, ACTP_PASSWORD] = OldPw then begin
      CfgAccounts.Value[UID, ACTP_PASSWORD] := NewPw;
      Result := 1
   end else begin
      Log ( LOGID_WARN, TrGlF (kLog, 'OLE.ControlChangePassword.Fails',
             'OLE-Change of password for account "%s" failed', AccountName))
   end
end;

function TApp.ControlGetHsmPath2: WideString;
begin

end;

function TApp.ControlRunScript(const Script, Parameters: WideString;
  Wait: WordBool): Integer;
begin
   StartNewScript (Script, Parameters, Wait);
   Result := 0;
end;

function TApp.ControlGetStatus(Typ, Server: Integer): Integer;
begin
  Result := LocalServerGetState (Typ, Server)
end;

function TApp.ControlRunPurgeGroup(const GrpName: WideString): Integer;
begin
   if ControlIsIdle and (Not ArchivMode) then begin
      TThreadPurge.Create( HAM_PURGEOPT_DONEWS , GrpName, NIL).resume;
      Result := 0;
   end else begin
      Result := -1;
   end;
end;

{JW}
function TApp.ControlNewsJobsGetServer(Index: Integer): WideString;
begin
   Result := NewsJobs.GetServer( Index );
end;
function TApp.ControlNewsJobsGetParam(Index: Integer): WideString;
begin
   Result := NewsJobs.GetPar( Index );
end;
function TApp.ControlNewsJobsGetType(Index: Integer): Integer;
begin
   Result := NewsJobs.GetType( Index );
end;
function TApp.ControlNewsJobsGetPriority(Index: Integer): Integer;
begin
   Result := NewsJobs.GetPriority( Index );
end;
function TApp.ControlNewsJobsGetCounter(): Integer;
begin
   Result := NewsJobs.GetCounter();
end;
function TApp.ControlNewsJobsSetPriority(Index: Integer; Priority: Integer): Integer;
begin
   Result := NewsJobs.SetPriority( Index,Priority );
end;
function TApp.ControlNewsJobsDelete(Index: Integer): Integer;
begin
   Result := NewsJobs.Delete( Index );
end;

function TApp.ControlNewsJobsAdd(const Server: WideString; Typ: Integer;
                            const Parameter: WideString; Priority: Integer): Integer;
begin
   Result := 0;
   If Not ArchivMode then try
      Result := NewsJobs.Add(Server,TJobType(Typ),Parameter,Priority)
   except
   end
end;

{JW}

function  TApp.ControlGetProcessIdentifier: WideString;
begin
   Result :=MutexString;
end;

function TApp.ControlNewMail(const MailFrom, MailTo,
  MailText: WideString): Integer;
begin
   Result := -1;
   If ArchivMode then Exit;
   if NewMail(MailFrom,MailTo,MailText) then result:=0
end;

{JW} {SSL}
function TApp.ControlRunFetchMail(const Server: WideString; const Port: WideString;
                                  const User: WideString; const Pass: WideString;
                                  const DestUser: WideString): Integer;
var  DefUser, FiltSect: String;
     i: Integer;
begin
     Result := -1;
     If ArchivMode then Exit;
     AllShutdownReq := False;
     DefUser  := DestUser;
     FiltSect := '';
     i := Pos( ',', DefUser );
     if i>0 then begin
        FiltSect := copy( DefUser, i+1, Length(DefUser)-i );
        DefUser  := copy( DefUser, 1, i-1 );
     end;
     TThreadPop3Fetch.Create( Server, Port, User, Pass, DefUser,
        FiltSect, '?', 0, 0, '' ).resume; {MG}{SSL}
     Result := 0;
end;

function TApp.ControlRunSendMail(const Server: WideString; const Port: WideString;
                                 const FromSelection: WideString):
Integer;
begin
   Result := -1;
   If ArchivMode then Exit;
   AllShutdownReq := False;
   TThreadSmtpSend.Create( Server, Port, '', '', FromSelection, '',
      False, 0, 0, '' ).resume; {MG}{SSL}
   Result := 0
end;

function TApp.ControlRunSendMailAuth(const Server: WideString; const Port: WideString;
                                     const User: WideString; const Pass: WideString;
                                     const FromSelection: WideString):
Integer;
begin
   Result := -1;
   If ArchivMode then Exit;
   AllShutdownReq := False;
   TThreadSmtpSend.Create( Server, Port, User, Pass, FromSelection, '',
      True, 0, 0, '' ).resume; {MG}{SSL}
   Result := 0
end;

{JW} {Feed}

function TApp.NewsJobsFeed(const Servername,
  reGrpSelect: WideString): Integer;
begin
   If ArchivMode
      then Result := 0
      else Result := NewsJobs.AddFeedSrv( Servername, reGrpSelect );
end;
{JW}

{HSR} {OLE-Header}

function FormatXOver( s: String ) : String;
Var i: Integer;
begin
   Result := s;
   For i := 1 to Length(Result) do
      If Result[i] IN[#0, #9, #13, #10] then Result[i] := ' '
end;

function TApp.NewsArtXOver(GrpHdl, ArtNo: integer) : widestring;
var  sLines: String;
     ArtSize  : Integer;
//function TSrvNNTPCli.Cmd_XOVER( Par: String ): Boolean;
begin
   Result:='';
   With TArticle.Create do try
      ArtSize  := 4096;
      Text := ArticleBase.ReadArticleSized( GrpHdl, ArtNo, ArtSize );
      sLines   := Header['Lines:'];
      if (sLines='') or (FullBody = '') then begin
         Text := ArticleBase.ReadArticle( GrpHdl, ArtNo );
         sLines := inttostr( CountLines(FullBody) );
      end;

      If Text > '' then begin
         // 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References 6:Bytes 7:Lines [8:Xref]
         Result := inttostr(ArtNo) + #9
            + FormatXOver(Header['Subject:']) + #9
            + FormatXOver(Header['From:']) + #9
            + FormatXOver(Header['Date:']) + #9
            + FormatXOver(Header['Message-ID:']) + #9
            + FormatXOver(Header['References:']) + #9
            + FormatXOver(inttostr( ArtSize )            ) + #9
            + FormatXOver(sLines                         ) + #9
            + FormatXOver(Header['Xref:']) + #9
            + FormatXOver(Header['X-Hamster-Info:'])
            + CRLF;
      end
   finally
      Free
   end
end; {
function TApp.NewsArtXOver(GrpHdl, ArtNo: integer) : widestring;
var  Temp2: String; pEndHeader: Integer;

   Function Get (Const Headername: String): String;
   Var i, p: Integer;
   begin
      Result := '';
      p := Pos(CRLF+Headername, Temp2);
      If p > 0 then begin
         p := p + Length(CRLF+Headername);
         If Temp2[p] = ' ' then Inc(p);
         For i := p to pEndHeader-1 do begin
            Case Temp2[i] of
               #0, #9, #10: Result := Result + ' ';
               #13: If (Temp2[i+2]<>#9) and (Temp2[i+2]<>' ')
                       then break
                       else Result := Result + ' ';
               else Result := Result + Temp2[i]
            end
         end
      end
   end;

Var Temp, sLines: String;
    bFullHeader: Boolean;
    i, Anz, ArtSize: Integer;
begin
   Result:='';
   With TArticle.Create do try
      ArtSize  := 4096;
      Temp := ArticleBase.ReadArticleSized( GrpHdl, ArtNo, ArtSize );
      Temp2 := CRLF+LowerCase(Temp);
      pEndHeader := Pos(CRLF+CRLF, Temp);
      If pEndHeader > 0
         then sLines  := Get('lines:')
         else sLines := '';
      if (sLines='') or (pEndHeader=0) then begin
         Temp := ArticleBase.ReadArticle( GrpHdl, ArtNo );
         Temp2 := CRLF+LowerCase(Temp);
         pEndHeader := Pos(CRLF+CRLF, Temp);
         Anz := 0;
         For i := pEndHeader + 4 to Length(Temp) do If Temp[i]=#13 then Inc(Anz);
         sLines := inttostr( Anz )
      end;

      If Text > '' then begin
         // 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References 6:Bytes 7:Lines [8:Xref]
         Result := inttostr(ArtNo) + #9
            + FormatXOver( Get( 'subject:'    ) ) + #9
            + FormatXOver( Get( 'from:'       ) ) + #9
            + FormatXOver( Get( 'date:'       ) ) + #9
            + FormatXOver( Get( 'message-id:' ) ) + #9
            + FormatXOver( Get( 'references:' ) ) + #9
            + FormatXOver( inttostr( ArtSize )            ) + #9
            + FormatXOver( sLines                         ) + #9
            + FormatXOver( Get( 'xref: '       ) ) + #9
            + FormatXOver( Get('x-hamster-info:' ) )
            + CRLF;
      end
   finally
      Free
   end
end;
}
function TApp.NewsGrpXOver(GrpHdl, start, finish: Integer) : WideString;
//function TSrvNNTPCli.Cmd_XOVER( Par: String ): Boolean;
Var j, ArtSize: Integer; sLines: String;
begin
   Result := '';
   With TArticle.Create do try
      for j:=start to finish do begin
         ArtSize  := 4096;
         Text := ArticleBase.ReadArticleSized( GrpHdl, j, ArtSize );
         sLines   := Header['Lines:'];
         if (sLines='') or (FullBody = '') then begin
            Text := ArticleBase.ReadArticle( GrpHdl, j );
            sLines := inttostr( CountLines(FullBody) )
         end;
         If Text>'' then begin
            // 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References 6:Bytes 7:Lines [8:Xref]
            Result := Result + inttostr(j) + #9
               + FormatXOver( Header['Subject:']) + #9
               + FormatXOver( Header['From:']) + #9
               + FormatXOver( Header['Date:']) + #9
               + FormatXOver( Header['Message-ID:']) + #9
               + FormatXOver( Header['References:']) + #9
               + FormatXOver( inttostr( ArtSize )            ) + #9
               + FormatXOver( sLines ) + #9
               + FormatXOver( Header['Xref:']) + #9
               + FormatXOver( Header['X-Hamster-Info:'])
               + CRLF;
         end;
      end;
   finally
      Free
   end
end;

{/HSR}

{HSR} {OLE-Newsgroup-Informations}
function TApp.NewsGrpInformation(GrpHdl: Integer) : WideString;

  function FormatXOver( s: String ) : String;
  Var i: Integer;
  begin
     Setlength(Result, Length(s));
     For i := 1 to Length(s) do
        If s[i] IN[#9, #10, #13] then Result[i] := ' '
                                 else Result[i] := s[i]
  end;

var
  GrpName, SrvForGrp: string;
  c: Char;
  i: Integer;
begin

  GrpName := ArticleBase.Name[GrpHdl];
  SrvForGrp := '';
  for i:=0 to CfgHamster.PullCount-1 do begin
    if LowerCase(CfgHamster.PullGroup[i]) = GrpName then begin
        If SrvForGrp > '' then SrvForGrp := SrvForGrp + ',';
        SrvForGrp := SrvForGrp + CfgHamster.PullServer[i]
    end
  end;

  With TInifile.create(PATH_GROUPS + GrpName + '\data' + EXT_CFG) do try
     c := (ReadString( 'Setup', 'type', '' )+'y')[1];
     If not( c in ['y','n','m','g'] ) then c := 'y';
     Result :=
        FormatXOver( ReadString( 'Info',   'Description', '' ) ) + #9
        + FormatXOver( ReadString( 'Setup', 'pull.limit',
                       CfgIni.ReadString( 'Setup',  'pull.limit', IntToStr(Def_Pull_Limit)) ) ) + #9
        + FormatXOver( ReadString( 'Setup', 'purge.articles.keepdays',
                       CfgIni.ReadString( 'Setup',  'purge.articles.keepdays', IntToStr(Def_Purge_Articles_KeepDays)) ) ) + #9
        + FormatXOver( ReadString( 'Setup',  'postserver', '' ) ) + #9
        + FormatXOver( c ) + #9
        + SrvForGrp + #9
        + GrpName
  finally
    free
  end
end;
{/HSR}

function TApp.ControlGetNewsErrPath: WideString;
begin
     Result := PATH_NEWS_ERR
end;

function TApp.NewsGrpNameByHandle(GrpHdl: Integer): WideString;
begin
   Result := ArticleBase.Name[GrpHdl]
end;

function TApp.ClearXCounter(AbNr, BisNr: Integer): Integer;
Var i: Integer;
begin
   If (AbNr >= 0) and (BisNr <= 9) and (AbNr <= BisNr) then begin
      Result := 0;
      For i := AbNr to BisNr do SetCounter (CntCustomValues[i], 0)
   end else Result := -1
end;

function TApp.DecXCounter(Nr, Value: Integer): Integer;
begin
   If (Nr >= 0) and (Nr <= 9) then begin
      Result := 0;
      SetCounter (CntCustomValues[Nr], CntCustomValues[Nr]-Value)
   end else Result := -1
end;

function TApp.IncXCounter(Nr, Value: Integer): Integer;
begin
   If (Nr >= 0) and (Nr <= 9) then begin
      Result := 0;
      SetCounter (CntCustomValues[Nr], CntCustomValues[Nr]+Value)
   end else Result := -1
end;

function TApp.SetXCounter(Nr, Value: Integer): Integer;
begin
   If (Nr >= 0) and (Nr <= 9) then begin
      Result := 0;
      SetCounter (CntCustomValues[Nr], Value)
   end else Result := -1
end;

function TApp.XCounter(Nr: Integer): Integer;
begin
   If (Nr >= 0) and (Nr <= 9)
      then Result := GetCounter (CntCustomValues[Nr])
      else Result := 0
end;

{MG}{SSL}
function TApp.ControlRunFetchMailTLS(const Server, Port, User, Pass,
  Destuser: WideString; SSLMode, SSLVerify: SYSINT;
  const SSLCaFile: WideString): Integer;
var  DefUser, FiltSect: String;
     i: Integer;
begin
     Result := -1;
     If ArchivMode then Exit;

     AllShutdownReq := False;
     DefUser  := DestUser;
     FiltSect := '';
     i := Pos( ',', DefUser );
     if i>0 then begin
        FiltSect := copy( DefUser, i+1, 255 );
        DefUser  := copy( DefUser, 1, i-1 );
     end;
     TThreadPop3Fetch.Create( Server, Port, User, Pass, DefUser,
        FiltSect, '?', SSLMode, SSLVerify, SSLCaFile ).resume;
     Result := 0
end;

function TApp.ControlRunSendMailAuthTLS(const Server, Port, User, Pass,
  FromSelection: WideString; SSLMode, SSLVerify: SYSINT;
  const SSLCaFile: WideString): Integer;
begin
   Result := -1;
   If ArchivMode then Exit;
   AllShutdownReq := False;
   TThreadSmtpSend.Create( Server, Port, User, Pass, FromSelection, '',
      True, SSLMode, SSLVerify, SSLCaFile ).resume;
   Result := 0
end;

function TApp.ControlRunSendMailTLS(const Server, Port,
  FromSelection: WideString; SSLMode, SSLVerify: SYSINT;
  const SSLCaFile: WideString): Integer;
begin
   Result := -1;
   If ArchivMode then Exit;
   AllShutdownReq := False;
   TThreadSmtpSend.Create( Server, Port, '', '', FromSelection, '',
      False,  SSLMode, SSLVerify, SSLCaFile).resume;
   Result := 0
end;

function TApp.RasGetConnection: WideString;
begin
   Result := RasDynGetConnection
end;

function TApp.RasGetIP: WideString;
begin
   if RasDynIsConnected
      then Result := RasDynGetPppIp(RasDynGetConn)
      else Result := ''
end;

function TApp.RasListEntries: WideString;
Var sl: TStringList;
begin
   Result := '';
   sl := TStringList.Create;
   try
      if RasDynEnumPhonebookEntries( sl ) then Result := sl.Text
   finally
      sl.free
   end
end;

function TApp.ControlAddGroup(const Group: WideString): Integer;
begin
   Result := -1;
   If Not ArchivMode then begin
      If CfgHamster.ActiveAdd( Group ) then begin
         Result := 0;
         Log( LOGID_SYSTEM, TrGlF(kLog, 'OLE.GroupAdded.OK',
            'New Group "%s" created by OLE', Group) )
      end else begin
         Log( LOGID_WARN, TrGlF(kLog, 'OLE.GroupAdded.failed',
            'Creation of new group "%s" by OLE failed', Group) )
      end
   end
end;

function TApp.ControlDelGroup(const Group: WideString): Integer;
begin
   Result := -1;
   If Not ArchivMode then begin
      If CfgHamster.ActiveDel( Group ) then begin
         Result := 0;
         Log( LOGID_SYSTEM, TrGlF(kLog, 'OLE.GroupDeleted.OK',
            'Group "%s" deleted by OLE', Group) )
      end else begin
         Log( LOGID_WARN, TrGlF(kLog, 'OLE.GroupDeleted.failed',
            'Deletion of group "%s" by OLE failed', Group) )
      end
   end
end;

function TApp.ControlAddPull(const Server, Group: WideString): Integer;
begin
   Result := -1;
   If Not ArchivMode then begin
      If CfgHamster.PullAdd( Server, Group ) then begin
         Result := 0;
         Log( LOGID_SYSTEM, TrGlF(kLog, 'OLE.PullAdded.OK',
            'New pull for group "%s" from server "%s" created by OLE', [Group, Server]) )
      end else begin
         Log( LOGID_WARN, TrGlF(kLog, 'OLE.PullAdded.failed',
            'Creation of new pull for group "%s" from server "%s" by OLE failed', [Group, Server]) )
      end
   end
end;

function TApp.ControlDelPull(const Server, Group: WideString): Integer;
begin
   Result := -1;
   If Not ArchivMode then begin
      If CfgHamster.PullDel( Server, Group ) then begin
         Result := 0;
         Log( LOGID_SYSTEM, TrGlF(kLog, 'OLE.PullDeleted.OK',
            'Pull for group "%s" from server "%s" deleted by OLE', [Group, Server]) )
      end else begin
         Log( LOGID_WARN, TrGlF(kLog, 'OLE.PullDeleted.failed',
            'Deletion of pull for group "%s" from server "%s" by OLE failed', [Group, Server]) )
      end
   end
end;

function TApp.ControlRotateLog: Integer;
begin
   Result := 0;
   Logfile.RotateLog
end;

function TApp.hs2AsyncCommand(const Command: WideString): Integer;
Var sl: TStringList; s: String; p: Integer;
begin
   sl := TStringList.Create;
   try
      sl.Add ('#!hs2');
      If FileExists2 ( PATH_HSM + CFGFILE_SHELLMOD ) then sl.Add ('#!load hs2shell.hsm');
      s := Command;
      Repeat
         p := Pos('', s);
         If p > 0
            then begin sl.Add (Copy(s, 1, p-1)); Delete(s, 1, p) end
            else begin sl.Add (s); s := '' end
      until s = '';
      sl.Add ('quit');
      StartScriptLines ( sl, false )
   finally
      sl.free
   end
end;

initialization
{JW} {OLE-Server}
//  initialization
//   nach hamster.dpr verlagert
//  TAutoObjectFactory.Create(ComServer, TApp, Class_App, ciMultiInstance);
{MG}{SSL}
end.

