// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cAccount;

interface

uses Classes, IniFiles, FileCtrl;

const
  // account-/group-properties
  ACTP_USERNAME      = 'Username';
  ACTP_PASSWORD      = '!Password';
  ACTP_FULLNAME      = 'Fullname';
  ACTP_NEWSPOST      = 'NewsPost';
  ACTP_NEWSREAD      = 'NewsRead';
  ACTP_MAILBOX       = 'Mailbox';
  ACTP_MAILSEND      = 'MailSend';
  ACTP_MAILADDR      = 'MailAddr';
  ACTP_NEWSPEER      = 'NewsPeer';
  ACTP_GROUP         = 'Group';                                            //HSR //Groups
  ACTP_NEWNEWS      = 'NewsNewNews'; // 1=Enabled {JH}
  ACTP_CANCEL        = 'NewsCancel'; // 1=Enabled //HSR //ModeCancel 
  ACTP_REMOTECONTROL = 'RemoteControl';
  GRP_Name           = '';
  GRP_DESC           = 'Description';                                      //HSR     .
  GRP_NEWSPOST       = 'NewsPost';                                         //HSR     .
  GRP_NEWSREAD       = 'NewsRead';                                         //HSR     .
  GRP_MAILBOX        = 'Mailbox';                                          //HSR     .
  GRP_MAILSEND       = 'MailSend';                                         //HSR     .
  GRP_MAILADDR       = 'MailAddr';                                         //HSR     .
  GRP_NEWSPEER       = 'NewsPeer';                                         //HSR     .
  GRP_NEWNEWS        = 'NewsNews';
  GRP_NEWSCANCEL     = 'NewsCancel';
  GRP_REMOTE         = 'Remote';
  GRPM_SECTION    = 'COMMON';                                              //HSR     .
  GRPM_GROUPMAX   = 'GroupMax';                                            //HSR     .
  GRPM_SECTION2   = '0';                                                   //HSR //Groups

  {IMAP}
  MBTP_NONE          = 0; //not used
  MBTP_POP3          = 1; //"normal"/default
  MBTP_IMAP          = 2; //New (IMAP)
  {/IMAP}

  ACTID_NEW     = -1;
  ACTID_INVALID = -1;
  ACTID_ADMIN   =  1;

  LOCALMAILTYPE_NORMAL  = 0; // known local recipient
  LOCALMAILTYPE_INVALID = 1; // .invalid -> assume local, send to admin
  LOCALMAILTYPE_UNKNOWN = 2; // unknown local recipient -> notify admin

Const AccountChars: Set of Char
         =  ['!', '#', '$', '&', '''', '+', '-', '_',
             '0'..'9', '=', '.', 'A'..'Z', 'a'..'z'];
Function NormalizeAccountName (Const Account: String): String;

type
  TAccounts = class
    private
      LockedMailboxes: TList;
      LockedIMAPboxes: TStringList; //IMAP
      AuthLogins: TStringList;
      procedure SetValue( UserID: Integer; PropertyName, PropertyValue: String );
      function  GetValue( UserID: Integer; PropertyName: String ): String;
      procedure SetValueOfGroup( GroupID: Integer; PropertyName, PropertyValue: String );  //HSR //Groups
      function  GetValueOfGroup( GroupID: Integer; PropertyName: String ): String;
      procedure TestAccounts;  //HSR //Groups
    public
      ActFile   : TIniFile;
      GrpFile   : TIniFile;                                                //HSR //Groups
      FUserIDMax: Integer;
      FGroupIDMax:Integer;                                                 //HSR //Groups
      property  UserIDMax: Integer read FUserIDMax;
      property  Value[ UserID: Integer; PropertyName: String ]: String read GetValue write SetValue;
      property  ValueOfGroup[ GroupID: Integer; PropertyName: String ]: String read GetValueOfGroup write SetValueOfGroup; //HSR //Groups

      function  Add( UserID: Integer ): Integer;
      procedure Delete( UserID: Integer );
      procedure GroupDelete( GroupID: Integer );                           //HSR //Groups

      function  MailboxPath( UserID: Integer ): String;
      function  HasMailbox( UserID: Integer ): Boolean;
      function  MailboxLock( UserID: Integer; LockIt: Boolean ): Boolean;
      function  IsLocalMailbox( MailAddr: String;
         out UserID, LocalType: Integer ; Out bLocal: boolean ): Boolean;

      {IMAP}
      function  HasIMAPbox( UserID: Integer ): Boolean;
      function  IMAPMailboxLock( MailboxPath: String; LockIt: Boolean ): Boolean;
      procedure SetIMAPMailbox( MailboxPath: String; Mailbox: TObject );
      function  GetIMAPMailbox( MailboxPath: String ): TObject;
      procedure InitIMAPMailbox( UserID: Integer );
      {/IMAP}

      procedure LogAuthenticatedLogin( FromIP: LongInt );
      function  ChkAuthenticatedLogin( FromIP: LongInt ): Boolean;

      procedure SnapshotOfIDs( Strings: TStrings );
      function  UserIDOf( UserName: String ): Integer;
      function  GroupIDOf( GroupName: String ): Integer;                   //HSR //Groups
      function  IsUniqueUsername( UserName: String ): Boolean;
      function  IsUniqueGroupName( GroupName: String ): Boolean;           //HSR //Groups
      function  LoginID( UserName, Password: String ): Integer;

      procedure Lock;
      procedure Unlock;

      constructor Create;
      destructor Destroy; override;
  end;

function ExtractMailAddr( MailAddr: String ): String;
function IsLocalDomain( Domain: String ): Boolean;

implementation

uses Windows, SysUtils, uTools, Global, cPCRE, uBlowfish, uDateTime,
     cStdForm, cLogfile, cIMAPMailboxIndex;

const
  ACT_FILENAME    = 'Accounts.!!!';
  GRP_FILENAME    = 'Groups.!!!';                                          //HSR //Groups
  ACT_CODEKEY     = 'Just to make it "unreadable", not "safe"!';
  ACTPW_NOACCESS  = ' ';
  ACTPW_NOTNEEDED = '';
  ACTM_SECTION    = 'Common';
  ACTM_USERIDMAX  = 'UserIDMax';

var
  CS_ACCOUNTS: TRTLCriticalSection;

function ExtractMailAddr( MailAddr: String ): String;
var  k: Integer;
begin
     Result := TrimWhSpace( MailAddr );
     if copy(Result,1,1)='<' then begin
        System.Delete( Result, 1, 1 );
        k := Pos( '>', Result );
        if k>0 then Result:=copy(Result,1,k-1);
     end;
end;

function IsLocalDomain( Domain: String ): Boolean;
var  regex: TPCRE;
     re   : String;
begin
   // if not handled prior, force some domains to always be local-only
   Result := True;
   if Domain='' then exit; // no domain
   if Pos( '.', Domain )=0 then exit; // no valid domain
   if copy(Domain,length(Domain)-7,8)='.invalid' then exit; // .*\.invalid


   If Def_MID_FQDN_local then begin {JW} {FQDN_MID_Local}
      If lowercase(Domain)=lowercase(trim(Def_FQDNForMIDs)) then Exit
   end;
   If lowercase(Domain)=lowercase(trim(Def_FQDN)) then exit; //Domain is local

   // otherwise, the domain is always assumed non-local
   // unless the given 'local domain'-regex matches
   Result := False;
   if Def_IsLocalDomain=''  then exit; // not set

   // finally, the given domain-regex decides
   regex := TPCRE.Create( False, PCRE_CASELESS );
   try
      re := '^(' + Def_IsLocalDomain + ')$';
      try
         if regex.Match( PChar(re), PChar(Domain) ) then Result:=True;
      except
         on E:Exception do Log( LOGID_ERROR, TrGlF(kLog, 'Error.LocalDomain_failed',
               'IsLocalDomain failed for %s: %s', [Re, E.Message] ) );
      end
   finally
      regex.Free
   end
end;

function EncodeProperty( UserID: Integer; buf: String ): String;
var  i: Integer;
     s: String;
begin
     s := HamBlowfishEncipher( inttostr(UserID) + ACT_CODEKEY, buf );
     buf := '';
     for i:=1 to length(s) do begin
        buf := buf + inttohex( ord(s[i]), 2 );
     end;
     Result := buf;
end;

function DecodeProperty( UserID: Integer; buf: String ): String;
var  i: Integer;
     s: String;
begin
     s := '';
     for i:=1 to length(buf) div 2 do begin
        s := s + chr( strtoint( '$' + copy(buf,i*2-1,2) ) );
     end;
     if s='' then
        Result := #7#0#13#10
     else
        Result := HamBlowfishDecipher( inttostr(UserID) + ACT_CODEKEY, s );
end;

function TAccounts.Add( UserID: Integer ): Integer;
begin
     Lock;
     try
        Result := ACTID_INVALID;
        if UserID=ACTID_NEW then UserID := FUserIDMax + 1;

        // delete all properties of old account
        if UserID<=FUserIDMax then Delete( UserID );

        // create account by setting some properties
        Value[ UserID, ACTP_USERNAME ] := TrGl('Accounts', 'Newuser.Name', 'newuser') + inttostr(UserID);
        Value[ UserID, ACTP_PASSWORD ] := ACTPW_NOACCESS;
        Value[ UserID, ACTP_FULLNAME ] := TrGl('Accounts', 'Newuser.FullNameNr', 'New User #') + inttostr(UserID);
//        Value[ UserID, ACTP_GROUP ] := TrGl('Accounts','Newuser.Group','Admin');  //HSR //Groups
        Value[ UserID, ACTP_GROUP ] := 'Admin';                            //HSR //Groups

        // adjust global values
        if UserID>FUserIDMax then begin
           FUserIDMax := UserID;
           ActFile.WriteInteger( ACTM_SECTION, ACTM_USERIDMAX, FUserIDMax );
        end;

        Result := UserID;

     finally Unlock; end;
end;

procedure TAccounts.Delete( UserID: Integer );
begin
     Lock;
     try
        // delete all properties of account by deleting whole ini-section
        if UserID<=FUserIDMax then ActFile.EraseSection( inttostr(UserID) );
     finally Unlock; end;
end;

{HSR} {GROUPS}
procedure TAccounts.GroupDelete( GroupID: Integer );
 { TODO 1 -oHSR -cGroups : Timer-Problem? (Messagbox nach 2.try/i:=) (lock???) }
var
  i:integer; s: String;
//Groups.!!! bereinigen (Abschnitt GRPM_SECTION2 ('0')) {HSR}
begin
   Lock;
   try
      FGroupIDMax := GrpFile.ReadInteger( GRPM_SECTION, GRPM_GROUPMAX, 1);
      For i := GroupID to FGroupIDMax-1 do begin
         Grpfile.WriteString(GRPM_SECTION2, IntToStr(i),
            GrpFile.Readstring(GRPM_SECTION2, IntToStr(i+1),''));
        s := GRP_DESC;     ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_NEWSPOST; ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_NEWSREAD; ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_MAILBOX;  ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_MAILSEND; ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_MAILADDR; ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_NEWSPEER; ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_NEWNEWS;  ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_NEWSCANCEL; ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ];
        s := GRP_REMOTE;   ValueOfGroup[ i, s ] := ValueOfGroup [ i+1, s ]
      end;
      GrpFile.DeleteKey (GRPM_SECTION2, IntToStr(FGroupIDMax));
      GrpFile.EraseSection(inttostr(FGroupIDMax));
      Dec (FGroupIDMax);
      GrpFile.WriteInteger( GRPM_SECTION, GRPM_GROUPMAX, FGroupIDMax);
   finally
      Unlock
   end
end;

procedure TAccounts.SetValueOfGroup( GroupID: Integer; PropertyName, PropertyValue: String ); //HSR
begin
     Lock;
     try
        if copy(PropertyName,1,1)='!' then PropertyValue:=EncodeProperty(GroupID,PropertyValue);
        GrpFile.WriteString( inttostr(GroupID), PropertyName, PropertyValue );
     finally Unlock; end;
end;

function TAccounts.GetValueOfGroup( GroupID: Integer; PropertyName: String ): String;   //HSR
begin
   If PropertyName = GRP_Name then begin
      Result := GrpFile.ReadString( GRPM_SECTION2, IntToStr(GroupID), '')
   end else begin
     Lock;
     try
        Result := '';
        try
           Result := GrpFile.ReadString( inttostr(GroupID), PropertyName, '' );
           if copy(PropertyName,1,1)='!' then Result:=DecodeProperty(GroupID,Result);
        except
           on E:Exception do
              Log( LOGID_ERROR, 'Account.GetValueOfGroup.Exception: ' + E.Message );
        end;
     finally Unlock; end;
   end
end;
{/HSR}

procedure TAccounts.SetValue( UserID: Integer; PropertyName, PropertyValue: String );
begin
     Lock;
     try
        if copy(PropertyName,1,1)='!' then PropertyValue:=EncodeProperty(UserID,PropertyValue);
        ActFile.WriteString( inttostr(UserID), PropertyName, PropertyValue );
     finally Unlock; end;
end;

function TAccounts.GetValue( UserID: Integer; PropertyName: String ): String;
begin
     Lock;
     try
        Result := '';
        try
           Result := ActFile.ReadString( inttostr(UserID), PropertyName, '' );
{JW} {Accounts}
//           if copy(PropertyName,1,1)='!' then Result:=DecodeProperty(UserID,Result);

           if (PropertyName<>'') and (Result<>'') then
             if copy(PropertyName,1,1)='!' then Result:=DecodeProperty(UserID,Result);
{JW}
        except
           on E:Exception do
              Log( LOGID_ERROR, 'Account.GetValue.Exception: ' + E.Message );
        end;
     finally Unlock end
end;

function TAccounts.MailboxPath( UserID: Integer ): String;
var  s: String;
begin
     Lock;
     try
        s := Value[ UserID, ACTP_USERNAME ];
        if s='' then Result:=''
                else Result := PATH_MAILS + s + '\';
     finally Unlock; end;
end;

function TAccounts.HasMailbox( UserID: Integer ): Boolean;
begin
   Lock;
   try
      if UserID=ACTID_ADMIN
         then Result := True
         else Result := StrToInt(Value[ UserID, ACTP_MAILBOX ]) = MBTP_POP3; //IMAP

      if Result then begin
         if not DirectoryExists( MailboxPath(UserID) ) then begin
            ForceDirectories( MailboxPath(UserID) )
         end
      end else begin
         if StrToInt(Value[ UserID, ACTP_MAILBOX ]) = MBTP_NONE then begin//IMAP_not_delete
            if DirectoryExists( MailboxPath(UserID) ) then begin
               if not RemoveDir( MailboxPath(UserID) ) then begin
                  Log( LOGID_WARN, TrGlF('Accounts', 'Mailbox_undeleteble',
                     'Mailbox "%s" could not be deleted!', Value[UserID, ACTP_USERNAME]))
               end
            end
         end
      end
   finally
      Unlock
   end
end;

function TAccounts.MailboxLock( UserID: Integer; LockIt: Boolean ): Boolean;
var  i: Integer;
begin
     Lock;
     try
        Result := False;
        i := LockedMailboxes.IndexOf( Pointer(UserID) );
        if LockIt then begin
           if i<0 then begin
              LockedMailboxes.Add( Pointer(UserID) );
              Result := True;
           end;
        end else begin
           if i>=0 then begin
              LockedMailboxes.Delete( i );
              Result := True;
           end;
        end;
     finally Unlock end;
end;

function TAccounts.IsLocalMailbox( MailAddr: String;
   out UserID, LocalType: Integer; Out bLocal: boolean ): Boolean;
var  UserIDs        : TStringList;
     TmpUserID, i, k: Integer;
     MailAddrs, MailNam, MailDom, s: String;
     prs            : TParser;
     IsLocal        : Boolean;
begin
   Result    := False;
   UserID    := ACTID_INVALID;
   MailAddr  := ExtractMailAddr( MailAddr );
   LocalType := LOCALMAILTYPE_NORMAL;
   IsLocal   := false;

   Lock;

   try
      // 1.) check list of assigned addresses if local mailboxes are enabled
      UserIDs := TStringList.Create;
      prs     := TParser.Create;
      try
         SnapshotOfIDs( UserIDs );
         for i:=0 to UserIDs.Count-1 do begin
            TmpUserID := LongInt( UserIDs.Objects[i] );
            if HasMailbox( TmpUserID ) or HasIMAPbox( TmpUserID ) then begin
               MailAddrs := Value[ TmpUserID, ACTP_MAILADDR ];
               if MailAddrs<>'' then begin
                  prs.Parse( MailAddrs, ' ' );
                  k := 0;
                  repeat
                     s := prs.sPart( k, '' );
                     if s > '' then begin
                        if CompareText( s, MailAddr )=0 then begin
                           Result := True;
                           UserID := TmpUserID;
                           break;
                        end;
                        inc( k )
                     end       
                  until (s = '') or Result
               end;
            end;
         end
      finally
         prs.Free;
         UserIDs.Free
      end;

      // 2.) check for local domains
      if UserID=ACTID_INVALID then begin
         i := RScan ( '@', MailAddr );
         if i=0 then begin
            MailNam := MailAddr;
            MailDom := ''; // no domain -> always local
            IsLocal := True;
         end else begin
            MailNam := copy( MailAddr, 1, i-1 );
            MailDom := copy( MailAddr, i+1, Length(MailAddr)-i );

            if copy( MailDom, length(MailDom)-7, 8 )='.invalid' then begin
               IsLocal := True;
               LocalType := LOCALMAILTYPE_INVALID;
            end else begin
               IsLocal := IsLocalDomain( MailDom );
            end;
         end;

         if IsLocal then begin
            Result := True;
            UserID := UserIdOf( MailNam );
            if UserID<>ACTID_ADMIN then begin
               if not (HasMailbox( UserID ) or HasIMAPbox( UserID )) then UserID:=ACTID_INVALID
            end;

            if UserID=ACTID_INVALID then begin
               UserID := ACTID_ADMIN;
               if LocalType=LOCALMAILTYPE_NORMAL then LocalType:=LOCALMAILTYPE_UNKNOWN;
            end;
         end;
      end;

      bLocal := IsLocal

   finally Unlock; end;
end;

{IMAP}
function TAccounts.HasIMAPbox( UserID: Integer ): Boolean; //HSR //IMAP104
begin
  if UserID=ACTID_ADMIN
     then Result := false
     else Result := (StrToInt(Value[ UserID, ACTP_Mailbox ]) = MBTP_IMAP )
end;

function TAccounts.IMAPMailboxLock( MailboxPath: String; LockIt: Boolean ): Boolean;
var  i: Integer;
begin
     Lock;
     try
        Result := False;
        i := LockedIMAPboxes.IndexOf( MailboxPath );
        if LockIt then begin
           if i<0 then begin
              LockedIMAPboxes.Add( MailboxPath );
              Result := True;
           end;
        end else begin
           if i>=0 then begin
              LockedIMAPboxes.Delete( i );
              Result := True;
           end;
        end;
     finally Unlock end;
end;

procedure TAccounts.SetIMAPMailbox( MailboxPath: String; Mailbox: TObject );
var  i: Integer;
begin
     Lock;
     try
        i := LockedIMAPboxes.IndexOf( MailboxPath );
        if i >= 0 then LockedIMAPboxes.Objects[i] := Mailbox;
     finally Unlock end
end;

function TAccounts.GetIMAPMailbox( MailboxPath: String ): TObject;
var  i: Integer;
begin
     Result := nil;
     Lock;
     try
        i := LockedIMAPboxes.IndexOf( MailboxPath );
        if i >= 0 then Result := LockedIMAPboxes.Objects[i];
     finally Unlock end
end;

procedure TAccounts.InitIMAPMailbox( UserID: Integer );

     function RemoveFileIfExists( FileName: String ): Boolean;
     begin
          if FileExists2( FileName )
             then Result := DeleteFile( FileName )
             else Result := True
     end;
     
begin
   Lock;
   try
      if HasMailbox( UserID ) then begin
         ForceDirectories(MailboxPath(UserID));
         if not ( RemoveFileIfExists( MailboxPath(UserID) + IMAPINDEX_FILENAME )
              and RemoveFileIfExists( MailboxPath(UserID) + IMAPSTATUS_FILENAME ) )
         then begin
            Log( LOGID_WARN, Format( 'IMAP-Mailbox "%s" could not be initialized!',
                 [Value[ UserID, ACTP_USERNAME]]))
         end
      end;
   finally
      Unlock
   end
end;
{/IMAP}

procedure TAccounts.LogAuthenticatedLogin( FromIP: LongInt );
var  ValidTil, i: LongInt;
begin
     Lock;
     try
        // add/extend authentication period for address
        ValidTil := DateTimeToUnixTime( NowGMT ) + Def_SmtpAfterPop3Period;
        i := AuthLogins.IndexOf( inttohex(FromIP,8) );
        if i<0 then begin
           AuthLogins.AddObject( inttohex(FromIP,8), Pointer(ValidTil) );
        end else begin
           AuthLogins.Objects[i] := Pointer( ValidTil );
        end;

        // remove expired periods
        ValidTil := DateTimeToUnixTime( NowGMT );
        for i:=AuthLogins.Count-1 downto 0 do begin
           if LongInt(AuthLogins.Objects[i])<ValidTil then AuthLogins.Delete(i);
        end;
     finally Unlock; end;
end;

function TAccounts.ChkAuthenticatedLogin( FromIP: LongInt ): Boolean;
var  i: Integer;
begin
     Lock;
     try
        Result := False;
        i := AuthLogins.IndexOf( inttohex(FromIP,8) );
        if i>=0 then begin
           if LongInt(AuthLogins.Objects[i])>=DateTimeToUnixTime(NowGMT) then begin
              Result := True;
           end else begin
              AuthLogins.Delete( i ); // expired
           end;
        end;
     finally Unlock; end;
end;

procedure TAccounts.SnapshotOfIDs( Strings: TStrings );
var  i: Integer;
     s: String;
begin
     Lock;
     try
        try
           ActFile.ReadSections( Strings );
           With Strings do begin
              For i := Count-1 downto 0 do begin
                 s := Strings[i];
                 if s='' then Delete(i)
                 else if (s[1]<'1') or (s[1]>'9') then Delete( i )
                 else Objects[i] := Pointer( strtoint(s) )
              end;
           end;
        except
           on E:Exception do
              Log( LOGID_ERROR, 'Account.SnapshotOfIDs.Exception: ' + E.Message );
        end;
     finally Unlock; end;
end;

function TAccounts.UserIDOf( UserName: String ): Integer;
var  TS: TStringList;
     i : Integer;
begin
     Lock;
     try
        Result := ACTID_INVALID;
        try
           TS := TStringList.Create;
           SnapshotOfIDs( TS );
           Username := NormalizeAccountName(Username);
           for i:=0 to TS.Count-1 do begin
              if CompareText(Value[LongInt(TS.Objects[i]),ACTP_USERNAME],UserName)=0 then begin
                 Result := LongInt(TS.Objects[i]);
                 break;
              end;
           end;
           TS.Free;
        except
           on E:Exception do
              Log( LOGID_ERROR, 'Account.UserIDOf.Exception: ' + E.Message );
        end;
     finally Unlock; end;
end;

{HSR} {GROUPS}
function TAccounts.GroupIDOf( GroupName: String ): Integer;                //HSR
var i : Integer;
begin
     Lock;
     try
        Result := ACTID_INVALID;
        try
    //       messagebox(0,Pchar(groupname),'Group.GroupIdOf',0);
           for i:=1 to GrpFile.ReadInteger(GRPM_SECTION, GRPM_GROUPMAX,1) do begin
              if CompareText(GrpFile.ReadString(GRPM_SECTION2,IntToStr(i),' '),GroupName)=0 then begin
                 Result := i;
                 break;
              end;
           end;
        except
           on E:Exception do
              Log( LOGID_ERROR, 'Account.GroupIDOf.Exception: ' + E.Message );
        end;
     finally Unlock; end;
end;

function TAccounts.IsUniqueGroupName( GroupName: String ): Boolean;        //HSR {ACC-GRP}
var
   i   : integer;
begin
     Lock;
     try
       Result:=true;
       for i:=1 to GrpFile.ReadInteger( GRPM_SECTION, GRPM_GROUPMAX,1) do begin
         if (Grpfile.ReadString(GRPM_SECTION2,IntToStr(i),'Error')=GroupName) then Result:=false;
       end;
     finally Unlock; end;
end;
{/HSR}

function TAccounts.IsUniqueUsername( UserName: String ): Boolean;
begin
     Lock;
     try
        Result := ( UserIDOf(UserName) = ACTID_INVALID );
     finally Unlock; end;
end;

function TAccounts.LoginID( UserName, Password: String ): Integer;
var  pw: String;
begin
     Lock;
     try
        Result := UserIDOf( UserName );
        if Result<>ACTID_INVALID then begin
           pw := Value[ Result, ACTP_PASSWORD ];
           if pw=ACTPW_NOACCESS then Result:=ACTID_INVALID;
           if pw<>ACTPW_NOTNEEDED then begin
              if pw<>Password then Result:=ACTID_INVALID;
           end;
        end;
     finally Unlock; end;
end;

procedure TAccounts.Lock;
begin
     EnterCriticalSection( CS_ACCOUNTS );
end;

procedure TAccounts.Unlock;
begin
     LeaveCriticalSection( CS_ACCOUNTS );
end;

constructor TAccounts.Create;
begin
     inherited Create;

     ActFile := TIniFile.Create( PATH_BASE + ACT_FILENAME );
     GrpFile := TIniFile.Create( PATH_BASE + GRP_FILENAME );               //HSR //Groups

     LockedMailboxes := TList.Create;

     LockedIMAPboxes := TStringList.Create; //IMAP
     LockedIMAPboxes.Sorted := True; //IMAP
     {$IFDEF VER140} // Delphi 6
     LockedIMAPboxes.CaseSensitive := False; //IMAP
     {$ENDIF}

     AuthLogins := TStringList.Create;
     AuthLogins.Sorted := True;

     FUserIDMax := ActFile.ReadInteger( ACTM_SECTION, ACTM_USERIDMAX, 0 );
     if FUserIDMax<=0 then begin
        // create common section at top of file
        ActFile.WriteInteger( ACTM_SECTION, ACTM_USERIDMAX, UserIDMax );

        // create default admin-account
        Add( ACTID_ADMIN );
        Value[ ACTID_ADMIN, ACTP_USERNAME ] := 'admin';
        Value[ ACTID_ADMIN, ACTP_PASSWORD ] := ACTPW_NOTNEEDED;
        Value[ ACTID_ADMIN, ACTP_FULLNAME ] := 'Hamster Administrator';
        Value[ ACTID_ADMIN, ACTP_NEWSPOST ] := '.*';
        Value[ ACTID_ADMIN, ACTP_NEWSREAD ] := '.*';
        Value[ ACTID_ADMIN, ACTP_MAILBOX  ] := '1';
        Value[ ACTID_ADMIN, ACTP_MAILSEND ] := '1';
        Value[ ACTID_ADMIN, ACTP_NEWSPEER ] := '1';
//     end;
{HSR}
        Value[ ACTID_ADMIN, ACTP_GROUP    ] := 'Admin';
     end;
     // create default groups.!!!
     FGroupIDMax := GrpFile.Readinteger( GRPM_SECTION, GRPM_GROUPMAX,0);
     if FGroupIDMax<1 then begin
       GrpFile.WriteInteger( GRPM_SECTION, GRPM_GROUPMAX, 1);
       ValueOfGroup[ 0,          '1'           ] := 'Admin';
       ValueOfGroup[ 1,          GRP_DESC      ] := 'Administration';
//       ValueOfGroup[ 1,          ACTP_PASSWORD ] := ACTPW_NOTNEEDED;
       ValueOfGroup[ 1,          GRP_NEWSPOST  ] := '.*';
       ValueOfGroup[ 1,          GRP_NEWSREAD  ] := '.*';
       ValueOfGroup[ 1,          GRP_MAILBOX   ] := '1';
       ValueOfGroup[ 1,          GRP_MAILSEND  ] := '1';
//       GRP_MAILADDR       = 'MailAddr';
//       GrpFile.WriteString('1','GroupAccount','hamsteradmin');

     end;
{/HSR}

     TestAccounts
end;

{TGL}
Procedure TAccounts.TestAccounts;
Var sl: TStringList;
    i, Nr: Integer;
    s, s2: String;
    f: File;
begin
     // Test Name of Mailboxes
     sl := TStringList.Create;
     try
        SnapshotOfIDs( sl );
        for i:=0 to sl.Count-1 do begin
           Nr := LongInt( sl.Objects[i] );
           s := Value[ Nr, ACTP_USERNAME ];
           s2 := NormalizeAccountName (s);
           If s <> s2 then begin
              try
                 AssignFile(f, PATH_MAILS + s);
                 FileMode := 2;
                 Rename(f, PATH_MAILS + s2)
              except end;
              Value[ Nr, ACTP_USERNAME ] := s2
           end
        end
     finally
        sl.Free
     end
end;
{TGL}

destructor TAccounts.Destroy;
begin
     ActFile.Free;
     GrpFile.Free;                                                         //HSR //Groups
     LockedMailboxes.Free;
     LockedIMAPBoxes.Free;
     AuthLogins.Free;
     inherited Destroy;
end;

Function NormalizeAccountName (Const Account: String): String;
Var i: Integer;
begin
   SetLength(Result, Length(Account));
   For i := 1 to Length(Result) do
      If Account[i] IN AccountChars
         then Result[i] := Account[i]
         else Result[i] := '-'
end;

initialization
  InitializeCriticalSection( CS_ACCOUNTS );

finalization
  DeleteCriticalSection( CS_ACCOUNTS );

end.
