// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cArticleFile; // Storage for a single newsgroup

// ----------------------------------------------------------------------------
// Contains a class to access and modify the files of a single newsgroup. This
// class is not used directly, it is covered by the class in "cArtFiles.pas".
// ----------------------------------------------------------------------------

interface

uses Windows, uTools, SysUtils, Classes, IniFiles, cIndexRc, uDateTime;

const
  ARTFILEREAD       = fmOpenRead      or fmShareDenyNone;
  ARTFILEWRITE      = fmOpenReadWrite or fmShareDenyWrite;
  ARTFILEEXCLUSIVE  = fmOpenReadWrite or fmShareExclusive;

{JW} {ArtFile}
// const
//  EXT_DAT = '.dat';
//  EXT_IDX = '.idx';
//  EXT_CFG = '.ini';
{JW}


type
  TCompactCanPurgeFunc = function ( Data: PChar; PurgePar: LongInt ) : Boolean of object;

  PIndexRec = ^TIndexRec;
  TIndexRec = record
     DatKey : LongInt;
     DatPos : LongInt;
  end;

  TArticleFile = class
    private
      CS_THISFILE: TRTLCriticalSection;
      IndexFile: TIndexedRecords;
      FDriveIdx: Byte;
      FFileBase: String;
      FGroupname : String;
      FSdat: TFileStreamEx;
      UseFileMode  : Word;
      PurgeCount : Integer;
      GroupIni   : TIniFile;

{JW} {N2M}
      function  GetModerator: String;
{JW}
      function  GetDatKey( Index: Integer ): LongInt;
      function  GetDatPos( Index: Integer ): LongInt;

      procedure WriteRec( Key: LongInt; const Data; DataSize: LongInt );
      function  ReadSize( Key: LongInt ): Integer;
      procedure ReadData( Key: LongInt; var Data; MaxSize: Integer );
      Function Delete ( Const Key: LongInt ): boolean;

      function  Compact( CanPurgeFunc: TCompactCanPurgeFunc;
                         KeepDays: LongInt;
                         var NewKeyMin, NewKeyMax: LongInt): Boolean;
      function  PurgeForReset( PArtText: PChar; KeepMaxDays: LongInt ) : Boolean; // JAWO 12.08.01 (Reset Group)

      function  GetLocalMin: Integer;
      procedure SetLocalMin( NewLocalMin: Integer );
      function  GetLocalMax: Integer;
      procedure SetLocalMax( NewLocalMax: Integer );
      function  GetServerMin( Server: String ): Integer;
      procedure SetServerMin( Server: String; NewArtMin: Integer );
      function  GetServerMax( Server: String ): Integer;
      procedure SetServerMax( Server: String; NewArtMin: Integer );
      function  GetServerLow( Server: String ): Integer;            // JAWO 12.01.2001 (lowest server artno)
      procedure SetServerLow( Server: String; NewArtLow: Integer ); // JAWO 12.01.2001 (lowest server artno)
{JW} {Feed}
      function  GetFeederLast( Server: String ): Integer;
      procedure SetFeederLast( Server: String; NewArtMin: Integer );
{JW}
      function  GetPullLimit( Server: String ): Integer;
      function  GetPurgeKeepDays: Integer;
      procedure SetPurgeKeepDays( NewKeepDays: Integer );
      function  GetLastClientRead: TDateTime;
      procedure SetLastClientRead( NewClientRead: TDateTime );
      function  GetDescription: String;
      procedure SetDescription( NewDescription: String );

      function  IsOldEnoughToPurge( PArtText: PChar; KeepMaxDays: LongInt ) : Boolean;
      function GetCount: LongInt;

    public
      property Count: LongInt read GetCount;
      property  DatKey[ Index:Integer ]: LongInt read GetDatKey;
      property  DatPos[ Index:Integer ]: LongInt read GetDatPos;
      property GroupName: String read FGroupname;

      function  GetProp( PropName: String; DefaultValue: String ): String;
      procedure SetProp( PropName: String; NewValue: String );

      property  LocalMin: Integer read GetLocalMin write SetLocalMin;
      property  LocalMax: Integer read GetLocalMax write SetLocalMax;
      property  ServerMin[ Server:String ]: Integer read GetServerMin write SetServerMin;
      property  ServerMax[ Server:String ]: Integer read GetServerMax write SetServerMax;
      property  ServerLow[ Server:String ]: Integer read GetServerLow write SetServerLow; // JAWO 12.01.2001 (lowest server artno)
{JW} {Feed}
      property  FeederLast[ Server:String ]: Integer read GetFeederLast write SetFeederLast;
{JW}
      property  PullLimit[ Server:String ]: Integer read GetPullLimit;
      property  PurgeKeepDays: Integer read GetPurgeKeepDays write SetPurgeKeepDays;
      property  LastClientRead: TDateTime read GetLastClientRead write SetLastClientRead;
      property  Description: String read GetDescription write SetDescription;
      procedure SetFirstPullDone( Server: String );
      function  DTCreated: TDateTime;
{JW} {N2M}
      property  Moderator: String read GetModerator;
{JW}
      function  IndexOfKey( Key: LongInt ): LongInt;
      function  Open( fmMode: Word ): Boolean; virtual;
      procedure Close; virtual;
      procedure Flush; virtual;

      function  ReserveArtNo: LongInt;
      function  ReadArticle( ArtNo: Integer; var MaxSize: Integer ): String;
      function  WriteArticle( ArtNo: Integer; const ArtText: String ): LongInt;
      Function  DeleteArticle( Const ArtNo: Integer ): boolean;

      procedure Purge;
      procedure PurgeReset; // JAWO 12.08.01 (Reset Group)
      procedure Reindex;

      procedure EnterThisFile;
      procedure LeaveThisFile;

      constructor Create( AGroupName: String );
      destructor Destroy; override;
  end;

implementation

uses Global, cArticle, cFiltersNews, cStdForm, cLogfile;

const
  DATMASK_SIZE     = $00FFFFFF;
  DATMASK_RESERVED = $7F000000;
  DATMASK_DELETED  = LongInt($80000000);

// --------------------------------------------------------- TArticleFile -----

function TArticleFile.GetCount: LongInt;
begin
     if Assigned(IndexFile) then Result := IndexFile.Count
                            else Result := 0;
end;

procedure TArticleFile.WriteRec( Key: LongInt; const Data; DataSize: LongInt );
var  idx: Integer;
     siz: LongInt;
     IRec : TIndexRec;
     DPos : LongInt;
begin
   try
      if DataSize>DATMASK_SIZE then DataSize:=DATMASK_SIZE;
      idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
      if idx<0 then begin
         IRec.DatKey := Key;
         IRec.DatPos := FSdat.Size;
         IndexFile.RecAdd( idx, IRec );
      end else begin
         DPos := DatPos[idx];
         if (DPos<0) or (DPos>FSdat.Size) then begin // Datenmll
            DPos := FSdat.size;
            siz  := 0;
         end else begin
            FSdat.Seek( DPos, soFromBeginning );
            FSdat.Read( siz, sizeof(siz) );
         end;
         if DataSize<>siz then begin
            siz := siz or DATMASK_DELETED;
            FSdat.Seek( DPos, soFromBeginning );
            FSdat.Write( siz, sizeof(siz) );
            PIndexRec( IndexFile.RecPtr(idx) )^.DatPos := FSdat.size;
            IndexFile.Changed := True;
         end;
      end;
      try FSdat.Seek( DatPos[idx], soFromBeginning );
      except Log( LOGID_ERROR, 'cArtFile.WriteRec.Seek Exception' ); raise end;
      try FSdat.Write( DataSize, sizeof(DataSize) );
      except Log( LOGID_ERROR, 'cArtFile.WriteRec.WriteKey Exception' ); raise end;
      try FSdat.Write( Data, DataSize )
      except Log( LOGID_ERROR, 'cArtFile.WriteRec.WriteRec Exception' ); raise end;
   except
      on E:Exception do Log( LOGID_ERROR, 'cArtFile.WriteRec: ' + E.Message );
   end;
end;

Function TArticleFile.Delete ( Const Key: LongInt ): Boolean;
var  idx: Integer;
     siz: LongInt;
begin
   Result := false;
   idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
   if idx<0 then exit;
   try
      FSdat.Seek( DatPos[idx], soFromBeginning );
      FSdat.Read( siz, sizeof(siz) );
      siz := siz or DATMASK_DELETED;
      FSdat.Seek( DatPos[idx], soFromBeginning );
      FSdat.Write( siz, sizeof(siz) )
   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'cArtFile.Delete: ' + E.Message );
         exit
      end
   end;
   try
      IndexFile.RecDelete( idx );
   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'cArtFile.Delete.IndexDelete: ' + E.Message );
         exit
      end
   end;
   Result := true;
end;

function TArticleFile.Compact( CanPurgeFunc: TCompactCanPurgeFunc;
                               KeepDays: LongInt;
                               var NewKeyMin, NewKeyMax: LongInt ): Boolean;
var  OldPos, NewPos   : LongInt;
     RecKey, RecSize  : LongInt;
     RecData          : PChar;
     IsDeleted        : Boolean;
     DebugInfo        : String;
     TestIdx, MaxLimit: LongInt;
begin
   Result := False;
   try
     Log( LOGID_DEBUG, TrGlF(kLog, 'Debug.Compact', 'Compact: %s', [FFileBase] ) );

     // save changes; close files if currently open
     Close;

     // open exclusive
     if not Open( ARTFILEEXCLUSIVE ) then begin
        Log( LOGID_WARN, TrGlF(kLog, 'Warning.CompactCancelled',
               'Compact cancelled; %s is in use.', [FFileBase] ));
        exit;
     end;

     MaxLimit  := FSdat.Size - 1;
     NewPos    := 0;
     NewKeyMin := NewKeyMax + 1;
     NewKeyMax := 0;

     // sort index entries by position in .dat file
     IndexFile.Sort( sizeof(LongInt) {=Offset TIndexRec.DatPos} );  //JH //Purge-Fix

     TestIdx := 0;
     while TestIdx<IndexFile.Count do begin

        // get old values
        RecKey := DatKey[TestIdx];
        OldPos := DatPos[TestIdx];

        if (OldPos>=0) and (OldPos<=MaxLimit-3) then begin
           FSdat.Seek( OldPos, soFromBeginning );
           FSdat.Read( RecSize, sizeof(RecSize) );
        end else begin
           // damaged pointer in index-file
           Log( LOGID_WARN, TrGlF(kLog, 'Warning.DamagedIndPointerDeleted',
                'Damaged index-pointer deleted (ArtNo=%s)', [inttostr(RecKey)]) );
           RecSize := DATMASK_DELETED;
        end;

        IsDeleted := ( RecSize and DATMASK_DELETED ) <> 0;
        RecSize   := RecSize and DATMASK_SIZE;
        if (OldPos+3+RecSize)>MaxLimit then begin
           // damaged pointer in data-file
           Log( LOGID_WARN, TrGlF(kLog, 'Warning.DamagedDataPointerDeleted',
               'Damaged data-pointer deleted (ArtNo=%s)', [inttostr(RecKey)]) );
           IsDeleted := True;
        end;

        DebugInfo := inttostr(RecKey) + ': ('
                   + inttostr(OldPos) + '/' + inttostr(RecSize) + ') -> ';

        // old data is marked as deleted
        if IsDeleted then DebugInfo:=DebugInfo + 'DEL';

        RecData := nil;

        if not(IsDeleted) and (RecSize>0) then begin
           // read old data
           GetMem( RecData, RecSize + 1 );
           RecData^ := #0;
           FSdat.Read( RecData^, RecSize );

           if RecData^<#32 then begin
              DebugInfo := DebugInfo + 'INVALID';
              FreeMem( RecData, RecSize + 1 );
              RecData := nil;
           end else begin
              // check, if old data should be purged
              if Assigned(CanPurgeFunc) then begin
                 RecData[RecSize] := #0; // ASCIIZ
                 if CanPurgeFunc( RecData, KeepDays ) then begin
                    DebugInfo := DebugInfo + 'PURGE';
                    FreeMem( RecData, RecSize + 1 );
                    RecData := nil;
                 end;
              end;
           end;
        end;

        if Assigned(RecData) then begin
           // article is valid and should be kept

           // adjust key-bounds
           if NewKeyMin=0 then NewKeyMin:=RecKey;
           if NewKeyMax=0 then NewKeyMax:=RecKey;
           if RecKey<NewKeyMin then NewKeyMin:=RecKey;
           if RecKey>NewKeyMax then NewKeyMax:=RecKey;

           if OldPos=NewPos then begin
              // remains at old position in .dat-file
              DebugInfo := DebugInfo + 'KEEP';
           end else begin
              // move to new position in .dat-file
              DebugInfo := DebugInfo + 'MOVE ' + inttostr(NewPos);
              PIndexRec( IndexFile.RecPtr(TestIdx) )^.DatPos := NewPos; // update index-entry
              IndexFile.Changed := True;
              FSdat.Seek ( NewPos, soFromBeginning );
              FSdat.Write( RecSize, sizeof(RecSize) );
              FSdat.Write( RecData^, RecSize );
           end;

           // adjust position for next data-record
           NewPos := NewPos + sizeof(RecSize);
           NewPos := NewPos + RecSize;

           FreeMem( RecData, RecSize + 1 );

           inc( TestIdx );

        end else begin

           // article is invalid or should be purged -> delete index-entry
           IndexFile.RecDelete( TestIdx );

        end;

     end;

     if NewKeyMin=0 then NewKeyMin:=1;

     // cut unused space at end of .dat-file
     FSdat.Seek( NewPos, soFromBeginning );
     SetEndOfFile( FSdat.Handle );

     // save new index
     IndexFile.Changed := True;
     IndexFile.Sort; // re-sort index by article numbers //JH //Purge-Fix
     IndexFile.FlushToFile;

     // close files again
     Close;

     Result := True;
   except
     on E:Exception do Log( LOGID_ERROR, 'cArtFile.Compact: ' + E.Message );
   end
end;

function TArticleFile.Open( fmMode: Word ) : Boolean;
begin
   Result := False;
   UseFileMode := fmMode;
   Close;
   try
      FSdat := TFileStreamEx.Create( FFileBase + EXT_DAT, UseFileMode );
      If FFileBase[2]=':'
         then FDriveIdx := Ord(FFileBase[1])-Ord('A')+1
         else FDriveIdx := 0;
      Result   := true;
   except
      on E:EFOpenError do begin
         Result := False;
         Log( LOGID_DEBUG, FFileBase + EXT_DAT + ': ' + E.Message );
         FSdat := nil;
      end;
      on E:Exception do begin

         Log( LOGID_DEBUG, FFileBase + EXT_DAT + ': ' + E.Message );
      end;
   end;
   IndexFile := TIndexedRecords.Create( FFileBase + EXT_IDX,
                                        sizeof(TIndexRec) ); // , True );
   IndexFile.LoadFromFile;
// JAWO 13.08.01 (Purge) removed (will be set inside LoadFromFile):
//     IndexFile.Sort;              // to be removed in future versions
//     IndexFile.Changed := False;  // to be removed with .sort
   if (IndexFile.Count=0) and (FSdat.Size>0) then Reindex;
end;

procedure TArticleFile.Flush;
begin
   try
      if Assigned(IndexFile) then IndexFile.FlushToFile;
      if Assigned(FSdat)     then FlushFileBuffers( FSdat.Handle );
   except
      on E:Exception do Log( LOGID_ERROR, 'cArtFile.flush: ' + E.Message );
   end
end;

procedure TArticleFile.Close;
begin
   try
     Flush;
     if Assigned( IndexFile ) then begin
        IndexFile.Free;
        IndexFile := nil;
     end;
     if FSdat<>nil then begin
        FSdat.Free;
        FSdat := nil;
     end
   except
     on E:Exception do Log( LOGID_ERROR, 'cArtFile.Close: ' + E.Message )
   end
end;

function TArticleFile.IndexOfKey( Key: LongInt ): LongInt;
begin
   Result := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
end;

function TArticleFile.GetLocalMin: Integer;
begin
     Result := GroupIni.ReadInteger( 'Ranges', 'Local.Min', 1 );
end;

procedure TArticleFile.SetLocalMin( NewLocalMin: Integer );
begin
     GroupIni.WriteInteger( 'Ranges', 'Local.Min', NewLocalMin );
end;

function TArticleFile.GetLocalMax: Integer;
begin
     Result := GroupIni.ReadInteger( 'Ranges', 'Local.Max', 0 );
end;

procedure TArticleFile.SetLocalMax( NewLocalMax: Integer );
begin
     GroupIni.WriteInteger( 'Ranges', 'Local.Max', NewLocalMax );
end;

function TArticleFile.GetServerMin( Server: String ): Integer;
begin
//     Result := GroupIni.ReadInteger( 'Ranges', Server+'.Min', 0 );
     Result := GroupIni.ReadInteger( 'Ranges', Server+'.Min', -1 ); // instead of 0 as default //JAWO //XOver420
end;

procedure TArticleFile.SetServerMin( Server: String; NewArtMin: Integer );
begin
     GroupIni.WriteInteger( 'Ranges', Server+'.Min', NewArtMin );
end;

function TArticleFile.GetServerMax( Server: String ): Integer;
begin
     Result := GroupIni.ReadInteger( 'Ranges', Server+'.Max', 0 );
end;

procedure TArticleFile.SetServerMax( Server: String; NewArtMin: Integer );
begin
     GroupIni.WriteInteger( 'Ranges', Server+'.Max', NewArtMin );
end;

{JAWO} {(lowest server artno)}
function TArticleFile.GetServerLow( Server: String ): Integer;
begin
     Result := GroupIni.ReadInteger( 'Ranges', Server+'.Low', 0 );
end;

procedure TArticleFile.SetServerLow( Server: String; NewArtLow: Integer );
begin
     GroupIni.WriteInteger( 'Ranges', Server+'.Low', NewArtLow );
end;
{/JAWO}

{JW} {Feed}
function TArticleFile.GetFeederLast( Server: String ): Integer;
begin
     Result := GroupIni.ReadInteger( 'Ranges', Server+'.LastFeed', 0 );
end;

procedure TArticleFile.SetFeederLast( Server: String; NewArtMin: Integer );
begin
     GroupIni.WriteInteger( 'Ranges', Server+'.LastFeed', NewArtMin );
end;
{JW}

procedure TArticleFile.SetFirstPullDone( Server: String );
begin
     GroupIni.WriteBool( 'Ranges', Server+'.FirstPullDone', True );
end;

function TArticleFile.GetProp( PropName: String; DefaultValue: String ): String;
begin
     Result := GroupIni.ReadString( 'Setup', PropName, DefaultValue );
end;

procedure TArticleFile.SetProp( PropName: String; NewValue: String );
begin
     GroupIni.WriteString( 'Setup', PropName, NewValue );
end;

function TArticleFile.GetPullLimit( Server: String ): Integer;
begin
     if not GroupIni.ReadBool( 'Ranges', Server+'.FirstPullDone', False ) then begin
        Result := Def_Pull_Limit_First;
        exit;
     end;

     Result := GroupIni.ReadInteger( 'Setup', 'pull.limit', Def_Pull_Limit );
end;

function TArticleFile.GetPurgeKeepDays: Integer;
begin
     Result := GroupIni.ReadInteger( 'Setup', 'purge.articles.keepdays', Def_Purge_Articles_KeepDays );
end;

procedure TArticleFile.SetPurgeKeepDays( NewKeepDays: Integer );
begin
     GroupIni.WriteInteger( 'Setup', 'purge.articles.keepdays', NewKeepDays );
end;

function TArticleFile.GetLastClientRead: TDateTime;
var  s: String;
begin
     s := GroupIni.ReadString( 'Info', 'LastClientRead', '' );
     if s='' then Result:=0 else Result:=TimeStampToDateTime( s );
end;

procedure TArticleFile.SetLastClientRead( NewClientRead: TDateTime );
begin
   If Not ArchivMode then
      GroupIni.WriteString( 'Info', 'LastClientRead', DateTimeToTimeStamp( NewClientRead ) );
end;

function TArticleFile.GetDescription: String;
begin
     Result := GroupIni.ReadString( 'Info', 'Description', '' );
end;

procedure TArticleFile.SetDescription( NewDescription: String );
begin
     GroupIni.WriteString( 'Info', 'Description', NewDescription );
end;

function TArticleFile.DTCreated: TDateTime;
var  h: String;
begin
     h := GroupIni.ReadString( 'Info', 'Created', '' );
     if h='' then Result := NowGMT
             else Result := TimeStampToDateTime( h );
end;

function TArticleFile.GetDatKey( Index: Integer ): LongInt;
begin
     if (Index>=0) and (Index<IndexFile.Count) then
        Result := PIndexRec( IndexFile.RecPtr(Index) )^.DatKey
     else
        Result := -1;
end;

function TArticleFile.GetDatPos( Index: Integer ): LongInt;
begin
     if (Index>=0) and (Index<IndexFile.Count) then
        Result := PIndexRec( IndexFile.RecPtr(Index) )^.DatPos
     else
        Result := -1;
end;

function TArticleFile.ReadSize( Key: LongInt ): Integer;
var  idx: Integer;
     siz: LongInt;
begin
     try
        Result := 0;
        idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
        if idx<0 then exit;
        FSdat.Seek( DatPos[idx], soFromBeginning );
        FSdat.Read( siz, sizeof(siz) );
        Result := siz and DATMASK_SIZE;
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'cArtFile.ReadSize: ' + E.Message );
           Result := 0;
        end;
     end;
end;

procedure TArticleFile.ReadData( Key: LongInt; var Data; MaxSize: Integer );
var  idx, dpos: Integer;
     siz: LongInt;
begin
     try
        idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
        if idx<0 then exit;

        dpos := DatPos[idx];
        if dpos<0 then exit;

        FSdat.Seek( dpos, soFromBeginning );
        FSdat.Read( siz, sizeof(siz) );
        siz := siz and DATMASK_SIZE;
        if siz>MaxSize then siz:=MaxSize;
        FSdat.Read( Data, siz );
     except
        on E:Exception do Log( LOGID_ERROR, 'cArtFile.ReadData: ' + E.Message );
     end;
end;

function TArticleFile.ReadArticle( ArtNo: Integer; var MaxSize: Integer ): String;
var  siz, rsiz: Integer;
     buf: PChar;
begin
     try
        siz := ReadSize( ArtNo );
        rsiz := siz;
        if (MaxSize>0) and (siz>MaxSize) then siz:=MaxSize;
        MaxSize := rsiz; // return real size

        if siz<=0 then begin
           Result := '';
        end else begin
           GetMem( buf, siz+1 );
           ReadData( ArtNo, buf^, siz );
           (buf+siz)^ := #0;
           Result := String( buf );
           FreeMem( buf, siz+1 );
        end;
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'cArtFile.ReadArticle: ' + E.Message );
           Result := '';
        end;
     end;
end;

function TArticleFile.ReserveArtNo: LongInt;
begin
     LocalMax := LocalMax + 1;
     Result := LocalMax;
end;

function TArticleFile.WriteArticle( ArtNo: Integer; const ArtText: String ): LongInt;
Var NeededSize: Integer;
begin
   NeededSize := Length(ArtText) + 10240;
   If DiskFree(FDriveIdx) < NeededSize then begin
      raise Exception.Create('Article can not be saved - not enough empty space on drive '
        + Chr(FDriveIdx+Ord('A')-1)+':')
   end;
   try
      if ArtNo<=0 then begin
         LocalMax := LocalMax + 1;
         ArtNo := LocalMax;
      end;
      if LocalMin=0 then LocalMin:=ArtNo;
      if LocalMax=0 then LocalMax:=ArtNo;
      if ArtNo<LocalMin then LocalMin:=ArtNo;
      if ArtNo>LocalMax then LocalMax:=ArtNo;
      WriteRec( ArtNo, ArtText[1], length(ArtText) );
      Result := ArtNo;
   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'cArtFile.WriteArticle: ' + E.Message );
         Result := 0;
      end;
   end;
end;

Function TArticleFile.DeleteArticle( Const ArtNo: Integer ): boolean;
begin
     Result := Delete( ArtNo );
end;

function TArticleFile.IsOldEnoughToPurge( PArtText: PChar; KeepMaxDays: LongInt ) : Boolean;
var  Art : TArticle;
     Base: TDateTime;
     Days: LongInt;
     MID : String;
begin
     Result := False;
     if KeepMaxDays<=0 then exit;

     Art := TArticle.Create;
     try
        Art.Text := String( PArtText );
        Base := Art.GetReceivedDT;
        MID  := Art['Message-ID:']
     finally
        Art.Free
     end;

     if MID='' then begin // damaged?
        inc(PurgeCount);
        Result:=True;
        exit;
     end;

     Days := Trunc( Now - Base );
//     if Days>KeepMaxDays then begin
        if Days>=KeepMaxDays then begin        //HSR
        inc(PurgeCount);
        Result:=True;
        exit;
     end;
end;

procedure TArticleFile.Purge;
var  AMin, AMax: LongInt;
begin
     PurgeCount := 0;

     if PurgeKeepDays>0 then begin
        AMin := LocalMin;
        AMax := LocalMax;
        if Compact( IsOldEnoughToPurge, PurgeKeepDays, AMin, AMax ) then begin
           LocalMin := AMin;
           if AMax>LocalMax then LocalMax := AMax;
        end;
     end;

     Log( LOGID_INFO, TrGlF(kLog, 'Info.PurgeResult', 'Purge %s (%sd): %s articles purged.',
                      [GroupName, inttostr(PurgeKeepDays), inttostr(PurgeCount)]) );
end;

procedure TArticleFile.Reindex;
var  CurPos, FileLen, RecSize, RecKey: Integer;
     NewKeyMin, NewKeyMax, idx: Integer;
     IRec     : TIndexRec;
     RecData  : PChar;
     IsDeleted: Boolean;
     Art      : TArticle;
     s: String;
     Cnt, i: Integer;
begin
   try
     Log( LOGID_Debug, TrGlF(kLog, 'Debug.AutoReIndex',
        'Auto-reindex of %s started due to missing index-file.', [s]) );

     IndexFile.Clear;
     CurPos  := 0;
     Cnt     := 0;
     FileLen := FSdat.Size;

     NewKeyMin := LocalMin;
     NewKeyMax := LocalMax;
     if NewKeyMin<1 then NewKeyMin:=1;
     if NewKeyMax<NewKeyMin then NewKeyMax:=NewKeyMin-1;

     while CurPos<FileLen do begin
        FSdat.Seek( CurPos, soFromBeginning );
        FSdat.Read( RecSize, sizeof(RecSize) );

        IsDeleted := ( RecSize and DATMASK_DELETED ) <> 0;
        RecSize   := RecSize and DATMASK_SIZE;
        if (CurPos+RecSize)>=FileLen then break; // damaged pointer in data-file

        if not IsDeleted then begin
           RecKey := NewKeyMax + 1; // default: assign next article-number

           GetMem( RecData, RecSize + 1 );
           RecData^ := #0;
           FSdat.Read( RecData^, RecSize );
           RecData[RecSize] := #0; // ASCIIZ
           Art := TArticle.Create;
           Art.Text := String( RecData );
           s  := LowerCase( Art['Xref:']);
           Art.Free;
           FreeMem( RecData, RecSize + 1 );

           if s<>'' then begin
              i := Pos( LowerCase(Groupname)+':', s );
              if i>0 then begin
                 System.Delete( s, 1, i+length(GroupName) );
                 i := PosWhSpace( s );
                 if i>0 then s := copy(s,1,i-1);
                 if s<>'' then begin
                    i := strtointdef( s, -1 );
                    if (i>=NewKeyMin) and (i<=NewKeyMax) then RecKey:=i; // use Xref-number
                 end;
              end;
           end;

           if NewKeyMin=0 then NewKeyMin:=RecKey;
           if NewKeyMax=0 then NewKeyMax:=RecKey;
           if RecKey<NewKeyMin then NewKeyMin:=RecKey;
           if RecKey>NewKeyMax then NewKeyMax:=RecKey;

           IRec.DatKey := RecKey;
           IRec.DatPos := CurPos;
           IndexFile.RecAdd( idx, IRec );
           inc( Cnt );
        end;

        inc( CurPos, sizeof(RecSize) );
        inc( CurPos, RecSize );
     end;

     // save new index
     IndexFile.Changed := True;
     IndexFile.FlushToFile;

     // save new bounds
     LocalMin := NewKeyMin;
     LocalMax := NewKeyMax;

     Log( LOGID_DEBUG, TrGlF(kLog, 'Debug.AutoReIndex.done',
            'Auto-reindex of %s done; %s articles recovered.',
             [Groupname, inttostr(Cnt)]));
   except
     on E:Exception do Log( LOGID_ERROR, 'cArtFile.ReIndex: ' + E.Message )
   end
end;

procedure TArticleFile.EnterThisFile;
begin
     EnterCriticalSection( CS_THISFILE );
end;

procedure TArticleFile.LeaveThisFile;
begin
     LeaveCriticalSection( CS_THISFILE );
end;


constructor TArticleFile.Create( AGroupName: String );
begin
     inherited Create;

     FFileBase := PATH_GROUPS + AGroupName + '\data';

     // create path and files if they don't exist already
     ForceFileExists( FFileBase + EXT_DAT );
     ForceFileExists( FFileBase + EXT_IDX );

     UseFileMode := ARTFILEEXCLUSIVE;
     FSdat       := nil;
     IndexFile   := nil;

     InitializeCriticalSection( CS_THISFILE );

     FGroupName := AGroupName;

     GroupIni := TIniFile.Create( PATH_GROUPS + AGroupName + '\data' + EXT_CFG );

     if GroupIni.ReadString( 'Info', 'Created', '' )='' then begin
        GroupIni.WriteString( 'Info', 'Created', DateTimeToTimeStamp(NowGMT) )
     end;
end;

destructor TArticleFile.Destroy;
begin
   Close;
   DeleteCriticalSection( CS_THISFILE );
   GroupIni.Free;
   inherited Destroy;
end;

function TArticleFile.PurgeForReset(PArtText: PChar;
  KeepMaxDays: Integer): Boolean;
begin
  inc(PurgeCount);
  Result := true;
end;

procedure TArticleFile.PurgeReset;
var  AMin, AMax: LongInt;
begin
     PurgeCount := 0;

     if PurgeKeepDays>0 then begin
        AMin := LocalMin;
        AMax := LocalMax;
        if Compact( PurgeForReset, PurgeKeepDays, AMin, AMax ) then begin
           LocalMin := AMin;
           if AMax>LocalMax then LocalMax := AMax;
        end;
     end;

     Log( LOGID_INFO, Format(TrGl(kLog, 'Info.PurgeResult', 'Purge %s (%sd): %s articles purged.'),
                      [GroupName, inttostr(PurgeKeepDays), inttostr(PurgeCount)]) );
end;

{JW} {N2M}
function TArticleFile.GetModerator: String;
begin
     Result := GroupIni.ReadString( 'Setup', 'Moderator', '' );
end;
{JW}

end.

