// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cChunkRc;

interface

uses SysUtils, Classes, Windows, uTools;

type
  TChunkMPtrArr     = array[0..255] of PChar;
  PChunkMPtrArr     = ^TChunkMPtrArr;
  TChunkSizeArr     = array[0..255] of LongInt;
  PChunkSizeArr     = ^TChunkSizeArr;
  TChunkChangedArr  = array[0..255] of Boolean;
  PChunkChangedArr  = ^TChunkChangedArr;
  TChunkCritSectArr = array[0..255] of TRTLCriticalSection;
  PChunkCritSectArr = ^TChunkCritSectArr;

  TChunkedRecords = class
    private
      FPCritSect: PChunkCritSectArr;
      CHUNK_BITS: Integer;
      CHUNK_SHR : Integer;
      FPMemPtr  : PChunkMPtrArr;
      FPMemSize : PChunkSizeArr;
      FPUseSize : PChunkSizeArr;
      FRecSize  : LongInt;
      FSorted   : Boolean;
      FPChanged : PChunkChangedArr;
      FFileBase : String;
      FFileExt  : String;

      function  GetSumCount: LongInt;
      procedure RecKeyFindPos( const Data;
                               KeySize: LongInt;
                               WantInsertPos: Boolean;
                               out Chunk: Byte;
                               out Index: Integer );
      procedure QuickSort( Chunk: Byte; L, R: Integer );
      procedure SetSorted( NewSorted: Boolean );

      function  RecPtr( Chunk: Byte; Index: Integer ): PChar;

      procedure RecAdd( out Chunk: Byte; out Index: Integer; const Data );
      procedure RecIns( Chunk: Byte; Index: Integer );
      procedure RecInsert( Chunk: Byte; Index: Integer; const Data );
      procedure RecDelete( Chunk: Byte; Index: Integer );

      function  RecKeyInsPosOf( Chunk: Byte; KeySize: LongInt; const Data ): Integer;

    protected
      CHUNK_MAX : Integer;

      function  ChunkFilename( Chunk: Byte ): String;
      function  ChunkCount( Chunk: Byte ): LongInt;
      function  ChunkOf( const Data ): Byte;
      procedure RecGet( Chunk: Byte; Index: Integer; var Data );
      procedure RecSet( Chunk: Byte; Index: Integer; const Data );
      function  RecKeyIndexOf ( Chunk: Byte; KeySize: LongInt; const Data ): Integer;

    public
      property  Count  : LongInt read GetSumCount;
      property  Sorted : Boolean read FSorted  write SetSorted;
      property  RecSize: LongInt read FRecSize;
      property  ChunkBits: Integer read CHUNK_BITS;

      procedure Add   ( const Data );
      procedure Remove( const Data );
      procedure RemoveKey( const Data; KeySize: LongInt );
      function  ContainsKey( const Data; KeySize: LongInt ): Boolean;

      procedure Clear;
      procedure Pack;
      procedure Sort;

      procedure Enter( Chunk: Byte );
      procedure Leave( Chunk: Byte );

      procedure LoadFromFile; virtual;
      procedure SaveToFile;

      constructor Create( AFileBase, AFileExt : String;
                          AChunkBits, ARecSize: LongInt;
                          ASorted             : Boolean );
      destructor Destroy; override;
  end;

implementation

// ---------------------------------------------------------------- Tools -----

function MemCompare( p1, p2: PChar; Length: Integer ): Integer;
var  i: Integer;
begin
     Result := 0;

     for i:=1 to Length do begin
        if p1^<p2^ then begin
           Result := -1;
           break;
        end else begin
           if p1^>p2^ then begin
              Result := +1;
              break;
           end;
        end;
        inc( p1 );
        inc( p2 );
     end;
end;

// ------------------------------------------------------ TChunkedRecords -----

function TChunkedRecords.ChunkFilename( Chunk: Byte ): String;
begin
     if CHUNK_BITS=0 then
        Result := FFileBase + FFileExt
     else
        Result := FFileBase
                + inttostr( CHUNK_BITS )
                + inttohex( Chunk, 2 )
                + FFileExt;
end;

function TChunkedRecords.ChunkOf( const Data ): Byte;
begin
     Move( Data, Result, 1 );
     Result := Result shr CHUNK_SHR;
end;

procedure TChunkedRecords.QuickSort( Chunk: Byte; L, R: Integer );
var  I, J     : Integer;
     DP, DI, DJ: Pointer;
begin
     GetMem( DI, FRecSize );
     GetMem( DJ, FRecSize );
     GetMem( DP, FRecSize );
     repeat
        I := L;
        J := R;
        RecGet( Chunk, (L+R) div 2 , DP^ );   // RecPtr->RecGet // P:= -> , DP^
        repeat
           while MemCompare( RecPtr(Chunk,I), DP, FRecSize ) < 0 do inc( I );
           while MemCompare( RecPtr(Chunk,J), DP, FRecSize ) > 0 do dec( J );
           if I <= J then begin
              RecGet( Chunk, I, DI^ );
              RecGet( Chunk, J, DJ^ );
              RecSet( Chunk, J, DI^ );
              RecSet( Chunk, I, DJ^ );
              inc( I );
              dec( J );
           end;
        until I > J;
        if L < J then QuickSort( Chunk, L, J );
        L := I;
     until I >= R;
     FreeMem( DI, FRecSize );
     FreeMem( DJ, FRecSize );
     FreeMem( DP, FRecSize );
end;

procedure TChunkedRecords.Sort;
var  Chunk: Byte;
begin
   for Chunk:=0 to CHUNK_MAX do begin
      Enter( Chunk );
      try
         if FPUseSize^[Chunk]>0 then QuickSort( Chunk, 0, ChunkCount(Chunk)-1 );
      finally Leave(Chunk) end;
   end;
end;

procedure TChunkedRecords.SetSorted( NewSorted: Boolean );
begin
   if NewSorted and not FSorted then Sort;
   FSorted := NewSorted;
end;

function TChunkedRecords.ChunkCount( Chunk: Byte ): LongInt;
begin
     Enter( Chunk );
     try
        if FRecSize>0 then Result := FPUseSize^[Chunk] div FRecSize
                      else Result := 0;
     finally Leave(Chunk) end;
end;

function TChunkedRecords.GetSumCount: LongInt;
var  Chunk: Byte;
begin
   Result := 0;
   if FRecSize>0 then begin
      for Chunk:=0 to CHUNK_MAX do inc( Result, ChunkCount(Chunk) );
   end;
end;

procedure TChunkedRecords.Clear;
var  Chunk: Byte;
begin
   for Chunk:=0 to CHUNK_MAX do begin
      Enter( Chunk );
      try
         FPUseSize^[Chunk] := 0;
         FPChanged^[Chunk] := True;
      finally Leave( Chunk ) end;
   end;
end;

procedure TChunkedRecords.Pack;
var  Chunk: Byte;
begin
   for Chunk:=0 to CHUNK_MAX do begin
      Enter( Chunk );
      try
         if FPUseSize^[Chunk]<FPMemSize^[Chunk] then begin
            FPMemSize^[Chunk] := FPUseSize^[Chunk];
            ReallocMem( FPMemPtr^[Chunk], FPMemSize^[Chunk] );
         end;
      finally Leave( Chunk ) end;
   end;
end;

function TChunkedRecords.RecPtr( Chunk: Byte; Index: Integer ): PChar;
begin
     Enter( Chunk );
     try
        Result := FPMemPtr^[Chunk] + Index * FRecSize;
     finally Leave( Chunk ) end;
end;

procedure TChunkedRecords.RecGet( Chunk: Byte; Index: Integer; var Data );
begin
     Enter(Chunk);
     try
        if (Index>=0) and (Index<ChunkCount(Chunk)) then begin
           Move( RecPtr(Chunk,Index)^, Data, FRecSize );
        end;
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.RecSet( Chunk: Byte; Index: Integer; const Data );
begin
     Enter( Chunk );
     try
        if (Index>=0) and (Index<ChunkCount(Chunk)) then begin
           Move( Data, RecPtr(Chunk,Index)^, FRecSize );
           FPChanged^[Chunk] := True;
        end;
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.RecAdd( out Chunk: Byte; out Index: Integer; const Data );
begin
     Chunk := ChunkOf( Data );
     Enter( Chunk );
     try
        if FSorted then begin
           Index := RecKeyInsPosOf( Chunk, RecSize, Data );
        end else begin
           Index := ChunkCount(Chunk);
        end;
        RecInsert( Chunk, Index, Data );
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.RecIns( Chunk: Byte; Index: Integer );
var  reserved: LongInt;
begin
     Enter( Chunk );
     try
        if (Index>=0) and (Index<=ChunkCount(Chunk)) then begin
           inc( FPUseSize^[Chunk], FRecSize );
           if FPUseSize^[Chunk]>FPMemSize^[Chunk] then begin
              reserved := ChunkCount(Chunk) div 100; // pre-alloc for 1% of current count
              if reserved<1 then reserved:=1;
              FPMemSize^[Chunk] := FPUseSize^[Chunk] + FRecSize * reserved;
              ReallocMem( FPMemPtr^[Chunk], FPMemSize^[Chunk] );
           end;
           if Index<>ChunkCount(Chunk)-1 then begin
              Move( RecPtr(Chunk,Index)^,
                    RecPtr(Chunk,Index+1)^,
                    FRecSize*(ChunkCount(Chunk)-Index-1) );
           end;
           FPChanged^[Chunk] := True;
        end;
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.RecInsert( Chunk: Byte; Index: Integer; const Data );
begin
     Enter( Chunk );
     try
        if (Index>=0) and (Index<=ChunkCount(Chunk)) then begin
           RecIns( Chunk, Index );
           RecSet( Chunk, Index, Data );
        end;
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.RecDelete( Chunk: Byte; Index: Integer );
begin
     Enter( Chunk );
     try
        if (Index>=0) and (Index<ChunkCount(Chunk)) then begin
           if Index<>ChunkCount(Chunk)-1 then begin
              Move( RecPtr(Chunk,Index+1)^,
                    RecPtr(Chunk,Index)^,
                    FRecSize*(ChunkCount(Chunk)-Index-1) );
           end;
           dec( FPUseSize^[Chunk], FRecSize );
           FPChanged^[Chunk] := True;
        end;
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.RecKeyFindPos( const Data;
                                         KeySize: LongInt;
                                         WantInsertPos: Boolean;
                                         out   Chunk: Byte;
                                         out   Index: Integer );
var  i, res, min, max, InsPos: Integer;
     P: PChar;
begin
     Chunk := ChunkOf( Data );
     Enter( Chunk );
     try

     if FSorted then begin

        Index  := -1;
        InsPos := 0;

        min := 0;
        max := ChunkCount(Chunk)-1;

        repeat
           if min>max then break;
           i := ( min + max ) div 2;

           res := MemCompare( RecPtr(Chunk,i), @Data, KeySize );

           if res<0 then begin
              min := i+1;
              InsPos := min; //=compared-pos.+1
           end else begin
              if res>0 then begin
                 InsPos := i;
                 max := i-1;
              end else begin
                 InsPos := i; //=compared-pos.
                 Index  := i; // = already in list
                 break;
              end;
           end;

        until False;

        if WantInsertPos then Index:=InsPos;

     end else begin

        if WantInsertPos then begin
           Index := ChunkCount(Chunk); // append
        end else begin
           Index := -1;
           P := RecPtr( Chunk, 0 );
           for i:=0 to ChunkCount(Chunk)-1 do begin
              if CompareMem( P, @Data, KeySize ) then begin
                 Index := i;
                 break;
              end;
              inc( P, FRecSize );
           end;
        end;

     end;

     finally Leave(Chunk) end;
end;

function TChunkedRecords.RecKeyInsPosOf( Chunk: Byte; KeySize: LongInt; const Data ): Integer;
begin
   RecKeyFindPos( Data, KeySize, True, Chunk, Result );
end;

function TChunkedRecords.RecKeyIndexOf( Chunk: Byte; KeySize: LongInt; const Data ): Integer;
begin
   RecKeyFindPos( Data, KeySize, False, Chunk, Result );
end;

procedure TChunkedRecords.Add( const Data );
var  Chunk: Byte;
     Index: Integer;
begin
   RecAdd( Chunk, Index, Data );
end;

procedure TChunkedRecords.RemoveKey( const Data; KeySize: LongInt );
var  Chunk: Byte;
     Index: Integer;
begin
     Chunk := ChunkOf( Data );
     Enter( Chunk );
     try
        Index := RecKeyIndexOf( Chunk, KeySize, Data );
        if Index>=0 then RecDelete( Chunk, Index );
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.Remove( const Data );
begin
     RemoveKey( Data, RecSize );
end;

function TChunkedRecords.ContainsKey( const Data; KeySize: LongInt ): Boolean;
var  Chunk: Byte;
begin
     Chunk  := ChunkOf( Data );
     Enter( Chunk );
     try
        Result := ( RecKeyIndexOf( Chunk, KeySize, Data ) >= 0 );
     finally Leave(Chunk) end;
end;

procedure TChunkedRecords.LoadFromFile;
var  FileStream: TFileStream;
     Chunk     : Byte;
begin
   Clear;
   for Chunk:=0 to CHUNK_MAX do begin
      Enter( Chunk );
      try
         if FileExists2( ChunkFilename(Chunk) ) then begin
            FileStream := TFileStream.Create( ChunkFilename(Chunk), fmOpenRead );
            FreeMem( FPMemPtr^[Chunk], FPMemSize^[Chunk] );
            FPUseSize^[Chunk] := ( (FileStream.Size+FRecSize-1) div FRecSize ) * FRecSize;
            FPMemSize^[Chunk] := FPUseSize^[Chunk];
            FPMemPtr^ [Chunk] := AllocMem( FPMemSize^[Chunk] );
            FileStream.Read( FPMemPtr^[Chunk]^, FPUseSize^[Chunk] );
            FileStream.Free;
         end;
         FPChanged^[Chunk] := False;
      finally Leave(Chunk) end;
   end;
end;

procedure TChunkedRecords.SaveToFile;
var  FileStream: TFileStream;
     Chunk     : Byte;
begin
     for Chunk:=0 to CHUNK_MAX do begin
        Enter( Chunk );
        try
           if FPChanged^[Chunk] then begin
              FileStream := TFileStream.Create( ChunkFilename(Chunk), fmCreate );
              FileStream.Write( FPMemPtr^[Chunk]^, FPUseSize^[Chunk] );
              FlushFileBuffers( FileStream.Handle );
              FileStream.Free;
              FPChanged^[Chunk] := False;
           end;
        finally Leave( Chunk ) end;
     end;
end;

procedure TChunkedRecords.Enter( Chunk: Byte );
begin
     EnterCriticalSection( FPCritSect^[Chunk] );
end;

procedure TChunkedRecords.Leave( Chunk: Byte );
begin
     LeaveCriticalSection( FPCritSect^[Chunk] );
end;

constructor TChunkedRecords.Create( AFileBase, AFileExt : String;
                                    AChunkBits, ARecSize: LongInt;
                                    ASorted             : Boolean );
var  Chunk: Byte;
begin
     inherited Create;

     if AChunkBits<0 then AChunkBits:=0;
     if AChunkBits>8 then AChunkBits:=8;
     CHUNK_BITS := AChunkBits;
     CHUNK_MAX  := ( 1 shl CHUNK_BITS ) - 1;
     CHUNK_SHR  := 8 - CHUNK_BITS;

     FRecSize  := ARecSize;
     FSorted   := ASorted;
     FFileBase := AFileBase;
     FFileExt  := AFileExt;

     GetMem( FPUseSize,  (CHUNK_MAX+1) * sizeof(LongInt) );
     GetMem( FPMemSize,  (CHUNK_MAX+1) * sizeof(LongInt) );
     GetMem( FPMemPtr ,  (CHUNK_MAX+1) * sizeof(PChar  ) );
     GetMem( FPChanged,  (CHUNK_MAX+1) * sizeof(Boolean) );
     GetMem( FPCritSect, (CHUNK_MAX+1) * sizeof(TRTLCriticalSection) );

     for Chunk:=0 to CHUNK_MAX do begin
        FPChanged^[Chunk] := False;
        FPUseSize^[Chunk] := 0;
        FPMemSize^[Chunk] := FRecSize;
        FPMemPtr^ [Chunk] := AllocMem( FPMemSize^[Chunk] );
        InitializeCriticalSection( FPCritSect^[Chunk] );
     end;
end;

destructor TChunkedRecords.Destroy;
var  Chunk: Byte;
begin
     for Chunk:=0 to CHUNK_MAX do begin
        FreeMem( FPMemPtr^[Chunk], FPMemSize^[Chunk] );
        DeleteCriticalSection( FPCritSect^[Chunk] );
     end;
     FreeMem( FPUseSize,  (CHUNK_MAX+1) * sizeof(LongInt) );
     FreeMem( FPMemSize,  (CHUNK_MAX+1) * sizeof(LongInt) );
     FreeMem( FPMemPtr ,  (CHUNK_MAX+1) * sizeof(PChar  ) );
     FreeMem( FPChanged,  (CHUNK_MAX+1) * sizeof(Boolean) );
     FreeMem( FPCritSect, (CHUNK_MAX+1) * sizeof(TRTLCriticalSection) );

     inherited Destroy;
end;

end.
