// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cClientBase;

interface

uses Classes, ScktComp, uSSL; {MG}{SSL}

const
   csDISCONNECTED  = 0;
   csCONNECTING    = 1;
   csCONNECTED     = 2;
   csLOGIN         = 3;
   csREADY         = 4;
   csDISCONNECTING = 5;
   csERROR         = 99;

   SPECIAL_ACCEPTWEAKENDOFLIST  = $00000001; // SendCmndGetList;
   {JW} {remove workaround for Telda}
   // SPECIAL_ACCEPTNODOTENDOFLIST = $00000002; // SendCmndGetList;
   SPECIAL_ACCEPTSMTPMULTILINE  = $00000004; // SendCmnd;
   SPECIAL_ACCEPTPOP3MULTILINE  = $00000008; // SendCmnd; {JW} {SASL}

type
   TNewDataCallback = procedure(DataSize:integer) of object; {kms}
   TClientSocketBase = class( TClientSocket )
      protected
         FServer     : String;
         FPort       : String;
         {MG}{SSL}
         FSSLVerify  : Integer;
         FSSLCaFile  : String;
         SSL         : TSSLConnection;
         FSSLCiphers : String; {MG}{CipherList}
         {/SSL}
         RcvBufIn    : String;
         RcvBuffer   : TStringList;
         procedure HandleReceivedData( NewData: String );
         function  ReceiveData( TimeoutMS: Integer; CloseOnFail: Boolean ): Boolean;
         Procedure ConnectionStatistic (Const AInifile: String; Const ok: Boolean);
      public
         {JW}
         Serv_RemoteTimeoutConnect : Integer;
         Serv_RemoteTimeoutCommand : integer;
         {JW}
         Greeting    : String;
         ClientState : Integer;
         ResultCmnd  : String;
         ResultLine  : String;
         ResultList  : TStringList;

         function  ResultListFollows: Boolean; virtual; abstract;
         procedure SendData( Data: String ); virtual;
         procedure SendCmnd( Cmnd: String; Special: Integer; FilterLog: Boolean ); virtual;
         procedure SendCmndGetList( Cmnd: String; Special: Integer ); overload; virtual; {kms}
         procedure SendCmndGetList( Cmnd: String; Special: Integer ; NewDataCallback: TNewDataCallback); overload; virtual; {kms}

         {MG}{SSL}
         // function  Connect( AServer, APort: String ): Boolean;
         function  Connect( Const AServer, APort, AIniFile: String;
            Const AUseSSL: Boolean; Const ASSLVerify: Integer; Const ASSLCaFile: String ): Boolean;
         function  StartSSL: Boolean;
         function  NegotiateSSL: Boolean; virtual; abstract;
         {/SSL}
         function  Login( AUser, APass: String ): Boolean; virtual; abstract;
         procedure Disconnect;

         procedure MyOnLookup    ( Sender: TObject; Socket: TCustomWinSocket );
         procedure MyOnConnecting( Sender: TObject; Socket: TCustomWinSocket );
         procedure MyOnConnect   ( Sender: TObject; Socket: TCustomWinSocket );
         procedure MyOnDisconnect( Sender: TObject; Socket: TCustomWinSocket );
         procedure MyOnError     ( Sender: TObject; Socket: TCustomWinSocket;
                                   ErrorEvent: TErrorEvent;
                                   var ErrorCode: Integer );

         constructor Create(AOwner: TComponent); override;
         destructor  Destroy; override;
   end;

implementation

uses Windows, SysUtils, uTools, Global, uWinSock, cStdForm, IniFiles, Config,
     tBase, cLogfile;

procedure TClientSocketBase.HandleReceivedData( NewData: String );
const LogLimit=120;
var  j   : Integer;
     Line: String;
begin
   IncCounter( CntByteIn, length(NewData) );
   RcvBufIn := RcvBufIn + NewData;

   if (length(NewData)<=LogLimit) {or ((LOGFILEMASK and LOGID_FULL)<>0)} then begin
      Log( LOGID_DEBUG, Self.Classname + '.Received: ' + NewData )
   end else begin
      Log( LOGID_DEBUG, Self.Classname + '.Received: ' + copy(NewData,1,LogLimit-6) + ' [...]' );
   end;
   repeat
      j := Pos( #10, RcvBufIn );
      if j=0 then break;
      Line := copy( RcvBufIn, 1, j-1 );
      If j>0 then System.Delete( RcvBufIn, 1, j );
      while (Line<>'') and (Line[length(Line)]=#13) do System.Delete(Line,length(Line),1);
      RcvBuffer.Add( Line );
   until RcvBufIn='';
end;

function TClientSocketBase.ReceiveData( TimeoutMS: Integer; CloseOnFail: Boolean ): Boolean;
const BufSize = 1024;
var  SocketStream: TWinSocketStream;
     Data: String;
     ByteIn: Integer;
begin
   Result := False;

   SocketStream := TWinSocketStream.Create( Socket, TimeoutMS );
   try
      try
         while Assigned(Socket) and (Socket.Connected) do begin

            If AllShutDownReq or ThreadControl.Shutdown then break;
            if ClientState in [csERROR] then break;
            {MG}{SSL}
            if not ( ( Assigned(SSL) and SSL.HasPendingData ) or
               SocketStream.WaitForData( TimeoutMS ) ) then break;
            {/SSL}

            SetLength( Data, BufSize );
            FillChar( Data[1], BufSize, 0 );

            ByteIn := 0;
            try
               if Socket.Connected then begin
                  if SSL=NIL {MG}{SSL}
                     then ByteIn := SocketStream.Read( Data[1], BufSize )
                     else ByteIn := SSL.Read( PChar(Data), BufSize ); {MG}{SSL}
               end;
            except
               ByteIn := 0;
            end;
            if ByteIn = 0 then break;

            HandleReceivedData( copy( Data, 1, ByteIn ) );

            if RcvBuffer.Count>0 then begin
               Result := True;
               break;
            end;
         end;

      except
         on E:Exception do begin
            Log( LOGID_ERROR, TrGlF(kLog, 'ReceivingData.Error',
                   'Exception receiving data from %s:', [FServer]) );
            Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
         end;
      end;
   finally
      SocketStream.Free
   end;

   if not Result then begin
      Log( LOGID_WARN, TrGlF(kLog, 'ReceivingData.failed',
           'Receiving data from %s failed', [fServer]) );
      if not ClientState in [csDISCONNECTED,csDISCONNECTING] then ClientState:=csERROR;
      if CloseOnFail then begin
         if Assigned(Socket) then begin
            try if Socket.Connected then Socket.Close; except end;
         end;
      end;
   end;
end;

procedure TClientSocketBase.SendData( Data: String );
var  SocketStream: TWinSocketStream;
     snd: Integer;
begin
   SocketStream := TWinSocketStream.Create( Socket, Serv_RemoteTimeoutCommand*1000 );
   try
      try
         while Data>'' do begin
            if not( ClientState in [csREADY, csLOGIN] ) then exit;
            if SSL=NIL {MG}{SSL}
               then snd := SocketStream.Write( Data[1], length(Data) )
               else snd := SSL.Write( PChar(Data), length(Data) ); {MG}{SSL}
            if snd < 0 then break; {MG}{SSL}
            if snd>0 then begin
               IncCounter( CntByteOut, snd );
               System.Delete( Data, 1, snd );
            end;
         end
      except
         on E:Exception do begin
            ClientState:=csERROR;
            Log( LOGID_ERROR, TrGlF(kLog, 'SendingData.Error',
                     'Exception sending data to %s:', [FServer]) );
            Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
         end;
      end
   finally
      SocketStream.Free
   end
end;

procedure TClientSocketBase.SendCmnd( Cmnd : String; Special: Integer; FilterLog: Boolean );
var  CmndLog   : String;
     ReplyLine : String;
     EndOfReply: Boolean;
begin
   ResultCmnd := Cmnd;
   ResultLine := '';
   ResultList.Clear;

   if not( ClientState in [csREADY, csLOGIN] ) then exit;

   CmndLog := Cmnd;
   if FilterLog then begin
      CmndLog := '[...authorization...]';
      if copy(Cmnd,1, 5)='USER '          then CmndLog:=copy(Cmnd,1, 5)+'[...]';
      if copy(Cmnd,1, 5)='PASS '          then CmndLog:=copy(Cmnd,1, 5)+'[...]';
      if copy(Cmnd,1,14)='AUTHINFO USER ' then CmndLog:=copy(Cmnd,1,14)+'[...]';
      if copy(Cmnd,1,14)='AUTHINFO PASS ' then CmndLog:=copy(Cmnd,1,14)+'[...]';
   end;
   Log( LOGID_DETAIL, Self.Classname + ' > ' + CmndLog );

   // send command
   SendData( Cmnd + #13#10 );

   // receive reply
   EndOfReply := False;
   try
      while Assigned(Socket) and (Socket.Connected) do begin
         if not( ClientState in [csREADY, csLOGIN] ) then break;

         while RcvBuffer.Count>0 do begin
            ReplyLine := RcvBuffer[0];
            RcvBuffer.Delete(0);

            if (Special and SPECIAL_ACCEPTSMTPMULTILINE)<>0 then begin
               if ResultLine<>'' then ResultLine:=#13#10+ResultLine;
               ResultLine := ReplyLine + ResultLine; // note: reverse order
               if copy( ReplyLine, 4, 1 )<>'-' then begin
                  EndOfReply := True;
                  break;
               end
            end else
{JW} {SASL}
            If (Special and SPECIAL_ACCEPTPOP3MULTILINE)<>0 then begin
               ResultLine := ResultLine +ReplyLine ;
               if copy( ReplyLine, 1, 1 )='.' then begin
                  EndOfReply := True;
                  break;
               end;
{JW}
            end else begin
               ResultLine := ReplyLine;
               EndOfReply := True;
               break;
            end;
         end;
         if EndOfReply then break;

         if not ReceiveData( Serv_RemoteTimeoutCommand*1000, True ) then break;
      end;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, TrGlF(kLog, 'SendingCommand.Error',
               'Exception sending command to %s:', [FServer] ) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
      end;
   end;

   if not EndOfReply then begin
      if not ClientState in [csDISCONNECTED,csDISCONNECTING] then ClientState:=csERROR;
      if Assigned(Socket) then begin
         try if Socket.Connected then Socket.Close; except end;
      end;
   end else begin
      Log( LOGID_DETAIL, Self.ClassName + ' < ' + ResultLine );
   end;
end;

procedure TClientSocketBase.SendCmndGetList( Cmnd: String; Special: Integer ); {.kms}
var NewDataCallback: TNewDataCallback;
begin
  NewDataCallback:=nil;
  SendCmndGetList( Cmnd, Special, NewDataCallback);
end; {/kms}

procedure TClientSocketBase.SendCmndGetList( Cmnd: String; Special: Integer;
                 NewDataCallback: TNewDataCallback); {kms}
var  ListLine, s: String;
{JW} {remove workaround for Telda}
//     EndOfReply, CloseOnFail: Boolean;
     EndOfReply: Boolean;
     i: Integer;
     LoadedDataSize:Integer; {kms}
begin
   LoadedDataSize:=0; {kms}
   // send command
   SendCmnd( Cmnd, Special, False );
   if not( ClientState in [csREADY, csLOGIN] ) then exit;
   if not ResultListFollows then exit;

  {JW} {remove workaround for Telda}
   // handle special case:
   //    (former?^H) POP3-LIST at Telda.net (no final '.' on 0 msgs)
   // if (Special and SPECIAL_ACCEPTNODOTENDOFLIST)<>0 then begin
   //    if UpperCase(copy(ResultLine,1,6))='+OK 0 ' then begin
   //       if (RcvBuffer.Count=0) and (length(RcvBufIn)=0) then exit;
   //    end;
   // end;

   // receive list til end-of-list (CRLF "." CRLF) or timeout
   EndOfReply := False;
   {JW} {remove workaround for Telda}
   // CloseOnFail := ( (Special and SPECIAL_ACCEPTNODOTENDOFLIST) = 0 );
   try
      while Assigned(Socket) and (Socket.Connected) do begin
         if not( ClientState in [csREADY, csLOGIN] ) then break;

         while RcvBuffer.Count>0 do begin
            ListLine := RcvBuffer[0];
            RcvBuffer.Delete(0);

            if ListLine='.' then begin
               EndOfReply := True;
               break;
            end;

            // handle special case:
            //    (former?) POP3-LIST at Germanynet: end-of-list='  .CRLF  '
            //    (former?) POP3-LIST at uunet: #0...#0'.'
            if (Special and SPECIAL_ACCEPTWEAKENDOFLIST)<>0 then begin
               s := '';
               for i:=1 to length(ListLine) do begin
                  if ListLine[i] in [#33..#126] then s:=s+ListLine[i];
               end;
               if s='.' then begin
                  EndOfReply := True;
                  RcvBufIn := '';
                  RcvBuffer.Clear;
                  break;
               end;
            end;

            if copy(ListLine,1,2)='..' then System.Delete(ListLine,1,1);
            ResultList.Add( ListLine );
            inc(LoadedDataSize, length(ListLine)+2); {kms}
            if Assigned(NewDataCallback) then NewDataCallback(LoadedDataSize); 
         end;
         if EndOfReply then break;

         {JW} {remove workaround for Telda}
         // if not ReceiveData( Serv_RemoteTimeoutCommand*1000, CloseOnFail ) then break;
         if not ReceiveData( Serv_RemoteTimeoutCommand*1000, True) then break
      end;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, TrGlF(kLog, 'ReceivingList.Error',
               'Exception receiving list from %s:', [FServer] ) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
      end;
   end;

   // no (valid) list -> give up
   if not EndOfReply then begin
      if not ClientState in [csDISCONNECTED,csDISCONNECTING] then ClientState:=csERROR;
      if Assigned(Socket) then begin
         try if Socket.Connected then Socket.Close; except end;
      end;
   end;
end;

procedure TClientSocketBase.MyOnLookup(Sender: TObject; Socket: TCustomWinSocket);
begin
   Log( LOGID_DETAIL, Self.Classname + '.OnLookup from '
                                    +'Hamster to '
                                    + self.FServer+':'
                                    + self.FPort+' use '
                                    + self.host+':'
                                    + IntToStr(self.port));
   Log( LOGID_DEBUG, Self.Classname + '.OnLookup with '
                                    +'local socket '
                                    + self.socket.LocalHost);
   ClientState := csCONNECTING;
end;

procedure TClientSocketBase.MyOnConnecting(Sender: TObject; Socket: TCustomWinSocket);
begin
   Log( LOGID_DETAIL, Self.Classname + '.OnConnecting from '
                                    +'Hamster to '
                                    + self.FServer+':'
                                    + self.FPort+' use '
                                    + self.host+':'
                                    + IntToStr(self.port));
   Log( LOGID_DEBUG, Self.Classname + '.OnConnecting with '
                                    +'local socket '
                                    + self.socket.LocalHost);
   ClientState := csCONNECTING;
end;

procedure TClientSocketBase.MyOnConnect(Sender: TObject; Socket: TCustomWinSocket);
begin
   Log( LOGID_DETAIL, Self.Classname + '.OnConnect from '
                                    +'Hamster to '
                                    + self.FServer+':'
                                    + self.FPort+' use '
                                    + self.host+':'
                                    + IntToStr(self.port));
   If ((Logfile.ViewMask or Logfile.FileMask) and LOGID_DEBUG)>0 then begin
      Log( LOGID_DEBUG, Self.Classname + '.OnConnect with '
                                    +'local socket '
                                    + self.socket.LocalHost+':'
                                    + IntToStr(self.Socket.LocalPort)
                                    +' with IP-address '
                                    + self.socket.LocalAddress);
      Log( LOGID_DEBUG, Self.Classname + '.OnConnect with '
                                    + 'remote socket '
                                    + self.socket.RemoteHost+':'
                                    + IntToStr(self.Socket.RemotePort)
                                    +' with IP-address '
                                    + self.Socket.RemoteAddress)
   end;
   ClientState := csCONNECTED;
end;

procedure TClientSocketBase.MyOnDisconnect(Sender: TObject; Socket: TCustomWinSocket);
begin
   Log( LOGID_DEBUG, Self.Classname + '.OnDisconnect' );
   ClientState := csDISCONNECTING;
   if not(Active) then ClientState := csDISCONNECTED;
end;

procedure TClientSocketBase.MyOnError(Sender: TObject; Socket: TCustomWinSocket;
                          ErrorEvent: TErrorEvent; var ErrorCode: Integer);
begin
   Log( LOGID_WARN, Self.Classname + '.OnError '
                    + inttostr(ErrorCode) + ' ' + WinsockErrTxt( ErrorCode ) );
   ClientState := csERROR;
   try if Socket.Connected then Socket.Close except end;
   ErrorCode := 0;
end;

function  TClientSocketBase.Connect( Const AServer, APort, AIniFile: String;
   Const AUseSSL: Boolean; Const ASSLVerify: Integer; Const ASSLCaFile: String ): Boolean;
Var s: String; x: Integer;
begin
   Result  := False;
   FServer := AServer;
   FPort   := APort;

   {MG}{SSL}
   FSSLVerify := ASSLVerify;
   if ASSLCaFile <> ''
      then FSSLCaFile := ASSLCaFile
      else FSSLCaFile := CfgIni.ReadString( 'SSL', 'CaFile', '' );
   {/SSL}

   Serv_RemoteTimeoutConnect := Def_RemoteTimeoutConnect;
   Serv_RemoteTimeoutCommand := Def_RemoteTimeoutCommand;
   With TIniFile.Create( AInifile ) do try
      FSSLCiphers := ReadString( 'NNTP', 'SSLCiphers', Def_SSLCipherString ); {MG}{CipherList}
      s := ReadString ( 'Setup', 'Remote.Timeout.Connect', ReadString ( 'Setup', 'RemoteTimeoutConnect', ''));
      If s > '' then try Serv_RemoteTimeoutConnect := StrToInt(s) except end;
      s := ReadString ( 'Setup', 'Remote.Timeout.Command', ReadString ( 'Setup', 'RemoteTimeoutCommand', ''));
      If s > '' then try Serv_RemoteTimeoutCommand := StrToInt(s) except end;
   finally
      free
   end;

   Log( LOGID_DEBUG, Self.Classname + '.Connect ' + FServer + ' ' + FPort );
   if ClientState<>csDISCONNECTED then exit;

   if (AServer>='0') and (AServer<='99') then Address := AServer
                                         else Host    := AServer;
   If Not ConvertPort( APort, x ) then Exit;
   Port := x;
   try
      RcvBuffer.Clear;
      RcvBufIn := '';
      Open;

      if Assigned(Socket) and (Socket.Connected) then begin
         if AUseSSL and ( not StartSSL ) then begin
            ClientState := csERROR;
            raise Exception.Create( 'Error starting SSL' )
         end;
         while Assigned(Socket) and (Socket.Connected) do begin
            if not ReceiveData( Serv_RemoteTimeoutConnect*1000, True ) then break;

            if RcvBuffer.Count>0 then begin
               ResultLine := RcvBuffer[0];
               RcvBuffer.Delete(0);
               if RcvBuffer.Count>0 then begin
                  ResultLine := ResultLine + '[...]' + RcvBuffer[RcvBuffer.Count-1];
               end;
               RcvBuffer.Clear;
               RcvBufIn := '';
               Greeting := ResultLine;
               Result := True;
               break;
            end
         end
      end
   except
      on E:Exception do begin
         Log( LOGID_ERROR, TrGlF(kLog, 'Connecting.Error',
              'Exception connecting to %s:', [FServer] ) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
      end;
   end;

   If Not Result then begin
      Log( LOGID_WARN, TrGlF(kLog, 'Connection.Failed',
              'Connection to %s failed!', [FServer]) );
      if Assigned(Socket) then begin
         try if Socket.Connected then Socket.Close; except end;
      end;
      ClientState := csDISCONNECTED;
   end
end;

procedure TClientSocketBase.Disconnect;
var  cnt: Integer;
begin
   Log( LOGID_DEBUG, Self.Classname + '.Disconnect' );

   if ClientState in [csLOGIN, csREADY] then SendCmnd( 'QUIT', 0, False );

   if Assigned(SSL) then SSL.Shutdown; {MG}{SSL}

   if Active then begin
      Log( LOGID_DEBUG, TrGlF(kLog, 'SocketBase.Disconnect.DebugInfo',
         'Close: %s,%s:%s,%s', [Host, Address, Service, inttostr(Port) ] ) );
      Close;
      cnt := 0;
      while not ClientState in [csDISCONNECTED,csERROR] do begin
         Sleep(250);
         inc( cnt );
         if cnt>=20 then break;
      end;
   end;
end;

{MG}{SSL}
function TClientSocketBase.StartSSL: Boolean;
Var  Ctx: TSSLContext;
     Options: Integer;
const
     SSL_VERIFY_NONE = 0;
     SSL_VERIFY_PEER = 1;
     SSL_ACCEPT_ONLY_LOCAL_CERTS = 8;
begin
   Result := False;
   Log( LOGID_DEBUG, Self.Classname + '.StartSSL' );

   if not SSLReady then begin
      Log( LOGID_ERROR, TrGl(kLog, 'SSL.NotReady', 'OpenSSL libraries not ready') );
      exit;
   end;

   Options := SSL_VERIFY_NONE;
   if FSSLVerify > 0 then Options := SSL_VERIFY_PEER;
   if FSSLVerify = 3 then Options := Options or SSL_ACCEPT_ONLY_LOCAL_CERTS;

   // First, we create a new SSL context to set our options
   Ctx := TSSLContext.Create( False, Options, FSSLCaFile,
                              CfgIni.ReadString( 'SSL', 'CaPath', '' ) );
   if Ctx.Context = nil then begin
      FreeAndNil( Ctx );
      Log( LOGID_ERROR, TrGl(kLog, 'SSL.CtxNil',
           'Creating SSL context failed') );
      exit;
   end;
   Ctx.SetOptions( FSSLCiphers ); {MG}{CipherList}

   // Now we create a new SSL structure for data exchange
   // it inherits the settings of the underlying context
   SSL := TSSLConnection.Create( Ctx.Context );
   if SSL.Ssl = nil then begin
      FreeAndNil( SSL );
      Log( LOGID_ERROR, TrGl(kLog, 'SSL.SslNil',
           'Creating SSL connection structure failed') );
      exit;
   end;

   // We don't need the context any more
   Ctx.Destroy;

   // Do the TLS/SSL handshake
   if SSL.Connect( Socket.SocketHandle ) then begin
      SSL.Info;
      Result := True;
   end else begin
      FreeAndNil( SSL );
      Log( LOGID_WARN, TrGl(kLog, 'SSL.ConnectionFailed',
           'SSL connection failed') );
   end
end;
{/SSL}

constructor TClientSocketBase.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);

   Log( LOGID_DEBUG, Self.Classname + '.Create' );

   ClientType   := ctBlocking;
   OnLookup     := MyOnLookup;
   OnConnecting := MyOnConnecting;
   OnConnect    := MyOnConnect;
   OnDisconnect := MyOnDisconnect;
   OnError      := MyOnError;

   ClientState  := csDISCONNECTED;
   RcvBuffer    := TStringList.Create;
   RcvBufIn     := '';
   ResultList   := TStringList.Create;
   Greeting     := '';
   FServer      := '';
   FPort        := '';
   {MG}{SSL}
   FSSLVerify   := 0;
   FSSLCaFile   := '';
   SSL          := nil;
   {/SSL}
end;

destructor TClientSocketBase.Destroy;
begin
   Log( LOGID_DEBUG, Self.Classname + '.Destroy' );

   if Assigned(SSL) then SSL.Free; {MG}{SSL}
   try if Active then Close; except end;
   if Assigned(RcvBuffer ) then RcvBuffer.Free;
   if Assigned(ResultList) then ResultList.Free;

   inherited Destroy;
end;

procedure TClientSocketBase.ConnectionStatistic(const AInifile: String;
  const ok: Boolean);
Var x: integer;  
begin
   With TIniFile.Create( AIniFile ) do try
      If ok then begin
         try x := ReadInteger('Statistic', 'Connections.ok', 0) except x := 0 end;
         try WriteInteger('Statistic', 'Connections.ok', x+1) except end;
         try WriteString('Statistic', 'LastConnect.ok', FormatDateTime('yyyy-mm-dd hh:mm:ss', Now)) except end;
         try x := ReadInteger('Statistic', 'LastConnect.Count.ok', 0) except x := 0 end;
         try WriteInteger('Statistic', 'LastConnect.Count.ok', x+1) except end;
         try WriteInteger('Statistic', 'LastConnect.Count.failed', 0) except end;
      end else begin
         try x := ReadInteger('Statistic', 'Connections.failed', 0) except x := 0 end;
         try WriteInteger('Statistic', 'Connections.failed', x+1) except end;
         try WriteString('Statistic', 'LastConnect.failed', FormatDateTime('yyyy-mm-dd hh:mm:ss', Now)) except end;
         try x := ReadInteger('Statistic', 'LastConnect.Count.failed', 0) except x := 0 end;
         try WriteInteger('Statistic', 'LastConnect.Count.failed', x+1) except end;
         try WriteInteger('Statistic', 'LastConnect.Count.ok', 0) except end;
      end;
   finally free end
end;

end.
