// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cClientNNTP;

interface

{JW} {SSL}
uses Classes, Dialogs, cClientBase, cFiltersNews, Windows, uSSL; {MG}{SSL}
{JW}

type
   TClientSocketNNTP = class( TClientSocketBase )
      public
         ResultCode: Integer;
         function  ResultListFollows: Boolean; override;
         procedure SendCmnd( Cmnd : String; Special: Integer; FilterLog: Boolean ); override;
        {JW} {Mode Reader}
         function  LoginMode( AUser, APass: String; Mode:Boolean ): Boolean; virtual;
         function  Login( AUser, APass: String): Boolean; override;
         function  NegotiateSSL: Boolean; override; {MG}{SSL}
         {JW}
   end;

   TClientNNTP = class
      private
         Server, Port, User, Pass: String;
         ServerDir  : String;
         NNTP: TClientSocketNNTP;
         ScoreFile: TFiltersNews;
         ServerXOverList: array[0..10] of Integer;
         SSLMode, SSLVerifyLevel: Integer; SSLCaFile: String; {MG}{SSL}
      public
         procedure FeedNewsForGroup(GroupName:String; GroupHandle: Integer );
         function  State : Integer;
         procedure Connect(Mode:Boolean);
         procedure Disconnect;
         procedure GetServerInfos;
         procedure GetNewGroups;
         procedure GetListOfMIDs;
         procedure GetNewArticlesForGroup( GroupName: String; GroupHandle: Integer );
         function  PostArticle( PostFile, ArtText: String; var SrvResult: String ): Boolean;
         constructor Create( AServer, APort, AUser, APass: String; {MG}{SSL}
                             ASSLMode, ASSLVerifyLevel: Integer;
                             ASSLCaFile: String );
         destructor Destroy; override;
   end;


   tArtCurLoading = class (tStringList)     //HRR: List for Articles currently loading
    public
     constructor Create;
     function CanLoad(MsgId:String):Boolean;
     procedure ArtLoaded(MsgId:String);
   end;

Var
   ArtCurLoading : tArtCurLoading = nil;

implementation

uses SysUtils, FileCtrl, Global, Config, uTools, uCRC32, cArtFiles,
     IniFiles, cArticle, uDateTime, cStdForm, cPCRE, tBase, cLogFile,
     cActions;

{ TClientSocketNNTP }
     
procedure TClientSocketNNTP.SendCmnd( Cmnd : String; Special: Integer; FilterLog: Boolean );
begin
   inherited SendCmnd( Cmnd, Special, FilterLog );
   if ResultLine='' then ResultLine := '999 (timeout or connection lost)';
   ResultCode := strtoint( copy( ResultLine, 1, 3 ) );
end;

function TClientSocketNNTP.ResultListFollows: Boolean;
begin
   Result := ( ResultCode < 400 );
end;

{JW} {Mode Reader}
function TClientSocketNNTP.Login( AUser, APass : String ): Boolean;
begin
  result:=LoginMode( AUser, APass, True )
end;

{JW} {Mode Reader}
// function TClientSocketNNTP.Login( AUser, APass : String ): Boolean;
function TClientSocketNNTP.LoginMode( AUser, APass : String; Mode:Boolean ): Boolean;
begin
   Result := False;
   Log( LOGID_DEBUG, 'NntpClient.Login' );
   if ClientState<>csCONNECTED then exit;

   ClientState := csLOGIN;
{JW} {Mode Reader}
   if Mode and Def_NNTPClient_ModeReader  then
      SendCmnd( 'MODE READER', 0, False );
{JW}
   if (AUser<>'') and (APass<>'') then begin
      SendCmnd( 'AUTHINFO USER ' + AUser, 0, True );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode >= 400 then begin ClientState:=csERROR; exit; end;

      SendCmnd( 'AUTHINFO PASS ' + APass, 0, True );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode < 400 then ClientState := csREADY
                          else begin ClientState:=csERROR; exit; end;
   end;
{JW} {Feed}
//   SendCmnd( 'MODE READER', 0, False );
{JW}
   if ClientState in [csERROR, csDISCONNECTED] then exit;
   if ResultCode < 400 then ClientState := csREADY;

   if ClientState=csREADY then Result:=True;
end;

{MG}{SSL}
// As there is not yet an official standard for "NNTP over TLS", the STARTTLS
// command is only compatible with the INN news server (V 2.3.0 and higher)
function TClientSocketNNTP.NegotiateSSL : Boolean;
begin
     Result := False;
     if ClientState<>csCONNECTED then exit;
     ClientState := csLOGIN;
     SendCmnd( 'STARTTLS', 0, False );
     if ResultCode = 382 then begin
        Log( LOGID_DEBUG, Format(TrGl(kLog, 'SSL.Handshake',
             '[%s] starting SSL/TLS handshake ...'), [FServer]) );
        Result := StartSSL;
     end else
        Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.ProtoNotSupported',
             '[%s] Server does not support %s over SSL'), [FServer, 'NNTP']) );
end;
{/SSL}

{ TClientNNTP }

function TClientNNTP.State : Integer;
begin
   if Assigned( NNTP ) then Result := NNTP.ClientState
                       else Result := csDISCONNECTED;
end;

{JW} {Mode Reader}
procedure TClientNNTP.Connect (Mode: Boolean);
Var s: String;
    ConnectWithSSL: Boolean; {MG}{SSL}
    ok: Boolean;
begin
   if Assigned( NNTP ) then Disconnect;
   NNTP := TClientSocketNNTP.Create( nil );
   {MG}{SSL}
   ConnectWithSSL := SSLMode = 1;
   ok := false;
   if NNTP.Connect( Server, Port, PATH_SERVER + Server + '\' + SRVFILE_INI,
                    ConnectWithSSL, SSLVerifyLevel, SSLCaFile )
   then begin
      if State = csCONNECTED then begin
         if SSLMode = 2 then NNTP.NegotiateSSL;
         if SSLMode = 3 then begin
            if not NNTP.NegotiateSSL then begin
               Disconnect; EXIT
            end
         end;
         if State = csLOGIN then NNTP.ClientState := csCONNECTED;
         {/SSL}
         {Mode Reader}
         If NNTP.LoginMode( User, Pass, Mode ) then begin
            ok := true
         end else begin
            if strtoint(copy( NNTP.Greeting, 1, 3 ) ) > 201 then begin {MMG} {NNTP-Greeting auswerten}
               s := NNTP.Greeting;
               Log( LOGID_WARN, TrGlF (kLog, 'Login.failed', 'Login failed %s', s ));
            end else begin
               s := NNTP.ResultLine;
               if s<>'' then s:=' ("'+s+'")';
               Log( LOGID_WARN, TrGlF (kLog, 'Login.failed', 'Login failed %s', [s] ))
            end
         end;
         if (NNTP.Greeting > '') and DirExists2(PATH_SERVER + Server) then begin
            HamFileRewriteLine( PATH_SERVER + Server + '\' +
                                SRVFILE_GREETING, NNTP.Greeting );
         end
      end;
   end;
   NNTP.ConnectionStatistic ( PATH_SERVER + Server + '\' + SRVFILE_INI, ok)
end;

procedure TClientNNTP.Disconnect;
begin
   if not Assigned( NNTP ) then exit;
   try NNTP.Disconnect except end;
   FreeAndNil (NNTP)
end;

procedure TClientNNTP.GetServerInfos;
var  SrvFile: String;
     ReloadGroups, ReloadDescs: Boolean;

   procedure GetList( Cmnd, Filename: String );
   begin
      if State in [csERROR, csDISCONNECTED] then exit;
      Log( LOGID_DETAIL, TrGlF(kLog, 'NNTP.LoadingCommand',
         '[%s] Loading <%s> ...', [Server, Cmnd] ) );
      NNTP.SendCmndGetList( Cmnd, 0 );
      if State in [csERROR, csDISCONNECTED] then exit;
      if NNTP.ResultList.Count=0 then begin
         if Cmnd='LIST' then begin
            if NNTP.ResultCode=215 then ///ff.
               Log( LOGID_WARN, TrGl(kLog, 'NNTP.NoNGsAvailableOnLIST', 'No newsgroups available on "LIST"') )
            else
               Log( LOGID_WARN, TrGlF(kLog, 'NNTP.CouldntLoad', 'Couldn''t load "%s" (%s)',
                   [Cmnd, NNTP.ResultLine] ));
            exit;
         end else begin
            NNTP.ResultList.Insert( 0, '# Command: ' + NNTP.ResultCmnd );
            NNTP.ResultList.Insert( 1, '# Result : ' + NNTP.ResultLine );
            Log( LOGID_DEBUG, TrGlF(kLog, 'NNTP.CouldntLoad', 'Couldn''t load "%s" (%s)',
                [Cmnd, NNTP.ResultLine] ) )
         end;
      end;

      try
         NNTP.ResultList.SaveToFile( Filename );
      except
         on E: Exception do begin
            Log( LOGID_ERROR, TrGlF(kLog, 'NNTP.CouldntSave.Error',
                  'Couldn''t save "%s" to "%s"', [Cmnd, FileName] ) );
            Log( LOGID_ERROR, TrGlF(kGlobal, 'ERROR_msg', 'ERROR: %s', E.Message ));
         end;
      end;
   end;

begin
   SrvFile := ServerDir + SRVFILE_HELPTEXT;
   if not FileExists2( SrvFile ) then GetList( 'HELP', SrvFile );

   SrvFile := ServerDir + SRVFILE_OVERVIEWFMT;
   if not FileExists2( SrvFile ) then GetList( 'LIST OVERVIEW.FMT', SrvFile );

   With TIniFile.Create( ServerDir + SRVFILE_INI ) do try
      ReloadGroups := ReadBool( 'Pull', 'ReloadGroups', False );
      ReloadDescs  := ReadBool( 'Pull', 'ReloadDescs',  False );
   finally
      Free
   end;
   if not FileExists2( ServerDir + SRVFILE_GROUPS ) then ReloadGroups:=True;

   if ReloadGroups then begin
      // load complete list of available groups
      SrvFile := ServerDir + SRVFILE_GROUPS;
      GetList( 'LIST', SrvFile );
      With TIniFile.Create( ServerDir + SRVFILE_INI ) do try
         if Uppercase(ReadString( 'Newgroups', 'LastGMT', '' ))<>'NEVER' then begin //HRR
           WriteString( 'Newgroups', 'LastGMT', DateTimeToTimeStamp(NowGMT) );
         end;                                                                              //HRR
         WriteBool  ( 'Pull', 'ReloadGroups', False );
         WriteString( 'Pull', 'ReloadGroupsLast', DateTimeToTimeStamp(Now) )
      finally
         Free
      end;
      GlobalListMarker( glTODO );
   end else begin
      // load new groups only
      GetNewGroups;
   end;

   if ReloadDescs then begin
      SrvFile := ServerDir + SRVFILE_GRPDESCS;
      GetList( 'LIST NEWSGROUPS', SrvFile );
      With TIniFile.Create( ServerDir + SRVFILE_INI ) do try
         WriteBool  ( 'Pull', 'ReloadDescs', False );
         WriteString( 'Pull', 'ReloadDescsLast', DateTimeToTimeStamp(Now) )
      finally
         free
      end;
      GlobalListMarker( glTODO );
   end;
end;

procedure TClientNNTP.GetNewGroups;
var  LastGMT : TDateTime;
     s, cmd, NewGroups, GrpName, GrpDesc: String;
     i, GrpCurr, DscCurr, LoadDescs: Integer;
     sl, slD : TStringList;
     RegExp: TPCRE;
     b, TestRegExp: Boolean;
begin
   if not( State in [csREADY, csLOGIN] ) then exit;

   if DateTimeToTimeStamp(NowGMT)<'19990906' then begin
      Log( LOGID_WARN, TrGlF(kLog, 'NNTP.GetNewGroups.InvalidPCDate',
          'Get new groups on %s - invalid pc-date: %s',
          [Server, DateTimeToTimeStamp(NowGMT)] ) );
      exit;
   end;

   sl  := TStringList.Create;
   slD := TStringList.Create;
   With TIniFile.Create( ServerDir + SRVFILE_INI ) do try
      s := ReadString( 'Newgroups', 'LastGMT', '' );
      if Uppercase(s)='NEVER' then exit;
      if s<>'' then LastGMT := TimeStampToDateTime( s )
               else LastGMT := NowGMT;

      If LastGMT > NowGMT then begin
         Log( LOGID_WARN, TrGlF(kLog, 'NNTP.GetNewGroups.Error.LastAccessAfterNow',
             'Get new groups on %s - invalid date assumed, last access after today: Last=%s Now=%s',
             [Server, DateTimeToTimeStamp(LastGMT), DateTimeToTimeStamp(NowGMT)] ) );
         exit;
      end;

      If NowGMT-LastGMT > 100 {days} then begin
         Log( LOGID_WARN, TrGlF(kLog, 'NNTP.GetNewGroups.Error.LastAccessTooLongAgo',
             'Get new groups on %s - invalid date assumed, last access %s days ago: Last=%s Now=%s',
             [Server, IntToStr(Trunc(NowGMT-LastGMT)), DateTimeToTimeStamp(LastGMT), DateTimeToTimeStamp(NowGMT)] ) );
         exit;
      end;

      // Load descriptions of new groups? (0=auto-detect, 1=always, 2=never)
      LoadDescs := ReadInteger( 'Newgroups', 'LoadDescs', 0 );

      If ReadInteger( 'Newgroups', '4-Digit-Years', 0 )=1 then begin
         s := FormatDateTime( 'yyyymmdd hhnnss', LastGMT ) + ' GMT';
      end else begin
         s := FormatDateTime( 'yymmdd hhnnss',   LastGMT ) + ' GMT';
      end;
      LastGMT := NowGMT;
      cmd := 'NEWGROUPS ' + s;
      NNTP.SendCmndGetList( cmd, 0 );

      if NNTP.ResultCode < 400 then begin

         WriteString( 'Newgroups', 'LastGMT', DateTimeToTimeStamp(LastGMT) );

         if NNTP.ResultList.Count>0 then begin
            NewGroups := NNTP.ResultList.Text;

            // add new groups to group-list (no "dupe"-check!)
            sl.Text := NewGroups;
            HamFileAppendList( ServerDir + SRVFILE_GROUPS, sl );
            GlobalListMarker( glTODO );

            // load descriptions of new groups
            sl.Text := NewGroups;
            slD.Clear;
            if LoadDescs <> 2 then begin
               for GrpCurr:=0 to sl.Count-1 do begin
                  GrpName := sl[GrpCurr];
                  i := PosWhSpace( GrpName );
                  if i>0 then GrpName:=copy(GrpName,1,i-1);

                  NNTP.SendCmndGetList( 'LIST NEWSGROUPS ' + GrpName, 0 );
                  if not( State in [csREADY, csLOGIN] ) then break;
                  If AllShutDownReq or ThreadControl.Shutdown then break;
                  if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin // Timeout
                     Log( LOGID_WARN, TrGlF(kLog, 'Warning.PullCancelled.NoReplyForListNGs',
                        'Pull cancelled, no reply for LIST NEWSGROUPS (%s)', [Server] ));
                     try Disconnect except end;
                     break;
                  end;

                  GrpDesc := '';
                  if (NNTP.ResultCode<400) and (NNTP.ResultList.Count>0) then begin
                     if (LoadDescs=0) and (NNTP.ResultList.Count>10) then begin
                        // Server returned a large list, so it looks like the server
                        // does not support LIST NEWSGROUPS with a groupname pattern.
                        // Disable immediate loading of descriptions to prevent from
                        // further trials:
                        LoadDescs := 2; // never
                        WriteInteger( 'Newgroups', 'LoadDescs', LoadDescs );
                        break;
                     end;
                     for DscCurr:=0 to NNTP.ResultList.Count-1 do begin
                        s := NNTP.ResultList[DscCurr];
                        i := PosWhSpace( s );
                        if i>0 then begin
                           if AnsiCompareText( GrpName, copy(s,1,i-1) )=0 then begin
                              GrpDesc := TrimWhSpace( copy( s, i+1, Length(s)-i ) );
                              break;
                           end;
                        end;
                     end
                  end;
                  slD.Add( GrpDesc )
               end
            end;

            // post note to hamster.misc
            sl.Text := NewGroups;
            s := '';
            RegExp := TPCRE.Create (true, 0);
            try
               If Def_FilterNewGroupsInfo>'' then begin
                  try
                     RegExp.Compile (PChar(Def_FilterNewGroupsInfo));
                     TestRegExp := true
                  except
                     TestRegexp := false
                  end;
               end else begin
                  TestRegexp := false
               end;
               for i:=0 to sl.Count-1 do begin
                  If TestRegexp then begin
                     RegExp.Compile (PChar(Def_FilterNewGroupsInfo));
                     b := RegExp.Exec (PChar(sl[i]), -1)
                  end else b := true;
                  If b then begin
                     s := s + sl[i] + #13#10;
                     GrpDesc := '';
                     if i<slD.Count then begin
                        if slD[i]<>'' then GrpDesc:=slD[i];
                     end;
                     if GrpDesc='' then GrpDesc:='?';
                     s := s + ' ' + #9 + GrpDesc + #13#10;
                     //ShowMessage('Meldung ber '+sl[i]+' kommt nun!')
                  end else begin
                     //ShowMessage('Keine Meldung ber: '+sl[i])
                  end
               end
            finally
               RegExp.free
            end;
            If s > '' then begin
               s := s + #13#10
                      + '-- ' + #13#10
                      + 'Result of "' + cmd + '"' + #13#10
                      + 'Sent at ' + FormatDateTime( 'yymmdd hhnnss', LastGMT ) + ' GMT' + #13#10;
               SaveInInternalGroup( INTERNALGROUP_NEWGROUPS,
                  TrGlF(kLog, 'NNTP.NewGroupsInfo.HeaderSubject', '[%s] New groups', Server),
                  s )
            end;

            // save new groups in global list of groups (may cause duplicates,
            // which will be fixed when rebuilding global lists)
            sl.Text := NewGroups;
            s := '';
            for GrpCurr:=0 to sl.Count-1 do begin
               GrpName := sl[GrpCurr];
               i := PosWhSpace( GrpName );
               if i>0 then GrpName:=copy(GrpName,1,i-1);
               if GrpCurr<slD.Count then GrpDesc:=slD[GrpCurr]
                                    else GrpDesc:='?';
               if GrpDesc='' then GrpDesc:='?';
               s := s + GrpName + #9 + GrpDesc + #13#10;
            end;
            sl.Text := s;
            HamFileAppendList ( ServerDir + SRVFILE_GRPDESCS, sl );
            EnterCriticalSection( CS_MAINTENANCE );
            HamFileAppendList ( PATH_SERVER + SRVFILE_ALLDESCS, sl );
            GlobalListMarker( glTODO );
            LeaveCriticalSection( CS_MAINTENANCE );
         end;

      end;
   finally
      Free;
      sl.Free; slD.free
   end
end;

procedure TClientNNTP.GetListOfMIDs;
var  MessageID    : String;
     MidList      : TStringList;
     LfdArticle   : TArticle;
     MidListLfd   : Integer;
     IgnoreHistory: Boolean;
begin
   if not FileExists2( ServerDir + SRVFILE_GETMIDLIST ) then exit;

   MidList := TStringList.Create;
   MidList.LoadFromFile( ServerDir + SRVFILE_GETMIDLIST );
   DeleteFile( ServerDir + SRVFILE_GETMIDLIST );

   LfdArticle := TArticle.Create;

   try
      for MidListLfd:=MidList.Count-1 downto 0 do begin
         If AllShutDownReq or ThreadControl.Shutdown then break;
         if State in [csERROR, csDISCONNECTED] then break;

         MessageID     := MidList[MidListLfd];
         IgnoreHistory := False;
         if copy(MessageID,1,1)='!' then begin
            IgnoreHistory := True;
            System.Delete( MessageID, 1, 1 );
         end;

         if (MessageID<>'') and not IgnoreHistory then begin
            if NewsHistory.ContainsMID(MessageID) then begin
               Log( LOGID_Debug, TrGlF(kLog, 'NNTP.GetByMID.AlreadyInHistory',
                  '[%s] GetByMID %s: already in history', [Server, MessageID] ) );
               MidList.Delete( MidListLfd );
               MessageID:='';
            end;
         end;

         if MessageID<>'' then begin
            NNTP.SendCmndGetList( 'ARTICLE ' + MessageID, 0 );
            if State in [csERROR, csDISCONNECTED] then break;
            Log( LOGID_Debug, TrGlF(kLog, 'NNTP.GetByMid.Result',
               '[%s] GetByMID %s: %s', [Server, MessageID, NNTP.ResultLine] ) );

            if NNTP.ResultCode < 400 then begin
               IncCounter( CntArtNew , 1 );
               LfdArticle.Text := NNTP.ResultList.Text;
               if SaveArticle( LfdArticle, '', -1,
                               '(none-getbymid)', IgnoreHistory, '' ) then begin
                  IncCounter( CntArtLoad, 1 );
                  IncCounter( CntArtByMID, 1 ); //HSR //GetByMid-Counter
               end else begin
                  IncCounter( CntArtHist, 1 );
               end;
            end else begin
               if NNTP.ResultCode>=500 then break;
            end;
            MidList.Delete( MidListLfd );

         end;
      end;

   finally
      if MidList.Count>0 then begin
         HamFileAppendList ( ServerDir + SRVFILE_GETMIDLIST, MidList );
      end;
      MidList.Free;
      LfdArticle.Free;
   end;
end;

{JW} {Feed}
procedure TClientNNTP.FeedNewsForGroup(GroupName:String; GroupHandle: Integer );
var
   WeHaveFrom,WeHaveTo,i,j:Integer;
   FullArticle,Info,MessageID,LineStr,Path:String;
   Article : TArticle;
   boDONE  : boolean;
   TS      : TStringList;
begin
   Article := TArticle.Create;
   TS := TStringList.Create;
   try
      Info := '[' + Server + ', ' + GroupName + '] ';
      Log( LOGID_Detail, Info + TrGl(kLog, 'Feed.newArticles', 'Feed new articles' ) );
      // read pointer for allready send article
      WeHaveFrom:=ArticleBase.FeederLast[GroupHandle,Server] + 1;
      // check if exist expired article
      If WeHaveFrom < ArticleBase.LocalMin[GroupHandle] then begin
         WeHaveFrom:=ArticleBase.LocalMin[GroupHandle]
      end;
      // read local article pointer
      WeHaveTo :=ArticleBase.LocalMax[GroupHandle];
      // check if exist article to send else exit function
      If WeHaveFrom > WeHaveTo then begin
         Log( LOGID_Detail, Info + TrGl(kLog, 'Feed.NoNewArticles', 'No NewArticles to feed'));
         exit;
      end;
      // loop send all unsend article
      for i:=WeHaveFrom to WeHaveTo do begin
         Log( LOGID_DETAIL, Info + TrGlF(kLog, 'Feed.NewArticleNo', 'Feed new article #%s', IntToStr(i) ) );
         FullArticle := ArticleBase.ReadArticle( GroupHandle, i );
         if FullArticle > '' then begin
            Article.Text := FullArticle;
            EnterCriticalSection(CS_LOCK_InFeed);
            try
               // check path to prevent refeeding
               Path:=Article['Path:'];
               boDone:=False;
               RE_Split( Path, '!', 0, -1, TS );
               for j:=0 to TS.Count-1 do begin
                  if AnsiCompareText( TS[j], Server )=0 then begin
                  Log( LOGID_DETAIL, TrGlF(kLog, 'Detail.Peer.Refeed',
                       'Article from this server received: %s', [MessageID]));
                     boDONE := True;
                     break;
                  end;
               end;
               // read Message-ID for IHAVE question
               MessageID:=Article['Message-ID:'];
               if (MessageID<>'') and not boDone then begin
                  if AllShutDownReq or ThreadControl.Shutdown then exit;
                  if not( State in [csREADY, csLOGIN] ) then exit;
                  NNTP.SendCmnd( 'IHAVE ' + MessageID, 0, False );
                  if (NNTP.ResultCode=480) then begin
                    Log( LOGID_WARN, TrGlF(kLog, 'Warning.Peer.NotAllowed',
                      'Article %s could not be sent: "%s"', [MessageID, NNTP.ResultLine]));
                    // if user has not permission for feed the peer then exit
                    try Disconnect except end;
                    exit;
                  end;
                  if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin // Timeout
                     // Timeout
                     Log( LOGID_WARN, TrGlF(kLog, 'Warning.FeedCancelled.NoReplyForIHave',
                         'Feed cancelled, no reply for IHAVE (%s)', Server));
                     LineStr := ' ('+ TrGLF(kLog, 'NNTP.ArtNo_GroupName.Info',
                        'ArtNo=%s, Group=%s', [IntToStr(i), GroupName]) + ')';
                     Log( LOGID_WARN, '[' + Server + '] "' + NNTP.ResultCmnd + '"'
                                        + ' -> "' + NNTP.ResultLine + '"' + LineStr );
                     try Disconnect except end;
                     exit;
                  end;
                  // increment send pointer only if not permission or other
                  // not classified error
                  ArticleBase.FeederLast[GroupHandle,Server]:=i;
                  if (NNTP.ResultCode=436) or (NNTP.ResultCode=437) then begin
                     Log( LOGID_WARN, TrGlF(kLog, 'Warning.Feed.Failed',
                          'Feed failed: %s', MessageID));
                     if (NNTP.ResultCode=437)
                         then LineStr := ' ('+ TrGLF(kLog, 'NNTP.ArtNo_GroupName.Info',
                                   'ArtNo=%s, Group=%s', [IntToStr(i), GroupName]) + ')'
                         else LineStr := '';
                     Log( LOGID_WARN, '[' + Server + '] "' + NNTP.ResultCmnd + '"'
                                        + ' -> "' + NNTP.ResultLine + '"' +   LineStr );
                  end;
                  // peer not wanted the article
                  if (NNTP.ResultCode=435) then begin
                     Log( LOGID_DETAIL, TrGlF(kLog, 'Detail.Peer.NoWanted',
                       'Article no wanted: %s', MessageID));
                  end;
                  // peer wanted the article
                  if NNTP.ResultCode=335 then begin
                     Log( LOGID_INFO, Info + TrGlF(kLog, 'Feed.NewArticle.MID',
                       'Feed new article %s', MessageId ) );
                     Log( LOGID_INFO, Info + TrGlF(kLog, 'Feed.NewArticle.No+Group',
                         'Feed new article #%s %s', [IntToStr(i), GroupName] ) );
                     TS.text := FullArticle;
                     for J:=0 to ts.Count-1 do begin
                        LineStr := ts[j];
                        if LineStr>'' then If LineStr[1]='.' then LineStr:='.'+LineStr;
                        NNTP.SendData( LineStr + #13#10 );
                        if not( State in [csREADY, csLOGIN] ) then break;
                     end;
                     // send finish token for transfer article
                     NNTP.SendCmnd( '.', 0, False );
                     //  check if article transfer failed
                     if (NNTP.ResultCode=436) or (NNTP.ResultCode=437) then begin
                        Log( LOGID_WARN, TrGlF(kLog, 'Warning.Feed.Failed',
                              'Feed failed: %s', MessageID));
                        if (NNTP.ResultCode=437)
                           then LineStr := ' ('+ TrGLF(kLog, 'NNTP.ArtNoGroupName.Info',
                                         'ArtNo=%s, Group=%s', [IntToStr(i), GroupName]) + ')'
                           else LineStr := '';
                        Log( LOGID_WARN, '[' + Server + '] "'   + NNTP.ResultCmnd + '"'
                                        + ' -> "' + NNTP.ResultLine + '"' + LineStr );
                     end;
                  end;
               end;
            finally
              LeaveCriticalSection(CS_LOCK_InFeed);
            end;
         end;
      end;
      // set send pointer only if feeding terminate normal
      ArticleBase.FeederLast[GroupHandle,Server]:=WeHaveTo;
   finally
      Article.Free;
      TS.Free
   end
end;
{JW}

procedure TClientNNTP.GetNewArticlesForGroup( GroupName: String;
                                              GroupHandle: Integer );
const
  MaxTries       : integer = 8;   //HRR: how often try to load an article which is
                                  //currently loading by an other thread
  MidPos         : integer = 4;   // position of Message-ID in overview string
  ParseChar      : Char    = #9;  // separator char in overview string

  function SelectGroup (GroupName, Info: String) : Boolean;
  //--------------------------------------------------------
  var  onContinue : Boolean;
  begin
     NNTP.SendCmnd( 'GROUP ' + GroupName, 0, False );
     onContinue := ((State in [csREADY, csLOGIN]) and (not (AllShutDownReq or ThreadControl.Shutdown)));
     if onContinue then begin
        if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin // Timeout
           Log( LOGID_WARN, TrGlF(kLog, 'Warning.PullCancelled.NoReplyForGroup',
              'Pull cancelled, no reply for GROUP (%s)', Server));
           try Disconnect except end;
           onContinue := False;
        end;
        if onContinue then begin
           if NNTP.ResultCode >= 400 then begin
              Log( LOGID_WARN, TrGlF(kLog, 'Warning.Pull.Failed',
                   'Pull failed: %s', Groupname));
              Log( LOGID_WARN, '[' + Server + '] "'   + NNTP.ResultCmnd + '"'
                                            + ' -> "' + NNTP.ResultLine + '"' );
              SaveInInternalGroup( INTERNALGROUP_PULLERRORS,
                 Info + TrGl(kLog, 'NNTP.PullErrorInfo.HeaderSubject', 'Pull error!'),
                 TrGlF(kLog, 'NNTP.PullErrorInfo.Body', 'GROUP-reply was: ^m%s', NNTP.ResultLine ) );
              onContinue := False;
           end;
        end;
     end;
     Result := onContinue;
  end;

  function InitGroupInfo (out SrvHasMin, SrvHasMax : Integer;
  //----------------------------------------------------------
                              GroupName, Info: String) : Boolean;
  var  onContinue : Boolean; // replaces 'SeemsOk'
       Parser     : TParser;
       i          : Integer;
       s          : string;
  begin
     onContinue := True;
     Parser := TParser.Create;
     Parser.Parse( NNTP.ResultLine, ' ' ); // 0:err 1:count 2:first 3:last 4:name
     i         := Parser.iPart( 0, -1 );  if         i<0 then onContinue := False;
     i         := Parser.iPart( 1, -1 );  if         i<0 then onContinue := False;

//     SrvHasMin := Parser.iPart( 2, -1 );  if SrvHasMin<0 then onContinue := False;
//     SrvHasMax := Parser.iPart( 3, -1 );  if SrvHasMax<0 then onContinue := False;
     SrvHasMin := Parser.iPart( 2, -2 );  if SrvHasMin<=-2 then onContinue := False; // allow -1 as valid value //JAWO //XOver420
     SrvHasMax := Parser.iPart( 3, -2 );  if SrvHasMax<=-2 then onContinue := False; // allow -1 as valid value //JAWO //XOver420

     Parser.Free;
     if onContinue then begin
        s := 'Old .Min/.Max: ' + inttostr(ArticleBase.ServerMin[GroupHandle,Server])
                         + '-' + inttostr(ArticleBase.ServerMax[GroupHandle,Server])
                 + '  Limit: ' + inttostr(ArticleBase.PullLimit[GroupHandle,Server]);
        Log( LOGID_DEBUG, Info + s );
        s := 'Srv .Min/.Max: ' + inttostr(SrvHasMin)
                         + '-' + inttostr(SrvHasMax);
        Log( LOGID_DEBUG, Info + s );
        if (i=0) then onContinue := False; // no articles at all, so break //JAWO //XOver420
     end else begin
        Log( LOGID_WARN, TrGlF (kLog, 'Warning.Pull.Cancelled',
             'Pull cancelled:  %s', Groupname));
        Log( LOGID_WARN, TrGlF(kLog, 'Warning.Pull.InvGroupReply',
             'Invalid GROUP-reply assumed:  "%s"', NNTP.ResultLine));
        SaveInInternalGroup( INTERNALGROUP_PULLERRORS,
           Info + TrGl(kLog, 'NNTP.PullErrorInfo.HeaderSubject', 'Pull error!'),
           TrGlF(kLog, 'NNTP.PullErrorInfo.Body_InvalidGroupReply', 'Invalid GROUP-reply assumed: ^m%s', NNTP.ResultLine ) );
        try Disconnect except end;
        // onContinue := False;
     end;
     Result := onContinue;
  end;

  function GetRanges (out WeWantMin, WeWantMax, SrvHasMin, SrvHasMax : Integer;
  //----------------------------------------------------------------------------
                          GroupHandle: Integer; GroupName, Info: String) : Boolean;
  var  onContinue : Boolean;
       i          : Integer;
       s          : string;
  begin                                        // (initiates: SrvHasMin, SrvHasMax)
     onContinue := InitGroupInfo (SrvHasMin, SrvHasMax, GroupName, Info);
     if onContinue then begin      // find out artno.-range to load
        WeWantMin := ArticleBase.ServerMin[GroupHandle,Server] + 1; // default: from (last loaded + 1)
        WeWantMax := SrvHasMax;                                     // default: to (current max.)
        if WeWantMin<SrvHasMin then WeWantMin:=SrvHasMin; // adjust for expire-gap or new group
        if WeWantMin>SrvHasMax+100 then begin // server restarted with new numbers?
           s := 'Old('     + inttostr(ArticleBase.ServerMin[GroupHandle,Server])
                 + '-'     + inttostr(ArticleBase.ServerMax[GroupHandle,Server])
              + ') Limit(' + inttostr(ArticleBase.PullLimit[GroupHandle,Server])
              + ') New('   + inttostr(SrvHasMin) + '-' + inttostr(SrvHasMax) + ')';
           Log( LOGID_DEBUG, Info + TrGlF(kLog, 'NNTP.Renumber_triggered', 'Renumber triggered: %s', s ) );
           WeWantMin := SrvHasMin;
           WeWantMax := SrvHasMax;
        end;
        i := ArticleBase.PullLimit[GroupHandle,Server]; // adjust for pull-limit per session
        if (i<>0) and ( (WeWantMax-WeWantMin)>abs(i) ) then begin
           if i>0 then WeWantMin := WeWantMax - i + 1  // only (limit) newest
                  else WeWantMax := WeWantMin - i - 1; // only (limit) oldest
        end;
        if WeWantMin<SrvHasMin then WeWantMin:=SrvHasMin;
        if WeWantMax>SrvHasMax then WeWantMax:=SrvHasMax;
        ArticleBase.ServerMin[GroupHandle,Server] := WeWantMin - 1; // save first art-no we have (maybe ficticious)
        ArticleBase.ServerMax[GroupHandle,Server] := SrvHasMax;     // save last available art-no. (for info only)
        ArticleBase.ServerLow[GroupHandle,Server] := SrvHasMin;     //save lowest available art-no. (for info only) //JAWO 12.01.2001 (lowest server artno)
        {JAWO} {XOver420}
        if (WeWantMin > WeWantMax) then begin
          onContinue := False;   // no new articles
        {/JAWO}
        end;
     end;
     Result := onContinue;
  end;

  function  GetOverview (var OverView: TStringList; var ByNumOnly: Boolean;
  //------------------------------------------------------------------------
                             LetsGetMin, LetsGetMax: Integer) : Boolean;
  var  onContinue : Boolean;
       OvrLfd     : integer;
       s: String;
  begin
     {JW} {Overview size}
     s := 'XOVER ' + inttostr(LetsGetMin) + '-' + inttostr(LetsGetMax);
     try
        NNTP.SendCmndGetList( s, 0 );
     except
        Log( LOGID_WARN, 'Overview failed by '+s)
     end;
     {JW}
     // XOVER: 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References
     //              6:Bytes 7:Lines [8:Xref full]
     // NNTP.SendCmndGetList( 'XHDR Message-ID '+i2s(GetMin)+'-'+i2s(GetMax) );
     // XHDR:  0:No. 1:(Message-ID|'(none)')
     // MidPos:=1; ParseChar:=' ';
     onContinue := ((State in [csREADY, csLOGIN]) and (not (AllShutDownReq or ThreadControl.Shutdown)));
     if onContinue then begin
        if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin // Timeout
           Log( LOGID_WARN, TrGlF(kLog, 'Warning.Pull.NoReply',
                'Pull cancelled, no reply for %s (%s)', ['XOVER', Server] ));
           try Disconnect except end;
           onContinue := False;
        end;
        if onContinue then begin  // prepare overview-list for scoring/download
           if NNTP.ResultCode >= 400 then begin
              Log( LOGID_INFO, TrGl(kLog, 'Info.XOverResult', 'XOVER-result')+': ' + NNTP.ResultLine );
              Log( LOGID_INFO, TrGl(kLog, 'Info.XOverFailedLoadbyNr',
                   'XOVER-failed -> ignore scorefile, load by number' ) );
              ByNumOnly := True;
              for OvrLfd:=LetsGetMin to LetsGetMax do OverView.Add( inttostr(OvrLfd) );
           end else begin
              ByNumOnly := False;
              Overview.Text := NNTP.ResultList.Text;
           end;
        end;
     end;
     Result := onContinue;
  end;

  function GetNewArticles (var Overview: TStringList; var FirstPullDone, IsSkipped: Boolean;
  //-----------------------------------------------------------------------------------------
                           var NewLowMark: Integer; Const MoreCount: Integer; ByNumOnly, LastTry: Boolean;
                           GroupHandle: Integer; GroupName, Info: String) : Boolean;

        function CalcScore (out ActMsgId: String; var Parser: TParser) : Integer;
        //........................................................................
        var  Score    : integer;
             XOverRec : TXOverRec;
             s        : String;
        begin
           Score := 0;
           ActMsgId := Parser.sPart( MidPos, '' ); //used for ArtCurLoading
           if Score>=0 then begin // check if article is already in history
              s := Parser.sPart( MidPos, '' ); // Message-ID
              if (s='') or NewsHistory.ContainsMID(s) then begin
                 Score := -10000; // already in history
                 IncCounter( CntArtHist, 1 );
              end;
           end;
           if Score>=0 then begin // check scorefile
              XOverToXOverRec( XOverRec, Parser, ServerXOverList );
              Score := ScoreFile.ScoreBeforeLoad( XOverRec, nil );// PG
              // Score := ScoreFile.ScoreForXOverRec( XOverRec ); // PG
              if Score<0 then IncCounter( CntArtKill, 1 );
           end;
           Result := Score;
        end;

        function CheckCurLoading (ActMsgId : String; LastTry: Boolean) : Boolean;
        //........................................................................
        var b : Boolean;
        begin
           b := (LastTry or (ActMsgId=''));
           if b then begin // Load article, if last try or empty MsgId (ByNunOnly)
              if LastTry then
                 Log( LOGID_DETAIL, TrGlF(kLog, 'NNTP.ArtCurLoading.perhapsbyannotherthread',
                    'ArtCurLoading: perhaps at another Thread %s', ActMsgId));
           end else begin //testing: is article currently in progress by an other thread?
              EnterCriticalSection(CS_TEST_ARTICLE);
              b:=ArtCurLoading.CanLoad(ActMsgId);
              LeaveCriticalSection(CS_TEST_ARTICLE);
           end;
           Result := b;
        end;

        procedure UncheckCurLoading (ActMsgId: String; LastTry: Boolean);
        //................................................................
        var b : Boolean;
        begin
           b := (LastTry or (ActMsgId=''));
           if (not b) then begin // no last try and no empty MsgId
              EnterCriticalSection(CS_TEST_ARTICLE);
              ArtCurLoading.ArtLoaded(ActMsgId);
              LeaveCriticalSection(CS_TEST_ARTICLE);
           end;
        end;

        function KillArticle (var Overview: TStringList; var FirstPullDone: Boolean;
        //...........................................................................
                                   Const OvrLfd, Score: Integer;
                                   Const ScoreSave: Integer = 0;
                                   Const ScoreLoad: Integer = 0
                                   ) : Boolean;
        var  onContinue : Boolean;
             s : string;
        begin
           onContinue := True;
           Overview.Objects[OvrLfd]:=nil;
           if Score=-10000 then begin
              Log( LOGID_DEBUG, TrGlF(kLog, 'NNTP.GetArticles.SkippedByHistory',
                 'Skipped (History): %s', Overview[OvrLfd] ) );
           end else begin
              if Score>=Def_ScoreLogLimit then begin
                 s := Server + #9 + GroupName
                             + #9 + inttostr(Score)
                             + #9 + Overview[OvrLfd];
                 If ScoreLoad > 0 then begin
                    s := s + #9 + IntToStr(ScoreSave)
                           + #9 + IntToStr(ScoreLoad)
                 end;
                 LogFile.Enter;
                 try
                    HamFileAppendLine ( PATH_GROUPS + CFGFILE_SCORELOG, s );
                 finally
                    LogFile.Leave;
                 end;
              end else begin
                 Log( LOGID_DEBUG, TrGlF(kLog, 'NNTP.GetArticles.SkippedbyScorefile',
                    'Skipped (score=%s): %s', [inttostr(Score), Overview[OvrLfd]] ) );
              end;
           end;
           FirstPullDone := True;
           Result := onContinue;
        end;

        //...................................................................
        function StoreArticle (var Overview: TStringList;
                               var FirstPullDone: Boolean;
                               Const WeWantCur, OvrLfd, Score, GroupHandle: Integer;
                               Const GroupName, Info: String) : Boolean;
        Var FinalScore, ScoreAL: integer; t: string; Art: TArticle;
            OverrideGroups: String;
        begin
           Result := false;
           Overview.Objects[OvrLfd]:=nil;

           NNTP.SendCmndGetList( 'ARTICLE ' + inttostr(WeWantCur), 0 );
           if not( State in [csREADY, csLOGIN] ) then Exit;
           If AllShutDownReq or ThreadControl.Shutdown then Exit;
           if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin
              Log( LOGID_WARN, TrGlF(kLog, 'Warning.Pull.NoReply',
                   'Pull cancelled, no reply for %s (%s)' , ['ARTICLE', Server] ));
              try Disconnect except end;
              exit
           end;

           if NNTP.ResultCode < 400 then begin
              if length(NNTP.ResultList.Text)<32 then NNTP.ResultCode:=423;
              if NNTP.ResultList.Count<5 then NNTP.ResultCode:=423;
           end;

           if NNTP.ResultCode < 400 then begin
              // get loaded article
              Art := TArticle.Create;
              try
                 Art.Text := NNTP.ResultList.Text;
                 // get "Score-After-Load" value for article
                 ScoreAL := ScoreFile.ScoreAfterLoad( Art, nil );
                 // calculate final score value (= before + after load)
                 FinalScore := Score + ScoreAL;
                 if FinalScore<-9999 then FinalScore:=-9999;
                 if FinalScore>+9999 then FinalScore:=+9999;
                 // save article if "Score-After-Load" is also >= 0
                 if FinalScore >= 0 then begin
                    // Add Path-header
                    if Def_News_AddPath and (Def_FQDN>'')then begin
                       t := Art['Path:'];
                       If t > '' then Art['Path'] := Def_FQDN+'!'+t
                    end;
                    OverrideGroups := '';
                    If Actions.Exists ( atNewsInNNTP ) then begin
                       Case ModifyMessage( atNewsInNNTP, Art.Text, t) of
                          mcOriginal: ;
                          mcChanged: begin
                             Art.Text := t;
                             OverrideGroups := Art['!OverrideGroups']
                          end;
                          mcDeleted: Exit;
                       end;
                       If OverrideGroups>'' then Art.DeleteHeader('!OverrideGroups')
                    end;

                    // Save article
                    if SaveArticle( Art, GroupName, GroupHandle,
                       inttostr(FinalScore)
                         + ' ScoreLoad='+IntToStr(Score)
                         + ' ScoreSave='+IntToStr(ScoreAL),
                       False,
                       OverrideGroups )
                    then begin
                       IncCounter( CntArtLoad, 1 );
                    end else begin
                       IncCounter( CntArtHist, 1 );
                    end;
                    FirstPullDone := true
                 end else begin
                    KillArticle (Overview, FirstPullDone, OvrLfd, FinalScore,
                         Score, ScoreAL);
                    IncCounter( CntArtKill, 1 )
                 end
              finally
                 Art.free
              end;

           end else begin
              // 423 no such article number in this group
              // 430 no such article found
              IncCounter( CntArtHist, 1 );
              if (NNTP.ResultCode<>423) and (NNTP.ResultCode<>430) then exit
           end;
           Result := true
        end;

  var  onContinue : Boolean;
       OvrLfd     : integer;
       Parser     : TParser;
       WeWantCur: Integer;
       Score    : integer;
       ActMsgId, s: String;
  begin //// function GetNewArticles
     // 'FirstPullDone', 'IsSkipped' and 'NewLowMark' must be initialised previous
     onContinue := True;
     Parser := TParser.Create;
     try
        for OvrLfd:=0 to Overview.Count-1 do begin
           if Overview.Objects[OvrLfd]=pointer(1) then begin
              onContinue := (State in [csREADY, csLOGIN])
                  and Not (AllShutDownReq or ThreadControl.Shutdown);
              if (not onContinue) then begin
                 break; // just skipping the for-loop for saving time
              end else begin
                 if (MoreCount>0) // JAWO 22.08.01  (MoreCount: 'if ...')
                    then s := ' (+'+inttostr(MoreCount)+')'
                    else s := '';
                 Log( LOGID_STATUS, Info + TrGlF(kLog, 'Info.LoadingArticle',
                      'Loading article %s of max. %s...' , [inttostr(OvrLfd+1),
                      inttostr(Overview.Count)]) + s);    // JAWO 22.08.01 (MoreCount: ' + s') {kms}
                 IncCounter( CntArtNew, 1 );
                 // XOVER: 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References
                 //              6:Bytes 7:Lines [8:Xref full]
                 // XHDR:  0:No. 1:(Message-ID|'(none)')
                 Parser.Parse( Overview[OvrLfd], ParseChar );
                 WeWantCur := Parser.iPart( 0, -1 );
                 if WeWantCur>=0 then begin
                    Score := 0;
                    ActMsgId:='';
                    if not ByNumOnly then begin
                       Score := CalcScore (ActMsgId, Parser); // (initiates: ActMsgId)
                    end;
                    if (Score < 0) then begin
                       onContinue := KillArticle (Overview, FirstPullDone, OvrLfd, Score);
                    end else begin // i.e. (Score>=0)
                       if CheckCurLoading (ActMsgId, LastTry) then begin
                          try
                            onContinue := StoreArticle (Overview, FirstPullDone, WeWantCur,
                                                        OvrLfd, Score, GroupHandle, GroupName, Info);
                          finally
                            UncheckCurLoading (ActMsgId, LastTry);
                          end;
                       end else begin
                          IsSkipped:=True;    // at least one article is skipped
                          IncCounter( CntArtNew, -1 );
                          Log( LOGID_DETAIL, TrGlF(kLog, 'NNTP.ArtCurrentlyLoading.Skip',
                              'Article %s currently loading: already in progress by another thread', ActMsgId ) );
                       end;
                    end; // if Score<0 else
                    If onContinue and (not IsSkipped) and (NewLowMark < WeWantCur) then NewLowMark := WeWantCur;
                end; // if WeWantCur>=0
              end;
           end; // if Overview.Objects
        end; // for
     finally
        Parser.Free;
     end;
     Result := onContinue;
  end;

var  onContinue : Boolean;
     Overview, Skipped : TStringList;
     Info : String;
     NewLowMark, OvrLfd, ActTry : Integer;
     SrvHasMin, SrvHasMax : Integer;
     WeWantMin, WeWantMax : Integer;
     LetsGetMin, LetsGetMax, i : Integer;
     LetsGetParts : Boolean;
     ByNumOnly, FirstPullDone : Boolean;
     IsSkipped : Boolean;
     LetsGetSizeMin : Integer;
     LetsGetSizeMax : Integer;

begin //// procedure TClientNNTP.GetNewArticlesForGroup
   //HSR: Default-Einstellungen/ini-Einstellugen verwenden
   LetsGetSizeMin := Def_parts_size_Min; // default block size of parted pulls
   LetsGetSizeMax := Def_parts_size_Max; // max. block size of parted pulls (single/last one)

   onContinue := ((State in [csREADY, csLOGIN]) and (not (AllShutDownReq or ThreadControl.Shutdown)));
   if onContinue then begin      // select group
      Info := '[' + Server + ', ' + GroupName + '] ';
      Log( LOGID_DETAIL, Info + TrGl(kLog, 'NNTP.GetNewArticles', 'GetNewArticles' ) );
{JW} {Mode Reader}
//      {JW} {Feed}{TEST}   { TODO 1 -oJW/HSR -cFeed : TEST-JAWO + FEED }
//      if Def_NNTPClient_ModeReader and not ModeReader then begin
//        ModeReader:=True;
//        NNTP.SendCmnd( 'MODE READER', 0, False );
//      end;
     {JW}
      onContinue := SelectGroup (GroupName, Info);
   end;

   if onContinue then begin  // (initiates: WeWantMin, WeWantMax, SrvHasMin, SrvHasMax)
      onContinue := GetRanges (WeWantMin, WeWantMax, SrvHasMin, SrvHasMax,
                               GroupHandle, GroupName, Info);
   end;

   if onContinue then begin
      FirstPullDone := False;
      NewLowMark := -1;
      Overview := TStringList.Create;
      Skipped := TStringList.Create;

      try
         IsSkipped := False;
         LetsGetParts := ((WeWantMin + LetsGetSizeMax) <= WeWantMax); // shall we pull in parts?
         LetsGetMax := WeWantMin - 1;
         while (onContinue and (LetsGetMax < WeWantMax)) do begin
            LetsGetMin := LetsGetMax + 1;
            if ((LetsGetMin+LetsGetSizeMax) <= WeWantMax) and def_parts_make then begin
               { TODO 1 -oHSR -cPartsforOverview : Parts an-/ausschalten }
               LetsGetMax := LetsGetMin + LetsGetSizeMin - 1; // pull next parts
            end else begin
               LetsGetMax := WeWantMax; // pull the rest (or all)
            end;
            if LetsGetParts then begin
               Log( LOGID_INFO, Info + TrGlF(kLog, 'Info.OverviewhasWePart',
                  'Server has %s-%s. We want now %s-%s (of %s-%s).',
                  [inttostr(SrvHasMin),
                  inttostr(SrvHasMax), inttostr(LetsGetMin), inttostr(LetsGetMax),
                  inttostr(WeWantMin), inttostr(WeWantMax) ]) );
            end else begin
               Log( LOGID_INFO, Info + TrGlF(kLog, 'Info.OverviewhasWeWant',
                  'Server has %s-%s. We want %s-%s.',
                  [inttostr(SrvHasMin),
                  inttostr(SrvHasMax), inttostr(WeWantMin), inttostr(WeWantMax) ]) );
            end;

            onContinue := GetOverview (OverView, ByNumOnly, LetsGetMin, LetsGetMax);
            if onContinue then begin            // (initiates: OverView, ByNumOnly)
               ScoreFile.SelectSections( GroupName );  // load scorelist for current group
               if OverView.Count>0 then begin          // load articles
                  Log( LOGID_INFO, Info + TrGlF(kLog, 'Info.LoadingArticles',
                          'Loading up to %s articles ...' , inttostr(Overview.Count)));
               end else begin
                  if (not IsSkipped) then NewLowMark := WeWantMax;
                  Log( LOGID_INFO, Info + TrGl(kLog, 'Info.NoArticlesToLoad', 'No articles to load.' ) );
               end;
               for OvrLfd:=0 to Overview.Count-1 do Overview.Objects[OvrLfd]:=Pointer(1);  //Mark Articles to load
               onContinue := GetNewArticles (Overview, FirstPullDone, IsSkipped, NewLowMark,
                 (WeWantMax-LetsGetMax), ByNumOnly, false, GroupHandle, GroupName, Info);  // JAWO 22.08.01 (MoreCount: '(WeWantMax-LetsGetMax),')
               if onContinue then begin
                  for OvrLfd:=0 to Overview.Count-1 do begin // which articles have been skipped?
                     if (Overview.Objects[OvrLfd] = Pointer(1)) then begin
                        Skipped.Add (Overview[OvrLfd]);
                     end;
                  end;
               end;
            end;
         end; // while

         IsSkipped := (Skipped.Count > 0);
         if (onContinue and IsSkipped) then begin
            ActTry := 1; // first try we had just done above
            Log( LOGID_DETAIL, TrGlf(kLog, 'NNTP.ArtCurLoading.totalySkipped',
               'ArtCurLoading: (%s) totaly skipped %s articles above Nr %s',
               [IntToStr(1), inttostr(Skipped.Count), inttostr(NewLowMark)] ) );
            for OvrLfd:=0 to Skipped.Count-1 do Skipped.Objects[OvrLfd]:=Pointer(1);  //Mark Articles to load
            while (onContinue and IsSkipped and (ActTry < MaxTries)) do begin
               Inc(ActTry);          // wait an increasing bit of time for other
               sleep (ActTry * 500); // threads to retrieve the skipped messages
               IsSkipped := False;
               onContinue := GetNewArticles (Skipped, FirstPullDone, IsSkipped, NewLowMark,
                                 0, ByNumOnly, (ActTry=MaxTries), GroupHandle, GroupName, Info);
               i := 0;
               for OvrLfd:=0 to Skipped.Count-1 do if (Skipped.Objects[OvrLfd]=Pointer(1)) then inc(i);
               Log( LOGID_DETAIL, TrGlf(kLog, 'NNTP.ArtCurLoading.totalySkipped.1',
                  'ArtCurLoading: (%s) totaly skipped %s articles above Nr %s',
                  [IntToStr(ActTry), inttostr(i), inttostr(NewLowMark)] ) );
            end;
            if (onContinue and (not IsSkipped) and (NewLowMark < WeWantMax)) then begin
               Log( LOGID_DETAIL, TrGlf(kLog, 'NNTP.ArtCurLoading.NewLowMarkFromTo',
                  'NewLowMark increased from %s to %s', [inttostr(NewLowMark), inttostr(WeWantMax) ] ) );
               NewLowMark := WeWantMax;
            end;
         end; // if onContinue
      finally
         Skipped.Free;
         Overview.Free;
      end;

      // even if NOT onContinue:
      if FirstPullDone then begin  // first pull for Group/Server is from now on assumed as "done"
         ArticleBase.SetFirstPullDone( GroupHandle, Server );
      end;

      if NewLowMark>=1 then begin  // save last art-no processed
         ArticleBase.ServerMin[GroupHandle,Server] := NewLowMark;
      end;
   end; // if onContinue
end;


function TClientNNTP.PostArticle( PostFile, ArtText: String; var SrvResult: String ): Boolean;
var  Info, s: String;
     LineNo : Integer;
     LineStr: String;
     ArtList: TStringList;
begin
   Result    := False;
   SrvResult := '';

   if not( State in [csREADY, csLOGIN] ) then exit;
   if ArtText='' then exit;

   Info := '[' + Server + '] ';
   Log( LOGID_DETAIL, Info + TrGl(kLog, 'NNTP.PostArticle', 'PostArticle' ) );

   NNTP.SendCmnd( 'POST', 0, False );

   if NNTP.ResultCode = 340 then begin

      // Artikeltext senden
      ArtList := TStringList.Create;
      ArtList.Text := ArtText;
      for LineNo:=0 to ArtList.Count-1 do begin
         LineStr := ArtList[LineNo];
         if copy(LineStr,1,1)='.' then LineStr:='.'+LineStr;
         NNTP.SendData( LineStr + #13#10 );
         if not( State in [csREADY, csLOGIN] ) then break;
      end;
      ArtList.Free;

      // Ende-Kennung schicken
      NNTP.SendCmnd( '.', 0, False );
      if NNTP.ResultCode = 240 then Result:=True;

   end;

   SrvResult := NNTP.ResultLine;

   // save result in .log
   s := DateTimeToTimeStamp( Now ) + ' '
      + 'File='   + ExtractFilename( PostFile ) + ' '
      + 'Server=' + Server + ',' + Port + ' '
      + 'Result=' + NNTP.ResultLine;
   HamFileAppendLine ( PATH_LOGS + 'NewsOut.log', s );
end;

{JW} {SSL}
constructor TClientNNTP.Create( AServer, APort, AUser, APass: String;
                                ASSLMode, ASSLVerifyLevel: Integer;
                                ASSLCaFile: String );
begin
   inherited Create;
   Server := AServer;
   Port   := APort;
   User   := AUser;
   Pass   := APass;
   {MG}{SSL}
   SSLMode        := ASSLMode;
   SSLVerifyLevel := ASSLVerifyLevel;
   SSLCaFile      := ASSLCaFile;
   {/SSL}
   NNTP   := nil;

   ServerDir := PATH_SERVER + Server;
   if not DirectoryExists(ServerDir) then ForceDirectories( ServerDir );
   ServerDir := ServerDir + '\';

   ScoreFile := TFiltersNews.Create( PATH_BASE + CFGFILE_SCORES );

   Move( XOVERLIST_DEFAULT[0], ServerXOverList[0], sizeof(XOVERLIST_DEFAULT) );
end;

destructor TClientNNTP.Destroy;
begin
   if Assigned(NNTP) then Disconnect;
   if Assigned(ScoreFile) then ScoreFile.Free;
   inherited Destroy;
end;

{ tArtCurLoading }
constructor tArtCurLoading.Create;
begin
   inherited create;
   Sorted:=true;
   Duplicates:=dupIgnore
end;

function tArtCurLoading.CanLoad(MsgId:String):Boolean;
begin
   Result := IndexOf(MsgId) < 0;
   If Result then Add(MsgId)
end;

procedure tArtCurLoading.ArtLoaded(MsgId:String);
var i : Integer;
begin
   if Find(MsgId,i) then delete(i);
end;

Initialization
   ArtCurLoading := tArtCurLoading.Create
Finalization
   ArtCurLoading.free
end.
