// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cClientPOP3;

interface
{JW} {SSL}
uses Classes, cClientBase, cFiltersMail, cArticle, uSSL; {MG}{SSL}
{JW}

const
   SRVFILE_HELPTEXT     = 'help.txt';
   SRVFILE_GREETING     = 'greeting.txt';

type
   TClientSocketPOP3 = class( TClientSocketBase )
     private
       ServerDir: String;
     public
       ResultCode: Integer;
       constructor Create (Const AServerDir: String ); reintroduce;
       function  ResultListFollows: Boolean; override;
       procedure SendCmnd( Cmnd : String; Special: Integer; FilterLog: Boolean ); override;
       function  Login( AUser, APass: String ): Boolean; override;
       function  NegotiateSSL: Boolean; override; {MG}{SSL}
   end;

   TClientPOP3 = class
      private
         Server, Port, User, Pass, LocalUser, FilterSection: String;
         ServerDir  : String;
         POP3       : TClientSocketPOP3;
         FilterFile : TFiltersMail;
         LoadingMailSize: Integer;                    {.kms}
         LoadingMailLogline: String;
         LoadingMailLastprogress: Integer;
         SSLMode, SSLVerifyLevel: Integer; SSLCaFile: String; {MG}{SSL}
         procedure NewDataCallback(DataSize:integer); {/kms}
      public
         function  State : Integer;
         procedure Connect;
         procedure Disconnect;

         procedure GetServerInfos;
         procedure SendNotification( NotifyType: Integer;
                                     DestUsers: TStringList;
                                     NotifyReason, Scoring: String;
                                     Msg: TArticle;
                                     SizeOfMsg: Integer );
         Function GetNewMails( LeaveOnServer: Boolean;
                               DefaultDestUser: String;
                               DefFilterSection: String;
                               LoadMax: Integer ): boolean;

         {MG}{SSL}
         constructor Create( AServer, APort, AUser, APass,
                             ALocalUser, AFilterSection: String;
                             ASSLMode, ASSLVerifyLevel: Integer;
                             ASSLCaFile: String );
         {/SSL}
         destructor Destroy; override;
   end;

implementation

uses SysUtils, FileCtrl, Global, cArtFiles, cAccount, Config, uTools, cPCRE,
     uMD5, uSha1, uEncoding, uDateTime, cStdForm, cMailRouter,
     cLogFile, inifiles, tBase, cActions;

const
   NOTIFYTYPE_TEST   = 0;
   NOTIFYTYPE_KILL   = 1;
   NOTIFYTYPE_IGNORE = 2;
   NOTIFYTYPE_GLOBALSIZE_IGNORE = 10;
   NOTIFYTYPE_GLOBALSIZE_KILL   = 11;
   NOTIFYTYPE_GLOBALSIZE_NOTIFY = 12;

   UIDLMARKER_NOTIFY = 'notified-';

   RESULT_OK = 0;

{ TClientSocketPOP3 }
   
procedure TClientSocketPOP3.SendCmnd( Cmnd : String; Special: Integer; FilterLog: Boolean );
begin
   inherited SendCmnd( Cmnd, Special, FilterLog );
   if ResultLine='' then begin
      ResultCode := 999;
      ResultLine := '-ERR (timeout or connection lost)';
   end else begin
      if copy(ResultLine,1,1)='+' then begin
         ResultCode := RESULT_OK; // +OK
      end else begin
         ResultCode := 500; // -ERR
      end;
   end;
end;

function TClientSocketPOP3.ResultListFollows: Boolean;
begin
   Result := ( ResultCode = RESULT_OK );
end;

function TClientSocketPOP3.Login( AUser, APass: String ): Boolean;
var  APOP_Stamp, s, cmp: String;
     UseAPOP: Boolean;
     i, j: Integer;
     sRemote, sLocal, sWanted, SASL_Remote, SASL_Wanted, Mechanism: String;
     UseCRAMMD5, UseCRAMSHA1, UseLogin, UsePlain, UseSASL, UseAuth: Boolean;
     TS: TStringList;
begin
   Result := False;
   Log( LOGID_DEBUG, Self.ClassName + '.Login' );
   if ClientState<>csCONNECTED then exit;

   ClientState := csLOGIN;

   if (AUser<>'') then begin
      UseAPOP := False;
      APOP_Stamp := '';
      UseCRAMMD5 := False;
      UseCRAMSHA1:= False;
      UseLogin   := False;
      UsePlain   := False;
      UseSASL    := False;
      UseAuth    := False;
      if copy(APass,1,5)='PASS:' then begin
         System.Delete( APass, 1, 5 );
      end else
      if copy(APass,1,5)='APOP:' then begin
         UseAPOP := true;
         System.Delete( APass, 1, 5 );
         APOP_Stamp := RE_Extract( Greeting, '(<.*?\@.*?>)', 0 );
         If APOP_Stamp = '' then begin      // fix for APOP secure bug
            Log( LOGID_ERROR, FServer+': APOP-command not supported');
            ClientState:=csERROR;
            Exit
         end
      end else
      if copy(APass,1,9)='CRAM-MD5:' then begin
         UseCRAMMD5 := True; System.Delete( APass, 1, 9 )
      end else
      if copy(APass,1,6)='PLAIN:' then begin
         UsePlain := True; System.Delete( APass, 1, 6 )
      end else
      if copy(APass,1,6)='LOGIN:' then begin
         UseLogin := True; System.Delete( APass, 1, 6 )
      end else
      if copy(APass,1,5)='SASL:' then begin
         UseSASL := True; System.Delete( APass, 1, 5 )
      end else
      if copy(APass,1,5)='AUTH:' then begin
         UseAuth   := True; System.Delete( APass, 1, 5 )
      end else
      if copy(APass,1,10)='CRAM-SHA1:' then begin
         UseCRAMSHA1:= True; System.Delete( APass, 1,10 );
      end;

      if UseAUTH and (ClientState=csLOGIN) then begin
         Log( LOGID_DETAIL, 'Authentification with mechanism "SASL" ' );
         SendCmnd( 'AUTH', 8, True );
         if (ClientState in [csERROR, csDISCONNECTED]) or
            (ResultCode<>RESULT_OK) then begin
             Log(LOGID_ERROR,FServer+
             ': AUTH command required for AUTH mechanism not supportet ');
             ClientState:=csERROR;
             exit;
         end;
         SASL_WANTED := POP3LocalSASL;
         If FileExists2(ServerDir + SRVFILE_INI) then begin
            With TIniFile.Create(ServerDir + SRVFILE_INI ) do try
               SASL_WANTED := ReadString('POP3','SASL', SASL_WANTED );
            finally Free end
         end;
         TS := TStringList.Create;
         try
            TS.Text := ResultLine;
            for i:=0 to TS.Count-1 do begin
               s := UpperCase( copy( TS[i], 5, 999 ) ); // skip '250[- ]'
               if pos('CRAM-MD5',s)>0 then
                  SASL_REMOTE:=SASL_REMOTE+' CRAM-MD5';
               if pos('CRAM-SHA1',s)>0 then
                  SASL_REMOTE:=SASL_REMOTE+' CRAM-SHA1';
               if pos('PLAIN',s)>0 then
                  SASL_REMOTE:=SASL_REMOTE+' PLAIN';
               if pos('LOGIN',s)>0 then
                  SASL_REMOTE:=SASL_REMOTE+' LOGIN';
            end;
            trim(SASL_Remote);
         finally
            TS.Free;
         end;
         sRemote := UpperCase( SASL_REMOTE );  // supported by server
         sLocal  := UpperCase( POP3LocalSASL  );  // supported by Hamster
         sWanted := UpperCase( SASL_WANTED );  // preference of user
         if sWanted='' then sWanted := sLocal; // no preference,
                                               //use Hamster-default
         Mechanism := '';
         while sWanted<>'' do begin
            i := PosWhSpace( sWanted );
            if i=0 then begin
               s := sWanted;
               sWanted := '';
            end else begin
               s := copy( sWanted, 1, i-1 );
               System.Delete( sWanted, 1, i );
            end;
            if Pos( ' ' + s + ' ', ' ' + sLocal + ' ' ) = 0 then begin
               Log(LOGID_WARN, FServer+': Invalid SASL-setting "'+
                   SASL_WANTED + '"' );
            end else begin
               if Pos( ' ' + s + ' ', ' ' + sRemote + ' ' ) > 0 then begin
                  Mechanism := s;
                  break;
               end;
            end;
         end;
         if mechanism='' then begin
            Log(LOGID_WARN, FServer + ': SASL not found use trial and error');
            {JW} {RFC1734 Fallback}
            if pos('CRAM-MD5',UpperCase( SASL_WANTED ))<>0 then UseCRAMMD5 :=True;
            if pos('CRAM-SHA1',UpperCase( SASL_WANTED ))<>0 then UseCRAMSHA1:=True;
            if pos('LOGIN',UpperCase( SASL_WANTED ))<>0 then UseLogin   :=True;
            if pos('PLAIN',UpperCase( SASL_WANTED ))<>0 then UsePlain   := True;
            {JW}
         end else UseAuth:=False;
         if mechanism='CRAM-MD5'  then UseCRAMMD5:=TRUE;
         if mechanism='CRAM-SHA1' then UseCRAMSHA1:=TRUE;
         if mechanism='LOGIN'     then UseLogin:=TRUE;
         if mechanism='PLAIN'     then UsePlain:=TRUE;
      end;

      if UseSASL and (ClientState=csLOGIN) then begin
         Log( LOGID_DETAIL, 'Authentification with mechanism "SASL" ' );
         SendCmnd( 'CAPA', 8, True );
         if (ClientState in [csERROR, csDISCONNECTED]) or
            (ResultCode<>RESULT_OK) then begin
              Log( LOGID_ERROR, FServer +
                    ': CAPA command required for SASL not supported ');
              ClientState:=csERROR;
              exit;
         end;
         SASL_WANTED := POP3LocalSASL;
         If FileExists2(ServerDir + SRVFILE_INI) then begin
            With TIniFile.Create(ServerDir + SRVFILE_INI) do try
               SASL_WANTED := ReadString('POP3','SASL', SASL_WANTED );
            finally Free end
         end;
         TS := TStringList.Create;
         try
            TS.Text := ResultLine;
            for i:=0 to TS.Count-1 do begin
               s := UpperCase( copy( TS[i], 5, 999 ) ); // skip '250[- ]'
               if (copy(s,1,4)='SASL') and
                  (length(s)>5) and
                  (s[5] in [' ','=']) then begin
                   SASL_REMOTE:=TrimWhSpace(copy(s,6,999));
                   break;
               end;
               // Workarround for damage remote implemetion
               if pos('CRAM-MD5',s)>0 then begin
                  SASL_REMOTE:='CRAM-MD5';
                  break;
               end;
               // Workaround for damage remote implemetion
               if pos('CRAM-SHA1',s)>0 then begin
                  SASL_REMOTE:='CRAM-SHA1';
                  break;
               end;
            end;
         finally
            TS.Free;
         end;
         sRemote := UpperCase( SASL_REMOTE );  // supported by server
         sLocal  := UpperCase( POP3LocalSASL  );  // supported by Hamster
         sWanted := UpperCase( SASL_WANTED );  // preference of user
         if sWanted='' then sWanted := sLocal; // no preference,
                                               //use Hamster-default
         Mechanism := '';
         while sWanted<>'' do begin
            i := PosWhSpace( sWanted );
            if i=0 then begin
               s := sWanted;
               sWanted := '';
            end else begin
               s := copy( sWanted, 1, i-1 );
               System.Delete( sWanted, 1, i );
            end;
            if Pos( ' ' + s + ' ', ' ' + sLocal + ' ' ) = 0 then begin
               Log( LOGID_WARN,FServer+': Invalid SASL-setting "'+
                    SASL_WANTED+'"');
            end else begin
               if Pos( ' ' + s + ' ', ' ' + sRemote + ' ' ) > 0 then begin
                  Mechanism := s;
                  break;
               end;
            end;
         end;
         if mechanism='' then begin
            Log( LOGID_ERROR, FServer +
                 ': SASL not supportet by remote server');
            ClientState:=csERROR;
            exit;
         end;
         if mechanism='CRAM-MD5'  then UseCRAMMD5:=TRUE;
         if mechanism='CRAM-SHA1' then UseCRAMSHA1:=TRUE;
         if mechanism='LOGIN'     then UseLogin:=TRUE;
         if mechanism='PLAIN'     then UsePlain:=TRUE;
      end;
      if UseCRAMMD5 and (ClientState=csLOGIN) then begin
         Log(LOGID_DETAIL,'Authentification with mechanism "CRAM-MD5" ');
         SendCmnd( 'AUTH CRAM-MD5', 0, True );
         if (ClientState in [csERROR, csDISCONNECTED]) or
            (ResultCode<>RESULT_OK) then begin
            if UseAuth then begin
               Log( LOGID_DETAIL, FServer +
                    ': Mechanism CRAM-MD5 not supportet ')
            end else begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/1 failed with "' +
                    ResultLine + '"' );
               ClientState:=csERROR;
               exit;
            end;
         end else begin
            s:=copy(ResultLine,PosWhSpace( ResultLine )+1,500);
            s:=DecodeB64( s[1], length(s) );
            s:=MD5HMAC( APass,s );
            s:=MD5toHex( s );
            s:=AUser+' '+s;
            s := EncodeB64( s[1], length(s) ); // Username + Passwort
            SendCmnd( s, 0, True );
            if ClientState in [csERROR, csDISCONNECTED] then exit;
            if ResultCode<>RESULT_OK then begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/2 failed with "' +
                    ResultLine + '"' );
               ClientState:=csERROR;
               exit;
            end else ClientState := csREADY;
         end;
      end;

      if UseCRAMSHA1 and (ClientState=csLOGIN) then begin
         Log(LOGID_DETAIL,'Authentification with mechanism "CRAM-SHA1" ');
         SendCmnd( 'AUTH CRAM-SHA1', 0, True );
         if (ClientState in [csERROR, csDISCONNECTED]) or (ResultCode<>RESULT_OK) then begin
            if UseAuth then begin
               Log( LOGID_DETAIL, FServer + ': Mechanism CRAM-SHA1 not supported');
               Log( LOGID_WARN, FServer + ': secure crypto mechanism not found or supported');
               Log( LOGID_WARN, FServer + ': use insecure mechanism with clear text passwords! ');
            end else begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/1 failed with "' +
                    ResultLine + '"' );
               ClientState:=csERROR;
               exit;
            end;
         end else begin
            s:=copy(ResultLine,PosWhSpace( ResultLine )+1,500);
            s:=DecodeB64( s[1], length(s) );
            s:=HMAC_SHA1( s, APass );
            s:=AUser+' '+s;
            s := EncodeB64( s[1], length(s) ); // Username + Passwort
            SendCmnd( s, 0, True );
            if ClientState in [csERROR, csDISCONNECTED] then exit;
            if ResultCode<>RESULT_OK then begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/2 failed with "' +
                    ResultLine + '"' );
               ClientState:=csERROR;
               exit;
            end else ClientState := csREADY;
         end;
      end;

      if UseLogin and (ClientState=csLOGIN) then begin
         Log( LOGID_DETAIL, 'Authentification with mechanism "LOGIN" ' );
         SendCmnd( 'AUTH LOGIN', 0, True );
         if (ClientState in [csERROR, csDISCONNECTED]) or
            (ResultCode<>RESULT_OK) then begin
            if UseAuth then
               Log( LOGID_DETAIL, FServer +
                    ': Mechanism LOGIN not supportet ')
            else begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/1 failed with "' +
                    ResultLine + '"' );
               ClientState:=csERROR;
               exit;
            end;
         end else begin
            s := EncodeB64( AUser[1], length(AUser) ); // Username
            SendCmnd( s, 0, True );
            if ClientState in [csERROR, csDISCONNECTED] then exit;
            if ResultCode<>RESULT_OK then begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/2 failed with "' +
                    ResultLine + '"' );
               ClientState:=csERROR;
               exit;
            end;
            s := EncodeB64( APass[1], length(APass) ); // Password
            SendCmnd( s, 0, True );
            if ClientState in [csERROR, csDISCONNECTED] then exit;
            if ResultCode<>RESULT_OK then begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/3 failed with "' +
                    ResultLine + '"' );
               ClientState:=csERROR;
               exit;
            end else ClientState := csREADY;
         end;
      end;

      if UsePLAIN and (ClientState=csLOGIN) then begin
         Log( LOGID_DETAIL, 'Authentification with mechanism "PLAIN" ' );
         SendCmnd( 'AUTH PLAIN', 0, True );
         if (ClientState in [csERROR, csDISCONNECTED]) or
            (ResultCode<>RESULT_OK) then begin
            if UseAuth then begin
               Log( LOGID_DETAIL, FServer +
                    ': Mechanism PLAIN not supportet ');
               Log( LOGID_ERROR, FServer +
                    ': No Auth mechanism found');
            end else begin
               Log( LOGID_ERROR, FServer +
                    ': AUTH/1 failed with "' +
                    ResultLine + '"' );
            end;
            ClientState:=csERROR;
            exit;
         end;
         s := chr(0)+AUser+chr(0)+APASS;
         s := EncodeB64( s[1], length(s) ); // Username + Passwort
         SendCmnd( s, 0, True );
         if ClientState in [csERROR, csDISCONNECTED] then exit;
         if ResultCode<>RESULT_OK then begin
            Log( LOGID_ERROR, FServer +
                 ': AUTH/2 failed with "' +
                 ResultLine + '"' );
            ClientState:=csERROR;
            exit;
         end else ClientState := csREADY;
      end;
{JW}
      if UseAPOP then begin
         Log( LOGID_DETAIL, 'Authentification with mechanism "APOP" ' );
         Log( LOGID_DEBUG, 'APOP-timestamp="' + APOP_Stamp + '"' );
         s := MD5ofStr( APOP_Stamp + APass );
         cmp := '';
         for j:=1 to length(s) do cmp := cmp + lowercase( inttohex( ord(s[j]), 2 ) );
         SendCmnd( 'APOP ' + AUser + ' ' + cmp, 0, True );
         if ClientState in [csERROR, csDISCONNECTED] then exit;
         if ResultCode=RESULT_OK then begin
            ClientState := csREADY
         end else begin
            ClientState:=csERROR;
            Exit
         end
      end;

      if ClientState=csLOGIN then begin
         Log( LOGID_DETAIL, 'Login without authentification' );
         SendCmnd( 'USER ' + AUser, 0, True );
         if ClientState in [csERROR, csDISCONNECTED] then exit;
         if ResultCode<>RESULT_OK then begin ClientState:=csERROR; exit; end;

         SendCmnd( 'PASS ' + APass, 0, True );
         if ClientState in [csERROR, csDISCONNECTED] then exit;
         if ResultCode=RESULT_OK then ClientState := csREADY
                                 else begin ClientState:=csERROR; exit; end;
      end;
   end else begin
      ClientState := csREADY;
   end;
   if ClientState=csREADY then Result:=True;
end;

{MG}{SSL}
function TClientSocketPOP3.NegotiateSSL : Boolean;  // RFC 2595
begin
     Result := False;
     if ClientState<>csCONNECTED then exit;
     ClientState := csLOGIN;
     SendCmnd( 'STLS', 0, False );
     if ResultCode = RESULT_OK then begin
           Log( LOGID_DEBUG, Format(TrGl(kLog, 'SSL.Handshake',
             '[%s] starting SSL/TLS handshake ...'), [FServer]) );
        Result := StartSSL;
     end else
        Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.ProtoNotSupported',
             '[%s] Server does not support %s over SSL'), [FServer, 'POP3']) );
end;
{/SSL}

constructor TClientSocketPOP3.Create(const AServerDir: String);
begin
   inherited Create (NIL);
   ServerDir := AServerDir
end;

{ TClientPOP3 }

function TClientPOP3.State : Integer;
begin
   if Assigned( POP3 ) then Result := POP3.ClientState
                       else Result := csDISCONNECTED;
end;

procedure TClientPOP3.Connect;
var  s: String;
     ConnectWithSSL: Boolean; {MG}{SSL}
     ok: Boolean;
begin
   if Assigned( POP3 ) then Disconnect;
   POP3 := TClientSocketPOP3.Create( ServerDir );
   ConnectWithSSL := SSLMode = 1;
   ok := false;
   if POP3.Connect( Server, Port, ServerDir + SRVFILE_INI,
                    ConnectWithSSL, SSLVerifyLevel, SSLCaFile )
   then begin
      if State = csCONNECTED then begin
         if SSLMode = 2 then POP3.NegotiateSSL;
         if SSLMode = 3 then begin
            if not POP3.NegotiateSSL then begin Disconnect; exit; end;
         end;
         if State = csLOGIN then POP3.ClientState := csCONNECTED;
         If POP3.Login( User, Pass ) then begin
            ok := true
         end else begin
            s := POP3.ResultLine;
            if s<>'' then s:=' ("'+s+'")';
            Log( LOGID_WARN, TrGlF(kLog, 'LoginFailed', 'Login failed %s', s ) );
         end;
         if POP3.Greeting > '' then begin
            If DirExists2(ExcludeTrailingBackslash(ServerDir)) then begin
               HamFileRewriteLine( ServerDir + SRVFILE_GREETING, POP3.Greeting )
            end
         end
      end
   end;
   POP3.ConnectionStatistic ( ServerDir + SRVFILE_INI, ok)
end;

procedure TClientPOP3.Disconnect;
begin
   if not Assigned( POP3 ) then exit;
   try POP3.Disconnect; except end;
   FreeAndNIL(POP3)
end;


procedure TClientPOP3.GetServerInfos;

   procedure GetList( Cmnd, Filename: String );
   begin
      if State in [csERROR, csDISCONNECTED] then exit;
      if FileExists2( Filename ) then exit;

      Log( LOGID_INFO, TrGlF(kLog, 'Info.ServerLoadingCmd',
         '[%s] Loading <%s> ...' , [Server, Cmnd]));

      POP3.SendCmndGetList( Cmnd, 0 );
      if State in [csERROR, csDISCONNECTED] then exit;

      if POP3.ResultList.Count=0 then begin
         POP3.ResultList.Insert( 0, '# Command: ' + POP3.ResultCmnd );
         POP3.ResultList.Insert( 1, '# Result : ' + POP3.ResultLine );
      end;

      try POP3.ResultList.SaveToFile( Filename ); except end;
   end;

begin
   // deactivated, because some servers do not handle it correctly:
   // GetList( 'HELP', ServerDir + SRVFILE_HELPTEXT );
end;


procedure TClientPOP3.SendNotification( NotifyType: Integer;
                                        DestUsers: TStringList;
                                        NotifyReason, Scoring: String;
                                        Msg: TArticle;
                                        SizeOfMsg: Integer );
var  i: Integer;
     Subject, Body: String;
     MsgIsMail: Boolean;
begin
    MsgIsMail := True;

   case NotifyType of
      NOTIFYTYPE_KILL: begin
         Subject := '[Hamster] '+TrGl('Mailinfo', 'Mail.killed-filter.Subject', 'Mail killed!');
         Body := '[Hamster]' + #13#10#13#10
               + TrGlF('Mailinfo', 'mail.killed-filter.body',
                    'A mail-message on %s was deleted due to mail-filters.',
                    Server+' (' + User + ')');
         If NotifyReason > '' then Body := Body + #13#10 + NotifyReason;
         If Scoring > '' then Body := Body + #13#10 + 'Scoring:'+Scoring + #13#10;
         Body := Body + #13#10 + 'Subject: '+Msg['Subject']
                      + #13#10 + 'From: '+Msg['From']
                      + #13#10 + 'To: '+Msg['To']
                      + #13#10 + TrGlF('MailInfo', 'SizeOfMailInBytes',
                                 'Size of mail: %s bytes', FormatFloat(',0', SizeOfMsg ))
                      + #13#10
      end;
      NOTIFYTYPE_IGNORE: begin
         Subject := '[Hamster] '+TrGl('Mailinfo', 'Mail.ignored-filter.Subject', 'Mail ignored!');
         Body := '[Hamster]' + #13#10#13#10
               + TrGlF('Mailinfo', 'mail.ignored-filter.body',
                    'A mail-message on %s was ignored due to mail-filters.',
                    Server+' (' + User + ')');
         If NotifyReason > '' then Body := Body + #13#10 + NotifyReason;
         If Scoring > '' then Body := Body + #13#10 + 'Scoring:'+Scoring + #13#10;
         Body := Body + #13#10 + 'Subject: '+Msg['Subject']
                      + #13#10 + 'From: '+Msg['From']
                      + #13#10 + 'To: '+Msg['To']
                      + #13#10 + TrGlF('MailInfo', 'SizeOfMailInBytes',
                                 'Size of mail: %s bytes', FormatFloat(',0', SizeOfMsg ))
                      + #13#10
      end;

      NOTIFYTYPE_GLOBALSIZE_KILL: begin
         Subject := '[Hamster] '+TrGl('Mailinfo', 'Mail.killed-globalsize.Subject',
                 'Mail killed!');
         Body := '[Hamster]' + #13#10#13#10
               + TrGlF('Mailinfo', 'Mail.killed-globalsize.body',
                    'A mail-message on %s was deleted due to global size-limit.',
                    Server+' (' + User + ')')
               + #13#10 + Msg.Text;
         MsgIsMail := false
      end;

      NOTIFYTYPE_GLOBALSIZE_IGNORE: begin
         Subject := '[Hamster] '+TrGl('Mailinfo', 'Mail.ignored-globalsize.Subject',
                 'Mail ignored!');
         Body := '[Hamster]' + #13#10#13#10
               + TrGlF('Mailinfo', 'Mail.ignored-globalsize.body',
                    'A mail-message on %s was ignored due to global size-limit.',
                    Server+' (' + User + ')')
               + #13#10 + Msg.Text;
         MsgIsMail := false
      end;

      NOTIFYTYPE_GLOBALSIZE_NOTIFY: begin
         Subject := '[Hamster] '+TrGl('Mailinfo', 'Mail.notice-globalsize.Subject',
                 'Mail notification!');
         Body := '[Hamster]' + #13#10#13#10
               + TrGlF('Mailinfo', 'Mail.notice-globalsize.body',
                    'A mail-message from %s was loaded.',
                    Server+' (' + User + ')')
               + #13#10 + Msg.Text;
         MsgIsMail := false
      end;

      else begin
         Subject := '[Hamster] '+TrGl('Mailinfo', 'Mail.notice-test.Subject',
                 'Notification-test!');
         Body := '[Hamster]' + #13#10#13#10
               + TrGlF('Mailinfo', 'Mail.notice-test.body',
                    'Please ignore, a mail-message on %s was ... just tested.',
                    Server+' (' + User + ')')
               + #13#10
      end;
   end;

   if MsgIsMail then begin
      Body := Body + #13#10
                   + '---------- ' + TrGlF('Mailinfo', 'HeadersAndFirstXLinesFollows',
                   'Headers and first %s lines of message follow', inttostr(Def_NoOfTopLinesToLoad))
                   + ': ----------'
                   + #13#10#13#10 + Msg.Text;
   end;

   for i:=0 to DestUsers.Count-1 do begin
      SendInfoMail( DestUsers[i], Subject, Body );
   end;
end;

procedure TClientPop3.NewDataCallback(DataSize:integer); {.kms}
begin
  if LoadingMailLastProgress<>trunc(DataSize/LoadingMailSize*100) then begin
    LoadingMailLastProgress:=trunc(DataSize/LoadingMailSize*100);
    Log(LOGID_STATUS, LoadingMailLogline+' - '+inttostr(LoadingMailLastProgress)+'%');
  end;
end; {/kms}

Function TClientPOP3.GetNewMails( LeaveOnServer: Boolean;
                                  DefaultDestUser: String;
                                  DefFilterSection: String;
                                  LoadMax: Integer ): boolean;
var  i, OvrLfd, Score, FilterScore, WeWantCur, SizeOfMsg, DefaultDestUserID, LoadedMails: Integer;
     OriginalMsg, MessageID,
        NotifyReason, AccountReason, PostToReason, Scoring,
        s, Info, UIDL: String;
     MayDelete, HadTimeout, RetrDone, DoNotify, IgnoreIt,
        KillIt, UseTopHeaders, UseTOPCmd: Boolean;
     ListUIDL, Overview, DestUsers, DestGroups, DestNotifys: TStringList;
     Parser : TParser;
     Msg : TArticle;
begin
   Result := false;
   if not( State in [csREADY, csLOGIN] ) then exit;
   Info := '[' + Server + ' (' + User + '), POP3] ';
   Log( LOGID_INFO, Info + TrGl(kLog, 'Info.GetNewMails', 'GetNewMail' ) );
   With TIniFile.Create(ServerDir + SRVFILE_INI ) do try
      UseTOPCmd := ReadBool('POP3','UseTOPCommand', true );
   finally Free end;
   if DefaultDestUser='' then DefaultDestUser:=LocalUser;
   if DefaultDestUser='' then DefaultDestUser:=Def_Postmaster;
   If DefFilterSection = '' then DefFilterSection := FilterSection;
   If DefFilterSection = '' then DefFilterSection := '*';
   DefaultDestUserID := CfgAccounts.UserIDOf( DefaultDestUser );
   if DefaultDestUserID=ACTID_INVALID then begin
      Log( LOGID_ERROR, Info + TrGlF(kLog, 'POP3.UnknownDefaultUser',
          'Unknown default user "%s"!', DefaultDestUser ) );
      exit;
   end;
   If not ((CfgAccounts.HasMailbox( DefaultDestUserID ))
       or (CfgAccounts.HasIMAPbox(DefaultDestUserID)))
   then begin
      Log( LOGID_ERROR, Info + TrGlF (kLog, 'POP3.DefaultUserhasNoMailbox',
         'User "%s" has no mailbox!', DefaultDestUser ) );
      exit;
   end;
   // get specific filter-list for current selection
   FilterFile.SelectSections( DefFilterSection );
   // get list of available mails
   POP3.SendCmndGetList( 'LIST', SPECIAL_ACCEPTWEAKENDOFLIST ); // or SPECIAL_ACCEPTNODOTENDOFLIST );
   if not( State in [csREADY, csLOGIN] ) then exit;
   if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
      Log( LOGID_WARN, TrGlF(kLog, 'POP3.GetNewMailsCancelled.NoReplyForLIST',
         'Get new mail cancelled, no reply for LIST (%s)', Server ) );
      try Disconnect except end;
      exit;
   end;
   if POP3.ResultCode<>RESULT_OK then begin
      Log( LOGID_WARN, TrGlF(kLog, 'POP3.GetNewMailsCancelled.FalseReplyForLIST.2',
         'Get new mail failed: LIST (%s)', Server ) );
      Log( LOGID_WARN, Server + ': "' + POP3.ResultCmnd + '" -> "' + POP3.ResultLine + '"' );
      exit;
   end;
   if AllShutDownReq or ThreadControl.Shutdown then exit;
   // load mails
   Overview := TStringList.Create;
   Parser := TParser.Create;
   Msg := TArticle.Create;
   ListUIDL := TStringList.Create;
   DestUsers := TStringList.Create;
   DestGroups := TStringList.Create;
   DestNotifys := TStringList.Create;
   try
      Overview.Text := POP3.ResultList.Text;
      DestUsers.Sorted     := True;
      DestUsers.Duplicates := dupIgnore;
      DestGroups.Sorted     := True;
      DestGroups.Duplicates := dupIgnore;
      DestNotifys.Sorted     := True;
      DestNotifys.Duplicates := dupIgnore;
      HadTimeout := False;

      if OverView.Count=0 then begin
         Log( LOGID_INFO, Info + TrGl(kLog, 'Info.NoMailsToLoad', 'No mail to load.' ) );
      end else begin
         Log( LOGID_INFO, Info + TrGlF(kLog, 'Info.LoadingUpMails',
         'Loading up to %s mail ...', inttostr(Overview.Count) ));
         // Filter out already loaded mails
         if Def_FilterByUIDL then begin
            // Get UIDLs of all available mails
            POP3.SendCmndGetList( 'UIDL', SPECIAL_ACCEPTWEAKENDOFLIST ); // or SPECIAL_ACCEPTNODOTENDOFLIST );
            if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
               Log( LOGID_WARN, TrGlF(kLog, 'POP3.GetNewMailsCancelled.NoReplyForUIDL',
                  'Get new mail cancelled, no reply for UIDL (%s)', Server) );
               try Disconnect except end;
               HadTimeout := True;
            end else begin
               if POP3.ResultCode=RESULT_OK then begin
                  ListUIDL.Text := POP3.ResultList.Text;
                  for i:=0 to ListUIDL.Count-1 do begin
                     Parser.Parse( ListUIDL[i], ' ' );
                     s := TrimWhSpace( Parser.SPart(1,'') ); // UIDL
                     if s='' then begin
                        ListUIDL.Objects[i] := Pointer( -1 );
                        ListUIDL.Strings[i] := ''
                     end else begin
                        ListUIDL.Objects[i] := Pointer( Parser.iPart(0,-1) ); // No.
                        ListUIDL.Strings[i] := Server + '::' + s;
                     end;
                  end;
               end;
            end;
         end;
      end;

      LoadedMails := 0;
      for OvrLfd:=0 to Overview.Count-1 do begin

         If (LoadMax > 0) and (LoadedMails = LoadMax) then break;
         if HadTimeout then break;
         if not( State in [csREADY, csLOGIN] ) then break;
         if AllShutDownReq or ThreadControl.Shutdown then break;

         // LIST: 0:No. 1:Bytes
         Parser.Parse( Overview[OvrLfd], ' ' );
         WeWantCur := Parser.iPart( 0, -1 );
         SizeOfMsg := Parser.iPart( 1,  1 );
         FilterScore := 0;

         if WeWantCur>0 then begin
            Score := 0;
            UIDL  := '';
            MayDelete := False;

            // Check, if already loaded
            for i:=0 to ListUIDL.Count-1 do begin
               if LongInt(ListUIDL.Objects[i])=WeWantCur then begin
                  UIDL := ListUIDL.Strings[i];
                  if UIDL<>'' then begin
                     if MailHistory.ContainsUIDL( UIDL ) then begin
                        Score:=-10000; // already loaded
                        Log( LOGID_DEBUG, Info + '#=' + inttostr(WeWantCur)
                                               + ' UIDL=' + UIDL
                                               + ' Score=(ignored, already known)'  );
                     end;
                  end;
                  break;
               end;
            end;

            // Check, if mail exceeds global size-limits
            if (Score>=0) and (SizeOfMsg>1) then begin
               i := NOTIFYTYPE_GLOBALSIZE_NOTIFY;
               if (Def_MailSizeIgnore>0) and (SizeOfMsg>Def_MailSizeIgnore) then begin
                  Score := -1;
                  i := NOTIFYTYPE_GLOBALSIZE_IGNORE;
               end;
               if (Def_MailSizeKill>0) and (SizeOfMsg>Def_MailSizeKill) then begin
                  Score := -2;
                  i := NOTIFYTYPE_GLOBALSIZE_KILL;
                  MayDelete := True;
               end;
               if (Def_MailSizeNotify>0) and (SizeOfMsg>Def_MailSizeNotify) then begin
                  DoNotify := True;
                  if UIDL<>'' then begin
                     if MailHistory.ContainsUIDL( UIDLMARKER_NOTIFY+UIDL ) then DoNotify:=False; // already done
                  end;
                  if DoNotify then begin
                    {JW} {Postmaster}
                     DestNotifys.Text := Def_Postmaster;
//                     DestNotifys.Text := 'admin';
                    {/JW}
                     Msg.Text := TrGlF('MailInfo', 'SizeOfMailInBytes',
                        'Size of mail: %s bytes', FormatFloat(',0', SizeOfMsg ));
                     SendNotification( i, DestNotifys, '', '', Msg, 0 );
                     if UIDL<>'' then MailHistory.AddUIDL( UIDLMARKER_NOTIFY+UIDL, 0 ); // mark as notified
                  end;
               end;
            end;

            if Score<0 then begin

               Log( LOGID_INFO, Info + TrGlF(kLog, 'Info.IgnoreMailNr',
                  'ignore mail #%s/%s', [inttostr(WeWantCur), inttostr(Overview.Count) ] ));
               // if Score<=-10000 then MayDelete := True;

            end else begin

               // pre-load headers with TOP
               RetrDone := False;
               POP3.ResultCode := 500;
               if FilterFile.TOP_makes_sense and UseTOPCmd then begin
                  Log( LOGID_INFO, Info + TrGlF(kLog, 'Info.RetrieveMailTop',
                      'retrieve mail-top #%s/%s', [inttostr(WeWantCur), inttostr(Overview.Count) ] ) );
                  POP3.SendCmndGetList( 'TOP ' + inttostr(WeWantCur)
                                         + ' ' + inttostr(Def_NoOfTopLinesToLoad), 0 );
                  if not( State in [csREADY, csLOGIN] ) then break;
                  if AllShutDownReq or ThreadControl.Shutdown then break;
                  if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
                     Log( LOGID_WARN, TrGlF(kLog, 'POP3.GetNewMailsCancelled.NoReplyForTOP',
                     'Get new mail cancelled, no reply for TOP (%s)', Server ) );
                     try Disconnect except end;
                     {HadTimeout := True;}
                     break;
                  end;
               end;

               // if TOP is not needed or not supported, retrieve full mail
               if POP3.ResultCode<>RESULT_OK then begin

                  LoadingMailsize:=SizeOfMsg;  {.kms}
                  LoadingMailLastProgress:=-1;
                  LoadingMailLogline:=Info + TrGlF(kLog,'Info.RetrieveMailNr',
                     'retrieve mail #%s/%s (%s bytes)',
                     [inttostr(WeWantCur), inttostr(Overview.Count), FormatFloat(',0', SizeOfMsg) ] );
                  Log( LOGID_INFO, LoadingMailLogline);
                  POP3.SendCmndGetList( 'RETR ' + inttostr(WeWantCur), 0, NewDataCallback ); {/kms}

                  if not( State in [csREADY, csLOGIN] ) then break;
                  if AllShutDownReq or ThreadControl.Shutdown then break;
                  if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
                     Log( LOGID_WARN, TrGlF(kLog, 'POP3.GetNewMailsCancelled.NoReplyForRetr1',
                     'Get mew mail cancelled, no reply for RETR.1 (%s)', Server ) );
                     try Disconnect except end;
                     {HadTimeout := True;}
                     break;
                  end;
                  if POP3.ResultCode<>RESULT_OK then break;
                  RetrDone := True;
                  SizeOfMsg := length( POP3.ResultList.Text );
               end;

               // Action for check mail header
               KillIt := false;
               UseTopHeaders := false;
               Msg.Text := POP3.ResultList.Text;
               If Actions.Exists ( atMailHeader ) then begin
                  Case ModifyMessage( atMailHeader, Msg.FullHeader, s) of
                     mcOriginal: ;
                     mcChanged: begin Msg.FullHeader := s; UseTopHeaders := true end;
                     mcDeleted: Killit := true;
                  end
               end;

               // get results of mail-filters
               If Not KillIt
                  then FilterFile.FilterMail(
                            Msg, SizeOfMsg,
                            DefaultDestUser,
                            IgnoreIt, KillIt,
                            DestNotifys, DestUsers, DestGroups,
                            NotifyReason, AccountReason, PostToReason, Scoring,
                            FilterScore);

               if KillIt then begin // don't load, but delete
                  IgnoreIt  := True;
                  MayDelete := True;
               end;

               if IgnoreIt and (DestNotifys.Count>0) then begin // don't load, but notify
                  DoNotify := True;
                  if UIDL<>'' then begin
                     if MailHistory.ContainsUIDL( UIDLMARKER_NOTIFY+UIDL ) then DoNotify:=False; // already done
                  end;
                  if DoNotify then begin
                     if DestNotifys.Count=0 then DestNotifys.Add( Def_Postmaster);
                     if KillIt then SendNotification(NOTIFYTYPE_KILL, DestNotifys, NotifyReason, Scoring, Msg,SizeOfMsg)
                               else SendNotification(NOTIFYTYPE_IGNORE,DestNotifys, NotifyReason,Scoring, Msg,SizeOfMsg);
                     if UIDL<>'' then MailHistory.AddUIDL( UIDLMARKER_NOTIFY+UIDL, 0 ); // mark as notified
                  end;
               end;

               if not IgnoreIt then begin

                   // retrieve complete mail if not done already
                   if not RetrDone then begin
                      LoadingMailsize:=SizeOfMsg; {.kms}
                      LoadingMailLastProgress:=-1;
                      LoadingMailLogline:=Info + TrGlF(kLog, 'Info.RetrieveMailNr',
                         'retrieve mail #%s/%s (%s bytes)',
                         [inttostr(WeWantCur), inttostr(Overview.Count), FormatFloat(',0', SizeOfMsg) ] );
                      Log( LOGID_INFO, LoadingMailLogline);
                      POP3.SendCmndGetList( 'RETR ' + inttostr(WeWantCur), 0, NewDataCallback ); {/kms}
                      if not( State in [csREADY, csLOGIN] ) then break;
                      if AllShutDownReq or ThreadControl.Shutdown then break;
                      if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
                         Log( LOGID_WARN, TrGlF(kLog, 'POP3.GetNewMailsCancelled.NoReplyForRETR',
                         'Get new mail cancelled, no reply for RETR (%s)', Server ) );
                         try Disconnect except end;
                         {HadTimeout := True;}
                         break;
                      end;
                      If UseTopHeaders then s:=Msg.FullHeader;
                      Msg.Text := POP3.ResultList.Text;
                      If UseTopHeaders then Msg.FullHeader:=s;
                   end;

                   // save result in .log
                   s := DateTimeToTimeStamp( Now ) + ' '
                      + 'Server=' + Server + ',' + Port + ' '
                      + 'Result=' + POP3.ResultLine;
                   HamFileAppendLine ( PATH_LOGS + 'MailIn.log', s );
                   s := ^I + 'From: ' + Msg['From']
                           + ', To: ' + Msg['To']
                           + ', Date: ' + Msg['Date']
                           + ', Lines: ' + Msg['Lines']
                           + ', Subject: ' + Msg['Subject'];
                   HamFileAppendLine ( PATH_LOGS + 'MailIn.log', s );

                   if POP3.ResultCode<>RESULT_OK then break;

                   IncCounter( CntMailNew, 1 );
                   LoadedMails := LoadedMails + 1;

                   // get/add Message-ID
                   {If UseTopHeaders then s:=Msg.FullHeader;
                   Msg.Text := POP3.ResultList.Text;
                   If UseTopHeaders then Msg.FullHeader:=s;}
                   MessageID := Msg['Message-ID:'];
                   if MessageID='' then begin
                      MessageID := MidGenerator( Def_FQDNForMIDs );
                      Msg ['Message-ID:'] := MessageID
                   end;
                   If AccountReason > '' then Msg.AddHeader('X-Hamster-Account-Reason:', 'Matching lines in mailfilt.hst'+AccountReason);
                   If PostToReason > '' then Msg.AddHeader('X-Hamster-PostTo-Reason:', PostToReason);
                   If Scoring > '' then Msg.AddHeader('X-Hamster-Scoring:', 'Matching lines in mailfilt.hst'+Scoring);

                   // add filter-results
                   s := '';
                   for i:=0 to DestUsers.Count-1 do begin
                      if s<>'' then s:=s+',';
                      if pos('@',DestUsers[i])=0 then  //HRR: Add only local accounts
                        s := s + 'account:' + DestUsers[i];
                   end;
                   for i:=0 to DestGroups.Count-1 do begin
                      if s<>'' then s:=s+',';
                      s := s + 'news:' + DestGroups[i];
                   end;
                   if s > '' then Msg.AddHeader('X-Hamster-To:', s);

                   // add info-line
                   Msg[OUR_X_HEADER] :=
                        'Score='    + inttostr(FilterScore) + ' '
                      + 'UIDL='     + UIDLGenerator + ' '
                      + 'Received=' + DateTimeToTimeStamp(Now); // =base for purge
                   // post to newsgroups (local only)
                   OriginalMsg := Msg.Text;
                   for i:=0 to DestGroups.Count-1 do begin
                      if SaveMailToNews( DestGroups[i], OriginalMsg ) then begin
                         if DestUsers.Count=0 then begin
                            MayDelete := True;
                            if UIDL<>'' then MailHistory.AddUIDL( UIDL, 0 ); // mark as loaded
                         end;
                      end else begin
                         Log( LOGID_ERROR, TrGlF(kLog, 'POP3.CouldNotPostMailToGroup',
                           'Could not post mail to %s', DestGroups[i] ) );
                         if DestUsers.Count=0 then DestUsers.Add( Def_Postmaster);
                         MayDelete := False;
                         break;
                      end;
                   end;
                   Msg.Text := OriginalMsg;

                   // store in local mailbox(es)
                   if DestUsers.Count>0 then begin
                       with TRouter.Create do try
                          MailSource:=msIncoming;
                          MailType:=mtPOP;
                          InIp:=POP3.Socket.RemoteAddr.sin_addr.s_Addr;
                          InIpStr:=POP3.Socket.RemoteAddress;
                          MailTo.AddStrings(DestUsers);
                          MailText.Text:=Msg.Text;
                          MailRemoveMids:='';
                          MailAddXHamster:=False;
                          MailAddReceived:=Def_POP3_AddReceived; {JW} {Def_POP3_AddReceived}
                          GenerateMailMID:=GENERATEMID_NEVER; //Msg-Id allready generated
                          MailFrom:=FilterEmailOfFrom(Msg['Return-Path']);
                          if pos('@',MailFrom)=0 then MailFrom:=FilterEmailOfFrom(Msg['Sender:']);
                          if pos('@',MailFrom)=0 then MailFrom:=FilterEmailOfFrom(Msg['From:']);
                          if pos('@',MailFrom)=0 then begin
                            if Def_FQDN<>''
                               then MailFrom:='Hamsterrouter@'+Def_FQDN
                               else MailFrom:='Hamsterrouter@unknown.invalid'
                          end;
                          if not Execute then begin
                            MayDelete:=False;
                            Log( LOGID_ERROR, TrGlF(kLog, 'POP3.CouldNotSaveMail_RouterSays',
                              'Could not save mail, Router says: %s', ResultStr ) )
                           end else begin
                            MayDelete:=True;
                            if UIDL<>'' then MailHistory.AddUIDL( UIDL, 0 ); // mark as loaded
                          end
                       finally
                          Free
                       end
                   end
               end; // if not IgnoreIt...

            end; // if Score...

            if MayDelete and not LeaveOnServer then begin
               // delete mail on server
               POP3.SendCmnd( 'DELE ' + inttostr(WeWantCur), 0, False );
               if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
                  Log( LOGID_WARN, TrGlF(kLog, 'POP3.GetNewMailsCancelled.NoReplyForDELE',
                  'GetNewMail cancelled, no reply for DELE (%s)', Server ) );
                  try Disconnect except end;
                  {HadTimeout := True;}
                  break;
               end;
               if POP3.ResultCode<>RESULT_OK then begin
                  Log( LOGID_ERROR, Info + TrGlF(kLog, 'POP3.MailDeletionFailed',
                       'Mail deletion failed (%s)', Overview[OvrLfd] ) );
                  break;
               end
            end

         end // if WeWantCur>0 then ...
      end;
      MailHistory.SaveToFile;
      Result := LoadedMails > 0
   finally
     {HSR} {IMAP-Folder 03}
      for i := DestUsers.count-1 downto 0 do
        if assigned(pMailToObj(DestUsers.Objects[i])) then
          dispose(pMailToObj(DestUsers.Objects[i]));
      {/HSR}
      DestUsers.Free;
      DestGroups.Free;
      DestNotifys.Free;
      Msg.Free;
      Parser.Free;
      ListUIDL.Free;
      Overview.Free;
   end
end;

{JW} {SSL}
constructor TClientPOP3.Create( AServer, APort, AUser, APass,
                                ALocalUser, AFilterSection: String;
                                ASSLMode, ASSLVerifyLevel: Integer;
                                ASSLCaFile: String );
Var p: Integer;
begin
   inherited Create;
   Server := AServer;
   Port   := APort;
   User   := AUser;
   Pass   := APass;
   LocalUser := ALocalUser;
   FilterSection := AFilterSection;
   {MG}{SSL}
   SSLMode        := ASSLMode;
   SSLVerifyLevel := ASSLVerifyLevel;
   SSLCaFile      := ASSLCaFile;
   {/SSL}
   POP3   := nil;
   ServerDir := IncludeTrailingBackslash ( PATH_SERVER + POP3ServerNameToPath(Server) );
   p := Pos('/', Server); If p > 0 then Delete(Server, p, Length(Server)-p+1);
   FilterFile := TFiltersMail.Create( PATH_BASE + CFGFILE_MFILTER );
end;

destructor TClientPOP3.Destroy;
begin
   if Assigned(POP3) then Disconnect;
   if Assigned(FilterFile) then FilterFile.Free;

   inherited Destroy;
end;

end.
