// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cFiltersBase;

interface

uses SysUtils, Classes, Global, cPCRE, uTools;

type
  TFilterPatternBase = class
    SelectType : Char;    // '+' | '-' | ' '
    IsRegex    : Boolean;
    Pattern    : String;
  end;

  TFilterLineBase = class
    private
    protected
      fLineNo: Integer;
      fComment: String;
      fFilterPatterns: TList;
      DoMimeDecode : Boolean; // '~'?
      LastSetError : String;
      function GetPatternCount: Integer;
      function GetPatternItem( Index: Integer ): TFilterPatternBase;
    public
      property  PatternCount: Integer read GetPatternCount;
      property  PatternItem[ Index: Integer ]: TFilterPatternBase read GetPatternItem;
      function  PatternAdd( Pat: TFilterPatternBase ): Integer;
      procedure PatternDelete( Index: Integer );
      function  SetFilterLine( Const LineNo: Integer; Const FilterLine: String ): Boolean;
         virtual; abstract;
      function  AsString: String; virtual; abstract;
      Property  LastError: String read LastSetError;
      procedure Clear;

      constructor Create;
      destructor Destroy; override;
  end;

  TFiltersBase = class
  private
    protected
      fFilterFilename: String;
      fFilterFile : TStringList;
      fFilterLines: TList;
      RegExSection: TPCRE;
      RegExFilter : TPCRE;
      function GetLinesCount: Integer;
      function GetLinesItem( Index: Integer ): TFilterLineBase;
    public
      property  LinesCount: Integer read GetLinesCount;
      property  LinesItem[ Index: Integer ]: TFilterLineBase read GetLinesItem;

      function  LinesAdd( Const LineNo: Integer; Const LineText: String ): Integer;
         virtual; abstract;
      procedure LinesDelete( Const Index: Integer );
      procedure Clear;

      function  IsFilterLine( Const Line: String ): Boolean; virtual; abstract;
      procedure SelectSections( Const SectionIdentifier: String ); virtual;
      function  SelectedLines: String;

      constructor Create( AFilterFilename: String );
      destructor Destroy; override;
  end;

function MatchSimpleString( TestStr, Pattern: String ): Boolean;
function MatchSimpleNumber( TestStr, Pattern: String ): Boolean;
function MatchSimple( TestStr, Pattern: String ): Boolean;

Procedure MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE; Const Testmode: Boolean;
   Var Result: Boolean; Var Error: String  ); overload;
function MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE ): Boolean; overload;

Function CheckExpire ( Var Filterline, LastSetError: String; Var Expire: TDateTime; Out bError: boolean ): boolean;

implementation

Uses cLogfile, cStdForm;

// --------------------------------------------------------------- Tools ------

function MatchSimpleString( TestStr, Pattern: String ): Boolean;
begin
     if Pattern='*' then begin // '*' matcht alles
        Result := True;
     end else begin
        Result := ( Pos( LowerCase(Pattern), LowerCase(TestStr) ) > 0 );
     end;
end;

function MatchSimpleNumber( TestStr, Pattern: String ): Boolean;
var  iBase, iTest: Integer;
     vgl         : Char;
begin
     try
        if TestStr='' then iBase := 0
                      else iBase := strtoint( TestStr );
        If Length(Pattern)<2 then Abort;
        iTest := strtoint( copy(Pattern,2,9) );
        vgl := Pattern[1];

        Result := False;
        if (vgl='>') and (iBase>iTest) then Result:=True;
        if (vgl='<') and (iBase<iTest) then Result:=True;
        if (vgl='=') and (iBase=iTest) then Result:=True;
     except

        Result := False;
     end;
end;

function MatchSimple( TestStr, Pattern: String ): Boolean;
begin
     if Pattern[1]='%' then begin
        Result := MatchSimpleNumber( TestStr, copy(Pattern, 2, Length(Pattern)-1) );
     end else begin
        if TestStr='' then begin
           if Pattern='*' then Result:=True else Result:=False;
        end else begin
           Result := MatchSimpleString( TestStr, Pattern );
        end;
     end;
end;

Procedure MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE; Const Testmode: Boolean;
   Var Result: Boolean; Var Error: String  ); 
const ONE=0; YES=1; NO=2;
var  NeedOneOf, HaveOneOf: Boolean;
     Matches : Boolean;
     MPatterns, Pattern: String;
     i: Integer;
     SelectType: Char;
begin
     Error := '';
     Result := True;

     MPatterns := Patterns;
     NeedOneOf := False;
     HaveOneOf := False;
     Matches   := False;

     while MPatterns<>'' do begin
        MPatterns := TrimWhSpace( MPatterns );
        i := PosWhSpace( MPatterns );
        if i=0 then begin
           Pattern   := MPatterns;
           MPatterns := '';
        end else begin
           Pattern  := copy( MPatterns, 1, i-1 );
           System.Delete( MPatterns, 1, i );
        end;

        // Pattern:
        //    ['+'|'-']  '{'  regex-pattern   '}'
        //    ['+'|'-'] ['"'] simple-pattern ['"']

        SelectType := ' ';
        if (Pattern>'') and (Pattern[1] in ['+','-']) then begin
           SelectType := Pattern[1];
           System.Delete( Pattern, 1, 1 );
           if Pattern='' then Error := 'Missing pattern after +/-:' + Patterns
        end;

        if (SelectType<>' ') or not(HaveOneOf) then begin
           if copy(Pattern,1,1)='{' then begin
              if Pattern[length(Pattern)]='}' then System.Delete(Pattern,length(Pattern),1);
              System.Delete( Pattern, 1, 1 );
              try
                 Matches := RegEx.Match( PChar(Pattern), PChar(TestStr) );
              except
                 on E: Exception do begin
                    Error := 'Regex-error in {' + Pattern + '}:' + E.Message;
                    Matches := False;
                 end;
              end;
           end else begin
              if copy(Pattern,1,1)='"' then begin
                 System.Delete( Pattern, 1, 1 );
                 if Pattern[length(Pattern)]='"' then System.Delete(Pattern,length(Pattern),1);
              end;
              Matches := MatchSimpleString( TestStr, Pattern );
           end;

           case SelectType of
              '+': if Not TestMode then
                     If not Matches then begin Result:=False; break; end;
              '-': if Not TestMode then
                     If Matches     then begin Result:=False; break; end;
              ' ': begin
                      NeedOneOf := True;
                      if Matches then HaveOneOf:=True;
                   end;
           end;
        end;
     end;

     if NeedOneOf and not HaveOneOf then Result:=False;
end;

function MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE ): Boolean;
Var Fehler: String; ok: Boolean;
begin
   MatchPatterns( TestStr, Patterns, RegEx, false, ok, Fehler);
   Result := ok;
   If Fehler > '' then Log( LOGID_ERROR, Fehler );
end;

// -------------------------------------------------------- TFilterLine ------

function TFilterLineBase.GetPatternCount: Integer;
begin
     Result := fFilterPatterns.Count;
end;

function TFilterLineBase.GetPatternItem( Index: Integer ): TFilterPatternBase;
begin
     if (Index>=0) and (Index<PatternCount) then begin
        Result := fFilterPatterns[ Index ];
     end else begin
        Result := nil;
     end;
end;

function TFilterLineBase.PatternAdd( Pat: TFilterPatternBase ): Integer;
begin
     Result := fFilterPatterns.Add( pat );
end;

procedure TFilterLineBase.PatternDelete( Index: Integer );
begin
     if (Index>=0) and (Index<PatternCount) then begin
        TFilterPatternBase( fFilterPatterns[ Index ] ).Free;
        fFilterPatterns.Delete( Index );
     end;
end;


procedure TFilterLineBase.Clear;
begin
     while PatternCount>0 do PatternDelete( PatternCount-1 );
end;

constructor TFilterLineBase.Create;
begin
     inherited Create;
     fFilterPatterns := TList.Create;
     Clear;
end;

destructor TFilterLineBase.Destroy;
begin
     Clear;
     fFilterPatterns.Free;
     inherited Destroy;
end;

// -------------------------------------------------------- TFiltersBase ------

function TFiltersBase.GetLinesCount: Integer;
begin
     Result := fFilterLines.Count;
end;

function TFiltersBase.GetLinesItem( Index: Integer ): TFilterLineBase;
begin
     if (Index>=0) and (Index<LinesCount) then begin
        Result := fFilterLines[ Index ];
     end else begin
        Result := nil;
     end;
end;

procedure TFiltersBase.LinesDelete( Const Index: Integer );
begin
     if (Index>=0) and (Index<LinesCount) then begin
        TFilterLineBase( fFilterLines[ Index ] ).Free;
        fFilterLines.Delete( Index );
     end;
end;

procedure TFiltersBase.Clear;
begin
     while LinesCount>0 do LinesDelete( LinesCount-1 );
end;

procedure TFiltersBase.SelectSections( Const SectionIdentifier: String );
var  LineNo  : Integer;
     LineText: String;
     GroupOK : Boolean;

   procedure DoLine;
   var  j, k: Integer;
   begin
      if IsFilterLine( LineText ) then begin // filter-line
         if GroupOK then LinesAdd( LineNo, LineText );
         exit;
      end;
      if (LineText>'') and (LineText[1]='[') then begin // group-selection
         System.Delete( LineText, 1, 1 );
         k := 0;
         for j:=1 to length(LineText) do begin
            case LineText[j] of
               '{': inc(k);
               '}': dec(k);
               ']': if (k=0) then begin
                       LineText := copy( LineText, 1, j-1 );
                       break;
                    end;
            end;
         end;
         GroupOK := MatchPatterns( SectionIdentifier, LineText, RegExSection );
         exit;
      end;
      if (LineText > '') and (LineText[1] in ['#',';']) then exit; // comment
      Log( LOGID_WARN, 'Filterfile-line ignored: ' + LineText );
      Log( LOGID_WARN, 'Filterfile: ' + fFilterFilename );
      Log( LOGID_WARN, 'Reason: starts with invalid char' );
   end;

begin
   Clear;
   GroupOK := False;
   for LineNo:=0 to fFilterFile.Count-1 do begin
      LineText := TrimWhSpace( fFilterFile[LineNo] );
      if LineText<>'' then DoLine;
   end;
end;


function TFiltersBase.SelectedLines: String;
var  LineNo: Integer;
begin
     Result := '';
     for LineNo:=0 to LinesCount-1 do begin
        Result := Result + LinesItem[LineNo].AsString + #13#10;
     end;
end;

constructor TFiltersBase.Create( AFilterFilename: String );
var  s  : String;
begin
   inherited Create;

   fFilterFile  := TStringList.Create;
   fFilterLines := TList.Create;
   Clear;

   fFilterFilename := aFilterFilename;
   if fFilterFilename<>'' then begin
      try
         if FileExists2( fFilterFilename ) then begin
            With TFileStream.Create( fFilterFilename,
                                       fmOpenRead or fmShareDenyNone )
            do try
               SetLength(s, Size);
               If Size > 0 then Read( s[1], Size );
               FFilterFile.Text := s;
            finally
               Free;
            end;
         end;
      except
         on E: Exception do begin
            Log( LOGID_WARN, TrGlF(kLog, 'FilterFile.couldnt-load',
               'Couldn''t load filter-file %s', fFilterFilename ) );
            Log( LOGID_WARN, TrGlF(kLog, 'FilterFile.couldnt-load.error',
               'Load-Error: %s', E.Message ) );
         end;
      end;
   end;
   RegExSection := TPCRE.Create( False, PCRE_CASELESS );
   RegExFilter  := TPCRE.Create( False, 0 );
end;

destructor TFiltersBase.Destroy;
begin
   Clear;
   if Assigned(fFilterFile ) then fFilterFile.Free;
   if Assigned(fFilterLines) then fFilterLines.Free;
   if Assigned(RegExFilter ) then RegExFilter.Free;
   if Assigned(RegExSection) then RegExSection.Free;
   inherited Destroy;
end;

Function CheckExpire ( Var Filterline, LastSetError: String; Var Expire: TDateTime; Out bError: boolean ): boolean;
Const kExpire = 'Expire:';
Var j: Integer;
begin
   Result := false;
   bError := true;
   If LowerCase(Copy(FilterLine, 1, Length(kExpire)))=LowerCase(kExpire) then begin
      If Length(Filterline) >= Length(kExpire) + 8 then begin
         For j := 1 to 8 do If Not (Filterline[Length(kExpire)+j] IN['0'..'9']) then begin
            LastSetError:='invalid char "'+Filterline[Length(kExpire)+j]+'" in expire-date';
            Exit
         end;
         System.Delete(Filterline, 1, Length(kExpire));
         try
            Expire := EncodeDate (StrToInt(Copy(Filterline, 1, 4)),
                                  StrToInt(Copy(Filterline, 5, 2)),
                                  StrToInt(Copy(Filterline, 7, 2)))
         except
            LastSetError:='invalid expire-date "'+Copy(Filterline, 1, 8)+'"';
            Exit
         end;
         System.Delete(Filterline, 1, 8);
         Result := true
      end else begin
         LastSetError:='invalid expire, date to short - use "Expire:yyyymmdd"';
         Exit
      end
   end;
   bError := false
end;

end.
