// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cFiltersMail;

interface

uses SysUtils, Classes, Global, cArticle, cFiltersBase, cPCRE;

type
  TFilterPatternMail = class( TFilterPatternBase )
    FilterHeader: String;
    ExtractField: boolean;
  end;

Type
  TMailfilterAction = (mfaINVALID, mfaLOAD, mfaKILL, mfaIGNORE, mfaNOTIFY, mfaNOTIFYOFF,
     mfaDEFAULT, mfaSET, mfaADD, mfaDEL, mfaADDACCOUNTS, mfaPOSTTO, mfaADDDEFAULT,
     mfaSetScore, mfaAddScore);
Const
  MailfilterAction_Names: Array[TMailfilterAction] of String
     = ('#invalid#', 'load', 'kill', 'ignore', 'notify', 'notfiyoff',
         'default', 'set', 'add', 'del', 'addaccounts', 'postto', 'adddefault',
         'setscore', 'addscore');

Type
  TFilterLineMail = class( TFilterLineBase )
    private
      procedure GetHeaderValue(const TestHdr: String; Mail: TArticle; Out s: String);
      procedure GetHeaderValues(const TestHdr: String; Mail: TArticle; Values: TStrings);
    protected
      function MatchesAccounts( Mail: TArticle; var ResultAccounts: String ): Boolean;
      function TestHeaderValue( Const HdrVal: String; Mail: TArticle; RE: TPCRE ): Boolean;
    public
      IsFinal      : Boolean; // '='?
      DoAllHeaders : Boolean; // '*'?
      ActionID     : TMailfilterAction;
      ActionPars   : String;
      DefaultField : String;
      Expire       : TDateTime;
      // Next fields had to be set before executing MatchesMailHeaders, used by GetHeaderValue(s)
      SizeOfMsg, Score: Integer;
      //
      function  MatchesMailHeaders( Mail: TArticle; RE: TPCRE; var ResultAccounts: String ): Boolean;
      function  SetFilterLine( Const LineNo: Integer; Const FilterLine: String ): Boolean; override;
      function  AsString: String; override;
  end;

  TFiltersMail = class( TFiltersBase )
    public
      TOP_makes_sense  : Boolean;
      Filter_bytes_only: Boolean;
      Procedure Test;
      function  LinesAdd( Const LineNo: Integer; Const LineText: String ): Integer; override;
      function  IsFilterLine( Const Line: String ): Boolean; override;
      procedure SelectSections( Const SectionIdentifier: String ); override;
      procedure FilterMail(
         Mail: TArticle; Const SizeOfMsg  : Integer;
         DefaultUser      : String;
         Var ResultIgnore, ResultKillIt : Boolean;
         ResultNotifys, ResultUsers, ResultGroups: TStrings;
         Var NotifyReason, AccountReason, PostToReason, Scoring: String;
         Out Score: Integer);
      Procedure Purge;
  end;

implementation

uses uTools, cAccount, cMailrouter, Config, uEncoding, cLogFile, cStdForm;

const
  ANY_SENDER        = 'Any-Sender:';
  ANY_SENDER_MAX    = 6;
  ANY_SENDER_LIST   : array[0..ANY_SENDER_MAX] of String =
                      ( 'From:', 'Apparently-From:', 'Sender:', 'Reply-To:',
                        'X-Sender', 'Envelope-From:', 'X-Envelope-From:' );

  ANY_RECIPIENT     = 'Any-Recipient:';
  ANY_RECIPIENT_MAX = 5;
  ANY_RECIPIENT_LIST: array[0..ANY_RECIPIENT_MAX] of String =
                      ( 'To:', 'Apparently-To:', 'CC:', 'BCC:',
                        'Envelope-To:', 'X-Envelope-To:' );

procedure ExpandFilterGroups( HdrName: String; TS: TStrings );
var  i: Integer;
begin
   If HdrName = '' then Exit;
   if HdrName[length(HdrName)]<>':' then HdrName:=HdrName+':';
   if CompareText( ANY_SENDER, HdrName )=0 then begin
      for i:=0 to ANY_SENDER_MAX do TS.Add( ANY_SENDER_LIST[i] );
   end else
   if CompareText( ANY_RECIPIENT, HdrName )=0 then begin
      for i:=0 to ANY_RECIPIENT_MAX do TS.Add( ANY_RECIPIENT_LIST[i] );
   end else
   begin
      TS.Add( HdrName )
   end
end;

// ------------------------------------------------------ TFilterLineMail -----

function TFilterLineMail.MatchesAccounts( Mail: TArticle; var ResultAccounts: String ): Boolean;
var  TestHdrs, MailAddrs: TStringList;
     TestHdr, DefStr, s: String;
     CurrHdr, LastPos, UID, LTY, i: Integer;
     bLocal: Boolean;

     procedure AddMailAddress( TestStr: String );
     var  at: Integer;
     begin
          TestStr := TrimWhSpace( TestStr );
          if length(TestStr)<3 then exit;
          if PosWhSpace( TestStr )>0 then exit;

          if TestStr[1]='<' then System.Delete( TestStr, 1, 1 );
          if TestStr[length(TestStr)]='>' then System.Delete( TestStr, length(TestStr), 1 );

          at := Pos( '@', TestStr );
          if (at<2) or (at=length(TestStr)) then exit;

          MailAddrs.Add( TestStr );
     end;

     procedure AddMailAddresses( HdrVal: String );
     // add all strings which look like mail-addresses to accounts-testlist
     var  p: Integer;
          addr: String;
     begin
          addr := '';
          p    := 1;

          while p<=length(HdrVal) do begin
             if HdrVal[p] in [#9,' ','<','>'] then begin
                AddMailAddress( addr );
                addr := '';
             end else begin
                addr := addr + HdrVal[p];
             end;

             inc( p );
          end;

          AddMailAddress( addr );
     end;
begin
   TestHdrs := TStringList.Create;
   MailAddrs := TStringList.Create;
   try
     ExpandFilterGroups( DefaultField, TestHdrs );
     MailAddrs.Sorted := True;
     MailAddrs.Duplicates := dupIgnore;
     for CurrHdr:=0 to TestHdrs.Count-1 do begin
        TestHdr := TestHdrs[CurrHdr];
        LastPos := -1;
        repeat
           DefStr := Mail.HeaderAfterIdx( TestHdr, LastPos );
           If DefStr > '' then begin
              If DoMimeDecode then DefStr:=DecodeHeadervalue(DefStr);
              AddMailAddresses( DefStr )
           end
        until (DefStr = '') or (not DoAllHeaders);
     end;

     for i:=0 to MailAddrs.Count-1 do begin
        if MailAddrs[i]<>'' then begin
           if CfgAccounts.IsLocalMailbox( MailAddrs[i], UID, LTY, bLocal ) then begin
              s := CfgAccounts.Value[ UID, ACTP_USERNAME ];
              if ResultAccounts<>'' then ResultAccounts:=ResultAccounts+',';
              ResultAccounts := ResultAccounts + s;
           end else begin
              Log( LOGID_DEBUG, 'MailFilter.addaccounts: "' + MailAddrs[i] + '" unknown.' );
           end;
        end;
     end;

     if ResultAccounts<>'' then Log( LOGID_DEBUG, 'MailFilter.addaccounts.final: "' + ResultAccounts + '"' );
     Result := ( ResultAccounts <> '' )
   finally
     MailAddrs.Free;
     TestHdrs.Free;
   end
end;

function TFilterLineMail.TestHeaderValue( Const HdrVal: String; Mail: TArticle; RE: TPCRE ): Boolean;
var  Matches, NeedOneOf, HaveOneOf: Boolean;
     PatNo: Integer;
     TestStr: String;
     Pat: TFilterPatternMail;
     Pattern: String;
begin
     Result := True;

     NeedOneOf := False;
     HaveOneOf := False;
     Matches   := False;

     for PatNo:=0 to PatternCount-1 do begin
        Pat := TFilterPatternMail( PatternItem[ PatNo ] );

        If Pat.ExtractField
           then Pattern := Mail[Pat.Pattern]
           else Pattern := Pat.Pattern;
        If Pattern = '' then continue;

        if CompareText( Pat.FilterHeader, DefaultField )=0
           then TestStr := HdrVal
           else GetHeaderValue(Pat.FilterHeader,Mail,TestStr);

        if (Pat.SelectType<>' ') or not(HaveOneOf) then begin
           if Pat.IsRegex then begin
              try
                 RE.OptCompile := PCRE_CASELESS;
                 Matches := RE.Match( PChar(Pattern), PChar(TestStr) );
              except
                 on E: Exception do begin
                    Log( LOGID_ERROR, 'Regex-error in {' + Pattern + '}:' + E.Message );
                    Matches := False;
                 end;
              end;
           end else begin
              Matches := MatchSimple( TestStr, Pattern );
           end;

           case Pat.SelectType of
              '+': if not Matches then begin Result:=False; break; end;
              '-': if Matches     then begin Result:=False; break; end;
              ' ': begin
                      NeedOneOf := True;
                      if Matches then HaveOneOf:=True;
                   end;
           end;
        end;
     end;

     if NeedOneOf and not HaveOneOf then Result:=False;
end;

procedure TFilterLineMail.GetHeaderValue(const TestHdr: String; Mail: TArticle; Out s: String);
begin
   If CompareText(TestHdr, 'Bytes:')=0 then s := IntToStr(SizeOfMsg)
   else If CompareText(TestHdr, 'Score:')=0 then s := IntToStr(Score)
   else If CompareText(TestHdr, 'Top:')=0 then s := Mail.FullBody
   else s := Mail[TestHdr]
end;
Procedure TFilterLineMail.GetHeaderValues(Const TestHdr: String; Mail: TArticle; Values: TStrings);
Var LastPos: Integer; s: String;
begin
   Values.Clear;
   If CompareText(TestHdr, 'Bytes:')=0 then Values.Add(IntToStr(SizeOfMsg))
   else If CompareText(TestHdr, 'Score:')=0 then Values.Add(IntToStr(Score))
   else If CompareText(TestHdr, 'Top:')=0 then Values.Add(Mail.FullBody)
   else If Mail.HeaderExists(TestHdr) then begin
      LastPos := -1;
      Repeat
         s := Mail.HeaderAfterIdx(TestHdr, LastPos);
         If s > '' then begin
            If DoMimeDecode then s:=DecodeHeadervalue(s);
            Values.Add(s)
         end
      until (s='') or (Not DoAllHeaders)
   end
end;

function TFilterLineMail.MatchesMailHeaders( Mail: TArticle; RE: TPCRE;
   var ResultAccounts: String ): Boolean;
var  i, CurrHdr: Integer; TestHdrs, HdrValues: TStringList;
begin
   ResultAccounts := '';
   If ActionID = mfaADDACCOUNTS then begin
      // test given header-name[s] for known mail-accounts
      Result := MatchesAccounts( Mail, ResultAccounts )
   end else begin
      // test given patterns
      Result := Defaultfield = '';
      TestHdrs := TStringList.Create;
      HdrValues:= TStringList.Create;
      try
         ExpandFilterGroups( DefaultField, TestHdrs );
         for CurrHdr:=0 to TestHdrs.Count-1 do begin
            GetHeaderValues(TestHdrs[CurrHdr], Mail, HdrValues);
            For i := 0 to HdrValues.Count-1 do begin
               // test given patterns
               Result := TestHeaderValue( HdrValues[i], Mail, RE ); // test patterns
               If Result then break
            end;
            if Result then break
         end;
      finally
         TestHdrs.Free;
         HdrValues.Free;
      end
   end
end;

Type TVTyp = (vtText, vtField, vtRegex);

function TFilterLineMail.SetFilterLine(Const LineNo: Integer; Const FilterLine: String ): Boolean;
Type TFilterMode = (fmStart, fmCommand, fmCommandExt, fmCommandPar,
            fmDefField1, fmDefField2, fmDefField3,
            fmPattern0, fmPattern1, fmPattern2, fmPatternField, fmPatternText,
            fmExpireDate0, fmExpireDate );
Var Mode: TFilterMode;
    Err, s, TmpFilterHeader: String;
    i, p, Stack: Integer;
    done: Boolean;
    c, TmpSelectType, StopChar: Char;
    pat: TFilterPatternMail;

   Procedure InitPattern;
   begin
      Mode := fmPattern0;
      TmpSelectType     := ' ';
      TmpFilterHeader   := '';
      StopChar          := ' ';
      Stack             := 0
   end;

   Procedure CheckExpire(Const StrDate: String);
   begin
      If Expire > 0 then begin
         Err := 'Only one expire-expression allowed in one line'
      end else
      If Length(StrDate)<>8 then begin
         Err := 'Invalid date in expire-expression - format must be YYYYMMDD'
      end else try
         Expire := EncodeDate(StrToInt(Copy(StrDate, 1, 4)),
                              StrToInt(Copy(StrDate, 5, 2)),
                              StrToInt(Copy(StrDate, 7, 2)));
      except
         Err := 'Invalid date "'+s+'" in expire-expression - format must be YYYYMMDD'
      end
   end;

Const Commentchars = ['#', ';'];
      WhiteSpace = [' ',^I];
      ActionChars = ['A'..'Z', 'a'..'z'];
      HeaderChars = [#33..#57, #59..#126];
Var iMFA: TMailfilterAction;
begin
   Err := '';
   Clear;
   fLineNo      := LineNo;
   fComment     := '';
   IsFinal      := False;
   ActionID     := mfaINVALID;
   ActionPars   := '';
   DoMimeDecode := False;
   DoAllHeaders := False;
   DefaultField := '';
   Expire       := 0;
   Mode := fmStart;
   p := 0;
   For i := 1 to Length(Filterline) + 1 do begin
      If i <= Length(Filterline) then c := Filterline[i] else c := ' ';
      Case Mode of
         // Analysis begins
         fmStart:
         If c IN Commentchars then begin fComment := Copy(Filterline, i, Length(Filterline)); break end
         else If c IN WhiteSpace then
         else If c IN ActionChars then begin p := i; Mode := fmCommand end
         else If c = '=' then begin
            If IsFinal then Err := 'Doubled final-marker "="'
                       else IsFinal := true;
         end else Err := 'Action expected';
         // Check command Part 1/3
         fmCommand: begin
            done := true;
            If c IN Commentchars then begin fComment := Copy(Filterline, i, Length(Filterline)); break end
            else If c IN WhiteSpace then Mode := fmCommandExt
            else if c IN ActionChars then done := false
            else If c = '(' then Mode := fmCommandPar;
            If done then begin
               s := LowerCase(Copy(Filterline, p, i-p));
               ActionID := mfaINVALID;
               For iMFA := Low(TMailfilteraction) to High(TMailfilteraction) do begin
                  If s=Mailfilteraction_Names[iMFA] then begin
                     ActionID := iMFA;
                     break
                  end
               end;
               If ActionID = mfaInvalid then Err := 'Unknown action "'+s+'"';
               p := i+1
            end
         end;
         fmCommandPar: If c = ')' then begin
            ActionPars := TrimWhSpace(Copy(Filterline, p, i-p));
            Mode := fmDefField1
         end;
         fmCommandExt: Case c of
            '(': begin p := i+1; Mode := fmCommandPar end;
            '~': begin DoMimeDecode := true; Mode:=fmDefField2; p:=i+1 end;
            '*': begin DoAllHeaders := true; Mode:=fmDefField3; p:=i+1 end;
            else begin
               If c IN Commentchars then begin fComment := Copy(Filterline, i, Length(Filterline)); break end
               else If c IN WhiteSpace then
               else If c IN HeaderChars then begin Mode:=fmDefField3; p:=i end
               else Err := 'invalid char "'+c+'" after action.'
            end;
         end;
         // -----------------------------------------------------
         fmDefField1: Case c of
            '~': begin DoMimeDecode := true; Mode:=fmDefField2; p:=i+1 end;
            '*': begin DoAllHeaders := true; Mode:=fmDefField3; p:=i+1 end;
            else begin
               If c IN Commentchars then begin fComment := Copy(Filterline, i, Length(Filterline)); break end
               else If c IN WhiteSpace then
               else If c IN HeaderChars then begin Mode:=fmDefField3; p:=i end
               else Err := 'invalid char "'+c+'" after action.'
            end;
         end;
         fmDefField2: Case c of
            '*': begin DoAllHeaders := true; Mode:=fmDefField3; p:=i+1 end;
            else begin
               If c IN Commentchars then begin fComment := Copy(Filterline, i, Length(Filterline)); break end
               else If c IN WhiteSpace then
               else If c IN HeaderChars then begin Mode:=fmDefField3; p:=i end
               else Err := 'invalid char "'+c+'" after action.'
            end
         end;
         fmDefField3: begin
            done := false;
            Case c of
               ':': done := true;
               else begin
                  If c IN Commentchars then begin fComment := Copy(Filterline, i, Length(Filterline)); break end
                  else If c IN WhiteSpace then done := true
                  else If c IN HeaderChars then 
                  else Err := 'invalid char "'+c+'" in def-field'
               end
            end;
            If done then begin
               If (c=':') and (LowerCase(Copy(Filterline, p, i-p))='expire') then begin
                  Mode := fmExpireDate0
               end else begin
                  DefaultField := Copy(Filterline, p, i-p)+':';
                  InitPattern
               end
            end
         end;
         // ----------------------------------------
         fmExpireDate0: If c IN['0'..'9'] then begin
            Mode := fmExpireDate;
            p := i
         end else begin
            Err := 'Invalid expire-expression - date must follow directly after "expire:"'
         end;
         fmExpireDate: begin
            done := false;
            If c IN WhiteSpace then done := true
            else If Not (c IN ['0'..'9']) then begin
               Err := 'Invalid char "'+c+'" in expire-expression'
            end;
            If done then begin
               CheckExpire(Copy(Filterline, p, i-p));
               If DefaultField = ''
                  then Mode := fmDefField1
                  else InitPattern
            end
         end;
         // ----------------------------------------
         fmPattern0: Case c of
            '+', '-': begin TmpSelectType := c; Mode := fmPattern1 end;
            '@': begin p := i+1; Mode := fmPatternField end;
            '$', '"': begin p := i+1; Mode := fmPatternText; StopChar := c end;
            '{': begin p := i+1; Mode := fmPatternText; StopChar := '}' end;
            else begin
               If c IN Commentchars then begin fComment := Copy(Filterline, i, Length(Filterline)); break end
               else If c IN WhiteSpace then
               else begin p := i; Mode := fmPatternText end
            end
         end;
         fmPattern1: Case c of
            '@': begin p := i+1; Mode := fmPatternField end;
            '$', '"': begin p := i+1; Mode := fmPatternText; StopChar := c end;
            '{': begin p := i+1; Mode := fmPatternText; StopChar := '}' end;
            else begin
               If c IN WhiteSpace then
               else begin p := i; Mode := fmPatternText end
            end
         end;
         fmPattern2: Case c of
            '$', '"': begin p := i+1; Mode := fmPatternText; StopChar := c end;
            '{': begin p := i+1; Mode := fmPatternText; StopChar := '}' end;
            else begin
               If c IN WhiteSpace then
               else begin p := i; Mode := fmPatternText end
            end
         end;
         fmPatternField: Case c of
            ':': begin
                    TmpFilterHeader := Copy(Filterline, p, i-p+1);
                    Mode := fmPattern2;
                 end;
            else begin
               If c IN HeaderChars then
               else Err := 'invalid char "'+c+'" in special fieldname'
            end
         end;
         fmPatternText: begin
            done := false;
            If (c IN WhiteSpace) and (StopChar = ' ') then done := true
            else If (c=StopChar) and (Stack<=0) then done := true
            else If StopChar='}' then begin
               If c = '{' then Inc(Stack)
               else If c = '}' then Dec(Stack)
            end;
            If done then begin
               s := Copy(Filterline, p, i-p);
               If (StopChar=' ') and (LowerCase(Copy(s, 1, 7))='expire:')
                  and (TmpSelectType = ' ') and (TmpFilterHeader='')
               then begin
                  CheckExpire(Copy(s, 8, Length(s)-7))
               end else
               If (StopChar = ' ') and (s<>'*') and (Copy(s,1,1)<>'%') then begin
                  Err := 'Missing "" around text-pattern "'+s+'"'
               end else begin
                  pat := TFilterPatternMail.Create;
                  pat.SelectType   := TmpSelectType;
                  pat.IsRegex      := StopChar = '}';
                  pat.ExtractField := StopChar = '$';
                  pat.Pattern      := s;
                  If TmpFilterHeader = ''
                     then pat.FilterHeader := DefaultField
                     else pat.FilterHeader := TmpFilterHeader;
                  PatternAdd( pat );
               end;
               InitPattern
            end
         end;
      end;
      If Err>'' then break
   end;
   If Err = '' then Case Mode of
      fmStart: If IsFinal then Err := 'Missing action after "="';
      fmCommandExt, fmDefField1, fmPattern0: ;
      fmCommand: Err := 'Internal error, parser ended in impossible state #'+IntToStr(Ord(fmCommand));
      fmCommandPar: Err := 'Missing ")" to end action-parameter';
      fmDefField2, fmDefField3: Err := 'Missing default field';
      fmPattern1, fmPattern2, fmPatternText: Err := 'Missing pattern';
      fmPatternField: Err := 'Missing end of pattern-field';
      fmExpireDate0, fmExpireDate: Err := 'Missing end of expire-rule';
      else Err := 'Internal error, no defined reaction for end-state #'+IntToStr(Ord(fmCommand));
   end;
   If (Err = '') then begin
      If ActionID=mfaADDACCOUNTS then begin
         If DefaultField = '' then Err := 'Missing field containing accounts'
         else If fFilterPatterns.Count>0 then Err := 'No pattern allowed for this type of action';
      end else begin
         If (DefaultField > '') and (fFilterPatterns.Count=0) then Err := 'Default-field defined without any pattern'
      end
   end;
   Result := Err = '';
   If Result
      then LastSetError := ''
      else LastSetError := '"'+Copy(Filterline,1,i)+'" => '+Err
end;

function TFilterLineMail.AsString: String;
var  Pat  : TFilterPatternMail;
     PatNo: Integer;
begin
     Result := IntToStr(fLineNo+1)+': ';

     if IsFinal then Result:=Result+'=';
     Result := Result + MailfilterAction_names[ActionID];
     Result := Result + '(' + ActionPars + ')';

     if DefaultField<>'' then begin
        Result := Result + ' ';
        if DoMimeDecode then Result:=Result+'~';
        if DoAllHeaders then Result:=Result+'*';
        Result := Result + DefaultField;
        for PatNo:=0 to PatternCount-1 do begin
           Pat := TFilterPatternMail( PatternItem[ PatNo ] );
           Result := Result + ' ';
           if Pat.SelectType in ['+','-'] then Result:=Result+Pat.SelectType;
           if CompareText( Pat.FilterHeader, DefaultField )<>0 then begin
              Result := Result + '@' + Pat.FilterHeader
           end;
           if Pat.IsRegEx then begin
              Result := Result + '{' + Pat.Pattern + '}'
           end else begin
              if Copy(Pat.Pattern, 1, 1)<>'%' then begin
                 Result := Result + '"' + Pat.Pattern + '"';
              end else begin
                 Result := Result + Pat.Pattern;
              end;
           end;
        end;
     end;
     If fComment > '' then Result := Result + ' '+fComment
end;

// --------------------------------------------------------- TFiltersMail -----

function TFiltersMail.LinesAdd( Const LineNo: Integer; Const LineText: String ): Integer;
var  lin: TFilterLineMail;
begin
     lin := TFilterLineMail.Create;
     if lin.SetFilterLine( LineNo, LineText ) then begin
        Result := fFilterLines.Add( lin );
     end else begin
        Log( LOGID_WARN, 'Filter-line ignored: ' + LineText );
        Log( LOGID_WARN, 'Reason: ' + lin.LastSetError );
        lin.Free;
        Result := -1;
     end;
end;

function TFiltersMail.IsFilterLine( Const Line: String ): Boolean;
begin
   Result := (Line>'') and (Line[1] in ['=','a'..'z','A'..'Z'])
end;

procedure TFiltersMail.SelectSections( Const SectionIdentifier: String );
var  LineNo: Integer;
     Line  : TFilterLineMail;
begin
   inherited SelectSections( SectionIdentifier );
   TOP_makes_sense := False;
   for LineNo:=0 to LinesCount-1 do begin
      Line := TFilterLineMail( LinesItem[ LineNo ] );
      if Line.ActionID in [mfaKILL, mfaIGNORE] then begin
         TOP_makes_sense := True;
         break;
      end;
   end;
end;

procedure TFiltersMail.FilterMail(
   Mail: TArticle; Const SizeOfMsg  : Integer;
   DefaultUser      : String;
   Var ResultIgnore, ResultKillIt : Boolean;
   ResultNotifys, ResultUsers, ResultGroups: TStrings;
   Var NotifyReason, AccountReason, PostToReason, Scoring: String;
   Out Score: Integer);

   Function ParseDest(s : string; var Usr, Folder, Flags : string) : Boolean;
   var i : integer;
   begin
     s := StringReplace(s, '/', '\', [rfReplaceAll, rfIgnoreCase]);
     i := pos(':', s);  //Parse Flags out
     if (i>0) then begin
        Flags := copy(s, i+1, length(s));
        s := trim(copy(s, 1, i-1));
     end else begin
        Flags := ''
     end;

     i := pos('\', s);  //Parse Folders out
     if (i>0) then begin
        Folder := copy(s, i+1, length(s));
        s := trim(copy(s, 1, i-1));
     end else begin
        Folder := ''
     end;

     Usr    := trim(s); //Rest is user-part
     Result := (Usr<>'')
   end;

   Procedure ClearResultUsers;
   Var i: integer;
       IMAP_MTObj: PMailToObj;
   begin
      {HSR} {IMAP-Folder 03}
      For i := 0 to ResultUsers.count-1 do begin
         IMAP_MTObj := pMailToObj(ResultUsers.Objects[i]);
         If Assigned(IMAP_MTObj) then Dispose(IMAP_MTObj)
      end;
      {/HSR}
      ResultUsers.Clear
   end;

var  LineNo, i, k  : Integer;
     Line          : TFilterLineMail;
     Parser        : TParser;
     ResAccounts, s: String;
     WantNotify    : Boolean;
     IMAP_MTObj    : pMailToObj;
     Usr, Folder, Flags : String;
begin
   ResultIgnore := False;
   ResultKillIt := False;
   WantNotify   := False;
   ResultNotifys.Clear;
   ClearResultUsers;
   ResultGroups.Clear;
   NotifyReason := '';
   AccountReason := '';
   PostToReason := '';
   Scoring := '';
   Score := 0;
   Parser := TParser.Create;
   try
     if DefaultUser='' then DefaultUser:=Def_Postmaster;
     for LineNo:=0 to LinesCount-1 do begin
        Line := TFilterLineMail( LinesItem[ LineNo ] );
        Line.SizeOfMsg := SizeOfMsg;
        Line.Score := Score;
        if Line.MatchesMailHeaders( Mail, RegexFilter, ResAccounts ) then begin
           Log( LOGID_DEBUG, 'MailFilter (+): ' + Line.AsString );
           case Line.ActionID of
              // undo a previous kill/ignore
              mfaLOAD: begin
                 ResultIgnore := False;
                 ResultKillIt := False;
                 NotifyReason := '';
              end;
              // don't load, but delete
              mfaKILL: begin
                 ResultKillIt := True;
                 NotifyReason := NotifyReason + '-> ' + Line.AsString + #13#10;
              end;
              // don't load, don't delete
              mfaIGNORE: begin
                 ResultKillIt := False;
                 ResultIgnore := True;
                 NotifyReason := NotifyReason + '-> ' + Line.AsString + #13#10;
              end;
              // notify if "killed" or "ignored"
              mfaNOTIFY: begin
                 WantNotify := True;
                 ResultNotifys.Clear;
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if s<>'' then ResultNotifys.Add( s );
                 end;
              end;
              // do NOT notify anyone
              mfaNOTIFYOFF: begin   // JAWO 21.09.01 (Notify none)
                 WantNotify := false;
                 ResultNotifys.Clear;
              end;
              // new default-user
              mfaDEFAULT: begin
                 DefaultUser := Line.ActionPars;
              end;
              // set/start new recipient-list
              mfaSET: begin
                 AccountReason := AccountReason + #13#10+' '+Line.AsString;
                 ClearResultUsers;
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if ParseDest(s, Usr, Folder, Flags) then begin
                      new(IMAP_MTObj);
                      IMAP_MTObj^.IMAPFolder := Folder;
                      IMAP_MTObj^.IMAPFlags  := Flags;
                      ResultUsers.AddObject( Usr, Pointer(IMAP_MTObj));
                    end
                 end;
              end;
              // add to recipient-list
              mfaADD: begin
                 AccountReason := AccountReason + #13#10+' '+Line.AsString;
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if ParseDest(s, Usr, Folder, Flags) then begin
                       if (ResultUsers.IndexOf( Usr ) < 0) then begin
                          new(IMAP_MTObj);
                          IMAP_MTObj^.IMAPFolder := Folder;
                          IMAP_MTObj^.IMAPFlags  := Flags;
                          ResultUsers.AddObject( Usr, tObject(IMAP_MTObj))
                       end
                    end
                 end
              end;
              // delete from recipient-list
              mfaDEL: begin
                 AccountReason := AccountReason + #13#10+' '+Line.AsString;
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    ParseDest(s, Usr, Folder, Flags);
                    k := ResultUsers.IndexOf( Usr );
                    if k>=0 then begin
                       if assigned(pMailToObj(ResultUsers.Objects[k])) then
                         dispose(pMailToObj(ResultUsers.Objects[k]));
                       ResultUsers.Delete( k )
                    end;
                 end;
              end;
              // add matching accounts to recipient-list
              mfaADDACCOUNTS: begin
                 AccountReason := AccountReason + #13#10+' '+Line.AsString;
                 Parser.Parse( ResAccounts, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if ParseDest(s, Usr, Folder, Flags) then begin
                       new(IMAP_MTObj);
                       IMAP_MTObj^.IMAPFolder := Folder;
                       IMAP_MTObj^.IMAPFlags  := Flags;
                       ResultUsers.AddObject( Usr, tObject(IMAP_MTObj));
                    end
                 end
              end;
              // add matching accounts to recipient-list
              mfaADDDEFAULT: begin
                 AccountReason := AccountReason + #13#10+' '+Line.AsString;
                 ResultUsers.Add( DefaultUser  );
              end;
              // post to newsgroups
              mfaPOSTTO: begin
                 PostToReason := Line.AsString;
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if s<>'' then ResultGroups.Add( s );
                 end;
              end;
              // Scoring
              mfaSETSCORE: begin
                 Score := StrToInt(Line.ActionPars);
                 Scoring := Scoring + #13#10+' '+Line.AsString
              end;
              mfaADDSCORE: begin
                 Score := Score + StrToInt(Line.ActionPars);
                 Scoring := Scoring + #13#10+' '+Line.AsString
              end;
           end;
           if Line.IsFinal then break; // line preceded with "="
        end else begin
           {Log( LOGID_FULL, 'MailFilter (-): ' + Line.AsString );}
        end
     end;
     // use default-user, if no recipients/newsgroups were selected
     if (ResultUsers.Count=0) and (ResultGroups.Count=0) then begin
        if DefaultUser<>'' then begin
           Parser.Parse( DefaultUser, ',' );
           for i:=0 to Parser.Count-1 do begin
              s := TrimWhSpace( Parser.sPart( i, '' ) );
              if s<>'' then ResultUsers.Add( s );
           end;
        end;
     end;

     // fill list of notification-recipients if none were given with "notify()"
     if WantNotify and (ResultIgnore or ResultKillIt) then begin
        // send notifications to determined recipients ...
        if ResultNotifys.Count=0 then ResultNotifys.Text:=ResultUsers.Text;
        // .. or finally to "postmaster"
        if ResultNotifys.Count=0 then ResultNotifys.Add( Def_Postmaster );
     end
   finally
     Parser.Free
   end
end;

procedure TFiltersMail.Purge;
Var ok: Boolean; LineNo: Integer; LineText: String;
begin
   ok := false;
   for LineNo:=fFilterFile.Count-1 downto 0 do begin
      LineText := TrimWhSpace( fFilterFile[LineNo] );
      if LineText > '' then If IsFilterLine( LineText ) then begin
         With TFilterLineMail.Create do try
            If SetFilterLine( LineNo, LineText ) then begin
               If (Expire > 0) and (Expire <= Now) then begin
                  If Def_Mail_ExpiredFilterentries_Delete then begin
                     fFilterFile.Delete (LineNo);
                     LogFile.Add ( LOGID_INFO, TrGlF(kLog, 'Scorefile.ExpiredEntry.Deleted',
                        'Expired Scorefile-Entry deleted: "%s"', LineText))
                  end else begin
                     fFilterFile[LineNo] := '# Expired! # '+LineText;
                     LogFile.Add ( LOGID_INFO, TrGlF(kLog, 'Scorefile.ExpiredEntry.commented',
                        'Expired Scorefile-Entry commented out: "%s"', LineText))
                  end;
                  ok := true
               end
            end
         finally Free end
      end
   end;
   If ok then fFilterFile.SaveToFile ( fFilterFilename );
end;

procedure TFiltersMail.Test;
var  LineNo, Counter: Integer;
     Line  : TFilterLineMail;
     LineText: String;
begin
   If fFilterFile.Count = 0 then Exit;
   Log( LOGID_SYSTEM, 'Mail-filter-file '+fFilterFilename+': Testing' );
   Line := TFilterLineMail.Create;
   Counter := 0;
   try
      for LineNo:=0 to fFilterFile.Count-1 do begin
         LineText := TrimWhSpace( fFilterFile[LineNo] );
         if IsFilterLine( LineText ) then begin
            if Not Line.SetFilterLine( LineNo, LineText ) then begin
               Log( LOGID_WARN, 'Error in line '+IntToStr(LineNo+1)+' of Mail-filter-file: ' + LineText );
               Log( LOGID_WARN, 'Error description: ' + line.LastSetError );
               Inc(Counter)
            end
         end else
         If (LineText>'') and (Not (LineText[1] IN['[', ';', '#'])) then begin
            Log( LOGID_WARN, 'Error in line '+IntToStr(LineNo+1)+' of Mail-filter-file: ' + LineText );
            Log( LOGID_WARN, 'Error description: No comment, no section and no action');
            Inc(Counter)
         end
      end
   finally
      Line.free
   end;
   If Counter = 0
      then Log( LOGID_SYSTEM, 'Mail-filter-file '+fFilterFilename+': Test OK')
      else Log( LOGID_SYSTEM, 'Mail-filter-file '+fFilterFilename+': '+IntToStr(Counter)+' errors found');
end;

end.
