// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cHistoryNews;

interface

uses SysUtils, Classes, Windows, uTools, cChunkRc;

const MD5HashLen = 16;
type  MD5Hash = array[1..MD5HashLen] of Char;

type
  TInfoProc = Procedure (Const Info: String);
  TNewsHistoryRecords = class( TChunkedRecords )
    private
      //fLoaded: Boolean;
      procedure MID2Hash( MID: String; out Hash: MD5Hash );
    public
      function  ContainsMID( MID: String ) : Boolean;
      function  AddEntryDupes( MID: String; GroupHash, ArtNo: LongInt;
                          PurgeBase: LongInt ): Boolean;
      function  AddEntryFirst( MID: String; GroupHash, ArtNo: LongInt;
                               PurgeBase: LongInt ): Boolean;
      procedure RemoveEntry( MID: String; GroupHash, ArtNo: LongInt );

      function  LocateMID( MID: String; out Groupname: String; out ArtNo: Integer ): Boolean;
      function  LocateMIDInGroup(MID, Groupname: String; out ArtNo: Integer): Boolean;

      procedure Rebuild (Const InfoProc: TInfoProc = NIL);
      procedure Purge;
      procedure PurgeReset( GrpHdl: LongInt );// JAWO 12.08.01 (Reset Group)

      procedure LoadFromFile; override;

      constructor Create( AFilePath: String; AChunkBits: Integer );
      destructor Destroy; override;
  end;

implementation

uses Global, Config, cArticle, cArtFiles, uMD5, uCRC32, dSplash, uDateTime,
     cStdForm, cLogFile;


// ------------------------------------------------------ THistoryRecords -----

const
  HISTORY_FILEBASE = 'Hist';
  HISTORY_FILEEXT  = '.dat';

  HISTKEYLEN_MID       = MD5HashLen;                          // multiple for crossposts
  HISTKEYLEN_MIDGRP    = HISTKEYLEN_MID    + sizeof(LongInt); // multiple after 'import -ih'
  HISTKEYLEN_MIDGRPART = HISTKEYLEN_MIDGRP + sizeof(LongInt); // always unique

type
  PHistoryEntry = ^THistoryEntry;
  THistoryEntry = record
    HashMid  : MD5Hash; //  16
    HashGrp  : LongInt; //   4
    ArtNo    : LongInt; //   4
    PurgeBase: LongInt; //   4
  end;                  // =28 Bytes

procedure TNewsHistoryRecords.MID2Hash( MID: String; out Hash: MD5Hash );
var  s: String;
begin
     s := MD5ofStr( MID );
     System.Move( s[1], Hash[1], MD5HashLen );
end;

function TNewsHistoryRecords.ContainsMID( MID: String ) : Boolean;
var  HE: THistoryEntry;
begin
     //If Not fLoaded then LoadFromFile;
     FillChar( HE, sizeof(HE), 0 );
     MID2Hash( MID, HE.HashMid );
     Result := ContainsKey( HE, HISTKEYLEN_MID );
end;

function TNewsHistoryRecords.LocateMID( MID: String;
                                    out Groupname: String;
                                    out ArtNo: Integer ): Boolean;
var  HE: THistoryEntry;
     Chunk: Byte;
     GroupHash: LongInt;
     Index, i: Integer;
begin
   //If Not fLoaded then LoadFromFile;
   Result := False;

   FillChar( HE, sizeof(HE), 0 );
   MID2Hash( MID, HE.HashMid );

   Chunk := ChunkOf( HE );
   Enter( Chunk );
   try
      Index := RecKeyIndexOf( Chunk, HISTKEYLEN_MID, HE );

      if Index>=0 then begin
         RecGet( Chunk, Index, HE );
         GroupHash := HE.HashGrp;
         ArtNo     := HE.ArtNo;

         Groupname := '';
         for i:=0 to CfgHamster.ActiveCount-1 do begin
            if StrToCRC32(LowerCase(CfgHamster.ActiveName[i]))=GroupHash then begin
               Groupname := CfgHamster.ActiveName[i];
               break;
            end;
         end;
         if Groupname<>'' then begin
            // calling program has to check further,
            // if Groupname:ArtNo is *still* valid
            Result := True;
         end;
      end;

   finally Leave(Chunk) end;
end;


function TNewsHistoryRecords.LocateMIDInGroup(  MID, Groupname: String;
   out ArtNo: Integer ): Boolean;
var  HE: THistoryEntry;
     Chunk: Byte;
     Index: Integer;
begin
   //If Not fLoaded then LoadFromFile;
   Result := False;
   FillChar( HE, sizeof(HE), 0 );
   MID2Hash( MID, HE.HashMid );
   HE.HashGrp := StrToCRC32(lowercase(Groupname));
   Chunk := ChunkOf( HE );
   Enter( Chunk );
   try
      Index := RecKeyIndexOf( Chunk, HISTKEYLEN_MIDGRP, HE );
      if Index>=0 then begin
         RecGet( Chunk, Index, HE );
         ArtNo     := HE.ArtNo;
         Result := true
      end
   finally Leave(Chunk) end;
end;

function TNewsHistoryRecords.AddEntryDupes( MID: String;
                                        GroupHash, ArtNo: LongInt;
                                        PurgeBase: LongInt ): Boolean;
var  HE: THistoryEntry;
begin
     //If Not fLoaded then LoadFromFile;
     FillChar( HE, sizeof(HE), 0 );
     MID2Hash( MID, HE.HashMid );
     HE.HashGrp := GroupHash;
     HE.ArtNo   := ArtNo;

     if ContainsKey( HE, HISTKEYLEN_MIDGRPART ) then begin // unique?
        Result := False;
     end else begin
        Result := True;
        if PurgeBase=0 then PurgeBase:=DateTimeToUnixTime(Now);
        HE.PurgeBase := PurgeBase;
        Add( HE );
     end;
end;

function TNewsHistoryRecords.AddEntryFirst( MID: String;
                                        GroupHash, ArtNo: LongInt;
                                        PurgeBase: LongInt ): Boolean;
begin
     //If Not fLoaded then LoadFromFile;
     if ContainsMid( MID ) then begin
        Result := False;
     end else begin
        Result := AddEntryDupes( MID, GroupHash, ArtNo, PurgeBase );
     end;
end;

procedure TNewsHistoryRecords.RemoveEntry( MID: String; GroupHash, ArtNo: LongInt );
var  HE: THistoryEntry;
begin
     //If Not fLoaded then LoadFromFile;
     FillChar( HE, sizeof(HE), 0 );
     MID2Hash( MID, HE.HashMid );
     HE.HashGrp := GroupHash;
     HE.ArtNo   := ArtNo;
     RemoveKey( HE, HISTKEYLEN_MIDGRPART );
end;

procedure TNewsHistoryRecords.Purge;
var  Idx : Integer;
     Base: TDateTime;
     Days: LongInt;
     PurgeCount: Integer;
     HE: THistoryEntry;
     Chunk: Byte;
begin
     //If Not fLoaded then LoadFromFile;
     SaveToFile;
     PurgeCount := 0;

     for Chunk:=0 to CHUNK_MAX do begin
        Enter( Chunk );
        try
           Idx := ChunkCount(Chunk) - 1;
           while Idx>=0 do begin
              RecGet( Chunk, Idx, HE );
              Base := UnixTimeToDateTime( HE.PurgeBase );
              Days := Trunc( Now - Base );

              if (Days>Def_Purge_History_KeepDays) and (Def_Purge_History_KeepDays>0) then begin
                 RemoveKey( HE, HISTKEYLEN_MIDGRPART );
                 inc( PurgeCount );
              end;

              dec( Idx );
           end;
        finally Leave(Chunk) end;
     end;

     SaveToFile;

     Log( LOGID_INFO, TrGlF(kLog, 'Info.PurgeHistory.Result',
        'Purge History (%sd): %s entries purged.',
        [inttostr(Def_Purge_History_KeepDays), inttostr(PurgeCount)]));
end;

procedure TNewsHistoryRecords.Rebuild (Const InfoProc: TInfoProc = NIL);

   Function AddEntry(MID: String; GroupHash, ArtNo, PurgeBase: Integer): Boolean;
   var  HE: THistoryEntry;
   begin
      FillChar( HE, sizeof(HE), 0 );
      MID2Hash( MID, HE.HashMid );
      HE.HashGrp := GroupHash;
      HE.ArtNo   := ArtNo;
      Result := True;
      if PurgeBase=0 then PurgeBase:=DateTimeToUnixTime(Now);
      HE.PurgeBase := PurgeBase;
      Add( HE );
   end;

var  LfdGrp, LfdArt, ArtMin, ArtMax: Integer;
     GrpHdl : LongInt;
     Article: TArticle;
     GrpHash: LongInt;
     MessageID, s: String;
begin
     //If Not fLoaded then LoadFromFile;
     try
        try
           Sorted := false;

           Log( LOGID_SYSTEM, TrGl(kLog, 'Rebuild.History', 'Rebuild history ...') );
           Clear;

           Article := TArticle.Create;
           try
              for LfdGrp:=0 to CfgHamster.ActiveCount-1 do begin
                 GrpHdl := ArticleBase.Open( CfgHamster.ActiveName[LfdGrp] );
                 if GrpHdl>=0 then begin
                    GrpHash := StrToCRC32( LowerCase(ArticleBase.Name[GrpHdl]) );
                    ArtMin  := ArticleBase.LocalMin[GrpHdl];
                    ArtMax  := ArticleBase.LocalMax[GrpHdl];

                    if (ArtMin>0) and (ArtMax>0) then begin
                       s := TrGlF(kLog, 'Info.RebuildHistory.GroupInfo', 'Rebuild history: %s, %s articles',
                            [CfgHamster.ActiveName[LfdGrp], inttostr(ArticleBase.Count[GrpHdl])]);
                       Log( LOGID_INFO, s);
                       If Assigned(InfoProc) then begin
                          s := TrGlF(kLog, 'Info.RebuildHistory.GroupInfoCRLF', 'Rebuild history:^m%s (%s articles)',
                               [CfgHamster.ActiveName[LfdGrp], inttostr(ArticleBase.Count[GrpHdl])]);
                          InfoProc (s)
                       end;
                       for LfdArt:=ArtMin to ArtMax do begin
                          try
                             Article.Text := ArticleBase.ReadArticle( GrpHdl, LfdArt );
                          except
                             on E:Exception do begin
                                Log( LOGID_ERROR, 'Error in History.R.RA: ' + E.Message );
                                Log( LOGID_ERROR, 'Error at: ' + CfgHamster.ActiveName[LfdGrp] + ':' + inttostr(LfdArt) );
                             end;
                          end;
                          try
                             if Article.Text<>'' then begin
                                MessageID := Trim( Article.Header['Message-ID:'] );
                                if MessageID<>'' then begin
                                   AddEntry( MessageID, GrpHash, LfdArt,
                                             DateTimeToUnixTime(Article.GetReceivedDT) );
                                end;
                             end;
                          except
                             on E:Exception do Log( LOGID_ERROR, 'Error in History.R.AMOR: ' + E.Message );
                          end;
                       end;
                    end;
                    ArticleBase.Close( GrpHdl );
                 end else begin
                    Log( LOGID_ERROR, 'Cannot open ' + CfgHamster.ActiveName[LfdGrp] );
                 end;
              end
           finally
              Article.Free
           end;
           // if Count=0 then begin
              // AddMIDOnRebuild( '<dummy@mid>', DT2UnixTime( Now, 0 ), 0, 0 );
           // end;

           Log( LOGID_INFO, TrGl(kLog, 'Info.RebuildHistory.Sort', 'Sort history ... ' ) );
        finally
           Sorted := true
        end;

        Log( LOGID_INFO, TrGl(kLog, 'Info.RebuildHistory.Save', 'Save rebuilt history ... ' ) );
        SaveToFile;

        Log( LOGID_SYSTEM, TrGlF(kLog, 'Info.RebuildHistory.Result',
            'History rebuilt (%s articles).', [inttostr(Count)]) );

     except
        on E:Exception do Log( LOGID_ERROR, TrGlF(kLog, 'Rebuild.History.Exception',
           'Error in History.Rebuild: %s', E.Message ) )
     end;
end;

procedure TNewsHistoryRecords.LoadFromFile;
var  Chunk      : Byte;
     AutoRebuild: Boolean;
begin

     AutoRebuild := False;
     for Chunk:=0 to CHUNK_MAX do begin
        if not FileExists2( ChunkFilename(Chunk) ) then begin
           AutoRebuild := True;
           break;
        end;
     end;

     if AutoRebuild then begin
        Log( LOGID_WARN, TrGl(kLog, 'Rebuild.WarningInProgress', 'Rebuilding history-file!') );
        SplashOn( TrGl(kMessages, 'History_rebuild_wait', 'History will be rebuild.^MPlease wait!') );
        Rebuild ( SplashOn );
        SplashOff;
     end else begin
        inherited LoadFromFile;
     end;

     //fLoaded := true

end;

constructor TNewsHistoryRecords.Create( AFilePath: String; AChunkBits: Integer );
begin
     if AFilePath='' then AFilePath:='.\';
     AFilePath := ExpandFilename( AFilePath );
     if AFilePath[length(AFilePath)]<>'\' then AFilePath:=AFilePath+'\';

     inherited Create( AFilePath + HISTORY_FILEBASE, HISTORY_FILEEXT,
                       AChunkBits, sizeof(THistoryEntry), True );
     //fLoaded := false
     LoadFromFile
end;

destructor TNewsHistoryRecords.Destroy;
begin
   SaveToFile;
   inherited Destroy;
end;

procedure TNewsHistoryRecords.PurgeReset(GrpHdl: Integer);
var  Idx : Integer;
     PurgeCount: Integer;
     HE: THistoryEntry;
     Chunk: Byte;
     GrpHash: LongInt;
begin
     SaveToFile;
     PurgeCount := 0;
     if GrpHdl>=0 then begin
        GrpHash := StrToCRC32( LowerCase(ArticleBase.Name[GrpHdl]) );

        for Chunk:=0 to CHUNK_MAX do begin
           Enter( Chunk );
           try
              Idx := ChunkCount(Chunk) - 1;
              while Idx>=0 do begin
                 RecGet( Chunk, Idx, HE );

                 if (HE.HashGrp=GrpHash) then begin
                    RemoveKey( HE, HISTKEYLEN_MIDGRPART );
                    inc( PurgeCount );
                 end;

                 dec( Idx );
              end;
           finally Leave(Chunk) end;
        end;

        SaveToFile;
     end;

     Log( LOGID_INFO, TrGlF(kLog, 'Info.PurgeHistory.Result',
        'Purge History (%sd): %s entries purged.', ['-', inttostr(PurgeCount)]));
        
end;

end.

