// ============================================================================
// Basic script-engine for "Hamster-scripts (*.hsc)"
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cHscEngine; // Basic script-engine for "Hamster-scripts (*.hsc)"

// ----------------------------------------------------------------------------
// Contains a basic script-engine with accompanying tools and classes. It has
// some interfaces to redirect output and to enhance the script-language with
// application-specific functions.
// These interfaces are used in Hamster's "tScript.pas" to redirect output to
// the log-window and to implement its various Ham*-functions.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows, uTools, cResControl, cStdForm;

type
   THscEngine = class;

   THscObject = class
      private
         FEngine   : THscEngine;
         FLastError: Integer;

      public
         property Engine: THscEngine read FEngine;

         function SelfError: Boolean;
         function AnyError : Boolean;
         procedure Error( const AErrNum: Integer; const AErrMsg: String );

         constructor Create( const AEngine: THscEngine );
   end;

   THscVarTypes = ( hvtEmpty, hvtInteger, hvtString );

   EHscVarError = class(Exception);

   THscVariant = class
      private
         FValTyp: THscVarTypes;
         FValInt: Integer;
         FValStr: String;
         FConst : Boolean;

         function  GetAsVar: Variant;
         procedure SetAsVar( const NewValue: Variant );
         function  GetAsInt: Integer;
         procedure SetAsInt( const NewValue: Integer );
         function  GetAsBool: Boolean;
         procedure SetAsBool( const NewValue: Boolean );
         function  GetAsStr: String;
         procedure SetAsStr( const NewValue: String );
         function  GetAsPtr: Integer;

      public
         property TypOf: THscVarTypes read FValTyp;
         property AsVar: Variant read GetAsVar write SetAsVar;
         property AsBool: Boolean read GetAsBool write SetAsBool;
         property AsInt: Integer read GetAsInt write SetAsInt;
         property AsStr: String  read GetAsStr write SetAsStr;
         property AsPtr: Integer read GetAsPtr;
         property IsConst: Boolean read FConst Write FConst;

         procedure Assign( const HV: THscVariant );
         function  Unassigned: Boolean;
         procedure Unassign;
         procedure ToInt;
         procedure ToStr;

         constructor Create;                              overload;
         constructor Create( const AValue: Integer; Const AConst: boolean ); overload;
         constructor Create( const AValue: String; Const AConst: Boolean ); overload;
         constructor Create( const AValue: THscVariant; Const AConst: boolean ); overload;
   end;

   THscVariantArray = class
      private
         FCount: Integer;
         FArray: array of THscVariant;

         function GetItem( const Index: Integer ): THscVariant;

      public
         property Count: Integer read FCount;
         property Item[ const Index: Integer]: THscVariant
                                               read GetItem; default;

         constructor Create( const ACount: Integer );
         destructor Destroy; override;
   end;

   THscVariable = class
      private
         FContext: Integer;
         FName   : String;
         FValue  : THscVariant;

      public
         property Context: Integer     read FContext;
         property Name   : String      read FName;
         property Value  : THscVariant read FValue;

         constructor Create( const AContext: Integer;
                             const AName   : String;
                             const AValue  : THscVariant;
                             Const AConst  : boolean );
         destructor Destroy; override;
   end;

   THscVariables = class( THscObject )
      private
         FList: TList;

         function  GetCount: Integer;
         function  GetVarIndex( const Name: String ): Integer;
         function  GetValue( const Name: String ): THscVariant;

         function  IndexOf( const AContext: Integer;
                            const AName   : String ): Integer;
         procedure Add( const AContext: Integer;
                        const AName   : String;
                        const AValue  : THscVariant;
                        Const AConst  : boolean );

      public
         property Count: Integer read GetCount;
         property Value[ const Name: String ]: THscVariant read GetValue; default;
         Function Items (const Index: Integer): THscVariable;
         procedure DefValue( const AName: String; const AValue: THscVariant;
            Const AConst: boolean );
         function  Dump( const Index: Integer ): String;
         procedure ClearContext( const Context: Integer );
         procedure Clear;

         constructor Create( const AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscLists = class( THscObject )
      private
         FLists: TList;

         function  GetCount: Integer;
         function  GetList( const Index: Integer ): TStringListEx;
         function  GetListItem( const ListIndex, ItemIndex: Integer ): String;
         procedure SetListItem( const ListIndex, ItemIndex: Integer;
                                const NewValue: String );
         function  GetListTag( const ListIndex, ItemIndex: Integer ): Integer;
         procedure SetListTag( const ListIndex, ItemIndex, NewValue: Integer );

      public
         property Count: Integer read GetCount;
         property List    [ const Index: Integer ]: TStringListEx read GetList;
         property ListItem[ const ListIndex, ItemIndex: Integer ]: String
                            read GetListItem write SetListItem;
         property ListTag [ const ListIndex, ItemIndex: Integer ]: Integer
                            read GetListTag write SetListTag;

         function ListAlloc ( const Sorted, Duplicates: Boolean ): Integer;
         function ListFree  ( const Index: Integer ): Integer;
         function ListExists( const Index: Integer ): Boolean;
         function ListClear ( const Index: Integer ): Integer;

         procedure Clear;

         constructor Create( const AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscExpTypes = (
      hetINT, // Number
      hetSTR, // String
      hetVAR, // Variable
      hetFUN, // Function
      hetOP0, // "||"
      hetOP1, // "&&"
      hetOP2, // "|"
      hetOP3, // "^"
      hetOP4, // "&"
      hetOP5, // "==" / "!=" [ / "=" / "!" / "<>"
      hetOP6, // "<" / ">" / "<=" / ">="
      hetOP7, // "<<" / ">>"
      hetOP8, // "+" / "-"
      hetOP9, // "*" / "/" / "%"
      hetUOP, // "+" / "-" / "!" / "~" (unary)
      hetGON, // "("
      hetGOF, // ")"
      hetERR  // error-marker
   );

   THscExpressionPart = class
      public
         EText : String;
         EType : THscExpTypes;
         EValue: THscVariant;

         constructor Create( const AExpPat: THscExpressionPart ); overload;
         constructor Create( const AText  : String;
                             const AType  : THscExpTypes;
                             const AValue : Integer            ); overload;
         constructor Create( const AText  : String;
                             const AType  : THscExpTypes;
                             const AValue : String             ); overload;
         destructor Destroy; override;
   end;

   THscExpressionParts = class
      private
         FList: TList;

         function GetCount: Integer;
         function GetItem( const Index: Integer ): THscExpressionPart;

      public
         property Count: Integer read GetCount;
         property Item[ const Index: Integer ]: THscExpressionPart
                                                read GetItem; default;

         procedure Add( const AText : String;
                        const AType : THscExpTypes;
                        const AValue: Integer      ); overload;
         procedure Add( const AText : String;
                        const AType : THscExpTypes;
                        const AValue: String       ); overload;
         procedure Clear;
         procedure Delete( const Index: Integer );
         procedure Assign( const EP: THscExpressionParts );

         constructor Create;
         destructor Destroy; override;
   end;

   THscExpression = class( THscObject )
      private
         FParts: THscExpressionParts;

         procedure ParseExpression( const pExpression: PChar );
         procedure Solve_VariablesAndFunctions;
         procedure Solve_ReduceParts;

      public
         procedure Solve( const Expression: String;
                          const Result    : THscVariant );

         constructor Create( const AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscExpressions = class( THscObject )
      private
         FExps: TStringList_ExactMatch;

         function GetParsedExp( const Index: Integer ): THscExpressionParts;

      public
         property ParsedExp[ const Index: Integer ]: THscExpressionParts
                                                     read GetParsedExp;

         function  IndexOf( const AExpression: String ): Integer;
         procedure Add    ( const AExpression: String;
                            const AParsedExp : THscExpressionParts );
         procedure Clear;

         constructor Create( const AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscSchedEntry = class( THscObject )
      private
         FLastTriggered: Integer;
         FKeyword      : String;
         FFromTime     : String;
         FTilTime      : String;
         FAtDays       : String;
         FEveryMins    : Integer;
         FImmediate    : Boolean;

      public
         property Keyword       : String  read FKeyword;
         property FromTime      : String  read FFromTime;
         property TilTime       : String  read FTilTime;
         property AtDays        : String  read FAtDays;
         property EveryMins     : Integer read FEveryMins;
         property Immediate     : Boolean read FImmediate;

         function Trigger: Boolean;
         property LastTriggered : Integer read FLastTriggered;

         constructor Create( const AEngine: THscEngine;
                             const AKeyword: String;
                             const AFromTime, ATilTime, AAtDays: String;
                             const AEveryMins: Integer;
                             const AImmediate: Boolean );
   end;

   THscScheduler = class( THscObject )
      private
         FList: TList;
      public
         procedure Clear;
         function  Add( const Keyword: String;
                        FromTime, TilTime, AtDays: String;
                        EveryMins: Integer;
                        const Immediate: Boolean ): Integer;
         function  Check: String;

         function  GetCount: Integer;

         function  GetEntrySub( const Entry: Integer): String;
         function  GetEntryAtDays(const Entry: Integer): String;
         function  GetEntryEveryMins(const Entry: Integer): Integer;
         function  GetEntryFromTime(const Entry: Integer): String;
         function  GetEntryTilTime(const Entry: Integer): String;

         constructor Create( const AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscParsedFunc = class( THscObject )
      private
         FName  : String;
         FMarker: LongWord;
         FParms : TStringList;
         FNameOK: Boolean;

         procedure DoFuncParse( const LineStr: String );

      public
         property  FuncName  : String      read FName;
         property  FuncMarker: LongWord    read FMarker;
         property  FuncPars  : TStringList read FParms;
         property  FuncNameOK: Boolean     read FNameOK write FNameOK;

         function  FuncIs( const TestFunc: String;
                           const ParMin, ParMax: Integer ): Boolean; overload;
         function  FuncIs( const TestFunc: LongWord;
                           const ParMin, ParMax: Integer ): Boolean; overload;

         procedure ParH( const Result: THscVariant;
                         const Index : Integer      ); overload;
         procedure ParH( const Result: THscVariant;
                         const Index : Integer;
                         const DefVal: Integer      ); overload;
         procedure ParH( const Result: THscVariant;
                         const Index : Integer;
                         const DefVal: String       ); overload;

         function ParV( const Index: Integer; const DefVal: Variant ): Variant;
         function ParS( const Index: Integer  ): String; overload;
         function ParS( const Index: Integer; const DefVal: String  ): String; overload;
         function ParI( const Index: Integer  ): Integer; overload;
         function ParI( const Index: Integer; const DefVal: Integer ): Integer; overload;
         function ParX( const Index: Integer; const DefVal: String  ): String;

         constructor Create( const AEngine: THscEngine;
                             const LineStr: String );
         destructor Destroy; override;
   end;

   THscParsedFuncs = class( THscObject )
      private
         FList: TStringList_ExactMatch;

         function GetParsedFunc( const Index: Integer ): THscParsedFunc;

      public
         property ParsedFunc[ const Index: Integer ]: THscParsedFunc
                                                      read GetParsedFunc;

         function  IndexOf( const AExpression: String ): Integer;
         function  Add    ( const AExpression: String ): THscParsedFunc;
         procedure Clear;

         constructor Create( const AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscScript = class( THscObject )
      private
         FLines   : TStringList;
         FScrLines: TList;    
         FModRefs : TList;
         FModules : TStringList;
         FCurrLine: Integer;

         function  GetCount: Integer;
         function  GetText : String;
         function  GetLine   ( const Index: Integer ): String;
         function  GetLineNo ( const Index: Integer ): Integer;
         function  GetModID  ( const Index: Integer ): Integer;
         function  GetModName( const Index: Integer ): String;
         function  GetScrLine( const Index: Integer ): THscParsedFunc; 
         function  GetMarkers( const Index: Integer ): LongWord;
         procedure SetMarkers( const Index: Integer;
                               const NewMarkers: LongWord );

      public
         property Count: Integer read GetCount;
         property Text : String read GetText;
         property Lines  [ const Index: Integer ]: String  read GetLine;
         property LineNo [ const Index: Integer ]: Integer read GetLineNo;
         property ModID  [ const Index: Integer ]: Integer read GetModID;
         property ModName[ const Index: Integer ]: String  read GetModName;
         property ScrLine[ const Index: Integer ]: THscParsedFunc 
                                                   read GetScrLine;
         property Markers[ const Index: Integer ]: LongWord read  GetMarkers
                                                            write SetMarkers;

         function  IsModuleLoaded( const ModName: String ): Boolean;
         function  StartNewModule( const ModName: String ): Integer;
         function  Add( const Line: String ): Integer;
         procedure AddLines( const sl: TStringList );
         procedure Clear;

         constructor Create( const AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscEngine = class( THscObject )
      private
         FScriptName : String;
         FScript     : THscScript;
         FVariables  : THscVariables;
         FScheduler  : THscScheduler;
         FHscLists   : THscLists;
         FContextID  : Integer;
         FLoopStack  : TList;
         FIPtrStack  : TList;
         FExpressions: THscExpressions;
         FParsedFuncs: THscParsedFuncs;
         FCurrLine   : String;
         FCurrPos    : Integer;
         FNextPos    : Integer;
         FStopEvent  : THandle;
         FTraceIsOn  : Boolean;
         FDebugLevel : Integer;
         FTerminated : Boolean;
         FSyncCmd    : String;
         FSyncPars   : TStringList; 
         FSyncResult : Variant;
         FErrNum     : Integer;
         FErrMsg     : String;
         FErrModule  : String;
         FErrLineNo  : Integer;
         FErrLine    : String;
         FErrSender  : String;
         FErrCatch   : Boolean;
         FPathHSM    : String;
         FScriptParams: TStringList;
         FResControl  : TResController;
         FDllLastError: Integer;
         FParentThread: TThread;

         procedure AddError( const Sender : TObject;
                             const AErrNum: Integer;
                             const AErrMsg: String );
         function  GetContext: Integer;
         procedure IPtrPush;
         procedure IPtrPop;
         procedure LoopStackPush( const Value: Integer );
         function  LoopStackPop ( const Count: Integer ): Integer;
         function  LoopStackPeek( const Depth: Integer ): Integer;
         procedure EnterContext;
         procedure LeaveContext;
         function  IndexOfLabel( const LabelName: String ): Integer;
         function  IndexOfSub( const SubName: String ): Integer;
         procedure Trace( const TracePos: Integer; const TraceText: String );
         procedure SolveFunctionStr( const FuncStr: String;
                                     const Result: THscVariant );
         procedure SolveFunction   ( const ParsedFunc: THscParsedFunc;
                                     const Result: THscVariant );
         procedure SolveExpression ( const Expression: String;
                                     const Result: THscVariant );
         procedure SyncExecute;
         procedure ExecuteNextPos( const Result: THscVariant );
         procedure Execute( const Result: THscVariant );

      protected
         procedure Engine_Error  ( const Engine: THscEngine;
                                   const Sender: TObject;
                                   const ErrNum: Integer;
                                   const ErrMsg: String;
                                   const CurrMod: String;
                                   const CurrPos: Integer;
                                   const CurrLine: String ); virtual; abstract;
         procedure Engine_Trace  ( const Engine: THscEngine;
                                   const CurrMod: String;
                                   const CurrPos: Integer;
                                   const CurrLine: String ); virtual; abstract;
         procedure Engine_Warning( const Engine: THscEngine;
                                   const PrintStr: String ); virtual; abstract;
         procedure Engine_Print  ( const Engine: THscEngine;
                                   const PrintStr: String ); virtual; abstract;
         procedure Engine_SyncExec( const Engine: THscEngine;
                                    const X: TThreadMethod); virtual; abstract;
         procedure Engine_Func( const ParsedFunc: THscParsedFunc;
                                const Result: THscVariant ); virtual; abstract;
         function  Engine_StartScript( const ScriptFile: String;
                                       const Params    : String;
                                       const WaitForEnd: Boolean ): Integer;
                                       virtual; abstract;

      public
         property StopEvent : THandle        read FStopEvent write FStopEvent;
         property Variables : THscVariables  read FVariables;
         property ResControl: TResController read FResControl;
         property Context   : Integer        read GetContext;
         property ScriptName: String         read FScriptName;

         procedure ExecuteFromList( const AScriptName: String;
                                    const AScriptList: TStringList;
                                    const AParamsText: String;
                                    const Result: THscVariant );

         constructor Create( AParentThread: TThread; const APathHSM: String );
         destructor Destroy; override;
   end;

function IsVariable( const s: String ): Boolean;

implementation

uses IniFiles, Clipbrd, FileCtrl,
     // -----
     cPCRE, uRasDyn, dInput, uDateTime, cLogFile, tBase, cArticle,
     uDynDll, uCRC32, uMD5, dPopUpBox , uSHA1, global, uWinSock
     {$IFDEF H_NEED_VARIANTS} , Variants {$ENDIF} ;

const
   GLOBAL_CONTEXT = 0;

   HSCERR_NOERROR           = 0;
   HSCERR_ENGINEEXCEPTION   = 3;
   HSCERR_INVALIDEXPRESSION = 4;
   HSCERR_UNSUPPORTED       = 5;
   HSCERR_VARNOTINITIALIZED = 6;
   HSCERR_SYNTAX            = 7;
   HSCERR_LABELNOTFOUND     = 8;
   HSCERR_LOOPSTACK         = 9;
   HSCERR_IFTHENELSE        = 10;
   HSCERR_LOOPNOTFOUND      = 11;
   HSCERR_LIMITEXCEEDED     = 12;
   HSCERR_UNKNOWNVAR        = 13;
   HSCERR_INVALID           = 14;
   HSCERR_INITIALIZEFAILED  = 15;
   HSCERR_USERDEFINED       = 1000;

   SMARKER_MASK_LINENO = $00FFFFFF;
   SMARKER_MASK_MARKER = $FF000000;

   SMARKER_CHECKED   = $80000000;
   SMARKER_NOSPECIAL = SMARKER_CHECKED;
   SMARKER_IF        = SMARKER_CHECKED or $01000000;
   SMARKER_ELSE      = SMARKER_CHECKED or $02000000;
   SMARKER_ENDIF     = SMARKER_CHECKED or $03000000;
   SMARKER_DO        = SMARKER_CHECKED or $04000000;
   SMARKER_LOOP      = SMARKER_CHECKED or $05000000;
   SMARKER_WHILE     = SMARKER_CHECKED or $06000000;
   SMARKER_ENDWHILE  = SMARKER_CHECKED or $07000000;
   SMARKER_REPEAT    = SMARKER_CHECKED or $08000000;
   SMARKER_UNTIL     = SMARKER_CHECKED or $09000000;
   SMARKER_FOR       = SMARKER_CHECKED or $0A000000;
   SMARKER_ENDFOR    = SMARKER_CHECKED or $0B000000;
   SMARKER_SUB       = SMARKER_CHECKED or $0C000000;
   SMARKER_LABEL     = SMARKER_CHECKED or $0D000000;
   SMARKER_ELSEIF    = SMARKER_CHECKED or $0E000000;

function MarkerInArray( const Marker: LongWord;
                        const MarkerArray: array of LongWord ): Boolean;
var  i: Integer;
begin
   Result := False;
   for i:=0 to High(MarkerArray) do begin
      if MarkerArray[i]=Marker then begin Result:=True; exit end;
   end;
end;

function CmdToMarker( const Cmd: String ): LongWord;
begin
   if pos( ','+Cmd+',',
           ',if,else,elseif,endif,do,loop,while,endwhile,repeat,until,for,endfor,sub,label,' ) = 0 then begin
      Result := SMARKER_NOSPECIAL;
      exit;
   end;

   if      Cmd='if'       then Result:=SMARKER_IF
   else if Cmd='else'     then Result:=SMARKER_ELSE
   else if Cmd='elseif'   then Result:=SMARKER_ELSEIF
   else if Cmd='endif'    then Result:=SMARKER_ENDIF
   else if Cmd='do'       then Result:=SMARKER_DO
   else if Cmd='loop'     then Result:=SMARKER_LOOP
   else if Cmd='while'    then Result:=SMARKER_WHILE
   else if Cmd='endwhile' then Result:=SMARKER_ENDWHILE
   else if Cmd='repeat'   then Result:=SMARKER_REPEAT
   else if Cmd='until'    then Result:=SMARKER_UNTIL
   else if Cmd='for'      then Result:=SMARKER_FOR
   else if Cmd='endfor'   then Result:=SMARKER_ENDFOR
   else if Cmd='sub'      then Result:=SMARKER_SUB
   else if Cmd='label'    then Result:=SMARKER_LABEL
   else                        Result:=SMARKER_NOSPECIAL;
end;

function MarkerToCmd( const Marker: LongWord ): String;
begin
   case Marker of
      SMARKER_IF      : Result:='if';
      SMARKER_ELSE    : Result:='else';
      SMARKER_ELSEIF  : Result:='elseif';
      SMARKER_ENDIF   : Result:='endif';
      SMARKER_DO      : Result:='do';
      SMARKER_LOOP    : Result:='loop';
      SMARKER_WHILE   : Result:='while';
      SMARKER_ENDWHILE: Result:='endwhile';
      SMARKER_REPEAT  : Result:='repeat';
      SMARKER_UNTIL   : Result:='until';
      SMARKER_FOR     : Result:='for';
      SMARKER_ENDFOR  : Result:='endfor';
      SMARKER_SUB     : Result:='sub';
      SMARKER_LABEL   : Result:='label';
      else              Result:='';
   end;
end;

function ReadLineCmd( const s: String ): String;
var  i: Integer;
begin
   Result := LowerCase( s );
   i := PosWhSpace( Result ); if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '(', Result );   if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '#', Result );   if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '{', Result );   if i>0 then Result:=copy(Result,1,i-1);
   if copy(Result,1,1)='$' then Result:='set';
end;

function ExecuteProcess( CommandLine     : String;
                         WorkingDir      : String;
                         SWShowWindow    : Integer;
                         WaitForEnd      : Boolean;
                         var WaitExitCode: Integer;
                         StopEvent       : Integer;
                         Out Failed: Integer ): Integer;
var  StartupInfo : TStartUpInfo;
     ProcessInfo : TProcessInformation;
     WaitObjects : TWOHandleArray;
     WaitObjCount: Integer;
     pWorkingDir : PChar;
     res: DWord;
begin
     Result := 0;
     WaitExitCode := -1;
     Failed := 0;

     if WorkingDir='' then pWorkingDir:=nil
                      else pWorkingDir:=PChar(WorkingDir);

     FillChar( StartUpInfo, sizeof(StartUpInfo), 0 );
     StartupInfo.cb := Sizeof( StartupInfo );
     StartupInfo.dwFlags     := STARTF_USESHOWWINDOW;
     StartupInfo.wShowWindow := SWShowWindow;

     FillChar( ProcessInfo, sizeof(ProcessInfo), 0 );

     if CreateProcess( nil, PChar(CommandLine), nil, nil, False,
                       NORMAL_PRIORITY_CLASS, nil, pWorkingDir,
                       StartUpInfo, ProcessInfo )
     then begin

        if WaitForEnd then begin
           WaitObjects[0] := ProcessInfo.hProcess;
           WaitObjects[1] := StopEvent;
           if StopEvent=0 then WaitObjCount:=1 else WaitObjCount:=2;
           res := WaitForMultipleObjects( WaitObjCount, @WaitObjects,
                                          False, INFINITE );
           case res of
              WAIT_OBJECT_0  : Result :=  0; // ok
              WAIT_OBJECT_0+1: Result := -1; // stop-event
              WAIT_FAILED    : Result := GetLastError; // error
           end;

           if Result=0 then begin
              GetExitCodeProcess( ProcessInfo.hProcess, res );
              WaitExitCode := Integer(res)
           end

        end;

        CloseHandle( ProcessInfo.hThread  );
        CloseHandle( ProcessInfo.hProcess );

     end else begin
        Result := GetLastError;
        if Result=0 then Result:=-1;
        Failed := Result;
     end;
end;

// ----------------------------------------------------------- THsc-Tools -----

const
   CSET_DIGITS     = ['0'..'9'];
   CSET_LETTERS    = ['A'..'Z','a'..'z'];
   CSET_HEXDIGITS  = ['0'..'9','A'..'F','a'..'f'];
   CSET_IDENTIFIER = ['a'..'z','A'..'Z','0'..'9','_'];

procedure SkipEmbeddedComment( var pch: PChar );
begin
   if pch^<>'{' then exit;
   inc( pch );
   while pch^<>#0 do begin
      if pch^='}' then begin inc(pch); break; end;
      inc( pch );
   end;
end;

function TrimWSPEC( const s : String ) : String;
var  pb, pe: PChar;
begin
   SetLength( Result, 0 );

   pb := PChar( s );
   repeat
      case pb^ of
         #9, ' ': inc( pb );
         '{'    : begin
                     while not( pb^ in [#0,'}'] ) do inc( pb );
                     if pb^=#0 then exit else inc(pb);
                  end;
         else     break;
      end;
   until False;
   if pb^=#0 then exit;

   pe := strend( pb ) - 1;
   repeat
      case pe^ of
         #9, ' ': dec( pe );
         '}'    : begin
                     while (pe>pb) and (pe^<>'{') do dec( pe );
                     if pb=pe then exit else dec(pe);
                  end;
         else     break;
      end;
   until False;

   // Result := copy( pb, 1, pe-pb+1 );
   SetString( Result, pb, pe-pb+1 );
end;

procedure SkipWSPC( var pch: PChar );
begin
   while pch^<>#0 do begin
      case pch^ of
        ' ', #9: inc( pch );
        '#'    : begin pch:=strend(pch); break end;
        '{'    : begin
                    while not( pch^ in [#0,'}'] ) do inc(pch);
                    if pch^=#0 then break else inc(pch);
                 end;
        else     break;
      end;
   end;
end;

function VarToHscVarType( const V: Variant ): THscVarTypes;
begin
   case VarType( V ) of
      varInteger, varByte, varSmallInt, varBoolean:
         Result := hvtInteger;
      varString, varOleStr:
         Result := hvtString;
      else
         Result := hvtEmpty;
   end;
end;

function IsScalar( const het: THscExpTypes ): Boolean;
begin
   Result := ( het in [ hetINT, hetSTR ] );
end;

function IsOperator( const het: THscExpTypes ): Boolean;
begin
   Result := ( het in [ hetOP0..hetOP9 ] );
end;

function IsSimpleNumber( const s: String ): Boolean;
var  i: Integer;
begin
   Result := s>'';
   for i:=1 to length(s) do begin
      if not( s[i] in CSET_DIGITS ) then begin
         Result := false;
         exit
      end
   end;
end;

function IsSimpleString( const s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   if s='' then exit;
   if s[1]<>'"' then exit;
   if s[length(s)]<>'"' then exit;
   for i:=2 to length(s)-1 do begin
      if s[i]='"' then exit;
   end;
   Result := True;
end;

function IsIdentifier( const s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   if s='' then exit;
   if not( s[1] in CSET_LETTERS ) then exit;
   for i:=2 to length(s) do begin
      if not( s[i] in CSET_IDENTIFIER ) then exit;
   end;
   Result := True;
end;

function IsVariable( const s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   if length(s)<2 then exit;
   if s[1]<>'$' then exit;
   if not( s[2] in CSET_LETTERS ) then exit;
   for i:=3 to length(s) do begin
      if not( s[i] in CSET_IDENTIFIER ) then exit;
   end;
   Result := True;
end;

function WantIdentifier( const HscObject: THscObject; var pch: PChar ): String;
begin
   Result := '';
   if not( pch^ in CSET_LETTERS ) then begin // 1st char
      HscObject.Error( HSCERR_SYNTAX, 'Identifier expected: ' + String(pch) );
      exit;
   end;
   while pch^ in CSET_IDENTIFIER do begin
      AddChar( Result, pch^ );
      inc( pch );
   end;
end;

function WantVariable( const HscObject: THscObject; var pch: PChar ): String;
begin
   Result := '';
   if ( pch^<>'$' ) or not( (pch+1)^ in CSET_LETTERS ) then begin
      HscObject.Error( HSCERR_SYNTAX, 'Variable expected: ' + String( pch ) );
      exit;
   end;
   inc( pch );
   Result := '$' + WantIdentifier( HscObject, pch );
end;

function WantString( const HscObject: THscObject;
                     var   pch: PChar;
                     const KillDoubleQuotes: Boolean ): String;
begin
   Result := '';
   if pch^<>'"' then begin
      HscObject.Error( HSCERR_SYNTAX, 'String expected: ' + String( pch ) );
      exit;
   end;

   Result := '"';
   inc( pch );

   while pch^<>#0 do begin
      AddChar( Result, pch^ );
      if pch^='"' then begin
         inc( pch );
         if pch^='"' then begin        
            If Not KillDoubleQuotes then AddChar( Result, pch^ ) 
         end else
            break;
      end;
      inc( pch );
   end;

   if (length(Result)<2) or (Result[length(Result)]<>'"') then begin
      HscObject.Error( HSCERR_SYNTAX, 'Missing end of string: ' + Result );
      exit;
   end;
end;

function WantNumber( const HscObject: THscObject; var pch: PChar ): String;
var  CharSet: set of Char;
begin
   Result := '';
   if not( pch^ in CSET_DIGITS ) then begin
      HscObject.Error( HSCERR_SYNTAX, 'Number expected: ' + String( pch ) );
      exit;
   end;

   CharSet := CSET_DIGITS;
   Result := pch^;
   inc( pch );

   if (Result='0') and (pch^='x') then begin
      if (pch+1)^ in CSET_HEXDIGITS then begin
         Result := '$' + (pch+1)^;
         CharSet := CSET_HEXDIGITS;
         inc( pch, 2 );
      end;
   end;

   while pch^ in CharSet do begin
      AddChar( Result, pch^ );
      inc( pch );
   end;

   if Result[1]='$' then Result := inttostr( strtoint( Result ) );
end;

function WantOperator( const HscObject: THscObject;
                       var   pch: PChar;
                       var   OpClass: THscExpTypes ): String;
var  c2: Char;
begin
   Result := '';

   c2 := (pch+1)^;

   case pch^ of
      '+','-':     begin Result:=pch^; OpClass:=hetOP8; end;
      '*','/','%': begin Result:=pch^; OpClass:=hetOP9; end;
      '^':         begin Result:=pch^; OpClass:=hetOP3; end;
      '|':         if c2='|' then begin
                      Result:='||'; OpClass:=hetOP0;
                      inc( pch );
                   end else begin
                      Result:='|'; OpClass:=hetOP2;
                   end;
      '&':         if c2='&' then begin
                      Result:='&&'; OpClass:=hetOP1;
                      inc( pch );
                   end else begin
                      Result:='&'; OpClass:=hetOP4;
                   end;
      '=':         begin
                      Result:='=='; OpClass:=hetOP5;
                      if c2='=' then inc( pch );
                   end;
      '!':         begin
                      Result:='!='; OpClass:=hetOP5;
                      if c2='=' then inc( pch );
                   end;
      '<':         case c2 of
                      '>': begin Result:='!='; OpClass:=hetOP5; inc( pch ); end;
                      '=': begin Result:='<='; OpClass:=hetOP6; inc( pch ); end;
                      '<': begin Result:='<<'; OpClass:=hetOP7; inc( pch ); end;
                      else begin Result:='<';  OpClass:=hetOP6; end;
                   end;
      '>':         case c2 of
                      '=': begin Result:='>='; OpClass:=hetOP6; inc( pch ); end;
                      '>': begin Result:='>>'; OpClass:=hetOP7; inc( pch ); end;
                      else begin Result:='>';  OpClass:=hetOP6; end;
                   end;
      else         begin
                      HscObject.Error( HSCERR_INVALIDEXPRESSION,
                                       'Operator expected: ' + String( pch ) );
                      exit;
                   end;
   end;

   inc( pch );
end;

function WantParens( const HscObject: THscObject; var pch: PChar ): String;
var  Depth: Integer;
begin
   Result := '';
   if pch^<>'(' then begin
      HscObject.Error( HSCERR_SYNTAX, 'Missing "(": ' + String( pch ) );
      exit;
   end;

   Result := '(';
   Depth := 1;
   inc( pch );

   while pch^<>#0 do begin
      if HscObject.SelfError then exit;
      case pch^ of
         '(': inc( Depth );
         ')': begin
                 dec( Depth );
                 if Depth=0 then begin
                    AddChar( Result, pch^ ); 
                    inc(pch);
                    break;
                 end;
              end;
         '"': begin
                 Result := Result + WantString( HscObject, pch, false );
                 continue;
              end;
      end;
      AddChar( Result, pch^ ); 
      inc( pch );
   end;

   if Depth>0 then HscObject.Error( HSCERR_SYNTAX, 'Missing ")": ' + Result );
end;

function WantFunction( const HscObject: THscObject; var pch: PChar ): String;
begin
   Result := '';
   if not( pch^ in CSET_LETTERS ) then begin // 1st char
      HscObject.Error( HSCERR_SYNTAX, 'Function expected: ' + String( pch ) );
      exit;
   end;
   Result := WantIdentifier( HscObject, pch );
   SkipWSPC( pch );
   if pch^='(' then Result := Result + WantParens( HscObject, pch );
end;

function WantExpression( const HscObject: THscObject; var pch: PChar ): String;
var  pOrg  : PChar;
     s     : String;
     het   : THscExpTypes;
     WantOp: Boolean;
     Parens: Integer;
begin
   Result := '';
   pOrg   := pch;
   WantOp := False;
   Parens := 0;
   SkipWSPC( pch );

   while pch^ <> #0 do begin
      if HscObject.SelfError then break;

      if WantOp then begin

         case pch^ of
            #9, ' ' : inc( pch ); // skip whitespace
            '{'     : SkipEmbeddedComment( pch );
            ','     : begin inc(pch); break; end; // param-separator
            ')'     : begin
                         AddChar( Result, pch^ ); 
                         inc( pch );
                         dec( Parens );
                      end;
            else      begin
                         s := WantOperator( HscObject, pch, het );
                         Result := Result + s;
                         WantOp := False;
                      end;
         end;

      end else begin

         case pch^ of
            #9, ' ' : inc( pch ); // skip whitespace
            '{'     : SkipEmbeddedComment( pch );
            ','     : begin inc(pch); break; end; // param-separator
            '('     : begin
                         AddChar( Result, pch^ ); 
                         inc( pch );
                         inc( Parens );
                      end;
            '+','-',
            '!','~' : begin
                         AddChar( Result, pch^ ); 
                         inc( pch );
                      end;
            '0'..'9': begin // number
                         s := WantNumber( HscObject, pch );
                         Result := Result + s;
                         WantOp := True;
                      end;
            '"'     : begin // string
                         s := WantString( HscObject, pch, false );
                         Result := Result + s;
                         WantOp := True;
                      end;
            '$'     : begin // variable
                         s := WantVariable( HscObject, pch );
                         Result := Result + s;
                         WantOp := True;
                      end;
            'a'..'z',
            'A'..'Z': begin // function
                         s := WantFunction( HscObject, pch );
                         Result := Result + s;
                         WantOp := True;
                      end;
            else begin
                    HscObject.Error( HSCERR_INVALIDEXPRESSION,
                           'Invalid expression #1: ' + String( pOrg ) );
                    break;
                 end;
         end;

      end;

      SkipWSPC( pch );
   end;

   if HscObject.AnyError then exit;

   if Parens>0 then HscObject.Error( HSCERR_INVALIDEXPRESSION,
                                     'Missing ")": ' + String( pOrg ) );
   if Parens<0 then HscObject.Error( HSCERR_INVALIDEXPRESSION,
                                     'Too many ")": ' + String( pOrg ) );
end;

procedure WantSplitFunction( const HscObject: THscObject;
                             var pcLine: PChar;
                             var Cmd   : String;
                             var Pars  : TStringList );
var  Par, s: String;
     pcTemp: PChar;
begin
   Cmd := '';
   Pars.Clear;

   SkipWSPC( pcLine );
   if pcLine^=#0 then exit; // empty or comment-only line

   // get command
   if pcLine^ in CSET_LETTERS then begin
      Cmd := WantIdentifier( HscObject, pcLine );
      if HscObject.SelfError then exit;

      SkipWSPC( pcLine );
      if pcLine^=#0 then exit; // no parameters, "()" may be ommitted

      if pcLine^='(' then begin
         // get pars
         s := WantParens( HscObject, pcLine ); // "(" params ")"
         if HscObject.SelfError then exit;
         System.Delete( s, 1, 1 );
         System.Delete( s, length(s), 1 );
      end else begin
         s := pcLine;
         while pcLine^<>#0 do Inc(pcline)
         {HscObject.Error( HSCERR_SYNTAX, '"(" expected: ' + Cmd + '<(?>' + String(pcLine));
         exit;}
      end;

      pcTemp := PChar( s );
      while pcTemp^<>#0 do begin
         Par := WantExpression( HscObject, pcTemp );
         if HscObject.AnyError then break;
         Pars.Add( Par );
      end;

   end else if pcLine^='$' then begin
      Cmd := WantVariable( HscObject, pcLine );
      if HscObject.SelfError then exit;

      SkipWSPC( pcLine );
      if pcLine^<>'=' then begin
         HscObject.Error( HSCERR_SYNTAX, '"=" expected: ' + Cmd + '<=?>' + String(pcLine));
         exit;
      end;
      inc( pcLine );

      Pars.Add( Cmd ); // convert: "$Variable=..." => "set($Variable,...)"
      Cmd := 'set';
      Pars.Add( WantExpression(HscObject, pcLine) );

   end else begin
      HscObject.Error( HSCERR_SYNTAX, '"Function()" or "$Variable=" expected: ' + String(pcLine));
      exit;
   end;

   if HscObject.SelfError then exit;
   SkipWSPC( pcLine );
   if pcLine^=#0 then exit;
   HscObject.Error( HSCERR_SYNTAX, 'End of line expected: ' + String(pcLine));
end;

procedure HscCalculateUnary( const HscObject: THscObject;
                             const Op : String;
                             const V1 : THscVariant;
                             var ETypeResult: THscExpTypes;
                             const Result: THscVariant );
begin
   ETypeResult := hetERR;

   try
      case V1.TypOf of

         hvtInteger: begin
            ETypeResult := hetINT;
            if      OP='+' then Result.AsInt := V1.AsInt
            else if OP='-' then Result.AsInt := -V1.AsInt
            else if OP='!' then Result.AsInt := Integer( V1.AsInt = 0 )
            else if OP='~' then Result.AsInt := Integer( not V1.AsInt )
            else begin
               HscObject.Error( HSCERR_UNSUPPORTED,
                                'Unsupported integer-uoperator: ' + OP );
               Result.Unassign;
               ETypeResult := hetERR;
            end;
         end;


         hvtString: begin
            ETypeResult := hetSTR;
            HscObject.Error( HSCERR_UNSUPPORTED,
                             'Unsupported string-uoperator: ' + OP );
            Result.Unassign;
            ETypeResult := hetERR;
         end;

         hvtEmpty: begin
            HscObject.Error( HSCERR_UNSUPPORTED, 'Undefined uoperand: ' + OP );
            Result.Unassign;
            ETypeResult := hetERR;
         end;

      end;

   except
      on E: Exception do begin
         HscObject.Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
         Result.Unassign;
         ETypeResult := hetERR;
      end;
   end;
end;

procedure HscCalculate( const HscObject: THscObject;
                        const V1 : THscVariant;
                        const Op : String;
                        const V2 : THscVariant;
                        var ETypeResult: THscExpTypes;
                        const Result: THscVariant );
var  ResType: THscVarTypes;
     i1, i2 : Integer;
     s1, s2 : String;
begin
   ETypeResult := hetERR;

   try
      ResType := V1.TypOf;
      if (ResType<>hvtEmpty) and (V2.TypOf=hvtString) then ResType:=hvtString;

      case ResType of

         hvtInteger: begin
            ETypeResult := hetINT;
            i1 := V1.AsInt;
            i2 := V2.AsInt;
            if      OP='+'  then Result.AsInt := i1 + i2
            else if OP='-'  then Result.AsInt := i1 - i2
            else if OP='*'  then Result.AsInt := i1 * i2
            else if OP='/'  then Result.AsInt := i1 div i2
            else if OP='%'  then Result.AsInt := i1 mod i2
            else if OP='<<' then Result.AsInt := i1 shl i2
            else if OP='>>' then Result.AsInt := i1 shr i2
            else if OP='^'  then Result.AsInt := i1 xor i2
            else if OP='&'  then Result.AsInt := i1 and i2
            else if OP='|'  then Result.AsInt := i1 or  i2
            else if OP='||' then Result.AsInt := Integer( (i1<>0) or  (i2<>0) )
            else if OP='&&' then Result.AsInt := Integer( (i1<>0) and (i2<>0) )
            else if OP='==' then Result.AsInt := Integer( i1 =  i2 )
            else if OP='!=' then Result.AsInt := Integer( i1 <> i2 )
            else if OP='<=' then Result.AsInt := Integer( i1 <= i2 )
            else if OP='>=' then Result.AsInt := Integer( i1 >= i2 )
            else if OP='<'  then Result.AsInt := Integer( i1 <  i2 )
            else if OP='>'  then Result.AsInt := Integer( i1 >  i2 )
            else begin
               HscObject.Error( HSCERR_UNSUPPORTED,
                                'Unsupported integer-operator: ' + OP );
               Result.Unassign;
               ETypeResult := hetERR;
            end;
         end;


         hvtString: begin
            ETypeResult := hetSTR;
            s1 := V1.AsStr;
            s2 := V2.AsStr;
            if OP='+'  then Result.AsStr := s1 + s2
            else begin
               ETypeResult := hetINT;
               if      OP='==' then Result.AsInt := Integer( s1 =  s2 )
               else if OP='!=' then Result.AsInt := Integer( s1 <> s2 )
               else if OP='<=' then Result.AsInt := Integer( s1 <= s2 )
               else if OP='>=' then Result.AsInt := Integer( s1 >= s2 )
               else if OP='<'  then Result.AsInt := Integer( s1 <  s2 )
               else if OP='>'  then Result.AsInt := Integer( s1 >  s2 )
               else begin
                  HscObject.Error( HSCERR_UNSUPPORTED,
                                   'Unsupported string-operator: ' + OP );
                  Result.Unassign;
                  ETypeResult := hetERR;
               end;
            end;
         end;

         hvtEmpty: begin
            HscObject.Error( HSCERR_UNSUPPORTED, 'Undefined operands: ' + OP );
            Result.Unassign;
            ETypeResult := hetERR;
         end;

      end;

   except
      on E: Exception do begin
         HscObject.Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
         Result.Unassign;
         ETypeResult := hetERR;
      end;
   end;
end;


// ----------------------------------------------------------- THscObject -----

function THscObject.SelfError : Boolean;
begin
   Result := ( FLastError <> 0 );
end;

function THscObject.AnyError: Boolean;
begin
   if Assigned( FEngine ) then begin
      Result := FEngine.SelfError;
   end else begin
      Result := SelfError;
   end;
end;

procedure THscObject.Error( const AErrNum: Integer; const AErrMsg: String );
begin
   try
      if Assigned( FEngine ) then FEngine.AddError( Self, AErrNum, AErrMsg );
      if FLastError = HSCERR_NOERROR then FLastError := AErrNum;
   except
   end;
end;

constructor THscObject.Create( const AEngine: THscEngine );
begin
   inherited Create;
   FEngine := AEngine;
   FLastError := HSCERR_NOERROR;
end;

// ---------------------------------------------------------- THscVariant -----

const
   SHInvalidVarCast = 'Invalid use of unassigned value!';

procedure THscVariant.Assign( const HV: THscVariant );
begin
   If FConst then begin
      raise Exception.Create ('"Const"-Values can''t be changed')
   end else begin
      if Assigned( HV ) then begin
         FValTyp := HV.FValTyp;
         case FValTyp of
            hvtInteger: FValInt := HV.FValInt;
            hvtString : FValStr := HV.FValStr;
         end;
      end else begin
         Unassign;
      end
   end
end;

procedure THscVariant.Unassign;
begin
   FValTyp := hvtEmpty;
end;

function THscVariant.Unassigned: Boolean;
begin
   Result := ( FValTyp = hvtEmpty );
end;

procedure THscVariant.ToInt;
begin
   case FValTyp of
      hvtString : AsInt := SysUtils.strtoint(FValStr);
      hvtEmpty  : raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

procedure THscVariant.ToStr;
begin
   case FValTyp of
      hvtInteger: AsStr := inttostr(FValInt);
      hvtEmpty  : raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

function THscVariant.GetAsVar: Variant;
begin
   case FValTyp of
      hvtEmpty  : Result := Unassigned;
      hvtInteger: Result := AsInt;
      hvtString : Result := AsStr;
   end;
end;

procedure THscVariant.SetAsVar( const NewValue: Variant );
begin
   If FConst then begin
      raise Exception.Create ('"Const"-Values can''t be changed')
   end else begin
      case VarToHscVarType( NewValue ) of
         hvtEmpty  : Unassign;
         hvtInteger: AsInt := NewValue;
         hvtString : AsStr := NewValue;
      end
   end
end;

function THscVariant.GetAsInt: Integer;
begin
   case FValTyp of
      hvtInteger: Result := FValInt;
      hvtString : Result := SysUtils.strtoint(FValStr);
      else        raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

procedure THscVariant.SetAsInt(const NewValue: Integer);
begin
   If FConst then begin
      raise Exception.Create ('"Const"-Values can''t be changed')
   end else begin
      FValTyp := hvtInteger;
      FValInt := NewValue;
   end
end;
function  THscVariant.GetAsBool: Boolean;
begin
   Result := GetAsInt <> 0
end;
procedure THscVariant.SetAsBool( const NewValue: Boolean );
begin
   If FConst then begin
      raise Exception.Create ('"Const"-Values can''t be changed')
   end else begin
      If NewValue then SetAsInt (1) else SetAsInt (0)
   end
end;

function THscVariant.GetAsStr: String;
begin
   case FValTyp of
      hvtString : Result := FValStr;
      hvtInteger: Result := inttostr(FValInt);
      else        raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

procedure THscVariant.SetAsStr(const NewValue: String);
begin
   If FConst then begin
      raise Exception.Create ('"Const"-Values can''t be changed')
   end else begin
      FValTyp := hvtString;
      FValStr := NewValue;
   end
end;

function THscVariant.GetAsPtr: Integer;
begin
   case FValTyp of
      hvtInteger: Result := Integer(@FValInt);
      hvtString : Result := Integer(PChar(FValStr));
      else        Result := 0;
   end;
end;

constructor THscVariant.Create;
begin
   inherited Create;
   Unassign;
end;

constructor THscVariant.Create( const AValue: Integer; Const AConst: boolean );
begin
   inherited Create;
   AsInt := AValue;
   FConst := AConst
end;

constructor THscVariant.Create( const AValue: String; Const AConst: boolean );
begin
   inherited Create;
   AsStr := AValue;
   FConst := AConst
end;

constructor THscVariant.Create( const AValue: THscVariant; Const AConst: boolean );
begin
   inherited Create;
   Assign( AValue );
   FConst := AConst
end;


// ----------------------------------------------------- THscVariantArray -----

constructor THscVariantArray.Create( const ACount: Integer );
var  i: Integer;
begin
   inherited Create;
   FCount := ACount;
   SetLength( FArray, FCount );
   for i:=0 to FCount-1 do FArray[i] := THscVariant.Create;
end;

destructor THscVariantArray.Destroy;
var  i: Integer;
begin
   for i:=0 to FCount-1 do FArray[i].Free;
   FArray := nil;
   inherited Destroy;
end;

function THscVariantArray.GetItem( const Index: Integer ): THscVariant;
begin
   Result := FArray[ Index ];
end;


// --------------------------------------------------------- THscVariable -----

constructor THscVariable.Create( const AContext: Integer;
                                 const AName   : String;
                                 const AValue  : THscVariant;
                                 Const AConst  : boolean );
begin
   inherited Create;
   FContext := AContext;
   FName    := AName;
   FValue   := THscVariant.Create( AValue, AConst );
end;

destructor THscVariable.Destroy;
begin
   FValue.Free;
   inherited Destroy;
end;


// -------------------------------------------------------- THscVariables -----

function SortByContextAndName( Item1, Item2: Pointer ): Integer;
begin
   Result := THscVariable(Item2).Context - THscVariable(Item1).Context;
   if Result = 0 then Result := CompareText( THscVariable(Item1).Name,
                                             THscVariable(Item2).Name );
end;

procedure THscVariables.Add( const AContext: Integer; const AName: String;
   const AValue: THscVariant; Const AConst: boolean );
begin
   FList.Add( THscVariable.Create( AContext, AName, AValue, AConst ) );
   FList.Sort( SortByContextAndName );
end;

function THscVariables.IndexOf( const AContext: Integer; const AName: String ): Integer;
var  L, R, I, C: Integer;
begin
   Result := -1;

   L := 0;
   R := FList.Count - 1;

   while L <= R do begin
     I := (L + R) shr 1;
     C := AContext - THscVariable(FList[I]).Context;
     if C=0 then C := CompareText( THscVariable(FList[I]).Name, AName );
     if      C < 0 then L := I + 1
     else if C > 0 then R := I - 1
     else begin Result := I; break; end;
   end;
end;

function THscVariables.GetCount: Integer;
begin
   Result := FList.Count;
end;

function THscVariables.Dump( const Index: Integer ): String;
begin
   with THscVariable( FList[Index] ) do begin
      if Context = GLOBAL_CONTEXT
         then Result := '->' + Name + '='
         else Result := inttostr(Context) + '->' + Name + '=';
      case Value.TypOf of
         hvtInteger: Result := Result + inttostr( Value.AsInt );
         hvtString : Result := Result + '"' + Value.AsStr + '"';
         else        Result := Result + '(unassigned)';
      end;
   end;
end;

function THscVariables.GetVarIndex( const Name: String ): Integer;
var  Ctx: Integer;
begin
   Ctx := Engine.Context;
   Result := IndexOf( Ctx, Name ); // local

   if Result < 0 then begin
      if Ctx <> GLOBAL_CONTEXT then Result := IndexOf( GLOBAL_CONTEXT, Name );
      if Result < 0 then begin
         Engine.Error( HSCERR_UNKNOWNVAR, 'Unknown variable: ' + Name );
      end;
   end;
end;

function THscVariables.GetValue( const Name: String ): THscVariant;
var  i: Integer;
begin
   i := GetVarIndex( Name );
   if i < 0 then Result := nil
            else Result := THscVariable( FList[i] ).Value;
end;

procedure THscVariables.DefValue( const AName : String;
                                  const AValue: THscVariant;
                                  Const AConst: boolean );
var  i  : Integer;
     Ctx: Integer;
begin
   Ctx := Engine.Context;
   i := IndexOf( Ctx, AName ); // local
   if i < 0 then begin
      Add( Ctx, AName, AValue, AConst )
   end else begin
      THscVariable( FList[i] ).Value.Assign( AValue );
      THscVariable( FList[i] ).Value.IsConst := AConst
   end
end;

procedure THscVariables.ClearContext( const Context: Integer );
var  i: Integer;
begin
   i := FList.Count - 1;
   while i >= 0 do begin
      if THscVariable( FList[i] ).Context = Context then begin
         THscVariable( FList[i] ).Free;
         FList.Delete( i );
      end;
      dec( i );
   end;
end;

procedure THscVariables.Clear;
var  i: Integer;
begin
   for i:=0 to FList.Count-1 do begin
      THscVariable( FList[i] ).Free;
   end;
   FList.Clear;
end;

constructor THscVariables.Create( const AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FList := TList.Create;
end;

destructor THscVariables.Destroy;
var  i: Integer;
begin
   for i:=0 to FList.Count-1 do THscVariable( FList[i] ).Free;
   FList.Free;
   inherited Destroy;
end;

// ------------------------------------------------------------ THscLists -----

function THscLists.GetCount: Integer;
begin
   Result := FLists.Count;
end;

function THscLists.GetList( const Index: Integer ): TStringListEx;
begin
   if ListExists( Index ) then begin
      Result := FLists[Index];
   end else begin
      Engine.Error( HSCERR_INVALID, 'Invalid list-handle: ' + inttostr(Index) );
      Result := nil;
   end;
end;

function THscLists.ListExists( const Index: Integer ): Boolean;
begin
   Result := False;
   if (Index>=0) and (Index<FLists.Count) then begin
      if Assigned( FLists[Index] ) then Result := True;
   end;
end;

function THscLists.ListAlloc( const Sorted, Duplicates: Boolean ): Integer;
var  i: Integer;
begin
   Result := -1;

   for i:=0 to FLists.Count-1 do begin
      if not Assigned( FLists[i] ) then begin
         FLists[i] := TStringList.Create;
         Engine.ResControl.Add( TResHscListEntry.Create( Self, i ) );
         Result := i;
         break;
      end;
   end;

   if Result<0 then begin
      if FLists.Count>99 then begin
         Engine.Error( HSCERR_LIMITEXCEEDED, 'Too many lists (max. 99)!' );
         Result := -1;
      end else begin
         i := FLists.Add( TStringList.Create );
         Engine.ResControl.Add( TResHscListEntry.Create( Self, i ) );
         Result := i
      end;
   end;

   if (Result>=0) and Sorted then begin
      List[Result].Sorted := True;
      if Duplicates then List[Result].Duplicates:=dupAccept
                    else List[Result].Duplicates:=dupIgnore;
   end;
end;

function THscLists.ListFree( const Index: Integer ): Integer;
begin
   Result := -1;
   if ListExists(Index) then begin
      List[Index].Free;
      FLists[Index] := nil;
      FEngine.ResControl.Remove( RESID_HscListEntry, Index );
      Result := 0;
   end;
end;

function THscLists.ListClear( const Index: Integer ): Integer;
begin
   Result := -1;
   if ListExists(Index) then begin List[Index].Clear; Result:=0; end;
end;

function THscLists.GetListItem( const ListIndex, ItemIndex: Integer ): String;
begin
   Result := '';
   if ListExists(ListIndex) then begin
      if (ItemIndex>=0) and (ItemIndex<List[ListIndex].Count) then begin
         Result := List[ListIndex][ItemIndex];
      end;
   end;
end;

procedure THscLists.SetListItem( const ListIndex, ItemIndex: Integer; const NewValue: String );
begin
   if ListExists(ListIndex) then begin
      try
         while ItemIndex>=List[ListIndex].Count do List[ListIndex].Add( '' );
         List[ListIndex][ItemIndex] := NewValue;
      except
         On E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
      end;
   end;
end;

function THscLists.GetListTag( const ListIndex, ItemIndex: Integer ): Integer;
begin
   Result := 0;
   if ListExists(ListIndex) then begin
      if (ItemIndex>=0) and (ItemIndex<List[ListIndex].Count) then begin
         Result := Integer( List[ListIndex].Objects[ItemIndex] );
      end;
   end;
end;

procedure THscLists.SetListTag( const ListIndex, ItemIndex, NewValue: Integer );
begin
   if ListExists(ListIndex) then begin
      try
         while ItemIndex>=List[ListIndex].Count do List[ListIndex].Add( '' );
         List[ListIndex].Objects[ItemIndex] := Pointer( NewValue );
      except
         On E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
      end;
   end;
end;

procedure THscLists.Clear;
var  i: Integer;
begin
   for i:=0 to FLists.Count-1 do ListFree( i );
   FLists.Clear;
end;

constructor THscLists.Create( const AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FLists := TList.Create;
end;

destructor THscLists.Destroy;
begin
   if Assigned( FLists ) then begin
      Clear;
      FLists.Free;
   end;
   inherited Destroy;
end;


// --------------------------------------------------- THscExpressionPart -----

constructor THscExpressionPart.Create( const AExpPat: THscExpressionPart );
begin
   inherited Create;
   EText  := AExpPat.EText;
   EType  := AExpPat.EType;
   EValue := THscVariant.Create( AExpPat.EValue, false );
end;

constructor THscExpressionPart.Create( const AText : String;
                                       const AType : THscExpTypes;
                                       const AValue: Integer );
begin
   inherited Create;
   EText  := AText;
   EType  := AType;
   EValue := THscVariant.Create( AValue, false );
end;

constructor THscExpressionPart.Create( const AText : String;
                                       const AType : THscExpTypes;
                                       const AValue: String );
begin
   inherited Create;
   EText  := AText;
   EType  := AType;
   EValue := THscVariant.Create( AValue, false );
end;

destructor THscExpressionPart.Destroy;
begin
   EValue.Free;
   inherited Destroy;
end;


// -------------------------------------------------- THscExpressionParts -----

procedure THscExpressionParts.Add( const AText : String;
                                   const AType : THscExpTypes;
                                   const AValue: Integer );
begin
   FList.Add( THscExpressionPart.Create( AText, AType, AValue ) );
end;

procedure THscExpressionParts.Add( const AText : String;
                                   const AType : THscExpTypes;
                                   const AValue: String );
begin
   FList.Add( THscExpressionPart.Create( AText, AType, AValue ) );
end;

procedure THscExpressionParts.Assign( const EP: THscExpressionParts );
var  i: Integer;
begin
   Clear;
   for i:=0 to EP.Count-1 do FList.Add( THscExpressionPart.Create( EP[i] ) );
end;

procedure THscExpressionParts.Clear;
var  i: Integer;
begin
   for i:=0 to FList.Count-1 do THscExpressionPart( FList[ i ] ).Free;
   FList.Clear;
end;

constructor THscExpressionParts.Create;
begin
   inherited Create;
   FList := TList.Create;
end;

procedure THscExpressionParts.Delete( const Index: Integer );
begin
   THscExpressionPart( FList[ Index ] ).Free;
   FList.Delete( Index );
end;

destructor THscExpressionParts.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;

function THscExpressionParts.GetCount: Integer;
begin
   Result := FList.Count;
end;

function THscExpressionParts.GetItem(const Index: Integer): THscExpressionPart;
begin
   Result := THscExpressionPart( FList[ Index ] );
end;


// ------------------------------------------------------- THscExpression -----

procedure THscExpression.ParseExpression( const pExpression: PChar );
var  pCurr : PChar;
     s     : String;
     het   : THscExpTypes;
     WantOp: Boolean;
begin
   FParts.Clear;
   pCurr  := PExpression;
   WantOp := False;

   while pCurr^ <> #0 do begin
      if SelfError then break;

      if WantOp then begin

         case pCurr^ of
            #9, ' ' : inc( pCurr ); // skip whitespace
            '{'     : SkipEmbeddedComment( pCurr );
            ')'     : begin
                         FParts.Add( pCurr^, hetGOF, 0 );
                         inc( pCurr );
                      end;
            else      begin
                         s := WantOperator( Self, pCurr, het );
                         FParts.Add( s, het, 0 );
                         WantOp := False;
                      end;
         end;

      end else begin

         case pCurr^ of
            #9, ' ' : inc( pCurr ); // skip whitespace
            '{'     : SkipEmbeddedComment( pCurr );
            '('     : begin
                         FParts.Add( pCurr^, hetGON, 0 );
                         inc( pCurr );
                      end;
            '+','-',
            '!','~' : begin
                         FParts.Add( pCurr^, hetUOP, 0 );
                         inc( pCurr );
                      end;
            '0'..'9': begin // number
                         s := WantNumber( Self, pCurr );

                         // special case: -2147483648 (=$80000000)
                         if (s='2147483648') and (FParts.Count>0) then
                            if FParts[FParts.Count-1].EType=hetUOP then
                               if FParts[FParts.Count-1].EText='-' then begin
                                  // convert to a safe: (not $7FFFFFFF)
                                  FParts[FParts.Count-1].EText := '~';
                                  s := '2147483647'; {=$7FFFFFFF}
                               end;

                         FParts.Add( s, hetINT, strtoint(s) );
                         WantOp := True;
                      end;
            '"'     : begin // string
                         s := WantString( Self, pCurr, true );
                         if copy(s,1,1)='"' then System.Delete(s,1,1);
                         if copy(s,length(s),1)='"' then System.Delete(s,length(s),1);
                         FParts.Add( s, hetSTR, s );
                         WantOp := True;
                      end;
            '$'     : begin // variable
                         s := WantVariable( Self, pCurr );
                         FParts.Add( s, hetVAR, 0 );
                         WantOp := True;
                      end;
            'a'..'z',
            'A'..'Z': begin // function
                         s := WantFunction( Self, pCurr );
                         FParts.Add( s, hetFUN, 0 );
                         WantOp := True;
                      end;
            else begin
                    Error( HSCERR_INVALIDEXPRESSION,
                           'Invalid expression #2: ' + String( pExpression ) );
                    break;
                 end;
         end;

      end
   end
end;

procedure THscExpression.Solve_VariablesAndFunctions;

   procedure DoSolveVariable( const Index: Integer );
   begin
      FParts[Index].EValue.Assign( FEngine.Variables[ FParts[Index].EText ] );
      case FParts[Index].EValue.TypOf of
         hvtInteger: FParts[ Index ].EType := hetINT;
         hvtString : FParts[ Index ].EType := hetSTR;
         else Error( HSCERR_VARNOTINITIALIZED,
                     'Variable not initialized: ' + FParts[Index].EText );
      end;
   end;

   procedure DoSolveFunction( const Index: Integer );
   begin
      FEngine.SolveFunctionStr( FParts[Index].EText, FParts[Index].EValue );
      case FParts[Index].EValue.TypOf of
         hvtInteger: FParts[ Index ].EType := hetINT;
         hvtString : FParts[ Index ].EType := hetSTR;
         else Error( HSCERR_UNSUPPORTED,
                     'Unknown function: ' + FParts[Index].EText );
      end;
   end;

var  PartNo: Integer;
begin
   for PartNo := 0 to FParts.Count - 1 do begin
      if      FParts[PartNo].EType=hetVAR then DoSolveVariable( PartNo )
      else if FParts[PartNo].EType=hetFUN then DoSolveFunction( PartNo );
   end;
end;

procedure THscExpression.Solve_ReduceParts;
var  PartNo: Integer;
     ETypeResult: THscExpTypes;
     OP: String;
     HV1, HV2: THscVariant;
     OK: Boolean;
begin
   PartNo := 0;

   try
      while PartNo <= FParts.Count-2 do begin

         case FParts[ PartNo ].EType of
         
            hetINT, hetSTR:
               // Scalar Operator Scalar => Scalar (del.) (del.)
               if  IsOperator( FParts[PartNo+1].EType )
               and IsScalar  ( FParts[PartNo+2].EType ) then begin

                  OK := True;
                  if (PartNo+3) < FParts.Count then begin
                     // followed by operator with higher priority?
                     if IsOperator( FParts[PartNo+3].EType ) then begin
                        if FParts[PartNo+1].EType
                         < FParts[PartNo+3].EType then OK:=False;
                     end;
                  end;
                  if (PartNo-1) >= 0 then begin
                     // preceded by operator with higher priority?
                     if IsOperator( FParts[PartNo-1].EType ) then begin
                        if FParts[PartNo+1].EType
                         < FParts[PartNo-1].EType then OK:=False;
                     end;
                  end;

                  if OK then begin

                     HV1 := FParts[ PartNo   ].EValue;
                     OP  := FParts[ PartNo+1 ].EText;
                     HV2 := FParts[ PartNo+2 ].EValue;
                     HscCalculate( Self, HV1, Op, HV2, ETypeResult, HV1 );
                     FParts[PartNo].EType := ETypeResult;
                     if SelfError then break;

                     FParts.Delete( PartNo+2 );
                     FParts.Delete( PartNo+1 );

                  end else begin
                     inc( PartNo, 2 ); // also skip operator
                  end;

               end else begin
                  inc( PartNo );
               end;

            hetUOP:
               // Unary-Operator Scalar => (del.) Scalar
               if IsScalar( FParts[PartNo+1].EType ) then begin

                  OP  := FParts[ PartNo   ].EText;
                  HV1 := FParts[ PartNo+1 ].EValue;
                  HscCalculateUnary( Self, Op, HV1, ETypeResult, HV1 );
                  FParts[PartNo+1].EType := ETypeResult;
                  if SelfError then break;

                  FParts.Delete( PartNo );

               end else begin
                  inc( PartNo );
               end;

            hetGON:
               // "(" Scalar ")" => (del.) Scalar (del.)
               if  ( FParts[PartNo+2].EType = hetGOF )
               and IsScalar( FParts[PartNo+1].EType )  then begin
                  FParts.Delete( PartNo+2 );
                  FParts.Delete( PartNo   );
               end else begin
                  inc( PartNo );
               end;

            hetGOF:
               // "(" Scalar ")" => (del.) Scalar (del.)
               if  ( FParts[PartNo-2].EType = hetGON )
               and IsScalar( FParts[PartNo-1].EType ) then begin
                  FParts.Delete( PartNo   );
                  FParts.Delete( PartNo-2 );
               end else begin
                  inc( PartNo );
               end;

            else
               inc( PartNo );

         end;

      end;

   except
      // leads to 'invalid expression'
   end;
end;

procedure THscExpression.Solve( const Expression: String;
                                const Result    : THscVariant );
var  OldCount, i: Integer;
begin
   // is it an already known expression?
   i := FEngine.FExpressions.IndexOf( Expression );
   if i >= 0 then begin
      // yes: get already parsed expression from cache
      FParts.Assign( FEngine.FExpressions.ParsedExp[i] );
   end else begin
      // no: parse new expression and store it in cache
      ParseExpression( PChar( Expression ) );
      FEngine.FExpressions.Add( Expression, FParts );
   end;
   if AnyError then exit;

   // solve variables and functions in expression
   Solve_VariablesAndFunctions;
   if AnyError then exit;

   // solve expression parts step by step until we have the final result
   while FParts.Count>1 do begin
      OldCount := FParts.Count;
      Solve_ReduceParts;
      if FParts.Count=OldCount then begin
         Error( HSCERR_INVALIDEXPRESSION, 'Invalid expression: ' + Expression );
         break;
      end;
      if AnyError then break;
   end;

   // return remaining scalar as expression's result
   if (not SelfError) and (FParts.Count > 0) then begin  ///NEW///
      Result.Assign( FParts[0].EValue );
   end;
end;

constructor THscExpression.Create( const AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FParts := THscExpressionParts.Create;
end;

destructor THscExpression.Destroy;
begin
   FParts.Free;
   inherited Destroy;
end;


// ------------------------------------------------------ THscExpressions -----

function THscExpressions.GetParsedExp( const Index: Integer ): THscExpressionParts;
begin
   Result := THscExpressionParts( FExps.Objects[Index] );
end;

function THscExpressions.IndexOf( const AExpression: String ): Integer;
begin
   Result := FExps.IndexOf( AExpression );
end;

procedure THscExpressions.Add( const AExpression: String;
                               const AParsedExp: THscExpressionParts );
var  EP: THscExpressionParts;
begin
   EP := THscExpressionParts.Create;
   EP.Assign( AParsedExp );
   FExps.AddObject( AExpression, EP );
end;

procedure THscExpressions.Clear;
var  i: Integer;
begin
   for i:=0 to FExps.Count-1 do TStringList( FExps.Objects[i] ).Free;
   FExps.Clear;
end;

constructor THscExpressions.Create(const AEngine: THscEngine);
begin
   inherited Create( AEngine );
   FExps := TStringList_ExactMatch.Create;
   FExps.Sorted := True;
   FExps.Duplicates := dupIgnore;
end;

destructor THscExpressions.Destroy;
begin
   Clear;
   FExps.Free;
   inherited Destroy;
end;


// ------------------------------------------------------ THscParsedFuncs -----

function THscParsedFuncs.GetParsedFunc(const Index: Integer): THscParsedFunc;
begin
   Result := THscParsedFunc( FList.Objects[Index] );
end;

function THscParsedFuncs.IndexOf( const AExpression: String ): Integer;
begin
   Result := FList.IndexOf( AExpression );
end;

function THscParsedFuncs.Add( const AExpression: String ): THscParsedFunc;
begin
   Result := THscParsedFunc.Create( FEngine, AExpression );
   FList.AddObject( AExpression, Result );
end;

procedure THscParsedFuncs.Clear;
var  i: Integer;
begin
   for i:=0 to FList.Count-1 do THscParsedFunc( FList.Objects[i] ).Free;
   FList.Clear;
end;

constructor THscParsedFuncs.Create(const AEngine: THscEngine);
begin
   inherited Create( AEngine );
   FList := TStringList_ExactMatch.Create;
   FList.Sorted := True;
   FList.Duplicates := dupIgnore;
end;

destructor THscParsedFuncs.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;


// ------------------------------------------------------- THscSchedEntry -----

function THscSchedEntry.Trigger: Boolean;
var  dNow: TDateTime;
     iNow: Integer;
     sNow: String;
     i: Integer;
begin
   Result := False;

   dNow := Now;
   iNow := DateTimeToUnixTime( dNow );

   if ( iNow - FLastTriggered ) < 60 then exit; // max. once per minute

   i := DayOfWeek( dNow ) - 1;
   if i=0 then i:=7;
   if (i<=length(FAtDays)) and (FAtDays[i]='0') then exit;

   sNow := FormatDateTime( 'hh"."nn', dNow );
   if FFromTime<=FTilTime then begin
      if (sNow<FFromTime) or  (sNow>FTilTime) then exit;
   end else begin
      if (sNow<FFromTime) and (sNow>FTilTime) then exit;
   end;

   i := ( iNow - FLastTriggered ) div 60;
   if i<FEveryMins then exit;

   Result := True;
   if Result then FLastTriggered := iNow;
end;

constructor THscSchedEntry.Create( const AEngine: THscEngine;
                                   const AKeyword: String;
                                   const AFromTime, ATilTime, AAtDays: String;
                                   const AEveryMins: Integer;
                                   const AImmediate: Boolean );
begin
   inherited Create( AEngine );

   FKeyword   := AKeyword;
   FFromTime  := AFromTime;
   FTilTime   := ATilTime;
   FAtDays    := AAtDays;
   FEveryMins := AEveryMins;
   FImmediate := AImmediate;

   if FImmediate then FLastTriggered := 0
                 else FLastTriggered := DateTimeToUnixTime( Now );
end;

// -------------------------------------------------------- THscScheduler -----

function THscScheduler.Add( const Keyword: String;
                            FromTime, TilTime, AtDays: String;
                            EveryMins: Integer;
                            const Immediate: Boolean ): Integer;
var  HSE: THscSchedEntry;
begin
   while length(FromTime)<5 do FromTime:=FromTime+' ';
   if TilTime='' then TilTime:=FromTime;
   while length(TilTime )<5 do TilTime :=TilTime +'Z';
   FromTime[3] := '.';
   TilTime[3]  := '.';
   while length(AtDays)<7 do AtDays:=AtDays+'1';
   if EveryMins<0 then EveryMins:=0;

   HSE := THscSchedEntry.Create( Engine, Keyword,
                                 FromTime, TilTime, AtDays,
                                 EveryMins, Immediate );
   Result := FList.Add( HSE );
end;

procedure THscScheduler.Clear;
begin
   while FList.Count>0 do begin
      THscSchedEntry( FList[ FList.Count-1 ] ).Free;
      FList.Delete( FList.Count-1 );
   end;
end;

function THscScheduler.Check: String;
var  i: Integer;
begin
   Result := '';
   for i:=0 to FList.Count-1 do begin
      if THscSchedEntry( FList[i] ).Trigger then begin
         Result := THscSchedEntry( FList[i] ).Keyword;
         break;
      end;
   end;
end;

constructor THscScheduler.Create( const AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FList := TList.Create;
end;

destructor THscScheduler.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;

{JW} {AtEntrys}
function THscScheduler.GetCount: Integer;
begin
  Result := Flist.Count;
end;

function THscScheduler.GetEntrySub(const Entry:Integer): String;
begin
  Result := THscSchedEntry( FList[Entry] ).Keyword;
end;

function THscScheduler.GetEntryFromTime (const Entry:Integer): String;
begin
  Result := THscSchedEntry( FList[Entry] ).FromTime;
end;

function THscScheduler.GetEntryTilTime (const Entry:Integer): String;
begin
  Result := THscSchedEntry( FList[Entry] ).TilTime;
end;

function THscScheduler.GetEntryAtDays (const Entry:Integer): String;
begin
  Result := THscSchedEntry( FList[Entry] ).AtDays;
end;

function THscScheduler.GetEntryEveryMins (const Entry:Integer): Integer;
begin
  Result := THscSchedEntry( FList[Entry] ).EveryMins
end;

{JW}
// ------------------------------------------------------- THscParsedFunc -----

procedure THscParsedFunc.DoFuncParse( const LineStr: String );
var  pLineStr: PChar;
begin
   pLineStr := PChar( LineStr );
   WantSplitFunction( Self, pLineStr, FName, FParms );
   FName   := LowerCase( FName );
   FMarker := CmdToMarker( FName );
   FNameOK := False;
end;

function THscParsedFunc.FuncIs( const TestFunc: LongWord;
                                const ParMin, ParMax: Integer ): Boolean;
begin
   Result := False;

   if TestFunc=FuncMarker then begin
      FNameOK := True;
      if FuncPars.Count<ParMin then
         Error( HSCERR_SYNTAX, 'Missing parameters (min. '
                               + inttostr(ParMin) + '): ' + FuncName )
      else if FuncPars.Count>ParMax then
         Error( HSCERR_SYNTAX, 'Too many parameters (max. '
                               + inttostr(ParMax) + '): ' + FuncName )
      else
         Result := True;
   end;
end;

function THscParsedFunc.FuncIs( const TestFunc: String;
                                const ParMin, ParMax: Integer ): Boolean;
begin
   Result := False;
   if length(TestFunc) <> length(FuncName) then exit;
   if TestFunc=FuncName then begin
      FNameOK := True;
      if FuncPars.Count<ParMin then
         Error( HSCERR_SYNTAX, 'Missing parameters (min. '
                               + inttostr(ParMin) + '): ' + FuncName )
      else if FuncPars.Count>ParMax then
         Error( HSCERR_SYNTAX, 'Too many parameters (max. '
                               + inttostr(ParMax) + '): ' + FuncName )
      else
         Result := True;
   end;
end;

function THscParsedFunc.ParX( const Index : Integer;
                              const DefVal: String ): String;
begin
   if Index<FuncPars.Count then Result := FuncPars[Index]
                           else Result := DefVal;
end;

procedure THscParsedFunc.ParH( const Result: THscVariant;
                               const Index : Integer );
begin
   if Index<FuncPars.Count then Engine.SolveExpression(FuncPars[Index],Result)
                           else Result.Unassign;
end;

procedure THscParsedFunc.ParH( const Result: THscVariant;
                               const Index : Integer;
                               const DefVal: Integer );
begin
   if Index<FuncPars.Count then Engine.SolveExpression(FuncPars[Index],Result)
                           else Result.AsInt := DefVal;
end;

procedure THscParsedFunc.ParH( const Result: THscVariant;
                               const Index : Integer;
                               const DefVal: String );
begin
   if Index<FuncPars.Count then Engine.SolveExpression(FuncPars[Index],Result)
                           else Result.AsStr := DefVal;
end;

function THscParsedFunc.ParV( const Index: Integer; const DefVal: Variant ): Variant;
var  HV: THscVariant;
begin
   if Index<FuncPars.Count then begin
      HV := THscVariant.Create;
      try
         Engine.SolveExpression( FuncPars[Index], HV );
         Result := HV.AsVar;
      finally
         HV.Free;
      end;
   end else
      Result := DefVal;
end;

function THscParsedFunc.ParS( const Index: Integer ): String;
var  HV: THscVariant;
begin
   if Index<FuncPars.Count then begin
      HV := THscVariant.Create;
      try
         Engine.SolveExpression( FuncPars[Index], HV );
         Result := HV.AsStr
      finally
         HV.Free;
      end;
   end else
      Result := ''
end;

function THscParsedFunc.ParS( const Index: Integer; const DefVal: String ): String;
var  HV: THscVariant;
begin
   if Index<FuncPars.Count then begin
      HV := THscVariant.Create;
      try
         Engine.SolveExpression( FuncPars[Index], HV );
         If HV.FValTyp = hvtEmpty
            then Result := DefVal
            else Result := HV.AsStr;
      finally
         HV.Free;
      end;
   end else
      Result := DefVal;
end;

function THscParsedFunc.ParI( const Index: Integer ): Integer;
var  HV: THscVariant;
begin
   if Index<FuncPars.Count then begin
      HV := THscVariant.Create;
      try
         Engine.SolveExpression( FuncPars[Index], HV );
         Result := HV.AsInt;
      finally
         HV.Free;
      end;
   end else
      Result := 0
end;

function THscParsedFunc.ParI( const Index: Integer; const DefVal: Integer ): Integer;
var  HV: THscVariant;
begin
   if Index<FuncPars.Count then begin
      HV := THscVariant.Create;
      try
         Engine.SolveExpression( FuncPars[Index], HV );
         If HV.FValTyp = hvtEmpty
            then Result := DefVal
            else Result := HV.AsInt;
      finally
         HV.Free;
      end;
   end else
      Result := DefVal;
end;

constructor THscParsedFunc.Create( const AEngine: THscEngine; const LineStr: String );
begin
   inherited Create(AEngine);
   FNameOK := True;
   FName   := '';
   FMarker := SMARKER_NOSPECIAL;
   FParms  := TStringList.Create;
   try DoFuncParse( LineStr ) except {should never be reached} end;
end;

destructor THscParsedFunc.Destroy;
begin
   if Assigned(FParms) then FParms.Free;
   inherited Destroy;
end;

// ----------------------------------------------------------- THscScript -----

function THscScript.GetCount: Integer;
begin
   Result := FLines.Count;
end;

function THscScript.GetText: String;
begin
   Result := FLines.Text;
end;

function THscScript.GetLine( const Index: Integer ): String;
begin
   Result := FLines[ Index ];
end;

function THscScript.GetLineNo( const Index: Integer ): Integer;
begin
   Result := Integer( FLines.Objects[ Index ] ) and SMARKER_MASK_LINENO;
end;

function THscScript.GetMarkers( const Index: Integer ): LongWord;
begin
   Result := LongWord( FLines.Objects[ Index ] ) and SMARKER_MASK_MARKER;
end;

function THscScript.GetScrLine( const Index: Integer ): THscParsedFunc;
begin
   if not Assigned( FScrLines[Index] ) then begin
      FScrLines[Index] := THscParsedFunc.Create( Engine, FLines[Index] );
   end;
   Result := FScrLines[Index]
end;

procedure THscScript.SetMarkers( const Index: Integer;
                                 const NewMarkers: LongWord );
begin
   FLines.Objects[ Index ] := Pointer(
      LongWord( FLines.Objects[ Index ] ) or (NewMarkers and SMARKER_MASK_MARKER)
   );
end;

function THscScript.GetModID( const Index: Integer ): Integer;
begin
   Result := Integer( FModRefs[ Index ] );
end;

function THscScript.GetModName( const Index: Integer ): String;
begin
   Result := FModules[ ModID[Index] ];
end;

function THscScript.IsModuleLoaded( const ModName: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   for i:=0 to FModules.Count-1 do begin
      if CompareText( ModName, FModules[i] )=0 then begin
         Result := True;
         break;
      end;
   end;
end;

function THscScript.StartNewModule( const ModName: String ): Integer;
begin
   Result := FModules.AddObject( ModName, Pointer(FLines.Count) );
   FCurrLine := 0;
end;

function THscScript.Add( const Line: String ): Integer;
var  k: Integer;
     s, h: String;
begin
   // remove leading/trailing whitespace and comments
   s := TrimWSPEC( Line );
   if (copy(s,1,1)='#') and (copy(s,2,1)<>'!') then s := '';

   // add line
   inc( FCurrLine );
   Result := FLines.AddObject( s, Pointer(FCurrLine and SMARKER_MASK_LINENO) );
   FScrLines.Add( nil );
   FModRefs.Add( Pointer(FModules.Count-1) );
   Markers[Result] := CmdToMarker( ReadLineCmd( s ) );

   // combine continuation lines
   if (Result>0) and (s<>'') then begin
      if copy(s,length(s),1)<>'_' then begin // last line?
         k := Result - 1;
         repeat
            h := FLines[ k ]; // preceding line
            if copy(h,length(h),1)<>'_' then break; // not continued

            // append to previous line, then clear
            h[ length(h) ] := ' ';
            Flines[k] := h + FLines[k+1];
            FLines[k+1] := '';
            Markers[k+1] := SMARKER_NOSPECIAL;

            dec( k );
         until k < 0;
      end;
   end;
end;

procedure THscScript.AddLines( const sl: TStringList );
var  i: Integer;
begin
   for i:=0 to sl.Count-1 do Add( sl[i] );
end;

procedure THscScript.Clear;
var  i: Integer;
begin
   if Assigned( FScrLines ) then begin
      for i:=0 to FScrLines.Count-1 do begin
         if Assigned( FScrLines[i] ) then begin
            THscParsedFunc( FScrLines[i] ).Free;
         end;
      end;
      FScrLines.Clear;
   end;
   FLines.Clear;
   FModRefs.Clear;
   FModules.Clear;
   FCurrLine := 0;
end;

constructor THscScript.Create( const AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FLines    := TStringList.Create;
   FModRefs  := TList.Create;
   FModules  := TStringList.Create;
   FScrLines := TList.Create;
   FCurrLine := 0;
end;

destructor THscScript.Destroy;
begin
   if Assigned( FScrLines ) then begin
      Clear;
      FScrLines.Free;
   end;
   if Assigned( FModules ) then FModules.Free;
   if Assigned( FModRefs ) then FModRefs.Free;
   if Assigned( FLines   ) then FLines  .Free;
   inherited Destroy;
end;

// ----------------------------------------------------------- THscEngine -----

procedure THscEngine.AddError( const Sender: TObject;
                               const AErrNum: Integer;
                               const AErrMsg: String );
begin
   if FLastError=HSCERR_NOERROR then begin
      FErrNum    := AErrNum;
      FErrMsg    := AErrMsg;
      FErrModule := FScript.ModName[FCurrPos];
      FErrLineNo := FScript.LineNo [FCurrPos];
      FErrLine   := FScript.Lines  [FCurrPos];
      FErrSender := Sender.ClassName;

      if not FErrCatch then begin
         FLastError := AErrNum;
         if AErrNum <> HSCERR_NOERROR then
            Engine_Error( Engine, Sender,
                          FErrNum, FErrMsg, FErrModule, FErrLineNo, FErrLine );
      end;
   end;
end;

procedure THscEngine.IPtrPush;
begin
   FIPtrStack.Add( Pointer( FCurrPos ) );
   FIPtrStack.Add( Pointer( FNextPos ) );
end;

procedure THscEngine.IPtrPop;
begin
   if FIPtrStack.Count>1 then begin
      FNextPos := Integer( FIPtrStack[ FIPtrStack.Count-1 ] );
      FIPtrStack.Delete( FIPtrStack.Count-1 );
      FCurrPos := Integer( FIPtrStack[ FIPtrStack.Count-1 ] );
      FIPtrStack.Delete( FIPtrStack.Count-1 );
   end;
end;

procedure THscEngine.LoopStackPush( const Value: Integer );
begin
   FLoopStack.Add( Pointer(Value) );
end;

function THscEngine.LoopStackPop( const Count: Integer ): Integer;
var  i: Integer;
begin
   Result := Integer( FLoopStack[ FLoopStack.Count - 1 ] );
   for i:=1 to Count do FLoopStack.Delete( FLoopStack.Count - 1 );
end;

function THscEngine.LoopStackPeek( const Depth: Integer ): Integer;
begin
   Result := Integer( FLoopStack[ FLoopStack.Count - Depth - 1 ] );
end;

function THscEngine.GetContext: Integer;
begin
   Result := FContextID;
end;

procedure THscEngine.EnterContext;
begin
   inc( FContextID );
end;

procedure THscEngine.LeaveContext;
begin
   if FContextID <= 0 then raise Exception.Create('FContextID out of range!');
   FVariables.ClearContext( Context );
   dec( FContextID );
end;

procedure THscEngine.Trace( const TracePos: Integer; const TraceText: String );
begin
   if FTraceIsOn then begin
      Engine_Trace(
         Engine, FScript.ModName[TracePos], FScript.LineNo[TracePos], TraceText
      );
   end;
end;

procedure THscEngine.SyncExecute;
var  s: String;
     i: Integer;
     b: Boolean;
begin
   FSyncResult := Unassigned;

   if FSyncCmd='msgbox' then begin
      FSyncResult := MessageBox( 0, PChar( FSyncPars[0] ),
                                    PChar( FSyncPars[1] ),
                                    strtoint( FSyncPars[2] ) );
      exit;
   end;

   if FSyncCmd='inputbox' then begin
      s := FSyncPars[2];
      b := InputDlgStr( FSyncPars[0], FSyncPars[1], s, 0 );
      FSyncResult := s;
      if b then FSyncPars[0]:='1' else FSyncPars[0]:='0';
      exit;
   end;

   if FSyncCmd='inputpw' then begin
      s := FSyncPars[2];
      b := InputDlgPwd( FSyncPars[0], FSyncPars[1], s, 0 );
      FSyncResult := s;
      if b then FSyncPars[0]:='1' else FSyncPars[0]:='0';
      exit;
   end;

   if FSyncCmd='listbox' then begin
      i := strtoint( FSyncPars[2] );
      b := InputDlgList( FSyncPars[0], FSyncPars[1], i, FSyncPars[3], 0 );
      FSyncResult := i;
      if b then FSyncPars[0]:='1' else FSyncPars[0]:='0';
      exit;
   end;

   if FSyncCmd='popupbox' then begin
      FSyncResult := PopupBox (PChar( FSyncPars[0] ),
                               PChar( FSyncPars[1] ),
                               strtoint( FSyncPars[2] ),
                               strtoint( FSyncPars[3] ),  
                               strtoint( FSyncPars[4] ) );
      exit;
   end;

end;

procedure THscEngine.SolveExpression( const Expression: String;
                                      const Result    : THscVariant );
var  EX: THscExpression;
begin
   try
      // optimization for some simple, often used expressions
      if IsSimpleNumber( Expression ) then begin // number
         Result.AsInt := strtoint( Expression );
         exit;
      end else if IsSimpleString( Expression ) then begin // string
         Result.AsStr := copy( Expression, 2, length(Expression)-2 );
         exit;
      end else if IsVariable( Expression ) then begin // variable
         Result.Assign( FVariables[ Expression ] );
         //if not Result.Unassigned then
         exit
      end;
   except
      // just fall through, i.e. let THscExpression handle the error
   end;

   EX := THscExpression.Create( Engine );
   try
      EX.Solve( Expression, Result );
   finally
      EX.Free;
   end;
end;

function THscEngine.IndexOfLabel( const LabelName: String ): Integer;
var  i, j: Integer;
     s   : String;
begin
   Result := -1;
   for i:=0 to FScript.Count-1 do begin
      if FScript.Markers[i] = SMARKER_LABEL then begin

         s := TrimWhSpace( FScript.Lines[i] );
         System.Delete( s, 1, 5 );

         j := Pos( '(', s );
         if j=0 then continue;
         System.Delete( s, 1, j );

         j := Pos( ')', s );
         if j=0 then continue;
         s := TrimWhSpace( copy(s,1,j-1) );

         if CompareText( s, LabelName ) = 0 then begin
            Result := i;
            break;
         end;

      end;
   end;
end;

function THscEngine.IndexOfSub( const SubName: String ): Integer;

   function Testname( i: Integer ): Boolean;
   var  t: String;
        p: PChar;
   begin
      Result := False;
      t := FScript.Lines[i];
      p := PChar( t ) + 3{='sub'};
      SkipWSPC( p );
      if strlicomp( @SubName[1], p, length(SubName) ) = 0 then begin
         inc( p, length( SubName ) );
         SkipWSPC( p );
         if (p^=#0) or (p^='(') then begin
            Result := True;
            exit;
         end;
      end;
   end;

var  i: Integer;
     m: LongWord;
begin
   Result := -1;

   for i:=FNextPos to FScript.Count-1 do begin
      m := FScript.Markers[i];
      if (m=SMARKER_SUB) and Testname(i) then begin Result:=i; exit end;
   end;

   for i:=FNextPos-1 downto 0 do begin
      m := FScript.Markers[i];
      if (m=SMARKER_SUB) and Testname(i) then begin Result:=i; exit end;
   end;
end;

procedure THscEngine.SolveFunctionStr( const FuncStr: String;
                                       const Result : THscVariant );
var  ParsedFunc: THscParsedFunc;
     i: Integer;
begin
   i := FEngine.FParsedFuncs.IndexOf( FuncStr );
   if i >= 0 then begin
      ParsedFunc := FEngine.FParsedFuncs.ParsedFunc[i];
   end else begin
      ParsedFunc := FEngine.FParsedFuncs.Add( FuncStr );
   end;

   SolveFunction( ParsedFunc, Result );
end;

procedure THscEngine.SolveFunction( const ParsedFunc: THscParsedFunc;
                                    const Result    : THscVariant );

   procedure do_decodetime;
   var  dt: TDateTime;
        yy, mm, dd, hh, nn, ss, ms: Word;
   begin
      with ParsedFunc do begin
         Result.AsInt := ParI( 0, 0 );
         dt := UnixTimeToDateTime( Result.AsInt );
         DecodeDate( dt, yy, mm, dd );
         DecodeTime( dt, hh, nn, ss, ms );
         if IsVariable(ParX(1,'')) then Variables[ParX(1,'')].AsInt:=yy;
         if IsVariable(ParX(2,'')) then Variables[ParX(2,'')].AsInt:=mm;
         if IsVariable(ParX(3,'')) then Variables[ParX(3,'')].AsInt:=dd;
         if IsVariable(ParX(4,'')) then Variables[ParX(4,'')].AsInt:=hh;
         if IsVariable(ParX(5,'')) then Variables[ParX(5,'')].AsInt:=nn;
         if IsVariable(ParX(6,'')) then Variables[ParX(6,'')].AsInt:=ss;
         if IsVariable(ParX(7,'')) then Variables[ParX(7,'')].AsInt:=DayOfWeek(dt)
      end
   end;

   procedure do_encodetime;
   var  dt: TDateTime;
        yy, mm, dd, hh, nn, ss: Word;
   begin
      with ParsedFunc do begin
         yy := ParI( 0, 0 );
         mm := ParI( 1, 0 );
         dd := ParI( 2, 0 );
         hh := ParI( 3, 0 );
         nn := ParI( 4, 0 );
         ss := ParI( 5, 0 );
         dt := EncodeDate( yy, mm, dd ) + EncodeTime( hh, nn, ss, 0 );
         Result.AsInt := DateTimeToUnixTime( dt )
      end
   end;

var VTransfer: THscVariantArray;
    VSubVars : array of String;
    VSubRefs : array of Boolean;
    HV1, HV2 : THscVariant;
    i, j, k, l, Timeout: Integer;
    s, t, u, v: String;
    OK: Boolean;
    RasDialPars: TRasDialParams;
    SR: TSearchRec;
    TS: TStringList;
    WaitObjects: TWOHandleArray;
    het: THscExpTypes;
begin
   Result.Unassign;

   with ParsedFunc do try

      if FuncName='' then exit;

      FuncNameOK := False;

      // invoke built-in functions
      case FuncName[1] of

       'a': if FuncIs( 'abs', 1, 1 ) then begin
               Result.AsInt := abs( ParI( 0 ) );
            // Sheduling
            end else if FuncIs( 'atadd', 2, 6 ) then begin
               s := ParX( 0, '?' ); // sub-name
               if IsIdentifier(s) then i:=IndexOfSub(s) else i:=-1;
               if i<0 then begin
                  Error( HSCERR_UNSUPPORTED, 'Sub not found: ' + s );
                  Result.AsInt := -1;
               end else begin
                  t := ParS( 1 ); // from-time
                  u := ParS( 2, t  ); // til-time
                  v := ParS( 3, '1111111' ); // days
                  i := ParI( 4, 0  ); // min's
                  j := ParI( 5, 1  ); // immediate
                  Result.AsInt := FScheduler.Add( s, t, u, v, i, (j<>0) );
               end;
            end else if FuncIs( 'atclear', 0, 0 ) then begin
               FScheduler.Clear;
               Result.AsInt := 0;
            end else if FuncIs( 'atexecute', 0, 1 ) then begin
               Result.AsInt := 0;
               Timeout := ParI( 0, -1 );
               repeat
                  // check and execute all times and intervals
                  repeat
                     s := FScheduler.Check;
                     if s<>'' then begin
                        i := IndexOfSub( s );
                        if i>=0 then begin
                           // invoke function within script
                           IPtrPush;
                           Trace( i, FScript.Lines[i] );
                           FNextPos := i+1;
                           EnterContext;
                           ExecuteNextPos( Result );
                           Result.ToInt;
                           LeaveContext;
                           IPtrPop;
                        end else Error( HSCERR_UNSUPPORTED, 'Sub not found: ' + s );
                     end;
                     if AnyError or (Result.AsInt<>0) then break;
                  until s='';
                  if AnyError or (Result.AsInt<>0) then break;

                  // sleep for up to 30 seconds
                  i := 30000;
                  if (Timeout>=0) and (Timeout<i) then i:=Timeout;
                  if FStopEvent<>0 then begin
                     if ThreadShutDown(false,(WaitForSingleObject(FStopEvent,i)=WAIT_OBJECT_0)) then break; //HSR //One_SD_S
                  end else begin
                     if ThreadShutDown(false) then break; //HSR //One_SD_S
                     Sleep(i);
                  end;
                  if Timeout>=0 then begin
                     dec( Timeout, i );
                     if Timeout<=0 then begin Result.AsInt:=-1; break; end;
                  end;
               until False;
            end else if FuncIs( 'atcount', 0, 0 ) then begin
               Result.AsInt := FScheduler.GetCount
            end else if FuncIs( 'atsubfunction', 1, 1 ) then begin
               Result.AsStr := FScheduler.GetEntrySub(ParI(0));
            end else if FuncIs( 'atondays', 1, 1 ) then begin
               Result.AsStr := FScheduler.GetEntryAtDays(ParI(0));
            end else if FuncIs( 'ateverymins', 1, 1 ) then begin
               Result.AsInt := FScheduler.GetEntryEveryMins(ParI(0));
            end else if FuncIs( 'atfrom', 1, 1 ) then begin
               Result.AsStr := FScheduler.GetEntryFromTime(ParI(0));
            end else if FuncIs( 'atuntil', 1, 1 ) then begin
               Result.AsStr := FScheduler.GetEntryTilTime(ParI(0));
            // Article-Functions
            end else if FuncIs( 'artalloc', 0, 1 ) then begin
               s := ParS( 0, '' );
               i := Integer( TArticle.Create );
               Engine.ResControl.Add( TResDelphiObject.Create( TArticle(i) ) );
               if s<>'' then TArticle( i ).Text := s;
               Result.AsInt := i;
            end else if FuncIs( 'artfree', 1, 1 ) then begin
               i := ParI( 0, 0 );
               MsgCheck( i );
               TArticle( i ).Free;
               Engine.ResControl.Remove( RESID_DelphiObject, i );
               Result.AsInt := 0;
            end else if FuncIs( 'artload', 2, 2 ) then begin
               i := ParI( 0, 0  );
               s := ParS( 1, '' );
               MsgCheck( i );
               Result.AsInt := TArticle( i ).LoadFromFile( s );
            end else if FuncIs( 'artsave', 2, 2 ) then begin
               i := ParI( 0, 0  );
               s := ParS( 1, '' );
               MsgCheck( i );
               Result.AsInt := TArticle( i ).SaveToFile( s );
            end else if FuncIs( 'artgettext', 1, 1 ) then begin
               i := ParI( 0, 0 );
               MsgCheck( i );
               Result.AsStr := TArticle( i ).Text;
            end else if FuncIs( 'artsettext', 2, 2 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               MsgCheck( i );
               TArticle( i ).Text := s;
               Result.AsInt := 0;
            end else if FuncIs( 'artgetheaders', 1, 1 ) then begin
               i := ParI( 0, 0 );
               MsgCheck( i );
               Result.AsStr := TArticle( i ).FullHeader;
            end else if FuncIs( 'artsetheaders', 2, 2 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               MsgCheck( i );
               TArticle( i ).FullHeader := s;
               Result.AsInt := 0;
            end else if FuncIs( 'artgetbody', 1, 1 ) then begin
               i := ParI( 0, 0 );
               MsgCheck( i );
               Result.AsStr := TArticle( i ).FullBody;
            end else if FuncIs( 'artsetbody', 2, 2 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               MsgCheck( i );
               TArticle( i ).FullBody := s;
               Result.AsInt := 0;
            end else if FuncIs( 'artgetheader', 2, 3 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               j := ParI( 2, 0 );
               MsgCheck( i );
               if j=0 then Result.AsStr := TArticle(i).HeaderWOCRLF[s]
                      else Result.AsStr := TArticle(i).Header[s];
            end else if FuncIs( 'artaddheader', 3, 3 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               t := ParS( 2, '' );
               MsgCheck( i );
               TArticle( i ).AddHeader( s, t );
               Result.AsInt := 0;
            end else if FuncIs( 'artsetheader', 3, 4 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               t := ParS( 2, '' );
               u := ParS( 3, '' );
               MsgCheck( i );
               With TArticle( i ) do begin
                  If u > '' then RenameHeader(s, u);
                  Header[s] := t
               end;
               Result.AsInt := 0;
            end else if FuncIs( 'artdelheader', 2, 2 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               MsgCheck( i );
               TArticle( i ).DeleteHeader( s );
               Result.AsInt := 0;
            end else if FuncIs( 'artheaderexists', 2, 2 ) then begin
               i := ParI( 0, 0 );
               s := ParS( 1, '' );
               MsgCheck( i );
               Result.AsInt := iif( TArticle( i ).HeaderExists( s ), 1, 0 );
            end;

       'b': if FuncIs( 'beep', 0, 1 ) then begin
               Result.AsInt:=Integer(MessageBeep(LongWord(ParI(0,LongInt($FFFFFFFF)))));
            end;

       'c': if FuncIs( 'chr', 1, 1 ) then begin
               Result.AsStr := chr( ParI( 0 ) and $FF );
            end else if FuncIs( 'copy', 2, 3 ) then begin
               s := ParS( 0 );
               i := ParI( 1 );
               j := ParI( 2, length(s) );
               Result.AsStr := copy( s, i, j );
            end else if FuncIs( 'clipread', 0, 0) then begin
               if Clipboard.HasFormat(CF_TEXT)
                  then Result.AsStr := Clipboard.AsText
                  else Result.AsStr := ''
            end else if FuncIs( 'clipwrite', 1, 1) then begin
               Result.AsStr := ParS( 0 ); // variable
               Clipboard.clear;
               Clipboard.AsText := Result.AsStr;
            end else if FuncIs( 'clearxcounter', 1, 2 ) then begin
               j := ParI(0);
               k := ParI(1, j);
               If (j >= 0) and (j <= 9) and (k >= 0) and (k <= 9) then begin
                  For i := j to k do SetCounter ( CntCustomValues[i], 0 );
                  Result.asInt := 0
               end else Result.asInt := -1;
            end;

       'd': if FuncIs( 'dec', 1, 2 ) then begin
               s := ParX( 0, '?' );
               if IsVariable( s ) then begin
                  HV1 := Variables[ s ];
                  Result.AsInt := HV1.AsInt - ParI( 1, 1 );
                  HV1.AsInt := Result.AsInt;
               end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
            end else if FuncIs( 'diskfreekb', 1, 1 ) then begin
               s := UpperCase(ParS( 0 ));
               If (s > '') and (s[1] IN ['A'..'Z'])
                  then Result.AsInt := Integer(DiskFree(Ord(s[1])-Ord('A')+1) div 1024)
                  else Result.AsInt := -1
            end else if FuncIs( 'decxcounter', 1, 2 ) then begin
               j := ParI(0);
               If (j >= 0) and (j <= 9) then begin
                  SetCounter ( CntCustomValues[j], CntCustomValues[j] - ParI(1, 1) );
                  Result.asInt := 0
               end else Result.asInt := -1;
            end else if FuncIs( 'delete', 2, 3 ) then begin
               s := ParS( 0 );
               i := ParI( 1 );
               j := ParI( 2, length(s) );
               System.Delete( s, i, j );
               Result.AsStr := s;
            end else if FuncIs( 'decodetime', 2, 8 ) then begin
               do_decodetime;
            end else if FuncIs( 'digest', 2, 3 ) then begin
               i := ParI( 0  ); // type
               s := ParS( 1 ); // data
               j := ParI( 2, 0  ); // as hex
               case i of
                  0: if j=0 then Result.AsInt := StrToCRC32(s)
                            else Result.AsStr := LowerCase(IntToHex(StrToCRC32(s),8));
                  1: if j=0 then Result.AsStr := MD5OfStr(s)
                            else Result.AsStr := MD5toHex( MD5OfStr(s) );
                  2: if j=0 then Result.AsStr := SHA1OfStr(s)
                            else Result.AsStr := SHA1toHex( SHA1OfStr(s) );
                  else Error( HSCERR_SYNTAX, 'First parameter is invalid.' );
               end;
            end else if FuncIs( 'direxists', 1, 1 ) then begin
               try
                  if DirExists2(ParS(0)) then Result.AsInt:=1 else Result.AsInt:=0;
               except
                  Result.AsInt := 0;
               end;
            end else if FuncIs( 'dirmake', 1, 1 ) then begin
               Result.AsInt := 0;
               try MkDir( ParS(0) ) except Result.AsInt:=-1 end;
            end else if FuncIs( 'dirremove', 1, 1 ) then begin
               Result.AsInt := 0;
               try RmDir( ParS(0) ) except Result.AsInt:=-1 end;
            end else if FuncIs( 'dirchange', 1, 1 ) then begin
               Result.AsInt := 0;
               try ChDir( ParS(0) ) except Result.AsInt:=-1 end;
            end else if FuncIs( 'dircurrent', 0, 0 ) then begin
               Result.AsStr := '';
               try GetDir( 0, s ); Result.AsStr:=s except Result.AsStr:='' end;
               if s > '' then Result.AsStr:=IncludeTrailingBackslash(s)
                         else Result.AsStr:=''
            end else if FuncIs( 'dirwindows', 0, 0 ) then begin
               Result.AsStr := GetWindowsPath;
            end else if FuncIs( 'dirsystem', 0, 0 ) then begin
               Result.AsStr := GetSystemPath;
            end else if FuncIs( 'dllload', 1, 1 ) then begin
               s := ParS( 0 ); // path/name of dll to load
               Result.AsInt := Integer( DynDllLoad( s,DWORD(FDllLastError) ) );
               if Result.AsInt <> 0 then
                  Engine.ResControl.Add( TResDynamicDll.Create(Result.AsInt) );
            end else if FuncIs( 'dllfree', 1, 1 ) then begin
               i := ParI( 0 ); // handle of dll to free
               Result.AsInt := Integer( DynDllFree( i ) );
               Engine.ResControl.Remove( RESID_DynamicDll, i );
            end else if FuncIs( 'dllcall', 1, 12 ) then begin
               s := ParS( 0 ); // declaration of dll-proc
               Result.AsInt := Integer( DynDllCall( s,
                  [ DWORD(ParI(1,0)), DWORD(ParI(2,0)), DWORD(ParI(3,0)), DWORD(ParI( 4,0)),
                    DWORD(ParI(5,0)), DWORD(ParI(6,0)), DWORD(ParI(7,0)), DWORD(ParI( 8,0)),
                    DWORD(ParI(9,0)), DWORD(ParI(10,0)), DWORD(ParI(11,0)) ],
                  DWORD(FDllLastError) ));
            end else if FuncIs( 'dlllasterror', 0, 0 ) then begin
               Result.AsInt := FDllLastError;
            end else if FuncIs( 'deletehostsentry', 2, 2 ) then begin
                if (ParS(0)<>'') and  (ParS(0)<>'')
                   then Result.asInt:=deletehostsentry(ParS( 0, '' ),ParS( 1, '' ))
                   else Result.asInt:=-1;
            end;

       'e': if FuncIs( 'encodetime', 6, 6 ) then begin
               do_encodetime;
            end else if FuncIs( 'eval', 1, 1 ) then begin
               s := ParS( 0 );
               SolveExpression( s, Result );
            end else if FuncIs( 'execute', 1, 5 ) then begin
               s := ParS( 0 ); // command-line
               t := ParS( 1, '' ); // working-dir
               i := ParI( 2, SW_SHOWNORMAL ); // showwindow
               j := ParI( 3, 1 ); // waitflag (default=wait)
               Result.AsInt := ExecuteProcess( s, t, i, (j<>0), k, FStopEvent, l);
               If l > 0 then begin
                  Engine_Warning ( Engine, TrGlF(kLog, 'Warning.OSErrorOnExecute',
                     'OS-Error #%s ("%s") when execute "%s"',
                     [IntToStr(l), WindowsErrorText(l), s] ) )
               end;
               if (Result.AsInt=0) and (FuncPars.Count>=5) then begin
                  s := ParX( 4, '' ); // var for exitcode
                  if IsVariable(s) then Variables[s].AsInt := k;
               end;
            // Error-handling
            end else if FuncIs( 'errcatch', 0, 1 ) then begin
               if FuncPars.Count>0 then begin
                  FErrCatch  := ( ParI( 0, 0 ) <> 0 );
                  FErrNum    := HSCERR_NOERROR;
                  FErrMsg    := 'No error';
                  FErrModule := '';
                  FErrLineNo := -1;
                  FErrLine   := '';
                  FErrSender := '';
               end;
               Result.AsInt := Integer( FErrCatch );
            end else if FuncIs( 'errnum', 0, 0 ) then begin
               Result.AsInt := FErrNum;
            end else if FuncIs( 'errmsg', 0, 0 ) then begin
               Result.AsStr := FErrMsg;
            end else if FuncIs( 'errmodule', 0, 0 ) then begin
               Result.AsStr := FErrModule;
            end else if FuncIs( 'errlineno', 0, 0 ) then begin
               Result.AsInt := FErrLineNo;
            end else if FuncIs( 'errline', 0, 0 ) then begin
               Result.AsStr := FErrLine;
            end else if FuncIs( 'errsender', 0, 0 ) then begin
               Result.AsStr := FErrSender;
            // Events
            end else if FuncIs( 'eventcreate', 1, 4 ) then begin
               s := ParS( 0 ); // event-name
               i := ParI( 1, 1  ); // manual reset
               j := ParI( 2, 0  ); // initial state
               {JW} {critical event}
               EnterCriticalSection(CS_Event);
               Result.asInt := Integer( CreateEvent( nil, i=0, j<>0, PChar(s) ) );
               LeaveCriticalSection(CS_Event);
               {JW}
               i := GetLastError;
               s := ParX( 3, '?' ); // error-var
               if IsVariable( s ) then Variables[ s ].asInt := i;
            end else if FuncIs( 'eventwait', 1, 2)  then begin
               i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(ParS(0)));
               if i<>0 then begin
                  WaitObjects[0]:=i;
                  WaitObjects[1]:=FStopEvent;
                  if FStopEvent=0 then j:=1 else j:=2;
                  k:=ParI(1,-1);
                  If k=-1
                     then Result.asInt:=Integer(WaitForMultipleObjects(j, @WaitObjects,False,INFINITE))
                     else Result.asInt:=Integer(WaitForMultipleObjects(j, @WaitObjects,False,k));
                  if (Result.asInt<>Wait_Timeout)
                      and (Result.asInt<>Wait_Object_0)
                      and (Result.asInt<>Wait_Object_0+1) {JW} {Multiple Event}
                  then begin
                      Result.asInt:=Integer(GetLastError)
                  end;
                  CloseHandle(i);
               end else begin
                  Result.asInt:=Integer(GetLastError)
               end
            end else if FuncIs( 'eventmultiplewait', 0, 10 ) then begin
               Timeout := ParI( 0, Integer(INFINITE) );
               i := FuncPars.Count - 1{Timeout} + 1{FStopEvent};
               WaitObjects[0] := FStopEvent;
               for k:=1 to FuncPars.Count-1 do WaitObjects[k]:=ParI(k,0);
               k := WaitForMultipleObjects( i, @WaitObjects, False, Timeout );
               case DWORD( k ) of
                  WAIT_FAILED  : begin Result.AsInt:=-2; FDllLastError:=GetLastError end;
                  WAIT_TIMEOUT : Result.AsInt := -1; // timeout
                  WAIT_OBJECT_0: Result.AsInt :=  0; // FStopEvent
                  WAIT_OBJECT_0+1 .. WAIT_OBJECT_0+10:
                     Result.AsInt := k - WAIT_OBJECT_0; // one of given objects
               end;
            end else if FuncIs( 'eventreset', 1, 1)  then begin
               i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(
               ParS(0)));
               if i <> 0 then begin
                  {JW} {critical event}
                  EnterCriticalSection(CS_Event);
                  if ResetEvent(i)
                     then Result.asInt:=0
                     else Result.asInt:=Integer(GetLastError);
                  CloseHandle(i);
                  LeaveCriticalSection(CS_Event);
                  {JW}
               end else begin
                  Result.asInt:=-1
               end;
            end else if FuncIs( 'eventset', 1, 1)  then begin
               i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(ParS(0)));
               if i <> 0 then begin
                  {JW} {critical event}
                  EnterCriticalSection(CS_Event);
                  if SetEvent(i)
                     then Result.asInt:=0
                     else Result.asInt:=Integer(GetLastError);
                  CloseHandle(i);
                  LeaveCriticalSection(CS_Event);
                  {JW}
               end else Result.asInt:=-1;
            end else if FuncIs( 'eventpulse', 1, 1)  then begin
               i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(
               ParS(0)));
               if i <> 0 then begin
                  {JW} {critical event}
                  EnterCriticalSection(CS_Event);
                  if PulseEvent(i)
                     then result.asInt:=0
                     else result.asInt:=Integer(GetLastError);
                  CloseHandle(i);
                  LeaveCriticalSection(CS_Event);
                  {JW}
               end else result.asInt:=-1;
            end else if FuncIs( 'eventclose', 1, 1)  then begin
               {JW} {critical event}
               EnterCriticalSection(CS_Event);
               If CloseHandle(ParI(0))
                  then result.asInt := 1
                  else result.asInt := 0;
               LeaveCriticalSection(CS_Event);
               {JW}
            end;

       'f': if FuncIs( 'false', 0, 0 ) then begin
               Result.AsInt := 0;
            end else if FuncIs( 'fileexists', 1, 1 ) then begin
               try
                  if FileExistsByPattern(ParS(0)) then Result.AsInt:=1 else Result.AsInt:=0;
                  // Andere Alternative: FileExists2
               except
                  Result.AsInt := 0;
               end;
            end else if FuncIs( 'filedelete', 1, 1 ) then begin
               Result.AsInt := 0;
               s := ParS( 0 );
               if not Windows.DeleteFile( PChar(s) ) then Result.AsInt:=Integer(GetLastError);
            end else if FuncIs( 'filerename', 2, 2 ) then begin
               Result.AsInt := 0;
               s := ParS( 0 );
               t := ParS( 1 );
               if not Windows.MoveFile( PChar(s), PChar(t) ) then Result.AsInt:=Integer(GetLastError);
            end else if FuncIs( 'filecopy', 2, 2 ) then begin
               Result.AsInt := 0;
               s := ParS( 0 );
               t := ParS( 1 );
               if not Windows.CopyFile( PChar(s), PChar(t), False ) then Result.AsInt:=Integer(GetLastError);

            end else if FuncIs( 'filesize', 1, 1 ) then begin
               Result.AsInt := -1;
               if FindFirst( ParS(0), faAnyFile, SR ) = 0 then begin
                  Result.AsInt := SR.Size;
                  SysUtils.FindClose( SR );
               end;
            end else if FuncIs( 'filetime', 1, 1 ) then begin
               Result.AsInt := -1;
               if FindFirst( ParS(0), faAnyFile, SR ) = 0 then begin
                  Result.AsInt := DateTimeToUnixTime( FileDateToDateTime(SR.Time) );
                  SysUtils.FindClose( SR );
               end;
            end;

       'g': if FuncIs( 'gosub', 1, 1 ) then begin
               s := ParX( 0, '?' );
               i := IndexOfLabel( s );
               if IsIdentifier(s) and (i>=0) then begin
                  IPtrPush;
                  FNextPos := i;
                  ExecuteNextPos( Result );
                  IPtrPop;
               end else Error( HSCERR_LABELNOTFOUND, 'Label not found: ' + s );
            end else if  FuncIs( 'getenvironment', 1, 1 ) or FuncIs( 'getenv', 1, 1 ) then begin
               Result.AsStr:='';
               t := ParS( 0 );
               i := GetEnvironmentVariable(PChar(t),NIL,0);
               SetLength(s, i-1);
               GetEnvironmentVariable(PChar(t),@s[1],i);
               Result.AsStr := s
            end else if FuncIs( 'getprocessidentifier', 0, 0 ) then begin
               Result.AsStr := Mutexstring
            end;

       'h': if FuncIs( 'hex', 1, 2 ) then begin
               Result.AsStr := inttohex( ParI(0), ParI(1,1) );
            end;

       'i': if FuncIs( 'inc', 1, 2 ) then begin
               s := ParX( 0, '?' );
               if IsVariable( s ) then begin
                  HV1 := Variables[ s ];
                  Result.AsInt := HV1.AsInt + ParI( 1, 1 );
                  HV1.AsInt := Result.AsInt;
               end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
            end else if FuncIs( 'incxcounter', 1, 2 ) then begin
               j := ParI(0);
               If (j >= 0) and (j <= 9) then begin
                  SetCounter ( CntCustomValues[j], CntCustomValues[j] + ParI(1, 1) );
                  Result.asInt := 0
               end else Result.asInt := -1;
            end else if FuncIs( 'isint', 1, 1 ) then begin
               Result.AsInt := Integer( VarToHscVarType( ParV(0,Unassigned) ) = hvtInteger );
            end else if FuncIs( 'isstr', 1, 1 ) then begin
               Result.AsInt := Integer( VarToHscVarType( ParV(0,Unassigned) ) = hvtString );
            end else if FuncIs( 'int', 1, 2 ) then begin
               s := ParS(0); // expression
               try
                  Result.AsInt := SysUtils.StrToInt( s );
               except
                  ParH( Result, 1, 0 ); // error default
               end;
            end else if FuncIs( 'iif', 3, 3 ) then begin
               if ParI(0)<>0 then ParH( Result, 1, 0 )
                             else ParH( Result, 2, 0 );
            end else if FuncIs( 'icase', 3, 999 ) then begin
               Result.AsInt := 0;
               HV1 := THscVariant.Create;
               HV2 := THscVariant.Create;
               try
                  ParH( HV1, 0, 0 );
                  i := 1;
                  while (i+1<FuncPars.Count) and not(AnyError) do begin
                     OK := False;
                     if LowerCase( ParX(i,'') )='else' then begin
                        OK := True;
                     end else begin
                        ParH( HV2, i, 0 );
                        HscCalculate( Self, HV1, '==', HV2, het, HV2 );
                        if (het=hetINT) and (HV2.AsInt<>0) then OK:=True;
                     end;
                     if OK then begin
                        ParH( Result, i+1, 0 );
                        break;
                     end;
                     inc( i, 2 );
                  end;
               finally
                  HV2.Free;
                  HV1.Free;
               end;
            end else if FuncIs( 'iniread', 4, 4 ) then begin
               s := ParS( 0, '' ); // ini-filename
               t := ParS( 1, '' ); // section
               u := ParS( 2 ); // identifier
               v := ParS( 3 ); // default-value
               if s='' then s:='HScripts.ini';
               if t='' then t:='All';
               With TIniFile.Create( ExpandFilename(s) ) do try
                  Result.AsStr := ReadString( t, u, v )
               finally Free end;
            end else if FuncIs( 'iniwrite', 4, 4 ) then begin
               s := ParS( 0, '' ); // ini-filename
               t := ParS( 1, '' ); // section
               u := ParS( 2 ); // identifier
                // value
               if s='' then s:='HScripts.ini';
               if t='' then t:='All';
               try
                  With TIniFile.Create( ExpandFilename(s) ) do try
                     WriteString( t, u, ParS( 3, '' ) );
                  finally Free end;
                  Result.AsInt := 0
               except
                  Result.AsInt := -1
               end
            end else if FuncIs( 'inidelete', 3, 3 ) then begin
               s := ParS( 0, '' ); // ini-filename
               t := ParS( 1, '' ); // section
               u := ParS( 2 ); // identifier
               if s='' then s:='HScripts.ini';
               if t='' then t:='All';
               try
                  With TIniFile.Create( ExpandFilename(s) ) do try
                     DeleteKey( t, u )
                  finally Free end;
                  Result.AsInt := 0
               except
                  Result.AsInt := -1
               end
            end else if FuncIs( 'inierasesection', 2, 2 ) then begin
               s := ParS( 0 ); // ini-filename
               t := ParS( 1 ); // section
               if s='' then s:='HScripts.ini';
               if t='' then t:='All';
               try
                  With TIniFile.Create( ExpandFilename(s) ) do try
                     EraseSection( t )
                  finally Free end;
                  Result.AsInt := 0
               except
                  Result.AsInt := -1
               end
            end else if FuncIs( 'inputbox', 1, 4 ) then begin
               s := ParS( 0 );
               t := ParS( 1, 'Hamster-Script' );
               u := ParS( 2, '' );
               v := ParX( 3, '?' );
               FSyncCmd := 'inputbox';
               FSyncPars.Clear;
               FSyncPars.Add( t );
               FSyncPars.Add( s );
               FSyncPars.Add( u );
               FSyncResult := Unassigned;
               Engine_SyncExec( Engine, SyncExecute );
               Result.AsVar := FSyncResult;
               if IsVariable(v) then Variables[v].AsInt := strtoint(FSyncPars[0]);
            end else if FuncIs( 'inputpw', 1, 4 ) then begin
               s := ParS( 0 );
               t := ParS( 1, 'Hamster-Script' );
               u := ParS( 2, '' );
               v := ParX( 3, '?' );
               FSyncCmd := 'inputpw';
               FSyncPars.Clear;
               FSyncPars.Add( t );
               FSyncPars.Add( s );
               FSyncPars.Add( u );
               FSyncResult := Unassigned;
               Engine_SyncExec( Engine, SyncExecute );
               Result.AsVar := FSyncResult;
               if IsVariable(v) then Variables[v].AsInt := strtoint(FSyncPars[0]);
            end;

       'l': if FuncIs( 'len', 1, 1 ) then begin
               Result.AsInt := length( ParS(0) );
            end else if FuncIs( 'lowercase', 1, 1 ) then begin
               Result.AsStr := LowerCase( ParS(0) );
            end else if FuncIs( 'listalloc', 0, 2 ) then begin
               i := ParI( 0, 0 ); // sorted (def.=not)
               j := ParI( 1, 1 ); // duplicates (def.=allowed)
               Result.AsInt := FHscLists.ListAlloc( i<>0, j<>0 );
            end else if FuncIs( 'listfree', 1, 1 ) then begin
               Result.AsInt := FHscLists.ListFree( ParI(0) );
            end else if FuncIs( 'listexists', 1, 2 ) then begin
               if FHscLists.ListExists( ParI(0) )
                  then Result.AsInt:=1
                  else Result.AsInt:=0;
               if (Result.AsInt<>0) and (FuncPars.Count>1) then begin
                  i := ParI( 1, 0 );
                  if (i<0) or (i>=FHscLists.List[ParI(0,-1)].Count) then Result.AsInt:=0;
               end;
            end else if FuncIs( 'listclear', 1, 1 ) then begin
               Result.AsInt := 0;
               FHscLists.List[ ParI(0) ].Clear;
            end else if FuncIs( 'listcount', 1, 1 ) then begin
               Result.AsInt := FHscLists.List[ ParI(0) ].Count;
            end else if FuncIs( 'listget', 2, 2 ) then begin
               Result.AsStr := FHscLists.ListItem[ ParI(0), ParI(1) ];
            end else if FuncIs( 'listset', 2, 3 ) then begin
               Result.AsStr := ParS( 2, '' );
               FHscLists.ListItem[ ParI(0), ParI(1) ] := Result.AsStr;
            end else if FuncIs( 'listgettag', 2, 2 ) then begin
               Result.AsInt := FHscLists.ListTag[ ParI(0), ParI(1) ];
            end else if FuncIs( 'listsettag', 3, 3 ) then begin
               Result.AsInt := ParI( 2 );
               FHscLists.ListTag[ ParI(0), ParI(1) ] := Result.AsInt;
            end else if FuncIs( 'listgetkey', 2, 2 ) then begin
               Result.AsStr := FHscLists.List[ ParI(0) ].Values[ ParS(1) ];
            end else if FuncIs( 'listsetkey', 2, 3 ) then begin
               Result.AsStr := ParS( 2, '' );
               FHscLists.List[ ParI(0) ].Values[ ParS(1) ] := Result.AsStr;
            end else if FuncIs( 'listadd', 1, 2 ) then begin
               Result.AsInt := FHscLists.List[ ParI(0) ].Add( ParS(1,'') );
            end else if FuncIs( 'listdelete', 2, 2 ) then begin
               Result.AsInt := 0;
               FHscLists.List[ ParI(0) ].Delete( ParI(1) );
            end else if FuncIs( 'listinsert', 2, 3 ) then begin
               Result.AsStr := ParS( 2, '' );
               FHscLists.List[ ParI(0) ].Insert( ParI(1), Result.AsStr );
            end else if FuncIs( 'listsort', 1, 1 ) then begin
               Result.AsInt := 0;
               FHscLists.List[ ParI(0) ].Sort;
            end else if FuncIs( 'listsettext', 2, 2 ) then begin
               Result.AsStr := ParS( 1 );
               FHscLists.List[ ParI(0) ].Text := Result.AsStr;
            end else if FuncIs( 'listgettext', 1, 1 ) then begin
               Result.AsStr := FHscLists.List[ ParI(0) ].Text;
            end else if FuncIs( 'listindexof', 2, 2 ) then begin
               Result.AsInt := FHscLists.List[ ParI(0) ].IndexOf( ParS(1) );
            end else if FuncIs( 'listload', 2, 2 ) then begin
               i := ParI( 0 );
               s := ParS( 1 );
               if Assigned( FHscLists.List[i] ) then begin
                  if FileExists2(s) then begin
                     try
                        FHscLists.List[i].LoadFromFile( s );
                        Result.AsInt := 0 // OK
                     except
                        On E:Exception do begin
                           Log( LOGID_INFO, '{' + FScriptName + '} '
                            + 'Listload failed: '+ E.Message );
                           Result.AsInt := -2 // Error when Loading
                        end
                     end
                  end else begin
                     Log( LOGID_INFO, '{' + FScriptName + '} Listload failed: "'+s+'" doesn''t exists' );
                     Result.AsInt := -3 // File doesn't exists
                  end
               end else begin
                  Result.AsInt := -1; // Unassigned listhandle
               end;
            end else if FuncIs( 'listsave', 2, 2 ) then begin
               i := ParI( 0 );
               s := ParS( 1 );
               if Assigned( FHscLists.List[i] ) then begin
                  try
                     FHscLists.List[i].SaveToFile( s );
                     Result.AsInt := 0;
                  except
                     On E:Exception do begin
                        Log( LOGID_INFO, '{' + FScriptName + '} '
                         + 'Listsave failed: '+ E.Message );
                        Result.AsInt := -2 // Error when Saving
                     end
                  end;
               end else begin
                  Result.AsInt := -1; // Unassigned List-Handle
               end;
            end else if FuncIs( 'listappend', 2, 2 ) then begin
               i := ParI( 0 );
               s := ParS( 1 );
               if Assigned( FHscLists.List[i] ) then begin
                  try
                     FHscLists.List[i].AppendToFile( s );
                     Result.AsInt := 0;
                  except
                     On E:Exception do begin
                        Log( LOGID_INFO, '{' + FScriptName + '} '
                         + 'Listappend failed: '+ E.Message );
                        Result.AsInt := -2 // Error when Saving
                     end
                  end;
               end else begin
                  Result.AsInt := -1; // Unassigned List-Handle
               end;
            end else if FuncIs( 'listfiles', 2, 3 )
                     or FuncIs( 'listdirs',  2, 3 )
            then begin
               Result.AsInt := 0;
               i := ParI( 0 );
               s := ParS( 1 );
               j := ParI( 2, 0 );
               try
                  if j=0 then t:='' else begin
                     t := ExtractFilePath( ExpandFilename( s ) );
                  end;
                  if FuncIs( 'listfiles', 2, 3 ) then k:=faAnyFile-faDirectory
                                                 else k:=faDirectory;
                  if FindFirst( s, faAnyFile-faVolumeID, SR ) = 0 then begin
                     repeat
                        OK := True;
                        if (k<>faDirectory) then begin
                           if (SR.Attr and faDirectory) <> 0 then OK:=False;
                        end else begin
                           if (SR.Attr and faDirectory) = 0 then OK:=False;
                        end;
                        if (k=faDirectory) and (SR.Name[1]='.') then OK:=False;
                        if OK then FHscLists.List[ i ].Add( t + SR.Name );
                     until FindNext( SR ) <> 0;
                     SysUtils.FindClose( SR );
                  end;
                  Result.AsInt := FHscLists.List[ i ].Count;
               except
                  Result.AsInt := -2;
               end;
            end else if FuncIs( 'listrasentries', 1, 1 ) then begin
               Result.AsInt := 0;
               i := ParI( 0 );
               TS := FHscLists.List[ i ];
               if Assigned(TS) then begin
                  if RasDynEnumPhonebookEntries( TS ) then begin
                     Result.AsInt := TS.Count;
                  end;
               end;
            end else if FuncIs( 'listbox', 2, 5 ) then begin
               Result.AsStr := '';
               i := ParI( 0 ); // list
               if not FHscLists.ListExists(i) then begin
                  Error( HSCERR_INVALIDEXPRESSION, 'List expected as first parameter!' );
               end else begin
                  s := ParS( 1 ); // prompt
                  t := ParS( 2, 'Hamster-Script' ); // caption
                  u := inttostr( ParI( 3, -1 ) ); // default-index
                  v := ParX( 4, '?' ); // code-var
                  FSyncCmd := 'listbox';
                  FSyncPars.Clear;
                  FSyncPars.Add( t );
                  FSyncPars.Add( s );
                  FSyncPars.Add( u );
                  FSyncPars.Add( FHscLists.List[ ParI(0,-1) ].Text );
                  FSyncResult := Unassigned;
                  Engine_SyncExec( Engine, SyncExecute );
                  Result.AsVar := FSyncResult;
                  if IsVariable(v) then Variables[v].AsInt := strtoint(FSyncPars[0]);
               end;
            end else if FuncIs( 'localhostname', 0, 0 ) then begin
               Result.asStr:=LookupLocalHostName;
            end else if FuncIs( 'localhostaddr', 0, 0 ) then begin
               Result.asStr:=nAddrToStr(LookupLocalHostAddr);
            end else if FuncIs( 'lookuphostaddr', 1, 1 ) then begin
               Result.asStr:=nAddrToStr(LookupHostAddr(ParS(0)));
            end else if FuncIs( 'lookuphostname', 1, 1 ) then begin
               Result.asStr:=LookupHostName(StrToAddr(Pars(0)));
            end;

       'm': if FuncIs( 'memalloc', 1, 1 ) then begin
               i := ParI( 0 ); // size
               k := Integer( GlobalAlloc( GMEM_FIXED or GMEM_ZEROINIT, i ) );
               Result.AsInt := k;
               if k<>0 then Engine.ResControl.Add( TResGlobalMemory.Create( k ) );
            end else if FuncIs( 'memfree', 1, 1 ) then begin
               i := ParI( 0 ); // pointer
               Engine.ResControl.Remove( RESID_GlobalMemory, i );
               Result.AsInt := Integer( GlobalFree( i ) );
            end else if FuncIs( 'memforget', 1, 1 ) then begin
               i := ParI( 0 ); // pointer
               Engine.ResControl.Remove( RESID_GlobalMemory, i );
               Result.AsInt := i;
            end else if FuncIs( 'memsize', 1, 1 ) then begin
               i := ParI( 0 ); // pointer
               Result.AsInt := Integer( GlobalSize( i ) );
            end else if FuncIs( 'memsetint', 2, 2 ) then begin
               i := ParI( 0 );      // destination-pointer
               Result.AsInt := ParI( 1 ); // source-int.
               PInteger( i )^ := Result.AsInt;
            end else if FuncIs( 'memgetint', 1, 1 ) then begin
               i := ParI( 0 ); // source-pointer
               Result.AsInt := PInteger( i )^;
            end else if FuncIs( 'memsetstr', 2, 3 ) then begin
               i := ParI( 0 );           // destination-pointer
               s := ParS( 1 );          // source-str.
               j := ParI( 2, length(s)+1 ); // max. size (+1=#0)
               Result.AsStr := s;
               s := s + #0;
               if j > length(s) then j := length(s);
               if j > 0 then Move( s[1], Pointer(i)^, j );
            end else if FuncIs( 'memgetstr', 1, 2 ) then begin
               i := ParI( 0 );  // source-pointer
               j := ParI( 1, -1 ); // max. size; -1=ASCIIZ
               if j=-1 then begin
                  s := String( PChar(i) );
               end else begin
                  SetLength( s, j );
                  If j > 0 then Move( Pointer(i)^, s[1], j );
               end;
               Result.AsStr := s;
            end else if FuncIs( 'memvarptr', 1, 1 ) then begin
               s := ParX( 0, '?' );
               if IsVariable( s ) then begin
                  Result.AsInt := Variables[ s ].AsPtr;
               end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
            end else if FuncIs( 'msgbox', 1, 3 ) then begin
               // Note: Par?() may also call SyncExecute, so pars are saved to vars first
               s := ParS( 0 );
               t := ParS( 1, 'Hamster-Script' );
               i := ParI( 2, MB_OK or MB_ICONINFORMATION );
               FSyncCmd := 'msgbox';
               FSyncPars.Clear;
               FSyncPars.Add( s );
               FSyncPars.Add( t );
               FSyncPars.Add( inttostr(i) );
               FSyncResult := Unassigned;
               Engine_SyncExec( Engine, SyncExecute );
               Result.AsVar := FSyncResult;
            end;

       'o': if FuncIs( 'ord', 1, 1 ) then begin
               s := ParS( 0 ) + #0;
               Result.AsInt := ord( s[1] );
            end;

       'p': if FuncIs( 'pos', 2, 3 ) then begin
               i := ParI( 2, 1 );
               if i=1 then
                  Result.AsInt := Pos( ParS(0), ParS(1) )
               else begin
                  s := ParS(1,'');
                  System.Delete( s, 1, i-1 );
                  Result.AsInt := Pos( ParS(0), s );
                  if Result.AsInt>0 then Result.AsInt := Result.AsInt + i - 1;
               end;
            end else if FuncIs( 'paramcount', 0, 0 ) then begin
               Result.AsInt := FScriptParams.Count - 1;
            end else if FuncIs( 'paramstr', 1, 1 ) then begin
               i := ParI( 0 );
               Result.AsStr := '';
               if (i>=0) and (i<FScriptParams.Count) then Result.AsStr:=FScriptParams[i];
            end else if FuncIs( 'print', 0, 999 ) then begin
               Result.AsStr := '';
               for i:=0 to FuncPars.Count-1 do Result.AsStr := Result.AsStr + ParS(i,'');
               Engine_Print( Engine, Result.AsStr );
            end else if FuncIs( 'popupbox', 1, 6 ) then begin
               s := ParS( 0 );
               t := ParS( 1, 'Hamster-Script' );
               i := ParI( 2, MB_OK or MB_ICONINFORMATION );
               j := ParI( 3, 10 );
               k := ParI( 4, IDOK );
               FSyncCmd := 'popupbox';
               FSyncPars.Clear;
               FSyncPars.Add( s );
               FSyncPars.Add( t );
               FSyncPars.Add( inttostr(i) );
               FSyncPars.Add( inttostr(j) );
               FSyncPars.Add( inttostr(k) );
               FSyncResult := Unassigned;
               Engine_SyncExec( Engine, SyncExecute );
               Result.AsVar := FSyncResult
            end;

       'r': if FuncIs( 'random', 1, 1 ) then begin
               Result.AsInt := PRNG( ParI(0) ); {JW+MG}{PRNG}
            end else if FuncIs( 'randomize', 0, 0 ) then begin
               Result.AsBool := SeedPRNG; {JW+MG}{PRNG}
            end else if FuncIs( 'replace', 3, 5 ) then begin
               s := ParS( 0 ); // base-string
               t := ParS( 1 ); // string to find
               u := ParS( 2 ); // replacement
               i := ParI( 3, 0  ); // replace all
               j := ParI( 4, 0  ); // ignore case
               Result.AsStr := '';
               if j=0 then v:=s else begin v:=LowerCase(s); t:=LowerCase(t) end;
               while v<>'' do begin
                  k := Pos( t, v );
                  if k=0 then begin Result.AsStr:=Result.AsStr+s; break; end;
                  Result.AsStr := Result.AsStr + copy( s, 1, k-1 ) + u;
                  System.Delete( s, 1, k + length(t) - 1 );
                  if i=0 then begin Result.AsStr:=Result.AsStr+s; break; end;
                  System.Delete( v, 1, k + length(t) - 1 );
               end;
            end else if FuncIs( 're_match', 2, 2 ) then begin
               s := ParS( 0 );
               t := ParS( 1 );
               if RE_Match( s, t, PCRE_CASELESS ) then Result.AsInt:=1 else Result.AsInt:=0;
            end else if FuncIs( 're_extract', 2, 2 ) then begin
               s := ParS( 0 );
               t := ParS( 1 );
               Result.AsStr := RE_Extract( s, t, PCRE_CASELESS );
            end else if FuncIs( 're_parse', 3, 999 )
                     or FuncIs( 're_split', 4, 999 )  then begin
               Result.AsInt := 0; // false
               s := ParS( 0 );
               t := ParS( 1 );
               TS := TStringList.Create;
               try
                  if FuncName='re_parse' then
                     RE_Parse( s, t, PCRE_CASELESS, TS )
                  else
                     RE_Split( s, t, PCRE_CASELESS, FuncPars.Count-2, TS );
                  for i:=2 to FuncPars.Count-1 do begin
                     s := ParX( i, '?' );
                     if IsVariable( s ) then begin
                        if (i-2)<TS.Count then t:=TS[i-2] else t:='';
                        Variables[ s ].AsStr := t;
                     end;
                  end;
                  Result.AsInt := 1; // true
               finally
                  TS.Free;
               end;
            end else if FuncIs( 'rasgetconnection', 0, 0 ) then begin
               Result.AsStr := RasDynGetConnection;
            end else if FuncIs( 'rasisconnected', 0, 0 ) then begin
               if RasDynIsConnected then Result.AsInt:=1 else Result.AsInt:=0;
            end else if FuncIs( 'rasdial', 1, 4 ) then begin
               if RasDynInit then begin
                  s := ParS( 0 ); // entry
                  t := ParS( 1, '' ); // user
                  u := ParS( 2, '' ); // pass
                  v := ParX( 3, '' ); // hconn-var
                  FillChar( RasDialPars, sizeof(RasDialPars), 0 );
                  RasDialPars.dwSize := sizeof( RasDialPars );
                  strpcopy( RasDialPars.szEntryName, s );
                  strpcopy( RasDialPars.szUserName,  t );
                  strpcopy( RasDialPars.szPassword,  u );
                  j := 0; // conn-hdl
                  Result.AsInt := RasDynDial( nil, nil, RasDialPars, 0, nil, j );
                  if (Result.AsInt<>0) and (j<>0) then begin RasDynHangup(j); j:=0 end;
                  if (v<>'') and IsVariable(v) then Variables[ v ].AsInt := j;
               end else Result.AsInt:=-1; // no RAS installed
            end else if FuncIs( 'rasgetip', 0, 0 ) then begin
               if RasDynIsConnected
                  then Result.AsStr := RasDynGetPppIp(RasDynGetConn)
                  else Result.AsStr := ''
            end else if FuncIs( 'rashangup', 0, 1 ) then begin
               if RasDynInit then begin
                  i := ParI( 0, 0 );
                  if i=0 then i := RasDynGetConn;
                  if i<>0 then Result.AsInt := RasDynHangUp( i ) else Result.AsInt := 0;
               end else Result.AsInt:=-1; // no RAS installed
            end else if FuncIs( 'runscript', 1, 3 ) then begin
               s := ParS( 0 ); // script
               t := ParS( 1, '' ); // parameters
               j := ParI( 2, 1 );  // waitflag (default=wait)
               Result.AsInt := Engine_StartScript( s, t, (j<>0) );
            end;

       's': if FuncIs( 'set', 2, 2 ) then begin
               s := ParX( 0, '?' );
               if IsVariable( s ) then begin
                  ParH( Result, 1 );
                  Variables[ s ].Assign( Result )
               end else begin
                  Error( HSCERR_SYNTAX, 'Variable expected: ' + s )
               end
            end else if FuncIs( 'setxcounter', 2, 2 ) then begin
               j := ParI(0);
               If (j >= 0) and (j <= 9) then begin
                  SetCounter ( CntCustomValues[j], ParI(1) );
                  Result.asInt := 0
               end else Result.asInt := -1;
            end else if FuncIs( 'sgn', 1, 1 ) then begin
               Result.AsInt := sgn( ParI( 0 ) );
            end else if FuncIs( 'str', 1, 3 ) then begin
               Result.AsStr := ParS( 0 );
               i := ParI( 1, 1 );
               t := copy( ParS(2,'0') + '0', 1, 1 );
               while length(Result.AsStr)<i do Result.AsStr := t + Result.AsStr;
            end else if FuncIs( 'sethostsentry_byname', 1, 2 ) then begin
               if ParS(0)<>''
                  then Result.asInt:=sethostsentry_byname(ParS( 0 ),ParS( 1, '' ))
                  else result.asInt:=-1
            end else if FuncIs( 'sethostsentry_byaddr', 1, 2 ) then begin
               if ParS(0)<>''
                  then result.asInt:=sethostsentry_byaddress(ParS( 1, '' ),ParS( 0 ))
                  else result.asInt:=-1
            end else if FuncIs( 'scriptpriority', 1, 1 ) then begin
               J:=ParI(0);
               FParentThread.Priority := TThreadPriority(j);
               Result.AsInt := 0;
            end else if FuncIs( 'syserrormessage', 1, 1 ) then begin
               J:=ParI(0);
               Result.AsStr := SysErrorMessage ( J )
            end else if FuncIs( 'stopthread', 1, 1 ) then begin
               If ThreadControl.Stop ( ParS(0) )
                  then Result.AsInt := 0
                  else Result.AsInt := -1
            end;

       't': if FuncIs( 'true', 0, 0 ) then begin
               Result.AsInt := 1;
            end else if FuncIs( 'trim', 1, 2 ) then begin
               s := ParS( 0 );
               if FuncPars.Count=1 then begin
                  Result.AsStr := Trim( s );
               end else begin
                  t := ParS( 1, '' );
                  while (s<>'') and (t<>'') do begin
                     if      pos(s[1        ],t)>0 then System.Delete(s,1        ,1)
                     else if pos(s[length(s)],t)>0 then System.Delete(s,length(s),1)
                     else break;
                  end;
                  Result.AsStr := s;
               end;
            end else if FuncIs( 'ticks', 0, 0 ) then begin
               Result.AsInt := Integer(GetTickCount);
            end else if FuncIs( 'time', 0, 0 ) then begin
               Result.AsInt := DateTimeToUnixTime( Now );
            end else if FuncIs( 'timegmt', 0, 0 ) then begin
               Result.AsInt := DateTimeToUnixTime( NowGMT );
            end;

       'u': if FuncIs( 'uppercase', 1, 1 ) then begin
               Result.AsStr := UpperCase( ParS(0) );
            end;

       'v': if FuncIs( 'varset', 2, 100 ) then begin
               ParH( Result, FuncPars.Count-1 );
               For i := 0 to FuncPars.Count-2 do begin
                  s := ParX( i, '?' );
                  if IsVariable( s )
                     then Variables.DefValue( s, Result, false )
                     else Error( HSCERR_SYNTAX, 'Variable expected: ' + s )
               end
            end;

       'w': if FuncIs( 'warning', 1, 999 ) then begin
               Result.AsStr := ParS(0);
               for i:=1 to FuncPars.Count-1 do Result.AsStr := Result.AsStr + ParS(i,'');
               Engine_Warning( Engine, Result.AsStr );
            end;

       'x': if FuncIs( 'xcounter', 1, 1 ) then begin
               j := ParI(0);
               If (j >= 0) and (j <= 9)
                  then Result.asInt := GetCounter ( CntCustomValues[j] )
                  else Result.asInt := 0
            end;

      end;

      // was it one of the built-in functions?
      if FuncNameOK then exit; // yes

      // invoke external functions
      Engine_Func( ParsedFunc, Result );
      if FuncNameOK then exit;

      // invoke function within script
      i := IndexOfSub( FuncName );
      if i<0 then begin
         Error( HSCERR_LABELNOTFOUND, 'Sub not found: ' + FuncName );
         exit;
      end else begin
         // save old and prepare new execution state
         FuncNameOK := True;
         IPtrPush;
         FNextPos := i + 1;
         Trace( i, FScript.Lines[i] );

         // eval and save transfer-vars in current var-context
         VTransfer := THscVariantArray.Create( FuncPars.Count );
         SetLength( VSubVars, FuncPars.Count );
         SetLength( VSubRefs, FuncPars.Count );
         try
         
            s := FScript.Lines[i];
            for i:=0 to FuncPars.Count-1 do ParH( VTransfer[i], i, 0 );
            i := Pos( '(', s );
            if i<=0 then begin
               s := '';
            end else begin
               System.Delete(s,1,i);
               i := Pos( ')', s ); if i>0 then s:=copy(s,1,i-1);
            end;
            for i:=0 to FuncPars.Count-1 do begin
               j := Pos( ',', s );
               if j>0 then begin
                  t := TrimWhSpace( copy(s,1,j-1) );
                  System.Delete( s, 1, j );
               end else begin
                  t := TrimWhSpace( s );
                  s := '';
               end;
               if t='' then begin
                  Error( HSCERR_SYNTAX, 'Too many parameters: '+FuncName ); exit;
               end;
               if t[1]='*' then VSubRefs[ i ] := True
                           else VSubRefs[ i ] := False;
               if t[1]='*' then System.Delete(t,1,1);
               if IsVariable(t) then
                  VSubVars[ i ] := t
               else begin
                  Error( HSCERR_SYNTAX, 'Variable expected: ' + t ); exit;
               end;
            end;
            if TrimWhSpace(s)<>'' then begin
               Error( HSCERR_SYNTAX, 'Missing parameters: '+FuncName ); exit;
            end;

            // enter new var-context and create transfer-vars
            EnterContext;
            for i:=0 to FuncPars.Count-1 do begin
               FVariables.DefValue( VSubVars[ i ], VTransfer[ i ], false );
            end;

            // execute sub
            ExecuteNextPos( Result );

            // save resulting 'by reference'-vars
            for i:=0 to FuncPars.Count-1 do begin
               if VSubRefs[ i ] then begin
                  VTransfer[ i ].Assign( FVariables[ VSubVars[ i ] ] );
               end;
            end;

            // return to previous context and set 'by reference'-vars
            LeaveContext;
            for i:=0 to FuncPars.Count-1 do begin
               if VSubRefs[ i ] then begin
                  s := ParX( i, '?' );
                  if IsVariable( s ) then
                     FVariables[ s ].Assign( VTransfer[ i ] )
                  else
                     Error( HSCERR_SYNTAX, 'Variable (byref) expected: ' + s );
               end;
            end;

         finally
            VTransfer.Free;
            VSubVars  := nil;
            VSubRefs  := nil;
            IPtrPop;
         end;
      end;

      if not( FuncNameOK ) then begin
         Error( HSCERR_UNSUPPORTED, 'Unkown function: ' + FuncName );

      end else if not( AnyError ) and ( Result.Unassigned ) then begin
         Error( HSCERR_UNSUPPORTED, 'Invalid function-syntax: ' + FuncName );
         
      end;

   except
      on E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
   end;
end;

procedure THscEngine.ExecuteNextPos( const Result: THscVariant );

   function SetNextPosToCorrespondingLine(
               const IncDepth, ExitOn1, DecAndExitOn0: array of LongWord
            ): Boolean;
   var  Depth, Idx: Integer;
        s         : String;
        m         : LongWord;
   begin
      Result := False;
      Depth  := 1;
      Idx    := FNextPos;

      while Idx < FScript.Count do begin
         m := FScript.Markers[Idx];

         if m <> SMARKER_NOSPECIAL then begin
            if MarkerInArray( m, IncDepth ) then begin
               inc( Depth );
            end else begin
               if ( Depth = 1 ) and ( High(ExitOn1) >= 0 ) then begin
                  if MarkerInArray( m, ExitOn1 ) then begin
                     FNextPos := Idx + 1;
                     Result := True;
                     Trace( Idx, FScript.Lines[Idx] );
                     break;
                  end;
               end;
               if MarkerInArray( m, DecAndExitOn0 ) then begin
                  dec( Depth );
                  if Depth = 0 then begin
                     FNextPos := Idx + 1;
                     Result := True;
                     Trace( Idx, FScript.Lines[Idx] );
                     break;
                  end;
               end;
            end;
         end;
         inc( Idx );
      end;

      if not Result then begin
         s := '';
         for m:=0 to High(DecAndExitOn0) do s:=s+'/'+MarkerToCmd( DecAndExitOn0[m] );
         for m:=0 to High(ExitOn1) do s:=s+'/'+MarkerToCmd( ExitOn1[m] );
         System.Delete( s, 1, 1 );
         Error( HSCERR_LOOPNOTFOUND, 'Missing ' + s + ':' + FCurrLine );
      end;
   end;

   procedure Exec_Do_While_Repeat( const ParsedLine: THscParsedFunc );
   var  i: Integer;
   begin
      with ParsedLine do begin
         i := 1; // True
         if FuncPars.Count = 1 then i := ParI( 0, 0 ); // while
         if i <> 0 then begin
            if FLoopStack.Count > 999 then
               Error( HSCERR_LOOPSTACK, 'Loop-stack overflow!' )
            else begin
               LoopStackPush( FCurrPos );
               LoopStackPush( Integer(SMARKER_DO) );
            end;
         end else begin
            // continue after corresponding Loop
            SetNextPosToCorrespondingLine(
               [SMARKER_DO,SMARKER_WHILE,SMARKER_REPEAT], [],
               [SMARKER_LOOP,SMARKER_ENDWHILE,SMARKER_UNTIL] );
         end;
      end;
   end;

   procedure Exec_Loop_EndWhile_Until( const ParsedLine: THscParsedFunc );
   var  i: Integer;
   begin
      with ParsedLine do begin
         if FLoopStack.Count < 2 then begin
            Error( HSCERR_LOOPSTACK,
                   'Loop/EndWhile/Until without Do/While/Until' );
         end else begin
            i := 1; // True
            if FuncPars.Count > 0 then begin // until
               i := ParI( 0, 0 );
               if i=0 then i:=1 else i:=0;
            end;
            if i <> 0 then begin
               if Cardinal(LoopStackPeek(0)) <> SMARKER_DO then begin
                  Error( HSCERR_LOOPSTACK,
                         'Loop/EndWhile/Until without Do/While/Until' );
                  exit;
               end;
               FNextPos := LoopStackPeek(1);
            end;
            LoopStackPop( 2 );  
         end;
      end;
   end;

var  ParsedLine: THscParsedFunc;
     i, j, i1, i2, StackHeight: Integer;
     s: String;
     OK, CheckElseIf: Boolean;
     HV: THscVariant;
begin
   Result.AsInt := 0;
   FCurrPos := FNextPos;
   StackHeight := FLoopStack.Count;
   CheckElseIf := false;
   try
      while FCurrPos<FScript.Count do begin
         if FTerminated then break;
         If ThreadShutDown (false) then break;

         FNextPos   := FCurrPos + 1;
         FCurrLine  := FScript.Lines[ FCurrPos ];
         ParsedLine := FScript.ScrLine[ FCurrPos ];
         if length(FCurrLine)>0 then Trace( FCurrPos, FCurrLine );

         with ParsedLine do try
            if FuncName<>'' then begin

               FuncNameOK := False;

               case FuncName[1] of

                'a': if FuncIs( 'assert', 1, 999 ) then begin
                        if ParI( 0 )=0 then begin
                           s := '';
                           for i:=1 to FuncPars.Count-1 do s:=s+ParS(i,'');
                           if s='' then s:='Assertion failed!';
                           Error( HSCERR_USERDEFINED, s );
                           Result.AsInt := 255;
                           FTerminated := True;
                           break;
                        end;
                     end;

                'b': if FuncIs( 'break', 0, 1 ) then begin
                        if FLoopStack.Count < StackHeight+2 then begin
                           Error( HSCERR_LOOPSTACK, 'Break without Do/While/Repeat/For' );
                        end else begin
                           i := 1; // True
                           if FuncPars.Count>0 then i := ParI( 0, 0 );
                           if i<>0 then begin
                              if Cardinal(LoopStackPeek(0))=SMARKER_DO then
                                 LoopStackPop( 2 )
                              else begin
                                 LoopStackPop( 5 );
                              end;
                              // continue after corresponding Loop
                              SetNextPosToCorrespondingLine(
                                 [SMARKER_DO,SMARKER_WHILE,SMARKER_REPEAT,SMARKER_FOR], [],
                                 [SMARKER_LOOP,SMARKER_ENDWHILE,SMARKER_UNTIL,SMARKER_ENDFOR] );
                           end;
                        end;
                     end;

                'c': if FuncIs( 'continue', 0, 1 ) then begin
                        if FLoopStack.Count < StackHeight+2 then begin
                           Error( HSCERR_LOOPSTACK, 'Continue without Do/While/Repeat/For' );
                        end else begin
                           i := 1; // True
                           if FuncPars.Count>0 then i := ParI( 0, 0 );
                           if i<>0 then begin
                              if Cardinal(LoopStackPeek(0))=SMARKER_DO then begin
                                 FNextPos := LoopStackPeek(1);
                                 LoopStackPop( 2 );
                              end else begin
                                 // continue with corresponding EndFor
                                 SetNextPosToCorrespondingLine(
                                    [SMARKER_FOR], [], [SMARKER_ENDFOR] );
                                 dec( FNextPos ); // one back to EndFor-line
                              end;
                           end;
                        end;
                     end else
                     if FuncIs( 'const', 2, 2 ) then begin
                        ParH( Result, 1 );
                        s := ParX( 0, '?' );
                        if IsVariable( s )
                           then Variables.DefValue( s, Result, true )
                           else Error( HSCERR_SYNTAX, 'Variable expected: ' + s )
                     end else If FuncIs( 'constenum', 1, 100 ) then begin
                        For i := 0 to FuncPars.Count-1 do begin
                           Result.asInt := i;
                           s := ParX( i, '?' );
                           if IsVariable( s )
                              then Variables.DefValue( s, Result, true )
                              else Error( HSCERR_SYNTAX, 'Variable expected: ' + s )
                        end
                     end;

                'd': if FuncIs( SMARKER_DO, 0, 0 ) then begin
                        Exec_Do_While_Repeat( ParsedLine );
                     end else if FuncIs( 'dump', 0, 1 ) then begin
                        Engine_Print( Engine, 'dump follows:' );
                        if Context = GLOBAL_CONTEXT then
                           Engine_Print( Engine, '.context: (global)' )
                        else
                           Engine_Print( Engine, '.context: ' + inttostr(Context) );
                        for i:=0 to FVariables.Count-1 do
                           If Not FVariables.Items(i).Value.IsConst then
                              Engine_Print( Engine, '.var: ' + FVariables.Dump(i) );
                        If ParI(0, 0) = 1 then
                           for i:=0 to FVariables.Count-1 do
                              If FVariables.Items(i).Value.IsConst then
                                 Engine_Print( Engine, '.const: ' + FVariables.Dump(i) );
                        for i:=0 to FHscLists.Count-1 do begin
                           if FHscLists.ListExists(i) then begin
                              Engine_Print( Engine, '.list: ' + inttostr(i)
                              + ' (' + inttostr(FHscLists.List[i].Count) + ' entries)' );
                           end;
                        end;
                     end else if FuncIs( 'debug', 1, 999 ) then begin
                        i := ParI( 0 ); // level
                        if FuncPars.Count=1 then begin
                           FDebugLevel := i;
                        end else begin
                           if (i<=FDebugLevel) and (FDebugLevel>0) then begin
                              s := ''; for i:=1 to FuncPars.Count-1 do s:=s+ParS(i,'');
                              Engine_Print( Engine, s );
                           end;
                        end;
                     end;

                'e': if FuncIs( SMARKER_ENDIF, 0, 0 ) then begin
                        // ignore
                     end else if FuncIs( SMARKER_ELSE, 0, 0 ) then begin
                        // continue after corresponding endif
                        SetNextPosToCorrespondingLine( [SMARKER_IF], [], [SMARKER_ENDIF] );
                     end else if FuncIs( SMARKER_ELSEIF, 1, 1 ) then begin
                        If CheckElseIf then begin
                           CheckElseIf := false;
                           i := ParI( 0, 0 );
                           if i=0 then begin // False
                              SetNextPosToCorrespondingLine(
                                 [SMARKER_IF], [SMARKER_ELSE, SMARKER_ELSEIF], [SMARKER_ENDIF] );
                              If FScript.Markers[FNextPos-1] = SMARKER_ELSEIF then begin
                                 CheckElseIf := true;
                                 Dec(FNextpos)
                              end
                           end
                        end else begin
                           // continue after corresponding endif
                           SetNextPosToCorrespondingLine( [SMARKER_IF], [], [SMARKER_ENDIF] )
                        end
                     end else if FuncIs( SMARKER_ENDFOR, 0, 0 ) then begin
                        if FLoopStack.Count < StackHeight+5 then begin
                           Error( HSCERR_LOOPSTACK, 'EndFor without For' );
                        end else if Cardinal(LoopStackPeek(0))<>SMARKER_FOR then begin
                           Error( HSCERR_LOOPSTACK, 'EndFor without For' );
                        end else begin
                           HV := THscVariant( LoopStackPeek(2) ); // var's value
                           j  := LoopStackPeek(3); // step
                           i2 := LoopStackPeek(4); // limit2
                           i1 := HV.AsInt + j; // next val
                           HV.AsInt := i1; // update loop-var
                           if j > 0 then OK := (i1 <= i2) else OK := (i1 >= i2);
                           if OK then begin // continue loop
                              FNextPos := LoopStackPeek(1);
                           end else begin // end loop
                              LoopStackPop( 5 );
                           end;
                        end;
                     end else if FuncIs( SMARKER_ENDWHILE, 0, 0 ) then begin
                        Exec_Loop_EndWhile_Until( ParsedLine );
                     end else if FuncIs( 'endsub', 0, 0 ) then begin
                        Result.AsInt := 0;
                        break;
                     end else if FuncIs( 'entercontext', 0, 1 ) then begin
                        EnterContext; // not used any more: ParS(0,'?')
                     end else if FuncIs( 'error',  0, 999 ) then begin
                        s := '';
                        for i:=0 to FuncPars.Count-1 do s:=s+ParS(i,'');
                        if s='' then s:='User defined error!';
                        Error( HSCERR_USERDEFINED, s );
                        if AnyError then begin
                           Result.AsInt := 255;
                           FTerminated := True;
                           break;
                        end;
                     end;

                'f': if FuncIs( SMARKER_FOR, 3, 4 ) then begin
                        s  := ParX( 0, '?' ); // loop-var
                        i1 := ParI( 1, 0 );   // 1st limit
                        i2 := ParI( 2, 0 );   // 2nd limit
                        j  := ParI( 3, 1 );   // step
                        if not IsVariable( s ) then begin
                           Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
                           break;
                        end;
                        HV := Variables[s]; // get var (=ptr to its value)
                        HV.AsInt := i1;           // init var
                        if j > 0 then OK := (i1 <= i2) else OK := (i1 >= i2);
                        if OK then begin // start loop
                           if FLoopStack.Count>999 then begin
                              Error( HSCERR_LOOPSTACK, 'Loop-stack overflow!' )
                           end else begin
                              LoopStackPush( i2 ); // (4)
                              LoopStackPush( j  ); // (3)
                              LoopStackPush( Integer(HV) ); // (2)
                              LoopStackPush( FNextPos );    // (1)
                              LoopStackPush( Integer( SMARKER_FOR ) ); // (0)
                           end;
                        end else begin // end loop
                           // continue after corresponding Loop
                           SetNextPosToCorrespondingLine(
                              [SMARKER_FOR], [], [SMARKER_ENDFOR] );
                        end;
                     end;

                'g': if FuncIs( 'goto', 1, 1 ) then begin
                        s := ParX( 0, '?' );
                        i := IndexOfLabel( s );
                        if IsIdentifier(s) and (i>=0) then FNextPos:=i
                        else Error( HSCERR_LABELNOTFOUND,
                                    'Label not found: ' + s );
                     end;

                'i': if FuncIs( SMARKER_IF, 1, 1 ) then begin
                        i := ParI( 0, 0 );
                        if i=0 then begin // False
                           // continue after corresponding else/endif
                           SetNextPosToCorrespondingLine(
                              [SMARKER_IF], [SMARKER_ELSE, SMARKER_ELSEIF], [SMARKER_ENDIF] );
                           If FScript.Markers[FNextPos-1] = SMARKER_ELSEIF then begin
                              CheckElseIf := true;
                              Dec(FNextpos)
                           end
                        end;
                     end;

                'l': if FuncIs( SMARKER_LOOP, 0, 0 ) then begin
                        Exec_Loop_EndWhile_Until( ParsedLine );
                     end else if FuncIs( SMARKER_LABEL, 1, 1 ) then begin
                        if not IsIdentifier( ParX(0,'?') ) then
                           Error( HSCERR_INVALID, 'Invalid label: '+ParX(0,'') );
                     end else if FuncIs( 'leavecontext', 0, 0 ) then begin
                        LeaveContext;
                     end;

                'q': if FuncIs( 'quit', 0, 1 ) then begin
                        Result.AsInt := ParI( 0, 0 );
                        FTerminated := True;
                        break;
                     end;

                'r': if FuncIs( SMARKER_REPEAT, 0, 0 ) then begin
                        Exec_Do_While_Repeat( ParsedLine );
                     end else if FuncIs( 'return', 0, 1 ) then begin
                        ParH( Result, 0, 0 );
                        if (Result.Unassigned) and (FuncPars.Count>0) then
                           Error( HSCERR_INVALIDEXPRESSION,
                                  'Undefined return-value' );
                        break;
                     end;

                's': if FuncIs( 'sleep', 1, 1 ) then begin
                        i := ParI( 0 );
                        if FStopEvent=0 then Sleep( i )
                                        else WaitForSingleObject( FStopEvent, i );
                     end;

                't': if FuncIs( 'trace', 1, 1 ) then begin
                        FTraceIsOn := ( ParI(0) <> 0 );
                     end;

                'u': if FuncIs( SMARKER_UNTIL, 1, 1 ) then begin
                        Exec_Loop_EndWhile_Until( ParsedLine );
                     end;

                'v': if FuncIs( 'var', 1, 999 ) then begin
                        for i:=0 to FuncPars.Count-1 do begin
                           s := ParX( i, '?' );
                           if IsVariable( s ) then begin
                              Variables.DefValue( s, NIL, false );
                           end else Error( HSCERR_SYNTAX,
                                           'Variable expected: ' + s );
                        end;
                     end;

               'w': if FuncIs( SMARKER_WHILE,  1, 1 ) then begin
                       Exec_Do_While_Repeat( ParsedLine );
                    end;

               end;

               if not( FuncNameOK or AnyError ) then begin
                  SolveFunction( ParsedLine, Result );
                  if Result.Unassigned and not AnyError then begin
                     Error( HSCERR_UNSUPPORTED, 'Unknown statement: '+FuncName );
                  end;
               end;

               if AnyError then break;

               if FStopEvent<>0 then begin
                  if WaitForSingleObject( FStopEvent, 0 )=WAIT_OBJECT_0 then begin
                     Log( LOGID_SYSTEM, 'Script stopped.' );
                     break;
                  end;
               {HSR} {One_SD_S}
               end else begin
                  if ThreadShutDown(true,false) then begin
                     Log( LOGID_SYSTEM, 'Script stopped.' );
                     break;
                  end;
               {/HSR}
               end;
            end

         except
            on E:Exception do Error( HSCERR_ENGINEEXCEPTION,
                                     'Exception: ' + E.Message );
         end;

         FCurrPos := FNextPos;
      end
   finally
      If FLoopStack.Count > StackHeight then begin
         LoopStackPop(FLoopStack.Count-StackHeight)
      end
   end
end;

procedure THscEngine.Execute( const Result: THscVariant );
const WAS_INITIALIZED = Pointer( 0 );
var InitMods, ExitMods: TStringList;

   procedure PreProcess( const FromLine, ToLine: Integer; const CurrentModule: String );
   var  LineNo, i: Integer;
        Cmd, Par : String;
        TS: TStringList;
   begin
      for LineNo:=FromLine to ToLine do begin
         FCurrPos  := LineNo;
         FCurrLine := Trim(FScript.Lines[LineNo]);

         if copy( FCurrLine, 1, 2 ) = '#!' then begin

            Trace( FCurrPos, FCurrLine );

            Cmd := copy( FCurrLine, 3, Length(FCurrLine)-2 );
            Par := '';
            i := Pos( ' ', Cmd );
            if i>0 then begin
               Par := copy( Cmd, i+1, Length(Cmd)-i );
               Cmd := copy( Cmd, 1, i-1 );
            end;
            Cmd := LowerCase( Cmd );
            Par := TrimWSPEC( Par );

            if (Cmd='hs2') and (Par='') then begin
               // ignore

            end else if (Cmd='trace') and (Par<>'') then begin
               FTraceIsOn := ( strtointdef( Par, 0 ) = 1 );

            end else if (Cmd='debug') and (Par<>'') then begin
               FDebugLevel := strtointdef( Par, 0 );

            end else if (Cmd='load') and (Par<>'') then begin
               if Copy(Par, 2, 1)<>':' then begin
                  if Copy(Par, 1, 1) = '\'
                     then Par := Copy( FPathHSM, 1, 2) + Par
                     else Par := FPathHSM + Par
               end;
               Par := ExpandFilename( Par );
               if not( FScript.IsModuleLoaded( Par ) ) then begin
                  TS := TStringList.Create;
                  try
                     if FileExists2( Par ) then begin
                        TS.LoadFromFile( Par );
                        FScript.Add( 'quit {module-separator}' );
                        FScript.StartNewModule( Par );
                        i := FScript.Count;
                        FScript.AddLines( TS );
                        PreProcess( i, FScript.Count-1, Par );
                     end else begin
                        Error( HSCERR_INITIALIZEFAILED,
                               'Module not found: ' + Par );
                        break;
                     end;
                  finally
                     TS.Free;
                  end;
               end;

            end else if (Cmd='initialize') and (Par='') then begin
               InitMods.AddObject( CurrentModule, Pointer( LineNo+1 ) );

            end else if (Cmd='finalize') and (Par='') then begin
               if InitMods.IndexOf( CurrentModule ) >= 0 then begin
                  ExitMods.AddObject( CurrentModule, Pointer( LineNo+1 ) );
               end else begin
                  // Not allowed, as we use InitMods.Objects[] later to
                  // recognize modules, which were successfully initialized.
                  Error( HSCERR_INITIALIZEFAILED,
                         '#!finalize without #!initialize in ' + CurrentModule );
               end;

            end else if (Copy(Cmd, 1, 5)='menu:') then begin

              // Eintrag ins Hamstermenu

            end else begin
               Error( HSCERR_SYNTAX,
                      'Invalid pre-processor command: ' + FCurrLine );
               break;
            end;

         end;
      end;
   end;

   procedure InitializeModules;
   // Initialize modules from "deepest" to "highest" one.
   var  No: Integer;
        HV: THscVariant;
   begin
      HV := THscVariant.Create;
      try
         for No := 0 to InitMods.Count-1 do begin
            if AnyError then break;

            FNextPos := Integer( InitMods.Objects[ No ] );
            HV.Unassign;
            ExecuteNextPos( HV );

            if HV.AsInt = 0 then begin
               InitMods.Objects[ No ] := WAS_INITIALIZED;
            end else begin
               Error( HSCERR_INITIALIZEFAILED,
                      'Initialization of module ' + InitMods[ No ]
                      + ' failed with error-code ' + inttostr(HV.AsInt) + '!' );
               break;
            end;
         end;

      finally
         HV.Free;
      end;
   end;

   procedure FinalizeModules;
   // Finalize initialized modules from "highest" to "deepest" one.
   // To give modules the chance to clean up their ressources, all previous
   // errors and errors now caused by finalization are completely ignored.
   var  No, i: Integer;
        Module   : String;
        HV       : THscVariant;
        SaveStopEvent: THandle;
        SaveLastError: Integer;
   begin
      SaveStopEvent := FStopEvent;
      SaveLastError := FLastError;
      HV := THscVariant.Create;

      try
         for No := ExitMods.Count-1 downto 0 do begin
            try
               FTerminated := False;
               FLastError  := 0;
               FStopEvent  := 0;

               Module := ExitMods[ No ];
               i := InitMods.IndexOf( Module );

               if (i>=0) and (InitMods.Objects[i]=WAS_INITIALIZED) then begin
                  FNextPos := Integer( ExitMods.Objects[ No ] );
                  ExecuteNextPos( HV );
               end;
            except {ignore} end;
         end;

      finally
         HV.Free;
         FStopEvent := SaveStopEvent;
         FLastError := SaveLastError;
      end;
   end;

begin
   Result.AsInt := 255;

   try
      // initialize script-engine
      FVariables.Clear;
      FHscLists.Clear;
      FScheduler.Clear;
      FHscLists.Clear;
      FLoopStack.Clear;
      FIPtrStack.Clear;
      FExpressions.Clear;
      FParsedFuncs.Clear;
      FLastError := 0;
      FTraceIsOn  := False;
      FDebugLevel := 0;
      FTerminated := False;

      FErrNum     := HSCERR_NOERROR;
      FErrMsg     := 'No error';
      FErrModule  := '';
      FErrLineNo  := -1;
      FErrLine    := '';
      FErrSender  := '';
      FErrCatch   := False;

      // pre-process script (e.g. load modules)
      InitMods := TStringList.Create;
      ExitMods := TStringList.Create;
      try
         PreProcess( 0, FScript.Count-1, '' );

         // initialize modules from bottom to top
         InitializeModules;

         // run main script
         if not AnyError then begin
            FCurrPos    := 0;
            FNextPos    := 0;
            ExecuteNextPos( Result );
         end;

         // finalize modules
         FinalizeModules;

      finally
         InitMods.Free;
         ExitMods.Free;
      end

   except
      on E:Exception do Error( HSCERR_ENGINEEXCEPTION,
                               'Exception: ' + E.Message );
   end;
end;

procedure THscEngine.ExecuteFromList( const AScriptName: String;
                                      const AScriptList: TStringList;
                                      const AParamsText: String;
                                      const Result: THscVariant );
begin
   Result.Unassign;

   try
      FScriptName := AScriptName;
      FScriptParams.Text := AParamsText;

      FScript.Clear;
      FScript.StartNewModule( FScriptName );
      FScript.AddLines( AScriptList );

      Execute( Result );

   except
      on E:Exception do Error( HSCERR_ENGINEEXCEPTION,
                               'Exception: ' + E.Message );
   end;
end;

constructor THscEngine.Create( AParentThread: TThread; const APathHSM: String );
begin
   inherited Create( Self );
   FParentThread := AParentThread;
   FVariables    := THscVariables.Create( Engine );
   FScheduler    := THscScheduler.Create( Engine );
   FHscLists     := THscLists.Create( Engine );
   FLoopStack    := TList.Create;
   FIPtrStack    := TList.Create;
   FExpressions  := THscExpressions.Create( Self );
   FParsedFuncs  := THscParsedFuncs.Create( Self );
   FScript       := THscScript.Create( Self );
   FSyncPars     := TStringList.Create;
   FStopEvent    := 0;
   FContextID    := GLOBAL_CONTEXT;
   FPathHSM      := APathHSM;
   FScriptParams := TStringList.Create;
   FResControl   := TResController.Create;
   SeedPRNG
end;

destructor THscEngine.Destroy;
var  s: String;
begin
   try
      s := FResControl.AutoDestroyAll;
      if s <> '' then begin
         Log( LOGID_WARN, '{' + FScriptName + '} '
                            + 'Ressources not released by script: ' + s );
      end;
   except end;

   if Assigned( FSyncPars     ) then FSyncPars.Free;
   if Assigned( FScript       ) then FScript.Free;
   if Assigned( FHscLists     ) then FHscLists.Free;
   if Assigned( FScheduler    ) then FScheduler.Free;
   if Assigned( FVariables    ) then FVariables.Free;
   if Assigned( FLoopStack    ) then FLoopStack.Free;
   if Assigned( FIPtrStack    ) then FIPtrStack.Free;
   if Assigned( FExpressions  ) then FExpressions.Free;
   if Assigned( FParsedFuncs  ) then FParsedFuncs.Free;
   if Assigned( FScriptParams ) then FScriptParams.Free;
   if Assigned( FResControl   ) then FResControl.Free;
   inherited Destroy;
end;

function THscVariables.Items(const Index: Integer): THscVariable;
begin
   Result := THscVariable( FList[Index] )
end;

end.
