unit cHscHamster;

interface

uses SysUtils, Classes, cSyncObjects, cHscEngine;

type
   TThreadSynchronize = procedure ( Method: TThreadMethod ) of object;

   THscEngineHamster = class( THscEngine )
      private
         FThreadSynchronize: TThreadSynchronize;
         FOutputBuffer     : TThreadStringList;

         procedure BufferOutput  ( const s: String );

      protected
         procedure Engine_Error( const Engine: THscEngine;
                                 const Sender: TObject;
                                 const ErrNum: Integer;
                                 const ErrMsg: String;
                                 const CurrMod: String;
                                 const CurrPos: Integer;
                                 const CurrLine: String ); override;
         procedure Engine_Trace( const Engine: THscEngine;
                                 const CurrMod: String;
                                 const CurrPos: Integer;
                                 const CurrLine: String ); override;
         procedure Engine_Warning( const Engine: THscEngine;
                                   const PrintStr: String ); override;
         procedure Engine_Print( const Engine: THscEngine;
                                 const PrintStr: String ); override;
         procedure Engine_SyncExec( const Engine: THscEngine;
                                    const X: TThreadMethod ); override;
         procedure Engine_Func( const ParsedFunc: THscParsedFunc;
                                const Result: THscVariant ); override;
         function  Engine_StartScript( const ScriptFile: String;
                                       const Params    : String;
                                       const WaitForEnd: Boolean ): Integer; override;

      public
         property OutputBuffer: TThreadStringList read FOutputBuffer;

         constructor Create( AParentThread: TThread;
                             const APathHSM: String;
                             const AThreadSynchronize: TThreadSynchronize;
                             const AOutputBuffer: TThreadStringList );
   end;


implementation

uses Windows, Global, Config, uTools, cLogFile, cArticle, cFiltersNews, cStdForm,
     tMaintenance, cPasswordFile, tTransfer, cArtFiles, cResControl, tScript,
     cMailRouter, uEncoding, uDateTime, tBase, uCommands, cNewsJobs,
     cAccount
     {$IFDEF H_NEED_VARIANTS} , Variants {$ENDIF} ;


// ---------------------------------------------------- THscEngineHamster -----

procedure THscEngineHamster.BufferOutput( const s: String );
begin
   if not Assigned( FOutputBuffer ) then exit;

   with FOutputBuffer.LockList do try
      while Count > 10000 do Delete( 0 );
      Add( s );
   finally
      FOutputBuffer.UnlockList;
   end;
end;

procedure THscEngineHamster.Engine_Error( const Engine: THscEngine;
                                          const Sender: TObject;
                                          const ErrNum: Integer;
                                          const ErrMsg: String;
                                          const CurrMod: String;
                                          const CurrPos: Integer;
                                          const CurrLine: String );
var  s: String;
begin
   If CurrMod > '' then begin
      s := Format(
           'Error in line %s of script-file "%s"', [inttostr(CurrPos), CurrMod] );
      Log( LOGID_ERROR, s );
      BufferOutput( 'E> ' + s )
   end;
   s := Format(
        'Error %s: %s [%s]', [inttostr( ErrNum ), ErrMsg, Sender.ClassName ] );
   Log( LOGID_ERROR, s );
   BufferOutput( 'E> ' + s );
   s := Format(
        'Error-line: %s', [TrimWhSpace(CurrLine)] );
   Log( LOGID_ERROR, s );
   BufferOutput( 'E> ' + s );
end;

procedure THscEngineHamster.Engine_Trace( const Engine: THscEngine;
                                          const CurrMod: String;
                                          const CurrPos: Integer;
                                          const CurrLine: String );
begin
   Log( LOGID_INFO, '[' + CurrMod + ':' + inttostr(CurrPos) + ']: ' + CurrLine );
end;

procedure THscEngineHamster.Engine_Warning( const Engine: THscEngine;
                                            const PrintStr: String );
var  s: String;
begin
   s := '!> ' + PrintStr;
   Log( LOGID_WARN, s );
   BufferOutput( s );
end;

procedure THscEngineHamster.Engine_Print( const Engine: THscEngine;
                                          const PrintStr: String );
var  s: String;
begin
   s := '> ' + PrintStr;
   Log( LOGID_INFO, s );
   BufferOutput( s );
end;

procedure THscEngineHamster.Engine_SyncExec( const Engine: THscEngine;
                                             const X: TThreadMethod );
begin
   if Assigned(FThreadSynchronize) then FThreadSynchronize( X )
                                   else X;
end;


procedure THscEngineHamster.Engine_Func( const ParsedFunc: THscParsedFunc;
                                         const Result    : THscVariant );
var  i, j, LfdServer, TimeOut, LfdThread: Integer;
     s, s1, s2, s3, t, SrvList: String;
     Identifier, ConnectionID, Username,  Password: String;
     Server, Port, User, Pass, DestUser, FiltSect,
     FromSel, ToSel, LeaveOnServer: String;
     Art: TArticle;
     ScoreFile: TFiltersNews;
     XOverRec : TXOverRec;
     OK: Boolean;
     SSLMode, SSLVerify: Integer; SSLCaFile: String; {MG}{SSL}

     {JW} {purge event}
      Function CheckPurge (Const bShowWarning: Boolean): boolean;
      begin
          Result := WAIT_OBJECT_0=WaitForSingleObject(EventPurge,0);
          If Result and bShowWarning
             then Log(LogID_Warn, TrGl(kLog, 'Script.Abort.PurgeDatabase',
                  'critical state detected: Hamster purges database, hs2 command aborted'))
      end;
      {JW}

begin
   with ParsedFunc do begin

      Result.Unassign;
      If Funcname = '' then Exit;

      Case FuncName[1] of
         'a': if FuncIs( 'addlog', 2, 3 ) then begin  // = hamaddlog
                 Result.AsStr := ParS(0);
                 if ParI(2, 1)=0 then ScriptAddLog( Result.AsStr, ParI(1) )
                                 else ScriptAddLog( '{Script ' + ScriptName + '} '
                                                  + Result.AsStr, ParI(1) );
               end;
         'd': if FuncIs( 'decodemimeheaderstring', 1, 2) then begin
                  s:=ParS(0);
                  Result.AsStr:=DecodeHeaderValue(s, s2);
                  if FuncPars.Count > 1 then begin
                     s1 := ParX( 1, '' );
                     If IsVariable( s1 )
                        then Variables[ s1 ].AsStr :=s2
                        else Log(LOGID_Error, 'Second optional parameter for "DecodeMimeHeaderString" has to be a variable')
                  end
               end;
          'g': if FuncIs( 'gettasksactive', 0, 0) then begin
                  EnterCriticalSection(CS_Counter);
                  Result.asInt:=CntActiveTasks;
                  LeaveCriticalSection(CS_Counter)
               end else if FuncIs( 'gettaskswait', 0, 0) then begin
                  EnterCriticalSection(CS_Counter);
                  Result.asInt:=CntWaitTasks;
                  LeaveCriticalSection(CS_Counter);
               end else if FuncIs( 'gettasksrun', 0, 0) then begin
                  EnterCriticalSection(CS_Counter);
                  Result.asInt:=CntActiveTasks-CntWaitTasks;
                  LeaveCriticalSection(CS_Counter)
               end;
          'r': if FuncIs( 'raslasterror',  0, 0 ) then begin
                  Result.asInt := RasDialer.LastError;
               end else if FuncIs( 'rfctimezone', 0, 0) then begin
                  Result.asStr := NowRfcTimezone;
               end;
          's': if FuncIs( 'settasklimiter', 1, 1) then begin
                  Result.AsInt := Def_MaxTasks;
                  Def_MaxTasks := ParI(0)
                end
      end;

      If Copy(FuncName,1,3) <> 'ham' then Exit;

      if FuncIs( 'hamversion', 0, 1 ) then begin
         i := ParI(0, 0);
         Case i of
            0: Result.AsStr := GetExeVersion;
            1: Result.AsStr := GetMyStringFileInfo('ProductName','Hamster');
            2: Result.AsStr := GetMyStringFileInfo('Maintainer','Unknown');
            3: Result.AsStr := GetMyStringFileInfo('Download','Unknown');
            4: Result.AsStr := GetMyStringFileInfo('Comments','Unknown');
            else Result.AsStr := ''
         end;
      {JW} {ScriptFQDN}
      end else if FuncIs( 'hammainfqdn', 0, 0 ) then begin
         Result.AsStr  := Def_FQDN;
      end else if FuncIs( 'hammidfqdn', 0, 0 ) then begin
         Result.AsStr  := Def_FQDNforMIDs;
      end else if FuncIs( 'hamenvelopefrom', 0, 0 ) then begin
         Result.AsStr  := Def_EnvelopeFrom;
      end else if FuncIs( 'hampostmaster', 0, 0 ) then begin
         Result.AsStr  := Def_Postmaster;
      end else if FuncIs( 'hamusenetacc', 0, 0 ) then begin
         Result.AsStr  := Def_Usenet;
      {JW}
      end else if FuncIs('hamcheckpurge',0,0) then begin
         Result.AsBool := CheckPurge(false)
      end else if FuncIs( 'hamexepath', 0, 0 ) then begin
         Result.AsStr := UpperCase(ExtractFilePath(ParamStr(0)));
      end else if FuncIs( 'hampath', 0, 0 ) then begin
         Result.AsStr := PATH_BASE;
      end else if FuncIs( 'hamhscpath', 0, 0 ) then begin
         Result.AsStr := PATH_HSC;
      end else if FuncIs( 'hamhsmpath', 0, 0 ) then begin
         Result.AsStr := PATH_HSM;
      end else if FuncIs( 'hamrcpath', 0, 0 ) then begin
         Result.AsStr := PATH_HSC_RC;
      end else if FuncIs( 'hamlogspath', 0, 0 ) then begin
         Result.AsStr := PATH_LOGS;
      end else if FuncIs( 'hamserverpath', 0, 0 ) then begin
         Result.AsStr := PATH_SERVER;
      end else if FuncIs( 'hamgroupspath', 0, 0 ) then begin
         Result.AsStr := PATH_GROUPS;
      end else if FuncIs( 'hammailpath', 0, 0 ) then begin
         Result.AsStr := PATH_MAILS;
      end else if FuncIs( 'hamnewsoutpath', 0, 0 ) then begin
         Result.AsStr := PATH_NEWS_OUT;
      end else if FuncIs( 'hamnewserrpath', 0, 0 ) then begin
         Result.AsStr := PATH_NEWS_ERR;
      end else if FuncIs( 'hammailsoutpath', 0, 0 ) then begin
         Result.AsStr := PATH_MAIL_OUT;

      end else if FuncIs( 'hammessage', 1, 2 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, ParI(0), ParI(1,0) );
      end else if FuncIs( 'hamshutdown', 0, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_HAMSTER_EXIT, ParI(0,0) );
      end else if FuncIs( 'hamstopalltasks', 0, 0 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_STOPALLTASKS, 0 );
      end else if FuncIs( 'hamresetcounters', 0, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_RESET_COUNTERS, ParI(0,0) );
      end else if FuncIs( 'hamnntpserver', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_LOCALNNTP_ONOFF, ParI(0) );
      end else if FuncIs( 'hamsmtpserver', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_LOCALSMTP_ONOFF, ParI(0) );
      end else if FuncIs( 'hampop3server', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_LOCALPOP3_ONOFF, ParI(0) );
      end else if FuncIs( 'hamimapserver', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_LOCALIMAP_ONOFF, ParI(0) );
      end else if FuncIs( 'hamrecoserver', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_LOCALRECO_ONOFF, ParI(0) );
      end else if FuncIs( 'hammainwindow', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_SHOWWINDOW, ParI(0) );
      end else if FuncIs( 'hamtrayicon', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_SHOWICON, ParI(0) );
      end else if FuncIs( 'hamreloadconfig', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_CONFIG_RELOAD, ParI(0) );
      end else if FuncIs( 'hamreloadipaccess', 0, 0 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_CONFIG_IPAccess, 0 );
      end else if FuncIs( 'hamdialogeditdirs', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_DIRS, ParI(0) );
      end else if FuncIs( 'hamdialogscripts', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_SCRIPTS, ParI(0) );
      end else if FuncIs( 'hamdialognewskillfilelog', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_NEWSKILLFILELOG, ParI(0) );
      end else if FuncIs( 'hamdialogaddpull', 1, 1 ) then begin
         Result.AsInt := SendMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_PULLNEWGROUPS, ParI(0) );

      end else if FuncIs( 'hamrotatelog', 0, 0 ) then begin
         Result.AsInt := 0;
         Logfile.RotateLog
      end else if FuncIs( 'hamaddlog', 2, 3 ) then begin
         Result.AsStr := ParS(0);
         if ParI(2, 1)=0 then ScriptAddLog( Result.AsStr, ParI(1) )
                         else ScriptAddLog( '{' + ScriptName + '} '
                                          + Result.AsStr, ParI(1) );
      end else if FuncIs( 'hamflush', 0, 0 ) then begin
         Result.AsInt := 0;
         If Not ArchivMode then NewsHistory.SaveToFile;

      end else if FuncIs( 'hamthreadcount', 0, 0 ) then begin
         Result.AsInt := GetCounter(CntActiveTasks);
      end else if FuncIs( 'hamisidle', 0, 0 ) then begin
         if GetCounter(CntActiveTasks)<=0 then Result.AsInt:=1 else Result.AsInt:=0;
      end else if FuncIs( 'hamwaitidle', 0, 1 ) then begin
         TimeOut := ParI( 0, -1 );
         Result.AsInt := 0;
         // Todo: replace by events (esp. evtHamster.IsIdle)
         while GetCounter(CntActiveTasks)>0 do begin
            i := 500;
            if Timeout>=0 then begin
               i := TimeOut;
               if i>500 then i:=500;
            end;
            if ThreadShutDown(false,(WaitForSingleObject(EVT_STOPSCRIPT,i)=WAIT_OBJECT_0)) then begin //HSR //One_SD_S
               Result.AsInt := 2;
               break;
            end;
            if AllShutDownReq or ThreadControl.Shutdown then break;
            if TimeOut>=0 then begin
               TimeOut := TimeOut - i;
               if TimeOut<=0 then begin Result.AsInt:=1; break; end;
            end;
         end;

      end else if FuncIs( 'hampurge', 0, 2 ) then begin
         Result.AsInt := -1;
         If Not ArchivMode then begin
            if GetCounter(CntActiveTasks)<=0 then begin
               i := ParI(0,HAM_PURGEOPT_DOALL);
               s:=ParS(1,'');
               TThreadPurge.Create( i , s, NIL).resume;
               Result.AsInt := 0
            end else begin
               Log(LogID_Warn, TrGl(kLog, 'hampurge.Abort.activetasks',
                  '"hampurge" cancelled, because there are active tasks'))
            end
         end;

      end else if FuncIs( 'hamrebuildgloballists', 0, 0 ) then begin
         Result.AsInt := -1;
         If Not ArchivMode then begin
            if GetCounter(CntActiveTasks)<=0 then begin
               TThreadRebuildGlobalLists.Create.resume;
               Result.AsInt := 0;
            end else begin
               Log(LogID_Warn, TrGl(kLog, 'hamrebuild.Abort.activetasks',
                  '"hamrebuildgloballists" cancelled, because there are active tasks'))
            end
         end
      end else if FuncIs( 'hamrebuildhistory', 0, 0 ) then begin
         Result.AsInt := -1;
         If Not ArchivMode then begin
            if GetCounter(CntActiveTasks)<=0 then begin
               TThreadHistoryRebuild.Create.resume;
               Result.AsInt := 0
            end else begin
               Log(LogID_Warn, TrGl(kLog, 'hamrebuildhistory.Abort.activetasks',
                  '"hamrebuildhistory" cancelled, because there are active tasks'))
            end
         end

      end else if FuncIs( 'hamsetlogin', 3, 3 ) then begin
         Result.AsInt := -1;
         Identifier := ParS(0);
         Username   := ParS(1);
         Password   := ParS(2);
         OK := True;
         if (copy(Username,1,1)='$') and (Password='') then begin
            s := Username;
            if not PasswordFile.UsePassword( s, Username, Password ) then begin
               Log( LOGID_WARN, TrGlF(kLog, 'Warning.PW$XX.MissingUsernamePW',
                  'Missing username/password for "%s"!', s) );
               OK := False;
            end;
         end;
         if OK then begin
            Result.AsInt := 0;
            PasswordFile.SavePassword( False, Identifier, Username, Password );
         end;

      end else if FuncIs( 'hamnewsjobscheckactive', 0, 1 ) then begin {JW} {news check info}
         If ArchivMode then Result.AsInt := 0
         else If CheckPurge(true) then Result.AsInt := 0
         else Result.AsInt := NewsJobs.CheckInfo( ParS(0,'') );

      end else if FuncIs( 'hamnewspull', 0, 1 ) then begin
         Result.AsInt := 0;
         If Not ArchivMode and Not CheckPurge(true) then begin
            SrvList := ParS(0,'');
            With TStringList.Create do begin
               With CfgHamster do begin
                  If SrvList = '' then begin
                     for LfdServer:=0 to ServerCount-1 do Add (IntToStr(lfdServer))
                  end else begin
                     Text := SplitServerList (SrvList);
                     For LfdServer := Count-1 downto 0 do begin
                         i := ServerIndexof[Strings[LfdServer]];
                         If i < 0 then begin
                            Log (LOGID_WARN, TrGlF(kLog, 'Skript.UnknownServerForHamNewspull',
                               'HamNewsPull: Unknown NNTP-Server "%s"!', Strings[LfdServer]));
                            Delete(LfdServer)
                         end else begin
                            Strings[LfdServer] := IntToStr(i)
                         end
                     end
                  end;
                  For i := 0 to Count-1 do begin
                     LfdServer := StrToInt(Strings[i]);
                     NewsJobs.AddPullSrv( ServerName[LfdServer], '' );
                     for LfdThread:=1 to ServerPullThreads(LfdServer,True) do begin
                        if AllShutDownReq or ThreadControl.Shutdown then begin
                           Result.asInt:=-1;
                           break
                        end;
                        NewsJobs.AddPullSrv( ServerName[LfdServer], '' );
                        if Result.asInt=0 then TThreadNewsJobs.Create( LfdServer ).resume;
                     end
                  end
               end
            end
         end
      end else if FuncIs( 'hamnewspost', 0, 1 ) then begin
         Result.AsInt := 0;
         If Not ArchivMode and Not CheckPurge(true) then begin
            SrvList := ParS(0,'');
            If SrvList > '' then begin
               With TStringList.Create do try
                  Text := SplitServerlist(SrvList);
                  SrvList := '';
                  For i := 0 to Count-1 do begin
                     If CfgHamster.ServerIndexOf [Strings[i]] >= 0
                        then SrvList := SrvList + ';' + Strings[i]
                        else Log (LOGID_WARN, TrGlF(kLog, 'Skript.UnknownServerForHamNewspost',
                               'HamNewsPost: Unknown NNTP-Server "%s"!', Strings[i]))
                  end;
                  If SrvList > '' then SrvList := SrvList + ';'
                                  else Result.asInt := -1
               finally Free end
            end;
            if (Result.asInt = 0)
               and (CfgHamster.ServerCount > 0)
               and FileExists2( PATH_NEWS_OUT+'*.'+CfgIni.ReadString( 'Setup', 'news.ext.out', 'msg' ) )
            then begin
               TThreadNewsPostSimulate.Create( SrvList ).resume
            end
         end

      end else if FuncIs( 'hamnewsjobsclear',   0, 0 ) then begin
         Result.AsInt := NewsJobs.Clear;
      end else if FuncIs( 'hamnewsjobspulldef', 0, 1 ) then begin
         If ArchivMode
            then Result.AsInt := 0
            else Result.AsInt := NewsJobs.AddPullDef( ParS( 0, '' ) ); //JW //NoFound
      end else if FuncIs( 'hamnewsjobspostdef', 0, 1 ) then begin
         If ArchivMode
            then Result.AsInt := 0
            else Result.AsInt := NewsJobs.AddPostDef( ParS( 0, '' ) );
      end else if FuncIs( 'hamnewsjobspull',    1, 2 ) then begin
         If ArchivMode
            then Result.AsInt := 0
            else Result.AsInt := NewsJobs.AddPullSrv( ParS(0), ParS(1,'') );
      end else if FuncIs( 'hamnewsjobspost',    1, 3 ) then begin
         If ArchivMode
            then Result.AsInt := 0
            else Result.AsInt := NewsJobs.AddPostSrv( ParS(0), ParS(1,''), ParS(2,'') );
      end else if FuncIs( 'hamnewsjobsstart',   0, 1 ) then begin
         If ArchivMode then Result.AsInt := 0
         else If CheckPurge(true) then Result.AsInt := 0
         else Result.AsInt := NewsJobs.StartThreads( ParS(0,'') );

      end else if FuncIs( 'hamnewsjobsgetserver', 1, 1 ) then begin
         Result.asStr := NewsJobs.GetServer( ParI(0) );
      end else if FuncIs( 'hamnewsjobsgetparam', 1, 1 ) then begin
         Result.asStr := NewsJobs.GetPar( ParI(0) );
      end else if FuncIs( 'hamnewsjobsgettype', 1, 1 ) then begin
         Result.asInt := NewsJobs.GetType( ParI(0) );
      end else if FuncIs( 'hamnewsjobsgetpriority', 1, 1 ) then begin
         Result.asInt := NewsJobs.GetPriority( ParI(0) );
      end else if FuncIs( 'hamnewsjobsgetcounter', 0, 0)  then begin
         Result.asInt := NewsJobs.GetCounter;
      end else if FuncIs( 'hamnewsjobsdelete', 1, 1)  then begin
          Result.asInt := NewsJobs.Delete( ParI(0) );
      end else if FuncIs( 'hamnewsjobssetpriority', 2, 2)  then begin
         Result.asInt := NewsJobs.SetPriority(ParI(0),ParI(1))
      end else if FuncIs( 'hamnewsjobsadd', 4, 4)  then begin
         Result.asInt := -1;
         If not ArchivMode then try
            Result.asInt := NewsJobs.Add(ParS(0),TJobType(ParI(1)),ParS(2),ParI(3))
         except
         end
{JW} {Feed}
      end else if FuncIs( 'hamnewsjobsfeed',    1, 2 ) then begin
         If ArchivMode
            then Result.asInt := 0
            else Result.asInt := NewsJobs.AddFeedSrv( ParS(0), ParS(1,'') );
{JW}

      end else if FuncIs( 'hamfetchmail', 1, 10 ) then begin
         Result.AsInt := 0;
         If Not ArchivMode then begin
            Server   := ParS(0);
            if Server='' then Result.AsInt:=-1;
            Port     := ParS(1,'');
            User     := ParS(2,'');
            Pass     := ParS(3,'');
            DestUser := ParS(4,'');
            FiltSect := ParS(5,'');
            LeaveOnServer := ParS(6,'?');
            {MG}{SSL}
            SSLMode   := ParI(7,0);
            SSLVerify := ParI(8,0);
            SSLCaFile := ParS(9,'');
            if Result.AsInt=0 then begin
                TThreadPop3Fetch.Create( Server, Port, User, Pass,
                   DestUser, FiltSect, LeaveOnServer,
                   SSLMode, SSLVerify, SSLCaFile ).resume
            end
            {/SSL}
         end;
      end else if FuncIs( 'hamsendmail', 1, 7 ) then begin
         Result.AsInt := -1;
         If Not ArchivMode then begin
            Server  := ParS(0);
            if Server>'' then begin
               Port    := ParS(1,'');
               FromSel := ParS(2,'');
               ToSel   := ParS(3,'');
               {MG}{SSL}
               SSLMode   := ParI(4,0);
               SSLVerify := ParI(5,0);
               SSLCaFile := ParS(6,'');
               TThreadSmtpSend.Create( Server, Port, '', '', FromSel, ToSel,{JW} {SMTP Auth 08.12.00}
                  False, SSLMode, SSLVerify, SSLCaFile ).resume;
               Result.asInt := 0
            end
         end
      end else if FuncIs( 'hamsendmailauth', 1, 9 ) then begin
         Result.AsInt:=-1;
         If Not ArchivMode then begin
            Server  := ParS(0);
            if Server>'' then begin
               Port    := ParS(1,'');
               User    := ParS(2,'');
               Pass    := ParS(3,'');
               FromSel := ParS(4,'');
               ToSel   := ParS(5,'');
               SSLMode   := ParI(6,0);
               SSLVerify := ParI(7,0);
               SSLCaFile := ParS(8,'');
               TThreadSmtpSend.Create( Server, Port, User, Pass, FromSel, ToSel,
                  True, SSLMode, SSLVerify, SSLCaFile ).resume;
               Result.AsInt := 0;
            end
         end
      end else if FuncIs( 'hammailexchange', 0, 2 ) then begin
         Result.AsInt := -1;
         If Not ArchivMode then begin
            SrvList := ParS( 0, '' );
            if SrvList<>'' then SrvList := ';' + LowerCase(SrvList) + ';';
            If (CfgHamster.Pop3ServerCount + CfgHamster.SmtpServerCount) > 0 then begin
               TThreadPop3AndSmtp.Create( SrvList, ParI(1, 0) ).resume;
               Result.AsInt := 0;
            end
         end
      end else if FuncIs('hamnewmail',3,3) then begin  //HRR: send a new mail
         Result.asInt := -1;
         s1:=ParS(0);
         s2:=ParS(1);
         s3:=ParS(2);
         if Not ArchivMode then If NewMail(s1,s2,s3) then Result.asInt:=0;

      end else if FuncIs('hamaddgroup',1,1) then begin
         Result.asInt := -1;
         s1 := ParS(0);
         If Not ArchivMode then begin
            If CfgHamster.ActiveAdd( s1 ) then begin
               Result.asInt := 0;
               Log( LOGID_SYSTEM, TrGlF(kLog, 'Script.GroupAdded.OK',
                  'New Group "%s" created by script', s1) )
            end else begin
               Log( LOGID_WARN, TrGlF(kLog, 'Script.GroupAdded.failed',
                  'Creation of new group "%s" by script failed', s1) )
            end
         end
      end else if FuncIs('hamdelgroup',1,1) then begin
         Result.asInt := -1;
         s1 := ParS(0);
         If Not ArchivMode then begin
            If CfgHamster.ActiveDel( s1 ) then begin
               Result.asInt := 0;
               Log( LOGID_SYSTEM, TrGlF(kLog, 'Script.GroupDeleted.OK',
                  'Group "%s" deleted by script', s1) )
            end else begin
               Log( LOGID_WARN, TrGlF(kLog, 'Script.GroupDeleted.failed',
                  'Deletion of group "%s" by script failed', s1) )
            end
         end
      end else if FuncIs('hamaddpull',2,2) then begin
         Result.asInt := -1;
         s1 := ParS(0);
         s2 := ParS(1);
         If Not ArchivMode then begin
            If CfgHamster.PullAdd( s1, s2 ) then begin
               Result.asInt := 0;
               Log( LOGID_SYSTEM, TrGlF(kLog, 'Script.PullAdded.OK',
                  'New pull for group "%s" from server "%s" created by script', [s2, s1]) )
            end else begin
               Log( LOGID_WARN, TrGlF(kLog, 'Script.PullAdded.failed',
                  'Creation of new pull for group "%s" from server "%s" by script failed', [s2, s1]) )
            end
         end
      end else if FuncIs('hamdelpull',2,2) then begin
         Result.asInt := -1;
         s1 := ParS(0);
         s2 := ParS(1);
         If Not ArchivMode then begin
            If CfgHamster.PullDel( s1, s2 ) then begin
               Result.asInt := 0;
               Log( LOGID_SYSTEM, TrGlF(kLog, 'Script.PullDeleted.OK',
                  'Pull for group "%s" from server "%s" deleted by script', [s2, s1]) )
            end else begin
               Log( LOGID_WARN, TrGlF(kLog, 'Script.PullDeleted.failed',
                  'Deletion of pull for group "%s" from server "%s" by script failed', [s2, s1]) )
            end
         end
      end else if FuncIs( 'hamrasdial', 1, 3 ) then begin
         Result.AsInt := -1;
         ConnectionID := ParS(0);
         Username     := ParS(1,'');
         Password     := ParS(2,'');
         OK := True;
         if (copy(Username,1,1)='$') and (Password='') then begin
            s := Username;
            if not PasswordFile.UsePassword( s, Username, Password ) then begin
               Log( LOGID_WARN, TrGlF(kLog, 'Warning.PW$XX.MissingUsernamePW',
                  'Missing username/password for "%s"!', s) );
               OK := False;
            end;
         end;
         if OK then Result.AsInt := RasDialer.Dial( ConnectionID,Username,Password );
      end else if FuncIs( 'hamrashangup', 0, 0 ) then begin
         Result.AsInt := RasDialer.HangUp;

      end else if FuncIs( 'hamgroupcount', 0, 0 ) then begin
         Result.AsInt := CfgHamster.ActiveCount;
      end else if FuncIs( 'hamgroupname',  1, 1 ) then begin
         i := ParI( 0 );
         If (i >= 0) and (i < CfgHamster.ActiveCount )
            then Result.AsStr := CfgHamster.ActiveName[ i ]
            else Result.AsStr := '';
      end else if FuncIs( 'hamgroupnamebyhandle',  1, 1 ) then begin
         i := ParI( 0 );
         Result.AsStr := ArticleBase.Name[i]
      end else if FuncIs( 'hamgroupindex', 1, 1 ) then begin
         s := ParS( 0 );
         Result.AsInt := CfgHamster.ActiveIndexOf[ s ];
      end else if FuncIs( 'hamgroupopen',  1, 1 ) then begin
         s := ParS( 0 );
         Result.AsInt := -1;
         If Not CheckPurge(true) then begin
            if CfgHamster.ActiveIndexOf[s]>=0 then begin
                i := ArticleBase.Open( s );
                If i >= 0 then begin
                   Engine.ResControl.Add( TResHamsterGroup.Create( i ) );
                   Result.AsInt := i
                end
            end
         end;
      end else if FuncIs( 'hamgroupclose', 1, 1 ) then begin
         Result.AsInt := 0;
         i := ParI( 0 );
         if i>=0 then begin
            Engine.ResControl.Remove( RESID_HamsterGroup, i );
            ArticleBase.Close( i )
         end
      end else if FuncIs( 'hamartcount', 1, 1 ) then begin
         i := ParI( 0 );
         Result.AsInt := ArticleBase.Count[ i ];
      end else if FuncIs( 'hamartnomax', 1, 1 ) then begin
         i := ParI( 0 );
         Result.AsInt := ArticleBase.LocalMax[ i ];
      end else if FuncIs( 'hamartnomin', 1, 1 ) then begin
         i := ParI( 0 );
         Result.AsInt := ArticleBase.LocalMin[ i ];
      end else if FuncIs( 'hamarttext',  2, 2 ) then begin
         i := ParI( 0 );
         j := ParI( 1 );
         Result.AsStr := ArticleBase.ReadArticle( i, j );
      end else if FuncIs( 'hamarttextexport',  2, 2 ) then begin
         i := ParI( 0 );
         j := ParI( 1 );
         Result.AsStr := ArticleBase.ReadArticle( i, j );
         if Result.AsStr<>'' then begin
            With TArticle.Create do try
               Text := Result.AsStr;
               Result.AsStr := AsExportFormat
            finally
               Free
            end
         end;

      end else if FuncIs( 'hamartimport',  1, 4 ) then begin
         Result.asInt := 0;
         If Not ArchivMode and Not CheckPurge(true) then begin
            s := ParS( 0 ); // article
            t := ParS( 1, '' ); // override groups
            i := ParI( 2, 0  ); // ignore history
            j := ParI( 3, 0  ); // mark no-archive
            Result.AsInt := Integer( ImportArticle( s, t, (i<>0), (j<>0) ) )
         end;

      end else if FuncIs( 'hamartdeletemid',  1, 1 ) then begin
         Result.asInt := -1;
         If Not ArchivMode and Not CheckPurge(true) then begin
            s := ParS( 0 ); // MID
            if ArticleBase.DeleteArticleByMID( s, true ) then Result.AsInt:=0
         end
      end else if FuncIs( 'hamartdeletemidingroup',  2, 2 ) then begin
         Result.asInt := -1;
         If Not ArchivMode and Not CheckPurge(true) then begin
            s := ParS( 0 ); // MID
            t := ParS( 1 ); // Group
            if ArticleBase.DeleteArticleByMIDAndGroup( s, t, true ) then Result.AsInt:=0
         end
      end else if FuncIs( 'hamartdeletenringroup',  2, 2 ) then begin
         Result.asInt := -1;
         If Not ArchivMode and Not CheckPurge(true) then begin
            i := ParI( 0 ); // Nr
            t := ParS( 1 ); // Group
            Result.AsInt := -2;
            If i >= 0 then
               If ArticleBase.DeleteArticleByNrAndGroup( i, t )
                  then Result.AsInt := 0
         end

      end else if FuncIs( 'hamartlocatemid',  1, 3 ) then begin
         Result.AsInt := -1;
         s := ParS( 0 ); // MID
         if NewsHistory.LocateMID( s, t, i ) then begin
            s := ParX( 1, '' ); // var for groupname
            if IsVariable(s) then Engine.Variables.Value[s].AsStr := t;
            s := ParX( 2, '' ); // var for artno
            if IsVariable(s) then Engine.Variables.Value[s].AsInt := i;
            Result.AsInt := 0;
         end;
      end else if FuncIs( 'hamartlocatemidingroup',  1, 3 ) then begin
         Result.AsInt := -1;
         s := ParS( 0 ); // MID
         t := ParS( 1, '' ); // Group
         if NewsHistory.LocateMIDInGroup( s, t, i ) then begin
            s := ParX( 2, '' ); // var for artno
            if IsVariable(s) then Engine.Variables.Value[s].AsInt := i;
            Result.AsInt := 0;
         end;

      end else if FuncIs( 'hamscorelist',  1, 1 ) then begin
         s := ParS( 0 ); // grpname
         ScoreFile := TFiltersNews.Create( PATH_BASE + CFGFILE_SCORES );
         ScoreFile.SelectSections( s );
         Result.AsStr := ScoreFile.SelectedLines;
         ScoreFile.Free;

      end else if FuncIs( 'hamscoretest',  2, 4 ) then begin
         Result.AsInt := 0;
         s := ParS( 0 ); // grpname
         t := ParS( 1 ); // arttext
         ScoreFile := TFiltersNews.Create( PATH_BASE + CFGFILE_SCORES );
         try
            ScoreFile.SelectSections( s );
            Art := TArticle.Create;
            try
               Art.Text := t;
               t := '';
               ArticleToXOverRec( Art, XOverRec );
               case ParI( 3, 0 ) of
                  1: Result.AsInt := ScoreFile.ScoreBeforeLoad( XOverRec, @t );
                  2: Result.AsInt := ScoreFile.ScoreAfterLoad( Art, @t );
                  else Result.AsInt := ScoreFile.ScoreBeforeLoad( XOverRec, @t )
                                     + ScoreFile.ScoreAfterLoad( Art, @t );
               end;
            finally Art.Free end;
            s := ParX( 2, '' ); // var for matchlog
            if IsVariable(s) then Engine.Variables.Value[s].AsStr := t;
         finally ScoreFile.Free end

      end else if FuncIs( 'hamgetstatus', 1, 2 ) then begin
         i := ParI(0);
         j := ParI(1,0);
         Result.AsInt := LocalServerGetState( i, j )

      end else if FuncIs( 'hamchangepassword', 3, 3 ) then begin
         Result.asInt := 2;
         If ArchivMode then Exit;
         i := CfgAccounts.UserIDOf(ParS(0));
         Result.AsInt := 0;
         If CfgAccounts.Value[i, ACTP_PASSWORD] = ParS(1) then begin
            CfgAccounts.Value[i, ACTP_PASSWORD] := ParS(2);
            Result.asInt := 1
         end else begin
            Log ( LOGID_WARN, TrGlF (kLog, 'hs2.hamChangePassword.Fails',
                   'hs2-Change of password for account "%s" failed', ParS(0)))
         end
      end else begin
         Result.Unassign;
      end
   end
end;

function THscEngineHamster.Engine_StartScript( const ScriptFile: String;
                                               const Params: String;
                                               const WaitForEnd: Boolean ): Integer;
var  ScriptThread: TThread;
begin
    Result := StartNewScript( ScriptFile, Params, WaitForEnd,
                              ScriptThread, False, True );
end;

constructor THscEngineHamster.Create( AParentThread: TThread;
                                      const APathHSM: String;
                                      const AThreadSynchronize: TThreadSynchronize;
                                      const AOutputBuffer: TThreadStringList );
begin
   inherited Create( AParentThread, APathHSM );
   FThreadSynchronize := AThreadSynchronize;
   FOutputBuffer      := AOutputBuffer;
   StopEvent          := EVT_STOPSCRIPT;
end;

end.
