unit cIMAPMailbox;

interface

uses Windows, cIMAPMailboxIndex, Classes, uDateTime;

type
  TMessageSet = array of Integer;
  TStoreMode  = set of ( smAdd, smReplace, smDelete );

  tOnNewMess = procedure of Object; //HSR //IDLE
  tOnExpunge = procedure (Number : Integer) of Object; //HSR //IDLE

  pIMAPNotification = ^tIMAPNotification;
  tIMAPNotification = record //HSR //IDLE
    OnNewMess : tOnNewMess;
    OnExpunge : tOnExpunge;
  end;

  TImapMailbox = class
    private
      fCritSection : TRTLCriticalSection;
      fIndex       : TImapMailboxIndex;
      fPath        : String;
      fStatus      : TMbxStatus;
      fUsers       : TList;
      fReadOnly    : Boolean; //ClientRO
      function  GetStatus: TMbxStatus;
      function  ExtractParameter( var Params: String ): String;
      procedure AddMessage( Flags: String; TimeStamp: TUnixTime );
      function  StringToFlagMask( Flags: String ): TFlagMask;
      function  FlagMaskToString( FlagMask: TFlagMask ): String;
      function  GetPossFlags: String;
    public
      function  RebuildStatusFile: TMbxStatus;
      function  StrToMsgSet( s: String; UseUID: Boolean ): TMessageSet;
      procedure WriteStatus;

      property  Status         : TMBxStatus read  fStatus;
      property  Path           : String     read  fPath;
      property  MBReadOnly     : Boolean    read  fReadOnly write fReadOnly; //ClientRO //ToDo: 'write' von ReadOnly lschen oder umfunktionieren 
      property  GetUIDnext     : LongInt    read  fStatus.UIDNext;
      property  GetUIDvalidity : TUnixTime  read  fStatus.UIDvalidity;
      property  PossFlags      : String     read  GetPossFlags;
      procedure Lock;
      procedure Unlock;
      procedure RemoveRecentFlags;
      procedure AddUser( Notify : pIMAPNotification );
      procedure RemoveUser( Notify : pIMAPNotification; out NoUsersLeft: Boolean );
      procedure Expunge( ExcludeFromResponse: pIMAPNotification );

      function  Search( Charset, Criteria: String; UseUID: Boolean): String;
      function  Fetch( Idx: Integer; MsgDat: String; var Success: Boolean ): String;
      function  CopyMessage( MsgSet: TMessageSet; Destination: TImapMailbox ): Boolean;

//      function  Store( Idx: Integer; Flags: TFlagMask; Mode: TStoreMode ): TFlagMask;
      function  Store( Idx: Integer; Flags: String; Mode: TStoreMode ): String;
//      function  AppendMessage( MsgTxt: String; Flags: TFlagMask; TimeStamp: TUnixTime ): String;
      function  AppendMessage( MsgTxt: String; Flags: String; TimeStamp: TUnixTime ): String;

      function  AreValidFlags(Flags: String) : Boolean;

      procedure AddIncomingMessage(Const Flags : String = '');
      procedure SendMailboxUpdate;

      constructor Create( APath: String );
      destructor Destroy; override;
  end;

//------------------------------------------------------------------------------
implementation

uses uTools, Sysutils, cLogFile, cAccount, uIMAPUtils, cIMAPMessage, Config, uEncoding, FileCtrl;

const
     FLAGNONE     : TFlagMask =  0;
     FLAGSEEN     : TFlagMask =  1;
     FLAGANSWERED : TFlagMask =  2;
     FLAGFLAGGED  : TFlagMask =  4;
     FLAGDELETED  : TFlagMask =  8;
     FLAGDRAFT    : TFlagMask = 16;
     FLAGRECENT   : TFlagMask = 32;

     FLAGAPPROVE  : TFlagMask = 64;  //Moder
     FLAGDECLINE  : TFlagMask = 128; //Moder

// --------------------------------------------------------- TImapMailbox -----
function TImapMailbox.StringToFlagMask( Flags: String ): TFlagMask;
begin
     Result := FLAGNONE;
     Flags := uppercase(Flags);
     if pos( '\SEEN',     Flags ) > 0 then Result := Result or FLAGSEEN;
     if pos( '\ANSWERED', Flags ) > 0 then Result := Result or FLAGANSWERED;
     if pos( '\FLAGGED',  Flags ) > 0 then Result := Result or FLAGFLAGGED;
     if pos( '\DELETED',  Flags ) > 0 then Result := Result or FLAGDELETED;
     if pos( '\DRAFT',    Flags ) > 0 then Result := Result or FLAGDRAFT;
     if pos( '\RECENT',   Flags ) > 0 then Result := Result or FLAGRECENT;
     //Moder
     if pos( '\APPROVE',  Flags ) > 0 then Result := Result or FLAGAPPROVE;
     if pos( '\DECLINE',  Flags ) > 0 then Result := Result or FLAGDECLINE;
end;

function TImapMailbox.FlagMaskToString( FlagMask: TFlagMask ): String;
begin
     Result := '';

     //Moder
     if FlagMask and FLAGAPPROVE  = FLAGAPPROVE  then Result := Result + ' \Approve';
     if FlagMask and FLAGDECLINE  = FLAGDECLINE  then Result := Result + ' \Decline';


     if FlagMask and FLAGSEEN     = FLAGSEEN     then Result := Result + ' \Seen';
     if FlagMask and FLAGANSWERED = FLAGANSWERED then Result := Result + ' \Answered';
     if FlagMask and FLAGFLAGGED  = FLAGFLAGGED  then Result := Result + ' \Flagged';
     if FlagMask and FLAGDELETED  = FLAGDELETED  then Result := Result + ' \Deleted';
     if FlagMask and FLAGDRAFT    = FLAGDRAFT    then Result := Result + ' \Draft';
     if FlagMask and FLAGRECENT   = FLAGRECENT   then Result := Result + ' \Recent';
     if Result<>'' then Delete( Result, 1, 1 );
     Result := '(' + Result + ')';
end;

function  TImapMailbox.AreValidFlags(Flags: String) : Boolean;
begin
  Result := not (pos('\RECENT', Flags)>0)
end;

function TImapMailbox.GetPossFlags: String;
begin
  Result := '(\Answered \Flagged \Deleted \Seen \Draft)'
end;

function TImapMailbox.GetStatus: TMbxStatus;
var  RebuildNeeded : Boolean;
begin
     RebuildNeeded := False;
     FillChar( Result, SizeOf(Result), 0 );
     if FileExists2( fPath + IMAPSTATUS_FILENAME ) then begin
        with TFileStream.Create( fPath + IMAPSTATUS_FILENAME, fmOpenRead ) do try
           if Read( Result, SizeOf(Result) ) <> SizeOf(Result) then begin
              Log( LOGID_WARN, 'Error reading imap mailbox status file.' );
              RebuildNeeded := True;
           end
        finally Free end
     end else RebuildNeeded := True;
     if RebuildNeeded then Result := RebuildStatusFile;
end;

function TImapMailbox.RebuildStatusFile: TMbxStatus;
var  Status     : TMbxStatus;
     MR         : TMessageRec;
     Chunk      : Byte;
     i          : Integer;
begin
     Log( LOGID_INFO, 'Rebuilding mailbox status file.' );

     FillChar( Status, SizeOf(Status), 0 );
     Status.UIDvalidity := DateTimeToUnixTime( NowGMT );
     Status.UIDnext     := 1;

     FillChar( MR, sizeof(MR), 0 );
     Chunk := fIndex.pubChunkOf( MR );
     try
       Lock;
       fIndex.Enter( Chunk );
       try
          Status.Messages := fIndex.Count;
          for i := 0 to Status.Messages-1 do begin
             fIndex.pubRecGet( Chunk, i, MR );
             if MR.Flags and FLAGSEEN <> FLAGSEEN then inc( Status.Unseen );
             if MR.Flags and FLAGRECENT = FLAGRECENT then inc( Status.Recent );
             // if MR.UID >= Status.UIDnext then Status.UIDnext := MR.UID + 1;
          end;
          {MG}{IMAP-UID-ByteOrder}
          fIndex.pubRecGet( Chunk, Status.Messages-1, MR );
          if SwitchByteOrder( MR.UID ) >= Status.UIDnext then begin
             Status.UIDnext := SwitchByteOrder( MR.UID ) + 1
          end;
          {/IMAP-UID-ByteOrder}
       finally
          fIndex.Leave( Chunk );
       end;
     finally
       Unlock;
     end;

     Result := Status;
end;

procedure TImapMailbox.WriteStatus;
begin
   try
     Lock;
     with TFileStream.Create( fPath + IMAPSTATUS_FILENAME, fmCreate ) do try
        if Write( fStatus, SizeOf(fStatus) ) <> SizeOf(fStatus) then
           Log( LOGID_ERROR, 'Error writing imap mailbox status to disk.' );
     finally
        Free;
     end;
   finally
     Unlock;
   end
end;

procedure TImapMailbox.Expunge( ExcludeFromResponse: pIMAPNotification );
var  Chunk : Byte;
     i,j   : Integer;
     MR    : TMessageRec;
     ServerThread : pIMAPNotification;
     FN: String;
begin
   if fReadOnly then exit;
   try
     Lock;
      for Chunk:=0 to fIndex.propCHUNK_MAX do begin
         fIndex.Enter( Chunk );
         try
            for i := fIndex.Count-1 downto 0 do begin
               fIndex.pubRecGet( Chunk, i, MR );
               if (MR.Flags and FLAGDELETED) = FLAGDELETED then begin
                  FN := fPath + IntToStr(SwitchByteOrder(MR.UID)) + '.msg';
                  If SysUtils.DeleteFile( FN ) then begin
                     if fIndex.ContainsKey(MR, INDEXKEYLEN_UID ) then begin
                       dec( fStatus.Messages );
                       if (MR.Flags and FLAGSEEN) <> FLAGSEEN then dec( fStatus.Unseen );
                       if (MR.Flags and FLAGRECENT) = FLAGRECENT then dec( fStatus.Recent );
                     end;

                     fIndex.RemoveKey( MR, INDEXKEYLEN_UID );
                     for j := 0 to fUsers.Count-1 do begin
                        try
                           if Assigned( fUsers.Items[j] ) then begin
                              ServerThread := pIMAPNotification( fUsers.Items[j] );
                              if ServerThread <> ExcludeFromResponse then begin
                                 ServerThread^.OnExpunge( i+1 )
                              end
                           end
                        except
                        end
                     end
                  end else begin
                     if FileExists2( FN ) then
                       Log( LOGID_ERROR, 'Error deleting imap message file ' + FN + ' in use?' )
                     else begin
                       Log( LOGID_WARN, 'Error deleting imap message file ' + FN + ': Deleted outside' );
                       fIndex.RemoveKey( MR, INDEXKEYLEN_UID )
                     end
                  end
               end
            end
         finally
            fIndex.Leave(Chunk)
         end
      end;
      fIndex.SaveToFile
   finally
      Unlock
   end;
   WriteStatus
end;

function TImapMailbox.Store( Idx: Integer; Flags: String; Mode: TStoreMode ): String;
var
  FlagMsk : tFlagMask;
begin
  if fReadOnly then begin
    Result := FlagMaskToString(fIndex.GetFlags( Idx ));
    exit
  end;
  try
     Lock;
     FlagMsk := StringToFlagMask(Flags);
     if FlagMsk and FLAGSEEN = FLAGSEEN then begin
        if fIndex.GetFlags(Idx) and FLAGSEEN = FLAGSEEN then begin
           if Mode = [smDelete] then inc( fStatus.Unseen )
        end else begin
           if Mode <= [smReplace, smAdd] then dec( fStatus.Unseen )
        end
     end else begin
        if (fIndex.GetFlags(Idx) and FLAGSEEN = FLAGSEEN) then
           if Mode = [smReplace] then inc( fStatus.Unseen );
     end;

     if      Mode = [smReplace] then FlagMsk := fIndex.SetFlags   ( Idx, FlagMsk )
     else if Mode = [smDelete]  then FlagMsk := fIndex.RemoveFlags( Idx, FlagMsk )
     else if Mode = [smAdd]     then FlagMsk := fIndex.AddFlags   ( Idx, FlagMsk )
     else                            FlagMsk := fIndex.GetFlags   ( Idx ); // just in case ...
     Result := FlagMaskToString(FlagMsk);
  finally
     Unlock;
  end;
end;

procedure TImapMailbox.RemoveRecentFlags;
var  i: Integer;
begin
  if fReadOnly then exit;
  for i := 0 to fIndex.Count-1 do fIndex.RemoveFlags( i, FLAGRECENT );
  fStatus.Recent := 0;
end;

procedure TImapMailbox.AddMessage( Flags: String; TimeStamp: TUnixTime );
var FMFlags : tFlagMask;
begin
  try
     Lock;
     FMFlags := StringToFlagMask(Flags);
     fIndex.AddEntry( GetUIDnext, FMFlags, TimeStamp );
     inc( fStatus.UIDnext );
     inc( fStatus.Messages );
     if FMFlags and FLAGSEEN <> FLAGSEEN then inc( fStatus.Unseen );
     if FMFlags and FLAGRECENT = FLAGRECENT then inc( fStatus.Recent );
  finally
     Unlock;
  end;
end;

function TImapMailbox.CopyMessage( MsgSet: TMessageSet; Destination: TImapMailbox ): Boolean;
var  i: Integer;
     FileName: String;
begin
  if Destination.MBReadOnly then begin
    Result := false;
    exit
  end;
  try
     Lock;
     Result := True;
     for i := 0 to High(MsgSet) do begin
        FileName := fPath + fIndex.GetUIDStr(MsgSet[i]-1) + '.msg';
        if not FileExists2( Filename ) then begin Result := False; continue end;
        if Windows.CopyFile( PChar(Filename), PChar( Destination.Path +
           IntToStr(Destination.GetUIDnext) + '.msg' ), True )
        then begin {MG}{ImapLog}
           Log( LOGID_DETAIL, 'Message file ' + FileName + ' copied to ' + Destination.Path );
           Destination.AddMessage( FlagMaskToString(fIndex.GetFlags(MsgSet[i]-1) and not FLAGRECENT),
                                   fIndex.GetTimeStamp(MsgSet[i]-1) )
        end else begin
           Log( LOGID_WARN, 'Error copying message ' + FileName +' to ' + Destination.Path );
           Result := False;
        end
     end;
  finally
     Unlock;
  end;
  Destination.SendMailboxUpdate;
end;

function TImapMailbox.AppendMessage( MsgTxt: String; Flags: String;
                                     TimeStamp: TUnixTime ): String;
var  Bytes    : Integer;
     FileName : String;
begin
  Result   := 'NO APPEND error: [Read-Only] ';
  if fReadOnly then exit; //ClientRO

  Result   := 'NO APPEND error';

  FileName := fPath + IntToStr( GetUIDnext ) + '.msg';
  try
     Lock;
     try
        with TFileStream.Create( FileName, fmCreate ) do try
           If MsgTxt > ''
              then Bytes := Write( MsgTxt[1], Length(MsgTxt) )
              else Bytes := 0
        finally Free end;
        if Bytes = Length(MsgTxt) then begin
           AddMessage( Flags, TimeStamp );
           Log( LOGID_DETAIL, 'Message appended to imap mailbox "' + fPath + '" with Flags: ' + Flags);
           Result := 'OK APPEND completed';
           SendMailboxUpdate;
        end else begin
           Log( LOGID_ERROR, 'Error appending message to imap mailbox "' + fPath + '"' );
        end;
     except
        on E: Exception do
           Log( LOGID_ERROR, 'Couldn''t append message: ' + E.Message );
     end;
  finally
     Unlock;
  end;
end;

procedure TImapMailbox.AddIncomingMessage(Const Flags : String = '');
begin
   if Flags = ''
      then AddMessage( FlagMaskToString(FLAGRECENT), DateTimeToUnixTime(NowGMT) )
      else AddMessage( Flags, DateTimeToUnixTime(NowGMT) );
   SendMailboxUpdate;
end;

procedure TImapMailbox.SendMailboxUpdate;
var  i: Integer;
//     FirstUser: Boolean;
begin
//     FirstUser := True;
  for i := 0 to fUsers.Count-1 do try
    if Assigned( fUsers.Items[i] ) then begin
      pIMAPNotification(fUsers.Items[i])^.OnNewMess;
{
           TSrvIMAPCli( Users.Items[i] ).SendRes( IntToStr(GetMessages) + ' EXISTS');
           if FirstUser then begin
              TSrvIMAPCli( Users.Items[i] ).SendRes( IntToStr(GetRecent) + ' RECENT');
              RemoveRecentFlags;
              FirstUser := False;
           end }
    end
  except Continue end;
  if not fReadOnly then
    RemoveRecentFlags;
end;

function TImapMailbox.ExtractParameter(var Params: String): String;
var  i: Integer;
begin
     Params := TrimWhSpace( Params );
     i := PosWhSpace( Params );
     if (i > 0) and (Pos( '[', Params ) < i) and (Pos( ']', Params ) > i) then begin
        i := Pos( ']', Params ) + 1;
        while i <= Length( Params ) do begin
           if Params[i] in [#9,' '] then break;
           inc( i )
        end
     end;
     if (i > 0) then begin
        Result := Uppercase( TrimQuotes( copy( Params, 1, i-1 ) ) );
        Params := TrimWhSpace( copy( Params, i+1, length(Params)-i ) );
     end else begin
        Result := Uppercase( TrimQuotes( Params ) );
        Params := '';
     end
end;

type tIntArray = array of Integer;

function TImapMailbox.Search( Charset, Criteria: String; UseUID: Boolean): String;
  function privSearch( Charset, Criteria: String; UseUID: Boolean; Found : tIntArray): String;
       function FoundToString( SearchResult: TIntArray ): String;
       var  i: Integer;
            s: String;
       begin
            s := '';
            if UseUID then
               for i := 0 to Length(SearchResult)-1 do
                  s := s + ' ' + fIndex.GetUIDStr( SearchResult[i] )
            else
               for i := 0 to Length(SearchResult)-1 do
                  s := s + ' ' + IntToStr( SearchResult[i] + 1 );
            Result := s;
       end;


       procedure FindFlags( WantedFlag: TFlagMask; Exclude: Boolean; var Found: TIntArray );
       var  i, j: Integer;
       begin
            j := 0;
            for i := 0 to Length( Found ) - 1 do
               if (fIndex.GetFlags(Found[i]) and WantedFlag = WantedFlag)
                  xor Exclude then begin
                  Found[j] := Found[i];
                  inc( j )
               end;
            SetLength( Found, j )
       end;

       procedure FindDate( DateText: String; Exclude: Boolean; When: String; var Found: TIntArray );
       var  i, j: Integer;
            Date, MyDate: Int64;
       begin
            j := 0;
            Date := Trunc( ImapDateTextToDateTime( DateText ) );
            for i := 0 to Length( Found ) - 1 do begin
               MyDate := Trunc( UnixTimeToDateTime( fIndex.GetTimeStamp(Found[i]) ) );
               if When = 'BEFORE' then
                  if ( MyDate >= Date ) xor Exclude then continue;
               if When = 'ON' then
                  if ( MyDate <> Date ) xor Exclude then continue;
               if When = 'SINCE' then
                  if ( MyDate <= Date ) xor Exclude then continue;
               Found[j] := Found[i];
               inc( j )
            end;
            SetLength( Found, j )
       end;

       procedure FindLarger( Size: Integer; Exclude: Boolean; var Found: TIntArray );
       var  i, j: Integer;
            MyMail : TImapMessage;
       begin
            j := 0;
            MyMail := TImapMessage.Create;
            try
               for i := 0 to Length( Found ) - 1 do begin
                  MyMail.LoadFromFile( fPath + fIndex.GetUIDStr(Found[i]) + '.msg' );
                  if ( Length(MyMail.FullBody) > Size ) xor Exclude then begin
                     Found[j] := Found[i];
                     inc( j )
                  end;
                  MyMail.Text := '';
               end
            finally MyMail.Free end;
            SetLength( Found, j )
       end;

       procedure FindMessageSet( MsgSet: TMessageSet; Exclude: Boolean; var Found: TIntArray );
       var  i, j, k: Integer;
            GotIt: Boolean;
       begin
            j := 0;
            for i := 0 to Length( Found ) - 1 do begin
               GotIt := False;
               for k := 0 to High(MsgSet) do
                  if MsgSet[k] = Found[i]+1 then begin
                     GotIt := True;
                     break;
                  end;
               if GotIt xor Exclude then begin
                  Found[j] := Found[i];
                  inc( j );
               end;
            end;
            SetLength( Found, j )
       end;

       procedure FindHeader( FieldName, Value: String; Exclude: Boolean; var Found: TIntArray);
       var  i, j   : Integer;
            Hdr    : String;
            MyMail : TImapMessage;
            temp : integer;
            cset:string; //JW //IMAP SEARCH
       begin
            j := 0;
            if FieldName = '' then begin FieldName := Value; Value := ''; end;
            if FieldName <> '' then begin
               if FieldName[Length(FieldName)] = ':' then Hdr := FieldName
               else Hdr := FieldName + ':';
               MyMail := TImapMessage.Create;
               try
                  for i := 0 to Length( Found ) - 1 do begin
                     MyMail.LoadFromFile( fPath + fIndex.GetUIDStr(Found[i]) + '.msg' );
                     temp := -1;
                     if ( ( (Value='') and
                            (MyMail.HeaderAfterIdx(Hdr,temp)>'') )
  {JW} {IMAP SEARCH}   // possible Bug?
  //                        or ( Pos(MyMail.Header[Hdr],Value) > 0 ) )
                          or (Pos(Value,
                             DecodeHeadervalue(MyMail.Header[Hdr],cset))>0))
                        xor Exclude then begin

                          if  (Charset='') or
                              (Charset=Trim(Uppercase(cset))) then begin
                             Found[j] := Found[i];
                             inc( j );
                          end;
  {JW}
                     end;
                     MyMail.Text := '';
                  end
               finally MyMail.Free end
            end;
            SetLength( Found, j )
       end;

       procedure FindText( Value: String; IncludeHeader, Exclude: Boolean; var Found: TIntArray);
       var  i, j   : Integer;
            MyMail : TImapMessage;
       begin
            j := 0;
            MyMail := TImapMessage.Create;
            try
               for i := 0 to Length( Found ) - 1 do begin
                  MyMail.LoadFromFile( fPath + fIndex.GetUIDStr(Found[i]) + '.msg' );
                  if IncludeHeader then begin
  {JW} {IMAP SEARCH}
                     if (Pos( Value, MyMail.Text) = 0) xor
                         Exclude then continue;
  {JW}
                  end else begin
                     if (Pos( MyMail.FullBody, Value ) = 0) xor Exclude then continue;
                  end;
                  Found[j] := Found[i];
                  inc( j );
                  MyMail.Text := '';
               end;
            finally MyMail.Free end;
            SetLength( Found, j )
       end;

       procedure FindSent( DateText, When: String; Exclude: Boolean; var Found: TIntArray);
       var  i, j         : Integer;
            Date, MyDate : Int64;
            MyMail       : TImapMessage;
       begin
            j := 0;
            Date := Trunc( ImapDateTextToDateTime( DateText ) );
            MyMail := TImapMessage.Create;
            try
               for i := 0 to Length( Found ) - 1 do begin
                  MyMail.LoadFromFile( fPath + fIndex.GetUIDStr(Found[i]) + '.msg' );
                  MyDate := Trunc( RfcDateTimeToDateTimeGMT(
                                   MyMail.Header['Date:'] ) );
                  if When = 'BEFORE' then
                     if ( MyDate >= Date ) xor Exclude then continue;
                  if When = 'SINCE' then
                     if ( MyDate <= Date ) xor Exclude then continue;
                  if When = 'ON' then
                     if ( MyDate <> Date ) xor Exclude then continue;
                  Found[j] := Found[i];
                  inc( j );
                  MyMail.Text := '';
               end;
            finally MyMail.Free end;
            SetLength( Found, j )
       end;

  var
       SearchKey: String;
       Negation: Boolean;
       SearchResult: TIntArray;
       i : integer;
  begin
     Negation := False;
     SearchKey := ExtractParameter( Criteria );
     if SearchKey = 'AND' then
        SearchKey := ExtractParameter( Criteria );

     if SearchKey = 'NOT' then begin
        Negation := True;
        SearchKey := ExtractParameter( Criteria );
     end;
{
     if SearchKey = 'OR' then begin

        TODO : Support fr OR einfgen!

     end;
}
     SetLength(SearchResult, length(Found));
     for i := 0 to length(found)-1 do
       SearchResult[i] := Found[i];

     if SearchKey = 'ALL' then SearchResult := Found
     else if SearchKey = 'SEEN' then
        FindFlags( FLAGSEEN, Negation, SearchResult )
     else if SearchKey = 'ANSWERED' then
        FindFlags( FLAGANSWERED, Negation, SearchResult )
     else if SearchKey = 'FLAGGED' then
        FindFlags( FLAGFLAGGED, Negation, SearchResult )
     else if SearchKey = 'DELETED' then
        FindFlags( FLAGDELETED, Negation, SearchResult )
     else if SearchKey = 'DRAFT' then
        FindFlags( FLAGDRAFT, Negation, SearchResult )
     else if SearchKey = 'RECENT' then
        FindFlags( FLAGRECENT, Negation, SearchResult )
     else if SearchKey = 'NEW' then begin
        FindFlags( FLAGRECENT, Negation, SearchResult );
        FindFlags( FLAGSEEN, not Negation, SearchResult )
     end else if SearchKey = 'OLD' then
        FindFlags( FLAGRECENT, not Negation, SearchResult )
     else if SearchKey = 'UNANSWERED' then
        FindFlags( FLAGANSWERED, not Negation, SearchResult )
     else if SearchKey = 'UNDELETED' then
        FindFlags( FLAGDELETED, not Negation, SearchResult )
     else if SearchKey = 'UNDRAFT' then
        FindFlags( FLAGDRAFT, not Negation, SearchResult )
     else if SearchKey = 'UNFLAGGED' then
        FindFlags( FLAGFLAGGED, not Negation, SearchResult )
     else if SearchKey = 'UNSEEN' then
        FindFlags( FLAGSEEN, not Negation, SearchResult )
     else if SearchKey = 'BEFORE' then
        FindDate( ExtractParameter( Criteria ), Negation, 'BEFORE', SearchResult )
     else if SearchKey = 'ON' then
        FindDate( ExtractParameter( Criteria ), Negation, 'ON', SearchResult )
     else if SearchKey = 'SINCE' then
        FindDate( ExtractParameter( Criteria ), Negation, 'SINCE', SearchResult )
     else if SearchKey = 'LARGER' then
        FindLarger( StrToInt( ExtractParameter( Criteria ) ), Negation, SearchResult )
     else if SearchKey = 'SMALLER' then
        FindLarger( StrToInt( ExtractParameter( Criteria ) ), not Negation, SearchResult )
     else if SearchKey = 'HEADER' then
        FindHeader( ExtractParameter( Criteria ), ExtractParameter( Criteria ), Negation, SearchResult )
     else if SearchKey = 'FROM' then
        FindHeader( 'From:', ExtractParameter( Criteria ), Negation, SearchResult )
     else if SearchKey = 'TO' then
        FindHeader( 'To:', ExtractParameter( Criteria ), Negation, SearchResult )
     else if SearchKey = 'CC' then
        FindHeader( 'CC:', ExtractParameter( Criteria ), Negation, SearchResult )
     else if SearchKey = 'BCC' then
        FindHeader( 'BCC:', ExtractParameter( Criteria ), Negation, SearchResult )
     else if SearchKey = 'SUBJECT' then
        FindHeader( 'Subject:', ExtractParameter( Criteria ), Negation, SearchResult )
     else if SearchKey = 'BODY' then
        FindText( ExtractParameter( Criteria ), False, Negation, SearchResult )
     else if SearchKey = 'TEXT' then
        FindText( ExtractParameter( Criteria ), True, Negation, SearchResult )
     else if (SearchKey = 'SENTON') or (SearchKey = 'SENTBEFORE')
                                    or (SearchKey = 'SENTSINCE') then begin
        System.Delete( SearchKey, 1, 4 );
        FindSent( ExtractParameter( Criteria ), SearchKey, Negation, SearchResult )
     end else if (SearchKey = 'KEYWORD') or (SearchKey = 'UNKEYWORD') then
        begin
           ExtractParameter( Criteria );
           Log( LOGID_WARN, 'Imap search key ''' + SearchKey + ''' not implemented yet!' );
           //TODO: Search: Keyword
        end
     else if SearchKey = 'UID' then
        FindMessageSet( StrToMsgSet( ExtractParameter( Criteria ), True ), Negation, SearchResult )
     else
        FindMessageSet( StrToMsgSet( SearchKey, False ), Negation, SearchResult );

     if Criteria <> '' then Result := privSearch( Charset, Criteria, UseUID, SearchResult )
     else Result := 'SEARCH' + FoundToString( SearchResult )
  end;

var  i : integer;
  temp : tIntArray; //previous as param
begin
 {previous in DoSearch}
 SetLength( temp, Status.Messages );
 for i := 0 to Status.Messages - 1 do
   temp[i] := i;
 {/}
 Result := privSearch(charset, criteria, UseUID, temp);
end;

function TImapMailbox.Fetch( Idx: Integer; MsgDat: String; var Success: Boolean ): String;
      function MakeLiteral( Txt: String ): String;
      begin
           Result := '{' + IntToStr( Length(Txt) ) + '}' + #13#10 + Txt
      end;

var  Filename, Args, DataItem, Data: String;
     MyMail: TImapMessage;

     procedure AddDataValue( NewValue: String );
     begin
          Data := Data + ' ' + DataItem + ' ' + NewValue
     end;

begin
     Args    := MsgDat;
     Data    := '';

     Filename := fPath + fIndex.GetUIDStr( Idx ) + '.msg';
     if not FileExists2( Filename ) then exit;

     MyMail := TImapMessage.Create;
     try
        MyMail.LoadFromFile( Filename );

        repeat
           DataItem := ExtractParameter( Args );

           if DataItem = 'FLAGS' then
              AddDataValue( FlagMaskToString( fIndex.GetFlags(Idx) ) )
           else if DataItem = 'INTERNALDATE' then
              AddDataValue( '"' + DateTimeGMTToImapDateTime( UnixTimeToDateTime(
                            fIndex.GetTimeStamp(Idx) ), '+0000' ) + '"' )
           else if DataItem = 'UID' then
              AddDataValue( fIndex.GetUIDStr(Idx) )
           else if DataItem = 'ENVELOPE' then
              AddDataValue( MyMail.Envelope )
           else if DataItem = 'RFC822' then
              AddDataValue( MakeLiteral( MyMail.Text ) )
           else if DataItem = 'RFC822.HEADER' then
              AddDataValue( MakeLiteral( MyMail.FullHeader + #13#10 ) )
           else if DataItem = 'RFC822.TEXT' then
              AddDataValue( MakeLiteral( MyMail.FullBody ) )
           else if DataItem = 'RFC822.SIZE' then
              AddDataValue( IntToStr( Length(MyMail.Text) ) )
           else if DataItem = 'BODYSTRUCTURE' then
              AddDataValue( MyMail.BodyStructure( True ) )
           else if DataItem = 'BODY' then
              AddDataValue( MyMail.BodyStructure( False ) )
           else if Copy( DataItem, 1, 4 ) = 'BODY' then begin
              if Copy( DataItem, 5, 5 ) = '.PEEK' then
                 System.Delete( DataItem, 5, 5 )
              else if fIndex.GetFlags( Idx ) and FLAGSEEN <> FLAGSEEN then begin
                 // The \Seen flag is implicitly set; if this causes the flags to
                 // change they SHOULD be included as part of the FETCH responses.
                 fIndex.SetFlags( Idx, FLAGSEEN );
                 Args := Args + ' FLAGS';
              end;
              AddDataValue( MakeLiteral( MyMail.BodySection( DataItem ) ) )
           end else begin
              Log( LOGID_WARN, 'Unsupported Imap FETCH parameter: ''' + DataItem + '''' );
              Success := False
           end;

        until Args = '';

     finally MyMail.Free end;

     System.Delete( Data, 1, 1 );
     Result := IntToStr(Idx+1) + ' FETCH (' + Data + ')'
end;

procedure TImapMailbox.AddUser( Notify : pIMAPNotification );
//procedure TImapMailbox.AddUser( User: TSrvIMAPCli );
begin
     Log( LOGID_DEBUG, 'TImapMailbox.AddUser (current user count: '
                  + IntToStr(fUsers.Count) +')' ); {MG}{ImapLog}
     fUsers.Add( Notify );
end;

procedure TImapMailbox.RemoveUser( Notify : pIMAPNotification; out NoUsersLeft: Boolean );
//procedure TImapMailbox.RemoveUser( User: TSrvIMAPCli; out NoUsersLeft: Boolean );
var  i : Integer;
begin
     Log( LOGID_DEBUG, 'TImapMailbox.RemoveUser (current user count: '
                   + IntToStr(fUsers.Count) +')' ); {MG}{ImapLog}
     NoUsersLeft := False;
     i := fUsers.IndexOf( Notify );
     if i >= 0 then fUsers.Delete( i );
     if fUsers.Count = 0 then NoUsersLeft := True;
end;

procedure TImapMailbox.Lock;
begin
     EnterCriticalSection( fCritSection );
end;

procedure TImapMailbox.Unlock;
begin
     LeaveCriticalSection( fCritSection );
end;

constructor TImapMailbox.Create( APath: String );
begin
   inherited Create;
   Log( LOGID_DEBUG, 'TImapMailbox.Create ' + APath ); {MG}{ImapLog}
   InitializeCriticalSection( fCritSection );
   fPath   := APath;
   ForceDirectories(APath); 
   fUsers  := TList.Create;
   fIndex  := TImapMailboxIndex.Create( fPath );
   fStatus := GetStatus;
   fReadOnly := false; //ClientRO
   //ToDo: Read-Only einzelner Mailboxen kann hier gesteuert werden.
   if not FileExists2( fPath + IMAPINDEX_FILENAME ) then
      fIndex.Rebuild( fPath, fStatus );
end;

destructor TImapMailbox.Destroy;
begin
     Log( LOGID_DEBUG, 'TImapMailbox.Destroy' ); {MG}{ImapLog}
     WriteStatus;
     if Assigned(fIndex) then fIndex.Free;
     if fUsers.Count > 0 then
        Log( LOGID_ERROR, 'TImapMailbox.Destroy: imap mailbox is still in use!' );
     fUsers.Free; // TODO: User direkt aufrumen oder warnen?
     CfgAccounts.IMAPMailboxLock( fPath, False );
     DeleteCriticalSection( fCritSection );

     inherited;
end;

function TImapMailbox.StrToMsgSet(s: String; UseUID: Boolean): TMessageSet;

   function GetSet( s: String ): TMessageSet;
   var  i, j, Start, Finish: Integer;
   begin
        i := Pos( ':', s );
        if i > 0 then begin
           Start  := StrToInt( copy( s, 1, i-1 ) );
           System.Delete( s, 1, i );

           if (s = '*') or (s = '4294967295') then begin
              if UseUID then Finish := fIndex.GetUID( Status.Messages - 1 )
              else Finish := Status.Messages;
           end else Finish := StrToInt( s );
           if Finish < Start then Finish := Start;
           SetLength( Result, Finish-Start+1 );

           j := 0;
           for i := Start to Finish do begin
              if UseUID then begin
                 Result[j] := fIndex.GetIndex( i ) + 1;
                 if Result[j] = 0 then continue;
              end else Result[j] := i;
              inc( j );
           end;
           if j < High(Result) then SetLength( Result, j+1 );
           while (length(Result)>0) and  (Result[High(Result)]=0) do
             SetLength(Result, High(Result));
        end else begin
           SetLength( Result, 1 );
           if UseUID then Result[0] := fIndex.GetIndex( StrToInt(s) ) + 1
           else Result[0] := StrToInt(s);
        end;
   end;

   procedure AppendSet( var Dest: TMessageSet; MsgSet: TMessageSet );
   var  i, j: Integer;
   begin
        i := Length( Dest );
        SetLength( Dest, i + Length(MsgSet) );
        for j := 0 to High(MsgSet) do Dest[j+i] := MsgSet[j];
   end;

var  i : Integer;
begin
   SetLength( Result, 0 );
   s := TrimWhSpace( s );
   If s > '' then begin
      i := Pos( ',', s );
      while i > 0 do begin
         AppendSet( Result, GetSet( copy( s, 1, i-1 ) ) );
         Log(LOGID_DEBUG, 'IMAPMsgSet: ' + IntToStr(Result[High(Result)]) + ' + ' + s);
         System.Delete( s, 1, i );
         s := TrimWhSpace( s );
         i := Pos( ',', s );
      end;
      AppendSet( Result, GetSet( s ) )
   end
end;

end.

