unit cIMAPMailboxIndex;

interface

uses uDateTime, cChunkRc;

type
  TMbxStatus = record
    UIDvalidity : TUnixTime;  // Unique identifier of the mailbox
    UIDnext     : LongInt;    // Next UID
    Messages    : LongInt;    // All messages
    Unseen      : LongInt;    // Unseen Messages
    Recent      : LongInt;    // Recent messages
  end;

  TFlagMask = LongInt;   // Bitmask containing status flags of the message
  TMessageRec = record
    UID       : LongInt;    //    4
    TimeStamp : TUnixTime;  //    4
    Flags     : TFlagMask;  //    1
  end;                      //  = 9 Bytes


  TImapMailboxIndex = class( TChunkedRecords )
    private
      procedure GetRecord( Index: Integer; out MR: TMessageRec );

      function GetCHUNK_MAX: Integer;
      procedure SetCHUNK_MAX(const Value: Integer);
    public
      function  AddEntry( UID: LongInt; Flags: TFlagMask; TimeStamp: TUnixTime ): Boolean;
      procedure RemoveEntry( UID: LongInt );
      function  SetFlags( Index: Integer; Flags: TFlagMask ): TFlagMask;
      function  GetFlags( Index: Integer ): TFlagMask;
      function  AddFlags( Index: Integer; Flags: TFlagMask ): TFlagMask;
      function  RemoveFlags( Index: Integer; Flags: TFlagMask ): TFlagMask;
      function  GetUID( Index: Integer ): LongInt;
      function  GetUIDStr( Index: Integer ): String;
      function  GetTimeStamp( Index: Integer ): TUnixTime;
      function  GetIndex( UID: LongInt ): Integer;
      procedure Rebuild( Mailbox: String; var Status: TMbxStatus );

      //Things needed outsie of protected tChunkedRecords (cIMAPMailbox)
      property  propCHUNK_MAX : Integer read GetCHUNK_MAX write SetCHUNK_MAX;
      function  pubChunkOf( const Data ): Byte;
      procedure pubRecGet( Chunk: Byte; Index: Integer; var Data );

      constructor Create( AFilePath: String );
      destructor Destroy; override;
  end;


const
     IMAPSUBSCRIBED_FILENAME = 'Subscr.dat';
     IMAPINDEX_FILEBASE      = 'Mails';
     IMAPINDEX_FILEEXT       = '.dat';
     IMAPINDEX_FILENAME      = IMAPINDEX_FILEBASE+IMAPINDEX_FILEEXT;
     IMAPINDEX_OLDFILENAME   = 'Mailbox.dat';
     IMAPSTATUS_FILENAME     = 'Status.dat';
Const INDEXKEYLEN_UID = sizeof(LongInt);



implementation

uses
 Sysutils, cLogfile, Global, Windows, uIMAPUtils, Classes, uTools;


// ---------------------------------------------------- TImapMailboxIndex -----

procedure TImapMailboxIndex.GetRecord( Index: Integer; out MR: TMessageRec );
var  Chunk: Byte;
begin
     Chunk := ChunkOf( MR );
     Enter( Chunk );
     try RecGet( Chunk, Index, MR );
     finally Leave(Chunk) end;
end;

procedure TImapMailboxIndex.Rebuild( Mailbox: String; var Status: TMbxStatus );
var  UID        : LongInt;
     SR         : TSearchRec;
begin
     Log( LOGID_INFO, 'Rebuilding index for imap mailbox ''' + Mailbox + '''.' );

     try
        Clear;

        Status.Messages := 0;
        Status.Recent   := 0;
        Status.Unseen   := 0;

        try
          EnterCriticalSection( CS_LOCK_MAILBOX_ALL );
          if (SysUtils.FindFirst( Mailbox + '*.msg', faAnyFile, SR ) = 0) then begin
             repeat
                UID := StrToIntDef( Copy( SR.Name, 1, Length(SR.Name)-4 ), 0 );
                if UID > 0 then begin
                   inc( Status.Messages );
                   inc( Status.Recent );
                   inc( Status.Unseen );
                   if UID >= Status.UIDnext then Status.UIDnext := UID + 1;
                   AddEntry( UID, 32 {FLAGRECENT},
                             DateTimeToUnixTime(FileDateToDateTime(SR.Time)) );
                end;
             until SysUtils.FindNext( SR ) <> 0;
             SysUtils.FindClose( SR );
          end;
        finally
          LeaveCriticalSection( CS_LOCK_MAILBOX_ALL );
        end;

        Sort;
        SaveToFile;

     except
        on E:Exception do
           Log( LOGID_ERROR, 'ImapMailboxIndex.Rebuild-Exception: ' + E.Message );
     end;
end;

function TImapMailboxIndex.AddEntry( UID: LongInt; Flags: TFlagMask;
                                     TimeStamp: TUnixTime ): Boolean;
var  MR: TMessageRec;
begin
     FillChar( MR, sizeof(MR), 0 );
     // MR.UID := UID;
     MR.UID := SwitchByteOrder( UID ); {MG}{IMAP-UID-ByteOrder}

     if ContainsKey( MR, INDEXKEYLEN_UID ) then begin // unique?
        Result := False;
     end else begin
        Result := True;
        if TimeStamp = 0 then TimeStamp := DateTimeToUnixTime( NowGMT );
        MR.TimeStamp := TimeStamp;
        MR.Flags     := Flags;
        Add( MR );
     end;
     SaveToFile
end;

procedure TImapMailboxIndex.RemoveEntry( UID: LongInt );
var  MR: TMessageRec;
begin
     FillChar( MR, sizeof(MR), 0 );
     // MR.UID    := UID;
     MR.UID    := SwitchByteOrder( UID ); {MG}{IMAP-UID-ByteOrder}
     RemoveKey( MR, INDEXKEYLEN_UID );
end;

function TImapMailboxIndex.SetFlags(Index: Integer; Flags: TFlagMask): TFlagMask;
var  MR: TMessageRec;
     Chunk: Byte;
begin
     FillChar( MR, sizeof(MR), 0 );
     Chunk := ChunkOf( MR );
     Enter( Chunk );
     try
        RecGet( Chunk, Index, MR );
        MR.Flags := Flags;
        RecSet( Chunk, Index, MR );
        Result := MR.Flags;
     finally Leave(Chunk) end
end;

function TImapMailboxIndex.GetFlags( Index: Integer ): TFlagMask;
var  MR: TMessageRec;
begin
     FillChar( MR, sizeof(MR), 0 );
     GetRecord( Index, MR );
     Result := MR.Flags
end;

function TImapMailboxIndex.AddFlags(Index: Integer; Flags: TFlagMask): TFlagMask;
begin
     Result := SetFlags( Index, GetFlags(Index) or Flags )
end;

function TImapMailboxIndex.RemoveFlags(Index: Integer; Flags: TFlagMask): TFlagMask;
begin
     Result := SetFlags( Index, GetFlags(Index) and not Flags )
end;

function TImapMailboxIndex.GetUID( Index: Integer ): LongInt;
var  MR: TMessageRec;
begin
     FillChar( MR, sizeof(MR), 0 );
     GetRecord( Index, MR );
     // Result := MR.UID
     Result := SwitchByteOrder( MR.UID ) {MG}{IMAP-UID-ByteOrder}
end;

function TImapMailboxIndex.GetUIDStr( Index: Integer ): String;
begin
     Result := IntToStr( GetUID( Index ) )
end;

function TImapMailboxIndex.GetTimeStamp( Index: Integer ): TUnixTime;
var  MR: TMessageRec;
begin
     FillChar( MR, sizeof(MR), 0 );
     GetRecord( Index, MR );
     Result := MR.TimeStamp
end;

function TImapMailboxIndex.GetIndex(UID: LongInt): Integer;
var  MR: TMessageRec;
     Chunk: Byte;
begin
     FillChar( MR, sizeof(MR), 0 );
     // MR.UID := UID;
     MR.UID := SwitchByteOrder( UID ); {MG}{IMAP-UID-ByteOrder}
     Chunk := ChunkOf( MR );
     Enter( Chunk );
     try
        Result := RecKeyIndexOf( Chunk, INDEXKEYLEN_UID, MR );
     finally Leave(Chunk) end;
end;

Procedure Convert (Const OldFile, NewFile: String);
Type
  TEntry = packed Record TurnIt: Array[1..4] of Byte; ok: Array[1..8] of Byte end;
Var
  Entry1, Entry2: TEntry;
  Source, Dest: TFileStream;
  i, j, Anz: Integer;
begin
   Source := NIL; Dest := NIL;
   try
      Source := TFileStream.Create(OldFile, fmOpenRead);
      Dest := TFileStream.Create(NewFile, fmCreate);
      Anz := Source.Size div SizeOf(TEntry);
      For i := 1 to Anz do begin
         Source.ReadBuffer(Entry1, SizeOf(TEntry));
         Entry2 := Entry1;
         For j := 1 to 4 do Entry2.TurnIt[5-j] := Entry1.TurnIt[j];
         Dest.WriteBuffer(Entry2, SizeOf(TEntry))
      end;
   finally
      Source.Free;
      Dest.Free
   end;
   // DeleteFile(OldFile)
end;

constructor TImapMailboxIndex.Create( AFilePath: String );
begin
   if AFilePath='' then AFilePath:='.\';
   AFilePath := IncludeTrailingBackslash(ExpandFilename( AFilePath ));
   If Not FileExists2( AFilePath + IMAPINDEX_FILENAME ) and
      FileExists2( AFilePath + IMAPINDEX_OLDFILENAME )
   then begin
      Convert ( AFilePath + IMAPINDEX_OLDFILENAME, AFilePath + IMAPINDEX_FILENAME)
   end;
   inherited Create( AFilePath + IMAPINDEX_FILEBASE, IMAPINDEX_FILEEXT,
                     0, sizeof(TMessageRec), True );
   LoadFromFile
end;

destructor TImapMailboxIndex.Destroy;
begin
     SaveToFile;
     inherited;
end;


function TImapMailboxIndex.pubChunkOf(const Data): Byte;
begin
  Result := ChunkOf(Data)
end;

procedure TImapMailboxIndex.pubRecGet(Chunk: Byte; Index: Integer; var Data);
begin
  RecGet(Chunk, Index, Data)
end;

function TImapMailboxIndex.GetCHUNK_MAX: Integer;
begin
  Result := CHUNK_MAX
end;

procedure TImapMailboxIndex.SetCHUNK_MAX(const Value: Integer);
begin
  CHUNK_MAX := Value
end;

end.

