unit cLogfile;

interface

uses Windows, SysUtils, Classes, FileCtrl;

const
   LOGID_ERROR     = $8000;
   LOGID_WARN      = $4000;
   LOGID_SYSTEM    = $0800;
   LOGID_INFO      = $0080;
   LOGID_DETAIL    = $0040;
   LOGID_DEBUG     = $0008;
   LOGID_STATUS    = $0004;
   LOGID_FULL      = $0001;

function LogIdToMarker( const ID: Integer ): String; //HSR //Logfile

type
   TLogFile = class
      // {E/L} = not thread safe, i.e. .Enter/.Leave required while using
      private
         FLock: TRTLCriticalSection;
         FStrm: TFileStream;
         FViewBuffer: TStringList;
         FTaskBuffer: TStringList;
         FFileBuffer: TStringList;
         FLogPath: String;
         FViewMax, FFileMax: Integer;
         FViewMask, FFileMask, FTaskMask: Integer;
         FTaskChanges: Integer;
         FLastRotateLog: TDateTime;
         FCached, FFirst: boolean;
         FFormatLogfileName: String;
         FSkip_ConectionLost: boolean;  // JAWO 26.01.02
         procedure DoOpen;                   {E/L}
         procedure DoClose;                  {E/L}
         procedure DoAppend( Line: String ); {E/L}
         procedure DoRotateLog;              {E/L}
         procedure SetCached(const Value: boolean); {E/L}

         procedure SetInt( Index: Integer; NewValue: Integer );
         procedure SetStr( Index: Integer; NewValue: String  );

         function GetTaskChanged: Boolean;
         function GetTaskCount: Integer;                  {E/L}
         function GetTaskLine( Index: Integer ): String;  {E/L}
         function GetViewCount: Integer;                  {E/L}
         function GetViewLine( Index: Integer ): String;  {E/L}
         function GetViewType( Index: Integer ): Integer; {E/L}
         Procedure DoAdd( const ID: Integer; const OrgMsg: String; Const ThreadID: LongWord );
      public
         property  ViewMax : Integer index 0 read FViewMax  write SetInt;
         property  FileMax : Integer index 1 read FFileMax  write SetInt;
         property  ViewMask: Integer index 2 read FViewMask write SetInt;
         property  FileMask: Integer index 3 read FFileMask write SetInt;
         property  TaskMask: Integer index 4 read FTaskMask write SetInt;

         property  LogPath: String index 0 read FLogPath write SetStr;
         function  Logfilename(const Nr: Integer): String;

         property  Cached: boolean read FCached Write SetCached;

         property  Skip_ConectionLost: boolean read FSkip_ConectionLost write FSkip_ConectionLost;  // JAWO 26.01.02

         property  TaskChanged: Boolean read GetTaskChanged;
         property  TaskCount: Integer read GetTaskCount;               {E/L}
         property  TaskLine[Index: Integer]: String read GetTaskLine;  {E/L}

         property  ViewCount: Integer read GetViewCount;               {E/L}
         property  ViewLine[Index: Integer]: String  read GetViewLine; {E/L}
         property  ViewType[Index: Integer]: Integer read GetViewType; {E/L}
         procedure ViewClear;

         procedure Enter;
         procedure Leave;

         procedure SetTask( const TaskIdent, Msg: String );
         procedure RemoveTask( const TaskIdent: String );
         procedure Add( const ID: Integer; const OrgMsg: String );
         procedure AddMain( const ID: Integer; const OrgMsg: String );
         function  LastLines( const MaxKByte: Integer ): String;
         procedure RotateLog;
         procedure Cancel;

         procedure LoadINISettings;

         constructor Create( ALogPath: String );
         destructor Destroy; override;
   end;

Function LogFile: TLogFile;

procedure MainLog( const ID: Integer; const OrgMsg: String );
procedure Log( const ID: Integer; const OrgMsg: String );

implementation

Uses Config, Global, uTools;

Var FLog: TLogfile;

Function LogFile: TLogFile;
begin
   If Not Assigned(FLog) then FLog := TLogFile.Create( PATH_LOGS );
   Result := FLog
end;

Procedure MainLog( const ID: Integer; const OrgMsg: String );
begin
   LogFile.AddMain( ID, OrgMsg );
end;

procedure Log( const ID: Integer; const OrgMsg: String );
begin
   LogFile.Add( ID, OrgMsg );
end;

{ TLog }

constructor TLogFile.Create(ALogPath: String);
begin
   inherited Create;
   InitializeCriticalSection( FLock );
   FLastRotateLog := 0;
   FStrm    := nil;
   FLogPath := ALogPath;
   FViewBuffer   := TStringList.Create;
   FTaskBuffer   := TStringList.Create;
   FFileBuffer   := TStringList.Create;
   FFormatLogfileName := '';
   FTaskChanges  := 0;
   FSkip_ConectionLost := false;  // JAWO 26.01.02
   FViewMax  := 500;
   FFileMax  := 14;
   FViewMask := LOGID_ERROR or LOGID_WARN or LOGID_SYSTEM or LOGID_INFO;
   FFileMask := LOGID_ERROR or LOGID_WARN or LOGID_SYSTEM or LOGID_INFO or LOGID_DETAIL;
   FTaskMask := LOGID_ERROR or LOGID_WARN or LOGID_SYSTEM or LOGID_INFO or LOGID_STATUS;
   LoadINISettings;
   FFirst := true
end;

Procedure TLogFile.LoadINISettings;
begin
   With CfgIni do begin
      TaskMask := strtoint( '$' + ReadString ( 'Setup', 'log.thread.mask',IntToHex(TaskMask,4) ) );{kms}
      ViewMask := strtoint( '$' + ReadString ( 'Setup', 'log.view.mask', inttohex( ViewMask, 4 ) ) );
      FileMask := strtoint( '$' + ReadString ( 'Setup', 'log.file.mask', inttohex( FileMask, 4 ) ) );
      Skip_ConectionLost := Readbool('Setup', 'log.IgnoreConnectionLost', Def_IgnoreConnectionLost);  // JAWO 26.01.02
      FFormatLogfileName := ReadString('Setup', 'log.FormatFileName', FFormatLogfileName);
      try Format(FFormatLogfileName, [0]) except FFormatLogfileName := '' end;
      If ArchivMode then FileMask := 0;
      FileMax  := ReadInteger( 'Setup', 'log.file.max',  FileMax );
      ViewMax := ReadInteger( 'Setup', 'log.view.max',  ViewMax );
      If ViewMax < 10 then ViewMax :=  10;
      if ViewMax > 9999 then ViewMax := 9999
   end
end;

destructor TLogFile.Destroy;
begin
   Enter;
   try
      Add( LOGID_DETAIL, 'Closing log file.' );
      FViewBuffer.Free;
      FTaskBuffer.Free;
      FFileBuffer.Free;
      if Assigned(FStrm) then FStrm.Free;
   except
   end;
   Leave;
   DeleteCriticalSection( FLock );
   inherited;
end;

Function TLogFile.Logfilename(Const Nr: Integer): String;
begin
   If FFormatLogfileName > ''
      then Result := FLogPath + Format(FFormatLogfileName, [Nr])
      else Result := FLogPath + IntToStr(Nr) + '.log'
end;

procedure TLogFile.DoOpen;
begin
   if Assigned( FStrm ) then exit;

   if not FileExists2( Logfilename(0) ) then begin
      try
         FStrm := TFileStream.Create( Logfilename(0), fmCreate or fmShareExclusive );
         FreeAndNil( FStrm );
      except
         FStrm := nil;
      end;
   end;

   try
      FStrm := TFileStream.Create( Logfilename(0), fmOpenReadWrite or fmShareDenyNone );
   except
      FStrm := nil;
   end;
end;

procedure TLogFile.DoClose;
begin
   if Assigned( FStrm ) then FreeAndNil( FStrm );
end;

procedure TLogFile.DoAppend(Line: String);
Var l: Integer;
begin
   l := Length(Line);
   If l = 0 then Exit;
   If FCached then begin
      FFileBuffer.Add (Line)
   end else begin
      If FFirst then begin
         FFirst := false;
         If FLastRotateLog = 0 then DoRotateLog
      end;
      DoOpen;
      if not Assigned( FStrm ) then exit;
      try
         FStrm.Seek( 0, soFromEnd );
         FStrm.Write( Line[1], length(Line) );
         If (l<2) or ((Line[l-1]<>#13) and (Line[l]<>#10)) then FStrm.Write( #13#10, 2 )
      except
      end;
      DoClose
   end
end;
procedure TLogFile.SetCached(const Value: boolean);
begin
  If Value <> FCached then begin
     FCached := Value;
     If (Not Value) and (FFileBuffer.Count>0) then begin
        DoAppend ( FFileBuffer.Text );
        FFileBuffer.Clear
     end
  end
end;

procedure TLogFile.DoRotateLog;
Var i, Missed: Integer;
begin
   try
      FLastRotateLog := Now;
      i := FileMax-1;
      Missed := 0;
      Repeat
         If FileExists2 ( Logfilename(i) ) then begin
            DeleteFile( Logfilename(i) );
            Missed := 0
         end else begin
            Inc(Missed)
         end;
         Inc(i)
      until Missed > 2;
      for i:=FileMax-1-1 downto 0 do begin
          RenameFile( Logfilename(i),
                      Logfilename(i+1) );
      end;
   except
   end;
end;

function TLogFile.LastLines(const MaxKByte: Integer): String;
var  p, l: Integer;
begin
   Result := '';
   
   Enter;
   try
      DoOpen;
      if not Assigned( FStrm ) then exit;

      try
         p := FStrm.Seek( 0, soFromEnd ) - ( MaxKByte shl 10 );
         if p < 0 then begin
            p := 0;
            l := FStrm.Size;
         end else begin
            l := MaxKByte shl 10;
         end;

         SetLength( Result, l );
         FStrm.Seek( p, soFromBeginning );
         if l>0 then FStrm.Read( Result[1], l );

         if p > 0 then begin
            p := Pos( #10, Result );
            if p > 0 then System.Delete( Result, 1, p );
         end;

      finally
         DoClose;
      end;

   finally
      Leave;
   end;
end;

procedure TLogFile.AddMain (const ID: Integer; const OrgMsg: String);
begin
   DoAdd ( ID, OrgMsg, MainThreadID )
end;

procedure TLogFile.Add(const ID: Integer; const OrgMsg: String);
begin
   DoAdd ( ID, OrgMsg, GetCurrentThreadID )
end;

function LogIdToMarker( const ID: Integer ): String; //HSR //Logfile
begin
   case ID of
      LOGID_ERROR : Result := 'ERR';
      LOGID_WARN  : Result := 'WAR';
      LOGID_SYSTEM: Result := 'Sys';
      LOGID_INFO  : Result := 'I  ';
      LOGID_DETAIL: Result := 'D  ';
      LOGID_DEBUG : Result := 'd  ';
      LOGID_STATUS: Result := 's  ';
//         LOGID_FULL  : Result := 'f  ';
      else          Result := '???';
   end
end;

procedure TLogFile.DoAdd(const ID: Integer; const OrgMsg: String; Const ThreadID: LongWord);
var  ThdID, Msg, Timestamp: String;
begin
     if ( ID and (ViewMask or FileMask or TaskMask) ) = 0 then exit;

     ThdID := lowercase(IntToHex(ThreadID,1));
     Timestamp := FormatDateTime( 'yyyy/mm/dd hh:nn:ss', Now );
     Msg   := Timestamp
            + ' ' + '{' + ThdID + '}'
            + ' ' + OrgMsg;

     if (ID and TaskMask) > 0 then begin
        SetTask( ThdID, OrgMsg );
     end;

     if (ID and ViewMask) > 0 then begin
        Enter;
        try
           If Length(Msg) > 1025
              then FViewBuffer.AddObject( Copy(Msg, 1, 1020)+'...', Pointer(ID) )
              else FViewBuffer.AddObject( Msg, Pointer(ID) );
           if FViewBuffer.Count>ViewMax then begin
              while FViewBuffer.Count > ViewMax-10 do FViewBuffer.Delete(0);
           end;
        finally
           Leave;
        end;
     end;

     if (ID and FileMask) > 0 then begin
        Enter;
        try
           If Not fCached then begin
              if Trunc(Now) <> Trunc(FLastRotateLog) then DoRotateLog
           end;
           DoAppend( Timestamp
                     + ' ' + LogIdToMarker(ID)
                     + ' {' + ThdID + '} '
                     + StringReplace(OrgMsg, #10, #10#9, [rfReplaceAll] ) )
        finally
           Leave;
        end;
     end;
end;

procedure TLogFile.RemoveTask(const TaskIdent: String);
var  i : Integer;
begin
   Enter;
   try
      i := FTaskBuffer.IndexOfName( TaskIdent );
      if i>=0 then FTaskBuffer.Delete( i );
      InterlockedExchange( FTaskChanges, 1 )
   finally
      Leave;
   end;
end;

procedure TLogFile.SetTask(const TaskIdent, Msg: String);
begin
   Enter;
   try
      FTaskBuffer.Values[ TaskIdent ] := Msg;
      InterlockedExchange( FTaskChanges, 1 )
   finally
      Leave;
   end;
end;

procedure TLogFile.SetInt(Index, NewValue: Integer);
begin
   Enter;
   try
      case Index of
         0: FViewMax  := NewValue;
         1: FFileMax  := NewValue;
         2: FViewMask := NewValue;
         3: FFileMask := NewValue;
         4: FTaskMask := NewValue;
      end;
   finally
      Leave;
   end;
end;

procedure TLogFile.SetStr(Index: Integer; NewValue: String);
begin
   Enter;
   try
      case Index of
         0: FLogPath := NewValue;
      end;
   finally
      Leave;
   end;
end;

procedure TLogFile.Enter;
begin
   EnterCriticalSection( FLock );
end;

procedure TLogFile.Leave;
begin
   LeaveCriticalSection( FLock );
end;

function TLogFile.GetTaskCount: Integer;
begin
   Result := FTaskBuffer.Count;
end;

function TLogFile.GetTaskLine(Index: Integer): String;
begin
   Result := FTaskBuffer[Index];
end;

function TLogFile.GetViewCount: Integer;
begin
   Result := FViewBuffer.Count;
end;

function TLogFile.GetViewLine(Index: Integer): String;
begin
   Result := FViewBuffer[Index];
end;

function TLogFile.GetViewType(Index: Integer): Integer;
begin
   Result := Integer( FViewBuffer.Objects[Index] );
end;

procedure TLogFile.ViewClear;
begin
   Enter;
   try
      FViewBuffer.Clear;
   finally
      Leave;
   end;
end;

function TLogFile.GetTaskChanged: Boolean;
begin
   Result := ( InterlockedExchange( FTaskChanges, 0 ) <> 0 );
end;

procedure TLogFile.RotateLog;
begin
   Enter;
   try
      DoRotateLog;
   finally
      Leave;
   end;
end;

procedure TLogFile.Cancel;
begin
   FFileBuffer.Clear;
end;

initialization
   FLog := nil;

finalization

   if Assigned( FLog ) then begin
      FLog.Cached := false;
      FLog.DoClose;
      FreeAndNil( FLog )
   end;

end.

