unit cMailAlias;
//Joern Weber Mai 2001
interface

uses
  classes, sysutils;

type
 tMailAlias = class
 private
   slKey, slValue: TStringList;
   procedure   AddToLog(ID:Word; Msg:String);
   procedure   Load;
   function    CheckAlias(const OriginAddress: String; Const LogInfo: boolean): string;
   Procedure   CheckDefaultEntries;
   Function    Search (Const Key: String; Var Value: String): Boolean;
 public
   constructor Create;
   procedure   Reload;
   Procedure   Deactivate (Const Entry: String);
   Function    Resolve (Const OrgAdr: String; Const LogInfo: Boolean = true): String;
   Function    Exists(const Nam: String; Var Entry: String): Boolean;
   Procedure   GetAliasesForAccount (Const Account: String; sl: TStrings);
   Procedure   LoadFromFile (sl: TStrings);
   Procedure   SaveIntoFile (sl: TStrings);
   destructor  Destroy; override;
 end;

Function MailAlias : tMailalias;

implementation

Uses uTools, config, cAccount, global, cLogfile, cStdForm;

Const FQDNWildCard = '%FQDN%';

Var fMailalias: tMailalias = NIL;

Function MailAlias : tMailalias;
begin
   If Not Assigned(fMailalias)
      then fMailalias := TMailalias.Create;
   Result := fMailalias
end;

function tMailAlias.CheckAlias(Const OriginAddress: String; Const LogInfo: boolean): string;
Var s : string;
begin
   If Search ( OriginAddress, s ) then begin
      Result := s;
      If LogInfo then AddToLog(LOGID_DETAIL,TrGlF(kLog, 'mailalias.aliasfoundforaddress',
         'Alias "%s" found for address %s', [s, OriginAddress]));
   end else begin
      Result := OriginAddress;
      If LogInfo then AddToLog(LOGID_DETAIL,TrGlF(kLog, 'mailalias.noaliasfoundforaddress',
         'No Alias found for address %s', OriginAddress ) );
   end;
{/JW}
end;

procedure tMailAlias.AddToLog(ID:Word; Msg:String);
begin
  Log(Id, TrGlF(kLog, 'mailalias: xxx', 'Mail alias: %s', Msg ) );
end;

Procedure tMailAlias.CheckDefaultEntries;
Var sl: TStringList; changed, NewFile: boolean;

  Procedure Test (Const Alias, Dest: String);
  var UserID, LocalType, i: integer; bInsert, bLocal: Boolean; s: String;
  begin
    bInsert := Not ( CfgAccounts.IsLocalMailbox(Alias, UserID, LocalType, bLocal)
                     and (LocalType=LOCALMAILTYPE_NORMAL) );
    If bInsert then begin
       For i := 0 to sl.Count-1 do begin
          If Copy(sl[i], 1, Length(Alias)+1) = Alias+'=' then begin
             bInsert := false;
             break
          end
       end;
       For i := 0 to sl.Count-1 do begin
          If Copy(sl[i], 1, 2+Length(Alias)+1) = '# '+Alias+'=' then begin
             bInsert := false;
             s := Copy(sl[i], 3, Length(sl[i])-2);
             sl[i] := s;
             If Not NewFile then AddToLog(LOGID_WARN, TrGlF(kLog, 'mailalias.recreatedefault',
                'recreate default mailalias "%s"', s ));
             changed := true;
             break
          end
       end
    end;
    If bInsert then begin
       s := Alias + '=' + Dest;
       sl.Add (s);
       If Not NewFile then AddToLog(LOGID_WARN, TrGlF(kLog, 'mailalias.recreatedefault',
          'recreate default mailalias "%s"', s ));
       changed := true
    end
  end;

begin
   sl := TStringList.Create;
   try
      changed := false;
      NewFile := Not FileExists2(PATH_BASE + CFGFILE_ALIAS);
      If NewFile
         then changed := true
         else sl.LoadFromFile(PATH_BASE + CFGFILE_ALIAS);
      Test ( 'news', 'admin' );
      Test ( 'abuse', 'admin' );
      Test ( 'usenet', 'admin' );
      Test ( 'postmaster', 'admin' );
      Test ( 'news@'+FQDNWildCard, 'news' );
      Test ( 'abuse@'+FQDNWildCard, 'abuse' );
      Test ( 'usenet@'+FQDNWildCard, 'usenet' );
      Test ( 'postmaster@'+FQDNWildCard, 'postmaster' );
      Test ( 'admin@'+FQDNWildCard, 'admin' );
      Test ( 'local-hamster-info@'+FQDNWildCard, 'admin' );
      Test ( 'local-smtp@'+FQDNWildCard, 'admin' );
      If changed then sl.SaveToFile(PATH_BASE + CFGFILE_ALIAS)
   finally
      sl.free
   end
end;

Function tMailAlias.Resolve (Const OrgAdr: String; Const LogInfo: Boolean = true): String;
Var Found: Boolean; Iterations: Integer; s: String;
begin
   Result := LowerCase(OrgAdr);
   Iterations := 0;
   Repeat
     Found := true;
     s:= LowerCase(CheckAlias(Result, LogInfo));
     if s <> Result
        then begin Result := s; Inc(Iterations) end
        else found := false
   until (Not found) or (Iterations > 100);
   If Iterations > 100
      then AddToLog(LOGID_WARN, TrGlf(kLog, 'mailalias.crossedaliases',
         'Some aliases must be crossed, because "%s" isn''t resolved after 100 steps!',
         OrgAdr ) )
end;

procedure tMailAlias.Load;

   Function ExtFQDN(Const s: String): String;
   Var p: Integer;
   begin
      p := Pos(FQDNWildcard, s);
      If p > 0 then begin
         If Def_FQDN > '' then begin
            Result := s;
            Delete (Result, p, Length(FQDNWildcard));
            Insert (Def_FQDN, Result, p)
         end else begin
            Result := ''
         end
      end else begin
         Result := s
      end
   end;

Var i, p, UserID, LocalType: Integer; s, Key, Value: String; bLocal: Boolean;
begin
   CheckDefaultEntries;
   slKey.Clear; slKey.sorted := true;
   slValue.Clear; slValue.sorted := false;
   With TStringList.Create do try
      LoadFromFile(PATH_BASE + CFGFILE_ALIAS);
      For i := 0 to Count-1 do begin
         s := Trim(Strings[i]);
         If (s > '') and (Not (s[1] IN['#', ';'])) then begin
            p := Pos('=', s);
            If p > 0 then begin
               Key := Trim(ExtFQDN(Copy(s, 1, p-1)));
               Value := FilterEmailOfFrom(Trim(ExtFQDN(Copy(s, p+1, Length(s)-p))));
               If (Key > '') and (Value > '') then begin
                  slKey.AddObject ( Key, Pointer(slValue.Add(Value)) )
               end
            end else begin
               AddToLog(LOGID_WARN, TrGlF(kLog, 'mailalias.novalidentry',
                  '"%s" is no valid entry in malias.hst!', s))
            end
         end
      end
   finally
      free
   end;
   // Check integrity
   For i := 0 to slKey.Count-1 do begin
      Key := slKey[i];
      Value := slValue[Longint(slKey.Objects[i])];
      s := Resolve(Value, false);
      If Not (CfgAccounts.IsLocalMailbox(s, UserID, LocalType, bLocal)
          and (LocalType=LOCALMAILTYPE_NORMAL)) and bLocal
      then begin
         AddToLog(LOGID_ERROR, TrGlF(kLog, 'mailalias.novalidentry.unabletoresolve',
           '"%s=%s" is no valid entry, because it can''t be resolved to an existing user!',
           [key, Value] ))
      end;
      If CfgAccounts.IsLocalMailbox(Key, UserID, LocalType, bLocal)
         and (LocalType=LOCALMAILTYPE_NORMAL)
         and (Resolve(Key, false) <> Key)
      then begin
         AddToLog(LOGID_WARN, TrGlF(kLog, 'mailalias.entryhidesuser',
            'Entry "%s" hides user "%s".', [Value, Key]))
      end
   end
end;

procedure tMailAlias.Reload;
begin
  AddToLog(LOGID_INFO,TrGlF(kLog, 'mailalias.reload', 'Reload alias file with %s',
     PATH_BASE + CFGFILE_ALIAS));
  Load
end;


constructor tMailAlias.Create;
begin
  inherited Create;
  slKey := TStringList.Create;
  slValue := TStringList.Create;
  Load
end;

destructor tMailAlias.Destroy;
begin
  slKey.Free; slValue.Free;
  inherited Destroy;
end;

procedure tMailAlias.LoadFromFile(sl: TStrings);
begin
   CheckDefaultEntries;
   sl.LoadFromFile(PATH_BASE + CFGFILE_ALIAS);
end;

procedure tMailAlias.SaveIntoFile(sl: TStrings);
begin
   sl.SaveToFile(PATH_BASE + CFGFILE_ALIAS);
   Reload
end;

function tMailAlias.Exists(const Nam: String; Var Entry: String): Boolean;
begin
   Result := Search (Nam, Entry)
end;

procedure tMailAlias.Deactivate(const Entry: String);
Var sl: TStringList; i: Integer; Done: Boolean;
begin
   sl := TStringList.Create;
   try
      LoadFromFile (sl);
      Done := false;
      For i := 0 to sl.Count-1 do begin
         If LowerCase(sl[i]) = LowerCase(Entry) then begin
            sl[i] := '# '+sl[i]; Done := true
         end
      end;
      If Done then begin
         SaveIntoFile (sl);
         Reload
      end
   finally sl.Free end
end;

function tMailAlias.Search(const Key: String; var Value: String): Boolean;
Var i: Integer;
begin
   i := slKey.IndexOf(Key);
   Result := i >= 0;
   If Result then Value := slValue[Longint(slKey.Objects[i])]
             else Value := '' 
end;

procedure tMailAlias.GetAliasesForAccount(const Account: String; sl: TStrings);
Var i, j: integer; temAcc : string;
begin
   temAcc := lowercase(Account);
   sl.Clear;
   for i := 0 to slValue.count-1 do begin
      if lowercase(slValue[i])=temAcc then begin
         For j := 0 to slKey.Count-1 do begin
            If Integer(slKey.Objects[j]) = i then begin
               sl.Add(slKey[j])
            end
         end
      end
   end
end;

initialization
finalization
   If Assigned(fMailAlias) then fMailalias.Free;
end.


