// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cNewsJobs; // Job-based handling of news-transfers

// ----------------------------------------------------------------------------
// Container which handles and controls all news transfers based on a list of
// jobs.
// ----------------------------------------------------------------------------

interface

uses SysUtils, Classes, Windows;

Type
   TJobType = ( JOBTYPE_INVALID,
                JOBTYPE_SRVINFOS,
                JOBTYPE_NEWSPOST,
                JOBTYPE_GETBYMID,
                JOBTYPE_NEWSPULL,
                JOBTYPE_NEWSFEED );
const
   JOBPRIO_SRVINFOS = MaxInt - 1;
   JOBPRIO_NEWSPOST = MaxInt - 2;
   JOBPRIO_GETBYMID = MaxInt - 3;
   JOBPRIO_NEWSFEED = MaxInt - 4;

   {JW}
type
   TNewsJobDef = Record
        ID: Integer;
        Server: String;
        Typ: TJobType;
        Par: String;
        Prio: Integer
   end;
   TNewsJob = class
      private
         FDef: TNewsJobDef;
      public
         property JobServer  : String  read FDef.Server;
         property JobType    : TJobType read FDef.Typ;
         property JobPar     : String  read FDef.Par;
         property JobPriority: Integer read FDef.Prio write FDef.Prio;
         constructor Create( Const ID: Integer;
                             const AJobServer  : String;
                             const AJobType    : TJobType;
                             const AJobPar     : String;
                             const AJobPriority: Integer );
   end;

   TNewsJobDefs = Array of TNewsJobDef;

   TNewsJobList = class
      private
         FLock   : TRTLCriticalSection;
         FJobList: TList;
         FResort : Boolean;
         AktID: Integer;
         function  GetCount: Integer;
      public
         {JW}
         function GetServer( Index: Integer ): String;
         function GetPar( Index: Integer ): String;
         function GetType( Index: Integer ): Integer;
         function GetPriority( Index: Integer ): Integer;
         function GetCounter(): Integer;
         Procedure GetList(Var Defs: TNewsJobDefs);
         {JW}
         {JW} {Prio}
         function SetPriority( Index,NewPriority: Integer ): Integer;
         function Delete( Index: Integer ): Integer;
         {JW}
         property  Count: Integer read GetCount;
         procedure Clear;
         procedure Sort;
         procedure Enter;
         procedure Leave;
         function  JobIndexID ( Const ID          : Integer): Integer;
         function  JobIndexSTP( const AJobServer  : String;
                                const AJobType    : TJobType;
                                const AJobPar     : String  ): Integer;
         function  JobIndexST ( const AJobServer  : String;
                                const AJobType    : TJobType ): Integer;
         function  JobIndexTP ( const AJobType    : TJobType;
                                const AJobPar     : String  ): Integer;
         procedure JobSet     ( const AJobServer  : String;
                                const AJobType    : TJobType;
                                const AJobPar     : String;
                                const AJobPriority: Integer );
         function  JobGet     ( const AJobServer  : String;
                                var   AJobType    : TJobType;
                                var   AJobPar     : String  ): Boolean;
         procedure JobDelay   ( const AJobServer  : String;
                                const AJobType    : TJobType;
                                const AJobPar     : String;
                                JobsToSkip        : Integer );
         function MaxThreadsFor( AJobServer: String ): Integer;
         constructor Create;
         destructor Destroy; override;
   end;

   TNewsJobs = class
      private
         FJobList   : TNewsJobList;
         FGroupPrios: TStringList;
         function  GetGroupPrio( Groupname: String ): Integer;
         procedure SetGroupPrio( Groupname: String; NewPriority: Integer );
      public
         function GetServer(Index: Integer ): String;
         function GetPar(Index: Integer ): String;
         function GetType(Index: Integer ): Integer;
         function GetPriority(Index: Integer ): Integer;
         function GetCounter(): Integer;
         function SetPriority(Index,NewPriority: Integer ): Integer;
         function Delete(Index: Integer ): Integer;
         function Add(Servername:String;Typ:TJobType;Param:String;Priority:Integer): Integer;
         property JobList: TNewsJobList read FJobList;
         property GroupPriority[ Groupname:String ]: Integer read  GetGroupPrio
                                                             write SetGroupPrio;
         function MaxThreadsFor( Servername: String ): Integer;

         function Clear: Integer;
         function CheckInfo( ServerList: String ): Integer;
         function AddPullSrv( Servername, reGrpSelect: String ): Integer;
         function AddFeedSrv( Servername, reGrpSelect: String ): Integer;
         function AddPostSrv( Servername, reGrpSelect, reMsgSelect: String ): Integer;
         function AddPullDef( ServerList: String ): Integer;
         function AddPostDef( ServerList: String ): Integer;
         function StartThreads( ServerList: String ): Integer;

         constructor Create;
         destructor Destroy; override;
   end;

Function JobInfo (Const Typ: TJobType; Const Par: String): String;

// ----------------------------------------------------------------------------

implementation

uses Global, Config, cPCRE, cArticle, uTools, tTransfer, cStdForm, cLogFile;

function IsServerSelected( ServerList, TestSrv, TestPort: String ): Boolean;
begin
   if ServerList='' then begin
      Result := True;
   end else begin
      Result := Pos( ';' + LowerCase(TestSrv+','+TestPort) + ';',
                     ';' + LowerCase(ServerList) + ';' ) > 0;
   end;
end;

// ------------------------------------------------------------- TNewsJob -----




constructor TNewsJob.Create( Const ID: Integer;
                             const AJobServer  : String;
                             const AJobType    : TJobType;
                             const AJobPar     : String;
                             const AJobPriority: Integer );
begin
   inherited Create;
   FDef.ID       := ID;
   FDef.Server   := AJobServer;
   FDef.Typ      := AJobType;
   FDef.Par      := AJobPar;
   FDef.Prio     := AJobPriority;
end;


// --------------------------------------------------------- TNewsJobList -----

{JW}
function TNewsJobList.GetServer( Index: Integer ): String;
begin
   with TNewsJob( FJobList[Index] ) do begin
      Result:=JobServer;
   end;
end;
function TNewsJobList.GetPar( Index: Integer ): String;
begin
   with TNewsJob( FJobList[Index] ) do begin
      Result:=JobPar;
   end;
end;
function TNewsJobList.GetType( Index: Integer ): Integer;
begin
   with TNewsJob( FJobList[Index] ) do begin
      Result:=Ord(JobType);
   end;
end;
function TNewsJobList.GetPriority( Index: Integer ): Integer;
begin
   with TNewsJob( FJobList[Index] ) do begin
      Result:=JobPriority;
   end;
end;
function TNewsJobList.GetCounter(): Integer;
begin
      Result:=FJobList.count;
end;
{JW}
{JW} {Prio}
function TNewsJobList.SetPriority( Index,NewPriority: Integer ): Integer;
begin
   if assigned( FJobList[Index]) then begin
      with TNewsJob( FJobList[Index] ) do begin
         JobPriority:=NewPriority;
         result:=0;
      end;
      FResort := true
   end else result:=1;
end;

function TNewsJobList.Delete( Index: Integer ): Integer;
begin
   if assigned( FJobList[Index]) then begin
      TNewsJob( FJobList[Index] ).Free;
      FJobList.Delete( Index );
      result:=0;
   end
   else result:=1;
end;

{JW}
procedure TNewsJobList.Enter;
begin
   EnterCriticalSection( FLock );
end;

procedure TNewsJobList.Leave;
begin
   LeaveCriticalSection( FLock );
end;

function TNewsJobList.GetCount: Integer;
begin
   Enter;
   try
      Result := FJobList.Count;
   finally
      Leave;
   end;
end;

procedure TNewsJobList.Clear;
begin
   Enter;
   try
      while FJobList.Count>0 do begin
         TNewsJob( FJobList[FJobList.Count-1] ).Free;
         FJobList.Delete( FJobList.Count-1 );
      end;
      FResort := False;
   finally
      Leave;
   end;
end;

function  TNewsJobList.JobIndexID ( Const ID: Integer): Integer;
var  Index: Integer;
begin
   Result := -1;
   Enter;
   try
      for Index:=0 to FJobList.Count-1 do
         with TNewsJob( FJobList[Index] ) do If FDef.ID = ID then begin
            Result := Index;
            break;
         end;
   finally
      Leave;
   end;
end;

function TNewsJobList.JobIndexSTP( const AJobServer: String;
                                   const AJobType  : TJobType;
                                   const AJobPar   : String ): Integer;
var  Index: Integer;
begin
   Result := -1;
   Enter;
   try
      for Index:=0 to FJobList.Count-1 do
         with TNewsJob( FJobList[Index] ) do
            if ( CompareText( JobServer, AJobServer )=0 )
               and (JobType=AJobType) and (JobPar=AJobPar) then begin
               Result := Index;
               break;
            end;
   finally
      Leave;
   end;
end;

function TNewsJobList.JobIndexST( const AJobServer  : String;
                                  const AJobType    : TJobType ): Integer;
var  Index: Integer;
begin
   Result := -1;
   Enter;
   try
      for Index:=0 to FJobList.Count-1 do
         with TNewsJob( FJobList[Index] ) do
            if ( CompareText( JobServer, AJobServer )=0 )
               and (JobType=AJobType) then begin
               Result := Index;
               break;
            end;
   finally
      Leave;
   end;
end;

function TNewsJobList.JobIndexTP( const AJobType: TJobType;
                                  const AJobPar : String ): Integer;
var  Index: Integer;
begin
   Result := -1;
   Enter;
   try
      for Index:=0 to FJobList.Count-1 do
         with TNewsJob( FJobList[Index] ) do
            if (JobType=AJobType) and (JobPar=AJobPar) then begin
               Result := Index;
               break;
            end;
   finally
      Leave;
   end;
end;

function SortByPriority( Item1, Item2: Pointer ): Integer;
var  A: TNewsJob absolute Item1;
     B: TNewsJob absolute Item2;
begin
   if      A.JobPriority < B.JobPriority then Result := +1
   else if A.JobPriority > B.JobPriority then Result := -1
   else                                       Result :=  0
end;

procedure TNewsJobList.Sort;
begin
   if FResort then begin
      Enter;
      try
         FJobList.Sort( SortByPriority );
         FResort := False;
      finally
         Leave;
      end;
   end;
end;

procedure TNewsJobList.GetList(Var Defs: TNewsJobDefs);
Var Index: Integer;
begin
   Enter;
   try
      if FResort then Sort;
      SetLength(Defs, FJobList.Count);
      for Index:=0 to FJobList.Count-1 do begin
         Defs[Index] := TNewsJob( FJobList[Index] ).FDef
      end;
   finally
      Leave
   end
end;

function TNewsJobList.JobGet( const AJobServer: String;
                              var   AJobType  : TJobType;
                              var   AJobPar   : String ): Boolean;
var  Index: Integer;
begin
   Result   := False;
   AJobType := JOBTYPE_INVALID;

   Enter;
   try
      if FResort then Sort;
      for Index:=0 to FJobList.Count-1 do begin
         with TNewsJob( FJobList[Index] ) do begin
            if CompareText( JobServer, AJobServer )=0 then begin
               AJobType := JobType;
               AJobPar  := JobPar;
               Result   := True;
               TNewsJob( FJobList[Index] ).Free;
               FJobList.Delete( Index );
               break;
            end;
         end;
      end;
   finally
      Leave;
   end;
end;

procedure TNewsJobList.JobSet( const AJobServer  : String;
                               const AJobType    : TJobType;
                               const AJobPar     : String;
                               const AJobPriority: Integer );
var  Index: Integer;
begin
   Enter;
   try
      Index := JobIndexSTP( AJobServer, AJobType, AJobPar );
      if Index>=0 then begin
         TNewsJob( FJobList[Index] ).JobPriority := AJobPriority;
      end else begin
         If AktID = MAXINT then AktID := 1 else AktID := AktID + 1;
         FJobList.Add(
            TNewsJob.Create( AktID, AJobServer, AJobType, AJobPar, AJobPriority )
         );
      end;
      FResort := True;
   finally
      Leave;
   end;
end;

procedure TNewsJobList.JobDelay( const AJobServer: String;
                                 const AJobType  : TJobType;
                                 const AJobPar   : String;
                                 JobsToSkip      : Integer );
var  Index, AJobPriority: Integer;
begin
   Enter;
   try
      AJobPriority := -1;
      for Index:=0 to FJobList.Count-1 do begin
         with TNewsJob( FJobList[Index] ) do begin
            if CompareText( JobServer, AJobServer )=0 then begin
               AJobPriority := JobPriority - 1;
               dec( JobsToSkip );
               if JobsToSkip<=0 then break;
            end;
         end;
      end;
      JobSet( AJobServer, AJobType, AJobPar, AJobPriority );
   finally
      Leave;
   end;
end;

function TNewsJobList.MaxThreadsFor( AJobServer: String ): Integer;
var  Index   : Integer;
     HaveJobs: Boolean;
begin
   Result := 0;
   Enter;
   try
      HaveJobs := False;
      for Index:=0 to FJobList.Count-1 do begin
         with TNewsJob( FJobList[Index] ) do begin
            if CompareText( JobServer, AJobServer )=0 then begin
               HaveJobs := True;
               if JobType=JOBTYPE_NEWSPULL then begin
                  inc( Result );
                  if Result>=4 then break;
               end;
            end;
         end;
      end;
      if HaveJobs and (Result=0) then Result:=1;
   finally
      Leave;
   end;
end;

constructor TNewsJobList.Create;
begin
   inherited Create;
   InitializeCriticalSection( FLock );
   FJobList := TList.Create;
   FResort := False;
end;

destructor TNewsJobList.Destroy;
begin
   if Assigned( FJobList ) then begin Clear; FJobList.Free end;
   DeleteCriticalSection( FLock );
   inherited Destroy;
end;


// ------------------------------------------------------------ TNewsJobs -----

{JW}
function TNewsJobs.GetServer( Index: Integer ): String;
begin
   Result:=FJobList.GetServer(Index);
end;
function TNewsJobs.GetPar( Index: Integer ): String;
begin
   Result:=FJobList.GetPar(Index);
end;
function TNewsJobs.GetType( Index: Integer ): Integer;
begin
   Result:=FJobList.GetType(Index);
end;
function TNewsJobs.GetPriority( Index: Integer ): Integer;
begin
   Result:=FJobList.GetPriority(Index);
end;
function TNewsJobs.GetCounter(): Integer;
begin
   Result:=FJobList.GetCounter();
end;
{JW}
{JW} {Prio}
function TNewsJobs.SetPriority( Index,NewPriority: Integer ): Integer;
begin
   FJobList.Enter;
   try
      Result:=FJobList.SetPriority(Index,NewPriority);
   finally
      FJobList.Leave;
   end;
end;

function TNewsJobs.Delete( Index: Integer ): Integer;
begin
   FJobList.Enter;
   try
      Result:=FJobList.delete(Index);
   finally
      FJobList.Leave;
   end;
end;

function TNewsJobs.Add(Servername:String;Typ:TJobType;Param:String;Priority: Integer): Integer;

begin
   FJobList.Enter;
   try
      FJobList.JobSet( Servername, Typ, Param, Priority);
      result:=0;
   finally
      FJobList.Leave;
   end;
end;
{JW}


function TNewsJobs.GetGroupPrio( Groupname: String ): Integer;
var  i: Integer;
begin
   FJobList.Enter;
   try
      i := FGroupPrios.IndexOf( GroupName );
      if i<0 then Result := 0
             else Result := Integer( FGroupPrios.Objects[i] );
   finally
      FJobList.Leave;
   end;
end;

procedure TNewsJobs.SetGroupPrio( Groupname: String; NewPriority: Integer );
var  i: Integer;
begin
   FJobList.Enter;
   try
      i := FGroupPrios.IndexOf( GroupName );
      if i<0 then FGroupPrios.AddObject( Groupname, Pointer(NewPriority) )
             else FGroupPrios.Objects[i] := Pointer(NewPriority);
   finally
      FJobList.Leave;
   end;
end;

function TNewsJobs.MaxThreadsFor( Servername: String ): Integer;
begin
   Result := FJobList.MaxThreadsFor( Servername );
end;

function TNewsJobs.Clear: Integer;
begin
   Result := 0;
   FJobList.Enter;
   try
      FJobList.Clear;
   finally
      FJobList.Leave;
   end;
end;

function TNewsJobs.CheckInfo(ServerList: String): Integer;
var  LfdServer : Integer;
     ServerName: String;
begin
   Result := 0;
   CfgHamster.Lock.BeginRead;
   try
      for LfdServer:=0 to CfgHamster.ServerCount-1 do begin
         ServerName := CfgHamster.ServerName[LfdServer];
         if IsServerSelected( ServerList, ServerName, CfgHamster.ServerPort[LfdServer] ) then begin
           // get server infos (initial group-list, new groups, ...)
           FJobList.JobSet( Servername, JOBTYPE_SRVINFOS, '', JOBPRIO_SRVINFOS );
           Log(LOGID_DETAIL, 'Add job to check active for '+Servername);
           result :=result+1;
         end;
      end;
   finally ///
      CfgHamster.Lock.EndRead; ///
   end; ///
end;

function TNewsJobs.AddPullSrv( Servername, reGrpSelect: String ): Integer;
var  Groupname: String;
     i: Integer;
begin
   Result := 0;
   CfgHamster.Lock.BeginRead; ///
   FJobList.Enter;
   try
      // get server infos (initial group-list, new groups, ...)
      If Def_NNTPAutoGetServerInfos then begin {JW} {news check info}
         FJobList.JobSet( Servername, JOBTYPE_SRVINFOS, '', JOBPRIO_SRVINFOS )
      end;
      // get list of Message-IDs
      if FileExists2( PATH_SERVER + Servername + '\' + SRVFILE_GETMIDLIST ) then begin
         FJobList.JobSet( Servername, JOBTYPE_GETBYMID, '', JOBPRIO_GETBYMID );
      end;

      // pull newsgroups
      for i:=0 to CfgHamster.PullCount-1 do begin
         if CompareText( CfgHamster.PullServer[i], Servername )=0 then begin
            Groupname := CfgHamster.PullGroup[i];
            if (reGrpSelect='')
            or RE_Match( PChar(Groupname), PChar(reGrpSelect), PCRE_CASELESS ) then begin
               FJobList.JobSet( Servername, JOBTYPE_NEWSPULL,
                                Groupname, GroupPriority[Groupname] );
               inc( Result );
            end;
         end;
      end;
   finally
      FJobList.Leave;
      CfgHamster.Lock.EndRead; ///
   end;
end;

{JW} {Feed}
function TNewsJobs.AddFeedSrv( Servername, reGrpSelect: String ): Integer;
var  Groupname: String;
     i: Integer;
begin
   Result := 0;
   CfgHamster.Lock.BeginRead; ///
   FJobList.Enter;
   try
      Servername:=Trim(Servername);
      // feed newsgroups
      for i:=0 to CfgHamster.ActiveCount-1 do begin
         GroupName:=CfgHamster.ActiveName[i];
         if (reGrpSelect='') or
            RE_Match( PChar(GroupName), PChar(reGrpSelect), PCRE_CASELESS )
            then begin
            Log( LOGID_INFO,  'Add group '+GroupName+' on '+Servername+' to list of news jobs');
            FJobList.JobSet( Servername, JOBTYPE_NEWSFEED,
                             GroupName, JOBPRIO_NEWSFEED );
            inc( Result );
         end;
      end;
   finally
      FJobList.Leave;
      CfgHamster.Lock.EndRead;
   end;
end;

{JW}  {NoFound}
function TNewsJobs.AddPullDef ( ServerList: String ): Integer;
var  LfdServer, offs, i, j: Integer;
     PullServer, s: String;
begin
   Result := 0;
   CfgHamster.Lock.BeginRead;
   FJobList.Enter;
   try
      With TStringList.Create do try
         if ServerList = '' then begin
            for LfdServer:=0 to CfgHamster.ServerCount-1 do begin
               Add ( CfgHamster.ServerName[LfdServer] )
            end
         end else begin
            Text := SplitServerList ( Serverlist );
            offs := 0;
            For i := 0 to Count-1 do begin
               j := i - offs;
               s := Strings[j];
               LfdServer := CfgHamster.ServerIndexOf [ s ];
               If LfdServer >= 0 then begin
                  Strings[j] := CfgHamster.ServerName[LfdServer]
               end else begin
                  Log ( LOGID_WARN, TrGlF (kLog, 'AddPullDef.UnknownServer',
                        'AddPullDef: Unknown server "%s"!', s ) );
                  Delete(j); Inc(offs)
               end
            end
         end;
         for LfdServer:=0 to Count-1 do begin
            PullServer := Strings[LfdServer];
            // refresh Pull-List for server
            inc( Result, AddPullSrv( PullServer, '' ) );
            Log(LOGID_DETAIL, TrGlF(kLog, 'AddPullDef.Added', 'Server %s add to job list', Pullserver ) )
         end
      finally
         free
      end
   finally
      FJobList.Leave;
      CfgHamster.Lock.EndRead;
   end;
end;
{JW}

function TNewsJobs.AddPostSrv( Servername, reGrpSelect, reMsgSelect: String ): Integer;
var  Art     : TArticle;
     Parser  : TParser;
     iFindRes, iGroup, iLine: Integer;
     PostSR  : TSearchRec;
     PostFile, PostServer, PostGroup, sLine: String;
begin
   Result := 0;
   Art    := TArticle.Create;
   Parser := TParser.Create;

   EnterCriticalSection( CS_LOCK_NEWSOUT );
   CfgHamster.Lock.BeginRead;
   FJobList.Enter;
   try
      // check all messages waiting to be posted
      iFindRes := SysUtils.FindFirst( PATH_NEWS_OUT + '*.msg', faAnyFile, PostSR );

      while iFindRes=0 do begin
         PostFile := PATH_NEWS_OUT + PostSR.Name;
         Art.Text := '';

         // load message, if it's not already assigned to any server
         if FJobList.JobIndexTP( JOBTYPE_NEWSPOST, PostFile ) < 0 then begin
            try Art.LoadFromFile(PostFile) except Art.Text:='' end;
         end;

         // if message is not assigned yet and could be loaded: check msg&server
         if Art.Text<>'' then begin
            Log( LOGID_INFO, TrGlF(kLog, 'News.checkarticle',
              'Check article %s', PostFile ) );
            PostServer := '';

            // check newsgroup-selection
            sLine := Art.Header['Newsgroups:'];
            if (reGrpSelect='')
            or RE_Match( PChar(sLine), PChar(reGrpSelect), PCRE_CASELESS ) then begin

               // check, if given server is a valid post-server for any group
               Parser.Parse( Art.Header['Newsgroups:'], ',' );
               iGroup := 0;
               repeat
                  PostGroup  := TrimWhSpace( Parser.sPart( iGroup, '' ) );
                  if PostGroup='' then break;
                  if CfgHamster.IsPostServerFor( Servername, PostGroup ) then begin
                     PostServer := Servername;
                     break;
                  end;
                  inc( iGroup );
               until PostServer<>'';

            end;

            // check, if any header matches the given regular expression
            if (PostServer<>'') and (reMsgSelect<>'') then begin
               PostServer := '';
               With TStringList.Create do try
                  Text := Art.FullHeader;
                  For iLine := 0 to Count-1 do begin
                     sLine := Strings[iLine];
                     if RE_Match( PChar(sLine), PChar(reMsgSelect), PCRE_CASELESS ) then begin
                        PostServer := Servername;
                        break;
                     end
                  end
               finally
                  Free
               end
            end;

            // create post-job for the message if a postserver was found for it
            if PostServer<>'' then begin
               Log( LOGID_INFO, TrGlF(kLog, 'News.ArticleAcceptedPostToServer',
                  '%s accepted; will post to: %s', [PostFile, PostServer] ) );
               If Def_NNTPAutoGetServerInfos then begin {JW} {news check info}
                  FJobList.JobSet( PostServer, JOBTYPE_SRVINFOS, '', JOBPRIO_SRVINFOS )
               end;
               FJobList.JobSet( PostServer, JOBTYPE_SRVINFOS, '',       JOBPRIO_SRVINFOS );
               FJobList.JobSet( PostServer, JOBTYPE_NEWSPOST, PostFile, JOBPRIO_NEWSPOST );
               inc( Result );
            end;
         end;

         iFindRes := SysUtils.FindNext( PostSR );
      end;

      SysUtils.FindClose( PostSR );

   finally
      FJobList.Leave;
      CfgHamster.Lock.EndRead;
      LeaveCriticalSection( CS_LOCK_NEWSOUT );
      Parser.Free;
      Art.Free;
   end;
end;

function TNewsJobs.AddPostDef( ServerList: String ): Integer;
var  Art     : TArticle;
     Parser  : TParser;
     iFindRes, iGroup, iServer, PostSrvIdx: Integer;
     PostSR  : TSearchRec;
     s, PostFile, PostServer, PostGroup, TestServer: String;
begin
   Result := 0;
   Art    := TArticle.Create;
   Parser := TParser.Create;

   EnterCriticalSection( CS_LOCK_NEWSOUT );
   CfgHamster.Lock.BeginRead;
   FJobList.Enter;
   try
      // Check Server in List [TGL]
      if ServerList > '' then begin
         With TStringList.Create do try
            Text := SplitServerList (ServerList);
            ServerList := '';
            For iServer := 0 to Count-1 do begin
               s := Strings[iServer];
               If CfgHamster.ServerIndexof[s] >= 0 then begin
                  ServerList := ServerList + ';' + s;
               end else begin
                  Log ( LOGID_WARN, TrGlF(kLog, 'AddPostDef.UnknownServer',
                     'AddPostDef: Unknown server "%s"!', s))
               end
            end;
            If ServerList = ''
               then exit
               else ServerList := ServerList + ';'
         finally
            free
         end
      end;

      // check all messages waiting to be posted
      iFindRes := SysUtils.FindFirst( PATH_NEWS_OUT + '*.msg', faAnyFile, PostSR );

      while iFindRes=0 do begin
         PostFile := PATH_NEWS_OUT + PostSR.Name;
         Art.Text := '';

         // load message, if it's not already assigned to any server
         if FJobList.JobIndexTP( JOBTYPE_NEWSPOST, PostFile ) < 0 then begin
            try Art.LoadFromFile(PostFile) except Art.Text:='' end;
         end;

         // if message is not assigned yet and could be loaded: find post-server
         if Art.Text<>'' then begin
            Log( LOGID_INFO, TrGlF(kLog, 'News.checkarticle',
              'Check article %s', PostFile ) );

            // get postserver-name based on newsgroup-names
            Parser.Parse( Art.Header['Newsgroups:'], ',' );
            iGroup := 0;

            repeat
               PostServer := '';
               PostGroup  := TrimWhSpace( Parser.sPart( iGroup, '' ) );
               if PostGroup='' then break;

               // use preferred postserver, if configured and selected
               PostServer := CfgHamster.ActivePostServer[ PostGroup  ];
               PostSrvIdx := CfgHamster.ServerIndexOf   [ PostServer ];
               if not IsServerSelected( ServerList, PostServer,
                  CfgHamster.ServerPort[PostSrvIdx] ) then PostServer:='';

               // look for alternative, selected postservers
               if PostServer='' then begin
                  for iServer:=0 to CfgHamster.ServerCount-1 do begin
                     TestServer := CfgHamster.ServerName[iServer];
                     if CfgHamster.IsPostServerFor( TestServer, PostGroup ) then begin
                        // alternative server found, but is it selected?
                        if IsServerSelected( ServerList, TestServer,
                           CfgHamster.ServerPort[iServer] ) then begin
                           PostServer := TestServer;
                           break;
                        end;
                     end;
                  end;
               end;

               inc( iGroup );
            until PostServer<>'';

            // create post-job for the message if a postserver was found for it
            if PostServer<>'' then begin
               Log( LOGID_INFO, TrGlF(kLog, 'News.ArticleAcceptedPostToServer',
                  '%s accepted; will post to: %s', [PostFile, PostServer] ) );
               If Def_NNTPAutoGetServerInfos then begin {JW} {news check info}
                  FJobList.JobSet( PostServer, JOBTYPE_SRVINFOS, '',       JOBPRIO_SRVINFOS )
               end;
               FJobList.JobSet( PostServer, JOBTYPE_NEWSPOST, PostFile, JOBPRIO_NEWSPOST );
               inc( Result );
            end;
         end;

         iFindRes := SysUtils.FindNext( PostSR );
      end;

      SysUtils.FindClose( PostSR );

   finally
      FJobList.Leave;
      CfgHamster.Lock.EndRead;
      LeaveCriticalSection( CS_LOCK_NEWSOUT );
      Parser.Free;
      Art.Free;
   end;
end;

function TNewsJobs.StartThreads( ServerList: String ): Integer;
var  LfdServer, LfdThread, i: Integer;
     ServerName: String;
begin
   Result := 0;
   CfgHamster.Lock.BeginRead;
   try
      for LfdServer:=0 to CfgHamster.ServerCount-1 do begin
         ServerName := CfgHamster.ServerName[LfdServer];
         if IsServerSelected( ServerList, ServerName,
                              CfgHamster.ServerPort[LfdServer] ) then begin
            i := CfgHamster.ServerPullThreads( LfdServer, True );
            for LfdThread:=1 to i do begin
               TThreadNewsJobs.Create( LfdServer).resume;
               inc( Result );
               Sleep( 100 );
            end;
         end;
      end;
   finally ///
      CfgHamster.Lock.EndRead; ///
   end; ///
end;

constructor TNewsJobs.Create;
begin
   inherited Create;
   FJobList := TNewsJobList.Create;
   FGroupPrios := TStringList.Create;
   FGroupPrios.Sorted := True;
   FGroupPrios.Duplicates := dupIgnore;
end;

destructor TNewsJobs.Destroy;
begin
   if Assigned(FGroupPrios) then FGroupPrios.Free;
   if Assigned(FJobList)    then FJobList.Free;
   inherited Destroy;
end;

// ----------------------------------------------------------------------------

Function JobInfo (Const Typ: TJobType; Const Par: String): String;
begin
   Case Typ of
      JOBTYPE_SRVINFOS: Result := 'Get serverinfo';
      JOBTYPE_NEWSPOST: Result := 'Post '+Par;
      JOBTYPE_GETBYMID: Result := 'Get articles by MID';
      JOBTYPE_NEWSPULL: Result := 'Pull new articles from '+Par;
      JOBTYPE_NEWSFEED: Result := 'Feeding '+Par;
      else Result := '<unknown>'
   end
end;

end.
