unit cServerBase;

interface

{$INCLUDE Compiler.inc}

Uses Classes, ScktComp, uSSL; {MG}{SSL}

type
  TSrvBaseCli = class;
  TClassSrvBaseCli = class of TSrvBaseCli;

  TSrvBase = class( TServerSocket )
    private
      MaxClients      : Integer;
      MaxClientsPerIP : Integer;
      IPAccessScope   : LongInt;
      ClassSrvBaseCli : TClassSrvBaseCli;
      function  GetActiveConnections: Integer;
    protected
      function SockDesc( Socket: TCustomWinSocket; Const EventDesc: String ): String;
      function ClientAccess( ClientSocket: TServerClientWinSocket ): LongInt;
      procedure MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                              var SocketThread: TServerClientThread);
    public
      {MG}{SSL}
      SSLCtx : TSSLContext;
      OpenSSLContext : Pointer;
      procedure InitSSLContext;
      {/SSL}
      property ActiveConnections: Integer read GetActiveConnections;
      constructor Create( AOwner: TComponent;
                          Const AServerBind: String;
                          Const AServerPort: Integer;
                          Const AMaxClients: Integer;
                          Const AMaxClientsPerIP: Integer;
                          Const AIPAccessScope: LongInt;
                          AClassSrvBaseCli: TClassSrvBaseCli ); reintroduce;
      destructor Destroy; override;
  end;

  TSrvBaseCli = class(TServerClientThread)
    protected
      IPAccessScope: LongInt;
      Function ShutDownReq: Boolean;
    public
      constructor Create( ASocket: TServerClientWinSocket;
                          Const AIPAccessScope: LongInt;
                          Const AOpenSSLContext: Pointer ); virtual; {MG}{SSL}
  end;

  TSrvRefuseCli = class(TSrvBaseCli)
    public
      Reply: String;
      procedure ClientExecute; override;
      destructor Destroy; override;
  end;

  TSrvWorkCli = class(TSrvBaseCli)
    protected
      IPAccess       : LongInt;
      WaitForCmd     : Boolean;
      BufInRaw       : String;
      BufInStrm      : String {TMemoryStream};
      BufInStrmLen   : Integer;
      ClientID       : String;
      LimitLineLen   : Integer;
      LimitTextSize  : Integer;
      HadLineTooLong : Boolean;
      HadTextTooLarge: Boolean;
      CurrentInactTimeOut : Integer;
      {MG}{SSL}
      OpenSSLCtx : Pointer;
      SSL        : TSSLConnection;
      {/SSL}
      procedure CheckClientAccess;

    public
      function SockDesc( Const EventDesc: String ): String;

      function  ReceiveData( Const TimeoutMS: Integer ): String;
      procedure SendData  ( Const Txt: String );
      procedure SendResult( Const Txt: String );
      procedure SendQuoted( Const Txt: String );
      procedure SendQuotedMultiLine( Const Txt: String );
      function  SendRequest( Const Txt: String ): String;

      procedure SendGreeting; virtual;

      function  HandleData: String; virtual;
      procedure HandleCommand( Const CmdLine: String ); virtual;

      procedure ClientSocketError( Sender: TObject; Socket: TCustomWinSocket;
                                   ErrorEvent: TErrorEvent; var ErrorCode: Integer);
      procedure ClientExecute; override;
      {MG}{SSL}
      function StartSSL: Boolean;
      property SSLContext: Pointer read OpenSSLCtx;
      property SSLConnection: TSSLConnection read SSL;

      constructor Create( ASocket: TServerClientWinSocket;
                          Const AIPAccessScope: LongInt;
                          Const AOpenSSLContext: Pointer ); override;
      {/SSL}
      destructor Destroy; override;
  end;

const
   R_INFO     = 100;

   R_OK0      = 200;
   R_OK1      = 201;
   R_AUTHOK   = 280;

   R_AUTHCHLG_SMTP = 334;
   R_AUTHCHLG_RECO = 380;

   R_FAILED0  = 400;
   R_FAILED1  = 401;
   R_AUTHREQ  = 480;

   R_SYNTAX   = 500;
   R_UNKNOWN  = 501;
   R_NOPERM   = 502;
   R_SYSERR   = 503;
   R_AUTHFAIL = 580;

function Res( ResCode: Integer; ResText: String ): String;

{
function Local_SASL_Login( const SrvWorkCli: TSrvWorkCli;
                           const SrvChallengeReply: String;
                           const AUTH_Stamp, Mechanism, Parameters: String;
                           var   CurrentUserID: Integer;
                           var   CurrentUserName: String ): Boolean;
}

implementation

uses SysUtils, Windows, Global, cIPAccess, uTools, Winsock, uWinSock,
     Config, cAccount, uMD5, uEncoding, cLogfile, tBase, cStdForm;

function Res( ResCode: Integer; ResText: String ): String;
begin
   Result := Format( '%3d %s', [ ResCode, ResText ] );
end;
     
// ----------------------------------------------------------------------------

(*
function Local_SASL_Login( const SrvWorkCli: TSrvWorkCli;
                           const SrvChallengeReply: String;
                           const AUTH_Stamp, Mechanism, Parameters: String;
                           var   CurrentUserID: Integer;
                           var   CurrentUserName: String ): Boolean;
// SASL authentication for local SMTP-, POP3- and RECO-servers.
const SASL_CANCEL = '*';
      SASL_EMPTY  = '=';
var  s, u, p: String;
begin
   Result := False;
   CurrentUserName := '';
   CurrentUserID   := ACTID_INVALID;

   if Mechanism='LOGIN' then begin

      if Parameters = '' then begin
         s := 'Username:';
         s := EncodeB64( s[1], length(s) );
         s := SrvWorkCli.SendRequest( SrvChallengeReply + s );
      end else begin
         s := Parameters;
      end;
      if (s='') or (s=SASL_CANCEL) then exit;
      u := DecodeB64( s[1], length(s) );

      s := 'Password:';
      s := EncodeB64( s[1], length(s) );
      s := SrvWorkCli.SendRequest( SrvChallengeReply + s );
      if (s='') or (s=SASL_CANCEL) then exit;
      p := DecodeB64( s[1], length(s) );

   end else if Mechanism='PLAIN' then begin

      if Parameters = '' then begin
         s := SrvWorkCli.SendRequest( SrvChallengeReply + SASL_EMPTY );
      end else begin
         s := Parameters;
      end;
      if (s='') or (s=SASL_CANCEL) then exit;
      if not AUTH_PLAIN_Decode( s, u, p ) then exit;

   end else if Mechanism='CRAM-MD5' then begin

      s := EncodeB64( AUTH_Stamp[1], length(AUTH_Stamp) );
      s := SrvWorkCli.SendRequest( SrvChallengeReply + s );
      if (s='') or (s=SASL_CANCEL) then exit;

      if not AUTH_CRAM_MD5_Decode( s, u, p ) then exit;
      if (u='') or (p='') then exit;

      CurrentUserID := CfgAccounts.UserIDOf( u );
      if CurrentUserID = ACTID_INVALID then exit;

      s := CfgAccounts.Value[ CurrentUserID, ACTP_PASSWORD ];
      if s='' then exit;

      if p <> HMAC_MD5( AUTH_Stamp, s ) then exit;
      p := s;

   end;

   if (u<>'') and (p<>'') then begin
      CurrentUserID := CfgAccounts.LoginID( u, p );
      if CurrentUserID <> ACTID_INVALID then begin
         CurrentUserName := u;
         Result := True;
      end;
   end;
end;

*)

// ---------------------------------------------------------- TSrvBaseCli -----

constructor TSrvBaseCli.Create( ASocket: TServerClientWinSocket;
                                Const AIPAccessScope: Integer;
                                Const AOpenSSLContext: Pointer ); {MG}{SSL}
begin
   inherited Create( True, ASocket );
   IPAccessScope := AIPAccessScope;
end;


// -------------------------------------------------------- TSrvRefuseCli -----

procedure TSrvRefuseCli.ClientExecute;
begin
   if Assigned(ClientSocket) then try
      Log( LOGID_WARN, TrGl(kLog, 'Connection.refused', 'Connection refused:')
                     + ClientSocket.RemoteAddress
                     + ':' + inttostr( ClientSocket.LocalPort )
                     + '  ' + Reply );
      if (Reply<>'') and ClientSocket.Connected then ClientSocket.SendText( Reply + CRLF );
      ClientSocket.Close;
   except
   end;

   KeepInCache := False;
   Terminate;
end;

destructor TSrvRefuseCli.Destroy;
begin
   Logfile.RemoveTask ( lowercase(inttohex(GetCurrentThreadID,1)) );
   inherited;
end;


// ---------------------------------------------------------- TSrvWorkCli -----

procedure TSrvWorkCli.CheckClientAccess;
var  raddr: LongInt;
begin
   IPAccess := IPACC_ACCESS_NA;
   if ClientSocket=nil then exit;
   try
      raddr := ClientSocket.RemoteAddr.sin_addr.S_addr;
      IPAccess := IPAccessCheck.GetAccess( raddr, IPAccessScope );

      Log( LOGID_DEBUG, SockDesc( '.CheckClientAccess' )
                   + IPAccessScopeStr( IPAccessScope )
                   + '(' + inet_ntoa( ClientSocket.RemoteAddr.sin_addr ) + ')'
                   + ' -> ' + IPAccessAccessStr( IPAccess ) );
                   // + '0x'     + inttohex( ntohl(raddr), 8 )
                   // + ', 0x'   + inttohex( IPAccessScope, 1 )
                   // + ' -> 0x' + inttohex( IPAccess, 1 ) );
   except
      IPAccess := IPACC_ACCESS_NA;
   end;
end;

function TSrvWorkCli.SockDesc( Const EventDesc: String ): String;
begin
     try
        if ClientSocket=nil then
           Result := Self.Classname + EventDesc + '(socket closed): '
        else
           Result := Self.Classname + EventDesc + '(' + inttostr(ClientSocket.SocketHandle)+'): ';
     except
        Result := Self.Classname + EventDesc + '(?Socket closed?): ';
     end;
end;

function TSrvWorkCli.ReceiveData( Const TimeoutMS: Integer ): String;
var  SocketStream: TWinSocketStream;
     Data: array[0..1023] of Char;
     ByteIn, i: Integer;
begin
   Result := '';

   SocketStream := TWinSocketStream.Create( ClientSocket, TimeoutMS );
   try

      ByteIn := 0;

      try
         while Assigned(ClientSocket) and (ClientSocket.Connected) do begin
            if ShutDownReq or Terminated then break;
            if ClientSocket=nil then break;
            if not ClientSocket.Connected then break;

            {MG}{SSL}
            if not ( ( Assigned(SSL) and SSL.HasPendingData ) or
                     SocketStream.WaitForData(TimeoutMS) ) then begin
               // If there are no buffered data inside ssl and we
               // didn't get any data after ? seconds then close the connection
            {/SSL}
              Log( LOGID_WARN, TrGl (kLog, 'Connection.timeout',
                   'Connection closed (timeout):') + ClientID );
              FreeAndNil( SocketStream );
              if ClientSocket.Connected then ClientSocket.Close;
              Terminate;
              exit;
            end;

            FillChar( Data, SizeOf(Data), 0 );
            ByteIn := 0;
            try
               if ClientSocket.Connected then begin
                  if SSLConnection=NIL {MG}{SSL}
                     then ByteIn := SocketStream.Read( Data, SizeOf(Data) )
                     else ByteIn := SSL.Read( Data, SizeOf(Data) ); {MG}{SSL}
               end;
            except
               ByteIn := 0;
            end;
            if ByteIn = 0 then break;

            if ((LogFile.FileMask or LogFile.ViewMask) and LOGID_DETAIL)<>0 then begin
               Log( LOGID_DEBUG, SockDesc('.Recv2') + Data );
            end;

            Result := Result + Data;
            i := Pos( #10, Result );
            if i>=0 then break;
         end;

      except
         on E:Exception do begin
            Log( LOGID_ERROR, 'ReceiveData.Exception: ' + E.Message );
         end;
      end;

      if ByteIn = 0 then begin
         If Not LogFile.Skip_ConectionLost then begin
            Log( LOGID_WARN, TrGl (kLog, 'Connection.lost', 'Connection lost:') +
                ' ' + ClientID ) // JAWO 26.01.02
         end;
         if Assigned(ClientSocket) then try
            if ClientSocket.Connected then ClientSocket.Close
         except end;
      end;

   finally
      If Assigned(SocketStream) then SocketStream.Free
   end;
end;

procedure TSrvWorkCli.SendData( Const Txt: String );
var  SocketStream: TWinSocketStream;
     snd: Integer;
     s: String;
begin
   if ClientSocket=nil then exit;
   s := txt;
   SocketStream := TWinSocketStream.Create( ClientSocket, 60000 );
   try
      try
         while Assigned(ClientSocket) and ClientSocket.Connected and (s > '') do begin
            if SSLConnection=nil {MG}{SSL}
               then snd := SocketStream.Write( s[1], length(s) )
               else snd := SSL.Write( PChar( s ), length(s)  ); {MG}{SSL}
            if snd < 0 then break; {MG}{SSL}
            If snd > 0 then System.Delete( s, 1, snd )
         end
      except
         on E:Exception do
            Log( LOGID_DEBUG, SockDesc('.SendData') + E.Message );
      end
   finally
      SocketStream.Free;
   end;
end;

procedure TSrvWorkCli.SendResult( Const Txt: String );
begin
   if Txt='.' then Log( LOGID_DETAIL, '< ' + Txt )
              else Log( LOGID_INFO,   '< ' + Txt );
   SendData( Txt + CRLF );
end;

procedure TSrvWorkCli.SendQuoted( Const Txt: String );
Var s: String;
begin
   if ClientSocket=nil then exit;
   if copy(Txt,1,1)='.'
      then s := '.'+txt // quote leading dot
      else s := txt;
   if ((LogFile.FileMask or Logfile.ViewMask) and LOGID_DETAIL)<>0 then begin
      Log( LOGID_DEBUG, SockDesc('.SendT') + s )
   end;
   SendData( s + CRLF )
end;

function TSrvWorkCli.SendRequest( Const Txt: String ): String;
begin
   SendResult( Txt );
   Result := ReceiveData( CurrentInactTimeOut );
end;

function TSrvWorkCli.HandleData: String;
begin
   Result := '';
   if ClientSocket=nil then exit;
   if not ClientSocket.Connected then exit;
   Result := '+ Data received';
   if HadLineTooLong or HadTextTooLarge then Result:='- Limit exceeded';
end;

procedure TSrvWorkCli.HandleCommand( Const CmdLine: String );
begin
   if ClientSocket=nil then exit;
   
   try
      if not ClientSocket.Connected then exit;

      if HadLineTooLong or HadTextTooLarge then begin
         SendResult( '- Limit exceeded' );
         exit;
      end;

      if UpperCase(CmdLine)='QUIT' then begin
         SendResult( '+ BYE' );
         Sleep( Def_LocalTimeoutQuitDelay );
         ClientSocket.Close;
         Terminate;
         exit;
      end;

      if UpperCase(CmdLine)='DATA' then begin
         WaitForCmd := False;
         SetLength(BufInStrm, 0) {.Clear}; BufInStrmLen := 0;
         SendResult( '+ Ready to receive data; end with CRLF.CRLF' );
         exit;
      end;

      if UpperCase(CmdLine)='HELP' then begin
         SendResult( '+ List follows' );
         SendQuoted( 'DATA' );
         SendQuoted( 'HELP' );
         SendQuoted( 'QUIT' );
         SendResult( '.' );
         exit;
      end;

      SendResult( '- Unknown command: ' + CmdLine );

   except
      on E: Exception do
         Log( LOGID_DEBUG, SockDesc('.HandleCommand.Exception') + E.Message );
   end;
end;

procedure TSrvWorkCli.ClientSocketError( Sender: TObject; Socket: TCustomWinSocket;
                                    ErrorEvent: TErrorEvent; var ErrorCode: Integer);
begin
     Log( LOGID_WARN, SockDesc('.ClientSocketError')
                    + inttostr(ErrorCode) + ' '
                    + WinsockErrTxt( ErrorCode ) );
     ErrorCode := 0; // mark as "handled"
     Terminate;      // terminate client-thread
end;

procedure TSrvWorkCli.ClientExecute;
var  Data: array[0..1023] of Char;
     SocketStream: TWinSocketStream;
     LineIn, Reply : String;
     LineEnd, ByteIn, i : Integer;
begin
   Sleep( 0 );

   try
      SendGreeting;
      Logfile.SetTask( '(client ' + lowercase(inttohex(GetCurrentThreadID,1))+')', ClientID );
   except
      on E:Exception do begin
         Log( LOGID_WARN, TrGl (kLog, 'Connection.greeting.failed',
              'Connection failed (SendGreeting):') + ' ' + ClientID ); // JAWO 26.01.02
         Log( LOGID_WARN, TrGl (kLog, 'Error', 'Error') + ': ' + E.Message );
         Terminate;
      end;
   end;

   while not Terminated do try
      if ClientSocket=nil then exit;
      if not ClientSocket.Connected then exit;

      SocketStream := TWinSocketStream.Create( ClientSocket, CurrentInactTimeOut );
      try
         {MG}{SSL}
         if ( Assigned(SSL) and SSL.HasPendingData ) or
            SocketStream.WaitForData( CurrentInactTimeOut )
         then begin
         {/SSL}
            FillChar(Data, SizeOf(Data), 0);
            ByteIn := 0;
            if ClientSocket<>nil then begin
               try
                  if ClientSocket.Connected then begin
                    {MG}{SSL}
                    if SSLConnection=nil
                       then ByteIn := SocketStream.Read( Data, SizeOf(Data) )
                       else ByteIn := SSL.Read( Data, SizeOf(Data) );
                    {/SSL}
                  end;
               except
                  ByteIn := 0;
               end;
            end;
            if ByteIn = 0 then begin
              if Not LogFile.Skip_ConectionLost then begin
                 Log( LOGID_WARN, TrGl (kLog, 'Connection.lost', 'Connection lost:') +
                      ' ' + ClientID ); // JAWO 26.01.02
              end;
              if Assigned(ClientSocket) then begin
                 if ClientSocket.Connected then ClientSocket.Close;
              end;
              Terminate;
            end;

            BufInRaw := BufInRaw + Data;

            repeat
               LineEnd  := Pos( CRLF, BufInRaw );
               if (LineEnd=0) and (LimitLineLen>0) and (length(BufInRaw)>LimitLineLen) then begin
                  HadLineTooLong := True;
                  LineEnd := LimitLineLen + 1;
               end;

               if LineEnd>0 then begin

                  // ggf. virtuellen LineIn-Handler
                  LineIn := copy( BufInRaw, 1, LineEnd-1 );
                  System.Delete ( BufInRaw, 1, LineEnd+1 );

                  if WaitForCmd then begin
                     HandleCommand( LineIn );
                  end else begin
                     if LineIn='.' then begin
                        WaitForCmd := True;
                        SetLength(BufInStrm, BufInStrmLen); {TGL}
                        Reply := HandleData;
                        SetLength(BufInStrm, 0) {.Clear}; BufInStrmLen := 0;
                        if Reply<>'' then SendResult( Reply );
                     end else begin
                        if copy(LineIn,1,2)='..' then System.Delete(LineIn,1,1);
                        {if length(LineIn)>0 then}
                           If BufInStrmLen + length(LineIn) + 2 > Length(BufInStrm) then begin
                              If BufInStrmLen > 128000
                                 then SetLength(BufInStrm, BufInStrmLen + 163840)
                                 else SetLength(BufInStrm, BufInStrmLen + length(LineIn) + 2 + 16384)
                           end;
                           For i := 1 to Length(LineIn) do
                              BufInStrm[BufInStrmLen + i] := LineIn[i];
                           BufInStrm[BufInStrmLen + length(LineIn) + 1] := CRLF[1];
                           BufInStrm[BufInStrmLen + length(LineIn) + 2] := CRLF[2];
                           Inc (BufInStrmLen, length(LineIn) + 2);
                        {if length(LineIn)>0 then BufInStrm.Write( LineIn[1], length(LineIn) );
                        BufInStrm.Write( CRLF, 2 );}
                        if (LimitTextSize>0) and (BufInStrmLen{.Size}>LimitTextSize) then begin
                           HadTextTooLarge := True;
                           Reply := HandleData;
                           SetLength(BufInStrm, 0) {.Clear}; BufInStrmLen := 0;
                           if Reply<>'' then SendResult( Reply );
                        end;
                     end;
                  end;

                  if ClientSocket=nil then Terminate
                  else if not ClientSocket.Connected then Terminate;

               end;

               if HadLineTooLong or HadTextTooLarge then begin
                  if HadTextTooLarge then Reply:='textsize-limit exceeded'
                                     else Reply:='linelength-limit exceeded';
                  Log( LOGID_WARN, SockDesc('.Recv') + TrGl (kLog, 'Connection.terminated',
                       'Connection terminated:') + ' ' + Reply );  // JAWO 26.01.02
                  if ClientSocket<>nil then begin
                     if ClientSocket.Connected then ClientSocket.Close;
                  end;
                  Terminate;
               end;

            until (LineEnd=0) or Terminated;

         end else begin
           // If we didn't get any data after ? seconds then close the connection
           Log( LOGID_WARN, TrGl (kLog, 'Connection.timeout',
                'Connection closed (timeout):') + ' '  + ClientID );
           if ClientSocket.Connected then ClientSocket.Close;
           Terminate;
         end;
      finally
         SocketStream.Free;
      end;
   except
      Terminate;
   end;
end;

procedure TSrvWorkCli.SendGreeting;
begin
   IPAccess := IPACC_ACCESS_ALL;
   SendResult( '+ Base-server ready.' );
end;

constructor TSrvWorkCli.Create( ASocket: TServerClientWinSocket;
                                Const AIPAccessScope: Longint;
                                Const AOpenSSLContext: Pointer );
begin
   inherited Create( ASocket, AIPAccessScope, AOpenSSLContext);

   LimitLineLen    := 1000;
   LimitTextSize   := 1000000;
   HadLineTooLong  := False;
   HadTextTooLarge := False;
   {MG}{SSL}
   OpenSSLCtx      := AOpenSSLContext;
   SSL             := nil;
   {/SSL}
   CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;

   try
      BufInRaw   := '';
      BufInStrm  := ''; BufInStrmLen := 0;
      WaitForCmd := True;
      IPAccess   := 0;
      ClientSocket.OnErrorEvent := ClientSocketError;
      ClientID   := ClientSocket.RemoteAddress
                  + ':' + inttostr( ClientSocket.LocalPort )
                  + ' (' + inttostr( ClientSocket.SocketHandle ) + ')';
      Log( LOGID_INFO, TrGlF (kLog, 'Client.connected', 'Client %s connected', ClientID ) );
   except
      on E: Exception do
         Log( LOGID_ERROR, Self.ClassName + '.Create-Exeption: ' + E.message );
   end;

   Logfile.RemoveTask( lowercase(inttohex(GetCurrentThreadID,1)) );
end;

destructor TSrvWorkCli.Destroy;
begin
     Log( LOGID_INFO, TrGlF (kLog, 'Client.disconnected', 'Client %s disconnected', ClientID ) );
     BufInStrm:='';
     Log( LOGID_DEBUG, Self.Classname + '.Destroy' );

     if Assigned(SSL) then begin SSL.Shutdown; SSL.Free end; {MG}{SSL}

     LogFile.RemoveTask( '(client '+lowercase(inttohex(Self.ThreadID,1))+')' ); /// rein
     LogFile.RemoveTask( lowercase(inttohex(Self.ThreadID,1)) ); /// rein
     
     inherited;
end;


// ------------------------------------------------------------- TSrvBase -----

function TSrvBase.SockDesc( Socket: TCustomWinSocket; Const EventDesc: String ): String;
begin
     try
        Result := Self.Classname + EventDesc + '(' + inttostr(Socket.SocketHandle)+'): ';
     except
        Result := '(?TSrvBase.SockDesc?)' + EventDesc + '(?Socket_closed?): ';
     end;
end;

function TSrvBase.ClientAccess( ClientSocket: TServerClientWinSocket ): LongInt;
var  RAddr: LongInt;
begin
   Result := IPACC_ACCESS_NA;
   if ClientSocket=nil then exit;

   RAddr  := ClientSocket.RemoteAddr.sin_addr.S_addr;
   Result := IPAccessCheck.GetAccess( raddr, IPAccessScope );

   Log( LOGID_DEBUG, SockDesc( ClientSocket, '.ClientAccess' )
                   + IPAccessScopeStr( IPAccessScope )
                   + '(' + inet_ntoa( ClientSocket.RemoteAddr.sin_addr ) + ')'
                   + ' -> ' + IPAccessAccessStr( Result ) );
                   // + '0x'     + inttohex( ntohl(raddr), 8 )
                   // + ', 0x'   + inttohex( IPAccessScope, 1 )
                   // + ' -> 0x' + inttohex( IPAccess, 1 ) );
end;

function TSrvBase.GetActiveConnections: Integer;
begin
   if Assigned(Socket) then Result := Socket.ActiveConnections
                       else Result := 0;
end;

procedure TSrvBase.MyOnGetThread( Sender: TObject;
                                  ClientSocket: TServerClientWinSocket;
                                  var SocketThread: TServerClientThread);
var  i, Cnt: Integer;
begin
   Log( LOGID_DEBUG, Self.Classname + '.OnGetThread' );

   try
      if (MaxClients>0) and (ActiveConnections>MaxClients) then begin
         // server's connection limit is reached -> refuse
         SocketThread := TSrvRefuseCli.Create( ClientSocket, IPAccessScope, nil );
         TSrvRefuseCli(SocketThread).Reply :=
            'Connection limit reached - try again later.';

      end else if ClientAccess( ClientSocket ) = IPACC_ACCESS_NA then begin
         // unwanted IP -> refuse
         SocketThread := TSrvRefuseCli.Create( ClientSocket, IPAccessScope, nil );
         TSrvRefuseCli(SocketThread).Reply :=
            'Permission denied - do not try again.';

      end else begin
         Cnt := 0;
         // count number of active connections from same client
         If MaxClientsPerIP > 0 then try
            for i:=0 to Socket.ActiveConnections-1 do
               with Socket.Connections[i] do
                  if RemoteAddress=ClientSocket.RemoteAddress then inc(Cnt);
         except
         end;

         if (MaxClientsPerIP>0) and (Cnt>MaxClientsPerIP) then begin
            // user's connection limit is reached -> refuse
            SocketThread := TSrvRefuseCli.Create( ClientSocket, IPAccessScope, nil );
            TSrvRefuseCli(SocketThread).Reply :=
               'Connection rejected - more than '+IntToStr(MaxClientsPerIP)
               +' open connections from you not allowed!';
         end else begin
            // create a worker thread for connection
            SocketThread := ClassSrvBaseCli.Create( ClientSocket,
                IPAccessScope, OpenSSLContext ); {MG}{SSL}
         end;

      end;

      // start client thread
      SocketThread.Resume;

   except
      on E: Exception do
         Log( LOGID_ERROR, Self.Classname + '.OnGetThread: ' + E.Message );
   end;
end;

{MG}{SSL}
procedure TSrvBase.InitSSLContext;
var
     CaFile, CaPath, PrivateKey: String;
     Verify, Options: Integer;
const
     SSL_VERIFY_NONE = $0;
     SSL_VERIFY_PEER = $1;
     SSL_VERIFY_FAIL_IF_NO_PEER_CERT = $2;
     SSL_VERIFY_CLIENT_ONCE = $4;
     SSL_ACCEPT_ONLY_LOCAL_CERTS = $8;
begin
     Log( LOGID_DEBUG, Self.Classname + '.InitSSLContext' );

     // Get the configuration data
     CaFile     := CfgIni.ReadString  ( 'SSL', 'CAFile', '' );
     CaPath     := CfgIni.ReadString  ( 'SSL', 'CAPath', '' );
     Verify     := CfgIni.ReadInteger ( 'SSL', 'VerifyLevel', 0 );
     PrivateKey := CfgIni.ReadString  ( 'SSL', 'PrivateKeyPair', '' );

     if PrivateKey = '' then begin
        Log( LOGID_WARN, TrGl(kLog, 'SSL.NoKey',
             'No key pair specified - can not start SSL') );
        exit;
     end;

     Options := SSL_VERIFY_NONE;
     if Verify > 0 then Options := Options or SSL_VERIFY_PEER;
     if Verify > 1 then Options := Options or SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
     if Verify = 3 then Options := Options or SSL_ACCEPT_ONLY_LOCAL_CERTS;

     SSLCtx := TSSLContext.Create( True, Options, CaFile, CaPath );
     OpenSSLContext := SSLCtx.Context;
     if OpenSSLContext = nil then begin
        FreeAndNil( SSLCtx );
        Log( LOGID_ERROR, TrGl(kLog, 'SSL.CtxNil',
             'Creating SSL context failed') );
        exit;
     end;

     SSLCtx.SetOptions( Def_SSLCipherString ); {MG}{CipherList}

     if not SSLCtx.LoadPrivateKeyPair( PChar(PrivateKey) ) then begin
        FreeAndNil( SSLCtx );
        Log( LOGID_WARN, TrGl(kLog, 'SSL.LoadKeyError',
             'Loading SSL key pair failed') );
     end else
        Log( LOGID_DEBUG, TrGl(kLog, 'SSL.LoadKeySuccess',
             'SSL Key pair loaded') );
end;
{/SSL}

constructor TSrvBase.Create( AOwner: TComponent;
                             Const AServerBind: String;
                             Const AServerPort: Integer;
                             Const AMaxClients: Integer;
                             Const AMaxClientsPerIP: Integer;
                             Const AIPAccessScope: LongInt;
                             AClassSrvBaseCli: TClassSrvBaseCli );
var  s: String;
begin
     if AServerBind='' then s:='0.0.0.0' else s:=AServerBind;
     Log( LOGID_DEBUG, Self.Classname + '.Create(' + s + ', ' + inttostr(AServerPort)+')' );
     inherited Create(AOwner);

     ServerType       := stThreadBlocking;
     Port             := AServerPort;
     OnGetThread      := MyOnGetThread;
     MaxClients       := AMaxClients;
     MaxClientsPerIP  := AMaxClientsPerIP;
     IPAccessScope    := AIPAccessScope;
     ClassSrvBaseCli  := AClassSrvBaseCli;
     if AServerBind<>'' then Address := AServerBind;
end;

destructor TSrvBase.Destroy;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Destroy' );
     FreeAndNil( SSLCtx ); {MG}{SSL}
     if Active then Close;
     inherited Destroy;
end;

// ----------------------------------------------------------------------------

function TSrvWorkCli.StartSSL: Boolean;
begin
     Result := False;
     Log( LOGID_DEBUG, Self.Classname + '.StartSSL' );

     // Now we create a new SSL structure for data exchange
     // it inherits the settings of the underlying context
     SSL := TSSLConnection.Create( SSLContext );
     if SSL.Ssl = nil then begin
        FreeAndNil( SSL );
        Log( LOGID_ERROR, TrGl(kLog, 'SSL.SslNil',
             'Creating SSL connection structure failed') );
        exit;
     end;

     // Do the TLS/SSL handshake
     if SSL.Accept( ClientSocket.SocketHandle ) then begin
        SSL.Info;
        Result := True;
     end else begin
        FreeAndNil( SSL );
        Log( LOGID_WARN, TrGl(kLog, 'SSL.ConnectionFailed',
             'SSL connection failed') );
     end;
end;

function TSrvBaseCli.ShutDownReq: Boolean;
begin
   Result := AllShutDownReq or ThreadControl.Shutdown
end;

procedure TSrvWorkCli.SendQuotedMultiLine(const Txt: String);
Var s, Ziel: String; p: Integer;
begin
   if ClientSocket=nil then exit;
   Ziel := '';
   s := Txt;
   If Length(s) >= 2 then begin
      If (s[Length(s)-1] <> #13) or (s[Length(s)] <> #10) then s := s + #13#10 
   end;
   Repeat
      p := Pos(#13#10+'.', s);
      If p > 0 then begin
         // Copy with ".", don't delete "." => result is needed ".."
         Ziel := Ziel + Copy(s, 1, p+2);
         Delete(s, 1, p+1);
      end else begin
         Ziel := Ziel + s
      end
   until p = 0;
   if ((LogFile.FileMask  or LogFile.ViewMask) and LOGID_DETAIL)<>0 then begin
      Log( LOGID_DEBUG, SockDesc('.SendT') + Ziel )
   end;
   SendData(Ziel);
end;

end.
