// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cServerBaseAlt;

interface
implementation
end.

(*

uses Classes, ScktComp, cStdForm;

type
  TSrvBase = class( TServerSocket )
    protected
      function SockDesc( Socket: TCustomWinSocket; EventDesc: String ): String;
      procedure MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                              var SocketThread: TServerClientThread); virtual;
    public
      constructor CreateWithPort(AOwner: TComponent; AServerPort: Integer;
                                 AAddress:String );
      function CheckClientAccess( IPAccessScope: LongInt ;
                                  ClientSocket:
                                  TServerClientWinSocket):LongInt;
      destructor Destroy; override;
  end;

type
  TSrvBaseCli = class(TServerClientThread)
    protected
      CurrentThreadID:Longword;
      CurrentInactTimeOut : Integer;
      IPAccess   : LongInt;
      WaitForCmd : Boolean;
      BufInRaw   : String;
      BufInStrm  : String {TMemoryStream};
      BufInStrmLen: Integer;
      ClientID   : String;
      LimitLineLen   : Integer;
      LimitTextSize  : Integer;
      HadLineTooLong : Boolean;
      HadTextTooLarge: Boolean;
    public
      procedure CheckClientAccess( IPAccessScope: LongInt );
      function SockDesc( EventDesc: String ): String;

      function  ReceiveData( TimeoutMS: Integer ): String;
      procedure SendData  ( Txt: String );
      procedure SendResult( Txt: String );
      procedure SendQuoted( Txt: String );
      function  SendRequest( Txt: String ): String;

      procedure SendGreeting; virtual;

      function  HandleData: String; virtual;
      procedure HandleCommand( CmdLine: String ); virtual;

      procedure ClientSocketError( Sender: TObject; Socket: TCustomWinSocket;
                                   ErrorEvent: TErrorEvent; var ErrorCode: Integer);
      procedure ClientExecute; override;

      constructor Create(CreateSuspended: Boolean; ASocket: TServerClientWinSocket);
      destructor Destroy; override;
  end;

  TSrvBaseCli = class(TServerClientThread)
    protected
      IPAccessScope: LongInt;
    public
      constructor Create( CreateSuspended: Boolean;
                          ASocket: TServerClientWinSocket;
                          AIPAccessScope: LongInt ); virtual;
  end;
  
implementation

uses SysUtils, Windows, Global, cIPAccess, uTools, Winsock, uWinSock;

// ------------------------------------------------- TSrvBaseCli -----

procedure TSrvBaseCli.CheckClientAccess( IPAccessScope: LongInt );
var  raddr: LongInt;
begin
     if ClientSocket=nil then exit;

     raddr := ClientSocket.RemoteAddr.sin_addr.S_addr;
     IPaccess := IPAccessCheck.GetAccess( raddr, IPAccessScope );

     Log( LOGID_DEBUG, SockDesc( '.CheckClientAccess' )
                     + '0x'     + inttohex( ntohl(raddr), 8 )
                     + ', 0x'   + inttohex( IPAccessScope, 1 )
                     + ' -> 0x' + inttohex( IPAccess, 1 ) );
end;

function TSrvBaseCli.SockDesc( EventDesc: String ): String;
begin
     try
        if ClientSocket=nil then
           Result := Self.Classname + EventDesc + '(socket closed): '
        else
           Result := Self.Classname + EventDesc + '(' + inttostr(ClientSocket.SocketHandle)+'): ';
     except
        Result := Self.Classname + EventDesc + '(?Socket closed?): ';
     end;
end;

function TSrvBaseCli.ReceiveData( TimeoutMS: Integer ): String;
var  SocketStream: TWinSocketStream;
     Data: array[0..1023] of Char;
     ByteIn, i: Integer;
begin
   Result := '';

   SocketStream := TWinSocketStream.Create( ClientSocket, TimeoutMS );
   ByteIn := 0;

   try
      while Assigned(ClientSocket) and (ClientSocket.Connected) do begin
         if ShutDownReq or Terminated then break;
         if ClientSocket=nil then break;
         if not ClientSocket.Connected then break;

         if not SocketStream.WaitForData( TimeoutMS ) then begin
           // If we didn't get any data after ? seconds then close the connection
           Log( LOGID_WARN, 'Connection closed (timeout):' + ClientID );
           SocketStream.Free;
           if ClientSocket.Connected then ClientSocket.Close;
           Terminate;
           exit;
         end;

         FillChar( Data, SizeOf(Data), 0 );
         ByteIn := 0;
         try
            if ClientSocket.Connected then begin
               ByteIn := SocketStream.Read( Data, SizeOf(Data) );
            end;
         except
            ByteIn := 0;
         end;
         if ByteIn = 0 then break;

         {if (LOGFILEMASK and LOGID_FULL)<>0 then begin
            Log( LOGID_DEBUG, SockDesc('.Recv2') + Data );
         end;}

         Result := Result + Data;
         i := Pos( #10, Result );
         if i>=0 then break;
      end;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'ReceiveData.Exception: ' + E.Message );
      end;
   end;

   if ByteIn = 0 then begin
     Log( LOGID_WARN, TrGl (kLog, 'Connection.lost', 'Connection lost') + ':' + ClientID );
     if ClientSocket<>nil then begin
        try if ClientSocket.Connected then ClientSocket.Close; except end;
     end;
   end;

   SocketStream.Free;
end;

procedure TSrvBaseCli.SendData( Txt: String );
var  SocketStream: TWinSocketStream;
     snd: Integer;
begin
   if ClientSocket=nil then exit;

   SocketStream := TWinSocketStream.Create( ClientSocket, 60000 );

   try
      while Assigned(ClientSocket) do begin
         if not ClientSocket.Connected then break;
         if Txt='' then break;
         if ClientSocket=nil then break;
         snd := SocketStream.Write( Txt[1], length(Txt) );
         if ClientSocket=nil then break;
         if not ClientSocket.Connected then break;
         if snd>0 then System.Delete( Txt, 1, snd );
      end;
   except
      on E:Exception do
         Log( LOGID_DEBUG, SockDesc('.SendData') + E.Message );
   end;

   SocketStream.Free;
end;

procedure TSrvBaseCli.SendResult( Txt: String );
begin
   if Txt='.' then Log( LOGID_DETAIL, '< ' + Txt )
              else Log( LOGID_INFO,   '< ' + Txt );
   SendData( Txt + CRLF );
end;

procedure TSrvBaseCli.SendQuoted( Txt: String );
begin
   if ClientSocket=nil then exit;
   if copy(Txt,1,1)='.' then Txt:='.'+Txt; // quote leading dot
   {if (LOGFILEMASK and LOGID_FULL)<>0 then begin
      Log( LOGID_DEBUG, SockDesc('.SendT') + Txt );
   end;}
   SendData( Txt + CRLF );
end;

function TSrvBaseCli.SendRequest( Txt: String ): String;
begin
   SendResult( Txt );
   Result := ReceiveData( Def_LocalTimeoutInactivity*60000 );
end;

function TSrvBaseCli.HandleData: String;
begin
   Result := '';
   if ClientSocket=nil then exit;
   if not ClientSocket.Connected then exit;
   Result := '+ Data received';
   if HadLineTooLong or HadTextTooLarge then Result:='- Limit exceeded';
end;

procedure TSrvBaseCli.HandleCommand( CmdLine: String );
begin
   if ClientSocket=nil then exit;
   
   try
      if not ClientSocket.Connected then exit;

      if HadLineTooLong or HadTextTooLarge then begin
         SendResult( '- Limit exceeded' );
         exit;
      end;

      if UpperCase(CmdLine)='QUIT' then begin
         SendResult( '+ BYE' );
         Sleep( Def_LocalTimeoutQuitDelay );
         ClientSocket.Close;
         Terminate;
         exit;
      end;

      if UpperCase(CmdLine)='DATA' then begin
         WaitForCmd := False;
         SetLength(BufInStrm, 0) {.Clear}; BufInStrmLen := 0;
         SendResult( '+ Ready to receive data; end with CRLF.CRLF' );
         exit;
      end;

      if UpperCase(CmdLine)='HELP' then begin
         SendResult( '+ List follows' );
         SendQuoted( 'DATA' );
         SendQuoted( 'HELP' );
         SendQuoted( 'QUIT' );
         SendResult( '.' );
         exit;
      end;

      SendResult( '- Unknown command: ' + CmdLine );

   except
      on E: Exception do
         Log( LOGID_DEBUG, SockDesc('.HandleCommand.Exception') + E.Message );
   end;
end;

procedure TSrvBaseCli.ClientSocketError( Sender: TObject; Socket: TCustomWinSocket;
                                    ErrorEvent: TErrorEvent; var ErrorCode: Integer);
begin
     Log( LOGID_WARN, SockDesc('.ClientSocketError')
                    + inttostr(ErrorCode) + ' '
                    + WinsockErrTxt( ErrorCode ) );
     ErrorCode := 0; // mark as "handled"
     Terminate;      // terminate client-thread
end;

procedure TSrvBaseCli.ClientExecute;
var  Data: array[0..1023] of Char;
     SocketStream: TWinSocketStream;
     LineIn, Reply : String;
     LineEnd, ByteIn, i : Integer;
begin
     try
        SendGreeting;
        Log_SetTask( '(client ' + lowercase(inttohex(GetCurrentThreadID,1))+')', ClientID );
 {JW}  {CurrentThreadID}
        CurrentThreadID:=GetCurrentThreadID;
{JW} except
        on E:Exception do begin
           Log( LOGID_WARN, 'Connection failed (SendGreeting):' + ClientID );
           Log( LOGID_WARN, 'ERROR: ' + E.Message );
        end;
     end;

     while not Terminated do
     try
        if ClientSocket=nil then exit;
        if not ClientSocket.Connected then exit;
{JW} {Login}
        SocketStream := TWinSocketStream.Create( ClientSocket,
                                                 CurrentInactTimeOut );
        try
           if SocketStream.WaitForData( CurrentInactTimeOut ) then begin
{JW}
              FillChar(Data, SizeOf(Data), 0);
              ByteIn := 0;
              if ClientSocket<>nil then begin
                 try
                    if ClientSocket.Connected then begin
                       ByteIn := SocketStream.Read( Data, SizeOf(Data) );
                    end;
                 except
                    ByteIn := 0;
                 end;
              end;
              if ByteIn = 0 then begin
                Log( LOGID_WARN, TrGl (kLog, 'Connection.lost', 'Connection lost') + ':' + ClientID );
                if ClientSocket<>nil then begin
                   if ClientSocket.Connected then ClientSocket.Close;
                end;
                Terminate;
              end;

              BufInRaw := BufInRaw + Data;

              repeat
                 LineEnd  := Pos( CRLF, BufInRaw );
                 if (LineEnd=0) and (LimitLineLen>0) and (length(BufInRaw)>LimitLineLen) then begin
                    HadLineTooLong := True;
                    LineEnd := LimitLineLen + 1;
                 end;

                 if LineEnd>0 then begin

                    // ggf. virtuellen LineIn-Handler
                    LineIn := copy( BufInRaw, 1, LineEnd-1 );
                    System.Delete ( BufInRaw, 1, LineEnd+1 );

                    {if (LOGFILEMASK and LOGID_FULL)<>0 then begin
                       Log( LOGID_DEBUG, SockDesc('.Recv') + LineIn );
                    end;}

                    if WaitForCmd then begin
                       HandleCommand( LineIn );
                    end else begin
                       if LineIn='.' then begin
                          WaitForCmd := True;
                          SetLength(BufInStrm, BufInStrmLen); {TGL}
                          Reply := HandleData;
                          SetLength(BufInStrm, 0) {.Clear}; BufInStrmLen := 0;
                          if Reply<>'' then SendResult( Reply );
                       end else begin
                          if copy(LineIn,1,2)='..' then System.Delete(LineIn,1,1);
                          {if length(LineIn)>0 then}
                             If BufInStrmLen + length(LineIn) + 2 > Length(BufInStrm) then begin
                                If BufInStrmLen > 128000
                                   then SetLength(BufInStrm, BufInStrmLen + 163840)
                                   else SetLength(BufInStrm, BufInStrmLen + length(LineIn) + 2 + 16384)
                             end;
                             For i := 1 to Length(LineIn) do
                                BufInStrm[BufInStrmLen + i] := LineIn[i];
                             BufInStrm[BufInStrmLen + length(LineIn) + 1] := CRLF[1];
                             BufInStrm[BufInStrmLen + length(LineIn) + 2] := CRLF[2];
                             Inc (BufInStrmLen, length(LineIn) + 2);
                          {if length(LineIn)>0 then BufInStrm.Write( LineIn[1], length(LineIn) );
                          BufInStrm.Write( CRLF, 2 );}
                          if (LimitTextSize>0) and (BufInStrmLen{.Size}>LimitTextSize) then begin
                             HadTextTooLarge := True;
                             Reply := HandleData;
                             SetLength(BufInStrm, 0) {.Clear};
                             BufInStrmLen := 0;
                             if Reply<>'' then SendResult( Reply );
                          end;
                          {JW} {HadLineTooLong}
                          if HadLineTooLong and (Reply<>'') then SendResult( Reply );
                          {JW}
                       end
                    end;

                    if ClientSocket=nil then Terminate
                    else if not ClientSocket.Connected then Terminate;

                 end;

                 if HadLineTooLong or HadTextTooLarge then begin
                    if HadTextTooLarge then Reply:='textsize-limit exceeded'
                                       else Reply:='linelength-limit exceeded';
                    Log( LOGID_WARN, SockDesc('.Recv') + 'Connection terminated: ' + Reply );
                    if ClientSocket<>nil then begin
                       if ClientSocket.Connected then ClientSocket.Close;
                    end;
                    Terminate;
                 end;

              until (LineEnd=0) or Terminated;

           end else begin
             // If we didn't get any data after ? seconds then close the connection
             Log( LOGID_WARN, 'Connection closed (timeout):' + ClientID );
             if ClientSocket.Connected then ClientSocket.Close;
             Terminate;
           end;
        finally
           SocketStream.Free;
        end;
     except
        Terminate;
     end;
end;

procedure TSrvBaseCli.SendGreeting;
begin
     IPAccess := IPACC_ACCESS_ALL;
     SendResult( '+ Base-server ready.' );
end;

constructor TSrvBaseCli.Create(CreateSuspended: Boolean; ASocket: TServerClientWinSocket);
begin
     Log( LOGID_DEBUG, Self.Classname + '.Create' );

     inherited Create( True, ASocket );

     {JW} {Login}
     CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
     {JW}

     LimitLineLen    := 1000;
     LimitTextSize   := 1000000;
     HadLineTooLong  := False;
     HadTextTooLarge := False;
     try
        BufInRaw   := '';
        BufInStrm  := ''{TMemoryStream.Create}; BufInStrmLen := 0;
        WaitForCmd := True;
        IPAccess   := 0;
        ClientSocket.OnErrorEvent := ClientSocketError;
        ClientID   := ClientSocket.RemoteAddress
                    + ':' + inttostr( ClientSocket.LocalPort )
                    + ' (' + inttostr( ClientSocket.SocketHandle ) + ')';
        Log( LOGID_INFO, 'Client ' + ClientID + ' connected' );

        if not CreateSuspended then Resume;
     except
     end;
     Log_RemoveTask( lowercase(inttohex(GetCurrentThreadID,1)) );
end;

destructor TSrvBaseCli.Destroy;
begin
     Log_RemoveTask( '(client '+lowercase(inttohex(CurrentThreadID,1))+')'); {JW}  {CurrentThreadID}
     Log( LOGID_INFO, TrGlF(kLog, 'Client.disconnected', 'Client %s disconnected', ClientID) );
     BufInStrm:=''{.Free};
     Log( LOGID_DEBUG, Self.Classname + '.Destroy' );
     Log_RemoveTask( lowercase(inttohex(CurrentThreadID,1)) ); {JW}  {CurrentThreadID}
     inherited;
end;

// ------------------------------------------------------------- TSrvBase -----

function TSrvBase.SockDesc( Socket: TCustomWinSocket; EventDesc: String ): String;
begin
     try
        Result := Self.Classname + EventDesc + '(' + inttostr(Socket.SocketHandle)+'): ';
     except
        Result := '(?TSrvBase.SockDesc?)' + EventDesc + '(?Socket_closed?): ';
     end;
end;

procedure TSrvBase.MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                                    var SocketThread: TServerClientThread);
begin
     Log( LOGID_DEBUG, Self.Classname + '.OnGetThread' );
     SocketThread := TSrvBaseCli.Create(False, ClientSocket);
end;

constructor TSrvBase.CreateWithPort(AOwner: TComponent; AServerPort: Integer;
                                    AAddress: String );
begin
     Log( LOGID_DEBUG, Self.Classname + '.Create(' +
            AAddress+':'+inttostr(AServerPort)+')' );
     inherited Create(AOwner);
     ServerType      := stThreadBlocking;
     Port            := AServerPort;
     OnGetThread     := MyOnGetThread;
     Address         := AAddress;
end;

destructor TSrvBase.Destroy;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Destroy' );
     if Active then Close;
     inherited Destroy;
end;

// ----------------------------------------------------------------------------

{JW} {Terminate Socket}
function TSrvBase.CheckClientAccess(IPAccessScope: Integer;
  ClientSocket: TServerClientWinSocket): LongInt;
var  raddr: LongInt;
begin
     Result := 0;
     if ClientSocket=nil then exit;
     // IP-Adresse zum Socket auslesen
     raddr := ClientSocket.RemoteAddr.sin_addr.S_addr;
     // Ip-Adresse berprfen
     Result := IPAccessCheck.GetAccess( raddr, IPAccessScope );
     Log( LOGID_DETAIL, 'Check IPAccess '
                      + '0x'     + inttohex( ntohl(raddr), 8 )
                      + ', 0x'   + inttohex( IPAccessScope, 1 )
                      + ' -> 0x' + inttohex( Result, 1 ) );
end;
{JW}

end. *)
