// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cServerNNTP;

interface

uses ScktComp, cServerBase, Classes;

type
  TSrvNNTP = class( TSrvBase )
    public
      constructor Create( AOwner: TComponent );
      {procedure MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                              var SocketThread: TServerClientThread); override;}
  end;

type
  TSrvNNTPCli = class(TSrvWorkCli)
    private
      CurrentUserID  : Integer;
      CurrentUserName: String;
      CurrentGroup   : LongInt;
      CurrentArtNo   : Integer;
      UserMayPost    : Boolean;
      UserMayNewNews : Boolean;
      UserMayCancel  : Boolean;
      UserMayPeer    : Boolean;
      PeerState      : Boolean;
      StreamMode     : Boolean;
      CancelState    : Boolean; //HSR //ModeCancel
      GroupList      : TStringList;
      RecommendedMID : String;
      PermRead, PermPost: String;
      function  ReloadGroupList: Boolean;
      function  LoginUser( Password: String ): String;
      procedure SetCurrentGroup( Groupname: String );
      function  HandlePeeringData: String; {JW} {Peer}
      function  TraceContent: String; {JW} {Peer}
      function  HandleGatewayGroup(Const Article, Group: String;
         Const Moderated: boolean): String; {JW} {N2M}
      function  HandleCancelData(LineIn: String): String;
      Function ProcessCancel(Const local: boolean; Const DataBuf, DestControl,
         DestSupersedes, DestFrom, DestSender: string; var ErrorMsg: string): boolean;
    public
      procedure SendGreeting; override;
      function  HandleData: String; override;
      procedure HandleCommand( Const Command: String ); override;
      procedure ClientExecute; override; //HSR //ModeCancel
      function Cmd_ARTINFOS ( Const Cmd, Par: String ): Boolean;
      function Cmd_LASTNEXT ( Const Cmd, Par: String ): Boolean;
      function Cmd_QUIT     ( Const Par: String ): Boolean;
      function Cmd_AUTHINFO ( Const Par: String ): Boolean;
      function Cmd_POST     ( Const Par: String ): Boolean;
      function Cmd_SLAVE    ( Const Par: String ): Boolean;
      function Cmd_GROUP    ( Const Par: String ): Boolean;
      function Cmd_NEWGROUPS( Const Par: String ): Boolean;
      function Cmd_LIST     ( Const Par: String ): Boolean;
      function Cmd_LISTGROUP( Const Par: String ): Boolean;
      function Cmd_MODE     ( Const Par: String ): Boolean;
      function Cmd_DATE     ( Const Par: String ): Boolean;
      function Cmd_XOVER    ( Const Par: String ): Boolean;
      function Cmd_XPAT     ( Const Par: String ): Boolean;
      function Cmd_XHDR     ( Const Par: String ): Boolean;
      Function Cmd_STARTTLS ( Const Par: String ): Boolean;
      function Cmd_HELP     ( Const Par: String ): Boolean;
{JW} {Peer}
      function Cmd_CHECK    ( Const Par: String ): Boolean;
      function Cmd_TAKETHIS ( Const Par: String ): Boolean;
      function Cmd_IHAVE    ( Const Par: String ): Boolean;
{JW}
      function Cmd_NEWNEWS  ( Const Par: String ): Boolean; {JH}
      constructor Create( ASocket: TServerClientWinSocket;
                          Const AIPAccessScope: LongInt;
                          Const ASSLContext: Pointer ); override; {MG}{SSL}
      destructor Destroy; override;
  end;

  const
    PERM_NOTH = 0;
    PERM_READ = 1;
    PERM_POST = 2;

  function GetPermissionForGroup( Group, PermPost, PermRead: String ): LongInt;

implementation

uses SysUtils, Windows, uTools, Global, Config, cIPAccess, cArticle, cArtFiles,
     cAccount, cPCRE, uCRC32, uDateTime, cMailrouter, cStdForm,
     cActions, cLogfile;

// ---------------------------------------------------------- TSrvNNTPCli -----

function GetPermissionForGroup( Group, PermPost, PermRead: String ): LongInt;
var  rex: TPCRE;
     prs: TParser;

     function HasPermFor( Perm: String ): Boolean;
     var  i   : Integer;
          s, p: String;
     begin
          Result := False;

          prs.Parse( Perm, ' ' );
          i := 0;
          repeat
             s := prs.sPart( i, '' );
             if s<>'' then begin
                try
                   if s[1]='!' then begin
                      p := copy( s, 2, Length(s)-1 );
                      if rex.Match( PChar(p), PChar(Group) ) then break;
                   end else begin
                      if rex.Match( PChar(s), PChar(Group) ) then begin
                         Result := True;
                         break;
                      end;
                   end;
                except
                   on E:Exception do begin
                      Log( LOGID_ERROR, 'Invalid group-permission: ' + Perm
                                      + ' Msg=' + E.Message );
                      break;
                   end;
                end;

                inc( i );
             end;
          until s='';
     end;

begin
     Result := PERM_NOTH;

     try
        // optimize for some common settings
        if PermRead='' then exit; //PERM_NOTH
        if PermRead='.*' then begin
           if PermPost=''   then begin Result:=PERM_READ; exit; end;
           if PermPost='.*' then begin Result:=PERM_POST; exit; end;
        end;

        prs := TParser.Create;
        rex := TPCRE.Create( False, PCRE_CASELESS );
        if HasPermFor(PermRead) then begin
           Result := PERM_READ;
           if HasPermFor(PermPost) then Result:=PERM_POST;
        end;

        rex.Free;
        prs.Free;
     except
        on E: Exception do begin
           Log( LOGID_ERROR, 'GetPermissionForGroup: ' + E.Message );
        end;
     end;
end;

function TSrvNNTPCli.LoginUser( Password: String ): String;
begin
     Result := '503 System-error, check logfile. [0]';
     try
        SetCurrentGroup( '' );
        if Assigned(GroupList) then GroupList.Clear;
        UserMayPost := False;
        UserMayNewNews := False;
        UserMayPeer := False;
        UserMayCancel  := False; //HSR //ModeCancel
        RecommendedMID := '';
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.LoginUser-Exception #1: ' + E.Message );
           Result := '503 System-error, check logfile. [1]';
           exit;
        end;
     end;

     try
        Result := '482 Authentication rejected';
        if CurrentUserName='' then exit;

        CurrentUserID := CfgAccounts.LoginID( CurrentUserName, Password );
        if CurrentUserID=ACTID_INVALID then begin
           CurrentUserName := '';
           Result := '502 No permission';
           exit;
        end;
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.LoginUser-Exception #2: ' + E.Message );
           Result := '503 System-error, check logfile. [2]';
           exit;
        end;
     end;

     try
        PermPost := CfgAccounts.Value[ CurrentUserID, ACTP_NEWSPOST ];
        PermRead := CfgAccounts.Value[ CurrentUserID, ACTP_NEWSREAD ];
        UserMayNewNews := ( CfgAccounts.Value[ CurrentUserID, ACTP_NEWNEWS ] = '1' );
        UserMayCancel  := ( CfgAccounts.Value[ CurrentUserID, ACTP_CANCEL   ] = '1' ); //HSR //ModeCancel

     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.LoginUser-Exception #3: ' + E.Message );
           Result := '503 System-error, check logfile. [3]';
           exit;
        end;
     end;

     // authentication ok, build user-specific list of newsgroups
{JW} {Login} {Peer}
     if ReloadGroupList then begin
        Result := '281 Authentication accepted';
        CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
        UserMayPeer := CfgAccounts.Value[ CurrentUserID, ACTP_NEWSPEER ]='1';
        end
{JW}
     else Result := '503 System-error, check logfile. [4]';
end;

function TSrvNNTPCli.ReloadGroupList: Boolean;
var  i: Integer;
     g: String;
     p: LongInt;
begin
   Result := False;

   try
      if not Assigned(GroupList) then exit;
      UserMayPost := False;
      GroupList.Clear;
      for i:=0 to CfgHamster.ActiveCount-1 do begin
         g := CfgHamster.ActiveName[i];
         p := GetPermissionForGroup(g,PermPost,PermRead);
         if p<>PERM_NOTH then GroupList.AddObject( g, Pointer( p ) );
         if p=PERM_POST then UserMayPost:=True;
      end;
      Result := True;
   except
      on E:Exception do begin
         Log( LOGID_ERROR, Self.ClassName + '.ReloadGroupList: ' + E.Message );
         exit;
      end;
   end;
end;

procedure TSrvNNTPCli.SetCurrentGroup( Groupname: String );
begin
     try
        if CurrentGroup>=0 then begin
           ArticleBase.Close( CurrentGroup );
           CurrentGroup := -1;
        end;

        if Groupname='' then exit;

        CurrentGroup := ArticleBase.Open( Groupname );
        CurrentArtNo := ArticleBase.LocalMin[ CurrentGroup ];
     except
        on E:Exception do
           Log( LOGID_ERROR,
                'TSrvNNTPCli.SetCurrentGroup-Exception: ' + E.Message );
     end;
end;

{JW} {Peer}
function TSrvNNTPCli.TraceContent: String;
var  DT: TDateTime;
begin
      if Def_FQDN=''
         then Result := 'localhost'
         else Result := Def_FQDN;
      DT := NowGMT;
      Result := Result + ' ' + inttostr( DateTimeToUnixTime( DT ) )
                       + ' ' + Format( '%u', [GetCurrentThreadID] )
                       + ' ' + ClientSocket.RemoteAddress
                       + ' ('
                         + inttostr( CurrentUserID ) + ' '
                         + lowercase( inttohex(GetCurrentThreadID,1) ) + ' '
                         + FormatDateTime( 'yyyy"."mm"."dd hh":"nn":"ss', DT )
                       + ')';
end;


function TSrvNNTPCli.HandlePeeringData: String;
var  DataBuf, DestGrp, DestMid, DestLin, GrpNam, DestDate: String;
     DestPath, DestXref, DestFup2, DestControl, DestSupersedes,
        DestFrom, DestSubj, DestSender: String;
     LfdGrp, GrpHdl, KnownGroups,i: Integer;
     boDONE, OK: Boolean;
     Parser: TParser;
     Xrefs: TList;
     Xref: TXrefArtNo;
     TS: TStringList;
begin
  If StreamMode
     then Result := '439 article transfer failed'
     else Result := '437 article rejected - do not try again';

  PeerState:=False;
  boDONE := False;
  try
     EnterCriticalSection(CS_LOCK_InFeed);
     try
        if not ClientSocket.Connected then exit;
        if HadTextTooLarge then begin
           if not StreamMode then Result := '437 article rejected - do not try again (textsize-limit exceeded) '
           else Result := '439 article transfer failed';
           HadTextTooLarge:=false; {JW} {HadLineTooLong}
           exit;
        end;
        if HadLineTooLong then begin
           if not StreamMode then Result := '437 article rejected - do not try again (linelength-limit exceeded) '
           else Result := '439 article transfer failed';
           HadLineTooLong:=false; {JW} {HadLineTooLong}
           exit;
        end;
        if Length(BufInStrm){.Size}<=0 then begin
           if not StreamMode then Result := '437 article rejected - do not try again (missing ... all) '
           else Result := '439 article transfer failed';
           exit;
        end;
        DataBuf := BufInStrm {SetLength( DataBuf, BufInStrm.Size )};

//---------------------- Check and add header -------------------------------

        With TArticle.Create do try
           Text := DataBuf;
           if FullBody = '' then begin
              If not StreamMode
                 then Result := '437 article rejected - do not try again (missing body) '
                 else Result := '439 article transfer failed (missing body) ';
              exit
           end;
           DestGrp     := Header['Newsgroups:'];
           DestFup2    := Header['Followup-To:'];
           DestMid     := Header['Message-ID:'];
           DestLin     := Header['Lines:'];
           DestDate    := Header['Date:'];
           DestControl := Header['Control:'];
           DestSupersedes
                       := Header['Supersedes:'];
           DestSender  := Header['Sender:'];
           DestFrom    := Header['From:'];
           DestSubj    := Header['Subject:'];
           DestPath    := Header['Path:'];
           DeleteHeader('Xref:');
           DataBuf := Text
        finally
           Free
        end;

        // check Message-ID
        if DestMid<>'' then begin
           if NewsHistory.ContainsMID( DestMid ) then begin
              if not StreamMode then Result := '437 article rejected - do not try again (Message-ID already in history) '+DestMid
              else Result := '439 article transfer failed '+DestMid;
              exit;
           end;
        end;

        // check some headers
        if (DestFrom='')  or (DestGrp='') or (DestMid='')then begin // {JW} {DestSubj}
           if not StreamMode then Result := '437 article rejected - do not try again (missing From|Newsgroups|Message-ID)'
           else Result := '439 article transfer failed '+DestMid;
           exit;
        end;
        if DestDate<>'' then begin
           if RfcDateTimeToDateTimeGMT(DestDate)<=EncodeDate( 1980, 1, 1 ) then begin
              if not StreamMode then Result := '437 article rejected - do not try again (invalid Date) '+DestMid
              else Result := '439 article transfer failed '+DestMid;
              exit;
           end;
        end;
        if DestPath='' then begin
           if Def_Check_Path then begin {JW} {Check Path}
              if not StreamMode
                 then Result := '437 article rejected - do not try again (not Path found) '+DestMid
                 else Result := '439 article transfer failed '+DestMid;
              exit;
           end else begin
              DestPath := 'not-for-mail';
              With TArticle.Create do try
                 Text := DataBuf;
                 Header['Path:'] := DestPath;
                 DataBuf := Text
              finally
                 Free
              end;
           end; {JW}
        end else begin
           if Def_FQDN>'' then begin
              TS := TStringList.Create;
              try
                 RE_Split( DestPath, '!', 0, -1, TS );
                 for i:=0 to TS.Count-1 do begin
                    if AnsiCompareText( TS[i], Def_FQDN )=0 then begin
                       If not StreamMode then Result := '437 article rejected - do not try again (invalid "Path:") '+DestMid
                       else Result := '439 article transfer failed '+DestMid;
                       boDONE := True;
                       break;
                    end;
                 end;
                 if not boDONE then begin
                    DestPath := Def_FQDN + '!' + DestPath;
                    With TArticle.Create do try
                       Text := DataBuf;
                       If Header['Path'] > '' then begin
                          Header['Path'] := DestPath;
                          DataBuf := Text
                       end
                    finally
                       Free
                    end
                 end;
              finally
                 TS.Free;
              end;
           end else begin
              if not StreamMode then begin
                 Result := '436 local configuration error  - try again later.(No local FQDN set - expand path by peering impossible) '+DestMid;
                 Log( LOGID_ERROR, '436 local configuration error  - try again later. (No local FQDN set - unable to create PATH-Header)')
              end else begin
                 Result := '439 article transfer failed '+DestMid
              end;
              exit;
           end;
        end;
        if boDONE then Exit;

//-----------------------End check and add header ----------------------------
//-----------------------Check newsgroups-------------------------------------
        KnownGroups   := 0;
        Parser := TParser.Create;
        try
           Parser.Parse( DestGrp, ',' );
           LfdGrp := 0;
           repeat
              GrpNam := Parser.sPart( LfdGrp, '' );
              if GrpNam='' then begin
                 if LfdGrp=0 then begin
                    if not StreamMode then Result := '437 article rejected - do not try again (no newsgroups) '+DestMid
                    else Result := '439 article transfer failed '+DestMid;
                    boDONE := True;
                 end;
                 break;
              end;
              if CfgHamster.ActiveIndexOf[GrpNam]<0 then begin
                 if not IsNewsgroup( GrpNam ) then begin
                    // posting to invalid group
                    if not StreamMode then Result := '437 article rejected - do not try again (invalid newsgroup "'+GrpNam+'") '+DestMid
                    else Result := '439 article transfer failed '+DestMid;
                    boDONE := True;
                 end else begin
                    // posting to unknown group
                    Log( LOGID_DETAIL, '(Cross-) Posting to unknown group: ' + GrpNam );
                 end;
              end else begin
                 inc( KnownGroups );
                 i := GroupList.IndexOf(GrpNam);
                 if i>=0 then begin
                    if LongInt(GroupList.Objects[i])<>PERM_POST then i:=-1;
                 end;
                 if i<0 then begin
                    Log( LOGID_Detail, 'Posting to unknown group(s): ' + DestMid); {JW} {unknown Groups}
                    if not StreamMode then Result := '437 article rejected - do not try again  (no permission to transfer to '+GrpNam+') '+DestMid
                    else Result := '439 article transfer failed '+DestMid;
                    boDONE := True;
                 end;
              end;
              inc( LfdGrp );
           until False;
           {JW junk feeder}
           // if not boDONE and (KnownGroups=0) then begin
           // article feed to unknown groups only if junkfeeder requiered
           // in other case reject article
           if not boDONE and (KnownGroups=0) and (not Def_Junkfeeder) then begin
              Log( LOGID_Detail, 'Posting to unknown group(s): ' + DestMid); {JW} {unknown Groups}
              if not StreamMode then Result := '437 article rejected - do not try again (no valid groups) '+DestMid
              else Result := '439 article transfer failed '+DestMid;
              boDONE := True;
           end;
           if boDONE then exit;
//-------------------End check newsgroups-------------------------------------

           // Process Cancel, if this message is one
           If ProcessCancel(false, DataBuf, DestControl, DestSupersedes, DestFrom, DestSender, result)
              and Def_News_FeededCancelDelete
           then begin
              Exit
           end;
           
//---------------------------Save article-------------------------------------
            // Create Xref-header by opening groups and reserving article-numbers
           DestXref := '';
           Xrefs := TList.Create;
           try
              LfdGrp := 0;
              If KnownGroups > 0 then Repeat
                 GrpNam := Parser.sPart( LfdGrp, '' );
                 if GrpNam > '' then begin
                    If CfgHamster.ActiveIndexOf[GrpNam] >= 0 then begin
                       GrpHdl := ArticleBase.Open( GrpNam );
                       if GrpHdl>=0 then begin
                          Xref := TXrefArtNo.Create;
                          Xref.GrpNam := GrpNam;
                          Xref.GrpHdl := GrpHdl;
                          Xref.ArtNo  := ArticleBase.ReserveArtNo( Xref.GrpHdl );
                          Xrefs.Add( Xref );
                          DestXref := DestXref + ' ' + GrpNam + ':' + inttostr( Xref.ArtNo );
                       end;
                    end;
                    inc( LfdGrp )
                 end
              until GrpNam = '';
              {JW junk feeder}
              // generate Xref for unknown group if junk feeder requiered
              if (KnownGroups=0) and Def_Junkfeeder then begin
                 GrpNam:= INTERNALGROUP[ INTERNALGROUP_UNKNOWNGROUP ];
                 GrpHdl := ArticleBase.Open( GrpNam );
                 Xref := TXrefArtNo.Create;
                 Xref.GrpNam := GrpNam;
                 Xref.GrpHdl := GrpHdl;
                 Xref.ArtNo  := ArticleBase.ReserveArtNo( Xref.GrpHdl );
                 Xrefs.Add( Xref );
                 DestXref := DestXref + ' ' + GrpNam + ':' + IntToStr(Xref.ArtNo )
              end;
              // Note: already existing Xref would have been removed above
              DataBuf := 'Xref: ' + XREF_HOSTNAME + DestXref + #13#10 + DataBuf;
              // save article in local groups
              for LfdGrp:=0 to Xrefs.Count-1 do begin
                 Xref := TXrefArtNo( Xrefs[LfdGrp] );
                 try
                    ArticleBase.WriteArticle( Xref.GrpHdl, Xref.ArtNo, DataBuf );
                    NewsHistory.AddEntryDupes( DestMid, StrToCRC32(LowerCase(Xref.GrpNam)), Xref.ArtNo, 0 );
                    OK := True;
                 except
                    OK := False;
                 end;
                 if OK then begin
                    if not boDONE then begin
                       If not StreamMode then Result := '235 article transferred ok '+DestMid
                       else Result := '239 article transferred ok '+DestMid
                    end;
                    boDONE := True;
                 end else begin
                    if not boDONE then begin
                       if not StreamMode then Result := '437 article rejected - do not try again (couldn''t save) '+DestMid
                       else Result := '439 article transfer failed '+DestMid;
                       Log( LOGID_ERROR, 'couldn''t save the article or store to history' );
                    end;
                    boDONE := True;
                 end;
              end;
              // close groups and free Xref-entries and -list
           finally
              for LfdGrp:=Xrefs.Count-1 downto 0 do begin
                 Xref := TXrefArtNo( Xrefs[LfdGrp] );
                 if Assigned(Xref) then begin
                    ArticleBase.Close( Xref.GrpHdl );
                    Xref.Free;
                 end;
              Xrefs.Delete( LfdGrp );
              end;
              Xrefs.Free;
           end;
        finally
           Parser.Free;
        end;
        if not boDONE then begin
           If not StreamMode
              then Result := '437 article rejected - do not try again'
              else Result := '439 article transfer failed '+DestMid
        end
//-----------------------End save article-------------------------------------
     except
        on E:Exception do
           Log( LOGID_ERROR, 'TSrvNNTPCli.HandlePeeringData.Exception: ' + E.Message );
     end;
  finally
     LeaveCriticalSection(CS_LOCK_InFeed);
  end
end;
{JW} {N2M}

// newsgroups to mail gateway
function TSrvNNTPCli.HandleGatewayGroup(Const Article, Group: String;
   Const Moderated: boolean): String; {JW} {N2M}
var  GrpHdl: LongInt; s, Receiver: String;
begin
   With TRouter.Create do try
      MailText.Text:=Article; // copy messages body
      MailFrom:=MailText['From'];  // copy E-Mail adresss sender
      // if group found read property
      GrpHdl := ArticleBase.Open( Group );
      if GrpHdl>=0 then begin
         try
            Receiver:=ArticleBase.Moderator[GrpHdl]; // read e-mail adresse
            // if destination E-Mail address not empty then add to mail
            if receiver > '' then begin
               MailText.AddHeader('To:', Receiver);
            end else begin
               Log( LOGID_Error,'missing moderator in moderated group or gateway address' );
               result:='441 posting failed (missing moderator in moderated group or gateway address)';
               Exit
            end
         finally
            ArticleBase.Close( GrpHdl )
         end
      end else begin
        Log( LOGID_Error,'Unable to open group '+Group+' and find moderator address' );
        result:='441 posting failed (unable to open group '+Group+' and find moderator address)';
        exit
      end;
      // if gate Article then add gate header
      If Not Moderated then begin
         With Mailtext do begin
            // save newsgroups header to x-header
            RenameHeader('Newsgroups', 'X-Newsgroups');
            // save path header to x-header
            RenameHeader('Path', 'X-Path');
            // Message-ID sicherheitshalber anpassen
            If Def_GatewayMIDExtension > '' then begin
               s := Header['Message-ID'];
               If s > '' then begin
                  If s[1] = '<'
                     then s := '<' + Def_GatewayMIDExtension + Copy(s, 2, Length(s)-1)
                     else s := Def_GatewayMIDExtension + s;
                  Header['Message-ID'] := s
               end
            end
         end
      end;
      // compose mail
      MailTo.Add(receiver);
      MailType := mtNews2Mail;
      MailSource := msOutgoing;
      InIP    := ClientSocket.RemoteAddr.sin_addr.S_addr;
      InIpStr := ClientSocket.RemoteAddress;
      UserID  := CurrentUserID;
      // send mail with mailrouter
      if Execute then begin
         result:='';
         Log( LOGID_INFO, TrGlF(kLog, 'Info.News2Mail.Done',
              'News for group %s was gated to mail address %s', [group, receiver] ) );
      end else begin
         result:='441 posting failed when trying to send as mail to '+receiver
      end
   finally
      Free
   end
end;
{JW}

function TSrvNNTPCli.HandleData: String;
var  Parser: TParser;
     Xrefs: TList;
     Xref: TXrefArtNo;
     TS, Gateways: TStringList;
     DestLines, LfdGrp, GrpHdl, KnownGroups, UnknownGroups, p, i, idx: Integer;
     boDONE, HasLocal, HasRemote, HasFeedOnly, FeedOnly, OK, Moderated: Boolean;
     GrpType: Char;
     DataBuf, DestGrp, DestMid, DestLin, GrpNam, DestDate,
       DestPath, DestRef, DestXref, DestFup2, DestControl, DestSupersedes,
       DestFrom, DestSubj, DestSender, DestApproved,
       CancelMID, ModeratedGroup, s: String;
     ErrorMsg: String;
begin
   ErrorMsg := '441 posting failed (unknown reason, see logfile)';
   moderated:=False;
   feedonly:=false;
   HasFeedOnly:=false;
   try
      // if client a peer for feeding then activate the
      // special function to handle peering
      If PeerState then begin
         ErrorMsg := HandlePeeringData;
         exit;
      end;
      try
         //------------------ check header and formats ---------------------------------
         // Check if client live
         if not ClientSocket.Connected then exit;
         // Check Length of article
         if HadTextTooLarge then begin
            ErrorMsg := '441 posting failed (textsize-limit exceeded)';
            exit;
         end;
         // check has article lines with more as 998 character
         if HadLineTooLong then begin
            ErrorMsg := '441 posting failed (linelength-limit exceeded)';
            exit;
         end;
         // Error, if Length of article 0
         if Length(BufInStrm)<=0 then begin
            ErrorMsg := '441 posting failed (missing ... all)';
            exit;
         end;
         // execute a a action if exist
         If Actions.Exists ( atNewsPreprocess ) then begin
            Case ModifyMessage ( atNewsPreprocess, BufInStrm, s ) of
               mcOriginal: ;
               mcChanged: BufInStrm := s;
               mcDeleted: begin ErrorMsg := '240 article deleted by news.preprocess'; Exit end;
            end
         end;
         // Check if boddy exist JW 22.03.02
         if pos(CRLF+CRLF,BufInStrm)=0 then begin
            ErrorMsg := '441 posting failed (missing header/body-separator)';
            exit;
         end;
         // Copy full Article into DataBuf
         DataBuf := BufInStrm;
         // Create Article-Object
         With TArticle.Create do try
            Text := DataBuf;
            //  Check if article has RFC1036 format
            if FullBody = '' then begin
               ErrorMsg := '441 posting failed (missing body)';
               exit;
            end;
            //  add X-Posting-Agent header
            if Def_News_AddXHamster then begin
               If Header['X-Posting-Agent']='' then Header['X-Posting-Agent'] := OUR_VERINFO
            end;
            //  add Trace Header
            if Def_News_AddXHTrace > '' then begin
               RenameHeader( Def_News_AddXHTrace, 'X-Old-'+Def_News_AddXHTrace );
               AddHeaderFirst(Def_News_AddXHTrace, TraceContent )
            end;
            // add or extend User-Agent Header
            if Def_News_AddUserAgent then begin
                Header['User-Agent:'] := Trim(Header['User-Agent:']
                                         + ' Hamster/'+GetExeVersion)
            end;
            // Copy several Header-lines
            DestGrp     := Header['Newsgroups:'];
            DestFup2    := Header['Followup-To:'];
            DestMid     := Header['Message-ID:'];
            DestRef     := Header['References:'];
            DestLin     := Header['Lines:'];
            DestDate    := Header['Date:'];
            DestLines   := CountLines(FullBody);
            DestControl := Header['Control:'];
            DestSupersedes
                        := Header['Supersedes:'];
            DestFrom    := Header['From:'];
            DestSubj    := Header['Subject:'];
            DestPath    := Header['Path:'];
            DestSender  := Header['Sender:'];
            DestApproved:= Header['Approved:'];
            // delete Xref-Header if exist
            DeleteHeader ('Xref:');
            // recopy  article to buffer
            DataBuf := Text
         finally
            Free
         end;
         // From-, Newsgroup- and Subject-Header must exists else reject article
         if (DestFrom='') or (DestSubj='') or (DestGrp='') then begin
            ErrorMsg := '441 posting failed (missing From|Subject|Newsgroups)';
            exit;
         end;
         //  check format of From-Header
         i := Pos( '@', DestFrom );
         if i=0 then begin
            ErrorMsg := '441 posting failed (invalid From, missing "@")';
            exit;
         end else begin
            s := copy( DestFrom, i+1, Length(DestFrom)-i );
            if Pos( '.', s ) = 0 then begin
               ErrorMsg := '441 posting failed (invalid From, missing "." in domain-part)';
               exit
            end
         end;
         // check Date format
         if DestDate<>'' then begin
            if RfcDateTimeToDateTimeGMT(DestDate)<=EncodeDate( 1980, 1, 1 ) then begin
               ErrorMsg := '441 posting failed (invalid Date)';
               exit;
            end;
         end;
         // If MID not exist and recommended MID exist then add recommended MID
         if (DestMid='') and (RecommendedMID>'') then begin
            DestMid := RecommendedMID;
            RecommendedMID := '';
            DataBuf := 'Message-ID: ' + DestMid + #13#10 + DataBuf
         end;
         //  If MID not exist then check if MID required and add
         if DestMid='' then begin
            // local injetion required MID always
            if (Def_GenerateNewsMID<>GENERATEMID_NEVER) or Def_News_LocalInjection then begin
               // If exist FQDN then generate MID else reject article
               if Def_FQDNforMIDs<> '' then begin
                  DestMid := MidGenerator( Def_FQDNforMIDs )
               end else begin
                  ErrorMsg := '436 local configuration error  - try again later. (no local FQDN set - unable to create MID)';
                  Log( LOGID_ERROR, '436 local configuration error  - try again later. (No local FQDN set - unable to create MID)');
                  exit
               end;
               //  add MID to Article
               DataBuf := 'Message-ID: ' + DestMid + #13#10 + DataBuf;
            end
         end;
         //  check if article alread exists in history
         If (DestMid > '') and NewsHistory.ContainsMID( DestMid ) then begin
            ErrorMsg := '441 posting failed (Message-ID already in history)';
            exit
         end;

         // ----------- Check group list, News2Mail-Gateway etc ------------------
         boDONE        := False;
         HasLocal      := False;
         HasRemote     := False;
         KnownGroups   := 0;
         UnknownGroups := 0;
         Parser := TParser.Create;
         Gateways := TStringList.Create;
         try
            Parser.Parse( DestGrp, ',' );
            LfdGrp := 0;
            // ------------  Count groups, check for moderated groups etc  ---------------
            repeat
               GrpNam := Parser.sPart( LfdGrp, '' );
               If GrpNam > '' then begin
                  // count unknown newsgroups to prevent exezessive crossposts
                  if CfgHamster.ActiveIndexOf[GrpNam]<0 then begin
                     // check format of name of newsgroups and wheater it is a known local group
                     If IsNewsgroup( GrpNam ) then begin
                        // check if post to unknown groups allow
                        if Def_LocalNntpPostToUnknown then begin
                           inc( UnknownGroups );
                           Log( LOGID_WARN, '(Cross-)Posting to unknown group: ' + GrpNam );
                        end else begin
                           ErrorMsg := '441 posting failed (unknown newsgroup '+GrpNam+')';
                           boDONE := True;
                        end
                     end else begin
                        // posting to invalid group
                        ErrorMsg := '441 posting failed (invalid newsgroup "'+GrpNam+'")';
                        boDONE := True
                     end;
                  end else begin
                     // count known group
                     inc( KnownGroups );
                     // check if group by use on feeder only
                     feedonly:=false;
                     GrpHdl := ArticleBase.Open( GrpNam );
                     if GrpHdl>=0 then try
                        feedonly := (ArticleBase.GetProp( GrpHdl, 'feedonly', '0' )='1')
                     finally
                        ArticleBase.Close( GrpHdl )
                     end;
                     // if group by use on feeder only set the flag for "save article
                     // only to local groups"
                     if feedonly then begin
                        // save article for feeder groups only local
                        HasLocal :=True;
                        HasRemote:=False;
                        HasFeedOnly:=True;
                     end else begin
                        // if group not "feed olnly" group then test if
                        // group local with check if group in active listing
                        if CfgHamster.ActivePostServer[GrpNam]=''
                           then HasLocal :=True
                           else HasRemote:=True
                     end;
                     // check the permission to post for group, i=-1 means no permission
                     i := GroupList.IndexOf(GrpNam);
                     if (i>=0) and (LongInt(GroupList.Objects[i])<>PERM_POST) then begin
                        i:=-1
                     end;
                     // if group not found or no permission to post then reject article
                     if i<0 then begin
                        ErrorMsg := '440 posting failed (no permission to post to '+GrpNam+')';
                        boDONE := True;
                     end;
                     // check if group read only, gateway or moderated group
                     GrpHdl := ArticleBase.Open( GrpNam );
                     if GrpHdl>=0 then try
                        GrpType := (ArticleBase.GetProp( GrpHdl, 'type', '' )+'y')[1];
                        Case GrpType of
                           'n': begin // group is read only
                                   ErrorMsg:='440 posting failed (posting not allowed to group ' +GrpNam+')';
                                   boDONE := True;
                                end;
                           'g': begin // group is a news2mail-gateway
                                   dec( KnownGroups ); // for correct x-post handling!
                                   Gateways.Add (GrpNam) // add to list
                                end;
                           'm': begin // group is moderated
                                   If DestApproved='' then begin // gate only not approved messages
                                      // for correct x-post handling!
                                      dec( KnownGroups );
                                      // only first moderated group is used
                                      If Not moderated then begin
                                         ModeratedGroup:=GrpNam;
                                         Moderated := true
                                      end
                                   end
                                end;
                        end { Case }
                     finally
                        ArticleBase.Close( GrpHdl )
                     end; {if GrpHdl>=0 }
                  end;  // known group
                  inc( LfdGrp )
               end; // GrpNam > ''
            until GrpNam = '';
            // reject article if newsgroup header empty
            If LfdGrp=0 then begin
               ErrorMsg := '441 posting failed (no newsgroups)';
               boDONE := True
            end;

            //------ handle moderated groups and news to mails gateway ---------

            // check if article go to moderation or in gate to E-Mail
            // and is not failed by newsgroups header check
            If (moderated or (Gateways.Count > 0)) and (not boDONE) then begin
               // check if group for gate to E-mail available
               If Gateways.Count > 0 then begin
                  // loop over all groups will gate to E-Mail
                  For i := 0 to Gateways.Count-1 do begin
                     s :=  HandleGatewayGroup(DataBuf, Gateways[i], false); // activate news to mails gateway
                     if s > '' then begin
                        Log( LOGID_WARN, 'News to Mail Gateway terminate with error');
                        ErrorMsg:=s;
                        exit;
                     end
                  end;
                  // delete gated groups from newsgroups header and list
                  With TArticle.Create do try
                     Text := DataBuf;
                     s := Header['Newsgroups:'];
                     if s > '' then begin
                        s := ','+s+',';
                        // delete space character from newsgroups header
                        Repeat
                           p := Pos(', ', s);
                           If p > 0 then System.Delete(s, p+1, 1)
                        until p = 0;
                        // delete group from list of gateways
                        For i := 0 to Gateways.Count-1 do begin
                           p := Pos(','+Gateways[i]+',', s);
                           If p > 0 then System.Delete(s, p+1, Length(Gateways[i])+1)
                        end;
                        // Set changed newsgroups header
                        Header['Newsgroups:'] := Copy(s, 2, Length(s)-2)
                     end;
                     // recopy temporary buffer
                     DataBuf := Text
                  finally Free end
               end;
               // check if moderated group availlable
               if moderated then begin
                  // activate news to mails gateway
                  s :=  HandleGatewayGroup(DataBuf,ModeratedGroup,true);
                  if s > '' then  begin
                     Log( LOGID_WARN, 'Gateway for moderated groups terminate with error');
                     ErrorMsg:=s;
                     exit
                  end
               end;
               // if no "normal" group exists or post to moderated group then terminate
               if moderated or (KnownGroups=0) then  begin
                  ErrorMsg := '240 article sent as e-mail';
                  exit;
               end
            end;

            // ----------------- check crosspost and path header -------------------------
            // reject article with contain only unknown groups
            if not boDONE and (KnownGroups=0) then begin
               ErrorMsg := '441 posting failed (no valid group(s))';
               Log( LOGID_WARN, 'reject article because newsgroups-header contains no valid group(s)');
               boDONE := True;
            end;
            // reject crosposts without followup header
            if not boDONE and (UnknownGroups>2) then begin
               if ((DestFUp2='') or (Pos(',',DestFup2)>0)) and
                  ((DestRef = '') or (UnknownGroups > Def_MaxUnknownGroupsInRe))
               then begin
                  Log( LOGID_WARN, 'reject article with crosspost and without FollowUp-To header');
                  ErrorMsg := '441 posting failed (crossposted to more than '
                      +IntToStr(Def_MaxUnknownGroupsInRe)+' unknown groups without FollowUp-To)';
                  boDONE := True;
               end;
            end;
            // reject crosspost over more than five groups
            if not boDONE and (KnownGroups+UnknownGroups>5) then begin
               if (DestFUp2='') or (Pos(',',DestFup2)>0) then begin
                  Log( LOGID_WARN,'reject excessive crosspost to more than 5 groups');
                  ErrorMsg := '441 posting failed (crossposted to more than 5 groups)';
                  boDONE := True;
               end else begin
                  Log( LOGID_WARN, 'Was this mega-crossposting REALLY necessary?! :-(' );
               end;
            end;
            // terminate if error state available
            if boDONE then exit;

            // if local injetion required and article go to upstream then
            // add flag vor local save
            if Def_News_LocalInjection and HasRemote then HasLocal := True;
            // add Path header if optionally configured or required for
            // local injection or feeding only groups
            if (Def_News_AddPath and (Def_FQDN>'')) or Def_News_LocalInjection or HasFeedOnly then begin
               // if path empty add spoiler and FQDN
               if DestPath='' then begin
                  DestPath := 'not-for-mail';
                  // check FQDN available then add FQDN or reject article
                  if Def_FQDN<>'' then begin
                     DestPath:=Def_FQDN+'!'+DestPath
                  end else begin
                     ErrorMsg := '436 local configuration error  - try again later. (No local FQDN set - unable to create PATH-Header)';
                     Log( LOGID_ERROR, '436 local configuration error  - try again later. (No local FQDN set - unable to create PATH-Header)');
                     exit
                  end;
                  // add Path-Header to article
                  DataBuf := 'Path: ' + DestPath + #13#10 + DataBuf;
               end else begin
                  // if Path header not empty then check if contain path
                  // header the own FQDN, in this case reject article
                  TS := TStringList.Create;
                  try
                     RE_Split( DestPath, '!', 0, -1, TS );
                     for i:=0 to TS.Count-1 do begin
                        //  if found own FQDN reject article
                        if AnsiCompareText( TS[i], Def_FQDN )=0 then begin
                           Log( LOGID_ERROR, 'Article contains own FQDN in PATH-Header, message loop detected!');
                           ErrorMsg := '441 posting failed (invalid "Path:")';
                           boDone := true;
                           break
                        end;
                     end;
                  finally
                     TS.Free;
                  end;
                  // terminate function if error detected
                  if boDone then exit;
                  // check if FQDN for Path expansion available
                  if Def_FQDN<>'' then begin
                     DestPath := Def_FQDN + '!' + DestPath;
                     With TArticle.Create do try
                        Text := DataBuf;
                        If Header['Path:']>'' then begin
                           Header['Path:'] := DestPath;
                           DataBuf := Text
                        end
                     finally Free end;
                  end else begin
                     ErrorMsg := '436 local configuration error  - try again later. (No local FQDN set - unable to expand PATH-Header)';
                     Log( LOGID_ERROR, '436 local configuration error  - try again later. (No local FQDN set - unable to create PATH-Header)');
                     exit
                  end
               end
            end;

            //----------- remote post - save groups in News.out directory------------------
            LfdGrp := 0;
            // loop over all groups remotelly and locally !
            If HasRemote and (Not boDone) then begin
               Repeat
                  GrpNam := Parser.sPart( LfdGrp, '' );
                  if GrpNam > '' then begin
                     // save group only if server with group with downstream found
                     if (CfgHamster.ActivePostServer[GrpNam]<>'') then begin
                        // save article
                        if SaveUniqueNewsMsg( PATH_NEWS_OUT, DataBuf, Def_GenerateNewsMID ) then begin
                           IncCounter( CntOutboxChk, 1 );
                           // generate exit message
                           if DestMid='' then ErrorMsg:='240 article posted ok'
                                         else ErrorMsg:='240 article posted ok '+DestMid;
                           // if local injection not use then set exit state
                           if not Def_News_LocalInjection
                              then boDONE := True
                              else break;
                        end else begin
                           Log( LOGID_ERROR, 'Internal error when saving article in News.Out');
                           ErrorMsg := '441 posting failed (couldn''t save in News.Out)';
                           boDONE := True;
                        end;
                     end;
                     inc( LfdGrp )
                  end;
               until boDone or (GrpNam = '')
            end;
            if boDONE then  exit;

            // ------------------- locally article save in database ----------------------
            if HasLocal then begin
               // add Hamster's info header
               DataBuf := OUR_X_HEADER + ' Received=' + DateTimeToTimeStamp(Now) + #13#10 + DataBuf;
               // add Lines-Header
               if DestLin='' then begin
                  DataBuf := 'Lines: ' + inttostr(DestLines) + #13#10 + DataBuf;
               end;
               // add Date Header
               if DestDate='' then begin
                  DestDate := DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone );
                  DataBuf := 'Date: ' + DestDate + #13#10 + DataBuf;
               end;
               // Message-ID header empty then add or reject
               if DestMid='' then begin
                  // if using feeder and not FQDN available reject message
                  if HasFeedOnly and (Def_FQDN='')  then begin
                     ErrorMsg := '436 local configuration error  - try again later. (no local FQDN set - unable to generate MID, article cannot be stored in feed-only-group)';
                     Log( LOGID_ERROR, '436 Local configuration error  - try again later. (No local FQDN set - unable to create MID, article cannot be stored in feed-only-group)');
                     exit;
                  end else begin
                     // generate Message-ID
                     if Def_FQDNforMIDs>''
                        then DestMid := MidGenerator( Def_FQDNforMIDs )
                        else DestMid := MidGenerator( 'hamster.local.invalid' );
                     DataBuf := 'Message-ID: ' + DestMid + #13#10 + DataBuf;
                  end
               end;
               // if Path emtpty them expand path
               // Notice: By use HasFeedonly and Localinjetion is not empty
               // in all case. we have check it on other point of this function
               if DestPath='' then begin
                  // add spoiler
                  DestPath := 'not-for-mail';
                  // add fqdn if available
                  if Def_FQDN<>'' then begin
                     DestPath:=Def_FQDN+'!'+DestPath;
                     DataBuf := 'Path: ' + DestPath + #13#10 + DataBuf;
                  end
               end else
               // check Path header if use own FQDN if not already down
               if ((not Def_News_AddPath) or (Def_FQDN=''))
                  and (not Def_News_LocalInjection)
                  and (not HasFeedOnly) then
               begin
                  // make it only if FQDN available
                  if Def_FQDN > '' then begin
                     // Search for FQDN in Path
                     TS := TStringList.Create;
                     try
                        RE_Split( DestPath, '!', 0, -1, TS );
                        for i:=0 to TS.Count-1 do begin
                           if AnsiCompareText( TS[i], Def_FQDN )=0 then begin
                              boDONE := True;
                              break;
                           end;
                        end
                     finally
                        TS.Free;
                     end;
                     // if FQDN not found then expand PATH
                     if not boDONE then begin
                        DestPath := Def_FQDN + '!' + DestPath;
                        With TArticle.Create do try
                           Text := DataBuf;
                           If Header['Path:']>'' then begin
                              Header['Path:'] := DestPath;
                              DataBuf := Text
                           end
                        finally
                           Free;
                        end
                     end {not boDONE}
                  end {Def_FQDNforMID>''}
               end
            end; {HasLocal}

            // Execute cancel Messages locally
            If HasLocal then begin
               If ProcessCancel(true, DataBuf, DestControl, DestSupersedes, DestFrom, DestSender, ErrorMsg) then begin
                  if (not HasFeedOnly) and (not Def_News_Localinjection) then exit
               end
            end;

            // Action for groups go locally
            If Actions.Exists ( atNewsLocal ) then begin
               Case ModifyMessage( atNewslocal, DataBuf, s) of
                  mcOriginal: ;
                  mcChanged: DataBuf := s;
                  mcDeleted: exit
               end
            end;

            //  ---------- Generate Xref Header and save article in local groups ----------
            DestXref := '';
            Xrefs := TList.Create;
            try
               // repeat parsing of newsgroups, the "Newsgroup:" header
               // can changed by gateway
               With TArticle.Create do try
                  Text := DataBuf;
                  Parser.Parse( Header['Newsgroups:'], ',' );
               finally
                  Free;
               end;
               LfdGrp := 0;
               Repeat
                  GrpNam := Parser.sPart( LfdGrp, '' );
                  if GrpNam > '' then begin
                     // save groups only locally or by using local-injection feature
                     if (CfgHamster.ActivePostServer[GrpNam]='')
                        or Def_News_LocalInjection or feedonly
                     then begin
                        If CfgHamster.ActiveIndexOf[GrpNam] >= 0 then begin
                           GrpHdl := ArticleBase.Open( GrpNam ); // Group is closed later
                           if GrpHdl>=0 then begin
                              // create a XRef object and add to xrefs list
                              Xref := TXrefArtNo.Create;
                              Xref.GrpNam := GrpNam;
                              Xref.GrpHdl := GrpHdl;
                              Xref.ArtNo  := ArticleBase.ReserveArtNo( Xref.GrpHdl );
                              Xrefs.Add( Xref );
                              DestXref := DestXref + ' ' + GrpNam + ':' + inttostr( Xref.ArtNo )
                           end else begin
                              Log( LOGID_ERROR, 'internal error when saving article to local group '+GrpNam);
                              ErrorMsg := '441 local-posting failed (internal error, couldn''t open group)';
                              exit;
                           end
                        end else begin
                           ErrorMsg := '441 local-posting failed (internal error, couldn''t found group)';
                           exit;
                        end;
                     end;
                     inc( LfdGrp )
                  end
               until GrpNam = '';
               // Note: already existing Xref would have been removed above
               DataBuf := 'Xref: ' + XREF_HOSTNAME + DestXref + #13#10 + DataBuf;
               // Loop for save article locally
               for LfdGrp:=0 to Xrefs.Count-1 do begin
                  Xref := TXrefArtNo( Xrefs[LfdGrp] );
                  OK:=True;
                  // Save article
                  try
                     ArticleBase.WriteArticle( Xref.GrpHdl, Xref.ArtNo, DataBuf );
                  except
                     OK := False;
                     ErrorMsg := '441 local-posting failed (couldn''t save)';
                     Log( LOGID_ERROR, 'Article itself or history-entry couldn''t be saved' );
                     boDONE := True;
                  end;
                  // Save in history
                  if OK then try
                     NewsHistory.AddEntryDupes( DestMid, StrToCRC32(LowerCase(Xref.GrpNam)), Xref.ArtNo, 0 );
                  except
                     OK := False;
                     ErrorMsg := '441 local-posting failed (couldn''t save)';
                     Log( LOGID_ERROR, 'couldn''t add the Messages-ID to History' );
                     boDONE := True;
                  end;
                  // Saved!
                  if OK then begin
                     Log( LOGID_DETAIL, 'Article saved in local group' );
                     if not boDONE then ErrorMsg := '240 article posted local';
                     boDONE := True
                  end
               end
            finally
               // close groups and free Xref-entries and -list
               for LfdGrp:=Xrefs.Count-1 downto 0 do begin
                  Xref := TXrefArtNo( Xrefs[LfdGrp] );
                  if Assigned(Xref) then begin
                     ArticleBase.Close( Xref.GrpHdl );
                     Xref.Free;
                   end;
                  Xrefs.Delete( LfdGrp );
               end;
               Xrefs.Free
            end;
         finally
            Gateways.Free;
            Parser.Free;
         end;
         if not boDONE then ErrorMsg := '441 posting failed'
      except
         on E:Exception do begin
            Log( LOGID_ERROR, 'TSrvNNTPCli.HandleData.Exception: ' + E.Message );
            ErrorMsg := '441 posting failed (Exception: '+E.Message+')'
         end
      end
   finally
      Result := ErrorMsg
   end
end;

Function TSrvNNTPCli.ProcessCancel(Const local: boolean; Const DataBuf, DestControl,
   DestSupersedes, DestFrom, DestSender: string; var ErrorMsg: string): boolean;
Var s, CancelMID: String;
begin
   result := false;  // HB
   If DestSupersedes > '' then
      CancelMID := DestSupersedes
   else
   if Pos( 'cancel', Lowercase(DestControl) )=1 then begin
      CancelMID := TrimWhSpace( copy(DestControl, 8, Length(DestControl)-7) );
      result := true; // this is a cancel-message
   end
   else Exit;

   // process cancels?
   if (not local) and (not Def_News_FeededCancel) then exit;

   Log( LOGID_INFO, 'Check cancel message');
   With TArticle.Create do try
      Text := ArticleBase.ReadArticleByMID( CancelMID );
      // Article of Cancel-Message found?
      If HeaderCount = 0 then begin
         if local then
            ErrorMsg := '441 local-cancel failed (MID not found)'
         else begin
            // no error-message; article will be stored normally (without processing)
            Log( LOGID_DETAIL, 'cancel for unknown article' + CancelMid);
            // Original-Nachricht als Bekannt abweisen,
            // wenn sie spaeter angeboten wird
            if not Def_News_FeededCancelVerify then begin
               NewsHistory.AddEntryDupes(CancelMID, 0, 0, 0)
            end
         end;
         exit; // do not delete anything
      end else begin
         Log( LOGID_DETAIL, 'Cancel: Message '+CancelMID+' found');
         // veriy_cancel
         if local or Def_News_FeededCancelVerify then
            if (ParseMail(Header['From:']) <> ParseMail(DestFrom))
            and (ParseMail(Header['Sender:']) <> ParseMail(DestSender))
            and (ParseMail(Header['From:']) <> ParseMail(DestSender)) then begin
               Log( LOGID_DETAIL, 'Sender-mismatch on cancel: '
                         +'Article-From: ' + Header['From:'] + ']'
                         +'Article-Sender: ' + Header['Sender:'] + ']'
                         +'Cancel-From: [' + DestFrom + ']'
                         +'Cancel-Sender: [' + DestSender + ']');
               if local then
                  ErrorMsg := '441 local-cancel failed (From-mismatch)';
               exit; // do not delete anything
         end;
         // is this a control-message and we are not allowed to cancel control messages?
         if (not Def_News_FeededCancelControlMsg) and (Header['Control:'] <> '') then begin
            Log( LOGID_DETAIL, 'Cancel: Message '+CancelMID+' not processed because original article is control-message.');
         end;
      end;

      // If article found then delete it in database
      if ArticleBase.DeleteArticleByMID( CancelMID, false ) then begin
         // log the job to internal group and log file
         Log( LOGID_INFO, 'message '+CancelMID+' cancelled from'+Header['From:']);
         if local then begin
            ErrorMsg := '240 local article cancelled';
            // save cancel message for admin
            s := 'Note: Message ' + CancelMID + ' was cancelled.' + #13#10;
            s := s + #13#10 + 'Headers of cancelled-message:' + #13#10#13#10;
            s := s + QuotedHeader('> ');
            Text := DataBuf;
            s := s + #13#10 + 'Cancel-message:' + #13#10#13#10;
            s := s + QuotedText(': ');
            SaveInInternalGroup( INTERNALGROUP_CANCELNOTIFY, '[Hamster] Cancel', s );
         end;
      end else begin
         Log( LOGID_ERROR, 'cancel message '+CancelMID+' failed by delete in database');
         ErrorMsg := '441 local-cancel failed (DelByMid)'
      end;
   finally
      Free
   end;
end;

procedure TSrvNNTPCli.HandleCommand( Const Command: String );
var  LogCmdLine, Cmd, Par: String;
     j: Integer;
begin
     try
        if not ClientSocket.Connected then exit;

        // Extract command
        j := PosWhSpace( Command );
        if j=0 then begin
           Cmd := UpperCase( Command );
           Par := '';
        end else begin
           Cmd := UpperCase  ( copy( Command, 1, j-1 ) );
           Par := TrimWhSpace( copy( Command, j+1, 512 ) );
        end;

        if (Cmd='AUTHINFO') and (UpperCase(copy(Par,1,4))='PASS') then begin
           LogCmdLine := 'AUTHINFO PASS [...]';
        end else begin
           LogCmdLine := Command;
        end;

        Log( LOGID_INFO, '> ' + LogCmdLine );
        if Command='' then exit;

        if HadTextTooLarge then begin
           SendResult( '503 Textsize-limit exceeded' );
           exit;
        end;
        if HadLineTooLong then begin
           SendResult( '503 Linelength-limit exceeded' );
           exit;
        end;

        // commands (no authentication required)
        if Cmd='DATE'     then if Cmd_DATE    ( Par ) then exit;
        if Cmd='HELP'     then if Cmd_HELP    ( Par ) then exit;
        if Cmd='QUIT'     then if Cmd_QUIT    ( Par ) then exit;
        if Cmd='STARTTLS' then if Cmd_STARTTLS( Par ) then exit;

        // refuse connections without TLS
        if (Def_LocalNntpTlsMode=2) and not Assigned(SSLConnection) then begin
           SendResult( '480 TLS connection required - try STARTTLS' );
           exit;
        end;

        if Cmd='AUTHINFO' then if Cmd_AUTHINFO( Par ) then exit;
        if Cmd='MODE'     then if Cmd_MODE    ( Par ) then exit;
        if Cmd='SLAVE'    then if Cmd_SLAVE   ( Par ) then exit;

        if (not Def_ListRequireAuth) and (Cmd='LIST') then begin
           if Cmd_LIST ( Par ) then exit
        end;

       {JW}  {Peer}
        if (Cmd='IHAVE')    and not Def_IHAVE_Auth then if Cmd_IHAVE   ( Par ) then exit;
        if (Cmd='TAKETHIS') and not Def_IHAVE_Auth then if Cmd_TAKETHIS( Par ) then exit;
        if (Cmd='CHECK')    and not Def_IHAVE_Auth then if Cmd_CHECK   ( Par ) then exit;

        // check authentication
{JW} {Peer}
        if CurrentUserID=ACTID_INVALID then begin
           if  ((cmd='LIST') and Def_ListRequireAuth)
               or (Pos( '|'+Cmd+'|',
                   '|ARTICLE|BODY|GROUP|HEAD|LAST|LIST|LISTGROUP|'
                   + 'NEWGROUPS|NEXT|POST|STAT|XHDR|XOVER|'
                   + 'XPAT|NEWNEWS|IHAVE|')>0)
           then begin
              SendResult( '480 Authentication required' );
              exit;
           end;
        end;

        // commands (authentication required)
        if Cmd='ARTICLE'   then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='BODY'      then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='GROUP'     then if Cmd_GROUP    (      Par ) then exit;
        if Cmd='HEAD'      then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='LAST'      then if Cmd_LASTNEXT ( Cmd, Par ) then exit;
        If(Cmd='LIST') and Def_ListRequireAuth
                           then if Cmd_LIST     (      Par ) then exit;
        if Cmd='LISTGROUP' then if Cmd_LISTGROUP(      Par ) then exit;
        if Cmd='NEWGROUPS' then if Cmd_NEWGROUPS(      Par ) then exit;
        if Cmd='NEXT'      then if Cmd_LASTNEXT ( Cmd, Par ) then exit;
        if Cmd='POST'      then if Cmd_POST     (      Par ) then exit;
        if Cmd='STAT'      then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='XHDR'      then if Cmd_XHDR     (      Par ) then exit;
        if Cmd='XOVER'     then if Cmd_XOVER    (      Par ) then exit;
        if Cmd='XPAT'      then if Cmd_XPAT     (      Par ) then exit;
        if Cmd='NEWNEWS'   then if Cmd_NEWNEWS  (      Par ) then exit;
{JW} {Peer}
        if (Cmd='IHAVE') and
           Def_IHAVE_Auth    then if Cmd_IHAVE    (      Par ) then exit;
        if (Cmd='TAKETHIS') and
           Def_IHAVE_Auth    then if Cmd_TAKETHIS (      Par ) then exit;
        if (Cmd='CHECK') and
           Def_IHAVE_Auth    then if Cmd_CHECK    (      Par ) then exit;
{JW}
        // unknown (sub-) command
        SendResult( '500 Command not implemented.' );
        Log( LOGID_INFO, 'Unsupported NNTP-command: ' + Command );
     except
        on E: Exception do begin
           Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
           Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + LogCmdLine );
        end;
     end;
end;

{MG}{SSL}
// As there is not yet an official standard for "NNTP over TLS", the STARTTLS
// command is only compatible with the INN news server (V 2.3.0 and higher)
function TSrvNNTPCli.Cmd_STARTTLS( Const Par: String ): Boolean;
begin
     Result := True;
     if not SSLReady then begin
        SendResult( '500 Command not implemented' );
        exit;
     end;
     if SSLContext = nil then begin
        SendResult( '580 Error initializing TLS' );
        exit;
     end;
     if SSLConnection = nil then begin
        SendResult( '382 Begin TLS negotiation' );
        if StartSSL then begin
           CurrentUserID   := ACTID_INVALID;
           CurrentUserName := '';
        end else
           SendResult( '580 STARTTLS failed');
     end else
        SendResult( '483 Already successfully executed STARTTLS' );
end;
{/SSL}

function TSrvNNTPCli.Cmd_ARTINFOS( Const Cmd, Par: String ): Boolean;

   Procedure SearchArticleNextTime (Const MID: String; Const IgnoreHist: boolean);
   Var j, ArtNo: Integer; s, GrpNam: String;
   begin
      SendResult( '430 '+TrGl(kLog, 'Info.MIDtoGetMids', 'Missing article will be searched on all servers on next pull'));
      If IgnoreHist then begin
         While NewsHistory.LocateMID( MID, GrpNam, ArtNo ) do begin
            NewsHistory.RemoveEntry( MID, StrToCRC32(LowerCase(GrpNam)), ArtNo )
         end
      end;
      For j:=0 to CfgHamster.ServerCount-1 do begin
         s := CfgHamster.ServerPath[j] + SRVFILE_GETMIDLIST;
         With TStringlist.Create do try
            If FileExists2(s) then LoadFromFile(s);
            If (IndexOf(MID)<0) and (IndexOf('!'+MID)<0) then begin
               Add(MID); SaveToFile(s)
            end
         finally free end
      end;
      Actions.Exec ( atSearchMID, MID )
   end;

   Procedure SendPartofArticle (Const Cmd, Buf: String);
   Var p: Integer;
   begin
      if Cmd='ARTICLE' then begin
         SendQuotedMultiLine(Buf)
      end else begin
         p := Pos(#13#10#13#10, Buf);
         if Cmd='HEAD' then begin
             If p > 0 then SendQuotedMultiLine(Copy(Buf, 1, p-1))
                      else SendQuotedMultiLine(buf);
          end else
          if Cmd='BODY' then begin
              If p > 0 then SendQuotedMultiLine(Copy(Buf, p+4, Length(Buf)));
          end
      end
   end;

var  s, h, Buf: String;
     j, TempGrpHdl: Integer;
begin
   Result := True;

   if copy(Par,1,1)='<' then begin

      // ARTICLE (selection by message-id)

      if not NewsHistory.LocateMID( Par, s, j ) then begin
         If Def_LocalNntpSearchUnknownMIDs
            then SearchArticleNextTime (Par, false)
            else SendResult( '430 no such article found (history)');
         exit
      end;
      Log( LOGID_DEBUG, Par + ' -> ' + s + ' #' + inttostr(j) );

      Buf := '';
      TempGrpHdl := ArticleBase.Open( s );
      if TempGrpHdl>=0 then begin
         if j<ArticleBase.LocalMin[TempGrpHdl] then s:='';
         if j>ArticleBase.LocalMax[TempGrpHdl] then s:='';
         if s<>'' then Buf := ArticleBase.ReadArticle( TempGrpHdl, j );
         ArticleBase.Close( TempGrpHdl );
      end else begin
         s := '';
      end;

      if (s='') or (Buf='') then begin
         If Def_LocalNntpSearchUnknownMIDs
            then SearchArticleNextTime (Par, true)
            else SendResult( '430 no such article found (group)' );
      end else begin
         h := ' ' + inttostr(j) + ' ' + Par + ' ';

         if Cmd='ARTICLE' then s:='220' + h + 'article'
         else if Cmd='HEAD'    then s:='221' + h + 'head'
         else if Cmd='BODY'    then s:='222' + h + 'body'
         else if Cmd='STAT'    then s:='223' + h + 'stat';
         SendResult( s );
         SendPartofArticle (Cmd, Buf);
         if Cmd<>'STAT' then SendResult( '.' )
      end

   end else begin

      // ARTICLE (selection by number)

      if (CurrentGroup<0) or (ArticleBase.Name[CurrentGroup]='') then begin
         SendResult( '412 no newsgroup has been selected' );
         exit;
      end;
      if CurrentArtNo=0 then begin
         SendResult( '420 no current article has been selected' );
         exit;
      end;

      if Par<>'' then j := strtoint( Par )
                 else j := CurrentArtNo;
      Buf := '';
      if (j>0) and (j>=ArticleBase.LocalMin[CurrentGroup])
               and (j<=ArticleBase.LocalMax[CurrentGroup]) then begin
         Buf := ArticleBase.ReadArticle( CurrentGroup, j );
         If Buf > '' then CurrentArtNo := j;
      end;
      If Buf = '' then begin
         SendResult( '423 no such article number in this group' );
         exit;
      end;

      With TArticle.Create do try
         Text := Buf;
         h := Header['Message-ID:']
      finally
         free
      end;
      If h = '' then h := '<0>';

      h := ' ' + inttostr(j) + ' ' + h + ' ';
      if Cmd='ARTICLE' then s:='220' + h + 'article';
      if Cmd='HEAD'    then s:='221' + h + 'head';
      if Cmd='BODY'    then s:='222' + h + 'body';
      if Cmd='STAT'    then s:='223' + h + 'stat';
      SendResult( s );
      SendPartofArticle (Cmd, Buf);
      SendResult( '.' );
   end
end;

function TSrvNNTPCli.Cmd_LASTNEXT( Const Cmd, Par: String ): Boolean;
var  h, s: String;
     j: Integer;
begin
     Result := True;

     if (CurrentGroup<0) or (ArticleBase.Name[CurrentGroup]='') then begin
        SendResult( '412 no newsgroup selected' );
        exit;
     end;
     if CurrentArtNo=0 then begin
        SendResult( '420 no current article has been selected' );
        exit;
     end;

     h := '';
     j := CurrentArtNo;
     repeat
        if Cmd='LAST' then begin
           dec( j );
           if (j<1) or (j<ArticleBase.LocalMin[CurrentGroup]) then break;
        end else begin
           inc( j );
           if j>ArticleBase.LocalMax[CurrentGroup] then break;
        end;
        With TArticle.Create do try
           Text := ArticleBase.ReadArticle( CurrentGroup, j );
           h := Header['Message-ID:']
        finally
           Free
        end
     until (h<>'');

     if h<>'' then begin
        CurrentArtNo := j;
        s := '223' + ' '
           + inttostr(CurrentArtNo) + ' '
           + h + ' '
           + 'article selected';
        SendResult( s );
     end else begin
        if Cmd='LAST' then SendResult( '422 no previous article in this group' )
                      else SendResult( '421 no next article in this group' ); // RFC 977 {JH}
     end;
end;

function TSrvNNTPCli.Cmd_QUIT( Const Par: String ): Boolean;
begin
     Result := True;

     if ClientSocket.Connected then SendResult( '205 Closing connection.' );
     Sleep( Def_LocalTimeoutQuitDelay );
     try
        if ClientSocket.Connected then ClientSocket.Close;
     except
        on E:Exception do Log(LOGID_DEBUG, 'Exception on .Close: ' + E.Message );
     end;
     Terminate;
end;

function TSrvNNTPCli.Cmd_AUTHINFO( Const Par: String ): Boolean;
var  s: String;
begin
     Result := True;
     try
        if UpperCase(copy(Par,1,4))='USER' then begin
           CurrentUserName := TrimWhSpace( copy( Par,6,Length(Par)-5) );
           CurrentUserID   := ACTID_INVALID;
           SendResult( '381 More authentication information required' );
           exit;
        end;
        if UpperCase(copy(Par,1,4))='PASS' then begin
           s := LoginUser( TrimWhSpace( copy( Par,6,Length(Par)-5 ) ) );
           SendResult( s );
           if (copy(s,1,3)<>'281') then Cmd_Quit('');  //JW 27.12.00 NNTP->Auto Logout
           exit;
        end;

        SendResult( '501 Command not supported' );
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'Nntp.Cmd_AUTHINFO-Exception: ' + E.Message );
        end;
     end;
end;

function TSrvNNTPCli.Cmd_POST( Const Par: String ): Boolean;
var  s: String;
begin
    Result := True;
    if UserMayPost and
       ((IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW) then begin
       WaitForCmd := False;
       PeerState:=False;
       SetLength(BufInStrm, 0) {.Clear};
       if (Def_GenerateNewsMID<>GENERATEMID_NEVER) then begin
          RecommendedMID := MidGenerator( Def_FQDNforMIDs );
          s := '340 OK, recommended ID ' + RecommendedMID;
       end else begin
          RecommendedMID := '';
          s := '340 send article to be posted. End with <CR-LF>.<CR-LF>';
       end;
       SendResult( s );
    end else begin
       SendResult( '440 posting not allowed' );
    end;
end;

{JW} {Peer}
function TSrvNNTPCli.Cmd_IHAVE( Const Par: String ): Boolean;
begin
     Result := True;
     if UserMayPeer or (not Def_IHAVE_Auth) and
        ((IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW) then begin
        If not Def_IHAVE_Auth then
           CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
        SetLength(BufInStrm, 0) {.Clear};
        if NewsHistory.ContainsMID( Par ) then begin
           PeerState:=False;
           WaitForCmd := True;
           SendResult( '435 article not wanted - do not send it');
           exit;
        end else begin
           PeerState:=True;
           WaitForCmd := False;
           SendResult( '335 send article to be transferred. End with <CR-LF>.<CR-LF>');
           exit;
        end;
     end else begin
        SendResult( '480 Transfer permission denied' );
     end;
end;

// Funktion nach einer Vorlage von  Heiko Studt
function TSrvNNTPCli.Cmd_CHECK ( Const Par: String ): Boolean;
begin
  result:=true;
  if Par='' then begin SendResult( '500 There is no MID!'); exit; end;
  if UserMayPeer or (not Def_IHAVE_Auth) and
     ((IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW) then begin
     If not Def_IHAVE_Auth then
        CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
     SetLength(BufInStrm, 0) {.Clear};
     if copy(Par,1,1)='<' then begin
        if not NewsHistory.ContainsMID( Par ) then begin
           WaitForCmd := True;
           SendResult( '238 no such article found (history). Please send '+Par );
           Log( LOGID_DETAIL,'Message-ID Check, wanted: '+Par);
        end else begin
           WaitForCmd := True;
           SendResult( '438 this article was found in history. Don''t send '+Par);
           Log( LOGID_DETAIL,'Message-ID Check, not wanted: '+Par);
        end;
     end else begin
        SendResult( '400 not accepting article, wrong Message-ID '+Par);
        Log( LOGID_WARN,'Wrong Message-ID received: '+Par);
     end;
  end else begin
     SendResult( '480 posting not allowed '+Par );
  end;
end;

function TSrvNNTPCli.Cmd_TAKETHIS ( Const Par: String ): Boolean;
begin
     result:=true;
     if UserMayPeer or (not Def_IHAVE_Auth) and
        ((IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW) then begin
        If not Def_IHAVE_Auth then
           CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
        SetLength(BufInStrm, 0) {.Clear};
        PeerState:=True;
        WaitForCmd := False;
        exit;
     end else begin
        SendResult( '480 Transfer permission denied '+Par );
     end;
end;


{JW}


function TSrvNNTPCli.Cmd_SLAVE( Const Par: String ): Boolean;
begin
     Result := True;
     SendResult( '202 slave status noted' );
end;

function TSrvNNTPCli.Cmd_GROUP( Const Par: String ): Boolean;
var  OK: Boolean;
     j: Integer;
     s: String;
begin
     Result := True;

     OK := False;
     for j:=0 to GroupList.Count-1 do begin
        if GroupList[j]=Par then begin OK:=True; break; end;
     end;
     if not OK then begin
        SendResult( '411 no such news group' );
        exit;
     end;

     // 211 101 100 200 group.name
     // 0:errno 1:count 2:first 3:last 4:name
     SetCurrentGroup( Par );
     try ArticleBase.LastClientRead[CurrentGroup] := Now except end;

     s := '211' + ' '
        + inttostr( ArticleBase.Count   [CurrentGroup] ) + ' '
        + inttostr( ArticleBase.LocalMin[CurrentGroup] ) + ' '
        + inttostr( ArticleBase.LocalMax[CurrentGroup] ) + ' '
        + ArticleBase.Name[CurrentGroup];
     SendResult( s );
end;

function TSrvNNTPCli.Cmd_NEWGROUPS( Const Par: String ): Boolean;
var  j, TempGrpHdl: Integer;
     s: String;
     DT: TDateTime;
     Typ: Char;
begin
     Result := True;

     // NEWGROUPS YYMMDD HHMMSS [GMT]
     j := Pos( ' ', Par );
     if j=0 then j:=length(Par)+1;
     s := copy(Par,1,j-1) + copy( TrimWhSpace(copy(Par,j+1,99)) ,1,6);
     case j of
        6: begin // YMMDD, Y=[0-9] -> YYYYMMDD
              if s[1] in ['0'..'9'] then s:='200'+s;
           end;
        7: begin // YYMMDD -> YYYYMMDD
              if strtoint(copy(s,1,2))>=50 then s:='19'+s else s:='20'+s;
           end;
        8: begin // .YYMMDD -> YYYYMMDD
              System.Delete( s, 1, 1 );
              if strtoint(copy(s,1,2))>=50 then s:='19'+s else s:='20'+s;
           end;
     end;

     DT := TimeStampToDateTime(s);
     if (length(s)<>14) or (DT=0) then begin
        SendResult( '431 Unexpected date-format (' + Par + ')' );
        exit;
     end;

     if ReloadGroupList then begin
        SendResult( '231 list of new newsgroups follows' );
        for j:=0 to GroupList.Count-1 do begin
           TempGrpHdl := ArticleBase.Open( GroupList[j] );
           if TempGrpHdl>=0 then begin
              if ArticleBase.DTCreated(TempGrpHdl)>=DT then begin
                 // group last first
                 Typ := (ArticleBase.GetProp( TempGrpHdl, 'type', '' ) + 'y')[1];
                 Case Typ of
                   'y', 'n', 'm': ;
                   'g': Typ := 'm';
                   else Typ := 'y'
                 end;
                 s := ArticleBase.Name[TempGrpHdl] + ' '
                    + inttostr( ArticleBase.LocalMax[TempGrpHdl] ) + ' '
                    + inttostr( ArticleBase.LocalMin[TempGrpHdl] ) + ' '
                    + Typ;
                 SendQuoted( s );
              end;
              ArticleBase.Close( TempGrpHdl );
           end;
        end;
        SendResult( '.' );
     end else begin
        SendResult( '503 System-error, check logfile. [ng]' );
     end;
end;

function NntpDateTimeToDateTime( DateStr, TimeStr: String ): TDateTime; {JH}
// Convert a NEWGROUPS/NEWNEWS-timepoint ('YYMMDD','HHNNSS') into TDateTime.
begin
   Result := 0;

   case length(DateStr) of
      5: begin // YMMDD, Y=[0-9] -> YYYYMMDD
            if DateStr[1] in ['0'..'9'] then DateStr:='200'+DateStr else exit;
         end;
      6: begin // YYMMDD -> YYYYMMDD
            if strtointdef(copy(DateStr,1,2),0)>=50 then DateStr:='19'+DateStr
                                                    else DateStr:='20'+DateStr;
         end;
      7: begin // 1YYMMDD -> YYYYMMDD
            System.Delete( DateStr, 1, 1 );
            if strtointdef(copy(DateStr,1,2),0)>=50 then DateStr:='19'+DateStr
                                                    else DateStr:='20'+DateStr;
         end;
   end;

   if length(DateStr)<>8 then exit;
   if length(TimeStr)<>6 then exit;

   Result := TimeStampToDateTime( DateStr + TimeStr );
end;

function GroupMatchesPatterns( const GroupName  : String; {JH}
                               const WildMatPats: TStringList ): Boolean;
// Check, if GroupName is matched by the wildmat-patterns given in WildMatPats.
var  i: Integer;
     w: String;
     g: PChar;
begin
   // optimize: all or none are selected
   if WildMatPats.Count = 1 then begin
      if WildMatPats[0] = '*'  then begin Result := True;  exit end;
      if WildMatPats[0] = '!*' then begin Result := False; exit end;
   end;

   // test patterns - last matching pattern wins
   Result := False;
   g := PChar( GroupName );
   for i := WildMatPats.Count-1 downto 0 do begin
       w := WildMatPats[i];
       if copy(w,1,1) = '!' then begin // '!' as 1st char means 'not'
          if WildMat( g, PChar( copy(w,2,999) ) ) then break;
       end else begin
          if WildMat( g, PChar(w) ) then begin Result := True; break end;
       end;
   end;
end;

function SplitXrefGroups( const XrefValue : String; {JH}
                          const XrefGroups: TStringList;
                          WantHost: Boolean = False ): Integer;
// Splits the value of a Xref:-header ('host WSP grp:art WSP grp:art ...').
// XrefGroups.Strings will be filled with the group-names, .Objects with the
// corresponding article-numbers. If WantHost is True, the leading host-name
// is returned in XrefGroups[0] (if .Sorted=False!) with a number of -1.
var  j, n: Integer;
     Xref, s: String;
begin
   Result := 0;
   XrefGroups.Clear;
   Xref := XrefValue;

   j := PosWhSpace( Xref );
   if j=0 then exit;
   if WantHost then
      XrefGroups.AddObject( copy( Xref, 1, j ), Pointer(-1) ); // host, -1
   System.Delete( Xref, 1, j );
   Xref := TrimWhSpace( Xref );

   while Xref <> '' do begin
      j := PosWhSpace( Xref );
      if j=0 then begin
         s := Xref;
         Xref := '';
      end else begin
         s := copy( Xref, 1, j-1 );
         System.Delete( Xref, 1, j );
         Xref := TrimWhSpace( Xref );
      end;

      j := Pos( ':', s );
      if j>0 then begin
         n := strtointdef( copy( s, j+1, Length(s)-j ), 0 );
         s := copy( s, 1, j-1 );
         XrefGroups.AddObject( s, Pointer(n) ); // groupname, article-number
      end;
   end;

   Result := XrefGroups.Count;
end;

function TSrvNNTPCli.Cmd_LIST( Const Par: String ): Boolean;
var  j, TempGrpHdl: Integer;
     s, Pattern, GrpNam, Cmd: String;
begin
   Result := True;
   Cmd := UpperCase(Par);
   if Par='' then begin
      SendResult( '215 list of newsgroups follows' );
      for j:=0 to GroupList.Count-1 do begin
         TempGrpHdl := ArticleBase.Open( GroupList[j] );
         if TempGrpHdl>=0 then begin
            // group last first p
            s := ArticleBase.Name[TempGrpHdl] + ' '
               + inttostr( ArticleBase.LocalMax[TempGrpHdl] ) + ' '
               + inttostr( ArticleBase.LocalMin[TempGrpHdl] ) + ' '
               + ArticleBase.GetProp( TempGrpHdl, 'type', 'y' );
            SendQuoted( s );
            ArticleBase.Close( TempGrpHdl );
         end;
      end;
      SendResult( '.' )
   end else
   if Cmd='OVERVIEW.FMT' then begin
      SendResult( '215 information follows' );
      SendQuoted( 'Subject:' );
      SendQuoted( 'From:' );
      SendQuoted( 'Date:' );
      SendQuoted( 'Message-ID:' );
      SendQuoted( 'References:' );
      SendQuoted( 'Bytes:' );
      SendQuoted( 'Lines:' );
      SendQuoted( 'Xref:full' );
      SendResult( '.' );
   end else
   if Cmd='EXTENSIONS' then begin
      // SendResult( '501 Bad command use' );
      SendResult( '202-Extensions supported:' );
      SendQuoted( ' ' + 'AUTHINFO' );
      SendQuoted( ' ' + 'LISTGROUP' );
      SendResult( '.' );
   end else
   if Cmd='ACTIVE.TIMES' then begin
      SendResult( '215 list of newsgroups follows' );
      for j:=0 to GroupList.Count-1 do begin
         TempGrpHdl := ArticleBase.Open( GroupList[j] );
         if TempGrpHdl>=0 then begin
            // group last first p
            s := ArticleBase.Name[TempGrpHdl] + ' '
               + inttostr( DateTimeToUnixTime(ArticleBase.DTCreated(TempGrpHdl)) ) + ' '
               + 'local';
            SendQuoted( s );
            ArticleBase.Close( TempGrpHdl );
         end;
      end;
      SendResult( '.' );
   end else
   If copy(Cmd,1,6)='ACTIVE' then begin
      j := PosWhSpace( Par );
      if j=0 then Pattern:=''
             else Pattern:=TrimWhSpace( copy(Par,j+1,Length(Par)-j) );
      SendResult( '215 list of newsgroups follows' );
      for j:=0 to GroupList.Count-1 do begin
         GrpNam := GroupList[j];
         if Pattern<>'' then begin
            if not WildMat( PChar(GrpNam), PChar(Pattern) ) then GrpNam:='';
         end;
         if GrpNam<>'' then  begin
            TempGrpHdl := ArticleBase.Open( GroupList[j] );
            if TempGrpHdl>=0 then begin
               // group last first p
               s := ArticleBase.Name[TempGrpHdl] + ' '
                  + inttostr( ArticleBase.LocalMax[TempGrpHdl] ) + ' '
                  + inttostr( ArticleBase.LocalMin[TempGrpHdl] ) + ' '
                  + ArticleBase.GetProp( TempGrpHdl, 'type', 'y' );
               SendQuoted( s );
               ArticleBase.Close( TempGrpHdl );
            end;
         end;
      end;
      SendResult( '.' );
   end else
   If copy(Cmd,1,10)='NEWSGROUPS' then begin
      j := PosWhSpace( Par );
      if j=0 then Pattern:=''
             else Pattern:=TrimWhSpace( copy(Par,j+1,Length(Par)-j) );
      SendResult( '215 information follows' );
      for j:=0 to GroupList.Count-1 do begin
         GrpNam := GroupList[j];
         if Pattern<>'' then begin
            if not WildMat( PChar(GrpNam), PChar(Pattern) ) then GrpNam:='';
         end;
         if GrpNam<>'' then  begin
            TempGrpHdl := ArticleBase.Open( GroupList[j] );
            if TempGrpHdl>=0 then begin
               s := ArticleBase.Name[TempGrpHdl] + #9
                  + ArticleBase.Description[TempGrpHdl];
               SendQuoted( s );
               ArticleBase.Close( TempGrpHdl );
            end;
         end;
      end;
      SendResult( '.' );
   end else
{HSR} {listmotd}
   If copy(Cmd,1,4)='MOTD' then begin
      SendResult( '215 Message of the day text..' );
      if fileexists2(PATH_BASE + 'hamstermotd.hst') then begin
         With TTextReader.Create(PATH_BASE + 'hamstermotd.hst', 1024) do try
            while not eof do SendQuoted( ReadLine )
         finally
            free
         end
      end;
      SendResult( '.' );
   end else begin
      Result := False
   end
end;

function TSrvNNTPCli.Cmd_LISTGROUP( Const Par: String ): Boolean;
var  OK: Boolean;
     j, vv, bb: Integer;
     s: String;
begin
     Result := True;

     if Par<>'' then begin
        OK := False;
        for j:=0 to GroupList.Count-1 do begin
           if GroupList[j]=Par then begin OK:=True; break; end;
        end;
        if not OK then begin
           SendResult( '411 no such news group' );
           exit;
        end;
        SetCurrentGroup(Par);
     end;

     if (CurrentGroup<0) or (ArticleBase.Name[CurrentGroup]='') then begin
        SendResult( '412 not currently in newsgroup' );
        exit;
     end;

     vv := ArticleBase.LocalMin[CurrentGroup];
     bb := ArticleBase.LocalMax[CurrentGroup];

     SendResult( '211 list of article numbers follows' );
     s := '';
     for j:=vv to bb do begin
        // if ClientData.CurrentGroup.ReadSize(j)>0 then begin
        if ArticleBase.GetKeyIndex(CurrentGroup,j)>=0 then begin
           if s<>'' then s:=s+#13#10;
           s := s + inttostr(j); // last CRLF is added by SendQuoted() below
        end;
     end;
     if s<>'' then SendQuoted( s );
     SendResult( '.' );
end;

function TSrvNNTPCli.Cmd_MODE( Const Par: String ): Boolean;
begin
   Result := True;
   if UpperCase(Par)='READER' then begin
      if (IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW then begin {JW} {Mode}
          if Def_Stream_Mode and StreamMode then begin
             StreamMode:=false;
             SendResult( '200 enter Mode Reader' );
          end else begin
             SendResult( '200 ignored' );
          end {JW}
      end else begin
         SendResult( '201 ignored' );
      end
   end else
   if UpperCase(Par)='STREAM' then begin
      if (UserMayPeer or (not Def_IHAVE_Auth)) and
         ((IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW) and
         Def_Stream_Mode then begin
         StreamMode:=True;
         If not Def_IHAVE_Auth then
            CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
         SendResult( '203 Streaming is OK' );
      end else begin
         SendResult( '500 ignored' );
      end;
   end else
   if UpperCase(Par)='CANCEL' then begin      //HSR //ModeCancel
      if (not StreamMode) and ((IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW)
          and UserMayCancel
      then begin
         WaitForCmd  := False;
         CancelState := true;
         SendResult('284 Send your Cancels!');
      end else begin
         SendResult('500 Not previleged!')
      end
   end else begin
      Result := False
   end
end;

function TSrvNNTPCli.Cmd_DATE( Const Par: String ): Boolean;
begin
     Result := True;
     SendResult( '111 ' + DateTimeToTimeStamp(NowGMT) );
end;

function TSrvNNTPCli.Cmd_XOVER( Const Par: String ): Boolean;

   function FormatXOver( const s: String ) : String;  /// ff.
   // replace (remaining) NUL, TAB, LF and CR with SP
   var  i: Integer;
   begin
      Result := s;
      If Def_News_XOverWithoutCRLF then begin
         For i:=Length(Result) downto 2 do begin
            If i <= Length(Result) then begin
               If (Result[i-1]=#13) and (Result[i]=#10) then begin
                  Delete (Result, i-1, 2)
               end
            end
         end
      end;
      for i:=1 to length(Result) do begin
         if Result[i] in [ #0, #9, #10, #13 ] then Result[i] := ' ';
      end
   end;

var  s, sLines: String;
     j, vv, bb, ArtSize: Integer;
     Art: TArticle;
begin
     Result := True;

     if (CurrentGroup<0) or (ArticleBase.Name[CurrentGroup]='') then begin
        SendResult( '412 No news group current selected' );
        exit;
     end;
     if (Par='') and (CurrentArtNo=0) then begin
        SendResult( '420 No article(s) selected' );
        exit;
     end;

     s := Par;
     if s='' then s:=inttostr( CurrentArtNo );

     j := Pos('-',s);
     if j>0 then begin
        vv := strtoint( copy(s,1,j-1) );
        s  := copy( s, j+1, Length(s)-j );
        if s<>'' then bb := strtoint( s )
                 else bb := ArticleBase.LocalMax[CurrentGroup];
     end else begin
        vv := strtoint( s );
        bb := vv;
     end;

     SendResult( '224 Overview information follows' );

     Art := TArticle.Create;
     for j:=vv to bb do begin
        ArtSize  := 4096;
        Art.Text := ArticleBase.ReadArticleSized( CurrentGroup, j, ArtSize );
        sLines   := Art.Header['Lines:'];
        if (sLines='') or (Art.FullBody = '') then begin
           Art.Text := ArticleBase.ReadArticle( CurrentGroup, j );
           sLines := inttostr( CountLines(Art.FullBody) );
        end;

        if Art.Text<>'' then begin
           // 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References 6:Bytes 7:Lines [8:Xref]
           s := inttostr(j) + #9
              + FormatXOver( Art['Subject:'    ] ) + #9
              + FormatXOver( Art['From:'       ] ) + #9
              + FormatXOver( Art['Date:'       ] ) + #9
              + FormatXOver( Art['Message-ID:' ] ) + #9
              + FormatXOver( Art['References:' ] ) + #9
              + FormatXOver( inttostr( ArtSize )            ) + #9
              + FormatXOver( sLines                         ) + #9
              + FormatXOver( Art['Xref:'       ] );
           SendQuoted( s );
        end;
     end;
     Art.Free;

     SendResult( '.' );
end;

function TSrvNNTPCli.Cmd_XPAT( Const Par: String ): Boolean;
var  Parser: TParser;
     HdrNam, HdrVal, Pattern, s: String;
     j, vv, bb, k, ArtSize: Integer;
     Art: TArticle;
     ok: Boolean;
begin
     Result := True;

     if (CurrentGroup<0) or (ArticleBase.Name[CurrentGroup]='') then begin
        SendResult( '412 No news group current selected' );
        exit;
     end;
     if (Par='') and (CurrentArtNo=0) then begin
        SendResult( '420 No article(s) selected' );
        exit;
     end;

     Parser := TParser.Create;
     Parser.Parse( Par, ' ' );

     HdrNam := Parser.sPart( 0, '' );
     s := Parser.sPart( 1, '' ); // range|mid
     if (HdrNam='') or (s='') or (pos('@',s)>0) then begin
        Parser.Free;
        SendResult( '501 Unknown/unsupported xpat-syntax' );
        exit;
     end;

     j := Pos('-',s);
     if j>0 then begin
        if s='-' then begin
           vv := ArticleBase.LocalMin[CurrentGroup];
           bb := ArticleBase.LocalMax[CurrentGroup];
        end else begin
           vv := strtoint( copy(s,1,j-1) );
           if vv<ArticleBase.LocalMin[CurrentGroup] then vv:=ArticleBase.LocalMin[CurrentGroup];

           s  := copy( s, j+1, Length(s)-j );
           if s<>'' then begin
              bb := strtoint( s );
              if bb>ArticleBase.LocalMax[CurrentGroup] then bb:=ArticleBase.LocalMax[CurrentGroup];
           end else begin
              bb := ArticleBase.LocalMax[CurrentGroup];
           end;
        end;
     end else begin
        vv := strtoint( s );
        bb := vv;
     end;

     SendResult( '221 Header follows' );

     Art := TArticle.Create;
     for j:=vv to bb do begin
        ArtSize  := 4096;
        Art.Text := ArticleBase.ReadArticleSized( CurrentGroup, j, ArtSize );
        If Art.FullBody = '' then Art.Text := ArticleBase.ReadArticle( CurrentGroup, j );

        HdrVal := Art[HdrNam + ':'];

        ok := False;
        if HdrVal<>'' then begin
           k := 2;
           repeat
              Pattern := Parser.sPart( k, '' );
              if (Pattern='') and (k=2) then begin
                 ok := True; // impliziter "*"
                 break;
              end;
              if Pattern='' then break;

              if WildMat( PChar(HdrVal), PChar(Pattern) ) then begin
                 ok := True;
                 break;
              end;
              inc(k);
           until False;
        end;

        if ok then begin
           s := inttostr(j) + ' ' + HdrVal;
           SendQuoted( s );
        end;
     end;
     Art.Free;
     Parser.Free;

     SendResult( '.' );
end;

function TSrvNNTPCli.Cmd_XHDR( Const Par: String ): Boolean;
var  j, vv, bb, ArtSize: Integer;
     HdrNam, ArtNr, s, h: String;
begin
   Result := True;
   j := Pos( ' ', Par );
   if j=0 then begin
      HdrNam := Trim(Par);
      ArtNr    := '';
   end else begin
      HdrNam := Trim( copy( Par, 1, j-1 ) );
      ArtNr  := Trim( copy( Par, j+1, Length(Par)-j ) );
   end;

   if copy(ArtNr,1,1)='<' then begin
      SendResult( '500 XHDR with Message-ID not implemented.' );
      Log( LOGID_INFO, 'Unsupported NNTP-server-command: ' + Par );
   end
   else
   if (CurrentGroup<0) or (ArticleBase.Name[CurrentGroup]='') then begin
      SendResult( '412 No news group current selected' );
   end
   else
   if (ArtNr='') and (CurrentArtNo=0) then begin
      SendResult( '420 No article(s) selected' );
   end
   else begin
      s := ArtNr;
      if s='' then s:=inttostr( CurrentArtNo );
      j := Pos('-',s);
      if j>0 then begin
         vv := strtoint( copy(s,1,j-1) );
         s  := copy( s, j+1, 255 );
         if s<>'' then bb := strtoint( s )
                  else bb := ArticleBase.LocalMax[CurrentGroup];
      end else begin
         vv := strtoint( s );
         bb := vv;
      end;
      SendResult( '221 Header follow' );
      With TArticle.Create do try
         for j:=vv to bb do begin
            ArtSize  := 4096;
            Text := ArticleBase.ReadArticleSized( CurrentGroup, j, ArtSize );
            If FullBody = '' then Text := ArticleBase.ReadArticle( CurrentGroup, j );
            If Text > '' then begin
               h := Header[HdrNam];
               if h='' then h:='(none)';
               s := inttostr(j) + ' ' + h;
               SendQuoted( s );
            end
         end
      finally
         Free
      end;
      SendResult( '.' )
   end
end;

function TSrvNNTPCli.Cmd_HELP( Const Par: String ): Boolean;
begin
     Result := True;
     SendResult( '100 Implemented commands follow:' );
     SendQuoted( '    authinfo user Name|pass Password' );
     SendQuoted( '    article [MessageID|Number]' );
     SendQuoted( '    body [MessageID|Number]' );
     SendQuoted( '    check MessageID' );
     SendQuoted( '    date' );
     SendQuoted( '    group newsgroup' );
     SendQuoted( '    head [MessageID|Number]' );
     SendQuoted( '    help' );
     SendQuoted( '    ihave MessageID' );
     SendQuoted( '    last' );
     SendQuoted( '    list [active|active.times|newsgroups|overview.fmt|motd]' );
     SendQuoted( '    listgroup [newsgroup]' );
     SendQuoted( '    mode reader|stream' );
     SendQuoted( '    newgroups yymmdd hhmmss ["GMT"]' );
     SendQuoted( '    newnews newsgroups date time ["GMT"|"UTC"]' ); {JW} {Help News}
     SendQuoted( '    next' );
     SendQuoted( '    post' );
     SendQuoted( '    quit' );
     SendQuoted( '    slave' );
     SendQuoted( '    stat [MessageID|Number]' );
     if Assigned(SSLContext) and (SSLConnection=nil) then begin
        SendQuoted( '    starttls' ); {MG}{SSL}
     end;
     SendQuoted( '    takethis MessageID' );
     SendQuoted( '    xhdr header [range]' );
     SendQuoted( '    xpat header range pat [pat...]' );
     SendQuoted( '    xover [range]' );
     SendResult( '.' );
end;

procedure TSrvNNTPCli.SendGreeting;

   procedure AutoLogin;
   var  s: String;
   begin
      // auto-login "nntpdefault/*"-user, if account exists
      if CurrentUserID=ACTID_INVALID then begin
         if CfgAccounts.UserIDOf('nntpdefault')<>ACTID_INVALID then begin
            CurrentUserName := 'nntpdefault';
            CurrentUserID   := ACTID_INVALID;
            s := LoginUser( '*' );
            If copy(s,1,3)<>'281' then begin {JW} {AutoLogin}
               SendResult( s );
               Cmd_Quit('');  //JW 27.12.00 NNTP->Auto Logout
               Log( LOGID_ERROR, 'Auto-Login "nntpdefault": ' + s )
            end else begin
               Log( LOGID_INFO, 'Auto-Login "nntpdefault": ' + s );
               CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
            end
         end;
      end;
   end;

begin
   {JW} {LocalLimit}
     if (GetCounter(CntNntpCli) > Def_Max_Local_NNTP_Servers) and
        (Def_Max_Local_NNTP_Servers>0) then begin
        SendResult( '502 server overload, try again later' );
        Log( LOGID_WARN, 'NNTP Server overload, too many clients' );
        Terminate;
        exit;
     end;
   {/JW}
   CheckClientAccess;

//PKR 22.12.00
   if (IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW then begin
      If Def_FQDN > '' then begin
         SendResult( '200 NNTP-Server '
                + GetMyStringFileInfo('ProductName','Hamster') + ' '
                + GetMyVersionInfo(true)
                + ' (post ok) on '
                + Def_FQDN + ' says: Hi!' )
      end else begin
         SendResult( '200 NNTP-Server '
                + GetMyStringFileInfo('ProductName','Hamster') + ' '
                + GetMyVersionInfo(true)
                + ' (post ok) says: Hi!' )
      end;

      AutoLogin;

   end else if (IPAccess and IPACC_ACCESS_RO)=IPACC_ACCESS_RO then begin
      If Def_FQDN > '' then begin
         SendResult( '201 NNTP-Server '
                + GetMyStringFileInfo('ProductName','Hamster') + ' '
                + GetMyVersionInfo(true)
                + ' (no post) on '
                + Def_FQDN + ' says: Hi!' )
      end else begin
         SendResult( '201 NNTP-Server '
                + GetMyStringFileInfo('ProductName','Hamster') + ' '
                + GetMyVersionInfo(true)
                + ' (no post) says: Hi!' );
      end;
      AutoLogin;

   end else begin
      Log( LOGID_WARN, TrGlF(kLog, 'Warning.NNTP.ConnectionRefused',
         'Connection refused, %s isn''t allowed to connect to NNTP-Server', ClientID) );
      if Assigned(ClientSocket) then
      try
         if ClientSocket.Connected then SendResult( '500 Permission denied - closing connection.' );
         if ClientSocket.Connected then ClientSocket.Close;
      except
      end;
      Terminate;

   end;
end;

function TSrvNNTPCli.Cmd_NEWNEWS( Const Par: String ): Boolean;
// NEWNEWS newsgroups date time ["GMT"|"UTC"] [<distributions>]
// Note: if <distributions> is given, it is ignored
var  GrpNo, ArtNo, RefNo, GrpHdl, ArtSize, ArtMin, ArtMax: Integer;
     sGroups, sDate, sTime, sGMT, MessageID: String;
     GroupPats, TS: TStringList;
     dtStart: TDateTime;
     Art: TArticle;
begin
   Result := True;

   if not UserMayNewNews then begin
      SendResult( '502 NEWNEWS-permission denied.' );
      exit;
   end;

   TS := TStringList.Create;
   GroupPats := TStringList.Create;
   Art := TArticle.Create;

   try
      // parse arguments
      ArgsWhSpace( Par, TS, 5 );
      sGroups := TS[0];
      sDate   := TS[1];
      sTime   := TS[2];
      if copy(TS[3],1,1)<>'<' then sGMT:=UpperCase(TS[3]) else sGMT:='';

      // parse newsgroups-arg.
      if ArgsWhSpace( ReplaceChar( sGroups, ',', ' ' ), GroupPats ) = 0 then begin
         SendResult( '501 Bad newsgroup specifier: ' + sGroups );
         exit;
      end;

      // parse date-arg.
      dtStart := NntpDateTimeToDateTime( sDate, sTime );
      if dtStart = 0 then begin
         SendResult( '501 Bad date: ' + sDate + ' ' + sTime + ' ' + sGMT );
         exit;
      end;
      if (sGMT='GMT') or (sGMT='UTC') then dtStart:=DateTimeGMTToLocal(dtStart);

      // loop through user's newsgroup-list and check if group is selected;
      // if selected, loop through all articles of the group and report its
      // new articles not reported so far

      SendResult( '230 New news follows' );

      for GrpNo:=0 to GroupList.Count-1 do begin

         if GroupMatchesPatterns( GroupList[GrpNo], GroupPats ) then begin

            GrpHdl := ArticleBase.Open( GroupList[GrpNo] );
            if GrpHdl>=0 then try

               ArtMin := ArticleBase.LocalMin[GrpHdl];
               ArtMax := ArticleBase.LocalMax[GrpHdl];
               
               for ArtNo := ArtMin to ArtMax do begin

                  ArtSize  := 4096;
                  Art.Text := ArticleBase.ReadArticleSized( GrpHdl, ArtNo, ArtSize );

                  if Art.HeaderCount > 1 then begin // still available?

                     if Art.FullBody = '' then Art.Text := ArticleBase.ReadArticle( GrpHdl, ArtNo );
                     if Art.GetReceivedDT >= dtStart then begin // a new one?
                        MessageID := Art['Message-ID:'];

                        // check if article was already reported
                        SplitXrefGroups( Art['Xref:'], TS );
                        TS.Sort; // sort Xref-groups like GroupList-groups
                        for RefNo:=0 to TS.Count-1 do begin
                           // is current group the first selected group?
                           if AnsiCompareText( TS[RefNo], GroupList[GrpNo] ) = 0 then break;
                           // was a previous Xref-group selected?
                           if GroupMatchesPatterns( TS[RefNo], GroupPats ) then begin
                              MessageID := ''; // don't report again
                              break;
                           end;
                        end;

                        if MessageID<>'' then SendQuoted( MessageID );
                     end;
                  end;
               end;
            finally
               ArticleBase.Close( GrpHdl );
            end;
         end;
      end;

      SendResult( '.' );

   finally
      Art.Free;
      GroupPats.Free;
      TS.Free;
   end;
end;

constructor TSrvNNTPCli.Create( ASocket: TServerClientWinSocket;
                                Const AIPAccessScope: LongInt;
                                Const ASSLContext: Pointer);
begin
   inherited Create( ASocket, AIPAccessScope, ASSLContext );

   CurrentInactTimeOut:=Def_LocalNNTPLoginTimeout;

   LimitLineLen  := Def_LocalLimitLineLenNntp;
   LimitTextSize := Def_LocalLimitTextSizeNntp;

   CurrentUserID   := ACTID_INVALID;
   CurrentUserName := '';
   CurrentGroup    := -1;
   CurrentArtNo    := 0;
   UserMayPost     := False;
   UserMayPeer     := False;
   UserMayNewNews  := False;
   UserMayCancel   := False;
   PeerState       := False;
   CancelState     := False; //HSR //ModeCancel     RecommendedMID  := '';
   PermRead        := '';
   PermPost        := '';

   GroupList := TStringList.Create;
   GroupList.Sorted := True;
   GroupList.Duplicates := dupIgnore;

   IncCounter(CntNntpCli,1);
end;

destructor TSrvNNTPCli.Destroy;
begin
     SetCurrentGroup( '' );
     if Assigned(GroupList) then begin GroupList.Free; GroupList := nil; end;

//     dec(CntNntpCli);
     IncCounter(CntNntpCli,-1);
     inherited;
end;

// ------------------------------------------------------------- TSrvNNTP -----

{procedure TSrvNNTP.MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                                    var SocketThread: TServerClientThread);
begin
     Log( LOGID_DEBUG, Self.Classname + '.OnGetThread' );
     // Create a new thread for connection
     try
        SocketThread := TSrvNNTPCli.Create(True, ClientSocket);
        SocketThread.Resume;
     except
        On E:Exception do Log( LOGID_ERROR, '.OnGetThread.Exception: ' + E.Message );
     end;
end;}
constructor TSrvNNTP.Create( AOwner: TComponent );
begin
   inherited Create(
      AOwner,
      CfgIni.ReadString ('Setup', 'local.nntp.serverbind', Def_LocalNNTPServerBind ),
      CfgIni.ReadInteger('Setup', 'local.port.nntp', DEF_LOCALNNTPServer_PORT  ),
      CfgIni.ReadInteger('Setup', 'MaxLocalNNTPServers', Def_Max_Local_NNTP_Servers),
      CfgIni.ReadInteger('Setup', 'MaxLocalNNTPServersPerIP', Def_Max_Local_NNTP_Servers_Per_IP),
      IPACC_SCOPE_NNTP,
      TSrvNNTPCli
   );
end;

// ----------------------------------------------------------------------------

function TSrvNNTPCli.HandleCancelData(LineIn: String): String;

  function DeleteMID(MessageID: String) : boolean;
  begin
     if ArchivMode
        then Result := false
        else Result := ArticleBase.DeleteArticleByMID( MessageID, true );
  end;

  function HasPrevilegue(MessageID: String) : Boolean;
  var
    Group      : String;
    Number, i  : Integer;
    TempGrpHdl : Integer;
  begin
    Result := false;
    if not NewsHistory.LocateMID( MessageID, Group, Number ) then exit;

    with tArticle.Create do try
      TempGrpHdl := ArticleBase.Open( Group );
      if TempGrpHdl>=0 then begin
         if Number<ArticleBase.LocalMin[TempGrpHdl] then Group:='';
         if Number>ArticleBase.LocalMax[TempGrpHdl] then Group:='';
         if Group<>'' then begin
           Text := ArticleBase.ReadArticle( TempGrpHdl, Number );
           Group := ',' + Header['NewsGroups:'] + ',';
         end;
         ArticleBase.Close( TempGrpHdl );
      end else begin
         Group := '';
      end;
    finally
      Free;
    end;

    if not (Group='') then begin
      for i := 0 to grouplist.count-1 do
        if (pos(','+grouplist[i]+',',Group)>0) then
        //ToDo: Pro Gruppe festlegbar machen, ob gecanelt werden darf.
        //Bisher: User darf in allen Gruppen Canceln, in der er auch lesen kann.
        //(Sofern er *berhaupt* MODE CANCEL benutzen darf)
        begin
          Result := true;
          break
        end;
    end;
  end;

begin
   WaitForCmd := true;
   CancelState := False;
   Result := '400 Unknown error!';
   try
      EnterCriticalSection(CS_LOCK_InCancel);
      try
         if not ClientSocket.Connected then exit;
         If LineIn <> '.' then begin
            LineIn := TrimWHSpace(LineIn);
            If pos('<',LineIn)=0 then LineIn := '<' + LineIn;
            If pos('>',LineIn)=0 then LineIn := LineIn + '>';
            if HasPrevilegue(LineIn) then begin
               if DeleteMID(LineIn) then begin
                  Result := ('289 Article deleted!');
                  Log ( LOGID_SYSTEM, 'Mode Cancel - '+LineIn+' deleted!')
               end else begin
                  Result := ('484 Article couldn''t be found');
                  Log ( LOGID_SYSTEM, 'Mode Cancel - '+LineIn+' couldn''t be found')
               end
            end else begin
               Result := ('484 You don''t have permission or article wasn''t found');
               Log ( LOGID_SYSTEM, 'Mode Cancel - No permission or '+LineIn+' wasn''t found')
            end;
            WaitForCmd := False;
            CancelState := true;
         end else begin
            Result := '201 MODE CANCEL successfull ended'; //'201 Test; End of MODE CANCEL';
         end
      except
        on E:Exception do Log( LOGID_ERROR, 'TSrvNNTPCli.HandleCancelData.Exception: ' + E.Message );
      end;
   finally
      LeaveCriticalSection(CS_LOCK_InCancel);
   end;
end;

procedure TSrvNNTPCli.ClientExecute; //Wegen MODE CANCEL hineingezogen
var  Data: array[0..1023] of Char;
     SocketStream: TWinSocketStream;
     LineIn, Reply : String;
     LineEnd, ByteIn, i : Integer;
begin
     try
        SendGreeting;
        Logfile.SetTask( '(client ' + lowercase(inttohex(GetCurrentThreadID,1))+')', ClientID );
     except
        on E:Exception do begin
           Log( LOGID_WARN, 'Connection failed (SendGreeting):' + ClientID );
           Log( LOGID_WARN, 'ERROR: ' + E.Message );
        end;
     end;

     while not Terminated do try
        if ClientSocket=nil then exit;
        if not ClientSocket.Connected then exit;
        SocketStream := TWinSocketStream.Create( ClientSocket,
                                                 CurrentInactTimeOut );
        try
           if SocketStream.WaitForData( CurrentInactTimeOut ) then begin
              FillChar(Data, SizeOf(Data), 0);
              ByteIn := 0;
              if ClientSocket<>nil then begin
                 try
                    if ClientSocket.Connected then begin
                       if SSLConnection=NIL {MG}{SSL}
                          then ByteIn := SocketStream.Read( Data, SizeOf(Data) )
                          else ByteIn := SSL.Read( Data, SizeOf(Data) ); {MG}{SSL}
                    end;
                 except
                    ByteIn := 0;
                 end;
              end;
              if ByteIn = 0 then begin
                If Not LogFile.Skip_ConectionLost then begin
                   Log( LOGID_WARN, TrGl (kLog, 'Connection.lost', 'Connection lost') + ':' + ClientID )
                end;
                if ClientSocket<>nil then begin
                   if ClientSocket.Connected then ClientSocket.Close;
                end;
                Terminate;
              end;

              BufInRaw := BufInRaw + Data;

              repeat
                 LineEnd  := Pos( CRLF, BufInRaw );
                 if (LineEnd=0) and (LimitLineLen>0) and (length(BufInRaw)>LimitLineLen) then begin
                    HadLineTooLong := True;
                    LineEnd := LimitLineLen + 1;
                 end;

                 if LineEnd>0 then begin

                    // ggf. virtuellen LineIn-Handler
                    LineIn := copy( BufInRaw, 1, LineEnd-1 );
                    System.Delete ( BufInRaw, 1, LineEnd+1 );

                    {if (LOGFILEMASK and LOGID_FULL)<>0 then begin
                       Log( LOGID_DEBUG, SockDesc('.Recv') + LineIn );
                    end;}

                    if WaitForCmd then begin
                       HandleCommand( LineIn );
                    end else begin
                      if not CancelState then //HSR //ModeCancel
                       if (LineIn='.') then begin
                          WaitForCmd := True;
                          SetLength(BufInStrm, BufInStrmLen); {TGL}
                          Reply := HandleData;
                          SetLength(BufInStrm, 0) {.Clear}; BufInStrmLen := 0;
                          if Reply<>'' then SendResult( Reply );
                       end else begin
                          if copy(LineIn,1,2)='..' then System.Delete(LineIn,1,1);
                          {if length(LineIn)>0 then}
                             If BufInStrmLen + length(LineIn) + 2 > Length(BufInStrm) then begin
                                If BufInStrmLen > 128000
                                   then SetLength(BufInStrm, BufInStrmLen + 163840)
                                   else SetLength(BufInStrm, BufInStrmLen + length(LineIn) + 2 + 16384)
                             end;
                             For i := 1 to Length(LineIn) do
                                BufInStrm[BufInStrmLen + i] := LineIn[i];
                             BufInStrm[BufInStrmLen + length(LineIn) + 1] := CRLF[1];
                             BufInStrm[BufInStrmLen + length(LineIn) + 2] := CRLF[2];
                             Inc (BufInStrmLen, length(LineIn) + 2);
                          {if length(LineIn)>0 then BufInStrm.Write( LineIn[1], length(LineIn) );
                          BufInStrm.Write( CRLF, 2 );}
                          if (LimitTextSize>0) and (BufInStrmLen{.Size}>LimitTextSize) then begin
                             HadTextTooLarge := True;
                             Reply := HandleData;
                             SetLength(BufInStrm, 0) {.Clear};
                             BufInStrmLen := 0;
                             if Reply<>'' then SendResult( Reply );
                          end;
                          {JW} {HadLineTooLong}
                          if HadLineTooLong and (Reply<>'') then SendResult( Reply );
                          {JW}
                       end
                      else begin //HSR //ModeCancel
                        Reply := HandleCancelData(LineIn);
                        SetLength(BufInStrm, 0) {.Clear}; BufInStrmLen := 0;
                        if Reply<>'' then SendResult( Reply )
                      end;

                    end;

                    if ClientSocket=nil then Terminate
                    else if not ClientSocket.Connected then Terminate;

                 end;

                 if HadLineTooLong or HadTextTooLarge then begin
                    if HadTextTooLarge then Reply:='textsize-limit exceeded'
                                       else Reply:='linelength-limit exceeded';
                    Log( LOGID_WARN, SockDesc('.Recv') + 'Connection terminated: ' + Reply );
                    if ClientSocket<>nil then begin
                       if ClientSocket.Connected then ClientSocket.Close;
                    end;
                    Terminate;
                 end;

              until (LineEnd=0) or Terminated;

           end else begin
             // If we didn't get any data after ? seconds then close the connection
             Log( LOGID_WARN, 'Connection closed (timeout):' + ClientID );
             if ClientSocket.Connected then ClientSocket.Close;
             Terminate;
           end;
        finally
           SocketStream.Free;
        end;
     except
        Terminate;
     end;
end;

end.

