// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cServerSMTP;

interface

uses Classes, ScktComp, cServerBase, cAccount;

type
  TSrvSMTP = class( TSrvBase )
    public
      constructor Create( AOwner: TComponent );
  end;

type
  TSrvSMTPCli = class(TSrvWorkCli)
    private
      IsESMTP: Boolean;
      AuthOK : Boolean; 
      CurrentUserID  : Integer;
      CurrentUserName: String;
      strHELO: String;
      strFROM: String;
      SASLMechanisms: String; {MG}{SSL+SASL}
      lstRCPT: TStringList;
    public
      procedure SendGreeting; override;
      function  Local_SASL_Login( Mechanism, Parameters: String ): Boolean;
      function  HandleData: String; override;
      procedure HandleCommand( Const CmdLine: String ); override;

      constructor Create( ASocket: TServerClientWinSocket;
                          Const AIPAccessScope: LongInt;
                          Const ASSLContext: Pointer ); override;
      destructor Destroy; override;
  end;

implementation

uses SysUtils, Windows, uTools, uWinSock, Global, Config, cIPAccess, cArticle,
     cArtFiles, uDateTime, uEncoding, cStdForm, cMailRouter, cLogfile, uMD5, uSHA1;

// ------------------------------------------------- Special Functions --------

{JW} {local mail required not permission}
Function AddressContainsLocalDomain(Const FullAddress: string): boolean;
var p1, p2, i: integer; Addr, Domain: string;
begin
   Result := false;
   p1 := pos('<',FullAddress);
   p2 := pos('>',FullAddress);
   If (p1 = 0) or (p2 = 0) or (p2 < p1) then Exit;
   Addr := Copy(FullAddress, p1+1, p2-p1-2+1);
   // decleare emty adress as non local
   if Addr > '' then begin
      i := RScan ( '@', Addr );
      // check if old style local address using
      if i = 0 then begin
         Result := True
      end else begin
         // extract domain part
        Domain := copy(Addr, i+1, Length(Addr)-i );
        // check for bad domains
        Result := (Domain > '') // Domain exists
           and (Pos( '.', Domain )>0) // is valid
           and ((Length(Domain)<8) or (copy(Domain,length(Domain)-7,8)<>'.invalid'))  // .*\.invalid
           and IsLocalDomain ( Domain )
      end
   end
end;
{JW}

// ---------------------------------------------------------- TSrvSMTPCli -----

function TSrvSMTPCli.Local_SASL_Login( Mechanism, Parameters: String ): Boolean;
var  s: String; TimeStamp, Hash: String; {JW} {SASL}
begin
   Result := False;
   CurrentUserName := '';
   CurrentUserID   := ACTID_INVALID;
   try
      if Mechanism='LOGIN' then begin
         If Parameters = '' then begin
            s := 'Username:';
            s := EncodeB64( s[1], length(s) );
            s := SendRequest( Res( R_AUTHCHLG_SMTP, s ) )
         end else begin
            s := Parameters
         end;
         if s='' then Exit;
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_INFO, '> ' + '[...]' );
         CurrentUserName := TrimWhSpace( s );
         CurrentUserID   := ACTID_INVALID;
         s := 'Password:';
         s := EncodeB64( s[1], length(s) );
         s := SendRequest( Res( R_AUTHCHLG_SMTP, s ) );
         if s='' then exit;
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_INFO, '> ' + '[...]' );
         CurrentUserID := CfgAccounts.LoginID( CurrentUserName, s );
         if CurrentUserID=ACTID_INVALID
            then CurrentUserName := ''
            else Result := True;
      end else
     {JW} {SASL}
      if Mechanism='PLAIN' then begin
         If Parameters = '' then begin
            TimeStamp := MidGenerator(Def_FQDNforMIDs);
            s := EncodeB64( TimeStamp[1], length(TimeStamp) );
            s := SendRequest( Res( R_AUTHCHLG_SMTP, s ) )
         end else begin
            s := Parameters
         end;
         if s='' then exit;
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_INFO, '> ' + '[...]' );
         CurrentUserName := TrimWhSpace( copy(s,pos(#0,s)+1,500));
         s:=TrimWhSpace( copy(CurrentUserName,
                         pos(#0,CurrentUserName)+1,500));
         CurrentUserName := TrimWhSpace( copy(CurrentUserName,1,
                                              pos(#0,CurrentUserName)-1));
         CurrentUserID   := ACTID_INVALID;
         CurrentUserID := CfgAccounts.LoginID( CurrentUserName, s );
         if CurrentUserID=ACTID_INVALID then CurrentUserName := ''
                                        else Result := True;
      end else
      if Mechanism='CRAM-MD5' then begin
         TimeStamp := MidGenerator(Def_FQDNforMIDs);
         s := EncodeB64( TimeStamp[1], length(TimeStamp) );
         s := SendRequest( Res( R_AUTHCHLG_SMTP, s ) );
         if s='' then exit;
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_INFO, '> ' + '[...]' );
         Hash:=TrimWhSpace( copy(s,PosWhSpace(s)+1,32));
         CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
         CurrentUserID   := CfgAccounts.UserIDOf(CurrentUserName);
         if CurrentUserID=ACTID_INVALID then begin
            CurrentUserName := ''
         end else begin
            s:=CfgAccounts.Value[CurrentUserID,ACTP_PASSWORD];
            s:=MD5HMAC( s,TimeStamp );
            s:=MD5toHex( s );
            if s=Hash then Result := True
                      else CurrentUserName := ''
         end
      end else
      if Mechanism='CRAM-SHA1' then begin
         TimeStamp := MidGenerator(Def_FQDNforMIDs);
         s := EncodeB64( TimeStamp[1], length(TimeStamp) );
         s := SendRequest( Res( R_AUTHCHLG_SMTP, s ) );
         if s='' then exit;
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_INFO, '> ' + '[...]' );
         Hash:=TrimWhSpace( copy(s,PosWhSpace(s)+1,40));
         CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
         CurrentUserID   := CfgAccounts.UserIDOf(CurrentUserName);
         if CurrentUserID=ACTID_INVALID then begin
            CurrentUserName := '';
            exit;
         end;
         s:=CfgAccounts.Value[CurrentUserID,ACTP_PASSWORD];
         s:=HMAC_SHA1( TimeStamp,s );
         if s=Hash then  Result := True
                   else CurrentUserName := '';
      end
   finally
      If Result then Result := CfgAccounts.Value[ CurrentUserID, ACTP_MAILSEND ] = '1';
      If not result then CurrentUserID:=ACTID_INVALID;  {JW} {Reqfix}
   end
end;

function TSrvSMTPCli.HandleData: String;
begin
   Result := '554 Transaction failed (unknown reason, see logfile)';
   try
      if not ClientSocket.Connected then exit;
      if HadTextTooLarge then begin
         Result := '554 Transaction failed (textsize-limit exceeded)';
         exit;
      end;
      if HadLineTooLong then begin
         Result := '554 Transaction failed (linelength-limit exceeded)';
         exit;
      end;
      if (Length(BufInStrm)<=4{42}) or (strFROM='') or (lstRCPT.Count=0) then begin
         // fail safe, should not happen due to prior tests
         Result := '554 Transaction failed (invalid MAIL/RCPT or DATA)';
         exit;
      end;
      With TRouter.Create do try
         MailFrom := strFROM;
         MailTo.AddStrings(lstRCPT);
         MailText.Text := BufInStrm;
         MailHelo :=StrHELO;
         InIP    :=ClientSocket.RemoteAddr.sin_addr.S_addr;
         InIpStr :=ClientSocket.RemoteAddress;
         {JW} {CurrendUserID}
         UserID  :=CurrentUserID;
         {JW}
         if IsESMTP then MailType:=mtESMTP else MailType:=mtSMTP;
         MailSource:=msOutgoing;
         Execute;
         Result := ResultStr;
      finally
         free
      end
   except
      on E: Exception do
         Log( LOGID_ERROR, SockDesc('.HandleData.Exception: ') + E.Message );
   end;
end;


procedure TSrvSMTPCli.HandleCommand( Const CmdLine: String );
var  Cmd, Par, SASL_Name, s: String; j: Integer;
begin
   try
      if not ClientSocket.Connected then exit;

      Log( LOGID_INFO, '> ' + CmdLine );
      if CmdLine='' then exit;

      if HadTextTooLarge then begin
         SendResult( '552 Textsize-limit exceeded' );
         exit;
      end;
      if HadLineTooLong then begin
         SendResult( '552 Linelength-limit exceeded' );
         exit;
      end;

      // Workaround for tired OutlookXP Clients
      if Def_SMTPDelay>0 then sleep(Def_SMTPDelay); {JW} {SMTP delay}

      // Extract command
      j := PosWhSpace( CmdLine );
      if j=0 then begin
         Cmd := UpperCase( CmdLine );
         Par := '';
      end else begin
         Cmd := UpperCase  ( copy( CmdLine, 1, j-1 ) );
         Par := TrimWhSpace( copy( CmdLine, j+1, 512 ) );
      end;

      // Implemented commands (RFC 821)
      // - Reihenfolge in der Abarbeitung der Kommandos gendert!]
      // - neu : QUIT - HELO - EHLO - HELP - STARTTLS |berprfung ob TLS erzwungen| AUTH usw...]
      // - alt : QUIT - HELO - EHLO - AUTH - ... - HELP - ...]

      if Cmd='QUIT' then begin
         if ClientSocket.Connected then SendResult( '221 closing connection - goodbye!' );
         Sleep( Def_LocalTimeoutQuitDelay );
         try
            if ClientSocket.Connected then ClientSocket.Close;
         except
            on E:Exception do Log(LOGID_DEBUG, 'Exception on .Close: ' + E.Message );
         end;
         Terminate;
      end else

      if Cmd='HELO' then begin
         IsESMTP := False;
         strHELO := TrimWhSpace( copy(CmdLine,5,Length(CmdLine)-4) );
         strFROM := '';
         lstRCPT.Clear;
         SetLength(BufInStrm, 0) {.Clear};
         if strHELO='' then SendResult( '501 Syntax error (missing domain name)' )
                       else SendResult( '250 helo ' + strHELO );
      end else

      if Cmd='EHLO' then begin
         IsESMTP := True;
         strHELO := TrimWhSpace( copy(CmdLine,5,Length(CmdLine)-4) );
         strFROM := '';
         lstRCPT.Clear;
         SetLength(BufInStrm, 0) {.Clear};
         if strHELO='' then begin
            SendResult( '501 Syntax error (missing domain name)' )
         end else begin
            if Def_FQDN='' then SendResult( '250-localhost')
                           else SendResult( '250-'+Def_FQDN);
            SendResult( '250-8BITMIME' );
            SendResult( '250-AUTH ' + SASLMechanisms );
            if pos(' LOGIN ', uppercase(' '+SASLMechanisms+' ')) > 0 then begin //HSR: AuthMoz neu
               SendResult( '250-AUTH=' + SASLMechanisms )
            end;
            if Assigned( SSLContext ) and not Assigned( SSLConnection ) then begin
               SendResult( '250-STARTTLS' )
            end;
            SendResult( '250 HELP' );
         end
      end else

      if Cmd='HELP' then begin
         SendResult( '214-Implemented commands follow:' );
         SendResult( '214-data' );
         SendResult( '214-helo' );
         SendResult( '214-ehlo' );
         SendResult( '214-mail' );
         SendResult( '214-noop' );
         SendResult( '214-quit' );
         SendResult( '214-rcpt' );
         {MG}{SSL}
         if Assigned(SSLContext) and (SSLConnection=nil) then begin
            SendResult( '214-rset' );
            SendResult( '214 starttls' );
         end else
            SendResult( '214 rset' );
      end else

      if Cmd='STARTTLS' then begin // RFC 2487
         if not SSLReady then begin
            SendResult( '502 Command not implemented' );
         end else
         if SSLContext = nil then begin
            SendResult( '454 TLS not available due to temporary error' );
         end else
         if SSLConnection = nil then begin
            SendResult( '220 Begin TLS negotiation' );
            if StartSSL then begin
               // The server MUST discard any knowledge obtained from the
               // client, such as the argument to the EHLO command, which was
               // not obtained from the TLS negotiation itself. [RFC 2487]
               CurrentUserID   := ACTID_INVALID;
               CurrentUserName := '';
               strHELO := '';
               strFROM := '';
               SASLMechanisms := Def_LocalSmtpSASL; {MG}{SSL+SASL}
            end;
         end else begin
            SendResult( '502 TLS session already active' );
         end
      end else

      if (Def_LocalSmtpTlsMode=2) and not Assigned(SSLConnection) then begin
         SendResult( '530 TLS connection required - try STARTTLS' )
      end else

      if Cmd='AUTH' then begin
         if not IsESMTP then begin
            SendResult( '503 Bad sequence of commands (AUTH)' );
            exit;
         end;
         j := Pos( ' ', Par );
         if j=0 then begin
            SASL_Name := Uppercase( Par );
            Par := '';
         end else begin
            SASL_Name := Uppercase( copy( Par, 1, j-1 ) );
            System.Delete( Par, 1, j );
         end;
         if Pos( ' ' + SASL_Name + ' ', ' ' + SASLMechanisms + ' ' )=0 then begin {MG}{SSL+SASL}
            SendResult( '504 Unrecognized authentication type' );
         end else
         if Local_SASL_Login( SASL_NAME, Par ) then begin
            SendResult( '235 Authentication successful.' );
            AuthOK := True;
            CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
         end else begin
            SendResult( '535 Authentication failed.' );
         end
      end else

      if Cmd='NOOP' then begin
         SendResult( '250 OK' );
      end else

      if Cmd='RSET' then begin
         strFROM := '';
         lstRCPT.Clear;
         SetLength(BufInStrm, 0) {.Clear};
         SendResult( '250 OK' );
      end else

      if Cmd='MAIL' then begin
         // reject mail if required auth is missed and local mail deliver is not used
         if not (AuthOK or Def_LocalMailReqNotAuth) then begin
            SendResult( '500 Permission denied' );
         end else
         if UpperCase(copy(CmdLine,1,10))<>'MAIL FROM:' then begin
            SendResult( '504 Command parameter not implemented' );
         end else
         {JW} {RFC 2881 Limit}
         if length(CmdLine)>510 then begin
            SendResult( '500 Syntax error, command too long' );
         end else begin
            s:=TrimWhSpace( copy(CmdLine,11,Length(CmdLine)-10) );
            if length(s)>256 then begin
               SendResult( '501 Parameter error, parameter too long' );
            end else
            {JW} {source routing and required brackets}
            if (pos('<',s) = 0) or (pos('>',s) = 0) then begin
               SendResult( '501 Syntax error, missing brackets' );
            end else begin
              // strip source routing (RFC 2821)
              if pos(':',s)>0 then s := '<'+copy(s,pos(':',s)+1, length(s)-pos(':',s));
              // store envelope sender address
              strFrom := s;
              lstRCPT.Clear;
              SetLength(BufInStrm, 0);
              SendResult( '250 OK' )
            end
         end;
      end else

      if Cmd='RCPT' then begin
         // check if 'mail from' command already execute
         if strFROM='' then begin
            SendResult( '503 Bad sequence of commands (missing FROM)' );
         end else
         {JW} {RFC 2881 Limit}
         if length(CmdLine)>510 then begin
            SendResult( '500 Syntax error, command too long' );
         end else
         // check command option
         if UpperCase(copy(CmdLine,1,8))<>'RCPT TO:' then begin
            SendResult( '504 Command parameter not implemented' );
         end else begin
            // Extract RCPT-To-Address
            s := TrimWhSpace( copy(CmdLine,9,Length(CmdLine)-8));
            {JW} {source routing and required brackets}
            if (pos('<',s) = 0) or (pos('>',s) = 0) then begin
               SendResult( '501 Syntax error, missing brackets' );
            end else begin
               // strip source routing (RFC 2821)
               if pos(':',s)>0 then s := '<'+copy(s,pos(':',s)+1, length(s)-pos(':',s));
               // RFC 2881 Limit
               if length(s)>256 then begin
                  SendResult( '501 Parameter error, parameter too long' );
               end else
               // reject non local mails if not auth ok and local mail deliver is used
               If not (AuthOK or (Def_LocalMailReqNotAuth and AddressContainsLocalDomain(s))) then begin
                  SendResult( '550 Permission denied for relaying' );
               end else begin
                  lstRCPT.Add( s );
                  SetLength(BufInStrm, 0);
                  SendResult( '250 OK' );
               end
            end
         end
      end else

      if Cmd='DATA' then begin
         if lstRCPT.Count=0 then begin
            SendResult( '503 Bad sequence of commands (missing RCPT)' );
         end else begin
            WaitForCmd := False;
            SetLength(BufInStrm, 0) {.Clear};
            SendResult( '354 Start mail input; end with <CRLF>.<CRLF>' )
         end

      end else begin
         SendResult( '502 Command not implemented.' );
         Log( LOGID_INFO, TrGl(kLog, 'Info.UnsupportedSMTPCommand',
            'Unsupported SMTP-command') + ': ' + CmdLine )
      end

   except
      on E: Exception do begin
         Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
         Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + CmdLine );
      end;
   end;
end;

procedure TSrvSMTPCli.SendGreeting;
var  OK: Boolean;
     Msg: String;
begin
     {JW} {LocalLimit}
     if (GetCounter(CntSmtpCli) > Def_Max_Local_SMTP_Servers) and
        (Def_Max_Local_SMTP_Servers>0) then begin
        SendResult( '502 server overload, try again latter' );
        Log( LOGID_WARN, 'SMTP Server overload, too many clients' );
        Terminate;
        exit;
     end;
     {/JW}
     CheckClientAccess; // ( IPACC_SCOPE_SMTP );

     OK  := True;
     Msg := '500 Permission denied - closing connection.';

     if not( AuthOK ) and Def_LocalAuthReqSmtp then begin
        // SMTP-after-POP3
        if CfgAccounts.ChkAuthenticatedLogin(
              ClientSocket.RemoteAddr.sin_addr.S_addr
           ) then begin
           AuthOK := True;
           {JW} {Login}
           CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
           {JW}
        end else begin
           OK  := False;
           Msg := '500 Permission denied (not authorized by POP3)';
        end;
     end;

//PKR 22.12.00
     if OK and ((IPAccess and IPACC_ACCESS_WO)=IPACC_ACCESS_WO) then begin
        If Def_FQDN > '' then
           SendResult( '220 SMTP-Server '
                    + GetMyStringFileInfo('ProductName','Hamster') + ' '
                    + GetMyVersionInfo(true)
                    + ' on ' + Def_FQDN + ' is ready.' )
        else
           SendResult( '220 SMTP-Server '
                    + GetMyStringFileInfo('ProductName','Hamster') + ' '
                    + GetMyVersionInfo(true) );

        exit; //OK
     end;
//ENDE PKR 22.12.00

     Log( LOGID_WARN, TrGlF(kLog, 'Warning.SMTP.ConnectionRefused',
        'Connection refused, %s isn''t allowed to connect to SMTP-Server', ClientID) );
     if ClientSocket<>nil then
     try
        if ClientSocket.Connected then SendResult( Msg );
        if ClientSocket.Connected then ClientSocket.Close;
     except
     end;
     Terminate;
end;

constructor TSrvSMTPCli.Create( ASocket: TServerClientWinSocket;
                                Const AIPAccessScope: LongInt;
                                Const ASSLContext: Pointer );
Var i: Integer;
begin
   inherited Create( ASocket, AIPAccessScope, ASSLContext );

   LimitLineLen  := Def_LocalLimitLineLenSmtp;
   LimitTextSize := Def_LocalLimitTextSizeSmtp;

   IsESMTP := False;
   AuthOK  := not( Def_LocalAuthReqSmtp or Def_LocalAuthReqESmtp );
{JW} {Login}
   If AuthOK
      then CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000
      else CurrentInactTimeOut:=Def_LocalSMTPLoginTimeout;
{JW}
   {MG}{SSL+SASL}
   SASLMechanisms := Def_LocalSmtpSASL;
   i := Pos( 'PLAIN', SASLMechanisms );
   if i > 0 then System.Delete( SASLMechanisms, i, 6 );
   i := Pos( 'EXTERNAL', SASLMechanisms );
   if i > 0 then System.Delete( SASLMechanisms, i, 9 );
   SASLMechanisms := TrimWhSpace( SASLMechanisms );
   {/SSL+SASL}

   CurrentUserID   := ACTID_INVALID;
   CurrentUserName := '';

   strHELO := '';
   strFROM := '';
   lstRCPT := TStringList.Create;

   IncCounter(CntSmtpCli,1);
end;

destructor TSrvSMTPCli.Destroy;
begin
     lstRCPT.Free;
//    dec(CntSmtpCli);
     IncCounter(CntSmtpCli,-1);
     inherited;
end;

// ------------------------------------------------------------- TSrvSMTP -----

constructor TSrvSMTP.Create;
begin
   inherited Create(
      AOwner,
      CfgIni.ReadString ('Setup', 'local.smtp.serverbind', Def_LocalSMTPServerBind ),
      CfgIni.ReadInteger('Setup', 'local.port.smtp', DEF_LOCALSMTPServer_PORT  ),
      CfgIni.ReadInteger('Setup', 'MaxLocalSMTPServers', Def_Max_Local_SMTP_Servers),
      CfgIni.ReadInteger('Setup', 'MaxLocalSMTPServersPerIP', Def_Max_Local_SMTP_Servers_Per_IP),
      IPACC_SCOPE_SMTP,
      TSrvSMTPCli
   );
end;

{procedure TSrvSMTP.MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                                    var SocketThread: TServerClientThread);
begin
     Log( LOGID_DEBUG, Self.Classname + '.OnGetThread' );
     // Create a new thread for connection
     try
        SocketThread := TSrvSMTPCli.Create(true, ClientSocket);
        SocketThread.Resume;
     except
        On E:Exception do Log( LOGID_ERROR, '.OnGetThread.Exception: ' + E.Message );
     end;
end;}

// ----------------------------------------------------------------------------

end.

