unit cStdForm;

// Find Translations: [^a-zA-Z0-9]tr(gl) *\(
// Find not translated log-entries: LOGID_[A-Z]+ *, *[^tT ]

interface

Uses Windows, Controls, extctrls, Forms, Classes, Dialogs, Graphics, Buttons;

Procedure TestLanguage;
Function SetLanguage: boolean;
Function Sprachdatei (Const s: String): String;
Function ExtractLanguageFromFile (Const s: String): String;

Var Sprache: String = '';

Const kMessages = 'Messages';
      kLog = 'Log';
      kGlobal = 'General';
      kNewEntries1 = '; ###################################';
      kNewEntries2 = '; ### NEW ENTRIES FOR TRANSLATION ###';
      kNewEntries3 = '; ###################################';

Function TrGl(Const Abschnitt, Key, Meldung: String): String;
Function TrGlF(Const Abschnitt, Key, Meldung: String; const Args: array of const): String; overload;
Function TrGlF(Const Abschnitt, Key, Meldung, Arg: String): String; overload;
Function HelpFAQID: String;
Function CommandMenuFileName: String;
Procedure SaveChangedLangFiles;

Type
   THForm = Class(TForm)
     private
       First: Boolean;
       Abschnitt: String;
       procedure TranslateAll (Const FirstTime: boolean);
       procedure GeneralFormKeyDown(Sender: TObject; var Key: Word;
         Shift: TShiftState);
       Procedure butHelpOnClick(Sender: TObject);
     protected
       Procedure DoCreate; override;
       Procedure DoShow; override;
       Procedure DoDestroy; override;
       Function  ChangeLanguageFor(c: TComponent): boolean; virtual;
       Procedure Help;
       Function HelpIDFor(Const Key: String): Integer;
     public
       Function Tr(Const Key, Meldung: String): String;
       function TrF(const Key, Meldung: String; const Args: array of const): String; overload;
       function TrF(const Key, Meldung, Arg: String): String; overload;
       Procedure SetFontFixed (F: TFont; Const yes: Boolean);
       Procedure IsFixedFont(Ctrls: Array of TControl);
       Procedure LoadLanguageAgain;
     end;

   TSection = class
    private
      FUsage: String;
      FOrg, FSorted: TStringList;
      function GetItems(i: Integer): String;
    public
      constructor Create;
      destructor Destroy; override;
      Procedure Add (Const s: String; Const bNew: boolean);
      Function Find (Const Key: String; Var Value: String): Boolean;
      Procedure Change(Const Key, newValue: String);
      Function Count: Integer;
      Property Items[i: Integer]: String read GetItems;
      Procedure PrepareFormEntries;
      Function RemarkUnusedFormEntries (Const Action: String): Boolean;
   end;

   TLanguage = Class
     private
       FFilename: String;
       FFileDate: TDateTime;
       FChanged: Boolean;
       FHeader, FSections, FSectionsSorted: TStringList;
       procedure SetFile(const Value: String);
       function HelpID(Const Key: String; Const CreateIfNotExist: boolean): String;
       Function HelpFAQID: String;
       function CommandMenu_Filename: String;
       procedure Check;
       Procedure Clear;
       function Entry(const Sect, Key, DefValue: String;
          Const AutoCreateBlanks: boolean): String;
       function ChangeEntry(const Sect, Key, NewValue: String): String;
       Function AddSection (Const Sect: String): TSection;
       procedure Save;
       procedure Load;
     public
       constructor Create;
       destructor Destroy; override;
       Property Filename: String Write SetFile;
       Function Find(Const Sect: String): TSection;
       procedure CorrectFormEntry(const Sect, Key, TestEntryValue: String);
       procedure CheckControlBounds(Const Sect: String; c: TControl);
       function FormEntry(const Sect, Key, DefValue: String): String;
       function TextEntry(const Sect, Key, DefValue: String): String;
       Function PropFontCharset (Const Sect: String): Integer;
       Function PropFontName (Const Sect: String): String;
       Function PropFontSize (Const Sect: String): Integer;
       Function FixFontCharset (Const Sect: String): Integer;
       Function FixFontName (Const Sect: String): String;
       Function FixFontSize (Const Sect: String): Integer;
       Function HelpFile: String;
       Procedure ReloadIfNewer;
       Procedure PrepareFormEntries (Const Sect: String);
       Procedure RemarkUnusedFormEntries (Const Sect: String);
   end;

implementation

Uses SysUtils, menus, StdCtrls, ComCtrls, uTools, 
     Global, DSprache, Config;

Var UpdateEn: boolean;
    Lang, ELang: TLanguage;

Procedure TestLanguage;
begin
  With CfgIni do begin
     Sprache := ReadString( 'Setup', 'Language', '' );
     UpdateEn := ReadString( 'Setup', 'Language.Update.en.dat', '0' ) = '1';
     If (Sprache = '') or Not FileExists2(Sprachdatei(Sprache)) then begin
        Sprache := 'en';
        If DialogSprache (Sprache)
           then WriteString ('Setup', 'Language', Sprache)
           else Sprache := 'en'
     end;
  end;
  Lang.FileName := Sprachdatei(Sprache);
  Application.HelpFile := PATH_BASE + ExtractFileName(Lang.HelpFile)
end;

Function SetLanguage: boolean;
begin
  With CfgIni do begin
     Sprache := ReadString( 'Setup', 'Language', '' );
     Result := DialogSprache (Sprache);
     If Result then WriteString ('Setup', 'Language', Sprache)
  end;
  Lang.Filename := Sprachdatei(Sprache);
  Application.HelpFile := PATH_BASE + ExtractFileName(Lang.Helpfile)
end;

{ THForm }

Function Sprachdatei (Const s: String): String;
begin
   Result := PATH_BASE
           + ChangeFileExt(ExtractFileName(ParamStr(0)),'')
           + '_' + s + '.dat'
end;

Function ExtractLanguageFromFile (Const s: String): String;
Var p1, p2, i: Integer;
begin
   p1 := 0; For i:=1 to Length(s) do If s[i]='_' then p1 := i;
   p2 := Pos('.dat', LowerCase(s));
   if (p1 > 0) and (p2 > p1)
      then Result := LowerCase(Copy(s, p1+1, p2-p1-1))
      else Result := ''  
end;

procedure THForm.DoCreate;
Var c: TComponent;
begin
   try
      First := true;
      Screen.Cursor := crHourGlass;
      Application.ProcessMessages;
      If HelpContext = 1 then HelpContext := 0;
      KeyPreview := true;
      OnKeyDown :=GeneralFormKeyDown;
      c := FindComponent ('butHelp');
      If Assigned(c) then begin
         If c is TButton then (c as TButton).OnClick := butHelpOnClick
      end;
      c := FindComponent ('mnuHelp');
      If Assigned(c) then begin
         If c is TMenuItem then With (c as TMenuItem) do begin
            If Count = 0 then OnClick := butHelpOnClick
         end
      end;
      Lang.ReloadIfNewer;
      If Assigned(ELang) then ELang.ReloadIfNewer;
      TranslateAll (true);
      inherited;
   except
      Screen.Cursor := crDefault;
      raise
   end
end;

procedure THForm.DoShow;
begin
   If First then begin
      First := false;
      Screen.Cursor := crDefault
   end;
   inherited
end;

procedure THForm.GeneralFormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
   If Key = vk_F1 then Help
end;

Function CRLFtoStrgM(Const Org: String): String;
Var i: Integer;
begin
   Result := Org;
   For i := 1 to Length(Result)-1 do begin
      If Result[i] = #13 then If Result[i+1]=#10 then begin
         Result[i] := '^'; Result[i+1] := 'm'
      end
   end
end;

Function StrgMtoCRLF (Const Org: String): String;
Var i: Integer;
begin
   Result := Org;
   For i := 1 to Length(Result)-1 do begin
      If Result[i] = '^' then If UpCase(Result[i+1])= 'M' then begin
         Result[i] := #13; Result[i+1] := #10
      end
   end
end;

procedure THForm.TranslateAll (Const FirstTime: boolean);
Var Bez: String; C: TComponent; i, j: Integer;

   Function Test(Const s, def: String): String;
   begin
      If FirstTime and (Sprache <> 'en') and UpdateEn then begin
         ELang.CorrectFormEntry(Abschnitt, s, CRLFtoStrgM(def));
         Lang.FormEntry (Abschnitt, s, def)
      end;
      Result := StrgMtoCRLF(Lang.FormEntry (Abschnitt, s, CRLFtoStrgM(def)));
   end;

Var s: String;   
begin
   HelpFile := Application.HelpFile;
   Abschnitt := Classname;
   If Firsttime then begin
      Lang.PrepareFormEntries (Abschnitt);
      If (Sprache <> 'en') and UpdateEn then ELang.PrepareFormEntries (Abschnitt)
   end;
   Font.Charset := Lang.PropFontCharset (Abschnitt);
   Font.Name := Lang.PropFontName (Abschnitt);
   Font.Size := Lang.PropFontSize (Abschnitt);

   Caption := Test('Caption', caption);
   Lang.CheckControlBounds(Abschnitt, self);
   
   For i:=0 to ComponentCount-1 do begin
      c := Components[i];
      // Use this method to avoid unwanted changes!
      If Not ChangeLanguageFor(c) then continue;
      If c is TLabel then With TLabel(c) do begin
         If Name > '' then begin
            Bez := 'LB.'+Name;
            Caption := Test(Bez, caption);
            Hint := Test(Bez+'.hint', hint)
         end
      end else
      If c is TPanel then With TPanel(c) do begin
         If Name > '' then begin
            Bez := 'PA.'+Name;
            If Trim(Caption) > '' then Caption := Test(Bez, caption);
            If Trim(Hint) > '' then Hint := Test(Bez+'.hint', hint)
         end
      end else
      If c is TGroupbox then With TGroupbox(c) do begin
         If Name > '' then begin
            Bez := 'GB.'+Name;
            Caption := ' '+Trim(Test(Bez, caption))+' ';
            If Caption = '  ' then Caption := '';
            Hint := Test(Bez+'.hint', hint)
         end
      end else
      If c is TRadioGroup then With TRadioGroup(c) do begin
         If Name > '' then begin
            Bez := 'RGB.'+Name;
            Caption := ' '+Trim(Test(Bez, caption))+' ';
            If Caption = '  ' then Caption := '';
            For j := 0 to Items.Count-1 do begin
               Bez := 'RGB.'+Name+'.Item'+InttoStr(j);
               Items[j] := Test(Bez, Items[j])
            end
         end
      end else
      If c is TButton then With TButton(c) do begin
         If Name > '' then begin
            Bez := 'BT.'+Name;
            Caption := Test(Bez, caption);
            Hint := Test(Bez+'.hint', hint)
         end
      end else
      If c is TSpeedButton then With TSpeedButton(c) do begin
         If Name > '' then begin
            Bez := 'SB.'+Name;
            Caption := Test(Bez, caption);
            Hint := Test(Bez+'.hint', hint)
         end
      end else
      If c is TMenuItem then With TMenuItem(c) do begin
         If (Name > '') and (Caption <> '-') then begin
            Bez := 'MI.'+Name;
            Caption := Test(Bez, caption);
            Hint := Test(Bez+'.hint', hint);
            s := Lang.HelpID(Abschnitt+'/'+Name, false);
            If s > '' then try
               HelpContext := StrToInt(s)
            except end
         end
      end else
      If c is TCheckbox then With TCheckbox(c) do begin
         If Name > '' then begin
            Bez := 'CB.'+Name;
            Caption := Test(Bez, caption);
            Hint := Test(Bez+'.hint', hint)
         end
      end else
      If c is TRadiobutton then With TRadiobutton(c) do begin
         If Name > '' then begin
            Bez := 'RB.'+Name;
            Caption := Test(Bez, caption);
            Hint := Test(Bez+'.hint', hint)
         end
      end else
      If c is TTabSheet then With TTabSheet(c) do begin
         Bez := 'TS.'+Name;
         Caption := Test(Bez, caption);
         Hint := Test(Bez+'.hint', hint)
      end else
      If c is THeaderControl then With THeaderControl(c) do begin
         For j := 0 to Sections.Count-1 do begin
            Bez := 'HC.'+Name+'.'+InttoStr(j);
            Sections[j].text := Test(Bez, Sections[j].text)
         end
      end else
      If c is TCombobox then With TCombobox(c) do begin
         For j := 0 to Items.Count-1 do begin
            Bez := 'CB.'+Name+'.Item'+InttoStr(j);
            Items[j] := Test(Bez, Items[j])
         end
      end else
      If c is TEdit then With TEdit(c) do begin
         Bez := 'ED.'+Name;
         Hint := Test(Bez+'.hint', hint)
      end else
      If c is TTabControl then With TTabControl(c) do begin
         For j := 0 to Tabs.Count-1 do begin
            Bez := 'TB.'+Name+'.Tab'+InttoStr(j);
            Tabs[j] := Test(Bez, Tabs[j])
         end
      end;
      // Grenanpassungen
      If c is TControl then Lang.CheckControlBounds(Abschnitt, TControl(c))
   end;
   If Firsttime then begin
      Lang.RemarkUnusedFormEntries(Abschnitt);
      If (Sprache <> 'en') and UpdateEn then ELang.RemarkUnusedFormEntries (Abschnitt)
   end
end;

Procedure THForm.LoadLanguageAgain;
begin
   TranslateAll (false)
end;

Function TrGlF(Const Abschnitt, Key, Meldung: String; const Args: array of const): String;
begin
   try
      Result := TrGl(Abschnitt, Key, Meldung);
      Result := Format (Result, Args)
   except
      Result := '#FmtErr#'+Result
   end
end;

Function TrGlF(Const Abschnitt, Key, Meldung, Arg: String): String; overload;
begin
   try
      Result := TrGl(Abschnitt, Key, Meldung);
      Result := Format (Result, [Arg])
   except
      Result := '#FmtErr#'+Result
   end
end;

function THForm.HelpIDFor(const Key: String): Integer;
Var s: String;
begin
   s := Lang.Entry ('HelpIDs', Abschnitt + '/' + Key, '', true);
   Result := 0;
   If s > '' then try Result := StrToInt(s) except end
end;

Function TrGl(Const Abschnitt, Key, Meldung: String): String;
Var Ab, K: String; i: Integer;
begin
   K := Key; Ab := Abschnitt;
   If Ab = '' then Ab := 'Global';
   If K = '' then begin
      For i := 1 to Length(Meldung) do Case Meldung[i] of
         'A'..'Z', 'a'..'z', '0'..'9': K := K + Meldung[i];
         ' ': If (K<>'') then If (K[length(K)]<>'_') and (K[length(K)]<>'.') then K := K + '_';
         else If (K<>'') then If (K[length(K)]<>'_') and (K[length(K)]<>'.') then K := K + '.'
      end;
      If K[Length(K)] IN ['_', '.'] then Delete(K, Length(K), 1)
   end;
   Result := StrgMtoCRLF(Lang.TextEntry (Ab, K, Meldung))
end;

Function THForm.TrF(Const Key, Meldung: String; const Args: array of const): String;
begin
   try
      Result := TrGl(Abschnitt, Key, Meldung);
      Result := Format (Result, Args)
   except
      Result := '#FmtErr#'+Result
   end
end;

Function THForm.TrF(Const Key, Meldung, Arg: String): String;
begin
   try
      Result := TrGl(Abschnitt, Key, Meldung);
      Result := Format (Result, [Arg])
   except
      Result := '#FmtErr#'+Result
   end
end;

Function THForm.Tr(Const Key, Meldung: String): String;
begin
   Result := TrGl(Abschnitt, Key, Meldung)
end;

function THForm.ChangeLanguageFor(c: TComponent): boolean;
begin
   Result := true
end;

procedure THForm.Help;
Var c: TComponent; ID: String;
begin
   ID := Lang.HelpID (Classname, true);
   If Copy(ID, 1, 1)='@' then begin
      Delete(ID, 1, 1);
      c := FindComponent(ID);
      ID := '';
      If Assigned(c) then begin
         If c is TPageControl then begin
            ID := Lang.HelpID (Classname + '/' + IntToStr((c as TPageControl).ActivePageIndex), true)
         end
      end
   end;
   If ID > ''
      then Application.HelpJump(ID)
      else Application.HelpCommand(HELP_CONTENTS, 0)
end;

procedure THForm.butHelpOnClick(Sender: TObject);
begin
   Help
end;

Function CommandMenuFileName: String;
begin
   Result := Lang.CommandMenu_Filename;
end;

Function HelpFAQID: String;
begin
   Result := Lang.HelpFAQID;
end;

procedure THForm.IsFixedFont(Ctrls: array of TControl);
Var i: Integer;
begin
   For i := Low(Ctrls) to High(Ctrls) do begin
      TLabel(Ctrls[i]).Font.Charset := Lang.FixFontCharset (Abschnitt);
      TLabel(Ctrls[i]).Font.Name := Lang.FixFontName (Abschnitt);
      TLabel(Ctrls[i]).Font.Size := Lang.FixFontSize (Abschnitt)
   end
end;

procedure THForm.SetFontFixed(F: TFont; const yes: Boolean);
begin
   With F do begin
      If yes then begin
         Charset := Lang.FixFontCharset (Abschnitt);
         Name := Lang.FixFontName (Abschnitt);
         Size := Lang.FixFontSize (Abschnitt)
      end else begin
         Charset := Lang.PropFontCharset (Abschnitt);
         Name := Lang.PropFontName (Abschnitt);
         Size := Lang.PropFontSize (Abschnitt)
      end
   end
end;

procedure THForm.DoDestroy;
begin
   If First then begin
      First := false;
      Screen.Cursor := crDefault
   end;
   inherited
end;

{ TLanguage }

procedure TLanguage.SetFile(const Value: String);
begin
   Save;
   FFileDate := 0;
   FFileName := Value;
end;

Procedure TLanguage.Clear;
begin
   FHeader.Clear;
   FSectionsSorted.Clear;
   While FSections.Count > 0 do begin
      TSection(FSections.Objects[0]).Free;
      FSections.Delete(0)
   end
end;

procedure TLanguage.CorrectFormEntry (Const Sect, Key, TestEntryValue: String);
Var s: String;
begin
   Check;
   s := FormEntry (Sect, Key, TestEntryValue);
   If s <> TestEntryValue then ChangeEntry(Sect, Key, TestEntryValue)
end;

Procedure TLanguage.Check;
begin
   If (FFileDate = 0) and (FFileName>'') then Load
end;

Procedure TLanguage.Load;
Var bSectStart: Boolean; s, s2: String;
begin
   Clear;
   FFileDate := FileAge ( FFileName );
   FChanged := false;
   If FileExists2(FFileName) then begin
      With TTextReader.Create(FFileName, 8192) do try
         While Not EOF do begin
            s := ReadLine;
            s2 := Trim(s);
            bSectStart := false;
            If s2 > '' then If (s2[1]='[') and (s2[Length(s2)]=']') then bSectStart := true;
            If bSectStart then begin
               FSectionsSorted.AddObject (LowerCase(Copy(s2, 2, Length(s2)-2)),
                 Pointer( FSections.AddObject (s, TSection.Create ) ) )
            end else begin
               If FSections.Count = 0
                  then FHeader.Add (s)
                  else TSection(FSections.Objects[FSections.Count-1]).Add (s, false)
            end
         end
      finally
         free
      end
   end
end;

function TLanguage.AddSection(const Sect: String): TSection;
begin
   Result := TSection.Create;
   FSectionsSorted.AddObject ( LowerCase(Sect),
     Pointer( FSections.AddObject ('['+Sect+']', Result ) ) );
end;

Procedure TLanguage.Save;
Var T: Textfile; i, j: Integer; LastEmpty: Boolean;
begin
   If FFileName = '' then Exit;
   If Not FChanged then Exit;
   If FSections.Count = 0 then Exit;
   try
      AssignFile( T, FFileName );
      FileMode := 1;
      Rewrite (T)
   except
      exit
   end;
   For i := 0 to FHeader.Count-1 do Writeln(T, FHeader[i]);
   For i := 0 to FSections.Count-1 do begin
      Writeln(T, FSections[i]);
      LastEmpty := true;
      With TSection(FSections.Objects[i]) do begin
         For j := 0 to Count-1 do begin
            Writeln(T, Items[j]);
            If (j = Count-1) then LastEmpty := Trim(Items[j]) = ''
         end;
         If Not LastEmpty then Writeln(T)
      end
   end;
   CloseFile(T);
   FFileDate := FileAge ( FFileName );
   FChanged := false
end;

Function TLanguage.Entry (Const Sect, Key, DefValue: String;
   Const AutoCreateBlanks: boolean): String;
Var AktSect: TSection; Value: String;
    IsEmpty: Boolean;
begin
   Result := DefValue;
   Check;
   IsEmpty := ((DefValue = '') or (DefValue = '/')) and (Not AutoCreateBlanks);
   AktSect := Find(Sect);
   If Not Assigned(AktSect) then begin
      If IsEmpty then Exit;
      AktSect := AddSection (Sect);
      FChanged := true
   end;
   If AktSect.Find (Key, Value) then begin
      Result := Value
   end else begin
      If IsEmpty then Exit;
      AktSect.Add (Key+'='+DefValue, true);
      FChanged := true
   end
end;

Function TLanguage.FormEntry (Const Sect, Key, DefValue: String): String;
begin
   Result := Entry(Sect+'/Form', Key, DefValue, false);
end;

Function TLanguage.TextEntry (Const Sect, Key, DefValue: String): String;
begin
   Result := Entry(Sect+'/Texte', Key, DefValue, false);
end;

Function TLanguage.HelpID (Const Key: String; Const CreateIfNotExist: boolean): String;
begin
   Result := Entry ('HelpIDs', Key, '', CreateIfNotExist)
end;

Function TLanguage.HelpFAQID: String;
begin
   Result := Entry ('HelpIDs', 'FAQ', '', true)
end;

Function TLanguage.CommandMenu_Filename: String;
begin
   Result := Entry ('Init', 'CommandMenu', 'HamsterCmds.dat', true);
   If Result = '' then Result := 'HamsterCmds.dat'
end;

constructor TLanguage.Create;
begin
  inherited;
  FHeader := TStringList.Create;
  FSections := TStringList.Create;
  FSectionsSorted := TStringList.Create;
  FSectionsSorted.Sorted := true
end;

destructor TLanguage.Destroy;
begin
  Save;
  Clear;
  FSectionsSorted.Free;
  FSections.Free;
  FHeader.Free;
  inherited
end;

function TLanguage.Find(const Sect: String): TSection;
Var p: integer;
begin
   p := FSectionsSorted.IndexOf (Sect);
   If p >= 0 then begin
      p := Longint(FSectionsSorted.Objects[p]);
      Result := TSection(FSections.Objects[p])
   end else begin
      Result := NIL
   end
end;

function TLanguage.HelpFile: String;

  Function Check(Const Liste: Array of String): String;
  Var i: Integer;
  begin
     Result := '';
     For i := Low(Liste) to High(Liste) do begin
        If fileexists(PATH_BASE + Liste[i]) then Result := Liste[i]
     end
  end;

Var Def: String;
begin
  Def := Check(['hamster.hlp', 'hamster_en.hlp', 'hamster_en.chm',
        ChangeFileExt(ExtractFileName(FFileName), '.hlp'),
        ChangeFileExt(ExtractFileName(FFileName), '.chm')]);
  Result := Entry ( 'Init', 'HelpFile', Def, false);
  If Not fileexists(PATH_BASE + Result) then Result := Def;
end;

function TLanguage.ChangeEntry(const Sect, Key, NewValue: String): String;
Var AktSect: TSection;
begin
   Check;
   AktSect := Find(Sect);
   If Assigned(AktSect) then begin
      AktSect.Change(Key, newValue);
      FChanged := true
   end
end;

procedure TLanguage.PrepareFormEntries (Const Sect: String);
Var AktSect: TSection;
begin
   Check;
   AktSect := Find(Sect+'/Form');
   If Assigned(AktSect) then AktSect.PrepareFormEntries
end;

procedure TLanguage.RemarkUnusedFormEntries (Const Sect: String);
Var AktSect: TSection;
begin
   Check;
   AktSect := Find(Sect+'/Form');
   If Assigned(AktSect) then begin
      If AktSect.RemarkUnusedFormEntries (
         Entry ( 'Init', 'ActionForUnusedEntries', 'C', false )
      ) then begin
         FChanged := true
      end
   end
end;

procedure TLanguage.ReloadIfNewer;
begin
   If FFileDate < FileAge ( FFileName ) then Load
end;

function TLanguage.PropFontName(Const Sect: String): String;
begin
   Result := Entry ( Sect, 'PropFont.Name', '', false );
   If Result = '' then Result := Entry ( 'Init', 'PropFont.Name', 'MS Sans Serif', true )
end;
function TLanguage.PropFontSize(Const Sect: String): Integer;
Var s: String;
begin
   s := Entry ( Sect, 'PropFont.Size', '', false );
   If s = '' then s := Entry ( 'Init', 'PropFont.Size', '8', true );
   try Result := StrToInt(s) except Result := 8 end
end;

function TLanguage.FixFontName(Const Sect: String): String;
begin
   Result := Entry ( Sect, 'FixFont.Name', '', false );
   If Result = '' then Result := Entry ( 'Init', 'FixFont.Name', 'Courier New', true )
end;

function TLanguage.FixFontSize(Const Sect: String): Integer;
Var s: String;
begin
   s := Entry ( Sect, 'FixFont.Size', '', false );
   If s = '' then s := Entry ( 'Init', 'FixFont.Size', '10', true );
   try Result := StrToInt(s) except Result := 10 end
end;

function TLanguage.FixFontCharset(const Sect: String): Integer;
Var s: String;
begin
   s := Entry ( Sect, 'FixFont.Charset', '', false );
   If s = '' then s := Entry ( 'Init', 'FixFont.Charset', '0', true );
   try Result := StrToInt(s) except Result := 0 end
end;

function TLanguage.PropFontCharset(const Sect: String): Integer;
Var s: String;
begin
   s := Entry ( Sect, 'PropFont.Charset', '', false );
   If s = '' then s := Entry ( 'Init', 'PropFont.Charset', '0', true );
   try Result := StrToInt(s) except Result := 0 end
end;

procedure TLanguage.CheckControlBounds(const Sect: String; c: TControl);
Var Key, s: String; i: Integer; R: TRect; Change: Boolean;
begin
   try
      If c is TForm then Key := '' else Key := c.Name+'.';
      R := C.BoundsRect;
      change := false;
      s := Entry ( Sect, Key+'AddLeft', '', false);
      If s > '' then begin
         i := StrToInt(s); change := true;
         R.Left := R.Left + i; R.Right := R.Right + i
      end;
      s := Entry ( Sect, Key+'AddTop', '', false);
      If s > '' then begin
         i := StrToInt(s); change := true;
         R.Top := R.Top + i; R.Bottom := R.Bottom + i
      end;
      s := Entry ( Sect, Key+'AddWidth', '', false);
      If s > '' then begin
         i := StrToInt(s); change := true;
         R.Right := R.Right + i
      end;
      s := Entry ( Sect, Key+'AddHeight', '', false);
      If s > '' then begin
         i := StrToInt(s); change := true;
         R.Bottom := R.Bottom + i
      end;
      If change then c.BoundsRect := R
   except end
end;

{ TSection }

procedure TSection.Add(const s: String; Const bNew: boolean);
Var p: Integer;
begin
   If FUsage > '' then FUsage := '';
   If bNew then begin
      If FOrg.IndexOf(kNewEntries2) < 0 then begin
         FOrg.Add(kNewEntries1);
         FOrg.Add(kNewEntries2);
         FOrg.Add(kNewEntries3);
      end;
      If FOrg.Count > 0 then If Trim(FOrg[FOrg.Count-1])='' then begin
         FOrg.Delete(FOrg.Count-1)
      end
   end;
   p := Pos('=', s);
   If p = 0 then begin
      FOrg.Add (s)
   end else begin
      FSorted.AddObject ( LowerCase(Trim(Copy(s, 1, p-1))),
                          Pointer ( FOrg.AddObject(s, Pointer(p+1)) ) )
   end
end;

procedure TSection.Change(const Key, newValue: String);
Var p, Start: Integer;
begin
   p := FSorted.IndexOf(Key);
   If p >= 0 then begin
      p := Longint(FSorted.Objects[p]);
      Start := Longint(FOrg.Objects[p]);
      FOrg[p] := Copy(FOrg[p], 1, Start-1) + NewValue
   end
end;

function TSection.Count: Integer;
begin
   Result := FOrg.Count
end;

constructor TSection.Create;
begin
  inherited;
  FOrg := TStringList.Create;
  FSorted := TStringList.Create;
  FSorted.sorted := true
end;

destructor TSection.Destroy;
begin
  FOrg.Free;
  FSorted.Free;
  inherited;
end;

function TSection.Find(const Key: String; var Value: String): Boolean;
Var p: Integer;
begin
   p := FSorted.IndexOf(Key);
   Result := p >= 0;
   If Result then begin
      If Length(FUsage) > p then FUsage[p+1] := 'X';
      p := Longint(FSorted.Objects[p]);
      Value := Copy(FOrg[p], Longint(FOrg.Objects[p]), Length(FOrg[p]))
   end
end;

function TSection.GetItems(i: Integer): String;
begin
   Result := FOrg[i]
end;

procedure TSection.PrepareFormEntries;
Var i: Integer;
begin
   SetLength(FUsage, FSorted.Count);
   For i := 1 to Length(FUsage) do FUsage[i] := ' '
end;

function TSection.RemarkUnusedFormEntries (Const Action: String): Boolean;
Const Unmark = '; #UNUSED# ';
Var i, j, p, p2: Integer; bDelete: Boolean;
begin
   Result := false;

   Case Upcase((Action+'C')[1]) of
      'C': bDelete := false;
      'D': bDelete := true;
      else Exit;
   end;

   For i := Length(FUsage) downto 1 do If FUsage[i] = ' ' then begin
      p := Longint(FSorted.Objects[i-1]);
      If bDelete then begin
         FOrg.Delete(p);
         Result := true;
         // Correction of sorted List
         For j := FSorted.Count-1 downto 0 do begin
            p2 := Longint(FSorted.Objects[j]);
            If p2 = p then FSorted.Delete(j)
            else If p2 > p then FSorted.Objects[j] := Pointer ( p2-1 )
         end
      end else begin
         If Copy(FOrg[p], 1, Length(Unmark)) <> Unmark then begin
            // Correct position of "="
            FOrg.Objects[p] := Pointer( Longint(FOrg.Objects[p]) + Length(Unmark) );
            // Unmark Entry
            FOrg.Strings[p] := Unmark + FOrg[p];
            Result := true
         end
      end
   end
end;

Procedure SaveChangedLangFiles;
begin
   If Assigned(Lang) then With Lang do If FChanged then Save;
   If Assigned(ELang) then With ELang do If FChanged then Save;
end;

initialization
   ELang := TLanguage.Create;
   ELang.Filename := Sprachdatei('en');
   Lang := TLanguage.Create;
finalization
   Lang.Free;
   ELang.Free
end.
