{MG}{IMAP}

unit cIMAPMessage; // Class to handle a single imap message.

// ----------------------------------------------------------------------------
// Contains a class to handle an single imap message, which allows i.e.
// RFC-822 and MIME-IMB parsing, and selective fetching of message attributes,
// texts, and portions thereof.
// ----------------------------------------------------------------------------

interface

uses Classes, cArticle, cLogFile;

type

  TImapMessage = class( TArticle )
    private
      ContentType    : String;
      ContentSubtype : String;
      BodyLines      : Integer;
      Parts          : Array of TImapMessage;
      function GetHeaderFields( Fields: String; Exclude: Boolean ): String;
      function GetBodyFields: String;
      function GetAddresses( HdrNam: String ): String;
      function GetAddressStructure( Address: String ): String;
      function GetParameter( HdrNam: String ): String;
      function GetBodyDisposition: String;
      function GetBodyMD5: String;
      function GetBodyLanguage: String;
      function GetEnvelope: String;
      function GetBodyLines: Integer;
      function GetBodySection( Section: String; Nested: Boolean;
                               var Offset, Maximum: Integer ): String;
      procedure ParseMessage;
    public
      property Envelope: String read GetEnvelope;
      property Lines: Integer read GetBodyLines;
      function BodySection( var Section: String ): String;
      function GetHeaderNString( HdrNam: String ): String;
      function BodyStructure( Extensible: Boolean ): String;
      procedure LoadFromFile( const FileName: String );
      destructor Destroy; override;
  end;

// ----------------------------------------------------------------------------

implementation

uses SysUtils, Global, uTools, uImapUtils, cAccount;

function TImapMessage.GetHeaderFields( Fields: String; Exclude: Boolean ): String;
  function DeleteEOL(Value : String) : String;
  var i : integer;
  begin
    Result := Value;
    for i := length(Result) downto 1 do
      if Result[i] in [#13, #10] then
        System.Delete(Result,i,1);
  end;

var  AllHeaders : String;
     Data       : String;
     s          : String;
     i          : Integer;
begin
     Result := '';
     Data := '';

     try
        //'Header1 Header2   Header3' --> ':Header1:Header2:Header3:'
        AllHeaders := ':' + Lowercase( Fields ) + ':';
        i := PosWhSpace( AllHeaders );
        while i > 0 do begin
           AllHeaders[i] := ':';
           while AllHeaders[i+1] in [#9,' '] do System.Delete( AllHeaders, i+1, 1 );
           i := PosWhSpace( AllHeaders )
        end;

        i := 0;
        while i < Headercount do begin
           s := Lowercase( Copy( HeaderLine[i], 1, Pos(':',HeaderLine[i])-1 ) );
           if s <> '' then begin
              if ( Pos(':'+s+':', AllHeaders) > 0 ) xor Exclude then begin
                 Data := Data + CRLF + TrimWhSpace( DeleteEOL(HeaderLine[i]) );
                 while i+1 < Headercount do begin //Mehrzeilige Header?
                                                  //ToDo: tArticle enthlt jetzt Unterstzzung dafr
                    s := Copy( HeaderLine[i+1], 1, 1 );
                    if not ( (s = ' ') or (s = #9) ) then break;
                    Data := Data + ' ' + TrimWhSpace(DeleteEOL(HeaderLine[i+1])) ;
                    inc( i )
                 end
              end
           end;
           inc( i )
        end;
        System.Delete(Data,1,2);
        Result := Trim( Data ) + CRLF;

     except
        Result := '';
     end;
end;

function TImapMessage.GetBodyFields: String;
begin
     Result := GetParameter( 'Content-Type:' ) + ' ' +
               GetHeaderNString( 'Content-ID:' ) + ' ' +
               GetHeaderNString( 'Content-Description:' ) + ' ' +
               GetHeaderNString( 'Content-Transfer-Encoding:' ) + ' ' +
               IntToStr( Length( FullBody ) )
end;

function TImapMessage.GetHeaderNString( HdrNam: String ): String;
begin
   Result := NString( HeaderWOCRLF[ HdrNam ] )
end;

function TImapMessage.GetAddresses( HdrNam: String ): String;
var  i : Integer;
     s : String;
begin
     s := HeaderWOCRLF[ HdrNam ];
     if s = '' then begin
        // default Sender: and Reply-To: to From:
        if (HdrNam = 'reply-to:') or (HdrNam = 'sender:') then
           s := HeaderWOCRLF[ 'from:' ];
     end;
     if s = '' then begin
        Result := 'NIL'
     end else begin
        // TODO
        // ----
        // [RFC-822] group syntax is indicated by a special form of address
        // structure in which the host name field is NIL.  If the mailbox
        // name field is also NIL, this is an end of group marker
        // (semi-colon in RFC 822 syntax).  If the mailbox name field is
        // non-NIL, this is a start of group marker, and the mailbox name
        // field holds the group name phrase.
        i := Pos( ',', s );
        while i > 0 do begin
           Result := Result + GetAddressStructure( TrimWhSpace( Copy( s, 1, i-1 ) ) );
           System.Delete( s, 1, i );
           i := Pos( ',', s )
        end;
        Result := '(' + Result + GetAddressStructure( TrimWhSpace( s ) ) + ')';
     end
end;

function TImapMessage.GetAddressStructure( Address: String ): String;
// An address structure is a parenthesized list that describes an electronic
// mail address.  The fields of an address structure are in the following
// order: personal name, [SMTP] at-domain-list (source route), mailbox name,
// and host name.
var  Name, Mailbox, Host, s : String;
     i : Integer;
begin
     Mailbox     := 'NIL';
     Host        := 'NIL';

     s := Address;
     i := PosWhSpace( s );
     if Pos( '@', s ) < i then begin       // gray@cac.washington.edu (Terry Gray)
        Name := TrimParentheses( Copy( s, i+1, Length(s)-i ) );
        SetLength( s, i-1 );
     end else begin                        // Terry Gray <gray@cac.washington.edu>
        for i := Length(s)-5 downto 1 do if s[i] in [#9,' '] then begin
           Name := TrimQuotes( Copy( s, 1, i-1 ) );
           System.Delete( s, 1, i );
           break
        end
     end;
     Name := NString( Name );
     s := ExtractMailAddr( s );
     i := Pos( '@', s );
     if i > 0 then begin
        Mailbox := NString( Copy( s, 1, i-1 ) );
        Host    := NString( Copy( s, i+1, Length(s) ) );
     end else
        Mailbox := NString( s );

     // ("Terry Gray" NIL "gray" "cac.washington.edu")
     Result := '(' + Name + ' NIL ' + Mailbox + ' ' + Host + ')'
end;

function TImapMessage.GetBodySection( Section: String; Nested: Boolean;
                                      var Offset, Maximum: Integer ): String;
// The text of a particular body section. The section specification is a set
// of zero or more part specifiers delimited by periods. A part specifier is
// either a part number or one of the following: HEADER, HEADER.FIELDS,
// HEADER.FIELDS.NOT, MIME, and TEXT. An empty section specification refers
// to the entire message, including the header.
var  Part       : Integer;
     SubSection : String;
     Fields     : String;
begin
     Result := '';

     SplitString( '.', Section, SubSection );

     // The TEXT part specifier refers to the text body of the message,
     // omitting the [RFC-822] header.
     if Section = 'TEXT' then begin
        Result := FullBody

     end else if Section = '' then begin
        if Nested then Result := FullBody
        else           Result := Text

     end else if Section = 'HEADER' then begin
        if SubSection = '' then Result := FullHeader + CRLF
        else if Copy( SubSection, 1, 6 ) = 'FIELDS' then begin
           SplitStringWhSpace( Subsection, Fields );
           Result := GetHeaderFields( TrimParentheses(Fields),
                                      Copy(SubSection,7,4) = '.NOT' ) + CRLF
        end

     // The MIME part specifier refers to the [MIME-IMB] header for this part.
     // The MIME part specifier MUST be prefixed by one or more numeric part specifiers.
     end else if Section = 'MIME' then begin
        if Nested then Result := FullHeader + CRLF
        else Log( LOGID_WARN, 'Invalid Client operation: Fetching body section - MIME '+
                              'part specifier not prefixed by numeric part specifier.' )

     end else begin
        Part := StrToInt( Section );
        if Part > 0 then begin
           if ContentType = 'MULTIPART' then begin
              if Part > Length( Parts ) then
                 Log( LOGID_WARN, 'Invalid Client operation: Fetching part '+
                      IntToStr(Part) + ' of ' + IntToStr(Length(Parts)) + ' part message.' )
              else Result := Parts[Part-1].GetBodySection( SubSection, True, Offset, Maximum );
           end else begin
              // Non-[MIME-IMB] messages, and non-multipart [MIME-IMB] messages
              // with no encapsulated message, only have a part 1.
              if Part = 1 then Result := FullBody
              else Log( LOGID_WARN, 'Invalid Client operation: Fetching part ' +
                        IntToStr(Part) + ' of non-multipart message.' );
           end;
        end else Log( LOGID_WARN, 'Invalid Client operation: ' +
                      'Fetching body section ' + Section + '.' );
     end;

     if Offset > 0 then begin
        System.Delete( Result, 1, Offset );
        Offset := 0;
     end;
     if Maximum > -1 then begin
        if Maximum < Length(Result) then SetLength( Result, Maximum );
        Maximum := -1;
     end;
end;

function TImapMessage.BodySection( var Section: String ): String;
var  Offset, Maximum, i: Integer;
     s: String;
begin
     Offset  :=  0;
     Maximum := -1;

     i := Pos('<', Section);
     if (i > 0) and (Section[Length(Section)] = '>') then begin
        s := Copy( Section, i, Length(Section)-i+1 );
        i := Pos( '.', s );
        if i <> 0 then begin
           Offset  := StrToIntDef( Copy( s, 2, i-2 ), 0 );
           Maximum := StrToIntDef( Copy( s, i+1, Length(s)-i-1 ), -1 );
           System.Delete( Section, Pos('<',Section)+i-1, Length(s)-i );
        end
     end;

     Result := GetBodySection( Copy( Section, 6, Pos(']',Section)-6 ),
                               False, Offset, Maximum )
end;

function TImapMessage.GetParameter( HdrNam: String ): String;
//  A parenthesized list of attribute/value pairs [e.g. ("foo" "bar" "baz" "rag")
//  where "bar" is the value of "foo" and "rag" is the value of  "baz"] as
//  defined in [MIME-IMB].
var  i, ContinuationCount: Integer;
     Params, s, Parameter, Value: String;
begin
     Result := '';
     ContinuationCount := 0;
     Params := HeaderWOCRLF[ HdrNam ];
     i := Pos( ';', Params );
     while i > 0 do begin
        System.Delete( Params, 1, i );
        i := Pos( ';', Params );
        if i = 0 then s := TrimWhSpace( Params )
        else s := TrimWhSpace( Copy( Params, 1, i-1 ) );

        i := Pos( '=', s );
        if i > 0 then begin
           Parameter := Uppercase( Copy( s, 1, i-1 ) );
           System.Delete( s, 1, i );
           if s = '' then break;
           Value := QuotedStringOrToken( s );

           if (Parameter = '') or (Value = '') then continue;

           // Decoding of parameter value continuations [RFC 2184]
           // TODO: ContinuationCount > 9
           // TODO: Correct handling of Character Set and Language Information
           i := Length(Parameter);
           if Parameter[i] = '*' then dec( i );
           if (Parameter[i-1] = '*') and
              (Parameter[i] = IntToStr(ContinuationCount) ) then begin
              if ContinuationCount = 0 then begin
                 SetLength( Parameter, i-2 );
                 Result := Result + '"' + Parameter + '" "' + Value + '" ';
              end else begin
                 System.Delete( Result, Length(Result)-1, 2 );
                 Result := Result + Value + '" ';
              end;
              inc( ContinuationCount );
           end else begin
              if ContinuationCount > 0 then ContinuationCount := 0;
              Result := Result + '"' + Parameter + '" "' + Value + '" ';
           end

        end;
        i := Pos( ';', Params )
     end;
     if Result = '' then Result := 'NIL'
     else Result := '(' + TrimWhSpace( Result ) + ')'
end;

function TImapMessage.GetBodyDisposition: String;   // RFC 2183
var  s: String;
     i: Integer;
begin
     s := HeaderWOCRLF[ 'Content-Disposition:' ];
     if s <> '' then begin
        i := Pos( ';', s );
        if i > 0 then SetLength( s, i-1 );
        Result := '("' + Uppercase( TrimWhSpace(s) ) + '" ' +
                  GetParameter( 'Content-Disposition:' ) + ')'
     end else
        Result := 'NIL'
end;

function TImapMessage.GetBodyMD5: String;
begin
     Result := GetHeaderNString( 'Content-MD5:' )
end;

function TImapMessage.GetBodyLanguage: String;
// A string or parenthesized list giving the body language value
// as defined in RFC 1766
var  s, Data : String;
     i : Integer;
begin
     s := HeaderWOCRLF[ 'Content-Language:' ];
     if s <> '' then begin
        i := Pos( ',', s );
        while i > 0 do begin
           Data := Data + ' "' + TrimWhSpace( Copy ( s, 1, i-1 ) ) + '"';
           System.Delete( s, 1, i );
           i := Pos( ',', s );
        end;
        if Data <> '' then Result := '(' + TrimWhSpace( Data ) + ')'
        else Result := '"' + s + '"'
     end else Result := 'NIL'
end;

function TImapMessage.GetEnvelope: String;
// A parsed representation of the [RFC-2822] envelope information (not
// to be confused with an [SMTP] envelope) of the message.

      function replace8BitChar(text:string;var change:boolean):string; //JW //IMAP 7Bit
      var i,j:word;
        begin
          change:=false;
          j:=length(text);
          for i:=1 to j do
            if (byte(text[i])>127) then begin
              text[i]:='?';
              change:=true;
            end;
          result:=text;
        end;

var change : Boolean;
begin
     Result := '(' + GetHeaderNString( 'date:' ) +
               ' ' + GetHeaderNString( 'subject:' ) +
               ' ' + GetAddresses( 'from:' ) +
               ' ' + GetAddresses( 'sender:' ) +
               ' ' + GetAddresses( 'reply-to:' ) +
               ' ' + GetAddresses( 'to:' ) +
               ' ' + GetAddresses( 'cc:' ) +
               ' ' + GetAddresses( 'bcc:' ) +
               ' ' + GetHeaderNString( 'in-reply-to:' ) +
               ' ' + GetHeaderNString( 'message-id:' ) + ')';
{JW} {IMAP 7Bit}
    if Def_IMAP8bitFilter then begin
       Result := replace8BitChar(Result, change);
       if change then
          Log( LOGID_WARN,'IMAP: 8-Bit char in envelope correted!');
    end;
{JW}

end;

function TImapMessage.GetBodyLines: Integer;
var  s : String;
     i : Integer;
begin
     if BodyLines < 0 then begin
        BodyLines := 0;
        s := FullBody;
        for i := 1 to Length(s)-1 do
           if (s[i] = #13) and (s[i+1] = #10) then inc( BodyLines );
     end;
     Result := BodyLines
end;

function TImapMessage.BodyStructure( Extensible: Boolean ): String;
var  Data : String;
     i    : Integer;
begin
     if ContentType = 'MULTIPART' then begin                        // Multipart
        // Multiple parts are indicated by parenthesis nesting.  Instead of
        // a body type as the first element of the parenthesized list there is
        // a nested body.  The second element of the parenthesized list is the
        // multipart subtype (mixed, digest, parallel, alternative, etc.). }
        Data := '(';
        for i := 0 to High( Parts ) do
            Data := Data + Parts[i].BodyStructure( Extensible );
        Data := Data + ' "' + ContentSubtype +'"';

        if Extensible then
           Data := Data + ' ' + GetParameter( 'Content-Type:' ) + ' ' +
                   GetBodyDisposition + ' ' + GetBodyLanguage;

     end else begin

        Data := '("' + ContentType + '" "' + ContentSubtype + '" ' + GetBodyFields;

        if ContentType = 'TEXT' then Data := Data + ' ' + IntToStr( Lines )

        else if (ContentType = 'MESSAGE') and (ContentSubtype = 'RFC822') then begin
           if Length(Parts) > 0 then
              Data := Data + ' ' + Envelope + ' ' +
                      Parts[0].BodyStructure( Extensible ) + ' ' +
                      IntToStr( Lines )
           else
              Log( LOGID_WARN, 'Error parsing RFC822 message: There is no message.' );
        end;

        if Extensible then Data := Data + ' ' + GetBodyMD5 + ' ' +
                                   GetBodyDisposition + ' ' + GetBodyLanguage;
     end;

     Result := Data + ')'
end;

procedure TImapMessage.ParseMessage;
var
     i, BoundaryLength : Integer;
     s, BoundaryString : String;
     CurrentPart       : Integer;
begin
     BodyLines := -1;            // don't count the body lines unless requested
     BoundaryString := '';
     SetLength( Parts, 0 );

     s := HeaderWOCRLF[ 'Content-Type:' ];
     if s = '' then s := 'text/plain';
     i := Pos( '/', s );
     if i > 0 then begin
        ContentType    := Uppercase( Copy( s, 1, i-1 ) );
        ContentSubtype := Uppercase( Copy( s, i+1, Length(s)-i ) );
        i := Pos( ';', ContentSubtype );
        if i > 0 then SetLength( ContentSubtype, i-1 );
     end else
        Log( LOGID_WARN, 'Error parsing Header ''Content-Type''.' );

     if ContentType = 'MULTIPART' then begin
        i := Pos( 'BOUNDARY=', uppercase(s) );
        if i > 0 then begin
           System.Delete( s, 1, i+8 );
           s := QuotedStringOrToken( s );

           // If a boundary delimiter line appears to end with white space,
           // the white space must be presumed to have been added by a gateway,
           // and must be deleted.
           while s[Length(s)] in [#9,' '] do System.Delete( s, Length(s), 1 );

           // The CRLF preceding the boundary delimiter line is conceptually
           // attached to the boundary so that it is possible to have a part
           // that does not end with a CRLF (line break).
           BoundaryString := CRLF + '--' + s;
           BoundaryLength := Length( BoundaryString );
        end else begin
           Log( LOGID_WARN, 'Error parsing Multipart message: ''boundary'' not found.' );
           exit
        end;

        CurrentPart := -1;
        s := FullBody;

        i := Pos( BoundaryString, s );
        while i > 0 do begin
           if CurrentPart > -1 then begin
              SetLength( Parts, CurrentPart + 1 );
              Parts[CurrentPart] := TImapMessage.Create;
              try
                 Parts[CurrentPart].Text := Copy( s, 1, i-1 );
                 Parts[CurrentPart].ParseMessage;
              except
                 on E:Exception do begin
                    Log( LOGID_ERROR, 'Exception parsing multipart message: ' + E.Message );
                    Parts[CurrentPart].Free;
                    SetLength( Parts, CurrentPart );
                 end
              end
           end;
           System.Delete( s, 1, i-1+BoundaryLength );
           if Copy(s,1,2) = '--' then break;                  // last body part
           i := Pos( CRLF, s );
           if i > 0 then System.Delete( s, 1, i+1 )
           else break;                                        // unexpected end
           i := Pos( BoundaryString, s );
           inc( CurrentPart )
        end
     end

     else if (ContentType = 'MESSAGE') and (ContentSubtype = 'RFC822') then begin
        try
           SetLength( Parts, 1 );
           Parts[0] := TImapMessage.Create;
           Parts[0].Text := FullBody;
           Parts[0].ParseMessage;
        except
           on E:Exception do begin
              Log( LOGID_ERROR, 'Exception parsing RFC822 message: ' + E.Message );
              if Length(Parts) > 0 then Parts[0].Free;
              SetLength( Parts, 0 );
           end
        end
     end
end;

procedure TImapMessage.LoadFromFile( const FileName: String );
Var s: String;
begin
   With TFileStream.Create( FileName, fmOpenRead or fmShareDenyWrite ) do try
      SetLength(s, Size);
      If Size > 0 then Read( s[1], Size );
   finally
      Free
   end;
   Text := S;
   ParseMessage
end;

destructor TImapMessage.Destroy;
var  i : Integer;
begin
     for i := 0 to High( Parts ) do FreeAndNil( Parts[i] );
     inherited Destroy;
end;

// ----------------------------------------------------------------------------

end.

