unit cServerIMAP; //HSR //IMAP

interface

uses ScktComp, Classes, Windows, cServerBase, IniFiles, uDateTime, cChunkRc,
     ExtCtrls, cImapMessage, cIMAPMailBox, cIMAPMailBoxIndex;

const IMAPSUBSCRIBED_FILENAME = 'Subscr.dat';

type
  TSrvIMAP = class( TSrvBase )
    public
      constructor Create( AOwner: TComponent );
      {procedure MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                              var SocketThread: TServerClientThread); override;}
  end;

  TSrvIMAPCli = class(TSrvWorkCli)
    private
      HierarchyDelimiter : Char;
      SubscribedMBs  : TStringList;
      CurrentUserID  : Integer;
      CurrentUserName: String;
      CurrentTag     : String;

      LiteralCount   : Integer;

      MailboxPath    : String;
      Selected       : TImapMailbox;
      fReadOnly      : Boolean;

      {HSR} {IDLE}
      SendNewMessages : Boolean;
      SendExpunge     : array of Integer;
      IdleState       : Boolean;
      CS_THR_IDLE     : TRTLCriticalSection;
      SelNotify       : pIMAPNotification;
      {/HSR}

      procedure NewMessages; //HSR //IDLE
      procedure NewExpunge(Number : Integer); //HSR //IDLE

      function LoginUser( Password: String; AuthMechanism : String ): String;
      function SafeString( Path: String ): Boolean;
      function ReplaceChars( S: String; OldChar, NewChar: Char ): String; {MG}{IMAP-List}
      function ReplacePathDelimiters( Path: String ): String;
      function GetStatus( Mailbox: String ): TMbxStatus;
      function RebuildStatusFile( Mailbox: String ): TMbxStatus;

      procedure SendResLit( Txt: String );
      procedure SendResTag( Txt: String);
      procedure SendRes( Txt: String);

      function  MBLogin(  var Mailbox: TImapMailbox; Path: String; LINotify : Boolean ): Boolean; //HSR //IDLE (Chg)
      procedure MBLogout( var Mailbox: TImapMailbox; LOSel : Boolean ); //HSR //IDLE (Chg)

      function  MBSelect( Mailbox: string; ReadOnly : Boolean ): boolean;
      function  MBCreate( Mailbox: string ): boolean;
      function  MBDelete( Mailbox: string ): boolean;
      function  MBExists( var Mailbox: string ): boolean;
      function  MBRename( OldName, NewName: String ): Boolean;

      procedure DoSearch( UseUID: Boolean; Par: String );
      procedure DoCopy( MsgSet: TMessageSet; Command, Destination: String );
      procedure DoStore( MsgSet: TMessageSet; Command, Par: String );
      procedure DoFetch( MsgSet: TMessageSet; Command, Par: String );
      procedure DoList( Par: String; LSub: Boolean ); {MG}{IMAP-List}
    public
      procedure ClientExecute; override;
      procedure SendGreeting; override;
      function  HandleData: String; override;
      procedure HandleCommand( Const CmdLine: String ); override;

      procedure Cmd_APPEND   ( Par: String );
      procedure Cmd_AUTHENTICATE( Par: String );
      procedure Cmd_CAPA     ( Par: String );
      procedure Cmd_CHECK    ( Par: String );
      procedure Cmd_CLOSE    ( Par: String );
      procedure Cmd_COPY     ( Par: String );
      procedure Cmd_CREATE   ( Par: String );
      procedure Cmd_DELETE   ( Par: String );
      procedure Cmd_EXAMINE  ( Par: String );
      procedure Cmd_EXPUNGE  ( Par: String );
      procedure Cmd_FETCH    ( Par: String );
      procedure Cmd_HELP     ( Par: String );
      procedure Cmd_ID       ( Par: String ); //JW //IMAP ID
      procedure Cmd_IDLE     ( Par: String ); //HSR //IDLE
      procedure Cmd_LIST     ( Par: String );
      procedure Cmd_LOGIN    ( Par: String );
      procedure Cmd_LOGOUT   ( Par: String );
      procedure Cmd_LSUB     ( Par: String );
      procedure Cmd_NCBrain  ( Par: String );
      procedure Cmd_NOOP     ( Par: String );
      procedure Cmd_RENAME   ( Par: String );
      procedure Cmd_SEARCH   ( Par: String );
      procedure Cmd_SELECT   ( Par: String );
      procedure Cmd_STARTTLS ( Par: String ); {MG}{SSL}
      procedure Cmd_STATUS   ( Par: String );
      procedure Cmd_STORE    ( Par: String );
      procedure Cmd_SUBSCRIBE( Par: String );
      procedure Cmd_UID      ( Par: String );
      procedure Cmd_UNSUBSCRIBE( Par: String );

      constructor Create( ASocket: TServerClientWinSocket;
                          Const AIPAccessScope: LongInt;
                          Const ASSLContext: Pointer ); override; {MG}{SSL}

      destructor Destroy; override;
  end;


// ----------------------------------------------------------------------------

implementation

uses SysUtils, uTools, uImapUtils, Global, Config, cIPAccess, cArtFiles,
     cAccount, cStdForm, cPCRE, cLogFile, FileCtrl,
     uEncoding, uMD5, uSHA1; //JW //IMAP-Auth


//----------------------------------------------------TOOLs--------
{MG}{Literal}
function CutFirstParam( var Parameters: String ): String;
var  i, Size : Integer;
begin
   Result := '';
   if Parameters = '' then exit;
   case Parameters[1] of
      '"' : begin
               for i := 2 to Length(Parameters) do begin            // Quoted string
                  if Parameters[i] = '"' then begin
                     Result := Copy( Parameters, 2, i-2 );
                     System.Delete( Parameters, 1, i );
                     Parameters := TrimWhSpace( Parameters );
                     exit
                  end
               end;
               // Not found
               Log( LOGID_DEBUG, 'Error parsing quoted string: No closing quotation mark' ) {MG}{ImapLog}
            end;
      '{' : for i := 2 to Length(Parameters) do begin           // Literal
               if Parameters[i] = '}' then begin
                  if Parameters[i-1] = '+' // Literal+ //NHB
                     then Size := StrToIntDef( Copy( Parameters, 2, i-3 ), -1 )
                     else Size := StrToIntDef( Copy( Parameters, 2, i-2 ), -1 );
                  if Length(Parameters) >= Size+i+2 then begin
                     Result := Copy( Parameters, i+3, Size );
                     System.Delete( Parameters, 1, i+3+Size );
                     Parameters := TrimWhSpace( Parameters );
                     exit
                  end else begin
                     Log( LOGID_DEBUG, 'Error parsing literal string: too short!' ) {MG}{ImapLog}
                  end
               end
            end;
   else
      begin                                               // Token
         i := PosWhSpace( Parameters );
         if i > 0 then begin
            Result := Copy( Parameters, 1, i-1 );
            System.Delete( Parameters, 1, i );
            Parameters := TrimWhSpace( Parameters )
         end else begin
            Result := Parameters;
            Parameters := ''
         end
      end
   end
end;
{/Literal}

// ------------------------------------------------------ TSrvIMAPCli --------

function TSrvIMAPCli.SafeString( Path: String ): Boolean;
var  i : Integer;
     SafeChars: Set of Char;
begin
     Result := False;
//     SafeChars := ['!', '#'..'.', '_', '0'..'9',     {MG}{IMAP-List}
//                   'A'..'Z', 'a'..'z', HierarchyDelimiter, ' '];
     SafeChars := [' ', '!', '#'..'.', '0'..'9', ';', '=',
                   '@'..'[', ']'..'{', '}', '~', HierarchyDelimiter];
     for i := 1 to Length(Path) do
        if not (Path[i] in SafeChars) then exit;
     Result := True;
end;

function TSrvIMAPCli.ReplacePathDelimiters( Path: String ): String;
begin
   Result := IncludeTrailingBackslash( ReplaceChars( Path, HierarchyDelimiter, '\' ) );
end;

function TSrvIMAPCli.GetStatus( Mailbox: String ): TMbxStatus;
var  RebuildNeeded : Boolean;
begin
     RebuildNeeded := False;
     FillChar( Result, SizeOf(Result), 0 );
     if not FileExists2( Mailbox + IMAPSTATUS_FILENAME ) then
        RebuildNeeded := True
     else begin
        with TFileStream.Create( Mailbox + IMAPSTATUS_FILENAME, fmOpenRead ) do try
           if Read( Result, SizeOf(Result) ) <> SizeOf(Result) then begin
              Log( LOGID_WARN, 'Error reading imap mailbox status file.' );
              RebuildNeeded := True;
           end
        finally Free end
     end;
     if RebuildNeeded then Result := RebuildStatusFile( Mailbox );
end;

function TSrvIMAPCli.RebuildStatusFile( Mailbox: String ): TMbxStatus;
begin
     FillChar( Result, SizeOf(Result), 0 );
     if CfgAccounts.IMAPMailboxLock( Mailbox, True ) then begin
        with TImapMailbox.Create( Mailbox ) do try
           Lock;
           Result := RebuildStatusFile
        finally
           Unlock; Free
        end;
     end else begin
        with TImapMailbox( CfgAccounts.GetIMAPMailbox( Mailbox ) ) do try
           Result := Status
        except end
     end
end;


//-------------------------------------------------- TSrvIMAPCli -----
//---------------------------------------------------Winsock--------

procedure TSrvIMAPCli.ClientExecute;
var  Data: array[0..1023] of Char;
     SocketStream: TWinSocketStream;
     Reply : String;
     LineEnd, ByteIn, i, j : Integer;
     BADCount : Integer; //HSR //IDLE
     EnhLit : Boolean; //HSR //Literal+     
begin
     BadCount := 0; //HSR //IDLE
     try
        SendGreeting;
        LogFile.SetTask( '(client ' + lowercase(inttohex(GetCurrentThreadID,1))+')', ClientID );
     except
        on E:Exception do begin
           Log( LOGID_WARN, TrGl (kLog, 'Connection.greeting.failed',
                'Connection failed (SendGreeting):') + ' ' + ClientID ); // JAWO 26.01.02
           Log( LOGID_WARN, TrGl (kLog, 'Error', 'Error') + ': ' + E.Message );
           Terminate;
        end;
     end;

     while not Terminated do
     try
        if ClientSocket=nil then exit;
        if not ClientSocket.Connected then exit;

        SocketStream := TWinSocketStream.Create( ClientSocket,
                                                 Def_LocalTimeoutInactivity*60000 );
        try
           {MG}{SSL}
           if ( Assigned(SSL) and SSL.HasPendingData ) or
              SocketStream.WaitForData( Def_LocalTimeoutInactivity*60000 )
           then begin
           {/SSL}
              FillChar(Data, SizeOf(Data), 0);
              ByteIn := 0;
              if ClientSocket<>nil then begin
                 try
                    if ClientSocket.Connected then begin
                    {MG}{SSL}
                    if SSLConnection=nil
                       then ByteIn := SocketStream.Read( Data, SizeOf(Data) )
                       else ByteIn := SSL.Read( Data, SizeOf(Data) );
                    {/SSL}
                    end;
                 except
                    ByteIn := 0;
                 end;
              end;
              if ByteIn = 0 then begin
                if Not LogFile.Skip_ConectionLost then begin
                   Log( LOGID_WARN, TrGl (kLog, 'Connection.lost', 'Connection lost:') +
                        ' ' + ClientID ); // JAWO 26.01.02
                end;
                if Assigned(ClientSocket) then begin
                   if ClientSocket.Connected then ClientSocket.Close;
                end;
                Terminate;
              end;

              BufInRaw := BufInRaw + Data;
              repeat
                 {MG}{Literal}
                 if (LimitTextSize > 0) and (BufInStrmLen+length(BufInRaw) > LimitTextSize)
                    then HadTextTooLarge := True;

                 LineEnd  := Pos( CRLF, BufInRaw );

                 if WaitForCmd and (LineEnd>0) and (Copy(BufInRaw,LineEnd-1,1)='}')
                    then for j := LineEnd-1 downto 1 do
                       if BufInRaw[j] = '{' then begin
                          // the literal starts here
                          EnhLit := copy(BufInRaw,LineEnd-2,1)='+'; //HSR //Literal+
                          if EnhLit then
                            LiteralCount := StrToIntDef( Copy(BufInRaw,j+1,LineEnd-j-3), -1 )
                          else
                            LiteralCount := StrToIntDef( Copy(BufInRaw,j+1,LineEnd-j-2), -1 );
                          Log(LOGID_DEBUG, 'IMAP: Literal of size ' + IntToStr(LiteralCount));
                          if LiteralCount > -1 then begin
                             if (LimitTextSize > 0) and (BufInStrmLen+LineEnd+LiteralCount
                                                         > LimitTextSize) then begin
                                HadTextTooLarge := True;
                                Reply := HandleData;
                                break
                             end;
                             if BufInStrmLen+LineEnd+LiteralCount+1 > Length(BufInStrm) then
                                SetLength( BufInStrm, BufInStrmLen+LineEnd+LiteralCount+1 );
                             for i := 1 to LineEnd+1 do
                                BufInStrm[BufInStrmLen+i] := BufInRaw[i];
                             Inc( BufInStrmLen, LineEnd+1 );
                             System.Delete ( BufInRaw, 1, LineEnd+1 );
                             WaitForCmd := False;
                             if not EnhLit then //HSR //Literal+
                               SendData( '+ Ready to receive ' + IntToStr(LiteralCount) +
                                         ' bytes of data' + CRLF );
                          end;
                          break
                       end;

                 if IdleState then begin //HSR //IDLE (Complete Part)
                   LineEnd := Pos( CRLF, BufInRaw );
                   Log(LOGID_Debug, copy(BufInRaw, 1, LineEnd - 1));
                   if LineEnd > 0 then begin
                     if UpperCase(copy(BufInRaw, 1, LineEnd - 1)) = 'DONE' then begin
                       IdleState := false;
                       BadCount := 0;
                       Reply := 'OK IDLE completed and ended';
                       Log(LOGID_DETAIL, 'IMAP: IDLE end');
                     end else begin
                         Log(LOGID_DEBUG, 'IMAP-Client BAD. Send information in IDLE-State: ' + copy(BufInRaw, 1, LineEnd - 1) );
                         inc(BADCount);
                         if BadCount>3 then begin //Prevent against bad clients
                           CurrentUserID   := ACTID_INVALID;
                           try
                             if Assigned(Selected) then MBLogout( Selected, true );

                             if Assigned(SubscribedMBs) then begin
                               SubscribedMBs.SaveToFile(MailboxPath + IMAPSUBSCRIBED_FILENAME);
                               FreeAndNil(SubscribedMBs);
                             end;
                           except
                           end;

                           if CurrentUserID>=0 then CurrentUserID := ACTID_INVALID;

                           Log(LOGID_DETAIL, 'IMAP: The client get disconnected');
                           try
                             if ClientSocket.Connected then begin
                               SendRes( 'BYE IMAP4rev1 closing connection - goodbye!' );
                               SendData(copy(BufInRaw, 1, pos(' ', BufInRaw)-1) + ' BAD You are in IDLE-state!');
                             end;
                             Sleep( Def_LocalTimeoutQuitDelay );

                             try
                               if ClientSocket.Connected then ClientSocket.Close;
                             except
                               on E:Exception do Log(LOGID_DEBUG, 'Exception on Socket.Close: ' + E.Message );
                             end;
                           finally
                             Terminate;
                           end;
                           break;
                         end;
                         SendData(copy(BufInRaw, 1, pos(' ', BufInRaw)-1) +  ' BAD You are in IDLE-state!');
                     end;

                     System.Delete(BufInRaw, 1, LineEnd + 1)
                   end
                 end else begin

                   if not WaitForCmd then begin
                      LineEnd := Length( BufInRaw );
                      if LineEnd < LiteralCount then begin
                         // we didn't read the whole literal
                         for i := 1 to LineEnd do
                            BufInStrm[BufInStrmLen+i] := BufInRaw[i];
                         SetLength( BufInRaw, 0 );
                         Inc( BufInStrmLen, LineEnd );
                         Dec( LiteralCount, LineEnd );
                      end else begin
                         // literal completed
                         for i := 1 to LiteralCount do
                            BufInStrm[BufInStrmLen+i] := BufInRaw[i];
                         System.Delete( BufInRaw, 1, LiteralCount );
                         Inc( BufInStrmLen, LiteralCount );
                         LiteralCount := 0;
                         WaitForCmd := True;
                      end;
                   end;

                   if WaitForCmd then begin
                      LineEnd := Pos( CRLF, BufInRaw );
                      if LineEnd > 0 then begin
                         if BufInStrmLen+LineEnd-1 > Length( BufInStrm )
                            then SetLength( BufInStrm, BufInStrmLen+LineEnd-1 );
                         for i := 1 to LineEnd - 1 do
                            BufInStrm[BufInStrmLen+i] := BufInRaw[i];
                         Inc( BufInStrmLen, LineEnd - 1 );
                         System.Delete( BufInRaw, 1, LineEnd+1 );
                         Reply := HandleData;
                      end;
                   end;
                 end;
                 if Reply <> '' then begin
                     SendResTag( Reply );
                     Reply := ''
                 end;

                 if HadTextTooLarge then begin
                    Log( LOGID_WARN, SockDesc('.Recv') + TrGl (kLog, 'Connection.terminated',
                         'Connection terminated:') + ' ' + Reply );  // JAWO 26.01.02
                    if ClientSocket<>nil then begin
                       if ClientSocket.Connected then ClientSocket.Close;
                    end;
                    Terminate;
                 end;
                 {/Literal}
              until (LineEnd=0) or Terminated;

           end else begin
             // If we didn't get any data after ? seconds then close the connection
             Log( LOGID_WARN, TrGl (kLog, 'Connection.timeout',
                  'Connection closed (timeout):') + ' '  + ClientID );

             if ClientSocket.Connected then ClientSocket.Close;
             Terminate;
           end;
        finally
           SocketStream.Free;
        end;
     except
        Terminate;
     end;
end;

procedure TSrvIMAPCli.SendResLit( Txt: String );
begin
   if Length( Txt ) > 250
      then Log( LOGID_DEBUG, '< * ' + Copy( Txt, 1, 250 ) + ' [...]' )
      else Log( LOGID_DEBUG, '< * ' + Txt );
   SendData( CurrentTag + ' ' + '{' + IntToStr(length(Txt + CRLF)) + '}' + CRLF );
   SendData( Txt + CRLF );
end;

procedure TSrvIMAPCli.SendRes( Txt: String);
begin
   if Length( Txt ) > 250
      then Log( LOGID_DEBUG, '< * ' + Copy( Txt, 1, 250 ) + ' [...]' )
      else Log( LOGID_DEBUG, '< * ' + Txt );
   SendData( '* ' + Txt + CRLF );
end;

procedure TSrvIMAPCli.SendResTag( Txt: String);
var i : integer;
begin
  {HSR} {IDLE}
  try
    EnterCriticalSection( CS_THR_IDLE );
    if Assigned(Selected) then try
      Selected.Lock;
      if (length(SendExpunge)>0) then begin
        for i := 0 to length(SendExpunge)-1 do
          SendRes( IntToStr(SendExpunge[i]) + ' EXPUNGE');
        if not SendNewMessages then //Don't send it double
          SendRes( IntToStr(Selected.Status.Messages) + ' EXISTS');
        SetLength(SendExpunge, 0)
      end;

      if SendNewMessages then begin
        SendRes( IntToStr(Selected.Status.Messages) + ' EXISTS');
        SendRes( IntToStr(Selected.Status.Recent)   + ' RECENT');
        SendNewMessages := false
      end;
    finally
      Selected.Unlock
    end;
  finally  
    LeaveCriticalSection( CS_THR_IDLE )
  end;
  {/HSR}

  if Length( Txt ) > 250
     then Log( LOGID_DEBUG, '< * ' + Copy( Txt, 1, 250 ) + ' [...]' )
     else Log( LOGID_DEBUG, '< * ' + Txt );
  SendResult( CurrentTag + ' ' + Txt )
end;

procedure TSrvIMAPCli.SendGreeting;
begin
     if (GetCounter(CntIMAPCli) > Def_Max_Local_IMAP_Servers) and
        (Def_Max_Local_IMAP_Servers>0) then begin
        SendRes( 'BYE server overload, try again later' );
        Log( LOGID_WARN, 'IMAP Server overload, too many clients (' +
                         IntToStr( GetCounter(CntIMAPCli) ) + ')' );
        Terminate;
        exit;
     end;

     CheckClientAccess;

     if (IPAccess and IPACC_ACCESS_RO)=IPACC_ACCESS_RO then begin
        If Def_FQDN > '' then begin
           SendResult( '* OK IMAP4rev1 Server '
                        + GetMyStringFileInfo('ProductName','Hamster') + ' ' // jawo //HSR23.5.02
                        + GetMyVersionInfo(true)
                        + ' on ' + Def_FQDN
                        + ' greets you!' )
        end else begin
           SendResult( '* OK IMAP4rev1 Server '
                        + GetMyStringFileInfo('ProductName','Hamster') + ' ' // jawo //HSR23.5.02
                        + GetMyVersionInfo(true)
                        + ' greets you!' );
        end;
        Log(LOGID_INFO, 'Send greetings...');
        exit; // OK
     end;

     Log( LOGID_WARN, TrGl(kLog, 'Warning.ConnectionRefused', 'Connection refused')+': ' + ClientID );
     if ClientSocket<>nil then
     try
        if ClientSocket.Connected then SendRes( 'BYE Permission denied - closing connection.' );
        if ClientSocket.Connected then ClientSocket.Close;
     except
     end;
     Terminate;
end;


//-----------------------------------Zustze zu den Kommandos--------

function TSrvIMAPCli.LoginUser( Password: String; AuthMechanism : String ): String;
begin
     Result := CurrentTag + 'BAD System-error, check logfile. [0]';
     CurrentUserID := ACTID_INVALID;
     try
        Result := 'NO Authentication rejected';
        if CurrentUserName='' then exit;

        CurrentUserID := CfgAccounts.LoginID( CurrentUserName, Password );
        if CurrentUserID=ACTID_INVALID then begin
           CurrentUserName := '';
           Result := 'NO No permission';
           exit;
        end;

        if not CfgAccounts.HasIMAPbox( CurrentUserID ) then begin
           CurrentUserID := ACTID_INVALID;
           CurrentUserName := '';
           Result := 'NO No IMAP-mailbox';
           Exit;
        end;

        MailBoxPath := CfgAccounts.MailboxPath( CurrentUserID );  {MG}{IMAP-List}
        ForceDirectories(MailBoxPath); //MB not created yet

        // read list of subscribed mailboxes
        SubscribedMBs  := TStringlist.create;
        if fileexists2(MailBoxPath + IMAPSUBSCRIBED_FILENAME) then
          SubscribedMBs.LoadFromFile(MailBoxPath + IMAPSUBSCRIBED_FILENAME);

        CurrentInactTimeOut := Def_LocalTimeoutInactivity*60000;
        // at least 30 minutes of inactivity timeout
        if CurrentInactTimeOut < 1800000 then CurrentInactTimeOut := 1800000;

        if AuthMechanism='' then
          Result := 'OK LOGIN completed.'
        else
          Result := 'OK ' + AuthMechanism + ' completed.'
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'IMAP.LoginUser-Exception: ' + E.Message );
           Result := 'BAD System-error, check logfile.';
        end;
     end;
end;

function  TSrvIMAPCli.MBLogin(  var Mailbox: TImapMailbox; Path: String; LINotify : Boolean ): Boolean; //HSR //IDLE (Chg)
begin
     Result := False;
     Log( LOGID_DEBUG, 'TSrvIMAPCli.MBLogin ' + Path ); {MG}{ImapLog}
     if Assigned( Mailbox ) then exit;
     try try
      EnterCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
        if CfgAccounts.IMAPMailboxLock( Path, True ) then begin
           Mailbox := TImapMailbox.Create( Path );
           CfgAccounts.SetIMAPMailbox( Path, Mailbox );
        end else
           Mailbox := TImapMailbox( CfgAccounts.GetIMAPMailbox( Path ) );
        {HSR} {IDLE}
        if LINotify then begin
          if SelNotify<>nil then begin
            Log(LOGID_ERROR, 'IMAP: Notification not deleted');
            dispose(SelNotify);
          end;
          new(SelNotify);
          SelNotify^.OnNewMess := NewMessages;
          SelNotify^.OnExpunge := NewExpunge;
          Mailbox.AddUser( SelNotify )
        end else
          Mailbox.AddUser( nil );
        {/HSR}

        Result := True
     finally
       LeaveCriticalSection( CS_IMAPMBCreate ) //HSR //MBCreate
     end 
     except
        on E:Exception do Log( LOGID_ERROR, 'IMAP.MBLogin-Exception: ' + E.Message )
     end
end;

procedure TSrvIMAPCli.MBLogout( var Mailbox: TImapMailbox; LOSel : Boolean ); //HSR //IDLE (Chg)
var  NoUsersLeft: Boolean;
begin
     if Assigned( Mailbox ) then begin
       try
         EnterCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
         if LOSel then begin
           Mailbox.RemoveUser( SelNotify, NoUsersLeft );
           dispose(SelNotify);
           SelNotify := nil
         end else
           Mailbox.RemoveUser( nil, NoUsersLeft );

         if NoUsersLeft then Mailbox.Free;
       finally
         LeaveCriticalSection( CS_IMAPMBCreate ) //HSR //MBCreate
       end;    
       Mailbox := nil
     end
end;

function TSrvIMAPCli.MBSelect( Mailbox: String; ReadOnly : Boolean ) : boolean;
var i : int64; //HSR //Integer Overflow
begin
     Result := false;

     //Unselect Mailbox
     if Assigned(Selected) then MBLogout( Selected, true );

     //Select Mailbox
     if (Mailbox='') or (not SafeString(Mailbox)) then exit;
     if uppercase(Mailbox)='INBOX' then Mailbox := MailboxPath
     else Mailbox := MailboxPath + ReplacePathDelimiters(Mailbox);
     if not DirectoryExists( Mailbox ) then exit;
     if FileExists2( Mailbox + 'NOSELECT' ) then exit;
     if not MBLogin( Selected, Mailbox, true ) then exit;

     try
        Selected.Lock;
        {HSR} {IDLE}
        SendNewMessages := false;
        SetLength(SendExpunge, 0);
        IdleState       := false;
        {/HSR}

        SendRes( IntToStr( Selected.Status.Messages ) + ' EXISTS');
        SendRes( IntToStr( Selected.Status.Recent ) + ' RECENT');
        i := Selected.Status.Messages - Selected.Status.Unseen + 1; //HSR //Integer Overflow
        if Selected.Status.Unseen > 0 then SendRes('OK [UNSEEN '
           + IntToStr(i)
           + '] First message-number unseen.');
        SendRes( 'OK [UIDVALIDITY ' + IntToStr( Selected.GetUIDvalidity ) + ']');
        SendRes('FLAGS '+Selected.PossFlags);
        Selected.MBReadOnly := Selected.MBReadOnly OR ReadOnly; //ClientRO //Soll die MB ReadOnly geffnet werden?
        if not Selected.MBReadOnly then Selected.RemoveRecentFlags;
        Result := True;
     finally
        Selected.Unlock;
     end
end;

function TSrvIMAPCli.MBCreate(Mailbox: string): boolean;
begin
   Result := false;
   if not SafeString( Mailbox ) then exit;
   if uppercase(Mailbox) = 'INBOX' then exit;
   Mailbox := MailboxPath + ReplacePathDelimiters(Mailbox);
   try
      if DirectoryExists( Mailbox ) then begin
         if FileExists2( Mailbox + 'NOSELECT' )
            then Result := SysUtils.DeleteFile( Mailbox + 'NOSELECT' )
            else exit
      end else begin
         Result := CreateDir( Mailbox ); //Win32Check (2.0.0.11 HB)
         Win32Check( Result);
      end
   except
     On E: Exception do begin
        Log(LOGID_ERROR, 'IMAP: CREATE failed: ' + E.Message);
        exit
     end;
   end
end;

function TSrvIMAPCli.MBDelete( Mailbox: string ): boolean;
var  Mbx: String;
     SR: TSearchRec;
begin
     Result := False;
     if (uppercase(Mailbox)='INBOX') or (not SafeString(Mailbox)) then exit;
     Mbx := Mailbox;
     if not MBExists( Mbx ) then exit;
     if Assigned(Selected) and (Selected.Path=Mbx) then MBLogout( Selected, true );
     if not CfgAccounts.IMAPMailboxLock( Mbx, True ) then begin
        Log( LOGID_WARN, 'Mailbox is still in use - DELETE refused.' );
        exit;
     end;
     try
        if SysUtils.FindFirst( Mbx + '*.msg', faAnyFile, SR ) = 0 then begin
           repeat
              if StrToIntDef( Copy( SR.Name, 1, Length(SR.Name)-4 ), 0 ) > 0 then
                 SysUtils.DeleteFile( Mbx + SR.Name );
           until SysUtils.FindNext( SR ) <> 0;
           SysUtils.FindClose( SR );
        end;
        SysUtils.DeleteFile( Mbx + IMAPINDEX_FILENAME );
        SysUtils.DeleteFile( Mbx + IMAPSTATUS_FILENAME );
        if not RemoveDir( Mbx ) then      // Unterverzeichnisse? ->  NOSELECT
           if FileCreate( Mbx + 'NOSELECT' ) = -1 then exit;
     except
       On E: Exception do begin
          Log(LOGID_ERROR, 'IMAP: DELETE failed: ' + E.Message);
          exit
       end;
     end;
     CfgAccounts.IMAPMailboxLock( Mbx, False );
     Result := True;
end;

function  TSrvIMAPCli.MBExists( var Mailbox: String ): Boolean;
begin
   if (Uppercase(MailBox)='INBOX') then begin
      MailBox := MailboxPath;
      Result := true;
   end else begin
     Mailbox := MailboxPath + ReplacePathDelimiters( Mailbox );
     Result := (DirectoryExists( Mailbox ) and not FileExists2( Mailbox + 'NOSELECT' ));
   end;
end;

function  TSrvIMAPCli.MBRename( OldName, NewName: String ): Boolean;
var  OldPath, NewPath : String;
     OldMbx,  NewMbx  : TImapMailbox;
     i                : Integer;
     Mbx : String; //Rename selected
begin
   Result := False;
   if uppercase(NewName) = 'INBOX' then exit;
   if not SafeString( OldName ) or not SafeString( NewName ) then exit;
   OldPath := MailboxPath + ReplacePathDelimiters( OldName );
   NewPath := MailboxPath + ReplacePathDelimiters( NewName );

   if uppercase(OldName) = 'INBOX' then begin
      if MBCreate( NewName ) then begin
         // move all messages in INBOX to the new mailbox
         if MBLogin( OldMbx, MailboxPath, false ) then begin
            try
               if MBLogin( NewMbx, NewPath, false ) then begin
                  try OldMbx.CopyMessage( OldMbx.StrToMsgSet('1:*',False), NewMbx )
                  finally MBLogout( NewMbx, false ) end;
                  for i := 0 to OldMbx.Status.Messages-1 do
                     OldMbx.Store( i , '\DELETED', [smAdd] );
                  OldMbx.Expunge( nil );
                  Result := True;
               end
            finally MBLogout( OldMbx, false ) end
         end
      end
   end else begin
      //Rename Selected
      Mbx := OldName;
      if not MBExists( Mbx ) then exit;
      if Assigned(Selected) and (Selected.Path=Mbx) then MBLogout( Selected, true );
      if not CfgAccounts.IMAPMailboxLock( Mbx, True ) then begin
         Log( LOGID_WARN, 'Mailbox is still in use - RENAME refused.' );
         exit;
      end;

      // rename mailbox
      if DirectoryExists( OldPath ) then try
         Result := RenameFile( copy( OldPath, 1, length(OldPath)-1 ),
                               copy( NewPath, 1, length(NewPath)-1 ) );


      except
         On E: Exception do Log(LOGID_ERROR, 'IMAP: RENAME failed: ' + E.Message);
      end
   end;
end;

procedure TSrvIMAPCli.DoSearch( UseUID: Boolean; Par: String );
var  i: Integer;
     SendS, Command, Charset: String;
begin
     if UseUID then Command := 'UID SEARCH' else Command := 'SEARCH';
     Charset := '';
     if uppercase( copy( Par, 1, 7 ) ) = 'CHARSET' then begin
        Delete( Par, 1, 8 );
        Par := TrimWhSpace( Par );
        i := PosWhSpace( Par );
        if (par='') or (i=0) then begin
          SendResTag('BAD ' + Command + ' charset argument missing');
          exit
        end;
        Charset := Uppercase( TrimQuotes( copy( Par, 1, i-1 ) ) );
        Par := TrimWhSpace( copy( Par, i+1, length(Par) ) );
{JW} {IMAP SEARCH}
        if (Charset = 'UTF-8') or (Charset = 'UTF-7') then begin
           SendResTag('NO ' + Command +
                      ' specified charset not supported');
           exit
        end
     end;
{JW}
     SendS := Selected.Search( Charset, Par, UseUID);
     if trim(SendS) <> '' then SendRes( SendS );
     SendResTag('OK ' + Command + ' completed')
end;

procedure TSrvIMAPCli.DoCopy( MsgSet: TMessageSet; Command, Destination: String );
var  DestMailbox : TImapMailbox;
begin
     DestMailBox := NIL;
     if not MBExists( Destination ) then begin
        SendResTag( 'NO [TRYCREATE] ' + Command + ' error: destination mailbox not known' );
        exit
     end;

     if not MBLogin( DestMailbox, Destination, false ) then begin
        SendResTag( 'NO ' + Command + ' error: can''t open destination mailbox' );
        exit
     end;

     try
        if Selected.CopyMessage( MsgSet, DestMailbox ) then begin
           SendResTag( 'OK ' + Command + ' completed' );
        end else
           SendResTag( 'NO ' + Command + ' error: can''t copy messages' );
        MBLogout( DestMailbox, false );
     except
        on E:Exception do
           Log( LOGID_ERROR, 'IMAP.DoCopy-Exception: ' + E.Message );
     end
end;

procedure TSrvIMAPCli.DoFetch( MsgSet: TMessageSet; Command, Par: String );
var  Success : Boolean;
     SendS, MsgDat  : String;
     i       : Integer;
begin
     Success := True;
     MsgDat := TrimParentheses( Uppercase( Par ) );

     // macros
     StringReplace( MsgDat, 'FAST', 'FLAGS INTERNALDATE RFC822.SIZE', [] );
     StringReplace( MsgDat, 'ALL',  'FLAGS INTERNALDATE RFC822.SIZE ENVELOPE', [] );
     StringReplace( MsgDat, 'FULL', 'FLAGS INTERNALDATE RFC822.SIZE ENVELOPE BODY', [] );

     // Server implementations MUST implicitly include the UID message data item
     // as part of any FETCH response caused by a UID command, regardless of
     // whether a UID was specified as a message data item to the FETCH.
     if ( Command = 'UID FETCH' ) and ( Pos( 'UID', MsgDat ) = 0 ) then
        MsgDat := MsgDat + ' UID';

     for i := 0 to High(MsgSet) do begin
        SendS := Selected.Fetch( MsgSet[i]-1, MsgDat, Success );
        if (trim(SendS) <> '') AND Success then SendRes ( SendS )
     end;
     if Success
        then SendResTag( 'OK ' + Command + ' is now completed' )
        else SendResTag( 'NO ' + Command + ' error' )
end;

procedure TSrvIMAPCli.DoStore( MsgSet: TMessageSet; Command, Par: String );
var  i: integer;
     MsgDat: string;
     Flags : String;
     Silent : Boolean;
     Mode : TStoreMode;
begin
     if Selected.MBReadOnly then begin //ClientRO
        SendResTag('NO selected mailbox is read-only.');
        exit;
     end;

     i := PosWhSpace( Par );
     MsgDat := Uppercase( TrimQuotes( copy( Par, 1, i ) ) );
     Flags  := Uppercase( TrimParentheses( copy( Par, i+1, length(Par)-i ) ) ); {MG}{Imap-Store}
     if trim(Flags)='' then begin
        SendResTag('BAD ' + Command + ' without Flag-Value!');
        exit;
     end;
     if not Selected.AreValidFlags(Flags) then begin
        SendResTag('NO The \Recent flag may not used as an argument in STORE!');
        exit;
     end;

     i := pos( '.', Msgdat);
     if (i>0) and (copy( MsgDat, i, 7 )='.SILENT') then begin
        Silent := True;
        MsgDat := copy( MsgDat, 1, i-1 );
     end else
        Silent := False;

     if      MsgDat = 'FLAGS'  then Mode := [smReplace]
     else if MsgDat = '+FLAGS' then Mode := [smAdd]
     else if MsgDat = '-FLAGS' then Mode := [smDelete]
     else begin
        SendResTag('NO ' + Command + ' with unknown message-data!');
        exit;
     end;

     for i := 0 to High(MsgSet) do begin
        Flags := Selected.Store( MsgSet[i]-1, Flags, Mode );
        if not Silent then SendRes( IntToStr(MsgSet[i]) + ' FETCH (FLAGS ' +
                                    Flags + ')' );
     end;
     Selected.WriteStatus;
     SendResTag('OK you''ve stored your flags now!')
end;

//---------------------------------------------------HANDLEs--------

function TSrvIMAPCli.HandleData: String;
var  i: Integer;
begin
     Result := 'BAD Command failed (unknown reason, see logfile)';
     if not Assigned( ClientSocket ) then exit;
     if not ClientSocket.Connected then exit;

     {MG}{Literal}
     SetLength( BufInStrm, BufInStrmLen );
     i := Pos( ' ', BufInStrm );
     CurrentTag := Copy( BufInStrm, 1, i-1 );
     System.Delete( BufInStrm, 1, i );

     if trim(CurrentTag)='' then begin //HSR //TAG-Miss
       CurrentTag := BufInStrm;
       System.Delete( BufInStrm, 1, length(CurrentTag) );
       Result := 'BAD Command failed (missing TAG)'
     end else begin
        if HadTextTooLarge then begin
           Result := 'Command failed (textsize-limit exceeded)'
        end else begin
           HandleCommand( BufInStrm );
           Result := ''
        end
     end;
     SetLength( BufInStrm, 0 );
     BufInStrmLen := 0;
     {/Literal}
end;

procedure TSrvIMAPCli.HandleCommand( Const CmdLine: String );
var  LogCmdLine, Cmd, Par: String;
     j: Integer;
     nCmdLine : String;
begin
     nCmdLine := CmdLine;
     try
        if not ClientSocket.Connected then exit;

        j := PosWhSpace( nCmdLine );
        if j=0 then begin
           Cmd := UpperCase( nCmdLine );
           Par := '';
        end else begin
           Cmd := UpperCase  ( copy( nCmdLine, 1, j-1 ) );
           Par := TrimWhSpace( copy( nCmdLine, j+1, length(nCmdLine) ) );
        end;

        if (Cmd='LOGIN') then
          LogCmdLine := 'LOGIN ' + copy( Par, 1, PosWhSpace(Par)-1 ) + ' [...]'
        else if Length( nCmdLine ) > 250
          then LogCmdLine := copy( nCmdLine, 1, 250 ) + ' [...]'
          else LogCmdLine := nCmdLine;

        Log( LOGID_DETAIL, '> ' + CurrentTag + ' ' + LogCmdLine );

        if CmdLine='' then exit;

        // Workaround for tired OutlookXP Clients
        if Def_IMAPDelay > 0 then sleep(Def_IMAPDelay);

        // commands (no authentication required)
        if Cmd='HELP'         then begin Cmd_HELP   ( Par ); exit end; //NOT IN THE STANDARD!!!
        if Cmd='CAPABILITY'   then begin Cmd_CAPA   ( Par ); exit end;
        if (Cmd='ID') AND
                  Def_IMAP_ID then begin Cmd_ID     ( Par ); exit end; //JW //IMAP ID
        if (Cmd='NETSCAPE') and
              Def_IMAPNCBrain then begin Cmd_NCBrain( Par ); exit end; //HSR //NCBrain

        if Cmd='NOOP'         then begin Cmd_NOOP   ( Par ); exit end;
        if Cmd='LOGOUT'       then begin Cmd_LOGOUT ( Par ); exit end;
        if CurrentUserID=ACTID_INVALID then begin
          if Cmd='STARTTLS'     then begin Cmd_STARTTLS    ( Par ); exit end; {MG}{SSL}
          if Cmd='AUTHENTICATE' then begin Cmd_AUTHENTICATE( Par ); exit end;
          if Cmd='LOGIN'        then begin Cmd_LOGIN       ( Par ); exit end;
        end;

        // check authentication
        if CurrentUserID=ACTID_INVALID then begin
           Log(LOGID_WARN, 'This command need authentication, but client is not authenticated yet: ' + nCmdLine);
           SendResTag( 'BAD Authentication required!' );
           exit
        end;

        // commands (authentication required)
        if Cmd='LIST'       then begin Cmd_LIST       ( Par ); exit end;
        if Cmd='SELECT'     then begin Cmd_SELECT     ( Par ); exit end;
        if Cmd='EXAMINE'    then begin Cmd_EXAMINE    ( Par ); exit end;
        if Cmd='SUBSCRIBE'  then begin Cmd_SUBSCRIBE  ( Par ); exit end;
        if Cmd='UNSUBSCRIBE'then begin Cmd_UNSUBSCRIBE( Par ); exit end;
        if Cmd='LSUB'       then begin Cmd_LSUB       ( Par ); exit end;
        if Cmd='STATUS'     then begin Cmd_STATUS     ( Par ); exit end;
        if Cmd='DELETE'     then begin Cmd_DELETE     ( Par ); exit end;
        if Cmd='CREATE'     then begin Cmd_CREATE     ( Par ); exit end;
        if Cmd='RENAME'     then begin Cmd_RENAME     ( Par ); exit end;
        if Cmd='APPEND'     then begin Cmd_APPEND     ( Par ); exit end;
        if Cmd='IDLE'       then begin Cmd_IDLE       ( Par ); exit end; //HSR //IDLE

        if not Assigned(Selected) then begin
           Log(LOGID_WARN, 'This command need a mailbox be selected, but there is no mailbox selected yet: ' + nCmdLine);
           SendResTag( 'BAD Selection of a mailbox is required!' );
           exit;
        end;

        //commands for "selected"-mode
        if Cmd='CHECK'    then begin Cmd_CHECK   ( Par ); exit end;
        if Cmd='CLOSE'    then begin Cmd_CLOSE   ( Par ); exit end;
        if Cmd='EXPUNGE'  then begin Cmd_EXPUNGE ( Par ); exit end;
        if Cmd='SEARCH'   then begin Cmd_SEARCH  ( Par ); exit end;
        if Cmd='FETCH'    then begin Cmd_FETCH   ( Par ); exit end;
        if Cmd='STORE'    then begin Cmd_STORE   ( Par ); exit end;
        if Cmd='COPY'     then begin Cmd_COPY    ( Par ); exit end;
        if Cmd='UID'      then begin Cmd_UID     ( Par ); exit end;

        // unknown (sub-) command
        SendResTag( 'BAD Command not implemented.' );
        Log( LOGID_WARN, TrGl(kLog, 'Info.UnsupportedIMAPCommand',
           'Unsupported IMAP-command') + ': ' + nCmdLine );

     except
        on E: Exception do begin
           Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
           Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + LogCmdLine );
        end;
     end;
end;


//-------------------------------------------------Kommandos--------

procedure TSrvIMAPCli.Cmd_HELP( Par: String );
var s : string;
begin
  if Par = '' then begin
    SendRes('Implemented Commands follows');
    s :=
               'APPEND msgs fgs dt' + CRLF +
               'AUTHENTICATE methode'+ CRLF + //JW //IMAP-Auth
               'CAPABILITY'         + CRLF +
               'CHECK'              + CRLF +
               'CLOSE'              + CRLF +
               'COPY msgs mbx'      + CRLF +
               'CREATE mbx'         + CRLF +
               'DELETE mbx'         + CRLF +
               'EXAMINE mbx'        + CRLF +
               'EXPUNGE'            + CRLF +
               'FETCH msgs fdat'    + CRLF +
               'HELP [topic]'       + CRLF + //!!
               'IDLE  ..  DONE'     + CRLF + //HSR //IDLE
               'LIST ref mbx'       + CRLF +
               'LOGIN usr pasw'     + CRLF +
               'LOGOUT'             + CRLF +
               'LSUB ref mbx'       + CRLF +
               'NOOP'               + CRLF +
               'RENAME mbx mbx'     + CRLF +
               'SEARCH'             + CRLF + //Parameter?
               'SELECT mbx'         + CRLF +
               'STARTTLS'           + CRLF +
               'STATUS mbx'         + CRLF +
               'STORE msgs sdat val'+ CRLF +
               '    flags[.silent]' + CRLF +
               '    +flags[.silent]'+ CRLF +
               '    -flags[.silent]'+ CRLF +
               'SUBSCRIBE mbx'      + CRLF +
               'UID cmd params'     + CRLF +
               '    COPY'           + CRLF +
               '    FETCH'          + CRLF +
               '    SEARCH'         + CRLF +
               '    STORE'          + CRLF +
               'UNSUBSCRIBE mbx';
    if Def_IMAP_ID then
      s := s + CRLF + 'ID params';
    SendResLit( s );

  end else begin
    par := uppercase(par);
    if par='APPEND' then
      SendResLit(
                 'Parameter:  mailbox name'                       + CRLF +
                 '            OPTIONAL flag list'                 + CRLF +
                 '            OPTIONAL date/time string'          + CRLF +
                 '            message literal'                    + CRLF +
                 ''                                               + CRLF +
                 'Adds a new message with flags and datetime set' + CRLF +
                 'into the given mailbox. The message itself is'  + CRLF +
                 'given literal.'
                );
    if par='AUTHENTICATE' then
      SendResLit(
                 'SASL secur authentifications.'  + CRLF +
                 ''
                 );
    if par='CAPABILITY' then
      SendResLit(
                 'Capability gives back some Flags/Params'         + CRLF +
                 'to handle "new" commands.'                       + CRLF +
                 'Try to get a list of all available extensions.'
                 );
    if par='CHECK' then
      SendResLit(
                 'CHECK returns a mailbox status update.'
                 );
    if par='CLOSE' then
      SendResLit(
                 'Deselects (closes) the current selected mailbox.' + CRLF +
                 '''EXPUNGE'' is called automatically if the'       + CRLF +
                 'mailbox is read-write'
                 );
    if par='COPY' then
      SendResLit(
                 ''
                );
    if par='CREATE' then
      SendResLit(
                 'Parameter: mailbox'                                          + CRLF +
                 'Creates a new mailbox. Hierachy-delimiter: ''/'''
                 );
    if par='DELETE' then
      SendResLit( ''
                );
    if par='EXAMINE' then
      SendResLit(
                 'Parameter: mailbox'                                          + CRLF +
                 'Selects another mailbox read-only.'                           + CRLF +
                 ''                                                            + CRLF +
                 'If that mailbox doesn''t exists, this command only deselect' + CRLF +
                 'selected mailbox. Hierachy-delimiter: ''/'''
                 );
    if par='EXPUNGE' then
      SendResLit(
                 'Deletes messages flagged with "/delete".' + CRLF +
                 'Use it for getting a ''clean'' mailbox.'
                 );
    if par='FETCH' then
      SendResLit( ''
                );
    if par='HELP' then
      SendResLit(
                 'This command <g>. It''s NOT defined by RFC!'                   + CRLF +
                 'If I''m ready it could be deleted. Also it could be changed'   + CRLF +
                 'and so on. Use it only with(for) telnet-experiments.'          + CRLF +
                 ''                                                              + CRLF +
                 'Parameter could be the command to which you want to get help'  + CRLF +
                 ''                                                              + CRLF +
                 'BTW: I couldn''t use "XHELP" ''cause nobody would find it.'
                 );
    {JW} {IMAP ID}
    if (par='ID') and Def_IMAP_ID then
      SendResLit(
                 'Parameter: Property of client or "nil" token. Ignored.' + CRLF +
                 'Gives out an ID of the server. It MUST NOT be used to ' + CRLF +
                 'change clients behavior!'
                 );
    {/JW}
    if par='IDLE' then //HSR //IDLE
      SendResLit( 'IDLE'                                       + CRLF +
                  'Wait til client sends DONE<CRLF>.'          + CRLF +
                  'While the server is in this state it sends' + CRLF +
                  'EXISTS and RECENT without request'          + CRLF +
                  'Client MUST NOT send any commands in this state'
                );

    if par='LIST' then
      SendResLit( ''
                );
    if par='LOGIN' then
      SendResLit(
                 'Normal authentifications. Need as parameters USER *and* PASS.' + CRLF +
                 '''Cause there is no preauth and AUTHENTICATE-command in this'  + CRLF +
                 'implementation, you have to use this command.'
                 );
    if par='LOGOUT' then
      SendResLit(
                 'Goes out of mailboxes and terminates the connection!' + CRLF +
                 'Use it like "QUIT" in POP3/SMTP/NNTP'
                 );
    if par='LSUB' then
      SendResLit( ''
                );
    if par='NOOP' then
      SendResLit(
                 'NOOP returns a mailbox status update.'       + CRLF +
                 'You can use the NOOP command to reset the '  + CRLF +
                 'inactivity autologout timer on the server.'
                 );
    if par='RENAME' then
      SendResLit( ''
                );
    if par='SEARCH' then
      SendResLit(
                 'Parameter: OPTIONAL [CHARSET] specification'        + CRLF +
                 '           searching criteria (one or more)'        + CRLF +
                 ''                                                   + CRLF +
                 'The SEARCH command searches the mailbox for'        + CRLF +
                 'messages that match the given searching criteria.'  + CRLF +
                 ''                                                   + CRLF +
                 'The defined search keys are as follows:'            + CRLF +
                 '  <message set>     Message numbers'                 + CRLF +
                 '  ALL               All messages'                    + CRLF +
                 '  Answered          Flag "/Answered"'                + CRLF +
                 '  BCC <string>      BCC in envelope-structure'       + CRLF +
                 '  Before <date>     Older messages then <date>'      + CRLF +
                 '  BODY <string>     Is <string> in body?'            + CRLF +
                 '  CC <string>       CC in envelope-structure'        + CRLF +
                 '  Deleted           Flag "/Deleted"'                 + CRLF +
                 '  Draft             Flag "/Draft"'                   + CRLF +
                 '  Flagged           Flag "/Flagged"'                 + CRLF +
                 '  FROM <string>     FROM in envelope-structure'      + CRLF +
                 '  Header <field> <string>'                          + CRLF +
                 '                    Contains header <field> <string>?' + CRLF +
                 '  Keyword <flag>    Flag <flag>'                     + CRLF +
                 '  Larger <n>        Larger than <n> octets?'         + CRLF +
                 '  New               Flag "/Recent" and not flag "/Seen"' + CRLF +
                 '  NOT <search-key>  Don''t match <search-key>'       + CRLF +
                 '  Old               Not flag "/Recent"'              + CRLF +
                 '  On <date>         Internal date = <date>?'         + CRLF +
                 '  OR <s-k1> <s-k2>  Matches <search-key1> OR <search-key2>' + CRLF +
                 '  Recent            Flag "/Recent"'                  + CRLF +
                 '  Seen              Flag "/Seen"'                    + CRLF +
                 '  Sentbefore <date> Is header "date" <  <date>?'     + CRLF +
                 '  Senton <date>     Is header "date" =  <date>?'     + CRLF +
                 '  Sentsince <date>  Is header "date" => <date>?'     + CRLF +
                 '  Since <date>      Internal date => <date?'         + CRLF +
                 '  Smaller <n>       Smaller than <n> octets?'        + CRLF +
                 '  SUBJECT <string>  SUBJECT in envelope-structure'   + CRLF +
                 '  Text <string>     Contains <string> in header or body?' + CRLF +
                 '  TO <string>       TO in envelope-structure'        + CRLF +
                 '  UID <message-set> Corresponding to the UID-Set?'   + CRLF +
                 '  Unanswered        Not flag "/Answered"'            + CRLF +
                 '  Undeleted         Not flag "/Deleted"'             + CRLF +
                 '  Undraft           Not flag "/Draft"'               + CRLF +
                 '  Unflagged         Not flag "/Flagged"'             + CRLF +
                 '  Unkeyword <flag>  Not flag <flag>'                 + CRLF +
                 '  Unseen            Not flag "/Seen"'
                );
    if par='SELECT' then
      SendResLit(
                 'Parameter: mailbox'                                          + CRLF +
                 'Select another mailbox read-write.'                          + CRLF +
                 'If that mailbox doesn''t exists, this command only deselect' + CRLF +
                 'selected mailbox. Hierachy-delimiter: ''/'''
                 );
    if par='STARTTLS' then
      SendResLit( ''
                );
    if par='STATUS' then
      SendResLit(
                 'Parameter:  mailbox name'                                        + CRLF +
                 '            status data item names'                              + CRLF +
                 ''                                                                + CRLF +
                 'Gets the status you want of the specified mailbox'               + CRLF +
                 '  MESSAGES       Number of all messages'                         + CRLF +
                 '  RECENT         Number of messages with "\Recent" flag'         + CRLF +
                 '  UIDNEXT        The UID that will be assigned to a new message' + CRLF +
                 '  UIDVALIDITY    The unique identifier validity value'           + CRLF +
                 '  UNSEEN         Number of messages without "\Seen" flag'
                );
    if par='STORE' then
      SendResLit(
                 'Arguments:  message set'                                         + CRLF +
                 '            message data item name'                              + CRLF +
                 '            value for message data item'                         + CRLF +
                 ''                                                                + CRLF +
                 'The STORE command alters data associated with a message '        + CRLF +
                 'in the mailbox.'                                                 + CRLF +
                 ''                                                                + CRLF +
                 'valid message data item names:'                                   + CRLF +
                 '     (FLAGS|+FLAGS|-FLAGS)[.SILENT]'                             + CRLF +
                 'valid values for message data item names:'                       + CRLF +
                 '     (\Deleted \Flagged \Seen \Answered \Marked)'                + CRLF
                );
    if par='SUBSCRIBE' then
      SendResLit( ''
                );
    if par='UID' then
      SendResLit( ''
                );
{
               '    COPY'           + CRLF +
               '    FETCH'          + CRLF +
               '    SEARCH'         + CRLF +
               '    STORE'          + CRLF +
}
    if par='UNSUBSCRIBE' then
      SendResLit( ''
                );

    s := '|APPEND|AUTHENTICATE|CAPABILITY|CHECK|' +
         'CLOSE|COPY|CREATE|DELETE|EXAMINE|EXPUNGE|'+
         'FETCH|HELP|IDLE|LIST|LOGIN|LOGOUT|LSUB|NOOP|' +
         'RENAME|SEARCH|SELECT|STARTTLS|STATUS|' +
         'STORE|SUBSCRIBE|UID|UNSUBSCRIBE|';
    If Def_IMAP_ID then //JW //IMAP ID
      s := s + 'ID|';
    IF not (pos( ('|'+par+'|'), s ) > 0) THEN
      SendResTag('NO I couldn''t find this command in my help-database!')
    else
      SendResTag('OK You''ve the help you want, haven''t you?');
  end;
end;

procedure TSrvIMAPCli.Cmd_APPEND( Par: String ); {MG}{Literal}
var
     Mailbox, TimeStr, MessageText: String;
     i           : Integer;
     Flags       : String;
     Time        : TUnixTime;
     DestMailbox : TImapMailbox;
begin
     Mailbox := CutFirstParam( Par );
     if (Mailbox='') or (Par='') then begin
         Log(LOGID_WARN, 'IMAP: APPEND: Missing arguments');
         SendResTag('BAD arguments missing for APPEND!');
         exit
     end;
     if not MBExists( Mailbox ) then begin
        Log(LOGID_WARN, 'IMAP: APPEND: Mailbox not known');
        SendResTag( 'NO [TRYCREATE] APPEND error: mailbox not known' );
        exit
     end;

     i := Pos(')',Par);
     if (i > 0) and (i < Pos('{',Par)) then begin
        Flags := trim(Copy( Par, Pos('(',Par) + 1, i - Pos('(',Par) -1 ));
        Par   := TrimLeft( copy( Par, i+1, length(Par)-i ) ); // NHB
     end else begin
        Flags := ''
     end;

     if Copy( Par, 1, 1 ) = '"'
        then TimeStr := CutFirstParam( Par )
        else TimeStr := '';
     if TimeStr <> ''
        then Time := DateTimeToUnixTime( ImapDateTimeToDateTime( TimeStr ) )
        else Time := DateTimeToUnixTime( nowGMT );

     if Par = '' then begin
         Log(LOGID_WARN, 'IMAP: APPEND: Message missing');
         SendResTag('BAD APPEND without message literal!');
         exit;
     end;
     MessageText := CutFirstParam( Par );

     if Assigned(Selected) and (Mailbox = Selected.Path) then begin
        SendResTag( Selected.AppendMessage( MessageText, Flags, Time ) );
        Log(LOGID_INFO, 'IMAP: Append: Message added')
     end else if not MBLogin( DestMailbox, Mailbox, false ) then begin
        SendResTag( 'NO APPEND error: can''t open destination mailbox' );
        Log(LOGID_INFO, 'IMAP: Append failed: Can''t open mailbox')
     end else begin
        try
           SendResTag( DestMailbox.AppendMessage( MessageText, Flags, Time ) );
           Log(LOGID_INFO, 'IMAP: Append: Message added (MB opened)')
        finally
           MBLogout( DestMailbox, false )
        end
     end
end;

procedure TSrvIMAPCli.Cmd_AUTHENTICATE( Par: String ); //JW //IMAP-Auth
var
{JW} {SASL-DIGEST}
    realm, nonce, cnonce, qop, username, nc, realm2,
    digesturi, response, a1, a2, rspauth :string;
{/JW}
    s, TimeStamp, Hash, pass: string;
begin
   CurrentUserName := '';
   CurrentUserID   := ACTID_INVALID;
   try
      par:=uppercase(par);
      Log( LOGID_DETAIL, 'Auth Parameter '+Par );
      if (par='LOGIN') and not Def_IMAP_DisableSASLLogin then begin
         s := 'Username:';
         s := '+ '+EncodeB64( s[1], length(s) );
         s := SendRequest( s);
         if s='' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN protocol error' );
            SendResTag('NO Authentification failed!');
            Exit;
         end;
         if s='*' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN cancelled by client' );
            SendResTag('BAD Authentification failed!');
            Exit;
         end;
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_INFO, '> ' +  s );
         CurrentUserName := TrimWhSpace( s );
         s := 'Password:';
         s := '+ '+EncodeB64( s[1], length(s) );
         s := SendRequest( s );
         if s='' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN protocol error' );
            SendResTag('NO Authentification failed!');
            Exit;
         end;
         if s='*' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN cancelled by client' );
            SendResTag('BAD Authentification failed!');
            Exit;
         end;
         s := DecodeB64( s[1], length(s) );
         CurrentUserID   := ACTID_INVALID;
         SendResTag(LoginUser( s, 'LOGIN' ));
      end else
      if (par='PLAIN') and Assigned(SSL) then begin
         TimeStamp := MidGenerator(Def_FQDNforMIDs);
         s := '+ '+EncodeB64( TimeStamp[1], length(TimeStamp) );
         s := SendRequest(s);
         if s='' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN protocoll error' );
            SendResTag('NO Authentification failed!');
            Exit;
         end;
         if s='*' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN cancel by client' );
            SendResTag('BAD Authentification failed!');
            Exit;
         end;
         s := DecodeB64( s[1], length(s) );
         CurrentUserName := TrimWhSpace( copy(s,pos(#0,s)+1,500));
         s:=TrimWhSpace( copy(CurrentUserName,
                         pos(#0,CurrentUserName)+1,500));
         CurrentUserName := TrimWhSpace( copy(CurrentUserName,1,
                                              pos(#0,CurrentUserName)-1));
         CurrentUserID   := ACTID_INVALID;
         SendResTag(LoginUser( s, 'PLAIN' ));
      end else

{JW} {SASL-DIGEST}    // portions by Olivier Zolli
      if Par = 'DIGEST-MD5' then begin
         // build challenge
         if Def_FQDN<> '' then
            realm := Def_FQDN
         else
            realm := 'localhost';
         s := 'realm="' + realm + '"';
         nonce:= HMAC_SHA1( nonce,
                  IntToHex(PRNG(MaxInt),8)+
                  IntToHex(PRNG(MaxInt),8)+
                  IntToHex(PRNG(MaxInt),8)+
                  IntToHex(PRNG(MaxInt),8));
         nonce := EncodeB64( nonce[1], length(nonce) );
         s := s + ',' + 'nonce="' + nonce + '"';
         qop := 'auth';
         s := s + ',' + 'qop="' + qop + '"';
         s := s + ',' + 'algorithm=' + 'md5-sess';
         Log( LOGID_DEBUG, 'DIGEST-MD5 challenge: ' + s );
         s := EncodeB64( s[1], length(s) );
         // send challenge, get response
         s := SendRequest( '+ '+s);
         if s='' then begin
           Log( LOGID_Error, 'empty response received');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         // check response, extract values
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_DEBUG, 'DIGEST-MD5 response: ' + s );
         // check username
         username:=ExtractQuotedParameter(s,'username');
         if username='' then begin
           Log( LOGID_Error, 'missing username in answer');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         CurrentUserName:=trim(username);
         CurrentUserID   := ACTID_INVALID;
         nonce:=ExtractQuotedParameter(s,'nonce');
         if nonce='' then begin
           Log( LOGID_Error, 'missing nonce in answer');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         cnonce:=ExtractQuotedParameter(s,'cnonce');
         if cnonce='' then begin
           Log( LOGID_Error, 'missing cnonce in answer');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         nc:=ExtractQuotedParameter(s,'nc');
         if nc <> '00000001' then begin
           Log( LOGID_Error, 'wrong nc value');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         qop:=ExtractQuotedParameter(s,'qop');
         if qop='' then
           qop:='auth'
         else
           if (lowercase(qop)<>'auth') then begin
             Log(LOGID_ERROR,
                 'not supported hash quality protection '+qop);
             SendResTag('NO Authentification failed!');
             exit;
           end;
         if pos('realm=',s)=0 then begin
           Log( LOGID_Error, 'missing realm in answer');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         realm2:=ExtractQuotedParameter(s,'realm');
         if realm2='' then begin
           Log( LOGID_Error, 'missing realm in answer');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         if realm2<>realm then begin
           Log( LOGID_Error, 'wrong realm in answer');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         digesturi:=ExtractQuotedParameter(s,'digest-uri');
         response:=ExtractQuotedParameter(s,'response');
         if length(response)<>32 then begin
           Log( LOGID_Error, 'wrong response length in answer');
           SendResTag('NO Authentification failed!');
           exit;
         end;
         // build expected response and compare with received one
        try
           CurrentUserID   := CfgAccounts.UserIDOf(CurrentUserName);
           if CurrentUserID=ACTID_INVALID then begin
              Log( LOGID_Error, 'login rejected, unknown user');
              CurrentUserName := '';
              SendResTag('NO Authentification failed!');
              exit;
           end;
           pass:=CfgAccounts.Value[CurrentUserID,ACTP_PASSWORD];
           a1 := MD5OfStr( username + ':' + realm + ':' + pass )
                 + ':' + nonce + ':' + cnonce;
           A2:='AUTHENTICATE:'+digesturi;
           s  := MD5toHex( MD5OfStr(
                  MD5toHex( MD5OfStr( A1 ) )
                  + ':' + nonce + ':' + nc + ':' + cnonce + ':' + qop
                  + ':' + MD5toHex( MD5OfStr( A2 ) )
               ) );
           if s<>response then begin
              Log( LOGID_Error, 'login rejected, wrong response value');
              SendResTag('NO Authentification failed!');
              CurrentUserName := '';
              CurrentUserID:=ACTID_INVALID;
              exit;
           end;
         except
           Log( LOGID_Error, 'unknown error');
           SendResTag('BAD Authentification failed!');
           //Soll hier ein 'NO' hin? IMHO wegen 'unbekannt' nicht.
           CurrentUserName := '';
           CurrentUserID:=ACTID_INVALID;
           exit;
        end;
        // build rspauth and send it
        a2 := ':' + digesturi;
        rspauth := MD5toHex( MD5OfStr(
                   MD5toHex( MD5OfStr( A1 ) )
                   + ':' + nonce + ':' + nc + ':' + cnonce + ':' + qop
                   + ':' + MD5toHex( MD5OfStr( A2 ) )
                   ) );
        s := 'rspauth=' + rspauth;
        Log( LOGID_DEBUG, 'DIGEST-MD5 rspauth: ' + s );
        s := EncodeB64( s[1], length(s) );
        s := SendRequest( '+ '+s);
        SendResTag(LoginUser( pass, 'DIGEST-MD5' ))
      end else
{JW}
      if par='CRAM-MD5' then begin
         TimeStamp := MidGenerator(Def_FQDNforMIDs);
         s := '+ '+EncodeB64( TimeStamp[1], length(TimeStamp) );
         s := SendRequest(s );
         if s='' then exit;
         s := DecodeB64( s[1], length(s) );
         if s='' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN protocoll error' );
            SendResTag('NO Authentification failed!');
            Exit;
         end;
         if s='*' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN cancel by client' );
            SendResTag('BAD Authentification failed!');
            Exit;
         end;
         Hash:=TrimWhSpace( copy(s,PosWhSpace(s)+1,32));
         CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
         CurrentUserID   := CfgAccounts.UserIDOf(CurrentUserName);
         if CurrentUserID=ACTID_INVALID then begin
            CurrentUserName := ''
         end else begin
            pass:=CfgAccounts.Value[CurrentUserID,ACTP_PASSWORD];
            s:=MD5HMAC( pass,TimeStamp );
            s:=MD5toHex( s );
            if s=Hash then
               SendResTag(LoginUser( pass, 'CRAM-MD5' ))
            else begin
               SendResTag('NO Authentification rejected!');
               CurrentUserID:=ACTID_INVALID;
               CurrentUserName := '';
            end;
         end;
      end  else
      if par='CRAM-SHA1' then begin
         TimeStamp := MidGenerator(Def_FQDNforMIDs);
         s := '+ '+EncodeB64( TimeStamp[1], length(TimeStamp) );
         s := SendRequest(s );
         if s='' then exit;
         s := DecodeB64( s[1], length(s) );
         if s='' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN protocoll error' );
            SendResTag('NO Authentification failed!');
            Exit;
         end;
         if s='*' then begin
            Log( LOGID_DETAIL, 'Auth LOGIN cancel by client' );
            SendResTag('BAD Authentification failed!');
            Exit;
         end;
         Hash:=TrimWhSpace( copy(s,PosWhSpace(s)+1,32));
         CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
         CurrentUserID   := CfgAccounts.UserIDOf(CurrentUserName);
         if CurrentUserID=ACTID_INVALID then begin
            CurrentUserName := ''
         end else begin
            pass:=CfgAccounts.Value[CurrentUserID,ACTP_PASSWORD];
            s:=HMAC_SHA1( pass,TimeStamp );
            s:=SHA1toHex( s );
            if s=Hash then
               SendResTag(LoginUser( pass, 'CRAM-SHA1' ))
            else begin
               SendResTag('NO Authentification rejected!');
               CurrentUserID:=ACTID_INVALID;
               CurrentUserName := '';
            end;
         end;
      end else begin
            CurrentUserName := '';
            CurrentUserID   := ACTID_INVALID;
            SendResTag('NO Unknown AUTH mechanism '+par);
            Log( LOGID_WARN, 'Unknown AUTH mechanism '+par );
      end;
   except
      CurrentUserID:=ACTID_INVALID;
      CurrentUserName := '';
   end;
   if CurrentUserID = ACTID_INVALID
      then Log(LOGID_WARN, 'IMAP: AUTHENTICATE: Failed')
      else Log(LOGID_INFO, 'IMAP: AUTHENTICATE: User '+CurrentUserName+' logged in');
end;
procedure TSrvIMAPCli.Cmd_CAPA( Par: String );
var capabilities: String;
begin
  if par<>'' then begin
    Log(LOGID_WARN, 'IMAP: CAPA: Too many arguments!');
    SendResTag('BAD I don''t know parameters for CAPABILITY!');
    exit;
  end;
  //---Standard-CAPAs--------------------------------
  capabilities := 'IMAP4rev1 '
                + 'AUTH=CRAM-SHA1 '
                + 'AUTH=CRAM-MD5 '   //JW //IMAP-Auth
                + 'AUTH=DIGEST-MD5 ' //JW //SASL-DIGEST
                + 'IDLE '            //HSR //IDLE
                + 'LITERAL+ ';        //HSR //Literal+


  //---Bedingte CAPAs--------------------------------
  if Def_IMAPNCBrain then
    capabilities := capabilities + 'X-NETSCAPE ';     //HSR //NCBrain

  if not Def_IMAP_DisableSASLLogin then
    capabilities := capabilities + 'AUTH=LOGIN '; //JW //IMAP-Auth

  {MG}{SSL}
  if not Assigned(SSLConnection) then begin
    if Assigned(SSLContext)   then capabilities := capabilities + 'STARTTLS ';
    if (Def_LocalImapTlsMode=2)
      and not Def_IMAP_DisableLogin then //HSR //LOGINDISABLED
          capabilities := capabilities + 'LOGINDISABLED ';
  end else capabilities := capabilities + 'AUTH=PLAIN ';
  {/SSL}

  if Def_IMAP_DisableLogin then //HSR //LOGINDISABLED
    capabilities := capabilities + 'LOGINDISABLED ';

  if Def_IMAP_ID  then //JW //IMAP ID
    capabilities := capabilities + 'ID ';

  //---Sending---------------------------------------
  SendRes ( 'CAPABILITY ' + trim(capabilities) );

  Log(LOGID_DETAIL, 'IMAP: Sent capabilities: ' + capabilities );
  SendResTag ('OK I''m ready sending capabilities!');
end;

procedure TSrvIMAPCli.Cmd_CHECK( Par: String );
begin
     if Par <> '' then begin
        Log(LOGID_WARN, 'IMAP: CHECK: Too many arguments!');
        SendResTag('BAD I don''t know parameters for CHECK!')
     end else begin
        if Assigned( Selected ) then begin
           try
              Selected.Lock;
              Selected.SendMailboxUpdate;
           finally
              Selected.Unlock;
           end;
        end;
        SendResTag ('OK CHECK completed.');
        Log(LOGID_DETAIL, 'IMAP: CHECK');
     end;
end;

procedure TSrvIMAPCli.Cmd_CLOSE( Par: String );
begin
     if Par <> '' then begin
        Log(LOGID_WARN, 'IMAP: CLOSE: Too many arguments!');
        SendResTag( 'BAD I don''t know parameters for CLOSE!' )
     end else begin
        try
           if not Selected.MBReadOnly then begin //ClientRO
              Log(LOGID_INFO, 'IMAP: Expunge (Close) of ' + Selected.Path);
              Selected.Expunge( SelNotify );
           end
        finally
           MBLogout( Selected, true );
           Log(LOGID_DETAIL, 'IMAP: CLOSE!');
           SendResTag( 'OK Mailbox closed.' );
        end
     end
end;

procedure TSrvIMAPCli.Cmd_COPY( Par: String );
var  MsgSetStr, Destination : String;
begin
     MsgSetStr   := CutFirstParam( Par );
     Destination := CutFirstParam( Par );
     if (MsgSetStr='') or (Destination='') then begin
        SendResTag( 'BAD COPY without message set / mailbox!' );
        Log(LOGID_WARN, 'IMAP: COPY: Missing arguments!');
     end else begin
        Log(LOGID_INFO, 'IMAP: COPY ' + MsgSetStr + '-->' + Destination);
        DoCopy( Selected.StrToMsgSet( MsgSetStr, False ), 'COPY', Destination )
     end
end;

procedure TSrvIMAPCli.Cmd_CREATE( Par: String );
var  Mailbox: String;
begin
     Mailbox := CutFirstParam( Par );
     if Mailbox = '' then begin
        SendResTag( 'BAD CREATE without mailbox!' );
        Log(LOGID_WARN, 'IMAP: CREATE: No mailbox given');
     end else if MBCreate( Mailbox ) then begin
        Log(LOGID_INFO, 'IMAP: CREATE: Mailbox "' + MailBox + '" created');
        SendResTag( 'OK Mailbox created!' )
     end else begin
        Log(LOGID_WARN, 'IMAP: CREATE: Mailbox "'+MailBox+'" not created');
        SendResTag( 'NO Mailbox not created!' )
     end
end;

procedure TSrvIMAPCli.Cmd_DELETE( Par: String );
var  Mailbox: String;
begin
     Mailbox := CutFirstParam( Par );
     if Mailbox = '' then begin
        Log(LOGID_WARN, 'IMAP: DELETE: Missing mailbox');
        SendResTag('BAD DELETE without mailbox!' )
     end else if MBDelete( Mailbox ) then begin
        Log(LOGID_INFO, 'IMAP: DELETE: Mailbox "' + MailBox + '" deleted');
        SendResTag( 'OK Mailbox deleted!' )
     end else begin
        Log(LOGID_WARN, 'IMAP: DELETE: Mailbox "'+MailBox+'" not deleted');
        SendResTag( 'NO Mailbox not deleted!' )
     end
end;

procedure TSrvIMAPCli.Cmd_EXAMINE( Par: String );
var  Mailbox: String;
begin
     Mailbox := CutFirstParam( Par );
     if Mailbox = '' then begin
        Log(LOGID_WARN, 'IMAP: EXAMINE: Missing mailbox');
        SendResTag('BAD EXAMINE without mailbox!' )
     end else begin
        if MBSelect( Mailbox, true ) then begin
           SendRes( 'OK [PERMANENTFLAGS ()] No permanent flags permitted' );
           SendResTag( 'OK [READ-ONLY] Mailbox opened' );
           Log(LOGID_INFO, 'IMAP: EXAMINE: Mailbox "'+MailBox+'" opened Read-Only');
        end else begin
           MBLogout( Selected, true );
           SendResTag( 'NO EXAMINE failed!' );
           Log(LOGID_WARN, 'IMAP: EXAMINE: Failed to open mailbox "'+MailBox+'" Read-Only');
        end
     end
end;

procedure TSrvIMAPCli.Cmd_EXPUNGE( Par: String );
begin
     if Par <> '' then begin
        Log(LOGID_WARN, 'IMAP: EXPUNGE: Too many arguments');
        SendResTag( 'BAD I don''t know parameters for EXPUNGE!' )
     end else if Selected.MBReadOnly then begin
        Log(LOGID_WARN, 'IMAP: EXPUNGE: Mailbox Read-Only');
        SendResTag( 'NO I can''t EXPUNGE (mailbox is read-only).' )
     end else begin
        Selected.Expunge( nil );
        Log(LOGID_INFO, 'IMAP: EXPUNGE: All marked messages are deleted');
        SendResTag( 'OK All deleted messages are removed.' );
     end
end;

procedure TSrvIMAPCli.Cmd_FETCH( Par: String );
var  MsgSetStr : String;
begin
     MsgSetStr := CutFirstParam( Par );
     if (MsgSetStr='') or (Par='') then begin
        Log(LOGID_WARN, 'IMAP: FETCH: Missing arguments');
        SendResTag( 'BAD FETCH without message set / data!' )
     end else begin
        Log(LOGID_DETAIL, 'IMAP: FETCH');
        DoFetch( Selected.StrToMsgSet( MsgSetStr, False ), 'FETCH', Par )
     end
end;

{JW} {IMAP ID}
procedure TSrvIMAPCli.Cmd_ID       ( Par: String );
var id: string;
begin
  if par='' then begin
    Log(LOGID_Detail, 'IMAP Client ID: missing argument!');
    SendResTag('BAD I''m missing parameters for ID!');
    exit;
  end;
  ID := '("name" "Hamster" '+
        '"version" "'+GetMyBuildInfo+'" '+
        '"os" "windows" '+
        '"os-version" "'+GetWinVerInfo+'" '+
        '"support-url" "news:hamster.de.misc")';

  SendRes ( 'ID ' + ID );
  Log(LOGID_DETAIL, 'IMAP: Sent ID: ' + ID );
  SendResTag ('OK ID completed!');
end;
{/JW}

procedure TSrvIMAPCli.Cmd_IDLE( Par: String ); //HSR //IDLE
begin
  if Par <> '' then begin
    Log(LOGID_WARN, 'IMAP: IDLE: Too many arguments!');
    SendResTag('BAD I don''t know parameters for IDLE!')
  end else begin
    if CurrentUserID<>ACTID_INVALID then begin
      Log(LOGID_DETAIL, 'IMAP: IDLE begin');
      SendData('+ You are IDLE now, changes will be sent immidiatelly' + CRLF);
      IdleState := true
    end else begin
      Log(LOGID_DETAIL, 'IMAP: IDLE didn''t begin');
      SendResTag ('NO Not authenticated.');
    end;
  end;
end;


{MG}{IMAP-List}
procedure TSrvIMAPCli.Cmd_LIST( Par: String );
begin
   Log(LOGID_DETAIL, 'IMAP: LIST');
   DoList( Par, False )
end;
{/IMAP-List}

procedure TSrvIMAPCli.Cmd_LOGIN( Par: String );
var  Pass      : string;
begin
     CurrentUserID   := ACTID_INVALID;
     CurrentUsername := '';
     {MG}{SSL}
     // disable the LOGIN command which uses clear-text passwords
     // unless encryption is active for security reasons
     if (Def_LocalImapTlsMode=2) and not Assigned(SSLConnection) then begin
        SendResTag('BAD TLS connection required for LOGIN - try STARTTLS');
        Log(LOGID_WARN, 'IMAP: LOGIN: TLS needed for this command by config');
        exit;
     end;
     {/SSL}

     if (Def_IMAP_DisableLogin) then begin
        SendResTag('BAD LOGIN is switched off by server-admin. Please use AUTHENTICATE.');
        Log(LOGID_WARN, 'IMAP: LOGIN: Disabled by config');
        exit;
     end;

     CurrentUsername := CutFirstParam( Par );
     Pass            := CutFirstParam( Par );
     if (CurrentUserName='') OR (Pass='') then begin
        SendResTag( 'BAD LOGIN without User / Pass!' );
        Log(LOGID_WARN, 'IMAP: LOGIN: Missing User/Pass');
     end else begin
        SendResTag( LoginUser( Pass, '' ) );
        Log(LOGID_INFO, 'IMAP: LOGIN: User ' + CurrentUserName + ' logged in');
     end
end;

procedure TSrvIMAPCli.Cmd_LOGOUT( Par: String );
begin
     Log(LOGID_INFO, 'IMAP: LOGOUT');
     CurrentUserID   := ACTID_INVALID;
     try
       if Assigned(Selected) then MBLogout( Selected, true );

       if Assigned(SubscribedMBs) then begin
         SubscribedMBs.SaveToFile(MailboxPath + IMAPSUBSCRIBED_FILENAME);
         FreeAndNil(SubscribedMBs);
       end;
     except
     end;

     if CurrentUserID>=0 then CurrentUserID := ACTID_INVALID;

     Log(LOGID_DETAIL, 'IMAP: Let the client disconnect-->disconnect');
     try
       if ClientSocket.Connected then begin
         SendRes( 'BYE IMAP4rev1 closing connection - goodbye!' );
         SendResTag( 'OK Closing.');
       end;

       Sleep( Def_LocalTimeoutQuitDelay );
       try
          if ClientSocket.Connected then ClientSocket.Close;
       except
          on E:Exception do Log(LOGID_DEBUG, 'Exception on Socket.Close: ' + E.Message );
       end;
     finally
       Terminate;
     end;
end;

{MG}{IMAP-List}
procedure TSrvIMAPCli.Cmd_LSUB( Par: String );
begin
   Log(LOGID_DETAIL, 'IMAP: LSUB');
   DoList( Par, True )
end;
{/IMAP-List}

{HSR} {NCBRain}
procedure TSrvIMAPCli.Cmd_NCBrain(Par: String);
const NCBURL = 'http://www.rimarts.co.jp'; 
begin //Got out of Cyrus-Source
  SendRes('OK [NETSCAPE]');
  SendRes('* VERSION 1.0 UNIX');
  SendRes('* ACCOUNT-URL "' + NCBURL + '"');
  SendResTag('OK Your brain is done now...');
end;
{/HSR}

procedure TSrvIMAPCli.Cmd_NOOP( Par: String );
begin
  if par<>'' then begin
    Log(LOGID_WARN, 'IMAP: NOOP: Too many arguments');
    SendResTag('BAD I don''t know parameters for NOOP!')
  end else begin
    if Assigned( Selected ) then begin
       try
         Selected.Lock;
         Selected.SendMailboxUpdate;
       finally
         Selected.Unlock;
       end;
    end;
    Log(LOGID_DETAIL, 'IMAP: NOOP');
    SendResTag ('OK Noop isn''t slow, is it? ;-)');
  end;
end;

procedure TSrvIMAPCli.Cmd_RENAME( Par: String );
var  OldName, NewName: String;
begin
     OldName := CutFirstParam( Par );
     NewName := CutFirstParam( Par );

     if (OldName='') or (NewName='') then begin
        SendResTag( 'BAD RENAME without existing / new name!' );
        Log(LOGID_WARN, 'IMAP: RENAME: Missing arguments')
     end else if MBRename( OldName, NewName ) then begin
        Log(LOGID_INFO, 'IMAP: RENAME: '+OldName+'-->'+NewName);
        SendResTag( 'OK Mailbox renamed.' )
     end else begin
        Log(LOGID_WARN, 'IMAP: RENAME failed');
        SendResTag( 'NO Mailbox not renamed!' )
     end
end;

procedure TSrvIMAPCli.Cmd_SEARCH( Par: String );
begin
   if par='' then begin
      Log(LOGID_WARN, 'IMAP: SEARCH: Missing arguments');
      SendResTag('BAD SEARCH without arguments!')
   end else begin
      Log(LOGID_DETAIL, 'IMAP: SEARCH');
      DoSearch( False, Par )
   end
end;

procedure TSrvIMAPCli.Cmd_SELECT( Par: String );
var  Mailbox: String;
begin
     if Assigned(Selected) then MBLogout( Selected, true ); //RFC!

     Mailbox := CutFirstParam( Par );
     if Mailbox = '' then begin
        Log(LOGID_WARN, 'IMAP: SELECT: Missing arguments');
        SendResTag( 'BAD SELECT without mailbox!' )
     end else begin
//        ReadOnly := False;
        if MBSelect( Mailbox, fReadOnly ) then begin
           if Selected.MBReadOnly then begin //ClientRO
             SendRes( 'OK [PERMANENTFLAGS ()] ' +
                      'Flags you can change permanently: NONE');

             SendResTag( 'OK [READ-ONLY] Mailbox opened' );
             Log(LOGID_INFO, 'IMAP: SELECT: '+MailBox+' opened (Read-Only)')
           end else begin
             SendRes( 'OK [PERMANENTFLAGS '+Selected.PossFlags+'] ' +
                      'Flags you can change permanently');

             SendResTag( 'OK [READ-WRITE] Mailbox opened' );
             Log(LOGID_INFO, 'IMAP: SELECT: '+MailBox+' opened (Read-Write)')
           end
        end else begin
           MBLogout( Selected, true );
           SendResTag( 'NO SELECT failed!' );
           Log(LOGID_WARN, 'IMAP: SELECT failed at '+mailbox);
        end
     end
end;

procedure TSrvIMAPCli.Cmd_STARTTLS( Par: String ); {MG}{SSL}
begin
    if not SSLReady then begin
       SendResTag( 'BAD Command not implemented.' );
       Log(LOGID_WARN, 'IMAP: STARTTLS not available by config');
    end else
    if Par<>'' then begin
       SendResTag( 'BAD I don''t know parameters for STARTTLS!' );
       Log(LOGID_WARN, 'IMAP: STARTTLS: Too many arguments');
    end else
    if SSLContext = nil then begin
       SendResTag( 'BAD TLS not available due to temporary error' );
       Log(LOGID_ERROR, 'IMAP: STARTTLS: SSLContext is wrong');
    end else
    if SSLConnection = nil then begin
       Log(LOGID_INFO, 'IMAP: STARTTLS: Beginning transaction with SSL');
       SendResTag( 'OK Begin TLS negotiation' );
       if StartSSL then begin
          if CurrentUserID<>ACTID_INVALID then CurrentUserID := ACTID_INVALID;
          CurrentUserName := '';
       end;
    end else begin
       Log(LOGID_WARN, 'IMAP: STARTTLS: TLS was allready started');
       SendResTag( 'BAD Command not permitted when TLS active' )
    end
end;

procedure TSrvIMAPCli.Cmd_STATUS( Par: String );
var  Mailbox, MbName, Status, StatusU, Erg : String;
     MbxStatus : TMbxStatus;
     i : Integer;
begin
     MailBox   := CutFirstParam( Par );
     if (Mailbox='') or (Par='') then begin
        Log(LOGID_WARN, 'IMAP: STATUS: Missing arguments');
        SendResTag('BAD STATUS without mailbox / status-data!');
        exit;
     end;
     MbName    := Mailbox;

     if not SafeString( Mailbox ) then begin
       Log(LOGID_WARN, 'IMAP: STATUS: Forbidden characters');
       SendResTag('BAD STATUS Mailbox parameter contains forbidden characters!');
       exit;
     end;
     if not MBExists( Mailbox ) then begin
       Log(LOGID_WARN, 'IMAP: STATUS: Mailbox "'+MailBox+'" don''t exists'); 
       SendResTag('NO STATUS error: mailbox does not exist!');
       exit
     end;

     MbxStatus := GetStatus( Mailbox );

     Status := TrimParentheses( TrimQuotes( Par ) ) + ' ';
     Erg := '';
     i := PosWhSpace( Status );
     repeat
       StatusU := uppercase(copy(Status,1,i-1));
       Status  := copy(Status,i+1,length(Status));

       if StatusU='MESSAGES' then
         Erg := Erg + ' MESSAGES '     + IntToStr(MbxStatus.Messages);

       if StatusU='RECENT' then
         Erg := Erg + ' RECENT '       + IntToStr(MbxStatus.Recent);

       if StatusU='UIDNEXT' then
         Erg := Erg + ' UIDNEXT '      + IntToStr(MbxStatus.UIDNext);

       if StatusU='UIDVALIDITY' then
         Erg := Erg + ' UIDVALIDITY '  + IntToStr(MbxStatus.UIDValidity);

       if StatusU='UNSEEN' then
         Erg := Erg + ' UNSEEN '       + IntToStr(MbxStatus.Unseen);

       i := PosWhSpace( Status );
     until i=0;

     SendRes('STATUS "' + MbName + '" (' + Trim(Erg) + ')');
     SendResTag('OK You now have the status!');
     Log(LOGID_DETAIL, 'IMAP: STATUS');
end;

procedure TSrvIMAPCli.Cmd_STORE( Par: String );
var  MsgSetStr: String;
begin
     MsgSetStr := CutFirstParam( Par );
     if (MsgSetStr='') or (Par='') then begin
        Log(LOGID_WARN, 'IMAP: STORE: Missing arguments');
        SendResTag( 'BAD STORE arguments missing!' )
     end else begin
        Log(LOGID_DETAIL, 'IMAP: STORE parameters');
        DoStore( Selected.StrToMsgSet( MsgSetStr, False ), 'STORE', Par )
     end
end;

{IMAP-List}
procedure TSrvIMAPCli.Cmd_SUBSCRIBE( Par: String );
var  Mailbox: String;
begin
     Mailbox := uppercase( CutFirstParam(Par) );
     if SubscribedMBs.IndexOf(Mailbox) = -1 then SubscribedMBs.Add(Mailbox);
     Log(LOGID_DETAIL, 'IMAP: SUBSCRIBE: ' + MailBox);
     SendResTag( 'OK Mailbox subscribed' );
end;
{/IMAP-List}

procedure TSrvIMAPCli.Cmd_UNSUBSCRIBE( Par: String );
var  Idx: Integer;
     MailBox: String; 
begin
     MailBox := UpperCase(CutFirstParam(Par));
     Idx := SubscribedMBs.IndexOf( Mailbox );
     if Idx = -1 then begin
        Log(LOGID_INFO, 'IMAP: UNSUBSCRIBE: ' + MailBox + ' failed: Not yet subscribed');
        SendResTag( 'NO This mailbox is not subscribed yet' )
     end else begin
        SubscribedMBs.Delete( Idx );
        SendResTag( 'OK Mailbox unsubscribed' );
        Log(LOGID_DETAIL, 'IMAP: UNSUBSCRIBE: ' + MailBox);
     end;
end;

procedure TSrvIMAPCli.Cmd_UID( Par: String );
var
     i : integer;
     Command, CmdParams: String;
     MsgSet: TMessageSet;
begin
     SetLength( MsgSet, 0 );

     i := PosWhSpace( Par );
     if (par='') or (i=0) then begin
        Log(LOGID_WARN, 'IMAP: UID: Missing arguments');
        SendResTag('BAD UID without Command/Cmd-Params!');
        exit;
     end;
     Command   := Uppercase( TrimQuotes( copy( Par, 1, i-1 ) ) );
     CmdParams := TrimQuotes( copy( Par, i+1, length(Par) ) );

     Log(LOGID_DETAIL, 'IMAP: UID-' + Command);

     if Command = 'SEARCH' then begin
        DoSearch( True, CmdParams )
     end else begin
        i := PosWhSpace( CmdParams );
        if i = 0 then begin
           Log(LOGID_WARN, 'IMAP: UID-'+Command+': Missing arguments');
           SendResTag( 'BAD UID ' + Command + ': not enough arguments!');
           exit
        end;

        MsgSet := Selected.StrToMsgSet( copy( CmdParams, 1, i-1 ), True );
        CmdParams := TrimQuotes(TrimWhSpace( copy( CmdParams, i+1, length(CmdParams) ) ) );

        if
           Command = 'COPY'  then DoCopy( MsgSet, 'UID COPY', CmdParams)
        else if
           Command = 'STORE' then DoStore( MsgSet, 'UID STORE', CmdParams )
        else if
           Command = 'FETCH' then DoFetch( MsgSet, 'UID FETCH', CmdParams)
        else begin
           Log(LOGID_WARN, 'IMAP: UID-'+Command+' not known');
           SendResTag('BAD I don''t know this UID-command!')
        end
     end
end;

//#####################################################################
constructor TSrvIMAPCli.Create( ASocket: TServerClientWinSocket;
                                Const AIPAccessScope: LongInt;
                                Const ASSLContext: Pointer );
begin
   inherited Create( ASocket, AIPAccessScope, ASSLContext );
   fReadOnly := not (IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW; //ClientRO

   HierarchyDelimiter := '/';
   CurrentInactTimeOut:=Def_LocalIMAPLoginTimeout;
   Selected := nil;

   LiteralCount := 0;
   LimitTextSize := Def_LocalLimitTextSizeIMAP;

   {HSR} {IDLE}
   InitializeCriticalSection( CS_THR_IDLE );
   SendNewMessages := false;
   SetLength(SendExpunge, 0);
   IdleState       := false;
   {/HSR}

   CurrentUserID   := ACTID_INVALID;
   CurrentUserName := '';

   IncCounter(CntIMAPCli,1);
end;

destructor TSrvIMAPCli.Destroy;
begin
   if Assigned(Selected) then MBLogout( Selected, true );
   if Assigned(SubscribedMBs) then begin //Leak1
      SubscribedMBs.SaveToFile(MailboxPath + IMAPSUBSCRIBED_FILENAME);
      FreeAndNil(SubscribedMBs);
   end;
   IncCounter(CntIMAPCli,-1);
   DeleteCriticalSection( CS_THR_IDLE );
   inherited;
end;

// ------------------------------------------------------------- TSrvIMAP -----

constructor TSrvIMAP.Create;
begin
   inherited Create(
      AOwner,
      CfgIni.ReadString ('Setup', 'local.IMAP.serverbind', Def_LocalIMAPServerBind ),
      CfgIni.ReadInteger('Setup', 'local.port.IMAP', DEF_LOCALIMAPServer_PORT  ),
      CfgIni.ReadInteger('Setup', 'MaxLocalIMAPServers', Def_Max_Local_IMAP_Servers),
      CfgIni.ReadInteger('Setup', 'MaxLocalIMAPServersPerIP', Def_Max_Local_IMAP_Servers_Per_IP),
      IPACC_SCOPE_IMAP,
      TSrvIMAPCli
   );
end;

// ----------------------------------------------------------------------------

function TSrvIMAPCli.ReplaceChars(S: String; OldChar,
  NewChar: Char): String;
var i : integer;
begin
  for i := 1 to length(s) do if s[i]=OldChar then s[i] := NewChar;
  Result := s
end;

{MG}{IMAP-List}
procedure TSrvIMAPCli.DoList( Par: String; LSub: Boolean );

     procedure SendList( Txt: String );
     var  s: String;
     begin
          if LSub then s := 'LSUB (' else s := 'LIST (';
          if FileExists2( MailBoxPath + ReplacePathDelimiters( Txt ) + 'NOSELECT' )
             then s := s + '\NOSELECT';
          SendRes( s + ') "' + HierarchyDelimiter + '" "' + Txt + '"' );
     end;

     procedure ScanFolders( RegEx, Dir, Base: String );
     var  SR : TSearchRec;
          Found: String;
     begin
          if SysUtils.FindFirst( Dir + '*.*', faDirectory, SR ) = 0 then begin
             repeat
                if ((SR.Attr and faDirectory)<>0) and (SR.Name<>'.') and (SR.Name<>'..') then begin
                   Found := Base + SR.Name;
                   if ( not LSub or (SubscribedMBs.IndexOf(Found) > -1) )
                      and RE_Match( Found, RegEx, PCRE_CASELESS ) then SendList( Found );
                   ScanFolders( RegEx, Dir + SR.Name + '\', Found + HierarchyDelimiter );
                end;
             until SysUtils.FindNext( SR ) <> 0;
             SysUtils.FindClose( SR )
          end
     end;

var  i : Integer;
     Reference, Mailbox, Pattern, RegEx : String;
begin
   i := PosWhSpace( Par );
   if (par='') or (i=0) then begin
      SendResTag('BAD missing reference/mailbox parameter!');
      exit;
   end;
   Reference := TrimQuotes( copy( Par, 1, i-1 ) );
   Mailbox   := TrimQuotes( copy( Par, i+1, length(Par) ) );

   if not SafeString( Reference ) then begin
      SendResTag( 'BAD reference parameter contains forbidden characters!' );
      exit;
   end;

   if (Mailbox = '') then begin
      SendRes( 'LIST (\NOSELECT) "' + HierarchyDelimiter + '" "' +
               Copy( Reference, 1, Pos(HierarchyDelimiter,Reference)- 1 ) + '"' )
   end else begin
      if not SafeString( Mailbox ) then begin
         SendResTag('BAD mailbox parameter contains forbidden characters!' );
         exit;
      end;

      Pattern := Reference + Mailbox;
      RegEx := '^';
      for i := 1 to Length( Pattern ) do begin
         case Pattern[i] of
            '*' : RegEx := RegEx + '.*';
            '%' : RegEx := RegEx + '[^' + HierarchyDelimiter + ']*';
            '+', '-', '.', '$', '(', ')': RegEx := RegEx+'\'+Pattern[i];
            else  RegEx := RegEx + Pattern[i];
         end
      end;
      RegEx := RegEx + '$';

      if RE_Match( 'INBOX', RegEx, PCRE_CASELESS ) then SendList( 'INBOX' );

      ScanFolders( RegEx, MailboxPath, '' );
   end;

   SendResTag( 'OK You have now the List!' )
end;
{/IMAP-List}

procedure TSrvIMAPCli.NewMessages; //HSR //IDLE
begin
  try
    EnterCriticalSection( CS_THR_IDLE );
    if IdleState then begin
      if Assigned(Selected) then begin
        SendRes( IntToStr(Selected.Status.Messages) + ' EXISTS');
        SendRes( IntToStr(Selected.Status.Recent)   + ' RECENT')
      end
    end else
      SendNewMessages := true
  finally
    LeaveCriticalSection( CS_THR_IDLE )
  end
end;

procedure TSrvIMAPCli.NewExpunge(Number : Integer); //HSR //IDLE
begin
  try
    EnterCriticalSection( CS_THR_IDLE );
    if IdleState then begin
      if Assigned(Selected) then begin
        SendRes( IntToStr(Number) + ' EXPUNGE')
       end
    end else begin
      SetLength(SendExpunge, length(SendExpunge)+1);
      SendExpunge[length(SendExpunge)-1] := Number
    end
  finally
    LeaveCriticalSection( CS_THR_IDLE )
  end
end;

end.


