// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit dGroupSelectFromServer;

interface

uses Windows, Messages, SysUtils, Classes, Graphics, Forms, Controls, StdCtrls,
  Buttons, ExtCtrls, ComCtrls, cStdForm;

type
  TDescsFilterThread = class( TThread )
    protected
      lstDescs : TListBox;
      lblCount : TLabel;
      MatchText: String;
      GrpInfo  : String;
      ErrMsg   : String;
      ForNewsserver: String;
      GrList: TStrings;
      procedure ListInit;
      procedure ListExit;
      procedure ShowCount;
      procedure ListAdd;
      procedure Execute; override;
    public
      constructor Create( AlstDescs : TListBox;
                          AlblCount : TLabel;
                          AMatchText: String;
                          AGrList   : TStrings);
  end;

type
  TGetGroupPullsForServerDlg = class(THForm)
    pnlBase: TPanel;
    frRight: TPanel;
    hdrGroups: THeaderControl;
    lstGroups: TListBox;
    Panel1: TPanel;
    Label1: TLabel;
    emSelGroup: TEdit;
    pnlFooter: TPanel;
    OKBtn: TButton;
    CancelBtn: TButton;
    lblDescsCount: TLabel;
    cbType: TComboBox;
    frLeft: TPanel;
    labSubscribed: TLabel;
    Panel5: TPanel;
    butSubscribe: TButton;
    butHelp: TButton;
    Splitter1: TSplitter;
    SplitPulls: TSplitter;
    frDeadPulls: TPanel;
    Panel4: TPanel;
    Label2: TLabel;
    hdrPulls2: THeaderControl;
    lstPulls2: TListBox;
    labLeftDown: TPanel;
    butUnsubscribe2: TButton;
    Panel2: TPanel;
    labSubscAndExisting: TPanel;
    labUnsubscribed: TLabel;
    hdrPulls1: THeaderControl;
    lstPulls1: TListBox;
    Panel3: TPanel;
    butUnsubscribe1: TButton;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure emSelGroupChange(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure lstDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure hdrGroupsSectionResize(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure hdrPullsSectionResize(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure FormActivate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure butUnsubscribe2Click(Sender: TObject);
    procedure butSubscribeClick(Sender: TObject);
    procedure Panel3Resize(Sender: TObject);
    procedure labLeftDownResize(Sender: TObject);
    procedure Panel5Resize(Sender: TObject);
    procedure butUnsubscribe1Click(Sender: TObject);
    procedure lstPulls1MouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
  private
    { Private declarations }
    slAvGroups, slSelGroups, slDeadPulls, slDelDeadPulls, slOrg: TStringlist;
    First: boolean;
    DescsFilterThread: TDescsFilterThread;
    procedure ShowDescs( StartIt: Boolean );
    procedure ShowAvSel;
  public
    { Public declarations }
    ForNewsserver, AddGroups, DelGroups: String;
  end;

Function GroupPullsForServer (Const ANewsserver: String; Var Add, Del: String): boolean;

implementation

{$R *.DFM}

uses Global, Config, cPCRE, uTools;

Var
  LastType: Integer = 0;
  LastSelGroup : String = ''; // persistent per session

// ------------------------------------------------------------ functions -----

Function GroupPullsForServer (Const ANewsserver: String; Var Add, Del: String): boolean;
Var s: String;
begin
   With TGetGroupPullsForServerDlg.Create(nil) do try
      ForNewsserver := ANewsserver;
      s := Caption;
      If Pos('%s', s)>0 then Caption := Format(s, [ANewsserver]);
      Result := ShowModal=mrOK;
      If Result then begin
         Add := AddGroups; Del := DelGroups
      end
   finally
      free
   end
end;

// --------------------------------------------------- TDescsFilterThread -----

procedure TDescsFilterThread.ListInit;
begin
     ErrMsg := '';
     lstDescs.Clear;
     lstDescs.Color := clLtGray;
     ShowCount;
end;

Const Anzeigelos = 100;

procedure TDescsFilterThread.ListExit;
begin
   if not Terminated then begin
      lstDescs.Color := clWindow;
      ShowCount;
   end;
end;

procedure TDescsFilterThread.ShowCount;
begin
   if lstDescs.Items.Count>=32000 then begin
      lblCount.Caption := TrGl('TGetGroupNameDlg', 'Hits', 'Hits') + ': >32000';
   end else begin
      lblCount.Caption := TrGl('TGetGroupNameDlg', 'Hits', 'Hits') + ': ' + inttostr( lstDescs.Items.Count );
   end;
end;

procedure TDescsFilterThread.ListAdd;
begin
   if lstDescs.Items.Count>=32000 then begin
      ShowCount;
      exit;
   end else begin
      try lstDescs.Items.Add( GrpInfo ) except end;
      if (lstDescs.Items.Count mod 100)=0 then ShowCount;
   end;
end;

procedure TDescsFilterThread.Execute;
var  LfdGrp: Integer;
     RegExGroup: TPCRE;
begin
     Synchronize( ListInit );
     RegExGroup := TPCRE.Create( True, PCRE_CASELESS );

     try
        RegExGroup.Compile( PChar(MatchText) );

        For LfdGrp:=0 to GrList.Count-1 do begin
           If Terminated then break;
           GrpInfo := GrList[LfdGrp];
           if MatchText='' then begin
              Synchronize( ListAdd );
           end else begin
              if RegExGroup.Exec( PChar(GrpInfo), 0 ) then Synchronize( ListAdd );
           end
        end

     except
     {JW} {Overflow}
//   on E: Exception do begin ErrMsg:=E.Message; Synchronize(ListAdd) end;
      end;
     {/JW}
     RegExGroup.Free;
     FreeAndNil(GrList);
     Synchronize( ListExit )
end;

constructor TDescsFilterThread.Create( AlstDescs : TListBox;
                                       AlblCount : TLabel;
                                       AMatchText: String;
                                       AGrList: TStrings);
begin
     inherited Create( True );

     lstDescs  := AlstDescs;
     lblCount  := AlblCount;
     MatchText := AMatchText;
{JK} {GrpInfo}
//     GrList    := AGrList;
     GrList:=TStringList.Create;
     GrList.Assign(AGrList);
{/JK}
     FreeOnTerminate := False;
     Resume;
end;

// ----------------------------------------------------- TGetGroupNameDlg -----

procedure TGetGroupPullsForServerDlg.FormCreate(Sender: TObject);
Var x: Integer;
begin
     First := true;
     DescsFilterThread := nil;
     slSelGroups := TStringlist.Create;
     slDeadPulls := TStringList.Create;
     slDelDeadPulls := TStringList.Create;
     slAvGroups := TStringlist.Create;
     slOrg := TStringlist.Create;

     labSubscribed.Font.Style := [fsBold];
     labUnsubscribed.Font.Style := [fsBold];

     LoadWindowState( Self, 'SelPulls' );

     With hdrGroups.Sections[0] do Width := CfgIni.ReadInteger( 'SelPulls', 'DescsWidth0', Width );
     With hdrPulls1.Sections[0] do Width := CfgIni.ReadInteger( 'SelPulls', 'PullsWidth0', Width );
     With hdrPulls2.Sections[0] do Width := hdrPulls1.Sections[0].Width;

     x := lstGroups.Canvas.TextHeight('Gg');
     lstGroups.ItemHeight := x;
     lstPulls1.ItemHeight := x;
     lstPulls2.ItemHeight := x;
     cbType.ItemIndex := LastType;
     emSelGroup.Text  := LastSelGroup;

     { // Klappt nicht wegen THeaderelementen drber...
     SendMessage( lstDescs1.Handle, LB_SETHORIZONTALEXTENT, 2000, 0 );
     SendMessage( lstDescs2.Handle, LB_SETHORIZONTALEXTENT, 2000, 0 );}
end;

procedure TGetGroupPullsForServerDlg.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
     ShowDescs( False );
     SaveWindowState( Self, 'SelPulls' );
end;

procedure TGetGroupPullsForServerDlg.ShowDescs( StartIt: Boolean );
Var s, s2: String; i: Integer;
begin
     if Assigned( DescsFilterThread ) then begin
        DescsFilterThread.Terminate;
        DescsFilterThread.WaitFor;
        FreeAndNil(DescsFilterThread)
     end;

     if StartIt then begin
        LastType := cbType.Itemindex;
        LastSelGroup := emSelGroup.Text;
        s2 := emSelGroup.Text; s := s2;
        If s2 > '' then Case LastType of
           1: s := EscRegEx(s2);
           2: s := '^'+EscRegEx(s2);
           3: begin
                 s := '^';
                 For i:=1 to Length(s2)-1 do s := s + s2[i] + '[^. ]*\.';
                 s := s + s2[Length(s2)]
              end
        end;
        DescsFilterThread := TDescsFilterThread.Create(
           lstGroups, lblDescsCount, s, slAvGroups)
     end;
end;

procedure TGetGroupPullsForServerDlg.emSelGroupChange(Sender: TObject);
begin
     ShowDescs( True );
end;

procedure TGetGroupPullsForServerDlg.lstDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var  t, g, d: String;
     j, x   : Integer;
     R      : TRect;
     myHdrDescs: THeaderControl; //JAWO //010428
begin
     with Control as TListBox do With Canvas do begin
        t := Items[Index];
        j := Pos( #9, t );
        if j>0 then begin
           g := copy( t, 1, j-1 );
           d := copy( t, j+1, 999 );
        end else begin
           g := t;
           d := '';
        end;

        x := LongInt(Items.Objects[Index]);
        If x = 0 then begin
           If CfgHamster.ExistPullServer( g ) then x := 1
                                              else x := 2;
           Items.Objects[Index] := Pointer(x)
        end;
        If x = 1 then begin
           if [odSelected]<=State then Font.Color := clYellow
                                  else Font.Color := clBlue;
        end else
        If x = 3 then begin
           if [odSelected]<=State then Font.Color := clYellow
                                  else Font.Color := clPurple;
        end;

        FillRect(Rect);
        R := Rect;
{JAWO} {010428}

        if (Control=lstPulls1) then myHdrDescs := hdrPulls1
        else if (Control=lstPulls2) then myHdrDescs := hdrPulls2
        else myHdrDescs := hdrGroups;
        
//        R.Right := hdrDescs1.Sections[0].Width;
        R.Right := myHdrDescs.Sections[0].Width;
{/JAWO}
        if R.Right>4 then dec(R.Right,4);
        TextRect( R, R.Left, R.Top, g );

        R := Rect;
//        inc( R.Left, hdrDescs1.Sections[0].Width );
        inc( R.Left, myHdrDescs.Sections[0].Width ); //JAWO //010428 
        TextOut( R.Left, R.Top, d );
     end;
end;

procedure TGetGroupPullsForServerDlg.hdrGroupsSectionResize(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
     lstGroups.Refresh;
     CfgIni.WriteInteger( 'SelPulls', 'DescsWidth0', hdrGroups.Sections[0].Width );
end;

procedure TGetGroupPullsForServerDlg.hdrPullsSectionResize(
  HeaderControl: THeaderControl; Section: THeaderSection);
Var x: Integer;
begin
   x := HeaderControl.Sections[0].Width;
   If HeaderControl = HdrPulls1 then begin
      lstPulls1.Refresh;
      HdrPulls2.Sections[0].Width := x
   end else begin
      lstPulls2.Refresh;
      HdrPulls1.Sections[0].Width := x
   end;
   CfgIni.WriteInteger( 'SelPulls', 'PullsWidth0', x )
end;

procedure TGetGroupPullsForServerDlg.FormActivate(Sender: TObject);
var  i, j, p: Integer;
     Descs, Pulls: TStringList;
     GroupFilename, GroupsDesc, s, s2, s3, last: String;
begin
   If First then begin
      First := false;
      Descs := TStringList.Create;
      Pulls := TStringList.Create;
      try
         Screen.Cursor := crHourGlass;
         lstGroups.Color := clLtGray;
         lstPulls1.Color := clLtGray;
         lstPulls2.Color := clLtGray;
         Application.ProcessMessages;

         With CfgHamster do
            for i:=0 to PullCount-1 do
               if lowercase(PullServer[i])=lowercase(ForNewsServer)
                  then Pulls.Add (lowercase(PullGroup[i]));

         GroupFilename := PATH_SERVER + ForNewsServer + '\groups.txt';
         If FileExists2(GroupFileName) then slAvGroups.LoadFromFile( GroupFileName );
         slAvGroups.Sorted := true; slAvGroups.Sorted := false;

         GroupsDesc := PATH_SERVER + ForNewsServer + '\grpdescs.txt';
         If FileExists2(GroupsDesc) then Descs.LoadFromFile( GroupsDesc );
         Descs.Sorted := true; Descs.Sorted := false;

         last := '';
         For i := slAvGroups.Count-1 downto 0 do begin
            s := slAvGroups [i];
            p := PosWhSpace (s);
            If p > 0 then Delete(s, p, Length(s)-p+1);
            If (s = '') or (s = last) then
               slAvGroups.Delete(i)
            else begin
               last := s; s2 := '';
               For j := Descs.Count-1 downto 0 do begin
                  s3 := Descs[j]; If s3 < s then break;
                  Descs.Delete(j);
                  If s = Copy(s3, 1, Length(s)) then begin
                     s2 := s3; break
                  end
               end;
               If s2 = '' then s2 := '?'
               else begin
                  p := PosWhSpace (s2);
                  If p > 0 then s2 := Copy(s2, p+1, Length(s2)-p);
                  While (Copy(s2, 1, 1)+'x')[1] IN[#9, ' '] do Delete(s2, 1, 1)
               end;
               s2 := s + #9 + s2;
               p := Pulls.IndexOf (LowerCase(s));
               If p >= 0 then begin
                  slAvGroups.Delete(i);
                  slSelGroups.Add (s2);
                  Pulls.Delete (p)
               end else begin
                  slAvGroups[i] := s2
               end
            end
         end;
         slOrg.Assign (slSelGroups);
         slDeadPulls.Assign (Pulls);
         frDeadPulls.Visible := slDeadPulls.Count > 0;
         SplitPulls.Visible := slDeadPulls.Count > 0;
      finally
         Screen.Cursor := crDefault;
         descs.free; Pulls.free
      end;
      ShowAvSel;
      lstPulls1.Color := clWindow;
      lstPulls2.Color := clWindow;
      ShowDescs( True )
   end
end;

procedure TGetGroupPullsForServerDlg.FormDestroy(Sender: TObject);
begin
   slAvGroups.free; slSelGroups.free;
   slDeadPulls.Free; slDelDeadPulls.Free;
   slOrg.free
end;

procedure TGetGroupPullsForServerDlg.ShowAvSel;
begin
   lstPulls1.Items.Assign (slSelGroups);
   lstPulls2.Items.Assign (slDeadPulls)
end;

procedure TGetGroupPullsForServerDlg.FormResize(Sender: TObject);
begin
   frRight.Width := ClientWidth div 2;
   With CancelBtn do Left := Parent.Width - Width - Top;
   With OKBtn do Left := CancelBtn.Left - Top - Width;
end;

procedure TGetGroupPullsForServerDlg.butUnsubscribe1Click(Sender: TObject);
Var i: Integer;
begin
   With lstPulls1 do With Items do begin
      For i:=Count-1 downto 0 do begin
         If Selected[i] then begin
            lstGroups.Items.InsertObject(0, Items[i], Pointer(3));
            Delete(i);
            slAvGroups.Insert(0, slSelGroups[i]);
            slSelGroups.Delete(i)
         end
      end
   end
end;

procedure TGetGroupPullsForServerDlg.butUnsubscribe2Click(Sender: TObject);
Var i: Integer; 
begin
   With lstPulls2 do With Items do begin
      For i:=Count-1 downto 0 do begin
         If Selected[i] then begin
            Delete(i);
            slDelDeadPulls.Add (slDeadPulls[i]);
            slDeadPulls.Delete(i)
         end
      end
   end
end;

procedure TGetGroupPullsForServerDlg.butSubscribeClick(Sender: TObject);
Var i, p: Integer; s: String;
begin
   With lstGroups do With Items do begin
      For i:=Count-1 downto 0 do begin
         If Selected[i] then begin
            s := Items[i];
            lstPulls1.Items.InsertObject(0, s, Pointer(3));
            Delete(i);
            slSelGroups.Insert(0, s);
            p := slAvGroups.IndexOf(s);
            If p >= 0 then slAvGroups.Delete(p)
         end
      end
   end
end;

procedure TGetGroupPullsForServerDlg.OKBtnClick(Sender: TObject);
Var s: String; i, p: Integer;
begin
   AddGroups := '';
   DelGroups := '';
   For i := 0 to slSelGroups.Count-1 do begin
      s := slSelGroups[i];
      p := slOrg.IndexOf(s);
      If p<0 then begin
         { Neuer Pull }
         p := Pos(#9, s); If p > 0 then s := Copy(s, 1, p-1);
         AddGroups := AddGroups + s + ','
      end else begin
         slOrg.delete(p)
      end
   end;
   For i:=0 to slOrg.Count-1 do begin
      { Pull lschen }
      s := slOrg[i];
      p := Pos(#9, s); If p > 0 then s := Copy(s, 1, p-1);
      DelGroups := DelGroups + s + ','
   end;
   For i:=0 to slDelDeadPulls.Count-1 do begin
      { Pull lschen }
      s := slDelDeadPulls[i];
      p := Pos(#9, s); If p > 0 then s := Copy(s, 1, p-1);
      DelGroups := DelGroups + s + ','
   end;
end;


procedure TGetGroupPullsForServerDlg.Panel3Resize(Sender: TObject);
begin
   With butUnsubscribe1 do Width := (Sender as TPanel).ClientWidth - 2 * left
end;

procedure TGetGroupPullsForServerDlg.labLeftDownResize(Sender: TObject);
begin
   With butUnsubscribe2 do Width := (Sender as TPanel).ClientWidth - 2 * left
end;

procedure TGetGroupPullsForServerDlg.Panel5Resize(Sender: TObject);
begin
   With butSubscribe do Width := (Sender as TPanel).ClientWidth - 2 * left
end;

procedure TGetGroupPullsForServerDlg.lstPulls1MouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
Var P: TPoint; pTab, idx: Integer; s: String;
begin
   If Sender is TListbox then begin
      With Sender as TListbox do begin
         P.x := x;
         P.y := y;
         idx := ItemAtPos (P, true);
         s := '';
         If idx >= 0 then try s := Items[idx] except end;
         pTab := Pos(^I, s);
         If pTab > 0 then s := Copy(s, 1, pTab-1)+': "'+Copy(s,pTab+1,Length(s)-pTab)+'"';
         If s <> Hint then begin
            Hint := s;
            Application.CancelHint;
            Application.ActivateHint (P)
         end
      end
   end
end;

end.
