// ============================================================================
// Popup Box for Delphi
// Copyright (c) 2001, Wolfgang Jth. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================
// Description:
//
//   function PopupBox (const AText, ACaption: string; ADlgType: TMsgDlgType;
//                      AButtons: TMsgDlgButtons; ASeconds: TStyles;
//                      ADefault: TModalResult): TModalResult;
//
// This function uses MessageDlg-like parameter types.
//
// The parameters 'AText' and 'ACaption' are containing the body message and
// the window caption. The message is not restricted to 255 chars and will
// automatically be wrapped. If the caption contains an empty string, the
// standard title (given by delphi).
//
// The parameter 'ADlgType' describes the kind of message box including the
// shown bitmap and the standard title.
//
// The parameter 'AButtons' defines the buttons contained in the message
// window. As this type represents a set, multiple buttons are possible.
//
// The parameter 'AStyle' can specify some additional style definitions like
// stTOPMOST (which is at the moment the only flag available). 
//
// The parameter 'ASeconds' contains the time (in seconds), after which
// an automatic timeout occurs. On values equal or below zero no timeout
// will be generated.
//
// The parameter 'ADefault' contains the value automatically returned on
// timeout. The default value is mrOk.
//
// If an error occured, the function will return 0. On timeout, the specified
// value will be returned, else the value of the the button pressed by the
// user.
//                *************************************
//
//    function PopupBox (const AText, ACaption: string;
//                       AStyle, ASeconds, ADefault: integer): integer;
//
// In opposite of above, this function here uses MessageBox-like parameter
// types.
//
// The parameters 'AText', 'ACaption' and ASeconds correspond to the above
// function (see there).
//
// The value of 'AStyle' is a combination of several flags defining the
// kind of message box, as well as the displayed buttons.
//
// The parameter 'ADefault' contains the value automatically returned on
// timeout. The default value is mrOk.
//
// If an error occured, the function will return 0. On timeout, the specified
// value will be returned, else the value of the the button pressed by the
// user.
//
// ============================================================================

unit dPopupBox;

interface

uses
  Windows, Controls, Forms, Dialogs, ExtCtrls;

type
   TStyle = (stTOPMOST);
   TStyles = Set of TStyle;
   TPopupBox = class
   private
      FText: string;
      FCaption: string;
      FDlgType: TMsgDlgType;
      FButtons: TMsgDlgButtons;
      FDefault: TModalResult;
      FStyle: TStyles;  // JAWO 19.01.02 (TOPMOST)
      function GetSeconds: integer;
      procedure SetSeconds (ASeconds: integer);
   public
      property Text: string read FText write FText;
      property Caption: string read FCaption write FCaption;
      property DlgType: TMsgDlgType read FDlgType write FDlgType;
      property Buttons: TMsgDlgButtons read FButtons write FButtons;
      property Seconds: integer read GetSeconds write SetSeconds;
      property Default: TModalResult read FDefault write FDefault;
      function Popup: TModalResult;
      property Style: TStyles read FStyle write FStyle;  // JAWO 19.01.02 (TOPMOST)
   private
      FMsgBox: TForm;
      FTimer: TTimer;
      TimerStop: Boolean;
      procedure DoTimeout (Sender: TObject);
   public
      constructor Create;
      destructor Destroy; override;
   end;

function PopupBox (const AText, ACaption: string;
                   AStyle, ASeconds, ADefault: integer): integer; overload;


function PopupBox (const AText, ACaption: string; ADlgType: TMsgDlgType;
                   AButtons: TMsgDlgButtons;
                   AStyle: TStyles;
                   ASeconds: integer;
                   ADefault: TModalResult = mrOk): TModalResult; overload;

implementation


function PopupBox (const AText, ACaption: string;
//================================================
                   AStyle, ASeconds, ADefault: integer): integer;

      function DlgStyle (AStyle: integer): TStyles;  // JAWO 19.01.02 (TOPMOST)
      //--------------------------------------------
      begin
         Result := [];
         if ( ((AStyle AND MB_SYSTEMMODAL) > 0) OR ((AStyle AND MB_TOPMOST) > 0) )
            then Result := Result + [stTOPMOST];
      end;

      function DlgButtons (AStyle: integer): TMsgDlgButtons;
      //-----------------------------------------------------
      begin
         case (AStyle AND $000F) of
            $01 : Result := [mbOK, mbCancel];
            $02 : Result := mbAbortRetryIgnore;
            $03 : Result := mbYesNoCancel;
            $04 : Result := [mbYes, mbNo];
            $05 : Result := [mbRetry, mbCancel];
            else  Result := [mbOK];
         end;
      end;

      function DlgType (AStyle: integer): TMsgDlgType;
      //-----------------------------------------------
      begin
         case (AStyle AND $00F0) of
            $10 : Result := mtError;
            $20 : Result := mtConfirmation;
            $30 : Result := mtWarning;
            $40 : Result := mtInformation;
            else  Result := mtCustom;
         end;
      end;

      function DlgDefButton (AStyle: integer): TModalResult;
      //-----------------------------------------------------
      begin
         case (AStyle AND $0F00) of
             $000 : Result := 0;
             else   Result := 0;
          end;
      end;

      function DlgDefault (ADefault: integer): TModalResult;
      //-----------------------------------------------------
      const
         MINVAL = $00;
         MAXVAL = $0A;
         RetVal: array [MINVAL..MAXVAL] of TModalResult = (mrNone, mrOK, mrCancel,
                  mrAbort, mrRetry, mrIgnore, mrYes, mrNo, mrAll, mrNoToAll, mrYesToAll);
      begin
         if ((ADefault<MINVAL) OR (ADefault>MAXVAL))
            then Result := RetVal[$00]
            else Result := RetVal[ADefault];
      end;

      function DlgResult (AResult: TModalResult): integer;
      //---------------------------------------------------
      const
         MINVAL = mrNone;
         MAXVAL = mrYesToAll;
         RetVal: array [MINVAL..MAXVAL] of integer = (0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10);
      begin
         if ((AResult<MINVAL) OR (AResult>MAXVAL))
            then Result := RetVal[$00]
            else Result := RetVal[AResult];
      end;

var myDlgType: TMsgDlgType;
    myButtons: TMsgDlgButtons;
    myDefault: TModalResult;
    myStyle  : TStyles;  // JAWO 19.01.02 (TOPMOST)
begin //=== PopupBox ===
   myDlgType := DlgType (AStyle);
   myButtons := DlgButtons (AStyle);
   myStyle   := DlgStyle (AStyle);  // JAWO 19.01.02 (TOPMOST)
   myDefault := DlgDefault (ADefault);
   myDefault := PopupBox (AText, ACaption, myDlgType, myButtons, myStyle, ASeconds, myDefault);
   Result := DlgResult (myDefault);
end;


function PopupBox (const AText, ACaption: string; ADlgType: TMsgDlgType;
//======================================================================
                   AButtons: TMsgDlgButtons; AStyle: TStyles; ASeconds: integer;
                   ADefault: TModalResult): TModalResult;
begin
   try
      with TPopupBox.Create do
         try
            Text := AText;
            Caption := ACaption;
            DlgType := ADlgType;
            Buttons := AButtons;
            Style := AStyle;  // JAWO 19.01.02 (TOPMOST)
            Seconds := ASeconds;
            Default := ADefault;
            Result := Popup;
         finally
            Free;
         end;
   except
      Result := ADefault;
   end;
end;

constructor TPopupBox.Create;
//============================
begin
   inherited Create;
   FText := '';
   FDlgType := mtCustom;
   FButtons := [mbOk];
   FDefault := mrOk;
   FMsgBox := nil;
   TimerStop := false;
   FTimer := TTimer.Create (nil);
   with FTimer do begin
      Enabled  := False;
      OnTimer  := DoTimeout;
      Interval := 10000;
   end;
end;

destructor TPopupBox.Destroy;
//============================
begin
   FTimer.Free;
   inherited Destroy;
end;

procedure TPopupBox.DoTimeout (Sender: TObject);
//===============================================
begin
   FTimer.Enabled := false;
   if Assigned (FMsgBox) then begin
      TimerStop := true;
      FMsgBox.Close
   end
end;

function TPopupBox.Popup: TModalResult;
//======================================
begin
   try
      try
         FMsgBox := CreateMessageDialog (FText, FDlgType, FButtons);
         With FMsgBox do try
            if (FCaption>'') then Caption := FCaption;
            Position := poScreenCenter;
            if (mbCancel in FButtons)
               then BorderIcons := BorderIcons + [biSystemMenu]
               else BorderIcons := BorderIcons - [biSystemMenu];
            if stTOPMOST IN FStyle
               then FormStyle := fsStayOnTop;  // JAWO 19.01.02 (TOPMOST)
            with FTimer do try
               if (Interval>0) then Enabled := true;
               Result := ShowModal;
               If TimerStop then Result := FDefault
            finally
               Enabled := false;  // should be set in all cases
            end;
         finally
            Free
         end;
      finally
         FMsgBox := nil; // should be set in all cases
      end;
   except
      Result := mrNone;
   end;
end;

function TPopupBox.GetSeconds: integer;
//======================================
begin
   Result := FTimer.Interval div 1000;
end;

procedure TPopupBox.SetSeconds (ASeconds: integer);
//==================================================
const MAXSECONDS = $400000;
begin
   with FTimer do
      if (ASeconds<=0)
         then Interval := 0
         else if (ASeconds>MAXSECONDS)
            then Interval := Cardinal(MAXSECONDS) * 1000
            else Interval := ASeconds * 1000;
end;



end.

