{***************************************************************
 *
 * Unit Name: tMaintenance
 * Purpose  :
 * Author   :
 * History  :
 *
 ****************************************************************}

// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit tMaintenance; // Threads for local maintenance.

// ----------------------------------------------------------------------------
// Contains threads for local maintenance. 
// ----------------------------------------------------------------------------

interface

uses Windows, Forms, SysUtils, Classes, tBase, IniFiles;

const
  HAM_PURGEOPT_DOALL        = $FF;
  HAM_PURGEOPT_DONEWS       = $1;
  HAM_PURGEOPT_DOHISTORY    = $2;
  HAM_PURGEOPT_DOKILLS      = $4;
  HAM_PURGEOPT_DOMHISTORY   = $8;
  HAM_PURGEOPT_DOMAILINLOG  = $10;
  HAM_PURGEOPT_DOMAILOUTLOG = $20;
  HAM_PURGEOPT_DONEWSOUTLOG = $40;
  HAM_PURGEOPT_DORASDIALLOG = $80;

Type
  TProcedure = Procedure;

Procedure PurgeNow;

type
  TThreadPurge = class( TTaskThread )
    protected
      ToDoBits: Integer;
      Groupname: String;
      procedure ExecutePurge;
      procedure Execute; override;
    public
      constructor Create( Const AToDoBits: Integer; Const AGroupname: String;
         AOnTerminate: TNotifyEvent );
    destructor destroy; override;
  end;

  TThreadHistoryRebuild = class( TTaskThread )
    protected
      procedure Execute; override;
    public
      constructor Create;
  end;

  TThreadRebuildGlobalLists = class( TTaskThread )
    protected
      TheList  : TList;
      procedure ListAdd( GName, GDesc: String; IsActive: Boolean );
      procedure ListTrimGroups;
      procedure ListTrimDescs;
      procedure ListSave;
      procedure Execute; override;
    public
      constructor Create;
    destructor destroy; override;
  end;

  TThreadPurgeReset = class( TThreadPurge ) // JAWO 12.08.01 (Reset Group)
    private
      bResetIni: Boolean;
    protected
      procedure ExecutePurgeReset;
      procedure Execute; override;
    public
      constructor Create( Const AGroupname: String; Const ResetIni: Boolean );
  end;

  // Ein neuer Thread, der die Statistik erstellt. Der .Execute-Teil
  // entspricht im wesentlichen dem bisherigen Stand aus Config.pas.

  TThreadStatistics = class( TTaskThread )
    private
      MakeStats: Boolean;
    protected
    public
      procedure Execute; override;
      constructor Create;
     destructor destroy; override;
  end;

implementation

uses Global, Config, cArtFiles, uTools, uDateTime, cStdForm, cLogFile;

// --------------------------------------------------------- TThreadPurge -----

Procedure PurgeNow;
begin
   AllShutdownReq := False;
   IncCounter(CanCloseNow,1);
   Log( LOGID_SYSTEM, TrGl(kLog, 'System.Purge.Start', 'Starting purge ...') );
   TThreadPurge.Create( $FFFF , '', NIL).resume;
   IncCounter(CanCloseNow,-1)
end;

{ TThreadPurge }

procedure TThreadPurge.ExecutePurge;

  Procedure PurgeLogFile(Const Bit: Integer; Const FileName, key: String; Const LineDefault: Integer);
  Var i, m: Integer;
  begin
     if (ToDoBits and Bit)<>0 then begin
        if not ShutDownReq then begin
           m := CfgIni.ReadInteger('Setup', Key, LineDefault);
           if (m > 0) and FileExists2( PATH_LOGS + FileName ) then begin
              TLog( LOGID_INFO, TrGlF(kLog, 'Info.PurgeLogfile.Start',
                 'Logfile %s is purging now...', FileName ));
              With TStringList.Create do try
                 LoadFromFile(PATH_LOGS + FileName);
                 i := Count-m;
                 If i > 0 then begin
                    While Count > m do Delete(0);
                    SaveToFile(PATH_LOGS + FileName)
                 end else begin
                   i := 0
                 end
              finally Free end;
              If i = 0 then begin
                 TLog( LOGID_INFO, TrGlF(kLog, 'Info.PurgeLogfile.result.0',
                    'Logfile %s purged: No lines deleted', [FileName] ))
              end else begin
                 TLog( LOGID_INFO, TrGlF(kLog, 'Info.PurgeLogfile.result',
                    'Logfile %s purged: first %s lines deleted', [FileName, IntToStr(i)] ))
              end
           end
        end
     end
  end;

var  LfdGroup: Integer;
     GrpHdl  : LongInt;
     StrLst  : TStringList;
     i       : Integer;
     Parser  : TParser;
     Base    : TDateTime;
     Days    : LongInt;
     PurgeCount: Integer;
     GrpName : String;
begin
     if (ToDoBits and HAM_PURGEOPT_DONEWS)<>0 then begin
        StrLst := TStringList.Create;
        for LfdGroup:=0 to CfgHamster.ActiveCount-1 do begin
           if ShutDownReq then break;
           GrpName := CfgHamster.ActiveName[LfdGroup];
           If (GroupName = '') or (LowerCase(GrpName) = LowerCase(Groupname)) then begin
              TLog( LOGID_STATUS, TrGlF(kLog, 'Info.GroupStartPurge',
                 'Group "%s" (%s/%s) is purging now...' ,
                 [GrpName, IntToStr(LfdGroup+1), IntToStr(CfgHamster.ActiveCount)] ));
              if ArticleBase.UseCount( GrpName ) = 0 then begin
                 GrpHdl := ArticleBase.Open( GrpName );
                 ArticleBase.Purge( GrpHdl );
                 ArticleBase.Close( GrpHdl );
              end else begin
                 TLog( LOGID_INFO, TrGlF(kLog, 'Info.GroupdelayedIsInUse',
                    '%s delayed: group is in use.', GrpName));
                 StrLst.Add( GrpName );
              end
           end
        end;
        for LfdGroup:=0 to StrLst.Count-1 do begin
           if ShutDownReq then break;
           GrpName := StrLst[LfdGroup];
           Sleep( 1000 );
           if ArticleBase.UseCount( GrpName ) = 0 then begin
              TLog( LOGID_STATUS, TrGlF(kLog, 'Info.SkippedGroupStartPurge',
                 'Group "%s" is purging now...', GrpName));
              GrpHdl := ArticleBase.Open( GrpName );
              ArticleBase.Purge( GrpHdl );
              ArticleBase.Close( GrpHdl );
           end else begin
              TLog( LOGID_INFO, TrGlF(kLog, 'Info.GroupskippedStillInUse',
                 '%s skipped: group still in use.', GrpName));
           end;
        end;
        StrLst.Free;
     end;

     if (ToDoBits and HAM_PURGEOPT_DOHISTORY)<>0 then begin
        if not ShutDownReq then begin
           TLog( LOGID_INFO, TrGl(kLog, 'Info.Grouphistory.Purge',
              'Newshistory is purging now...' ));
           NewsHistory.Purge
        end;
     end;

     if (ToDoBits and HAM_PURGEOPT_DOMHISTORY)<>0 then begin
        if not ShutDownReq then begin
           TLog( LOGID_INFO, TrGl(kLog, 'Info.Mailhistory.purge',
              'Mailhistory is purging now...' ));
           MailHistory.Purge
        end
     end;

     if (ToDoBits and HAM_PURGEOPT_DOKILLS)<>0 then begin
        if not ShutDownReq then begin
           PurgeCount := 0;

           if FileExists2( PATH_GROUPS + CFGFILE_SCORELOG ) then begin
              TLog( LOGID_INFO, TrGl(kLog, 'Info.PurgeKillsLog.Start',
                 'News-kills-log is purging now...' ));
              Parser := TParser.Create;
              StrLst := TStringList.Create;
              StrLst.LoadFromFile( PATH_GROUPS + CFGFILE_SCORELOG );

              for i:=StrLst.Count-1 downto 0 do begin
                 Parser.Parse( StrLst[i], #9 );
                 // 0:Server 1:Group 2:Score 3-:Overview
                 // 3:No. 4:Subject 5:From 6:Date 7:Message-ID 8:References 9:Bytes 10:Lines [11:Xref]
                 Base := RfcDateTimeToDateTimeGMT( Parser.sPart(6,'') );
                 Days := Trunc( Now - Base );
                 if (Days>Def_Purge_Kills_KeepDays) and (Def_Purge_Kills_KeepDays>0) then begin
                    StrLst.Delete( i );
                    inc( PurgeCount );
                 end;
              end;

              if PurgeCount>0 then StrLst.SaveToFile( PATH_GROUPS + CFGFILE_SCORELOG );
              StrLst.Free;
              Parser.Free;
           end;

           Log( LOGID_INFO, TrGlF(kLog, 'Info.PurgeKillsLog.result',
             'Purge Kills.log (%sd): %s entries purged.',
             [inttostr(Def_Purge_Kills_KeepDays), inttostr(PurgeCount)]))
        end;
     end;

     PurgeLogFile(HAM_PURGEOPT_DOMAILINLOG,  'MailIn.log', 'purge.mailinlog.keeplines',  0);
     PurgeLogFile(HAM_PURGEOPT_DOMAILOUTLOG, 'MailOut.log','purge.mailoutlog.keeplines', 0);
     PurgeLogFile(HAM_PURGEOPT_DONEWSOUTLOG, 'NewsOut.log','purge.newsoutlog.keeplines', 0);
     PurgeLogFile(HAM_PURGEOPT_DORASDIALLOG, 'RasDial.log','purge.rasdiallog.keeplines', 0);
end;

procedure TThreadPurge.Execute;
begin
     TLog( LOGID_SYSTEM, TrGl(kLog, 'Thread.Start', 'Start' ) );
     InterlockedIncrement( CriticalState );
     try
       {JW} {critical event}
        EnterCriticalSection(CS_Event);
        SetEvent(EventPurge); {JW} {purge event}
        LeaveCriticalSection(CS_Event);
        {JW}
        ExecutePurge;
     finally
        InterlockedDecrement( CriticalState );
        {JW} {critical event}
        EnterCriticalSection(CS_Event);
        ResetEvent(EventPurge); {JW} {purge event}
        LeaveCriticalSection(CS_Event);
        {JW}
     end;
     TLog( LOGID_SYSTEM, TrGl(kLog, 'Thread.End', 'End' ) );
end;

constructor TThreadPurge.Create( Const AToDoBits: Integer;
      Const AGroupname: String; AOnTerminate: TNotifyEvent );
Var Info: String;
begin
   If AGroupname = ''
      then Info := TrGl(kLog, 'Thread.PurgeAllGroups', '{purge all groups}')
      else Info := TrGlF(kLog, 'Thread.PurgeSingleGroup', '{purge %s}', AGroupname);
   inherited Create( Info );
   ToDoBits := AToDoBits;
   GroupName := AGroupName;
   OnTerminate := AOnTerminate;
end;

destructor TThreadPurge.destroy;
begin
   If Groupname = ''
      then MainLog (LOGID_SYSTEM, TrGl(kLog, 'PurgeAllGroups.end', 'All groups purged.') )
      else MainLog (LOGID_SYSTEM, TrGlF(kLog, 'PurgeGroup.end', 'Group %s purged.', Groupname));
   inherited;
end;

// ------------------------------------------------ TThreadHistoryRebuild -----

procedure TThreadHistoryRebuild.Execute;
begin
     TLog( LOGID_SYSTEM, TrGl(kLog, 'Thread.Start', 'Start' ) );
     InterlockedIncrement( CriticalState );
     try
        NewsHistory.Rebuild;
     finally
        InterlockedDecrement( CriticalState );
     end;
     TLog( LOGID_SYSTEM, TrGl(kLog, 'Thread.End', 'End' ) );
end;

constructor TThreadHistoryRebuild.Create;
begin
   inherited Create( '{rebuild history}' );
end;

// ----------------------------------------------- TThreadRebuildPullList -----

type
   TTPullListEntry = class
      GroupName : String;
      GroupDesc : String;
      IsActive  : Boolean;
   end;

function SortPLE( Item1, Item2: Pointer ): Integer;
var  A: TTPullListEntry absolute Item1;
     B: TTPullListEntry absolute Item2;
begin
     Result := CompareText( A.GroupName, B.GroupName );
end;

procedure TThreadRebuildGlobalLists.ListAdd( GName, GDesc: String; IsActive: Boolean );
var  PLE: TTPullListEntry;
begin
     PLE := TTPullListEntry.Create;
     PLE.GroupName := GName;
     PLE.GroupDesc := GDesc;
     PLE.IsActive  := IsActive;
     TheList.Add( PLE );
end;

procedure TThreadRebuildGlobalLists.ListTrimGroups;
var  A, B : TTPullListEntry;
     Curr : Integer;
begin
     // sort all entries by GroupName
     TheList.Sort( SortPLE );

     // remove entries with same GroupName
     Curr := TheList.Count-1;
     while Curr>0 do begin
        A := TheList[ Curr-1 ];
        B := TheList[ Curr   ];
        if CompareText(A.GroupName,B.GroupName)=0 then begin
           B.Free;
           TheList.Delete( Curr );
        end;
        dec( Curr );
     end;
end;

procedure TThreadRebuildGlobalLists.ListTrimDescs;
var  A, B : TTPullListEntry;
     Curr : Integer;
begin
     // sort all entries by GroupName
     TheList.Sort( SortPLE );

     // remove entries with same GroupName, but conserve descriptions
     Curr := TheList.Count-1;
     while Curr>0 do begin
        A := TheList[ Curr-1 ];
        B := TheList[ Curr   ];
        if CompareText(A.GroupName,B.GroupName)=0 then begin
           if length(A.GroupDesc)<length(B.GroupDesc) then A.GroupDesc:=B.GroupDesc;
           if B.IsActive then A.IsActive:=True;
           B.Free;
           TheList.Delete( Curr );
        end;
        dec( Curr );
     end;
end;

procedure TThreadRebuildGlobalLists.ListSave;
var  i, cnt: Integer;
     s, g: String;
begin
   cnt := 0;
   try
      With TFileStream.Create(PATH_SERVER + SRVFILE_ALLDESCS, fmCreate ) do try
         g := '';
         for i:=0 to TheList.Count-1 do with TTPullListEntry( TheList[i] ) do begin
            if IsActive then begin
               if GroupName<>g then begin
                  inc( cnt );
                  if GroupDesc='' then s := GroupName + #9 + '?'
                                  else s := GroupName + #9 + GroupDesc;
                  Write(s[1], Length(s));
                  g := GroupName;
               end
            end
         end
      finally
         free
      end
   except
      on E:Exception do TLog( LOGID_ERROR, TrGl(kLog, 'Error', 'ERROR')+': ' + E.Message );
   end;
   TLog( LOGID_INFO, TrGlF(kLog, 'Info.RebuildGlLists.XEntries', '%s entries.', inttostr(cnt)) );
end;

procedure TThreadRebuildGlobalLists.Execute;
var  SrvCurr, GrpCurr: Integer;
     SrvName, SrvPath, GrpName, GrpDesc: String;
     TS: TStringList;
     j: Integer;
begin
     TLog( LOGID_SYSTEM, TrGl(kLog, 'System.Start', 'Start' ) );

     EnterCriticalSection( CS_MAINTENANCE );
     AllShutDownReq := False; ///
     if not GlobalListMarker( glTEST ) then begin
        LeaveCriticalSection( CS_MAINTENANCE );
        TLog( LOGID_INFO, TrGl(kLog, 'Info.RebuildGlLists.AlreadyUpToDate', 'Lists are already up to date.') );
        TLog( LOGID_SYSTEM, TrGl(kLog, 'System.End', 'End' ) );
        exit;
     end;

     TheList   := TList.Create;
     TS        := TStringList.Create;

     try
        for SrvCurr:=0 to CfgHamster.ServerCount-1 do begin
           if ShutDownReq then break;
           SrvName := CfgHamster.ServerName[ SrvCurr ];
           SrvPath := CfgHamster.ServerPath[ SrvCurr ];

           if FileExists2( SrvPath + SRVFILE_GROUPS ) then begin
              TS.LoadFromFile( SrvPath + SRVFILE_GROUPS );
              TLog( LOGID_INFO, SrvName + ' ' + SRVFILE_GROUPS
                              + ' (' + inttostr(TS.Count) + ') ...' );
              for GrpCurr:=0 to TS.Count-1 do begin
                 GrpName := TS[ GrpCurr ];
                 j := PosWhSpace( GrpName );
                 if j>0 then GrpName := copy( GrpName, 1, j-1 );
                 if GrpName<>'' then ListAdd( GrpName, '', True );
              end;
              TS.Clear;
              ListTrimGroups;
           end;
        end;

        for SrvCurr:=0 to CfgHamster.ServerCount-1 do begin
           if ShutDownReq then break;
           SrvName := CfgHamster.ServerName[ SrvCurr ];
           SrvPath := CfgHamster.ServerPath[ SrvCurr ];

           if FileExists2( SrvPath + SRVFILE_GRPDESCS ) then begin
              TS.LoadFromFile( SrvPath + SRVFILE_GRPDESCS );
              if TS.Count<=3 then begin
                 if copy(TS[0],1,1)='#' then TS.Clear; // error-marker
              end;
              TLog( LOGID_INFO, SrvName + ' ' + SRVFILE_GRPDESCS
                              + ' (' + inttostr(TS.Count) + ') ...' );
              for GrpCurr:=0 to TS.Count-1 do begin
                 GrpName := TS[ GrpCurr ];
                 GrpDesc := '';
                 j := PosWhSpace( GrpName );
                 if j>0 then begin
                    GrpDesc := TrimWhSpace( copy( GrpName, j+1, Length(GrpName)-j ) );
                    GrpName := copy( GrpName, 1, j-1 );
                 end;
                 if GrpName<>'' then ListAdd( GrpName, GrpDesc, False );
              end;
              TS.Clear;
              ListTrimDescs;
           end;
        end;

        if not ShutDownReq then ListSave;
        if not ShutDownReq then GlobalListMarker( glDONE );

        for GrpCurr:=TheList.Count-1 downto 0 do begin
           TTPullListEntry( TheList[GrpCurr] ).Free;
           TheList.Delete( GrpCurr );
        end;

     finally
        LeaveCriticalSection( CS_MAINTENANCE );
        TS.Free;
        TheList.Free;
     end;

     TLog( LOGID_SYSTEM, TrGl(kLog, 'System.End', 'End' ) );
end;

constructor TThreadRebuildGlobalLists.Create;
begin
   inherited Create( '{rebuild global lists}' );
end;

destructor TThreadRebuildGlobalLists.destroy;
begin
   MainLog (LOGID_SYSTEM, TrGl(kLog, 'RebuildGlobalLists.end', 'Rebuild global lists ready.') );
   inherited destroy
end;

// --------------------------------------------------------- TThreadPurgeReset -----
                                                         // JAWO 12.08.01 (Reset Group)

constructor TThreadPurgeReset.Create( Const AGroupname: String; Const ResetIni: Boolean );
begin
   bResetIni := ResetIni;
   inherited Create( HAM_PURGEOPT_DONEWS, AGroupName, NIL );
end;

procedure TThreadPurgeReset.Execute;
begin
   TLog( LOGID_SYSTEM, TrGl(kLog, 'TThreadPurgeReset.Start', 'Start' ) );
   InterlockedIncrement( CriticalState );
   try
      {JW} {critical event}
      EnterCriticalSection(CS_Event);
      SetEvent(EventPurge); {JW} {purge event}
      LeaveCriticalSection(CS_Event);
      {JW}
      ExecutePurgeReset;
   finally
      InterlockedDecrement( CriticalState );
      {JW} {critical event}
      EnterCriticalSection(CS_Event);
      ResetEvent(EventPurge); {JW} {purge event}
      LeaveCriticalSection(CS_Event);
      {JW}
   end;
   TLog( LOGID_SYSTEM, TrGl(kLog, 'TThreadPurgeReset.End', 'End' ) );
end;

procedure TThreadPurgeReset.ExecutePurgeReset;
var  LfdGroup: Integer;
     GrpHdl  : LongInt;
     GrpName : String;
     LocalMax: Integer;
begin
  for LfdGroup:=0 to CfgHamster.ActiveCount-1 do begin
     if ShutDownReq then break;
     GrpName := CfgHamster.ActiveName[LfdGroup];               // reset only, if exactly
     If (LowerCase(GrpName) = LowerCase(Groupname)) then begin // one group is specified
        if ArticleBase.UseCount( GrpName ) = 0 then begin
           GrpHdl := ArticleBase.Open( GrpName );
           ArticleBase.PurgeReset( GrpHdl );
           ArticleBase.Close( GrpHdl );
           NewsHistory.PurgeReset( GrpHdl );
           If bResetIni then begin
              With TIniFile.Create(PATH_GROUPS + GroupName + '\data' + EXT_CFG) do try
                 try LocalMax := ReadInteger('Ranges', 'Local.Max', 0) except LocalMax := 0 end;
                 EraseSection ( 'Ranges' );
                 If LocalMax > 0 then begin
                    WriteInteger('Ranges', 'Local.Min', LocalMax+1);
                    WriteInteger('Ranges', 'Local.Max', LocalMax);
                 end;
              finally
                 free
              end
           end;
           break;
        end else begin
           TLog( LOGID_INFO, Format(TrGl(kLog, 'Info.GroupskippedStillInUse',
              '%s skipped: group still in use.' ), [GrpName] ));
        end
     end
  end;
end;

{ TThreadStatistics }

constructor TThreadStatistics.Create;
begin
   inherited Create( '{create statistics}' );
end;

destructor TThreadStatistics.destroy;
begin
   If MakeStats
      then MainLog (LOGID_SYSTEM, TrGl(kLog, 'CreateStatistics.end', 'Statistics created.') )
      else MainLog (LOGID_SYSTEM, TrGl(kLog, 'CreateStatistics.skipped', 'Statistics skipped, already done.') );
   inherited
end;

procedure TThreadStatistics.Execute;
var  lGroupsInfo, lOutDated: TStringList;
     i, GrpHdl, LfdArt, Anz, j, p, v, SumArt: Integer;
     StatsNow, D: TDateTime;
     s, StatInfo: String;
begin
   // Stats already created today?
   EnterCriticalSection( CS_MAINTENANCE );
   try
      StatsNow := Now;
      MakeStats := CfgIni.ReadString( 'Stats', 'LastInfoMsg', '' )
                     <> FormatDateTime( 'dd"."mm"."yyyy', StatsNow );
      if MakeStats then
         CfgIni.WriteString( 'Stats', 'LastInfoMsg',
                             FormatDateTime( 'dd"."mm"."yyyy', StatsNow ) );
   finally
      LeaveCriticalSection( CS_MAINTENANCE );
   end;
   if not MakeStats then exit; // Nothing to do

   // Create stats
   TLog( LOGID_SYSTEM, TrGl(kLog, 'Thread.Start', 'Start' ) );
   InterlockedIncrement( CriticalState );

   try
      lGroupsInfo := TStringlist.Create;
      lOutDated   := TStringlist.Create;
      SumArt := 0;

      try
         for i:=0 to CfgHamster.ActiveCount-1 do begin
            GrpHdl := ArticleBase.Open( CfgHamster.ActiveName[i] );
            if GrpHdl>=0 then begin
               LfdArt := ArticleBase.Count[GrpHdl];
               // Group not used?
               D := Articlebase.LastClientRead [GrpHdl];
               If (StatsNow - D) >= 7 then begin
                  If D = 0 then begin
                     s := TrGl('Statistic', 'Groups.NotPulled.Never', 'Never');
                     Anz := 0
                  end else begin
                     s := FormatDateTime(ShortDateFormat, D);
                     Anz := Trunc(StatsNow-D)
                  end;
                  lOutDated.AddObject (s + ': ' + CfgHamster.ActiveName[i], Pointer(Anz))
               end;
               ArticleBase.Close( GrpHdl );
               inc( SumArt, LfdArt );
               s := CfgHamster.ActiveName[i]; Anz := 0;
               For j := 0 to CfgHamster.PullCount-1 do begin
                  If lowercase(CfgHamster.PullGroup[j])=lowercase(CfgHamster.ActiveName[i]) then begin
                     Inc(Anz); If Anz = 1 then s := s + ' (' else s := s + ', ';
                     s := s + CfgHamster.PullServer[j]
                  end;
               end;
               If Anz = 0
                  then s := s + ' ('+TrGl('Statistic', 'IsLocalGroup', 'local Group')+')'
                  else s := s + ')';
               lGroupsInfo.Add (Format( '%6d  %s', [LfdArt,s] ));
            end else begin
               Log( LOGID_ERROR, 'Could not open ' + CfgHamster.ActiveName[i] + '!' );
            end
         end;

         StatInfo := TrGl('Statistic', 'Groups.alphabetical', 'Groups in alphabetical order')+':'
                     +#13#10#13#10 + lGroupsInfo.text
                     +#13#10#13#10
                     +TrGl('Statistic', 'Groups.byArticleNumber', 'Groups sorted by number of articles')+':'
                     +#13#10#13#10;
         With lGroupsInfo do begin
            s := Text;
            For i := 1 to 2000 do begin
               p := -1; Anz := 0;
               For j := 0 to Count-1 do begin
                  v := StrToInt(Copy(Strings[j],1,6));
                  If v > Anz then begin p := j; Anz := v end
               end;
               If p >= 0 then begin
                  StatInfo := StatInfo + Format('%3d. %s', [i, Strings[p]])+#13#10;
                  Delete(p)
               end else break
            end;
            Text := s
         end;

         s := '';
         With lOutDated do begin
            For i := 0 to Count-1 do begin
               p := -1; Anz := 0;
               For j := 0 to Count-1 do begin
                  v := LongInt(Objects[j]);
                  If v = 0 then begin
                     p := j; break
                  end else
                  If v > Anz then begin
                     p := j; Anz := v
                  end
               end;
               If p >= 0 then begin
                  s := s + Strings[p] + #13#10;
                  Delete (p)
               end else break
            end
         end;
         If s > '' then
            StatInfo := StatInfo + #13#10#13#10
                       + TrGlF('Statistic', 'Groups.NotPulled',
                        'Groups not pulled from any client longer than %s days', ['7'])+':'
                       + #13#10#13#10
                       + s;

         StatInfo := StatInfo + #13#10#13#10 + Format( '%6d  %s', [SumArt,
             TrGl(kLog, 'Info.Articles.AllArticles', 'All articles')] ) + #13#10;

         If Assigned(NewsHistory) then begin
            StatInfo := StatInfo + #13#10 + Format( '%6d  %s', [NewsHistory.Count,
               TrGl(kLog, 'Info.History.Entries', 'Entries in History')] ) + #13#10;
         end;

         If Assigned(MailHistory) then begin
            StatInfo := StatInfo + #13#10 + Format( '%6d  %s', [MailHistory.Count,
               TrGl(kLog, 'Info.MailHistory.Entries', 'Entries in Mailhistory')] ) + #13#10;
         end;

         // save stats in internal group
         SaveInInternalGroup( INTERNALGROUP_STATISTICS, '[Hamster] Info', StatInfo );

      finally
         lGroupsInfo.free; lOutDated.free
      end

   finally
      InterlockedDecrement( CriticalState );
   end;

   TLog( LOGID_SYSTEM, TrGl(kLog, 'Thread.End', 'End' ) );
end;

Type
   TScoreLogEntry = Record
      Server, Group: String;
      ScoreValue, RefNr: Integer;
      Subject, From, Datum, MID, References: String;
      Byte, Lines: Integer;
      XRef: String;
   end;

Function  StrToScoreLogEntry(s: String): TScoreLogEntry;
Var p, Nr: Integer; akt: String;
begin
   FillChar (Result, SizeOf(Result), #0);
   Nr := 0;
   While s > '' do begin
      p := Pos(#9, s);
      If p = 0 then begin
         akt := s; s := ''
      end else begin
         akt := Copy(s, 1, p-1); Delete(s, 1, p)
      end;
      Inc(Nr);
      With Result do try
         Case Nr of
            1: Server := Akt;
            2: Group := Akt;
            3: ScoreValue := StrToInt(Akt);
            4: RefNr := StrToInt(Akt);
            5: Subject := Akt;
            6: From := Akt;
            7: Datum := AKt;
            8: MID := AKt;
            9: References := Akt;
            10: Byte := StrToInt(Akt);
            11: Lines := StrToInt(Akt);
            12: XRef := Akt;
         end
      except end
   end
end;

Procedure Test;
Var Log: TStringList; Entry: TScoreLogEntry;
    Gruppe, Artikel: String;
    i, offs: Integer;
begin
   If Not FileExists2(PATH_GROUPS + CFGFILE_SCORELOG_INFO) then Exit;
   Log := TStringList.Create;
   try
      Log.LoadFromFile ( PATH_GROUPS + CFGFILE_SCORELOG_INFO );
      While Log.Count > 0 do begin
         Entry := StrToScoreLogEntry(Log[0]);
         Gruppe := Entry.Group;
         Artikel := 'Newsgroups: '+Gruppe
                   +'Subject: Killed Articles on '+FormatDateTime(ShortDateFormat, Now)
                   +#13#10;
         offs := 0;
         For i := 0 to Log.Count-1 do begin
            Entry := StrToScoreLogEntry(Log[i-offs]);
            If Entry.Group = Gruppe then With Entry do begin
               Artikel := Artikel + #13#10
                   + 'Subject: '+Subject+ #13#10
                   + 'From: '+From + #13#10
                   + 'Score: '+IntToStr(ScoreValue)+ ', Message-ID: '+MID + #13#10
                   + 'References: '+References + #13#10
                   + 'Date: '+Datum + ', Bytes: '+IntToStr(Byte)+', Lines: '+IntToStr(Lines)+', Server: '+Server + #13#10;
               Log.Delete(i-offs); Inc(Offs)
            end
         end;
         Application.MessageBox(
            PChar(Artikel), NIL, 0)
      end
   finally
      Log.free
   end
end;

initialization
   {Test}
finalization
end.
