// ============================================================================
// Dynamic invokation of DLLs.
// Copyright (c) 2001, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uDynDll; // Dynamic invokation of DLLs.

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Windows, Classes;

const
   ERR_DYNDLL_INVALIDDECLARATION = 40000;
   ERR_DYNDLL_UNSUPPORTEDDLLTYPE = 40001;
   ERR_DYNDLL_EXECUTEFAILED      = 40002;

function DynDllLoad( const DllName: String;
                     var ErrCode: DWORD ): HMODULE;
function DynDllFree( const hdl: HMODULE ): DWORD;
function DynDllCall( const DllDeclaration : String;
                     dwPar: array of DWORD;
                     var ErrCode: DWORD ): DWORD;

implementation

uses cSyncObjects, cLogFile;

type
   // prototypes of supported function types:
   TDllProc_vd_vd   = procedure;                                       stdcall;
   TDllProc_dw_vd   = function                                : DWORD; stdcall;
   TDllProc_dw_1dw  = function( a                    : DWORD ): DWORD; stdcall;
   TDllProc_dw_2dw  = function( a,b                  : DWORD ): DWORD; stdcall;
   TDllProc_dw_3dw  = function( a,b,c                : DWORD ): DWORD; stdcall;
   TDllProc_dw_4dw  = function( a,b,c,d              : DWORD ): DWORD; stdcall;
   TDllProc_dw_5dw  = function( a,b,c,d,e            : DWORD ): DWORD; stdcall;
   TDllProc_dw_6dw  = function( a,b,c,d,e,f          : DWORD ): DWORD; stdcall;
   TDllProc_dw_7dw  = function( a,b,c,d,e,f,g        : DWORD ): DWORD; stdcall;
   TDllProc_dw_8dw  = function( a,b,c,d,e,f,g,h      : DWORD ): DWORD; stdcall;
   TDllProc_dw_9dw  = function( a,b,c,d,e,f,g,h,i    : DWORD ): DWORD; stdcall;
   TDllProc_dw_10dw = function( a,b,c,d,e,f,g,h,i,j  : DWORD ): DWORD; stdcall;
   TDllProc_dw_11dw = function( a,b,c,d,e,f,g,h,i,j,k: DWORD ): DWORD; stdcall;

type
   TDynDllList = class( TMultiReadExclusiveWriteStringList )
      protected
         function KeyOf(const S: string): String;
      public
         function LoadDll( const DllName: String; var ErrCode: DWORD ): HMODULE;
         function FreeDll( const hdl: HMODULE ): DWORD;
         constructor Create;
         destructor Destroy; override;
   end;

var
   DynDllList: TDynDllList;

threadvar
   hmemCallbackResult: HGLOBAL;

procedure AddCallbackResult( Par: DWORD );
// Stores callback-result in (thread's) callback-result-buffer. If the buffer
// is too small to store the new value, it is reallocated.
var  OldSize, NewSize: DWORD;
begin
   if hmemCallbackResult=0 then exit;

   // allocated size
   OldSize := GlobalSize( hmemCallbackResult );
   if OldSize=0 then exit;

   // size needed to store new value
   NewSize := ( PDWORD(hmemCallbackResult)^{old} + 1{new} + 1{cnt} ) * sizeof(DWORD);
   if NewSize > OldSize then begin // grow
      NewSize := ( NewSize + 256 );
      hmemCallbackResult := GlobalRealloc( hmemCallbackResult, NewSize, GMEM_MOVEABLE );
      if hmemCallbackResult=0 then exit;
   end;

   // increment counter, store value
   inc( PDWORD( hmemCallbackResult )^ );
   PDWORD( hmemCallbackResult + PDWORD(hmemCallbackResult)^ * sizeof(DWORD) )^ := Par;
end;

function DllCallback_Type1( Par1: DWORD ): BOOL; stdcall;
// callback of type 1: receives/stores one DWORD, always returns TRUE
begin
   AddCallbackResult( Par1 );
   Result := True;
end;

function DllCallback_Type2( Par1, Par2: DWORD ): BOOL; stdcall;
// callback of type 2: receives two DWORDs, stores first, always returns TRUE
begin
   AddCallbackResult( Par1 );
   Result := True;
end;

function DynDllSplitDeclaration( const DllDeclaration: String;
                                 var DllName: String;
                                 var DllProc: String;
                                 var DllResDef: Char;
                                 var DllProcDef: String): Boolean;
// Splits a DLL-declaration (''dll-name|proc-name|result-type|param-types')
// into its different parts. Returns False, if something is missing or is
// obviously wrong.
var  s: String;
     i: Integer;
begin
   Result := False;

   s := DllDeclaration; // 'dll-name|proc-name|result-type|params-type'

   i := Pos( '|', s );
   if i<=1 then exit;
   DllName := copy( s, 1, i-1 );
   if DLLName = '' then exit;
   System.Delete( s, 1, i );

   i := Pos( '|', s );
   if i<=1 then exit;
   DllProc := copy( s, 1, i-1 );
   if DllProc = '' then exit;
   System.Delete( s, 1, i );

   i := Pos( '|', s );
   if i<>2 then exit;
   DllResDef  := s[1];
   if not( DllResDef in ['n','v'] ) then exit;   

   DllProcDef := copy( s, i+1, Length(s)-i );
   if DllProcDef='' then exit;
   for i:=1 to length(DllProcDef) do begin
      if not( DllProcDef[i] in ['n','1'..'2'] ) then begin
         if DllProcDef[i]<>'v' then exit;
         if length(DllProcDef)>1 then exit;
      end;
   end;

   Result := True;
end;

function DynDllCallExecute( const DllName, DllProcName, DllResDef, DllProcDef: String;
                            dwPar: array of DWORD;
                            var ErrCode: DWORD ): DWORD;
// Invokes the DLL-function, either by using a pre-loaded DLL or by loading
// the DLL temporarily.
var  hDll    : HMODULE;
     pProc   : Pointer;
     DllType : String;
     DllIndex: Integer;
begin
   Result := 0;
   
   with DynDllList.BeginRead do try

      // Is DLL in (thread's) list of pre-loaded DLLs?
      DllIndex := IndexOf( DynDllList.KeyOf(DllName) );

      if DllIndex >= 0 then begin
         // Yes, use pre-loaded DLL
         Log( LOGID_DEBUG, 'DynDll: Use loaded library ' + DynDllList.KeyOf( DllName ) );
         hDll := HMODULE( Objects[DllIndex] );
      end else begin
         // No, load DLL temporarily
         Log( LOGID_DEBUG, 'DynDll: Load temp. library ' + DllName );
         hDll := LoadLibrary( PChar( DllName ) );
         if hDll=0 then begin ErrCode:=GetLastError; exit; end;
      end;

   finally
      DynDllList.EndRead;
   end;

   try
      // get address of function
      pProc := GetProcAddress( hDll, PChar( DllProcName ) );
      if not Assigned(pProc) then begin ErrCode:=GetLastError; exit; end;

      // invoke function
      DllType := DllResDef + '|' + DllProcDef;
      try
         if          DllType='v|v' then begin
            Result := 0;
            TDllProc_vd_vd( pProc );
         end else if DllType='n|v' then begin
            Result := TDllProc_dw_vd( pProc );
         end else if DllType='n|n' then begin
            Result := TDllProc_dw_1dw( pProc )
                         ( dwPar[0] );
         end else if DllType='n|nn' then begin
            Result := TDllProc_dw_2dw( pProc )
                         ( dwPar[0], dwPar[1] );
         end else if DllType='n|nnn' then begin
            Result := TDllProc_dw_3dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2] );
         end else if DllType='n|nnnn' then begin
            Result := TDllProc_dw_4dw( pProc )
                        ( dwPar[0], dwPar[1], dwPar[2], dwPar[3] );
         end else if DllType='n|nnnnn' then begin
            Result := TDllProc_dw_5dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2], dwPar[3], dwPar[4] );
         end else if DllType='n|nnnnnn' then begin
            Result := TDllProc_dw_6dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2], dwPar[3], dwPar[4],
                           dwPar[5] );
         end else if DllType='n|nnnnnnn' then begin
            Result := TDllProc_dw_7dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2], dwPar[3], dwPar[4],
                           dwPar[5], dwPar[6] );
         end else if DllType='n|nnnnnnnn' then begin
            Result := TDllProc_dw_8dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2], dwPar[3], dwPar[4],
                           dwPar[5], dwPar[6], dwPar[7] );
         end else if DllType='n|nnnnnnnnn' then begin
            Result := TDllProc_dw_9dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2], dwPar[3], dwPar[4],
                           dwPar[5], dwPar[6], dwPar[7], dwPar[8] );
         end else if DllType='n|nnnnnnnnnn' then begin
            Result := TDllProc_dw_10dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2], dwPar[3], dwPar[4],
                           dwPar[5], dwPar[6], dwPar[7], dwPar[8], dwPar[9] );
         end else if DllType='n|nnnnnnnnnnn' then begin
            Result := TDllProc_dw_11dw( pProc )
                         ( dwPar[0], dwPar[1], dwPar[2], dwPar[3], dwPar[4],
                           dwPar[5], dwPar[6], dwPar[7], dwPar[8], dwPar[9],
                           dwPar[10] );
         end else begin
            SetLastError( ERR_DYNDLL_UNSUPPORTEDDLLTYPE );
         end;

      except
         on E: Exception do begin
            Log( LOGID_ERROR, 'DynDll-Exception: ' + E.Message );
            SetLastError( ERR_DYNDLL_EXECUTEFAILED );
         end;
      end;

      ErrCode := GetLastError;

   finally
      if DllIndex < 0 then begin
         // free temporarily loaded DLL
         Log( LOGID_DEBUG, 'DynDll: Free temp. library ' + DllName );
         FreeLibrary( hDll );
      end;
   end;
end;

function DynDllCall( const DllDeclaration : String;
                     dwPar: array of DWORD;
                     var ErrCode: DWORD ): DWORD;
// checks parameters and invokes the given DLL-function
var  DllName, DllProc, DllProcDef: String;
     DllResDef: Char;
     i: Integer;
     pCallbackResultPtr: Pointer;
begin
   Result  := 0;
   ErrCode := 0;

   // split and check given DLL-declaration
   if not DynDllSplitDeclaration(
             DllDeclaration, DllName, DllProc, DllResDef, DllProcDef
          ) then begin
      ErrCode := ERR_DYNDLL_INVALIDDECLARATION;
      exit;
   end;

   // check/prepare callback-params
   pCallbackResultPtr := nil;
   for i:=1 to length(DllProcDef) do begin
      case DllProcDef[i] of
         '1': begin // callback-function of type 1
                 pCallbackResultPtr := Pointer( dwPar[i-1] );
                 PDWORD( pCallbackResultPtr )^ := 0;
                 dwPar[i-1] := DWORD( @DllCallback_Type1 );
                 DllProcDef[i] := 'n';
              end;
         '2': begin // callback-function of type 2
                 pCallbackResultPtr := Pointer( dwPar[i-1] );
                 PDWORD( pCallbackResultPtr )^ := 0;
                 dwPar[i-1] := DWORD( @DllCallback_Type2 );
                 DllProcDef[i] := 'n';
              end;
      end;
   end;

   if Assigned( pCallbackResultPtr ) then begin
      // allocate initial callback-result-buffer with counter set to 0
      hmemCallbackResult := GlobalAlloc( GMEM_FIXED or GMEM_ZEROINIT, 256 );
   end;

   // invoke DLL-function
   Result := DynDllCallExecute(
                DllName, DllProc, DllResDef, DllProcDef, dwPar, ErrCode
             );

   if Assigned( pCallbackResultPtr ) then begin
      // return pointer to final callback-result-buffer
      PDWORD( pCallbackResultPtr )^ := hmemCallbackResult;
   end;
end;

function DynDllLoad( const DllName: String; var ErrCode: DWORD ): HMODULE;
// load library and add it to internal list of loaded libraries
begin
   Result := DynDllList.LoadDll( DllName, ErrCode );
end;

function DynDllFree( const hdl: HMODULE ): DWORD;
// free library and remove it from internal list of loaded libraries
begin
   Result := DynDllList.FreeDll( hdl );
end;


{ TDynDllList }

function TDynDllList.KeyOf(const S: string): String;
// returns DLL-name qualified with thread's ID, which is used for .Strings[]
begin
   Result := S + ' (Thread-ID=' + inttostr( GetCurrentThreadID ) + ')';
end;

function TDynDllList.LoadDll( const DllName: String; var ErrCode: DWORD ): HMODULE;
// loads library and adds it to the list
begin
   Result := SafeLoadLibrary( DllName, SEM_FAILCRITICALERRORS );
   if Result=0 then begin
      ErrCode := GetLastError;
   end else begin
      Log( LOGID_DEBUG, 'DynDll: Load library ' + KeyOf(DllName) );
      with BeginWrite do try
         AddObject( KeyOf(DllName), Pointer(Result) );
      finally
         EndWrite;
      end;
   end;
end;

function TDynDllList.FreeDll( const hdl: HMODULE ): DWORD;
// frees library and removes it from the list
var  i: Integer;
begin
   with BeginWrite do try
      Result := DWORD( FreeLibrary(hdl) );
      for i:=0 to Count-1 do begin
         if HMODULE( Objects[i] ) = hdl then begin
            Log( LOGID_DEBUG, 'DynDll: Free library ' + Strings[i] );
            Delete( i );
            break;
         end;
      end;
   finally
      EndWrite;
   end;
end;

constructor TDynDllList.Create;
begin
   inherited Create( False, dupAccept );
end;

destructor TDynDllList.Destroy;
begin
   // free (orphaned) entries
   with BeginWrite do try
      while Count>0 do begin
         FreeLibrary( HMODULE( Objects[0] ) );
         Delete( 0 );
      end;
   finally
      EndWrite;
   end;

   inherited Destroy;
end;

initialization
   DynDllList := TDynDllList.Create;

finalization
   if Assigned( DynDllList ) then FreeAndNil( DynDllList );

end.

