// ============================================================================
// Various tools and functions
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uTools; // Various tools and functions

// ----------------------------------------------------------------------------
// Contains a really horrible, historically grown mix of various tools and
// functions ...
// ----------------------------------------------------------------------------

interface

{$R-}
{$Q-}

Uses Classes, Controls, StdCtrls, Windows, 
     ComCtrls, Commctrl; // Fr TreeNodeBold!

type
  TParser = class( TStringList )
  public
    ParsedLine: String;
    function sPart( PartNo: Integer; DefVal: String ): String;
    function iPart( PartNo: Integer; DefVal: Integer ): Integer;
    procedure Parse( ALine: String; ASplit: Char );
    procedure ParseQuoted( ALine: String; ASplit: Char );
  end;

  TTextReader = class
  private
    F: Textfile;
    FIntBuffer: String;
    FLineBuffer: String;
  public
    constructor Create (Const FileName: String; Const Buffersize: Integer = 128);
    destructor Destroy; override;
    Function ReadLine: String;
    Function EOF: Boolean;
  end;

  TFileListOption = (flFiles, flDirs, flSorted);
  TFileListOptions = Set of TFileListOption;
  TFileList = class
  private
    fList: TStringlist;
    fPath: String;
    Function GetFile(Const x: Integer): String;
    Function GetFileName(const x: Integer): String;
  public
    constructor Create (Const Path, FileMask: String; Const Attr: Integer;
       Const Options: TFileListOptions); 
    destructor Destroy; override;
    Function Count: Integer;
    Property Files[Const x: Integer]: String read GetFile; default;
    Property Filenames[Const x: Integer]: String read GetFileName;
  end;

  // TStringList_ExactMatch: Like TStringList, but uses CompareStr instead of
  // AnsiCompareText, so "a" and "A" will be different entries here.
  TStringList_ExactMatch = class( TStringList )
  public
    function Find(const S: string; var Index: Integer): Boolean; override;
    function IndexOf(const S: string): Integer; override;
    procedure Sort; override;
  end;

  TFileStreamEx = class( TFileStream ) // OJB
  public
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
  end;

  TStringListEx = class(TStringList) // OJB
  public
    procedure AppendToFile(const FileName: string);
  end;

function ParseString( PartNo: Integer; DefVal, Line: String; Split: Char ): String;

function strtoint( s: String ) : Integer;
function Split( Src: String; Sep: Char; Idx: Integer ) : String;
function TrimWhSpace( Const s : String ) : String;
function PosWhSpace( s : String ) : Integer;
function ArgsWhSpace( const Line: String; const Args: TStrings; Const FillupArgs: Integer = 0 ): Integer;
function ArgsWhSpaceDQuoted( const Line: String; const Args: TStrings; Const FillupArgs: Integer = 0 ): Integer;
Function CountCharInString (Const c: Char; Const s: String): Integer;
function FilterNameOfFrom( F: String ) : String;
function FilterEmailOfFrom(From:String):String;  //HRR
function ReadLineFromFile( Filename: String ) : String;
procedure ForceFileExists( Filename: String );
function LikeRadix32( l : Longint ): String;
function MIDGenerator( LocalPart: String ): String;
function UIDLGenerator: String;
function GetExeVersion : String;
function GetFileInfo( Filename: String ): String;
function GetWindowsPath: String;
function GetSystemPath: String;
function GetWinVerInfo: String;
function ReplaceChar( s: String; oldc, newc: Char ): String;
function GetFileSize( const Filename: String ): LongInt;
function WildMat( pText: PChar; pPattern: PChar ): Boolean;
function IsAtom( c: Char ): Boolean;
function IsDomain( s: String ): Boolean;
function IsNewsgroup( s: String ): Boolean;
function GetMyBuildInfo: String;
function GetMyVersionInfo (Const bBuildInParantheses: Boolean) : String;
function GetMyStringFileInfo( StringName, DefResult: String ): String;
function FileExists2(SearchFile: String): Boolean; {Joern Weber}
function DirExists2(SearchFile: String): Boolean; {Joern Weber}
Function FileSize (Const FileName: String): Integer;
Function StripFirst(Const S:String; Const Trenner: String = ','): String; // "Admin, Administratoraccount" => "Admin"
Procedure DoEnable (pc: TWinControl; Const b: Boolean; Const Tiefe: Integer = 1 );
function RScan(c:Char; s:String):integer;
function IsWindowsNT:boolean;
function IsWindowsXP:boolean;
Function WindowsErrorText(Const Nr: Integer): String;
Procedure StripURLType (Var s: String);
Function SplitServerList (Const SrvList: String): String;
function ConvertPort(Const Port:String; Out PortNr: Integer): boolean; {JW} {KonfPort}
function Winsock2Installed( Silent: Boolean = False ): Boolean; {JH}
procedure AddChar( var s: String; const c: Char ); // s := s + c
function FileExistsByPattern( const SearchFile: String ): Boolean;
Function IIf (Const b: Boolean; Const yes: String; Const no: String = ''): String; overload;
Function IIf (Const b: Boolean; Const yes: double; Const no: double = 0): Double; overload;
Function IIf (Const b: Boolean; Const yes: Longint = 1; Const no: LongInt = 0): LongInt; overload;
Function Sgn (Const x: Extended): Integer;
Function CountLines (Const s: String): Integer;
function SeedPRNG: Boolean; {JW+MG}{PRNG}
function PRNG( Range: Integer ): Integer; {JW+MG}{PRNG}
procedure SetTreeNodeBold(ANode: TTreeNode; Value: Boolean);
function CheckLatin1Charset(Const s: String):Boolean;
function ExtractQuotedParameter(container,parameter:string):String; {JW} {SASL-DIGEST}
Function StrToFloat2(Const s: String): Extended;

// ----------------------------------------------------------------------------

implementation

uses SysUtils, FileCtrl, Forms, uBlowfish, cPCRE, uCRC32, uDateTime,
     cStdForm, uMD5, Global, WinSock, TypInfo, uSSL, cLogFile;

// -------------------------------------------------------------- TParser -----

function TParser.sPart( PartNo: Integer; DefVal: String ): String;
begin
     if (PartNo>=0) and (PartNo<Count) then Result:=Strings[PartNo]
                                       else Result:=DefVal;
end;

function TParser.iPart( PartNo: Integer; DefVal: Integer ): Integer;
begin
     try
        if (PartNo>=0) and (PartNo<Count) then Result:=strtoint(Strings[PartNo])
                                          else Result:=DefVal;
     except
        Result := DefVal;
     end;
end;

procedure TParser.Parse( ALine: String; ASplit: Char );
var  i : Integer;
begin
     ParsedLine := ALine;

     Clear;
     if ASplit=' ' then ALine:=Trim(ALine);

     while ALine<>'' do begin
        i := Pos( ASplit, ALine );
        if i=0 then begin
           Add( ALine );
           ALine := '';
        end else begin
           Add( copy( ALine, 1, i - 1 ) );
           System.Delete( ALine, 1, i );
           if ASplit=' ' then ALine:=Trim(ALine);
        end;
     end;
end;

procedure TParser.ParseQuoted( ALine: String; ASplit: Char );
var  i : Integer;
     s : String;
begin
     ParsedLine := ALine;

     Clear;
     if ASplit=' ' then ALine:=Trim(ALine);

     while ALine<>'' do begin
        i := Pos( ASplit, ALine );
        if i=0 then begin
           s := ALine;
           if copy(s,1,1)='"' then System.Delete(s,1,1);
           if copy(s,length(s),1)='"' then System.Delete(s,length(s),1);
           Add( s );
           ALine := '';
        end else begin
           s := copy( ALine, 1, i - 1 );
           System.Delete( ALine, 1, i );
           if copy(s,1,1)='"' then begin
              System.Delete( s, 1, 1 );
              if copy( s, length(s), 1 )='"' then begin
                 System.Delete( s, length(s), 1 );
              end else begin
                 s := s + ' ';
                 i := Pos( '"', ALine );
                 if i=0 then begin
                    s := s + ALine;
                    ALine := '';
                 end else begin
                    s := s + copy( ALine, 1, i-1 );
                    System.Delete( ALine, 1, i );
                 end;
              end;
           end;
           Add( s );
           if ASplit=' ' then ALine:=Trim(ALine);
        end;
     end;
end;

function ParseString( PartNo: Integer; DefVal, Line: String; Split: Char ): String;
var  P: TParser;
begin
     P := TParser.Create;
     P.Parse( Line, Split );
     Result := P.sPart( PartNo, DefVal );
     P.Free;
end;

// ----------------------------------------------- TStringList_ExactMatch -----

function TStringList_ExactMatch.Find(const S: string; var Index: Integer): Boolean;
var  L, H, I, C: Integer;
begin
   Result := False;
   L := 0;
   H := GetCount - 1;
   while L <= H do begin
      I := ( L + H ) shr 1;
      C := CompareStr( Get(I), S );
      if C < 0 then
         L := I + 1
      else begin
         H := I - 1;
         if C = 0 then begin
            Result := True;
            if Duplicates <> dupAccept then L := I;
         end;
      end;
   end;
   Index := L;
end;

function TStringList_ExactMatch.IndexOf( const S: string ): Integer;
begin
   if not Sorted then begin
      for Result := GetCount - 1 downto 0 do
         if CompareStr( Get(Result), S ) = 0 then exit;
      Result := -1;
   end else begin
      if not Find( S, Result ) then Result := -1;
   end;
end;

function StringListCompare_ExactMatch( List: TStringList; I1, I2: Integer ): Integer;
begin
   Result := CompareStr( List[I1], List[I2] );
end;

procedure TStringList_ExactMatch.Sort;
begin
   CustomSort( StringListCompare_ExactMatch );
end;

// ----------------------------------------------------------------------------

function strtoint( s: String ) : Integer;
begin
   Result := strtointdef( s, 0 );
end;

function Split( Src: String; Sep: Char; Idx: Integer ) : String;
var  j : Integer;
begin
     repeat
        j := Pos( Sep, Src );

        if j=0 then begin
           Result := Src;
           if Result='' then break;
           Src    := '';
        end else begin
           Result := copy( Src, 1, j-1 );
           System.Delete( Src, 1, j );
        end;

        dec( Idx );
     until Idx=0;
end;

function TrimWhSpace( Const s : String ) : String;
Var i, p, l: Integer;
begin
   p := 1; l := Length(s);
   For i := p to l do begin
      If (s[i]=#9) or (s[i]=' ') then Inc(p) else break
   end;
   For i := l downto p do begin
      If (s[i]=#9) or (s[i]=' ') then Dec(l) else break
   end;
   If p > l then Result := ''
   else If (p = 1) and (l=Length(s)) then Result := s
   else Result := Copy(s, p, l-p+1)
end;

function PosWhSpace( s : String ) : Integer;
var  j: Integer;
begin
     Result := Pos( ' ', s );
     if Result=0 then begin
        Result := Pos( #9, s );
     end else begin
        j := Pos( #9, s );
        if (j>0) and (j<Result) then Result:=j;
     end;
end;

function ArgsWhSpace( const Line: String;
                      const Args: TStrings;
                      Const FillupArgs: Integer = 0 ): Integer;
var  pch: PChar;
     s  : String;
begin
   Args.Clear;

   pch := PChar( Line );
   while pch^ in [ #9, ' ' ] do inc( pch );

   while pch^<>#0 do begin
      s := '';
      while not( pch^ in [ #0, #9, ' ' ] ) do begin
         s := s + pch^;
         inc( pch );
      end;
      Args.AddObject( s, Pointer(1) );
      while pch^ in [ #9, ' ' ] do inc( pch );
   end;

   Result := Args.Count;
   while Args.Count<FillupArgs do Args.AddObject( '', nil );
end;

function ArgsWhSpaceDQuoted( const Line: String; const Args: TStrings;
   Const FillupArgs: Integer = 0 ): Integer;
var  pch: PChar;
     s  : String;
     isq: Boolean;
begin
   Args.Clear;

   pch := PChar( Line );
   while pch^ in [ #9, ' ' ] do inc( pch );

   while pch^<>#0 do begin
      s := '';

      if pch^='"' then begin
         isq := True;
         inc( pch );
      end else begin
         isq := False;
      end;

      while pch^ <> #0 do begin
         case pch^ of
            #9, ' ': if isq then begin
                        s := s + pch^;
                     end else begin
                        inc( pch );
                        break;
                     end;
            '"'    : if (pch+1)^='"' then begin
                        s := s + '"';
                        inc( pch );
                     end else begin
                        inc( pch );
                        break;
                     end;
            else     s := s + pch^;
         end;
         inc( pch );
      end;

      Args.AddObject( s, Pointer(1) );
      while pch^ in [ #9, ' ' ] do inc( pch );
   end;

   Result := Args.Count;
   while Args.Count<FillupArgs do Args.AddObject( '', nil );
end;

Function CountCharInString (Const c: Char; Const s: String): Integer;
Var i: integer;
begin
   Result := 0;
   For i := 1 to Length(s) do If s[i]=c then Inc(Result)
end;

function FilterNameOfFrom( F: String ) : String;
var  j : Integer;
     s : String;
begin
     s := F;
     j := Pos( '(', s );
     if j>0 then begin
        System.Delete( s, 1, j );
        j := Pos( ')', s );
        if j>0 then s:=copy(s,1,j-1);
     end else begin
        // From: "xxx" <yyy@zzz>
        j := Pos( '<', s );
        if j>=5 then s := Trim( copy(s,1,j-1) );
        if copy(s,1,1)='"' then System.Delete(s,1,1);
        if copy(s,length(s),1)='"' then System.Delete(s,length(s),1);
     end;

     s := Trim( s );
     if s='' then s:=F;
     Result := s;
end;

function FilterEmailOfFrom(From:String):String; //HRR
var p1,p2 : Integer;
    erg   : String;
    i     : Integer;
begin
  erg:=TrimWhSpace(From);
  repeat
    p1:=0; p2:=0;
    for i:=1 to length(erg) do begin
      if erg[i]='"' then begin
         p1:=p2; p2:=i;
      end;
    end;
    if p1<>0 then begin
      system.delete(erg,p1,p2-p1+1);
      system.insert('dummy',erg,p1);
    end;
  until p1=0;
  i:=pos('<',erg);
  if i>0 then begin
    //name <aaa@bbb>
    system.delete(erg,1,i);
    i:=pos('>',erg);
    if i>0 then erg:=copy(erg,1,i-1);
  end else begin
    i:=pos('(',erg);
    if i>0 then begin
      //aaa@bbb (name)
      erg:=copy(erg,1,i-1);
    end;
  end;
  result:=TrimWhSpace(erg);
end;

function ReadLineFromFile( Filename: String ) : String;
begin
   Result := '';
   If FileExists2( Filename ) then begin
      With TTextreader.Create(FileName) do try
         While Not EOF do begin
            Result := ReadLine;
            if copy(Result,1,1)<>';' then break
         end
      finally
         free
      end
   end
end;

procedure ForceFileExists( Filename: String );
var  STM : TFileStream;
begin
     try
        ForceDirectories( ExtractFilePath(Filename) );
        if not FileExists2(Filename) then begin
           STM := TFileStream.Create( Filename, fmCreate );
           STM.Free;
        end;
     except
     end;
end;

function LikeRadix32( l : Longint ): String;
const ALPHABET = '0123456789abcdefghijklmnopqrstuv';
var  p : Longint;
begin
     Result := '';
     repeat
        p := l and $1F;
        l := l shr 5;
        //Result := Result + copy(ALPHABET,p+1,1);
        Result := copy(ALPHABET,p+1,1) + Result;
     until l=0;
end;

function MIDGenerator( LocalPart: String ): String;
const OFFSET = 673416000;
{JW} {MID}
//const Counter    : LongInt = 0;
//      OldDateTime: LongInt = 0;
{JW}
var   NewDateTime: LongInt;
      i: LongInt;
begin
     NewDateTime := DateTimeToUnixTime( Now ) - OFFSET;
{JW} {MID}
     EnterCriticalSection( CS_MID );
     if NewDateTime=GetCounter(MID_OldDateTime) then begin
        IncCounter( MID_Counter,1 )
     end else begin
        SetCounter(MID_Counter,1);
        SetCounter(MID_OldDateTime,NewDateTime);
     end;
     LeaveCriticalSection( CS_MID );
{JW}
     if LocalPart='' then LocalPart:='FQDN-not-set';
     if Pos('.',LocalPart)=0 then begin
{JW} {MID}
        i := NewDateTime XOR LongInt(GetCurrentThreadID) XOR GetCounter(MID_Counter);
{JW}
        LocalPart := LocalPart
                   + '.h' + lowercase( inttohex( StrToCRC32(inttostr(i)),8) )
                   + '.' + 'invalid';
     end;
     {JW}  {MID-Crypt}
     if Def_crypt_mid then
         Result := '<'
              + MD5toHex(MD5ofStr(LikeRadix32( NewDateTime ) + '.'
              + LikeRadix32(GetCurrentThreadID) + '.'
{JW} {MID}
              + inttostr( GetCounter(MID_Counter) )))
{JW}
              + '@' + LocalPart
              + '>'
      else
         Result := '<'
              + LikeRadix32( NewDateTime ) + '.'
              + LikeRadix32(GetCurrentThreadID) + '.'
{JW} {MID}
              + inttostr( GetCounter(MID_Counter) )
{JW}
              + '@' + LocalPart
              + '>';
      {/JW}

{     Result := '<'
             + MD5toHex(MD5ofStr(LikeRadix32( NewDateTime ) + '.'
             + LikeRadix32(GetCurrentThreadID) + '.'
             + inttostr( Counter )))
             + '@' + LocalPart
             + '>'}
end;

function UIDLGenerator: String;
var  j: Integer;
begin
     Result := MIDGenerator( 'dum.my' );
     System.Delete( Result, 1, 1 ); // "<"
     j := Pos( '@', Result ); // "@dum.my>"
     if j>0 then Result:=copy(Result,1,j-1);
end;

function GetExeVersion : String;
var  vlen, dw: DWord;
     vstr    : Pointer;
     p       : Pointer;
     s       : String;
begin
   {Versionsnummer ermitteln}
   Result := '?.?.?.?';

   vlen := GetFileVersionInfoSize( PChar(Application.ExeName), dw );
   if vlen=0 then exit;

   GetMem( vstr, vlen + 1 );
   try
      if GetFileVersionInfo( PChar(Application.ExeName), dw, vlen, vstr ) then begin
         if VerQueryValue( vstr, '\', p, dw ) then begin
            s := inttostr( hiword(PVSFixedFileInfo(p)^.dwFileVersionMS) ) + '.'
               + inttostr( loword(PVSFixedFileInfo(p)^.dwFileVersionMS) ) + '.'
               + inttostr( hiword(PVSFixedFileInfo(p)^.dwFileVersionLS) ) + '.'
               + inttostr( loword(PVSFixedFileInfo(p)^.dwFileVersionLS) );
            Result := s;
         end;
      end
   finally
      FreeMem( vstr, vlen + 1 );
   end
end;

function GetFileInfo( Filename: String ): String;
var  vlen, dw : DWord;
     vstr, p: Pointer;
     TS     : TSearchRec;
begin
     if FindFirst( Filename, faAnyFile, TS ) <> 0 then begin
        Result := '('+TrGl('Tools', 'File.not.found', 'file not found')+')';
        exit;
     end;

     Result := FormatDateTime( 'dd/mm/yyyy hh:nn', FileDateToDateTime(TS.Time) );
     Result := Result + ', ' + inttostr( TS.Size );
     SysUtils.FindClose( TS );

     vlen := GetFileVersionInfoSize( PChar(Filename), dw );
     if vlen=0 then exit;

     GetMem( vstr, vlen + 1 );
     if GetFileVersionInfo( PChar(Filename), dw, vlen, vstr ) then begin
        if VerQueryValue( vstr, '\', p, dw ) then begin
           with PVSFixedFileInfo(p)^ do begin
               Result := Result + ', ' + inttostr( hiword(dwFileVersionMS) )
                                + '.'  + inttostr( loword(dwFileVersionMS) )
                                + '.'  + inttostr( hiword(dwFileVersionLS) )
                                + '.'  + inttostr( loword(dwFileVersionLS) );
           end;
        end;
     end;
     FreeMem( vstr, vlen + 1 );
end;

function GetWindowsPath: String;
var  p: array[0..256] of Char;
begin
  GetWindowsDirectory( p, 256 );
  Result := String(p);
  if Result='' then exit;
  if Result[length(Result)]<>'\' then Result:=Result+'\';
end;

function GetSystemPath: String;
var  p: array[0..256] of Char;
begin
  GetSystemDirectory ( p, 256 );
  Result := String(p);
  if Result='' then exit;
  if Result[length(Result)]<>'\' then Result:=Result+'\';
end;

function GetWinVerInfo: String;
var  VerInfo: TOSVersionInfo;
     s, h   : String;
begin
     Result := '';

     VerInfo.dwOSVersionInfoSize := sizeof(VerInfo);
     GetVersionEx( VerInfo );

     with VerInfo do begin
        s := 'Windows ';

        case dwPlatformId of
           VER_PLATFORM_WIN32s:
              s := s + '32s';
           VER_PLATFORM_WIN32_WINDOWS:
              if dwMinorVersion<10 then begin
                 s := s + '95'; // 4.0
                 if (dwBuildNumber and $FFFF)>=1111 then s:=s+'-OSR2';
              end else if dwMinorVersion<90 then begin
                 s := s + '98'; // 4.10
                 if (dwBuildNumber and $FFFF)>=2222 then s:=s+'-SE';
              end else 
                 s := s + 'ME'; // 4.90
           VER_PLATFORM_WIN32_NT:
              if dwMajorVersion<=4 then
                 s := s + 'NT'   // 3.51/4.0
              else if (dwMajorVersion=5) and (dwMinorVersion=0) then
                 s := s + '2000' // 5.0
              else
                 s := s + 'XP';  // 5.1
           else
              s := s + '?PlatformID=' + inttostr(dwPlatformId) + '?';
        end;

        s := s + ' (Vr. ' + inttostr(dwMajorVersion)
               + '.' + inttostr(dwMinorVersion)
               + '.' + inttostr(dwBuildNumber and $FFFF);
        h := String( szCSDVersion );
        if h<>'' then s := s + ', ' + h;
        s := s + ')';
        Result := s;
     end;
end;

function ReplaceChar( s: String; oldc, newc: Char ): String;
var  i: Integer;
begin
     if oldc<>newc then begin
        for i:=1 to length(s) do begin
           if s[i]=oldc then s[i]:=newc;
        end;
     end;

     Result := s;
end;

function GetFileSize( const Filename: String ): LongInt;
var  SR: TSearchRec;
begin
     if SysUtils.FindFirst( Filename, faAnyFile, SR )=0 then begin
        Result := SR.Size;
     end else begin
        Result := 0;
     end;
     SysUtils.FindClose( SR );
end;

const
  WIMA_TRUE  = 1;
  WIMA_FALSE = 0;
  WIMA_ABORT = -1;

function DoWildMat( pText: PChar; pPattern: PChar ): Integer;
// Adapted from INN 2.0 (wildmat.c, rev. 1.2) by Rich $alz (<rsalz@osf.org>).
var  cLast: Char;
     iMatched, iReverse: Integer;
begin
     while ( pPattern^ <> #0 ) do begin

	if ( pText^ = #0 ) and ( pPattern^ <> '*' ) then begin
	    Result := WIMA_ABORT;
            exit;
        end;

        if ( pPattern^ = '\' ) then begin

           // Literal match with following character.
           inc( pPattern );
           if ( pText^ <> pPattern^ ) then begin
              Result := WIMA_FALSE;
              exit;
           end;

        end else begin

           case pPattern^ of

              '?': ; // Match anything.

              '*': begin
                      repeat
                         inc( pPattern );
                      until pPattern^<>'*'; // Consecutive stars act just like one.

                      if (pPattern^ = #0) then begin
                          // Trailing star matches everything.
                          Result := WIMA_TRUE;
                          exit;
                      end;

                      while (pText^<>#0) do begin
                         iMatched := DoWildMat( pText, pPattern );
                         inc( pText );
                         if iMatched<>WIMA_FALSE then begin
                            Result := iMatched;
                            exit;
                         end;
                      end;

                      Result := WIMA_ABORT;
                      exit;
                   end;

              '[': begin
                      if (pPattern+1)^='^' then begin
                         // Inverted character class.
                         iReverse := WIMA_TRUE;
                         inc(pPattern);
                      end else begin
                         iReverse := WIMA_FALSE;
                      end;

                      iMatched := WIMA_FALSE;

                      if ((pPattern+1)^ = ']') or ((pPattern+1)^ = '-') then begin
                         // special case: first char of class is ']' or '-'
                         inc(pPattern);
                         if (pPattern^ = pText^) then iMatched := WIMA_TRUE;
                      end;

                      cLast := pPattern^;
                      while ((pPattern+1)^<>#0) and (pPattern^<>']') do begin
                         inc(pPattern);

                         // "This next line requires a good C compiler." *sigh*
                         if (pPattern^='-') and ((pPattern+1)^<>']') then begin
                            // char-range
                            inc(pPattern);
                            if (pText^ <= pPattern^) and (pText^ >= cLast) then iMatched:=WIMA_TRUE;
                         end else begin
                            // single char
                            if pText^=pPattern^ then iMatched:=WIMA_TRUE;
                         end;

                         cLast := pPattern^;
                      end;

                      if (iMatched = iReverse) then begin
                         Result := WIMA_FALSE;
                         exit;
                      end;
                   end;

              else
                 // match char
                 if (pText^ <> pPattern^) then begin
                     Result := WIMA_FALSE;
                     exit;
                 end;
           end;
        end;

        inc( pText );
        inc( pPattern );

     end;

     if (pText^ = #0) then Result := WIMA_TRUE
                      else Result := WIMA_FALSE;
end;

function WildMat( pText: PChar; pPattern: PChar ): Boolean;
begin
     Result := ( DoWildMat(pText,pPattern) = WIMA_TRUE );
end;

function IsAtom( c: Char ): Boolean;
begin
     {
     atom        =  1*<any CHAR except specials, SPACE and CTLs>
     CHAR        =  <any ASCII character>        ; (  0-177,  0.-127.)
     CTL         =  <any ASCII control           ; (  0- 37,  0.- 31.)
                     character and DEL>          ; (    177,     127.)
     SPACE       =  <ASCII SP, space>            ; (     40,      32.)
     specials    =  "(" / ")" / "<" / ">" / "@"  ; Must be in quoted-
                 /  "," / ";" / ":" / "\" / <">  ;  string, to use
                 /  "." / "[" / "]"              ;  within a word.
     }
     Result := False;
     if (c>=#33) and (c<=#126) then begin
        if Pos(c,'()<>@,;:\".[]')=0 then Result:=True;
     end;
end;

function IsDomain( s: String ): Boolean;
var  NeedAtom: Boolean;
     c: Char;
     i: Integer;
begin
     // Note: 'domain-literal' not supported here
     Result := True;
     NeedAtom := True;
     for i:=1 to length(s) do begin
        c := s[i];
        if NeedAtom then begin
           if IsAtom(c) then NeedAtom := False
                        else break;
        end else begin
           if c='.' then
              NeedAtom := True
           else
              if not IsAtom(c) then begin
                 Result:=False;
                 break;
              end;
        end;
     end;
     if NeedAtom then Result:=False;
end;

function IsNewsgroup( s: String ): Boolean;
var  NeedCS: Boolean;
     c: Char;
     i: Integer;
begin
          {
          Newsgroups-content = newsgroup-name *( ng-delim newsgroup-name)

          newsgroup-name = *FWS component *( "." component ) *FWS
          component = component-start [*component-rest component-start]

          component-start = lowercase / digit
          component-rest = component-start / "+" / "-" / "_"
          ng-delim = ","

          // added uppercase (MS Exchange)
          }
     Result := True;
     NeedCS := True;
     for i:=1 to length(s) do begin
        c := s[i];
        if NeedCS then begin
           if c in ['a'..'z','A'..'Z','0'..'9'] then NeedCS := False
                                                else break;
        end else begin
           if c='.' then
              NeedCS := True
           else
              if not (c in ['a'..'z','A'..'Z','0'..'9','+','-','_']) then begin
                 Result:=False;
                 break;
              end;
        end;
     end;
     if NeedCS then Result:=False;
end;

function GetMyBuildInfo: String;
var  vlen, dw: DWord;
     vstr, p : Pointer;
begin
   Result := '?.?';

   vlen := GetFileVersionInfoSize( PChar(Application.ExeName), dw );
   if vlen=0 then exit;

   GetMem( vstr, vlen + 1 );
   try
      if GetFileVersionInfo( PChar(Application.ExeName), dw, vlen, vstr ) then begin
         if VerQueryValue( vstr, '\', p, dw ) then begin
            Result :=  inttostr( hiword(PVSFixedFileInfo(p)^.dwFileVersionMS) ) + '.'
                     + inttostr( loword(PVSFixedFileInfo(p)^.dwFileVersionMS) ) + '.'
                     + inttostr( hiword(PVSFixedFileInfo(p)^.dwFileVersionLS) ) + '.'
                     + inttostr( loword(PVSFixedFileInfo(p)^.dwFileVersionLS) )
         end
      end
   finally
      FreeMem( vstr, vlen + 1 )
   end
end;

function GetMyVersionInfo (Const bBuildInParantheses: Boolean): String;
var  vlen, dw: DWord;
     vstr, p : Pointer;
begin
     Result := '?.?';

     vlen := GetFileVersionInfoSize( PChar(Application.ExeName), dw );
     if vlen=0 then exit;

     GetMem( vstr, vlen + 1 );
     try
        if GetFileVersionInfo( PChar(Application.ExeName), dw, vlen, vstr ) then begin
           if VerQueryValue( vstr, '\', p, dw ) then begin
              Result := TrGl('Tools', 'Version', 'Vr.')+' '
                               + inttostr( hiword(PVSFixedFileInfo(p)^.dwProductVersionMS) ) + '.'
                               + inttostr( loword(PVSFixedFileInfo(p)^.dwProductVersionMS) ) + ' ';
              If bBuildInParantheses then Result := Result + '(';
              Result := Result + TrGl('Tools', 'Build', 'Build')+' '
                               + inttostr( hiword(PVSFixedFileInfo(p)^.dwFileVersionMS) ) + '.'
                               + inttostr( loword(PVSFixedFileInfo(p)^.dwFileVersionMS) ) + '.'
                               + inttostr( hiword(PVSFixedFileInfo(p)^.dwFileVersionLS) ) + '.'
                               + inttostr( loword(PVSFixedFileInfo(p)^.dwFileVersionLS) );
              If bBuildInParantheses then Result := Result + ')'
           end;
        end
     finally
        FreeMem( vstr, vlen + 1 )
     end
end;

function GetMyStringFileInfo( StringName, DefResult: String ): String;
var  vlen, dw: DWORD;
     vstr, p : Pointer;
     tmp     : array[0..3] of Byte;
     LangID  : String;
begin
   Result := DefResult;
   vlen := GetFileVersionInfoSize( PChar(Application.ExeName), dw );
   if vlen=0 then exit;

   GetMem( vstr, vlen + 1 );
   if GetFileVersionInfo( PChar(Application.ExeName), dw, vlen, vstr ) then begin
      if VerQueryValue( vstr, '\VarFileInfo\Translation', p, dw ) then begin
         if dw>=4 then begin
            Move( p^, tmp[0], 4 );
            dw := tmp[2] or tmp[3] shl 8 or tmp[0] shl 16 or tmp[1] shl 24;
            LangID := inttohex( dw, 8 );
            if VerQueryValue( vstr, PChar('\StringFileInfo\'+LangID+'\'+StringName), p, dw ) then begin
               Result := String( PChar(p) )
            end
         end
      end
   end;
   FreeMem( vstr, vlen + 1 );
end;

// ----------------------------------------------------------------------------

function FileExists2(SearchFile: String): Boolean; {Joern Weber}

   Function IsFile(Const Find_Data: WIN32_FIND_DATA): boolean;
   begin
      Result := (Find_Data.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) = 0
   end;

var
    Handle    : dWord;
    Find_Data : WIN32_FIND_DATA;
begin
   Result := false;
   handle:=FindFirstFile(PChar(SearchFile),Find_Data);
   if handle<>INVALID_HANDLE_VALUE then begin
      try
         if Not IsFile(Find_Data) then begin
            while FindNextFile(handle,Find_Data) do begin
               If IsFile(Find_Data) then begin
                  Result:=True;
                  break;
               end
            end
         end else begin
            Result:=True
         end
      finally
         Windows.FindClose(Handle)
      end
   end
end;

function DirExists2(SearchFile: String): Boolean; {Joern Weber}
{
   Function IsDir(Const Find_Data: WIN32_FIND_DATA);
   begin
      Result := (Find_Data.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) = 0
   end;
}
var
    Handle    : dWord;
    Find_Data : WIN32_FIND_DATA;
begin
   Result:=False;
   handle:=FindFirstFile(PChar(ExcludeTrailingBackslash(SearchFile)),Find_Data);
   if handle<>INVALID_HANDLE_VALUE then try
      Repeat
         With Find_Data do begin
            If (dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY)=FILE_ATTRIBUTE_DIRECTORY then begin
               If (CFileName[0]+CFileName[1]<>'.'#0) and
                  (CFileName[0]+CFileName[1]+CFileName[2]<>'..'#0)
               then begin
                  Result := true;
                  exit
               end
            end
         end
      until Not FindNextFile(handle,Find_Data)
   finally
      Windows.FindClose(handle)
   end
end;

Function FileSize (Const FileName: String): Integer;
Var r: TSearchRec;
begin
   If Findfirst (FileName, faAnyfile, r) = 0 then begin
      Result := r.Size;
      FindClose(r)
   end else begin
      Result := 0
   end
end;

Function StripFirst(Const S:String; Const Trenner: String = ','): String;
Var p: Integer;
begin
   p := Pos(Trenner, s);
   If p > 0 then Result := Copy(s, 1, p-1) else Result := s
end;

Procedure DoEnable (pc: TWinControl; Const b: Boolean; Const Tiefe: Integer = 1 );
Var i: Integer; c: TControl; P: PPropInfo;
begin
   For i := 0 to pc.ControlCount-1 do begin
      c := pc.Controls[i];
      If (Tiefe > 1) and (c is TWinControl) then DoEnable ( c as TWinControl, b, Tiefe-1 );
      p := GetPropInfo(C.ClassInfo, 'Enabled', [tkEnumeration]);
      If Assigned(p) then begin
         If b then SetEnumProp (C, p, 'true')
              else SetEnumProp (C, p, 'false')
      end
   end
end;

function RScan(c:Char; s:String):integer;
var
  fc:PChar;
begin
  fc:=StrRScan(PChar(s),c);
  if fc<>Nil then
    Result:=fc-PChar(s)+1
  else
    Result:=0;
end;

function IsWindowsNT:boolean;
var
 OsVinfo   : TOSVERSIONINFO;
begin
 ZeroMemory(@OsVinfo,sizeOf(OsVinfo));
 OsVinfo.dwOSVersionInfoSize := sizeof(TOSVERSIONINFO);
 if GetVersionEx(OsVinfo) then
   Result:=OsVinfo.dwPlatformId = VER_PLATFORM_WIN32_NT
 else
   Result:=false;
end; {IsWindowsNT}

function IsWindowsXP:boolean; {JW}
Var OsVinfo: TOSVERSIONINFO;
begin
  ZeroMemory(@OsVinfo,sizeOf(OsVinfo));
  OsVinfo.dwOSVersionInfoSize := sizeof(TOSVERSIONINFO);
  GetVersionEx(OsVinfo);
  With OsVinfo do begin
     Result :=
        ( dwPlatFormID = 2 ) and
        ((dwMajorVersion > 5) or ((dwMajorVersion = 5) and (dwMinorVersion > 0)))
  end
end;

Function WindowsErrorText(Const Nr: Integer): String;
Var Buffer: Array[1..1024] of Char; i, Anz: Integer;
begin
   FillChar(Buffer[1], 1024, 0);
   Anz := FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NIL, Nr, 0, @Buffer[1], 1024, NIL);
   If Anz > 0 then begin
      Result := '';
      For i := 1 to Anz do If Not (Buffer[i] IN[#13,#10,#0]) then Result := Result + Buffer[i]
   end else begin
      Result := '???'
   end
end;

Procedure StripURLType (Var s: String);
Var p: Integer;
begin
   p := Pos('://', s);
   If p > 0 then Delete(s, 1, p+2)
end;

Function SplitServerList (Const SrvList: String): String;
Var l, s: String; p: Integer;
begin
   Result := '';
   If SrvList = '' then Exit;
   l := SrvList;
   Repeat
      p := Pos(';', l);
      If p > 0 then begin
         s := Copy(l, 1, p-1);
         Delete(l, 1, p)
      end else begin
         s := l;
         l := ''
      end;
      s := Trim(s);
      If s > '' then Result := Result + s + #13#10
   until l = ''
end;

{JW} {KonfPort}
function ConvertPort(Const Port:String; Out PortNr: Integer): Boolean;
var i: Short; bNumeric: Boolean;
begin
   Result := false;
   bNumeric := Port > '';
   for i:=1 to length(Port) do If bNumeric then begin
     if (Port[I]<'0') or (Port[I]>'9') then bNumeric:=false
   end;
   If bNumeric then begin
      PortNr := StrToInt(Port);
      If PortNr > 65536
         then Log( LOGID_ERROR, 'ports >65536 are not allowed! Port: '+Port)
         else Result := true
   end else begin
      if (getservbyname(PChar(Port),'')<>nil) then begin
         // resolve portname
         i:=getservbyname(PChar(Port),'')^.s_port;
            // reverse byte order for network
         PortNr := (Lo(i) shl 8)+Hi(i);
         Log( LOGID_DEBUG, 'Portname "'+port+'" resolved to '+IntToStr(PortNr));
         Result := true
      end else begin
         Log( LOGID_ERROR, TrGlF(kLog, 'portname.resolve.error',
            'Cannot resolve portname "%s" to a portnumber', port));
      end
   end
end;
{JW}

function Winsock2Installed( Silent: Boolean = False ): Boolean; {JH}
var  VerInfo: TOSVersionInfo;
     ws2_32_dll, vernr: String;
     vlen, dw : DWord;
     vstr, p: Pointer;
begin
   VerInfo.dwOSVersionInfoSize := sizeof(VerInfo);
   GetVersionEx( VerInfo );
   Result := ( VerInfo.dwPlatformId = VER_PLATFORM_WIN32_NT );
   if Result then exit;

   ws2_32_dll := GetSystemPath + 'ws2_32.dll';
   if FileExists2( ws2_32_dll ) then begin
      vlen := GetFileVersionInfoSize( PChar(ws2_32_dll), dw );
      if vlen=0 then exit;
      GetMem( vstr, vlen + 1 );
      if GetFileVersionInfo( PChar(ws2_32_dll), dw, vlen, vstr ) then begin
         if VerQueryValue( vstr, '\', p, dw ) then begin
            with PVSFixedFileInfo(p)^ do begin
                vernr := inttostr( hiword(dwFileVersionMS) )
                + '.'  + inttostr( loword(dwFileVersionMS) )
                + '.'  + inttostr( hiword(dwFileVersionLS) )
                + '.'  + inttostr( loword(dwFileVersionLS) );
            end;
            // all versions ok except a specific Win95-"beta"-version
            Result := ( vernr <> '4.10.0.1511' );
         end;
      end;
      FreeMem( vstr, vlen + 1 );
   end;
   if Result then exit;

   if Silent then exit;
   MessageBox( 0, 'This application requires Winsock2, which can be loaded '
                + 'from www.microsoft.com!',
               'Winsock2-Check', MB_OK or MB_ICONEXCLAMATION );
end;

procedure AddChar( var s: String; const c: Char ); // s := s + c {JH}
begin
   SetLength( s, length(s)+1 );
   s[ length(s) ] := c;
end;

function FileExistsByPattern( const SearchFile: String ): Boolean; {JH}
var  Handle  : DWORD;
     FindData: WIN32_FIND_DATA;
begin
   Result := False;
   Handle := FindFirstFile( PChar(SearchFile), FindData );
   if Handle <> INVALID_HANDLE_VALUE then begin
      repeat
         if (FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY)=0 then begin
            Result := True;
            break;
         end;
      until not FindNextFile( Handle, FindData );
      Windows.FindClose( Handle );
   end;
end;

Function IIf (Const b: Boolean; Const yes: String; Const no: String = ''): String;
begin
   If b then Result := yes else Result := no
end;

Function IIf (Const b: Boolean; Const yes: double; Const no: double = 0): Double;
begin
   If b then Result := yes else Result := no
end;

Function IIf (Const b: Boolean; Const yes: Longint = 1; Const no: LongInt = 0): LongInt;
begin
   If b then Result := yes else Result := no
end;

Function Sgn (Const x: Extended): Integer;
begin
   If x > 0 then Result := 1
   else If x < 0 then Result := -1
   else Result := 0
end;

Function CountLines (Const s: String): Integer;
Var i: Integer;
begin
   Result := 0;
   For i := 1 to Length(s) do If s[i] = #13 then Inc(Result);
   If (Length(s) > 2) and (Copy(s, Length(s)-1, 2)<>#13#10) then Inc(Result)
end;

function SeedPRNG: Boolean; {JW+MG}{PRNG}
var  Buffer : Array[0..3] of Char;
     BufLen : Integer;
begin
   Result := False;
   BufLen := SizeOf( Buffer );
   try
      FillChar( Buffer, BufLen, 0 );
      Result := SSLReady and SSL_RandBytes( Buffer, BufLen );
      If Not Result then begin
         // Use external file for seed if exist
         if FileExists2( CFGFILE_RANDSEED ) then begin
            with TFileStream.Create( CFGFILE_RANDSEED, fmOpenRead and fmShareDenyWrite ) do try
               Result := Read( Buffer, BufLen ) = BufLen
            finally Free end
         end
      end
   except
      Log( LOGID_ERROR, 'can not PRNG seed PRNG');
   end;
   if Result then begin
      // Seed PRNG with Open SSL
      RandSeed := LongInt( Buffer );
      Log( LOGID_Detail, 'PRNG seeded with Open SSL')
   end else begin
      Randomize;
      Log( LOGID_Detail, 'PRNG seeded with standard library')
   end
end;

function PRNG( Range: Integer ): Integer; {JW+MG}{PRNG}
var  Success : Boolean;
     Bits, Bytes, UnusedBits : Integer;
     Buffer : Array[0..3] of Char;
begin
   Result := -1;
   Success := False;
   // range must greater 0
   if Range < 1 then exit;
   if SSLReady then begin
      Bits := 0;
      // adapting the range for PRNG
      While (Range shr Bits) > 0 do inc( Bits );
      Bytes := ( Bits shr 3 ) + 1;
      UnusedBits := 8 - ( Bits mod 8 );
      // fill the result for the required range with bytes
      if Bytes <= 4 then begin
         While not Success do begin
            if not SSL_RandBytes( Buffer, Bytes ) then break;
            Result := Integer( Buffer ) shr UnusedBits;
            if Result < Range then Success := True
         end
      end;
      Log( LOGID_Detail, TrGl(kLog, 'SSL.Pseudo-random-number-generated',
         'Pseudo-random number generated with OpenSSL'))
   end;
   if Not Success then Result := Random( Range )
end;

procedure SetTreeNodeBold(ANode: TTreeNode; Value: Boolean);
var
  Item: TTVItem;
  Template: Integer;
begin
  if ANode = nil then Exit;
  if value then Template := -1 else Template := 0;
  with Item do begin
    mask := TVIF_STATE;
    hItem := ANode.ItemId;
    StateMask := TVIS_BOLD;
    State := StateMask and Template;
  end;
  TreeView_SetItem(ANode.TreeView.Handle, Item);
end;

{ TFileStreamEx }

{$IFDEF VER130}
Procedure RaiseLastOSError;
begin
   RaiseLastWin32Error
end;
{$ENDIF}

function TFileStreamEx.Read(var Buffer; Count: Integer): Longint;
begin
   Result := inherited Read( Buffer, Count );
   if Count <> Result then begin
      if (Result=0) and (GetLastError<>NO_ERROR) then RaiseLastOSError;
   end;
end;

function TFileStreamEx.Seek(Offset: Integer; Origin: Word): Longint;
begin
   Result := inherited Seek( Offset, Origin );
   if (Result=-1) and (GetLastError<>NO_ERROR) then RaiseLastOSError;
end;

function TFileStreamEx.Write(const Buffer; Count: Integer): Longint;
begin
   Result := inherited Write( Buffer, Count );
   if Count <> Result then begin
      if (Result=0) and (GetLastError<>NO_ERROR) then RaiseLastOSError;
   end;
end;

{ TStringListEx }

procedure TStringListEx.AppendToFile(const FileName: string);
Var Stream: TStream; parm: Word;
begin
  if FileExists2(FileName)
     then parm := fmOpenWrite
     else parm := fmCreate;
  Stream := TFileStreamEx.Create(FileName, parm);
  try
     Stream.Seek(0, soFromEnd);
     SaveToStream(Stream);
  finally
     Stream.Free;
  end
end;

{ TFileList }

constructor TFileList.Create (Const Path, FileMask: String; Const Attr: Integer;
       Const Options: TFileListOptions);
Var r: TSearchRec;
begin
   FList := TStringList.Create;
   fPath := IncludeTrailingBackslash(PATH);
   If FindFirst(fPATH + FileMask, Attr, r) = 0 then try
      Repeat
         If flFiles IN Options then begin
            If (r.Attr and faDirectory) = 0 then FList.Add (r.Name)
         end;
         If flDirs IN Options then begin
            If (r.Name[1]<>'.') and ((r.Attr and faDirectory)>0) then FList.Add(r.Name)
         end
      until FindNext(r)<>0
   finally
      FindClose(r)
   end;
   FList.Sorted := flSorted IN Options
end;

destructor TFileList.Destroy;
begin
  inherited;
  fList.Free;
end;

function TFileList.Count: Integer;
begin
   Result := FList.Count
end;

function TFileList.GetFile(const x: Integer): String;
begin
   Result := fPath + FList[x]
end;

function TFileList.GetFileName(const x: Integer): String;
begin
   Result := FList[x]
end;

function CheckLatin1Charset(Const s: String):Boolean;
var i: Integer;
begin
   result:=true;
   if Application.MainForm.Font.Charset=0 then begin
      For i:=1 to length(s) do begin
         if (s[i]>=#128) or (s[i]<#12) then begin
           result:=false; break;
         end
      end
   end else begin
      For i:=1 to length(s) do begin
         if (s[i]>=#128) then begin
           result:=false; break;
         end
      end
   end
end;

{JW} {SASL-DIGEST}

// DIGEST-MD5 Parser

// extract parameter from container string
// with format "paramerter=<value>,paramerter="<value>"" from
// given container string and return the value

function ExtractQuotedParameter(container,parameter:string):string;
begin
  result:='';
  // parameter in container not present
  if pos(parameter,container)=0 then exit;
  parameter:=trim(parameter);
  container:=trim(container);
  repeat
    // check the first parameter in container
    if copy(container+'=',1,length(parameter)+1)=parameter+'=' then begin
      result:=copy(container,length(parameter)+2,length(container));
      if result[1]='"' then begin   // check if dequoting required
        result:=copy(result,2,length(result)-1) ;
        result:=copy(result,1,pos('"',result)-1);
      end else
        // if not end of container cut the parameter
        if pos(',',result)<>0 then
          result:=copy(result,1,pos(',',result)-1);
      exit;
    end else
    // check if exist a next parameter
    if (pos('"',container)>0) and
        (pos(',',container)>pos('"',container)) then begin
      // remove current qoutet parameter
      container:=copy(container,pos('"',container)+1,length(container));
      container:=copy(container,pos('"',container)+1,length(container));
      if pos(',',container)>0 then // remove seprtor ","
        container:=copy(container,pos(',',container)+1,length(container));
    end else
    // remove currend unquotet parameter
    if pos(',',container)=0 then
      container:=''
    else
      container:=copy(container,pos(',',container)+1,length(container));
  until  container=''
end;

Function StrToFloat2(Const s: String): Extended;
Var p: Integer; s2: String;
begin
   s2 := s;
   If ThousandSeparator<>DecimalSeparator then begin
      Repeat
         p := Pos(ThousandSeparator, s2);
         If p > 0 then Delete(s2, p, 1)
      until p=0
   end;
   Result := StrToFloat(s2)
end;

{ TTextReader }

constructor TTextReader.Create(const FileName: String; Const Buffersize: Integer = 128);
begin
   FileMode := 0;
   If BufferSize > 128 then begin
      SetLength(FIntBuffer, Buffersize);
      SetTextBuf(F, FIntBuffer[1], Buffersize)
   end;
   AssignFile(F, FileName);
   Reset(F)
end;

destructor TTextReader.Destroy;
begin
  inherited;
  CloseFile(F)
end;

function TTextReader.EOF: Boolean;
begin
   Result := System.Eof(F) and (FLineBuffer = '')
end;

function TTextReader.ReadLine: String;
Var p: Integer;
begin
   If FLineBuffer = ''
      then Readln(F, Result)
      else Result := FLineBuffer;
   p := Pos(#10, Result);
   If p > 0 then begin
      FLineBuffer := Copy(Result, p+1, Length(Result)-p);
      Delete(Result, p, Length(Result));
   end
end;

end.
