// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit Config; // Configuration-classes and -functions

// ----------------------------------------------------------------------------
// Contains various configuration classes and functions
// ----------------------------------------------------------------------------

interface

uses Windows, SysUtils, Classes, IniFiles, Forms, FileCtrl, uTools, uRasDyn,
     cAccount, cHistoryNews, cHistoryMail, Winsock, cNewsJobs, cSyncObjects;

Procedure CheckObsoleteServerDirs;
Procedure CheckObsoleteGroupDirs;

const
   FICTGROUP_SRVINFOS   = '(ServerInfo)';
   FICTGROUP_LISTOFMIDS = '(ListOfMIDs)';

type
  TConfigList = class( TStringList ) // Sorted:=True; Duplicates:=dupIgnore;
    public
      procedure LoadFromFile(const FileName: string); override;
      constructor Create;
  end;

type
  TCfgHamster = class
    private
      FLock: TReaderWriterLock;

      FNntpServers      : TConfigList;
      FNntpServersInUse : TConfigList;
      FNewsgroups       : TConfigList;
      FNewsPulls        : TConfigList;
      FPop3Servers      : TConfigList;
      FSmtpServers      : TConfigList;

      function GetServerCount: Integer;
      function GetServerPath( Index: Integer ): String;
      function GetServerName( Index: Integer ): String;
      function GetServerPort( Index: Integer ): String;
      function GetServerUser( Index: Integer ): String;
      function GetServerPass( Index: Integer ): String;
      function GetServerIndexOf( FindServer: String ) : Integer;
      {MG}{SSL}
      function GetServerMode( Index: Integer ): Integer;
      function GetServerVerify( Index: Integer ): Integer;
      function GetServerCaFile( Index: Integer ): String;
      {/SSL}
      Function GetNNTPServertype( Index: Integer ): Integer;

      function GetServerPullList( Server: String ): String;
      function GetServerIsReadOnly( Server: String ): Boolean;

      function GetActiveCount: Integer;
      function GetActiveName( Index: Integer ): String;
      function GetActivePath( Index: Integer ): String;
      function GetActiveType(Groupname: String): Char;
      function GetActiveIndexOf( FindGroup: String ) : Integer;

      function GetPullCount: Integer;
      function GetPullServer( Index: Integer ): String;
      function GetPullGroup ( Index: Integer ): String;

      function GetPop3ServerCount: Integer;
      function GetPop3ServerPath( Index: Integer ): String;
      function GetPop3ServerName( Index: Integer ): String;
      function GetPop3ServerPort( Index: Integer ): String;
      function GetPop3ServerUser( Index: Integer ): String;
      function GetPop3ServerPass( Index: Integer ): String;
      function GetPop3ServerLocalUser( Index: Integer ): String;
      function GetPop3ServerFiltersection(Index: Integer): String;
      function GetPop3ServerIndexOf( FindServer: String ) : Integer;
      {MG}{SSL}
      function GetPop3ServerMode( Index: Integer ): Integer;
      function GetPop3ServerVerify( Index: Integer ): Integer;
      function GetPop3ServerCaFile( Index: Integer ): String;
      {/SSL}

      function GetSmtpServerCount: Integer;
      function GetSmtpServerPath( Index: Integer ): String;
      function GetSmtpServerName( Index: Integer ): String;
      function GetSmtpServerPort( Index: Integer ): String;
      function GetSmtpServerUser( Index: Integer ): String;
      function GetSmtpServerPass( Index: Integer ): String;
      function GetSmtpServerIndexOf( FindServer: String ) : Integer;
      function GetSmtpServerMode( Index: Integer ): Integer;
      function GetSmtpServerVerify( Index: Integer ): Integer;
      function GetSmtpServerCaFile( Index: Integer ): String;

    public
      property Lock: TReaderWriterLock read FLock;

      property ServerCount: Integer read GetServerCount;
      property ServerPath[Index: Integer]: String read GetServerPath;
      property ServerName[Index: Integer]: String read GetServerName;
      property ServerPort[Index: Integer]: String read GetServerPort;
      property ServerUser[Index: Integer]: String read GetServerUser;
      property ServerPass[Index: Integer]: String read GetServerPass;
      property ServerIndexOf[FindServer: String]: Integer read GetServerIndexOf;
      property ServerPullList[Server: String]: String read GetServerPullList;
      function ServerPullThreads( Index: Integer; LimitByJobs: Boolean ): Integer;
      property ServerIsReadOnly[Server: String]: Boolean read GetServerIsReadOnly;
      Function ServerUseInc( Index: Integer ): Boolean;
      Function ServerUseDec( Index: Integer ): Integer;
      {MG}{SSL}
      property ServerMode[Index: Integer]: Integer read GetServerMode;
      property ServerVerify[Index: Integer]: Integer read GetServerVerify;
      property ServerCaFile[Index: Integer]: String read GetServerCaFile;
      {/SSL}
      Property NNTPServertype[Index: Integer]: Integer read GetNNTPServertype;

      property ActiveCount: Integer read GetActiveCount;
      property ActivePath[Index: Integer]: String read GetActivePath;
      property ActiveName[Index: Integer]: String read GetActiveName;
      property ActiveType[Groupname: String]: Char read GetActiveType;
      property ActiveIndexOf[FindGroup: String]: Integer read GetActiveIndexOf;
      function ActiveAdd( Groupname: String ): Boolean;
      function ActiveDel( Groupname: String ): Boolean;
      Function GetActivePostServer(Const Groupname: String; Out Server, Newsdir: String): Boolean;

      property PullCount: Integer read GetPullCount;
      property PullServer[Index: Integer]: String read GetPullServer;
      property PullGroup [Index: Integer]: String read GetPullGroup;
      function IsPostServerFor( Const Servername, Groupname: String ): Boolean;
      function ExistPull( Const Servername, Groupname: String ): Boolean;
      function ExistPullServer(Const Groupname: String): Boolean;
      Function PullServerOf( Const Groupname: String ): String;
      function PullIndexOf( Const Servername, Groupname: String ): Integer;
      Function PullAdd(Const Servername, Groupname: String): Boolean;
      Function PullDel(Const Servername, Groupname: String): Boolean;

      property Pop3ServerCount: Integer read GetPop3ServerCount;
      property Pop3ServerPath[Index: Integer]: String read GetPop3ServerPath;
      property Pop3ServerName[Index: Integer]: String read GetPop3ServerName;
      property Pop3ServerPort[Index: Integer]: String read GetPop3ServerPort;
      property Pop3ServerUser[Index: Integer]: String read GetPop3ServerUser;
      property Pop3ServerPass[Index: Integer]: String read GetPop3ServerPass;
      property Pop3ServerLocalUser[Index: Integer]: String read GetPop3ServerLocalUser;
      property Pop3ServerFiltersection[Index: Integer]: String read GetPop3ServerFiltersection;
      property Pop3ServerIndexOf[FindServer: String]: Integer read GetPop3ServerIndexOf;
      {MG}{SSL}
      property Pop3ServerMode[Index: Integer]: Integer read GetPop3ServerMode;
      property Pop3ServerVerify[Index: Integer]: Integer read GetPop3ServerVerify;
      property Pop3ServerCaFile[Index: Integer]: String read GetPop3ServerCaFile;
      {/SSL}

      property SmtpServerCount: Integer read GetSmtpServerCount;
      property SmtpServerPath[Index: Integer]: String read GetSmtpServerPath;
      property SmtpServerName[Index: Integer]: String read GetSmtpServerName;
      property SmtpServerPort[Index: Integer]: String read GetSmtpServerPort;
      property SmtpServerUser[Index: Integer]: String read GetSmtpServerUser;
      property SmtpServerPass[Index: Integer]: String read GetSmtpServerPass;
      property SmtpServerIndexOf[FindServer: String]: Integer read GetSmtpServerIndexOf;
      property SmtpServerMode[Index: Integer]: Integer read GetSmtpServerMode;
      property SmtpServerVerify[Index: Integer]: Integer read GetSmtpServerVerify;
      property SmtpServerCaFile[Index: Integer]: String read GetSmtpServerCaFile;

      function GetUniqueMsgFilename( DestPath, Which: String ): String;

      procedure ReloadActiveList;
      procedure ReloadPullList;
      procedure ReloadConfiguration;

      constructor Create;
      destructor Destroy; override;
  end;

  TRasDialer = class
     private
       FLastError: Integer;
       OurRasConnection: THRasConn;
       OurRasDialParams: TRasDialParams;
       OurRasConnStatus: TRasConnStatus;
       StatTime : TDateTime;
       StatByte : LongInt;
       StatArts : LongInt;
       StatMails: LongInt;
       ConnID   : String;
       sDynamicIP: String;
       lDynamicIP: LongInt;
       function GetDynamicIP: LongInt;
     public
       property DynamicIP: LongInt read GetDynamicIP;
       function Dial( ConnectionID, UserName, Password: String ): Integer;
       function HangUp: Integer;
       property LastError: Integer read FLastError;
       function IsConnected: Boolean;
       function GetConnectionList( RasList: TStringList ): Boolean;
       function IsInstalled: Boolean;
       constructor Create;
       destructor Destroy; override;
  end;

  tExtInifile = class(tInifile)
  public
    procedure IRead(const Section, Ident: String; var Default: string); overload;
    procedure IRead(const Section, Ident: String; var Default: Integer); overload;
    procedure IRead(const Section, Ident: String; var Default: DWord); overload;
    procedure IRead(const Section, Ident: String; var Default: Boolean); overload;
    procedure IWrite(const Section, Ident: String; const Value: string); overload;
    procedure IWrite(const Section, Ident: String; const Value: Integer); overload;
    procedure IWrite(const Section, Ident: String; const Value: Boolean); overload;
  end;

Function POP3ServerNameToPath (Const s: String): String;

Type TSSLStatus = (ssOff, ssServer, ssAll);

Var
  CfgHamster : TCfgHamster = nil;
  CfgAccounts: TAccounts   = nil;
  NewsJobs   : TNewsJobs = nil;
  RasDialer  : TRasDialer = nil;
  NewsHistory: TNewsHistoryRecords = nil;
  MailHistory: TMailHistoryRecords = nil;

type
  TGlobalListMarker = ( glTODO, glDONE, glTEST );
  TConfigTyp = (ctAll, ctAutomatics, ctGeneral, ctNews, ctMail, ctUserPW, ctLocalServer);

Function CfgIni: TExtIniFile;

function  GlobalListMarker( Action: TGlobalListMarker ): Boolean;
function  GlobalGroupDesc( GroupName: String ): String;
Function  LoadWindowState( TheForm: TForm; Section: String ): Boolean;
procedure SaveWindowState( TheForm: TForm; Section: String );
procedure ConfigLoad (Typ: TConfigTyp);
procedure ConfigShutdown;

Procedure EditFile(Const s: String);

implementation

uses Global, cPasswordFile, cArtFiles, cArticle, cPCRE, uWinSock, cStdForm,
     ShellAPI, cLogfile, cActions, cFiltersnews, cFiltersmail,
     cMailAlias;

function GlobalListMarker( Action: TGlobalListMarker ): Boolean;
begin
     Result := True;
     EnterCriticalSection( CS_MAINTENANCE );
     case Action of
        glTODO: CfgIni.WriteBool( 'Marker', 'GlobalList', True  );
        glDONE: CfgIni.WriteBool( 'Marker', 'GlobalList', False );
        glTEST: Result:=CfgIni.ReadBool( 'Marker', 'GlobalList', True );
     end;
     LeaveCriticalSection( CS_MAINTENANCE );
end;

function GlobalGroupDesc( GroupName: String ): String;
var  g, d: String;
     i   : Integer;
begin
   Result := '';
   try
      if FileExists2( PATH_SERVER + SRVFILE_ALLDESCS ) then begin
         With TTextReader.Create( PATH_SERVER + SRVFILE_ALLDESCS, 4096 ) do try
            while not EOF do begin
               g := ReadLine;
               i := PosWhSpace( g );
               d := '';
               if i>0 then begin
                  d:=copy(g, i+1, Length(g)-i);
                  g:=copy(g, 1, i-1);
               end;
               if CompareText( g, GroupName )=0 then begin
                  Result := d;
                  break;
               end;
            end
         finally
            free
         end
      end;
   except
      Result := '';
   end;
end;

Function LoadWindowState( TheForm: TForm; Section: String ): Boolean;
var  i: Integer;
begin
//     if not Assigned(CfgIni) then exit;
     Result := true;

     i := CfgIni.ReadInteger( Section, 'Left',  -1 );
     if i<>-1 then TheForm.Left   := i else Result := false;
     i := CfgIni.ReadInteger( Section, 'Top',    -1 );
     if i<>-1 then TheForm.Top    := i else Result := false;
     i := CfgIni.ReadInteger( Section, 'Width',  -1 );
     if i>0 then TheForm.Width  := i else Result := false;
     i := CfgIni.ReadInteger( Section, 'Height', -1 );
     if i>0 then TheForm.Height := i else Result := false;

     Case CfgIni.ReadInteger( Section, 'State', -1 ) of
        1: TheForm.WindowState:=wsMinimized;
        2: TheForm.WindowState:=wsMaximized;
        else TheForm.WindowState:=wsNormal;
     end;
end;

procedure SaveWindowState( TheForm: TForm; Section: String );
begin
   try
     case TheForm.WindowState of
        wsNormal   : CfgIni.WriteInteger( Section, 'State', 0 );
        wsMinimized: CfgIni.WriteInteger( Section, 'State', 1 );
        wsMaximized: CfgIni.WriteInteger( Section, 'State', 2 );
     end;
     if TheForm.WindowState=wsNormal then begin
        CfgIni.WriteInteger( Section, 'Left',   TheForm.Left );
        CfgIni.WriteInteger( Section, 'Top',    TheForm.Top );
        CfgIni.WriteInteger( Section, 'Width',  TheForm.Width );
        CfgIni.WriteInteger( Section, 'Height', TheForm.Height );
     end;
   except
     Log(LOGID_WARN, TrGlF(kLog, 'Warning.WindowState.save',
        'Window-State of %s could not be saved in Hamster.ini', TheForm.Name))
   end
end;

{MG}{SSL}
function SSLCipherString: String;
var s: String;
begin
   s := 'ALL';
   with CfgIni do begin
      if not ReadBool( 'SSL', 'UseExportAlgorithms', False) then s := s + ':!EXP';
      if not ReadBool( 'SSL', 'LowEncryption', False) then s := s + ':!LOW';
      if not ReadBool( 'SSL', 'MediumEncryption', True) then s := s + ':!MEDIUM';
      if not ReadBool( 'SSL', 'HighEncryption', True) then s := s + ':!HIGH';
      if not ReadBool( 'SSL', 'UseADH', False) then s := s + ':!aNULL';
      if not ReadBool( 'SSL', 'UseRC2', True) then s := s + ':!RC2';
      if not ReadBool( 'SSL', 'UseRC4', True) then s := s + ':!RC4';
      if not ReadBool( 'SSL', 'UseIDEA', False) then s := s + ':!IDEA';
      if not ReadBool( 'SSL', 'UseDES', True) then s := s + ':!DES';
      if not ReadBool( 'SSL', 'Use3DES', True) then s := s + ':!3DES';
      if not ReadBool( 'SSL', 'UseAES', True ) then s := s + ':!AES'; {MG}{AES}
      if not ReadBool( 'SSL', 'UseMD5', True) then s := s + ':!MD5';
      if not ReadBool( 'SSL', 'UseSHA1', True) then s := s + ':!SHA';
   end;
   Result := s + ':+SSLv2:+ADH:@STRENGTH';
end;
{/SSL}

{ TConfigList }

procedure TConfigList.LoadFromFile(const FileName: string);
begin
     Clear;
     try
        if FileExists2( Filename ) then inherited LoadFromFile( Filename );
     except
     end;
end;

constructor TConfigList.Create;
begin
     inherited Create;
     Sorted     := True;
     Duplicates := dupIgnore;
end;

function TCfgHamster.GetServerCount: Integer;
begin
   FLock.BeginRead;
   try
      Result := FNntpServers.Count
   finally FLock.EndRead end;
end;

function TCfgHamster.GetServerPath( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
      Result := PATH_SERVER + ServerName[Index] + '\';
   finally FLock.EndRead end;
end;

function TCfgHamster.GetServerName( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
      if Index<0 then Result := ''
                 else Result := ParseString( 0, '', FNntpServers[Index], ',' );
   finally FLock.EndRead end;
end;

function TCfgHamster.GetServerPort( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
      if (Index>=0) and (Index<FNntpServers.Count) then
         Result := ParseString( 1, 'nntp', FNntpServers[Index], ',' )
      else
         Result := '119';
   finally FLock.EndRead end
end;

function TCfgHamster.GetServerPullList( Server: String ): String;
var  i : Integer;
     TS: TStringList;
begin
   TS := TStringList.Create;
   FLock.BeginRead;
   try
      for i:=0 to PullCount-1 do begin
         if PullServer[i]=Server then begin
            TS.Add( PullGroup[i] );
         end;
      end;
      Result := TS.Text
   finally
      FLock.EndRead;
      TS.Free
   end
end;

{MG}{SSL}
function TCfgHamster.GetServerMode( Index: Integer ): Integer;
begin
   With TIniFile.Create( GetServerPath(Index) + SRVFILE_INI ) do try
      Result := ReadInteger ( 'NNTP', 'SSLMode', 0 )
   finally Free end
end;

function TCfgHamster.GetServerVerify( Index: Integer ): Integer;
begin
   With TIniFile.Create( GetServerPath(Index) + SRVFILE_INI ) do try
      Result := ReadInteger ( 'NNTP', 'SSLVerifyLevel', 0 )
   finally Free end
end;

function TCfgHamster.GetServerCaFile( Index: Integer ): String;
begin
   With TIniFile.Create( GetServerPath(Index) + SRVFILE_INI ) do try
      Result := ReadString ( 'NNTP', 'SSLCaFile', '' );
   finally Free end
end;
{/SSL}

function TCfgHamster.GetServerIndexOf( FindServer: String ) : Integer;
var  i: Integer;
begin
   FLock.BeginRead;
   try
      Result := -1;

      i := Pos( ',', FindServer );
      if i>0 then FindServer:=copy(FindServer,1,i-1);

      for i:=0 to ServerCount-1 do begin
         if lowercase(ServerName[i])=lowercase(FindServer) then begin
            Result := i;
            break;
         end;
      end;
   finally FLock.EndRead end;
end;

function TCfgHamster.GetServerUser( Index: Integer ): String;
var  s: String;
begin
   FLock.BeginRead;
   try
     PasswordFile.UsePassword( ServerName[Index], Result, s );
   finally FLock.EndRead end;
end;

function TCfgHamster.GetServerPass( Index: Integer ): String;
var  s: String;
begin
   FLock.BeginRead;
   try
     PasswordFile.UsePassword( ServerName[Index], s, Result );
   finally FLock.EndRead end;
end;

function TCfgHamster.ServerPullThreads( Index: Integer; LimitByJobs: Boolean ): Integer;
var  i: Integer;
begin
  FLock.BeginRead;
  try
     With TIniFile.Create( ServerPath[Index] + SRVFILE_INI ) do try
        Result := ReadInteger( 'Pull', 'Threads', 1 )
     finally
        Free
     end;
     if Result<0 then Result:=0;
     if Result>4 then Result:=4;
     if LimitByJobs then begin
        i := NewsJobs.MaxThreadsFor( ServerName[Index] );
        if Result>i then Result:=i;
     end
  finally FLock.EndRead end;
end;

function TCfgHamster.GetServerIsReadOnly( Server: String ): Boolean;
begin
  FLock.BeginRead;
  try
     With TIniFile.Create( PATH_SERVER + Server + '\' + SRVFILE_INI ) do try
        Result := ReadInteger( 'Setup', 'ReadOnly', 0 ) = 1
     finally Free end
  finally FLock.EndRead end
end;

function TCfgHamster.ServerUseInc( Index: Integer ): Boolean;
var  Idx, Cnt: Integer;
begin
   FLock.BeginRead;
   try
     Idx := FNntpServersInUse.IndexOf( ServerName[Index] );
     if Idx<0 then Idx:=FNntpServersInUse.AddObject( ServerName[Index], Pointer(0) );
     Cnt := LongInt( FNntpServersInUse.Objects[Idx] );

     if Cnt<ServerPullThreads(Index,False) then begin
        Result := True;
        FNntpServersInUse.Objects[Idx] := Pointer( Cnt+1 );
     end else begin
        Result := False;
     end;
   finally FLock.EndRead end;
end;

Function TCfgHamster.ServerUseDec( Index: Integer ): Integer;
var  Idx, Cnt: Integer;
begin
   Result := -1;
   FLock.BeginRead;
   try
     Idx := FNntpServersInUse.IndexOf( ServerName[Index] );
     if Idx>=0 then begin
        Cnt := LongInt( FNntpServersInUse.Objects[Idx] );
        if Cnt>0 then FNntpServersInUse.Objects[Idx] := Pointer( Cnt-1 );
        Result := Integer(FNntpServersInUse.Objects[Idx])
     end;
  finally FLock.EndRead end;
end;

function TCfgHamster.GetActiveCount: Integer;
begin
   FLock.BeginRead;
   try
     Result := FNewsgroups.Count;
   finally FLock.EndRead end;
end;

function TCfgHamster.GetActiveName( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := ParseString( 0, '', FNewsgroups[Index], ',' );
   finally FLock.EndRead end;
end;

function TCfgHamster.GetActivePath( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := PATH_GROUPS + ActiveName[Index] + '\';
   finally FLock.EndRead end;
end;

function TCfgHamster.GetActiveType(Groupname: String): Char;
begin
   FLock.BeginRead;
   try
      With TIniFile.Create( PATH_GROUPS + GroupName + '\data' + EXT_CFG ) do try
         Result := ( ReadString( 'Setup', 'type', '' ) + 'y' ) [1]
      finally
         free
      end
   finally FLock.EndRead end;
end;

function TCfgHamster.GetActiveIndexOf( FindGroup: String ) : Integer;
var  i: Integer;
begin
   FLock.BeginRead;
   try
     Result := -1;
     for i:=0 to ActiveCount-1 do begin
        if lowercase(ActiveName[i])=lowercase(FindGroup) then begin
           Result := i;
           break;
        end;
     end;
   finally FLock.EndRead end;
end;

function TCfgHamster.ActiveAdd(Groupname: String): Boolean;
var  GrpHdl: Integer;
     Desc  : String;
     i: Integer;
begin
   Result := False;
   FLock.BeginWrite;
   try
      if ActiveIndexOf[ Groupname ] >= 0 then begin Result:=True; exit end;

      Groupname := LowerCase(Groupname);
      For i := 1 to Length(Groupname) do If Groupname[i] IN [#0..' '] then Exit;

      FNewsgroups.Add( Groupname );
      FNewsgroups.SaveToFile( PATH_BASE + CFGFILE_ACTIVE );

      GrpHdl := ArticleBase.Open( GroupName ); // open also creates dirs
      if GrpHdl < 0 then exit;
      try
         Desc := GlobalGroupDesc( GroupName );
         // ##### if Desc<>'' then ArticleBase.SetPropStr( GrpHdl, APS_DESCRIPT, APN_DESCRIPT, Desc );
      finally
         ArticleBase.Close( GrpHdl );
      end;

      NewsJobs.GroupPriority[ GroupName ] := MaxInt-4; // = "new group"
      Result := True;

   finally FLock.EndWrite end;
end;

function TCfgHamster.ActiveDel( Groupname: String ): Boolean;
var  Index: Integer;
     Server: String;
begin
   Result := False;
   FLock.BeginWrite;
   try
      Index := ActiveIndexOf[ Groupname ];
      if Index < 0 then exit;

      // remove group files and directory
      if ArticleBase.DeleteGroup( GroupName ) = -1 then exit; // in use

      // remove all pulls
      repeat
         Server := PullServerOf( Groupname );
         if Server <> '' then begin
            if not PullDel( Server, GroupName ) then break;
         end;
      until Server = '';

      // remove from active
      FNewsgroups.Delete( Index );
      FNewsgroups.SaveToFile( PATH_BASE + CFGFILE_ACTIVE );

      Result := True;
      
   finally FLock.EndWrite end;
end;

function TCfgHamster.GetPullCount: Integer;
begin
   FLock.BeginRead;
   try
     Result := FNewsPulls.Count;
   finally FLock.EndRead end;
end;

function TCfgHamster.GetPullServer( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := ParseString( 1, '', FNewsPulls[Index], ',' );
   finally FLock.EndRead end;
end;

function TCfgHamster.GetPullGroup( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := ParseString( 0, '', FNewsPulls[Index], ',' );
   finally FLock.EndRead end;
end;

function TCfgHamster.IsPostServerFor( Const Servername, Groupname: String ): Boolean;
var  i: Integer;
begin
   FLock.BeginRead;
   try
     Result := False;
     if ServerIsReadOnly[Servername] then exit;

     for i:=0 to PullCount-1 do begin
        if lowercase(PullGroup[i])=lowercase(Groupname) then begin
           if lowercase(PullServer[i])=lowercase(Servername) then begin
              Result := True;
              break;
           end;
        end;
     end;
   finally FLock.EndRead end;
end;

function TCfgHamster.ExistPull( Const Servername, Groupname: String ): Boolean;
var  i: Integer;
begin
   FLock.BeginRead;
   try
     Result := False;
     for i:=0 to PullCount-1 do begin
        if lowercase(PullGroup[i])=lowercase(Groupname) then begin
           if lowercase(PullServer[i])=lowercase(Servername) then begin
              Result := True;
              break;
           end;
        end;
     end;
   finally FLock.EndRead end
end;

function TCfgHamster.PullIndexOf( Const Servername, Groupname: String ): Integer;
var  i: Integer;
begin
   FLock.BeginRead;
   try
      Result := -1;
      for i:=0 to PullCount-1 do begin
         if lowercase(PullGroup[i])=lowercase(Groupname) then begin
            if lowercase(PullServer[i])=lowercase(Servername) then begin
               Result := i;
               break;
            end;
         end;
      end;
   finally FLock.EndRead end;
end;

function TCfgHamster.PullServerOf( Const Groupname: String ): String;
var  i: Integer;
begin
   Result := '';
   FLock.BeginRead;
   try
      for i:=0 to PullCount-1 do begin
         if lowercase(PullGroup[i])=lowercase(Groupname) then begin
            Result := PullServer[i];
            break;
         end;
      end;
   finally FLock.EndRead end;
end;

function TCfgHamster.ExistPullServer(Const Groupname: String): Boolean;
begin
   Result := ( PullServerOf( Groupname ) <> '' );
end;

function TCfgHamster.PullAdd(Const Servername, Groupname: String): Boolean;
begin
   FLock.BeginWrite;
   try
      Result := True;
      try
         if ExistPull( Servername, Groupname ) then exit;
         if ServerIndexOf[ Servername ] < 0 then begin Result:=False; exit end;
         if ActiveIndexOf[ Groupname ] < 0 then begin // add group
            if not ActiveAdd( Groupname ) then begin Result:=False; exit end;
         end;
         FNewsPulls.Add( Groupname + ',' + Servername );
         FNewsPulls.SaveToFile( PATH_BASE + CFGFILE_PULLS );
      except
         Result := False;
      end;
   finally FLock.EndWrite end;
end;

function TCfgHamster.PullDel(Const Servername, Groupname: String): Boolean;
var  i: Integer;
begin
   FLock.BeginWrite;
   try
      Result := True;
      try
         i := PullIndexOf( Servername, Groupname );
         if i < 0 then exit;
         FNewsPulls.Delete( i );
         FNewsPulls.SaveToFile( PATH_BASE + CFGFILE_PULLS );

      except
         Result := False;
      end;
   finally FLock.EndWrite end;
end;

function TCfgHamster.GetPop3ServerCount: Integer;
begin
   FLock.BeginRead;
   try
     Result := FPop3Servers.Count;
   finally FLock.EndRead end;
end;

Function POP3ServerNameToPath (Const s: String): String;
begin
   Result := s;
   While pos('/', Result) > 0
      do Result[pos('/', Result)] := '-'
end;

function TCfgHamster.GetPop3ServerPath( Index: Integer ): String;
begin
   Result := PATH_SERVER + POP3ServerNameToPath(Pop3ServerName[Index]) + '\';
end;

function TCfgHamster.GetPop3ServerName( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := ParseString( 0, '', FPop3Servers[Index], ',' )
   finally FLock.EndRead end
end;

function TCfgHamster.GetPop3ServerPort( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := ParseString( 1, 'pop3', FPop3Servers[Index], ',' )
   finally FLock.EndRead end
end;

function TCfgHamster.GetPop3ServerUser( Index: Integer ): String;
var  s: String;
begin
   FLock.BeginRead;
   try
     PasswordFile.UsePassword( Pop3ServerName[Index], Result, s )
   finally FLock.EndRead end
end;

function TCfgHamster.GetPop3ServerPass( Index: Integer ): String;
var  s: String;
begin
   FLock.BeginRead;
   try
     PasswordFile.UsePassword( Pop3ServerName[Index], s, Result )
   finally FLock.EndRead end
end;

function TCfgHamster.GetPop3ServerLocalUser( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
      With TIniFile.Create( POP3ServerPath[Index] + SRVFILE_INI ) do try
         Result := ReadString ('POP3', 'LocalUser', Def_Postmaster)
      finally free end
   finally FLock.EndRead end
end;

function TCfgHamster.GetPop3ServerFiltersection( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
      With TIniFile.Create ( POP3ServerPath[Index] + SRVFILE_INI ) do try
         Result := ReadString ('POP3', 'FilterSection', '*')
      finally free end
   finally FLock.EndRead end
end;

{MG}{SSL}
function TCfgHamster.GetPop3ServerMode( Index: Integer ): Integer;
var  SrvIni : TIniFile;
begin
     SrvIni := TIniFile.Create( Pop3ServerPath[Index] + SRVFILE_INI );
     Result := SrvIni.ReadInteger ( 'POP3', 'SSLMode', 0);
     SrvIni.Free;
end;

function TCfgHamster.GetPop3ServerVerify( Index: Integer ): Integer;
var  SrvIni : TIniFile;
begin
     SrvIni := TIniFile.Create( Pop3ServerPath[Index] + SRVFILE_INI );
     Result := SrvIni.ReadInteger ( 'POP3', 'SSLVerifyLevel', 0 );
     SrvIni.Free;
end;

function TCfgHamster.GetPop3ServerCaFile( Index: Integer ): String;
var  SrvIni : TIniFile;
begin
     SrvIni := TIniFile.Create( Pop3ServerPath[Index] + SRVFILE_INI );
     Result := SrvIni.ReadString ( 'POP3', 'SSLCaFile', '' );
     SrvIni.Free;
end;
{/SSL}

function TCfgHamster.GetPop3ServerIndexOf( FindServer: String ) : Integer;
var  i: Integer;
begin
   FLock.BeginRead;
   try
     Result := -1;

     i := Pos( ',', FindServer );
     if i>0 then FindServer:=copy(FindServer,1,i-1);
     if FindServer='' then exit;

     for i:=0 to Pop3ServerCount-1 do begin
        if lowercase(Pop3ServerName[i])=lowercase(FindServer) then begin
           Result := i;
           break;
        end;
     end
   finally FLock.EndRead end
end;

function TCfgHamster.GetSmtpServerCount: Integer;
begin
   FLock.BeginRead;
   try
     Result := FSmtpServers.Count;
   finally FLock.EndRead end
end;

function TCfgHamster.GetSmtpServerPath( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := PATH_SERVER + SmtpServerName[Index] + '\';
   finally FLock.EndRead end
end;

function TCfgHamster.GetSmtpServerName( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     if (Index<0) or (Index>=FSmtpServers.Count) then exit;
     Result := ParseString( 0, '', FSmtpServers[Index], ',' );
   finally FLock.EndRead end
end;

function TCfgHamster.GetSmtpServerPort( Index: Integer ): String;
begin
   FLock.BeginRead;
   try
     Result := ParseString( 1, 'smtp', FSmtpServers[Index], ',' );
   finally FLock.EndRead end
end;

function TCfgHamster.GetSmtpServerUser( Index: Integer ): String;
var  s: String;
begin
   FLock.BeginRead;
   try
     PasswordFile.UsePassword( SmtpServerName[Index], Result, s );
   finally FLock.EndRead end
end;

function TCfgHamster.GetSmtpServerPass( Index: Integer ): String;
var  s: String;
begin
   FLock.BeginRead;
   try
     PasswordFile.UsePassword( SmtpServerName[Index], s, Result );
   finally FLock.EndRead end
end;

{MG}{SSL}
function TCfgHamster.GetSmtpServerMode( Index: Integer ): Integer;
var  SrvIni : TIniFile;
begin
     SrvIni := TIniFile.Create( SmtpServerPath[Index] + SRVFILE_INI );
     Result := SrvIni.ReadInteger ( 'SMTP', 'SSLMode', 0);
     SrvIni.Free;
end;

function TCfgHamster.GetSmtpServerVerify( Index: Integer ): Integer;
var  SrvIni : TIniFile;
begin
     SrvIni := TIniFile.Create( SmtpServerPath[Index] + SRVFILE_INI );
     Result := SrvIni.ReadInteger ( 'SMTP', 'SSLVerifyLevel', 0 );
     SrvIni.Free;
end;

function TCfgHamster.GetSmtpServerCaFile( Index: Integer ): String;
var  SrvIni : TIniFile;
begin
     SrvIni := TIniFile.Create( SmtpServerPath[Index] + SRVFILE_INI );
     Result := SrvIni.ReadString ( 'SMTP', 'SSLCaFile', '' );
     SrvIni.Free;
end;
{/SSL}

function TCfgHamster.GetSmtpServerIndexOf( FindServer: String ) : Integer;
var  i: Integer;
begin
   FLock.BeginRead;
   try
     Result := -1;

     i := Pos( ',', FindServer );
     if i>0 then FindServer:=copy(FindServer,1,i-1);
     if FindServer='' then exit;

     for i:=0 to SmtpServerCount-1 do begin
        if lowercase(SmtpServerName[i])=lowercase(FindServer) then begin
           Result := i;
           break;
        end;
     end;
   finally FLock.EndRead end
end;

function TCfgHamster.GetUniqueMsgFilename( DestPath, Which: String ): String;
var  i, LfdNo: Integer;
     SR      : TSearchRec;
begin
   FLock.BeginRead;
   try
      LfdNo := CfgIni.ReadInteger( 'Setup', Which + '.number.lastused', 0 );
      inc( LfdNo );
      repeat
         Result := DestPath + inttostr(LfdNo) + '.'
           + CfgIni.ReadString( 'Setup', Which + '.ext.in', 'msg' );
         i := SysUtils.FindFirst( Result, faAnyFile, SR );
         SysUtils.FindClose( SR );
         if i<>0 then break;
         inc( LfdNo );
         if LfdNo>=99999999 then LfdNo:=1;
      until False;
      CfgIni.WriteInteger( 'Setup', Which + '.number.lastused', LfdNo );
   finally
      FLock.EndRead
   end;
end;

procedure TCfgHamster.ReloadActiveList;
var  OldCount, i: Integer;
begin
   FLock.BeginWrite;
   try
     FNewsgroups.Clear;
     FNewsgroups.LoadFromFile( PATH_BASE + CFGFILE_ACTIVE );
     OldCount := FNewsgroups.Count;

     // make sure, that internal-groups are in the list
     for i:=INTERNALGROUP_DEFAULT to INTERNALGROUP_LASTGROUP do begin
        FNewsgroups.Add( INTERNALGROUP[i] );
     end;
     if (OldCount<>FNewsgroups.Count) and (Not ArchivMode)
        then FNewsgroups.SaveToFile( PATH_BASE + CFGFILE_ACTIVE );
   finally FLock.EndWrite end;
end;

procedure TCfgHamster.ReloadPullList;
begin
   FLock.BeginWrite;
   try
     FNewsPulls.Clear;
     FNewsPulls.LoadFromFile( PATH_BASE + CFGFILE_PULLS );
   finally FLock.EndWrite end;
end;

procedure TCfgHamster.ReloadConfiguration;
var  i: Integer;
begin
   FLock.BeginWrite;
   try
    If Not ArchivMode then begin
       FNntpServers.LoadFromFile( PATH_BASE + CFGFILE_SERVER_NNTP );
       for i:=0 to ServerCount-1 do If Not DirExists2 (ServerPath[i]) then begin
          ForceDirectories( ServerPath[i] )
       end
    end;

    ReloadActiveList;
    If Not ArchivMode then for i:=0 to ActiveCount-1 do If Not DirExists2(ActivePath[i]) then begin
       ForceDirectories( ActivePath[i] )
    end;
    ReloadPullList;

    If Not ArchivMode then begin
       FPop3Servers.LoadFromFile( PATH_BASE + CFGFILE_SERVER_POP3 );
       for i:=0 to Pop3ServerCount-1 do If Not DirExists2( Pop3ServerPath[i]) then begin
          ForceDirectories( Pop3ServerPath[i] )
       end;
       ForceDirectories( PATH_MAILS + 'admin' );
       FSmtpServers.LoadFromFile( PATH_BASE + CFGFILE_SERVER_SMTP );
       for i:=0 to SmtpServerCount-1 do ForceDirectories( SmtpServerPath[i] )
    end
   finally FLock.EndWrite end;
end;

constructor TCfgHamster.Create;
begin
     inherited Create;

     FLock := TReaderWriterLock.Create;
     
     FNntpServers     := TConfigList.Create;
     FNntpServersInUse:= TConfigList.Create;
     FNewsgroups      := TConfigList.Create;
     FNewsPulls       := TConfigList.Create;
     FPop3Servers     := TConfigList.Create;
     FSmtpServers     := TConfigList.Create;

     ReloadConfiguration;
end;

destructor TCfgHamster.Destroy;
begin
   FLock.BeginWrite;
   try
      FNntpServers.Free;
      FNntpServersInUse.Free;
      FNewsgroups.Free;
      FNewsPulls.Free;
      FPop3Servers.Free;
      FSmtpServers.Free;
   finally
      FLock.Free;
   end;
   inherited Destroy;
end;

// ----------------------------------------------------------- TRasDialer -----

Var isDialing: Boolean = false;

function TRasDialer.Dial( ConnectionID, UserName, Password: String ): Integer;
var  i: Integer;
     u, p: String;
begin
   Result := -1;

   If isDialing then begin
      Log( LOGID_WARN, TrGl(kLog, 'Warning.RasDial.IsDialing',
            'RAS-Dial aborted - You are already dialing!' ) );
      exit;
   end;

   Actions.Exec ( atDUNBeforeDial, ConnectionID, '' );

   UserName:=Trim(UserName);
   Password:=Trim(Password);
   FLastError := -1;
   try
     isDialing := true;
     try
        if not RasDynInit then begin
           Log( LOGID_ERROR, TrGl(kLog, 'Error.RasDial.RASnotInstalled',
              'Ras.Dial failed: RAS not installed.' ) );
           exit;
        end;

        if OurRasConnection<>0 then Hangup;

        ConnID := ConnectionID;
        OurRasConnection := 0;
        OurRasDialParams.dwSize           := sizeof( OurRasDialParams );
        strpcopy( OurRasDialParams.szEntryName, ConnectionID );
        OurRasDialParams.szPhoneNumber    := '';
        OurRasDialParams.szCallbackNumber := '';

        if (UserName='') or (Password='') then begin
           if not PasswordFile.UsePassword( ConnectionID, u, p ) then exit;
           if UserName='' then UserName:=u;
           if Password='' then Password:=p;
        end;
        strpcopy( OurRasDialParams.szUserName, UserName );
        strpcopy( OurRasDialParams.szPassword, Password );

        OurRasDialParams.szDomain         := '';

        Log( LOGID_INFO, TrGl(kLog, 'Info.Dialing', 'Dialing ...') );
        i := RasDynDial( nil, // pointer to function extensions data
                         nil, // pointer to full path and filename of phonebook file
                         OurRasDialParams, // pointer to calling parameters data
                         0, // specifies type of RasDial event handler
                         nil, // specifies a handler for RasDial events
                         OurRasConnection ); // pointer to variable to receive connection handle
        {JW} {critical event}
        EnterCriticalSection(CS_Event);
        SetEvent(EventRASConnected);
        LeaveCriticalSection(CS_Event);
        {JW}

        if i=0 then begin
           FLastError := 0;
           OurRasConnStatus.dwSize  := sizeof( OurRasConnStatus );
           OurRasConnStatus.dwError := 0;
           repeat
              i := RasDynGetConnectStatus( OurRasConnection, OurRasConnStatus );
              if i<>0 then begin
                 FLastError := i;
                 Result := i;
                 Log( LOGID_ERROR, TrGlF (kLog, 'Error.RasDial.ErrorAonDialing',
                    'Error.A %s on dialing RAS-connection.', [inttostr(i)] ) );
                 RasDynHangup( OurRasConnection );
                 OurRasConnection := 0;
                break;
              end;
              Log( LOGID_DEBUG, TrGl(kLog, 'Debug.RasDial.ConnState',
                 'RasConnState') + '=' + inttostr(OurRasConnStatus.rasconnstate) );
              if OurRasConnStatus.rasconnstate=RASCS_Connected then begin
                 Result := 0;
                 break;
              end;
              if OurRasConnStatus.dwError<>0 then begin
                 FLastError := OurRasConnStatus.dwError;
                 Result := OurRasConnStatus.dwError;
                 Log( LOGID_ERROR, TrGlF (kLog, 'Error.RasDial.ErrorBonDialing',
                    'Error.A %s on dialing RAS-connection.', [inttostr(OurRasConnStatus.dwError)] ) );
                 RasDynHangup( OurRasConnection );
                 OurRasConnection := 0;
                 break;
              end;
              Sleep( 250 ); //???
           until False;

           // stat-start
           StatTime := Now;
           StatByte := GetCounter(CntByteIn) + GetCounter(CntByteOut);
           StatArts := GetCounter(CntArtLoad);
           StatMails:= GetCounter(CntMailNew);

           sDynamicIP := RasDynGetPppIp( OurRasConnection );
           if sDynamicIP='' then begin
              lDynamicIP := 0;
           end else begin
              lDynamicIP := inet_addr( PChar(sDynamicIP) );
              Log( LOGID_INFO, TrGlF(kLog, 'Info.RAS_PPP-IP', 'RAS PPP-IP = %s (%s)',
                               [sDynamicIP, LookupHostName(lDynamicIP) ] ));
           end;

           Actions.Exec ( atDUNConnected, IntToStr(lDynamicIP), '' );

        end else begin
           // #676 bei besetzt
           FLastError := i;
           Result := i;
           Log( LOGID_ERROR, TrGlF(kLog, 'Error.RASDial.ErrorOnDialing',
              'Error %s dialing RAS-connection.', [inttostr(i)]));
           u := RasDynErrTxt( i );
           if u<>'' then
              Log( LOGID_ERROR, TrGl(kLog, 'Error.RASDial.Error', 'RAS-Error')
                 + ' ' + inttostr(i) + ': ' + u );
           if OurRasConnection<>0 then RasDynHangup( OurRasConnection );
           OurRasConnection := 0;
           Actions.Exec ( atDUNDialFailed, ConnectionID, '' );

        end;
     except
        on E:Exception do begin
           if Result=0 then Result:=-2;
           if FLastError=0 then FLastError:=-2;
           Log( LOGID_ERROR, TrGlF(kLog, 'RasDynDial.Exception', 'RasDynDial-Exception: %s', E.Message) );
        end;
     end
   finally
     isDialing := false
   end
end;

function TRasDialer.GetDynamicIP: LongInt;
begin
     Result := lDynamicIP;
     if Result=0 then exit;
     if OurRasConnection=0 then begin Result:=0; exit; end;
     if RasDynGetConnectStatus( OurRasConnection, OurRasConnStatus )=0 then begin
        if OurRasConnStatus.rasconnstate<>RASCS_Connected then begin
           // not connected any more
           lDynamicIP := 0;
           Result := 0;
        end;
     end;
end;

function TRasDialer.HangUp: Integer;
const DT2SEC : Double = (24.0*60.0*60.0);
var  i, HangupRasConn: Integer;
     s: String;
begin
     Result     := -1;
     sDynamicIP := '';
     lDynamicIP := 0;
     if not RasDynInit then exit;

     Result     := 0;
     if OurRasConnection<>0 then begin
        try
           // stat-end
           StatTime := ( Now - StatTime ) * DT2SEC;
           if StatTime<=0.01 then StatTime:=0.01;
           StatByte  := GetCounter(CntByteIn) + GetCounter(CntByteOut) - StatByte;
           StatArts  := GetCounter(CntArtLoad) - StatArts;
           StatMails := GetCounter(CntMailNew) - StatMails;
           s := '';
           s := s + ' '+TrGl(kLog, 'Info.ConnectionStats.Secs', 'Secs')+'='  + FormatFloat('0',StatTime);
           s := s + ' '+TrGl(kLog, 'Info.ConnectionStats.Mails', 'Mails')+'=' + inttostr(StatMails);
           s := s + ' '+TrGl(kLog, 'Info.ConnectionStats.Arts', 'Arts')+'='  + inttostr(StatArts);
           s := s + ' '+TrGl(kLog, 'Info.ConnectionStats.Art/Secs', 'A/s')+'='   + FormatFloat('0.0',(StatArts+StatMails)/StatTime);
           s := s + ' '+TrGl(kLog, 'Info.ConnectionStats.Byte', 'Bytes')+'='  + inttostr(StatByte);
           s := s + ' '+TrGl(kLog, 'Info.ConnectionStats.Byte/Secs', 'B/s')+'='   + FormatFloat('0',StatByte/StatTime);
           s := s + ' '+TrGl(kLog, 'Info.ConnectionStats.RAS', 'RAS')+'='   + ConnID;
           Log( LOGID_INFO, TrGl(kLog, 'Info.ConnectionStats', 'Connection-stats')+':'+ s );
           HamFileAppendLine ( PATH_LOGS + 'RasDial.log',
                               FormatDateTime( 'yyyy/mm/dd hh:nn:ss', Now ) + s );
        except
        end;
     end;

     // hangup
     HangupRasConn    := OurRasConnection;
     if HangupRasConn=0 then HangupRasConn := RasDynGetConn; 
     OurRasConnection := 0;
     {JW} {critical event}
     EnterCriticalSection(CS_Event);
     PulseEvent(EventRASHangup);
     ResetEvent(EventRASConnected);
     LeaveCriticalSection(CS_Event);
     {JW}
     if HangupRasConn<>0 then begin
        try
           i := RasDynHangup( HangupRasConn );
           if i=0 then begin
              Log( LOGID_INFO, TrGl(kLog, 'Info.RAS.closed', 'RAS-connection closed.') );
           end else begin
              FLastError := i;
              Result := i;
              {JW} {RAS Warn}
              if  i=6 then begin
                Log( LOGID_WARN, TrGl(kLog, 'RAS.Warning.Invalidhandlebyclosing',
                     'Warning invalid handle by closing RAS-connection.') );
                Log( LOGID_WARN, TrGl(kLog, 'RAS.Warning.InvalidhandleAlreadyOpen',
                     'RAS-connection opened by hamster is already closed by other process' ) )
              end else
                Log( LOGID_ERROR, TrGlF(kLog, 'Ras.Error.Closing',
                     'Error %s closing RAS-connection.', inttostr(i)) )
             {/JW}
           end;
        except
           on E:Exception do begin
              if Result=0 then Result:=-2;
              if FLastError=0 then FLastError:=-2;
              Log( LOGID_ERROR, TrGlF(kLog, 'RAS.DynHangup.Exception',
                   'RasDynHangup-Exception: %s', E.Message ) )
           end;
        end;
     end else begin
        Log( LOGID_INFO, TrGl(kLog, 'Info.RASHangupIgnoredNoConnection',
           'RAS-hangup ignored; no connection'));
     end;

     Actions.Exec(atDUNHangup, '', '')
end;

function TRasDialer.IsConnected: Boolean;
begin
     Result := ( OurRasConnection <> 0 );
end;

function TRasDialer.GetConnectionList( RasList: TStringList ): Boolean;
begin
   Result := RasDynEnumPhonebookEntries( RasList );
end;

function TRasDialer.IsInstalled: Boolean;
begin
   RasDynInit;
   Result := ( RasDynDll <> 0 );
end;

constructor TRasDialer.Create;
begin
     inherited Create;
     // Hier noch kein RasDynInit, da Object global verwendet wird.
     OurRasConnection := 0;
     FLastError := 0;
     sDynamicIP := '';
     lDynamicIP := 0;
end;

destructor TRasDialer.Destroy;
begin
   // if OurRasConnection<>0 then // Hangup?
   inherited Destroy;
end;

// ----------------------------------------------------------------------------

procedure ConfigLoad (Typ: TConfigTyp);
var  LfdArt, i: Integer;
     s : String;
     GrpNew: Boolean;
begin
   Log( LOGID_INFO,  TrGl(kLog, 'Info.Loading.Config', 'Loading configuration ...') );

   // (re-) init CfgHamster
   if CfgHamster=nil then begin
      CfgHamster := TCfgHamster.Create;
      ArticleBase := TArticleBase.Create;
      Def_HistoryChunkBits := CfgIni.ReadInteger( 'Setup', 'history.chunkbits', Def_HistoryChunkBits );
      NewsHistory := TNewsHistoryRecords.Create( PATH_GROUPS, Def_HistoryChunkBits );
      MailHistory := TMailHistoryRecords.Create( PATH_MAILS );
      PasswordFile := TPasswordFile.Create;
      CfgAccounts := TAccounts.Create;
   end else begin
      CfgHamster.ReloadConfiguration;
   end;

   // init NewsJobCoordinator
   if NewsJobs=nil then begin
      NewsJobs := TNewsJobs.Create;
   end;

   // RasDialer initialisieren
   if RasDialer=nil then begin
      RasDialer := TRasDialer.Create;
      if not RasDialer.IsInstalled then Log( LOGID_INFO,
         TrGl(kLog, 'RAS.not_installed', 'Note: DUN/RAS is not installed (properly) - '
                  + 'Dialer functions will not work.' ) );
   end;

   // read settings

   With CfgIni do begin

      If Typ = ctAll then begin
         IRead('Setup', 'AdvancedConfiguration', Def_AdvancedConfiguration);
         IRead('Setup', 'local.nntp.checkpath', Def_Check_Path); {JW} {Check Path}
         IRead('Setup', 'local.nntp.feedjunk', Def_Junkfeeder); {JW junk feeder}
         IRead('Setup', 'MaxUnknownGroupsInRe', Def_MaxUnknownGroupsInRe);
         If Def_MaxUnknownGroupsInRe > 10 then Def_MaxUnknownGroupsInRe := 2;
         IRead('Setup', 'local.nntp.XOVERdelCRLF', Def_News_XOverWithoutCRLF);
         IRead( 'Setup', 'main.hidemenuitems', Def_HideMenuItems );
         IRead ( 'Setup', 'news2mail.MIDExtension', Def_GatewayMIDExtension );
         IRead ( 'Setup', 'news.scorefile.expiredentries.delete', Def_News_ExpiredFilterentries_Delete );
         IRead ( 'Setup', 'mail.scorefile.expiredentries.delete', Def_Mail_ExpiredFilterentries_Delete );
         ReloadActions
      end;

      If Typ IN[ctAll, ctGeneral] then begin
         LogFile.LoadINISettings;
         IRead ( 'Setup', 'log.file.refresh', LOGFILEREFRESH );
         IRead ( 'Setup', 'editor.app',    Def_EditorApp );
         IRead ( 'Setup', 'editor.params', Def_EditorParams );
         IRead ( 'Setup', 'remote.timeout.connect', Def_RemoteTimeoutConnect );
         IRead ( 'Setup', 'remote.timeout.command', Def_RemoteTimeoutCommand );
         IRead ( 'Setup', 'MaxTasks', Def_MaxTasks);
         IRead ( 'Main', 'ColoredTabs', ColoredTabs );
         IRead ( 'Main', 'TabColorErrorText', TabColorErrorText );
         IRead ( 'Main', 'TabColorErrorBrush', TabColorErrorBrush );
         IRead ( 'Main', 'TabColorWarningText', TabColorWarningText );
         IRead ( 'Main', 'TabColorWarningBrush', TabColorWarningBrush );
         IRead ( 'Setup', 'main.minimizeonclose', Def_MinimizeOnClose );
         IRead ( 'Setup', 'main.autoactivateerrorlog', Def_AutoActivateErrorLog );
         IRead ( 'Setup', 'local.codepage', Def_Codepage);
         {MG}{SSL}
         Def_SSLCipherString := SSLCipherString;
         IRead( 'SSL', 'UseSSLv3', Def_UseSSLv3);
         IRead( 'SSL', 'UseTLSv1', Def_UseTLSv1);
         {/SSL}
      end;

      If Typ IN[ctAll, ctAutomatics] then begin
         IRead ( 'Setup', 'purge.daily', Def_Purge_Daily );
         IRead ( 'Setup', 'FilterNewGroupsInfo', Def_FilterNewGroupsInfo);
      end;

      If Typ IN[ctAll, ctLocalServer] then begin
         IRead ( 'Setup', 'local.postmaster', Def_Postmaster);  //JW //Postmaster
         IRead ( 'Setup', 'local.usenet', Def_Usenet); {JW} {NNTPINFOMAIL}
         IRead ( 'Setup', 'local.nntp.infomail', Def_NNTPINFOMAIL); {JW} {NNTPINFOMAIL}
         IRead ( 'Setup', 'pop3.delay', Def_POP3Delay); {JW} {POP3 delay}
         IRead ( 'Setup', 'IMAP.delay',            Def_IMAPDelay); //IMAP
         IRead ( 'Setup', 'smtp.delay', Def_SMTPDelay); {JW} {SMTP delay}
         IRead ( 'Setup', 'MaxLocalNNTPServers',Def_Max_Local_NNTP_Servers); {JW} {LocalLimit}
         IRead ( 'Setup', 'MaxLocalPOP3Servers',Def_Max_Local_POP3_Servers); {JW} {LocalLimit}
         IRead ( 'Setup', 'MaxLocalIMAPServers',   Def_Max_Local_IMAP_Servers); //IMAP
         IRead ( 'Setup', 'MaxLocalSMTPServers',Def_Max_Local_SMTP_Servers); {JW} {LocalLimit}
         IRead ( 'Setup', 'FQDN', Def_FQDNforMIDs ); //JW //MID2
         IRead ( 'Setup', 'local.mail.LocalMIDFQDN', Def_MID_FQDN_local );{JW} {FQDN_MID_Local}
         IRead ( 'Setup', 'FQDN2', Def_FQDN ); //JW //MID2
         IRead ( 'Setup', 'local.port.nntp', Def_LocalNntpServer_Port );
         IRead ( 'Setup', 'local.port.pop3', Def_LocalPop3Server_Port );
         IRead ( 'Setup', 'local.port.IMAP',       Def_LocalIMAPServer_Port );//IMAP
         IRead ( 'Setup', 'local.port.smtp', Def_LocalSmtpServer_Port );
         IRead ( 'Setup', 'local.mail.islocaldomain', Def_IsLocalDomain );
         IRead ( 'Setup', 'local.authreq.smtp', Def_LocalAuthReqSmtp );
         IRead ( 'Setup', 'local.authreq.esmtp', Def_LocalAuthReqESmtp );
         IRead ( 'Setup', 'local.mail.reqnotauth', Def_LocalMailReqNotAuth);
         IRead ( 'Setup', 'local.authreq.closeiffalse', Def_LocalFalseAuthCloseConnection );
         Def_LocalSmtpSASL := SMTPSupportedSASL; IRead ( 'Setup', 'local.smtp.sasl', Def_LocalSmtpSASL );
         Def_LocalPOP3SASL := POP3SupportedSASL; IRead ( 'Setup', 'local.pop3.sasl',Def_LocalPOP3SASL ); {JW} {SASL}
         Def_LocalRECOSASL := ReCoSupportedSASL; IRead ( 'Setup','local.reco.sasl', Def_LocalRECOSASL );
         IRead ( 'Setup', 'DefaultEnvelopeFrom',Def_EnvelopeFrom);
         If Def_EnvelopeFrom = '' then Def_EnvelopeFrom := 'admin';
         IRead ( 'Setup', 'SendInfoMailLocalOnly', Def_SendInfoMailLocalOnly ); //JW //LOCAL_ONLY_ADMIN
         IRead ( 'Setup', 'purge.articles.keepdays', Def_Purge_Articles_KeepDays );
         IRead ( 'Setup', 'purge.history.keepdays',  Def_Purge_History_KeepDays );
         IRead ( 'Setup', 'purge.kills.keepdays',    Def_Purge_Kills_KeepDays );
         IRead ( 'Setup', 'purge.mhistory.keepdays', Def_Purge_MHistory_KeepDays );
         IRead ( 'Setup', 'mail.addreceived', Def_Mail_AddReceived );
         IRead ( 'Setup', 'pop3.addreceived', Def_POP3_AddReceived); {JW} {Def_POP3_AddReceived}
         If Def_FQDN = '' then Def_POP3_AddReceived := false;
         IRead ( 'Setup', 'mail.addxhamster', Def_Mail_AddXHamster );
         IRead ( 'Setup', 'local.mail.BounceIfUnknownUser', Def_BounceMailToSender);
         IRead ( 'Setup', 'local.timeout.inactivity', Def_LocalTimeoutInactivity ); {JW}   {LocalTimeoutInactivity}
         If Def_LocalTimeoutInactivity>35791 then Def_LocalTimeoutInactivity:=35791; {JW}   {LocalTimeoutInactivity}
         IRead ( 'Setup', 'local.NNTP.timeout.login', Def_LocalNNTPLoginTimeout ); {JW} {Login}
         IRead ( 'Setup', 'local.POP3.timeout.login', Def_LocalPOP3LoginTimeout ); {JW} {Login}
         IRead ( 'Setup', 'local.IMAP.timeout.login', Def_LocalIMAPLoginTimeout ); //IMAP
         IRead ( 'Setup', 'local.SMTP.timeout.login', Def_LocalSMTPLoginTimeout ); {JW} {Login}
         IRead ( 'Setup', 'local.ReCo.timeout.login', Def_LocalReCoLoginTimeout ); {JW} {Login}
         IRead ( 'Setup', 'local.timeout.quitdelay',  Def_LocalTimeoutQuitDelay );
         IRead ( 'Setup', 'local.nntp.posttounknown', Def_LocalNntpPostToUnknown );
         IRead ( 'Setup', 'local.nntp.SearchUnknownMIDs', Def_LocalNntpSearchUnknownMIDs );
         IRead ( 'Setup', 'generate.MID', Def_GenerateNewsMID );
         IRead ( 'Setup', 'news.generatedate',  Def_GenerateNewsDate );
         IRead ( 'SETUP', 'AddUserAgent', Def_News_AddUserAgent);
         IRead ( 'Setup', 'news.addxhamster', Def_News_AddXHamster );
         If Def_News_AddUserAgent then Def_News_AddXHamster := false;
         IRead ( 'Setup', 'news.addpath', Def_News_AddPath); {JW} {li}
         IRead ( 'Setup', 'news.localinjection',Def_News_LocalInjection );
         If (Def_FQDN = '') or (Def_FQDNforMIDs = '') or (Def_GenerateNewsMID<>1) or (Not Def_News_AddPath) then begin
            Def_News_LocalInjection := false
         end;
         IRead ( 'Setup','local.List.require.auth',Def_ListRequireAuth);
         IRead ( 'Setup', 'local.limit.linelen.nntp',  Def_LocalLimitLineLenNntp  );
         IRead ( 'Setup', 'local.limit.textsize.nntp', Def_LocalLimitTextSizeNntp );
         IRead ( 'Setup', 'local.limit.linelen.pop3',  Def_LocalLimitLineLenPop3  );
         IRead ( 'Setup', 'local.limit.textsize.pop3', Def_LocalLimitTextSizePop3 );
         IRead ( 'Setup', 'local.limit.textsize.IMAP', Def_LocalLimitTextSizeIMAP ); //IMAP
         IRead ( 'Setup', 'local.limit.linelen.smtp',  Def_LocalLimitLineLenSmtp  );
         IRead ( 'Setup', 'local.limit.textsize.smtp', Def_LocalLimitTextSizeSmtp );
         IRead ( 'Setup', 'mid.crypt', Def_crypt_mid ); //JW {MID-Crypt}
         If Def_FQDNforMIDs = '' then Def_crypt_mid := false;
         IRead ( 'Setup', 'score.loglimit', Def_ScoreLogLimit );
         IRead ( 'Setup', 'local.nntp.serverbind', Def_LocalNNTPServerBind);
         IRead ( 'Setup', 'local.pop3.serverbind', Def_LocalPOP3ServerBind);
         IRead ( 'Setup', 'local.IMAP.serverbind', Def_LocalIMAPServerBind); //IMAP
         IRead ( 'Setup', 'local.smtp.serverbind', Def_LocalSMTPServerBind);
         IRead ( 'Setup', 'local.reco.serverbind', Def_LocalReCoServerBind);
         IRead ( 'Setup', 'local.nntp.tlsmode', Def_LocalNntpTlsMode );
         IRead ( 'Setup', 'local.pop3.tlsmode', Def_LocalPop3TlsMode );
         IRead ( 'Setup', 'local.IMAP.tlsmode',    Def_LocalIMAPTlsMode ); //IMAP
         IRead ( 'Setup', 'local.smtp.tlsmode', Def_LocalSmtpTlsMode );
         IRead ( 'Setup', 'local.reco.usetls', Def_LocalRecoUseTls );
         IRead ( 'Setup', 'IMAP.SASL.LOGINDisable', Def_IMAP_DisableSASLLogin); //IMAP //JW //IMAP-AUTH
         IRead ( 'Setup', 'IMAP.LOGINDisable', Def_IMAP_DisableLogin); //IMAP
         IRead ( 'Setup', 'news.feededcancel', Def_News_FeededCancel );
         IRead ( 'Setup', 'news.feededsupersedes', Def_News_FeededSupersedes );
         IRead ( 'Setup', 'news.feededcancelverify', Def_News_FeededCancelVerify );
         IRead ( 'Setup', 'news.feededcancelcontrolmsg', Def_News_FeededCancelControlMsg);
         IRead ( 'Setup', 'IMAP.NcBrain', Def_IMAPNCBrain); //NCBrain
         IRead ( 'Setup', 'local.imap.id', Def_IMAP_ID); //JW //IMAP ID
         IRead ( 'Setup', 'imap.7bit', Def_IMAP8bitFilter); //JW //IMAP 7Bit
      end;

      If Typ IN[ctAll, ctUserPW] then begin
         IRead ( 'Setup', 'password.codebase', Def_PasswordCodeBase );
         IRead ( 'Setup', 'MaxScriptPWNumber', Def_MaxScriptPWNumber );
         MailAlias
      end;

      If Typ IN[ctAll, ctNews] then begin
         IRead ( 'Setup', 'preferred.postserver', Def_PostServer );
         IRead ( 'Setup', 'pull.limit', Def_Pull_Limit );
         IRead ( 'Setup', 'pull.limit.first', Def_Pull_Limit_First );
         IRead ( 'Setup', 'news.makeparts',Def_parts_make);
         IRead ( 'Setup', 'news.sizeofparts.min',Def_parts_size_Min);  // default block size of parted pulls
         IRead ( 'Setup', 'news.sizeofparts.max',Def_parts_size_Max);  // max. block size of parted pulls (single/last one)
         IRead ( 'Setup', 'nntp.ModeReader', Def_NNTPClient_ModeReader); {JW} {Feed}
         IRead ( 'Setup', 'ihave.auth', Def_IHAVE_Auth ); {JW} {Peer}
         IRead ( 'Setup', 'stream.mode', Def_Stream_Mode ); {JW} {Peer}
         IRead ( 'Setup', 'news.addxhtrace',  Def_News_AddXHTrace );
         if Def_News_AddXHTrace>'' then begin
            if Def_News_AddXHTrace[1]<>'X' then Def_News_AddXHTrace:='';
            s := '';
            for i:=1 to length(Def_News_AddXHTrace) do begin
               if Def_News_AddXHTrace[i] in ['A'..'Z','a'..'z','-'] then
                  s := s + Def_News_AddXHTrace[i];
            end;
            if s<>Def_News_AddXHTrace then Def_News_AddXHTrace:='';
            if Def_News_AddXHTrace='' then begin
               Log( LOGID_WARN, TrGl(kLog, 'Ini.NewsAddxhtrace.invalid',
                    'Invalid news.addxhtrace-setting in Hamster.ini!') )
            end
         end;
         IRead ( 'Setup', 'nntp.autogetserverinfos', Def_NNTPAutoGetServerInfos);
         IRead ( 'Setup', 'nntp.dropresidualjobs', Def_NNTPDropResidualJobs);
         // Initialize priorities for pulling news
         If Not ArchivMode then begin
            for i:=0 to CfgHamster.ActiveCount-1 do begin
               ArticleBase.EstimateValues( CfgHamster.ActiveName[i], LfdArt, GrpNew );
               if GrpNew
                  then NewsJobs.GroupPriority[CfgHamster.ActiveName[i]] := MaxInt-4 // pull new groups with a high priority
                  else NewsJobs.GroupPriority[CfgHamster.ActiveName[i]] := LfdArt
            end
         end
      end;

      If Typ IN[ctAll, ctMail] then begin                      
         IRead ( 'Setup', 'preferred.smtpserver', Def_SmtpServer );
         IRead ( 'Setup', 'mail.leaveonserver',      Def_LeaveMailsOnServer );
         IRead ( 'Setup', 'mail.filterbyuidl',       Def_FilterByUIDL );
         IRead ( 'Setup', 'mail.generate.MID', Def_GenerateMailMID );
         IRead ( 'Setup','mail.flupforgate',Def_Flup); {JW} {N2M}
         IRead ( 'Setup', 'mail.GateUseHamsterEnvelope', Def_Gate_UseHamsterEnvelope); {JW} {N2M}
         {JW} {MID BuG}
         if Def_FQDNforMIDs='' then begin
            Def_GenerateMailMID := GENERATEMID_NEVER;
            // TGL: No News-M-ID, if no FQDN is set
            If Def_GenerateNewsMID <> GENERATEMID_NEVER then begin
               Log( LOGID_WARN, TrGl(kLog, 'Ini.NoFQDN.NoMIDCreatable',
                  'Message-IDs for News not creatable, because FQDN is not set!') );
               Def_GenerateNewsMID := GENERATEMID_NEVER
            end
         end;
         if Def_FQDN='' then begin
            if Def_FQDNforMIDs>'' then XREF_HOSTNAME := Def_FQDNforMIDs
         end else begin
            XREF_HOSTNAME := Def_FQDN
         end;
         Def_Mail_RemoveMids := lowercase( CfgIni.ReadString ( 'Setup', 'mail.removemids', '' )); {JAWO 01.04.2001: mail.removemids case insenistive}
         IRead ( 'Setup', 'local.smtpafterpop3.period', Def_SmtpAfterPop3Period );
         IRead ( 'Setup', 'sendmail.attempts.max', Def_SendMailAttemptsMax );
         IRead ( 'Setup', 'sendmail.attempts.del', Def_SendMailAttemptsDel );
         IRead ( 'Setup', 'mail.filter.mailsize.ignore', Def_MailSizeIgnore );
         IRead ( 'Setup', 'mail.filter.mailsize.delete',  Def_MailSizeKill );
         IRead ( 'Setup', 'mail.filter.mailsize.notify', Def_MailSizeNotify );
         IRead ( 'Setup', 'mail.filter.toplines', Def_NoOfTopLinesToLoad );
         if Def_NoOfTopLinesToLoad<0 then Def_NoOfTopLinesToLoad:=0;
         IRead ( 'Setup', 'mail.alwaysuseuidl', Def_AlwaysUseUIDL ); //PWMail-KillLog 2003-06-06
         IRead ( 'Setup', 'mail.filter.mailsize.killlog', Def_MailSizeKillLog ); //PW Mail-KillLog 2003-06-06
      end
   end;

   If Typ IN[ctAll, ctNews] then begin
      Log( LOGID_INFO, TrGl(kLog, 'Info.Newsserver.Count', 'NewsServer.Count')+'=' + inttostr(CfgHamster.ServerCount) );
      Log( LOGID_INFO, TrGl(kLog, 'Info.Groups.Count', 'Groups.Count')+'='     + inttostr(CfgHamster.ActiveCount) );
      Log( LOGID_INFO, TrGl(kLog, 'Info.Pulls.Count', 'Pulls.Count')+'='      + inttostr(CfgHamster.PullCount) )
   end;
   If Typ IN[ctAll, ctMail] then begin
      Log( LOGID_INFO, TrGl(kLog, 'Info.Pop3Server.Count', 'Pop3Server.Count')+'=' + inttostr(CfgHamster.Pop3ServerCount) );
      Log( LOGID_INFO, TrGl(kLog, 'Info.SmtpServer.Count', 'SmtpServer.Count')+'=' + inttostr(CfgHamster.SmtpServerCount) )
   end;
   If (Typ IN[ctAll, ctNews]) and Assigned(NewsHistory) then begin
      Log( LOGID_INFO, TrGl(kLog, 'Info.History.Count', 'History.Count')+'=' + inttostr(NewsHistory.Count) );
   end;
   If (Typ IN[ctAll, ctMail]) and Assigned(MailHistory) then begin
      Log( LOGID_INFO, TrGl(kLog, 'Info.MailHistory.Count', 'MailHistory.Count')+'=' + inttostr(MailHistory.Count) );
   end;

   Log( LOGID_INFO,  TrGl(kLog, 'Info.Loading.Config.Done', 'Configuration loaded') );

   With TFiltersNews.Create( PATH_BASE + CFGFILE_SCORES ) do try Test finally Free end;
   With TFiltersMail.Create( PATH_BASE + CFGFILE_MFILTER ) do try Test finally Free end;

end;

Procedure EditFile(Const s: String);
Var EdApp, EdPar, Ext: String; i, x: Integer;
begin
   Ext := ExtractFileExt ( s );
   If Ext > '' then begin
      With CfgIni do begin
         EdApp := ReadString ( 'Setup', 'editor'+Ext+'.app', Def_EditorApp );
         EdPar := ReadString ( 'Setup', 'editor'+Ext+'.params', Def_EditorParams );
      end
   end else begin
      EdApp := Def_EditorApp;
      EdPar := Def_EditorParams;
   end;
   i := Pos( '%1', EdPar );                          
   if i>0 then EdPar := copy( EdPar, 1, i-1 )
                      + '"' + s + '"'
                      + copy( EdPar, i+2, 999 );
   if EdApp='' then begin EdApp:=EdPar; EdPar:='' end;
   x := ShellExecute( 0, 'open',
                 PChar(EdApp), PChar(EdPar),
                 PChar(ExtractFilePath(s)), SW_SHOWNORMAL );
   If x < 32 then begin
      Log ( LOGID_WARN, TrGlF(kLog, 'Editor.NotStartable',
         'Error %s when running "%s" with parameter "%s", running Notepad instead!',
         [IntToStr(x), EdApp, EdPar]));
      EdApp := 'Notepad.exe';
      EdPar := '"'+s+'"';
      x := ShellExecute( 0, 'open',
                 PChar(EdApp), PChar(EdPar),
                 PChar(ExtractFilePath(s)), SW_SHOWNORMAL );
      If x < 32 then begin
         Log ( LOGID_ERROR, TrGlF(kLog, 'Editor.Notepad.NotStartable',
            'Error #%s when running notepad.exe!', IntToStr(x)))
      end
   end
end;

procedure ConfigShutdown;
begin
   try
      if Assigned(NewsHistory)  then NewsHistory.Free;
      if Assigned(MailHistory)  then MailHistory.Free;
      if Assigned(CfgAccounts)  then begin CfgAccounts.Free;  CfgAccounts :=nil end;
      if Assigned(ArticleBase)  then begin ArticleBase.Free;  ArticleBase :=nil end;
      if Assigned(PasswordFile) then begin PasswordFile.Free; PasswordFile:=nil end;
      if Assigned(NewsJobs)     then begin NewsJobs.Free;     NewsJobs    :=nil end;
      if Assigned(RasDialer)    then begin RasDialer.Free;    RasDialer   :=nil end;
      if Assigned(CfgHamster)   then begin CfgHamster.Free;   CfgHamster  :=nil end;
//      if Assigned(CfgIni)       then begin CfgIni.Free;       CfgIni      :=nil end;
   except
      on E: Exception do Log( LOGID_WARN, TrGlF(kLog, 'Error.ConfigShutdown.Exception',
        'ConfigShutdown: %s', E.Message ) )
   end;
end;

Function SSLActivation(Const progPath, progfile, pemfile: String): TSSLStatus;
Var prog, pem: String;
begin
   Result := ssOff;
   prog:=IncludeTrailingBackslash(progpath)+progfile;
   pem:=IncludeTrailingBackslash(progpath)+pemfile;
   If prog > '' then If fileexists2(prog) then begin
      Result := ssServer;
      If pem > '' then If fileexists2(pem) then Result := ssAll
   end
end;

Var Ini: TExtIniFile = NIL;

Function CfgIni: TExtIniFile;
begin
   If Not Assigned(Ini) then begin
      If ArchivMode
         then Ini     := TExtIniFile.Create( 'Hamster.ini' )
         else Ini     := TExtIniFile.Create( PATH_BASE + 'Hamster.ini' );
   end;
   Result := Ini
end;

// Check Dirs

Procedure DeleteSysDirsFromDirList (Const MainDir: String; sl: TStrings);
Var i: Integer; s, V: String;
begin
   V := UpperCase
        ('::'+ PATH_HSC
        +'::'+ PATH_HSC_RC
        +'::'+ PATH_HSM
        +'::'+ PATH_LOGS
        +'::'+ PATH_SERVER
        +'::'+ PATH_MAILS
        +'::'+ PATH_MAIL_OUT
        +'::'+ PATH_GROUPS
        +'::'+ PATH_TRASH
        +'::'+ PATH_NEWS_OUT
        +'::'+ PATH_NEWS_ERR+'::' );
   For i := sl.Count-1 downto 0 do begin
      s := IncludeTrailingBackslash(IncludeTrailingBackslash(Maindir) + sl[i] );
      s := '::'+UpperCase(s)+'::';
      If Pos (s, V) > 0 then sl.Delete(i)
   end
end;

Procedure CheckObsoleteServerDirs;
Var r: TSearchrec;
    i, p: Integer;
    sl: TStringList;
begin
   sl := TStringList.Create;
   With sl do try
      If FindFirst ( PATH_SERVER + '*.*', faAnyfile, r) = 0 then try
         Repeat
            If (r.Name[1]<>'.') and ((r.Attr and faDirectory)>0) then Add (r.Name)
         until FindNext(r) <> 0
      finally
         FindClose(r)
      end;
      DeleteSysDirsFromDirList ( PATH_SERVER, sl );
      With CfgHamster do begin
         For i := 0 to ServerCount-1 do begin
            p := IndexOf ( ServerName[i] );
            If p >= 0 then Delete(p)
         end;
         For i := 0 to POP3ServerCount-1 do begin
            p := IndexOf ( POP3ServerNameToPath(POP3ServerName[i]) );
            If p >= 0 then Delete(p)
         end;
         For i := 0 to SMTPServerCount-1 do begin
            p := IndexOf ( SMTPServerName[i] );
            If p >= 0 then Delete(p)
         end;
      end;
      For i := 0 to Count-1 do begin
         LogFile.Add ( LOGID_WARN, TrGlF ( kLog, 'Check.ObsoleteServerDir.Found2',
             'Directory "%s" in the serverpath (%s) is not used.', [Strings[i], PATH_SERVER]) )
      end
   finally Free end
end;

Procedure CheckObsoleteGroupDirs;
Var r: TSearchrec;
    i, p: Integer;
    sl: TStringList;
begin
   sl := TStringList.Create;
   With sl do try
      If FindFirst ( PATH_GROUPS + '*.*', faAnyfile, r) = 0 then try
         Repeat
            If (r.Name[1]<>'.') and ((r.Attr and faDirectory)>0) then Add (r.Name)
         until FindNext(r) <> 0
      finally
         FindClose(r)
      end;
      DeleteSysDirsFromDirList ( PATH_GROUPS, sl );
      With CfgHamster do begin
         For i := 0 to ActiveCount-1 do begin
            p := IndexOf ( ActiveName[i] );
            If p >= 0 then Delete(p)
         end
      end;
      For i := 0 to Count-1 do begin
         LogFile.Add ( LOGID_WARN, TrGlF ( kLog, 'Check.ObsoleteGroupDir.Found2',
             'Directory "%s" in the groupspath (%s) is not used.', [Strings[i], PATH_GROUPS]) )
      end
   finally Free end
end;

{ tExtInifile }

procedure tExtInifile.IRead(const Section, Ident: String;
  var Default: Boolean);
begin
  Default := ReadBool(Section, Ident, Default)
end;
procedure tExtInifile.IRead(const Section, Ident: String;
  var Default: Integer);
begin
  Default := ReadInteger(Section, Ident, Default)
end;
procedure tExtInifile.IRead(const Section, Ident: String; var Default: DWord); 
begin
  Default := ReadInteger(Section, Ident, Default)
end;
procedure tExtInifile.IRead(const Section, Ident: String;
  var Default: string);
begin
  Default := ReadString(Section, Ident, Default)
end;

procedure tExtInifile.IWrite(const Section, Ident: String;
  const Value: Boolean);
begin
  WriteBool(Section, Ident, Value)
end;
procedure tExtInifile.IWrite(const Section, Ident: String;
  const Value: Integer);
begin
  WriteInteger(Section, Ident, Value)
end;
procedure tExtInifile.IWrite(const Section, Ident, Value: string);
begin
  WriteString(Section, Ident, Value)
end;

function TCfgHamster.GetActivePostServer(const Groupname: String;
  out Server, Newsdir: String): Boolean;
var  i: Integer;
     s, srv, prefMain, prefGroup: String;
begin
   FLock.BeginRead;
   try
     prefMain := Def_PostServer;
     i := Pos( ',', prefMain );
     if i>0 then prefMain:=copy(prefMain,1,i-1);

     With TIniFile.Create( PATH_GROUPS + GroupName + '\data' + EXT_CFG ) do try
        prefGroup := ReadString ( 'Setup', 'postserver', '' )
     finally
        free
     end;
     i := Pos( ',', prefGroup );
     if i>0 then prefGroup:=copy(prefGroup,1,i-1);

     Server := '';
     s := '('+TrGl('config', 'Groupname.NoPullMaybeLocal', 'no pullserver, maybe local')+')';

     for i:=0 to PullCount-1 do begin
        if lowercase(PullGroup[i])=lowercase(Groupname) then begin
           srv := PullServer[i];
           if not ServerIsReadOnly[srv] then begin
              if srv = PrefGroup then begin
                 Server := srv;
                 s := ' ('+TrGl('config', 'Groupname.GroupSpecifiedpostserver', 'is group-specified postserver')+')';
                 break
              end else if srv = PrefMain then begin
                 Server := srv;
                 s := ' ('+TrGl('config', 'Groupname.prefpostserver', 'is preferred postserver')+')';
                 If PrefGroup = '' then break
              end else If Server = '' then begin
                 Server := srv;
                 s := ' ('+TrGl('config', 'Groupname.pullserverforgroup', 'is a pullserver for group')+')';
                 If PrefGroup + PrefMain = '' then break; // no preferred
              end
           end
        end
     end;
     Result := Server > '';
     NewsDir := PATH_NEWS_OUT;
     If Result then With TIniFile.Create(PATH_SERVER + Server + '\' + SRVFILE_INI) do try
        If ReadBool('NNTP', 'UsePostDir', false) then begin
           NewsDir := ReadString('NNTP', 'PostDir', PATH_NEWS_OUT)
        end
     finally Free end;
     Log( LOGID_DEBUG, Groupname + '->' + Server + s + ', news.out: '+Newsdir )
   finally FLock.EndRead end;
end;

function TCfgHamster.GetNNTPServertype(Index: Integer): Integer;
begin
   With TIniFile.Create( GetServerPath(Index) + SRVFILE_INI ) do try
      Result := ReadInteger ( 'NNTP', 'Servertype', NNTPServerType_Full );
   finally Free end
end;

initialization
finalization
   If Assigned(ini) then Ini.free
end.
