// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit HKillsMain;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, ExtCtrls, Global, uTools, Menus, ClipBrd,
  cStdForm;

Procedure DialogKillsMain;
Procedure DialogKillfileLogClose;
Function  IsDialogKillfileLogOpen: Boolean;

Type
  TKillIndex = ( KILLIDX_SERVER, KILLIDX_GROUP, KILLIDX_SCORE, KILLIDX_NUMBER, KILLIDX_SUBJECT,
                 KILLIDX_FROM, KILLIDX_DATE, KILLIDX_MID, KILLIDX_REFS, KILLIDX_BYTES,
                 KILLIDX_LINES, KILLIDX_XREF );

Var
{  SORTNAMES: array[ KILLIDX_MIN .. KILLIDX_MAX ] of String = (
               'Server', 'Group', 'Score', 'Number', 'Subject', 'From',
               'Date', 'Message-ID', 'References', 'Bytes', 'Lines', 'Xref'
             ); }
  SORTSEQ  : array[ TKillIndex, 1..2 ] of TKillIndex = (
               (KILLIDX_GROUP, KILLIDX_DATE), (KILLIDX_SUBJECT, KILLIDX_DATE),
               (KILLIDX_GROUP, KILLIDX_DATE), (KILLIDX_GROUP, KILLIDX_DATE),
               (KILLIDX_GROUP, KILLIDX_DATE), (KILLIDX_GROUP, KILLIDX_DATE),
               (KILLIDX_GROUP, KILLIDX_SUBJECT), (KILLIDX_GROUP, KILLIDX_DATE),
               (KILLIDX_GROUP, KILLIDX_DATE), (KILLIDX_GROUP, KILLIDX_DATE),
               (KILLIDX_GROUP, KILLIDX_DATE), (KILLIDX_GROUP, KILLIDX_DATE)
             );

type
  TThreadInfo = Record MID, Refs, Subject: String end;
  TThreadInfos = Array of TThreadInfo;

  TfrmKillsMain = class(THForm)
    PopKills: TPopupMenu;
    mnuMark: TMenuItem;
    mnuDelete: TMenuItem;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Exit1: TMenuItem;
    N1: TMenuItem;
    N2: TMenuItem;
    mnuScoreTest: TMenuItem;
    N3: TMenuItem;
    mnuSelectAll: TMenuItem;
    N4: TMenuItem;
    mnuExit2: TMenuItem;
    Setsortsequence1: TMenuItem;
    N5: TMenuItem;
    mnuShowDetails: TMenuItem;
    N6: TMenuItem;
    mnuRetrieveThread: TMenuItem;
    menuDeleteThrea: TMenuItem;
    Help1: TMenuItem;
    mnuEntries: TMenuItem;
    pg: TPageControl;
    tsEntries: TTabSheet;
    lstResult: TListBox;
    hdrResult: THeaderControl;
    tsScorefile: TTabSheet;
    pnlEdit: TPanel;
    N7: TMenuItem;
    mnuSave: TMenuItem;
    mnuUndo: TMenuItem;
    tsSettings: TTabSheet;
    Label1: TLabel;
    emLogLimit: TEdit;
    Label23: TLabel;
    emKeepKill: TEdit;
    emmScore: TRichEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure mnuMarkClick(Sender: TObject);
    procedure mnuDeleteClick(Sender: TObject);
    procedure lstResultDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure hdrResultSectionResize(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure lstResultDblClick(Sender: TObject);
    procedure hdrResultSectionClick(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure Exit1Click(Sender: TObject);
    procedure Editscorefile1Click(Sender: TObject);
    procedure mnuScoreTestClick(Sender: TObject);
    procedure mnuSelectAllClick(Sender: TObject);
    procedure Setsortsequence1Click(Sender: TObject);
    procedure mnuRetrieveThreadClick(Sender: TObject);
    procedure mnuDeleteThreadClick(Sender: TObject);
    procedure Help1Click(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure emmScoreChange(Sender: TObject);
    procedure mnuSaveClick(Sender: TObject);
    procedure mnuUndoClick(Sender: TObject);
    procedure File1Click(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure lstResultKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
  private
    { Private-Deklarationen }
    KILLS, Retrieve: TStringList;
    KILLSORT : TKillIndex;
    ScorefileLoaded, InvertedSort: boolean;
    Changed  : Boolean;
    procedure ViewRefresh( Location: Integer );
    procedure ExtractSelectedHeaders(var List: TThreadInfos;
      var ListPos: Integer; Const CopyToMIDstxt: boolean);
    procedure Load;
    procedure Save;
    Procedure AddHeadersIntoScoreFile (HeaderList: TThreadInfos;
        Const SfSection, SfValue: String; Const Expire: Integer );
    procedure RetrieveLine(const Datei, Inhalt: String);
  public
    { Public-Deklarationen }
  end;

implementation

{$R *.DFM}

uses Config, cClientNNTP, HConfigScore, cArticle, cFiltersNews, dInput, uCRC32,
     uEncoding, uDateTime, cLogFile;

Var Dlg: TfrmKillsMain = NIL;

Procedure DialogKillsMain;
begin
  Dlg := TfrmKillsMain.Create(NIL);
  try
     Dlg.Showmodal
  finally
     FreeAndNil(Dlg)
  end
end;
Procedure DialogKillfileLogClose;
begin
   If Assigned(Dlg) then try
      Dlg.ModalResult := mrCancel
   except end
end;
Function IsDialogKillfileLogOpen: Boolean;
begin
   Result := Assigned(Dlg)
end;

procedure TfrmKillsMain.ViewRefresh( Location: Integer );
Var Parser: TParser;

   function SortValue( SortItem: TKillIndex ): String;
   begin
        Result := Lowercase( Parser.sPart( Ord(SortItem), '' ) );

        case SortItem of
           KILLIDX_SUBJECT: begin
              if copy(Result,1,4)='re: ' then begin
                 System.Delete(Result,1,4);
                 Result := Result + 'z';
              end;
              If Result > '' then Result := DecodeHeadervalue( Result );
           end;
           KILLIDX_FROM:
              If Result > '' then Result := FilterNameOfFrom( DecodeHeadervalue( Result ) )
                             else Result := '';
           KILLIDX_DATE:
              Result := FormatDateTime( 'yyyy.mm.dd hh:nn:ss', RfcDateTimeToDateTimeGMT(Result) );
           KILLIDX_SCORE, KILLIDX_NUMBER, KILLIDX_BYTES, KILLIDX_LINES:
              Result := Format( '%8d', [strtoint(Result)] );
       end;
   end;

var  Line, p, i: Integer;
     Srt: String; 
     LocTop: Integer;
     KILLSORT2, KILLSORT3: TKillIndex;
Const
   Sortchars = ' 0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
begin
   LocTop := lstResult.TopIndex;
   lstResult.Items.BeginUpdate;
   lstResult.Items.Clear;
   lstResult.Sorted := True;

   Srt := '';
   Parser := TParser.Create;
   try

      KILLSORT2 := SORTSEQ[KILLSORT, 1];
      KILLSORT3 := SORTSEQ[KILLSORT, 2];

      for Line:=0 to KILLS.Count-1 do begin
         if KILLS[Line]<>'' then begin

            Parser.Parse( KILLS[Line], #9 );
            Srt := SortValue( KILLSORT );
            if KILLSORT2<>KILLSORT  then Srt := Srt + '|' + SortValue( KILLSORT2 );
            if KILLSORT3<>KILLSORT2 then Srt := Srt + '|' + SortValue( KILLSORT3 );
            Srt := Srt + '|' + inttostr( StrToCRC32( KILLS[Line] ) );

            If InvertedSort then begin
               For i:=1 to Length(Srt) do begin
                  p := Pos(UpCase(Srt[i]), SortChars);
                  If p>0 then Srt[i]:=SortChars[Length(SortChars)-p+1]
               end
            end;

            try
               lstResult.Items.AddObject( Srt, Pointer(Line) )
            except
               Log( LOGID_WARN,
                       TrF( 'Killlog.TooManyEntries',
                            'There are too many entries in killfile - the last %s entries could not be shown.',
                       IntToStr(Kills.Count-Line-1)));
               break
            end
         end;
      end;

      if (Location>=0) and (Location<lstResult.Items.Count) then begin
         Application.ProcessMessages;
         if (LocTop>=0) and (LocTop<lstResult.Items.Count) then begin
            lstResult.TopIndex := LocTop;
         end;
         lstResult.Selected[Location] := True;
         lstResult.ItemIndex := Location;
      end;
      lstResult.Items.EndUpdate;
   finally
      Parser.Free;
   end
end;

procedure TfrmKillsMain.FormCreate(Sender: TObject);
var  i: Integer; ki: TKillIndex; M: TMenuItem; x: Word;
begin
   KILLS    := TStringList.Create;
   Retrieve := TStringList.Create;

   LoadWindowState( Self, 'KillList' );

   For i := 0 to PopKills.Items.Count-1-2 do begin
      M := TMenuItem.Create(self);
      With PopKills.Items[i] do begin
         M.Caption := Caption;
         M.ShortCut := ShortCut;
         M.OnClick := OnClick;
         M.Hint := Hint
      end;
      mnuEntries.Add ( M )
   end;

   for i:=0 to hdrResult.Sections.Count-1 do begin
      hdrResult.Sections[i].Width := CfgIni.ReadInteger( 'KillList',
         'ColWidth' + inttostr( i ), hdrResult.Sections[i].Width );
   end;

   for ki:=Low(TKillIndex) to High(TKillIndex) do begin
      x := Ord(SORTSEQ[ki, 1]) * 256 + Ord(SORTSEQ[ki, 2]);
      x := CfgIni.ReadInteger( 'KillList', 'SortSeq' + inttostr( Ord(ki) ), x);
      SORTSEQ[ki, 1] := TKillIndex(x div 256);
      SORTSEQ[ki, 2] := TKillIndex(x mod 256);
   end;

   Load;

   KILLSORT := TKillIndex(CfgIni.ReadInteger( 'KillList', 'SortBy', Ord(KILLIDX_GROUP) ));
   InvertedSort := CfgIni.ReadBool( 'KillList', 'SortInverted', false );
   ViewRefresh( 0 );
end;

procedure TfrmKillsMain.FormDestroy(Sender: TObject);
begin
   CfgIni.WriteInteger( 'KillList', 'SortBy', Ord(KILLSORT) );
   CfgIni.WriteBool( 'KillList', 'SortInverted', InvertedSort );
   While mnuEntries.Count > 0 do mnuEntries.Delete(0);
   Kills.Free;
   Retrieve.Free;
end;


procedure TfrmKillsMain.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
   if Changed then begin
      if Application.MessageBox( PChar(Tr('SaveChanges.ask', 'Save changes?')),
           PChar(Caption),
           MB_ICONQUESTION + MB_YESNO)=IDYes
      then Save
   end;
   SaveWindowState( Self, 'KillList' );
end;

Procedure TfrmKillsMain.Load;
begin
   // Killfile-Entries
   if FileExists2( PATH_GROUPS + CFGFILE_SCORELOG ) then begin
      KILLS.LoadFromFile( PATH_GROUPS + CFGFILE_SCORELOG );
   end;
   // Scorefile
   ScorefileLoaded := false;
   emmScore.Lines.BeginUpdate;
   try
      if FileExists2( PATH_BASE + CFGFILE_SCORES )
         then emmScore.Lines.LoadFromFile( PATH_BASE + CFGFILE_SCORES )
         else emmScore.Lines.Clear;
   finally
      emmScore.Lines.EndUpdate
   end;
   ScorefileLoaded := true;
   // Scorevalue
   emLogLimit.Text := inttostr( CfgIni.ReadInteger( 'Setup', 'score.loglimit', Def_ScoreLogLimit ));
   emKeepKill.Text  := inttostr( CfgIni.ReadInteger( 'Setup', 'purge.kills.keepdays', Def_Purge_Kills_KeepDays ) );
   // Status
   Changed  := False
end;

Procedure TfrmKillsMain.Save;
Var i: Integer;
begin
   // Save M-IDs to retrieve
   With Retrieve do begin
      For i := 1 to Count div 2 do begin
         HamFileAppendLine ( Strings[i*2-2], Strings[i*2-1] )
      end;
      Clear
   end;
   // Killfile-Entries
   KILLS.SaveToFile( PATH_GROUPS + CFGFILE_SCORELOG );
   // Scorefile
   If ScorefileLoaded then emmScore.Lines.SaveToFile( PATH_BASE + CFGFILE_SCORES );
   // Scorevalue
   try i := strtoint(TrimWhSpace(emLogLimit.Text)) except i := Def_ScoreLogLimit end;
   if i<>Def_ScoreLogLimit then begin
      Def_ScoreLogLimit := i;
      CfgIni.WriteInteger( 'Setup', 'score.loglimit', Def_ScoreLogLimit );
   end;
   try
      CfgIni.WriteInteger( 'Setup', 'purge.kills.keepdays', StrToInt(emKeepKill.Text) )
   except end;
   changed := false
end;

procedure TfrmKillsMain.lstResultDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var  R  : TRect;
     Line, Col: Integer;
     Parser: TParser;
     s, s2: String;

     procedure WriteCol( No: Integer; Txt: String );
     var  l, w : Integer;
     begin
          l := hdrResult.Sections.Items[No].Left;
          w := hdrResult.Sections.Items[No].Width;

          R.Left  := l + 1;
          R.Right := R.Left + w - 3;
          R.Top   := Rect.Top;
          R.Bottom:= Rect.Bottom;
          lstResult.Canvas.TextRect( R, R.Left, R.Top, Txt );

          lstResult.Canvas.MoveTo( l+w-1, R.Top    );
          lstResult.Canvas.LineTo( l+w-1, R.Bottom );
     end;
begin
   Line := LongInt( lstResult.Items.Objects[Index] );

   with lstResult.Canvas do begin

      if odSelected in State then begin
         Brush.Color:= clHighlight;
         Font.Color  := clYellow;
         Pen.Color := clLtGray;
      end else begin
         Brush.Color := clWindow;
         Font.Color   := clWindowText;
         Pen.Color := clLtGray;
      end;

      FillRect(Rect);

      Parser := TParser.Create;
      try
         Parser.Parse( KILLS[Line], #9 );

         for Col:=0 to hdrResult.Sections.Count-1 do begin
            s := Parser.sPart(Col,'');
            If s > '' then s2 := DecodeHeadervalue( s );
            case TKillIndex(Col) of
               KILLIDX_SUBJECT : WriteCol( Col, s2 );
               KILLIDX_FROM    : WriteCol( Col, FilterNameOfFrom( s2 ) );
               KILLIDX_DATE    : WriteCol( Col, FormatDateTime( 'dd.mm.yyyy hh:nn',
                                    DateTimeGMTToLocal( RfcDateTimeToDateTimeGMT(s) ) ) );
               else              WriteCol( Col, s );
            end;
         end;
      finally
         Parser.Free
      end
   end;
end;

procedure TfrmKillsMain.hdrResultSectionResize(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
     lstResult.Refresh;
     CfgIni.WriteInteger( 'KillList', 'ColWidth' + inttostr( Section.Index ), Section.Width );
end;

procedure TfrmKillsMain.lstResultDblClick(Sender: TObject);
var  i, k: Integer;
     s, h, Info: String;
     p   : TParser;
begin
     if lstResult.ItemIndex<0 then exit;
     if lstResult.ItemIndex>=lstResult.Items.Count then exit;

     i := LongInt( lstResult.Items.Objects[ lstResult.ItemIndex ] );
     s := KILLS[i];
     p := TParser.Create;
     try
       p.Parse( s, #9 );
       s := Tr('DblClickInfo.Newsserver', 'Newsserver')+': '
                  + p.sPart( Ord(KILLIDX_SERVER),'') + #13#10
          + Tr('DblClickInfo.Newsgroups', 'Newsgroups')+': '
                  + p.sPart( Ord(KILLIDX_GROUP),'') + #13#10
          + Tr('DblClickInfo.Score', 'Score (original)')+': '
                  + p.sPart( Ord(KILLIDX_SCORE),'');
       If p.sPart( Ord(High(TKillIndex))+1,'') > '' then begin
          s := s
          + ' (' + Tr('DblClickInfo.ScoreLoad', 'Score on Overview')+': '
                  + p.sPart( Ord(High(TKillIndex))+1,'') + ', '
          + Tr('DblClickInfo.ScoreSave', 'Score after Load')+': '
                  + p.sPart( Ord(High(TKillIndex))+2,'') + ')'
       end;
       s := s + #13#10 
          + Tr('DblClickInfo.Number', 'Number') + ': '
                  + p.sPart( Ord(KILLIDX_NUMBER),'') + #13#10#13#10;
       h := p.sPart( Ord(KILLIDX_SUBJECT),'');
       s := s + Tr('DblClickInfo.Subject', 'Subject')+': ' + h + #13#10;
       if Pos('=?',h)>0 then s := s + '~' + Tr('DblClickInfo.Subject', 'Subject')+': ' + DecodeHeadervalue(h) + #13#10;
       h := p.sPart( Ord(KILLIDX_FROM),'');
       s := s + Tr('DblClickInfo.From', 'From')+': ' + h + #13#10;
       if Pos('=?',h)>0 then s := s + '~' + Tr('DblClickInfo.From', 'From')+': ' + DecodeHeadervalue(h) + #13#10;
       s := s + #13#10;

       try
          k := Trunc( NowGMT - RfcDateTimeToDateTimeGMT(p.sPart(Ord(KILLIDX_DATE),'')) );
       except
          k := 0;
       end;

       s := s + Tr('DblClickInfo.Date', 'Date')+': '          + p.sPart( Ord(KILLIDX_DATE),'') + #13#10
              + Tr('DblClickInfo.Age', 'Age (current)')+': ' + inttostr(k) + #13#10#13#10;

       s := s + Tr('DblClickInfo.MID', 'Message-ID')+': ' + p.sPart( Ord(KILLIDX_MID),'') + #13#10
              + Tr('DblClickInfo.References', 'References')+': ' + p.sPart( Ord(KILLIDX_REFS),'') + #13#10#13#10
              + Tr('DblClickInfo.Bytes', 'Bytes')+': '      + p.sPart( Ord(KILLIDX_BYTES),'') + #13#10
              + Tr('DblClickInfo.Lines', 'Lines')+': '      + p.sPart( Ord(KILLIDX_LINES),'') + #13#10
             {+ 'Xref: '}      + p.sPart( Ord(KILLIDX_XREF),'');
     finally
       p.Free;
     end;

     Info := s;

     s := s + #13#10 + '______________________________________' + #13#10 + #13#10
            + Tr('CopyToClipboard.Question', 'Copy Info to clipboard?');

     If Application.MessageBox( PChar(s), PChar(Caption),
          MB_ICONQUESTION + MB_YESNO + MB_DEFBUTTON1) = IDYES
     then begin
        Clipboard.Clear;
        Clipboard.AsText := Info
     end;
end;

procedure TfrmKillsMain.hdrResultSectionClick(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
   If KILLSORT<>TKillIndex(Section.Index) then begin
      KILLSORT := TKillIndex(Section.Index);
      InvertedSort := false
   end else begin
      InvertedSort := Not InvertedSort
   end;
   ViewRefresh( 0 )
end;

procedure TfrmKillsMain.mnuMarkClick(Sender: TObject);
var  i, Line, Loc: Integer;
     Parser      : TParser;
     ServerDir   : String;
begin
     if lstResult.SelCount<=0 then exit;

     Parser := TParser.Create;
     try
        Loc := -1;
        for i:=0 to lstResult.Items.Count-1 do begin
           if lstResult.Selected[i] then begin
              Line := LongInt( lstResult.Items.Objects[i] );

              Parser.Parse( KILLS[Line], #9 );
              ServerDir := PATH_SERVER + Parser.sPart( Ord(KILLIDX_SERVER),'') + '\';
              RetrieveLine ( ServerDir + SRVFILE_GETMIDLIST,
                             Parser.sPart( Ord(KILLIDX_MID),'') );

              KILLS[Line] := ''; // mark for deletion below
              Loc := i + 1;
           end;
        end
     finally
        Parser.Free
     end;

     i := 0;
     while i<KILLS.Count do begin
        if KILLS[i]='' then begin
           KILLS.Delete( i );
           dec(Loc);
           Changed := True;
        end else begin
           inc(i);
        end;
     end;

     ViewRefresh( Loc );
end;

procedure TfrmKillsMain.mnuDeleteClick(Sender: TObject);
var  i, Line, Loc: Integer;
begin
     if lstResult.SelCount<=0 then exit;

     Loc := -1;
     for i:=0 to lstResult.Items.Count-1 do begin
        if lstResult.Selected[i] then begin
           Line := LongInt( lstResult.Items.Objects[i] );
           KILLS[Line] := ''; // mark for deletion below
           Loc := i + 1;
        end;
     end;

     i := 0;
     while i<KILLS.Count do begin
        if KILLS[i]='' then begin
           KILLS.Delete( i );
           dec(Loc);
           Changed := True;
        end else begin
           inc(i);
        end;
     end;

     ViewRefresh( Loc );
end;

procedure TfrmKillsMain.Exit1Click(Sender: TObject);
begin
     Close;
end;

procedure TfrmKillsMain.Editscorefile1Click(Sender: TObject);
begin
   With TfrmConfigScore.Create(NIL) do try
     ShowModal
   finally Free end
end;

procedure TfrmKillsMain.mnuScoreTestClick(Sender: TObject);
var  i : Integer;
     s, ml, ng, OrigScore : String;
     p : TParser;
     Art: TArticle;
     XOverRec : TXOverRec;
begin
   if lstResult.SelCount<>1 then begin
      Application.MessageBox (PChar(Tr('ScoreTest.Error.MarkOnlyOne',
         'Please mark ONE entry to be tested!')),
         PChar(Caption), MB_ICONINFORMATION + MB_OK );
      exit
   end;
   if lstResult.ItemIndex<0 then exit;
   if lstResult.ItemIndex>=lstResult.Items.Count then exit;

   i := LongInt( lstResult.Items.Objects[ lstResult.ItemIndex ] );
   s := KILLS[i];

   Art := TArticle.Create;
   p := TParser.Create;
   try
        p.Parse( s, #9 );
        OrigScore := p.sPart( Ord(KILLIDX_SCORE),'');
        Art.AddHeader( 'Newsgroups: ', p.sPart( Ord(KILLIDX_GROUP),'') );
        ng := p.sPart( Ord(KILLIDX_GROUP),'');
        i  := Pos( ',', ng );
        if i>0 then ng:=copy(ng,1,i-1);

        Art.AddHeader( 'Subject: '    , p.sPart( Ord(KILLIDX_SUBJECT),'') );
        Art.AddHeader( 'From: '       , p.sPart( Ord(KILLIDX_FROM),'') );
        Art.AddHeader( 'Date: '       , p.sPart( Ord(KILLIDX_DATE),'') );
        Art.AddHeader( 'Message-ID: ' , p.sPart( Ord(KILLIDX_MID),'') );
        Art.AddHeader( 'References: ' , p.sPart( Ord(KILLIDX_REFS),'') );
        Art.AddHeader( 'Lines: '      , p.sPart( Ord(KILLIDX_LINES),'') );
        Art.AddHeader( 'Xref: '       , p.sPart( Ord(KILLIDX_XREF),'') );
        Art.AddHeader( 'Bytes: '      , p.sPart( Ord(KILLIDX_BYTES),'') ); // only for simulation

        Art.FullBody := 'dummy-body';

        ArticleToXOverRec( Art, XOverRec );
     finally
        p.Free; Art.Free
     end;

     With TFiltersNews.Create( PATH_BASE + CFGFILE_SCORES ) do try
        SelectSections( ng );
        // i := ScoreForXOverRecLog( XOverRec, ml ); // PG
        i := ScoreBeforeLoad( XOverRec, @ml ); // PG
     finally
        Free
     end;

     s :=     Tr('ScoreTest.Info.TestGroup', 'Test-Group')+': ' + ng + #13#10;
     s := s + Tr('ScoreTest.Info.OrgScore', 'Orig.-Score')+': ' + OrigScore + #13#10;
     s := s + Tr('ScoreTest.Info.TestScore', 'Test-Score')+': ' + inttostr( i ) + #13#10#13#10;
     s := s + Tr('ScoreTest.Info.MatchList', 'Match-List')+':' + #13#10 + ml;
     Application.MessageBox( PChar(s), PChar(Caption), MB_ICONINFORMATION + MB_OK )
end;

procedure TfrmKillsMain.mnuSelectAllClick(Sender: TObject);
begin
     SendMessage( lstResult.Handle, LB_SELITEMRANGEEX, 0, lstResult.Items.Count-1 );
end;

procedure TfrmKillsMain.Setsortsequence1Click(Sender: TObject);
var  s: String;
     i, i2, i3: TKillIndex;
     Neu1, Neu2, Neu3, MinVal, MaxVal: Integer;
begin
   i2 := SORTSEQ[KILLSORT, 1];
   i3 := SORTSEQ[KILLSORT, 2];

   s := #13#10#13#10 + Tr('SortSequence.Fieldlist', 'Fields')+':' + #13#10;
   for i:=Low(TKillIndex) to High(TKillIndex) do begin
      s := s + inttostr(Ord(i)) + '=' + hdrResult.Sections[Ord(i)].Text;
      if i < High(TKillIndex) then s:=s+', ';
   end;
   s := s + #13#10#13#10;

   Neu1 := Ord(Killsort);
   Neu2 := Ord(i2);
   Neu3 := Ord(i3);
   MinVal := Ord(Low(TKillIndex));
   MaxVal := Ord(High(TKillIndex));
   if not InputDlgInt( TrF('SortSequence.Caption', 'Sort-sequence for "%s"', hdrResult.Sections[Neu1].text),
                       Tr('SortSequence.CurrentSequence', 'Current sequence')+':' + #13#10
                       + hdrResult.Sections[Neu1].text + ', '
                       + '>>' + hdrResult.Sections[Neu2].text + '<<, '
                       + hdrResult.Sections[Neu3].text
                       + s
                       + TrF('SortSequence.Enter2ndSortField', 'Enter second sort-field (%s..%s):', [IntToStr(MinVal), inttostr(MaxVal)]),
                       Neu2, MinVal, MaxVal, 0{HlpKillOverview_SortSequence} ) then exit;
   if not InputDlgInt( TrF('SortSequence.Caption', 'Sort-sequence for "%s"', hdrResult.Sections[Neu1].text),
                       Tr('SortSequence.CurrentSequence', 'Current sequence')+':' + #13#10
                         + hdrResult.Sections[Neu1].text + ', '
                         + hdrResult.Sections[Neu2].text + ', '
                         + '>>' + hdrResult.Sections[Neu3].text + '<<'
                         + s
                         + TrF('SortSequence.Enter3rdSortField', 'Enter third sort-field (%s..%s):', [inttostr(MinVal), IntToStr(MaxVal)]),
                       Neu3, MinVal, MaxVal, 0{HlpKillOverview_SortSequence} )
      then exit;

   SORTSEQ[KILLSORT, 1] := TKillIndex(Neu2);
   SORTSEQ[KILLSORT, 2] := TKillIndex(Neu3);
   CfgIni.WriteInteger( 'KillList', 'SortSeq' + inttostr( Neu1 ), Neu2*256 + Neu3 );
   ViewRefresh( 0 );
end;

{PW}{Kill_Reload_Thread}

Procedure TfrmKillsMain.ExtractSelectedHeaders(Var List: TThreadInfos;
    Var ListPos: Integer; Const CopyToMIDstxt: boolean);
Var Parser: TParser; Line, i1, i2, i3: Integer; AlreadyExisting: Boolean;
begin
   SetLength(List, 0);
   Parser := TParser.Create;
   try
      // mark selected lines for deletion and write entry to getbymids.txt
      ListPos := lstResult.Items.Count;
      i2:=0;
      for i1:=0 to lstResult.Items.Count-1 do begin
         if lstResult.Selected[i1] then begin
            Line := LongInt( lstResult.Items.Objects[i1] );
            Parser.Parse( KILLS[Line], #9 );
            If CopyToMIDstxt then begin
               RetrieveLine( PATH_SERVER + Parser.sPart( Ord(KILLIDX_SERVER),'') +
                             '\' + SRVFILE_GETMIDLIST,
                             Parser.sPart( Ord(KILLIDX_MID),'') )
            end;
            KILLS[Line] := ''; // mark for deletion below
            if ListPos > i1 then ListPos := i1;
            AlreadyExisting := False;
            for i3:=0 to i2-1 do begin
               if Parser.sPart( Ord(KILLIDX_MID),'') = List[i3].MID then begin
                  AlreadyExisting := True;
                  Break;
               end;
            end;
            if Not AlreadyExisting then begin
               inc( i2 );
               SetLength( List, i2 );
               With List[i2-1] do begin
                  MID := Parser.sPart( Ord(KILLIDX_MID),'');
                  Refs := Parser.sPart( Ord(KILLIDX_REFS),'');
                  Subject := Parser.sPart( Ord(KILLIDX_SUBJECT),'')
               end;
               if i2 >= lstResult.SelCount then Break;
            end;
         end;
      end;

      // delete answers to an selected posting
      for i1:=0 to High( List ) do
         If List[i1].MID > '' then
            for i2:=0 to High( List ) do
               If i1<>i2 then
                  if Pos( List[i1].MID, List[i2].Refs ) > 0 then
                     List[i2].MID := '';

      // mark all postings with the same message-id or answers for deletion
      For i1:=0 to lstResult.Items.Count-1 do If Kills[i1]> '' then begin
         Parser.Parse( KILLS[i1], #9 );
         for i2:=0 to High( List ) do With List[i2] do If MID > '' then begin
            if Parser.sPart( Ord(KILLIDX_MID),'') = MID then begin
               If CopyToMIDstxt then begin
                  RetrieveLine ( PATH_SERVER + Parser.sPart( Ord(KILLIDX_SERVER),'')+
                                 '\' + SRVFILE_GETMIDLIST,
                                 Parser.sPart( Ord(KILLIDX_MID),'') )
               end;
               KILLS[i1] := ''; // mark for deletion below
            end;
            if Pos( MID, Parser.sPart(Ord(KILLIDX_REFS),'') ) > 0 then begin
               If CopyToMIDstxt then begin
                  RetrieveLine ( PATH_SERVER + Parser.sPart( Ord(KILLIDX_SERVER),'')+
                                 '\' + SRVFILE_GETMIDLIST,
                                 Parser.sPart( Ord(KILLIDX_MID),'') )
               end;
               KILLS[i1] := ''; // mark for deletion below
            end;
         end;
      end;
   finally
      Parser.Free
   end;
   For i1 := Kills.Count-1 downto 0 do begin
      if KILLS[i1]='' then begin
         KILLS.Delete( i1 );
         Changed := True;
      end
   end
end;

Procedure TfrmKillsMain.RetrieveLine (Const Datei, Inhalt: String);
begin
   Retrieve.Add (Datei);
   Retrieve.Add (Inhalt);
end;

Function IsValidScoreValue (Const s: String): Boolean;
Var M, i: Integer; c: Char;
begin
   M := 0;
   For i := 1 to Length(s) do begin
      c := s[i];
      Case M of
         0: If c = '=' then Inc(M)
            else If c IN['+', '-'] then Inc(M, 2)
            else M := 99;
         1: If c IN['+', '-'] then Inc(M)
            else M := 99;
         2: If c IN['0'..'9'] then Inc(M)
            else M := 99;
         3: If Not (c IN['0'..'9']) then M := 99
      end;
      If M = 99 then break
   end;
   Result := M = 3
end;

Procedure FindOrCreateSection(Const Section: String; sl: TStrings;
   Var Von, Bis: Integer);
Var s: String; i: Integer; fertig: Boolean;
begin
   fertig := false;
   For i := 0 to sl.Count - 1 do begin
      s := Trim(sl[i]);
      If Von = 0 then begin
         If s = Section then begin
            fertig := true;
            Von := i+1;
            Bis := sl.Count-1
         end
      end else If s > '' then begin
         If (s[1] = '[') and (s[Length(s)] = ']') then begin
            Bis := i-1;
            break
         end
      end
   end;
   // Create, if doesn't exist
   If Not fertig then With sl do begin
      Insert( 0, '');
      Insert( 0, Section);
      Von := 1;
      Bis := 1;
   end;
   While Bis > sl.Count-1 do sl.Add('')
end;

Procedure TfrmKillsMain.AddHeadersIntoScoreFile (HeaderList: TThreadInfos;
   Const SfSection, SfValue: String; Const Expire: Integer );
Var //ScoreFile: TStringList;
    SfSectionBegin, SfSectionEnd, i1, i2: Integer;
    AlreadyExisting: Boolean;
begin
   //Scorefile := TStringlist.Create;
   emmScore.Lines.BeginUpdate;
   try
//      Scorefile.Assign (emmScore.Lines);
      FindOrCreateSection(SfSection, emmScore.Lines, SfSectionBegin, SfSectionEnd);
      for i1 := 0 to High( HeaderList ) do begin
         With HeaderList[i1] do begin
            If MID >'' then begin
               AlreadyExisting := False;
               for i2 := SfSectionBegin to SfSectionEnd do begin
                  if (Pos( '#', Trim( emmScore.Lines[i2] ) ) > 16)
                     and (Pos( 'references:', Lowercase( emmScore.Lines[i2] ) ) > 3)
                     and (Pos( MID, emmScore.Lines[i2] ) > 15)
                  then begin
                     AlreadyExisting := True;
                     Break
                  end
               end;
               if Not AlreadyExisting then begin
                  emmScore.Lines.Insert( SfSectionBegin,
                     SfValue + ' References: "' + MID + '" '
                     + 'Expire:'+FormatDateTime( 'yyyymmdd', Date + Expire )
                     + ' # Auto-generated '+
                     FormatDateTime('ddddd', Date) + ', Subject: ' + Subject )
               end
            end
         end
      end;
//      emmScore.Lines.BeginUpdate;
//      emmScore.Lines.Assign (Scorefile);
   finally
      emmScore.Lines.EndUpdate
//      Scorefile.Free
   end
end;

procedure TfrmKillsMain.mnuRetrieveThreadClick(Sender: TObject); // PW

  Procedure GetSectionAndValue ( Var SfSection, SfValue: String; Var Expire: Integer );
  begin
      SfSection := Trim( CfgIni.ReadString( 'Setup', 'score.RetrieveThread.section', '' ) );
      If SfSection = '' then SfSection := Def_KillLogRetrieveThreadSec;
      if (SfSection[1]<>'[') or (SfSection[Length(SfSection)]<>']') then begin
         SfSection := Def_KillLogRetrieveThreadSec;
         Log( LOGID_WARN, TrF( 'Killlog.RetrieveThread.InvalidSection',
                               'Invalid section-name for retrieving thread, using "%s"',
                               Def_KilllogRetrieveThreadSec))
      end;
      SfValue := Trim( CfgIni.ReadString( 'Setup', 'score.RetrieveThread.value', '' ) );
      if SfValue = '' then SfValue := Def_KillLogRetrieveThreadVal;
      If Not IsValidScoreValue(SfValue) then begin
         SfValue := Def_KillLogRetrieveThreadVal;
         Log( LOGID_WARN, TrF( 'Killlog.RetrieveThread.InvalidValue',
                               'Invalid scorefile-value for retrieving thread, using "%s"',
                               Def_KilllogRetrieveThreadVal ) );
      end;
      Expire := CfgIni.ReadInteger( 'Setup', 'score.RetrieveThread.expire',
                                   Def_KillLogRetrieveThreadExp );
   end;

var  Loc: Integer;
     SfSection, SfValue            : String;
     HeaderList                    : TThreadInfos;
     Expire                        : Integer; {PW}{write Expire}{13.06.2001}
// mnuRetrieveThreadClick
begin
   if lstResult.SelCount <= 0 then exit;
   // Get List of (unique) Message-IDs and kill all Lines with this Message-IDs
   ExtractSelectedHeaders(HeaderList, Loc, true);
   // Read INI-Settings
   GetSectionAndValue ( SfSection, SfValue, Expire );
   // Add to Scorefile
   AddHeadersIntoScoreFile (HeaderList, SfSection, SfValue, Expire );
   /// Update Dialog
   ViewRefresh( Loc );
end;

procedure TfrmKillsMain.mnuDeleteThreadClick(Sender: TObject); //PW

  Procedure GetSectionAndValue ( Var SfSection, SfValue: String; Var Expire: Integer );
  begin
      SfSection := Trim( CfgIni.ReadString( 'Setup', 'score.DeleteThread.section', '' ) );
      If SfSection = '' then SfSection := Def_KillLogDeleteThreadSec;
      if (SfSection[1]<>'[') or (SfSection[Length(SfSection)]<>']') then begin
         SfSection := Def_KillLogDeleteThreadSec;
         Log( LOGID_WARN,
              TrF( 'Killlog.DeleteThread.InvalidSection',
                   'Invalid section-name for deleting thread, using "%s"',
                   Def_KilllogDeleteThreadSec) );
      end;
      SfValue := Trim( CfgIni.ReadString( 'Setup', 'score.DeleteThread.value', '' ) );
      if SfValue = '' then SfValue := Def_KillLogDeleteThreadVal;
      If Not IsValidScoreValue(SfValue) then begin
         SfValue := Def_KillLogDeleteThreadVal;
         Log( LOGID_WARN, TrF( 'Killlog.DeleteThread.InvalidValue',
                               'Invalid scorefile-value for deleting thread, using "%s"',
                               Def_KilllogDeleteThreadVal) );
      end;
      Expire := CfgIni.ReadInteger( 'Setup', 'score.DeleteThread.expire',
                                   Def_KillLogDeleteThreadExp );
   end;

var  Loc: Integer;
     SfSection, SfValue            : String;
     HeaderList                    : TThreadInfos;
     Expire                        : Integer; {PW}{write Expire}{13.06.2001}
// mnuDeleteThreadClick     
begin
   if lstResult.SelCount <= 0 then exit;
   // Get List of (unique) Message-IDs and kill all Lines with this Message-IDs
   ExtractSelectedHeaders(HeaderList, Loc, false);
   // Read INI-Settings
   GetSectionAndValue ( SfSection, SfValue, Expire );
   // Add to Scorefile
   AddHeadersIntoScoreFile (HeaderList, SfSection, SfValue, Expire );
   /// Update Dialog
   ViewRefresh( Loc );
end;
{/PW}

procedure TfrmKillsMain.Help1Click(Sender: TObject);
begin
   Help
end;

procedure TfrmKillsMain.emmScoreChange(Sender: TObject);
begin
   changed := true
end;

procedure TfrmKillsMain.mnuSaveClick(Sender: TObject);
begin
   Save
end;

procedure TfrmKillsMain.mnuUndoClick(Sender: TObject);
begin
   Load;
   ViewRefresh( 0 )
end;

procedure TfrmKillsMain.File1Click(Sender: TObject);
begin
   mnuUndo.Enabled := changed;
   mnuSave.Enabled := changed
end;

procedure TfrmKillsMain.FormActivate(Sender: TObject);
begin
   OnActivate := NIL;
   pg.ActivePageIndex := 0;
   lstResult.SetFocus
end;

procedure TfrmKillsMain.lstResultKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
   If (Key = vk_Delete) and (Shift = []) then begin
      mnuDelete.OnClick(NIL); Key := 0
   end
end;

procedure TfrmKillsMain.FormKeyPress(Sender: TObject; var Key: Char);
begin
   If Key = #27 then begin Key := #0; Close end
end;

end.
