// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit HMailKillsMain;

interface

uses Windows, Menus, ComCtrls, StdCtrls, Classes, Controls, Forms, cStdForm,
  ExtCtrls;

Procedure DialogMailKillsMain;
Procedure DialogMailKillfileLogClose;
Function  IsDialogMailKillfileLogOpen: Boolean;

Type
  TKillIndex = ( KILLIDX_TYPE, KILLIDX_SERVER, KILLIDX_ACCOUNT, KILLIDX_UIDL,
                 KILLIDX_SCORE, KILLIDX_SUBJECT, KILLIDX_FROM, KILLIDX_TO,
                 KILLIDX_CC, KILLIDX_DATE, KILLIDX_MID, KILLIDX_REFS,
                 KILLIDX_BYTES, KILLIDX_LINES );

Var
  SORTSEQ  : array[ TKillIndex, 1..2 ] of TKillIndex = (
               (KILLIDX_SERVER, KILLIDX_ACCOUNT), (KILLIDX_ACCOUNT, KILLIDX_DATE),
               (KILLIDX_SERVER, KILLIDX_DATE), (KILLIDX_SERVER, KILLIDX_ACCOUNT),
               (KILLIDX_SERVER, KILLIDX_ACCOUNT), (KILLIDX_SERVER, KILLIDX_ACCOUNT),
               (KILLIDX_SERVER, KILLIDX_ACCOUNT), (KILLIDX_SERVER, KILLIDX_ACCOUNT),
               (KILLIDX_SERVER, KILLIDX_ACCOUNT), (KILLIDX_SERVER, KILLIDX_ACCOUNT),
               (KILLIDX_SERVER, KILLIDX_ACCOUNT), (KILLIDX_SERVER, KILLIDX_ACCOUNT),
               (KILLIDX_SERVER, KILLIDX_ACCOUNT), (KILLIDX_SERVER, KILLIDX_ACCOUNT)
             );

type
  TThreadInfo = Record MID, Refs, Subject: String end;
  TThreadInfos = Array of TThreadInfo;

  TfrmMailKillsMain = class(THForm)
    PopKills: TPopupMenu;
    mnuMarkDelete: TMenuItem;
    mnuDelete: TMenuItem;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Exit1: TMenuItem;
    N1: TMenuItem;
    N2: TMenuItem;
    N3: TMenuItem;
    mnuSelectAll: TMenuItem;
    mnuExit2: TMenuItem;
    Setsortsequence1: TMenuItem;
    N5: TMenuItem;
    mnuShowDetails: TMenuItem;
    Help1: TMenuItem;
    mnuEntries: TMenuItem;
    N7: TMenuItem;
    mnuSave: TMenuItem;
    mnuUndo: TMenuItem;
    Markforretrieval1: TMenuItem;
    pg: TPageControl;
    tsEntries: TTabSheet;
    tsSettings: TTabSheet;
    Label1: TLabel;
    Label23: TLabel;
    emToplines: TEdit;
    tsScorefile: TTabSheet;
    pnlEdit: TPanel;
    emmScore: TRichEdit;
    tabAccounts: TTabControl;
    hdrResult: THeaderControl;
    lstResult: TListBox;
    optNotifySubject_OldStyle: TRadioButton;
    optNotifySubject_ContainsSubject: TRadioButton;
    Label2: TLabel;
    chkGenerateXHamsterTo: TCheckBox;
    chkGenerateXHamsterAccountReason: TCheckBox;
    chkGenerateXHamsterPostToReason: TCheckBox;
    chkGenerateXHamsterScoring: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure mnuMarkRetrClick(Sender: TObject);
    procedure mnuDeleteClick(Sender: TObject);
    procedure lstResultDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure hdrResultSectionResize(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure lstResultDblClick(Sender: TObject);
    procedure hdrResultSectionClick(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure Exit1Click(Sender: TObject);
    procedure mnuSelectAllClick(Sender: TObject);
    procedure Setsortsequence1Click(Sender: TObject);
    procedure Help1Click(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure mnuSaveClick(Sender: TObject);
    procedure mnuUndoClick(Sender: TObject);
    procedure File1Click(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure lstResultKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure tabAccountsChange(Sender: TObject);
    procedure mnuMarkDeleteClick(Sender: TObject);
    procedure OnSettingChange(Sender: TObject);
    procedure emmScoreChange(Sender: TObject);
  private
    { Private-Deklarationen }
    KILLS: array of TStringList;
    CurrentKILLS: TStringList;
    Accounts, UIDLs: TStringList;
    KILLSORT : TKillIndex;
    InvertedSort, EntriesChanged, SettingsChanged, ScorefileChanged,
      ScorefileLoaded: Boolean;
    procedure ViewRefresh( Location: Integer );
    procedure Load;
    procedure Save;
    procedure Mark( UIDLMARKER: String );
  public
    { Public-Deklarationen }
  end;

implementation

{$R *.DFM}

uses Messages, Graphics, Clipbrd, SysUtils, uTools, uEncoding, uDateTime,
     uCRC32, cLogFile, Global, Config, dInput, cArticle, cFiltersMail;

Var Dlg: TfrmMailKillsMain = NIL;

Procedure DialogMailKillsMain;
begin
  Dlg := TfrmMailKillsMain.Create(NIL);
  try
     Dlg.Showmodal
  finally
     FreeAndNil(Dlg)
  end
end;
Procedure DialogMailKillfileLogClose;
begin
   If Assigned(Dlg) then try
      Dlg.ModalResult := mrCancel
   except end
end;
Function IsDialogMailKillfileLogOpen: Boolean;
begin
   Result := Assigned(Dlg)
end;

procedure TfrmMailKillsMain.ViewRefresh( Location: Integer );
Var Parser: TParser;

   function SortValue( SortItem: TKillIndex ): String;
   begin
        Result := Lowercase( Parser.sPart( Ord(SortItem), '' ) );

        case SortItem of
           KILLIDX_SUBJECT: begin
              if copy(Result,1,4)='re: ' then begin
                 System.Delete(Result,1,4);
                 Result := Result + 'z';
              end;
              If Result > '' then Result := DecodeHeadervalue( Result );
           end;
           KILLIDX_FROM:
              If Result > '' then Result := FilterNameOfFrom( DecodeHeadervalue( Result ) )
                             else Result := '';
           KILLIDX_DATE:
              Result := FormatDateTime( 'yyyy.mm.dd hh:nn:ss', RfcDateTimeToDateTimeGMT(Result) );
           KILLIDX_TYPE, KILLIDX_SCORE, KILLIDX_BYTES, KILLIDX_LINES:
              Result := Format( '%8d', [strtoint(Result)] );
       end;
   end;

var  Line, p, i: Integer;
     Srt: String; 
     LocTop: Integer;
     KILLSORT2, KILLSORT3: TKillIndex;
Const
   Sortchars = ' 0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
begin
   if not Assigned(CurrentKILLS) then begin
      lstResult.Items.Clear;
      exit;
   end;
   LocTop := lstResult.TopIndex;
   lstResult.Items.BeginUpdate;
   lstResult.Items.Clear;
   lstResult.Sorted := True;

   Srt := '';
   Parser := TParser.Create;
   try

      KILLSORT2 := SORTSEQ[KILLSORT, 1];
      KILLSORT3 := SORTSEQ[KILLSORT, 2];

      for Line:=0 to CurrentKILLS.Count-1 do begin
         if CurrentKILLS[Line]<>'' then begin

            Parser.Parse( CurrentKILLS[Line], #9 );
            Srt := SortValue( KILLSORT );
            if KILLSORT2<>KILLSORT  then Srt := Srt + '|' + SortValue( KILLSORT2 );
            if KILLSORT3<>KILLSORT2 then Srt := Srt + '|' + SortValue( KILLSORT3 );
            Srt := Srt + '|' + inttostr( StrToCRC32( CurrentKILLS[Line] ) );

            If InvertedSort then begin
               For i:=1 to Length(Srt) do begin
                  p := Pos(UpCase(Srt[i]), SortChars);
                  If p>0 then Srt[i]:=SortChars[Length(SortChars)-p+1]
               end
            end;

            try
               lstResult.Items.AddObject( Srt, Pointer(Line) )
            except
               Log( LOGID_WARN,
                       TrF( 'Killlog.TooManyEntries',
                            'There are too many entries in killfile - the last %s entries could not be shown.',
                       IntToStr(CurrentKILLS.Count-Line-1)));
               break;
            end;
         end;
      end;

      if (Location>=0) and (Location<lstResult.Items.Count) then begin
         Application.ProcessMessages;
         if (LocTop>=0) and (LocTop<lstResult.Items.Count) then begin
            lstResult.TopIndex := LocTop;
         end;
         lstResult.Selected[Location] := True;
         lstResult.ItemIndex := Location;
      end;
      lstResult.Items.EndUpdate;
   finally
      Parser.Free;
   end
end;

procedure TfrmMailKillsMain.FormCreate(Sender: TObject);
var  i: Integer; ki: TKillIndex; M: TMenuItem; x: Word;
   SearchResult: TSearchRec;
begin
   pg.ActivePageIndex := 0;
   
   UIDLs := TStringList.Create;
   Accounts := TStringList.Create;
   if FileExists( PATH_MAILS + CFGFILE_MAILKILLLOG ) then Accounts.Add( '*' );

   if FindFirst( PATH_MAILS + '*.*', faDirectory, SearchResult ) = 0 then begin
      repeat
         if FileExists( PATH_MAILS + SearchResult.Name + '\' + CFGFILE_MAILKILLLOG ) then begin
            if SearchResult.Name <> '.' then Accounts.Add( SearchResult.Name );
         end;
      until FindNext( SearchResult ) <> 0;
      FindClose( SearchResult );
   end;
   SetLength( KILLS, Accounts.Count );
   for i := 0 to Accounts.Count-1 do KILLS[i] := TStringlist.Create;

   tabAccounts.Tabs := Accounts;

   LoadWindowState( Self, 'MailKillList' );

   For i := 0 to PopKills.Items.Count-1-2 do begin
      M := TMenuItem.Create(self);
      With PopKills.Items[i] do begin
         M.Caption := Caption;
         M.ShortCut := ShortCut;
         M.OnClick := OnClick;
         M.Hint := Hint
      end;
      mnuEntries.Add ( M )
   end;

   for i:=0 to hdrResult.Sections.Count-1 do begin
      hdrResult.Sections[i].Width := CfgIni.ReadInteger( 'MailKillList',
         'ColWidth' + inttostr( i ), hdrResult.Sections[i].Width );
   end;

   for ki:=Low(TKillIndex) to High(TKillIndex) do begin
      x := Ord(SORTSEQ[ki, 1]) * 256 + Ord(SORTSEQ[ki, 2]);
      x := CfgIni.ReadInteger( 'MailKillList', 'SortSeq' + inttostr( Ord(ki) ), x);
      SORTSEQ[ki, 1] := TKillIndex(x div 256);
      SORTSEQ[ki, 2] := TKillIndex(x mod 256);
   end;

   //if Accounts.Count > 0 then
   Load;

   KILLSORT := TKillIndex(CfgIni.ReadInteger( 'MailKillList', 'SortBy', Ord(KILLIDX_SERVER) ));
   InvertedSort := CfgIni.ReadBool( 'MailKillList', 'SortInverted', false );
   ViewRefresh( 0 );
end;

procedure TfrmMailKillsMain.FormDestroy(Sender: TObject);
var i: Integer;
begin
   CfgIni.WriteInteger( 'MailKillList', 'SortBy', Ord(KILLSORT) );
   CfgIni.WriteBool( 'MailKillList', 'SortInverted', InvertedSort );
   While mnuEntries.Count > 0 do mnuEntries.Delete(0);
   for i := 0 to Length(KILLS)-1 do KILLS[i].Free;
   UIDLs.Free;
end;


procedure TfrmMailKillsMain.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
   if EntriesChanged or ScorefileChanged or SettingsChanged then begin
      if Application.MessageBox( PChar(Tr('SaveChanges.ask', 'Save changes?')),
           PChar(Caption),
           MB_ICONQUESTION + MB_YESNO)=IDYes
      then Save
   end;
   SaveWindowState( Self, 'MailKillList' );
end;

Procedure TfrmMailKillsMain.Load;
var i: Integer; b: Boolean; s: String;
begin
   UIDLs.Clear;

   // Killfile-Entries
   if Accounts.Count > 0 then begin
      CurrentKILLS := KILLS[0];
      if Accounts[0] = '*'
         then KILLS[0].LoadFromFile( PATH_MAILS + CFGFILE_MAILKILLLOG )
         else KILLS[0].LoadFromFile( PATH_MAILS + Accounts[0] + '\' + CFGFILE_MAILKILLLOG );
      for i := 1 to Length(KILLS)-1 do begin
         KILLS[i].LoadFromFile( PATH_MAILS + Accounts[i] + '\' + CFGFILE_MAILKILLLOG );
      end
   end;

   // Settings
   With CfgIni do begin
      emToplines.Text := IntToStr(ReadInteger('Setup', 'mail.filter.toplines', 20));
      b := ReadBool('Setup', 'mail.infomail.addheadertosubject', false);
      optNotifySubject_OldStyle.Checked := Not b;
      optNotifySubject_ContainsSubject.Checked := b;
      s := LowerCase(','+ReadString('Setup', 'mail.dont-generate-xheader', '')+',');
      chkGenerateXHamsterTo.checked := Pos(',to,', s)=0;
      chkGenerateXHamsterAccountReason.checked := Pos(',accountreason,', s)=0;
      chkGenerateXHamsterPostToReason.checked := Pos(',posttoreason,', s)=0;
      chkGenerateXHamsterScoring.checked := Pos(',scoring,', s)=0;
   end;

   // Scorefile
   ScorefileLoaded := false;
   emmScore.Lines.BeginUpdate;
   try
      if FileExists2( PATH_BASE + CFGFILE_MFILTER )
         then emmScore.Lines.LoadFromFile( PATH_BASE + CFGFILE_MFILTER )
         else emmScore.Lines.Clear;
   finally
      emmScore.Lines.EndUpdate
   end;
   ScorefileLoaded := true;

   EntriesChanged := false;
   SettingsChanged := false;
   ScorefileChanged := false
end;

Procedure TfrmMailKillsMain.Save;
type PStringList = ^TStringList;

    procedure SaveOrDelete( PList: PStringList; FileName: String );
    begin
       if PList.Count > 0 then PList.SaveToFile( FileName )
       else DeleteFile( FileName );
    end;

Var i: Integer; s: String;
begin
   If EntriesChanged then begin
      for i := 0 to UIDLs.Count-1 do begin
          MailHistory.AddUIDL( UIDLs[i], 0 );
      end;
      // Killfile-Entries
      if Accounts.Count > 0 then begin
         if Accounts[0] = '*'
            then SaveOrDelete( @KILLS[0], PATH_MAILS + CFGFILE_MAILKILLLOG )
            else SaveOrDelete( @KILLS[0], PATH_MAILS + Accounts[0] + '\' + CFGFILE_MAILKILLLOG );
         for i := 1 to Length(KILLS)-1 do begin
            SaveOrDelete( @KILLS[i], PATH_MAILS + Accounts[i] + '\' + CFGFILE_MAILKILLLOG );
         end;
      end;
      EntriesChanged := false
   end;

   If SettingsChanged then With CfgIni do begin
      With emToplines do begin
         If Text = '' then begin
            DeleteKey('Setup', 'mail.filter.toplines')
         end else try
            WriteInteger('Setup', 'mail.filter.toplines', StrToInt(Trim(Text)))
         except end
      end;
      WriteBool('Setup', 'mail.infomail.addheadertosubject', optNotifySubject_ContainsSubject.Checked);
      s := '';
      If Not chkGenerateXHamsterTo.checked then s := Iif(s>'', s+',') + 'To';
      If Not chkGenerateXHamsterAccountReason.checked then s := Iif(s>'', s+',') + 'AccountReason';
      If Not chkGenerateXHamsterPostToReason.checked then s := Iif(s>'', s+',') + 'PostToReason';
      If Not chkGenerateXHamsterScoring.checked then s := Iif(s>'', s+',') + 'Scoring';
      If s > '' then WriteString('Setup', 'mail.Dont-Generate-XHeader', s)
                else DeleteKey('Setup', 'mail.Dont-Generate-XHeader');
      SettingsChanged := false
   end;

   // Scorefile
   If ScorefileChanged then begin
      If ScorefileLoaded then emmScore.Lines.SaveToFile( PATH_BASE + CFGFILE_MFILTER );
      ScorefileChanged := false
   end;
end;

procedure TfrmMailKillsMain.lstResultDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var  R  : TRect;
     Line, Col: Integer;
     Parser: TParser;
     s: String;

     procedure WriteCol( No: Integer; Txt: String );
     var  l, w : Integer;
     begin
          l := hdrResult.Sections.Items[No].Left;
          w := hdrResult.Sections.Items[No].Width;

          R.Left  := l + 1;
          R.Right := R.Left + w - 3;
          R.Top   := Rect.Top;
          R.Bottom:= Rect.Bottom;
          lstResult.Canvas.TextRect( R, R.Left, R.Top, Txt );

          lstResult.Canvas.MoveTo( l+w-1, R.Top    );
          lstResult.Canvas.LineTo( l+w-1, R.Bottom );
     end;
begin
   Line := LongInt( lstResult.Items.Objects[Index] );

   with lstResult.Canvas do begin

      if odSelected in State then begin
         Brush.Color:= clHighlight;
         Font.Color  := clYellow;
         Pen.Color := clLtGray;
      end else begin
         Brush.Color := clWindow;
         Font.Color   := clWindowText;
         Pen.Color := clLtGray;
      end;

      FillRect(Rect);

      Parser := TParser.Create;
      try
         Parser.Parse( CurrentKILLS[Line], #9 );

         for Col:=0 to hdrResult.Sections.Count-1 do begin
            s := Parser.sPart(Col,'');
            if s <> '' then begin;
               case TKillIndex(Col) of
                  KILLIDX_TYPE: begin
                     case StrToIntDef( Parser.sPart( Ord(KILLIDX_TYPE), '0' ), 0 ) of
                        NOTIFYTYPE_TEST  : s := Tr( 'Type.Test', 'Test' );
                        NOTIFYTYPE_KILL  : s := Tr( 'Type.Kill', 'Filter-Kill' );
                        NOTIFYTYPE_IGNORE: s := Tr( 'Type.Ignore', 'Filter-Ignore' );
                        NOTIFYTYPE_GLOBALSIZE_IGNORE: s := Tr( 'Type.SizeIgnore', 'Size-Ignore' );
                        NOTIFYTYPE_GLOBALSIZE_KILL  : s := Tr( 'Type.SizeKill', 'Size-Kill' );
                        NOTIFYTYPE_GLOBALSIZE_NOTIFY: s := Tr( 'Type.SizeNotify', 'Size-Notify' );
                     end;
                  end;
                  KILLIDX_SUBJECT: s := DecodeHeadervalue( s );
                  KILLIDX_FROM:    s := FilterNameOfFrom( DecodeHeaderValue( s ) );
                  KILLIDX_DATE:    s := FormatDateTime( 'dd.mm.yyyy hh:nn',
                                        DateTimeGMTToLocal( RfcDateTimeToDateTimeGMT(s) ) );
               end;
            end;
            WriteCol( Col, s );
         end;
      finally
         Parser.Free
      end
   end;
end;

procedure TfrmMailKillsMain.hdrResultSectionResize(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
     lstResult.Refresh;
     CfgIni.WriteInteger( 'MailKillList', 'ColWidth' + inttostr( Section.Index ), Section.Width );
end;

procedure TfrmMailKillsMain.lstResultDblClick(Sender: TObject);
var  i: Integer;
     s, h, Info: String;
     p   : TParser;
begin
     if not Assigned(CurrentKILLS) then exit;
     if lstResult.ItemIndex<0 then exit;
     if lstResult.ItemIndex>=lstResult.Items.Count then exit;

     s := CurrentKILLS[LongInt( lstResult.Items.Objects[ lstResult.ItemIndex ] )];
     p := TParser.Create;
     try
       p.Parse( s, #9 );
       case StrToIntDef( p.sPart( Ord(KILLIDX_TYPE), '0' ), 0 ) of
          NOTIFYTYPE_TEST  : s := Tr( 'Type.Test', 'Test' );
          NOTIFYTYPE_KILL  : s := Tr( 'Type.Kill', 'Filter-Kill' );
          NOTIFYTYPE_IGNORE: s := Tr( 'Type.Ignore', 'Filter-Ignore' );
          NOTIFYTYPE_GLOBALSIZE_IGNORE: s := Tr( 'Type.SizeIgnore', 'Size-Ignore' );
          NOTIFYTYPE_GLOBALSIZE_KILL  : s := Tr( 'Type.SizeKill', 'Size-Kill' );
          NOTIFYTYPE_GLOBALSIZE_NOTIFY: s := Tr( 'Type.SizeNotify', 'Size-Notify' );
       end;
       s := Tr( 'DblClickInfo.Type', 'Type' ) + ':' + s + #13#10
          + Tr( 'DblClickInfo.Mailserver', 'Mailserver' ) + ': ' + p.sPart( Ord(KILLIDX_SERVER), '' ) + #13#10
          + Tr( 'DblClickInfo.Account', 'Account' ) + ': ' + p.sPart( Ord(KILLIDX_ACCOUNT), '' ) + #13#10
          + Tr( 'DblClickInfo.UIDL', 'UIDL' ) + ': ' + p.sPart( Ord(KILLIDX_UIDL), '' ) + #13#10
          + Tr( 'DblClickInfo.Score', 'Score (original)' ) + ': ' + p.sPart( Ord(KILLIDX_SCORE), '' ) + #13#10#13#10;

       if StrToIntDef( p.sPart( Ord(KILLIDX_TYPE), '' ), -1 ) < 10 then begin
          h := p.sPart( Ord(KILLIDX_SUBJECT), '' );
          s := s + Tr( 'DblClickInfo.Subject', 'Subject' ) + ': ' + h + #13#10;
          if Pos('=?',h)>0 then s := s + '~' + Tr( 'DblClickInfo.Subject', 'Subject' )+': ' + DecodeHeadervalue(h) + #13#10;

          h := p.sPart( Ord(KILLIDX_FROM),'');
          s := s + Tr('DblClickInfo.From', 'From')+': ' + h + #13#10;
          if Pos('=?',h)>0 then s := s + '~' + Tr('DblClickInfo.From', 'From')+': ' + DecodeHeadervalue(h) + #13#10;
          s := s
             + Tr('DblClickInfo.To', 'To')+': ' + p.sPart( Ord(KILLIDX_TO), '' ) + #13#10
             + Tr('DblClickInfo.CC', 'CC')+': ' + p.sPart( Ord(KILLIDX_CC), '' ) + #13#10;

          try
             i := Trunc( NowGMT - RfcDateTimeToDateTimeGMT(p.sPart(Ord(KILLIDX_DATE),'')) );
          except
             i := 0;
          end;
          s := s
             + Tr('DblClickInfo.Date', 'Date')+': '          + p.sPart( Ord(KILLIDX_DATE),'') + #13#10
             + Tr('DblClickInfo.Age', 'Age (current)')+': ' + inttostr(i) + #13#10#13#10

             + Tr('DblClickInfo.MID', 'Message-ID')+': ' + p.sPart( Ord(KILLIDX_MID),'') + #13#10
             + Tr('DblClickInfo.References', 'References')+': ' + p.sPart( Ord(KILLIDX_REFS),'') + #13#10#13#10
             + Tr('DblClickInfo.Bytes', 'Bytes')+': '      + p.sPart( Ord(KILLIDX_BYTES),'') + #13#10
             + Tr('DblClickInfo.Lines', 'Lines')+': '      + p.sPart( Ord(KILLIDX_LINES),'');
       end else begin
          s := s + Tr('DblClickInfo.Bytes', 'Bytes')+': '      + p.sPart( Ord(KILLIDX_BYTES),'');
       end;
     finally
       p.Free;
     end;

     Info := s;

     s := s + #13#10 + '______________________________________' + #13#10 + #13#10
            + Tr('CopyToClipboard.Question', 'Copy Info to clipboard?');

     If Application.MessageBox( PChar(s), PChar(Caption),
          MB_ICONQUESTION + MB_YESNO + MB_DEFBUTTON1) = IDYES
     then begin
        Clipboard.Clear;
        Clipboard.AsText := Info
     end;
end;

procedure TfrmMailKillsMain.hdrResultSectionClick(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
   If KILLSORT<>TKillIndex(Section.Index) then begin
      KILLSORT := TKillIndex(Section.Index);
      InvertedSort := false
   end else begin
      InvertedSort := Not InvertedSort
   end;
   ViewRefresh( 0 )
end;

procedure TfrmMailKillsMain.mnuMarkRetrClick(Sender: TObject);
begin
     Mark( UIDLMARKER_RETRIEVE );
end;

procedure TfrmMailKillsMain.mnuMarkDeleteClick(Sender: TObject);
begin
     Mark( UIDLMARKER_KILL );
end;

procedure TfrmMailKillsMain.Mark( UIDLMARKER: String );
var  i, Line, Loc, KillLog_Type: Integer;
     Parser            : TParser;
     UIDL              : String;
     ShowWarningNoUIDL,
     ShowWarningKill   : Boolean;
begin
     if not Assigned(CurrentKILLS) then exit;
     if lstResult.SelCount<=0 then exit;

     Parser := TParser.Create;
     ShowWarningNoUIDL := True;
     ShowWarningKill   := True;
     try
        Loc := -1;
        for i:=0 to lstResult.Items.Count-1 do begin
           if lstResult.Selected[i] then begin
              Line := LongInt( lstResult.Items.Objects[i] );

              Parser.Parse( CurrentKILLS[Line], #9 );
              KillLog_Type := StrToIntDef( Parser.sPart( Ord(KILLIDX_TYPE), '0' ), 0 );
              if (KillLog_Type = NOTIFYTYPE_KILL) or
                 (KillLog_Type = NOTIFYTYPE_GLOBALSIZE_KILL)
              then begin
                 if ShowWarningKill then begin
                    Application.MessageBox( PChar( Tr( 'message.MarkKill',
                       'At least one marked mail has already been killed on server. It cannot be marked for retrieval or kill.' ) ), PChar(Caption), mb_ICONINFORMATION );
                    ShowWarningKill := False;
                 end;
              end else begin
                 UIDL := Parser.sPart( Ord(KILLIDX_UIDL), '' );
                 if UIDL <> '' then UIDLs.Add( UIDLMARKER + UIDL )
                 else begin
                    if ShowWarningNoUIDL then begin
                       Application.MessageBox( PChar( Tr( 'message.noUIDL',
                          'At least one marked line has no UIDL. Mails without UIDL cannot be marked for retrieval or kill on server.' ) ), PChar(Caption), mb_ICONINFORMATION );
                       ShowWarningNoUIDL := False;
                    end;
                 end;
              end;

              CurrentKILLS[Line] := ''; // mark for deletion below
              Loc := i + 1;
           end;
        end
     finally
        Parser.Free
     end;

     i := 0;
     while i<CurrentKILLS.Count do begin
        if CurrentKILLS[i]='' then begin
           CurrentKILLS.Delete( i );
           dec(Loc);
           EntriesChanged := True;
        end else begin
           inc(i);
        end;
     end;

     ViewRefresh( Loc );
end;
procedure TfrmMailKillsMain.mnuDeleteClick(Sender: TObject);
var i, Line, Loc: Integer;
    Parser: TParser;
    ShowWarningDeleteIgnoreLine: Boolean;
    DeleteIgnoresOnServer:       Boolean;
    ShowWarningNoUIDL:           Boolean;
    UIDL: String;
begin
     if not Assigned(CurrentKILLS) then exit;
     if lstResult.SelCount<=0 then exit;

     ShowWarningDeleteIgnoreLine := True;
     ShowWarningNoUIDL           := True;
     DeleteIgnoresOnServer       := False;
     Parser := TParser.Create;
     try

        Loc := -1;
        for i:=0 to lstResult.Items.Count-1 do begin
           if lstResult.Selected[i] then begin
              Line := LongInt( lstResult.Items.Objects[i] );
              if def_filterbyuidl and
                 (    ( Copy( CurrentKILLS[Line], 1, 2 ) = '2'#9 {NOTIFYTYPE_IGNORE} )
                   or ( Copy( CurrentKILLS[Line], 1, 3 ) = '12'#9 {NOTIFYTYPE_GLOBALSIZE_NOTIFY} )
                  )
              then begin
                 if ShowWarningDeleteIgnoreLine then begin
                    if Application.MessageBox( PChar( Tr( 'message.DeleteIgnoreLine',
                          'At least one marked line is a mail which has been ignored. '+
                          'If you do not delete if on the server, it will remain there. '+
                          'Do you want to delete this mail on the server?'+#13#10#13#10+
                          'Warning: This will delete the mail ultimately!' ) ), PChar(Caption),
                          MB_YESNO ) = idyes
                     then DeleteIgnoresOnServer := True;
                    ShowWarningDeleteIgnoreLine := False;
                 end;
                 if DeleteIgnoresOnServer then begin
                    Parser.Parse( CurrentKILLS[Line], #9 );
                    UIDL := Parser.sPart( Ord(KILLIDX_UIDL), '' );
                    if UIDL <> '' then UIDLs.Add( UIDLMARKER_KILL + UIDL )
                    else begin
                       if ShowWarningNoUIDL then begin
                          Application.MessageBox( PChar( Tr( 'message.noUIDL',
                             'At least one marked line has no UIDL. Mails without UIDL cannot be marked for retrieval or kill on server.' ) ), PChar(Caption), mb_ICONINFORMATION );
                         ShowWarningNoUIDL := False;
                       end;
                    end;
                 end;
              end;
              CurrentKILLS[Line] := ''; // mark for deletion below
              Loc := i + 1;
           end;
        end;

     finally
        FreeAndNil( Parser );
     end;

     i := 0;
     while i<CurrentKILLS.Count do begin
        if CurrentKILLS[i]='' then begin
           CurrentKILLS.Delete( i );
           dec(Loc);
           EntriesChanged := True;
        end else begin
           inc(i);
        end;
     end;

     ViewRefresh( Loc );
end;

procedure TfrmMailKillsMain.Exit1Click(Sender: TObject);
begin
     Close;
end;

procedure TfrmMailKillsMain.mnuSelectAllClick(Sender: TObject);
begin
     SendMessage( lstResult.Handle, LB_SELITEMRANGEEX, 0, lstResult.Items.Count-1 );
end;

procedure TfrmMailKillsMain.Setsortsequence1Click(Sender: TObject);
var  s: String;
     i, i2, i3: TKillIndex;
     Neu1, Neu2, Neu3, MinVal, MaxVal: Integer;
begin
   i2 := SORTSEQ[KILLSORT, 1];
   i3 := SORTSEQ[KILLSORT, 2];

   s := #13#10#13#10 + Tr('SortSequence.Fieldlist', 'Fields')+':' + #13#10;
   for i:=Low(TKillIndex) to High(TKillIndex) do begin
      s := s + inttostr(Ord(i)) + '=' + hdrResult.Sections[Ord(i)].Text;
      if i < High(TKillIndex) then s:=s+', ';
   end;
   s := s + #13#10#13#10;

   Neu1 := Ord(Killsort);
   Neu2 := Ord(i2);
   Neu3 := Ord(i3);
   MinVal := Ord(Low(TKillIndex));
   MaxVal := Ord(High(TKillIndex));
   if not InputDlgInt( TrF('SortSequence.Caption', 'Sort-sequence for "%s"', hdrResult.Sections[Neu1].text),
                       Tr('SortSequence.CurrentSequence', 'Current sequence')+':' + #13#10
                       + hdrResult.Sections[Neu1].text + ', '
                       + '>>' + hdrResult.Sections[Neu2].text + '<<, '
                       + hdrResult.Sections[Neu3].text
                       + s
                       + TrF('SortSequence.Enter2ndSortField', 'Enter second sort-field (%s..%s):', [IntToStr(MinVal), inttostr(MaxVal)]),
                       Neu2, MinVal, MaxVal, 0{HlpKillOverview_SortSequence} ) then exit;
   if not InputDlgInt( TrF('SortSequence.Caption', 'Sort-sequence for "%s"', hdrResult.Sections[Neu1].text),
                       Tr('SortSequence.CurrentSequence', 'Current sequence')+':' + #13#10
                         + hdrResult.Sections[Neu1].text + ', '
                         + hdrResult.Sections[Neu2].text + ', '
                         + '>>' + hdrResult.Sections[Neu3].text + '<<'
                         + s
                         + TrF('SortSequence.Enter3rdSortField', 'Enter third sort-field (%s..%s):', [inttostr(MinVal), IntToStr(MaxVal)]),
                       Neu3, MinVal, MaxVal, 0{HlpKillOverview_SortSequence} )
      then exit;

   SORTSEQ[KILLSORT, 1] := TKillIndex(Neu2);
   SORTSEQ[KILLSORT, 2] := TKillIndex(Neu3);
   CfgIni.WriteInteger( 'MailKillList', 'SortSeq' + inttostr( Neu1 ), Neu2*256 + Neu3 );
   ViewRefresh( 0 );
end;

procedure TfrmMailKillsMain.Help1Click(Sender: TObject);
begin
   Help
end;

procedure TfrmMailKillsMain.mnuSaveClick(Sender: TObject);
begin
   Save
end;

procedure TfrmMailKillsMain.mnuUndoClick(Sender: TObject);
begin
   Load;
   ViewRefresh( 0 )
end;

procedure TfrmMailKillsMain.File1Click(Sender: TObject);
Var b: Boolean;
begin
   b := EntriesChanged or SettingsChanged or ScorefileChanged;
   mnuUndo.Enabled := b;
   mnuSave.Enabled := b
end;

procedure TfrmMailKillsMain.FormActivate(Sender: TObject);
begin
   OnActivate := NIL;
   lstResult.SetFocus
end;

procedure TfrmMailKillsMain.lstResultKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
   If (Key = vk_Delete) and (Shift = []) then begin
      mnuDelete.OnClick(NIL); Key := 0
   end
end;

procedure TfrmMailKillsMain.FormKeyPress(Sender: TObject; var Key: Char);
begin
   If Key = #27 then begin Key := #0; Close end
end;

procedure TfrmMailKillsMain.tabAccountsChange(Sender: TObject);
begin
     if tabAccounts.TabIndex = -1 then CurrentKILLS := nil
     else CurrentKILLS := KILLS[tabAccounts.TabIndex];
     ViewRefresh( 0 );
end;

procedure TfrmMailKillsMain.OnSettingChange(Sender: TObject);
begin
   Settingschanged := true
end;

procedure TfrmMailKillsMain.emmScoreChange(Sender: TObject);
begin
   ScorefileChanged := true
end;

end.
