// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cAccount;

interface

uses Classes, contnrs, IniFiles, FileCtrl, stdctrls;

Const AccountChars: Set of Char
         =  ['!', '#', '$', '&', '''', '-', '_',
             '0'..'9', '=', '.', 'A'..'Z', 'a'..'z'];

Function NormalizeAccountName (Const Account: String): String;

type
  // MBTP_NONE= 0; //not used MBTP_POP3= 1; //"normal"/default, MBTP_IMAP = 2;
  TMailboxtype = (mbtNone, mbtPOP3, mbtIMAP);
  TLocalmailtype = (lmtNormal,   // LOCALMAILTYPE_NORMAL=0; // known local recipient
                    lmtInvalid,  // LOCALMAILTYPE_INVALID = 1; //.invalid -> assume local, send to admin
                    lmtUnknown ); // LOCALMAILTYPE_UNKNOWN = 2; // unknown local recipient -> notify admin
  TTriValue = (vNo, vYes, vDefault);

type
  TAccounts  = class;    // Master-class
  TAccGroups = class;    // list of groups
  TAccGroup  = class;    // single group
  TAccUsers  = class;    // list of users/accounts
  TAccUser   = class;    // single user/account

  TAccounts = class
    private
      FUsers: TAccUsers;
      FGroups: TAccGroups;
      Procedure Load;
    public
      constructor Create;
      destructor Destroy; override;
      Property Users: TAccUsers read FUsers;
      Property Groups: TAccGroups read FGroups;
      function  IsLocalMailbox( MailAddr: String;
         out UserID: Integer; Out LocalType: TLocalmailtype; Out bLocal: boolean ): Boolean;
      function  IMAPMailboxLock( MailboxPath: String; LockIt: Boolean ): Boolean;
      procedure SetIMAPMailbox( MailboxPath: String; Mailbox: TObject );
      function  GetIMAPMailbox( MailboxPath: String ): TObject;
      procedure LogAuthenticatedLogin( FromIP: LongInt );
      function  ChkAuthenticatedLogin( FromIP: LongInt ): Boolean;
      function  LoginID( Const UserName, Password: String ): Integer;
      procedure Lock;
      procedure Unlock;
      Procedure  Reload;
  end;

  TAccGroups = Class
    private
      FParent: TAccounts;
      GrpFile: TIniFile;
      FList: TObjectList;
      function GetItem(const i: Integer): TAccGroup;
      Procedure Load;
      function _Add(const AID: Integer): TAccGroup;
      function GetMaxID: Integer;
      procedure SetMaxID(const Value: Integer);
    public
      constructor Create(AParent: TAccounts);
      destructor destroy; override;
      Function Count: Integer;
      Function Add(Const AName: String; Const ANr: Integer = 0): TAccGroup;
      Property MaxID: Integer read GetMaxID Write SetMaxID;
      Property Items[Const i: Integer]: TAccGroup read GetItem; default;
      Function Find(Const ID: Integer): TAccGroup; overload;
      Function Find(Const Name: String): TAccGroup; overload;
      Function IndexOf(Const ID: Integer): Integer; overload;
      Function IndexOf(Const Name: String): Integer; overload;
      Function IDOf(Const Name: String): Integer;
  end;

  TAccUsers = Class
    private
      FParent: TAccounts;
      ActFile: TIniFile;
      FList: TObjectList;
      LockedMailboxes: TList;
      LockedIMAPboxes: TStringList;
      AuthLogins: TStringList;
      Function _Add(Const AID: Integer): TAccUser;
      function GetItem(const i: Integer): TAccUser;
      function GetMaxID: Integer;
      procedure SetMaxID(const Value: Integer);
      Procedure Load;
    public
      constructor Create(AParent: TAccounts);
      destructor destroy; override;
      Function Count: Integer;
      Function Add(Const AUsername: String; Const AID: Integer = 0): TAccUser;
      Property Items[Const i: Integer]: TAccUser read GetItem; default;
      Property MaxID: Integer read GetMaxID Write SetMaxID;
      Function Find(Const ID: Integer): TAccUser; overload;
      Function Find(Const Name: String): TAccUser; overload;
      Function IndexOf(Const ID: Integer): Integer; overload;
      Function IndexOf(Const Name: String): Integer; overload;
      Function IDOf(Const Name: String): Integer;
  end;

  TAccGroup = class
    private
      FID: Integer;
      FMain: TAccounts;
      FGroups: TAccGroups;
      function GetMailboxtype: TMailboxtype;
      function GetBoolValue(Const Index: Integer): Boolean;
      function GetStrValue(Const Index: Integer): String;
      procedure SetMailboxtype(const Value: TMailboxtype);
      procedure SetBoolValue(const Index: Integer; const Value: Boolean);
      procedure SetStrValue(const Index: Integer; const Value: String);
      function GetUserOfGroup(idx: Integer): TAccUser;
      function GetName: String;
    public
      constructor Create(Const AID: Integer; Const AParent: TAccGroups);
      Procedure Delete;
      Function IsAdmin: Boolean;
      Property ID: Integer read FID;
      Property Name: String read GetName;
      Function UsersOfGroupCount: Integer;
      Property UsersOfGroup[idx: Integer]: TAccUser read GetUserOfGroup;
      // String-Values
      Property Description: String Index 0 read GetStrValue Write SetStrValue;
      Property DefaultNewsRead: String Index 1 read GetStrValue Write SetStrValue;
      Property DefaultNewsPost: String Index 2 read GetStrValue Write SetStrValue;
      // Boolean Values
      Property DefaultMailsend: Boolean Index 0 read GetBoolValue Write SetBoolValue;
      Property DefaultNewspeer: Boolean Index 1 read GetBoolValue Write SetBoolValue;
      Property DefaultNewNews: Boolean Index 2 read GetBoolValue Write SetBoolValue;
      Property DefaultModeCancel: Boolean Index 3 read GetBoolValue Write SetBoolValue;
      Property DefaultRemoteControl: Boolean Index 4 read GetBoolValue Write SetBoolValue;
      // Misc
      Property DefaultMailboxtype: TMailboxtype read GetMailboxtype Write SetMailboxtype;
  end;

  TAccUser = class
    private
      FID: Integer;
      FMain: TAccounts;
      FUsers: TAccUsers;
      function GetAccountGroup: TAccGroup;
      function GetMailboxtype: TMailboxtype;
      function GetOverride(Const Index: Integer): Boolean;
      function GetStrValue(Const Index: Integer): String;
      function GetTriValue(Const Index: Integer): TTriValue;
      procedure SetMailboxtype(const Value: TMailboxtype);
      procedure SetOverride(const Index: Integer; const Value: Boolean);
      procedure SetStrValue(const Index: Integer; const Value: String);
      procedure SetTriValue(const Index: Integer; const Value: TTriValue);
    public
      constructor Create(Const AID: Integer; Const AParent: TAccUsers);
      Function exists: boolean;
      // Properties
      Property ID: Integer read FID;
      Property Group: TAccGroup read GetAccountGroup;
      // String-Properties
      Property Username: String Index 0 read GetStrValue Write SetStrValue;
      Property Password: String Index 1 read GetStrValue Write SetStrValue;
      Property Fullname: String Index 2 read GetStrValue Write SetStrValue;
      Property Mailaddr: String Index 3 read GetStrValue Write SetStrValue;
      Property Groupname: String Index 4 read GetStrValue Write SetStrValue;
      Property NewsPostAccess: String Index 5 read GetStrValue Write SetStrValue;
      Property NewsReadAccess: String Index 6 read GetStrValue Write SetStrValue;
      // Trivalue-Properties
      Property Mailsend: TTriValue Index 0 read GetTriValue Write SetTriValue;
      Property Newspeer: TTriValue Index 1 read GetTriValue Write SetTriValue;
      Property NewsNews: TTriValue Index 2 read GetTriValue Write SetTriValue;
      Property ModeCancel: TTriValue Index 3 read GetTriValue Write SetTriValue;
      Property RemoteControl: TTriValue Index 4 read GetTriValue Write SetTriValue;
      // Bool-Properties
      Property OverrideNewsaccess: Boolean Index 0 read GetOverride Write SetOverride;
      Property OverrideMailboxtype: Boolean Index 1 read GetOverride Write SetOverride;
      // Misc Properties
      Property Mailboxtype: TMailboxtype read GetMailboxtype Write SetMailboxtype;
      // Permissions
      Function UseMailboxtype: TMailboxtype;
      Function MayNewsPost: String;
      Function MayNewsRead: String;
      Function MayMailsend: Boolean;
      Function MayNewspeer: Boolean;
      Function MayNewsNews: Boolean;
      Function MayModeCancel: Boolean;
      Function MayRemoteControl: Boolean;
      // Functions
      function  Path: String;
      function  HasMailbox: Boolean;
      function  MailboxLock( LockIt: Boolean ): Boolean;
      function  HasIMAPbox: Boolean;
      procedure InitIMAPMailbox;
      Function  CountMails: Integer;
      Function  IsAdmin: Boolean;
      Procedure Delete;
  end;

Function  GetTriValue(cb: TCombobox): TTriValue;
Procedure SetTriValue(cb: TCombobox; Const v: TTriValue);

const
  ACTID_INVALID = -1;
  ACTID_ADMIN   =  1;

implementation

uses Config, Windows, SysUtils, uTools, Global, cPCRE, uBlowfish, uDateTime,
     cStdForm, cLogfile, cIMAPMailboxIndex;

const
  // account-/group-properties
  ACTID_NEW     = -1;
const
  ACT_FILENAME    = 'Accounts.!!!';
  GRP_FILENAME    = 'Groups.!!!';                                          //HSR //Groups
  ACT_CODEKEY     = 'Just to make it "unreadable", not "safe"!';
  ACTPW_NOACCESS  = ' ';
  ACTPW_NOTNEEDED = '';
  ACTM_SECTION    = 'Common';
  ACTM_USERIDMAX  = 'UserIDMax';
  ACTM_GROUPIDMAX = 'GroupMax';

var
  CS_ACCOUNTS: TRTLCriticalSection;

function EncodeProperty( UserID: Integer; buf: String ): String;
var  i: Integer;
     s: String;
begin
     s := HamBlowfishEncipher( inttostr(UserID) + ACT_CODEKEY, buf );
     buf := '';
     for i:=1 to length(s) do begin
        buf := buf + inttohex( ord(s[i]), 2 );
     end;
     Result := buf;
end;

function DecodeProperty( UserID: Integer; buf: String ): String;
var  i: Integer;
     s: String;
begin
     s := '';
     for i:=1 to length(buf) div 2 do begin
        s := s + chr( strtoint( '$' + copy(buf,i*2-1,2) ) );
     end;
     if s='' then
        Result := #7#0#13#10
     else
        Result := HamBlowfishDecipher( inttostr(UserID) + ACT_CODEKEY, s );
end;

Function NormalizeAccountName (Const Account: String): String;
Var i: Integer;
begin
   SetLength(Result, Length(Account));
   For i := 1 to Length(Result) do
      If Account[i] IN AccountChars
         then Result[i] := Account[i]
         else Result[i] := '-'
end;

{ TAccGroup }

constructor TAccGroup.Create(const AID: Integer; const AParent: TAccGroups);
begin
   FID := AID;
   FGroups := AParent;
   FMain := FGroups.FParent 
end;

function TAccGroup.IsAdmin: Boolean;
begin
   If Self=NIL
      then Result := true
      else Result := ID=1
end;

procedure TAccGroup.Delete;
Var idx: Integer;
begin
   If isAdmin then begin
      raise Exception.Create('Admin-group can''t be deleted')
   end else
   If UsersOfGroupCount > 0 then begin
      raise Exception.Create('Groups can''t be deleted, because it contains '+IntToStr(UsersOfGroupCount)
        +' accounts')
   end else
   With FGroups.GrpFile do begin
      DeleteKey('0', IntToStr(ID));
      EraseSection(IntToStr(ID));
      idx := FGroups.IndexOf(ID);
      If idx >=0 then FGroups.FList.Delete(idx)
   end
end;

function TAccGroup.GetName: String;
begin
   Result := '';
   If Self=NIL then Exit;
   FMain.Lock;
   try
      Result := FGroups.GrpFile.ReadString('0', IntToStr(ID), '')
   finally
      FMain.Unlock
   end
end;

// Boolean Values

Const kGroupSettings_bool: Array [0..4] of String =
   ( 'MailSend', 'NewsPeer', 'NewsNews', 'NewsCancel', 'Remote' );
function TAccGroup.GetBoolValue(Const Index: Integer): Boolean;
begin
   If Self = NIL then begin
      Result := false;
      exit
   end;
   FMain.Lock;
   try
      Result := FGroups.GrpFile.ReadBool(IntToStr(ID), kGroupSettings_bool[Index], false)
   finally
      FMain.Unlock
   end
end;
procedure TAccGroup.SetBoolValue(const Index: Integer;
  const Value: Boolean);
begin
   If Self = NIL then Exit;
   FMain.Lock;
   try
      FGroups.GrpFile.WriteBool(IntToStr(ID), kGroupSettings_bool[Index], Value)
   finally
      FMain.Unlock
   end
end;

// String Values

Const kGroupSettings_str: Array [0..2] of String =
   ( 'Description', 'NewsRead', 'NewsPost');
function TAccGroup.GetStrValue(Const Index: Integer): String;
begin
   Case Index of
      0: Result := '';
      1, 2: Result := '.*';
   end;
   If Self = NIL then Exit;
   FMain.Lock;
   try
      Result := FGroups.GrpFile.ReadString(IntToStr(ID), kGroupSettings_str[Index], Result);
   finally
      FMain.Unlock
   end
end;
procedure TAccGroup.SetStrValue(const Index: Integer; const Value: String);
begin
   If Self = NIL then Exit;
   FMain.Lock;
   try
      FGroups.GrpFile.WriteString(IntToStr(ID), kGroupSettings_str[Index], Value)
   finally
      FMain.Unlock
   end
end;

Const kGroupSetting_Mailbox = 'Mailbox';

function TAccGroup.GetMailboxtype: TMailboxtype;
begin
   If Self = NIL then begin
      Result := mbtPOP3;
      exit
   end;
   FMain.Lock;
   try
      try
         Result := TMailboxtype(FGroups.GrpFile.ReadInteger(IntToStr(ID),
            kGroupSetting_Mailbox, Ord(mbtPOP3)))
      except
         Result := mbtPOP3
      end
   finally
      FMain.Unlock
   end
end;
procedure TAccGroup.SetMailboxtype(const Value: TMailboxtype);
begin
   If Self = NIL then Exit;
   FMain.Lock;
   try
      FGroups.GrpFile.WriteInteger(IntToStr(ID), kGroupSetting_Mailbox, Ord(Value))
   finally
      FMain.Unlock
   end
end;

function TAccGroup.UsersOfGroupCount: Integer;
Var i: Integer;
begin
   Result := 0;
   If Self=NIL then Exit;
   FMain.Lock;
   try
      For i := 0 to FMain.Users.Count-1 do begin
         If FMain.Users[i].Group = Self then Inc(Result)
      end
   finally
      FMain.Unlock
   end
end;

function TAccGroup.GetUserOfGroup(idx: Integer): TAccUser;
Var i, z: Integer;
begin
   Result := NIL;
   If Self = NIL then Exit;
   FMain.Lock;
   try
      z := idx;
      For i := 0 to FMain.Users.Count-1 do begin
         If FMain.Users[i].Group = Self then begin
            If z = 0 then begin
               Result := FMain.Users[i];
               break
            end else begin
               Dec(z)
            end
         end
      end
   finally
      FMain.Unlock
   end
end;

{ TAccGroups }

constructor TAccGroups.Create(AParent: TAccounts);
begin
   inherited Create;
   FParent := AParent;
   GrpFile := TIniFile.Create( PATH_BASE + GRP_FILENAME );
   FList := TObjectlist.Create;
   FList.OwnsObjects := true;
end;

destructor TAccGroups.destroy;
begin
   inherited;
   FList.Free;
   Grpfile.Free
end;

function TAccGroups.Count: Integer;
begin
   Result := FList.Count
end;

function TAccGroups._Add(Const AID: Integer): TAccGroup;
begin
   Result := TAccGroup.Create(AID, Self);
   FList.Add(Result)
end;

function TAccGroups.GetItem(const i: Integer): TAccGroup;
begin
   If (i >= 0) and (i <= Count-1)
      then Result := FList[i] as TAccGroup
      else Result := NIL
end;

function TAccGroups.IndexOf(const Name: String): Integer;
Var i: integer;
begin
   FParent.Lock;
   try
      Result := -1;
      try
         For i := 0 to Count-1 do begin
            if CompareText(Items[i].Name, Name)=0 then begin
               Result := i;
               break
            end
         end
      except
         on E:Exception do Log( LOGID_ERROR, 'AccGroups.IndexOf(Name).Exception: ' + E.Message );
      end
   finally
      FParent.Unlock
   end
end;

function TAccGroups.IndexOf(const ID: Integer): Integer;
Var i: integer;
begin
   FParent.Lock;
   try
      Result := -1;
      try
         For i := 0 to Count-1 do begin
            If Items[i].ID = ID then begin
               Result := i;
               break
            end
         end
      except
         on E:Exception do Log( LOGID_ERROR, 'AccGroups.IndexOf(ID).Exception: ' + E.Message );
      end
   finally
      FParent.Unlock
   end
end;

function TAccGroups.Add(Const AName: String; Const ANr: Integer = 0): TAccGroup;
Var Nr: Integer;
begin
   FParent.Lock;
   try
      If ANr = 0 then Nr := MaxID+1 else Nr := ANr;
      GrpFile.WriteString('0', IntToStr(Nr), AName);
      If Nr > MaxID then MaxID := Nr;
      Result := _Add(Nr);
   finally
      FParent.Unlock
   end
end;

function TAccGroups.Find(const ID: Integer): TAccGroup;
Var i: Integer;
begin
   i := IndexOf(ID);
   If i < 0
      then Result := NIL
      else Result := Items[i]
end;

function TAccGroups.Find(const Name: String): TAccGroup;
Var i: Integer;
begin
   i := IndexOf(Name);
   If i < 0
      then Result := NIL
      else Result := Items[i]
end;

function TAccGroups.IDOf(const Name: String): Integer;
Var i: Integer;
begin
   i := IndexOf(Name);
   If i < 0
      then Result := ACTID_INVALID
      else Result := Items[i].ID
end;

procedure TAccGroups.Load;
Var sl: TStringList; i: Integer;
begin
   FList.Clear;
   sl := TStringList.Create;
   With GrpFile do try
      ReadSection ( '0', sl );
      For i := 0 to sl.Count-1 do _Add(StrToInt(sl[i]))
   finally
      sl.free
   end
end;

function TAccGroups.GetMaxID: Integer;
begin
   Result := GrpFile.ReadInteger(ACTM_SECTION, ACTM_GROUPIDMAX, 0)
end;

procedure TAccGroups.SetMaxID(const Value: Integer);
begin
   GrpFile.WriteInteger(ACTM_SECTION, ACTM_GROUPIDMAX, Value)
end;

{ TAccUser }

function TAccUser.CountMails: Integer;
Var r: TSearchRec;
begin
   Result := 0;
   If FindFirst( Path + '*.'
       + CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ),
       faAnyFile-faDirectory, r ) = 0
   then try
      Repeat Inc(Result) until FindNext(r)<>0
   finally
      FindClose(r)
   end
end;

constructor TAccUser.Create(const AID: Integer; const AParent: TAccUsers);
begin
   FID := AID;
   FUsers := AParent;
   FMain := FUsers.FParent;
   FMain.Lock;
   try
      With FUsers.ActFile do begin
         If SectionExists(IntToStr(ID)) then begin
            If ReadString(IntToStr(ID), 'OverrideNewsAccess', '') = '' then begin
               OverrideNewsaccess := (MayNewsPost<>NewsPostAccess) or (MayNewsRead<>NewsReadAccess)
            end;
            If ReadString(IntToStr(ID), 'OverrideMailbox', '') = '' then begin
               OverrideMailboxtype := Mailboxtype <> UseMailboxtype
            end
         end
      end
   finally
      FMain.Unlock
   end
end;

procedure TAccUser.Delete;
Var Idx: Integer;
begin
   If isAdmin then begin
      raise Exception.Create('Admin can''t be deleted')
   end else
   If CountMails > 0 then begin
      raise Exception.Create('Account can''t be deleted, because it contains '+IntToStr(CountMails)
        +' mails')
   end else
   With FUsers.ActFile do begin
      OverrideMailboxtype := true;
      Mailboxtype := mbtNone;
      HasMailbox; // Remove Mailbox
      EraseSection(IntToStr(ID));
      Idx := FUsers.IndexOf(ID);
      If Idx >= 0 then FUsers.FList.Delete(Idx)
   end
end;

function TAccUser.exists: boolean;
begin
   Result := Self <> NIL
end;

function TAccUser.GetAccountGroup: TAccGroup;
begin
   Result := FMain.Groups.Find(Groupname)
end;

function TAccUser.GetMailboxtype: TMailboxtype;
begin
   try
      Result := TMailboxtype(FUsers.ActFile.ReadInteger(IntToStr(ID), 'Mailbox', 1))
   except
      Result := mbtPOP3
   end
end;
procedure TAccUser.SetMailboxtype(const Value: TMailboxtype);
begin
   FUsers.ActFile.WriteInteger(IntToStr(ID), 'Mailbox', Ord(Value))
end;

Const kUserSettings_bool: Array [0..1] of String =
   ( 'OverrideNewsAccess', 'OverrideMailbox' );
function TAccUser.GetOverride(Const Index: Integer): Boolean;
begin
   FMain.Lock;
   try
      Result := FUsers.ActFile.ReadBool(IntToStr(ID), kUserSettings_bool[Index], false);
   finally
      FMain.Unlock
   end
end;
procedure TAccUser.SetOverride(const Index: Integer; const Value: Boolean);
begin
   FMain.Lock;
   try
      FUsers.ActFile.WriteBool(IntToStr(ID), kUserSettings_bool[Index], Value);
   finally
      FMain.Unlock
   end
end;

Const kUserSettings_tri: Array [0..4] of String =
   ( 'MailSend', 'NewsPeer', 'NewsNewNews', 'NewsCancel', 'RemoteControl' );
function TAccUser.GetTriValue(Const Index: Integer): TTriValue;
begin
   FMain.Lock;
   try
      try
         Result := TTriValue(FUsers.ActFile.ReadInteger(IntToStr(ID),
            kUserSettings_tri[Index], Ord(vDefault)));
      except
         Result := vDefault
      end
   finally
      FMain.Unlock
   end
end;
procedure TAccUser.SetTriValue(const Index: Integer;
  const Value: TTriValue);
begin
   FMain.Lock;
   try
      FUsers.ActFile.WriteInteger(IntToStr(ID), kUserSettings_tri[Index], Ord(Value));
   finally
      FMain.Unlock
   end
end;

Const kUserSettings_str: Array [0..6] of String =
   ( 'Username', '!Password', 'Fullname', 'MailAddr', 'Group', 'NewsPost', 'NewsRead' );
function TAccUser.GetStrValue(Const Index: Integer): String;
begin
   FMain.Lock;
   try
      Case Index of
         4: Result := 'admin';
         5, 6: Result := '.*';
         else Result := ''
      end;
      Result := FUsers.ActFile.ReadString(IntToStr(ID), kUserSettings_str[Index], Result);
      If Index = 1 then Result := DecodeProperty(ID, Result)
   finally
      FMain.Unlock
   end
end;
procedure TAccUser.SetStrValue(const Index: Integer; const Value: String);
begin
   FMain.Lock;
   try
      With FUsers.ActFile do begin
         If Index = 1
            then WriteString(IntToStr(ID), kUserSettings_str[Index], EnCodeProperty(ID, Value))
            else WriteString(IntToStr(ID), kUserSettings_str[Index], Value);
      end
   finally
      FMain.Unlock
   end
end;

function TAccUser.HasIMAPbox: Boolean;
begin
  If ID=ACTID_ADMIN
     then Result := false
     else Result := (UseMailboxtype = mbtIMAP)
end;

function TAccUser.HasMailbox: Boolean;
begin
   FMain.Lock;
   try
      if isAdmin
         then Result := True
         else Result := UseMailboxtype = mbtPOP3;
      if Result then begin
         if not DirectoryExists( Path ) then ForceDirectories( Path )
      end else begin
         if UseMailboxtype = mbtNone then begin//IMAP_not_delete
            if DirectoryExists( Path ) then begin
               if not RemoveDir( Path ) then begin
                  Log( LOGID_WARN, TrGlF('Accounts', 'Mailbox_undeleteble',
                     'Mailbox "%s" could not be deleted!', Username ))
               end
            end
         end
      end
   finally
      FMain.Unlock
   end
end;

procedure TAccUser.InitIMAPMailbox;

     function RemoveFileIfExists( FileName: String ): Boolean;
     begin
          if FileExists2( FileName )
             then Result := DeleteFile( FileName )
             else Result := True
     end;

begin
   FMain.Lock;
   try
      if HasMailbox then begin
         ForceDirectories(Path);
         if not ( RemoveFileIfExists( Path + IMAPINDEX_FILENAME )
              and RemoveFileIfExists( Path + IMAPSTATUS_FILENAME ) )
         then begin
            Log( LOGID_WARN, Format( 'IMAP-Mailbox "%s" could not be initialized!', [Username]))
         end
      end;
   finally
      FMain.Unlock
   end
end;

function TAccUser.IsAdmin: Boolean;
begin
   Result := ID=ACTID_ADMIN
end;

function TAccUser.MailboxLock(LockIt: Boolean): Boolean;
var  i: Integer;
begin
   FMain.Lock;
   try
      Result := False;
      i := FUsers.LockedMailboxes.IndexOf( Pointer(ID) );
      if LockIt then begin
         if i<0 then begin
            FUsers.LockedMailboxes.Add( Pointer(ID) );
            Result := True;
         end;
      end else begin
         if i>=0 then begin
            FUsers.LockedMailboxes.Delete( i );
            Result := True;
         end;
      end;
   finally
      FMain.Unlock
   end
end;

function TAccUser.MayMailsend: Boolean;
begin
   If Mailsend = vDefault
      then Result := Group.DefaultMailsend
      else Result := (Mailsend = vYes)
end;

function TAccUser.MayModeCancel: Boolean;
begin
   If ModeCancel = vDefault
      then Result := Group.DefaultModeCancel
      else Result := (ModeCancel = vYes)
end;

function TAccUser.MayNewsNews: Boolean;
begin
   If NewsNews = vDefault
      then Result := Group.DefaultNewNews
      else Result := (NewsNews = vYes)
end;

function TAccUser.MayNewspeer: Boolean;
begin
   If Newspeer = vDefault
      then Result := Group.DefaultNewspeer
      else Result := (Newspeer = vYes)
end;

function TAccUser.MayNewsPost: String;
begin
   If OverrideNewsaccess
      then Result := NewsPostAccess
      else Result := Group.DefaultNewsPost
end;

function TAccUser.MayNewsRead: String;
begin
   If OverrideNewsaccess
      then Result := NewsReadAccess
      else Result := Group.DefaultNewsRead
end;

function TAccUser.MayRemoteControl: Boolean;
begin
   If RemoteControl = vDefault
      then Result := Group.DefaultRemoteControl
      else Result := (RemoteControl = vYes)
end;

function TAccUser.Path: String;
begin
   FMain.Lock;
   try
      Result := Username;
      If Result > '' then Result := PATH_MAILS + Result + '\'
   finally
      FMain.Unlock
   end
end;

function TAccUser.UseMailboxtype: TMailboxtype;
begin
   If OverrideMailboxtype
      then Result := Mailboxtype
      else Result := Group.DefaultMailboxtype
end;

{ TAccUsers }

function TAccUsers.Add(Const AUsername: String; Const AID: Integer = 0): TAccUser;
Var ID: Integer;
begin
   FParent.Lock;
   try
      If AID = 0 then ID := MaxID+1 else ID := AID;
      If ID > MaxID then MaxID := ID;
      Result := _Add(ID);
      Result.Username := AUsername
   finally
      FParent.Unlock
   end
end;

function TAccUsers.Count: Integer;
begin
   Result := FList.Count
end;

constructor TAccUsers.Create(AParent: TAccounts);
begin
   inherited Create;
   FParent := AParent;
   ActFile := TIniFile.Create( PATH_BASE + ACT_FILENAME );
   FList := TObjectlist.Create;
   FList.OwnsObjects := true;

   LockedMailboxes := TList.Create;

   LockedIMAPboxes := TStringList.Create; //IMAP
   LockedIMAPboxes.Sorted := True; //IMAP
   {$IFDEF VER140} // Delphi 6
   LockedIMAPboxes.CaseSensitive := False; //IMAP
   {$ENDIF}

   AuthLogins := TStringList.Create;
   AuthLogins.Sorted := True;
end;

destructor TAccUsers.destroy;
begin
   inherited;
   ActFile.Free;
   FList.Free;
   LockedMailboxes.Free;
   LockedIMAPBoxes.Free;
   AuthLogins.Free;
end;

function TAccUsers.Find(const ID: Integer): TAccUser;
Var i: Integer;
begin
   i := IndexOf(ID);
   If i < 0
      then Result := NIL
      else Result := Items[i]
end;

function TAccUsers.Find(const Name: String): TAccUser;
Var i: Integer;
begin
   i := IndexOf(Name);
   If i < 0
      then Result := NIL
      else Result := Items[i]
end;

function TAccUsers.GetItem(const i: Integer): TAccUser;
begin
   If (i >= 0) and (i <= Count-1)
      then Result := FList[i] as TAccUser
      else Result := NIL
end;

function TAccUsers.GetMaxID: Integer;
begin
   Result := ActFile.ReadInteger(ACTM_SECTION, ACTM_USERIDMAX, 0)
end;

procedure TAccUsers.SetMaxID(const Value: Integer);
begin
   ActFile.WriteInteger(ACTM_SECTION, ACTM_USERIDMAX, Value)
end;

function TAccUsers.IDOf(const Name: String): Integer;
Var i: Integer;
begin
   i := IndexOf(Name);
   If i < 0
      then Result := ACTID_INVALID
      else Result := Items[i].ID
end;

function TAccUsers.IndexOf(const Name: String): Integer;
Var i: integer;
begin
   FParent.Lock;
   try
      Result := -1;
      try
         For i := 0 to Count-1 do begin
            if CompareText(Items[i].UserName, Name)=0 then begin
               Result := i;
               break
            end
         end
      except
         on E:Exception do Log( LOGID_ERROR, 'AccUsers.IndexOf(Name).Exception: ' + E.Message );
      end
   finally
      FParent.Unlock
   end
end;

function TAccUsers.IndexOf(const ID: Integer): Integer;
Var i: integer;
begin
   FParent.Lock;
   try
      Result := -1;
      try
         For i := 0 to Count-1 do begin
            If Items[i].ID = ID then begin
               Result := i;
               break
            end
         end
      except
         on E:Exception do Log( LOGID_ERROR, 'AccUsers.IndexOf(ID).Exception: ' + E.Message );
      end
   finally
      FParent.Unlock
   end
end;

procedure TAccUsers.Load;
Var i: Integer;
begin
   FList.Clear;
   With ActFile do begin
      For i := 1 to MaxID do begin
         If SectionExists(IntToStr(i)) then _Add(i)
      end      
   end
end;

function TAccUsers._Add(const AID: Integer): TAccUser;
begin
   Result := TAccUser.Create(AID, Self);
   FList.Add(Result)
end;

{ TAccounts }

constructor TAccounts.Create;
Var i: Integer; f: File;
begin
   FGroups := TAccGroups.Create(Self);
   FUsers := TAccUsers.Create(Self);
   Load;
   With Groups do begin
      If Find(1)=NIL then With Add('Admin', 1) do begin
         Description := 'Administration';
         DefaultNewsRead := '.*';
         DefaultNewsPost := '.*';
         DefaultMailsend := true;
         DefaultMailboxtype := mbtPOP3;
      end
   end;
   With Users do begin
      If Find(ACTID_ADMIN)=NIL then With Add('admin', ACTID_ADMIN) do begin
         Fullname := 'Hamster Administrator';
         Groupname := 'Admin';
         OverrideNewsaccess := true;
         NewsReadAccess := Group.DefaultNewsRead;
         NewsPostAccess := Group.DefaultNewsPost;
         Mailboxtype := mbtPOP3
      end;
      // Change mailboxname if contains invalid chars
      For i := 0 to Count-1 do With Items[i] do begin
         If Username <> NormalizeAccountName(Username) then begin
            try
               AssignFile(f, PATH_MAILS + Username);
               FileMode := 2;
               Rename(f, PATH_MAILS + NormalizeAccountName(Username))
            except end;
            Username := NormalizeAccountName(Username)
         end
      end
   end;
end;

destructor TAccounts.Destroy;
begin
   inherited;
   FGroups.free;
   FUsers.free;
end;

function TAccounts.ChkAuthenticatedLogin(FromIP: Integer): Boolean;
var  i: Integer;
begin
     Lock;
     try
        Result := False;
        i := Users.AuthLogins.IndexOf( inttohex(FromIP,8) );
        if i>=0 then begin
           if LongInt(Users.AuthLogins.Objects[i])>=DateTimeToUnixTime(NowGMT) then begin
              Result := True;
           end else begin
              Users.AuthLogins.Delete( i ); // expired
           end;
        end;
     finally Unlock; end;
end;

function TAccounts.GetIMAPMailbox(MailboxPath: String): TObject;
var  i: Integer;
begin
   Result := nil;
   Lock;
   try
      i := Users.LockedIMAPboxes.IndexOf( MailboxPath );
      if i >= 0 then Result := Users.LockedIMAPboxes.Objects[i];
   finally Unlock end
end;

function TAccounts.IMAPMailboxLock(MailboxPath: String;
  LockIt: Boolean): Boolean;
var  i: Integer;
begin
   Lock;
   try
      Result := False;
      i := Users.LockedIMAPboxes.IndexOf( MailboxPath );
      if LockIt then begin
         if i<0 then begin
            Users.LockedIMAPboxes.Add( MailboxPath );
            Result := True;
         end;
      end else begin
         if i>=0 then begin
            Users.LockedIMAPboxes.Delete( i );
            Result := True;
         end;
      end;
   finally Unlock end;
end;

function TAccounts.IsLocalMailbox(MailAddr: String; out UserID: Integer;
  Out LocalType: TLocalmailtype; out bLocal: boolean): Boolean;
var  i, k           : Integer;
     MailAddrs, MailNam, MailDom, s: String;
     prs            : TParser;
     IsLocal        : Boolean;
begin
   Result    := False;
   UserID    := ACTID_INVALID;
   MailAddr  := ExtractMailAddr( MailAddr );
   LocalType := lmtNormal;
   IsLocal   := false;
   Lock;
   try
      // 1.) check list of assigned addresses if local mailboxes are enabled
      prs     := TParser.Create;
      try
         for i:=0 to Users.Count-1 do begin
            if Users[i].HasMailbox or Users[i].HasIMAPbox then begin
               MailAddrs := Users[i].Mailaddr;
               if MailAddrs>'' then begin
                  prs.Parse( MailAddrs, ' ' );
                  k := 0;
                  repeat
                     s := prs.sPart( k, '' );
                     if s > '' then begin
                        if CompareText( s, MailAddr )=0 then begin
                           Result := True;
                           UserID := Users[i].ID;
                           break;
                        end;
                        inc( k )
                     end       
                  until (s = '') or Result
               end;
            end;
         end
      finally
         prs.Free;
      end;
      // 2.) check for local domains
      if UserID=ACTID_INVALID then begin
         i := RScan ( '@', MailAddr );
         if i=0 then begin
            MailNam := MailAddr;
            MailDom := ''; // no domain -> always local
            IsLocal := True;
         end else begin
            MailNam := copy( MailAddr, 1, i-1 );
            MailDom := copy( MailAddr, i+1, Length(MailAddr)-i );
            if copy( MailDom, length(MailDom)-7, 8 )='.invalid' then begin
               IsLocal := True;
               LocalType := lmtInvalid;
            end else begin
               IsLocal := IsLocalDomain( MailDom );
            end;
         end;
         if IsLocal then begin
            Result := True;
            UserID := Users.IdOf( MailNam );
            if UserID<>ACTID_INVALID then begin
               if not (Users.Find(UserID).HasMailbox or Users.Find(UserID).HasIMAPbox)
                  then UserID:=ACTID_INVALID
            end;
            if UserID=ACTID_INVALID then begin
               UserID := ACTID_ADMIN;
               if LocalType=lmtNormal then LocalType := lmtUnknown
            end;
         end;
      end;
      bLocal := IsLocal
   finally
      Unlock
   end
end;

procedure TAccounts.Lock;
begin
   EnterCriticalSection( CS_ACCOUNTS )
end;

procedure TAccounts.LogAuthenticatedLogin(FromIP: Integer);
var  ValidTil, i: LongInt;
begin
     Lock;
     try
        // add/extend authentication period for address
        ValidTil := DateTimeToUnixTime( NowGMT ) + Def_SmtpAfterPop3Period;
        i := Users.AuthLogins.IndexOf( inttohex(FromIP,8) );
        if i<0 then begin
           Users.AuthLogins.AddObject( inttohex(FromIP,8), Pointer(ValidTil) );
        end else begin
           Users.AuthLogins.Objects[i] := Pointer( ValidTil );
        end;

        // remove expired periods
        ValidTil := DateTimeToUnixTime( NowGMT );
        for i:=Users.AuthLogins.Count-1 downto 0 do begin
           if LongInt(Users.AuthLogins.Objects[i])<ValidTil then Users.AuthLogins.Delete(i);
        end;
     finally Unlock; end;
end;

function TAccounts.LoginID(const UserName, Password: String): Integer;
var  pw: String;
begin
   Lock;
   try
      Result := Users.IDOf( UserName );
      if Result<>ACTID_INVALID then begin
         pw := Users.Find(Result).Password;
         if pw=ACTPW_NOACCESS then begin
            Log( LOGID_WARN, TrGlF(kLog, 'Login-failed.user-without-access',
               'User %s has no access but tried to login', Username));
            Result:=ACTID_INVALID;
         end else
         if pw<>ACTPW_NOTNEEDED then begin
            if pw<>Password then begin
               Log( LOGID_WARN, TrGlF(kLog, 'Login-failed.wrong-password',
                  'User %s tried to login with incorrect password', Username));
               Result:=ACTID_INVALID;
            end else begin
               Log( LOGID_DETAIL, TrGlF(kLog, 'Login-done.with-password',
                  'Successful login of User %s with correct password', intTostr(Result)));
            end
         end else begin
            Log( LOGID_DETAIL, TrGlF(kLog, 'Login-done.without-password',
               'Successful login of User %s without password', intTostr(Result)));
         end
      end
   finally Unlock; end;
end;

procedure TAccounts.SetIMAPMailbox(MailboxPath: String; Mailbox: TObject);
var  i: Integer;
begin
   Lock;
   try
      i := Users.LockedIMAPboxes.IndexOf( MailboxPath );
      if i >= 0 then Users.LockedIMAPboxes.Objects[i] := Mailbox;
   finally Unlock end
end;

procedure TAccounts.Unlock;
begin
   LeaveCriticalSection( CS_ACCOUNTS )
end;

procedure TAccounts.Load;
begin
   Groups.Load;
   Users.Load;
end;

procedure TAccounts.Reload;
begin
   try
      Lock;
      Load
   finally
      Unlock
   end
end;

{ Sonstiges }

Procedure SetTriValue(cb: TCombobox; Const v: TTriValue);
begin
   With cb, Items do begin
      Clear;
      AddObject ( TrGl(kGlobal, 'Choose.Default', '<Default>'), Pointer(Ord(vDefault)) );
      AddObject ( TrGl(kGlobal, 'Choose.Yes', 'Yes'), Pointer(Ord(vYes)) );
      AddObject ( TrGl(kGlobal, 'Choose.No', 'No'), Pointer(Ord(vNo)) );
      ItemIndex := IndexOfObject(Pointer(Ord(v)))
   end
end;
Function  GetTriValue(cb: TCombobox): TTriValue;
begin
   With cb, Items do begin
      If ItemIndex >= 0 then Result := TTriValue(Objects[ItemIndex])
                        else Result := vDefault
   end
end;

initialization
  InitializeCriticalSection( CS_ACCOUNTS );
finalization
  DeleteCriticalSection( CS_ACCOUNTS );
end.
