// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cArtFiles; // Storage for articles and mails

// ----------------------------------------------------------------------------
// Contains the class and various functions to access Hamster's message-base.
// ----------------------------------------------------------------------------

interface

uses Windows, SysUtils, Classes, FileCtrl, Global, cArticle, cArticleFile, 
     Shellapi, IniFiles, cActions;

Procedure CreateGroup( GroupName: String );

type
   TXrefArtNo = class
      GrpNam: String;
      GrpHdl: LongInt;
      ArtNo : Integer;
   end;

   TOpenArtFile = class
      ArtFile  : TArticleFile;
      UseCount : LongInt;
      GrpName  : String;
   end;

   TArticleBase = class
      private
         OpenArtFiles : TThreadList;
         function  EnterFile( GrpHdl: LongInt; var OAF: TOpenArtFile ): Boolean;
         function  GetCount( GrpHdl: LongInt ): LongInt;
         function  GetName( GrpHdl: LongInt ): String;
         function  GetLocalMin( GrpHdl: LongInt ): LongInt;
         function  GetLocalMax( GrpHdl: LongInt ): LongInt;
         function  GetServerMin( GrpHdl: LongInt; Server: String ): Integer;
         procedure SetServerMin( GrpHdl: LongInt; Server: String; NewArtMin: Integer );
         function  GetServerMax( GrpHdl: LongInt; Server: String ): Integer;
         procedure SetServerMax( GrpHdl: LongInt; Server: String; NewArtMax: Integer );
         function  GetServerLow( GrpHdl: LongInt; Server: String ): Integer;            // JAWO 12.01.2001 (lowest server artno)
         procedure SetServerLow( GrpHdl: LongInt; Server: String; NewArtLow: Integer ); // JAWO 12.01.2001 (lowest server artno)
{JW} {Feed}
         function  GetFeederLast( GrpHdl: LongInt; Server: String ): Integer;
         procedure SetFeederLast( GrpHdl: LongInt; Server: String; NewArtMin: Integer );
{JW}
         function  GetPullLimit( GrpHdl: LongInt; Server: String ): Integer;
         function  GetLastClientRead( GrpHdl: LongInt ): TDateTime;
         procedure SetLastClientRead( GrpHdl: LongInt; NewClientRead: TDateTime );
         function  GetDescription( GrpHdl: LongInt ): String;
         procedure SetDescription( GrpHdl: LongInt; NewDescription: String );
{JW} {N2M}
         function  GetModerator( GrpHdl: LongInt ): String;
{JW}
      public
         property  Count[ GrpHdl: LongInt ]: Integer read GetCount;
         property  Name [ GrpHdl: LongInt ]: String read GetName;
         property  LocalMin[ GrpHdl: LongInt ]: Integer read GetLocalMin;
         property  LocalMax[ GrpHdl: LongInt ]: Integer read GetLocalMax;
         property  ServerMin[ GrpHdl: LongInt; Server:String ]: Integer read GetServerMin write SetServerMin;
         property  ServerMax[ GrpHdl: LongInt; Server:String ]: Integer read GetServerMax write SetServerMax;
         property  ServerLow[ GrpHdl: LongInt; Server:String ]: Integer read GetServerLow write SetServerLow;  // JAWO 12.01.2001 (lowest server artno)
         property  PullLimit[ GrpHdl: LongInt; Server:String ]: Integer read GetPullLimit;
         procedure SetFirstPullDone( GrpHdl: LongInt; Server: String );
         property  LastClientRead[ GrpHdl: LongInt ]: TDateTime read GetLastClientRead write SetLastClientRead;
         property  Description[ GrpHdl: LongInt ]: String read GetDescription write SetDescription;
         property  Moderator[GrpHdl: LongInt ]: String read GetModerator; {JW} {N2M}
         property  FeederLast[ GrpHdl: LongInt; Server:String ]: Integer read GetFeederLast write SetFeederLast; {JW} {Feed}

         function  GetProp( GrpHdl: LongInt; PropName: String; DefaultValue: String ): String;
         procedure SetProp( GrpHdl: LongInt; PropName: String; NewValue: String );

         function  DTCreated( GrpHdl: LongInt ): TDateTime;
         function  GetKeyIndex( GrpHdl: LongInt; Key: LongInt ): Integer;

         function  ReadArticle ( GrpHdl: LongInt; ArtNo: Integer ): String;
         function  ReadArticleSized( GrpHdl: LongInt; ArtNo: Integer; var MaxSize: Integer ): String;
         function  ReserveArtNo( GrpHdl: LongInt ): LongInt;
         function  WriteArticle( GrpHdl: LongInt; ArtNo: Integer; const ArtText: String ): LongInt;
         Function DeleteArticle( Const GrpHdl: LongInt; Const ArtNo: Integer ): boolean;

         function  ReadArticleByMID( ReadMID: String ): String;

         function  DeleteArticleByMID( Const DelMID: String; Const DeleteHistEntries: boolean ): Boolean;
         function  DeleteArticleByMIDAndGroup ( Const DelMID, Groupname: String; Const DeleteHistEntries: boolean ): Boolean;
         Function DeleteArticleByNrAndGroup ( Const ArtNo: Integer; Const Groupname: String): boolean;
         function  UseCount( GroupName: String ): LongInt;

         function  Open( GroupName: String ): LongInt;
         procedure Close( GrpHdl: LongInt );
         procedure Purge( GrpHdl: LongInt );
         procedure PurgeReset( GrpHdl: LongInt ); // JAWO 12.08.01 (Reset Group)
         function  DeleteGroup( GroupName: String ): Integer;

         procedure EstimateValues( GroupName: String;
                                   var ArtCount: LongInt;
                                   var NewGroup: Boolean );
         constructor Create;
         destructor Destroy; override;
   end;

var
  ArticleBase : TArticleBase;

function SaveArticle(   const Article           : TArticle;
                        const CurrentGroupName  : String;
                        const CurrentGroupHandle: LongInt;
                        const ScoreMarker       : String;
                        const IgnoreHistory     : Boolean;
                        const OverrideGroups    : String ): Boolean;

function ImportArticle( const ArtText       : String;
                        const OverrideGroups: String;
                        const IgnoreHistory : Boolean;
                        const MarkNoArchive : Boolean ): Boolean;

procedure SaveInInternalGroup( InternalIndex: Integer;
                               Subject, ArtText: String );

function SaveUniqueNewsMsg( DestPath: String; MsgText: String;
                            GenerateMid: Integer ): Boolean;

function SaveMailToNews( Newsgroup: String; const MsgText: String ): Boolean;

Type TMessageChange = ( mcOriginal, mcChanged, mcDeleted );

Function ModifyMessage ( Const Typ: TActiontype; Const MsgText: String;
            Out NewText: String ): TMessageChange;

                                    // JAWO 20.12.2000 (local delivered mail)
Type TMailSource = (msIncoming, msOutgoing, msInternal, msLocal);

function SaveUniqueMailMsg( DestPath: String; MsgText: String; GenerateMid: Integer;
   Const Source: TMailsource;
   Const SaveToImap: Boolean; IMAPFolder: String; Const IMAPFlags: String;
   Var FileName, Account: String ): Boolean; //IMAP (Chg)


implementation

uses uTools, Config, cAccount, cPCRE, uCRC32, uDateTime, cStdForm, cMailRouter,
     cLogfile, cIMAPMailbox;

{---------------------------------------------------------- TArticleBase -----}

function TArticleBase.EnterFile( GrpHdl: LongInt; var OAF: TOpenArtFile ): Boolean;
begin
   Result := False;

   with OpenArtFiles.LockList do begin
      if (GrpHdl>=0) and (GrpHdl<Count) then begin
         OAF := TOpenArtFile( Items[GrpHdl] );
      end else begin
         OAF := nil;
      end;
      OpenArtFiles.UnlockList;
   end;

   if Assigned(OAF) then begin
      if Assigned(OAF.ArtFile) then begin
         OAF.ArtFile.EnterThisFile;
         Result := True;
      end;
   end;
end;

function TArticleBase.GetCount( GrpHdl: LongInt ): LongInt;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.Count;
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

function TArticleBase.GetName( GrpHdl: LongInt ): String;
begin
     with OpenArtFiles.LockList do begin
        if (GrpHdl>=0) and (GrpHdl<Count) then begin
           with TOpenArtFile( Items[GrpHdl] ) do Result := GrpName;
        end else begin
           Result := '';
        end;
        OpenArtFiles.UnlockList;
     end;
end;

function TArticleBase.GetLocalMin( GrpHdl: LongInt ): LongInt;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.LocalMin;
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

function TArticleBase.GetLocalMax( GrpHdl: LongInt ): LongInt;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.LocalMax;
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

function TArticleBase.ReadArticle( GrpHdl: LongInt; ArtNo: Integer ): String;
var  l: LongInt;
     OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      l := 0;
      Result := OAF.ArtFile.ReadArticle( ArtNo, l );
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := '';
   end;
end;

function TArticleBase.ReadArticleSized( GrpHdl: LongInt; ArtNo: Integer; var MaxSize: Integer ): String;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.ReadArticle( ArtNo, MaxSize );
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := ''
   end;
end;

function TArticleBase.ReserveArtNo( GrpHdl: LongInt ): LongInt;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.ReserveArtNo;
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

function TArticleBase.WriteArticle( GrpHdl: LongInt; ArtNo: Integer; const ArtText: String ): LongInt;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.WriteArticle( ArtNo, ArtText );
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

Function TArticleBase.DeleteArticle( Const GrpHdl: LongInt; Const ArtNo: Integer ): boolean;
var  OAF: TOpenArtFile;
begin
   Result := false;
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.DeleteArticle( ArtNo );
      OAF.ArtFile.LeaveThisFile;
   end;
end;

procedure TArticleBase.SetFirstPullDone( GrpHdl: LongInt; Server: String );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.SetFirstPullDone( Server );
      OAF.ArtFile.LeaveThisFile;
   end;
end;

function TArticleBase.GetKeyIndex( GrpHdl: LongInt; Key: LongInt ): Integer;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.IndexOfKey( Key );
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

function TArticleBase.DTCreated( GrpHdl: LongInt ): TDateTime;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.DTCreated;
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

function TArticleBase.GetServerMin( GrpHdl: LongInt; Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.ServerMin[Server];
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

procedure TArticleBase.SetServerMin( GrpHdl: LongInt; Server: String; NewArtMin: Integer );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.ServerMin[Server] := NewArtMin;
      OAF.ArtFile.LeaveThisFile;
   end;
end;

function TArticleBase.GetServerMax( GrpHdl: LongInt; Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.ServerMax[Server];
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

procedure TArticleBase.SetServerMax( GrpHdl: LongInt; Server: String; NewArtMax: Integer );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.ServerMax[Server]:=NewArtMax;
      OAF.ArtFile.LeaveThisFile;
   end;
end;

{JAWO} {(lowest server artno)}
function TArticleBase.GetServerLow( GrpHdl: LongInt; Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.ServerLow[Server];
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

procedure TArticleBase.SetServerLow( GrpHdl: LongInt; Server: String; NewArtLow: Integer );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.ServerLow[Server] := NewArtLow;
      OAF.ArtFile.LeaveThisFile;
   end;
end;
{/JAWO}

{JW} {Feed}

function TArticleBase.GetFeederLast( GrpHdl: LongInt; Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.FeederLast[Server];
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

procedure TArticleBase.SetFeederLast( GrpHdl: LongInt; Server: String; NewArtMin: Integer );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.FeederLast[Server] := NewArtMin;
      OAF.ArtFile.LeaveThisFile;
   end;
end;


{JW}

function TArticleBase.GetLastClientRead( GrpHdl: LongInt ): TDateTime;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.LastClientRead;
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

procedure TArticleBase.SetLastClientRead( GrpHdl: LongInt; NewClientRead: TDateTime );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.LastClientRead := NewClientRead;
      OAF.ArtFile.LeaveThisFile;
   end;
end;

function TArticleBase.GetProp( GrpHdl: LongInt; PropName: String; DefaultValue: String ): String;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.GetProp( PropName, DefaultValue );
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := '';
   end;
end;

procedure TArticleBase.SetProp( GrpHdl: LongInt; PropName: String; NewValue: String );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.SetProp( PropName, NewValue );
      OAF.ArtFile.LeaveThisFile;
   end;
end;

function TArticleBase.GetDescription( GrpHdl: LongInt ): String;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.Description;
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := '';
   end;
end;

procedure TArticleBase.SetDescription( GrpHdl: LongInt; NewDescription: String );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.Description := NewDescription;
      OAF.ArtFile.LeaveThisFile;
   end;
end;

{JW} {N2M}
function  TArticleBase.GetModerator( GrpHdl: LongInt ): String;
var  OAF: TOpenArtFile;
begin
    if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.Moderator;
      OAF.ArtFile.LeaveThisFile;
    end else begin
      Result := '';
   end;
end;
{JW}

function TArticleBase.GetPullLimit( GrpHdl: LongInt; Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      Result := OAF.ArtFile.PullLimit[Server];
      OAF.ArtFile.LeaveThisFile;
   end else begin
      Result := -1;
   end;
end;

procedure TArticleBase.Purge( GrpHdl: LongInt );
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.Purge;
      OAF.ArtFile.LeaveThisFile;
   end;
end;

function TArticleBase.UseCount( GroupName: String ): LongInt;
var  i: Integer;
begin
     Result := 0;

     with OpenArtFiles.LockList do begin
        // Gruppe schon in Liste der geffneten Gruppen?
        for i:=0 to Count-1 do begin
           if TOpenArtFile( Items[i] ).GrpName=GroupName then begin
              Result := TOpenArtFile( Items[i] ).UseCount;
              break;
           end;
        end;
        OpenArtFiles.UnlockList;
     end;
end;

function TArticleBase.Open( GroupName: String ): LongInt;
var  GrpHdl  : LongInt;
     ArtOpen : TOpenArtFile;
     i       : Integer;
     fmMode  : Word;
begin
   try
      with OpenArtFiles.LockList do begin
         GrpHdl  := -1;
         ArtOpen := nil;

         // is group already in list of open groups?
         for i:=0 to Count-1 do begin
            if TOpenArtFile( Items[i] ).GrpName=GroupName then begin
               GrpHdl  := i; // yes
               ArtOpen := Items[i];
               break;
            end;
         end;

         // add to list of open article-files
         if GrpHdl=-1 then begin
            ArtOpen := TOpenArtFile.Create;
            ArtOpen.ArtFile  := nil;
            ArtOpen.UseCount := 0;
            ArtOpen.GrpName  := GroupName;
            GrpHdl := Add( ArtOpen );
         end;

         // open group or increment usage-counter
         if ArtOpen.ArtFile=nil then begin
            ArtOpen.ArtFile := TArticleFile.Create( GroupName );
            If ArchivMode
               then fmMode := ARTFILEREAD
               else fmMode := ARTFILEWrite; // ARTFILEEXCLUSIVE;
            if ArtOpen.ArtFile.Open( fmMode ) then begin
               inc( ArtOpen.UseCount );
               Result := GrpHdl;
            end else begin
               ArtOpen.ArtFile.Free;
               ArtOpen.ArtFile := nil;
               Result := -1;
            end;
         end else begin
            inc( ArtOpen.UseCount );
            Result := GrpHdl;
         end;

         Log( LOGID_DEBUG, 'ArtBase.Open(' + GroupName + ') -> ' + inttostr(Result) );

      end;
   finally
      OpenArtFiles.UnlockList;
   end;
end;

procedure TArticleBase.Close( GrpHdl: LongInt );
begin
   with OpenArtFiles.LockList do begin
      if (GrpHdl>=0) and (GrpHdl<Count) then begin
         with TOpenArtFile( Items[GrpHdl] ) do begin

            // decrement usage-counter
            if UseCount>0 then dec( UseCount );

            if Assigned( ArtFile ) then begin
               if UseCount=0 then begin
                  // close article file if not in use any more
                  Log( LOGID_DEBUG, 'ArtBase.Close(' + inttostr(GrpHdl) + ')' );
                  try
                     ArtFile.Free;
                  except
                     Log( LOGID_WARN, 'ArtBase.Close(' + inttostr(GrpHdl) + ') failed!' );
                  end;
                  Artfile := nil
               end else begin
                  // flush changes (if any)
                  ArtFile.Flush
               end
            end
         end
      end;
      OpenArtFiles.UnlockList;
   end;
end;

function TArticleBase.ReadArticleByMID( ReadMID: String ): String;
var  GrpNam       : String;
     GrpHdl, ArtNo: Integer;
begin
     Result := '';
     if not NewsHistory.LocateMID( ReadMID, GrpNam, ArtNo ) then exit;

     GrpHdl := Open( GrpNam );
     if GrpHdl<0 then exit;
     Result := ReadArticle( GrpHdl, ArtNo );
     Close( GrpHdl );
end;

function TArticleBase.DeleteArticleByMID( Const DelMID: String; Const DeleteHistEntries: boolean ): Boolean;
begin
   Result := DeleteArticleByMIDAndGroup ( DelMid, '', DeleteHistEntries )
end;

function TArticleBase.DeleteGroup( GroupName: String ): Integer;
begin
     if UseCount( GroupName )>0 then begin
        Result := -1;
        Log( LOGID_WARN, TrGlF(kLog, 'Warning.DelGroup.InUse',
              'Group "%s" is in use and can''t be deleted!', [GroupName] ) );
        exit;
     end;

     try
        // delete "known" files
        DeleteFile( PATH_GROUPS + GroupName + '\data' + EXT_DAT );
        DeleteFile( PATH_GROUPS + GroupName + '\data' + EXT_IDX );
        DeleteFile( PATH_GROUPS + GroupName + '\data' + EXT_CFG );
        DeleteFile( PATH_GROUPS + GroupName + '\data.bak');

        // remove directory
        if RemoveDir( PATH_GROUPS + GroupName ) then begin
           Result := 0;
        end else begin
           Result := -2;
           Log( LOGID_WARN, TrGlF(kLog, 'Warning.DelGroup.DirUndeleteble',
                 'Couldn''t remove directory "%s"!', [PATH_GROUPS + GroupName] ) );
        end;
     except
        on E: Exception do begin
           Result := -9;
           Log( LOGID_WARN, TrGlF(kLog, 'Warning.DelGroup.Exception',
                 'Exception deleting group %s: %s', [GroupName, E.Message] ) );
        end;
     end;
end;

constructor TArticleBase.Create;
begin
     inherited Create;
     OpenArtFiles := TThreadList.Create;
end;

destructor TArticleBase.Destroy;
var  i: Integer;
begin
   // alle noch geffneten Dateien schlieen
   with OpenArtFiles.LockList do begin
      for i:=Count-1 downto 0 do begin
         with TOpenArtFile( Items[i] ) do begin
            while UseCount>0 do Close(i);
            TOpenArtFile( Items[i] ).Free;
         end;
      end;
      OpenArtFiles.UnlockList;
   end;
   OpenArtFiles.Free;
   inherited Destroy;
end;

{-----------------------------------------------------------------------------}

function SaveArticle( const Article           : TArticle;
                      const CurrentGroupName  : String;
                      const CurrentGroupHandle: LongInt;
                      const ScoreMarker       : String;
                      const IgnoreHistory     : Boolean;
                      const OverrideGroups    : String
                    ): Boolean;
var  MessageID, Newsgroups, LfdGroup, DestXref, s: String;
     GrpHdl, LfdGrp, i: Integer;
     HistoryCheck     : Boolean;
     DestGroups: TStringList;
     Xrefs     : TList;
     Xref      : TXrefArtNo;
begin
   Result := False;

   // get/create Message-ID
   MessageID := Article['Message-ID:'];
   if MessageID='' then begin
      MessageID := MidGenerator( Def_FQDNforMIDs );
      Article.AddHeaderFirst('Message-ID:', MessageID );
   end;

   // path
   s := Article['Path:'];
   If s = '' then Article.AddHeaderFirst( 'Path:', 'not-for-mail' );

   // precede article with Hamster's info-header
   Article.DeleteHeader(OUR_X_HEADER);
   s := 'Score='    + ScoreMarker + ' '
      + 'Received=' + DateTimeToTimeStamp(Now); // =timebase for purge
   Article.AddHeaderFirst( OUR_X_HEADER, s );

   // remove existing Xref-header
   Article.DeleteHeader('X-Old-Xref:');
   Article.RenameHeader ('Xref:', 'X-Old-Xref');

   // create Xref-header by opening groups and reserving article-numbers
   DestXref := '';
   Xrefs := TList.Create;

   if OverrideGroups='' then Newsgroups := Article['Newsgroups:']
                        else Newsgroups := OverrideGroups;

   // prepare list of groupnames
   DestGroups := TStringList.Create;
   DestGroups.Sorted := True;
   DestGroups.Duplicates := dupIgnore;

   try
      if CurrentGroupName<>'' then begin
         if CfgHamster.ActiveIndexOf[CurrentGroupName]>=0 then begin
            // add current group, even if not in Newsgroups (e.g. control.*)
            DestGroups.Add( CurrentGroupName );
         end;
      end;
      while Newsgroups<>'' do begin
         i := Pos( ',', Newsgroups );
         if i>0 then begin
            LfdGroup := copy( Newsgroups, 1, i-1 );
            System.Delete( Newsgroups, 1, i );
         end else begin
            LfdGroup   := Newsgroups;
            Newsgroups := '';
         end;
         LfdGroup := TrimWhSpace(LfdGroup);
         if CfgHamster.ActiveIndexOf[LfdGroup]>=0 then begin
            if LfdGroup<>'' then DestGroups.Add( LfdGroup );
         end;
      end;
      if DestGroups.Count=0 then begin // known groups?
         DestGroups.Add( INTERNALGROUP[ INTERNALGROUP_UNKNOWNGROUP ] );
      end;

//      EnterCriticalSection( CS_SAVE_ARTICLE ); //{JW} {Zombie} //HSR //Save_art_Crit_del
      try  //{JW} {Zombie}
        for i:=0 to DestGroups.Count-1 do begin
           LfdGroup := DestGroups[i];
           if LfdGroup=CurrentGroupName then GrpHdl:=CurrentGroupHandle
                                        else GrpHdl:=ArticleBase.Open(LfdGroup);
           if GrpHdl>=0 then begin
              Xref := TXrefArtNo.Create;
              Xref.GrpNam := LfdGroup;
              Xref.GrpHdl := GrpHdl;
              Xref.ArtNo  := ArticleBase.ReserveArtNo( Xref.GrpHdl );
              Xrefs.Add( Xref );
              DestXref := DestXref + ' ' + LfdGroup + ':' + inttostr( Xref.ArtNo );
           end;
        end;
        // Note: already existing Xref would have been removed above
        Article.AddHeaderAtPos('Xref:', XREF_HOSTNAME + DestXref, 1);

        // save article in local groups
        HistoryCheck := not IgnoreHistory;
//        if HistoryCheck then EnterCriticalSection( CS_SAVE_ARTICLE ); //{JW} {Zombie}
//      try                                                             //{JW} {Zombie}
         for LfdGrp:=0 to Xrefs.Count-1 do begin
            Xref := TXrefArtNo( Xrefs[LfdGrp] );

            try
               if HistoryCheck then begin
                  if not NewsHistory.AddEntryFirst(
                            MessageID, StrToCRC32(lowercase(Xref.GrpNam)),
                            Xref.ArtNo, 0
                         ) then begin
                     break;
                  end;
               end;

               if ArticleBase.WriteArticle( Xref.GrpHdl,
                                            Xref.ArtNo,
                                            Article.Text )<>0
               then begin
                  Result := True;

                  if HistoryCheck then begin
                     HistoryCheck := False;
//                     LeaveCriticalSection( CS_SAVE_ARTICLE );     //{JW} {Zombie}
                  end else begin
                     NewsHistory.AddEntryDupes( MessageID,
                                                StrToCRC32(lowercase(Xref.GrpNam)),
                                                Xref.ArtNo, 0 );
                  end;
               end else begin
                  break;
               end;
            except
               on E: Exception do begin
                  Log( LOGID_ERROR, 'SaveArticle.Exception: ' + E.Message );
                  break;
               end;
            end;
         end;
      finally
//         if HistoryCheck then LeaveCriticalSection( CS_SAVE_ARTICLE ); // {JW} {Zombie}
//         LeaveCriticalSection( CS_SAVE_ARTICLE ); // {JW} {Zombie} //HSR //Save_art_Crit_del
      end;

      // close groups and free Xref-entries and -list
      for LfdGrp:=Xrefs.Count-1 downto 0 do begin
         Xref := TXrefArtNo( Xrefs[LfdGrp] );
         if Xref.GrpNam<>CurrentGroupName then ArticleBase.Close( Xref.GrpHdl );
         Xrefs.Delete( LfdGrp );
         Xref.Free;
      end;
      Xrefs.Free;

   finally
      DestGroups.Free;
   end;

end;

function ImportArticle( const ArtText       : String;
                        const OverrideGroups: String;
                        const IgnoreHistory : Boolean;
                        const MarkNoArchive : Boolean ): Boolean;
var  Art       : TArticle;
     s         : String;
     ArtHdr    : String; //PW //header-control
     j         : integer;//PW //header-control

{PW} {header-control}

    function HdrMatch( re: String ): Boolean;
    var  regex: TPCRE;
    begin
         regex := TPCRE.Create( False, PCRE_CASELESS );
         regex.OptCompile := PCRE_CASELESS or PCRE_MULTILINE;
         Result := regex.Match( PChar(re), PChar(ArtHdr) );
         if not Result then Log(LOGID_WARN, TrGlF(kLog, 'Warning.Missing', 'WARNING: Missing %s', re ) );
         regex.Free
    end;

{/PW}

begin
   Result := False;
{PW} {header-control}
   j := Pos( #13#10#13#10, ArtText );      // header-separator
   if j=0 then exit;                       // not a valid article
   if copy(ArtText,1,5)='From ' then begin // mbox-format not supportedhere
      Log(LOGID_WARN, TrGl(kLog, 'Warning.import.mbox', 'WARNING: Importing Mbox aborted, invalid format.' ) );
      exit;
   end;

   ArtHdr := copy( ArtText, 1, j+1 );
   if not HdrMatch( '^Subject:\s.+' )       then exit;
   if not HdrMatch( '^From:\s.+' )          then exit;

    if OverrideGroups = ''                   then
       if not HdrMatch( '^Newsgroups:\s.+' ) then exit;
    if not HdrMatch( '^Date:\s.+' )           then exit;

   Art := TArticle.Create;
   try
      Art.Text := ArtText;
      if (Art.HeaderCount < 1) or (Art.FullBody = '') then exit;

      s := '(none-import)';
      if MarkNoArchive then s := s + ' NoArchive=1';

      if SaveArticle(
            Art, '', -1, s, IgnoreHistory, OverrideGroups
         ) then Result:=True;

   finally
      Art.Free;
   end;
end;


procedure SaveInInternalGroup( InternalIndex: Integer;
                               Subject, ArtText: String );
var  Header, MessageID: String;
     Art              : TArticle;
     Lines            : Integer;
begin
   MessageID := MidGenerator( Def_FQDNforMIDs );

   Lines    := CountLines(ArtText);

   Header := 'Subject: '    + Subject                      + #13#10
           + 'Date: '       + DateTimeGMTToRfcDateTime(
                              NowGMT, NowRfcTimezone )     + #13#10
           + 'From: '       + 'Hamster'                    + #13#10
           + 'Newsgroups: ' + INTERNALGROUP[InternalIndex] + #13#10
           + 'Message-ID: ' + MessageID                    + #13#10
           + 'Path: '       + 'not-for-mail'               + #13#10
           + 'Lines: '      + inttostr(Lines)              + #13#10;
   Art := TArticle.Create (Header, ArtText);
   try
      SaveArticle( Art, '', -1, '(none-local)', True, '' );
   finally
      Art.Free;
   end;
end;

{-----------------------------------------------------------------------------}

function SaveUniqueNewsMsg( DestPath: String; MsgText: String; GenerateMid: Integer ): Boolean;
Var Msg: TArticle; sMID, FN: String;
begin
   Result := False;
   DestPath:=IncludeTrailingBackslash(DestPath);
   if (GenerateMid=GENERATEMID_IFNOTSET) and (Pos('.',Def_FQDNforMIDs)>1) then begin
      Msg := TArticle.Create;
      try
         Msg.Text := MsgText;
         sMID := Msg['Message-ID:'];
         if sMID='' then begin
            sMID := MidGenerator( Def_FQDNforMIDs );
            Msg.AddHeaderFirst( 'Message-ID:', sMID );
         end;
         MsgText := Msg.Text
      finally
         Msg.Free
      end
   end;
   EnterCriticalSection( CS_LOCK_NEWSOUT );
   try
      try
         FN := CfgHamster.GetUniqueMsgFilename( DestPath, 'news' );
         With TFileStream.Create(FN, fmCreate) do try
            If MsgText > '' then Write(MsgText[1], Length(MsgText));
            Result := true
         finally
            free
         end;
         EnterCriticalSection(CS_Event);
         PulseEvent(EventNewsOut);
         LeaveCriticalSection(CS_Event);
         Actions.Exec ( atNewsOut, FN, '' )
      except
         on E: Exception do begin
            Log( LOGID_WARN, TrGlF(kLog, 'Warning.SaveNewsfile.Exception',
                  'Couldn''t save news file: %s', [E.Message] ) );
         end;
      end;
   finally
      LeaveCriticalSection( CS_LOCK_NEWSOUT );
   end;
end;

{-----------------------------------------------------------------------------}

function SaveMailToNews( Newsgroup: String; const MsgText: String ): Boolean;
var  Art: TArticle;
     s  : String;
begin
   Result := False;
   Art := TArticle.Create;
   try
      If Actions.Exists ( atNewsPreprocess ) then begin
         Case ModifyMessage ( atNewsPreprocess, MsgText, s ) of
            mcOriginal, mcChanged: Art.Text := s;
            mcDeleted: begin
               Result := true; Exit
            end;
         end
      end else begin
         Art.Text := MsgText
      end;

      Art.DeleteHeader('X-Newsgroups:');
      Art.RenameHeader('Newsgroups:', 'X-Newsgroups:');
      Art.AddHeaderFirst( 'Newsgroups:', Newsgroup );

      s := Art['References:'];
      if s='' then begin
         s := Art['In-Reply-To:'];
         if s > '' then begin
            s := RE_Extract( s, '\<.+?\@.+?\>', 0 );
            if s > '' then Art.AddHeaderFirst( 'References:', s );
         end;
      end;
{JW} {N2M}
      if Def_flup then begin
         s := Art['Followup-To:'];
         if s='' then Art.AddHeaderFirst('Followup-To:', 'poster' );
      end;
{JW}
{JW} {Path Gateway}
      if Def_FQDN<>''
         then Art.AddHeaderFirst( 'Path:', Def_FQDN+'!not-for-mail')
         else Art.AddHeaderFirst( 'Path:', 'not-for-mail' );
      { Date Gateway }
      s := Art['Date:'];
      if s='' then Art.AddHeaderFirst( 'Date:', DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone ));

      { Chip Verde postto Korrnews Processing }
      If Actions.Exists ( atNewsGateway ) then begin
         Case ModifyMessage ( atNewsGateway, Art.Text, s ) of
            mcChanged: Art.Text := s;
            mcDeleted: Exit;
         end
      end;
      if SaveArticle( Art, '', -1,
                      '(none-mail2news)', True, '' ) then Result:=True;

   finally
      Art.Free;
   end;
end;

{-----------------------------------------------------------------------------}

function SaveUniqueMailMsg( DestPath: String; MsgText: String; GenerateMid: Integer;
   Const Source: TMailsource;
   Const SaveToImap: Boolean; IMAPFolder: String; Const IMAPFlags: String;
   Var FileName, Account: String ): Boolean; //IMAP (Chg)
var  Msg  : TArticle;
     sMID, s: String;
     ImapMailbox: TImapMailbox; //IMAP
     ImapMbxNotInUse: Boolean; //IMAP
begin
     Result := False;
     DestPath:=IncludeTrailingBackslash(DestPath);
     Case Source of
        msIncoming: s := TrGl(kLog, 'Mailrouter.SaveMail.Typ0', 'Incoming');
        msOutgoing: s := TrGl(kLog, 'Mailrouter.SaveMail.Typ1', 'Outgoing');
        msInternal: s := TrGl(kLog, 'Mailrouter.SaveMail.Typ2', 'Internal');
        msLocal   : s := TrGl(kLog, 'Mailrouter.SaveMail.Typ3', 'Local');
        else s := 'Unknown'
     end;
     Log(LOGID_Detail,
         TrGlF( kLog, 'Mailrouter.SaveMail', 'Save mail to path "%0:s", Type of mail: %1:s', [DestPath, s]));
     if GenerateMid=GENERATEMID_IFNOTSET then begin
        Msg := TArticle.Create;
        try
           Msg.Text := MsgText;
           sMID := Msg['Message-ID:'];
           if sMID = '' then begin
              sMID := MidGenerator( Def_FQDNforMIDs );
              Msg.AddHeaderAfterSpecialHeaders('Message-ID:', sMid);
              MsgText := Msg.Text
           end
        finally
           Msg.Free
        end
     end;

     if DestPath=PATH_MAIL_OUT then EnterCriticalSection( CS_LOCK_MAILOUT_ADD )
                               else EnterCriticalSection( CS_LOCK_MAILBOX_ALL );
     try
        {IMAP}
        ImapMailbox := nil;
        If (DestPath<>PATH_MAIL_OUT) and SaveToImap then begin
           // IMAP-Postfach
           {HSR} {IMAP-Folder 03}
           while (length(IMAPFolder)>0) and (IMAPFolder[1] in ['/', '\', ' ']) do
             System.Delete(IMAPFolder, 1, 1);
           while (length(IMAPFolder)>0) and (IMAPFolder[length(IMAPFolder)] in ['/', '\', ' ']) do
             System.Delete(IMAPFolder, length(IMAPFolder),1);
           IMAPFolder := StringReplace(IMAPFolder, '/', '\', [rfReplaceAll, rfIgnoreCase]);
           If LowerCase(Copy(IMAPFolder, 1, 6)) = 'inbox\' then Delete(IMAPFolder, 1, 6);
           If LowerCase(IMAPFolder) = 'inbox' then IMAPFolder := '';
           if IMAPFolder>'' then begin
             if (pos('..', DestPath + IMAPFolder)=0) and (DirectoryExists(DestPath + IMAPFolder + '\'))
                then DestPath := DestPath + IMAPFolder + '\'
                else Log(LOGID_Warn, 'Folder "' + IMAPFolder + '" doesn''t exists in this IMAP-Mailbox. Mail is saved in INBOX');
           end;
           {/HSR}
           EnterCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
           try
              if CfgAccounts.IMAPMailboxLock( DestPath, True ) then begin
                 ImapMailbox := TImapMailbox.Create( DestPath );
                 if not Assigned( ImapMailbox ) then
                    raise Exception.Create( 'unable to create imap mailbox object' );
              end else begin
                 ImapMailbox := TImapMailbox( CfgAccounts.GetIMAPMailbox( DestPath ) );
                 if not Assigned( ImapMailbox ) then
                    raise Exception.Create( 'unable to get imap mailbox object' );
              end;
              ImapMailbox.AddUser( nil );
           finally
             LeaveCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
           end;
           ImapMailbox.Lock;
           Filename := DestPath + IntToStr( ImapMailbox.GetUIDnext ) + '.msg'
        end else begin
           // POP3-Postfach
           Filename := CfgHamster.GetUniqueMsgFilename( DestPath, 'mail' );
        end;
        If Pos(LowerCase(PATH_MAILS), LowerCase(Filename)) = 1
           then Account := LowerCase(ExcludeTrailingBackslash(
                              ExtractFilePath(ExtractRelativePath (PATH_MAILS, Filename))))
           else Account := '';

        With TFileStream.Create(FileName, fmCreate) do try
           If MsgText > '' then Write( MsgText[1], Length(MsgText) )
        finally
           free
        end;

        if SaveToImap then begin
           ImapMailbox.AddIncomingMessage(IMAPFlags);
           ImapMailbox.Unlock;
           EnterCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
           try
              ImapMailbox.RemoveUser( nil, ImapMbxNotInUse );
              if ImapMbxNotInUse then ImapMailbox.Free;
           finally
              LeaveCriticalSection( CS_IMAPMBCreate ); //HSR //MBCreate
           end
        end;
        {/IMAP}
        Result := True;
     except
        on E: Exception do begin
           Log( LOGID_WARN, TrGlF(kLog, 'Warning.SaveMailfile.Exception',
                 'Couldn''t save mail file: %s', [E.Message] ) );
        end;
     end;
     if DestPath=PATH_MAIL_OUT then LeaveCriticalSection( CS_LOCK_MAILOUT_ADD )
                               else LeaveCriticalSection( CS_LOCK_MAILBOX_ALL );
end;


// function StoreLocalInfoMail( LocalRecipient, Subject, MailBody: String ): Boolean;
// Durch SendInfoMail von HRR ersetzt


{-----------------------------------------------------------------------------}

// procedure RemoveMailMessageID( var MsgText: String );
// In den MailRouter verlagert.

procedure TArticleBase.EstimateValues(GroupName: String;
  var ArtCount: Integer; var NewGroup: Boolean);
var  SR: TSearchRec;
begin
   ArtCount := 0;
   NewGroup := True;
   if FindFirst( PATH_GROUPS + GroupName + '\data' + EXT_IDX, faAnyFile, SR ) = 0 then begin
      ArtCount := SR.Size div sizeof(TIndexRec);
      FindClose( SR );
      if ArtCount > 1 then
         NewGroup := False
      else With TIniFile.Create( PATH_GROUPS + GroupName + '\data' + EXT_CFG ) do try
         If ReadInteger('Ranges','Local.Max',0) > 1 then NewGroup := False
      finally Free end
   end
end;

procedure TArticleBase.PurgeReset(GrpHdl: Integer);
var  OAF: TOpenArtFile;
begin
   if EnterFile( GrpHdl, OAF ) then begin
      OAF.ArtFile.PurgeReset;
      OAF.ArtFile.LeaveThisFile;
   end
end;

Procedure CreateGroup( GroupName: String );
var  GrpHdl: Integer;
     Desc  : String;
begin
     GrpHdl := ArticleBase.Open( GroupName ); // open also creates dirs
     if GrpHdl>=0 then begin
        Desc := GlobalGroupDesc( GroupName );
        if Desc<>'' then ArticleBase.Description[GrpHdl] := Desc;
        ArticleBase.Close( GrpHdl );
     end else begin
        Log( LOGID_ERROR, TrGlF(kLog, 'Error.CreateGroup.Exception',
              'Could not create/open %s!', GroupName))
     end;
     With TStringList.Create do try
        sorted := true;
        LoadFromFile ( PATH_BASE + CFGFILE_ACTIVE );
        If IndexOf (GroupName) < 0 then begin
           Add ( Groupname );
           SaveToFile ( PATH_BASE + CFGFILE_ACTIVE )
        end
     finally
        free
     end
end;

//Start Chip Verde Mod - Modify Local Posts - Jan 07,2002
{-----------------------------------------------------------------------------}

Var TempPath: String = '';

Function ModifyMessage ( Const Typ: TActiontype; Const MsgText: String;
            Out NewText: String ): TMessageChange;
var  FN: String; p, L: Integer;
Const Prefix = '~HM'+#0;
begin
   Result := mcOriginal;
   try
      If TempPath = '' then begin
         SetLength(TempPath, MAX_PATH);
         L := GetTempPath(MAX_PATH, @TempPath[1]);
         SetLength(TempPath, L)
      end;
      SetLength(FN, MAX_PATH);
      If GetTempFileName( @TempPath[1], @Prefix[1], 0, @FN[1]) = 0 then exit;
      p := Pos(#0, FN); If p > 0 then SetLength(FN, p-1);
      With TFileStream.Create(FN, fmCreate) do try
         If MsgText>'' then Write(MsgText[1], Length(MsgText))
      finally
         free
      end;
      If Actions.Exec ( Typ, FN, '' ) then begin
         If FileExists(FN) then begin
            With TFileStream.Create(FN, fmOpenRead + fmShareDenyNone) do try
               SetLength(NewText, Size);
               If Size > 0 then Read (NewText[1], Size);
               If NewText <> MsgText then Result := mcChanged
            finally Free end;
            DeleteFile(FN)
         end else begin
            NewText := '';
            Result := mcDeleted
         end;
      end
    except
       on E: Exception do begin
          Log( LOGID_WARN, TrGlF(kLog, 'Warning.SaveNewsfile.Exception',
                'Couldn''t save news file: %s', [E.Message] ) );
       end
    end
end;


//End Chip Verde Mod - Modify Local Posts - Jan 07,2002
{-----------------------------------------------------------------------------}

function TArticleBase.DeleteArticleByMIDAndGroup(const DelMID,
  Groupname: String; const DeleteHistEntries: boolean): Boolean;
Var GrpNam: String; ArtNo: Integer;
    Xref, s: String; GrpHdl, j: Integer;
begin
   If Groupname = '' then begin
      Result := NewsHistory.LocateMID ( DelMID, GrpNam, ArtNo )
   end else begin
      GrpNam := Groupname;
      Result := NewsHistory.LocateMIDInGroup( DelMID, GrpNam, ArtNo );
   end;
   If Not Result then Exit;

   // delete history-entry
   If DeleteHistEntries then begin
      NewsHistory.RemoveEntry( DelMID, StrToCRC32(lowercase(GrpNam)), ArtNo )
   end;
   // get Xref of article and delete "main"-instance
   GrpHdl := Open( GrpNam );
   if GrpHdl<0 then exit;
   try
      With TArticle.Create do try
         Text := ReadArticle( GrpHdl, ArtNo );
         Xref := Header['Xref:'];
         If Text > '' then DeleteArticle( GrpHdl, ArtNo )
      finally
         Free
      end
   finally
      Close( GrpHdl )
   end;
   If Groupname = '' then begin
      // delete article-instances referenced in Xref
      j := PosWhSpace( Xref ); if j=0 then exit;
      System.Delete( Xref, 1, j ); // skip hostname
      while Xref<>'' do begin
         j := PosWhSpace( Xref );
         if j=0 then begin
            s := Xref; Xref := '';
         end else begin
            s := copy( Xref, 1, j-1 );
            System.Delete( Xref, 1, j ); // skip hostname
         end;
         j := Pos( ':', s );
         if j>0 then begin
            ArtNo := strtoint( copy( s, j+1, Length(s)-j ) );
            s := copy( s, 1, j-1 );
            if s<>GrpNam then begin // delete additional references
               GrpHdl := Open( s );
               if GrpHdl>=0 then begin
                  DeleteArticle( GrpHdl, ArtNo );
                  Close( GrpHdl );
                  If DeleteHistEntries then begin
                     NewsHistory.RemoveEntry(DelMID, StrToCRC32(lowerCase(s)), ArtNo)
                  end
               end
            end
         end
      end
   end
end;

Function TArticleBase.DeleteArticleByNrAndGroup(const ArtNo: Integer;
  Const Groupname: String): boolean;
Var GrpHdl: integer;
begin
   Result := false;
   GrpHdl := Open( GroupName );
   if GrpHdl<0 then exit;
   try
      Result := DeleteArticle( GrpHdl, ArtNo )
   finally
      Close( GrpHdl )
   end;
end;

end.
