// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cClientSMTP;

interface
{JW} {SSL}
uses Classes, ScktComp, cClientBase, uSSL; {MG}{SSL}
{JW}


const
   SRVFILE_GREETING     = 'greeting.txt';

type
   TSendMailResult = ( SMR_INPROGRESS, SMR_DELIVERED, SMR_TRYAGAINLATER,
                       SMR_ABORTED, SMR_DISCONNECT, SMR_CANTOPEN );

type
   TClientSocketSMTP = class( TClientSocketBase )
      public
         {JW} {SMTP 3.12.00}
         ScriptAuth: boolean;
         {JW}
         ResultCode: Integer;
         EHLOResult: String;
         function  ResultListFollows: Boolean; override;
         procedure SendCmnd( Cmnd : String; Special: Integer; FilterLog: Boolean ); override;
         procedure SendCmndGetList( Cmnd: String; Special: Integer ); override;
         function  Remote_SASL_Login( SASL_WANTED, SASL_REMOTE, AUser, APass: String ): Boolean;
         function  Login( AUser, APass: String ): Boolean; override;
         function  NegotiateSSL: Boolean; override; {MG}{SSL}
   end;

   TClientSMTP = class
      private
         Server, Port, User, Pass: String;
         SSLMode, SSLVerifyLevel: Integer; SSLCaFile: String; {MG}{SSL}
         SMTP: TClientSocketSMTP;
         function DoSendMailfile( Const MailOutFile: String; Out LogInfos: String ): TSendMailResult;
      public
         ScriptAuth: boolean; {JW} {SMTP 3.12.00}
         function  State : Integer;
         procedure Connect;
         procedure Disconnect;
         function  SendMailfile( Const MailFile: String ): Boolean;
         constructor Create( AServer, APort, AUser, APass: String ;
                             ASSLMode, ASSLVerifyLevel: Integer;
                             ASSLCaFile: String );
         destructor Destroy; override;
   end;

implementation

uses SysUtils, FileCtrl, uTools, uWinSock, Global, Config, cArticle, cArtFiles,
     uDateTime, uEncoding, IniFiles, cStdForm, cMailRouter, cLogFile, uMD5, uSha1;

procedure TClientSocketSMTP.SendCmnd( Cmnd : String; Special: Integer; FilterLog: Boolean );
begin
   inherited SendCmnd( Cmnd, Special or SPECIAL_ACCEPTSMTPMULTILINE, FilterLog );
   if ResultLine='' then ResultLine := '999 (timeout or connection lost)';
   ResultCode := strtoint( copy( ResultLine, 1, 3 ) );
end;

procedure TClientSocketSMTP.SendCmndGetList( Cmnd: String; Special: Integer );
begin
   inherited SendCmndGetList( Cmnd, Special or SPECIAL_ACCEPTSMTPMULTILINE );
end;

function TClientSocketSMTP.ResultListFollows: Boolean;
begin
   Result := False;
end;

function TClientSocketSMTP.Remote_SASL_Login( SASL_WANTED, SASL_REMOTE, AUser, APass: String ): Boolean;
var  Mechanism, sWanted, sLocal, sRemote, s: String;
     i: Integer;
     realm, nonce, algorithm, qop, digesturi, response, nc,
     A1, A2, cnonce, charset, mandatory_realm,
     mandatory_digesturi : String;
begin
   Result := False;

   sRemote := UpperCase( SASL_REMOTE );  // supported by server
   sLocal  := UpperCase( SMTPLocalSASL  );  // supported by Hamster
   sWanted := UpperCase( SASL_WANTED );  // preference of user
   if sWanted='' then sWanted := sLocal; // no preference, use Hamster-default

   Mechanism := '';

   while sWanted<>'' do begin
      i := PosWhSpace( sWanted );
      if i=0 then begin
         s := sWanted;
         sWanted := '';
      end else begin
         s := copy( sWanted, 1, i-1 );
         System.Delete( sWanted, 1, i );
      end;

      if Pos( ' ' + s + ' ', ' ' + sLocal + ' ' ) = 0 then begin
         Log( LOGID_WARN, FServer + ': Invalid SASL-setting "' + SASL_WANTED + '"' );
      end else begin
         if Pos( ' ' + s + ' ', ' ' + sRemote + ' ' ) > 0 then begin
            Mechanism := s;
            break;
         end;
      end;
   end;

   if Mechanism='DIGEST-MD5' then begin
      Log(LOGID_DETAIL, 'Authentification with mechanism "DIGEST-MD5" ');
      SendCmnd( 'AUTH DIGEST-MD5', 0, False );
      if (ClientState in [csERROR, csDISCONNECTED]) or (ResultCode<>334) then begin
         Log( LOGID_ERROR, FServer + ': AUTH/1 failed with "' + ResultLine + '"' );
         ClientState:=csERROR;
         exit;
      end else begin
         s:=copy(ResultLine,PosWhSpace( ResultLine )+1,500);
         s:=DecodeB64( s[1], length(s) );
         Log( LOGID_DEBUG, FServer + ' digest-challenge: '+s);
         charset:=ExtractQuotedParameter(s,'charset');
         if charset<>'' then Log( LOGID_WARN, FServer + ': server supported charset, but Hamster not');
         realm:=ExtractQuotedParameter(s,'realm');
         if pos('@',realm)=0 then begin
            if lowercase(trim(FServer))<>lowercase(trim(realm)) then begin
               Log( LOGID_WARN, FServer + ': requested servername differs from name in server-answer '+realm);
            end
         end else begin
            if lowercase(trim(FServer))<>lowercase(trim(copy(realm,pos('@',realm)+1,length(realm)))) then begin
               Log( LOGID_WARN, FServer + ': requested servername differs from name in server-answer '+realm);
            end
         end;
         nonce:=ExtractQuotedParameter(s,'nonce');
         if nonce='' then  begin
            Log( LOGID_ERROR, FServer + ': missing nonce in challenge');
            ClientState:=csERROR;
            exit;
         end;
         algorithm:=ExtractQuotedParameter(s,'algorithm');
         if algorithm='' then begin
            Log( LOGID_ERROR, FServer + ': missing algorithm in challenge');
            ClientState:=csERROR;
            exit;
         end;
         if 'md5-sess'<>algorithm then begin
            Log( LOGID_ERROR, FServer + ': not supported hash algorithm '+algorithm);
            ClientState:=csERROR;
            exit;
         end;
         qop:=ExtractQuotedParameter(s,'qop');
         if (qop='') or (Pos(',auth,', LowerCase(','+qop+','))>0) then begin
            qop:='auth'
         end else begin
            Log( LOGID_ERROR, FServer + ': not supported hash quality protection '+qop);
            ClientState:=csERROR;
            exit
         end;
         mandatory_realm:='';
         mandatory_digesturi:='';
         i := CfgHamster.SmtpServerIndexOf[ FServer ];
         if i>=0 then begin
            With TIniFile.Create( CfgHamster.SmtpServerPath[i] + SRVFILE_INI ) do try
               mandatory_realm := ReadString( 'SMTP', 'REALM', '' );
               mandatory_digesturi := ReadString( 'SMTP', 'DIGESTURI', '');
            finally
               Free
            end
         end;
         if mandatory_realm > '' then realm:=mandatory_realm;
         if mandatory_digesturi > ''
            then digesturi:=mandatory_digesturi
            else digesturi:='smtp/'+realm+'/'+FServer;
         Log( LOGID_DETAIL, FServer+' using parameter: '+realm+' '+nonce+' '+algorithm+' '+qop);
         cnonce:= SHA1ofStr( nonce+
                  IntToHex(PRNG(MaxInt),8)+IntToHex(PRNG(MaxInt),8)+
                  IntToHex(PRNG(MaxInt),8)+IntToHex(PRNG(MaxInt),8));
         cnonce:= EncodeB64(cnonce[1],length(cnonce));
         nc:='00000001';
         A1:=MD5ofStr(AUser+':'+realm+':'+APass)+':'+nonce+':'+cnonce;
         A2:='AUTHENTICATE:'+digesturi;
         response:=MD5toHex(MD5ofStr(MD5toHex(MD5ofStr(A1))+':'+
             nonce+':'+nc+':'+cnonce+':'+qop+':'+MD5toHex(MD5ofStr(A2))));
         s :='username="'+AUser+'",realm="'+realm+'",nonce="'+nonce+'",cnonce="'+cnonce
             +'",nc='+nc+',qop=auth'+',digest-uri="'+digesturi+'",response='+response;
         Log( LOGID_DETAIL, FServer+' digest-answer: '+s);
         s := EncodeB64( s[1], length(s));
         SendCmnd( s, 0, False );
         if ClientState in [csERROR, csDISCONNECTED] then exit;
         if ResultCode<>334 then begin
            Log( LOGID_ERROR, FServer + ': AUTH/2 failed with "' + ResultLine + '"' );
            ClientState:=csERROR;
            exit;
         end else begin
            ClientState := csREADY
         end;
         s:=copy(ResultLine,PosWhSpace( ResultLine )+1,500);
         s:=DecodeB64( s[1], length(s) );
         if length(s)=40 then begin
           s:=ExtractQuotedParameter(s,'rspauth')
         end else begin
           Log(LOGID_ERROR, FServer + ': wrong server response by login' );
           ClientState:=csERROR;
           exit;
         end;
         A2:=':'+digesturi;
         response:=MD5toHex(MD5ofStr(MD5toHex(MD5ofStr(A1))+':'+
            nonce+':'+nc+':'+cnonce+':'+qop+':'+MD5toHex(MD5ofStr(A2))));
         if response=s then begin
           SendCmnd( '', 0, True );
           result:=true;
         end else begin
           Log( LOGID_ERROR, FServer +': wrong server response by login,'+
               ' server know not the password' );
           Log( LOGID_ERROR, FServer +': respauth='+s );
           ClientState:=csERROR;
           exit;
         end;
      end;
   end else
   if Mechanism='PLAIN' then begin
      SendCmnd( 'AUTH PLAIN', 0, False );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode<>334 then begin
         Log(LOGID_ERROR,FServer+': AUTH LOGIN/1 failed with "'+
             ResultLine + '"' );
         exit;
      end;
      s:=chr(0)+AUser+chr(0)+APASS;
      s := EncodeB64( s[1], length(s) ); // Username + Passwort
      SendCmnd( s, 0, True );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode=235 then
         Result := True
      else
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/2 failed with "' +
              ResultLine + '"' );
   end else
   if Mechanism='CRAM-MD5' then begin
      SendCmnd( 'AUTH CRAM-MD5', 0, False );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode<>334 then begin
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/1 failed with "' +
              ResultLine + '"' );
         exit;
      end;
      i := PosWhSpace( ResultLine );
      s:=copy(ResultLine,i+1,500);
      s:=DecodeB64( s[1], length(s) );
      s:=MD5HMAC( APass,s );
      s:=MD5toHex( s );
      s:=AUser+' '+s;
      s := EncodeB64( s[1], length(s) ); // Username + Digest
      SendCmnd( s, 0, True );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode=235 then
         Result := True
      else
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/2 failed with "' +
              ResultLine + '"' );
   end else
   if Mechanism='CRAM-SHA1' then begin
      SendCmnd( 'AUTH CRAM-SHA1', 0, False );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode<>334 then begin
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/1 failed with "' +
         ResultLine + '"' );
         exit;
      end;
      i := PosWhSpace( ResultLine );
      s:=copy(ResultLine,i+1,500);
      s:=DecodeB64( s[1], length(s) );
      s:=HMAC_SHA1( s, APass );
      s:=AUser+' '+s;
      s := EncodeB64( s[1], length(s) ); // Username + Digest
      SendCmnd( s, 0, True );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode=235 then
         Result := True
      else
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/2 failed with "' +
              ResultLine + '"' );
   end else
{JW}
   if Mechanism='LOGIN' then begin

      SendCmnd( 'AUTH LOGIN', 0, False );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode<>334 then begin
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/1 failed with "' + ResultLine + '"' );
         exit;
      end;

      s := EncodeB64( AUser[1], length(AUser) ); // Username
      SendCmnd( s, 0, True );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode<>334 then begin
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/2 failed with "' + ResultLine + '"' );
         exit;
      end;

      s := EncodeB64( APass[1], length(APass) ); // Password
      SendCmnd( s, 0, True );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode=235 then begin
         Result := True;
      end else begin
         Log( LOGID_ERROR, FServer + ': AUTH LOGIN/3 failed with "' + ResultLine + '"' );
      end;

   end else begin

      Log( LOGID_ERROR, FServer + ': No valid SASL mechanism for AUTH found!' );
      Log( LOGID_ERROR, FServer + ': "' + SASL_REMOTE + '"'
                        + ' Hamster: "' + SMTPLocalSASL  + '"'
                        + ' User: "'    + SASL_WANTED + '"' );

   end;
end;

{MG}{SSL}
function TClientSocketSMTP.NegotiateSSL : Boolean;  // RFC 2487
var  i: Integer;
     TLSAnnounced: Boolean;
begin
   Result := False;
   if ClientState<>csCONNECTED then exit;
   ClientState := csLOGIN;
   SendCmnd( 'EHLO localhost', 0, False );
   if ClientState in [csERROR, csDISCONNECTED] then exit;
   if ResultCode = 250 then begin
      TLSAnnounced := false;
      With TStringList.Create do try
         Text := ResultLine;
          for i:=0 to Count-1 do begin
             if UpperCase( copy( Strings[i], 5, 12 ) ) = 'STARTTLS' then begin
                TLSAnnounced := true;
                break;
             end
          end
      finally
         free
      end;
      SendCmnd( 'STARTTLS', 0, False );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode = 220 then begin
         if Not TLSAnnounced then begin
            Log( LOGID_INFO, Format(TrGl(kLog, 'SSL.NotAnnouncedInEHLO',
                '[%s] Server supports TLS but doesn''t announce in EHLO reply!'),
                [FServer]) )
         end;
         Log( LOGID_DEBUG, Format(TrGl(kLog, 'SSL.Handshake',
              '[%s] starting SSL/TLS handshake ...'), [FServer]) );
         Result := StartSSL;
      end else begin
         Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.ProtoNotSupported',
              '[%s] Server does not support %s over SSL'), [FServer, 'SMTP']) )
      end
   end else begin
      Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.NoSMTPEhlo',
           '[%s] Server does not support SMTP service extensions'), [FServer]) )
   end
end;
{/SSL}

function TClientSocketSMTP.Login( AUser, APass : String ): Boolean;
var  HeloName: String;
     UseAUTH : Boolean;
     TS: TStringList;
     SASL_REMOTE, SASL_WANTED, s: String;
     i: Integer;
begin
   Result := False;
   Log( LOGID_DEBUG, Self.Classname + '.Login' );
   if ClientState<>csCONNECTED then exit;
   ClientState := csLOGIN;
   EHLOResult  := '';
{JW} {Heloname}
   if Def_FQDN<>''
      then HeloName := Def_FQDN
      else HeloName := LookupLocalHostName; //Socket.LocalHost;
{JW}
   if HeloName='' then HeloName := 'localhost';
   UseAUTH     := False;
   SASL_WANTED := '';
   if (AUser<>'') and (APass<>'') then begin

      // Kill Prefix / {JH}
      if copy(APass,1,5)='PASS:' then System.Delete( APass, 1, 5 )
      else if copy(APass,1,5)='APOP:' then System.Delete( APass, 1, 5 )
      else if copy(APass,1,9)='CRAM-MD5:' then System.Delete( APass, 1, 9 )
      else if copy(APass,1,6)='PLAIN:' then System.Delete( APass, 1, 6 )
      else if copy(APass,1,6)='LOGIN:' then System.Delete( APass, 1, 6 )
      else if copy(APass,1,5)='SASL:' then System.Delete( APass, 1, 5 )
      else if copy(APass,1,5)='AUTH:' then System.Delete( APass, 1, 5 )
      else if copy(APass,1,10)='CRAM-SHA1:' then System.Delete( APass, 1,10 );

      UseAUTH := True;   {JW} {SMTP 3.12.00}
      SASL_WANTED := SMTPLocalSASL; {JW} {SMTP 3.12.00}
      i := CfgHamster.SmtpServerIndexOf[ FServer ];
      if i>=0 then begin
         With TIniFile.Create( CfgHamster.SmtpServerPath[i] + SRVFILE_INI ) do try
            If not ScriptAUTH then UseAUTH :=  ReadBool  ( 'SMTP', 'AUTH', True);
            SASL_WANTED := ReadString( 'SMTP', 'SASL', SMTPLocalSASL )
         finally
            Free;
         end
      end;
   end;

   if UseAUTH then begin

      SendCmnd( 'EHLO ' + HeloName, 0, False );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode=250 then EHLOResult := ResultLine // CRLF-separated
                        else UseAUTH := False;

   end;

   if UseAUTH then begin

      // get SASL mechanisms supported by server
      SASL_REMOTE := '';
      TS := TStringList.Create;
      TS.Text := EHLOResult;
      for i:=0 to TS.Count-1 do begin
         s := UpperCase( copy( TS[i], 5, 999 ) ); // skip '250[- ]'
         if (copy(s,1,4)='AUTH') and (length(s)>5) and (s[5] in [' ','=']) then begin
            SASL_REMOTE := TrimWhSpace( copy( s, 6, 999 ) ); // skip 'AUTH[ =]'
            break;
         end;
      end;
      TS.Free;
      if SASL_REMOTE='' then begin
         Log( LOGID_ERROR, TrGlF(kLog, 'Error.SMTP.AuthNotAvailable',
                      'SMTP-Auth not available For server %s', fServer));
         Log( LOGID_ERROR, 'SMTP ' + FServer + ': ' + ResultLine );
         ClientState := csERROR;
         exit;
      end;

      if Remote_SASL_Login( SASL_WANTED, SASL_REMOTE, AUser, APass ) then begin
         ClientState := csREADY;
      end else begin
         ClientState := csERROR;
      end;

   end else begin

      SendCmnd( 'HELO ' + HeloName, 0, False );
      if ClientState in [csERROR, csDISCONNECTED] then exit;
      if ResultCode=250 then ClientState := csREADY;

   end;

   if ClientState=csREADY then Result:=True;
end;

function TClientSMTP.State: Integer;
begin
   if Assigned( SMTP ) then Result := SMTP.ClientState
                       else Result := csDISCONNECTED;
end;

procedure TClientSMTP.Connect;
var  s: String;
     ConnectWithSSL, ok: Boolean;
begin
   if Assigned( SMTP ) then Disconnect;
   ok := false;
   SMTP := TClientSocketSMTP.Create( nil );
   SMTP.ScriptAuth:=ScriptAuth;
   ConnectWithSSL := (SSLMode = 1);
   if SMTP.Connect( Server, Port, PATH_SERVER + Server + '\' + SRVFILE_INI,
                    ConnectWithSSL, SSLVerifyLevel, SSLCaFile )
   then begin
      if State = csCONNECTED then begin
         if SSLMode = 2 then SMTP.NegotiateSSL;
         if (SSLMode = 3) and (not SMTP.NegotiateSSL) then begin Disconnect; exit; end;
         if State = csLOGIN then SMTP.ClientState := csCONNECTED;
         If SMTP.Login( User, Pass ) then begin
            ok := true
         end else begin
            s := SMTP.ResultLine;
            if s<>'' then s:=' ("'+s+'")';
            Log( LOGID_WARN, 'Login failed ' + s );
         end;
         if (SMTP.Greeting>'') and DirExists2(PATH_SERVER + Server) then begin
             HamFileRewriteLine ( PATH_SERVER + Server + '\' +
                                  SRVFILE_GREETING, SMTP.Greeting );
         end
      end
   end;
   SMTP.ConnectionStatistic ( PATH_SERVER + Server + '\' + SRVFILE_INI, ok)
end;

procedure TClientSMTP.Disconnect;
begin
   if not Assigned( SMTP ) then exit;
   try SMTP.Disconnect except end;
   FreeAndNil(SMTP)
end;


function TClientSMTP.DoSendMailfile( Const MailOutFile: String; Out LogInfos: String ): TSendMailResult;
var  CmdLine, Cmd, CmdReplies, MailFrom, s : String;
     PreData, HadValidRcpt                 : Boolean;
     LineNo, i, k                          : Integer;
     TheMail                               : TStringlist;
     LastProgress                          : Integer; {kms}
     {JW} {Notify}
      Attempts                             : Integer;
      MailFile                             : String;
     {/JW}

     function TryToReset: Boolean;
     begin
        Result := False;
        SMTP.SendCmnd( 'RSET', 0, False );
        if SMTP.ResultCode=250 then Result:=True;
     end;

begin
   LogInfos := '';
   
   Result := SMR_DISCONNECT;

   if not FileExists2( MailOutFile ) then exit;
   if not( State in [csREADY, csLOGIN] ) then exit;

   Log( LOGID_DETAIL, '[' + Server + ', SMTP] SendMail '
                          + ExtractFilename(MailOutFile) );

   TheMail := TStringlist.Create;
   try
      try
         TheMail.LoadFromFile( MailOutFile )
      except
         Log( LOGID_Error, TrGlf(kLog, 'SMTP.SendMail.CantLoad',
            '[%s, SMTP] can not load Mailfile %s',
            [Server, ExtractFilename(MailOutFile)] ) );
         Result := SMR_CANTOPEN; //HSR //SendCantOpen
         Exit;
      end;

      PreData      := True;
      LineNo       := 0;
      MailFrom     := '';
      HadValidRcpt := False;
      CmdReplies   := '';

      Result := SMR_INPROGRESS;

      LastProgress := -1; {kms}

      while LineNo < TheMail.Count do begin

         if Result=SMR_ABORTED    then break;
         if Result=SMR_DISCONNECT then break;
         if not( State in [csREADY, csLOGIN] ) then break;

         s := TheMail[ LineNo ];

         if PreData and (copy(s,1,1)='!') then begin

            CmdLine := s;
            System.Delete( CmdLine, 1, 1 );

            i := PosWhSpace( CmdLine );
            if i=0 then Cmd:=CmdLine else Cmd:=copy(CmdLine,1,i-1);
            Cmd := UpperCase( Cmd );

            if (Cmd='MAIL') or (Cmd='RCPT') then begin

               i := Pos(':', CmdLine);
               While Copy(CmdLine, i+1, 1)=' ' do Delete(CmdLine, i+1, 1);
               
               CmdReplies := CmdReplies + '| ' + CmdLine + #13#10;
               SMTP.SendCmnd( CmdLine, 0, False );

               if (SMTP.ResultLine='') or (SMTP.ResultCode=999) then begin // Timeout
                  CmdReplies := CmdReplies + '| -> (timeout)' + #13#10;
                  Log( LOGID_ERROR, 'SendMail aborted: '
                                    + CmdLine + ' -> ' + 'Timeout (' + Server + ')' );
                  Result := SMR_DISCONNECT;
                  break;
               end;

               CmdReplies := CmdReplies + '| -> ' + SMTP.ResultLine + #13#10;

               case SMTP.ResultCode of

                  250, 251: begin // ok
                     // 250 Requested mail action okay, completed
                     // 251 User not local; will forward to <forward-path>
                     if Cmd='MAIL' then begin
                        MailFrom := TrimWhSpace(copy(CmdLine,11,99));
                        LogInfos := 'FROM:'+Mailfrom
                     end;
                     if Cmd='RCPT' then begin
                        // mark recipient as delivered
                        If HadValidRcpt
                           then LogInfos := LogInfos + ','
                           else LogInfos := LogInfos + ' TO:';
                        LogInfos := LogInfos + TrimWhSpace(copy(CmdLine,9,99));
                        TheMail[ LineNo ] := '!X-OK: ' + CmdLine;
                        HadValidRcpt := True;
                     end;
                  end;

   //               450, 451, 452, 500, 501, 550, 551, 552, 553: begin // delivery failed
                  {JW} {Attempts}
                  450, 451, 452, 500, 501, 550, 551, 552, 553, 571: begin // delivery faile
                  {/JW}
                     // 450 Requested mail action not taken: mailbox unavailable
                     // 451 Requested action aborted: local error in processing
                     // 452 Requested action not taken: insufficient system storage
                     // 500 Syntax error, command unrecognized
                     // 501 Syntax error in parameters or arguments
                     // 550 Requested action not taken: mailbox unavailable
                     //   | Unknown local part
                     //   | relaying to <x@y> prohibited by administrator
                     // 551 User not local; please try <forward-path>
                     // 552 Requested mail action aborted: exceeded storage allocation
                     // 553 Requested action not taken: mailbox name not allowed
                     // 571 Remote sending only allowed with authentification
                     if Cmd='MAIL' then begin
                        Log( LOGID_ERROR, 'SendMail aborted: '
                                          + CmdLine + ' -> ' + SMTP.ResultLine );
                        if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                        break;
                     end;
                     if Cmd='RCPT' then begin
                        Log( LOGID_WARN, 'SendMail failed: '
                                         + CmdLine + ' -> ' + SMTP.ResultLine );
                        Result := SMR_TRYAGAINLATER;
                     end;
                  end;

                  else begin
                     Log( LOGID_ERROR, 'SendMail aborted: '
                                       + CmdLine + ' -> ' + SMTP.ResultLine );
                     if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                     break;
                  end;

               end; // case SMTP.ResultCode of

            end; // if (Cmd='MAIL') or (Cmd='RCPT')

         end else begin

            if PreData then begin

               // first line after envelope-headers

               if MailFrom='' then begin
                  Log( LOGID_ERROR, 'SendMail aborted: No valid envelope-from!' );
                  if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                  break;
               end;
               if not(HadValidRcpt) then begin
                  Log( LOGID_ERROR, 'SendMail aborted: No valid recipients!' );
                  if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                  break;
               end;

               CmdReplies := CmdReplies + '| DATA' + #13#10;
               SMTP.SendCmnd( 'DATA', 0, False );

               if (SMTP.ResultLine='') or (SMTP.ResultCode=999) then begin // Timeout
                  CmdReplies := CmdReplies + '| -> (timeout)' + #13#10;
                  Log( LOGID_ERROR, 'SendMail aborted: '
                                    + 'DATA' + ' -> ' + 'Timeout (' + Server + ')' );
                  Result := SMR_DISCONNECT;
                  break;
               end;

               CmdReplies := CmdReplies + '| -> ' + SMTP.ResultLine + #13#10;
               if SMTP.ResultCode<>354 then begin
                  Log( LOGID_WARN, 'SendMail failed: '
                                   + 'DATA' + ' -> ' + SMTP.ResultLine );
                  if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                  break;
               end;

               PreData := False;

            end;

            if copy(s,1,1)='.' then s:='.'+s;
            SMTP.SendData( s + #13#10 );
            if not( State in [csREADY, csLOGIN] ) then break; // {JW} {break}

            if trunc(LineNo / TheMail.Count * 100)<>LastProgress then begin {.kms}
              LastProgress:=trunc(LineNo / TheMail.Count * 100);
              Log( LOGID_STATUS, '[' + Server + ', SMTP] SendMail '+
                                 ExtractFilename(MailOutFile) +
                                 ' (' + inttostr(LastProgress) + '%)' );
            end; {/kms}

          end;

         inc( LineNo );
      end;

      if State in [csREADY, csLOGIN] then begin
         if (Result<>SMR_ABORTED) and (Result<>SMR_DISCONNECT) then begin
            if not PreData then begin
               CmdReplies := CmdReplies + '| "." (end of data)' + #13#10;
               SMTP.SendCmnd( '.', 0, False ); // send "end of data"

               if (SMTP.ResultLine='') or (SMTP.ResultCode=999) then begin // Timeout
                  CmdReplies := CmdReplies + '| -> (timeout)' + #13#10;
                  Log( LOGID_ERROR, 'SendMail aborted: '
                                    + '"." (end of data)' + ' -> ' + 'Timeout (' + Server + ')' );
                  Result := SMR_DISCONNECT;
               end else begin
                  CmdReplies := CmdReplies + '| -> ' + SMTP.ResultLine + #13#10;
                  if SMTP.ResultCode=250 then begin
                     if Result=SMR_INPROGRESS then Result:=SMR_DELIVERED;
                     if (Result=SMR_TRYAGAINLATER) and HadValidRcpt then begin
                        // mark successful recipients as done in envelope-headers
                        TheMail.SaveToFile( MailOutFile );
                     end;
                  end else begin
                     Log( LOGID_WARN, 'SendMail failed: '
                                      + '"." (end of data)' + ' -> ' + SMTP.ResultLine );
                  end;
               end;
            end;
         end;
      end;

   {JW} {Notify}
         i := Pos( '.' + CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ),
            lowercase(MailOutFile) );
         if i>0 then MailFile:=copy(MailOutFile,1,i-1);
         i := Pos( '-', MailFile );
         if i=0 then begin
            Attempts := 0;
         end else begin
            Attempts := strtoint( copy(MailFile,i+1,3) );
         end;
      // notify sender of errors
      if (Result<>SMR_DELIVERED) and ((attempts=0) or (attempts=Def_SendMailAttemptsMax-1)) then begin

   {/JW}

   //   if Result<>SMR_DELIVERED then begin
         s := '[Hamster]' + #13#10#13#10
            + 'The following mail could not be delivered to any or all '
            + 'recipients:' + #13#10#13#10
            + '> ' + MailOutFile + #13#10#13#10
            + 'Used SMTP-server:' + #13#10#13#10
            + '> ' + Server + #13#10#13#10
            + 'Delivery-results:' + #13#10#13#10
            + CmdReplies + #13#10
            + 'Header-lines of undelivered mail:' + #13#10#13#10;
         k := TheMail.IndexOf('');
         If k < 0 then k := TheMail.Count-1;
         for i:=0 to k-1 do begin
            s := s + '| ' + TheMail[i] + #13#10;
         end;
         {JW} {InfoMail}
         if pos('<',Mailfrom) = 1 then Mailfrom:=Copy(MailFrom,2,MaxInt);
         if pos('>',Mailfrom) = length(MailFrom) then
           Mailfrom := Copy(MailFrom,1,length(MailFrom)-1);
         {/JW}
         SendInfoMail( MailFrom, '[Hamster] Mail-delivery failed!', '', s );
      end
   finally
      TheMail.Free
   end
end;

function TClientSMTP.SendMailfile( Const MailFile: String ): Boolean;
var  MailPath, MailName, ResLog, s, LogInfos: String;
     Attempts, i: Integer;
     SMResult: TSendMailResult;
begin
   Result := True; // =continue

   SMResult := DoSendMailfile( MailFile, LogInfos );
   if SMResult=SMR_INPROGRESS then SMResult:=SMR_DISCONNECT;

   case SMResult of
      SMR_DELIVERED    : ResLog:='250 OK.';
      SMR_TRYAGAINLATER: ResLog:='991 Invalid recipient.';
      SMR_ABORTED      : ResLog:='992 Aborted due to errors.';
      SMR_DISCONNECT   : ResLog:='993 Timeout or unrecoverable error.';
      SMR_CANTOPEN     : ResLog:='994 File couldn''t be opened.'; //HSR //SendCantOpen
   end;

   if SMResult=SMR_DELIVERED then begin

      // Mail could be delivered to all recipients, so delete it:
      Log( LOGID_INFO, TrGlF(kLog, 'Info.MailSent', 'Mail %s sent.', ExtractFilename(MailFile)));
      DeleteFile( MailFile );

   end else begin

      // Mail could not be delivered to any or all recipients, so keep the
      // mail in outbox to allow corrections in it and/or to try it again
      // later.
      // After a configurable number of (unsuccessful) attempts, the mailfile
      // is renamed to "*.err", so it's not sent again without prior user-
      // interaction.
      //
      // Note: Successful recipients were marked as such in the mailfile, so
      //       they won't get the mail again and again.
      // Note: Sender (or at least "admin") already got an error-notification.

      MailPath := ExtractFilePath( MailFile );
      MailName := ExtractFilename( MailFile );
      i := Pos( '.' + CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ),
         lowercase(MailName) );
      if i>0 then MailName:=copy(MailName,1,i-1);

      i := Pos( '-', MailName );
      if i=0 then begin
         Attempts := 0;
      end else begin
         Attempts := strtoint( copy(MailName,i+1,3) );
         MailName := copy( MailName, 1, i-1 );
      end;
      inc( Attempts );

      if Attempts>=Def_SendMailAttemptsMax then begin
         ResLog := '999 Delivery failed; no further attempts.';
         if Def_SendMailAttemptsDel then begin
            DeleteFile( MailFile );
         end else begin
            // rename file to prevent further attempts
            MailName := MailName + '-' + inttostr(Attempts-1);
            RenameFile( MailFile, MailPath + MailName + '.err' );
            // notify admin of orphaned file
            s := '[Hamster]' + #13#10#13#10
               + 'After ' + inttostr(Attempts) + ' attempts, an undeliverable '
               + 'mail was renamed to:' + #13#10#13#10
               + '> ' + MailPath + MailName + '.err' + #13#10#13#10
               + 'This file will NOT be deleted or sent out again by Hamster.' + #13#10;
            SendInfoMail( Def_Postmaster, '[Hamster] Mail-delivery aborted!', '', s );
         end;
      end else begin
         // mark file with new attempts-count
         MailName := MailName + '-' + inttostr(Attempts);
         RenameFile( MailFile, MailPath + MailName + '.' +
            CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ) );
      end;

      if SMResult=SMR_DISCONNECT then begin

         // Delivery failed with a fatal error (e.g. timeout) or with server
         // left in an unrecoverable, undefined state. Set error-markers to
         // force disconnect from current server.

         Result := False;
         SMTP.ClientState := csERROR;

      end;

   end;

   // Note result in mailout.log:
   s := DateTimeToTimeStamp( Now ) + ' '
      + 'File='   + ExtractFilename( MailFile ) + ' '
      + 'Server=' + Server + ',' + Port + ' '
      + 'Result=' + ResLog;
   HamFileAppendLine ( PATH_LOGS + 'MailOut.log', s );
   s := Chr(9) + LogInfos;
   HamFileAppendLine ( PATH_LOGS + 'MailOut.log', s );
   
end;

constructor TClientSMTP.Create( AServer, APort, AUser, APass: String;
                                ASSLMode, ASSLVerifyLevel: Integer;
                                ASSLCaFile: String );
begin
   inherited Create;
   Server := AServer;
   Port   := APort;
   User   := AUser;
   Pass   := APass;
   SSLMode        := ASSLMode;
   SSLVerifyLevel := ASSLVerifyLevel;
   SSLCaFile      := ASSLCaFile;
   SMTP   := nil;
end;

destructor TClientSMTP.Destroy;
begin
   if Assigned(SMTP) then Disconnect;
   inherited Destroy;
end;

end.

