// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cFiltersBase;

interface

uses SysUtils, Classes, Global, cPCRE, uTools;

CONST
  Name_INCLUDE1 = 'LOAD'; //HSR //INCLUDE
  Name_INCLUDE2 = 'INCLUDE'; //HSR //INCLUDE
type
  TFilterPatternBase = class
    SelectType : Char;    // '+' | '-' | ' '
    IsRegex    : Boolean;
    Pattern    : String;
  end;

  TFilterLineBase = class
    private
    protected
      fOrgLineNo: Integer;
      fOrgLineText, fOrgFilename: String;
      fFilterPatterns: TList;
      DoMimeDecode : Boolean; // '~'?
      LastSetError : String;
      function GetPatternCount: Integer;
      function GetPatternItem( Index: Integer ): TFilterPatternBase;
    public
      property  PatternCount: Integer read GetPatternCount;
      property  PatternItem[ Index: Integer ]: TFilterPatternBase read GetPatternItem;
      function  PatternAdd( Pat: TFilterPatternBase ): Integer;
      procedure PatternDelete( Index: Integer );
      function  SetFilterLine( Const AFilename: String; Const LineNo: Integer;
         Const FilterLine: String ): Boolean;
      virtual; abstract;
      function  AsString: String; 
      Property  LastError: String read LastSetError;
      procedure Clear;

      constructor Create;
      destructor Destroy; override;
  end;

  TFilterfile = class
  private
    slLines, slFiles: TStringList;
    function GetFilename(i: Integer): String;
    function GetFilenameNr(i: Integer): Integer;
    function GetLine(i: Integer): String;
    function GetLineNr(i: Integer): Integer;
    function GetUsedFile(i: Integer): String;
    function GetUsedFileContent(i: Integer): String;
    function GetRawLine(i: Integer): String;
    procedure SetRawLine(i: Integer; const Value: String);
  public
    constructor Create;
    destructor destroy; override;
    Function Count: Integer;
    Procedure Clear;
    Procedure Add(Const FileName: String; LineNr: Integer; Const Line: String);
    Procedure Delete(Const i: Integer);
    Property Lines[i: Integer]: String read GetLine; default;
    Property RawLines[i: Integer]: String read GetRawLine Write SetRawLine;
    Property Filename[i: Integer]: String read GetFilename;
    Property LineNr[i: Integer]: Integer read GetLineNr;
    // Filelist
    Function UsedFilesCount: Integer;
    Property UsedFiles[i: Integer]: String read GetUsedFile;
    Property UsedFilesContent[i: Integer]: String read GetUsedFileContent;
  end;

  TFiltersBase = class
    private
    protected
      fFilterFile : TFilterfile;
      fFilterLines: TList;
      fMainfilename: String;
      RegExSection: TPCRE;
      RegExFilter : TPCRE;
      function GetLinesCount: Integer;
      function GetLinesItem( Index: Integer ): TFilterLineBase;
      procedure Load( RekStufe : Byte; AFilterFilename: String ); //HSR //INCLUDE
      procedure Save; //HSR //INCLUDE
    public
      property  LinesCount: Integer read GetLinesCount;
      property  LinesItem[ Index: Integer ]: TFilterLineBase read GetLinesItem;

      function  LinesAdd( Const FileName: String; Const LineNo: Integer;
         Const LineText: String ): Integer; virtual; abstract;
      procedure LinesDelete( Const Index: Integer );
      procedure Clear;

      function  IsFilterLine( Const Line: String ): Boolean; virtual; abstract;
      procedure SelectSections( Const SectionIdentifier: String ); virtual;
      function  SelectedLines: String;

      constructor Create( AFilterFilename: String );
      destructor Destroy; override;
  end;

function MatchSimpleString( TestStr, Pattern: String ): Boolean;
function MatchSimpleNumber( TestStr, Pattern: String ): Boolean;
function MatchSimple( TestStr, Pattern: String ): Boolean;

Procedure MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE; Const Testmode: Boolean;
   Var Result: Boolean; Var Error: String  ); overload;
function MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE ): Boolean; overload;

Function CheckExpire ( Var Filterline, LastSetError: String; Var Expire: TDateTime; Out bError: boolean ): boolean;

implementation

Uses cLogfile, cStdForm;

Const Logsection = 'Filterfiles';

// --------------------------------------------------------------- Tools ------

function MatchSimpleString( TestStr, Pattern: String ): Boolean;
begin
     if Pattern='*' then begin // '*' matcht alles
        Result := True;
     end else begin
        Result := ( Pos( LowerCase(Pattern), LowerCase(TestStr) ) > 0 );
     end;
end;

function MatchSimpleNumber( TestStr, Pattern: String ): Boolean;
var  iBase, iTest, E: Integer;
     v: String;
begin
   Result := False;
   try
      if TestStr='' then begin
         iBase := 0
      end else begin
         Val(TestStr, iBase, E);
         If E <> 0 then exit
      end;
      If Length(Pattern)<2 then exit;
      v := Pattern[1];
      If Pattern[2] IN['<', '>', '='] then begin
         v := v + Pattern[2];
         Val(copy(Pattern,3,9), iTest, E);
      end else begin
         Val(copy(Pattern,2,9), iTest, E);
      end;
      If E <> 0 then Exit;
      // Vergleich selber
           If v = '>'  then Result := iBase>iTest
      else If v = '<'  then Result := iBase<iTest
      else If v = '='  then Result := iBase=iTest
      else If v = '<=' then Result := iBase<=iTest
      else If v = '>=' then Result := iBase>=iTest
      else If v = '<>' then Result := iBase<>iTest;
   except
   end;
end;

function MatchSimple( TestStr, Pattern: String ): Boolean;
begin
     if Pattern[1]='%' then begin
        Result := MatchSimpleNumber( TestStr, copy(Pattern, 2, Length(Pattern)-1) );
     end else begin
        if TestStr='' then begin
           if Pattern='*' then Result:=True else Result:=False;
        end else begin
           Result := MatchSimpleString( TestStr, Pattern );
        end;
     end;
end;

Procedure MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE; Const Testmode: Boolean;
   Var Result: Boolean; Var Error: String  ); 
const ONE=0; YES=1; NO=2;
var  NeedOneOf, HaveOneOf: Boolean;
     Matches : Boolean;
     MPatterns, Pattern: String;
     i: Integer;
     SelectType: Char;
begin
     Error := '';
     Result := True;

     MPatterns := Patterns;
     NeedOneOf := False;
     HaveOneOf := False;
     Matches   := False;

     while MPatterns<>'' do begin
        MPatterns := TrimWhSpace( MPatterns );
        i := PosWhSpace( MPatterns );
        if i=0 then begin
           Pattern   := MPatterns;
           MPatterns := '';
        end else begin
           Pattern  := copy( MPatterns, 1, i-1 );
           System.Delete( MPatterns, 1, i );
        end;

        // Pattern:
        //    ['+'|'-']  '{'  regex-pattern   '}'
        //    ['+'|'-'] ['"'] simple-pattern ['"']

        SelectType := ' ';
        if (Pattern>'') and (Pattern[1] in ['+','-']) then begin
           SelectType := Pattern[1];
           System.Delete( Pattern, 1, 1 );
           if Pattern='' then Error := 'Missing pattern after +/-:' + Patterns
        end;

        if (SelectType<>' ') or not(HaveOneOf) then begin
           if copy(Pattern,1,1)='{' then begin
              if Pattern[length(Pattern)]='}' then System.Delete(Pattern,length(Pattern),1);
              System.Delete( Pattern, 1, 1 );
              try
                 Matches := RegEx.Match( PChar(Pattern), PChar(TestStr) );
              except
                 on E: Exception do begin
                    Error := 'Regex-error in {' + Pattern + '}:' + E.Message;
                    Matches := False;
                 end;
              end;
           end else begin
              if copy(Pattern,1,1)='"' then begin
                 System.Delete( Pattern, 1, 1 );
                 if Pattern[length(Pattern)]='"' then System.Delete(Pattern,length(Pattern),1);
              end;
              Matches := MatchSimpleString( TestStr, Pattern );
           end;

           case SelectType of
              '+': if Not TestMode then
                     If not Matches then begin Result:=False; break; end;
              '-': if Not TestMode then
                     If Matches     then begin Result:=False; break; end;
              ' ': begin
                      NeedOneOf := True;
                      if Matches then HaveOneOf:=True;
                   end;
           end;
        end;
     end;

     if NeedOneOf and not HaveOneOf then Result:=False;
end;

function MatchPatterns( const TestStr, Patterns: String; RegEx: TPCRE ): Boolean;
Var Fehler: String; ok: Boolean;
begin
   MatchPatterns( TestStr, Patterns, RegEx, false, ok, Fehler);
   Result := ok;
   If Fehler > '' then Log( LOGID_ERROR, Fehler );
end;

// -------------------------------------------------------- TFilterLine ------

function TFilterLineBase.GetPatternCount: Integer;
begin
     Result := fFilterPatterns.Count;
end;

function TFilterLineBase.GetPatternItem( Index: Integer ): TFilterPatternBase;
begin
     if (Index>=0) and (Index<PatternCount) then begin
        Result := fFilterPatterns[ Index ];
     end else begin
        Result := nil;
     end;
end;

function TFilterLineBase.PatternAdd( Pat: TFilterPatternBase ): Integer;
begin
     Result := fFilterPatterns.Add( pat );
end;

procedure TFilterLineBase.PatternDelete( Index: Integer );
begin
     if (Index>=0) and (Index<PatternCount) then begin
        TFilterPatternBase( fFilterPatterns[ Index ] ).Free;
        fFilterPatterns.Delete( Index );
     end;
end;


procedure TFilterLineBase.Clear;
begin
     while PatternCount>0 do PatternDelete( PatternCount-1 );
end;

constructor TFilterLineBase.Create;
begin
     inherited Create;
     fFilterPatterns := TList.Create;
     Clear;
end;

destructor TFilterLineBase.Destroy;
begin
     Clear;
     fFilterPatterns.Free;
     inherited Destroy;
end;

// -------------------------------------------------------- TFiltersBase ------

{ TFiltersbase }

function TFiltersBase.GetLinesCount: Integer;
begin
     Result := fFilterLines.Count;
end;

function TFiltersBase.GetLinesItem( Index: Integer ): TFilterLineBase;
begin
     if (Index>=0) and (Index<LinesCount) then begin
        Result := fFilterLines[ Index ];
     end else begin
        Result := nil;
     end;
end;

procedure TFiltersBase.LinesDelete( Const Index: Integer );
begin
     if (Index>=0) and (Index<LinesCount) then begin
        TFilterLineBase( fFilterLines[ Index ] ).Free;
        fFilterLines.Delete( Index );
     end;
end;

procedure TFiltersBase.Clear;
begin
     while LinesCount>0 do LinesDelete( LinesCount-1 );
end;

procedure TFiltersBase.SelectSections( Const SectionIdentifier: String );
Var GroupOK : Boolean;

   procedure DoLine (Const FilterFileName: String; Const LineNo: Integer;
     LineText: String);
   var  j, k: Integer;
   begin
      if IsFilterLine( LineText ) then begin // filter-line
         if GroupOK then LinesAdd( Filterfilename, LineNo, LineText );
         exit;
      end;
      if (LineText>'') and (LineText[1]='[') then begin // group-selection
         System.Delete( LineText, 1, 1 );
         k := 0;
         for j:=1 to length(LineText) do begin
            case LineText[j] of
               '{': inc(k);
               '}': dec(k);
               ']': if (k=0) then begin
                       LineText := copy( LineText, 1, j-1 );
                       break;
                    end;
            end;
         end;
         GroupOK := MatchPatterns( SectionIdentifier, LineText, RegExSection );
         exit;
      end;
      if (LineText > '') and (LineText[1] in ['#',';']) then exit; // comment
      Log( LOGID_WARN, 'Filterfile-line ignored: ' + LineText );
      Log( LOGID_WARN, 'Filterfile: ' + FilterFilename );
      Log( LOGID_WARN, 'Reason: starts with invalid char' );
   end;

var i: Integer; Line: String;
begin
   Clear;
   GroupOK := False;
   With fFilterfile do begin
      For i:=0 to Count-1 do begin
         Line := TrimWhSpace( Lines[i] );
         if Line>'' then DoLine(FileName[i], LineNr[i], Line)
      end
   end
end;


function TFiltersBase.SelectedLines: String;
var  LineNo: Integer;
begin
     Result := '';
     for LineNo:=0 to LinesCount-1 do begin
        Result := Result + LinesItem[LineNo].AsString + #13#10;
     end;
end;

constructor TFiltersBase.Create( AFilterFilename: String );
begin
   inherited Create;

   fFilterFile  := TFilterfile.Create;
   fFilterLines := TList.Create;
   RegExSection := TPCRE.Create( False, PCRE_CASELESS );
   RegExFilter  := TPCRE.Create( False, 0 );

   Clear;
   fMainfilename := AFilterFileName;
   Load(0, AFilterFileName);
end;

destructor TFiltersBase.Destroy;
begin
   Clear;
   if Assigned(fFilterFile ) then fFilterFile.Free;
   if Assigned(fFilterLines) then fFilterLines.Free;
   if Assigned(RegExFilter ) then RegExFilter.Free;
   if Assigned(RegExSection) then RegExSection.Free;
   inherited Destroy;
end;

Function CheckExpire ( Var Filterline, LastSetError: String; Var Expire: TDateTime; Out bError: boolean ): boolean;
Const kExpire = 'Expire:';
Var j: Integer;
begin
   Result := false;
   bError := true;
   If LowerCase(Copy(FilterLine, 1, Length(kExpire)))=LowerCase(kExpire) then begin
      If Length(Filterline) >= Length(kExpire) + 8 then begin
         For j := 1 to 8 do If Not (Filterline[Length(kExpire)+j] IN['0'..'9']) then begin
            LastSetError:='invalid char "'+Filterline[Length(kExpire)+j]+'" in expire-date';
            Exit
         end;
         System.Delete(Filterline, 1, Length(kExpire));
         try
            Expire := EncodeDate (StrToInt(Copy(Filterline, 1, 4)),
                                  StrToInt(Copy(Filterline, 5, 2)),
                                  StrToInt(Copy(Filterline, 7, 2)))
         except
            LastSetError:='invalid expire-date "'+Copy(Filterline, 1, 8)+'"';
            Exit
         end;
         System.Delete(Filterline, 1, 8);
         Result := true
      end else begin
         LastSetError:='invalid expire, date to short - use "Expire:yyyymmdd"';
         Exit
      end
   end;
   bError := false
end;

procedure TFiltersBase.Load(RekStufe: Byte; AFilterFilename: String);
var
  FileName, UseFilename: String;
  tempTSL : tStringList;
  bCombine: Boolean;
  s : string;
  i : integer;
begin
   If RekStufe > 5 then begin
      Log(LOGID_ERROR, fMainfilename+': '+ TrGlF(Logsection, 'possible.include.recursion.detected',
        'You have reached the maximum include-depth, the file "%s" will not be included!',
        AFilterFilename));
      exit
   end;
   tempTSL := tStringList.Create;
   bCombine := false;
   AFilterfilename := AddPathIfNeeded(AFilterfilename, PATH_BASE);
   try
     if AFilterFilename>'' then begin
       if FileExists2( AFilterFilename ) then begin
          try
             With TFileStream.Create( AFilterFilename, fmOpenRead or fmShareDenyNone ) do try
                If Size > 0 then begin
                   SetLength(s, Size);
                   Read( s[1], Length(s) );
                   tempTSL.Text := s
                end
             finally
                Free
             end;
          except
             on E: Exception do begin
                Log( LOGID_WARN, 'Couldn''t load filter-file ' + AFilterFilename );
                Log( LOGID_WARN, 'Load-Error '+E.Classname+': ' + E.Message );
             end;
          end
       end else
       If RekStufe>0 then begin
          Log( LOGID_WARN, 'Couldn''t find included filter-file ' + AFilterFilename );
       end;
       // Parse and recursivly load
       for i := 0 to tempTSL.count-1 do begin
         If RekStufe = 0
            then UseFilename := ''
            else UseFilename := AFilterFileName;
         With FFilterFile do begin
            If bCombine
               then RawLines[Count-1] := RTrim(RawLines[Count-1]) + #13#10 + LTrim(tempTSL[i])
               else Add(UseFilename, i+1, tempTSL[i])
         end;
         s := TrimWHSpace(tempTSL[i]);
         bCombine := false;
         If (s > '') then If s[Length(s)]='_' then bCombine := true;
         If (copy(s, 1, 2)='#!') then begin
            s := TrimWHSpace(copy(s, 3, length(s)));
            if (Uppercase(copy(s, 1, length(Name_INCLUDE1))) = Name_INCLUDE1) then begin
              FileName := TrimWHSpace(copy(s, length(Name_INCLUDE1)+1, length(s)));
              Log(LOGID_WARN, TrGlF(Logsection, 'Filterfile.load_is_obsolete',
                'You should use "#!%s" instead of "#!%s" because '
                +'"#!%s" will not be supported in future versions anymore.',
                [Name_INCLUDE2, Name_INCLUDE1, Name_INCLUDE1]));
              Load(RekStufe+1, Filename);
            end else
            If (Uppercase(copy(s, 1, length(Name_INCLUDE2))) = Name_INCLUDE2) then begin
              FileName := TrimWHSpace(copy(s, length(Name_INCLUDE2)+1, length(s)));
              Load(RekStufe+1, Filename);
            end
         end;
       end
     end
   finally
     tempTSL.free;
   end;
end;

procedure TFiltersBase.Save;
Var s, AktFileName, SaveFilename: String; i: Integer;
begin
   With FFilterFile do begin
      For i := 0 to UsedFilesCount-1 do begin
         AktFileName := UsedFiles[i];
         If AktFileName > ''
            then SaveFilename := AktFilename
            else SaveFilename := fMainfilename;
         s := UsedFilesContent[i];
         With TFileStream.Create(SaveFileName, fmCreate) do try
            If Length(s)>0 then Write(s[1], Length(s))
         finally
            free
         end
      end
   end
end;

{ TFilterfile }

constructor TFilterfile.Create;
begin
  inherited;
  slLines := TStringList.Create;
  slFiles := TStringList.Create;
end;

destructor TFilterfile.destroy;
begin
  slLines.Free;
  slFiles.Free;
  inherited;
end;

procedure TFilterfile.Add(const FileName: String; LineNr: Integer;
  const Line: String);
Var FileNameNr: Integer;
begin
   FileNameNr := slFiles.IndexOf(FileName);
   If FileNameNr < 0 then FileNameNr := slFiles.Add(FileName);
   slFiles.Objects[FileNameNr] := Pointer(Longint(slFiles.Objects[FileNameNr])+1);
   slLines.AddObject ( Line, Pointer(FileNameNr + 100*LineNr) );
end;

function TFilterfile.GetRawLine(i: Integer): String;
begin
   Result := slLines[i]
end;
procedure TFilterfile.SetRawLine(i: Integer; const Value: String);
begin
   slLines[i] := Value
end;
function TFilterfile.GetLineNr(i: Integer): Integer;
begin
   Result := Longint(slLines.Objects[i]) div 100
end;
function TFilterfile.GetFilenameNr(i: Integer): Integer;
begin
   Result := Longint(slLines.Objects[i]) mod 100;
end;

function TFilterfile.GetFilename(i: Integer): String;
begin
   Result := slFiles[GetFilenameNr(i)];
end;

function TFilterfile.Count: Integer;
begin
   Result := slLines.Count
end;

procedure TFilterfile.Clear;
begin
   slLines.Clear;
   slFiles.Clear;
end;

procedure TFilterfile.Delete(const i: Integer);
Var idx: integer;
begin
   idx := GetFileNameNr(i);
   slLines.Delete(i);
   slFiles.Objects[idx] := Pointer(Longint(slFiles.Objects[idx])-1)
end;

function TFilterLineBase.AsString: String;
begin
   Result := IntToStr(fOrgLineNo);
   If fOrgFilename > '' then Result := Result+'('+ExtractFileName(fOrgFilename)+')';
   Result := Result+': ' + fOrgLineText
end;

function TFilterfile.UsedFilesCount: Integer;
begin
   Result := slFiles.Count
end;

function TFilterfile.GetUsedFile(i: Integer): String;
begin
   Result := slFiles[i]
end;

function TFilterfile.GetUsedFileContent(i: Integer): String;
Var j: integer;
begin
   Result := '';
   For j := 0 to Count-1 do begin
      If GetFilenameNr(j)=i then Result := Result + RawLines[j] + #13#10
   end
end;

function TFilterfile.GetLine(i: Integer): String;
Var p: Integer;
begin
   Result := TrimWhSpace(RawLines[i]);
   Repeat
      p := Pos('_'+#13#10, Result);
      If p > 0 then System.Delete(Result, p, 3)
   until p = 0
end;

end.
