{***************************************************************
 *
 * Unit Name: cFiltersNews
 * Purpose  :
 * Author   :
 * History  :
 *
 ****************************************************************}

// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cFiltersNews;

interface

uses Classes, cArticle, cPCRE, uTools, cFiltersBase;

const
  XOVER_Number     =  0;
  XOVER_Subject    =  1;
  XOVER_From       =  2;
  XOVER_Date       =  3;
  XOVER_MessageID  =  4;
  XOVER_References =  5;
  XOVER_Bytes      =  6;
  XOVER_Lines      =  7;
  XOVER_Xref       =  8;
  XOVER_Xpost      =  9;
  XOVER_Age        = 10;
  XOVER_MAX        = 10;
  XOVER_INVALID    = -1;

  XOVERLIST_MARKFULL    = $40;
  XOVERLIST_MARKUNKNOWN = $80;
  XOVERLIST_DEFAULT     : array[0..8] of Integer = (
                             // '08,01,02,03,04,05,06,07,48'
                             8, // count
                             XOVER_Subject, XOVER_From, XOVER_Date,
                             XOVER_MessageID, XOVER_References,
                             XOVER_Bytes, XOVER_Lines,
                             XOVER_Xref or XOVERLIST_MARKFULL
                          );

type
  PXOverRec = ^TXOverRec; // PG
  TXOverRec = record
    XFields: array[XOVER_Number..XOVER_MAX] of String;
  end;
  TFilterPatternNews = class( TFilterPatternBase )
     IsSameField: Boolean;
     XOverField : Integer;
     XFiltField : String;
  end;
  TFilterDataBase = Class
     public
        Function GetValue ( Const XOverField: Integer;
                            Const XFiltField: String;
                            Const DoMimeDecode: boolean ): String; Virtual; abstract; // PG
     end;
  TFilterDataXOver = Class (TFilterDataBase)
     private
        XOverRec: TXOverRec;
     public
        Function GetValue ( Const XOverField: Integer;
                            Const Unused: String;
                            Const DoMimeDecode: boolean ): String; override;
        constructor Create ( Akt: TXOverRec );
     end;
  TFilterDataArticle = Class (TFilterDataBase)
     private
        Article: TArticle;
     public
        Function GetValue ( Const WithoutUsage: Integer;
                            Const XFiltField: String;
                            Const DoMimeDecode: boolean ): String; override;
        constructor Create ( Akt: TArticle );
     end;


  TFilterGetValueFunc = function ( Const BaseData    : TXOverrec;  // PG
                                   const XOverField  : Integer;  // PG
                                   const XFiltField  : String;        // PG
                                   const DoMimeDecode: Boolean ): String;  // PG

  TFilterLineNews = class( TFilterLineBase )
    public
      IsFinal      : Boolean; // '='?
      ScoreValue   : Integer;
      Unless      : boolean;
      IsBeforeLoad: Boolean; // PG
      XOverField  : Integer; // PG
      XFiltField: String; // PG
      // DefaultField : Integer; // PG
      Expire       : TDateTime; // bei PG fehlend!
      function  MatchesData( const RE: TPCRE; FilterData: TFilterDataBase ): Boolean;   // PG
      // function  MatchesXOverRec( RE: TPCRE; XOverRec: TXOverRec ): Boolean; // PG
      function  SetFilterLine( Const AFilename: String; Const LineNo: Integer;
         Const AFilterLine: String ): Boolean; override;
  end;

  TFiltersNews = class( TFiltersBase )
    private
      FHasXOverLines, FHasXFiltLines: Boolean; // PG
    public
      property  HasXOverLines: Boolean read FHasXOverLines; // PG
      property  HasXFiltLines: Boolean read FHasXFiltLines; // PG
      procedure SelectSections( const SectionIdentifier: String ); override; // PG
      function  LinesAdd( Const FileName: String; Const LineNo: Integer;
         const LineText: String ): Integer; override;
      function  IsFilterLine( const Line: String ): Boolean; override;
      function  ScoreBeforeLoad( Const XOverRec: TXOverRec;      // PG
                                 const MatchLog: PString ): Integer;  // PG
      function  ScoreAfterLoad ( Const Article: TArticle; // PG
                                 const MatchLog: PString ): Integer; // PG

      Procedure Test; // nicht in PG
      Procedure Purge; // nicht in PG
      // function  ScoreForXOverRec( XOverRec: TXOverRec ): Integer; // PG
      // function  ScoreForXOverRecLog( XOverRec: TXOverRec; var MatchLog: String ): Integer; // PG
  end;

procedure XOverToXOverRec( var XOverRec: TXOverRec;
                           Const ParsedXOver : TParser;
                           Const XOverList   : array of integer );

procedure ArticleToXOverRec( Const Article: TArticle; var XOverRec: TXOverRec );

implementation

uses SysUtils, Global, uEncoding, uDateTime, cLogfile, cStdForm;

procedure XOverToXOverRec( var XOverRec: TXOverRec;
                           Const ParsedXOver : TParser;
                           Const XOverList   : array of integer );
var  PartNo, PartID: Integer;
     PartS         : String;
     IsFull        : Boolean;
     s             : String;
     i, k          : Integer;
begin
     // ParsedXOver (default):
     //    0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References 6:Bytes 7:Lines [8:Xref]
     // XOverList:
     //    [0]=Anzahl, [1..Anzahl]=Feldnummer [or XOVERLIST_MARKFULL] [or XOVERLIST_MARKUNKNOWN]

     for i:=XOVER_Number to XOVER_MAX do XOverRec.XFields[i]:='';
     XOverRec.XFields[XOVER_Number] := ParsedXOver.sPart( 0, '' );

     for PartNo:=1 to XOverList[0] do begin

        PartID := XOverList[PartNo];
        if (PartID and XOVERLIST_MARKFULL)<>0 then begin
           IsFull := True;
           PartID := PartID and not(XOVERLIST_MARKFULL);
        end else begin
           IsFull := False;
        end;

        if (PartID<0) or (PartID>XOVER_MAX) then PartID:=XOVERLIST_MARKUNKNOWN;

        if (PartID and XOVERLIST_MARKUNKNOWN)=0 then begin

           PartS := ParsedXOver.sPart( PartID, '' );

           if PartS<>'' then begin
              XOverRec.XFields[PartID] := PartS;

              if PartID=XOVER_Xref then begin // Xref->Xpost
                 s := LowerCase( PartS );
                 if IsFull and (copy(s,1,6)='xref: ') then System.Delete(s,1,6);
                 k := 0;
                 for i:=1 to length(s) do if s[i]=':' then inc(k);
                 if k=0 then k:=1;
                 XOverRec.XFields[XOVER_Xpost] := inttostr(k);
              end;

              if PartID=XOVER_Date then begin // Date->Age
                 try
                    k := Trunc( NowGMT - RfcDateTimeToDateTimeGMT( PartS, NowGMT ) );
                    XOverRec.XFields[XOVER_Age] := inttostr(k);
                 except
                    XOverRec.XFields[XOVER_Age] := '0';
                 end;
              end;
           end;

        end;
     end;
end;

function XFieldNameToNumber( Fieldname: String ): Integer;
begin
     if Fieldname[length(Fieldname)]=':' then System.Delete(Fieldname,length(Fieldname),1);
     Fieldname := UpperCase( Fieldname );

     if      Fieldname='NUMBER'     then Result := XOVER_Number
     else if Fieldname='SUBJECT'    then Result := XOVER_Subject
     else if Fieldname='FROM'       then Result := XOVER_From
     else if Fieldname='DATE'       then Result := XOVER_Date
     else if Fieldname='MESSAGE-ID' then Result := XOVER_MessageID
     else if Fieldname='REFERENCES' then Result := XOVER_References
     else if Fieldname='BYTES'      then Result := XOVER_Bytes
     else if Fieldname='LINES'      then Result := XOVER_Lines
     else if Fieldname='XREF'       then Result := XOVER_Xref
     else if Fieldname='XPOST'      then Result := XOVER_Xpost
     else if Fieldname='AGE'        then Result := XOVER_Age
     else                                Result := XOVER_INVALID;
end;

function XFieldNumberToName( Fieldnumber: Integer ): String;
begin
     case Fieldnumber of
        XOVER_Number     : Result:='Number';
        XOVER_Subject    : Result:='Subject';
        XOVER_From       : Result:='From';
        XOVER_Date       : Result:='Date';
        XOVER_MessageID  : Result:='Message-ID';
        XOVER_References : Result:='References';
        XOVER_Bytes      : Result:='Bytes';
        XOVER_Lines      : Result:='Lines';
        XOVER_Xref       : Result:='Xref';
        XOVER_Xpost      : Result:='Xpost';
        XOVER_Age        : Result:='Age';
        else               Result:='Invalid!';
     end;
end;

procedure ArticleToXOverRec( Const Article: TArticle; var XOverRec: TXOverRec );
var  i, k: Integer;
     s   : String;
begin
     for i:=XOVER_Number to XOVER_MAX do XOverRec.XFields[i]:='';

     with XOverRec do begin
        XFields[XOVER_Number]     := '42';
        XFields[XOVER_Subject]    := Article['Subject:'];
        XFields[XOVER_From]       := Article['From:'];
        XFields[XOVER_Date]       := Article['Date:'];
        XFields[XOVER_MessageID]  := Article['Message-ID:'];
        XFields[XOVER_References] := Article['References:'];

        XFields[XOVER_Bytes]      := Article['Bytes:']; // simulation-only
        if XFields[XOVER_Bytes]='' then begin
           XFields[XOVER_Bytes]   := inttostr( Length(Article.Text) );
        end;

        XFields[XOVER_Lines]      := Article['Lines:'];
        XFields[XOVER_Xref]       := Article['Xref:'];

        s := LowerCase( XFields[XOVER_Xref] ); // Xref->Xpost
        while copy(s,1,6)='xref: ' do System.Delete(s,1,6);
        k := 0;
        for i:=1 to length(s) do begin
           if s[i]=':' then inc(k);
        end;
        if k=0 then k:=1;
        XFields[XOVER_Xpost] := inttostr(k);

        try // Date->Age
           k := Trunc( NowGMT - RfcDateTimeToDateTimeGMT( XFields[XOVER_Date], NowGMT ) );
           XFields[XOVER_Age] := inttostr(k);
        except
           XFields[XOVER_Age] := '0';
        end;

     end;
end;

// ---------------------------------------------------------- TScoreLine ------

function TFilterLineNews.MatchesData( const RE: TPCRE; // PG
   FilterData: TFilterDataBase ): Boolean; // PG
const ONE=0; YES=1; NO=2;
var  Pat          : TFilterPatternNews;
     TestStr, DefStr : String;
     NeedOneOf, HaveOneOf: Boolean;
     PatNo: Integer;
     Matches : Boolean;
begin
   Result := True;
   DefStr := FilterData.GetValue( XOverField, XFiltField, DoMimeDecode ); // PG
   NeedOneOf := False;
   HaveOneOf := False;
   Matches   := False;
   for PatNo:=0 to PatternCount-1 do begin
      Pat := TFilterPatternNews( PatternItem[ PatNo ] );
      if Pat.IsSameField
         then TestStr := DefStr
         else TestStr := FilterData.GetValue( Pat.XOverField, Pat.XFiltField, DoMimeDecode );
      if (Pat.SelectType<>' ') or not(HaveOneOf) then begin
         if Pat.IsRegex then begin
            try
               RE.OptCompile := PCRE_CASELESS;
               Matches := RE.Match( PChar(Pat.Pattern), PChar(TestStr) );
            except
               on E: Exception do begin
                  Log( LOGID_ERROR, 'Regex-error in {' + Pat.Pattern + '}:' + E.Message );
                  Matches := False;
               end;
            end;
         end else begin
            Matches := MatchSimple( TestStr, Pat.Pattern );
         end;
         case Pat.SelectType of
            '+': if not Matches then begin Result:=False; break; end;
            '-': if Matches     then begin Result:=False; break; end;
            ' ': begin
                    NeedOneOf := True;
                    if Matches then HaveOneOf:=True;
                 end;
         end;
      end;
   end;
   if NeedOneOf and not HaveOneOf then Result:=False;
   If Unless then Result := Not Result 
end;

function TFilterLineNews.SetFilterLine( Const AFilename: String;
   Const LineNo: Integer; Const AFilterLine: String ): Boolean;
var  i, k: Integer;
     s: String;
     SelectType: Char;
     bError, IsRegex, WasSpecial: Boolean;
     // ScoreField: Integer; // PG
     TempIsSameField: Boolean; // PG
     TempXOverField: Integer; // PG
     TempXFiltField: String; // PG
     Filterline, Pattern: String;
     cEnd: Char;
     pat: TFilterPatternNews;
begin
   Filterline := AFilterline;

   Result := False;
   LastSetError := 'invalid line';

   Clear;
   fOrgLineNo   := LineNo;
   fOrgLineText := AFilterline;
   fOrgFilename := AFilename;
   IsFinal      := False;
   ScoreValue   := 0;
   DoMimeDecode := False;
   IsBeforeLoad := True; // PG
   XOverField   := XOVER_INVALID;
   XFiltField   := '';
   Expire       := 0;
   Unless       := false;

   FilterLine := TrimWhSpace( FilterLine );
   if FilterLine='' then exit;

   // ['?'] ['='] ('+'|'-') scorevalue WHSP ['~'] defaultfield WHSP pattern [WHSP pattern ...]

   // Score-After-Load marker // PG
   if FilterLine[1] = '?' then begin // PG
      IsBeforeLoad := False; // PG
      System.Delete( FilterLine, 1, 1 ); // PG
   end; // PG

   // Final-marker
   if FilterLine[1]='=' then begin
      IsFinal:=True;
      System.Delete( FilterLine, 1, 1 );
   end;

   // +/- Scorevalue
   i := PosWhSpace( FilterLine );
   if i<3 then begin LastSetError:='missing (+|-)scorevalue'; exit; end;
   s := copy( FilterLine, 1, i-1 );
   System.Delete( FilterLine, 1, i );
   FilterLine := TrimWhSpace( FilterLine );
   try
      ScoreValue := strtoint( s )
   except
      LastSetError:='Invalid Score-Value "'+s+'"';
      Exit
   end;

   // unless?
   If Copy(LowerCase(Filterline), 1, 6)='unless' then begin
      Unless := true;
      System.Delete(Filterline, 1, 6);
      FilterLine := TrimWhSpace( FilterLine );
   end;

   // Default-Field with optional MIME-Decode
   i := PosWhSpace( FilterLine );
   if i < 2 then begin LastSetError:='missing: [~]default-field'; exit end;
   s := copy( FilterLine, 1, i-1 );
   if (i=2) and (s[1]='~') then begin
      LastSetError:='missing default-field or invalid space between ~ and field';
      exit
   end;
   System.Delete( FilterLine, 1, i );
   FilterLine := TrimWhSpace( FilterLine );
   if (s>'') and (s[1]='~') then begin
      DoMimeDecode := True;
      System.Delete( s, 1, 1 );
   end;
   if IsBeforeLoad then begin
      XOverField := XFieldNameToNumber( s );
      if XOverField=XOVER_INVALID then begin LastSetError:='invalid default-field "'+s+'"'; exit; end; // PG
   end else begin
      s := LowerCase( s );
      if (s>'') and (s[length(s)]=':') then System.Delete( s, length(s), 1 );
      XFiltField := s;
   end;

   // One or more patterns of the following forms:
   //    ['+'|'-'] ['@' fieldname ':'] '{' regex-pattern '}'
   //    ['+'|'-'] ['@' fieldname ':'] '"' simple-pattern '"'
   //    ['+'|'-'] ['@' fieldname ':'] simple-pattern without WHSP
   //    Expire:yyyymmdd

   while FilterLine > '' do begin

      WasSpecial := CheckExpire ( Filterline, LastSetError, Expire, bError );
      If bError then Exit;

      If Not WasSpecial then begin
         SelectType := ' ';
         if FilterLine>'' then If FilterLine[1] in ['+','-'] then begin
            SelectType := FilterLine[1];
            System.Delete( FilterLine, 1, 1 );
         end;

         TempIsSameField := True; // PG
         TempXOverField  := XOverField; // PG
         TempXFiltField  := XFiltField; // PG
         // ScoreField := DefaultField; // PG

         if FilterLine > '' then If FilterLine[1]='@' then begin
            TempIsSameField := False; // PG
            i := Pos( ':', FilterLine );
            if i < 3 then begin LastSetError:='missing: "fieldname:" on "@"'; exit; end;
            s := copy( FilterLine, 2, i-2 );
            System.Delete( FilterLine, 1, i );
            // TGL: Benutzerfreundlichere Filterregeln...
            While (Filterline > '') and (Filterline[1] IN[' ', ^I]) do System.Delete(Filterline, 1, 1);
            if IsBeforeLoad then begin // PG
               TempXOverField := XFieldNameToNumber( s ); // PG
               if TempXOverField=XOVER_INVALID then begin LastSetError:='invalid field "'+s+'"'; exit; end; // PG
            end else begin // PG
               s := LowerCase( s ); // PG
               if (s>'') and (s[length(s)]=':') then System.Delete( s, length(s), 1 ); // PG
               TempXFiltField := s; // PG
            end; // PG
            // ScoreField := XFieldNameToNumber( s ); // PG
            // if ScoreField=XOVER_INVALID then begin LastSetError:='invalid field "'+s+'"'; exit; end; // PG
         end;

         if FilterLine='' then begin LastSetError:='missing: pattern'; exit; end;
         Pattern := '';

         if Copy(FilterLine,1,1)='{' then begin
            IsRegex := True;
            System.Delete( FilterLine, 1, 1 );
            k := 1;
            while FilterLine>'' do begin
               if FilterLine[1]='{' then inc(k);
               if FilterLine[1]='}' then begin
                  dec(k);
                  if k=0 then break;
               end;
               Pattern := Pattern + FilterLine[1];
               System.Delete( FilterLine, 1, 1 );
            end;
            if copy(FilterLine,1,1)='}' then System.Delete( FilterLine, 1, 1 );
         end else begin
            IsRegex := False;
            if FilterLine[1]='"' then begin
               cEnd := '"';
               System.Delete( FilterLine, 1, 1 );
               i := Pos( cEnd, FilterLine );
            end else begin
               cEnd := #32;
               if (FilterLine[1] = '%') then begin
                  While (Length(FilterLine)>2) and ((FilterLine[3] = #32) or (FilterLine[3] = #9)) do begin
                     System.Delete( FilterLine, 3, 1 )
                  end
               end;
               i := PosWhSpace( FilterLine ); {WJ}
            end;
            // falls EOL:
            If i <= 0 then i := length(FilterLine) + 1;
            Pattern := copy( FilterLine, 1, i-1 );
            System.Delete( FilterLine, 1, i-1 );
            if copy(FilterLine,1,1)=cEnd then System.Delete( FilterLine, 1, 1 );
         end;

         if Pattern='' then begin
            LastSetError:='missing: pattern/-delimiter'; exit;
         end;

         pat := TFilterPatternNews.Create;
         pat.SelectType := SelectType;
         pat.IsRegex    := IsRegex;
         pat.Pattern    := Pattern;
         pat.IsSameField  := TempIsSameField; // PG
         pat.XOverField   := TempXOverField; // PG
         pat.XFiltField   := TempXFiltField; // PG
         PatternAdd( pat )
      end;

      FilterLine := TrimWhSpace( FilterLine );
      if FilterLine >'' then begin
         if FilterLine[1] in ['#',';'] then FilterLine:=''; // rest of line is [valid] comment
      end;

   end;

   LastSetError := '';
   Result := True;
end;

(*
function TFilterLineNews.AsString: String;
var  Pat  : TFilterPatternNews;
     PatNo: Integer;
begin
     if PatternCount=0 then begin
        Result := '(line not set)';
        exit;
     end;

     Result := '';

     if not IsBeforeLoad then Result := Result + '?'; // PG
     if IsFinal then Result:=Result+'=';
     if ScoreValue>=0 then Result := Result+'+'
                      else Result := Result+'-';
     Result := Result + inttostr( abs(ScoreValue) );

     Result := Result + ' ';

     if DoMimeDecode then Result:=Result+'~';
     if IsBeforeLoad then Result := Result + XFieldNumberToName( XOverField ) // PG
                     else Result := Result + XFiltField; // PG
     // Result := Result + XFieldNumberToName( DefaultField ); // PG

     for PatNo:=0 to PatternCount-1 do begin
        Pat := TFilterPatternNews( PatternItem[ PatNo ] );

        Result := Result + ' ';
        if Pat.SelectType in ['+','-'] then Result:=Result+Pat.SelectType;

        if not Pat.IsSameField then begin // PG
           if IsBeforeLoad then begin // PG
              Result := Result + '@' + XFieldNumberToName( Pat.XOverField ) + ':'; // PG
           end else begin // PG
              Result := Result + '@' + Pat.XFiltField + ':'; // PG
           end; // PG
        end; // PG
        // if Pat.ScoreField<>DefaultField then begin // PG
        //    Result := Result + '@' + XFieldNumberToName( Pat.ScoreField ) + ':'; // PG
        // end; // PG

        if Pat.IsRegEx then begin
           Result := Result + '{' + Pat.Pattern + '}';
        end else begin
           if Pos(' ',Pat.Pattern)>0 then begin
              Result := Result + '"' + Pat.Pattern + '"';
           end else begin
              Result := Result + Pat.Pattern;
           end;
        end;
     end;
end;
*)

// ---------------------------------------------------------- TScoreFile ------

procedure TFiltersNews.SelectSections( const SectionIdentifier: String ); // PG
begin // PG
   FHasXOverLines := False; // PG
   FHasXFiltLines := False; // PG
   inherited; // PG
end; // PG

function TFiltersNews.LinesAdd( Const FileName: String; Const LineNo: Integer;
   Const LineText: String ): Integer;
var  lin: TFilterLineNews;
begin
     lin := TFilterLineNews.Create;
     if lin.SetFilterLine( FileName, LineNo, LineText ) then begin
        Result := fFilterLines.Add( lin );
        if lin.IsBeforeLoad then FHasXOverLines := true // PG
                            else FHasXFiltLines := True; // PG
     end else begin
        Log( LOGID_WARN, 'Scorefile-line ignored: ' + LineText );
        Log( LOGID_WARN, 'Reason: ' + lin.LastSetError );
        lin.Free;
        Result := -1;
     end;
end;

function TFiltersNews.IsFilterLine( Const Line: String ): Boolean;
begin
   Result := (Line>'') and (Line[1] in ['?', '=','+','-']) // PG
//   Result := (Line>'') and (Line[1] in ['=','+','-']) // PG
end;

function TFiltersNews.ScoreBeforeLoad( Const XOverRec: TXOverRec; // PG
                                       const MatchLog: PString ): Integer; // PG
// function TFiltersNews.ScoreForXOverRec( XOverRec: TXOverRec ): Integer; // PG
var  LineNo: Integer;
     Line  : TFilterLineNews;
     FilterData: TFilterDataXOver;
begin
     Result := 0;
     if Assigned(MatchLog) then MatchLog^ := ''; // PG
     if not FHasXOverLines then exit; // PG

     FilterData := TFilterDataXOver.Create(XOverRec);
     try
        for LineNo:=0 to LinesCount-1 do begin
           Line := TFilterLineNews( LinesItem[ LineNo ] );
           if Line.IsBeforeLoad then begin // PG
              If Line.MatchesData( RegexFilter, FilterData ) then begin // PG
              // if Line.MatchesXOverRec( RegexFilter, XOverRec ) then begin // PG
                 if Assigned(MatchLog) then MatchLog^ := MatchLog^ + Line.AsString + #13#10; // PG
                 if Line.IsFinal then begin
                    Result := Line.ScoreValue;
                    break;
                 end;
                 Result := Result + Line.ScoreValue;
              end
           end // PG
        end
     finally
        Filterdata.free
     end;

     if Result<-9999 then Result:=-9999;
     if Result>+9999 then Result:=+9999;
end;

{PG}
function TFiltersNews.ScoreAfterLoad( Const Article: TArticle;
                                      const MatchLog: PString ): Integer;
var  LineNo: Integer;
     Line  : TFilterLineNews;
     FilterData: TFilterDataArticle;
begin
   Result := 0;
   if Assigned(MatchLog) then MatchLog^ := '';
   if not FHasXFiltLines then exit;

   FilterData := TFilterDataArticle.Create (Article);
   try
      for LineNo:=0 to LinesCount-1 do begin
         Line := TFilterLineNews( LinesItem[ LineNo ] );
         if not Line.IsBeforeLoad then begin
            if Line.MatchesData( RegexFilter, FilterData ) then begin
               if Assigned(MatchLog) then MatchLog^ := MatchLog^ + Line.AsString + #13#10;
               if Line.IsFinal then begin
                  Result := Line.ScoreValue;
                  break;
               end;
               Result := Result + Line.ScoreValue;
            end;
         end;
      end;
   finally
      Filterdata.free
   end;

   if Result<-9999 then Result:=-9999;
   if Result>+9999 then Result:=+9999;
end;
{/PG}

{ // PG
function TFiltersNews.ScoreForXOverRecLog( XOverRec: TXOverRec; var MatchLog: String ): Integer;
var  LineNo: Integer;
     Line  : TFilterLineNews;
begin
     Result := 0;
     MatchLog := '';

     for LineNo:=0 to LinesCount-1 do begin
        Line := TFilterLineNews( LinesItem[ LineNo ] );
        if Line.MatchesXOverRec( RegexFilter, XOverRec ) then begin
           MatchLog := MatchLog + Line.AsString + #13#10;
           if Line.IsFinal then begin
              Result := Line.ScoreValue;
              break;
           end;
           Result := Result + Line.ScoreValue;
        end;
     end;

     if Result<-9999 then Result:=-9999;
     if Result>+9999 then Result:=+9999;
end;
}

procedure TFiltersNews.Test;
Var GroupOK : Boolean; Counter: Integer;

     procedure DoLine (Const Filterfilename: String; Const LineNo: Integer; Linetext: String);
     var  j, k: Integer; s: String; ok: Boolean; Error: String;
     begin
          if IsFilterLine( LineText ) then begin // filter-line
             if Not GroupOK then begin
                Log( LOGID_WARN, TrGlF(kGlobal, 'TestFilterFile.Ignored.Info',
                   'Filterfile "%s", line %s will be ignored: %s', [FilterFileName, IntToStr(LineNo), LineText]));
                s := TrGl(kGlobal, 'TestFilterFile.Ignored.Reason.NoGroup', 'No legal Grouppattern was defined above');
                Log( LOGID_WARN, TrGlF(kGlobal, 'TestFilterFile.Ignored.Reason', 'Reason: %s', s))
             end else begin
                With TFilterLineNews.Create do try
                   If Not SetFilterLine( Filterfilename, LineNo, LineText ) then begin
                      Log( LOGID_WARN, TrGlF(kGlobal, 'TestFilterFile.HasError.Info',
                         'Filterfile "%s", line %s will be ignored: %s', [FilterFileName, IntToStr(LineNo), LineText]));
                      Log( LOGID_WARN, TrGlF(kGlobal, 'TestFilterFile.HasError.Reason', 'Reason: %s', LastError) )
                   end
                finally Free end
             end;
             Exit
          end;

          if (LineText>'') and (LineText[1]='[') then begin // group-selection
             System.Delete( LineText, 1, 1 );
             k := 0; ok := false;
             for j:=1 to length(LineText) do begin
                case LineText[j] of
                   '{': inc(k);
                   '}': dec(k);
                   ']': if (k=0) then begin
                           LineText := copy( LineText, 1, j-1 );
                           ok := true;
                           break;
                        end;
                end;
             end;
             If Not ok then begin
                Log( LOGID_WARN, TrGlF(kGlobal, 'TestFilterFile.Groupline.NotOk',
                   'Filterfile "%s", line %s is no legal group-pattern: %s',
                   [FilterFileName, IntToStr(LineNo), LineText]));
                Inc(Counter)
             end else begin
                MatchPatterns( '', LineText, RegExSection, true, ok, Error );
                If Error > '' then begin
                   Log( LOGID_WARN, TrGlF(kGlobal, 'TestFilterFile.Groupline.NotOk',
                      'Filterfile "%s", line %s is no legal group-pattern: %s',
                      [FilterFileName, IntToStr(LineNo), '['+LineText+']']));
                   Log( LOGID_WARN, TrGlF(kGlobal, 'TestFilterFile.Groupline.Reason',
                      'Reason: %s', Error));
                   Inc(Counter)
                end else GroupOK := true
             end;
             exit;
          end;

          if (LineText > '') and (LineText[1] in ['#',';']) then exit; // comment

          Log( LOGID_WARN, 'TFilterfile-line ignored: ' + LineText );
          Log( LOGID_WARN, 'Filterfile: ' + FilterFilename );
          Log( LOGID_WARN, 'Reason: starts with invalid char' );
          Inc(Counter)
     end;

Var i: Integer; Line: String;
begin
   If fFilterFile.Count = 0 then Exit;
   Counter := 0;
   Log( LOGID_SYSTEM, 'Testing news-score-file' );
   try
      GroupOK := False;
      With fFilterFile do begin
         for i:=0 to Count-1 do begin
            Line := TrimWhSpace( Lines[i] );
            If Line > '' then DoLine(Filename[i], LineNr[i], Line)
         end
      end
   except
      On E:Exception do
         Log( LOGID_WARN, 'Internal error when testing Score-File: '+E.Message )
   end;
   If Counter = 0
      then Log( LOGID_INFO, 'News-score-file: Test OK')
      else Log( LOGID_SYSTEM, 'News-score-file: '+IntToStr(Counter)+' errors found');
end;

procedure TFiltersNews.Purge;
Var ok: Boolean; LineNo: Integer; LineText: String;
begin
   ok := false;
   for LineNo:=fFilterFile.Count-1 downto 0 do begin
      LineText := TrimWhSpace( fFilterFile[LineNo] );
      if LineText > '' then If IsFilterLine( LineText ) then begin
         With TFilterLineNews.Create do try
            If SetFilterLine( '', LineNo, LineText ) then begin
               If (Expire > 0) and (Expire <= Now) then begin
                  If Def_News_ExpiredFilterentries_Delete then begin
                     fFilterFile.Delete (LineNo);
                     LogFile.Add ( LOGID_INFO, TrGlF(kLog, 'Scorefile.ExpiredEntry.Deleted',
                        'Expired Scorefile-Entry deleted: "%s"', LineText))
                  end else begin
                     fFilterFile.RawLines[LineNo] := '# Expired! # '+fFilterFile.RawLines[LineNo];
                     LogFile.Add ( LOGID_INFO, TrGlF(kLog, 'Scorefile.ExpiredEntry.commented',
                        'Expired Scorefile-Entry commented out: "%s"', LineText))
                  end;
                  ok := true
               end
            end
         finally Free end
      end
   end;
   If ok then Save //HSR //INCLUDE
   //   If ok then fFilterFile.SaveToFile ( fFilterFilename );
end;

{ TFilterDataXOver }

constructor TFilterDataXOver.Create(Akt: TXOverRec);
begin
   XOverRec := Akt
end;

function TFilterDataXOver.GetValue(Const XOverField: Integer;
  Const Unused: String; const DoMimeDecode: boolean): String;
begin
   Result := XOverRec.XFields[ XOverField ];
   if (Result>'') and DoMimeDecode then Result := DecodeHeadervalue( Result );
end;

{ TFilterDataArticle }

constructor TFilterDataArticle.Create(Akt: TArticle);
begin
   Article := Akt
end;

function TFilterDataArticle.GetValue(const WithoutUsage: Integer;
  const XFiltField: String; const DoMimeDecode: boolean): String;
var
     s: String;
     i, k: Integer;
begin
   If XFiltField = 'bytes' then begin
      Result := inttostr( length( Article.Text ) );

   end else if XFiltField = 'age' then begin
      s := Article['date'];
      i := Trunc( NowGMT - RfcDateTimeToDateTimeGMT( s, NowGMT ) );
      Result := inttostr( i );

   end else if XFiltField = 'xpost' then begin
      s := LowerCase(Article['xref']);
      while copy(s,1,6)='xref: ' do System.Delete(s,1,6);
      k := 0;
      for i:=1 to length(s) do begin
         if s[i]=':' then inc(k);
      end;
      if k=0 then k:=1;
      Result := inttostr(k);

   end else if XFiltField = 'number' then begin
      s := LowerCase( Article['xref']);
      while copy(s,1,6)='xref: ' do System.Delete(s,1,6);
      i := pos( ':', s );
      if i > 0 then System.Delete( s, 1, i );
      i := PosWhSpace( s );
      if i > 0 then SetLength( s, i-1 );
      Result := inttostr( strtointdef( s, 0 ) );

   end else if XFiltField = 'header' then begin
      Result := Article.FullHeader;

   end else if XFiltField = 'body' then begin
      Result := Article.FullBody;

   end else if XFiltField = 'article' then begin
      Result := Article.Text;

   end else begin
      Result := Article[XFiltField];
      if (Result>'') and DoMimeDecode then Result := DecodeHeadervalue( Result );
   end;
end;

end.

