// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cHistoryMail;

interface

uses SysUtils, Classes, Windows, uTools, cChunkRc;


const MD5HashLen = 16;
type  MD5Hash = array[1..MD5HashLen] of Char;

type
  TMailHistoryRecords = class( TChunkedRecords )
    private
      procedure UIDL2Hash( UIDL: String; out Hash: MD5Hash ); 
    public
      function  ContainsUIDL( UIDL: String ) : Boolean;
      function  AddUIDL( UIDL: String; PurgeBase: LongInt ): Boolean;

      procedure Purge;

      constructor Create( AFilePath: String );
      destructor Destroy; override;
  end;

implementation

uses Global, Config, uMD5, uDateTime, cStdForm, cLogFile;

const
  MAILHISTORY_FILEBASE = 'MHistory';
  MAILHISTORY_FILEEXT  = '.dat';
  MAILHISTKEYLEN_UIDL  = MD5HashLen; // always unique

type
  PMailHistoryEntry = ^TMailHistoryEntry;
  TMailHistoryEntry = record
    HashUIDL : MD5Hash; //  16
    PurgeBase: LongInt; //   4
  end;                  // =28 Bytes

procedure TMailHistoryRecords.UIDL2Hash( UIDL: String; out Hash: MD5Hash );
var  s: String;
begin
     s := MD5ofStr( UIDL );
     System.Move( s[1], Hash[1], MD5HashLen );
end;

function TMailHistoryRecords.ContainsUIDL( UIDL: String ) : Boolean;
var  HE: TMailHistoryEntry;
begin
     FillChar( HE, sizeof(HE), 0 );
     UIDL2Hash( UIDL, HE.HashUIDL );
     Result := ContainsKey( HE, MAILHISTKEYLEN_UIDL );
end;

function TMailHistoryRecords.AddUIDL( UIDL: String; PurgeBase: LongInt ): Boolean;
var  HE: TMailHistoryEntry;
begin
     FillChar( HE, sizeof(HE), 0 );
     UIDL2Hash( UIDL, HE.HashUIDL );

     if ContainsKey( HE, MAILHISTKEYLEN_UIDL ) then begin
        Result := False;
     end else begin
        Result := True;
        if PurgeBase=0 then PurgeBase:=DateTimeToUnixTime(Now);
        HE.PurgeBase := PurgeBase;
        Add( HE );
     end;
end;

procedure TMailHistoryRecords.Purge;
var  Idx : Integer;
     Base: TDateTime;
     Days: LongInt;
     PurgeCount: Integer;
     HE: TMailHistoryEntry;
     Chunk: Byte;
begin
     SaveToFile;
     PurgeCount := 0;

     for Chunk:=0 to CHUNK_MAX do begin
        Enter( Chunk );
        try

           Idx := ChunkCount(Chunk) - 1;
           while Idx>=0 do begin
              RecGet( Chunk, Idx, HE );
              Base := UnixTimeToDateTime( HE.PurgeBase );
              Days := Trunc( Now - Base );

              if (Days>Def_Purge_MHistory_KeepDays) and (Def_Purge_MHistory_KeepDays>0) then begin
                 RemoveKey( HE, MAILHISTKEYLEN_UIDL );
                 inc( PurgeCount );
              end;

              dec( Idx );
           end;


        finally Leave(Chunk) end;
     end;

     SaveToFile;

     Log( LOGID_INFO, TrGlF(kLog, 'Info.PurgeMailHistory.Result',
              'Purge Mail-History (%sd): %s entries purged.',
              [inttostr(Def_Purge_MHistory_KeepDays), inttostr(PurgeCount)]))
end;

constructor TMailHistoryRecords.Create( AFilePath: String );
begin
     if AFilePath='' then AFilePath:='.\';
     AFilePath := ExpandFilename( AFilePath );
     if AFilePath[length(AFilePath)]<>'\' then AFilePath:=AFilePath+'\';

     inherited Create( AFilePath + MAILHISTORY_FILEBASE, MAILHISTORY_FILEEXT,
                       0, sizeof(TMailHistoryEntry), True );

     LoadFromFile;
end;

destructor TMailHistoryRecords.Destroy;
begin
   SaveToFile;
   inherited Destroy;
end;

end.
