// ============================================================================
// Basic script-engine for "Hamster-scripts (*.hsc)"
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cHscEngine_Alt; // Basic script-engine for "Hamster-scripts (*.hsc)"

// ----------------------------------------------------------------------------
// Contains a basic script-engine with accompanying tools and classes. It has
// some interfaces to redirect output and to enhance the script-language with
// application-specific functions.
// These interfaces are used in Hamster's "tScript.pas" to redirect output to
// the log-window and to implement its various Ham*-functions.
// ----------------------------------------------------------------------------

interface

uses SysUtils, Classes, Windows;

const
   HSCERR_NOERROR           = 0;
   HSCERR_STOPPED           = 1;
   HSCERR_ENGINEEXCEPTION   = 3;
   HSCERR_INVALIDEXPRESSION = 4;
   HSCERR_UNSUPPORTED       = 5;
   HSCERR_VARNOTINITIALIZED = 6;
   HSCERR_SYNTAX            = 7;
   HSCERR_LABELNOTFOUND     = 8;
   HSCERR_LOOPSTACK         = 9;
   HSCERR_IFTHENELSE        = 10;
   HSCERR_LOOPNOTFOUND      = 11;
   HSCERR_LIMITEXCEEDED     = 12;
   HSCERR_UNKNOWNVAR        = 13;
   HSCERR_INVALID           = 14;
   HSCERR_INITIALIZEFAILED  = 15;
   HSCERR_USERDEFINED       = 1000;

type
   THscEngine = class;

   THscObject = class
      private
         FEngine   : THscEngine;
         FLastError: Integer;
      public
         property Engine: THscEngine read FEngine;
         function SelfError: Boolean;
         function AnyError : Boolean;
         procedure Error( AErrNum: Integer; AErrMsg: String );
         constructor Create( AEngine: THscEngine );
   end;

   THscVarTypes = ( hvtEmpty, hvtInteger, hvtString );

   EHscVarError = class(Exception);

   THscVariant = class
      private
         FValTyp: THscVarTypes;
         FValInt: Integer;
         FValStr: String;

         function  GetAsVar: Variant;
         procedure SetAsVar( const NewValue: Variant );
         function  GetAsInt: Integer;
         procedure SetAsInt( const NewValue: Integer );
         function  GetAsStr: String;
         procedure SetAsStr( const NewValue: String );
         function  GetAsPtr: Integer;

      public
         property TypOf: THscVarTypes read FValTyp;
         property AsVar: Variant read GetAsVar write SetAsVar;
         property AsInt: Integer read GetAsInt write SetAsInt;
         property AsStr: String  read GetAsStr write SetAsStr;
         property AsPtr: Integer read GetAsPtr;

         procedure Assign( const HV: THscVariant );
         function  Unassigned: Boolean;
         procedure Unassign;
         procedure ToInt;
         procedure ToStr;

         constructor Create;                              overload;
         constructor Create( const AValue: Integer     ); overload;
         constructor Create( const AValue: String      ); overload;
         constructor Create( const AValue: THscVariant ); overload;
   end;

   THscScalar = class( THscObject )
      private
         FValue: Variant;
      public
         property Value: Variant read FValue write FValue;
         constructor Create( AEngine: THscEngine; AValue: Variant );
   end;

   THscScalars = class( THscObject )
      private
         FList: TList;
         function  GetValue( Index: Integer ): Variant;
         procedure SetValue( Index: Integer; NewValue: Variant );
      public
         property Value[ Index: Integer ]: Variant read GetValue write SetValue;
         constructor Create( AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscVariable = class( THscObject )
      private
         FName : String;
         FValue: Variant;
      public
         property Name : String  read FName;
         property Value: Variant read FValue write FValue;
         function Dump : String;
         constructor Create( AEngine: THscEngine; AName: String; AValue: Variant );
   end;

   THscVariables = class( THscObject )
      private
         FVarList: TStringList;
         function  GetCount: Integer;
         function  GetValue( Name: String ): Variant;
         procedure SetValue( Name: String; NewValue: Variant );
      public
         property Count: Integer read GetCount;
         property Value[ Name : String ]: Variant read GetValue write SetValue;
         procedure DefValue( Name: String; InitValue: Variant );
         function  Dump( Index: Integer ): String;
         procedure ClearContext( Context: String );
         procedure Clear;
         constructor Create( AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscLists = class( THscObject )
      private
         FLists: TList;
         FParamString: String;          // JAWO Params 4.10.2000 (new variables and functions)
         FParam: Integer;               // JAWO Params 4.10.2000 (new variables and functions)
         function ParamCount: Integer;  // JAWO Params 4.10.2000 (new variables and functions)
         procedure InitParams (ParamString : String); // JAWO Params 4.10.2000 (new variables and functions)
         function GetCount: Integer;
         function GetList( Index: Integer ): TStringList;
         function GetListItem( ListIndex, ItemIndex: Integer ): String;
         procedure SetListItem( ListIndex, ItemIndex: Integer; NewValue: String );
         function GetListTag( ListIndex, ItemIndex: Integer ): Integer;
         procedure SetListTag( ListIndex, ItemIndex: Integer; NewValue: Integer );
      public
         property Count: Integer read GetCount;
         property List[ Index: Integer ]: TStringList read GetList;
         property ListItem[ ListIndex, ItemIndex: Integer ]: String read GetListItem write SetListItem;
         property ListTag[ ListIndex, ItemIndex: Integer ]: Integer read GetListTag write SetListTag;
         function ListExists( Index: Integer ): Boolean;
         function ListAlloc( Sorted, Duplicates: Boolean ): Integer;
         function ListFree( Index: Integer ): Integer;
         function ListClear( Index: Integer ): Integer;
         procedure Clear;
         constructor Create( AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscExpression = class( THscObject )
      private
         FParts    : TStringList;
         FScalars  : THscScalars;
         FVariables: THscVariables;
         procedure Parse( pExpression: PChar );
         procedure RemovePart( Index: Integer );
         function  SolveVar( Index: Integer ): Variant;
         function  SolveFun( Index: Integer ): Variant;
         function  GetAsScalar( Index: Integer ): Variant;
         procedure Solve_UnaryoperatorScalar;
         procedure Solve_ScalarOperatorScalar;
         procedure Solve_ParenScalarParen;
      public
         function Solve( Expression: String ): Variant;
         constructor Create( AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscSchedEntry = class( THscObject )
      private
         FLastTriggered: Integer;
         FKeyword      : String;
         FFromTime     : String;
         FTilTime      : String;
         FAtDays       : String;
         FEveryMins    : Integer;
         FImmediate    : Boolean;
      public
         property Keyword: String  read FKeyword;
         function Trigger: Boolean;
         constructor Create( AEngine: THscEngine;
                             AKeyword: String;
                             AFromTime, ATilTime, AAtDays: String;
                             AEveryMins: Integer;
                             AImmediate: Boolean );
   end;

   THscScheduler = class( THscObject )
      private
         FList: TList;
      public
         procedure Clear;
         function  Add        ( Keyword: String;
                                FromTime, TilTime, AtDays: String;
                                EveryMins: Integer;
                                Immediate: Boolean ): Integer;
         function  Check: String;
         constructor Create( AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscScriptLine = class( THscObject ) // JAWO Quot.Marks 12.10.2000 (new object)
      private
         FNameOK: Boolean;
         FName  : String;
         FParms : TStringList;
      public
         property FuncName  : String read FName write FName;
         property FuncNameOK: Boolean read FNameOK write FNameOK;
         property FuncPars  : TStringList read FParms;
         function  FuncIs( TestFunc: String; ParMin, ParMax: Integer ): Boolean;
         function  ParX( Index: Integer; DefVal: String ): String;
         function  ParV( Index: Integer; DefVal: Variant ): Variant;
         function  ParI( Index: Integer; DefVal: Integer ): Integer;
         function  ParS( Index: Integer; DefVal: String ): String;
         procedure WantSplitFunction (var pcLine: PChar);
         function  WantExpression( var pch: PChar ): String;
         function  WantNumber( var pch: PChar ): String;
         function  WantOperator( var pch: PChar; var OpClass: Integer ): String;
         function  WantFunction( var pch: PChar ): String;
         function  WantVariable( var pch: PChar ): String;
         function  WantIdentifier( var pch: PChar ): String;
         function  WantParens( var pch: PChar ): String;
         function  WantString( var pch: PChar; Const KillDoubleQuotes: boolean ): String;
         constructor Create( AEngine: THscEngine);
         destructor Destroy; override;
   end;

   THscScript = class( THscObject )
      private
         FLines   : TStringList;
         FModRefs : TList;
         FModules : TStringList;
         FCurrLine: Integer;
         FSubRefs : TStringList;  // JAWO 28.11.2000 (speed acceleration)
         FSubDupes: TStringList;  // JAWO 28.11.2000 (subroutine declaration dupe check)
         function GetCount: Integer;
         function GetText : String;
         function GetLine   ( Index: Integer ): String;
         function GetLineNo ( Index: Integer ): Integer;
         function GetModID  ( Index: Integer ): Integer;
         function GetModName( Index: Integer ): String;
      public
         procedure AddSub( SubLine:   String; SubLineNo: Integer;
                           SubModule: string; SubModLine: Integer );  // JAWO 28.11.2000 (speed acceleration)

         property Count: Integer read GetCount;
         property Text : String read GetText;
         property Lines  [ Index: Integer ]: String  read GetLine; default;
         property LineNo [ Index: Integer ]: Integer read GetLineNo;
         property ModID  [ Index: Integer ]: Integer read GetModID;
         property ModName[ Index: Integer ]: String  read GetModName;

         function IsModuleLoaded( ModName: String ): Boolean;
         function StartNewModule( ModName: String ): Integer;
         function Add( s: String ): Integer;
         procedure AddLines( sl: TStringList );
         procedure Clear;

         constructor Create( AEngine: THscEngine );
         destructor Destroy; override;
   end;

   THscErrorEvent = procedure ( Engine  : THscEngine;
                                Sender  : TObject;
                                ErrNum  : Integer;
                                ErrMsg  : String;
                                CurrMod : String;
                                CurrPos : Integer;
                                CurrLine: String ) of object;

   THscTraceEvent = procedure ( Engine  : THscEngine;
                                CurrMod : String;
                                CurrPos : Integer;
                                CurrLine: String ) of object;

   THscPrintEvent = procedure ( Engine  : THscEngine;
                                PrintStr: String ) of object;

   THscAddLogEvent = procedure ( Engine  : THscEngine;
                                PrintStr: String;
                                Typ,
                                ShowScriptName: Integer ) of object;

   THscFuncEvent  = function  ( ScriptLine : THscScriptLine ): Variant of object;

   THscSyncExecEvent = procedure ( Engine: THscEngine;
                                   Method: TThreadMethod ) of object;


   THscHandleTyp = (kHscEngine_GroupHandle, kHscEngine_ListHandle);

   TLoopStackItem = Class
     private
        Line, NextLine: Integer;
        VarName: String;
        IncVal, EndVal: Integer;
   end;
   TLoopStack = Class
     private
       FList: TList;
     public
       constructor Create;
       destructor destroy; override;
       Procedure push (Const MyLine: Integer;
                       Const myNextLine: Integer = 0;
                       Const MyVarname: String = '';
                       Const myIncVal: Integer = 0;
                       Const myEndVal: Integer = 0 );
       Procedure Clear;
       Procedure pop;
       Function Count: Integer;
       Function Item: TLoopStackItem;
   end;

   THscEngine = class( THscObject )
      private
         FScript     : THscScript;
         FVariables  : THscVariables;
         FScheduler  : THscScheduler;
         FHscLists   : THscLists;
         FContexts   : TStringList;
         FContextID  : Integer;
         FLoopStack  : TLoopStack;
         FIPtrStack  : TList;
         FCurrLine   : String;
         FCurrPos    : Integer;
         FNextPos    : Integer;
         FOnError    : THscErrorEvent;
         FOnTrace    : THscTraceEvent;
         FOnPrint    : THscPrintEvent;
         FOnWarning  : THscPrintEvent;
         FOnAddLog   : THscAddLogEvent;
         FOnFunc     : THscFuncEvent;
         FOnSyncExec : THscSyncExecEvent;
         FStopEvent  : THandle;
         FTraceIsOn  : Boolean;
         FDebugLevel : Integer;
         FTerminated : Boolean;
         FSyncCmd    : String;
         FSyncPars   : TStringList;
         FSyncResult : Variant;
         FErrNum     : Integer;
         FErrMsg     : String;
         FErrModule  : String;
         FErrLineNo  : Integer;
         FErrLine    : String;
         FErrSender  : String;
         FErrCatch   : Boolean;
         GroupsInUse, ListsInUse: TStringList;
         procedure AddError( Sender: TObject; AErrNum: Integer; AErrMsg: String );
         function  LineCmd( Index: Integer ): String;
         function  GetContext: String;
         procedure IPtrPush;
         procedure IPtrPop;
      public
         property OnError   : THscErrorEvent    read FOnError    write FOnError;
         property OnTrace   : THscTraceEvent    read FOnTrace    write FOnTrace;
         property OnPrint   : THscPrintEvent    read FOnPrint    write FOnPrint;
         property OnWarning : THscPrintEvent    read FOnWarning  write FOnWarning;
         property OnAddLog  : THscAddLogEvent   read FOnAddLog   write FOnAddLog;
         property OnFunc    : THscFuncEvent     read FOnFunc     write FOnFunc;
         property OnSyncExec: THscSyncExecEvent read FOnSyncExec write FOnSyncExec;
         property StopEvent : THandle           read FStopEvent  write FStopEvent;
         property Variables : THscVariables     read FVariables;
         property Context   : String read GetContext;
         procedure EnterContext( NewContext: String );
         procedure LeaveContext;
         procedure Trace( TracePos: Integer; TraceText: String );
         function  SolveFunction( FuncStr: String ): Variant;
         function  SolveExpression( Expression: String ): Variant;
         function  IndexOfLabel( LabelName: String ): Integer;
         function  IndexOfSub( SubName: String ): Integer;
         procedure SyncExecute;
         function  ExecuteNextPos: Variant;
         function  Execute: Variant;

         { Fr Aufrumarbeiten / JW }
         procedure AddHandle(const Typ: THSCHandleTyp; handle:THandle);
         procedure RemoveHandle(const Typ: THscHandleTyp;handle:THandle);
         procedure GarbageCollection;

         function  ExecuteFromList( const ScriptName: String;
                                    const AScript: TStringList ): Variant;

         // JAWO Params 4.10.2000 (new functions)
         procedure InitParams (ParamString : String);
         constructor Create;
         destructor Destroy; override;
   end;

function HscVarToInt( V: Variant ): Integer;
function HscVarToStr( V: Variant ): String;
function HscVarType ( V: Variant ): Integer; // varEmpty, varInteger, varString

function IsVariable( s: String ): Boolean;

implementation

uses Forms, IniFiles, FileCtrl, cPCRE, uTools, uRasDyn, dInput, dPopUpBox, uDateTime,
     Config, Global, tScript, cStdForm, cArtFiles, main, uWinSock, cLogfile;

function ExecuteProcess( CommandLine     : String;
                         WorkingDir      : String;
                         SWShowWindow    : Integer;
                         WaitForEnd      : Boolean;
                         var WaitExitCode: Integer;
                         StopEvent       : Integer ): Integer;
var  StartupInfo : TStartUpInfo;
     ProcessInfo : TProcessInformation;
     WaitObjects : array[0..1] of LongInt;
     WaitObjCount: Integer;
     pWorkingDir : PChar;
     res: DWord;
begin
     Result := 0;
     WaitExitCode := -1;

     if WorkingDir='' then pWorkingDir:=nil
                      else pWorkingDir:=PChar(WorkingDir);

     FillChar( StartUpInfo, sizeof(StartUpInfo), 0 );
     StartupInfo.cb := Sizeof( StartupInfo );
     StartupInfo.dwFlags     := STARTF_USESHOWWINDOW;
     StartupInfo.wShowWindow := SWShowWindow;

     FillChar( ProcessInfo, sizeof(ProcessInfo), 0 );

     if CreateProcess( nil, PChar(CommandLine), nil, nil, False,
                       NORMAL_PRIORITY_CLASS, nil, pWorkingDir,
                       StartUpInfo, ProcessInfo ) then begin

        if WaitForEnd then begin
           WaitObjects[0] := ProcessInfo.hProcess;
           WaitObjects[1] := StopEvent;
           if StopEvent=0 then WaitObjCount:=1 else WaitObjCount:=2;
           res := WaitForMultipleObjects( WaitObjCount, @WaitObjects[0],
                                          False, INFINITE );
           case res of
              WAIT_OBJECT_0  : Result :=  0; // ok
              WAIT_OBJECT_0+1: Result := -1; // stop-event
              WAIT_FAILED    : Result := GetLastError; // error
           end;

           if Result=0 then begin
              GetExitCodeProcess( ProcessInfo.hProcess, res );
              WaitExitCode := res
           end

        end;

        CloseHandle( ProcessInfo.hThread  );
        CloseHandle( ProcessInfo.hProcess );

     end else begin
        Result := GetLastError;
        if Result=0 then Result:=-1;
     end;
end;

// ----------------------------------------------------------- THsc-Tools -----

const
   CSET_DIGITS     = ['0'..'9'];
   CSET_LETTERS    = ['A'..'Z','a'..'z'];
   CSET_HEXDIGITS  = ['0'..'9','A'..'F','a'..'f'];
   CSET_IDENTIFIER = ['a'..'z','A'..'Z','0'..'9','_'];

const
   ETYPE_INT = $01; // Number
   ETYPE_STR = $02; // String
   ETYPE_VAR = $03; // Variable
   ETYPE_FUN = $04; // Function
   ETYPE_OP0 = $10; // "||"
   ETYPE_OP1 = $11; // "&&"
   ETYPE_OP2 = $12; // "|"
   ETYPE_OP3 = $12; // "^"
   ETYPE_OP4 = $13; // "&"
   ETYPE_OP5 = $14; // "==" / "!=" [ / "=" / "!" / "<>"
   ETYPE_OP6 = $15; // "<" / ">" / "<=" / ">="
   ETYPE_OP7 = $16; // "<<" / ">>"
   ETYPE_OP8 = $17; // "+" / "-"
   ETYPE_OP9 = $18; // "*" / "/" / "%"
   ETYPE_UOP = $1F; // "+" / "-" / "!" / "~" (unary)
   ETYPE_GON = $20; // "("
   ETYPE_GOF = $21; // ")"
   ETYPE_ERR = $FF; // error-marker

procedure SkipEmbeddedComment( var pch: PChar );
begin
   if pch^<>'{' then exit;
   inc( pch );
   while pch^<>#0 do begin
      if pch^='}' then begin inc(pch); break; end;
      inc( pch );
   end;
end;

function TrimWSPEC( s: String ): String;
begin
   repeat
      s := TrimWhSpace( s );
      if s='' then break;
      if s[1]='{' then begin
         repeat System.Delete(s,1,1) until (s='') or (s[1]='}');
         if s<>'' then System.Delete(s,1,1);
      end else if s[length(s)]='}' then begin
         repeat System.Delete(s,length(s),1) until (s='') or (s[length(s)]='{');
         if s<>'' then System.Delete(s,length(s),1);
      end else break;
   until False;
   Result := s;
end;

procedure SkipWSPC( var pch: PChar );
begin
   while pch^<>#0 do begin
      case pch^ of
        ' ', #9: inc( pch );
        '#'    : pch := strend(pch);
        '{'    : SkipEmbeddedComment( pch );
        else     break;
      end;
   end;
end;

function HscVarType( V: Variant ): Integer;
begin
   case VarType( V ) of
      varInteger, varByte, varSmallInt, varBoolean:
         Result := varInteger;
      varString, varOleStr:
         Result := varString;
      else
         Result := varEmpty;
   end;
end;

function HscVarToInt( V: Variant ): Integer;
var  s: String;
begin
   case HscVarType( V ) of
      varInteger: Result := V;
      varString : begin
                     s := V;
                     if copy(s,1,2)='0x' then begin s[1]:='$'; System.Delete(s,2,1); end;
                     Result := strtoint( s );
                  end;
      else        Result := 0;
   end;
end;

function HscVarToStr( V: Variant ): String;
begin
   case HscVarType( V ) of
      varInteger: Result := inttostr( V );
      varString : Result := V;
      else        Result := '';
   end;
end;

function HscVarAsType( V: Variant; HscType: Integer ): Variant;
begin
   case HscType of
      varInteger: Result := HscVarToInt( V );
      varString : Result := HscVarToStr( V );
      else        Result := Unassigned;
   end;
end;

function IsSolveable( i: Integer ): Boolean;
begin
   Result := ( i in [ ETYPE_INT, ETYPE_STR, ETYPE_VAR, ETYPE_FUN ] );
end;

function IsOperator( i: Integer ): Boolean;
begin
   Result := ( i in [ ETYPE_OP0..ETYPE_OP9 ] );
end;

function IsIdentifier( s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   if s='' then exit;
   if not( s[1] in CSET_LETTERS ) then exit;
   for i:=2 to length(s) do begin
      if not( s[i] in CSET_IDENTIFIER ) then exit;
   end;
   Result := True;
end;

function IsVariable( s: String ): Boolean;
begin
   Result := False;
   if copy(s,1,1)<>'$' then exit;
   System.Delete( s, 1, 1 );
   Result := IsIdentifier( s );
end;

function HscCalculateUnary( HscObject: THscObject;
                            Op       : String;
                            V1       : Variant;
                            var ETypeResult: Integer ): Variant;
var  ResType: Integer;
begin
   Result      := Unassigned;
   ETypeResult := ETYPE_ERR;

   try
      ResType := HscVarType( V1 );

      case ResType of

         varInteger: begin
            ETypeResult := ETYPE_INT;
            if      OP='+' then Result := V1
            else if OP='-' then Result := -V1
            else if OP='!' then Result := Integer( V1 = 0 )
            else if OP='~' then Result := Integer( not V1 )
            else begin
               HscObject.Error( HSCERR_UNSUPPORTED,
                                'Unsupported integer-uoperator: ' + OP );
               Result      := Unassigned;
               ETypeResult := ETYPE_ERR;
            end;
         end;


         varString: begin
            ETypeResult := ETYPE_STR;
            HscObject.Error( HSCERR_UNSUPPORTED,
                             'Unsupported string-uoperator: ' + OP );
            Result      := Unassigned;
            ETypeResult := ETYPE_ERR;
         end;

         varEmpty: begin
            HscObject.Error( HSCERR_UNSUPPORTED, 'Undefined uoperand: ' + OP );
            Result      := Unassigned;
            ETypeResult := ETYPE_ERR;
         end;

      end;

   except
      on E: Exception do begin
         HscObject.Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
         Result      := Unassigned;
         ETypeResult := ETYPE_ERR;
      end;
   end;
end;

function HscCalculate( HscObject: THscObject;
                       V1       : Variant;
                       Op       : String;
                       V2       : Variant;
                       var ETypeResult: Integer ): Variant;
var  ResType: Integer;
     T1, T2 : Integer;
begin
   Result      := Unassigned;
   ETypeResult := ETYPE_ERR;

   try
      T1 := HscVarType( V1 );
      T2 := HscVarType( V2 );
      if T1<>varEmpty then ResType:=T1 else ResType:=T2;
      if (T1=varString) or (T2=varString) then ResType:=varString;
      if T1<>ResType then V1:=HscVarAsType( V1, ResType );
      if T2<>ResType then V2:=HscVarAsType( V2, ResType );

      case ResType of

         varInteger: begin
            ETypeResult := ETYPE_INT;
            if      OP='+'  then Result := V1 + V2
            else if OP='-'  then Result := V1 - V2
            else if OP='*'  then Result := V1 * V2
            else if OP='/'  then Result := V1 div V2
            else if OP='%'  then Result := V1 mod V2
            else if OP='<<' then Result := V1 shl V2
            else if OP='>>' then Result := V1 shr V2
            else if OP='^'  then Result := V1 xor V2
            else if OP='&'  then Result := V1 and V2
            else if OP='|'  then Result := V1 or  V2
            else if OP='||' then Result := Integer( (V1<>0) or  (V2<>0) )
            else if OP='&&' then Result := Integer( (V1<>0) and (V2<>0) )
            else if OP='==' then Result := Integer( V1 =  V2 )
            else if OP='!=' then Result := Integer( V1 <> V2 )
            else if OP='<=' then Result := Integer( V1 <= V2 )
            else if OP='>=' then Result := Integer( V1 >= V2 )
            else if OP='<'  then Result := Integer( V1 <  V2 )
            else if OP='>'  then Result := Integer( V1 >  V2 )
            else begin
               HscObject.Error( HSCERR_UNSUPPORTED,
                                'Unsupported integer-operator: ' + OP );
               Result      := Unassigned;
               ETypeResult := ETYPE_ERR;
            end;
         end;


         varString: begin
            ETypeResult := ETYPE_STR;
            if OP='+'  then Result := V1 + V2
            else begin
               ETypeResult := ETYPE_INT;
               if      OP='==' then Result := Integer( V1 =  V2 )
               else if OP='!=' then Result := Integer( V1 <> V2 )
               else if OP='<=' then Result := Integer( V1 <= V2 )
               else if OP='>=' then Result := Integer( V1 >= V2 )
               else if OP='<'  then Result := Integer( V1 <  V2 )
               else if OP='>'  then Result := Integer( V1 >  V2 )
               else begin
                  HscObject.Error( HSCERR_UNSUPPORTED,
                                   'Unsupported string-operator: ' + OP );
                  Result      := Unassigned;
                  ETypeResult := ETYPE_ERR;
               end;
            end;
         end;

         varEmpty: begin
            HscObject.Error( HSCERR_UNSUPPORTED, 'Undefined operands: ' + OP );
            Result      := Unassigned;
            ETypeResult := ETYPE_ERR;
         end;

      end;

   except
      on E: Exception do begin
         HscObject.Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
         Result      := Unassigned;
         ETypeResult := ETYPE_ERR;
      end;
   end;
end;

// ----------------------------------------------------------- THscObject -----

function THscObject.SelfError : Boolean;
begin
   Result := ( FLastError <> 0 );
end;

function THscObject.AnyError: Boolean;
begin
   if Assigned( FEngine ) then begin
      Result := FEngine.SelfError;
   end else begin
      Result := SelfError;
   end;
end;

procedure THscObject.Error( AErrNum: Integer; AErrMsg: String );
begin
   try
      if Assigned( FEngine ) then FEngine.AddError( Self, AErrNum, AErrMsg );
      if FLastError = HSCERR_NOERROR then FLastError := AErrNum;
   except
   end;
end;

constructor THscObject.Create( AEngine: THscEngine );
begin
   inherited Create;
   FEngine := AEngine;
   FLastError := HSCERR_NOERROR;
end;

// ----------------------------------------------------------- THscScalar -----

constructor THscScalar.Create( AEngine: THscEngine; AValue: Variant );
begin
   inherited Create( AEngine );
   FValue := AValue;
end;

// ---------------------------------------------------------- THscScalars -----

function THscScalars.GetValue( Index: Integer ): Variant;
begin
   if Index>=FList.Count then Result := Unassigned
                         else Result := THscScalar( FList[Index] ).Value;
end;

procedure THscScalars.SetValue( Index: Integer; NewValue: Variant );
begin
   while Index>=FList.Count do FList.Add( THscScalar.Create( Engine, Unassigned ) );
   THscScalar( FList[Index] ).Value := NewValue;
end;

constructor THscScalars.Create( AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FList := TList.Create;
end;

destructor THscScalars.Destroy;
begin
   if Assigned( FList ) then begin
      while FList.Count>0 do begin
         THscScalar( FList[FList.Count-1] ).Free;
         FList.Delete( FList.Count-1 );
      end;
      FList.Free;
   end;
   inherited Destroy;
end;

// --------------------------------------------------------- THscVariable -----

function THscVariable.Dump: String;
begin
   Result := FName + '=';
   case HscVarType( FValue ) of
      varInteger: Result := Result + inttostr( FValue );
      varString : Result := Result + '"' + FValue + '"';
      else        Result := Result + '(unassigned)';
   end;
end;

constructor THscVariable.Create( AEngine: THscEngine; AName: String; AValue: Variant );
begin
   inherited Create( AEngine );
   FName    := AName;
   FValue   := AValue;
end;

// -------------------------------------------------------- THscVariables -----

function THscVariables.GetCount: Integer;
begin
   Result := FVarList.Count;
end;

function THscVariables.Dump( Index: Integer ): String;
begin
   Result := THscVariable( FVarList.Objects[ Index ] ).Dump;
end;

function THscVariables.GetValue( Name: String ): Variant;
var  i  : Integer;
     Ctx: String;
begin
   if Assigned(Engine) then Ctx:=Engine.Context else Ctx:='';
   Name := LowerCase( Name );

   i := FVarList.IndexOf( Ctx + '->' + Name ); // local
   if (i<0) and (Ctx<>'') then i := FVarList.IndexOf( '->' + Name ); // global

   if i<0 then begin
      Result := Unassigned;
      if Assigned(Engine) then Engine.Error( HSCERR_UNKNOWNVAR, 'Unknown variable: ' + Name );
   end else begin
      Result := THscVariable( FVarList.Objects[i] ).Value;
   end;
end;

procedure THscVariables.SetValue( Name: String; NewValue: Variant );
var  i  : Integer;
     Ctx: String;
begin
   if Assigned(Engine) then Ctx:=Engine.Context else Ctx:='';
   Name := LowerCase( Name );

   i := FVarList.IndexOf( Ctx + '->' + Name ); // local
   if (i<0) and (Ctx<>'') then i := FVarList.IndexOf( '->' + Name ); // global

   if i<0 then begin
      if Assigned(Engine) then Engine.Error( HSCERR_UNKNOWNVAR, 'Unknown variable: ' + Name );
   end else begin
      THscVariable( FVarList.Objects[i] ).Value := NewValue;
   end;
end;

procedure THscVariables.DefValue( Name: String; InitValue: Variant );
var  i  : Integer;
     Ctx: String;
begin
   if Assigned(Engine) then Ctx:=Engine.Context else Ctx:='';
   Name := LowerCase( Name );

   i := FVarList.IndexOf( Ctx + '->' + Name ); // local

   if i<0 then begin
      FVarList.AddObject( Ctx + '->' + Name,
                          THscVariable.Create( Engine, Ctx+'->'+Name, InitValue ) );
   end else begin
      THscVariable( FVarList.Objects[i] ).Value := InitValue;
   end;
end;

procedure THscVariables.ClearContext( Context: String );
var  i: Integer;
     s: String;
begin
   i := 0;
   s := Context + '->';
   while i<FVarList.Count do begin
      if Pos( s, FVarList[i] )=1 then begin
         THscVariable( FVarList.Objects[i] ).Free;
         FVarList.Delete( i );
      end else begin
         inc( i );
      end;
   end;
end;

procedure THscVariables.Clear;
begin
   while FVarList.Count>0 do begin
      THscVariable( FVarList.Objects[FVarList.Count-1] ).Free;
      FVarList.Delete( FVarList.Count-1 );
   end;
end;

constructor THscVariables.Create( AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FVarList := TStringList.Create;
   FVarList.Sorted := True;
   FVarList.Duplicates := dupIgnore;
end;

destructor THscVariables.Destroy;
begin
   if Assigned( FVarList ) then begin
      Clear;
      FVarList.Free;
   end;
   inherited Destroy;
end;

// ------------------------------------------------------------ THscLists -----

function THscLists.GetCount: Integer;
begin
   Result := FLists.Count;
end;

function THscLists.GetList( Index: Integer ): TStringList;
begin
   if ListExists( Index ) then begin
      Result := FLists[Index];
   end else begin
      Engine.Error( HSCERR_INVALID, 'Invalid list-handle: ' + inttostr(Index) );
      Result := nil;
   end;
end;

function THscLists.ListExists( Index: Integer ): Boolean;
begin
   Result := False;
   if (Index>=0) and (Index<FLists.Count) then begin
      if Assigned( FLists[Index] ) then Result := True;
   end;
end;

function THscLists.ListAlloc( Sorted, Duplicates: Boolean ): Integer;
var  i: Integer;
begin
   Result := -1;

   for i:=0 to FLists.Count-1 do begin
      if not Assigned( FLists[i] ) then begin
         FLists[i] := TStringList.Create;
         Engine.AddHandle(kHscEngine_ListHandle,i);
         Result := i;
         break;
      end;
   end;

   if Result<0 then begin
      if FLists.Count>99 then begin
         Engine.Error( HSCERR_LIMITEXCEEDED, 'Too many lists (max. 99)!' );
         Result := -1;
      end else begin
         i := FLists.Add( TStringList.Create );
         Engine.AddHandle(kHscEngine_ListHandle,i);
         Result := i         
      end;
   end;

   if (Result>=0) and Sorted then begin
      List[Result].Sorted := True;
      if Duplicates then List[Result].Duplicates:=dupAccept
                    else List[Result].Duplicates:=dupIgnore;
   end;
end;

function THscLists.ListFree( Index: Integer ): Integer;
begin
   Result := -1;
   if ListExists(Index) then begin
      List[Index].Free;
      FLists[Index] := nil;
      Engine.RemoveHandle(kHscEngine_ListHandle,Index);
      Result := 0;
   end;
end;

function THscLists.ListClear( Index: Integer ): Integer;
begin
   Result := -1;
   if ListExists(Index) then begin List[Index].Clear; Result:=0; end;
end;

function THscLists.GetListItem( ListIndex, ItemIndex: Integer ): String;
begin
   Result := '';
   if ListExists(ListIndex) then begin
      if (ItemIndex>=0) and (ItemIndex<List[ListIndex].Count) then begin
         Result := List[ListIndex][ItemIndex];
      end;
   end;
end;

procedure THscLists.SetListItem( ListIndex, ItemIndex: Integer; NewValue: String );
begin
   if ListExists(ListIndex) then begin
      try
         while ItemIndex>=List[ListIndex].Count do List[ListIndex].Add( '' );
         List[ListIndex][ItemIndex] := NewValue;
      except
         On E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
      end;
   end;
end;

function THscLists.GetListTag( ListIndex, ItemIndex: Integer ): Integer;
begin
   Result := 0;
   if ListExists(ListIndex) then begin
      if (ItemIndex>=0) and (ItemIndex<List[ListIndex].Count) then begin
         Result := Integer( List[ListIndex].Objects[ItemIndex] );
      end;
   end;
end;

procedure THscLists.SetListTag( ListIndex, ItemIndex: Integer; NewValue: Integer );
begin
   if ListExists(ListIndex) then begin
      try
         while ItemIndex>=List[ListIndex].Count do List[ListIndex].Add( '' );
         List[ListIndex].Objects[ItemIndex] := Pointer( NewValue );
      except
         On E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
      end;
   end;
end;

procedure THscLists.Clear;
var  i: Integer;
begin
   for i:=0 to FLists.Count-1 do ListFree( i );
   FLists.Clear;
end;

// JAWO Params 4.10.2000 (new function)
function THscLists.ParamCount: Integer;
begin
   if ListExists( FParam ) then
      Result := List[FParam].Count - 1
   else
      Result := 0;  // no command line parameters ==> no list
end;


// JAWO Params 4.10.2000 (new function)
procedure THscLists.InitParams (ParamString : String);
begin

   if (FParam < 0) then
      FParam := ListAlloc( False, True );
   if (FParam >= 0) then begin
      List[FParam].Clear;
      List[FParam].Text := ParamString;
   end;
end;

constructor THscLists.Create( AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FLists := TList.Create;
   FParamString := '';            // JAWO Params 4.10.2000 (new code)
   FParam := -1;
end;

destructor THscLists.Destroy;
begin
   if Assigned( FLists ) then begin
      Clear;
      FLists.Free;
   end;
   inherited Destroy;
end;

// ------------------------------------------------------- THscExpression -----

procedure THscExpression.Parse( pExpression: PChar );
var  pCurr : PChar;
     s     : String;
     i     : Integer;
     WantOp: Boolean;
begin
   with THscScriptLine.Create(Engine) do try
      FParts.Clear;
      pCurr  := PExpression;
      WantOp := False;

      while pCurr^ <> #0 do begin
         if SelfError then break;

         if WantOp then begin

            case pCurr^ of
               #9, ' ' : inc( pCurr ); // skip whitespace
               '{'     : SkipEmbeddedComment( pCurr );
               ')'     : begin
                            FParts.AddObject( pCurr^, Pointer(ETYPE_GOF) );
                            inc( pCurr );
                         end;
               else      begin
                            s := WantOperator( pCurr, i );
                            FParts.AddObject( s, Pointer(i) );
                            WantOp := False;
                         end;
            end;

         end else begin

            case pCurr^ of
               #9, ' ' : inc( pCurr ); // skip whitespace
               '{'     : SkipEmbeddedComment( pCurr );
               '('     : begin
                            FParts.AddObject( pCurr^, Pointer(ETYPE_GON) );
                            inc( pCurr );
                         end;
               '+','-',
               '!','~' : begin
                            FParts.AddObject( pCurr^, Pointer(ETYPE_UOP) );
                            inc( pCurr );
                         end;
               '0'..'9': begin // number
                            s := WantNumber( pCurr );
                            FParts.AddObject( s, Pointer(ETYPE_INT) );
                            FScalars.Value[ FParts.Count-1 ] := strtoint(s);
                            WantOp := True;
                         end;
               '"'     : begin // string
                            s := WantString( pCurr, true );
                            if copy(s,1,1)='"' then System.Delete(s,1,1);
                            if copy(s,length(s),1)='"' then System.Delete(s,length(s),1);
                            FParts.AddObject( s, Pointer(ETYPE_STR) );
                            FScalars.Value[ FParts.Count-1 ] := s;
                            WantOp := True;
                         end;
               '$'     : begin // variable
                            s := WantVariable( pCurr );
                            FParts.AddObject( s, Pointer(ETYPE_VAR) );
                            WantOp := True;
                         end;
               'a'..'z',
               'A'..'Z': begin // function
                            s := WantFunction( pCurr );
                            FParts.AddObject( s, Pointer(ETYPE_FUN) );
                            WantOp := True;
                         end;
               else begin
                       Error( HSCERR_INVALIDEXPRESSION,
                              'Invalid expression #2: ' + String( pExpression ) );
                       break;
                    end;
            end;

         end
      end
   finally
      free
   end
end;

function THscExpression.SolveVar( Index: Integer ): Variant;
begin
   if not Assigned( FVariables ) then begin
      Error( HSCERR_UNSUPPORTED, 'No variables supported.' );
      exit;
   end;

   Result := FVariables.Value[ FParts[Index] ];
   if AnyError then exit;

   FScalars.Value[Index] := Result;
   case HscVarType(Result) of
      varInteger: FParts.Objects[Index] := Pointer(ETYPE_INT);
      varString : FParts.Objects[Index] := Pointer(ETYPE_STR);
      else        FParts.Objects[Index] := Pointer(ETYPE_ERR);
   end;
end;

function THscExpression.SolveFun( Index: Integer ): Variant;
begin
   if not Assigned( FEngine ) then begin
      Error( HSCERR_UNSUPPORTED, 'No functions supported.' );
      exit;
   end;

   Result := FEngine.SolveFunction( FParts[Index] );
   if AnyError then exit;
   FScalars.Value[Index] := Result;
   case HscVarType(Result) of
      varInteger: FParts.Objects[Index] := Pointer(ETYPE_INT);
      varString : FParts.Objects[Index] := Pointer(ETYPE_STR);
      else Error( HSCERR_UNSUPPORTED,
                  'Unknown function: ' + FParts[Index] );
   end;
end;

function THscExpression.GetAsScalar( Index: Integer ): Variant;
begin
   if      Integer(FParts.Objects[Index])=ETYPE_VAR then
      Result := SolveVar( Index )
   else if Integer(FParts.Objects[Index])=ETYPE_FUN then
      Result := SolveFun( Index )
   else
      Result := FScalars.Value[ Index ];
end;

procedure THscExpression.RemovePart( Index: Integer );
var  i: Integer;
begin
   for i:=Index+1 to FParts.Count-1 do begin
      FScalars.Value[i-1] := FScalars.Value[i];
   end;
   FParts.Delete( Index );
end;

procedure THscExpression.Solve_ParenScalarParen;
// "(" (Scalar/Var./Funct.) ")" => Scalar (del.) (del.)
var  PartNo: Integer;
begin
   PartNo := 0;
   while PartNo<=FParts.Count-1-2 do begin
      if LongInt(FParts.Objects[PartNo])=ETYPE_GON then begin
         if IsSolveable( LongInt( FParts.Objects[PartNo+1] ) ) then begin
            if LongInt( FParts.Objects[PartNo+2] )=ETYPE_GOF then begin
               RemovePart( PartNo+2 );
               RemovePart( PartNo   );
            end;
         end;
      end;
      inc( PartNo );
   end;
end;

procedure THscExpression.Solve_UnaryoperatorScalar;
// Unaryoperator (Scalar/Var./Funct.) => Scalar (del.)
var  PartNo, ETypeResult: Integer;
     OP: String;
     V1: Variant;
begin
     PartNo := 0;
     while PartNo<=FParts.Count-1-1 do begin

        if LongInt( FParts.Objects[PartNo] ) = ETYPE_UOP then begin
           if IsSolveable( LongInt( FParts.Objects[PartNo+1] ) ) then begin
              OP := FParts     [ PartNo   ];
              V1 := GetAsScalar( PartNo+1 ); if SelfError then break;

              V1 := HscCalculateUnary( Self, Op, V1, ETypeResult );
              if SelfError then break;

              FScalars.Value[PartNo] := V1;
              FParts.Objects[PartNo] := Pointer( ETypeResult );
              RemovePart( PartNo+1 );
           end;
        end;

        inc( PartNo );
     end;
end;

procedure THscExpression.Solve_ScalarOperatorScalar;
// (Scalar/Var./Funct.) Operator (Scalar/Var./Funct.) => Scalar (del.) (del.)
var  PartNo, ETypeResult: Integer;
     V1, V2: Variant;
     OP: String;
     OK: Boolean;
begin
     PartNo := 0;

     while PartNo<=FParts.Count-1-2 do begin
        if IsSolveable( LongInt( FParts.Objects[PartNo] ) ) then begin
           if IsOperator( LongInt( FParts.Objects[PartNo+1] ) ) then begin
              if IsSolveable( LongInt( FParts.Objects[PartNo+2] ) ) then begin

                 OK := True;
                 if (PartNo+3)<FParts.Count then begin
                    // followed by operator with higher priority?
                    if IsOperator( LongInt( FParts.Objects[PartNo+3] ) ) then begin
                       if LongInt(FParts.Objects[PartNo+1])<LongInt(FParts.Objects[PartNo+3]) then OK:=False;
                    end;
                 end;
                 if (PartNo-1)>=0 then begin
                    // preceded with operator of higher priority?
                    if IsOperator( LongInt( FParts.Objects[PartNo-1] ) ) then begin
                       if LongInt(FParts.Objects[PartNo+1])<LongInt(FParts.Objects[PartNo-1]) then OK:=False;
                    end;
                 end;

                 if OK then begin
                    V1 := GetAsScalar( PartNo   ); if SelfError then break;
                    OP := FParts     [ PartNo+1 ];
                    V2 := GetAsScalar( PartNo+2 ); if SelfError then break;

                    V1 := HscCalculate( Self, V1, Op, V2, ETypeResult );
                    if SelfError then break;

                    FScalars.Value[PartNo] := V1;
                    FParts.Objects[PartNo] := Pointer( ETypeResult );
                    RemovePart( PartNo+2 );
                    RemovePart( PartNo+1 );
                 end;
              end;
           end;
        end;

        inc( PartNo );
     end;
end;

function THscExpression.Solve( Expression: String ): Variant;
var  OldCount: Integer;
begin
   Result := Unassigned;
   FScalars := THscScalars.Create( Engine );

   try
      Parse( PChar( Expression ) );
      if AnyError then exit;

      while FParts.Count>1 do begin
         if AnyError then break;
         OldCount := FParts.Count;
         Solve_UnaryoperatorScalar;  // solve unary operators
         Solve_ScalarOperatorScalar; // solve operators
         Solve_ParenScalarParen;     // remove orphaned parens
         if FParts.Count=OldCount then begin
            Error( HSCERR_INVALIDEXPRESSION, 'Could''nt solve expression: ' + Expression );
            break;
         end;
      end;

      if not SelfError then Result := GetAsScalar(0);
   finally
      FScalars.Free;
   end;
end;

constructor THscExpression.Create( AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FParts := TStringList.Create;
   FVariables := FEngine.Variables;
end;

destructor THscExpression.Destroy;
begin
   FParts.Free;
   inherited Destroy;
end;

// ------------------------------------------------------- THscSchedEntry -----

function THscSchedEntry.Trigger: Boolean;
var  dNow: TDateTime;
     iNow: Integer;
     sNow: String;
     i: Integer;
begin
   Result := False;

   dNow := Now;
   iNow := DateTimeToUnixTime( dNow );

   if ( iNow - FLastTriggered ) < 60 then exit; // max. once per minute

   i := DayOfWeek( dNow ) - 1;
   if i=0 then i:=7;
   if (i<=length(FAtDays)) and (FAtDays[i]='0') then exit;

   sNow := FormatDateTime( 'hh"."nn', dNow );
   if FFromTime<=FTilTime then begin
      if (sNow<FFromTime) or  (sNow>FTilTime) then exit;
   end else begin
      if (sNow<FFromTime) and (sNow>FTilTime) then exit;
   end;

   i := ( iNow - FLastTriggered ) div 60;
   if i<FEveryMins then exit;

   Result := True;
   if Result then FLastTriggered := iNow;
end;

constructor THscSchedEntry.Create( AEngine: THscEngine;
                                   AKeyword: String;
                                   AFromTime, ATilTime, AAtDays: String;
                                   AEveryMins: Integer;
                                   AImmediate: Boolean );
begin
   inherited Create( AEngine );

   FKeyword   := AKeyword;
   FFromTime  := AFromTime;
   FTilTime   := ATilTime;
   FAtDays    := AAtDays;
   FEveryMins := AEveryMins;
   FImmediate := AImmediate;

   if FImmediate then FLastTriggered := 0
                 else FLastTriggered := DateTimeToUnixTime( Now );
end;

// -------------------------------------------------------- THscScheduler -----

function THscScheduler.Add( Keyword: String;
                            FromTime, TilTime, AtDays: String;
                            EveryMins: Integer;
                            Immediate: Boolean ): Integer;
var  HSE: THscSchedEntry;
begin
   while length(FromTime)<5 do FromTime:=FromTime+' ';
   if TilTime='' then TilTime:=FromTime;
   while length(TilTime )<5 do TilTime :=TilTime +'Z';
   FromTime[3] := '.';
   TilTime[3]  := '.';
   while length(AtDays)<7 do AtDays:=AtDays+'1';
   if EveryMins<0 then EveryMins:=0;

   HSE := THscSchedEntry.Create( Engine, Keyword,
                                 FromTime, TilTime, AtDays,
                                 EveryMins, Immediate );
   Result := FList.Add( HSE );
end;

procedure THscScheduler.Clear;
begin
   while FList.Count>0 do begin
      THscSchedEntry( FList[ FList.Count-1 ] ).Free;
      FList.Delete( FList.Count-1 );
   end;
end;

function THscScheduler.Check: String;
var  i: Integer;
begin
   Result := '';
   for i:=0 to FList.Count-1 do begin
      if THscSchedEntry( FList[i] ).Trigger then begin
         Result := THscSchedEntry( FList[i] ).Keyword;
         break;
      end;
   end;
end;

constructor THscScheduler.Create( AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FList := TList.Create;
end;

destructor THscScheduler.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;

// ------------------------------------------------------- THscScriptLine -----

                                        // JAWO Quot.Marks 12.10.2000
function THscScriptLine.FuncIs( TestFunc: String; ParMin, ParMax: Integer ): Boolean;
begin
   Result := False;
   if TestFunc=FuncName then begin
      FuncNameOK := True;
      if FuncPars.Count<ParMin then
         Error( HSCERR_SYNTAX, 'Missing parameters (min. '
                               + inttostr(ParMin) + '): ' + FuncName )
      else if FuncPars.Count>ParMax then
         Error( HSCERR_SYNTAX, 'Too many parameters (max. '
                               + inttostr(ParMax) + '): ' + FuncName )
      else
         Result := True;
   end;
end;

                                        // JAWO Quot.Marks 12.10.2000
function THscScriptLine.ParX( Index: Integer; DefVal: String ): String;
begin
   if Index<FuncPars.Count then Result := FuncPars[Index]
                           else Result := DefVal;
end;

                                        // JAWO Quot.Marks 12.10.2000
function THscScriptLine.ParV( Index: Integer; DefVal: Variant ): Variant;
begin
   if Index<FuncPars.Count then begin   // JAWO Quot.Marks 12.10.2000 (code modified)
      Result := Engine.SolveExpression( FuncPars[Index] );
      {if (HscVarType (Result) = varString) then
         Result := ClearQuotationmarks (Result);######################}
   end else
      Result := DefVal;
end;

                                        // JAWO Quot.Marks 12.10.2000
function THscScriptLine.ParI( Index: Integer; DefVal: Integer ): Integer;
begin
   if Index<FuncPars.Count then         // JAWO Quot.Marks 12.10.2000 (code modified)
      Result := HscVarToInt( Engine.SolveExpression( FuncPars[Index] ) )
   else
      Result := DefVal;
end;

                                        // JAWO Quot.Marks 12.10.2000
function THscScriptLine.ParS( Index: Integer; DefVal: String ): String;
begin
   if Index<FuncPars.Count then         // JAWO Quot.Marks 12.10.2000 (code modified)
      Result := {ClearQuotationmarks #####################(} HscVarToStr(
                           Engine.SolveExpression( FuncPars[Index] ) ){)}
   else
      Result := DefVal;
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantIdentifier, modified parameters)
function THscScriptLine.WantIdentifier( var pch: PChar ): String;
begin
   Result := '';
   if not( pch^ in CSET_LETTERS ) then begin // 1st char
                                        // JAWO Quot.Marks 12.10.2000 (changed:
                                        //              removed 'HscObject.' at Error functions)
      Error( HSCERR_SYNTAX, 'Identifier expected: ' + String( pch ) );
      exit;
   end;
   while pch^ in CSET_IDENTIFIER do begin
      Result := Result + pch^;
      inc( pch );
   end;
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantVariable, modified parameters)
function THscScriptLine.WantVariable( var pch: PChar ): String;
begin                                   // JAWO Quot.Marks 12.10.2000 (changed:
   Result := '';                        //              removed 'HscObject.' at Error functions
                                        //              removed 'HscObject, ' at WantXXX functions)
   if ( pch^<>'$' ) or not( (pch+1)^ in CSET_LETTERS ) then begin
      Error( HSCERR_SYNTAX, 'Variable expected: ' + String( pch ) );
      exit;
   end;
   inc( pch );
   Result := '$' + WantIdentifier( pch );
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantString, modified parameters)
function THscScriptLine.WantString( var pch: PChar; Const KillDoubleQuotes: boolean ): String;
begin
   Result := '';
   if pch^<>'"' then begin
                                        // JAWO Quot.Marks 12.10.2000 (changed:
                                        //              removed all 'HscObject.' at Error functions)
      Error( HSCERR_SYNTAX, 'String expected: ' + String( pch ) );
      exit;
   end;

   Result := '"';
   inc( pch );

   while pch^<>#0 do begin
      Result := Result + pch^;
      if pch^='"' then begin
         inc( pch );
         if pch^='"' then begin         // JAWO Quot.Marks 12.10.2000 (modified code)
            If Not KillDoubleQuotes then Result := Result + pch^
         end else
            break;
      end;
      inc( pch );
   end;

   if (length(Result)<2) or (Result[length(Result)]<>'"') then begin
      Error( HSCERR_SYNTAX, 'Missing end of string: ' + Result );
      exit;
   end;
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantNumber, modified parameters)
function THscScriptLine.WantNumber (var pch: PChar): String;
var  CharSet: set of Char;
begin
   Result := '';
   if not( pch^ in CSET_DIGITS ) then begin
                                        // JAWO Quot.Marks 12.10.2000 (changed:
                                        //              removed 'HscObject.' at Error functions)
      Error( HSCERR_SYNTAX, 'Number expected: ' + String( pch ) );
      exit;
   end;

   CharSet := CSET_DIGITS;
   Result := pch^;
   inc( pch );

   if (Result='0') and (pch^='x') then begin
      if (pch+1)^ in CSET_HEXDIGITS then begin
         Result := '$' + (pch+1)^;
         CharSet := CSET_HEXDIGITS;
         inc( pch, 2 );
      end;
   end;

   while pch^ in CharSet do begin
      Result := Result + pch^;
      inc( pch );
   end;

   Result := inttostr( strtoint( Result ) );
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantOperator, modified parameters)
function THscScriptLine.WantOperator (var pch: PChar; var OpClass: Integer): String;
var  c2: Char;
begin
   Result := '';

   c2 := (pch+1)^;

   case pch^ of
      '+','-':     begin Result:=pch^; OpClass:=ETYPE_OP8; end;
      '*','/','%': begin Result:=pch^; OpClass:=ETYPE_OP9; end;
      '^':         begin Result:=pch^; OpClass:=ETYPE_OP3; end;
      '|':         if c2='|' then begin
                      Result:='||'; OpClass:=ETYPE_OP0;
                      inc( pch );
                   end else begin
                      Result:='|'; OpClass:=ETYPE_OP2;
                   end;
      '&':         if c2='&' then begin
                      Result:='&&'; OpClass:=ETYPE_OP1;
                      inc( pch );
                   end else begin
                      Result:='&'; OpClass:=ETYPE_OP4;
                   end;
      '=':         begin
                      Result:='=='; OpClass:=ETYPE_OP5;
                      if c2='=' then inc( pch );
                   end;
      '!':         begin
                      Result:='!='; OpClass:=ETYPE_OP5;
                      if c2='=' then inc( pch );
                   end;
      '<':         case c2 of
                      '>': begin Result:='!='; OpClass:=ETYPE_OP5; inc( pch ); end;
                      '=': begin Result:='<='; OpClass:=ETYPE_OP6; inc( pch ); end;
                      '<': begin Result:='<<'; OpClass:=ETYPE_OP7; inc( pch ); end;
                      else begin Result:='<';  OpClass:=ETYPE_OP6; end;
                   end;
      '>':         case c2 of
                      '=': begin Result:='>='; OpClass:=ETYPE_OP6; inc( pch ); end;
                      '>': begin Result:='>>'; OpClass:=ETYPE_OP7; inc( pch ); end;
                      else begin Result:='>';  OpClass:=ETYPE_OP6; end;
                   end;
      else         begin
                                        // JAWO Quot.Marks 12.10.2000 (changed:
                                        //              removed 'HscObject.' at Error functions)
                      Error( HSCERR_INVALIDEXPRESSION,
                                       'Operator expected: ' + String( pch ) );
                      exit;
                   end;
   end;

   inc( pch );
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantParens, modified parameters)
function THscScriptLine.WantParens( var pch: PChar ): String;
var  Depth: Integer;
begin                                   // JAWO Quot.Marks 12.10.2000 (changed:
   Result := '';                        //              removed all 'HscObject.' at Error functions)
   if pch^<>'(' then begin              //              removed 'HscObject, ' at WantXXX functions)
      Error( HSCERR_SYNTAX, 'Missing "(": ' + String( pch ) );
      exit;
   end;

   Result := '(';
   Depth := 1;
   inc( pch );

   while pch^<>#0 do begin
      if SelfError then exit;
      case pch^ of
         '(': inc( Depth );
         ')': begin
                 dec( Depth );
                 if Depth=0 then begin
                    Result := Result + pch^;
                    inc(pch);
                    break;
                 end;
              end;
         '"': begin Result:=Result+WantString( pch, false ); continue; end;
      end;
      Result := Result + pch^;
      inc( pch );
   end;

   if Depth>0 then Error( HSCERR_SYNTAX, 'Missing ")": ' + Result );
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied,
                                        //              modified parameters)
function THscScriptLine.WantFunction( var pch: PChar ): String;
begin                                   // JAWO Quot.Marks 12.10.2000 (changed:
   Result := '';                        //              removed 'HscObject.' at Error functions
                                        //              removed all 'HscObject, ' at WantXXX functions)
   if not( pch^ in CSET_LETTERS ) then begin // 1st char
      Error( HSCERR_SYNTAX, 'Function expected: ' + String( pch ) );
      exit;
   end;
   Result := WantIdentifier( pch );
   SkipWSPC( pch );
   if pch^='(' then Result := Result + WantParens( pch );
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantExpression, modified parameters)
function THscScriptLine.WantExpression( var pch: PChar ): String;
var  pOrg  : PChar;
     s     : String;
     i     : Integer;
     WantOp: Boolean;
     Parens: Integer;
begin                                   // JAWO Quot.Marks 12.10.2000 (changed:
   Result := '';                        //              removed all 'HscObject.' at Error functions
   pOrg   := pch;                       //              removed all 'HscObject, ' at WantXXX functions)
   WantOp := False;
   Parens := 0;
   SkipWSPC( pch );

   while pch^ <> #0 do begin
      if SelfError then break;

      if WantOp then begin

         case pch^ of
            #9, ' ' : inc( pch ); // skip whitespace
            '{'     : SkipEmbeddedComment( pch );
            ','     : begin inc(pch); break; end; // param-separator
            ')'     : begin
                         Result := Result + pch^;
                         inc( pch );
                         dec( Parens );
                      end;
            else      begin
                         s := WantOperator( pch, i );
                         Result := Result + s;
                         WantOp := False;
                      end;
         end;

      end else begin

         case pch^ of
            #9, ' ' : inc( pch ); // skip whitespace
            '{'     : SkipEmbeddedComment( pch );
            ','     : begin inc(pch); break; end; // param-separator
            '('     : begin
                         Result := Result + pch^;
                         inc( pch );
                         inc( Parens );
                      end;
            '+','-',
            '!','~' : begin
                         Result := Result + pch^;
                         inc( pch );
                      end;
            '0'..'9': begin // number
                         s := WantNumber( pch );
                         Result := Result + s;
                         WantOp := True;
                      end;
            '"'     : begin // string
                         s := WantString( pch, false );
                         Result := Result + s;
                         WantOp := True;
                      end;
            '$'     : begin // variable
                         s := WantVariable( pch );
                         Result := Result + s;
                         WantOp := True;
                      end;
            'a'..'z',
            'A'..'Z': begin // function
                         s := WantFunction( pch );
                         Result := Result + s;
                         WantOp := True;
                      end;
            else begin
                    Error( HSCERR_INVALIDEXPRESSION,
                           'Invalid expression #1: ' + String( pOrg ) );
                    break;
                 end;
         end;

      end;

      SkipWSPC( pch );
   end;

   if SelfError then exit;

   if Parens>0 then Error( HSCERR_INVALIDEXPRESSION,
                                     'Missing ")": ' + String( pOrg ) );
   if Parens<0 then Error( HSCERR_INVALIDEXPRESSION,
                                     'Too many ")": ' + String( pOrg ) );
end;

                                        // JAWO Quot.Marks 12.10.2000 (function copied from
                                        //              WantSplitFunction, modified parameters)
procedure THscScriptLine.WantSplitFunction (var pcLine: PChar);
var  Par, s: String;
     pcTemp: PChar;
begin                                   // JAWO Quot.Marks 12.10.2000 (changed:
   FuncName := '';                      //              all 'cmd' -> 'FuncName'
   FuncPars.Clear;                      //              all 'Pars' -> 'FuncPars'
                                        //              removed all 'HscObject.' at Error functions
   SkipWSPC( pcLine );                  //              removed all 'HscObject, ' at WantXXX functions)
   if pcLine^=#0 then exit; // empty or comment-only line

   // get command
   if pcLine^ in CSET_LETTERS then begin
      FuncName := WantIdentifier( pcLine );
      if SelfError then exit;

      SkipWSPC( pcLine );
      if pcLine^=#0 then exit; // no parameters, "()" may be ommitted

      if pcLine^<>'(' then begin
         Error( HSCERR_SYNTAX, '"(" expected: ' + FuncName + '<(?>' + String(pcLine));
         exit;
      end;

      // get pars
      s := WantParens( pcLine ); // "(" params ")"
      if SelfError then exit;
      System.Delete( s, 1, 1 );
      System.Delete( s, length(s), 1 );

      pcTemp := PChar( s );
      while pcTemp^<>#0 do begin
         Par := WantExpression (pcTemp);
         if SelfError then break;
         FuncPars.Add( Par );
      end;

   end else if pcLine^='$' then begin
      FuncName := WantVariable( pcLine );
      if SelfError then exit;

      SkipWSPC( pcLine );
      if pcLine^<>'=' then begin
         Error( HSCERR_SYNTAX, '"=" expected: ' + FuncName + '<=?>' + String(pcLine));
         exit;
      end;
      inc( pcLine );

      FuncPars.Add( FuncName ); // convert: "$Variable=..." => "set($Variable,...)"
      FuncName := 'set';
      FuncPars.Add (WantExpression (pcLine));

   end else begin
      Error( HSCERR_SYNTAX, '"Function()" or "$Variable=" expected: ' + String(pcLine));
      exit;
   end;

   if SelfError then exit;
   SkipWSPC( pcLine );
   if pcLine^=#0 then exit;
   Error( HSCERR_SYNTAX, 'End of line expected: ' + String(pcLine));
end;

constructor THscScriptLine.Create (AEngine: THscEngine);
begin
   inherited Create (AEngine);
   FNameOK := true;
   FName   := '';
   FParms   := TStringList.Create;
end;

destructor THscScriptLine.Destroy;
begin
   if Assigned (FParms) then begin
      FParms.Free;
   end;
   inherited Destroy;
end;

// ----------------------------------------------------------- THscScript -----

procedure THscScript.AddSub( SubLine:   String; SubLineNo: Integer;     // JAWO 28.11.2000 (speed acceleration and
                             SubModule: string; SubModLine: Integer );  // subroutine declaration dupe check)
var s1, s2 : String;
    i1, i2 : integer;
begin
   s1 := LowerCase( TrimWSPEC( SubLine ) );
   if copy( s1, 1, 3 )='sub' then begin
      System.Delete( s1, 1, 3 );
      if (s1>'') and (s1[1] in [' ',#9]) then begin
         s1 := TrimWSPEC( s1 ) + ' '#9'(';  // find first arise of SPACE, TAB or '('
         i1 := Pos (' ', s1);               // (if none exists, use the appended ones)
         i2 := Pos (#9, s1);
         if ( i1 > i2 ) then
            i1 := i2;
         i2 := Pos ('(', s1);
         if ( i1 > i2 ) then
            i1 := i2;
         s1 := copy( s1, 1, i1-1 );         // separate the subroutine name
         s2 := SubModule + ' [' + inttostr(SubModLine) + '] = ' + s1;
         if ( FSubDupes.IndexOf(s2) < 0 ) then begin
            FSubDupes.Add ( s2 );
            i1 := FSubRefs.IndexOf(s1);
            if ( i1 < 0 ) then begin
               FSubRefs.AddObject( s1, Pointer(SubLineNo) );
            end else begin
               Log( LOGID_ERROR, '{' + SubModule + '} Error in line ' + inttostr(SubModLine) );
               Log( LOGID_ERROR, '{' + SubModule + '} Error: Multiple definition of:  ' + SubLine );
               i2 := Integer(FSubRefs.Objects[i1]);
               Log( LOGID_ERROR, '{' + SubModule + '} Error: Sub is already defined in:  ' +
                                 ModName[i2] + ', line ' + inttostr(LineNo[i2]) );
               if Assigned( FEngine ) then
                  FEngine.FTerminated := True;
            end;
         end;
      end;
   end;
end;



function THscScript.GetCount: Integer;
begin
   Result := FLines.Count;
end;

function THscScript.GetText: String;
begin
   Result := FLines.Text;
end;

function THscScript.GetLine( Index: Integer ): String;
begin
   Result := FLines[ Index ];
end;

function THscScript.GetLineNo( Index: Integer ): Integer;
begin
   Result := Integer( FLines.Objects[ Index ] );
end;

function THscScript.GetModID( Index: Integer ): Integer;
begin
   Result := Integer( FModRefs[ Index ] );
end;

function THscScript.GetModName( Index: Integer ): String;
begin
   Result := FModules[ ModID[Index] ];
end;

function THscScript.IsModuleLoaded( ModName: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   for i:=0 to FModules.Count-1 do begin
      if CompareText( ModName, FModules[i] )=0 then begin
         Result := True;
         break;
      end;
   end;
end;

function THscScript.StartNewModule( ModName: String ): Integer;
begin
   Result := FModules.AddObject( ModName, Pointer(FLines.Count) );
   FCurrLine := 0;
end;

function THscScript.Add( s: String ): Integer;
begin
   inc( FCurrLine );
   Result := FLines.AddObject( s, Pointer(FCurrLine) );
   FModRefs.Add( Pointer(FModules.Count-1) );
   AddSub( s, Result, ModName[Result], FCurrLine );  // JAWO 28.11.2000 (speed acceleration)
end;



procedure THscScript.AddLines( sl: TStringList );
var  i: Integer;
begin
   for i:=0 to sl.Count-1 do Add( sl[i] );
end;

procedure THscScript.Clear;
begin
   FLines.Clear;
   FModRefs.Clear;
   FModules.Clear;
   FSubRefs.Clear;   // JAWO 28.11.2000 (speed acceleration)
   FSubDupes.Clear;  // JAWO 28.11.2000 (subroutine declaration dupe check)
   FCurrLine := 0;
end;


constructor THscScript.Create( AEngine: THscEngine );
begin
   inherited Create( AEngine );
   FLines    := TStringList.Create;
   FModRefs  := TList.Create;
   FModules  := TStringList.Create;
   FSubRefs  := TStringList.Create;  // JAWO 28.11.2000 (speed acceleration)
   FSubDupes := TStringList.Create;  // JAWO 28.11.2000 (subroutine declaration dupe check)
   FCurrLine := 0;
end;

destructor THscScript.Destroy;
begin
   if Assigned( FSubDupes) then FSubDupes.Free; // JAWO 28.11.2000 (subroutine declaration dupe check)
   if Assigned( FSubRefs ) then FSubRefs.Free;  // JAWO 28.11.2000 (speed acceleration)
   if Assigned( FModules ) then FModules.Free;
   if Assigned( FModRefs ) then FModRefs.Free;
   if Assigned( FLines   ) then FLines  .Free;
   inherited Destroy;
end;

// ----------------------------------------------------------- THscEngine -----

procedure THscEngine.AddError( Sender: TObject; AErrNum: Integer; AErrMsg: String );
begin
   if FLastError=HSCERR_NOERROR then begin
      FErrNum    := AErrNum;
      FErrMsg    := AErrMsg;
      FErrModule := FScript.ModName[FCurrPos];
      FErrLineNo := FScript.LineNo[FCurrPos];
      FErrLine   := FScript[FCurrPos];
      FErrSender := Sender.ClassName;

      if not FErrCatch then begin
         FLastError := AErrNum;
         if Assigned( FOnError ) and ( AErrNum <> HSCERR_NOERROR ) then
            FOnError( Engine, Sender,
                      FErrNum, FErrMsg, FErrModule, FErrLineNo, FErrLine );
      end;
   end;
end;

procedure THscEngine.IPtrPush;
begin
   FIPtrStack.Add( Pointer( FCurrPos ) );
   FIPtrStack.Add( Pointer( FNextPos ) );
end;

procedure THscEngine.IPtrPop;
begin
   if FIPtrStack.Count>1 then begin
      FNextPos := Integer( FIPtrStack[ FIPtrStack.Count-1 ] );
      FIPtrStack.Delete( FIPtrStack.Count-1 );
      FCurrPos := Integer( FIPtrStack[ FIPtrStack.Count-1 ] );
      FIPtrStack.Delete( FIPtrStack.Count-1 );
   end;
end;

function THscEngine.GetContext: String;
begin
   if FContexts.Count>0 then Result := FContexts[ FContexts.Count-1 ]
                        else Result := '';
end;

procedure THscEngine.EnterContext( NewContext: String );
var  s: String;
begin
   if FContexts.Count>0 then s:=FContexts[FContexts.Count-1]+'.' else s:='';
   inc( FContextID );
   FContexts.Add( s + {LowerCase( NewContext ) +} inttostr(FContextID) );
end;

procedure THscEngine.LeaveContext;
begin
   if FContexts.Count>0 then begin
      FVariables.ClearContext( Context );
      FContexts.Delete( FContexts.Count-1 );
   end;
end;

procedure THscEngine.Trace( TracePos: Integer; TraceText: String );
begin
   if FTraceIsOn and Assigned(FOnTrace) then begin
      FOnTrace(
         Engine, FScript.ModName[TracePos], FScript.LineNo[TracePos], TraceText
      );
   end;
end;

procedure THscEngine.SyncExecute;
var  s: String;
     i: Integer;
     b: Boolean;
begin
   FSyncResult := Unassigned;

   if FSyncCmd='msgbox' then begin
      FSyncResult := Application.MessageBox( PChar( FSyncPars[0] ),
                                             PChar( FSyncPars[1] ),
                                             strtoint( FSyncPars[2] ) );
      exit;
   end;

   if FSyncCmd='inputbox' then begin
      s := FSyncPars[2];
      b := InputDlgStr( FSyncPars[0], FSyncPars[1], s, 0{HlpScriptInputBox} );
      FSyncResult := s;
      if b then FSyncPars[0]:='1' else FSyncPars[0]:='0';
      exit;
   end;

   if FSyncCmd='inputpw' then begin
      s := FSyncPars[2];
      b := InputDlgPwd( FSyncPars[0], FSyncPars[1], s, 0{HlpScriptInputBox} );
      FSyncResult := s;
      if b then FSyncPars[0]:='1' else FSyncPars[0]:='0';
      exit;
   end;

   if FSyncCmd='listbox' then begin
      i := strtoint( FSyncPars[2] );
      b := InputDlgList( FSyncPars[0], FSyncPars[1], i, FSyncPars[3], 0{HlpScriptInputListbox} );
      FSyncResult := i;
      if b then FSyncPars[0]:='1' else FSyncPars[0]:='0';
      exit;
   end;

   if FSyncCmd='popupbox' then begin
      FSyncResult := PopupBox (PChar( FSyncPars[0] ),
                               PChar( FSyncPars[1] ),
                               strtoint( FSyncPars[2] ),
                               strtoint( FSyncPars[3] ),
                               strtoint( FSyncPars[4] ) ); 
      exit;
   end;

end;

function THscEngine.SolveExpression( Expression: String ): Variant;
var  EX: THscExpression;
begin
   EX := THscExpression.Create( Engine );
   try
      Result := EX.Solve( Expression );
   finally
      EX.Free;
   end;
end;

function THscEngine.LineCmd( Index: Integer ): String;
var  i: Integer;
begin
   Result := LowerCase( TrimWSPEC( FScript[Index] ) );
   i := PosWhSpace( Result ); if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '(', Result );   if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '#', Result );   if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '{', Result );   if i>0 then Result:=copy(Result,1,i-1);
   if copy(Result,1,1)='$' then Result:='set';
end;

function THscEngine.IndexOfLabel( LabelName: String ): Integer;
var  i, j: Integer;
     s   : String;
begin
   Result := -1;
   for i:=0 to FScript.Count-1 do begin
      s := TrimWhSpace( FScript[i] );

      if CompareText( copy(s,1,5), 'label' ) = 0 then begin
         System.Delete( s, 1, 5 );

         j := Pos( '(', s );
         if j=0 then continue;
         System.Delete( s, 1, j );

         j := Pos( ')', s );
         if j=0 then continue;
         s := TrimWhSpace( copy(s,1,j-1) );

         if CompareText( s, LabelName ) = 0 then begin
            Result := i;
            break;
         end;

      end;
   end;
end;

function THscEngine.IndexOfSub( SubName: String ): Integer;  // JAWO 28.11.2000 (speed acceleration)
var  i   : Integer;
     s   : String;
begin
   Result := -1;
   s := LowerCase( SubName );

   with FScript.FSubRefs do begin
      i := IndexOf(s);
      if ( i >= 0 ) then
         Result := Integer ( Objects[ IndexOf(s) ] );
   end;
end;

function THscEngine.SolveFunction( FuncStr: String ): Variant;
Var Scriptline: THSCScriptline;

  {JW}
  WaitObjects : array[0..1] of LongInt;
  {JW}

   procedure do_decodetime;
   var  dt: TDateTime;
        yy, mm, dd, hh, nn, ss, ms: Word;
   begin
      with ScriptLine do begin
         Result := ParI( 0, 0 );
         dt := UnixTimeToDateTime( Result );
         DecodeDate( dt, yy, mm, dd );
         DecodeTime( dt, hh, nn, ss, ms );
         if IsVariable(ParX(1,'')) then Variables.Value[ParX(1,'')]:=yy;
         if IsVariable(ParX(2,'')) then Variables.Value[ParX(2,'')]:=mm;
         if IsVariable(ParX(3,'')) then Variables.Value[ParX(3,'')]:=dd;
         if IsVariable(ParX(4,'')) then Variables.Value[ParX(4,'')]:=hh;
         if IsVariable(ParX(5,'')) then Variables.Value[ParX(5,'')]:=nn;
         if IsVariable(ParX(6,'')) then Variables.Value[ParX(6,'')]:=ss;
         if IsVariable(ParX(7,'')) then Variables.Value[ParX(7,'')]:=DayOfWeek(dt)
      end
   end;

   procedure do_encodetime;
   var  dt: TDateTime;
        yy, mm, dd, hh, nn, ss: Word;
   begin
      with ScriptLine do begin
         yy := ParI( 0, 0 );
         mm := ParI( 1, 0 );
         dd := ParI( 2, 0 );
         hh := ParI( 3, 0 );
         nn := ParI( 4, 0 );
         ss := ParI( 5, 0 );
         dt := EncodeDate( yy, mm, dd ) + EncodeTime( hh, nn, ss, 0 );
         Result := DateTimeToUnixTime( dt )
      end
   end;

 var
    VTransfer  : Array of Variant;
    VSubVars   : Array of String;
    VSubRefs   : Array of Boolean;
    pFunction: PChar;
    i, j, k, Timeout: Integer;
    s, t, u, v: String;
    V1, V2: Variant;
    OK: Boolean;
    RasDialPars: TRasDialParams;
    SR: TSearchRec;
    TS: TStringList;
begin
   Result := Unassigned;
   ScriptLine := THscScriptLine.Create (self);
   with ScriptLine do try
      pFunction := PChar( FuncStr );
      WantSplitFunction( pFunction );
      if AnyError then exit;
      FuncName   := LowerCase( FuncName );
      FuncNameOK := False;
      try

         if FuncName='' then begin
            FuncNameOK := True;
         end else if FuncIs( 'getprocessidentifier', 0, 0 ) then begin
            result:=Mutexstring;
      {JW} {HOSTS}
         end else if FuncIs( 'sethostsentry_byname', 1, 2 ) then begin
            if ParS(0,'')<>''
               then result:=sethostsentry_byname(ParS( 0, '' ),ParS( 1, '' ))
               else result:=-1
         end else if FuncIs( 'sethostsentry_byaddr', 1, 2 ) then begin
            if ParS(0,'')<>''
               then result:=sethostsentry_byaddress(ParS( 1, '' ),ParS( 0, '' ))
               else result:=-1
      {JW} {delete hostsentry}
         end else if FuncIs( 'deletehostsentry', 2, 2 ) then begin
             if (ParS(0,'')<>'') and  (ParS(0,'')<>'')
                then result:=deletehostsentry(ParS( 0, '' ),ParS( 1, '' ))
                else result:=-1;
          end else if FuncIs( 'localhostname', 0, 0 ) then begin
            result:=LookupLocalHostName;
         end else if FuncIs( 'localhostaddr', 0, 0 ) then begin
            result:=nAddrToStr(LookupLocalHostAddr);
      {JW} {Lookup}
         end else if FuncIs( 'lookuphostaddr', 1, 1 ) then begin
            result:=nAddrToStr(LookupHostAddr(ParS(0,'')));
         end else if FuncIs( 'lookuphostname', 1, 1 ) then begin
            result:=LookupHostName(StrToAddr(Pars(0,'')));
      {JW}
         end else if FuncIs( 'eventcreate', 1, 4 ) then begin
            s := ParS( 0, '' ); // event-name
            i := ParI( 1, 1  ); // manual reset
            j := ParI( 2, 0  ); // initial state
            Result := Integer( CreateEvent( nil, i=0, j<>0, PChar(s) ) );
            i := GetLastError;
            s := ParX( 3, '?' ); // error-var
            if IsVariable( s ) then Variables.Value[ s ] := i;
         end else if FuncIs( 'eventwait', 1, 2)  then begin
            i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(ParS(0,'')));
            if i<>0 then begin
               WaitObjects[0]:=i;
               WaitObjects[1]:=FStopEvent;
               if FStopEvent=0 then j:=1 else j:=2;
               k:=ParI(1,-1);
               If k=-1 then
               result:=Integer(WaitForMultipleObjects(j,
                               @WaitObjects,False,INFINITE))
               else
               result:=Integer(WaitForMultipleObjects(j,
                               @WaitObjects,False,k));

               if (result<>Wait_Timeout) and (result<>Wait_Object_0) then
                   result:=Integer(GetLastError);    
               CloseHandle(i);
            end else  result:=Integer(GetLastError);
         end else if FuncIs( 'eventreset', 1, 1)  then begin
            i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(
            ParS(0,''))); if i <> 0 then begin
               if ResetEvent(i) then result:=0 else
               result:=Integer(GetLastError); CloseHandle(i);
            end else result:=-1;
         end else if FuncIs( 'eventset', 1, 1)  then begin
            i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(ParS(0,'')));
            if i <> 0 then begin
               if SetEvent(i) then result:=0 else
               result:=Integer(GetLastError); CloseHandle(i);
            end else result:=-1;
         end else if FuncIs( 'eventpulse', 1, 1)  then begin
            i:=OpenEvent(EVENT_MODIFY_STATE or SYNCHRONIZE,False,PChar(
            ParS(0,''))); if i <> 0 then begin
               if PulseEvent(i) then result:=0 else
               result:=Integer(GetLastError); CloseHandle(i);
            end else result:=-1;
         end else if FuncIs( 'eventclose', 1, 1)  then begin
            result:=CloseHandle(ParI(0,0));
       {/JW}
         end else if FuncIs( 'set', 2, 2 ) then begin
            s := ParX( 0, '?' );
            if IsVariable( s ) then begin
               Result := ParV( 1, Unassigned );
               Variables.Value[ s ] := Result;
            end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
         end else if FuncIs( 'varset', 2, 2 ) then begin
            s := ParX( 0, '?' );
            if IsVariable( s ) then begin
               Result := ParV( 1, Unassigned );
               Variables.DefValue( s, Result );
            end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
         end else if FuncIs( 'inc', 1, 2 ) then begin
            s := ParX( 0, '?' );
            if IsVariable( s ) then begin
               i := ParI( 1, 1 );
               Result := HscVarToInt( Variables.Value[ s ] ) + i;
               Variables.Value[ s ] := Result;
            end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
         end else if FuncIs( 'dec', 1, 2 ) then begin
            s := ParX( 0, '?' );
            if IsVariable( s ) then begin
               i := ParI( 1, 1 );
               Result := HscVarToInt( Variables.Value[ s ] ) - i;
               Variables.Value[ s ] := Result;
            end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );

         end else if FuncIs( 'true', 0, 0 ) then begin
            Result := 1;
         end else if FuncIs( 'false', 0, 0 ) then begin
            Result := 0;

         end else if FuncIs( 'isint', 1, 1 ) then begin
            Result := Integer( HscVarType( ParV(0,Unassigned) ) = varInteger );
         end else if FuncIs( 'isstr', 1, 1 ) then begin
            Result := Integer( HscVarType( ParV(0,Unassigned) ) = varString );

         end else if FuncIs( 'int', 1, 1 ) then begin
            Result := ParI( 0, 0 );
         end else if FuncIs( 'abs', 1, 1 ) then begin
            Result := abs( ParI( 0, 0 ) );
         end else if FuncIs( 'sgn', 1, 1 ) then begin
            Result := sgn( ParI( 0, 0 ) );

         end else if FuncIs( 'chr', 1, 1 ) then begin
            Result := chr( ParI( 0, 0 ) and $FF );
         end else if FuncIs( 'ord', 1, 1 ) then begin
            s := ParS( 0, '' ) + #0;
            Result := ord( s[1] );
         end else if FuncIs( 'str', 1, 3 ) then begin
            Result := ParS( 0, '' );
            i := ParI( 1, 1 );
            t := copy( ParS(2,'0') + '0', 1, 1 );
            while length(Result)<i do Result := t + Result;
         end else if FuncIs( 'hex', 1, 2 ) then begin
            Result := inttohex( ParI(0,0), ParI(1,1) );
         end else if FuncIs( 'len', 1, 1 ) then begin
            Result := length( ParS(0,'') );
         end else if FuncIs( 'pos', 2, 3 ) then begin
            i := ParI( 2, 1 );
            if i=1 then
               Result := Pos( ParS(0,''), ParS(1,'') )
            else begin
               s := ParS(1,'');
               System.Delete( s, 1, i-1 );
               Result := Pos( ParS(0,''), s );
               if Result>0 then inc( Result, i-1 );
            end;
         end else if FuncIs( 'copy', 2, 3 ) then begin
            s := ParS( 0, '' );
            i := ParI( 1, 1 );
            j := ParI( 2, length(s) );
            Result := copy( s, i, j );
         end else if FuncIs( 'delete', 2, 3 ) then begin
            s := ParS( 0, '' );
            i := ParI( 1, 1 );
            j := ParI( 2, length(s) );
            System.Delete( s, i, j );
            Result := s;
         end else if FuncIs( 'trim', 1, 2 ) then begin
            s := ParS( 0, '' );
            if FuncPars.Count=1 then begin
               Result := Trim( s );
            end else begin
               t := ParS( 1, '' );
               while (s<>'') and (t<>'') do begin
                  if      pos(s[1        ],t)>0 then System.Delete(s,1        ,1)
                  else if pos(s[length(s)],t)>0 then System.Delete(s,length(s),1)
                  else break;
               end;
               Result := s;
            end;
         end else if FuncIs( 'lowercase', 1, 1 ) then begin
            Result := LowerCase( ParS(0,'') );
         end else if FuncIs( 'uppercase', 1, 1 ) then begin
            Result := UpperCase( ParS(0,'') );
         end else if FuncIs( 'replace', 3, 5 ) then begin
            s := ParS( 0, '' ); // base-string
            t := ParS( 1, '' ); // string to find
            u := ParS( 2, '' ); // replacement
            i := ParI( 3, 0  ); // replace all
            j := ParI( 4, 0  ); // ignore case
            Result := '';
            if j=0 then v:=s else begin v:=LowerCase(s); t:=LowerCase(t) end;
            while v<>'' do begin
               k := Pos( t, v );
               if k=0 then begin Result:=Result+s; break; end;
               Result := Result + copy( s, 1, k-1 ) + u;
               System.Delete( s, 1, k + length(t) - 1 );
               if i=0 then begin Result:=Result+s; break; end;
               System.Delete( v, 1, k + length(t) - 1 );
            end;

         end else if FuncIs( 're_match', 2, 2 ) then begin
            s := ParS( 0, '' );
            t := ParS( 1, '' );
            if RE_Match( s, t, PCRE_CASELESS ) then Result:=1 else Result:=0;
         end else if FuncIs( 're_extract', 2, 2 ) then begin
            s := ParS( 0, '' );
            t := ParS( 1, '' );
            Result := RE_Extract( s, t, PCRE_CASELESS );
         end else if FuncIs( 're_parse', 3, 999 )
                  or FuncIs( 're_split', 4, 999 )  then begin
            Result := 0; // false
            s := ParS( 0, '' );
            t := ParS( 1, '' );
            TS := TStringList.Create;
            try
               if FuncName='re_parse' then
                  RE_Parse( s, t, PCRE_CASELESS, TS )
               else
                  RE_Split( s, t, PCRE_CASELESS, FuncPars.Count-2, TS );
               for i:=2 to FuncPars.Count-1 do begin
                  s := ParX( i, '?' );
                  if IsVariable( s ) then begin
                     if (i-2)<TS.Count then t:=TS[i-2] else t:='';
                     Variables.Value[ s ] := t;
                  end;
               end;
               Result := 1; // true
            finally
               TS.Free;
            end;

         end else if FuncIs( 'ticks', 0, 0 ) then begin
            Result := Integer(GetTickCount);
         end else if FuncIs( 'time', 0, 0 ) then begin
            Result := DateTimeToUnixTime( Now );
         end else if FuncIs( 'timegmt', 0, 0 ) then begin
            Result := DateTimeToUnixTime( NowGMT );
         end else if FuncIs( 'decodetime', 2, 8 ) then begin
            do_decodetime;
         end else if FuncIs( 'encodetime', 6, 6 ) then begin
            do_encodetime;
         end else if FuncIs( 'eval', 1, 1 ) then begin
            s := ParS( 0, '' );
            Result := SolveExpression( s );
         end else if FuncIs( 'iif', 3, 3 ) then begin
            if ParI(0,0)<>0 then Result := ParV( 1, 0 )
                            else Result := ParV( 2, 0 );
         end else if FuncIs( 'icase', 3, 999 ) then begin
            Result := 0;
            V1 := ParV( 0, 0 );
            i := 1;
            while (i+1<FuncPars.Count) and not(AnyError) do begin
               OK := False;
               if LowerCase( ParX(i,'') )='else' then begin
                  OK := True;
               end else begin
                  V2 := ParV( i, 0 );
                  V2 := HscCalculate( Self, V1, '==', V2, j );
                  if (j=ETYPE_INT) and (V2<>0) then OK:=True;
               end;
               if OK then begin
                  Result := ParV( i+1, 0 );
                  break;
               end;
               inc( i, 2 );
            end;

         end else if FuncIs( 'fileexists', 1, 1 ) then begin
            try
               if FileExists2(ParS(0,'')) then Result:=1 else Result:=0;
            except
               Result := 0;
            end;
         end else if FuncIs( 'filedelete', 1, 1 ) then begin
            Result := 0;
            s := ParS( 0, '' );
            if not Windows.DeleteFile( PChar(s) ) then Result:=Integer(GetLastError);
         end else if FuncIs( 'filerename', 2, 2 ) then begin
            Result := 0;
            s := ParS( 0, '' );
            t := ParS( 1, '' );
            if not Windows.MoveFile( PChar(s), PChar(t) ) then Result:=Integer(GetLastError);
         end else if FuncIs( 'filecopy', 2, 2 ) then begin
            Result := 0;
            s := ParS( 0, '' );
            t := ParS( 1, '' );
            if not Windows.CopyFile( PChar(s), PChar(t), False ) then Result:=Integer(GetLastError);

         end else if FuncIs( 'filesize', 1, 1 ) then begin
            Result := -1;
            if FindFirst( ParS(0,''), faAnyFile, SR ) = 0 then begin
               Result := SR.Size;
               SysUtils.FindClose( SR );
            end;
         end else if FuncIs( 'filetime', 1, 1 ) then begin
            Result := -1;
            if FindFirst( ParS(0,''), faAnyFile, SR ) = 0 then begin
               Result := DateTimeToUnixTime( FileDateToDateTime(SR.Time) );
               SysUtils.FindClose( SR );
            end;

         end else if FuncIs( 'direxists', 1, 1 ) then begin
            try
               if DirectoryExists(ParS(0,'')) then Result:=1 else Result:=0;
            except
               Result := 0;
            end;
         end else if FuncIs( 'dirmake', 1, 1 ) then begin
            Result := 0;
            try MkDir( ParS(0,'') ) except Result:=-1 end;
         end else if FuncIs( 'dirremove', 1, 1 ) then begin
            Result := 0;
            try RmDir( ParS(0,'') ) except Result:=-1 end;
         end else if FuncIs( 'dirchange', 1, 1 ) then begin
            Result := 0;
            try ChDir( ParS(0,'') ) except Result:=-1 end;
         end else if FuncIs( 'dircurrent', 0, 0 ) then begin
            Result := '';
            try GetDir( 0, s ); Result:=s except Result:='' end;
            if s > '' then Result:=IncludeTrailingBackslash(s)
                      else Result:=''
         end else if FuncIs( 'dirwindows', 0, 0 ) then begin
            Result := GetWindowsPath;
         end else if FuncIs( 'dirsystem', 0, 0 ) then begin
            Result := GetSystemPath;

         end else if FuncIs( 'iniread', 4, 4 ) then begin
            s := ParS( 0, '' ); // ini-filename
            t := ParS( 1, '' ); // section
            u := ParS( 2, '' ); // identifier
            v := ParS( 3, '' ); // default-value
            if s='' then s:='HScripts.ini';
            if t='' then t:='All';
            With TIniFile.Create( ExpandFilename(s) ) do try
               Result := ReadString( t, u, v );
            finally Free end;
         end else if FuncIs( 'iniwrite', 4, 4 ) then begin
            s := ParS( 0, '' ); // ini-filename
            t := ParS( 1, '' ); // section
            u := ParS( 2, '' ); // identifier
            Result := ParS( 3, '' ); // value
            if s='' then s:='HScripts.ini';
            if t='' then t:='All';
            With TIniFile.Create( ExpandFilename(s) ) do try
               WriteString( t, u, Result )
            finally Free end;
         end else if FuncIs( 'inidelete', 3, 3 ) then begin
            s := ParS( 0, '' ); // ini-filename
            t := ParS( 1, '' ); // section
            u := ParS( 2, '' ); // identifier
            Result := ''; // value
            if s='' then s:='HScripts.ini';
            if t='' then t:='All';
            With TIniFile.Create( ExpandFilename(s) ) do try
               DeleteKey( t, u )
            finally Free end;
         end else if FuncIs( 'inierasesection', 2, 2 ) then begin
            s := ParS( 0, '' ); // ini-filename
            t := ParS( 1, '' ); // section
            Result := ''; // value
            if s='' then s:='HScripts.ini';
            if t='' then t:='All';
            With TIniFile.Create( ExpandFilename(s) ) do try
               EraseSection( t )
            finally Free end;
         end else if FuncIs( 'execute', 1, 5 ) then begin
            s := ParS( 0, '' ); // command-line
            t := ParS( 1, '' ); // working-dir
            i := ParI( 2, SW_SHOWNORMAL ); // showwindow
            j := ParI( 3, 1 ); // waitflag (default=wait)
            Result := ExecuteProcess( s, t, i, (j<>0), k, FStopEvent );
            if (Result=0) and (FuncPars.Count>=5) then begin
               s := ParX( 4, '' ); // var for exitcode
               if IsVariable(s) then Variables.Value[s] := k;
            end;

         end else if FuncIs( 'beep', 0, 1 ) then begin
             result:=MessageBeep(ParI(0,0));

         end else if FuncIs( 'rasgetconnection', 0, 0 ) then begin
            Result := RasDynGetConnection;
         end else if FuncIs( 'rasisconnected', 0, 0 ) then begin
            if RasDynIsConnected then Result:=1 else Result:=0;
         end else if FuncIs( 'rasdial', 1, 4 ) then begin
            if RasDynInit then begin
               s := ParS( 0, '' ); // entry
               t := ParS( 1, '' ); // user
               u := ParS( 2, '' ); // pass
               v := ParX( 3, '' ); // hconn-var
               FillChar( RasDialPars, sizeof(RasDialPars), 0 );
               RasDialPars.dwSize := sizeof( RasDialPars );
               strpcopy( RasDialPars.szEntryName, s );
               strpcopy( RasDialPars.szUserName,  t );
               strpcopy( RasDialPars.szPassword,  u );
               j := 0; // conn-hdl
               Result := RasDynDial( nil, nil, RasDialPars, 0, nil, j );
               if (Result<>0) and (j<>0) then begin RasDynHangup(j); j:=0 end;
               if (v<>'') and IsVariable(v) then Variables.Value[ v ] := j;
            end else Result:=-1; // no RAS installed
         end else if FuncIs( 'rasgetip', 0, 0 ) then begin
            Result := '';
            if RasDynIsConnected then Result := RasDynGetPppIp(RasDynGetConn);
          end else if FuncIs( 'rashangup', 0, 1 ) then begin
            if RasDynInit then begin
               i := ParI( 0, 0 );
               if i=0 then i := RasDynGetConn;
               if i<>0 then Result := RasDynHangUp( i ) else Result := 0;
            end else Result:=-1; // no RAS installed

         end else if FuncIs( 'atclear', 0, 0 ) then begin
            FScheduler.Clear;
            Result := 0;
         end else if FuncIs( 'atadd', 2, 6 ) then begin
            s := ParX( 0, '?' ); // sub-name
            if IsIdentifier(s) then i:=IndexOfSub(s) else i:=-1;
            if i<0 then begin
               Error( HSCERR_UNSUPPORTED, 'Sub not found: ' + s );
               Result := -1;
            end else begin
               t := ParS( 1, '' ); // from-time
               u := ParS( 2, t  ); // til-time
               v := ParS( 3, '1111111' ); // days
               i := ParI( 4, 0  ); // min's
               j := ParI( 5, 1  ); // immediate
               Result := FScheduler.Add( s, t, u, v, i, (j<>0) );
            end;
         end else if FuncIs( 'atexecute', 0, 1 ) then begin
            Result  := 0;
            Timeout := ParI( 0, -1 );
            repeat
               // check and execute all times and intervals
               repeat
                  s := FScheduler.Check;
                  if s<>'' then begin
                     i := IndexOfSub( s );
                     if i>=0 then begin
                        // invoke function within script
                        IPtrPush;
                        Trace( i, FScript[i] );
                        FNextPos := i+1;
                        EnterContext( s );
                        Result := HscVarToInt( ExecuteNextPos );
                        LeaveContext;
                        IPtrPop;
                     end else Error( HSCERR_UNSUPPORTED, 'Sub not found: ' + s );
                  end;
                  if AnyError or (Result<>0) then break;
               until s='';
               if AnyError or (Result<>0) then break;

               // sleep for up to 30 seconds
               i := 30000;
               if (Timeout>=0) and (Timeout<i) then i:=Timeout;
               if FStopEvent<>0 then begin
                  if WaitForSingleObject(FStopEvent,i)=WAIT_OBJECT_0 then break;
               end else begin
                  Sleep(i);
               end;
               if Timeout>=0 then begin
                  dec( Timeout, i );
                  if Timeout<=0 then begin Result:=-1; break; end;
               end;
            until False;

         end else if FuncIs( 'listalloc', 0, 2 ) then begin
            i := ParI( 0, 0 ); // sorted (def.=not)
            j := ParI( 1, 1 ); // duplicates (def.=allowed)
            Result := FHscLists.ListAlloc( i<>0, j<>0 );
         end else if FuncIs( 'listfree', 1, 1 ) then begin
            Result := FHscLists.ListFree( ParI(0,-1) );
         end else if FuncIs( 'listexists', 1, 2 ) then begin
            if FHscLists.ListExists( ParI(0,-1) ) then Result:=1 else Result:=0;
            if (Result<>0) and (FuncPars.Count>1) then begin
               i := ParI( 1, 0 );
               if (i<0) or (i>=FHscLists.List[ParI(0,-1)].Count) then Result:=0;
            end;
         end else if FuncIs( 'listclear', 1, 1 ) then begin
            Result := 0;
            FHscLists.List[ ParI(0,-1) ].Clear;
         end else if FuncIs( 'listcount', 1, 1 ) then begin
            Result := FHscLists.List[ ParI(0,-1) ].Count;
         end else if FuncIs( 'listget', 2, 2 ) then begin
            Result := FHscLists.ListItem[ ParI(0,-1), ParI(1,-1) ];
         end else if FuncIs( 'listset', 2, 3 ) then begin
            Result := ParS( 2, '' );
            FHscLists.ListItem[ ParI(0,-1), ParI(1,-1) ] := Result;
         end else if FuncIs( 'listgettag', 2, 2 ) then begin
            Result := FHscLists.ListTag[ ParI(0,-1), ParI(1,-1) ];
         end else if FuncIs( 'listsettag', 3, 3 ) then begin
            Result := ParI( 2, 0 );
            FHscLists.ListTag[ ParI(0,-1), ParI(1,-1) ] := Result;
         end else if FuncIs( 'listgetkey', 2, 2 ) then begin
            Result := FHscLists.List[ ParI(0,-1) ].Values[ ParS(1,'') ];
         end else if FuncIs( 'listsetkey', 2, 3 ) then begin
            Result := ParS( 2, '' );
            FHscLists.List[ ParI(0,-1) ].Values[ ParS(1,'') ] := Result;
         end else if FuncIs( 'listadd', 1, 2 ) then begin
            Result := FHscLists.List[ ParI(0,-1) ].Add( ParS(1,'') );
         end else if FuncIs( 'listdelete', 2, 2 ) then begin
            Result := 0;
            FHscLists.List[ ParI(0,-1) ].Delete( ParI(1,-1) );
         end else if FuncIs( 'listinsert', 2, 3 ) then begin
            Result := ParS( 2, '' );
            FHscLists.List[ ParI(0,-1) ].Insert( ParI(1,-1), Result );
         end else if FuncIs( 'listsort', 1, 1 ) then begin
            Result := 0;
            FHscLists.List[ ParI(0,-1) ].Sort;
         end else if FuncIs( 'listsettext', 2, 2 ) then begin
            Result := ParS( 1, '' );
            FHscLists.List[ ParI(0,-1) ].Text := Result;
         end else if FuncIs( 'listgettext', 1, 1 ) then begin
            Result := FHscLists.List[ ParI(0,-1) ].Text;
         end else if FuncIs( 'listindexof', 2, 2 ) then begin
            Result := FHscLists.List[ ParI(0,-1) ].IndexOf( ParS(1,'') );
         end else if FuncIs( 'listload', 2, 2 ) then begin
            Result := -1;
            i := ParI( 0, -1 );
            s := ParS( 1, '' );
            if Assigned( FHscLists.List[i] ) then begin
               if FileExists(s) then begin
                  FHscLists.List[i].LoadFromFile( s );
                  Result := 0;
               end;
            end;
         end else if FuncIs( 'listsave', 2, 2 ) then begin
            Result := -1;
            i := ParI( 0, -1 );
            s := ParS( 1, '' );
            if Assigned( FHscLists.List[i] ) then begin
               try
                  FHscLists.List[i].SaveToFile( s );
                  Result := 0;
               except
               end;
            end;
         end else if FuncIs( 'listfiles', 2, 3 )
                  or FuncIs( 'listdirs',  2, 3 ) then begin
            Result := 0;
            i := ParI( 0, -1 );
            s := ParS( 1, '' );
            j := ParI( 2, 0 );
            try
               if j=0 then t:='' else begin
                  t := ExtractFilePath( ExpandFilename( s ) );
               end;
               if FuncIs( 'listfiles', 2, 3 ) then k:=faAnyFile-faDirectory
                                              else k:=faDirectory;
               if FindFirst( s, faAnyFile-faVolumeID, SR ) = 0 then begin
                  repeat
                     OK := True;
                     if (k<>faDirectory) then begin
                        if (SR.Attr and faDirectory) <> 0 then OK:=False;
                     end else begin
                        if (SR.Attr and faDirectory) = 0 then OK:=False;
                     end;
                     if (k=faDirectory) and (SR.Name[1]='.') then OK:=False;
                     if OK then FHscLists.List[ i ].Add( t + SR.Name );
                  until FindNext( SR ) <> 0;
                  SysUtils.FindClose( SR );
               end;
               Result := FHscLists.List[ i ].Count;
            except
               Result := -2;
            end;
         end else if FuncIs( 'listrasentries', 1, 1 ) then begin
            Result := 0;
            i := ParI( 0, -1 );
            TS := FHscLists.List[ i ];
            if Assigned(TS) then begin
               if RasDynEnumPhonebookEntries( TS ) then begin
                  Result := TS.Count;
               end;
            end;

         end else if FuncIs( 'print', 0, 999 ) then begin
            Result := '';
            for i:=0 to FuncPars.Count-1 do Result := Result + ParS(i,'');
            if Assigned( FOnPrint ) then FOnPrint( Engine, Result );
         end else if FuncIs( 'warning', 1, 999 ) then begin
            Result := '';
            for i:=0 to FuncPars.Count-1 do Result := Result + ParS(i,'');
            if Assigned( FOnWarning ) then FOnWarning( Engine, Result );
         end else if FuncIs( 'addlog', 2, 3 ) then begin
            Result := ParS(0, '');
            if Assigned( FOnAddLog ) then FOnAddLog( Engine, Result, ParI(1, 0), ParI(2, 1) )
         end else if FuncIs( 'msgbox', 1, 3 ) then begin
            // Note: Par?() may also call SyncExecute, so pars are saved to vars first
            s := ParS( 0, '' );
            t := ParS( 1, 'Hamster-Script' );
{JW} {msgbox}
            i := ParI( 2, MB_OK or MB_ICONINFORMATION);
{JW}
            FSyncCmd := 'msgbox';
            FSyncPars.Clear;
            FSyncPars.Add( s );
            FSyncPars.Add( t );
            FSyncPars.Add( inttostr(i) );
            FSyncResult := Unassigned;
            if Assigned(FOnSyncExec) then FOnSyncExec( Engine, SyncExecute )
                                     else SyncExecute;
            Result := FSyncResult;
         end else if FuncIs( 'inputbox', 1, 4 ) then begin
            s := ParS( 0, '' );
            t := ParS( 1, 'Hamster-Script' );
            u := ParS( 2, '' );
            v := ParX( 3, '?' );
            FSyncCmd := 'inputbox';
            FSyncPars.Clear;
            FSyncPars.Add( t );
            FSyncPars.Add( s );
            FSyncPars.Add( u );
            FSyncResult := Unassigned;
            if Assigned(FOnSyncExec) then FOnSyncExec( Engine, SyncExecute )
                                     else SyncExecute;
            Result := FSyncResult;
            if IsVariable(v) then Variables.Value[v] := strtoint(FSyncPars[0]);

         end else if FuncIs( 'popupbox', 1, 5 ) then begin
            s := ParS( 0, '' );
            t := ParS( 1, 'Hamster-Script' );
            i := ParI( 2, MB_OK or MB_ICONINFORMATION );
            j := ParI( 3, 10 );
            k := ParI( 4, IDOK );
            FSyncCmd := 'popupbox';
            FSyncPars.Clear;
            FSyncPars.Add( s );
            FSyncPars.Add( t );
            FSyncPars.Add( inttostr(i) );
            FSyncPars.Add( inttostr(j) );
            FSyncPars.Add( inttostr(k) );
            FSyncResult := Unassigned;
            if Assigned(FOnSyncExec) then FOnSyncExec( Engine, SyncExecute )
                                     else SyncExecute;
            Result := FSyncResult

         end else if FuncIs( 'inputpw', 1, 4 ) then begin
            s := ParS( 0, '' );
            t := ParS( 1, 'Hamster-Script' );
            u := ParS( 2, '' );
            v := ParX( 3, '?' );
            FSyncCmd := 'inputpw';
            FSyncPars.Clear;
            FSyncPars.Add( t );
            FSyncPars.Add( s );
            FSyncPars.Add( u );
            FSyncResult := Unassigned;
            if Assigned(FOnSyncExec) then FOnSyncExec( Engine, SyncExecute )
                                     else SyncExecute;
            Result := FSyncResult;
            if IsVariable(v) then Variables.Value[v] := strtoint(FSyncPars[0]);

         end else if FuncIs( 'listbox', 2, 5 ) then begin
            Result := '';
            i := ParI( 0, -1 ); // list
            if not FHscLists.ListExists(i) then begin
               Error( HSCERR_INVALIDEXPRESSION, 'List expected as first parameter!' );
            end else begin
               s := ParS( 1, '' ); // prompt
               t := ParS( 2, 'Hamster-Script' ); // caption
               u := inttostr( ParI( 3, -1 ) ); // default-index
               v := ParX( 4, '?' ); // code-var 
               FSyncCmd := 'listbox';
               FSyncPars.Clear;
               FSyncPars.Add( t );
               FSyncPars.Add( s );
               FSyncPars.Add( u );
               FSyncPars.Add( FHscLists.List[ ParI(0,-1) ].Text );
               FSyncResult := Unassigned;
               if Assigned(FOnSyncExec) then FOnSyncExec( Engine, SyncExecute )
                                        else SyncExecute;
               Result := FSyncResult;
               if IsVariable(v) then Variables.Value[v] := strtoint(FSyncPars[0]);
            end;

         end else if FuncIs( 'gosub', 1, 1 ) then begin
            s := ParX( 0, '?' );
            i := IndexOfLabel( s );
            if IsIdentifier(s) and (i>=0) then begin
               IPtrPush;
               FNextPos := i;
               Result := ExecuteNextPos;
               IPtrPop;
            end else Error( HSCERR_LABELNOTFOUND, 'Label not found: ' + s );

         end else if FuncIs( 'errcatch', 0, 1 ) then begin
            if FuncPars.Count>0 then begin
               FErrCatch  := ( ParI( 0, 0 ) <> 0 );
               FErrNum    := HSCERR_NOERROR;
               FErrMsg    := 'No error';
               FErrModule := '';
               FErrLineNo := -1;
               FErrLine   := '';
               FErrSender := '';
            end;
            Result := Integer( FErrCatch );
         end else if FuncIs( 'errnum', 0, 0 ) then begin
            Result := FErrNum;
         end else if FuncIs( 'errmsg', 0, 0 ) then begin
            Result := FErrMsg;
         end else if FuncIs( 'errmodule', 0, 0 ) then begin
            Result := FErrModule;
         end else if FuncIs( 'errlineno', 0, 0 ) then begin
            Result := FErrLineNo;
         end else if FuncIs( 'errline', 0, 0 ) then begin
            Result := FErrLine;
         end else if FuncIs( 'errsender', 0, 0 ) then begin
            Result := FErrSender;

         // JAWO Params 4.10.2000
         end else if FuncIs( 'runscript', 1, 3 ) then begin
            s := ParS( 0, '' ); // script
            t := ParS( 1, '' ); // parameters
            j := ParI( 2, 1 );  // waitflag (default=wait)
            Result := StartNewScript ( s, t, (j<>0) )
         end else if FuncIs( 'paramcount', 0, 0 ) then begin
            Result := FHscLists.ParamCount;
         end else if FuncIs( 'paramstr', 1, 1 ) then begin
            Result := FHscLists.ListItem[ FHscLists.FParam, ParI(0,-1) ];

         end else if FuncIs( 'xcounter', 1, 1 ) then begin
            j := ParI(0, -1);
            If (j >= 0) and (j <= 9)
               then Result := GetCounter ( CntCustomValues[j] )
               else Result := 0
         end else if FuncIs( 'clearxcounter', 1, 2 ) then begin
            j := ParI(0, -1);
            k := ParI(1, j);
            If (j >= 0) and (j <= 9) and (k >= 0) and (k <= 9) then begin
               For i := j to k do SetCounter ( CntCustomValues[i], 0 );
               Result := 0
            end else Result := -1;
         end else if FuncIs( 'setxcounter', 2, 2 ) then begin
            j := ParI(0, -1);
            If (j >= 0) and (j <= 9) then begin
               SetCounter ( CntCustomValues[j], ParI(1, 0) );
               Result := 0
            end else Result := -1;
         end else if FuncIs( 'incxcounter', 1, 2 ) then begin
            j := ParI(0, -1);
            If (j >= 0) and (j <= 9) then begin
               SetCounter ( CntCustomValues[j], CntCustomValues[j] + ParI(1, 1) );
               Result := 0
            end else Result := -1;
         end else if FuncIs( 'decxcounter', 1, 2 ) then begin
            j := ParI(0, -1);
            If (j >= 0) and (j <= 9) then begin
               SetCounter ( CntCustomValues[j], CntCustomValues[j] - ParI(1, 1) );
               Result := 0
            end else Result := -1;

         end else begin
            // invoke external extension
            if not( FuncNameOK ) then begin
               if Assigned ( FOnFunc ) then begin
                  Result := FOnFunc( Scriptline );
               end
            end;

            // invoke function within script
            if not( FuncNameOK ) then begin
               i := IndexOfSub( FuncName );
               if i<0 then begin
                  Error( HSCERR_LABELNOTFOUND, 'Sub not found: ' + FuncName );
                  exit;
               end else begin
                  // save old and prepare new execution state
                  FuncNameOK  := True;
                  IPtrPush;
                  FNextPos := i + 1;
                  Trace( i, FScript[i] );

                  // eval and save transfer-vars in current var-context
                  // VTransfer := VarArrayCreate( [ 0, FuncPars.Count-1 ], varVariant );
                  // VSubVars  := VarArrayCreate( [ 0, FuncPars.Count-1 ], varOleStr  );
                  // VSubRefs  := VarArrayCreate( [ 0, FuncPars.Count-1 ], varBoolean );
                  SetLength( VTransfer, FuncPars.Count );
                  SetLength( VSubVars,  FuncPars.Count );
                  SetLength( VSubRefs,  FuncPars.Count );
                  s := FScript[i];
                  for i:=0 to FuncPars.Count-1 do VTransfer[ i ] := ParV(i,0);
                  i := Pos( '(', s );
                  if i<=0 then begin
                     s := '';
                  end else begin
                     System.Delete(s,1,i);
                     i := Pos( ')', s ); if i>0 then s:=copy(s,1,i-1);
                  end;
                  for i:=0 to FuncPars.Count-1 do begin
                     j := Pos( ',', s );
                     if j>0 then begin
                        t := TrimWhSpace( copy(s,1,j-1) );
                        System.Delete( s, 1, j );
                     end else begin
                        t := TrimWhSpace( s );
                        s := '';
                     end;
                     if t='' then begin
                        Error( HSCERR_SYNTAX, 'Too many parameters: '+FuncName ); exit;
                     end;
                     if t[1]='*' then VSubRefs[ i ] := True
                                 else VSubRefs[ i ] := False;
                     if t[1]='*' then System.Delete(t,1,1);
                     if IsVariable(t) then
                        VSubVars[ i ] := t
                     else begin
                        Error( HSCERR_SYNTAX, 'Variable expected: ' + t ); exit;
                     end;
                  end;
                  if TrimWhSpace(s)<>'' then begin
                     Error( HSCERR_SYNTAX, 'Missing parameters: '+FuncName ); exit;
                  end;

                  // enter new var-context and create transfer-vars
                  EnterContext( FuncName );
                  for i:=0 to FuncPars.Count-1 do begin
                     FVariables.DefValue( VSubVars[ i ], VTransfer[ i ] );
                  end;

                  // execute sub
                  Result := ExecuteNextPos;

                  // save resulting 'by reference'-vars
                  for i:=0 to FuncPars.Count-1 do begin
                     if VSubRefs[ i ] then begin
                        VTransfer[ i ] := FVariables.Value[ VSubVars[ i ] ];
                     end;
                  end;

                  // return to previous context and set 'by reference'-vars
                  LeaveContext;
                  for i:=0 to FuncPars.Count-1 do begin
                     if VSubRefs[ i ] then begin
                        s := ParX( i, '?' );
                        if IsVariable( s ) then
                           FVariables.Value[ s ] := VTransfer[ i ]
                        else
                           Error( HSCERR_SYNTAX, 'Variable (byref) expected: ' + s );
                     end;
                  end;

                  VTransfer := nil;
                  VSubVars  := nil;
                  VSubRefs  := nil;
                  IPtrPop;
               end;
            end;

            if not( FuncNameOK ) then begin
               Error( HSCERR_UNSUPPORTED, 'Unkown function: ' + FuncName );
            end;

            if not( AnyError ) and VarIsEmpty( Result ) then begin
               Error( HSCERR_UNSUPPORTED, 'Invalid function-syntax: ' + FuncName );
            end;
         end;
      except
         on E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
      end;

   finally
      FreeAndNil (ScriptLine)
   end;
end;

function THscEngine.ExecuteNextPos: Variant;
var ScriptLine: THscScriptLine;        // JAWO Quot.Marks 12.10.2000 (new variable)

   function SetNextPosToCorrespondingLine( IncDepth,
                                           ExitOn1,
                                           DecAndExitOn0 : String ): Boolean;
   var  Depth, Idx: Integer;
        s         : String;
   begin
      Result := False;
      Depth  := 1;
      Idx    := FNextPos;

      while Idx<FScript.Count do begin
         s := '/' + LineCmd( Idx ) + '/';
         if Pos( s, '/' + IncDepth + '/' )>0 then begin
            inc( Depth );
         end else begin
            if (Depth=1) and (ExitOn1<>'') then begin
               if Pos( s, '/' + ExitOn1 + '/' )>0 then begin
                  FNextPos := Idx + 1;
                  Result := True;
                  Trace( Idx, FScript[Idx] );
                  break;
               end;
            end;
            if Pos( s, '/' + DecAndExitOn0 + '/' )>0 then begin
               dec( Depth );
               if Depth=0 then begin
                  FNextPos := Idx + 1;
                  Result := True;
                  Trace( Idx, FScript[Idx] );
                  break;
               end;
            end;
         end;
         inc( Idx );
      end;

      if not Result then begin
         s := DecAndExitOn0;
         if ExitOn1<>'' then s := ExitOn1 + '/' + s;
         Error( HSCERR_LOOPNOTFOUND, 'Missing ' + s + ':' + FCurrLine );
      end;
   end;

var  pcLine     : PChar;
     i, Step, j, CountMultiLines: Integer;
     s, t: String;
     OK: Boolean;
     V1, V2, V3: Variant;
begin
   Result := 0;
   ScriptLine := THscScriptLine.Create (self);
   with ScriptLine do try
      FCurrPos := FNextPos;

      while FCurrPos<FScript.Count do begin
         if FTerminated then break;

         FNextPos := FCurrPos + 1;
         FCurrLine := TrimWSPEC( FScript[ FCurrPos ] );
         CountMultiLines := 0;

         // Concat multi-lines
         if FCurrLine<>'' then begin
            while FCurrPos+1<FScript.Count do begin
               if FCurrLine[length(FCurrLine)]<>'_' then break;
               FCurrLine[length(FCurrLine)] := ' ';
               inc( FCurrPos );
               inc( FNextPos );
               Inc( CountMultiLines );
               FCurrLine := FCurrLine + TrimWSPEC( FScript[ FCurrPos ] );
            end;
         end;

         pcLine := PChar( FCurrLine );
         SkipWSPC( pcLine );

         if (pcLine^<>#0) then Trace( FCurrPos, FCurrLine );

         WantSplitFunction( pcLine );
         if AnyError then break;

         FuncName   := LowerCase( FuncName );
         FuncNameOK := False;

         try
            if FuncName='' then begin
               FuncNameOK := True;

            end else if FuncIs( 'quit', 0, 1 ) then begin
               Result := ParI( 0, 0 );
               FTerminated := True;
               break;
            end else if FuncIs( 'error',  0, 999 ) then begin
               s := '';
               for i:=0 to FuncPars.Count-1 do s:=s+ParS(i,'');
               if s='' then s:='User defined error!';
               Error( HSCERR_USERDEFINED, s );
               if AnyError then begin
                  Result := 255;
                  FTerminated := True;
                  break;
               end;
            end else if FuncIs( 'assert', 1, 999 ) then begin
               if ParI( 0, 0 )=0 then begin
                  s := '';
                  for i:=1 to FuncPars.Count-1 do s:=s+ParS(i,'');
                  if s='' then s:='Assertion failed!';
                  Error( HSCERR_USERDEFINED, s );
                  Result := 255;
                  FTerminated := True;
                  break;
               end;

            end else if FuncIs( 'return', 0, 1 )
                     or FuncIs( 'endsub', 0, 0 ) then begin
               Result := ParV( 0, 0 );
               if VarIsEmpty(Result) and (FuncPars.Count>0) then
                  Error( HSCERR_INVALIDEXPRESSION, 'Undefined return-value' );
               break;

            end else if FuncIs( 'var', 1, 999 ) then begin
               for i:=0 to FuncPars.Count-1 do begin
                  s := ParX( i, '?' );
                  if IsVariable( s ) then begin
                     Variables.DefValue( s, Unassigned );
                  end else Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
               end;
            end else if FuncIs( 'entercontext', 1, 1 ) then begin
               EnterContext( ParS(0,'?') );
            end else if FuncIs( 'leavecontext', 0, 0 ) then begin
               LeaveContext;

            end else if FuncIs( 'label', 1, 1 ) then begin
               if not IsIdentifier( ParX(0,'?') ) then
                  Error( HSCERR_INVALID, 'Invalid label: ' + ParX(0,'') );
            end else if FuncIs( 'goto', 1, 1 ) then begin
               s := ParX( 0, '?' );
               i := IndexOfLabel( s );
               if IsIdentifier(s) and (i>=0) then FNextPos:=i
               else Error( HSCERR_LABELNOTFOUND, 'Label not found: ' + s );

            end else if FuncIs( 'if', 1, 1 ) then begin
               i := ParI( 0, 0 );
               if i=0 then begin // False
                  // continue after corresponding else/endif
                  SetNextPosToCorrespondingLine( 'if', 'else', 'endif' );
               end;
            end else if FuncIs( 'else', 0, 0 ) then begin
               // continue after corresponding endif
               SetNextPosToCorrespondingLine( 'if', '', 'endif' );
            end else if FuncIs( 'endif', 0, 0 ) then begin
               // ignore

            end else if FuncIs( 'do',     0, 0 )
                     or FuncIs( 'while',  1, 1 )
                     or FuncIs( 'repeat', 0, 0 ) then begin
               i := 1; // True
               if FuncPars.Count=1 then i := ParI( 0, 0 ); // while
               if i<>0 then begin
                  if FLoopStack.Count>999 then
                     Error( HSCERR_LOOPSTACK, 'Loop-stack overflow!' )
                  else
                     // FLoopStack.Add( Pointer( FCurrPos-CountMultiLines ) );
                     FLoopStack.push ( FCurrPos-CountMultiLines );
               end else begin
                  // continue after corresponding Loop
                  SetNextPosToCorrespondingLine( 'do/while/repeat', '', 'loop/endwhile/until' );
               end;
            end else if FuncIs( 'loop',     0, 0 )
                     or FuncIs( 'endwhile', 0, 0 )
                     or FuncIs( 'until',    1, 1 ) then begin
               if FLoopStack.Count=0 then begin
                  Error( HSCERR_LOOPSTACK, 'Loop/EndWhile/Until without Do/While/Until' );
               end else begin
                  i := 1; // True
                  if FuncPars.Count>0 then begin // until
                     i := ParI( 0, 0 );
                     if i=0 then i:=1 else i:=0;
                  end;
                  if i<>0 then begin
                     // FNextPos := LongInt( FLoopStack[ FLoopStack.Count-1 ] );
                     FNextPos := FLoopStack.Item.Line;
                     if LineCmd(FNextPos)='for' then Error( HSCERR_LOOPSTACK, 'Loop/EndWhile/Until without Do/While/Until' );
                  end;
                  // FLoopStack.Delete( FLoopStack.Count-1 );
                  FLoopStack.pop
               end;
{
            end else if FuncIs( 'for', 3, 4 ) then begin
               OK := True; // default: start new loop
               if FLoopStack.Count>0 then begin
                  if LongInt( FLoopStack[ FLoopStack.Count-1 ] ) = FCurrPos then begin
                     OK := False; // continue/end loop
                     FLoopStack.Delete( FLoopStack.Count-1 );
                  end;
               end;
               s := ParX( 0, '?' ); // loop-var
               j := ParI( 3, 1   ); // step
               if OK then begin // start new loop
                  if not IsVariable( s ) then begin
                     Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
                     break;
                  end;
                  V1 := ParI( 1, 0 ); // 1st limit
               end else begin // continue/end loop
                  V1 := HscVarToInt( Variables.Value[s] ) + j; // new loop-var
               end;
               Variables.Value[s] := V1; // init/update loop-var
               V2 := ParV( 2, 0 ); // 2nd limit
               if j>=0 then t:='>' else t:='<';
               V2 := HscCalculate( Self, V1, t, V2, j );
               if (j=ETYPE_INT) and (V2<>0) then OK:=False // break
                                            else OK:=True; // continue
               if OK then begin
                  if FLoopStack.Count>999 then
                     Error( HSCERR_LOOPSTACK, 'Loop-stack overflow!' )
                  else
                     FLoopStack.Add( Pointer( FCurrPos-CountMultiLines ) );
               end else begin
                  // continue after corresponding Loop
                  SetNextPosToCorrespondingLine( 'for', '', 'endfor' );
               end;
            end else if FuncIs( 'endfor', 0, 0 ) then begin
               if FLoopStack.Count=0 then begin
                  Error( HSCERR_LOOPSTACK, 'EndFor without For' );
               end else begin
                  FNextPos := LongInt( FLoopStack[ FLoopStack.Count-1 ] );
                  if LineCmd(FNextPos)<>'for' then begin
                     Error( HSCERR_LOOPSTACK, 'EndFor without For' )
                  end;
                  // NOT: FDoLoopStack.Delete( FDoLoopStack.Count-1 );
                  // 'for' needs stack-entry to recognize already running loop
               end;

 }
            end else if FuncIs( 'for', 3, 4 ) then begin
               s := ParX( 0, '?' ); // loop-var
               Step := ParI( 3, 1   ); // step
               if not IsVariable( s ) then begin
                  Error( HSCERR_SYNTAX, 'Variable expected: ' + s );
                  break;
               end;
               V1 := ParI( 1, 0 );
               Variables.Value[s] := V1;
               V2 := ParV( 2, 0 ); // 2nd limit
               if Step>=0 then t:='>' else t:='<';
               V3 := HscCalculate( Self, V1, t, V2, j );
               if (j=ETYPE_INT) and (V3<>0) then OK:=False // break
                                            else OK:=True; // continue
               if OK then begin
                  if FLoopStack.Count>999 then
                     Error( HSCERR_LOOPSTACK, 'Loop-stack overflow!' )
                  else
                     // FLoopStack.Add( Pointer( FCurrPos-CountMultiLines ) );
                     FLoopStack.push ( FCurrPos-CountMultiLines, FCurrPos+1, s, Step, V2 );
               end else begin
                  // continue after corresponding Loop
                  SetNextPosToCorrespondingLine( 'for', '', 'endfor' );
               end;
            end else if FuncIs( 'endfor', 0, 0 ) then begin

               if FLoopStack.Count=0 then begin
                  Error( HSCERR_LOOPSTACK, 'EndFor without For' );
               end else begin
                  // FNextPos := LongInt( FLoopStack[ FLoopStack.Count-1 ] );
                  With FLoopStack.Item do begin
                     i := Line;
                     s := VarName; j := IncVal; V2 := EndVal
                  end;
                  if LineCmd(i)<>'for' then begin
                     Error( HSCERR_LOOPSTACK, 'EndFor without For' )
                  end;
                  V1 := HscVarToInt( Variables.Value[s] ) + j; // new loop-var
                  Variables.Value[s] := V1; // init/update loop-var
                  if j >= 0 then t:='>' else t:='<';
                  V2 := HscCalculate( Self, V1, t, V2, j );
                  if (j=ETYPE_INT) and (V2<>0) then OK:=False // break
                                               else OK:=True; // continue
                  if OK
                     then FNextPos := FLoopStack.Item.NextLine
                     else FLoopStack.pop
               end;

            end else if FuncIs( 'continue', 0, 1 ) then begin
               if FLoopStack.Count=0 then begin
                  Error( HSCERR_LOOPSTACK, 'Continue without Do/While/Repeat' );
               end else begin
                  i := 1; // True
                  if FuncPars.Count>0 then i := ParI( 0, 0 );
                  if i<>0 then begin
                     // FNextPos := LongInt( FLoopStack[ FLoopStack.Count-1 ] );
                     FNextPos := FLoopStack.Item.Line;
                     // if LineCmd(FNextPos)<>'for' then FLoopStack.Delete( FLoopStack.Count-1 );
                     if LineCmd(FNextPos) = 'for' then begin
                        FNextPos := FLoopStack.Item.NextLine;
                        SetNextPosToCorrespondingLine( 'for', '', 'endfor' );
                        Dec(FNextpos) 
                     end else begin
                        FLoopStack.pop
                     end
                  end;
               end;
            end else if FuncIs( 'break', 0, 1 ) then begin
               if FLoopStack.Count=0 then begin
                  Error( HSCERR_LOOPSTACK, 'Break without Do/While/Repeat/For' );
               end else begin
                  i := 1; // True
                  if FuncPars.Count>0 then i := ParI( 0, 0 );
                  if i<>0 then begin
                     //FLoopStack.Delete( FLoopStack.Count-1 );
                     FLoopStack.pop;
                     // continue after corresponding Loop
                     SetNextPosToCorrespondingLine( 'do/while/repeat/for', '', 'loop/endwhile/until/endfor' );
                  end;
               end;

            end else if FuncIs( 'sleep', 1, 1 ) then begin
               i := ParI( 0, 0 );
               if FStopEvent=0 then Sleep( i )
                               else WaitForSingleObject( FStopEvent, i );

            end else if FuncIs( 'trace', 1, 1 ) then begin
               FTraceIsOn := ( ParI(0,0) <> 0 );

            end else if FuncIs( 'dump', 0, 0 ) then begin
               if Assigned( FOnPrint ) then begin
                  FOnPrint( Engine, 'dump follows:' );
                  FOnPrint( Engine, '.context: ' + Context );
                  for i:=0 to FVariables.Count-1 do
                     FOnPrint( Engine, '.var: ' + FVariables.Dump(i) );
                  for i:=0 to FHscLists.Count-1 do begin
                     if FHscLists.ListExists(i) then begin
                        FOnPrint( Engine, '.list: ' + inttostr(i)
                        + ' (' + inttostr(FHscLists.List[i].Count) + ' entries)' );
                     end;
                  end;
               end;

            end else if FuncIs( 'debug', 1, 999 ) then begin
               i := ParI( 0, 0 ); // level
               if FuncPars.Count=1 then begin
                  FDebugLevel := i;
               end else begin
                  if (i<=FDebugLevel) and (FDebugLevel>0) and Assigned(FOnPrint) then begin
                     s := ''; for i:=1 to FuncPars.Count-1 do s:=s+ParS(i,'');
                     FOnPrint( Engine, s );
                  end;
               end;

            end else begin
               if not( FuncNameOK or AnyError ) then begin
                  if VarIsEmpty( SolveFunction( FCurrLine ) ) then begin
                     if not AnyError then
                        Error( HSCERR_UNSUPPORTED, 'Unknown statement: ' + FuncName );
                  end;
               end;
            end;

         except
            on E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
         end;

         if AnyError then break;
         if FStopEvent<>0 then begin
            if WaitForSingleObject( FStopEvent, 0 )=WAIT_OBJECT_0 then begin
               Error( HSCERR_STOPPED, 'Script stopped.' );
               break;
            end;
         end;

         FCurrPos := FNextPos;
      end;

   finally
      NewsHistory.SaveToFile; { JW }
      FreeAndNil(Scriptline);
   end;
end;

function THscEngine.Execute: Variant;
var InitMods: TStringList;

   procedure PreProcess( FromLine, ToLine: Integer );
   var  LineNo, i: Integer;
        Cmd, Par : String;
        TS: TStringList;
   begin
      for LineNo:=FromLine to ToLine do begin
         FCurrPos  := LineNo;
         FCurrLine := FScript[LineNo];

         if copy( FCurrLine, 1, 2 ) = '#!' then begin

            Trace( FCurrPos, FCurrLine );

            Cmd := copy( FCurrLine, 3, 255 );
            Par := '';
            i := Pos( ' ', Cmd );
            if i>0 then begin Par:=copy(Cmd,i+1,255); Cmd:=copy(Cmd,1,i-1); end;
            Cmd := LowerCase( Cmd );
            Par := TrimWSPEC( Par );

            if (Cmd='hs2') and (Par='') then begin
               // ignore

            end else if (Cmd='trace') and (Par<>'') then begin
               FTraceIsOn := ( strtointdef( Par, 0 ) = 1 );

            end else if (Cmd='debug') and (Par<>'') then begin
               FDebugLevel := strtointdef( Par, 0 );

            end else if (Cmd='load') and (Par<>'') then begin
               if not( FScript.IsModuleLoaded( Par ) ) then begin
                  TS := TStringList.Create;
                  try
                     { # TGL # }
                     If Copy(Par, 2, 1)<>':' then begin
                        If Copy(Par, 1, 1) = '\' then Par := Copy( PATH_HSM, 1, 2) + Par
                                                 else Par := PATH_HSM + Par
                     end;
                     Par := ExpandFilename( Par );
                     if FileExists( Par ) then begin
                        TS.LoadFromFile( Par );
                        FScript.Add( 'quit {module-separator}' );
                        FScript.StartNewModule( Par );
                        i := FScript.Count;
                        FScript.AddLines( TS );
                        PreProcess( i, FScript.Count-1 );
                     end else begin
                        Error( HSCERR_INITIALIZEFAILED, 'Module not found: ' + Par );
                        break;
                     end;
                  finally
                     TS.Free;
                  end;
               end;

            end else if (Cmd='initialize') and (Par='') then begin
               InitMods.AddObject( FCurrLine, Pointer( LineNo+1 ) );

            end else begin
               Error( HSCERR_SYNTAX, 'Invalid pre-processor command: ' + FCurrLine );
               break;
            end;

         end;
      end;
   end;

var  LineNo, i: Integer;
     Module   : String;
begin
   Result := 255;

   try
      // initialize script-engine
      FVariables.Clear;
      FScheduler.Clear;
      FHscLists.Clear;
      FContexts.Clear;
      FLoopStack.Clear;
      FIPtrStack.Clear;
      FLastError := 0;
      if not(Assigned(FOnWarning)) then FOnWarning:=FOnPrint;
      FTraceIsOn  := False;
      FDebugLevel := 0;
      FTerminated := False;

      FErrNum     := HSCERR_NOERROR;
      FErrMsg     := 'No error';
      FErrModule  := '';
      FErrLineNo  := -1;
      FErrLine    := '';
      FErrSender  := '';
      FErrCatch   := False;

      // JAWO Params 4.10.2000 (new code)
      FHscLists.Clear;
      FHscLists.InitParams ( FHscLists.FParamString );

      // pre-process script (e.g. load modules)
      InitMods := TStringList.Create;
      PreProcess( 0, FScript.Count-1 );

      // initialize modules from bottom to top
      while InitMods.Count>0 do begin
         if AnyError then break;
         Module := InitMods[ InitMods.Count-1 ];
         LineNo := Integer( InitMods.Objects[ InitMods.Count-1 ] );

         FNextPos := LineNo;
         i := HscVarToInt( ExecuteNextPos );
         if i <> 0 then begin
            Error( HSCERR_INITIALIZEFAILED,
                   'Initialization of module ' + Module
                   + ' failed with error-code ' + inttostr(i) + '!' );
            break;
         end;

         InitMods.Delete( InitMods.Count-1 );
      end;
      InitMods.Free;

      // run main script
      if not( AnyError ) then begin
         FCurrPos    := 0;
         FNextPos    := 0;
         Result := ExecuteNextPos;
      end;
   except
      on E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
   end;
end;

                                  // JAWO Params 4.10.2000 (new function)
procedure THscEngine.InitParams (ParamString : String);
begin
   try
      if Assigned ( FHscLists ) then begin
         FHscLists.FParamString := ParamString;
      end;
   except
      On E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
   end;
end;

function THscEngine.ExecuteFromList( const ScriptName: String;
                                     const AScript: TStringList ): Variant;
begin
   Result := Unassigned;

   try
      FScript.Clear;
      FScript.StartNewModule( ScriptName );
      FScript.AddLines( AScript );
      Result := Execute;
   except
      on E:Exception do Error( HSCERR_ENGINEEXCEPTION, 'Exception: ' + E.Message );
   end;
end;

constructor THscEngine.Create;
begin
   inherited Create( Self );
   FVariables  := THscVariables.Create( Engine );
   FScheduler  := THscScheduler.Create( Engine );
   FHscLists   := THscLists.Create( Engine );
   FContexts   := TStringList.Create;
   // FLoopStack  := TList.Create;
   FLoopStack  := TLoopStack.Create;
   FIPtrStack  := TList.Create;
   FScript     := THscScript.Create( Self );
   FSyncPars   := TStringList.Create;
   FStopEvent  := 0;
   FContextID  := 0;
   GroupsInUse := TStringList.Create;
   ListsInUse := TStringList.Create
end;

destructor THscEngine.Destroy;
begin
   if Assigned( FSyncPars  ) then FSyncPars.Free;
   if Assigned( FScript    ) then FScript.Free;
   if Assigned( FContexts  ) then FContexts.Free;
   if Assigned( FHscLists  ) then FHscLists.Free;
   if Assigned( FScheduler ) then FScheduler.Free;
   if Assigned( FVariables ) then FVariables.Free;
   if Assigned( FLoopStack ) then FLoopStack.Free;
   if Assigned( FIPtrStack ) then FIPtrStack.Free;
   if Assigned( GroupsInUse ) then GroupsInUse.Free;
   if Assigned( ListsInUse ) then ListsInUse.Free;
   inherited Destroy;
end;

procedure THscEngine.AddHandle(const Typ: THscHandleTyp; handle: THandle);
begin
  case typ of
     kHscEngine_GroupHandle: GroupsInUse.Add(IntToStr(handle));
     kHscEngine_ListHandle : ListsInUse.Add(IntToStr(handle));
  end
end;

procedure THscEngine.RemoveHandle(const typ: THscHandleTyp; handle: THandle);
begin
   case typ of
      kHscEngine_GroupHandle: If GroupsInUse.IndexOf(IntToStr(handle))>=0
         then GroupsInUse.Delete(GroupsInUse.IndexOf(IntToStr(handle)));
      kHscEngine_ListHandle : If ListsInUse.IndexOf(IntToStr(handle))>=0
         then ListsInUse.Delete(ListsInUse.IndexOf(IntToStr(handle)))
   end
end;

procedure THscEngine.GarbageCollection;
var k,j: Integer; s: String;
begin
   s := fScript.ModName[0];
   if GroupsInUse.Count>0 then begin
      Log( LOGID_WARN, '{'+s+'} ' + TrGlF(kLog, 'Warning.Script.UnClosed.Newsgroups',
                      '%s newsgroup(s) were opened in script, but not closed!',
                      IntToStr(Engine.GroupsInUse.Count)));
      while GroupsInUse.Count>0 do begin
         val(GroupsInUse[0],k,j);
         GroupsInUse.Delete(0);
         If (k>=0) AND (J=0) then begin
            Log( LOGID_WARN,'{'+s+'} !> '+ TrGlF (kLog, 'Warning.Script.UnClosed.Newsgroup.closed',
                          'Group %s was closed automaticly.', ArticleBase.Name[k]) );
            ArticleBase.Close( k );
         end;
      end;
   end;
   // Eine Liste ist zur Laufzeit des gabage collectors noch von der Engine
   // geffnet und darf nicht geschlossen und gemeldet werden.
   if ListsInUse.Count>1 then begin
      Log( LOGID_WARN, '{'+s+'} ' + TrGlF(kLog, 'Warning.Script.UnClosed.Lists',
                      '%s List(s) were opened in script, but not closed!',
                      IntToStr(Engine.ListsInUse.Count-1)));
      while ListsInUse.Count>1 do begin
         val(ListsInUse[0],k,j);
         ListsInUse.Delete(0);
         If (k>=0) AND (J=0) then begin
            FHscLists.ListFree(k);
         end
      end
   end
end;

{ TLoopStack }

procedure TLoopStack.Clear;
begin
   While Count>0 do pop;
end;

function TLoopStack.Count: Integer;
begin
   Result := FList.Count
end;

constructor TLoopStack.Create;
begin
   inherited;
   FList := TList.Create;
end;

destructor TLoopStack.destroy;
begin
   inherited;
   Clear;
   FList.Free
end;

function TLoopStack.Item: TLoopStackItem;
begin
   If Count=0
      then Result := NIL
      else Result := TLoopStackItem(FList[Count-1])
end;

procedure TLoopStack.pop;
begin
   TLoopStackItem(FList[Count-1]).Free;
   FList.Delete(Count-1)
end;

procedure TLoopStack.push(const MyLine: Integer;
  Const myNextLine: Integer = 0;
  const MyVarname: String = '';
  const myIncVal: Integer = 0;
  Const myEndVal: Integer = 0);
Var Item: TLoopStackItem;
begin
   Item := TLoopStackItem.Create;
   With Item do begin
       Line := MyLine; NextLine := MyNextline;
       VarName := MyVarname; IncVal := myIncVal; EndVal := myEndVal
   end;
   FList.Add(Pointer(Item))
end;

end.
