// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cIPAccess;

interface

uses SysUtils, Classes, uTools, Global, cSyncObjects;

const
  IPACC_SCOPE_NONE  = $0000;
  IPACC_SCOPE_NNTP  = $0001;
  IPACC_SCOPE_POP3  = $0010;
  IPACC_SCOPE_SMTP  = $0100;
  IPACC_SCOPE_RECO  = $1000;
  IPACC_SCOPE_IMAP  = $2000;
  IPACC_SCOPE_MAIL  = $2110;
  IPACC_SCOPE_ALL   = $FFFF;

  IPACC_ACCESS_NA   = $0000;
  IPACC_ACCESS_RO   = $0001;
  IPACC_ACCESS_WO   = $0002;
  IPACC_ACCESS_RW   = $0003;
  IPACC_ACCESS_ALL  = $FFFF;

  DefaultIPAccess_1 = 'ALL,NA,LOCAL,127.0.0.1';
  DefaultIPAccess_2 = 'ALL,RW,127.0.0.1';
  DefaultIPAccess_3 = 'ALL,RW,192.168.0.0,192.168.255.255';
  DefaultIPAccess_4 = 'ALL,NA,0.0.0.0,255.255.255.255';
  DefaultIPAccess = DefaultIPAccess_1 + #13#10 + DefaultIPAccess_2 + #13#10 + DefaultIPAccess_3 + #13#10 + DefaultIPAccess_4 + #13#10;

type
  TIPAccessEntry = ^RIPAccessEntry;
  RIPAccessEntry = record
    IpMin, IpMax, Scope, Access: LongInt;
  end;

  TIPAccessCheck = class
    private
      FLock: TReaderWriterLock;
      FReload: Boolean;
      IpAccList: TList;
      Replacement_for_LocalIPs: LongInt;
      procedure LoadRecs;
      procedure FreeRecs;
    public
      property WantReload: Boolean write FReload;
      function GetAccess( nChkIP: LongInt; ChkScope: LongInt ): LongInt;
      procedure reload; //JW //Reload IPaccess
      constructor Create;
      destructor Destroy; override;
  end;

Var
  IPAccessCheck: TIPAccessCheck = nil;

Function ConvertLineToEntry ( DefLine: String; Var Ip1, Ip2, Sc, Ac: Longint;
   Var EmptyLine, Local: boolean ): Boolean;
Function IP2Text ( Const IP: Longint ) : String;
Function IsValidIP ( Const s: String ): Boolean;
function GetLocalIPs: String;

function IPAccessScopeStr( Scope: LongInt ): String;
function IPAccessAccessStr( Access: LongInt ): String;

implementation

uses WinSock, uWinSock, Config, cLogFile;

function TIPAccessCheck.GetAccess( nChkIP: LongInt; ChkScope: LongInt ): LongInt;
var  hChkIP, i: Integer;
begin
     Result := IPACC_ACCESS_NA;
     If FReload then Reload;

     FLock.BeginRead;
     try
        hChkIP := ntohl( nChkIP );

        if (Replacement_for_LocalIPs<>0) and (hChkIP<>$7f000001) then begin
           if IsLocalHost( nChkIp ) then begin
              Log( LOGID_INFO, 'IP-Access: Treat local IP ' + hAddrToStr(hChkIP)
                             + ' like ' + hAddrToStr(Replacement_for_LocalIPs) );
              hChkIP := Replacement_for_LocalIPs;
           end;
        end;

        for i:=0 to IpAccList.Count-1 do begin
           with TIPAccessEntry(IpAccList[i])^ do begin
              if (ChkScope and Scope)=ChkScope then begin
                 if (inttohex(hChkIP,8) >= inttohex(IpMin,8)) and
                    (inttohex(hChkIP,8) <= inttohex(IpMax,8)) then begin // signed!
                    Result := Access;
                    break;
                 end;
              end;
           end;
        end;
   finally
      FLock.EndRead;
   end;
end;

{JW} {Reload IPaccess}
procedure TIPAccessCheck.reload;
begin
   FLock.BeginWrite;
   try
      FreeRecs;
      LoadRecs;
      FReload := False;
   finally
      FLock.EndWrite;
   end;
end;
{/JW}

procedure TIPAccessCheck.FreeRecs;
var  i: Integer;
begin
   for i:=0 to IpAccList.Count-1 do begin
      FreeMem( TIPAccessEntry(IpAccList[i]), sizeof(RIPAccessEntry) );
      IpAccList[i] := nil;
   end;
   IpAccList.Clear;
end;

function ConvertLineToEntry ( DefLine: String; Var Ip1, Ip2, Sc, Ac: Longint;
   Var EmptyLine, Local: boolean ): Boolean;
var  j: Integer; P: TParser; s, s1, s2: String;
begin
   Result := true;
   EmptyLine := true;
   Local := false;
   Ip1 := 0; Ip2 := 0; Sc := 0; Ac := 0;

   j := Pos(';',DefLine); if j>0 then DefLine:=copy(DefLine,1,j-1);
   j := Pos('#',DefLine); if j>0 then DefLine:=copy(DefLine,1,j-1);
   DefLine := TrimWhSpace( DefLine );
   if DefLine='' then exit;

   Result := False;
   EmptyLine := false;
   P := TParser.Create;
   try
      P.Parse( DefLine, ',' ); // (ALL|NNTP|POP3|SMTP),(0|1|2|3),ipfrom,ipto

      s := UpperCase( TrimWhSpace( P.sPart( 0, '' ) ) ); // Scope
      if s='' then Exit;
      if s='IMAP' then sc:=IPACC_SCOPE_IMAP //IMAP
      else if s='NNTP' then sc:=IPACC_SCOPE_NNTP
      else if s='POP3' then sc:=IPACC_SCOPE_POP3
      else if s='SMTP' then sc:=IPACC_SCOPE_SMTP
      else if s='MAIL' then sc:=IPACC_SCOPE_MAIL
      else if s='RECO' then sc:=IPACC_SCOPE_RECO
      else if s='IMAP' then sc:=IPACC_SCOPE_IMAP
      else if s='ALL'  then sc:=IPACC_SCOPE_ALL
      else Exit;

      s := UpperCase( TrimWhSpace( P.sPart( 1, '' ) ) ); // Access
      If s='NA' then ac := IPACC_ACCESS_NA
      else if s='RO' then ac:=IPACC_ACCESS_RO
      else if s='WO' then ac:=IPACC_ACCESS_WO
      else if s='RW' then ac:=IPACC_ACCESS_RW
      else if s='ALL' then ac:=IPACC_ACCESS_ALL
      else Exit;

      s1 := TrimWhSpace( P.sPart( 2, '' ) ); // Ip1
      s2 := TrimWhSpace( P.sPart( 3, s1 ) ); // Ip2 (default=Ip1)
      if (s1='') or (s2='') then exit;

      ip1 := 0;
      if UpperCase(s1)<>'LOCAL' then begin
         P.Parse( s1, '.' );
         s := P.sPart( 0, '' );  ip1 := ip1 or ( (strtointdef(s,0) and $FF) shl 24 );
         s := P.sPart( 1, '' );  ip1 := ip1 or ( (strtointdef(s,0) and $FF) shl 16 );
         s := P.sPart( 2, '' );  ip1 := ip1 or ( (strtointdef(s,0) and $FF) shl  8 );
         s := P.sPart( 3, '' );  ip1 := ip1 or ( (strtointdef(s,0) and $FF)        );
      end;

      ip2 := 0;
      P.Parse( s2, '.' );
      s := P.sPart( 0, '' );  ip2 := ip2 or ( (strtointdef(s,0) and $FF) shl 24 );
      s := P.sPart( 1, '' );  ip2 := ip2 or ( (strtointdef(s,0) and $FF) shl 16 );
      s := P.sPart( 2, '' );  ip2 := ip2 or ( (strtointdef(s,0) and $FF) shl  8 );
      s := P.sPart( 3, '' );  ip2 := ip2 or ( (strtointdef(s,0) and $FF)        );
   finally
      P.Free
   end;

   if UpperCase(s1)='LOCAL' then begin
      Local := true;
      Result := True;
      exit;
   end;

   if inttohex(ip1,8)>inttohex(ip2,8) then exit;

   Result := True;
end;

Function IP2Text ( Const IP: Longint ) : String;
begin
   Result := IntToStr((ip shr 24) and $FF)
             + '.' + IntToStr((ip shr 16) and $FF)
             + '.' + IntToStr((ip shr  8) and $FF)
             + '.' + IntToStr( ip and $FF )
end;

Function IsValidIP ( Const s: String ): Boolean;
Var i, z, tmp: Integer; c: Char; bSecond, bError: Boolean;
begin
   tmp := 0; z := 0;
   bSecond := false; bError := false;
   For i := 1 to Length(s) do begin
      c := s[i];
      If bSecond then begin
         If (c >= '0') and (c <= '9') then begin
            tmp := tmp*10 + Ord(c)-48;
            If tmp > 255 then bError := true
         end else
         If c = '.' then bSecond := false
         else bError := true
      end else begin
         If (c >= '0') and (c <= '9') then begin
            tmp := Ord(c)-48;
            bSecond := true;
            Inc(z)
         end else bError := true
      end;
      If bError then break
   end;
   Result := (Not bError) and bSecond and (z=4)
end;

procedure TIPAccessCheck.LoadRecs;
var TS: TStringList;
    i : Integer;

  procedure AddEntry( IpMin, IpMax, Scope, Access: LongInt );
  var  pIAE: TIPAccessEntry;
  begin
       GetMem( pIAE, sizeof(RIPAccessEntry) );
       pIAE^.IpMin  := IpMin;
       pIAE^.IpMax  := IpMax;
       pIAE^.Scope  := Scope;
       pIAE^.Access := Access;
       IpAccList.Add( pIAE );
  end;

  function AddLine( DefLine: String ): Boolean;
  var  ip1, ip2, sc, ac: LongInt;
       empty, local: Boolean;
  begin
     Result := ConvertLineToEntry ( DefLine, ip1, ip2, sc, ac, empty, local );
     If Result then begin
        If empty then
        else
        If local then Replacement_for_LocalIPs := ip2
        else
        AddEntry( ip1, ip2, sc, ac )
     end
  end;

begin
     Replacement_for_LocalIPs := 0;

     if FileExists2( PATH_BASE + CFGFILE_IPACCESS ) then begin
        TS := TStringList.Create;
        TS.LoadFromFile( PATH_BASE + CFGFILE_IPACCESS );
        for i:=0 to TS.Count-1 do begin
           if not AddLine( TS[i] ) then begin
              Log( LOGID_WARN, 'Invalid line in IPAccess.hst: ' + TS[i] );
           end;
        end;
        TS.Free;
     end;

     if IpAccList.Count=0 then begin
        AddLine( DefaultIPAccess_1 );
        AddLine( DefaultIPAccess_2 );
        AddLine( DefaultIPAccess_3 );
        AddLine( DefaultIPAccess_4 );
     end;
end;

constructor TIPAccessCheck.Create;
begin
   inherited Create;
   FLock := TReaderWriterLock.Create;
   IpAccList := TList.Create;
   Replacement_for_LocalIPs := 0;
   WantReload := True;
   Reload;
end;

destructor TIPAccessCheck.Destroy;
begin
   FLock.BeginWrite;  
   FreeRecs;
   IpAccList.Free;
   FLock.Free;
   inherited Destroy;
end;

function GetLocalIPs: String;
type PPInAddr= ^PInAddr;
var
  wsaData  : TWSAData;
  HostInfo : PHostEnt;
  HostName : Array[0..255] of Char;
  Addr     : PPInAddr;
begin
  Result:='';
  if WSAStartup($0102, wsaData) <> 0 then
    Exit;
  try
    if GetHostName(HostName, SizeOf(HostName)) <> 0 then
      Exit;
    HostInfo:= GetHostByName(HostName);
    if HostInfo=nil then
      Exit;
    Addr:=Pointer(HostInfo^.h_addr_list);
    if (Addr=nil) or (Addr^=nil) then
      Exit;
    Result:=StrPas(inet_ntoa(Addr^^))+#13#10;
    inc(Addr);
    while Addr^ <> nil do begin
      Result:=Result+StrPas(inet_ntoa(Addr^^))+#13#10;
      inc(Addr);
    end;
  finally
    WSACleanup;
  end;
end;

function IPAccessScopeStr( Scope: LongInt ): String;
begin
   case Scope of
      IPACC_SCOPE_NONE: Result := 'NONE';
      IPACC_SCOPE_NNTP: Result := 'NNTP';
      IPACC_SCOPE_POP3: Result := 'POP3';
      IPACC_SCOPE_IMAP: Result := 'IMAP'; //IMAP
      IPACC_SCOPE_SMTP: Result := 'SMTP';
      IPACC_SCOPE_MAIL: Result := 'MAIL';
      IPACC_SCOPE_RECO: Result := 'RECO';
      IPACC_SCOPE_ALL : Result := 'ALL';
      else              Result := '?'+inttohex(Scope,8)+'?';
   end;
end;

function IPAccessAccessStr( Access: LongInt ): String;
begin
   case Access of
      IPACC_ACCESS_NA : Result := 'NA';
      IPACC_ACCESS_RO : Result := 'RO';
      IPACC_ACCESS_WO : Result := 'WO';
      IPACC_ACCESS_RW : Result := 'RW';
      IPACC_ACCESS_ALL: Result := 'ALL';
      else              Result := '?'+inttohex(Access,8)+'?';
   end;
end;

initialization
   IPAccessCheck := TIPAccessCheck.Create;

finalization
   IPAccessCheck.Free;

end.


