// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cPasswordFile;

interface

uses SysUtils, Windows, Classes;

type
  TPasswordFile = class
    private
      FLock: TRTLCriticalSection;
      FileList: TStringList;
      TempList: TStringList;
    public
      procedure SavePassword( ToFile: Boolean; Identifier: String;
                              Username, Password: String );
      function  LoadPassword( Identifier: String;
                              var Username, Password: String ): Boolean;
      Procedure PasswordInfo ( Const ID, u1, p1: String; Var u2, p2: String);
      function  UsePassword ( Identifier: String;
                              var Username, Password: String ): Boolean;
      procedure Reload; //HSR //PassRL
      constructor Create;
      destructor Destroy; override;
  end;

Var
  PasswordFile: TPasswordFile = nil;

implementation

Uses Dialogs, Global, uTools, uCRC32, uBlowfish, dInput, cStdForm, cLogfile;

const
   PWCB_PC_OS_USER = 0;
   PWCB_PC_OS      = 1;
   PWCB_NONE       = 2;

function GetKeyword: String;
var  p: array[0..256] of char;
     dw: DWord;
     s: String;
     V: TOSVersionInfoA;
begin
     s := '';

     if Def_PasswordCodeBase in [PWCB_PC_OS_USER, PWCB_PC_OS] then begin
        dw := 256;
        if GetComputerName( p, dw ) then s := s + String( p );
     end else begin
        s := s + 'NoName';
     end;

     if Def_PasswordCodeBase in [PWCB_PC_OS_USER, PWCB_PC_OS] then begin
        V.dwOSVersionInfoSize := sizeof(V);
        if GetVersionEx( V ) then begin
           s := s + inttohex(V.dwBuildNumber,1) + inttohex(V.dwMinorVersion,1)
                  + inttohex(V.dwPlatformId,1)  + inttohex(V.dwMajorVersion,1);
        end;
     end else begin
        s := s + 'NoVer';
     end;

     if Def_PasswordCodeBase in [PWCB_PC_OS_USER] then begin
        dw := 256;
        if GetUserName( p, dw ) then s := s + String( p );
     end else begin
        s := s + 'NoName';
     end;

     s := s + 'Hamster';
     Result := s;

end;

function XRadix( l : LongInt ): String;
const ALPHABET = 'AbCd6efG5hij4kLmn3opqR2sTuv1Wxyz';
var  p : LongInt;
begin
     Result := '';
     repeat
        p := l and $1F;
        l := l shr 5;
        Result := Result + copy(ALPHABET,p+1,1);
     until l=0;
     while length(Result)<7 do Result:='i'+Result;
end;

procedure PWEncode( var buf: String );
var  i: Integer;
     s: String;
begin
     s := HamBlowfishEncipher( GetKeyword, buf );
     buf := '';
     for i:=1 to length(s) do begin
        buf := buf + inttohex( ord(s[i]), 2 );
     end;
end;

procedure PWDecode( var buf: String );
var  i: Integer;
     s: String;
begin
     s := '';
     for i:=1 to length(buf) div 2 do begin
        s := s + chr( strtoint( '$' + copy(buf,i*2-1,2) ) );
     end;
     buf := HamBlowfishDecipher( GetKeyword, s );
end;

function  TPasswordFile.UsePassword( Identifier: String;
                                     var Username, Password: String ): Boolean;
var  u, p: String;
     Save: Boolean;
begin
     Result   := False;
     Username := '';
     Password := '';

     u := '';
     p := '';
     if LoadPassword( Identifier, u, p ) then begin
        Save := False;
        if (u='?') then begin
           if not InputDlgStr( PChar(Identifier),
               PChar(TrGl(kGlobal, 'Username', 'Username')+':'), u, 0{HlpInputUserPassword} ) then exit;
           Save := True;
        end;
        if (p='?') then begin
           if not InputDlgPwd( PChar(Identifier),
               PChar(TrGl(kGlobal, 'Password', 'Password')+':'), p, 0{HlpInputUserPassword} ) then exit;
           Save := True;
        end;
        if Save then SavePassword( False, Identifier, u, p ); // fr akt. Session merken

        Result   := True;
        Username := u;
        Password := p;
     end;
end;

Procedure TPasswordFile.PasswordInfo ( Const ID, u1, p1: String; Var u2, p2: String);
begin
   If U1 = '' then U2 := TrGl(kGlobal, 'PW.None','{NONE}')
   else If U1 = '?' then U2 := TrGl(kGlobal, 'PW.Ask','{ASK}')
   else begin
      If ID[1]='$' then U2 := '"'+U1+'"'
                   else U2 := TrGl(kGlobal, 'PW.Set','{SET}')
   end;

   If P1 = '' then P2 := TrGl(kGlobal, 'PW.None','{NONE}')
   else if P1 = '?' then P2 := TrGl(kGlobal, 'PW.Ask','{ASK}')
   else P2 := TrGl(kGlobal, 'PW.Set','{SET}')
end;

function  TPasswordFile.LoadPassword( Identifier: String;
                                      var Username, Password: String ): Boolean;
var  AccountKey: String;
     AccountRec: String;
     j         : Integer;
begin
     Log( LOGID_DEBUG, 'Loading username/password for ' + Identifier );

     Result   := False;
     Username := '';
     Password := '';
     AccountKey := XRadix( StrToCRC32( Identifier ) );

     EnterCriticalSection( FLock );
     try
        AccountRec := TempList.Values[ AccountKey ];
        if AccountRec='' then begin
           Log( LOGID_DEBUG, '"' + Identifier + '" not in TempList, trying FileList.' );
           AccountRec := FileList.Values[ AccountKey ];
        end;
     finally
        LeaveCriticalSection( FLock );
     end;

     if AccountRec='' then begin
        Log( LOGID_DEBUG, '"' + Identifier + '" not in FileList, assume "not needed".' );
        Result := True; // 1.3.17.1
        exit;
     end;

     PWDecode( AccountRec );
     if AccountRec='' then begin
        Log( LOGID_DEBUG, 'PWDecode for "' + Identifier + '" failed.' );
        Username := '?';
        Password := '?';
        exit;
     end;

     j := Pos( #9, AccountRec );
     if j=0 then begin
        Log( LOGID_DEBUG, 'PWDecode for "' + Identifier + '" failed (no ID).' );
        Username := '?';
        Password := '?';
        exit;
     end;
     System.Delete( AccountRec, 1, j );

     j := Pos( #9, AccountRec );
     if j=0 then begin
        Log( LOGID_DEBUG, 'PWDecode for "' + Identifier + '" failed (no U/PW).' );
        Username := '?';
        Password := '?';
        exit;
     end;

     Username := copy( AccountRec, 1, j-1 );
     Password := copy( AccountRec, j+1, Length(AccountRec)-j );
     Result := True;
end;

procedure TPasswordFile.SavePassword( ToFile: Boolean; Identifier: String;
                                      Username, Password: String );
var  AccountKey: String;
     AccountRec: String;
begin
     AccountKey := XRadix( StrToCRC32( Identifier ) );
     AccountRec := Identifier + #9 + Username + #9 + Password;
     PWEncode( AccountRec );

     EnterCriticalSection( FLock );
     try
        if ToFile then begin
           FileList.Values[ AccountKey ] := AccountRec;
           FileList.SaveToFile( PATH_BASE + CFGFILE_PASS );
        end else begin
           TempList.Values[ AccountKey ] := AccountRec;
        end;
     finally
        LeaveCriticalSection( FLock );
     end;
end;

constructor TPasswordFile.Create;
begin
     inherited Create;
     InitializeCriticalSection( FLock );
     FileList := TStringList.Create;
     TempList := TStringList.Create;
     if FileExists2( PATH_BASE + CFGFILE_PASS ) then begin
        FileList.LoadFromFile( PATH_BASE + CFGFILE_PASS );
     end;
end;

destructor TPasswordFile.Destroy;
begin
     FileList.Free;
     TempList.Free;
     DeleteCriticalSection( FLock );
     inherited Destroy;
end;

procedure TPasswordFile.Reload;
begin
  EnterCriticalSection( FLock );
  try
    TempList.Clear;
    FileList.Clear;
    if FileExists2( PATH_BASE + CFGFILE_PASS ) then begin
      FileList.LoadFromFile( PATH_BASE + CFGFILE_PASS );
    end;
  finally
    LeaveCriticalSection( FLock );
  end;
end;

end.
