// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cResControl; // ressource controller

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows;

const
   RESID_HamsterGroup = 1;
   RESID_HscListEntry = 2;
   RESID_WinApiHandle = 3;
   RESID_GlobalMemory = 4;
   RESID_DynamicDll   = 5;
   RESID_StrNew       = 6;
   RESID_DelphiObject = 7;

type
   TResAbstract = class
      protected
         ResID, Entity: LongWord;
      public
         function AutoDestroy: String; virtual; abstract;
         constructor Create( aResID, aEntity: LongWord );
   end;

   TResHamsterGroup = class( TResAbstract )
      // x := ArticleBase.Open( s );
      // RC.Add( TRes_HamsterGroup.Create(x) );
      // ...
      // RC.Remove( RESID_HamsterGroup, x );
      // ArticleBase.Close( x );
      public
         function AutoDestroy: String; override;
         constructor Create( aGroupHandle: Integer );
   end;

   TResHscListEntry = class( TResAbstract )
      // x := FHscLists.ListAlloc(...)
      // RC.Add( TRes_HscListEntry.Create(Engine.FHscLists,x) );
      // ...
      // RC.Remove( RESID_HscListEntry, x );
      // FHscLists.ListFree( x );
      private
         HscLists: TObject;
      public
         function AutoDestroy: String; override;
         constructor Create( aHscLists: TObject; aHscListNo: Integer );
   end;

   TResWinApiHandle = class( TResAbstract )
      // x := ...
      // RC.Add( TRes_WinApiHandle.Create(x) );
      // ...
      // RC.Remove( RESID_WinApiHandle, x );
      // CloseHandle( x );
      public
         function AutoDestroy: String; override;
         constructor Create( aHandle: LongWord );
   end;

   TResDelphiObject = class( TResAbstract )
      public
         function AutoDestroy: String; override;
         constructor Create( o: TObject );
   end;

   TResGlobalMemory = class( TResAbstract )
      // x := GlobalAlloc(...)
      // RC.Add( TRes_GlobalMemory.Create(x) );
      // ...
      // RC.Remove( RESID_GlobalMemory, x );
      // GlobalFree( x );
      public
         function AutoDestroy: String; override;
         constructor Create( aMemory: HGLOBAL );
   end;

   TResDynamicDll = class( TResAbstract )
      public
         function AutoDestroy: String; override;
         constructor Create( aDll: HMODULE );
   end;

   TResStrNew = class( TResAbstract )
      public
         function AutoDestroy: String; override;
         constructor Create( pStr: Pointer );
   end;

   TResController = class
      // Engine:
      //    RC := TRessourceController.Create;
      //    ...
      //    s := RC.AutoDestroyAll;
      //    if s<>'' then Log( LOGID_WARN, s );
      //    RC.Free;
      private
         InUseList: TStringList;
      public
         procedure Add   ( ResObj: TResAbstract   );
         procedure Remove( ResID, Entity: LongWord );
         function AutoDestroyOne: String;
         function AutoDestroyAll: String;
         constructor Create;
         destructor Destroy; override;
   end;

implementation

uses cArtFiles, cHscEngine, uDynDll;

function ResKey( ResID: LongWord; Entity: LongWord ): String;
begin
   Result := inttostr( ResID ) + ':' + inttostr( Entity );
end;

{ TResAbstract }

constructor TResAbstract.Create(aResID, aEntity: LongWord);
begin
   inherited Create;
   ResID  := aResID;
   Entity := aEntity;
end;

{ TResHamsterGroup }

function TResHamsterGroup.AutoDestroy: String;
begin
   Result := 'Group ' + inttostr(Entity) + ' (' + ArticleBase.Name[Entity] + ')';
   ArticleBase.Close( Entity );
end;

constructor TResHamsterGroup.Create(aGroupHandle: Integer);
begin
   inherited Create( RESID_HamsterGroup, aGroupHandle );
end;

{ TResHscList }

function TResHscListEntry.AutoDestroy: String;
begin
   Result := 'HSC-list ' + inttostr(Entity);
   THscLists( HscLists ).ListFree( Entity );
end;

constructor TResHscListEntry.Create(aHscLists: TObject; aHscListNo: Integer);
begin
   inherited Create( RESID_HscListEntry, aHscListNo );
   HscLists := aHscLists;
end;

{ TRes_ApiHandle }

function TResWinApiHandle.AutoDestroy: String;
begin
   Result := 'API-handle ' + inttostr(Entity);
   CloseHandle( Entity );
end;

constructor TResWinApiHandle.Create(aHandle: LongWord);
begin
   inherited Create( RESID_WinApiHandle, aHandle );
end;

{ TResGlobalMemory }

function TResGlobalMemory.AutoDestroy: String;
begin
   Result := 'Global memory ' + inttostr(Entity);
   GlobalFree( Entity );
end;

constructor TResGlobalMemory.Create(aMemory: HGLOBAL);
begin
   inherited Create( RESID_GlobalMemory, aMemory );
end;

{ TResDynamicDll }

function TResDynamicDll.AutoDestroy: String;
begin
   Result := 'Dynamic dll ' + inttostr(Entity);
   DynDllFree( Entity );
end;

constructor TResDynamicDll.Create(aDll: HMODULE);
begin
   inherited Create( RESID_DynamicDll, aDll );
end;

{ TResStrNew }

function TResStrNew.AutoDestroy: String;
begin
   Result := 'Allocated string ' + inttostr(Entity);
   StrDispose( PChar( Entity ) );
end;

constructor TResStrNew.Create(pStr: Pointer);
begin
   inherited Create( RESID_StrNew, LongWord( pStr ) );
end;

{ TResDelphiObject }

function TResDelphiObject.AutoDestroy: String;
begin
   Result := 'Allocated Delphi-Object ' + TObject(Entity).classname + ' ' + IntToStr(Entity);
   TObject(Entity).free
end;

constructor TResDelphiObject.Create(o: TObject);
begin
   inherited Create( RESID_DelphiObject, LongWord( o ) );
end;

{ TResController }

constructor TResController.Create;
begin
   inherited Create;
   InUseList := TStringList.Create;
end;

destructor TResController.Destroy;
begin
   if Assigned(InUseList) then begin
      while InUseList.Count>0 do begin
         InUseList.Objects[ 0 ].Free;
         InUseList.Delete( 0 );
      end;
      InUseList.Free;
   end;
   inherited Destroy;
end;

procedure TResController.Remove( ResID, Entity: LongWord );
var  Index: Integer;
begin
   Index := InUseList.IndexOf( ResKey( ResID, Entity ) );
   if Index >= 0 then begin
      InUseList.Objects[ Index ].Free;
      InUseList.Delete( Index );
   end;
end;

procedure TResController.Add( ResObj: TResAbstract );
begin
   InUseList.AddObject( ResKey( ResObj.ResID, ResObj.Entity ), ResObj );
end;

function TResController.AutoDestroyOne: String;
var  Index: Integer;
     ResID, Entity: LongWord;
begin
   Result := '';

   // identify ressource
   Index  := InUseList.Count - 1;
   if Index < 0 then exit;
   ResID  := TResAbstract( InUseList.Objects[ Index ] ).ResID;
   Entity := TResAbstract( InUseList.Objects[ Index ] ).Entity;

   // release ressource
   try
      Result := TResAbstract( InUseList.Objects[ Index ] ).AutoDestroy;
   except
      on E: Exception do Result := Result + 'AutoDestroy-Exception: ' + E.Message;
   end;

   // remove ressource from list if not already done by .AutoDestroy
   Index := InUseList.IndexOf( ResKey( ResID, Entity ) );
   if Index >= 0 then begin
      try
         TResAbstract( InUseList.Objects[ Index ] ).Free;
      except
         on E: Exception do Result := Result + 'AutoDestroy-Exception-2: ' + E.Message;
      end;

      try
         InUseList.Delete( Index );
      except
         on E: Exception do Result := Result + 'AutoDestroy-Exception-3: ' + E.Message;
      end;
   end;
end;

function TResController.AutoDestroyAll: String;
begin
   Result := '';
   while InUseList.Count > 0 do begin
      if Result <> '' then Result := Result + ', ';
      Result := Result + AutoDestroyOne;
   end;
end;

end.
