// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cServerPOP3;

interface

uses Classes, ScktComp, cServerBase, cArticle;

type
  TSrvPOP3 = class( TSrvBase )
    public
      constructor Create( AOwner: TComponent ); 
      {procedure MyOnGetThread(Sender: TObject; ClientSocket: TServerClientWinSocket;
                              var SocketThread: TServerClientThread); override;}
  end;

type
  TSrvPOP3Cli = class(TSrvWorkCli)
    private
      APOP_Stamp     : String;
      CurrentUserID  : Integer;
      CurrentUserName: String;
      lstRETR     : TStringList;
      lstDELE     : TStringList;
      SASLMechanisms : String; {MG}{SSL+SASL}
      LoginState   : Boolean; {JW} {POP3 lock mailbox}
      function LoginUser( Password: String ): String;
      function LoadMailFile( MailSL: TArticle; MailFilename: String ): Boolean;
{JW} {SASL}
      function LoginSASL( Mechanism: String ): String;
      function LoginCRAMMD5( Hash,TimeStamp: String ): String;
      function LoginCRAMSHA1( Hash,TimeStamp: String ): String;
{JW}
    public
      procedure SendGreeting; override;
      function HandleData: String; override;
      procedure HandleCommand( Const CmdLine: String ); override;

      Procedure Cmd_QUIT    ( Const Par: String );
      Procedure Cmd_USER    ( Const Par: String );
      Procedure Cmd_PASS    ( Const Par: String );
      Procedure Cmd_APOP    ( Const Par: String );
      Procedure Cmd_RSET    ( Const Par: String );
      Procedure Cmd_STAT    ( Const Par: String );
      Procedure Cmd_LIST    ( Const Par: String );
      Procedure Cmd_RETR    ( Const Par: String );
      Procedure Cmd_TOP     ( Const Par: String );
      Procedure Cmd_UIDL    ( Const Par: String );
      Procedure Cmd_DELE    ( Const Par: String );
      Procedure Cmd_NOOP    ( Const Par: String );
      Procedure Cmd_HELP    ( Const Par: String );
      Procedure Cmd_CAPA    ( Const Par: String );
      Procedure Cmd_AUTH    ( Const Par: String );  {JW} {SASL}
      Procedure Cmd_STLS    ( Const Par: String ); {MG}{SSL}

      constructor Create( ASocket: TServerClientWinSocket;
                          Const AIPAccessScope: LongInt;
                          Const ASSLContext: Pointer ); override; {MG}{SSL}
      destructor Destroy; override;
  end;

implementation

uses SysUtils, Windows, uTools, Global, Config, cIPAccess, cArtFiles,
     cAccount, uMD5, uCRC32, cStdForm, cLogfile, uSHA1, uEncoding;

// ---------------------------------------------------------- TSrvPOP3Cli -----

function TSrvPOP3Cli.LoginUser( Password: String ): String;
var  No, res    : Integer;
     SR         : TSearchRec;
     MailBoxPath: String;
begin
     Result := '-ERR [SYS/PERM] System-error, check logfile. [0]';
     try
        if Assigned(lstRETR) then lstRETR.Clear;
        if Assigned(lstDELE) then lstDELE.Clear;

        Result := '-ERR [AUTH] Authentication rejected';
        if CurrentUserName='' then exit;

        CurrentUserID := CfgAccounts.LoginID( CurrentUserName, Password );
        if CurrentUserID=ACTID_INVALID then begin
           CurrentUserName := '';
           Result := '-ERR [AUTH] No permission';
           exit;
        end;

        // log authentication for SMTP-after-POP3
        if CfgAccounts.Users.Find(CurrentUserID).MayMailsend then begin
           try
              CfgAccounts.LogAuthenticatedLogin(
                 ClientSocket.RemoteAddr.sin_addr.S_addr
              );
           except
              on E:Exception do Log( LOGID_ERROR, 'Pop3.LogAuthLogin-Exception: ' + E.Message );
           end;
        end;

        if not CfgAccounts.Users.Find(CurrentUserID).HasMailbox then begin
           CurrentUserID := ACTID_INVALID;
           CurrentUserName := '';
           Result := '-ERR [AUTH] No mailbox';
           exit;
        end;

        // authentication ok, try to lock mailbox
        if not CfgAccounts.Users.Find(CurrentUserID).MailboxLock( True ) then begin
           Result := '-ERR [IN-USE] unable to lock mailbox';
           CurrentUserName := '';
           CurrentUserID := ACTID_INVALID;
           exit;
        end;

        // read list of available mails
        EnterCriticalSection( CS_LOCK_MAILBOX_ALL );
        try
           No := 0;
           MailBoxPath := CfgAccounts.Users.Find(CurrentUserID).Path;

           res := SysUtils.FindFirst( MailBoxPath + '*.' +
                  CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ), faAnyFile, SR );
           if res=0 then begin
              while res=0 do begin
                 inc( No ); // 1..n
                 lstRETR.AddObject( MailBoxPath + SR.Name, Pointer(No) );
                 res := SysUtils.FindNext( SR );
              end;
              SysUtils.FindClose( SR );
           end;
           Result := '+OK mailbox locked, ' + inttostr(lstRETR.Count) + ' messages';
           CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000; {JW} {Login}
           LoginState:=true {JW} {POP3 lock mailbox}
        except
           on E:Exception do begin
              Log( LOGID_ERROR, 'Pop3.LoginUser.ReadMailbox-Exception: ' + E.Message );
              Result := '-ERR [SYS/PERM] System-error, check logfile.'
           end;
        end;
        LeaveCriticalSection( CS_LOCK_MAILBOX_ALL );

     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'Pop3.LoginUser-Exception: ' + E.Message );
           Result := '-ERR System-error, check logfile.';
        end;
     end;
end;

function TSrvPOP3Cli.LoadMailFile( MailSL: TArticle; MailFilename: String ): Boolean;
begin
     Result := True;

     try
        MailSL.LoadFromFile( MailFilename );
     except
        on E: Exception do begin
           Log( LOGID_ERROR, TrGlF(kLog, 'Error.CouldntLoadMail',
              'Couldn''t load mail: %s', MailFilename) );
           Log( LOGID_ERROR, TrGl(kLog, 'Error', 'Error') + ': ' + E.Message );
           Result := False;
        end;
     end;
end;

function TSrvPOP3Cli.HandleData: String;
begin
     Result := '-ERR [SYS/PERM] (unknown reason, see logfile)';

     try
        if not ClientSocket.Connected then exit;

        if HadLineTooLong or HadTextTooLarge then Result:='-ERR Limit exceeded';

        Log( LOGID_DEBUG, 'Data ignored.' );
        exit;

     except
        on E: Exception do
           Log( LOGID_ERROR, SockDesc('.HandleData.Exception: ') + E.Message );
     end;
end;

procedure TSrvPOP3Cli.HandleCommand( Const CmdLine: String );
var  LogCmdLine, Cmd, Par, s: String;
     j: Integer;
begin
     try
        if not ClientSocket.Connected then exit;

        // Extract command
        j := PosWhSpace( CmdLine );
        if j=0 then begin
           Cmd := UpperCase( CmdLine );
           Par := '';
        end else begin
           Cmd := UpperCase  ( copy( CmdLine, 1, j-1 ) );
           Par := TrimWhSpace( copy( CmdLine, j+1, 512 ) );
        end;

        if (Cmd='PASS') then begin
           LogCmdLine := 'PASS [...]';
        end else begin
           if Cmd='APOP' then begin
              s := Par;
              j := PosWhSpace( s );
              if j>0 then s:=copy(s,1,j-1);
              LogCmdLine := 'APOP ' + s + ' [...]';
           end else begin
              LogCmdLine := CmdLine;
           end;
        end;

        Log( LOGID_INFO, '> ' + LogCmdLine );
        if CmdLine='' then exit;

        if HadTextTooLarge then begin
           SendResult( '-ERR Textsize-limit exceeded' );
           exit;
        end;
        if HadLineTooLong then begin
           SendResult( '-ERR Linelength-limit exceeded' );
           exit;
        end;

{JW} {POP3 delay}
        // Workaround for tired OutlookXP Clients
        if Def_POP3Delay > 0 then sleep(Def_POP3Delay);
{JW}

        // commands (no authentication required)
        {MG}{SSL}
        if Cmd='HELP' then begin Cmd_HELP( Par ); Exit end;
        if Cmd='QUIT' then begin Cmd_QUIT( Par ); Exit end;
        if Cmd='CAPA' then begin Cmd_CAPA( Par ); Exit end;
        if Cmd='STLS' then begin Cmd_STLS( Par ); Exit end; {MG}{SSL}

        if (Def_LocalPop3TlsMode=2) and not Assigned(SSLConnection) then begin
           SendResult( '-ERR [AUTH] TLS connection required - try STLS' );
           exit
        end;

        if Cmd='APOP' then begin Cmd_APOP( Par ); Exit end;
        if Cmd='PASS' then begin Cmd_PASS( Par ); Exit end;
        if Cmd='USER' then begin Cmd_USER( Par ); Exit end;
        if Cmd='AUTH' then begin Cmd_AUTH( Par ); Exit end; {JW} {SASL}
        {SSL}

        // check authentication
        if CurrentUserID=ACTID_INVALID then begin
           if Pos( '|'+Cmd+'|',
                   '|DELE|LIST|RETR|RSET|STAT|TOP|UIDL|NOOP|'
                 )>0 then begin
              SendResult( '-ERR [AUTH] Authentication required' );
              exit;
           end;
        end;

        // commands (authentication required)
        if Cmd='DELE' then begin Cmd_DELE ( Par ); Exit end;
        if Cmd='LIST' then begin Cmd_LIST ( Par ); Exit end;
        if Cmd='RETR' then begin Cmd_RETR ( Par ); Exit end;
        if Cmd='RSET' then begin Cmd_RSET ( Par ); Exit end;
        if Cmd='STAT' then begin Cmd_STAT ( Par ); Exit end;
        if Cmd='TOP'  then begin Cmd_TOP  ( Par ); Exit end;
        if Cmd='UIDL' then begin Cmd_UIDL ( Par ); Exit end;
        if Cmd='NOOP' then begin Cmd_NOOP ( Par ); Exit end;

        // unknown (sub-) command
        SendResult( '-ERR [SYS/PERM] Command not implemented.' );
        Log( LOGID_INFO, TrGl(kLog, 'Info.UnsupportedPOP3Command',
           'Unsupported POP3-command') + ': ' + CmdLine );

     except
        on E: Exception do begin
           Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
           Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + LogCmdLine );
        end;
     end;
end;

Procedure TSrvPOP3Cli.Cmd_CAPA( Const Par: String );   //HSR //CAPA1.0
begin
  if CurrentUserID=ACTID_INVALID then begin
    SendResult( '+OK Capability of non-authorization follows:' );

    SendQuoted( 'TOP' );
//       The TOP capability indicates the optional TOP command is
//       available.

    SendQuoted( 'USER' );
//       The USER capability indicates that the USER and PASS commands
//       are supported, although they may not be available to all users.

//RFC 2222
    SendQuoted('SASL '+Trim(SASLMechanisms)); {MG}{SSL+SASL}

// RFC 1734
    SendQuoted('AUTH');
{JW}

    SendQuoted( 'RESP-CODES' );  
//       The RESP-CODES capability indicates that any response text issued
//       by this server which begins with an open square bracket ("[") is
//       an extended response code (see section 8).
    SendQuoted( 'AUTH-RESP-CODES' );
//       Additional response codes for security RFC 3206

    SendQuoted( 'LOGIN-DELAY ' + IntToStr(Def_Login_Delay) );
//       minimum seconds between logins; optionally followed by USER in
//       AUTHENTICATION state.

//    SendQuoted( 'PIPELINING' );        //not implemented //don't use!
//       The PIPELINING capability indicates the server is capable of
//       accepting multiple commands at a time; the client does not have
//       to wait for the response to a command before issuing a subsequent
//       command.  If a server supports PIPELINING, it MUST process each
//       command in turn.  If a client uses PIPELINING, it MUST keep track
//       of which commands it has outstanding, and match server responses
//       to commands in order.  If either the client or server uses
//       blocking writes, it MUST not exceed the window size of the
//       underlying transport layer.

    SendQuoted( 'EXPIRE ' + Def_MailExpire );
//       server-guaranteed minimum retention days, or NEVER; optionally
//       followed by USER in AUTHENTICATION state

    SendQuoted( 'UIDL' );
//       The UIDL capability indicates that the optional UIDL command is
//       supported.

    SendQuoted( 'IMPLEMENTATION HamsterPOP3Server' );
//       It is often useful to identify an implementation of a particular
//       server (for example, when logging).  This is commonly done in the
//       welcome banner, but one must guess if a string is an
//       implementation ID or not.

    if Assigned(SSLContext) and (SSLConnection=nil) then
      SendQuoted( 'STLS' ); {MG}{SSL}
//       The POP3 STARTTLS extension adds the STLS command to POP3 servers.
//       [...] The capability name "STLS" indicates this command is present
//       and permitted in the current state.

  end else begin
    SendResult( '+OK Capability of authorization follows:' );
    SendQuoted( 'TOP' );                                      //Same as above!!
    SendQuoted( 'USER' );                                     //Same as above!!
{JW} {SASL}
//RFC 2222
    SendQuoted( 'SASL '+Trim(SASLMechanisms)); {MG}{SSL+SASL}
//RFC 1734
    SendQuoted( 'AUTH' );
    SendQuoted( 'RESP-CODES' );                             //Same as above!!
    SendQuoted( 'AUTH-RESP-CODES' );                        //Same as above!!
    SendQuoted( 'LOGIN-DELAY ' + IntToStr(Def_Login_Delay) + ' USER' ); //Should be changed for special account
//    SendQuoted( 'PIPELINING' );                             //Same as above!!
    SendQuoted( 'EXPIRE ' + trim(Def_MailExpire) + ' USER' );           //Should be changed for special account
    SendQuoted( 'UIDL' );                                     //Same as above!!
    SendQuoted( 'IMPLEMENTATION HamsterPOP3Server' );         //Same as above!!
  end;
  SendResult( '.' );
end;

{MG}{SSL}
Procedure TSrvPOP3Cli.Cmd_STLS( Const Par: String ); // RFC 2595
begin
   if not SSLReady then begin
      SendResult( '-ERR [AUTH] Command not implemented' );
      exit;
   end;
   if SSLContext = nil then begin
      SendResult( '-ERR [SYS/TEMP] TLS not available due to temporary error');
      exit;
   end;
   if SSLConnection = nil then begin
      SendResult( '+OK Begin TLS negotiation' );
      if StartSSL then begin
         SASLMechanisms := Def_LocalPOP3SASL; {MG}{SSL+SASL}
         if CurrentUserID<>ACTID_INVALID then begin
            CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False );
            CurrentUserID := ACTID_INVALID;
         end;
         CurrentUserName := '';
      end;
   end else begin
      SendResult( '-ERR [AUTH] Command not permitted when TLS active' )
   end
end;
{/SSL}

Procedure TSrvPOP3Cli.Cmd_QUIT( Const Par: String );
var  j: Integer;
begin
   LoginState:=False; {JW} {POP3 lock mailbox}
   if lstDELE.Count>0 then begin
      for j:=0 to lstDELE.Count-1 do begin
         SysUtils.DeleteFile( lstDELE[j] );
      end;
      lstDELE.Clear;
   end;

   if CurrentUserID>=0 then begin
      CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False );
      CurrentUserID := ACTID_INVALID;
   end;

   if ClientSocket.Connected then SendResult( '+OK closing connection - goodbye!' );

   Sleep( Def_LocalTimeoutQuitDelay );
   try
      if ClientSocket.Connected then ClientSocket.Close;
   except
      on E:Exception do Log(LOGID_DEBUG, 'Exception on Socket.Close: ' + E.Message );
   end;

   Terminate;
end;

Procedure TSrvPOP3Cli.Cmd_USER( Const Par: String );
begin
     If LoginState then begin
        SendResult( '-ERR [AUTH] command not allowed in this state' ) {JW}
     end else begin
        CurrentUserName := TrimWhSpace( Par );
        if CurrentUserID <> ACTID_INVALID then begin
           CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False )
        end;
        CurrentUserID   := ACTID_INVALID;
        SendResult( '+OK More authentication information required' )
     end
end;

Procedure TSrvPOP3Cli.Cmd_PASS( Const Par: String );
var  s: String;
begin
   If LoginState then begin
      SendResult( '-ERR [AUTH] command not allowed in this state' ); {JW}
   end else begin
      try
         s := LoginUser( TrimWhSpace( Par ) );
         SendResult( s );
      except
         on E:Exception do Log( LOGID_ERROR, 'Pop3.Cmd_PASS-Exception: ' + E.Message )
      end
   end
end;

Procedure TSrvPOP3Cli.Cmd_APOP( Const Par: String );
Var md5val, s, p, cmp: String; j: Integer;
begin
   If LoginState then begin
      SendResult( '-ERR command not allowed in this state' ); {JW} {POP3 lock mailbox}
   end else begin
      CurrentUserID := ACTID_INVALID;

      j := PosWhSpace( Par );
      if j=0 then begin
         SendResult( '-ERR Invalid params' );
         exit;
      end;
      CurrentUserName := copy( Par, 1, j-1 );
      md5val := TrimWhSpace( copy( Par, j+1, Length(Par)-j ) );

      CurrentUserID := CfgAccounts.Users.IDOf( CurrentUserName );
      if CurrentUserID=ACTID_INVALID then begin
         SendResult( '-ERR Authentication failed.' );
         exit;
      end;

      p := CfgAccounts.Users.Find(CurrentUserID).Password;
      s := MD5ofStr( APOP_Stamp + p );
      cmp := '';
      for j:=1 to length(s) do cmp := cmp + lowercase( inttohex( ord(s[j]), 2 ) );

      if cmp=md5val then begin
         s := LoginUser( p );
         SendResult( s );
         LoginState := true; {JW} {POP3 lock mailbox}
         CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000; {JW} {APOP Login}
      end else begin
         CurrentUserID := ACTID_INVALID;
         SendResult( '-ERR [AUTH] Authentication failed.' );
      end
   end
end;

Procedure TSrvPOP3Cli.Cmd_RSET( Const Par: String );
var  i, k: Integer;
begin
     while lstDELE.Count>0 do begin
        lstRETR.AddObject( lstDELE[0], lstDELE.Objects[0] );
        lstDELE.Delete( 0 );
     end;

     // resort by numbers
     for i:=0 to lstRETR.Count-2 do begin
        for k:=i+1 to lstRETR.Count-1 do begin
            if LongInt(lstRETR.Objects[i])>LongInt(lstRETR.Objects[k]) then begin
               lstRETR.Exchange( i, k );
            end;
        end;
     end;

     SendResult( '+OK mailbox has ' + inttostr(lstRETR.Count) + ' messages' );
end;

Procedure TSrvPOP3Cli.Cmd_STAT( Const Par: String );
var  cnt, byt, j: Integer;
begin
     cnt := 0;
     byt := 0;
     for j:=0 to lstRETR.Count-1 do begin
        inc( cnt );
        byt := byt + GetFileSize( lstRETR[j] );
     end;

     SendResult( '+OK' + ' ' + inttostr(cnt) + ' ' + inttostr(byt) );
end;

Procedure TSrvPOP3Cli.Cmd_LIST( Const Par: String );
var  No, j, byt: Integer;
     h: String;
begin
     if Par='' then begin
        SendResult( '+OK' + ' ' + inttostr(lstRETR.Count) + ' messages' );
        for j:=0 to lstRETR.Count-1 do begin
           byt := GetFileSize( lstRETR[j] );
           SendQuoted( inttostr(LongInt(lstRETR.Objects[j])) + ' ' + inttostr(byt) );
        end;
        SendResult( '.' );
        exit;
     end;

     No := strtoint( Par );
     h := '-ERR no such message';
     for j:=0 to lstRETR.Count-1 do begin
        if LongInt(lstRETR.Objects[j])=No then begin
           byt := GetFileSize( lstRETR[j] );
           h := '+OK' + ' ' + inttostr( LongInt(lstRETR.Objects[j]) ) + ' ' + inttostr(byt);
           break;
        end;
     end;
     SendResult( h );
end;

Procedure TSrvPOP3Cli.Cmd_RETR( Const Par: String );
var  OK: Boolean;
     Art: TArticle;
     No, j, k, byt: Integer;
begin
     OK := False;

     if Par<>'' then begin
        No := strtoint( Par );
        Art := TArticle.Create;

        for j:=0 to lstRETR.Count-1 do begin
           if LongInt(lstRETR.Objects[j])=No then begin
              //Art.LoadFromFile( lstRETR[j] );
              if not LoadMailFile( Art, lstRETR[j] ) then break;

              byt := GetFileSize( lstRETR[j] );
              SendResult( '+OK' + ' ' + inttostr(byt) + ' octets' );
              With TStringList.Create do try
                 Text := Art.Text;
                 for k:=0 to Count-1 do SendQuoted( Strings[k] )
              finally
                 free
              end;

              SendResult( '.' );
              OK := True;
              break;
           end;
        end;

        Art.Free;
     end;

     if not OK then SendResult( '-ERR no such message' );
end;

Procedure TSrvPOP3Cli.Cmd_TOP( Const Par: String );
var  j, k, No, Cnt, byt: Integer;
     OK: Boolean;
     Art: TArticle;
begin
   j := Pos( ' ', Par );
   if j=0 then begin
      SendResult( '-ERR invalid format for TOP' );
      exit;
   end;
   No  := strtoint( copy(Par,1,j-1) );
   Cnt := strtoint( copy(Par,j+1,Length(Par)-j) );
   OK  := False;

   Art := TArticle.Create;
   try
      for j:=0 to lstRETR.Count-1 do begin
         if LongInt(lstRETR.Objects[j])=No then begin
            if not LoadMailFile( Art, lstRETR[j] ) then break;

            byt := GetFileSize( lstRETR[j] );
            SendResult( '+OK' + ' ' + inttostr(byt) + ' octets' );
            With TStringList.Create do try
               Text := Art.FullHeader;
               for k := 0 to Count-1 do SendQuoted( Strings[k] );
               SendQuoted( '' );
               Text := Art.FullBody;
               for k := 0 to Count-1 do begin
                  If k >= Cnt then break;
                  SendQuoted( Strings[k] )
               end
            finally
               free
            end;
            SendResult( '.' );
            OK := True;
            break;
         end;
      end
   finally
      Art.Free
   end;
   if not OK then SendResult( '-ERR no such message' );
end;

Procedure TSrvPOP3Cli.Cmd_UIDL( Const Par: String );
var  Art: TArticle;
     OK : Boolean;
     No, j: Integer;
     h: String;
begin
     if Par<>'' then begin
        No := strtoint( Par );
        OK := False;
        Art := TArticle.Create;
        try
           for j:=0 to lstRETR.Count-1 do begin
              if LongInt(lstRETR.Objects[j])=No then begin
                 // Art.LoadFromFile( lstRETR[j] );
                 if not LoadMailFile( Art, lstRETR[j] ) then break;

                 h := Art.GetOurXHeader( 'UIDL' );
                 if h='' then h:=inttohex( StrToCRC32(Art.Text), 8 );

                 SendResult( '+OK' + ' ' + inttostr(no) + ' ' + h );
                 ok := True;
                 break;
              end;
           end
        finally
           Art.Free
        end;
        if not OK then SendResult( '-ERR no such message' );
        exit;
     end;


     SendResult( '+OK' );
     Art := TArticle.Create;
     try
        for j:=0 to lstRETR.Count-1 do begin
           // Art.LoadFromFile( lstRETR[j] );
           if LoadMailFile( Art, lstRETR[j] ) then begin
              h := Art.GetOurXHeader( 'UIDL' );
              if h='' then h:=inttohex( StrToCRC32(Art.Text), 8 );
              SendQuoted( inttostr(LongInt(lstRETR.Objects[j])) + ' ' + h );
           end;
        end;
        SendResult( '.' )
     finally
        Art.Free
     end
end;


Procedure TSrvPOP3Cli.Cmd_DELE( Const Par: String );
var  ok: Boolean;
     No, j: Integer;
begin
     ok := False;

     if Par<>'' then begin
        No := strtoint( Par );
        for j:=0 to lstRETR.Count-1 do begin
           if LongInt(lstRETR.Objects[j])=No then begin
              lstDELE.AddObject( lstRETR[j], lstRETR.Objects[j] );
              lstRETR.Delete( j );
              SendResult( '+OK' + ' ' + 'message ' + inttostr(no) + ' deleted' );
              ok := True;
              break;
           end;
        end;
     end;

     if not OK then SendResult( '-ERR no such message' );
end;

Procedure TSrvPOP3Cli.Cmd_NOOP( Const Par: String );
begin
   SendResult( '+OK localhost' );
end;

Procedure TSrvPOP3Cli.Cmd_HELP( Const Par: String );
begin
   SendResult( '+OK Implemented commands follow:' );

   SendQuoted( '    dele Number' );
   SendQuoted( '    help' );
   SendQuoted( '    list [Number]' );
   SendQuoted( '    pass Password' );
   SendQuoted( '    quit' );
   SendQuoted( '    retr Number' );
   SendQuoted( '    rset' );
   SendQuoted( '    stat' );
   if Assigned(SSLContext) and (SSLConnection=nil) then
      SendQuoted( '    stls' ); {MG}{SSL}
   SendQuoted( '    top Number Lines' );
   SendQuoted( '    uidl [Number]' );
   SendQuoted( '    user Username' );
{JW} {auth & capa in help}
   SendQuoted( '    capa' );
   SendQuoted( '    auth' );
{jw}

   SendResult( '.' );
end;

procedure TSrvPOP3Cli.SendGreeting;
begin
     {JW} {LocalLimit}
     if (GetCounter(CntPop3Cli) > Def_Max_Local_POP3_Servers) and
        (Def_Max_Local_POP3_Servers>0) then begin
        SendResult( '-ERR [SYS/TEMP] server overload, try again later' );
        Log( LOGID_WARN, 'POP3 Server overload, too many clients' );
        Terminate;
        exit;
     end;
     {/JW}
     CheckClientAccess;

{JW}  {APOP Bug}
     if (IPAccess and IPACC_ACCESS_RO)=IPACC_ACCESS_RO then begin
        If Def_FQDN > '' then
           SendResult( '+OK POP3-Server '
                        + GetMyStringFileInfo('ProductName','Hamster') + ' ' // jawo 27.09.2001
                        + GetMyVersionInfo(true)
                        + ' '+ Def_FQDN
                        + ' greets you! '
                        + APOP_Stamp )
        else
           SendResult( '+OK POP3-Server '
                        + GetMyStringFileInfo('ProductName','Hamster') + ' ' // jawo 27.09.2001
                        + GetMyVersionInfo(true)
                        + ' greets you! '
                        + APOP_Stamp);
        exit; // OK
     end;
{JW}
     Log( LOGID_WARN, TrGlF(kLog, 'Warning.POP3.ConnectionRefused',
        'Connection refused, %s isn''t allowed to connect to POP3-Server', ClientID) );
     if ClientSocket<>nil then
     try
        if ClientSocket.Connected then SendResult( '-ERR Permission denied - closing connection.' );
        if ClientSocket.Connected then ClientSocket.Close;
     except
     end;
     Terminate;
end;


constructor TSrvPOP3Cli.Create( ASocket: TServerClientWinSocket;
                                Const AIPAccessScope: LongInt;
                                Const ASSLContext: Pointer );
Var i: Integer;
begin
   inherited Create( ASocket, AIPAccessScope, ASSLContext );
   CurrentInactTimeOut:=Def_LocalPOP3LoginTimeout; {JW} {Login}
   LoginState:=False; {JW} {POP3 lock mailbox}
   LimitLineLen  := Def_LocalLimitLineLenPop3;
   LimitTextSize := Def_LocalLimitTextSizePop3;
   APOP_Stamp := MidGenerator( Def_FQDNforMIDs );
   {MG}{SSL+SASL}
   SASLMechanisms := Def_LocalPOP3SASL;
   i := Pos( 'PLAIN', SASLMechanisms );
   if i > 0 then System.Delete( SASLMechanisms, i, 6 );
   i := Pos( 'EXTERNAL', SASLMechanisms );
   if i > 0 then System.Delete( SASLMechanisms, i, 9 );
   SASLMechanisms := TrimWhSpace( SASLMechanisms );
   {/SSL+SASL}
   CurrentUserID   := ACTID_INVALID;
   CurrentUserName := '';
   lstRETR := TStringList.Create;
   lstDELE := TStringList.Create;
   IncCounter(CntPop3Cli,1);
end;

destructor TSrvPOP3Cli.Destroy;
begin
     if CurrentUserID>=0 then CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False );
     if Assigned(lstRETR) then begin lstRETR.Free; lstRETR:=nil; end;
     if Assigned(lstDELE) then begin lstDELE.Free; lstDELE:=nil; end;
     IncCounter(CntPop3Cli,-1);
     inherited;
end;

// ------------------------------------------------------------- TSrvPOP3 -----

constructor TSrvPOP3.Create;
begin
   inherited Create(
      AOwner,
      CfgIni.ReadString ('Setup', 'local.pop3.serverbind', Def_LocalPOP3ServerBind ),
      CfgIni.ReadInteger('Setup', 'local.port.pop3', DEF_LOCALPOP3Server_PORT  ),
      CfgIni.ReadInteger('Setup', 'MaxLocalPOP3Servers', Def_Max_Local_POP3_Servers),
      CfgIni.ReadInteger('Setup', 'MaxLocalPOP3ServersPerIP', Def_Max_Local_POP3_Servers_Per_IP),
      IPACC_SCOPE_POP3,
      TSrvPOP3Cli
   );
end;

// ----------------------------------------------------------------------------

Procedure TSrvPOP3Cli.Cmd_AUTH(Const Par: String);
var
   j         : integer;
   SASL_Name : string;
   s, AktPar : string;
begin
   AktPar := Par;
   if LoginState then begin
      SendResult( '-ERR [AUTH] command not allowed in this state' ) {JW}
   end else begin
      CurrentUserID := ACTID_INVALID;
      if aktpar='' then begin
         SendResult( '+OK list of SASL extensions follows' );
         s:=SASLMechanisms; // s:=Def_LocalPOP3SASL;
         while s<>'' do begin
            if PosWhSpace(s)=0 then begin
               SendQuoted(  s );
               s:='';
            end else  begin
               SendQuoted(TrimWhSpace(copy(s,1,PosWhSpace(s))));
               System.Delete(s,1,PosWhSpace(s));
            end;
         end;
         SendResult('.');
         exit;
      end;
      j := Pos( ' ', AktPar );
      if j=0 then begin
         SASL_Name := Uppercase( AktPar );
         AktPar := '';
      end else begin
         SASL_Name := Uppercase( copy( AktPar, 1, j-1 ) );
         System.Delete( AktPar, 1, j );
      end;
      if Pos( ' ' + SASL_Name + ' ',
              ' ' + SASLMechanisms  + ' ' )=0 then begin
         SendResult( '-ERR [AUTH] unsupported SASL authentication method' );
         exit;
      end;
      try
         s := LoginSASL( SASL_Name );
         SendResult( s );
      except
         on E:Exception do begin
            Log( LOGID_ERROR, 'Pop3.Cmd_PASS-Exception: ' + E.Message );
         end
      end
   end
end;

function TSrvPOP3Cli.LoginCRAMMD5(Hash, TimeStamp: String): String;
var  No, res    : Integer;
     SR         : TSearchRec;
     MailBoxPath: String;
     s          : string;
begin
     Result := '-ERR [AUTH/PERM] System-error, check logfile. [0]';

     try
        if Assigned(lstRETR) then lstRETR.Clear;
        if Assigned(lstDELE) then lstDELE.Clear;

        Result := '-ERR [AUTH] Authentication rejected';
        if CurrentUserName='' then exit;

        CurrentUserID   := CfgAccounts.Users.IDOf(CurrentUserName);
        if CurrentUserID=ACTID_INVALID then begin
           CurrentUserName := '';
           exit;
        end;
        s:=CfgAccounts.Users.Find(CurrentUserID).Password;
        s:=MD5HMAC( s,TimeStamp );
        s:=MD5toHex( s );
        if s<>Hash then  begin
           CurrentUserName := '';
           CurrentUserID:=ACTID_INVALID;
           Result := '-ERR [AUTH] No permission';
           exit;
        end;

        // log authentication for SMTP-after-POP3
        if CfgAccounts.Users.Find(CurrentUserID).MayMailsend then begin
           try
              CfgAccounts.LogAuthenticatedLogin(
                 ClientSocket.RemoteAddr.sin_addr.S_addr
              );
           except
              on E:Exception do Log( LOGID_ERROR,
                                     'Pop3.LogAuthLogin-Exception: ' +
                                      E.Message );
           end;
        end;

        if not CfgAccounts.Users.Find(CurrentUserID).HasMailbox then begin
           CurrentUserID := ACTID_INVALID;
           CurrentUserName := '';
           Result := '-ERR [AUTH] No mailbox';
           exit;
        end;

        // authentication ok, try to lock mailbox
        if not CfgAccounts.Users.Find(CurrentUserID).MailboxLock( True ) then begin
           Result := '-ERR [IN-USE] unable to lock mailbox';
           CurrentUserName := '';
           CurrentUserID := ACTID_INVALID;
           exit;
        end;

        // read list of available mails
        EnterCriticalSection( CS_LOCK_MAILBOX_ALL );
        try
           No := 0;
           MailBoxPath := CfgAccounts.Users.Find(CurrentUserID).Path;

           res := SysUtils.FindFirst( MailBoxPath + '*.' +
                  CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ),
                                     faAnyFile, SR );
           if res=0 then begin
              while res=0 do begin
                 inc( No ); // 1..n
                 lstRETR.AddObject( MailBoxPath + SR.Name, Pointer(No) );
                 res := SysUtils.FindNext( SR );
              end;
              SysUtils.FindClose( SR );
           end;
           Result := '+OK mailbox locked, ' + inttostr(lstRETR.Count)
                      + ' messages';
           LoginState:=true; {JW} {POP3 lock mailbox}
           CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000; {JW} {Login}
        except
           on E:Exception do begin
              Log( LOGID_ERROR, 'Pop3.LoginUser.ReadMailbox-Exception: '
                   + E.Message );
           end;
        end;
        LeaveCriticalSection( CS_LOCK_MAILBOX_ALL );

     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'Pop3.LoginUser-Exception: ' + E.Message );
           Result := '-ERR [SYS/PERM] System-error, check logfile.';
        end;
     end;
end;

function TSrvPOP3Cli.LoginCRAMSHA1(Hash, TimeStamp: String): String;
var  No, res    : Integer;
     SR         : TSearchRec;
     MailBoxPath: String;
     s          : string;
begin
     Result := '-ERR [SYS/PERM] System-error, check logfile. [0]';
     try
        if Assigned(lstRETR) then lstRETR.Clear;
        if Assigned(lstDELE) then lstDELE.Clear;
        Result := '-ERR [AUTH] Authentication rejected';
        if CurrentUserName='' then exit;
        CurrentUserID   := CfgAccounts.Users.IDOf(CurrentUserName);
        if CurrentUserID=ACTID_INVALID then begin
           CurrentUserName := '';
           exit;
        end;
        s:=CfgAccounts.Users.Find(CurrentUserID).Password;
        s:=HMAC_SHA1( TimeStamp,s );
        if s<>Hash then  begin
           CurrentUserName := '';
           CurrentUserID:=ACTID_INVALID;
           Result := '-ERR [AUTH] No permission';
           exit;
        end;
         // log authentication for SMTP-after-POP3
        if CfgAccounts.Users.Find(CurrentUserID).MayMailsend then begin
           try
              CfgAccounts.LogAuthenticatedLogin(
                 ClientSocket.RemoteAddr.sin_addr.S_addr
              );
           except
              on E:Exception do Log( LOGID_ERROR,
                                     'Pop3.LogAuthLogin-Exception: ' +
                                      E.Message );
           end;
        end;
        if not CfgAccounts.Users.Find(CurrentUserID).HasMailbox then begin
           CurrentUserID := ACTID_INVALID;
           CurrentUserName := '';
           Result := '-ERR [AUTH] No mailbox';
           exit;
        end;
        // authentication ok, try to lock mailbox
        if not CfgAccounts.Users.Find(CurrentUserID).MailboxLock( True ) then begin
           Result := '-ERR [IN-USE] unable to lock mailbox';
           CurrentUserName := '';
           CurrentUserID := ACTID_INVALID;
           exit;
        end;
        // read list of available mails
        EnterCriticalSection( CS_LOCK_MAILBOX_ALL );
        try
           No := 0;
           MailBoxPath := CfgAccounts.Users.Find(CurrentUserID).Path;
           res := SysUtils.FindFirst( MailBoxPath + '*.' +
                  CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ),
                                     faAnyFile, SR );
           if res=0 then begin
              while res=0 do begin
                 inc( No ); // 1..n
                 lstRETR.AddObject( MailBoxPath + SR.Name, Pointer(No) );
                 res := SysUtils.FindNext( SR );
              end;
              SysUtils.FindClose( SR );
           end;
           Result := '+OK mailbox locked, ' + inttostr(lstRETR.Count)
                      + ' messages';
           LoginState:=true; {JW} {POP3 lock mailbox}
           CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000; {JW} {Login}
        except
           on E:Exception do begin
              Log( LOGID_ERROR, 'Pop3.LoginUser.ReadMailbox-Exception: '
                   + E.Message );
           end;
        end;
        LeaveCriticalSection( CS_LOCK_MAILBOX_ALL );
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'Pop3.LoginUser-Exception: ' + E.Message );
           Result := '-ERR [SYS/PERM] System-error, check logfile.';
        end;
     end;
end;

function TSrvPOP3Cli.LoginSASL(Mechanism: String): String;
Var s, TimeStamp:string;
  realm, nonce, cnonce, qop, username, nc, realm2,
  digesturi, response, a1, a2, password, rspauth :string;
  No, res    : Integer;
  SR         : TSearchRec;
  MailBoxPath: String;
begin
  mechanism:=uppercase(mechanism);
  Result := '-ERR [AUTH] Authentication rejected';
  if mechanism='LOGIN' then begin
      s := 'Username:';
      s := EncodeB64( s[1], length(s) );
      s := SendRequest( '+ ' + s );
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      CurrentUserName := TrimWhSpace( s );
      s := 'Password:';
      s := EncodeB64( s[1], length(s) );
      s := SendRequest( '+ ' + s );
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      if CurrentUserID<>ACTID_INVALID then begin
         CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False );
      end;
      CurrentUserID   := ACTID_INVALID;
      result:=LoginUser(s);
  end else
  if mechanism='PLAIN' then begin
      s := SendRequest( '+');
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      s:=copy(s,pos(chr(0),s)+1,500);
      CurrentUserName := TrimWhSpace( copy(s,1,pos(chr(0),s)-1));
      s:=TrimWhSpace(copy(s,pos(chr(0),s)+1,500));
      if CurrentUserID<>ACTID_INVALID then begin
         CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False );
      end;
      CurrentUserID   := ACTID_INVALID;
      result:=LoginUser(s);
  end else
  if Mechanism = 'DIGEST-MD5' then begin
      // build challenge
      if Def_FQDN<> '' then realm := Def_FQDN
                       else realm := 'localhost';
      s := 'realm="' + realm + '"';
      nonce:= HMAC_SHA1( nonce,
               IntToHex(PRNG(MaxInt),8)+IntToHex(PRNG(MaxInt),8)+
               IntToHex(PRNG(MaxInt),8)+IntToHex(PRNG(MaxInt),8));
      nonce := EncodeB64( nonce[1], length(nonce) );
      s := s + ',' + 'nonce="' + nonce + '"';
      qop := 'auth';
      s := s + ',' + 'qop="' + qop + '"';
      s := s + ',' + 'algorithm=' + 'md5-sess';
      Log( LOGID_DEBUG, 'DIGEST-MD5 challenge: ' + s );
      s := EncodeB64( s[1], length(s) );
      // send challenge, get response
      s := SendRequest( '+ '+s);
      if s='' then begin
        Log( LOGID_Error, 'empty response received');
        exit;
      end;
      // check response, extract values
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_DEBUG, 'DIGEST-MD5 response: ' + s );
      // check username
      username:=ExtractQuotedParameter(s,'username');
      if username='' then begin
        Log( LOGID_Error, 'missing username in answer');
        exit;
      end;
      CurrentUserName:=trim(username);
      if CurrentUserID<>ACTID_INVALID then begin
         CfgAccounts.Users.Find(CurrentUserID).MailboxLock(False);
      end;
      CurrentUserID   := ACTID_INVALID;
      nonce:=ExtractQuotedParameter(s,'nonce');
      if nonce='' then begin
        Log( LOGID_Error, 'missing nonce in answer');
        exit;
      end;
      cnonce:=ExtractQuotedParameter(s,'cnonce');
      if cnonce='' then begin
        Log( LOGID_Error, 'missing cnonce in answer');
        exit;
      end;
      nc:=ExtractQuotedParameter(s,'nc');
      if nc <> '00000001' then begin
        Log( LOGID_Error, 'wrong nc value');
        exit;
      end;
      qop:=ExtractQuotedParameter(s,'qop');
      if qop='' then begin
         qop:='auth'
      end else begin
        if (lowercase(qop)<>'auth') then begin
          Log(LOGID_ERROR, 'not supported hash quality protection '+qop);
          exit;
        end
      end;
      realm2:=ExtractQuotedParameter(s,'realm');
      if realm2='' then begin
        Log( LOGID_Error, 'missing realm in answer');
        exit;
      end;
      if realm2<>realm then begin
        Log( LOGID_Error, 'wrong realm in answer');
        exit;
      end;
      digesturi:=ExtractQuotedParameter(s,'digest-uri');
      response:=ExtractQuotedParameter(s,'response');
      if length(response)<>32 then begin
        Log( LOGID_Error, 'wrong response length in answer');
        exit;
      end;
      // build expected response and compare with received one
      try
        if Assigned(lstRETR) then lstRETR.Clear;
        if Assigned(lstDELE) then lstDELE.Clear;
        CurrentUserID   := CfgAccounts.Users.IDOf(CurrentUserName);
        if CurrentUserID=ACTID_INVALID then begin
           Log( LOGID_Error, 'login rejected, unknown user');
           CurrentUserName := '';
           exit;
        end;
        password:=CfgAccounts.Users.Find(CurrentUserID).Password;
        a1 := MD5OfStr( username + ':' + realm + ':' + password )
              + ':' + nonce + ':' + cnonce;
        A2:='AUTHENTICATE:'+digesturi;
        s  := MD5toHex( MD5OfStr(
               MD5toHex( MD5OfStr( A1 ) )
               + ':' + nonce + ':' + nc + ':' + cnonce + ':' + qop
               + ':' + MD5toHex( MD5OfStr( A2 ) )
            ) );
        if s<>response then begin
           Log( LOGID_Error, 'login rejected, wrong response value');
           CurrentUserName := '';
           CurrentUserID:=ACTID_INVALID;
           Result := '-ERR [AUTH] No permission';
           exit;
        end;
        // read list of available mails
        EnterCriticalSection( CS_LOCK_MAILBOX_ALL );
        try
           No := 0;
           MailBoxPath := CfgAccounts.Users.Find( CurrentUserID ).Path;
           res := SysUtils.FindFirst( MailBoxPath + '*.' +
                  CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ),
                                     faAnyFile, SR );
           if res=0 then begin
              while res=0 do begin
                 inc( No ); // 1..n
                 lstRETR.AddObject( MailBoxPath + SR.Name, Pointer(No) );
                 res := SysUtils.FindNext( SR );
              end;
              SysUtils.FindClose( SR );
           end;
           Result := '+OK mailbox locked, ' + inttostr(lstRETR.Count)+ ' messages';
           LoginState:=true; {JW} {POP3 lock mailbox}
           CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000; {JW}
        finally
           LeaveCriticalSection( CS_LOCK_MAILBOX_ALL );
        end;
      except
        Log( LOGID_Error, 'unknown error');
        CurrentUserName := '';
        CurrentUserID:=ACTID_INVALID;
        exit;
      end;
      // build rspauth and send it
      a2 := ':' + digesturi;
      rspauth := MD5toHex( MD5OfStr(
                 MD5toHex( MD5OfStr( A1 ) )
                 + ':' + nonce + ':' + nc + ':' + cnonce + ':' + qop
                 + ':' + MD5toHex( MD5OfStr( A2 ) )
                 ) );
      s := 'rspauth=' + rspauth;
      Log( LOGID_DEBUG, 'DIGEST-MD5 rspauth: ' + s );
      s := EncodeB64( s[1], length(s) );
      s := SendRequest( '+ '+s);
  end else

  if mechanism='CRAM-MD5' then begin
      TimeStamp := MidGenerator(Def_FQDNforMIDs);
      s := EncodeB64( TimeStamp[1], length(TimeStamp) );
      s := SendRequest( '+ '+s);
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
      s:=TrimWhSpace( copy(s,PosWhSpace(s)+1,32));
      if CurrentUserID<>ACTID_INVALID then begin
         CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False );
      end;
      CurrentUserID   := ACTID_INVALID;
      result:=LoginCRAMMD5(s,TimeStamp);
  end else
  if mechanism='CRAM-SHA1' then begin
      TimeStamp := MidGenerator(Def_FQDNforMIDs);
      s := EncodeB64( TimeStamp[1], length(TimeStamp) );
      s := SendRequest( '+ '+s);
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
      s:=TrimWhSpace( copy(s,PosWhSpace(s)+1,40));  // Fix Arne Schloh
      if CurrentUserID<>ACTID_INVALID then begin
         CfgAccounts.Users.Find(CurrentUserID).MailboxLock( False );
      end;
      CurrentUserID   := ACTID_INVALID;
      result:=LoginCRAMSHA1(s,TimeStamp);
  end
end;

end.

