// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cServerRECO; // Remote Control Server

interface

{$INCLUDE Compiler.inc}

uses ScktComp, cServerBase, Classes;

type
  TSrvRECO = class( TSrvBase )
    public
      constructor Create( AOwner: TComponent );
  end;

type
  TSrvRECOCli = class(TSrvWorkCli)
    private
      CurrentUserID  : Integer;
      CurrentUserName: String;
    public
      procedure SendGreeting; override;
      function  HandleData: String; override;
      procedure HandleCommand( Const CmdLine: String ); override;
      function  Local_SASL_Login( Mechanism, Parameters: String ): Boolean; {JW} {RECO SASL}

      function Cmd_AUTH  ( Const AC: Integer; AV: TStringList ): Boolean;
      function Cmd_HELP  ( Const AC: Integer; AV: TStringList ): Boolean;
      function Cmd_LOG   ( Const AC: Integer; AV: TStringList ): Boolean;
      function Cmd_NEWS  ( Const AC: Integer; AV: TStringList ): Boolean;
      function Cmd_QUIT  ( Const AC: Integer; AV: TStringList ): Boolean;
      function Cmd_SCRIPT( Const AC: Integer; AV: TStringList ): Boolean;
      function Cmd_SERVER( Const AC: Integer; AV: TStringList ): Boolean;
      function Cmd_TASK  ( Const AC: Integer; AV: TStringList ): Boolean;

      constructor Create( ASocket: TServerClientWinSocket;
                          Const AIPAccessScope: LongInt;
                          Const ASSLContext: Pointer ); override;
      destructor Destroy; override;
  end;

implementation

uses SysUtils, Windows, uTools, Global, Config, cIPAccess, cArticle, cArtFiles,
     tBase, cAccount, uCRC32, cPCRE, uDateTime, tScript, uEncoding, cLogfile,
     uMD5, uSHA1;

type
   TCmdInfo = record
      CmdStr : String;
      MinPar : Byte;
      MaxPar : Byte;
      AuthReq: Byte;
   end;

const
   CmdCount = 8;
   CmdArray: array[ 0 .. CmdCount-1 ] of TCmdInfo = (
      ( CmdStr:'AUTH';   MinPar:2; MaxPar:4; AuthReq:0 ),
      ( CmdStr:'HELP';   MinPar:0; MaxPar:1; AuthReq:0 ),
      ( CmdStr:'LOG';    MinPar:1; MaxPar:5; AuthReq:1 ),
      ( CmdStr:'NEWS';   MinPar:2; MaxPar:4; AuthReq:1 ),
      ( CmdStr:'QUIT';   MinPar:0; MaxPar:0; AuthReq:0 ),
      ( CmdStr:'SCRIPT'; MinPar:1; MaxPar:9; AuthReq:1 ),
      ( CmdStr:'SERVER'; MinPar:2; MaxPar:2; AuthReq:1 ),
      ( CmdStr:'TASK';   MinPar:1; MaxPar:3; AuthReq:1 )
   );

function OptBoo( var AC: Integer;
                 var AV: TStringList;
                 const Opt: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   for i:=1 to AC do begin
      if CompareText( Opt, AV[i] ) = 0 then begin
         Result := True;
         AV.Delete( i );
         dec( AC );
         break;
      end;
   end;
end;

function OptStr( var AC: Integer;
                 var AV: TStringList;
                 const Opt: String;
                 const Default: String ): String;
var  i: Integer;
begin
   Result := Default;
   for i:=1 to AC-1 do begin
      if CompareText( Opt, AV[i] ) = 0 then begin
         Result := AV[ i+1 ];
         AV.Delete( i+1 );
         AV.Delete( i   );
         dec( AC, 2 );
         break;
      end;
   end;
end;

function OptInt( var AC: Integer;
                 var AV: TStringList;
                 const Opt: String;
                 const Default: Integer ): Integer;
begin
   Result := strtointdef( OptStr( AC, AV, Opt, '' ), Default );
end;

// ---------------------------------------------------------- TSrvRECOCli -----

function TSrvRECOCli.HandleData: String;
begin
   Result := Res( R_SYSERR, 'Unexpected data' );
   exit;
end;

procedure TSrvRECOCli.HandleCommand( Const CmdLine: String );
var  AC, i: Integer;
     AV: TStringList;
     LogCmdLine: String;
begin
     try
        if not ClientSocket.Connected then exit;

        AV := TStringList.Create;
        try
           // parse command and parameters
           // AC: number of arguments (i.e. not including AV[0])
           // AV[0]: command; AV[1]: 1st arg.; ...; AV[AC]: last arg.
           AC := ArgsWhSpaceDQuoted( CmdLine, AV, 2 ) - 1;
           AV[0] := UpperCase( AV[0] ); // command
           AV[1] := UpperCase( AV[1] ); // [action, i.e. 1st param]

           if AV[0] = 'AUTH' then begin
              LogCmdLine := 'AUTH [...]';
           end else begin
              LogCmdLine := CmdLine;
           end;

           Log( LOGID_INFO, '> ' + LogCmdLine );
           if CmdLine='' then exit;

           if HadTextTooLarge then begin
              SendResult( Res( R_SYSERR, 'Textsize-limit exceeded' ) );
              exit;
           end;
           if HadLineTooLong then begin
              SendResult( Res( R_SYSERR, 'Linelength-limit exceeded' ) );
              exit;
           end;

           // check and execute commands
           for i := 0 to CmdCount-1 do begin
              if AV[0] = CmdArray[i].CmdStr then begin

                 if ( CmdArray[i].AuthReq <> 0 ) and
                    ( CurrentUserID = ACTID_INVALID ) then begin
                    SendResult( Res( R_AUTHREQ, 'Authentication required' ) );
                    exit;
                 end;

                 if ( AC < CmdArray[i].MinPar ) or
                    ( AC > CmdArray[i].MaxPar ) then begin
                    SendResult( Res( R_SYNTAX, 'Syntax error (invalid parameter count)' ) );
                    exit;
                 end;

                 if AV[0]='AUTH'   then if Cmd_AUTH  ( AC, AV ) then exit;
                 if AV[0]='HELP'   then if Cmd_HELP  ( AC, AV ) then exit;
                 if AV[0]='LOG'    then if Cmd_LOG   ( AC, AV ) then exit;
                 if AV[0]='NEWS'   then if Cmd_NEWS  ( AC, AV ) then exit;
                 if AV[0]='QUIT'   then if Cmd_QUIT  ( AC, AV ) then exit;
                 if AV[0]='SCRIPT' then if Cmd_SCRIPT( AC, AV ) then exit;
                 if AV[0]='SERVER' then if Cmd_SERVER( AC, AV ) then exit;
                 if AV[0]='TASK'   then if Cmd_TASK  ( AC, AV ) then exit;

              end;
           end;

           // unknown (sub-) command
           SendResult( Res( R_UNKNOWN, 'Command not supported' ) );
           Log( LOGID_INFO, 'Unsupported remote control-command: ' + CmdLine );

        finally
           AV.Free;
        end;

     except
        on E: Exception do begin
           Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
           Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + LogCmdLine );
        end;
     end;
end;

function TSrvRECOCli.Cmd_HELP( Const AC: Integer; AV: TStringList ): Boolean;

   function eq( TestStr: String ): Boolean;
   begin
      Result := ( (AC=1) and (AV[1]=TestStr) );
   end;

var  i: Integer;
     s: String;
begin
     Result := True;

     // HELP [Command]
     SendResult( Res( R_INFO, 'Help text follows' ) );

     if eq('AUTH'  ) then
        SendQuoted( 'AUTH [-SIMPLE] Username Password'#13#10
                  + '   Login with given username and password.'#13#10
                  + 'AUTH -SASL Mechanism [InitialParameters]'#13#10
                  + '   Login with given SASL mechanism.'#13#10
                  + '   Supported mechanisms: ' + Def_LocalReCoSASL )
     else if eq('LOG') then
        SendQuoted( 'LOG LIST [-S Size] [-P Pattern]'#13#10
                  + '   Show end of current logfile.'#13#10
                  + '   -S: Size to read in KB (default 16)'#13#10
                  + '   -P: Only lines which match given regex pattern'#13#10
                  + 'LOG ROTATE'#13#10
                  + '   Start a new log file.' )
     else if eq('NEWS') then
        SendQuoted( 'NEWS (ADD|DEL) GROUP Groupname'#13#10
                  + '   Add/delete the given newsgroup.'#13#10
                  + 'NEWS (ADD|DEL) PULL Groupname Servername'#13#10
                  + '   Add/delete the given news-pull.'#13#10
                  + 'NEWS LIST (GROUP|PULL) [-P Pattern]'#13#10
                  + '   Return list of active groups/pulls.'#13#10
                  + '   -P: Only lines which match given regex pattern' )
     else if eq('QUIT') then
        SendQuoted( 'QUIT'#13#10
                  + '   Logout and terminate connection.' )
     else if eq('SCRIPT') then
        SendQuoted( 'SCRIPT LIST [-P Pattern]'#13#10
                  + '   List available scripts.'#13#10
                  + '   -P: Only lines which match given regex pattern'#13#10
                  + 'SCRIPT START [-W] Scriptname [Scriptparameters]'#13#10
                  + '   Start script with given parameters.'#13#10
                  + '   -W: Wait until finished; also shows script output'#13#10
                  + 'SCRIPT STOP'#13#10
                  + '   Stop all running scripts.' )
     else if eq('SERVER') then
        SendQuoted( 'SERVER (STATE|START|STOP|RESTART) (NNTP|POP3|SMTP|RECO|IMAP)'#13#10
                  + '   Get or change state of the given local server.' )
     else if eq('TASK') then
        SendQuoted( 'TASK (LIST [-P Pattern]|STOP ThreadID)'#13#10
                  + '   LIST: Show current task list.'#13#10
                  + '      -P: Only lines which match given regex pattern'#13#10
                  + '   STOP: Stop thread with ThreadID.' )
     else begin
        s := CmdArray[0].CmdStr;
        for i := 1 to CmdCount-1 do s := s + ', ' + CmdArray[i].CmdStr;
        SendQuoted( 'HELP [Command]'#13#10
                  + '   Show description for the given command.'#13#10
                  + '   Commands: ' + s );
     end;

     SendResult( '.' );
end;

function TSrvRECOCli.Cmd_NEWS( Const AC: Integer; AV: TStringList ): Boolean;
var  RE, s: String;
     TS: TStringList;
     ok: Boolean;
     i, Z: Integer;
begin
     Result := True;
     Z := AC;

     // NEWS (ADD|DEL) GROUP Groupname
     // NEWS (ADD|DEL) PULL Groupname Servername
     if (AV[1]='ADD') or (AV[1]='DEL') then begin
        s := UpperCase( AV[2] );

        if ( Z = 3 ) and ( s = 'GROUP' ) then begin
           if AV[1]='ADD' then ok := CfgHamster.ActiveAdd( AV[3] )
                          else ok := CfgHamster.ActiveDel( AV[3] );
           if ok then SendResult( Res( R_OK0, AV[1] + ' GROUP successful.' ) )
                 else SendResult( Res( R_FAILED0, AV[1] + ' GROUP failed.' ) );
           exit;

        end else if ( Z = 4 ) and ( s = 'PULL' ) then begin
          if AV[1]='ADD' then ok := CfgHamster.PullAdd( AV[4], AV[3] )
                          else ok := CfgHamster.PullDel( AV[4], AV[3] );
           if ok then SendResult( Res( R_OK0, AV[1] + ' PULL successful.' ) ) 
                 else SendResult( Res( R_FAILED0, AV[1] + ' PULL failed.' ) );
           exit;
        end;
     end;

     // NEWS LIST (GROUP|PULL) [-P Pattern]
     if AV[1]='LIST' then begin
        s := UpperCase( AV[2] );
        RE := OptStr( Z, AV, '-P', '.*' );

        if ( Z = 2 ) and ( (s='GROUP') or (s='PULL') ) then begin
           TS := TStringList.Create;
           try
              CfgHamster.Lock.BeginRead;
              try
                 if s='GROUP' then begin
                    for i:=0 to CfgHamster.ActiveCount-1 do
                       TS.Add( CfgHamster.ActiveName[i] );
                 end else begin
                    for i:=0 to CfgHamster.PullCount-1 do
                       TS.Add(      CfgHamster.PullGroup[i]
                             + #9 + CfgHamster.PullServer[i] );
                 end;
              finally
                 CfgHamster.Lock.EndRead;
              end;

              SendResult( Res( R_OK0, s + '-list follows' ) );
              for i:=0 to TS.Count-1 do begin
                 if RE_Match( TS[i], RE, PCRE_CASELESS ) then SendQuoted( TS[i] );
              end;
              SendResult( '.' );
           finally
              TS.Free;
           end;
           exit; 

        end;
     end;

     SendResult( Res( R_SYNTAX, 'Syntax error (invalid action/params)' ) );
end;

function TSrvRECOCli.Cmd_SCRIPT( Const AC: Integer; AV: TStringList ): Boolean;
var  ScriptThread: TThread;

   procedure SendScriptOutput;
   var  s: String;
   begin
     with ScriptThread as TThreadExecuteScript do begin
         with OutputBuffer.LockList do try
            while Count > 0 do begin
               s := Strings[ 0 ];
               Delete( 0 );
               SendQuoted( s );
            end;
         finally
            OutputBuffer.UnlockList;
         end;
      end
   end;

   Procedure RekSearch ( Const Path: String; sl: TStrings);
   Var SR: TSearchRec;
   begin
      if SysUtils.FindFirst( PATH_HSC_RC + Path + '*.*', faAnyFile, SR ) = 0 then begin
         repeat
            If (SR.Attr and faDirectory)=0  then begin
               If LowerCase(ExtractFileExt(SR.Name))='.hsc'
                  then sl.Add( Path + SR.Name )
            end else If SR.Name[1]<>'.' then begin
               // RekSearch ( Path + SR.Name + '\', sl )
            end
         until SysUtils.FindNext( SR ) <> 0;
         SysUtils.FindClose( SR )
      end
   end;

var  i, Z: Integer;
     RE, Scriptname, Scriptpars: String;
     WaitForEnd: Boolean;
     TS: TStringList;
begin
     Result := True;
     Z := AC;

     // SCRIPT START [-W] Scriptname [Scriptparameters]
     if ( Z >= 2 ) and ( AV[1] = 'START' ) then begin

        WaitForEnd := OptBoo( Z, AV, '-W' );

        Scriptname := av[2];
        if Pos( '.', Scriptname ) = 0 then Scriptname := Scriptname + '.hsc';

        if Pos('\', Scriptname) + Pos('/',Scriptname)
         + Pos('..',Scriptname) + Pos(':',Scriptname) > 0 then begin

           SendResult( Res( R_SYNTAX, 'Syntax error (path chars in scriptname)' ) );

        end else begin

           Scriptpars := '';
           If Z > 2 then ScriptPars := AV[3];
           for i:=4 to Z do Scriptpars := ' ' + Scriptpars + AV[i];
           i := StartNewScript( PATH_HSC_RC + Scriptname,
                                Scriptpars, False, ScriptThread,
                                WaitForEnd {BufferOutput},
                                Not WaitForEnd {FreeOnTerminate} );
           if i = 0 then begin
              if WaitForEnd then begin

                 SendResult( Res( R_OK1, 'Script started - waiting' ) );

                 try
                    while WaitForSingleObject( ScriptThread.Handle, 1000 ) = WAIT_TIMEOUT do begin
                       if (ClientSocket=nil) or not ClientSocket.Connected then begin
                          ScriptThread.FreeOnTerminate := True;
                          break;
                       end;
                       SendScriptOutput;
                    end;
                 except
                    ScriptThread.FreeOnTerminate := True;
                 end;

                 try SendScriptOutput except end;
                 try FreeAndNil( ScriptThread ) except end;

                 SendResult( '.' )
              end else begin
                 SendResult( Res( R_OK0, 'Script started' ) )
              end;

           end else begin
              if i=-2 then
                 SendResult( Res( R_FAILED0, 'Script not found' ) )
              else
                 SendResult( Res( R_FAILED1, 'Script start failed' ) );
           end;

        end;

     // SCRIPT LIST [-P Pattern]
     end else if ( Z >= 1 ) and ( AV[1] = 'LIST' ) then begin

        RE := OptStr( Z, AV, '-P', '.*' );

        SendResult( Res( R_OK0, 'List of available scripts follows' ) );

        TS := TStringList.Create;
        try
           TS.Sorted := True;
           TS.Duplicates := dupIgnore;

           RekSearch ( '', TS );

           for i:=0 to TS.Count-1 do try
              if RE_Match( TS[i], RE, PCRE_CASELESS ) then SendQuoted( TS[i] );
           except break end;

        finally
           TS.Free;
        end;
        SendResult( '.' );

     // SCRIPT STOP
     end else if ( Z = 1 ) and ( AV[1] = 'STOP' ) then begin
        {JW} {critical event}
        EnterCriticalSection(CS_Event);
        SetEvent( EVT_STOPSCRIPT );
        LeaveCriticalSection(CS_Event);
        {JW}
        SendResult( Res( R_OK0, 'Scripts stopped' ) )

     end else begin
        SendResult( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
     end;
end;

function TSrvRECOCli.Cmd_AUTH( Const AC: Integer; AV: TStringList ): Boolean;
var  SASL_Name, Par: String;
     Z, i: Integer;
begin
     Result := True;
     Z := AC;

     // AUTH [-SIMPLE] Username Password
     // AUTH -SASL Mechanism [InitialParameters]

     if OptBoo( Z, AV, '-SIMPLE' )
        then SASL_Name := ''
        else SASL_Name := UpperCase( OptStr( Z, AV, '-SASL', '' ) );

     // identify user
     if SASL_Name = '' then begin // SIMPLE

        CurrentUserName := AV[1];
        CurrentUserID := CfgAccounts.LoginID( AV[1], AV[2] );

     end else begin // SASL

        if Pos( ' ' + SASL_Name + ' ',
                ' ' + Def_LocalRECOSASL + ' ' )=0 then begin
           SendResult( Res( R_AUTHFAIL, 'Unrecognized authentication type' ) );
           exit;
        end;
        Par := '';
        If Z > 0 then Par := AV[1];
        for i:=2 to Z do Par := ' ' + Par + AV[i];
        If not Local_SASL_Login( SASL_NAME, Par ) then CurrentUserID := ACTID_INVALID
     end;

     // check user's rc-permission
     if CurrentUserID <> ACTID_INVALID then begin
        With CfgAccounts do try
           Lock;
           If Not Users.Find(CurrentUserID).MayRemoteControl then begin
              CurrentUserID := ACTID_INVALID;
              CurrentUserName := '';
              SendResult( Res( R_AUTHFAIL, 'No permission for remote control server.' ) );
              exit
           end
        finally
           Unlock
        end;
     end;
     // send final result
     if CurrentUserID <> ACTID_INVALID then begin
        SendResult( Res( R_AUTHOK, 'Authentication successful.' ) );
        CurrentInactTimeOut:=Def_LocalTimeoutInactivity*60000;
     end else begin
        SendResult( Res( R_AUTHFAIL, 'Authentication failed.' ) );
     end;
end;

function TSrvRECOCli.Cmd_QUIT( Const AC: Integer; AV: TStringList ): Boolean;
begin
     Result := True;

     // QUIT
     if ClientSocket.Connected then SendResult( Res( R_OK0, 'Closing connection.' ) );
     Sleep( Def_LocalTimeoutQuitDelay );
     try
        if ClientSocket.Connected then ClientSocket.Close;
     except
        on E:Exception do Log(LOGID_DEBUG, 'Exception on .Close: ' + E.Message );
     end;
     Terminate;
end;

function TSrvRECOCli.Cmd_LOG( Const AC: Integer; AV: TStringList ): Boolean;
var  Z, i, KB: Integer;
     RE: String;
     TS: TStringList;
begin
     Result := True;
     Z := AC;

     // LOG LIST [-S Size] [-P Pattern]
     if ( Z >= 1 ) and ( AV[1] = 'LIST' ) then begin

        KB := OptInt( Z, AV, '-S', 8 );
        RE := OptStr( Z, AV, '-P', '.*' );

        SendResult( Res( R_OK0, 'Logfile follows' ) );
        TS := TStringList.Create;
        try
           TS.Text := LogFile.LastLines( KB );
           for i:=0 to TS.Count-1 do try
              if RE_Match( TS[i], RE, PCRE_CASELESS ) then SendQuoted( TS[i] );
           except break end;
        finally
           TS.Free;
        end;
        SendResult( '.' );

     // LOG ROTATE
     end else if ( Z = 1 ) and ( AV[1] = 'ROTATE' ) then begin
        LogFile.RotateLog;
        SendResult( Res( R_OK0, 'Started new logfile.' ) );

     end else begin
        SendResult( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
     end;
end;

function TSrvRECOCli.Cmd_TASK( Const AC: Integer; AV: TStringList ): Boolean;
var  i, Z: Integer;
     RE: String;
     TS: TStringList;
begin
     Result := True;

     Z := AC;

     // TASK LIST [-P Pattern]
     if ( Z >= 1 ) and ( AV[1] = 'LIST' ) then begin

        RE := OptStr( Z, AV, '-P', '.*' );

        SendResult( Res( R_OK0, 'Current task list follows' ) );
        TS := TStringList.Create;
        try
           LogFile.Enter;
           try
              for i:=0 to LogFile.TaskCount-1 do TS.Add( LogFile.TaskLine[i] );
           finally
              LogFile.Leave;
           end;
           for i:=0 to TS.Count-1 do try
              if RE_Match( TS[i], RE, PCRE_CASELESS ) then SendQuoted( TS[i] );
           except break end
        finally
           TS.Free;
        end;
        SendResult( '.' );

     end else
     If ( AV[1] = 'STOP' ) then begin

        If Z = 2 then begin
           ThreadControl.Stop (AV[2]);
           SendResult( Res( R_OK0, 'STOP-Signal sended' ) );
        end else begin
           SendResult( Res( R_SYNTAX, 'Syntax error (Thread-ID missing)' ) )
        end

     end else begin
        SendResult( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
     end;
end;

function TSrvRECOCli.Cmd_SERVER( Const AC: Integer; AV: TStringList ): Boolean;
// SERVER (START|STOP|RESTART|STATE) (NNTP|POP3|SMTP|RECO|IMAP)
var  st: TLocalServerTypes;
begin
     Result := True;

     st := LocalServerStrToType( AV[2] );
     if st = stNONE then begin
        SendResult( Res( R_SYNTAX, 'Syntax error (unknown server type)' ) );
        exit;
     end;

     if ( AC = 2 ) and ( AV[1] = 'START' ) then begin
        if LocalServerActivate( st, saSTART ) then
           SendResult( Res( R_OK0, 'Server is started.' ) )
        else
           SendResult( Res( R_FAILED0, 'Server start failed.' ) );

     end else if ( AC = 2 ) and ( AV[1] = 'STOP' ) then begin
        if st=stRECO then begin
           SendResult( Res( R_OK0, 'Server will be stopped.' ) );
           PostMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_LOCALRECO_ONOFF, 0 );
        end else begin
           if LocalServerActivate( st, saSTOP ) then
              SendResult( Res( R_OK0, 'Server is stopped.' ) )
           else
              SendResult( Res( R_FAILED0, 'Server stop failed.' ) );
        end;

     end else if ( AC = 2 ) and ( AV[1] = 'RESTART' ) then begin
        if st=stRECO then begin
           SendResult( Res( R_OK0, 'Server will be restarted.' ) );
           PostMessage( hWndMainWindow, WM_HAMSTER, HAM_MSG_LOCALRECO_ONOFF, 2 );
        end else begin
           if LocalServerActivate( st, saRESTART ) then
              SendResult( Res( R_OK0, 'Server restarted.' ) )
           else
              SendResult( Res( R_FAILED0, 'Server restart failed.' ) );
        end;

     end else if ( AC = 2 ) and ( AV[1] = 'STATE' ) then begin
        if LocalServerIsActive( st ) then
           SendResult( Res( R_OK0, 'Server is started.' ) )
        else
           SendResult( Res( R_OK1, 'Server is stopped.' ) );

     end else begin
        SendResult( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
     end;
end;

procedure TSrvRECOCli.SendGreeting;
begin
   CheckClientAccess;

   if (IPAccess and IPACC_ACCESS_RW) = IPACC_ACCESS_RW then begin
      SendResult( Res( R_OK0, 'Hamster Remote Control, '
                + GetMyStringFileInfo('ProductName','Hamster') + ' '
                + GetMyVersionInfo(true) ) );

   end else begin                         
      Log( LOGID_WARN, 'Connection refused: ' + ClientID );
      if Assigned(ClientSocket) then
      try
         if ClientSocket.Connected then SendResult( Res( R_NOPERM, 'Permission denied - closing connection.' ) );
         if ClientSocket.Connected then ClientSocket.Close;
      except
      end;
      Terminate;

   end;
end;

constructor TSrvRECOCli.Create( ASocket: TServerClientWinSocket;
   Const AIPAccessScope: Longint; Const ASSLContext: Pointer );
begin
     inherited Create( ASocket, AIPAccessScope, ASSLContext ); {/SSL}

     CurrentInactTimeOut:=Def_LocalReCoLoginTimeout;

     LimitLineLen  := Def_LocalLimitLineLenReCo;
     LimitTextSize := Def_LocalLimitTextSizeReCo;

     CurrentUserID   := ACTID_INVALID;
     CurrentUserName := '';

     if Def_LocalRecoUseTls and not StartSSL then Terminate; {MG}{SSL}
end;

destructor TSrvRECOCli.Destroy;
begin
   inherited;
end;

// ------------------------------------------------------------- TSrvRECO -----

constructor TSrvRECO.Create;
begin
   inherited Create(
      AOwner,
      CfgIni.ReadString ('Setup', 'local.ReCo.serverbind', Def_LocalReCoServerBind ),
      CfgIni.ReadInteger('Setup', 'local.port.ReCo', DEF_LOCALReCoServer_PORT  ),
      CfgIni.ReadInteger('Setup', 'MaxLocalReCoServers', Def_Max_Local_ReCo_Servers),
      CfgIni.ReadInteger('Setup', 'MaxLocalReCoServersPerIP', Def_Max_Local_ReCo_Servers),
      IPACC_SCOPE_RECO,
      TSrvRECOCli
   );
end;

// ----------------------------------------------------------------------------

function TSrvRECOCli.Local_SASL_Login( Mechanism, Parameters: String ): Boolean;
var  s,TimeStamp,Hash: String;
     realm, nonce, cnonce, qop, username, nc, realm2,
     digesturi, response, a1, a2, password, rspauth :String;
begin
   Result := False;
   if Mechanism='LOGIN' then begin
      if Parameters = '' then begin
        s := 'Username:';
        s := EncodeB64( s[1], length(s) );
        s := SendRequest( Res( R_AUTHCHLG_RECO, s ) );
      end else begin
        s := Parameters;
      end;
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      CurrentUserName := TrimWhSpace( s );
      CurrentUserID   := ACTID_INVALID;
      s := 'Password:';
      s := EncodeB64( s[1], length(s) );
      s := SendRequest( '334 ' + s );
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      CurrentUserID := CfgAccounts.LoginID( CurrentUserName, s );
      if CurrentUserID=ACTID_INVALID then CurrentUserName := ''
                                     else Result := True;
   end else
{JW} {SASL}
   if Mechanism='PLAIN' then begin
      if Parameters = '' then begin
        TimeStamp := MidGenerator(Def_FQDNforMIDs);
        s := EncodeB64( TimeStamp[1], length(TimeStamp) );
        s := SendRequest( Res( R_AUTHCHLG_RECO, s ) );
      end else begin
        s := Parameters
      end;
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      CurrentUserName := TrimWhSpace( copy(s,pos(#0,s)+1,500));
      s:=TrimWhSpace( copy(CurrentUserName,
                      pos(#0,CurrentUserName)+1,500));
      CurrentUserName := TrimWhSpace( copy(CurrentUserName,1,
                                           pos(#0,CurrentUserName)-1));
      CurrentUserID   := ACTID_INVALID;
      CurrentUserID := CfgAccounts.LoginID( CurrentUserName, s );
      if CurrentUserID=ACTID_INVALID then CurrentUserName := ''
                                     else Result := True;
   end else

   if Mechanism='DIGEST-MD5' then begin
     // build challenge
     if Def_FQDN<> '' then realm := Def_FQDN
                      else realm := 'localhost';
     s := 'realm="' + realm + '"';
     nonce:= HMAC_SHA1( nonce,
                 IntToHex(PRNG(MaxInt),8)+IntToHex(PRNG(MaxInt),8)+
                 IntToHex(PRNG(MaxInt),8)+IntToHex(PRNG(MaxInt),8));
     nonce := EncodeB64( nonce[1], length(nonce) );
     s := s + ',' + 'nonce="' + nonce + '"';
     qop := 'auth';
     s := s + ',' + 'qop="' + qop + '"';
     s := s + ',' + 'algorithm=' + 'md5-sess';
     Log( LOGID_DEBUG, 'DIGEST-MD5 challenge: ' + s );
     s := EncodeB64( s[1], length(s) );
     // send challenge, get response
     s := SendRequest( Res( R_AUTHCHLG_RECO, s ));
     if s='' then begin
       Log( LOGID_Error, 'empty response received');
       exit;
     end;
     // check response, extract values
     s := DecodeB64( s[1], length(s) );
     Log( LOGID_DEBUG, 'DIGEST-MD5 response: ' + s );
     // check username
     username:=ExtractQuotedParameter(s,'username');
     if username='' then begin
       Log( LOGID_Error, 'missing username in answer');
       exit;
     end;
     CurrentUserName:=trim(username);
     CurrentUserID   := CfgAccounts.Users.IDOf(CurrentUserName);
     if CurrentUserID=ACTID_INVALID then begin
       CurrentUserName := ''
     end else begin
       nonce:=ExtractQuotedParameter(s,'nonce');
       if nonce='' then begin
         Log( LOGID_Error, 'missing nonce in answer');
         exit;
       end;
       cnonce:=ExtractQuotedParameter(s,'cnonce');
       if cnonce='' then begin
         Log( LOGID_Error, 'missing cnonce in answer');
         exit;
       end;
       nc:=ExtractQuotedParameter(s,'nc');
       if nc <> '00000001' then begin
         Log( LOGID_Error, 'wrong nc value');
         exit;
       end;
       qop:=ExtractQuotedParameter(s,'qop');
       if qop='' then begin
         qop:='auth'
       end else begin
         if (lowercase(qop)<>'auth') then begin
           Log(LOGID_ERROR,
               'not supported hash quality protection '+qop);
           exit;
         end
       end;
       realm2:=ExtractQuotedParameter(s,'realm');
       if realm2='' then begin
         Log( LOGID_Error, 'missing realm in answer');
         exit;
       end;
       if realm2<>realm then begin
         Log( LOGID_Error, 'wrong realm in answer');
         exit;
       end;
       digesturi:=ExtractQuotedParameter(s,'digest-uri');
       response:=ExtractQuotedParameter(s,'response');
       if length(response)<>32 then begin
         Log( LOGID_Error, 'wrong response length in answer');
         exit;
       end;
       // build expected response and compare with received one
       password:=CfgAccounts.Users.Find(CurrentUserID).Password;
       a1 := MD5OfStr( username + ':' + realm + ':' + password )
           + ':' + nonce + ':' + cnonce;
       A2:='AUTHENTICATE:'+digesturi;
       s  := MD5toHex( MD5OfStr(
              MD5toHex( MD5OfStr( A1 ) )
              + ':' + nonce + ':' + nc + ':' + cnonce + ':' + qop
              + ':' + MD5toHex( MD5OfStr( A2 ) )
              ) );
       if s<>response then begin
         Log( LOGID_Error, 'login rejected, wrong response value');
         CurrentUserName := '';
         CurrentUserID:=ACTID_INVALID;
         exit;
       end else begin
         result:=True
       end;
       // build rspauth and send it
       a2 := ':' + digesturi;
       rspauth := MD5toHex( MD5OfStr(
                     MD5toHex( MD5OfStr( A1 ) )
                     + ':' + nonce + ':' + nc + ':' + cnonce + ':' + qop
                     + ':' + MD5toHex( MD5OfStr( A2 ) )
                      ) );
       s := 'rspauth=' + rspauth;
       Log( LOGID_DEBUG, 'DIGEST-MD5 rspauth: ' + s );
       s := EncodeB64( s[1], length(s) );
       s := SendRequest( Res( R_AUTHCHLG_RECO, s ) )
     end;
   end else

   if Mechanism='CRAM-MD5' then begin
      TimeStamp := MidGenerator(Def_FQDNforMIDs);
      s := EncodeB64( TimeStamp[1], length(TimeStamp) );
      s := SendRequest( Res( R_AUTHCHLG_RECO, s ) );
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      Hash:=TrimWhSpace( copy(s,PosWhSpace(s)+1,32));
      CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
      CurrentUserID  := CfgAccounts.Users.IDOf(CurrentUserName);
      if CurrentUserID=ACTID_INVALID then begin
         CurrentUserName := '';
         exit;
      end;
      s:=CfgAccounts.Users.Find(CurrentUserID).Password;
      s:=MD5HMAC( s,TimeStamp );
      s:=MD5toHex( s );
      if s=Hash then  Result := True
      else CurrentUserName := '';
   end else
   if Mechanism='CRAM-SHA1' then begin
      TimeStamp := MidGenerator(Def_FQDNforMIDs);
      s := EncodeB64( TimeStamp[1], length(TimeStamp) );
      s := SendRequest( '334 ' + s );
      if s='' then exit;
      s := DecodeB64( s[1], length(s) );
      Log( LOGID_INFO, '> ' + '[...]' );
      Hash:=TrimWhSpace( copy(s,PosWhSpace(s)+1,40));
      CurrentUserName := TrimWhSpace( copy(s,1,PosWhSpace(s)-1));
      CurrentUserID   := CfgAccounts.Users.IDOf(CurrentUserName);
      if CurrentUserID=ACTID_INVALID then begin
         CurrentUserName := '';
      end else begin
         s:=CfgAccounts.Users.Find(CurrentUserID).Password;
         s:=HMAC_SHA1( TimeStamp,s );
         if s=Hash then Result := True
                   else CurrentUserName := ''
      end
   end;
{JW}
end;

end.

