unit cSyncObjects;

interface
uses SysUtils, Windows, Classes, syncobjs;

type
  TReaderWriterLock = class
      // Replacement for TReaderWriterLock, which causes
      // deadlocks in certain situation, especially when promoting a reader
      // to a writer lock.
      // Note: Until a safe replacement is available, this class just treats
      //       reader locks the same as writer locks.    { TODO : RW-Locks }

      private
         FLock: TCriticalSection;

      public
         procedure BeginRead;
         procedure EndRead;
         procedure BeginWrite;
         procedure EndWrite;

         constructor Create;
         destructor Destroy; override;
   end;

   TThreadStringList = class
      // TStringList with synchronized access like TThreadList.
      private
         FLock: TRTLCriticalSection;
         FList: TStringList;
         
      public
         function  LockList: TStringList;
         procedure UnlockList;

         procedure Clear;
         function  Add( Item: String ): Boolean;
         procedure Remove( Item: String );

         constructor Create( ASorted: Boolean; ADuplicates: TDuplicates );
         destructor Destroy; override;
   end;

   TMultiReadExclusiveWriteStringList = class
      // TStringList which allows multiple threads to "read" concurrently but
      // prevents from "writing" concurrently. In this case, "write" means
      // changing the StringList in any way (.Strings[]:='new', Clear, Add,
      // Delete, Sort, ...), and it's your responsibility to access the list
      // with the appropriate BeginRead/BeginWrite method.
      private
         FLock: TReaderWriterLock;
         FList: TStringList;

      public
         function  BeginRead: TStringList;
         procedure EndRead;

         function  BeginWrite: TStringList;
         procedure EndWrite;

         constructor Create( ASorted: Boolean; ADuplicates: TDuplicates );
         destructor Destroy; override;
   end;

implementation

{ TThreadStringList }

constructor TThreadStringList.Create;
begin
   inherited Create;
   InitializeCriticalSection( FLock );
   FList := TStringList.Create;
   FList.Sorted := ASorted;
   FList.Duplicates := ADuplicates;
end;

destructor TThreadStringList.Destroy;
begin
   FList.Free;
   DeleteCriticalSection( FLock );
   inherited Destroy;
end;

procedure TThreadStringList.Clear;
begin
   with LockList do try
      Clear;
   finally
      UnlockList;
   end;
end;

function TThreadStringList.Add(Item: String): Boolean;
begin
   with LockList do try
      Result := ( Duplicates <> dupIgnore) or ( IndexOf(Item) < 0 );
      if Result then Add( Item );
   finally
      UnlockList;
   end;
end;

procedure TThreadStringList.Remove(Item: String);
var  i: Integer;
begin
   with LockList do try
      i := IndexOf( Item );
      if i>=0 then Delete( i );
   finally
      UnlockList;
   end;
end;

function TThreadStringList.LockList: TStringList;
begin
   EnterCriticalSection( FLock );
   Result := FList;
end;

procedure TThreadStringList.UnlockList;
begin
   LeaveCriticalSection( FLock );
end;

{ TMultiReadExclusiveWriteStringList }

constructor TMultiReadExclusiveWriteStringList.Create(ASorted: Boolean;
  ADuplicates: TDuplicates);
begin
   inherited Create;
   FLock := TReaderWriterLock.Create;
   FList := TStringList.Create;
   FList.Sorted := ASorted;
   FList.Duplicates := ADuplicates;
end;

destructor TMultiReadExclusiveWriteStringList.Destroy;
begin
   FreeAndNil( FList );
   FLock.Free;
   inherited Destroy;
end;

function TMultiReadExclusiveWriteStringList.BeginRead: TStringList;
begin
   FLock.BeginRead;
   Result := FList;
end;

function TMultiReadExclusiveWriteStringList.BeginWrite: TStringList;
begin
   FLock.BeginWrite;
   Result := FList;
end;

procedure TMultiReadExclusiveWriteStringList.EndRead;
begin
   FLock.EndRead;
end;

procedure TMultiReadExclusiveWriteStringList.EndWrite;
begin
   FLock.EndWrite;
end;


{ TReaderWriterLock }

{ TReaderWriterLock }

constructor TReaderWriterLock.Create;
begin
   inherited Create;
   FLock := TCriticalSection.Create;
end;

destructor TReaderWriterLock.Destroy;
begin
   BeginWrite;
   FLock.Free;
   inherited Destroy;
end;

procedure TReaderWriterLock.BeginRead;
begin
   BeginWrite;
end;

procedure TReaderWriterLock.BeginWrite;
begin
   FLock.Enter;
end;

procedure TReaderWriterLock.EndRead;
begin
   EndWrite;
end;

procedure TReaderWriterLock.EndWrite;
begin
   FLock.Leave;
end;

end.
