// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit dGroupSelect;

interface

uses Windows, Messages, SysUtils, Classes, Graphics, Forms, Controls, StdCtrls,
  Buttons, ExtCtrls, ComCtrls, cStdForm;

function GetGroupPull (sl: TStrings): boolean;
Procedure DialogPullNewGroups;
Function  DialogPullNewGroupsOpen: boolean;

type
  TDescsFilterThread = class( TThread )
    private
      bError: Boolean;
    protected
      lstDescs : TListBox;
      lblCount : TLabel;
      MatchText: String;
      GrpInfo  : String;
      ErrMsg   : String;
      procedure ListInit;
      procedure ListExit;
      procedure ShowCount;
      procedure ListAdd;
      procedure Execute; override;
    public
      constructor Create( AlstDescs : TListBox;
                          AlblCount : TLabel;
                          AMatchText: String );
  end;

  TPullsFilterThread = class( TThread )
    protected
      SelGroups: TStringList;
      lstPulls : TListBox;
      lblCount : TLabel;
      MatchText: String;
      GrpInfo  : String;
      ErrMsg   : String;
      bError: Boolean;
      procedure ListInit;
      procedure ListExit;
      procedure ShowCount;
      procedure ListAdd;
      procedure Execute; override;
    public
      constructor Create( ASelGroups: TStringList;
                          AlstPulls : TListBox;
                          AlblCount : TLabel;
                          AMatchText: String);
      destructor Destroy; override;   //Gildas&Lamaiziere //Memo leck
  end;

type
  TGetGroupNameDlg = class(THForm)
    pnlBase: TPanel;
    Splitter1: TSplitter;
    pnlTop: TPanel;
    pnlBottom: TPanel;
    hdrPulls: THeaderControl;
    lstPulls: TListBox;
    hdrDescs: THeaderControl;
    lstDescs: TListBox;
    Panel1: TPanel;
    Label1: TLabel;
    emSelGroup: TEdit;
    Panel2: TPanel;
    Label2: TLabel;
    pnlFooter: TPanel;
    OKBtn: TButton;
    CancelBtn: TButton;
    lblDescsCount: TLabel;
    lblPullsCount: TLabel;
    cmdSelAllPulls: TButton;
    cmdSelAllDescs: TButton;
    cbType: TComboBox;
    cbSelServer: TComboBox;
    cbNewsserverSearchType: TComboBox;
    emSelServer: TEdit;
    butHelp: TButton;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure emSelGroupChange(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure lstPullsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure lstDescsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure emSelServerChange(Sender: TObject);
    procedure hdrDescsSectionResize(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure hdrPullsSectionResize(HeaderControl: THeaderControl;
      Section: THeaderSection);
    procedure cmdFindPullsClick(Sender: TObject);
    procedure cmdSelAllPullsClick(Sender: TObject);
    procedure cmdSelAllDescsClick(Sender: TObject);
    procedure lstDescsClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure cbNewsserverSearchTypeClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private declarations }
    First: boolean;
    TestedGroups: TStringlist;
    DescsFilterThread: TDescsFilterThread;
    PullsFilterThread: TPullsFilterThread;
    procedure ShowDescs( StartIt: Boolean );
    procedure ShowPulls( StartIt: Boolean );
  public
    { Public declarations }
    ResultList: TStrings;
  end;

implementation

{$R *.DFM}

uses Global, Config, cPCRE, uTools;

// ------------------------------------------------------------ functions -----

function GetGroupPull (sl: TStrings): boolean;
begin
   With TGetGroupNameDlg.Create(nil) do try
      Resultlist := sl;
      Result := ShowModal=mrOK
   finally
      free
   end
end;

Var Opened: Boolean = false;

Function DialogPullNewGroupsOpen: boolean;
begin
   Result := Opened
end;

Procedure DialogPullNewGroups;
Var s, Server, Group: String; i, p: Integer;
begin
   If Opened then Exit;
   With TGetGroupNameDlg.Create(nil) do try
      Opened := true;
      Resultlist := TStringList.Create;
      try
         If ShowModal=mrOK then begin
            For i := 0 to Resultlist.Count-1 do begin
               s := Resultlist[i];
               p := Pos(',', s);
               If p > 0 then begin
                  Group  := Copy(s, 1, p-1);
                  Server := Copy(s, p+1, Length(s)-p);
                  CfgHamster.PullAdd ( Server, Group )
               end
            end
         end
      finally
         Resultlist.free
      end
   finally
      Opened := false;
      free
   end
end;

// --------------------------------------------------- TDescsFilterThread -----

procedure TDescsFilterThread.ListInit;
begin
     ErrMsg := '';
     lstDescs.Clear;
     lstDescs.Color := clLtGray;
     ShowCount;
     bError := false
end;

procedure TDescsFilterThread.ListExit;
begin
     if not Terminated then begin
        lstDescs.Color := clWindow;
        lstDescs.Enabled := Not bError;
        ShowCount;
     end;
end;

procedure TDescsFilterThread.ShowCount;
begin
     if lstDescs.Items.Count>=32000 then begin
        lblCount.Caption := TrGl('TGetGroupNameDlg', 'Hits', 'Hits') + ': >32000';
     end else begin
        lblCount.Caption := TrGl('TGetGroupNameDlg', 'Hits', 'Hits') + ': ' + inttostr( lstDescs.Items.Count );
     end;
end;

procedure TDescsFilterThread.ListAdd;
begin
     if ErrMsg<>'' then begin
        try lstDescs.Items.Add( ErrMsg ) except end;
        ErrMsg := '';
        bError := true;
        exit;
     end;

     if lstDescs.Items.Count>=32000 then begin
        ShowCount;
        exit;
     end else begin
        try lstDescs.Items.Add( GrpInfo ) except end;
        if (lstDescs.Items.Count mod 100)=0 then ShowCount;
     end;
end;

procedure TDescsFilterThread.Execute;
var  LfdGrp: Integer;
     TS, Descs: TStringList;
     GroupFilename: String;
     RegExGroup: TPCRE;
begin
     Synchronize( ListInit );

     TS := TStringList.Create;
     Descs := TStringList.Create;
     RegExGroup := TPCRE.Create( True, PCRE_CASELESS );

     try
        RegExGroup.Compile( PChar(MatchText) );

        GroupFilename := PATH_SERVER + SRVFILE_ALLDESCS;
        if FileExists2( GroupFilename ) then begin
           TS.LoadFromFile( GroupFilename );

           for LfdGrp:=0 to TS.Count-1 do begin
              if Terminated then break;
              GrpInfo := TS[LfdGrp];
              if MatchText='' then begin
                 Synchronize( ListAdd );
              end else begin
                 if RegExGroup.Exec( PChar(GrpInfo), 0 ) then Synchronize( ListAdd );
              end;

           end;

        end;

     except
        on E: Exception do begin ErrMsg:=E.Message; Synchronize(ListAdd) end;
     end;

     TS.Free; Descs.free; RegExGroup.Free;
     Synchronize( ListExit );
end;

constructor TDescsFilterThread.Create( AlstDescs : TListBox;
                                       AlblCount : TLabel;
                                       AMatchText: String);
begin
     inherited Create( True );

     lstDescs  := AlstDescs;
     lblCount  := AlblCount;
     MatchText := AMatchText;

     FreeOnTerminate := False;
     Resume;
end;

// --------------------------------------------------- TPullsFilterThread -----

procedure TPullsFilterThread.ListInit;
begin
     ErrMsg := '';
     lstPulls.Clear;
     lstPulls.Color := clLtGray;
     ShowCount;
     bError := false
end;

procedure TPullsFilterThread.ListExit;
begin
     if not Terminated then begin
        lstPulls.Color := clWindow;
        lstPulls.Enabled := Not bError;
        ShowCount;
     end;
end;

procedure TPullsFilterThread.ShowCount;
begin
     if lstPulls.Items.Count>=32000 then begin
        lblCount.Caption := TrGl('TGetGroupNameDlg', 'Hits', 'Hits') + ': >32000';
     end else begin
        lblCount.Caption := TrGl('TGetGroupNameDlg', 'Hits', 'Hits') + ': ' + inttostr( lstPulls.Items.Count );
     end;
end;

procedure TPullsFilterThread.ListAdd;
begin
     if ErrMsg<>'' then begin
        try lstPulls.Items.Add( ErrMsg ) except end;
        ErrMsg := '';
        bError := true;
        exit;
     end;

     if lstPulls.Items.Count>=32000 then begin
        ShowCount;
        exit;
     end else begin
        try lstPulls.Items.Add( GrpInfo ) except end;
        if (lstPulls.Items.Count mod 100)=0 then ShowCount;
     end;
end;

function CreateDeltaList( GroupNames, GroupsList: TStrings ): TStringList;

   function GetWhPart( s: String ): String;
   var  j: Integer;
   begin
      j := PosWhSpace( s );
      if j=0 then Result:=s else Result:=copy(s,1,j-1);
   end;

var  i1, i2: Integer;
     c1, c2: Integer;
     g1, g2: String;
begin
   Result := TStringList.Create;
   Result.Sorted := True;
   Result.Duplicates := dupIgnore;

   i1 := 0;
   i2 := 0;
   c1 := GroupNames.Count;
   c2 := GroupsList.Count;

   if i1<c1 then g1:=GroupNames[i1];
   if i2<c2 then g2:=GetWhPart( GroupsList[i2] );

   Result.BeginUpdate;
   while (i1<c1) and (i2<c2) do begin
      case AnsiCompareText( g1, g2 ) of
         0: begin
               Result.Add( g1 );
               inc(i1); if i1<c1 then g1:=GroupNames[i1];
               inc(i2); if i2<c2 then g2:=GetWhPart( GroupsList[i2] );
            end;
        -1: begin
               inc(i1); if i1<c1 then g1:=GroupNames[i1];
            end;
        +1: begin
               inc(i2); if i2<c2 then g2:=GetWhPart( GroupsList[i2] );
            end;
      end;
   end;
   Result.EndUpdate;
end;

procedure TPullsFilterThread.Execute;
var  LfdSrv, LfdGrp: Integer;
     TS            : TStringList;
     GroupFilename : String;
     RegExGroup    : TPCRE;
     SrvName       : String;
     lstTemp       : TStringList;
begin
     Synchronize( ListInit );

     TS := TStringList.Create;
     RegExGroup := TPCRE.Create( True, PCRE_CASELESS );

     try
        if not Terminated then RegExGroup.Compile( PChar(MatchText) );

        for LfdSrv:=0 to CfgHamster.ServerCount-1 do begin
           if Terminated then break;
           SrvName := CfgHamster.ServerName[LfdSrv];
           GroupFilename := CfgHamster.ServerPath[LfdSrv] + 'groups.txt';
           if FileExists2( GroupFilename ) then begin

              TS.LoadFromFile( GroupFilename );
              if Terminated then break;

              TS.Sort;
              if Terminated then break;

              lstTemp := CreateDeltaList( SelGroups, TS );
              for LfdGrp:=0 to lstTemp.Count-1 do begin
                 if Terminated then break;
                 GrpInfo := lstTemp[LfdGrp] + ',' + SrvName;
                 if MatchText='' then begin
                    Synchronize( ListAdd );
                 end else begin
                    if RegExGroup.Exec( PChar(SrvName), 0 ) then Synchronize( ListAdd );
                 end;
              end;
              lstTemp.Free;

           end;

        end

     except
        on E: Exception do begin ErrMsg:=E.Message; Synchronize(ListAdd) end;
     end;

     TS.Free;
     RegExGroup.Free;
     Synchronize( ListExit );
end;

constructor TPullsFilterThread.Create( ASelGroups: TStringList;
                                       AlstPulls : TListBox;
                                       AlblCount : TLabel;
                                       AMatchText: String);
begin
     inherited Create( True );

     SelGroups := ASelGroups;
     lstPulls  := AlstPulls;
     lblCount  := AlblCount;
     MatchText := AMatchText;

     FreeOnTerminate := False;
     Resume;
end;

destructor TPullsFilterThread.Destroy;                       //Gildas&Lamaiziere //Memo leck
begin
  FreeAndNil(SelGroups);
  inherited destroy;
end;

// ----------------------------------------------------- TGetGroupNameDlg -----

procedure TGetGroupNameDlg.FormCreate(Sender: TObject);
Var i: Integer;
begin
   First := true;
   TestedGroups := TStringlist.Create;
   TestedGroups.sorted := true;
   DescsFilterThread := nil;
   PullsFilterThread := nil;

   With cbSelServer.Items do begin
      Clear;
      Add ('<'+Tr('FilterOnAllServer', 'All available newsserver')+'>');
      For i := 0 to CfgHamster.ServerCount-1 do Add (CfgHamster.ServerName[i])
   end;

   LoadWindowState( Self, 'SelPulls' );
   pnlTop.Height := CfgIni.ReadInteger( 'SelPulls', 'Split', pnlTop.Height );
   hdrDescs.Sections[0].Width := CfgIni.ReadInteger(
      'SelPulls', 'DescsWidth0', hdrDescs.Sections[0].Width
   );
   hdrPulls.Sections[0].Width := CfgIni.ReadInteger(
      'SelPulls', 'PullsWidth0', hdrPulls.Sections[0].Width
   );

   lstDescs.ItemHeight := lstDescs.Canvas.TextHeight('Gg');
   lstPulls.ItemHeight := lstPulls.Canvas.TextHeight('Gg');

   try
      cbType.ItemIndex := CfgIni.ReadInteger ('DlgGroupSelect', 'SearchType', 0);
      emSelGroup.Text  := CfgIni.ReadString ('DlgGroupSelect', 'SearchGroup', '');
      cbSelServer.ItemIndex := CfgIni.ReadInteger ('DlgGroupSelect', 'Server', 0);
      With cbNewsserverSearchType do try
         ItemIndex := CfgIni.ReadInteger ('DlgGroupSelect', 'NewsserverSearchType', 0);
         If ItemIndex < 0 then ItemIndex := 0
      except
         ItemIndex := 0
      end;
      emSelServer.text := CfgIni.ReadString ('DlgGroupSelect', 'SearchServer', '');
   except
      cbType.ItemIndex := 0;
      cbSelServer.ItemIndex := 0;
      emSelGroup.Text := '';
      cbNewsserverSearchType.ItemIndex := 0;
      emSelServer.Text := ''
   end;
   cbNewsserverSearchType.OnClick(nil)
end;
procedure TGetGroupNameDlg.FormDestroy(Sender: TObject);
begin
   CfgIni.WriteInteger ('DlgGroupSelect', 'SearchType', cbType.ItemIndex);
   CfgIni.WriteString ('DlgGroupSelect', 'SearchGroup', emSelGroup.Text);
   CfgIni.WriteString ('DlgGroupSelect', 'SearchServer', emSelServer.Text);
   CfgIni.WriteInteger ('DlgGroupSelect', 'NewsserverSearchType', cbNewsserverSearchType.ItemIndex);
   TestedGroups.free
end;

procedure TGetGroupNameDlg.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
   ShowDescs( False );
   ShowPulls( False );
   try CfgIni.WriteInteger( 'SelPulls', 'Split', pnlTop.Height ) except end;
   SaveWindowState( Self, 'SelPulls' );
end;

procedure TGetGroupNameDlg.ShowDescs( StartIt: Boolean );
Var s, s2: String; i: Integer;
begin
     if Assigned( DescsFilterThread ) then begin
        DescsFilterThread.Terminate;
        if Assigned(PullsFilterThread) then ShowPulls( False );
        DescsFilterThread.WaitFor;
        DescsFilterThread.Free;
        DescsFilterThread := nil;
     end;

     if StartIt then begin
        s2 := emSelGroup.Text; s := s2;
        If s2 > '' then Case cbType.ItemIndex of
           1: s := EscRegEx(s2);
           2: s := '^'+EscRegEx(s2);
           3: begin
                 s := '^';
                 For i:=1 to Length(s2)-1 do s := s + s2[i] + '[^. ]*\.';
                 s := s + s2[Length(s2)]
              end
        end;
        DescsFilterThread := TDescsFilterThread.Create(
           lstDescs, lblDescsCount, s)
     end;
end;

procedure TGetGroupNameDlg.ShowPulls( StartIt: Boolean );
var  SelGroups: TStringList;
     LfdGrp, j: Integer;
     s        : String;
begin
     if Assigned( PullsFilterThread ) then begin
        PullsFilterThread.Terminate;
        PullsFilterThread.WaitFor;
        PullsFilterThread.Free;
        PullsFilterThread := nil;
     end;

     if StartIt then begin

        // create list of selected group-names
        SelGroups := TStringList.Create;
        SelGroups.Sorted := True;
        SelGroups.Duplicates := dupIgnore;
        SelGroups.BeginUpdate;
        for LfdGrp:=0 to lstDescs.Items.Count-1 do begin
           if lstDescs.Selected[LfdGrp] then begin
              s := lstDescs.Items[LfdGrp];
              j := PosWhSpace( s );
              if j>0 then s:=copy(s,1,j-1);
              SelGroups.Add( s );
           end;
        end;
        SelGroups.EndUpdate;

        If cbNewsserverSearchType.ItemIndex = 1 then begin
           s := emSelServer.text
        end else begin
           If cbSelServer.ItemIndex = 0 then s := '' else s := cbSelServer.Text
        end;
        PullsFilterThread := TPullsFilterThread.Create(
           SelGroups, lstPulls, lblPullsCount, s
        );
     end;
end;

procedure TGetGroupNameDlg.emSelGroupChange(Sender: TObject);
begin
     ShowDescs( True );
end;

procedure TGetGroupNameDlg.emSelServerChange(Sender: TObject);
begin
   If Not First then ShowPulls( True );
end;

procedure TGetGroupNameDlg.OKBtnClick(Sender: TObject);
var  g, s: String;
     i, j: Integer;
begin
     ResultList.Clear;
     for i:=0 to lstPulls.Items.Count-1 do begin
        if lstPulls.Selected[i] then begin
           g := lstPulls.Items[i];
           s := '';
           j := Pos( ',', g );
           if j>0 then begin
              s := copy( g, j+1, 999 );
              g := copy( g, 1, j-1 );
              j := Pos( ',', s );
              if j>0 then s:=copy(s,1,j-1);
           end;
           ResultList.Add( g + ',' + s )
        end;
     end;
end;

procedure TGetGroupNameDlg.lstPullsDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var  t, g, s: String;
     j      : Integer;
     R      : TRect;
begin
     with (Control as TListBox).Canvas do begin
        t := (Control as TListBox).Items[Index];
        j := Pos( ',', t );
        if j>0 then begin
           g := copy( t, 1, j-1 );
           s := copy( t, j+1, 999 );
           j := Pos( ',', s );
           if j>0 then s:=copy(s,1,j-1);
        end else begin
           g := t;
           s := '';
        end;

        if CfgHamster.ExistPull( s, g ) then begin
           if [odSelected]<=State then Font.Color := clYellow
                                  else Font.Color := clBlue;
        end;

        FillRect(Rect);
        R := Rect;
        R.Right := hdrPulls.Sections[0].Width;
        if R.Right>4 then dec(R.Right,4);
        TextRect( R, R.Left, R.Top, g );

        R := Rect;
        inc( R.Left, hdrPulls.Sections[0].Width );
        TextOut( R.Left, R.Top, s );
     end;
end;

procedure TGetGroupNameDlg.lstDescsDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var  t, g, d: String;
     j, i   : Integer;
     R      : TRect;
begin
     with (Control as TListBox).Canvas do begin
        t := (Control as TListBox).Items[Index];
        j := Pos( #9, t );
        if j>0 then begin
           g := copy( t, 1, j-1 );
           d := copy( t, j+1, 999 );
        end else begin
           g := t;
           d := '';
        end;

        i := TestedGroups.IndexOf(g);
        If i<0 then begin
           i := TestedGroups.Add (g);
           If CfgHamster.ExistPullServer ( g ) then TestedGroups.Objects[i] := Pointer(1)
        end;
        if TestedGroups.Objects[i] = Pointer(1) then begin
           if [odSelected]<=State then Font.Color := clYellow
                                  else Font.Color := clBlue;
        end;

        FillRect(Rect);
        R := Rect;
        R.Right := hdrDescs.Sections[0].Width;
        if R.Right>4 then dec(R.Right,4);
        TextRect( R, R.Left, R.Top, g );

        R := Rect;
        inc( R.Left, hdrDescs.Sections[0].Width );
        TextOut( R.Left, R.Top, d );
     end;
end;

procedure TGetGroupNameDlg.hdrDescsSectionResize(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
     lstDescs.Refresh;
     CfgIni.WriteInteger( 'SelPulls', 'DescsWidth0', hdrDescs.Sections[0].Width );
end;

procedure TGetGroupNameDlg.hdrPullsSectionResize(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
     lstPulls.Refresh;
     CfgIni.WriteInteger( 'SelPulls', 'PullsWidth0', hdrPulls.Sections[0].Width );
end;

procedure TGetGroupNameDlg.cmdFindPullsClick(Sender: TObject);
begin
     ShowPulls( True );
end;

procedure TGetGroupNameDlg.cmdSelAllPullsClick(Sender: TObject);
begin
   With lstPulls do If Enabled then SendMessage( Handle, LB_SETSEL, LongInt(True), -1 );
end;

procedure TGetGroupNameDlg.cmdSelAllDescsClick(Sender: TObject);
begin
   With lstDescs do If Enabled then SendMessage( Handle, LB_SETSEL, LongInt(True), -1 );
   cmdFindPullsClick (nil)
end;

procedure TGetGroupNameDlg.lstDescsClick(Sender: TObject);
begin
   cmdFindPullsClick (nil)
end;

procedure TGetGroupNameDlg.FormActivate(Sender: TObject);
begin
   If First then begin
      First := false;
      ShowDescs( True )
   end
end;

procedure TGetGroupNameDlg.cbNewsserverSearchTypeClick(Sender: TObject);
Var b: boolean;
begin
   b := cbNewsserverSearchType.ItemIndex <> 1;
   emSelServer.visible := Not b;
   cbSelServer.visible := b;
   If Not First then ShowPulls( True )
end;

procedure TGetGroupNameDlg.FormResize(Sender: TObject);
Var x: Integer;
begin
   x := ((ClientHeight - pnlFooter.Height) div 3)*2;
   If x > 5 then pnlTop.Height := x;
   With CancelBtn do Left := Parent.Width - Width - Top;
   With OKBtn do Left := CancelBtn.Left - Top - Width;
end;

end.
