// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit tBase; // Base-classes for Hamster-threads

// ----------------------------------------------------------------------------
// Contains the base classes for the Hamster-threads.
// ----------------------------------------------------------------------------

interface

uses Windows, SysUtils, Classes;

type
   // base-class for all threads
   TBaseThread = class( TThread )
      protected
         ThreadName: String;
         procedure TLog( ID: Word; Msg: String );
         procedure AddToThreadList; virtual; //HSR //One_SD_TL
         Procedure RemoveFromThreadList;
         Function ShutDownReq: Boolean;
      public
         constructor Create( Const AThreadName: String;
                             Const DoFreeOnTerminate: Boolean = True );
         destructor Destroy; override;
   end;

   // base-class for all threads, that count as "active tasks"
   TTaskThread = class( TBaseThread )
      public
         procedure wait; {JW} {Tasks}
         constructor Create( Const AThreadName: String );
         destructor Destroy; override;
   end;

   {HSR} {One_SD}
   TThreadKind = (tkBase, tkClient, tkScript, tkRAS, tkPURGE, tkHistRebuild, tkGlobalRebuild, tkStatistics);
   PThreadInfo = ^TThreadInfo;
   TThreadInfo = Record
     ID: Cardinal;
     Kind: TThreadKind;
   end;
   TThreadControl = class
   private
     FShutdownList : TStringList;
     FThreadList: TThreadList;
     function IDToStr(const ID: Cardinal): String;
     Procedure Remove;
   public
     constructor Create;
     destructor destroy; override;
     Procedure Add ( Const AktID: Cardinal; Const AktKind: TThreadKind );
     Function Stop ( Const AktID: Cardinal ): boolean; overload;
     Function Stop ( Const AktIDStr: String ): boolean; overload;
     Function ShutDown (Const Delete: Boolean = false) : Boolean;
   end;

   Var ThreadControl: TThreadControl;
   Function ThreadShutDown (Const Delete: Boolean; Const default: Boolean = false): boolean;

implementation

uses Global, cClientNNTP, cLogFile, cStdForm;

{ TBaseThread }

procedure TBaseThread.TLog( ID: Word; Msg: String );
begin
   Log( ID, ThreadName + ' ' + Msg );
end;

constructor TBaseThread.Create( Const AThreadName: String;
   Const DoFreeOnTerminate: Boolean = True );
begin
   inherited Create( True ); // suspended

   ThreadName := AThreadName;
   If DoFreeOnTerminate then FreeOnTerminate := True;

   AddToThreadList; //HSR //One_SD_TL
end;

destructor TBaseThread.Destroy;
begin
   Sleep(1); // JH
   // remove thread from "Threads"-window
   Logfile.RemoveTask( LowerCase( IntToHex(ThreadID,1) ) );
   RemoveFromThreadList;
   inherited Destroy;
end;

procedure TBaseThread.AddToThreadList;
begin
   ThreadControl.Add ( ThreadID, tkBase )
end;

procedure TBaseThread.RemoveFromThreadList;
begin
   ThreadControl.Remove
end;

function TBaseThread.ShutDownReq: Boolean;
begin
   Result := AllShutDownReq or ThreadControl.Shutdown
end;

{ TTaskThread }

constructor TTaskThread.Create( Const AThreadName: String );
begin
   inherited Create( AThreadName );
   EnterCriticalSection( CS_COUNTER );
   try
      inc( CntActiveTasks );    // increment number of active tasks
      {JW} {critical event}
      EnterCriticalSection(CS_Event);
      ResetEvent( EVT_ISIDLE ); // signal: active tasks
      LeaveCriticalSection(CS_Event);
      {JW}
   finally
      LeaveCriticalSection( CS_COUNTER );
   end
end;

destructor TTaskThread.Destroy;
begin
   EnterCriticalSection( CS_COUNTER );
   dec( CntActiveTasks ); // decrement number of active tasks
   if CntActiveTasks<=0 then begin
      if ArtCurLoading<>nil then ArtCurLoading.Clear;  //HRR: delete possible remained MsgIds (i. e. Pull died)
      IncCounter( CntOutboxChk, 1 ); // re-count no. of msg's in outbox
      {JW} {critical event}
      EnterCriticalSection(CS_Event);
      SetEvent( EVT_ISIDLE );        // signal: idle, no active tasks
      LeaveCriticalSection(CS_Event);
      {JW}
   end;
   LeaveCriticalSection( CS_COUNTER );

   inherited Destroy;
end;

{JW} {Task}
procedure tTaskThread.Wait;

   function GetRunningTasks:integer;
   begin
    EnterCriticalSection(CS_Counter);
    result:=CntActiveTasks-CntWaitTasks;
    LeaveCriticalSection(CS_Counter);
   end;

begin
 if Def_MaxTasks > 0 then begin
   EnterCriticalSection(CS_Limiter);
   if (GetRunningTasks >= Def_MaxTasks) and (Def_MaxTasks > 0) then begin
     IncCounter(CntWaitTasks,1);
     TLog( LOGID_Info, 'Limiter: suspend task' );
     while (GetRunningTasks >= Def_MaxTasks) and (Def_MaxTasks > 0) and not ShutdownReq do begin
      LeaveCriticalSection(CS_Limiter);
      sleep(1000);
      EnterCriticalSection(CS_Limiter);
     end;
     TLog( LOGID_Info, 'Limiter: resume task' );
     IncCounter(CntWaitTasks,-1);
   end;
   LeaveCriticalSection(CS_Limiter);
 end;
end;

{ TThreadControl }

Function ThreadShutDown (Const Delete: Boolean; Const default: Boolean = false): boolean;
begin
   Result := ThreadControl.Shutdown(Delete) or default
end;

constructor TThreadControl.Create;
begin
   FShutDownList := TStringList.Create;
   FThreadList := tThreadList.Create;
end;

destructor TThreadControl.destroy;
begin
   FShutdownlist.free;
   FThreadList.Free;
   inherited
end;

procedure TThreadControl.Add(const AktID: Cardinal;
  const AktKind: TThreadKind);
Var Temp: PThreadInfo;
begin
   new(temp);
   Temp^.ID  := AktID;
   Temp^.Kind := AktKind;
   FThreadList.Add(temp)
end;

procedure TThreadControl.Remove;
Var temp : tList; i: Integer;
begin
   EnterCriticalSection(CS_SHUTDOWNLIST);
   try
      i := FShutDownList.IndexOf(IDToStr(GetCurrentThreadID));
      If i > -1 then FShutDownList.Delete(i);
      temp := FThreadList.LockList;
      try
        for i := temp.count-1 downto 0 do begin
          If TThreadInfo(temp.items[i]^).id = GetCurrentThreadID then begin
             Dispose(PThreadInfo(temp.items[i]));
             temp.Delete(i);
          end;
        end;
      finally
         FThreadList.unlocklist;
      end;
   finally
      LeaveCriticalSection(CS_SHUTDOWNLIST);
   end
end;

function TThreadControl.ShutDown(const Delete: boolean = false): Boolean;
var
  i, i2 : integer;
  temp : tList;
  lDel : Boolean;
  s : string;
begin
   Result := false;
   EnterCriticalSection(CS_SHUTDOWNLIST);
   try
      If FShutDownList.Count > 0 then begin
         i := FShutDownList.IndexOf(IDToStr(GetCurrentThreadID));
         If i > -1 then begin
            Result := true;
            if Delete then FShutDownList.Delete(i);
         end;
         temp := FThreadList.LockList;
         try
            for i := FShutDownList.count-1 downto 0 do begin //One_SD_TL
               lDel := true;
               for i2 := 0 to Temp.Count-1 do begin
                  s :=lowercase(IDToStr(tThreadInfo(temp.items[i2]^).id));
                  if s = FShutDownList[i] then begin
                     lDel := false;
                     break
                  end
               end;
               if lDel then FShutDownList.Delete(i)
            end
         finally
            FThreadList.UnLockList;
         end
      end
   finally
      LeaveCriticalSection(CS_SHUTDOWNLIST);
   end
end;

Function TThreadControl.IDToStr(Const ID: Cardinal): String;
begin
   Result := LowerCase(IntToHex(ID,MaxInt))
end;

Function TThreadControl.Stop(const AktID: Cardinal): boolean;
begin
   Result := Stop ( IDToStr(AktID) )
end;

Function TThreadControl.Stop(const AktIDStr: String): boolean;
begin
   EnterCriticalSection(CS_SHUTDOWNLIST);
   try
      Result := FShutDownList.indexOf(AktIDStr) < 0;
      If Result then begin
         Log ( LOGID_SYSTEM, TrGlF(kLog, 'Thread.STOP-Signal',
            'STOP-Signal for Thread %s sent!', AktIDStr));
         FShutDownList.Add (AktIDStr)
      end
   finally
      LeaveCriticalSection(CS_SHUTDOWNLIST);
   end
end;

initialization
   ThreadControl := TThreadControl.Create;
finalization
   FreeAndNil(ThreadControl)
end.
