// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit tScript; // Threads which runs scripts

// ----------------------------------------------------------------------------
// Contains the thread which executes scripts and the Hamster-related
// enhancements of the script-engine.
// ----------------------------------------------------------------------------

{$INCLUDE Compiler.inc}

interface

{JW} {SSL CMD}
uses Windows, SysUtils, Forms, Classes, {$IFDEF H_NEED_VARIANTS} Variants, {$ENDIF}
     tBase, cHscHamster, cHSCEngine, cSyncObjects;
{JW}

type
  TScriptmode = (smFile, smLine);
  TThreadExecuteScript = class( TBaseThread )
    private
      FScriptMode: TScriptmode;
      FScriptFilename: String;
      FScriptCommands: TStringlist;
      FParameterList: String; // JAWO Params 4.10.2000 (new variable)
      FOutputBuffer: TThreadStringList;
    protected
      procedure Execute; override;
    public
      property OutputBuffer: TThreadStringList read FOutputBuffer;
      constructor Create( const AScriptFile, AParamsText: String;
                          const DoBufferOutput: Boolean = False;
                          const DoFreeOnTerminate: Boolean = True ); overload;
      constructor Create( AScript: TStrings;
                          Const AParameterList: String;
                          const DoFreeOnTerminate: Boolean ); overload;
      destructor Destroy; override;
  end;

  function StartNewScript ( ScriptFile, ParameterList: String;
                            WaitForEnd : Boolean ): Integer; overload;
  function StartNewScript( const ScriptFile, ParamsText: String;
                           const WaitForEnd : Boolean;
                           out   ScriptThread: TThread;
                           const DoBufferOutput: Boolean = False;
                           const DoFreeOnTerminate: Boolean = True ): Integer; overload;
  function StartScriptLines ( AScript: TStrings; Const par: String; Const Wait: boolean ): Integer;

implementation

uses Global, Config, cPasswordFile, uTools, cArticle, cArtFiles, cFiltersNews,
     tTransfer, tMaintenance, Controls, Dialogs, XApp, cStdForm, Main,
     cMailRouter, cLogfile;

procedure TThreadExecuteScript.Execute;
var  MyScriptList: TStringList;
     MyScriptName: String;
     HSCResult: THscVariant;
begin
   If FScriptMode = smFile then begin
      TLog( LOGID_SYSTEM, TrGl(kLog, 'Skript.Start', 'Start') )
   end;

   AllShutdownReq := False;
   {JW} {critical event}
   EnterCriticalSection(CS_Event);
   ResetEvent( EVT_STOPSCRIPT );
   LeaveCriticalSection(CS_Event);
   {JW}

   MyScriptList := TStringList.Create;
   try
      If FScriptMode = smFile then begin
         MyScriptName := ExtractFilename( FScriptFilename );
         try
            MyScriptList.LoadFromFile( FScriptFilename );
         except
            TLog( LOGID_SYSTEM, TrGl(kLog, 'Skript.End', 'End' ) );
            TLog( LOGID_ERROR, TrGlF(kLog, 'Skript.ErrorOpening',
                  'Error opening %s', FScriptFilename) );
            exit
         end
      end else begin
         MyScriptName := '';
         MyScriptList.Assign ( FScriptCommands )
      end;

      HSCResult := THscVariant.Create( 0, false );
      with THscEngineHamster.Create( Self, PATH_HSM, Synchronize, FOutputBuffer ) do try
         ExecuteFromList( MyScriptName, MyScriptList, FParameterList, HSCResult );
      finally
         Free;
         HSCResult.Free
      end;

      NewsHistory.SaveToFile;

      if ShutDownReq and RasDialer.IsConnected then begin
         if MessageDlg( 'Hangup?', mtConfirmation, [mbYes,mbNo], 0 )=mrYes then begin
            RasDialer.HangUp
         end
      end;
   finally
      MyScriptList.Free;
      If FScriptMode = smFile then TLog( LOGID_SYSTEM, TrGl(kLog, 'Skript.End', 'End' ) )
   end;
end;

constructor TThreadExecuteScript.Create(
                          const AScriptFile, AParamsText: String;
                          const DoBufferOutput: Boolean = False;
                          const DoFreeOnTerminate: Boolean = True );
begin
     inherited Create( '{script ' + ExtractFilename(AScriptFile)+ '}',
        DoFreeOnTerminate );
     FScriptMode := smFile;
     FScriptFileName := AScriptFile;
     FScriptCommands := NIL;
     FParameterList := AParamsText;
     if DoBufferOutput
        then FOutputBuffer := TThreadStringList.Create( False, dupAccept )
        else FOutputBuffer := nil;
end;

constructor TThreadExecuteScript.Create( AScript: TStrings;
   Const AParameterList: String; Const DoFreeOnTerminate: boolean );
begin
   inherited Create( '{script}', DoFreeOnTerminate );
   FScriptMode := smLine;
   FScriptFilename := '';
   FScriptCommands := TStringList.Create;
   FScriptCommands.Assign(AScript);
   FParameterList := AParameterList;
   FOutputBuffer := NIL;
end;

destructor TThreadExecuteScript.Destroy;
begin
  If FScriptMode = smFile
     then MainLog (LOGID_SYSTEM, TrGlF(kLog, 'Script.end', 'Script %s ended.', FScriptFilename) )
     else MainLog (LOGID_SYSTEM, TrGl(kLog, 'Scriptline.end', 'Scriptline(s) executed.') );
  If Assigned(FScriptCommands) then FreeAndNil(FScriptCommands);
  inherited
end;

function StartNewScript( const ScriptFile, ParamsText: String;
                         const WaitForEnd : Boolean;
                         out   ScriptThread: TThread;
                         const DoBufferOutput: Boolean = False;
                         const DoFreeOnTerminate: Boolean = True ): Integer;
var  Filename, Params: String;
begin
   Result := -1;
   ScriptThread := nil;
   AllShutdownReq := False;

   Log( LOGID_SYSTEM, TrGl(kLog, 'Script.Starting', 'Starting script') + ': ' +
        ScriptFile + ', Wait=' + inttostr(Ord(WaitForEnd)));
   if pos (LF, ParamsText)<> 0 then begin
      Log( LOGID_DEBUG, TrGlF(kLog, 'Script.StartingPars',
          'Starting script: Parameters = "%s"',
          [copy (ParamsText, pos (LF, ParamsText)+1, Length(ParamsText))]))
   end else begin
      Log( LOGID_DEBUG,
            TrGl(kLog, 'Script.StartingNoPars', 'Starting script'))
   end;

   FileName := ScriptFile;
   If ExtractFileExt(FileName) = '' then FileName := FileName + '.hsc';
   Filename := AddPathIfNeeded(FileName, PATH_HSC);
   If Not FileExists2( Filename ) then begin
      Log( LOGID_ERROR, TrGlF(kLog, 'Script.Not_Found', 'Script "%s" not found!', Scriptfile) );
      Result := -2;
      exit
   end;
   if (Filename > '') then begin
      Params := ScriptFile;
      If ParamsText > '' then Params := Params + LF + ParamsText;
      if WaitForEnd then begin
         ScriptThread := TThreadExecuteScript.Create(
                         Filename, Params, DoBufferOutput,
                         false );
         ScriptThread.Resume;
         ScriptThread.WaitFor;
         FreeAndNil(ScriptThread)
      end else begin
         ScriptThread := TThreadExecuteScript.Create(
                         Filename, Params, DoBufferOutput,
                         DoFreeOnTerminate );
         ScriptThread.Resume
      end;
      Result := 0;
   end
end;

function StartNewScript ( ScriptFile, ParameterList: String;
                            WaitForEnd : Boolean ): Integer; overload;
var  ScriptThread: TThread;
begin
    Result := StartNewScript( ScriptFile, ParameterList, WaitForEnd,
                              ScriptThread, False, True );
end;

function StartScriptLines ( AScript: TStrings; Const Par: String; Const Wait: boolean ): Integer;
var  ScriptThread: TThread;
begin
   AllShutdownReq := False;
   Result := 0;
   If Wait then begin
      With TThreadExecuteScript.Create( AScript, '' + LF + Par, false) do try
         resume; WaitFor
      finally
         free
      end
   end else begin
      ScriptThread := TThreadExecuteScript.Create( AScript, '' + LF + Par, true);
      ScriptThread.resume;
      if Not Assigned(ScriptThread) then begin
         Log( LOGID_ERROR, TrGl(kLog, 'ScriptLines.Not_Created', 'Cannot create Thread for Scriptline(s)!') )
      end
   end
end;

end.

